/* This file is part of Cloudy and is copyright (C) 1978-2004 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*atmdat_H_rad_rec calculates state-specific recombination rates for hydrogen and hydrogen-like ions*/
#include "cddefines.h"
#include "atmdat.h"
/* NHYDRO_MAX_LEVEL defined 401 in cddefines.h */
float HRF[NHYDRO_MAX_LEVEL][9];
void HRFitData(void);
void HRFitData1(void);
void HRFitData2(void);
void HRFitData3(void);
void HRFitData4(void);
void HRFitData5(void);
void HRFitData6(void);
void HRFitData7(void);
void HRFitData8(void);
/*lint -e736 loss of precision 36 to 32 bits */

double atmdat_H_rad_rec(long int iz, 
	  long int n, 
	  double t)
{
	/*
	 * Version 4, October 9, 1997
	 ******************************************************************************
	 *** This subroutine calculates state-specific recombination rates 
	 *** for hydrogen and hydrogen-like ions.
	 *** Input parameters:  iz - atomic number 
	 ***          n  - shell number, from 0 to 400:
	 ***                                    0 - 1s
	 ***                                    1 - 2s
	 ***                                    2 - 2p
	 ***                                    3 - 3 
	 ***                                    ......
	 ***          t  - temperature, K
	 *** Output parameter:  r  - rate coefficient, cm^3 s^(-1)
	 *** If n is negative, the subroutine returns the total recombination 
	 *** rate coefficient
	 ******************************************************************************
	 */
	double rate,
	  TeScaled, 
	  x, 
	  x1, 
	  x2;
	static int lgFirst=TRUE;

#	ifdef DEBUG_FUN
	fputs( "<+>atmdat_H_rad_rec()\n", debug_fp );
#	endif

	/* do one time initialization of data if this is first call */
	if( lgFirst )
	{
		HRFitData();
		HRFitData1();
		HRFitData2();
		HRFitData3();
		HRFitData4();
		HRFitData5();
		HRFitData6();
		HRFitData7();
		HRFitData8();
		lgFirst = FALSE;
	}

	rate = 0.0;

	/* iz is charge, must be 1 or greater */
	ASSERT( iz > 0 );

	/* n is level number, must be less than dim or hydro vectors */
	ASSERT( n < NHYDRO_MAX_LEVEL );

	TeScaled = t/POW2((double)iz);

	if( n < 0 )
	{
		x1 = sqrt(TeScaled/3.148);
		x2 = sqrt(TeScaled/7.036e05);
		rate = (float)(7.982e-11/x1/pow(1.0 + x1,0.252)/pow(1.0 + x2,1.748));
	}
	else
	{
		x = log10(TeScaled);
		rate = (float)((HRF[n][0] + HRF[n][2]*x + HRF[n][4]*
		  x*x + HRF[n][6]*powi(x,3) + HRF[n][8]*powi(x,4))/
		  (1.0 + HRF[n][1]*x + HRF[n][3]*x*x + HRF[n][5]*
		  powi(x,3) + HRF[n][7]*powi(x,4)));
		rate = (float)(pow(10.0,rate)/TeScaled);
	}
	rate *= iz;
	
#	ifdef DEBUG_FUN
	fputs( " <->atmdat_H_rad_rec()\n", debug_fp );
#	endif

	return rate;
}

#ifdef _MSC_VER
#	pragma warning( disable : 4305 )/* disable truncation from double to float*/
#endif

/********************************************* */
void HRFitData(void)
{

#	ifdef DEBUG_FUN
	fputs( "<+>HRFitData()\n", debug_fp );
#	endif

	int j;
		{ static float _itmp0[] = {-1.0781359e01,-3.8790005e-01,4.6748385e00,
		  6.3898347e-02,-8.7297469e-01,-5.0923340e-03,8.1445619e-02,
		  2.4935490e-04,-3.9043790e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[0][j] = _itmp0[j];
			}
		}
		{ static float _itmp1[] = {-1.1614807e01,-4.1476052e-01,5.3074766e00,
		  7.3198786e-02,-1.0442882e00,-6.2653576e-03,1.0220322e-01,
		  3.1993430e-04,-5.1042098e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[1][j] = _itmp1[j];
			}
		}
		{ static float _itmp2[] = {-1.1179081e01,-3.9229646e-01,4.8858925e00,
		  6.4659417e-02,-9.1797817e-01,-4.8978538e-03,8.5295549e-02,
		  2.6802328e-04,-4.6109274e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[2][j] = _itmp2[j];
			}
		}
		{ static float _itmp3[] = {-1.1206241e01,-4.4055033e-01,5.4401345e00,
		  8.7431014e-02,-1.2008760e00,-7.4473079e-03,1.2504593e-01,
		  4.6410321e-04,-7.6113860e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[3][j] = _itmp3[j];
			}
		}
		{ static float _itmp4[] = {-1.1323596e01,-4.5838878e-01,5.6956887e00,
		  9.4245471e-02,-1.2998447e00,-7.9713343e-03,1.3644578e-01,
		  5.1810528e-04,-8.6822566e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[4][j] = _itmp4[j];
			}
		}
		{ static float _itmp5[] = {-1.1415392e01,-4.6823558e-01,5.8506689e00,
		  9.8131262e-02,-1.3590697e00,-8.1813205e-03,1.4233549e-01,
		  5.4651999e-04,-9.3300305e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[5][j] = _itmp5[j];
			}
		}
		{ static float _itmp6[] = {-1.1490612e01,-4.7406995e-01,5.9522166e00,
		  1.0064020e-01,-1.3980964e00,-8.2884254e-03,1.4570926e-01,
		  5.6291226e-04,-9.7414833e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[6][j] = _itmp6[j];
			}
		}
		{ static float _itmp7[] = {-1.1554831e01,-4.7439665e-01,5.9864464e00,
		  1.0160362e-01,-1.4165708e00,-8.1698345e-03,1.4599949e-01,
		  5.6771806e-04,-9.9944277e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[7][j] = _itmp7[j];
			}
		}
		{ static float _itmp8[] = {-1.1610895e01,-4.7473451e-01,6.0173483e00,
		  1.0262581e-01,-1.4353125e00,-8.0580283e-03,1.4635733e-01,
		  5.7397748e-04,-1.0264914e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[8][j] = _itmp8[j];
			}
		}
		{ static float _itmp9[] = {-1.1659862e01,-4.7320300e-01,6.0212073e00,
		  1.0306408e-01,-1.4434516e00,-7.9748463e-03,1.4574037e-01,
		  5.7312468e-04,-1.0345792e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[9][j] = _itmp9[j];
			}
		}
		{ static float _itmp10[] = {-1.1703726e01,-4.7109461e-01,6.0156460e00,
		  1.0338196e-01,-1.4491727e00,-7.9034464e-03,1.4497490e-01,
		  5.7136163e-04,-1.0390419e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[10][j] = _itmp10[j];
			}
		}
		{ static float _itmp11[] = {-1.1743682e01,-4.6861926e-01,6.0042748e00,
		  1.0367520e-01,-1.4545577e00,-7.8139901e-03,1.4410105e-01,
		  5.7011074e-04,-1.0450164e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[11][j] = _itmp11[j];
			}
		}
		{ static float _itmp12[] = {-1.1780066e01,-4.6552062e-01,5.9832773e00,
		  1.0383467e-01,-1.4571851e00,-7.7397432e-03,1.4303389e-01,
		  5.6721305e-04,-1.0457950e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[12][j] = _itmp12[j];
			}
		}
		{ static float _itmp13[] = {-1.1813470e01,-4.6189710e-01,5.9541488e00,
		  1.0391980e-01,-1.4579324e00,-7.6871593e-03,1.4190944e-01,
		  5.6342082e-04,-1.0426981e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[13][j] = _itmp13[j];
			}
		}
		{ static float _itmp14[] = {-1.1844401e01,-4.5789742e-01,5.9190369e00,
		  1.0395065e-01,-1.4573270e00,-7.6565151e-03,1.4080316e-01,
		  5.5903435e-04,-1.0365284e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[14][j] = _itmp14[j];
			}
		}
		{ static float _itmp15[] = {-1.1873433e01,-4.5425522e-01,5.8877001e00,
		  1.0408559e-01,-1.4586751e00,-7.6114950e-03,1.3982418e-01,
		  5.5618485e-04,-1.0345820e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[15][j] = _itmp15[j];
			}
		}
		{ static float _itmp16[] = {-1.1900671e01,-4.5046213e-01,5.8536797e00,
		  1.0419094e-01,-1.4595356e00,-7.5670909e-03,1.3881774e-01,
		  5.5313681e-04,-1.0319206e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[16][j] = _itmp16[j];
			}
		}
		{ static float _itmp17[] = {-1.1926344e01,-4.4671643e-01,5.8194909e00,
		  1.0432647e-01,-1.4608291e00,-7.5205681e-03,1.3783988e-01,
		  5.5044162e-04,-1.0300523e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[17][j] = _itmp17[j];
			}
		}
		{ static float _itmp18[] = {-1.1950244e01,-4.4250283e-01,5.7777996e00,
		  1.0446134e-01,-1.4605879e00,-7.5422316e-03,1.3712138e-01,
		  5.4702599e-04,-1.0221305e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[18][j] = _itmp18[j];
			}
		}
		{ static float _itmp19[] = {-1.1972892e01,-4.3764737e-01,5.7276578e00,
		  1.0446159e-01,-1.4581732e00,-7.5680362e-03,1.3624944e-01,
		  5.4222503e-04,-1.0105107e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[19][j] = _itmp19[j];
			}
		}
		{ static float _itmp20[] = {-1.1994494e01,-4.3314013e-01,5.6813703e00,
		  1.0454341e-01,-1.4572425e00,-7.5892373e-03,1.3551211e-01,
		  5.3849613e-04,-1.0016165e-02};
		for( j=0; j < 9; j++ )
		{
			HRF[20][j] = _itmp20[j];
			}
		}
		{ static float _itmp21[] = {-1.2015278e01,-4.2899099e-01,5.6394715e00,
		  1.0466056e-01,-1.4575908e00,-7.5819292e-03,1.3474566e-01,
		  5.3560530e-04,-9.9637723e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[21][j] = _itmp21[j];
			}
		}
		{ static float _itmp22[] = {-1.2035006e01,-4.2481855e-01,5.5964055e00,
		  1.0483605e-01,-1.4582521e00,-7.5989510e-03,1.3414820e-01,
		  5.3299713e-04,-9.9023366e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[22][j] = _itmp22[j];
			}
		}
		{ static float _itmp23[] = {-1.2053696e01,-4.2008513e-01,5.5454283e00,
		  1.0496928e-01,-1.4573236e00,-7.6615736e-03,1.3369676e-01,
		  5.2947429e-04,-9.7904904e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[23][j] = _itmp23[j];
			}
		}
		{ static float _itmp24[] = {-1.2072058e01,-4.1634554e-01,5.5075121e00,
		  1.0512653e-01,-1.4588331e00,-7.6498450e-03,1.3308986e-01,
		  5.2762119e-04,-9.7634634e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[24][j] = _itmp24[j];
			}
		}
		{ static float _itmp25[] = {-1.2089826e01,-4.1272256e-01,5.4709644e00,
		  1.0518475e-01,-1.4594760e00,-7.6469667e-03,1.3259415e-01,
		  5.2569772e-04,-9.7290250e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[25][j] = _itmp25[j];
			}
		}
		{ static float _itmp26[] = {-1.2106376e01,-4.0798819e-01,5.4187608e00,
		  1.0537831e-01,-1.4593349e00,-7.7079921e-03,1.3219580e-01,
		  5.2257784e-04,-9.6259927e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[26][j] = _itmp26[j];
			}
		}
		{ static float _itmp27[] = {-1.2122654e01,-4.0402237e-01,5.3766398e00,
		  1.0552075e-01,-1.4601488e00,-7.7304444e-03,1.3179220e-01,
		  5.2052730e-04,-9.5718084e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[27][j] = _itmp27[j];
			}
		}
		{ static float _itmp28[] = {-1.2138199e01,-3.9971721e-01,5.3294930e00,
		  1.0570740e-01,-1.4607050e00,-7.7860253e-03,1.3153680e-01,
		  5.1824795e-04,-9.4913989e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[28][j] = _itmp28[j];
			}
		}
		{ static float _itmp29[] = {-1.2153054e01,-3.9513275e-01,5.2780848e00,
		  1.0595281e-01,-1.4612490e00,-7.8747701e-03,1.3145217e-01,
		  5.1583396e-04,-9.3874624e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[29][j] = _itmp29[j];
			}
		}
		{ static float _itmp30[] = {-1.2167930e01,-3.9138097e-01,5.2382159e00,
		  1.0600332e-01,-1.4616532e00,-7.8839352e-03,1.3106517e-01,
		  5.1394018e-04,-9.3439203e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[30][j] = _itmp30[j];
			}
		}
		{ static float _itmp31[] = {-1.2181968e01,-3.8674060e-01,5.1858959e00,
		  1.0616534e-01,-1.4614609e00,-7.9713566e-03,1.3100468e-01,
		  5.1133166e-04,-9.2354259e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[31][j] = _itmp31[j];
			}
		}
		{ static float _itmp32[] = {-1.2195662e01,-3.8268334e-01,5.1408253e00,
		  1.0637771e-01,-1.4627500e00,-8.0266586e-03,1.3088179e-01,
		  5.0966616e-04,-9.1688121e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[32][j] = _itmp32[j];
			}
		}
		{ static float _itmp33[] = {-1.2209098e01,-3.7877434e-01,5.0977883e00,
		  1.0649961e-01,-1.4635035e00,-8.0614518e-03,1.3066287e-01,
		  5.0788955e-04,-9.1117192e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[33][j] = _itmp33[j];
			}
		}
		{ static float _itmp34[] = {-1.2222247e01,-3.7526307e-01,5.0597577e00,
		  1.0661513e-01,-1.4648734e00,-8.0657816e-03,1.3032517e-01,
		  5.0655485e-04,-9.0833800e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[34][j] = _itmp34[j];
			}
		}
		{ static float _itmp35[] = {-1.2234772e01,-3.7112862e-01,5.0129294e00,
		  1.0683463e-01,-1.4661298e00,-8.1284540e-03,1.3025747e-01,
		  5.0481694e-04,-9.0100151e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[35][j] = _itmp35[j];
			}
		}
		{ static float _itmp36[] = {-1.2247398e01,-3.6775589e-01,4.9767208e00,
		  1.0678117e-01,-1.4660431e00,-8.1257429e-03,1.2993622e-01,
		  5.0330535e-04,-8.9807259e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[36][j] = _itmp36[j];
			}
		}
		{ static float _itmp37[] = {-1.2259201e01,-3.6335087e-01,4.9258962e00,
		  1.0698568e-01,-1.4666287e00,-8.2299281e-03,1.3015290e-01,
		  5.0146756e-04,-8.8793254e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[37][j] = _itmp37[j];
			}
		}
		{ static float _itmp38[] = {-1.2270961e01,-3.5998505e-01,4.8886595e00,
		  1.0712655e-01,-1.4682977e00,-8.2445526e-03,1.2994249e-01,
		  5.0045061e-04,-8.8515356e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[38][j] = _itmp38[j];
			}
		}
		{ static float _itmp39[] = {-1.2282477e01,-3.5600036e-01,4.8435674e00,
		  1.0718326e-01,-1.4683224e00,-8.2968166e-03,1.2990940e-01,
		  4.9873453e-04,-8.7835230e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[39][j] = _itmp39[j];
			}
		}
		{ static float _itmp40[] = {-1.2293659e01,-3.5217479e-01,4.8001490e00,
		  1.0726941e-01,-1.4687748e00,-8.3475336e-03,1.2989838e-01,
		  4.9725367e-04,-8.7220585e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[40][j] = _itmp40[j];
			}
		}
		{ static float _itmp41[] = {-1.2304632e01,-3.4856406e-01,4.7594318e00,
		  1.0729563e-01,-1.4689101e00,-8.3811982e-03,1.2980621e-01,
		  4.9574644e-04,-8.6701745e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[41][j] = _itmp41[j];
			}
		}
		{ static float _itmp42[] = {-1.2315597e01,-3.4561223e-01,4.7275271e00,
		  1.0721066e-01,-1.4690294e00,-8.3508380e-03,1.2938663e-01,
		  4.9466337e-04,-8.6661167e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[42][j] = _itmp42[j];
			}
		}
		{ static float _itmp43[] = {-1.2326196e01,-3.4221151e-01,4.6894636e00,
		  1.0712558e-01,-1.4683368e00,-8.3666313e-03,1.2922251e-01,
		  4.9314817e-04,-8.6238608e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[43][j] = _itmp43[j];
			}
		}
		{ static float _itmp44[] = {-1.2336588e01,-3.3893821e-01,4.6529107e00,
		  1.0703895e-01,-1.4678200e00,-8.3703157e-03,1.2899411e-01,
		  4.9176638e-04,-8.5917758e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[44][j] = _itmp44[j];
			}
		}
		{ static float _itmp45[] = {-1.2346814e01,-3.3574671e-01,4.6173844e00,
		  1.0690797e-01,-1.4669774e00,-8.3640749e-03,1.2871175e-01,
		  4.9036625e-04,-8.5647600e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[45][j] = _itmp45[j];
			}
		}
		{ static float _itmp46[] = {-1.2356810e01,-3.3261508e-01,4.5824165e00,
		  1.0676638e-01,-1.4660238e00,-8.3574271e-03,1.2843376e-01,
		  4.8896379e-04,-8.5378066e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[46][j] = _itmp46[j];
			}
		}
}
void HRFitData1(void)
{
	int j;
		{ static float _itmp47[] = {-1.2366634e01,-3.2947436e-01,4.5472779e00,
		  1.0659764e-01,-1.4647938e00,-8.3490694e-03,1.2814914e-01,
		  4.8754769e-04,-8.5110562e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[47][j] = _itmp47[j];
			}
		}
		{ static float _itmp48[] = {-1.2376199e01,-3.2619360e-01,4.5100183e00,
		  1.0647124e-01,-1.4637210e00,-8.3589107e-03,1.2797438e-01,
		  4.8611662e-04,-8.4730862e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[48][j] = _itmp48[j];
			}
		}
		{ static float _itmp49[] = {-1.2385715e01,-3.2318798e-01,4.4765038e00,
		  1.0622041e-01,-1.4617842e00,-8.3383881e-03,1.2762918e-01,
		  4.8463870e-04,-8.4513947e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[49][j] = _itmp49[j];
			}
		}
		{ static float _itmp50[] = {-1.2395072e01,-3.2008186e-01,4.4417076e00,
		  1.0588000e-01,-1.4587654e00,-8.3213020e-03,1.2729914e-01,
		  4.8285781e-04,-8.4197382e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[50][j] = _itmp50[j];
			}
		}
		{ static float _itmp51[] = {-1.2403904e01,-3.1650776e-01,4.3997197e00,
		  1.0593068e-01,-1.4589423e00,-8.3707860e-03,1.2736160e-01,
		  4.8159316e-04,-8.3624544e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[51][j] = _itmp51[j];
			}
		}
		{ static float _itmp52[] = {-1.2412719e01,-3.1319660e-01,4.3613977e00,
		  1.0581488e-01,-1.4577944e00,-8.3896844e-03,1.2725171e-01,
		  4.8015680e-04,-8.3184354e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[52][j] = _itmp52[j];
			}
		}
		{ static float _itmp53[] = {-1.2421478e01,-3.1002456e-01,4.3250332e00,
		  1.0558445e-01,-1.4556801e00,-8.3895000e-03,1.2703200e-01,
		  4.7859695e-04,-8.2820375e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[53][j] = _itmp53[j];
			}
		}
		{ static float _itmp54[] = {-1.2430075e01,-3.0683792e-01,4.2883348e00,
		  1.0535434e-01,-1.4535110e00,-8.3931144e-03,1.2684156e-01,
		  4.7704115e-04,-8.2433457e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[54][j] = _itmp54[j];
			}
		}
		{ static float _itmp55[] = {-1.2438474e01,-3.0362332e-01,4.2509966e00,
		  1.0517012e-01,-1.4517003e00,-8.4072044e-03,1.2672499e-01,
		  4.7557554e-04,-8.2007274e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[55][j] = _itmp55[j];
			}
		}
		{ static float _itmp56[] = {-1.2446823e01,-3.0063188e-01,4.2166781e00,
		  1.0487986e-01,-1.4491082e00,-8.3915405e-03,1.2641545e-01,
		  4.7399857e-04,-8.1724580e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[56][j] = _itmp56[j];
			}
		}
		{ static float _itmp57[] = {-1.2455081e01,-2.9768002e-01,4.1828914e00,
		  1.0452262e-01,-1.4458462e00,-8.3713587e-03,1.2608610e-01,
		  4.7233200e-04,-8.1441868e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[57][j] = _itmp57[j];
			}
		}
		{ static float _itmp58[] = {-1.2462757e01,-2.9385799e-01,4.1365023e00,
		  1.0469372e-01,-1.4467616e00,-8.4668174e-03,1.2648870e-01,
		  4.7124206e-04,-8.0638248e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[58][j] = _itmp58[j];
			}
		}
		{ static float _itmp59[] = {-1.2470639e01,-2.9066688e-01,4.0990868e00,
		  1.0446084e-01,-1.4443944e00,-8.4800860e-03,1.2638119e-01,
		  4.6971423e-04,-8.0194455e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[59][j] = _itmp59[j];
			}
		}
		{ static float _itmp60[] = {-1.2478577e01,-2.8793618e-01,4.0679922e00,
		  1.0400717e-01,-1.4403269e00,-8.4382528e-03,1.2592086e-01,
		  4.6797944e-04,-8.0015389e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[60][j] = _itmp60[j];
			}
		}
		{ static float _itmp61[] = {-1.2486416e01,-2.8526971e-01,4.0376873e00,
		  1.0351269e-01,-1.4358788e00,-8.3878059e-03,1.2539995e-01,
		  4.6618766e-04,-7.9869265e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[61][j] = _itmp61[j];
			}
		}
		{ static float _itmp62[] = {-1.2494286e01,-2.8282446e-01,4.0105329e00,
		  1.0283953e-01,-1.4298381e00,-8.3079496e-03,1.2469480e-01,
		  4.6418348e-04,-7.9834526e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[62][j] = _itmp62[j];
			}
		}
		{ static float _itmp63[] = {-1.2501819e01,-2.8000638e-01,3.9778674e00,
		  1.0240918e-01,-1.4257513e00,-8.2772030e-03,1.2429468e-01,
		  4.6241222e-04,-7.9577072e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[63][j] = _itmp63[j];
			}
		}
		{ static float _itmp64[] = {-1.2509210e01,-2.7706131e-01,3.9433830e00,
		  1.0201304e-01,-1.4218240e00,-8.2637649e-03,1.2401871e-01,
		  4.6069108e-04,-7.9229325e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[64][j] = _itmp64[j];
			}
		}
		{ static float _itmp65[] = {-1.2516450e01,-2.7410242e-01,3.9084723e00,
		  1.0167847e-01,-1.4184698e00,-8.2563367e-03,1.2377772e-01,
		  4.5905972e-04,-7.8870673e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[65][j] = _itmp65[j];
			}
		}
		{ static float _itmp66[] = {-1.2523698e01,-2.7135918e-01,3.8765676e00,
		  1.0119164e-01,-1.4137731e00,-8.2213292e-03,1.2336019e-01,
		  4.5725374e-04,-7.8622969e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[66][j] = _itmp66[j];
			}
		}
		{ static float _itmp67[] = {-1.2530793e01,-2.6849079e-01,3.8428245e00,
		  1.0075468e-01,-1.4094019e00,-8.2013085e-03,1.2303955e-01,
		  4.5547090e-04,-7.8293653e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[67][j] = _itmp67[j];
			}
		}
		{ static float _itmp68[] = {-1.2537749e01,-2.6558509e-01,3.8083625e00,
		  1.0038110e-01,-1.4055945e00,-8.1924535e-03,1.2279880e-01,
		  4.5381184e-04,-7.7931294e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[68][j] = _itmp68[j];
			}
		}
		{ static float _itmp69[] = {-1.2544674e01,-2.6285797e-01,3.7763045e00,
		  9.9908859e-02,-1.4009424e00,-8.1595201e-03,1.2238880e-01,
		  4.5200714e-04,-7.7669360e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[69][j] = _itmp69[j];
			}
		}
		{ static float _itmp70[] = {-1.2551662e01,-2.6036021e-01,3.7476041e00,
		  9.9194907e-02,-1.3940214e00,-8.0890357e-03,1.2172686e-01,
		  4.4984609e-04,-7.7526402e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[70][j] = _itmp70[j];
			}
		}
		{ static float _itmp71[] = {-1.2558376e01,-2.5759929e-01,3.7148459e00,
		  9.8751232e-02,-1.3895355e00,-8.0656623e-03,1.2138607e-01,
		  4.4809876e-04,-7.7221384e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[71][j] = _itmp71[j];
			}
		}
		{ static float _itmp72[] = {-1.2564903e01,-2.5463727e-01,3.6790924e00,
		  9.8459512e-02,-1.3863566e00,-8.0763604e-03,1.2128927e-01,
		  4.4660235e-04,-7.6784319e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[72][j] = _itmp72[j];
			}
		}
		{ static float _itmp73[] = {-1.2571559e01,-2.5205341e-01,3.6488028e00,
		  9.7838454e-02,-1.3801751e00,-8.0242064e-03,1.2075010e-01,
		  4.4458217e-04,-7.6569631e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[73][j] = _itmp73[j];
			}
		}
		{ static float _itmp74[] = {-1.2578140e01,-2.4954867e-01,3.6194789e00,
		  9.7185925e-02,-1.3736923e00,-7.9644397e-03,1.2015500e-01,
		  4.4251181e-04,-7.6385066e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[74][j] = _itmp74[j];
			}
		}
		{ static float _itmp75[] = {-1.2584601e01,-2.4701615e-01,3.5895791e00,
		  9.6593536e-02,-1.3677512e00,-7.9145553e-03,1.1962981e-01,
		  4.4056110e-04,-7.6173665e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[75][j] = _itmp75[j];
			}
		}
		{ static float _itmp76[] = {-1.2590901e01,-2.4413516e-01,3.5548735e00,
		  9.6143499e-02,-1.3629293e00,-7.9058716e-03,1.1939797e-01,
		  4.3878731e-04,-7.5769532e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[76][j] = _itmp76[j];
			}
		}
		{ static float _itmp77[] = {-1.2597112e01,-2.4149412e-01,3.5231154e00,
		  9.5682263e-02,-1.3581367e00,-7.8829024e-03,1.1906875e-01,
		  4.3706052e-04,-7.5461273e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[77][j] = _itmp77[j];
			}
		}
		{ static float _itmp78[] = {-1.2603210e01,-2.3872180e-01,3.4894505e00,
		  9.5316388e-02,-1.3542082e00,-7.8779347e-03,1.1886695e-01,
		  4.3549182e-04,-7.5090025e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[78][j] = _itmp78[j];
			}
		}
		{ static float _itmp79[] = {-1.2609599e01,-2.3652078e-01,3.4642820e00,
		  9.4310246e-02,-1.3441550e00,-7.7662258e-03,1.1789787e-01,
		  4.3282984e-04,-7.5038783e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[79][j] = _itmp79[j];
			}
		}
		{ static float _itmp80[] = {-1.2615613e01,-2.3392205e-01,3.4328928e00,
		  9.3836337e-02,-1.3392009e00,-7.7387579e-03,1.1753230e-01,
		  4.3109822e-04,-7.4751796e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[80][j] = _itmp80[j];
			}
		}
		{ static float _itmp81[] = {-1.2621609e01,-2.3131607e-01,3.4015093e00,
		  9.3275614e-02,-1.3333091e00,-7.7050659e-03,1.1712413e-01,
		  4.2919294e-04,-7.4452772e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[81][j] = _itmp81[j];
			}
		}
		{ static float _itmp82[] = {-1.2627502e01,-2.2863489e-01,3.3689771e00,
		  9.2778794e-02,-1.3279967e00,-7.6827407e-03,1.1679579e-01,
		  4.2739016e-04,-7.4114702e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[82][j] = _itmp82[j];
			}
		}
		{ static float _itmp83[] = {-1.2633357e01,-2.2607903e-01,3.3380706e00,
		  9.2222534e-02,-1.3221276e00,-7.6485844e-03,1.1638605e-01,
		  4.2551736e-04,-7.3825135e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[83][j] = _itmp83[j];
			}
		}
		{ static float _itmp84[] = {-1.2639194e01,-2.2367841e-01,3.3092525e00,
		  9.1565289e-02,-1.3152970e00,-7.5939675e-03,1.1582534e-01,
		  4.2347968e-04,-7.3608533e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[84][j] = _itmp84[j];
			}
		}
		{ static float _itmp85[] = {-1.2644907e01,-2.2122909e-01,3.2795298e00,
		  9.1017880e-02,-1.3095304e00,-7.5569521e-03,1.1539772e-01,
		  4.2166820e-04,-7.3350207e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[85][j] = _itmp85[j];
			}
		}
		{ static float _itmp86[] = {-1.2650564e01,-2.1876413e-01,3.2495544e00,
		  9.0463698e-02,-1.3036540e00,-7.5195548e-03,1.1496159e-01,
		  4.1980861e-04,-7.3081395e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[86][j] = _itmp86[j];
			}
		}
		{ static float _itmp87[] = {-1.2656248e01,-2.1637219e-01,3.2207623e00,
		  8.9744836e-02,-1.2960892e00,-7.4614701e-03,1.1437573e-01,
		  4.1766278e-04,-7.2851880e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[87][j] = _itmp87[j];
			}
		}
		{ static float _itmp88[] = {-1.2661839e01,-2.1403044e-01,3.1924257e00,
		  8.9083403e-02,-1.2891260e00,-7.4065388e-03,1.1381108e-01,
		  4.1564534e-04,-7.2638104e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[88][j] = _itmp88[j];
			}
		}
		{ static float _itmp89[] = {-1.2667442e01,-2.1174929e-01,3.1650310e00,
		  8.8308848e-02,-1.2810075e00,-7.3349699e-03,1.1312044e-01,
		  4.1341304e-04,-7.2461348e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[89][j] = _itmp89[j];
			}
		}
		{ static float _itmp90[] = {-1.2672813e01,-2.0932512e-01,3.1351385e00,
		  8.7846808e-02,-1.2760009e00,-7.3083560e-03,1.1276199e-01,
		  4.1173844e-04,-7.2178305e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[90][j] = _itmp90[j];
			}
		}
		{ static float _itmp91[] = {-1.2678154e01,-2.0689476e-01,3.1051767e00,
		  8.7358326e-02,-1.2706968e00,-7.2809500e-03,1.1240203e-01,
		  4.1003560e-04,-7.1890308e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[91][j] = _itmp91[j];
			}
		}
		{ static float _itmp92[] = {-1.2683533e01,-2.0440832e-01,3.0747695e00,
		  8.6695574e-02,-1.2634950e00,-7.2420263e-03,1.1196411e-01,
		  4.0802188e-04,-7.1579888e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[92][j] = _itmp92[j];
			}
		}
		{ static float _itmp93[] = {-1.2688896e01,-2.0231242e-01,3.0494461e00,
		  8.5927054e-02,-1.2554336e00,-7.1600173e-03,1.1118125e-01,
		  4.0578062e-04,-7.1461019e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[93][j] = _itmp93[j];
			}
		}
}
void HRFitData2(void)
{
	int j;
		{ static float _itmp94[] = {-1.2694197e01,-2.0016593e-01,3.0233724e00,
		  8.5179284e-02,-1.2474972e00,-7.0928703e-03,1.1053173e-01,
		  4.0368337e-04,-7.1289749e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[94][j] = _itmp94[j];
			}
		}
		{ static float _itmp95[] = {-1.2699451e01,-1.9763219e-01,2.9922786e00,
		  8.4486149e-02,-1.2398840e00,-7.0514833e-03,1.1006483e-01,
		  4.0157454e-04,-7.0965532e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[95][j] = _itmp95[j];
			}
		}
		{ static float _itmp96[] = {-1.2704887e01,-1.9542164e-01,2.9661450e00,
		  8.3227798e-02,-1.2263921e00,-6.9460478e-03,1.0914782e-01,
		  3.9855702e-04,-7.0754103e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[96][j] = _itmp96[j];
			}
		}
		{ static float _itmp97[] = {-1.2710048e01,-1.9341168e-01,2.9416437e00,
		  8.2531497e-02,-1.2190948e00,-6.8588345e-03,1.0830297e-01,
		  3.9645022e-04,-7.0699537e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[97][j] = _itmp97[j];
			}
		}
		{ static float _itmp98[] = {-1.2715109e01,-1.9124776e-01,2.9149778e00,
		  8.1868276e-02,-1.2119284e00,-6.7981286e-03,1.0767777e-01,
		  3.9442052e-04,-7.0510376e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[98][j] = _itmp98[j];
			}
		}
		{ static float _itmp99[] = {-1.2720232e01,-1.8927985e-01,2.8911681e00,
		  8.0945246e-02,-1.2020578e00,-6.7088357e-03,1.0686056e-01,
		  3.9204193e-04,-7.0383926e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[99][j] = _itmp99[j];
			}
		}
		{ static float _itmp100[] = {-1.2725187e01,-1.8697031e-01,
		  2.8624456e00,8.0382571e-02,-1.1958658e00,-6.6616256e-03,
		  1.0632223e-01,3.9013205e-04,-7.0150336e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[100][j] = _itmp100[j];
			}
		}
		{ static float _itmp101[] = {-1.2730112e01,-1.8481848e-01,
		  2.8357635e00,7.9714708e-02,-1.1885269e00,-6.6159996e-03,
		  1.0584479e-01,3.8825095e-04,-6.9906744e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[101][j] = _itmp101[j];
			}
		}
		{ static float _itmp102[] = {-1.2735051e01,-1.8243988e-01,
		  2.8063266e00,7.8977324e-02,-1.1803323e00,-6.5677818e-03,
		  1.0532672e-01,3.8610981e-04,-6.9607552e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[102][j] = _itmp102[j];
			}
		}
		{ static float _itmp103[] = {-1.2739732e01,-1.8050966e-01,
		  2.7818241e00,7.8622170e-02,-1.1764390e00,-6.5317680e-03,
		  1.0488666e-01,3.8469775e-04,-6.9447402e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[103][j] = _itmp103[j];
			}
		}
		{ static float _itmp104[] = {-1.2744627e01,-1.7821079e-01,
		  2.7534480e00,7.7814102e-02,-1.1675094e00,-6.4694718e-03,
		  1.0425440e-01,3.8241854e-04,-6.9190636e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[104][j] = _itmp104[j];
			}
		}
		{ static float _itmp105[] = {-1.2749469e01,-1.7617977e-01,
		  2.7284224e00,7.6997980e-02,-1.1586157e00,-6.3922326e-03,
		  1.0350873e-01,3.8018939e-04,-6.9033392e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[105][j] = _itmp105[j];
			}
		}
		{ static float _itmp106[] = {-1.2754121e01,-1.7402913e-01,
		  2.7012751e00,7.6488346e-02,-1.1528647e00,-6.3630659e-03,
		  1.0314499e-01,3.7855521e-04,-6.8763983e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[106][j] = _itmp106[j];
			}
		}
		{ static float _itmp107[] = {-1.2758902e01,-1.7190276e-01,
		  2.6750190e00,7.5620264e-02,-1.1432902e00,-6.2892954e-03,
		  1.0242753e-01,3.7621209e-04,-6.8553095e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[107][j] = _itmp107[j];
			}
		}
		{ static float _itmp108[] = {-1.2763551e01,-1.7006454e-01,
		  2.6520872e00,7.4890412e-02,-1.1352857e00,-6.2221475e-03,
		  1.0177421e-01,3.7423181e-04,-6.8404377e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[108][j] = _itmp108[j];
			}
		}
		{ static float _itmp109[] = {-1.2768171e01,-1.6788185e-01,
		  2.6246567e00,7.4286737e-02,-1.1285245e00,-6.1711944e-03,
		  1.0120909e-01,3.7229306e-04,-6.8178363e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[109][j] = _itmp109[j];
			}
		}
		{ static float _itmp110[] = {-1.2772773e01,-1.6604213e-01,
		  2.6017046e00,7.3542908e-02,-1.1203743e00,-6.0932208e-03,
		  1.0044731e-01,3.7019985e-04,-6.8062712e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[110][j] = _itmp110[j];
			}
		}
		{ static float _itmp111[] = {-1.2777718e01,-1.6405818e-01,
		  2.5782454e00,7.1984068e-02,-1.1033458e00,-5.9388899e-03,
		  9.9106148e-02,3.6661720e-04,-6.7962422e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[111][j] = _itmp111[j];
			}
		}
		{ static float _itmp112[] = {-1.2782169e01,-1.6187979e-01,
		  2.5505383e00,7.1493290e-02,-1.0976995e00,-5.9120730e-03,
		  9.8753296e-02,3.6503130e-04,-6.7688031e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[112][j] = _itmp112[j];
			}
		}
		{ static float _itmp113[] = {-1.2786533e01,-1.6020589e-01,
		  2.5291200e00,7.0995674e-02,-1.0921558e00,-5.8699884e-03,
		  9.8308109e-02,3.6362084e-04,-6.7547909e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[113][j] = _itmp113[j];
			}
		}
		{ static float _itmp114[] = {-1.2791161e01,-1.5836388e-01,
		  2.5066085e00,6.9867603e-02,-1.0797631e00,-5.7439683e-03,
		  9.7141266e-02,3.6066718e-04,-6.7469878e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[114][j] = _itmp114[j];
			}
		}
		{ static float _itmp115[] = {-1.2795451e01,-1.5683571e-01,
		  2.4869962e00,6.9342732e-02,-1.0739563e00,-5.6833671e-03,
		  9.6520007e-02,3.5906475e-04,-6.7389519e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[115][j] = _itmp115[j];
			}
		}
		{ static float _itmp116[] = {-1.2799765e01,-1.5458080e-01,
		  2.4581382e00,6.8868391e-02,-1.0683447e00,-5.6754905e-03,
		  9.6341960e-02,3.5761393e-04,-6.7037246e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[116][j] = _itmp116[j];
			}
		}
		{ static float _itmp117[] = {-1.2804143e01,-1.5230910e-01,
		  2.4293978e00,6.8155415e-02,-1.0601218e00,-5.6365305e-03,
		  9.5899299e-02,3.5559351e-04,-6.6707791e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[117][j] = _itmp117[j];
			}
		}
		{ static float _itmp118[] = {-1.2808454e01,-1.5097314e-01,
		  2.4125738e00,6.7370392e-02,-1.0515565e00,-5.5348603e-03,
		  9.4947547e-02,3.5351014e-04,-6.6728336e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[118][j] = _itmp118[j];
			}
		}
		{ static float _itmp119[] = {-1.2813090e01,-1.4845622e-01,
		  2.3816895e00,6.6136532e-02,-1.0376852e00,-5.4316190e-03,
		  9.3970671e-02,3.5044577e-04,-6.6471626e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[119][j] = _itmp119[j];
			}
		}
		{ static float _itmp120[] = {-1.2817281e01,-1.4774430e-01,
		  2.3728304e00,6.5058097e-02,-1.0259345e00,-5.2792821e-03,
		  9.2604160e-02,3.4769412e-04,-6.6605825e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[120][j] = _itmp120[j];
			}
		}
		{ static float _itmp121[] = {-1.2821519e01,-1.4589632e-01,
		  2.3492885e00,6.4433761e-02,-1.0188626e00,-5.2247546e-03,
		  9.2037052e-02,3.4597234e-04,-6.6446606e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[121][j] = _itmp121[j];
			}
		}
		{ static float _itmp122[] = {-1.2825975e01,-1.4377499e-01,
		  2.3231606e00,6.3175820e-02,-1.0046961e00,-5.1110359e-03,
		  9.0978615e-02,3.4286492e-04,-6.6240961e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[122][j] = _itmp122[j];
			}
		}
		{ static float _itmp123[] = {-1.2830273e01,-1.4179285e-01,
		  2.2983122e00,6.2200990e-02,-9.9363565e-01,-5.0304634e-03,
		  9.0205699e-02,3.4043915e-04,-6.6031534e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[123][j] = _itmp123[j];
			}
		}
		{ static float _itmp124[] = {-1.2834367e01,-1.3986248e-01,
		  2.2734790e00,6.1612405e-02,-9.8680890e-01,-4.9769175e-03,
		  8.9603320e-02,3.3851157e-04,-6.5819244e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[124][j] = _itmp124[j];
			}
		}
		{ static float _itmp125[] = {-1.2838687e01,-1.3878451e-01,
		  2.2605224e00,6.0318887e-02,-9.7261924e-01,-4.8234053e-03,
		  8.8276260e-02,3.3580596e-04,-6.5941932e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[125][j] = _itmp125[j];
			}
		}
		{ static float _itmp126[] = {-1.2842663e01,-1.3698161e-01,
		  2.2370460e00,5.9850775e-02,-9.6710837e-01,-4.7813384e-03,
		  8.7770551e-02,3.3414972e-04,-6.5738978e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[126][j] = _itmp126[j];
			}
		}
		{ static float _itmp127[] = {-1.2846719e01,-1.3490482e-01,
		  2.2103941e00,5.9150971e-02,-9.5888424e-01,-4.7393828e-03,
		  8.7293297e-02,3.3212930e-04,-6.5425951e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[127][j] = _itmp127[j];
			}
		}
		{ static float _itmp128[] = {-1.2851072e01,-1.3286085e-01,
		  2.1852629e00,5.7786573e-02,-9.4346493e-01,-4.6162908e-03,
		  8.6174607e-02,3.2900393e-04,-6.5256553e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[128][j] = _itmp128[j];
			}
		}
		{ static float _itmp129[] = {-1.2855010e01,-1.3115966e-01,
		  2.1631446e00,5.7240546e-02,-9.3710995e-01,-4.5642629e-03,
		  8.5600451e-02,3.2729845e-04,-6.5091895e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[129][j] = _itmp129[j];
			}
		}
		{ static float _itmp130[] = {-1.2859076e01,-1.2951696e-01,
		  2.1423173e00,5.6334235e-02,-9.2683893e-01,-4.4621774e-03,
		  8.4609129e-02,3.2487098e-04,-6.5011480e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[130][j] = _itmp130[j];
			}
		}
		{ static float _itmp131[] = {-1.2862962e01,-1.2806115e-01,
		  2.1233704e00,5.5625848e-02,-9.1865236e-01,-4.3999031e-03,
		  8.3987780e-02,3.2303145e-04,-6.4860713e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[131][j] = _itmp131[j];
			}
		}
		{ static float _itmp132[] = {-1.2866885e01,-1.2593497e-01,
		  2.0958104e00,5.5144422e-02,-9.1284877e-01,-4.3816515e-03,
		  8.3711669e-02,3.2156793e-04,-6.4560063e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[132][j] = _itmp132[j];
			}
		}
		{ static float _itmp133[] = {-1.2870798e01,-1.2430360e-01,
		  2.0748169e00,5.4279868e-02,-9.0282679e-01,-4.3148766e-03,
		  8.3061904e-02,3.1944070e-04,-6.4350981e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[133][j] = _itmp133[j];
			}
		}
		{ static float _itmp134[] = {-1.2874683e01,-1.2223892e-01,
		  2.0480177e00,5.3818051e-02,-8.9729416e-01,-4.2875418e-03,
		  8.2696706e-02,3.1798150e-04,-6.4104544e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[134][j] = _itmp134[j];
			}
		}
		{ static float _itmp135[] = {-1.2878332e01,-1.2095545e-01,
		  2.0306039e00,5.3601392e-02,-8.9466864e-01,-4.2557106e-03,
		  8.2289040e-02,3.1698853e-04,-6.4024334e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[135][j] = _itmp135[j];
			}
		}
		{ static float _itmp136[] = {-1.2882322e01,-1.1863913e-01,
		  2.0011251e00,5.2658271e-02,-8.8358355e-01,-4.2008613e-03,
		  8.1718683e-02,3.1460490e-04,-6.3670999e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[136][j] = _itmp136[j];
			}
		}
		{ static float _itmp137[] = {-1.2886093e01,-1.1708276e-01,
		  1.9807477e00,5.2040864e-02,-8.7633514e-01,-4.1582384e-03,
		  8.1275731e-02,3.1307340e-04,-6.3485675e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[137][j] = _itmp137[j];
			}
		}
		{ static float _itmp138[] = {-1.2889717e01,-1.1604929e-01,
		  1.9667090e00,5.1607028e-02,-8.7133592e-01,-4.0865932e-03,
		  8.0520988e-02,3.1154527e-04,-6.3489736e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[138][j] = _itmp138[j];
			}
		}
		{ static float _itmp139[] = {-1.2893471e01,-1.1439819e-01,
		  1.9451270e00,5.1026333e-02,-8.6450821e-01,-4.0398128e-03,
		  8.0017231e-02,3.0997751e-04,-6.3321008e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[139][j] = _itmp139[j];
			}
		}
		{ static float _itmp140[] = {-1.2897036e01,-1.1299469e-01,
		  1.9261694e00,5.0722450e-02,-8.6076105e-01,-4.0136739e-03,
		  7.9664923e-02,3.0880058e-04,-6.3155042e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[140][j] = _itmp140[j];
			}
		}
		{ static float _itmp141[] = {-1.2900756e01,-1.1104361e-01,
		  1.9007665e00,5.0011914e-02,-8.5222423e-01,-3.9856685e-03,
		  7.9341255e-02,3.0698982e-04,-6.2807263e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[141][j] = _itmp141[j];
			}
		}
}
void HRFitData3(void)
{
	int j;
		{ static float _itmp142[] = {-1.2904379e01,-1.0948732e-01,
		  1.8800793e00,4.9630269e-02,-8.4762901e-01,-3.9567896e-03,
		  7.8983717e-02,3.0581685e-04,-6.2647904e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[142][j] = _itmp142[j];
			}
		}
		{ static float _itmp143[] = {-1.2907994e01,-1.0778527e-01,
		  1.8575341e00,4.9234398e-02,-8.4280735e-01,-3.9316192e-03,
		  7.8647867e-02,3.0453241e-04,-6.2438506e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[143][j] = _itmp143[j];
			}
		}
		{ static float _itmp144[] = {-1.2911791e01,-1.0570679e-01,
		  1.8308539e00,4.8353121e-02,-8.3242059e-01,-3.8773220e-03,
		  7.8097925e-02,3.0244072e-04,-6.2155398e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[144][j] = _itmp144[j];
			}
		}
		{ static float _itmp145[] = {-1.2915656e01,-1.0367294e-01,
		  1.8050619e00,4.7218010e-02,-8.1921118e-01,-3.7794977e-03,
		  7.7148400e-02,2.9961241e-04,-6.1926078e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[145][j] = _itmp145[j];
			}
		}
		{ static float _itmp146[] = {-1.2918971e01,-1.0285259e-01,
		  1.7930751e00,4.7145154e-02,-8.1816202e-01,-3.7605555e-03,
		  7.6867200e-02,2.9898452e-04,-6.1874539e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[146][j] = _itmp146[j];
			}
		}
		{ static float _itmp147[] = {-1.2922370e01,-1.0179729e-01,
		  1.7783837e00,4.6907444e-02,-8.1522393e-01,-3.7379954e-03,
		  7.6580144e-02,2.9820623e-04,-6.1788936e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[147][j] = _itmp147[j];
			}
		}
		{ static float _itmp148[] = {-1.2925926e01,-1.0003916e-01,
		  1.7551544e00,4.6439476e-02,-8.0953789e-01,-3.7079484e-03,
		  7.6202415e-02,2.9676565e-04,-6.1562588e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[148][j] = _itmp148[j];
			}
		}
		{ static float _itmp149[] = {-1.2929485e01,-9.8309949e-02,
		  1.7323883e00,4.5812018e-02,-8.0196106e-01,-3.6817442e-03,
		  7.5904183e-02,2.9519698e-04,-6.1271158e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[149][j] = _itmp149[j];
			}
		}
		{ static float _itmp150[] = {-1.2932901e01,-9.6801385e-02,
		  1.7119921e00,4.5564286e-02,-7.9880697e-01,-3.6652172e-03,
		  7.5636335e-02,2.9420515e-04,-6.1094943e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[150][j] = _itmp150[j];
			}
		}
		{ static float _itmp151[] = {-1.2936206e01,-9.5662840e-02,
		  1.6960744e00,4.5429509e-02,-7.9698467e-01,-3.6518404e-03,
		  7.5403735e-02,2.9348064e-04,-6.0978094e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[151][j] = _itmp151[j];
			}
		}
		{ static float _itmp152[] = {-1.2939630e01,-9.4225168e-02,
		  1.6767253e00,4.5031462e-02,-7.9211622e-01,-3.6262714e-03,
		  7.5085647e-02,2.9231334e-04,-6.0799159e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[152][j] = _itmp152[j];
			}
		}
		{ static float _itmp153[] = {-1.2942921e01,-9.3507692e-02,
		  1.6663682e00,4.4672649e-02,-7.8782737e-01,-3.5861547e-03,
		  7.4668594e-02,2.9137413e-04,-6.0767657e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[153][j] = _itmp153[j];
			}
		}
		{ static float _itmp154[] = {-1.2946257e01,-9.2675693e-02,
		  1.6547455e00,4.4167966e-02,-7.8187865e-01,-3.5148722e-03,
		  7.3932298e-02,2.8980157e-04,-6.0750726e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[154][j] = _itmp154[j];
			}
		}
		{ static float _itmp155[] = {-1.2950015e01,-9.0391479e-02,
		  1.6256981e00,4.3074708e-02,-7.6904678e-01,-3.4279511e-03,
		  7.3074177e-02,2.8716898e-04,-6.0502202e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[155][j] = _itmp155[j];
			}
		}
		{ static float _itmp156[] = {-1.2953408e01,-8.8867404e-02,
		  1.6053237e00,4.2561073e-02,-7.6279056e-01,-3.3953346e-03,
		  7.2694294e-02,2.8573457e-04,-6.0288389e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[156][j] = _itmp156[j];
			}
		}
		{ static float _itmp157[] = {-1.2956943e01,-8.7976813e-02,
		  1.5938628e00,4.1269448e-02,-7.4783772e-01,-3.2527705e-03,
		  7.1391009e-02,2.8278027e-04,-6.0277963e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[157][j] = _itmp157[j];
			}
		}
		{ static float _itmp158[] = {-1.2960416e01,-8.5727885e-02,
		  1.5643640e00,4.0816348e-02,-7.4220324e-01,-3.2284218e-03,
		  7.1024448e-02,2.8120971e-04,-5.9981486e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[158][j] = _itmp158[j];
			}
		}
		{ static float _itmp159[] = {-1.2963779e01,-8.4199898e-02,
		  1.5439276e00,4.0322408e-02,-7.3616827e-01,-3.2038621e-03,
		  7.0735134e-02,2.7997163e-04,-5.9769265e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[159][j] = _itmp159[j];
			}
		}
		{ static float _itmp160[] = {-1.2966952e01,-8.3230697e-02,
		  1.5301222e00,4.0180188e-02,-7.3426080e-01,-3.1899358e-03,
		  7.0515700e-02,2.7935259e-04,-5.9679970e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[160][j] = _itmp160[j];
			}
		}
		{ static float _itmp161[] = {-1.2970169e01,-8.1922732e-02,
		  1.5121366e00,3.9889395e-02,-7.3052442e-01,-3.1807255e-03,
		  7.0345335e-02,2.7841030e-04,-5.9471731e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[161][j] = _itmp161[j];
			}
		}
		{ static float _itmp162[] = {-1.2973417e01,-8.0568887e-02,
		  1.4936481e00,3.9643470e-02,-7.2740614e-01,-3.1544948e-03,
		  6.9985405e-02,2.7737793e-04,-5.9341961e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[162][j] = _itmp162[j];
			}
		}
		{ static float _itmp163[] = {-1.2976655e01,-7.9133324e-02,
		  1.4741757e00,3.9214551e-02,-7.2203594e-01,-3.1454836e-03,
		  6.9844894e-02,2.7632061e-04,-5.9090154e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[163][j] = _itmp163[j];
			}
		}
		{ static float _itmp164[] = {-1.2979909e01,-7.7718452e-02,
		  1.4550174e00,3.8812678e-02,-7.1704441e-01,-3.1283244e-03,
		  6.9619231e-02,2.7525748e-04,-5.8886558e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[164][j] = _itmp164[j];
			}
		}
		{ static float _itmp165[] = {-1.2983279e01,-7.5756460e-02,
		  1.4292192e00,3.8145758e-02,-7.0891368e-01,-3.0902065e-03,
		  6.9178462e-02,2.7343514e-04,-5.8592604e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[165][j] = _itmp165[j];
			}
		}
		{ static float _itmp166[] = {-1.2986597e01,-7.4546166e-02,
		  1.4130697e00,3.7432842e-02,-7.0048577e-01,-3.0105109e-03,
		  6.8380482e-02,2.7161685e-04,-5.8537601e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[166][j] = _itmp166[j];
			}
		}
		{ static float _itmp167[] = {-1.2989790e01,-7.3429137e-02,
		  1.3977282e00,3.6971614e-02,-6.9487977e-01,-2.9638058e-03,
		  6.7870811e-02,2.7027668e-04,-5.8433427e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[167][j] = _itmp167[j];
			}
		}
		{ static float _itmp168[] = {-1.2994373e01,-6.9232814e-02,
		  1.3474331e00,3.3121433e-02,-6.4997578e-01,-2.6562612e-03,
		  6.5177783e-02,2.6247231e-04,-5.7938080e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[168][j] = _itmp168[j];
			}
		}
		{ static float _itmp169[] = {-1.2997371e01,-6.8472899e-02,
		  1.3361125e00,3.3033863e-02,-6.4866334e-01,-2.6357728e-03,
		  6.4876556e-02,2.6179751e-04,-5.7879491e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[169][j] = _itmp169[j];
			}
		}
		{ static float _itmp170[] = {-1.3000412e01,-6.7792535e-02,
		  1.3259878e00,3.2849804e-02,-6.4630127e-01,-2.6079121e-03,
		  6.4542651e-02,2.6114471e-04,-5.7859533e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[170][j] = _itmp170[j];
			}
		}
		{ static float _itmp171[] = {-1.3003260e01,-6.7829102e-02,
		  1.3246375e00,3.2851961e-02,-6.4615655e-01,-2.5692834e-03,
		  6.4087257e-02,2.6069206e-04,-5.7971510e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[171][j] = _itmp171[j];
			}
		}
		{ static float _itmp172[] = {-1.3006524e01,-6.6123828e-02,
		  1.3020529e00,3.2185357e-02,-6.3805223e-01,-2.5192185e-03,
		  6.3542232e-02,2.5892514e-04,-5.7759113e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[172][j] = _itmp172[j];
			}
		}
		{ static float _itmp173[] = {-1.3009602e01,-6.5535143e-02,
		  1.2934536e00,3.1657096e-02,-6.3175803e-01,-2.4298525e-03,
		  6.2621817e-02,2.5726823e-04,-5.7819127e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[173][j] = _itmp173[j];
			}
		}
		{ static float _itmp174[] = {-1.3012690e01,-6.4464197e-02,
		  1.2786407e00,3.1103499e-02,-6.2493664e-01,-2.3970867e-03,
		  6.2281970e-02,2.5600567e-04,-5.7644141e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[174][j] = _itmp174[j];
			}
		}
		{ static float _itmp175[] = {-1.3015742e01,-6.3292615e-02,
		  1.2623187e00,3.0911241e-02,-6.2240219e-01,-2.3749035e-03,
		  6.1965335e-02,2.5516644e-04,-5.7541276e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[175][j] = _itmp175[j];
			}
		}
		{ static float _itmp176[] = {-1.3018823e01,-6.2251348e-02,
		  1.2478187e00,3.0633559e-02,-6.1896765e-01,-2.3310555e-03,
		  6.1454713e-02,2.5416550e-04,-5.7516024e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[176][j] = _itmp176[j];
			}
		}
		{ static float _itmp177[] = {-1.3021775e01,-6.1349817e-02,
		  1.2348622e00,3.0273836e-02,-6.1434722e-01,-2.3160807e-03,
		  6.1261188e-02,2.5321500e-04,-5.7333452e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[177][j] = _itmp177[j];
			}
		}
		{ static float _itmp178[] = {-1.3024689e01,-6.0537633e-02,
		  1.2229009e00,3.0137872e-02,-6.1240596e-01,-2.3084602e-03,
		  6.1106890e-02,2.5262887e-04,-5.7216748e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[178][j] = _itmp178[j];
			}
		}
		{ static float _itmp179[] = {-1.3027586e01,-5.9678543e-02,
		  1.2103324e00,2.9960839e-02,-6.0993612e-01,-2.3050862e-03,
		  6.1000217e-02,2.5199380e-04,-5.7059913e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[179][j] = _itmp179[j];
			}
		}
		{ static float _itmp180[] = {-1.3031168e01,-5.6708992e-02,
		  1.1724976e00,2.8762685e-02,-5.9562188e-01,-2.2367430e-03,
		  6.0328782e-02,2.4945661e-04,-5.6706360e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[180][j] = _itmp180[j];
			}
		}
		{ static float _itmp181[] = {-1.3034077e01,-5.5576283e-02,
		  1.1564863e00,2.8526813e-02,-5.9242380e-01,-2.2226607e-03,
		  6.0086787e-02,2.4843679e-04,-5.6511429e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[181][j] = _itmp181[j];
			}
		}
		{ static float _itmp182[] = {-1.3037005e01,-5.4630835e-02,
		  1.1429633e00,2.8292291e-02,-5.8934224e-01,-2.2043742e-03,
		  5.9833918e-02,2.4763323e-04,-5.6392886e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[182][j] = _itmp182[j];
			}
		}
		{ static float _itmp183[] = {-1.3039930e01,-5.3878319e-02,
		  1.1320028e00,2.7943395e-02,-5.8495444e-01,-2.1721653e-03,
		  5.9475660e-02,2.4669591e-04,-5.6312443e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[183][j] = _itmp183[j];
			}
		}
		{ static float _itmp184[] = {-1.3042773e01,-5.3207684e-02,
		  1.1218417e00,2.7845055e-02,-5.8351463e-01,-2.1474899e-03,
		  5.9135381e-02,2.4599661e-04,-5.6269937e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[184][j] = _itmp184[j];
			}
		}
		{ static float _itmp185[] = {-1.3045631e01,-5.2361719e-02,
		  1.1094748e00,2.7687121e-02,-5.8130652e-01,-2.1395434e-03,
		  5.8981530e-02,2.4538484e-04,-5.6145983e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[185][j] = _itmp185[j];
			}
		}
		{ static float _itmp186[] = {-1.3048620e01,-5.0999772e-02,
		  1.0908823e00,2.7193120e-02,-5.7508516e-01,-2.1304821e-03,
		  5.8867689e-02,2.4435803e-04,-5.5894139e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[186][j] = _itmp186[j];
			}
		}
		{ static float _itmp187[] = {-1.3051424e01,-5.0298367e-02,
		  1.0802363e00,2.7208241e-02,-5.7496727e-01,-2.1186783e-03,
		  5.8638353e-02,2.4388103e-04,-5.5841473e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[187][j] = _itmp187[j];
			}
		}
		{ static float _itmp188[] = {-1.3054798e01,-4.8396397e-02,
		  1.0560801e00,2.5615320e-02,-5.5606204e-01,-1.9646869e-03,
		  5.7181526e-02,2.4031432e-04,-5.5700387e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[188][j] = _itmp188[j];
			}
		}
}
void HRFitData4(void)
{
	int j;
		{ static float _itmp189[] = {-1.3057538e01,-4.7986466e-02,
		  1.0491396e00,2.5556719e-02,-5.5506003e-01,-1.9491425e-03,
		  5.6948952e-02,2.3983984e-04,-5.5664838e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[189][j] = _itmp189[j];
			}
		}
		{ static float _itmp190[] = {-1.3060360e01,-4.7135577e-02,
		  1.0367275e00,2.5400663e-02,-5.5287540e-01,-1.9418235e-03,
		  5.6803722e-02,2.3925587e-04,-5.5543855e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[190][j] = _itmp190[j];
			}
		}
		{ static float _itmp191[] = {-1.3063112e01,-4.6454512e-02,
		  1.0263287e00,2.5366202e-02,-5.5211705e-01,-1.9409299e-03,
		  5.6707036e-02,2.3886634e-04,-5.5445228e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[191][j] = _itmp191[j];
			}
		}
		{ static float _itmp192[] = {-1.3065994e01,-4.5264028e-02,
		  1.0097355e00,2.5085492e-02,-5.4843128e-01,-1.9375290e-03,
		  5.6615263e-02,2.3813079e-04,-5.5251624e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[192][j] = _itmp192[j];
			}
		}
		{ static float _itmp193[] = {-1.3068822e01,-4.4381630e-02,
		  9.9706441e-01,2.4764877e-02,-5.4431301e-01,-1.9086647e-03,
		  5.6266703e-02,2.3712622e-04,-5.5136797e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[193][j] = _itmp193[j];
			}
		}
		{ static float _itmp194[] = {-1.3071781e01,-4.2990677e-02,
		  9.7818369e-01,2.4277924e-02,-5.3824836e-01,-1.8774200e-03,
		  5.5910721e-02,2.3590148e-04,-5.4963217e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[194][j] = _itmp194[j];
			}
		}
		{ static float _itmp195[] = {-1.3074587e01,-4.2212937e-02,
		  9.6685761e-01,2.3972459e-02,-5.3434151e-01,-1.8410934e-03,
		  5.5485237e-02,2.3487664e-04,-5.4888208e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[195][j] = _itmp195[j];
			}
		}
		{ static float _itmp196[] = {-1.3077398e01,-4.1054782e-02,
		  9.5056415e-01,2.3765044e-02,-5.3149098e-01,-1.8443284e-03,
		  5.5445764e-02,2.3424011e-04,-5.4687210e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[196][j] = _itmp196[j];
			}
		}
		{ static float _itmp197[] = {-1.3080053e01,-4.0832713e-02,
		  9.4603688e-01,2.3733305e-02,-5.3084260e-01,-1.8237924e-03,
		  5.5167668e-02,2.3382150e-04,-5.4696943e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[197][j] = _itmp197[j];
			}
		}
		{ static float _itmp198[] = {-1.3084186e01,-3.7503101e-02,
		  9.0635663e-01,1.9529695e-02,-4.8126963e-01,-1.4090535e-03,
		  5.1408544e-02,2.2518923e-04,-5.4567936e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[198][j] = _itmp198[j];
			}
		}
		{ static float _itmp199[] = {-1.3086956e01,-3.7329234e-02,
		  9.0297139e-01,1.9049436e-02,-4.7545779e-01,-1.3067833e-03,
		  5.0344542e-02,2.2359312e-04,-5.4717502e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[199][j] = _itmp199[j];
			}
		}
		{ static float _itmp200[] = {-1.3089603e01,-3.7013296e-02,
		  8.9738065e-01,1.8718341e-02,-4.7115290e-01,-1.2700945e-03,
		  4.9938835e-02,2.2263105e-04,-5.4658791e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[200][j] = _itmp200[j];
			}
		}
		{ static float _itmp201[] = {-1.3092388e01,-3.6613312e-02,
		  8.9112645e-01,1.8137496e-02,-4.6399495e-01,-1.2164193e-03,
		  4.9429540e-02,2.2147842e-04,-5.4628281e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[201][j] = _itmp201[j];
			}
		}
		{ static float _itmp202[] = {-1.3095132e01,-3.5684671e-02,
		  8.7770873e-01,1.8105540e-02,-4.6331334e-01,-1.2082190e-03,
		  4.9243122e-02,2.2101958e-04,-5.4551652e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[202][j] = _itmp202[j];
			}
		}
		{ static float _itmp203[] = {-1.3097836e01,-3.4898277e-02,
		  8.6620337e-01,1.7686090e-02,-4.5786914e-01,-1.1992300e-03,
		  4.9138360e-02,2.2015675e-04,-5.4348572e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[203][j] = _itmp203[j];
			}
		}
		{ static float _itmp204[] = {-1.3100678e01,-3.3545244e-02,
		  8.4772015e-01,1.7206248e-02,-4.5183688e-01,-1.1484984e-03,
		  4.8546895e-02,2.1858729e-04,-5.4203244e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[204][j] = _itmp204[j];
			}
		}
		{ static float _itmp205[] = {-1.3103742e01,-3.0996004e-02,
		  8.1423676e-01,1.6590720e-02,-4.4414535e-01,-1.1375952e-03,
		  4.8375625e-02,2.1718058e-04,-5.3848224e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[205][j] = _itmp205[j];
			}
		}
		{ static float _itmp206[] = {-1.3106484e01,-2.9996075e-02,
		  8.0005884e-01,1.6287584e-02,-4.4018418e-01,-1.1180365e-03,
		  4.8120461e-02,2.1631015e-04,-5.3711827e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[206][j] = _itmp206[j];
			}
		}
		{ static float _itmp207[] = {-1.3109170e01,-2.9401589e-02,
		  7.9102898e-01,1.6042413e-02,-4.3697619e-01,-1.0843258e-03,
		  4.7724154e-02,2.1549418e-04,-5.3677033e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[207][j] = _itmp207[j];
			}
		}
		{ static float _itmp208[] = {-1.3111835e01,-2.8596591e-02,
		  7.7921110e-01,1.5828785e-02,-4.3405038e-01,-1.0748868e-03,
		  4.7568917e-02,2.1484101e-04,-5.3545022e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[208][j] = _itmp208[j];
			}
		}
		{ static float _itmp209[] = {-1.3114451e01,-2.8021712e-02,
		  7.7025521e-01,1.5722275e-02,-4.3242958e-01,-1.0645821e-03,
		  4.7393683e-02,2.1434083e-04,-5.3466437e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[209][j] = _itmp209[j];
			}
		}
		{ static float _itmp210[] = {-1.3117069e01,-2.7390355e-02,
		  7.6058865e-01,1.5608025e-02,-4.3071494e-01,-1.0549861e-03,
		  4.7225296e-02,2.1383847e-04,-5.3381613e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[210][j] = _itmp210[j];
			}
		}
		{ static float _itmp211[] = {-1.3119642e01,-2.6827678e-02,
		  7.5171244e-01,1.5532415e-02,-4.2941636e-01,-1.0551515e-03,
		  4.7154587e-02,2.1343310e-04,-5.3269491e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[211][j] = _itmp211[j];
			}
		}
		{ static float _itmp212[] = {-1.3122627e01,-2.4786793e-02,
		  7.2491205e-01,1.4539313e-02,-4.1723102e-01,-9.9324796e-04,
		  4.6536803e-02,2.1127122e-04,-5.2991458e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[212][j] = _itmp212[j];
			}
		}
		{ static float _itmp213[] = {-1.3126088e01,-2.1792339e-02,
		  6.8734562e-01,1.2260759e-02,-3.8998574e-01,-7.7876507e-04,
		  4.4500880e-02,2.0635551e-04,-5.2790716e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[213][j] = _itmp213[j];
			}
		}
		{ static float _itmp214[] = {-1.3128549e01,-2.1782210e-02,
		  6.8543190e-01,1.2256890e-02,-3.8955799e-01,-7.7181013e-04,
		  4.4369794e-02,2.0612964e-04,-5.2771848e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[214][j] = _itmp214[j];
			}
		}
		{ static float _itmp215[] = {-1.3130976e01,-2.1724079e-02,
		  6.8283385e-01,1.2247022e-02,-3.8899589e-01,-7.6713820e-04,
		  4.4247556e-02,2.0576685e-04,-5.2707074e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[215][j] = _itmp215[j];
			}
		}
		{ static float _itmp216[] = {-1.3133466e01,-2.1663373e-02,
		  6.8037415e-01,1.2241915e-02,-3.8860452e-01,-7.6182414e-04,
		  4.4145171e-02,2.0565845e-04,-5.2706273e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[216][j] = _itmp216[j];
			}
		}
		{ static float _itmp217[] = {-1.3135907e01,-2.1633258e-02,
		  6.7823213e-01,1.2186357e-02,-3.8752565e-01,-7.5918442e-04,
		  4.4071436e-02,2.0540993e-04,-5.2652583e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[217][j] = _itmp217[j];
			}
		}
		{ static float _itmp218[] = {-1.3138672e01,-2.0304639e-02,
		  6.6006476e-01,1.1816905e-02,-3.8286060e-01,-7.1675749e-04,
		  4.3565512e-02,2.0421870e-04,-5.2559432e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[218][j] = _itmp218[j];
			}
		}
		{ static float _itmp219[] = {-1.3141154e01,-2.0029264e-02,
		  6.5485919e-01,1.1772300e-02,-3.8195014e-01,-7.1192428e-04,
		  4.3452870e-02,2.0390841e-04,-5.2502602e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[219][j] = _itmp219[j];
			}
		}
		{ static float _itmp220[] = {-1.3143664e01,-1.9739542e-02,
		  6.4958817e-01,1.1640893e-02,-3.8004282e-01,-6.8975711e-04,
		  4.3168984e-02,2.0335412e-04,-5.2477098e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[220][j] = _itmp220[j];
			}
		}
		{ static float _itmp221[] = {-1.3146100e01,-1.9568518e-02,
		  6.4566112e-01,1.1627582e-02,-3.7948197e-01,-6.8958191e-04,
		  4.3103468e-02,2.0311653e-04,-5.2415049e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[221][j] = _itmp221[j];
			}
		}
		{ static float _itmp222[] = {-1.3148540e01,-1.9432614e-02,
		  6.4222771e-01,1.1611138e-02,-3.7891248e-01,-6.8437046e-04,
		  4.2987730e-02,2.0285635e-04,-5.2376096e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[222][j] = _itmp222[j];
			}
		}
		{ static float _itmp223[] = {-1.3150968e01,-1.9343788e-02,
		  6.3938451e-01,1.1611722e-02,-3.7854138e-01,-6.8243133e-04,
		  4.2907432e-02,2.0265492e-04,-5.2334894e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[223][j] = _itmp223[j];
			}
		}
		{ static float _itmp224[] = {-1.3153507e01,-1.8464608e-02,
		  6.2654173e-01,1.1540948e-02,-3.7732452e-01,-6.7847315e-04,
		  4.2769868e-02,2.0211814e-04,-5.2211611e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[224][j] = _itmp224[j];
			}
		}
		{ static float _itmp225[] = {-1.3155905e01,-1.8441848e-02,
		  6.2455273e-01,1.1501432e-02,-3.7645847e-01,-6.7208614e-04,
		  4.2650033e-02,2.0182622e-04,-5.2170381e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[225][j] = _itmp225[j];
			}
		}
		{ static float _itmp226[] = {-1.3158326e01,-1.8335026e-02,
		  6.2154204e-01,1.1458837e-02,-3.7557763e-01,-6.7073258e-04,
		  4.2591255e-02,2.0164398e-04,-5.2125729e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[226][j] = _itmp226[j];
			}
		}
		{ static float _itmp227[] = {-1.3160786e01,-1.8198408e-02,
		  6.1832982e-01,1.1222233e-02,-3.7240538e-01,-6.3405483e-04,
		  4.2175617e-02,2.0085413e-04,-5.2117007e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[227][j] = _itmp227[j];
			}
		}
		{ static float _itmp228[] = {-1.3164427e01,-1.1986890e-02,
		  5.3910154e-01,9.2938384e-03,-3.4894601e-01,-5.7495444e-04,
		  4.1586533e-02,1.9695863e-04,-5.1288703e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[228][j] = _itmp228[j];
			}
		}
		{ static float _itmp229[] = {-1.3166804e01,-1.1901799e-02,
		  5.3627676e-01,9.3546435e-03,-3.4930721e-01,-5.7174213e-04,
		  4.1474923e-02,1.9676672e-04,-5.1262407e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[229][j] = _itmp229[j];
			}
		}
		{ static float _itmp230[] = {-1.3169178e01,-1.1852906e-02,
		  5.3395212e-01,9.3853436e-03,-3.4931341e-01,-5.6747138e-04,
		  4.1364525e-02,1.9658523e-04,-5.1241680e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[230][j] = _itmp230[j];
			}
		}
		{ static float _itmp231[] = {-1.3171542e01,-1.1817374e-02,
		  5.3181034e-01,9.3707321e-03,-3.4875259e-01,-5.6380749e-04,
		  4.1271280e-02,1.9636675e-04,-5.1204315e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[231][j] = _itmp231[j];
			}
		}
		{ static float _itmp232[] = {-1.3173882e01,-1.1846778e-02,
		  5.3047019e-01,9.3941940e-03,-3.4864652e-01,-5.6056387e-04,
		  4.1174076e-02,1.9616346e-04,-5.1172795e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[232][j] = _itmp232[j];
			}
		}
		{ static float _itmp233[] = {-1.3176305e01,-1.1606558e-02,
		  5.2582210e-01,9.3994597e-03,-3.4840339e-01,-5.5341277e-04,
		  4.1036014e-02,1.9596161e-04,-5.1159482e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[233][j] = _itmp233[j];
			}
		}
		{ static float _itmp234[] = {-1.3178675e01,-1.1390689e-02,
		  5.2139878e-01,9.3737897e-03,-3.4770238e-01,-5.5306882e-04,
		  4.0969651e-02,1.9568326e-04,-5.1087299e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[234][j] = _itmp234[j];
			}
		}
		{ static float _itmp235[] = {-1.3181014e01,-1.1251786e-02,
		  5.1790005e-01,9.4191181e-03,-3.4785107e-01,-5.5706454e-04,
		  4.0937424e-02,1.9551867e-04,-5.1024719e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[235][j] = _itmp235[j];
			}
		}
		{ static float _itmp236[] = {-1.3183327e01,-1.1277574e-02,
		  5.1653129e-01,9.4258906e-03,-3.4752837e-01,-5.5724644e-04,
		  4.0881682e-02,1.9532644e-04,-5.0974791e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[236][j] = _itmp236[j];
			}
		}
}
void HRFitData5(void)
{
	int j;
		{ static float _itmp237[] = {-1.3185724e01,-1.0871968e-02,
		  5.0970376e-01,9.5123509e-03,-3.4824744e-01,-5.5959885e-04,
		  4.0814169e-02,1.9517577e-04,-5.0926777e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[237][j] = _itmp237[j];
			}
		}
		{ static float _itmp238[] = {-1.3188044e01,-1.0823163e-02,
		  5.0743014e-01,9.4705559e-03,-3.4733984e-01,-5.5831840e-04,
		  4.0750466e-02,1.9491515e-04,-5.0863549e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[238][j] = _itmp238[j];
			}
		}
		{ static float _itmp239[] = {-1.3190366e01,-1.0752993e-02,
		  5.0490189e-01,9.4199991e-03,-3.4633020e-01,-5.5658841e-04,
		  4.0683337e-02,1.9464413e-04,-5.0799223e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[239][j] = _itmp239[j];
			}
		}
		{ static float _itmp240[] = {-1.3192693e01,-1.0740140e-02,
		  5.0310194e-01,9.5389318e-03,-3.4745741e-01,-5.5158342e-04,
		  4.0548787e-02,1.9455553e-04,-5.0813369e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[240][j] = _itmp240[j];
			}
		}
		{ static float _itmp241[] = {-1.3195190e01,-9.7072888e-03,
		  4.8845306e-01,9.3511492e-03,-3.4485266e-01,-5.5055658e-04,
		  4.0476743e-02,1.9398899e-04,-5.0654854e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[241][j] = _itmp241[j];
			}
		}
		{ static float _itmp242[] = {-1.3198017e01,-7.1791112e-03,
		  4.5498148e-01,9.2394045e-03,-3.4345296e-01,-5.4242800e-04,
		  4.0252864e-02,1.9328151e-04,-5.0499164e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[242][j] = _itmp242[j];
			}
		}
		{ static float _itmp243[] = {-1.3200281e01,-7.1573881e-03,
		  4.5302418e-01,9.1663273e-03,-3.4211755e-01,-5.4209912e-04,
		  4.0199120e-02,1.9291339e-04,-5.0402354e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[243][j] = _itmp243[j];
			}
		}
		{ static float _itmp244[] = {-1.3202567e01,-7.1133603e-03,
		  4.5084053e-01,9.1117155e-03,-3.4105352e-01,-5.4061285e-04,
		  4.0136226e-02,1.9263318e-04,-5.0334241e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[244][j] = _itmp244[j];
			}
		}
		{ static float _itmp245[] = {-1.3204932e01,-6.4961631e-03,
		  4.4140086e-01,8.8527510e-03,-3.3745775e-01,-5.4832851e-04,
		  4.0192436e-02,1.9210628e-04,-5.0132163e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[245][j] = _itmp245[j];
			}
		}
		{ static float _itmp246[] = {-1.3207204e01,-6.4880061e-03,
		  4.3967658e-01,8.8279871e-03,-3.3676153e-01,-5.4782373e-04,
		  4.0136088e-02,1.9187114e-04,-5.0072474e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[246][j] = _itmp246[j];
			}
		}
		{ static float _itmp247[] = {-1.3209503e01,-6.4052162e-03,
		  4.3704134e-01,8.8740671e-03,-3.3699480e-01,-5.4390414e-04,
		  4.0027708e-02,1.9170826e-04,-5.0055399e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[247][j] = _itmp247[j];
			}
		}
		{ static float _itmp248[] = {-1.3211766e01,-6.3896747e-03,
		  4.3523371e-01,8.8470979e-03,-3.3627048e-01,-5.4279203e-04,
		  3.9963137e-02,1.9144527e-04,-4.9992329e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[248][j] = _itmp248[j];
			}
		}
		{ static float _itmp249[] = {-1.3214051e01,-6.2836045e-03,
		  4.3230957e-01,8.8227587e-03,-3.3560899e-01,-5.4020254e-04,
		  3.9880615e-02,1.9118447e-04,-4.9937619e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[249][j] = _itmp249[j];
			}
		}
		{ static float _itmp250[] = {-1.3216320e01,-6.2085832e-03,
		  4.2977908e-01,8.7957308e-03,-3.3490932e-01,-5.3598121e-04,
		  3.9780039e-02,1.9089734e-04,-4.9885544e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[250][j] = _itmp250[j];
			}
		}
		{ static float _itmp251[] = {-1.3219804e01,-8.3094143e-04,
		  3.6109191e-01,7.1909544e-03,-3.1571701e-01,-3.9715963e-04,
		  3.8309339e-02,1.8707608e-04,-4.9559507e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[251][j] = _itmp251[j];
			}
		}
		{ static float _itmp252[] = {-1.3222052e01,-8.2665752e-04,
		  3.5947871e-01,7.0797815e-03,-3.1395191e-01,-3.9614833e-04,
		  3.8269803e-02,1.8677131e-04,-4.9477997e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[252][j] = _itmp252[j];
			}
		}
		{ static float _itmp253[] = {-1.3224281e01,-8.2145236e-04,
		  3.5782874e-01,6.9841202e-03,-3.1236514e-01,-3.9609161e-04,
		  3.8234897e-02,1.8647456e-04,-4.9394039e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[253][j] = _itmp253[j];
			}
		}
		{ static float _itmp254[] = {-1.3226524e01,-8.1961689e-04,
		  3.5627136e-01,6.8920543e-03,-3.1085598e-01,-3.9074189e-04,
		  3.8142975e-02,1.8615567e-04,-4.9334988e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[254][j] = _itmp254[j];
			}
		}
		{ static float _itmp255[] = {-1.3228744e01,-8.1255275e-04,
		  3.5460719e-01,6.7967596e-03,-3.0927286e-01,-3.8746485e-04,
		  3.8067918e-02,1.8578689e-04,-4.9252193e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[255][j] = _itmp255[j];
			}
		}
		{ static float _itmp256[] = {-1.3230975e01,-8.0201513e-04,
		  3.5292658e-01,6.7503601e-03,-3.0832425e-01,-3.8490284e-04,
		  3.7995718e-02,1.8552488e-04,-4.9195485e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[256][j] = _itmp256[j];
			}
		}
		{ static float _itmp257[] = {-1.3233188e01,-7.9763756e-04,
		  3.5128805e-01,6.7413491e-03,-3.0781880e-01,-3.8347711e-04,
		  3.7922163e-02,1.8525014e-04,-4.9133282e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[257][j] = _itmp257[j];
			}
		}
		{ static float _itmp258[] = {-1.3235422e01,-7.8033918e-04,
		  3.4954587e-01,6.7629344e-03,-3.0774677e-01,-3.7996433e-04,
		  3.7828583e-02,1.8509531e-04,-4.9113375e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[258][j] = _itmp258[j];
			}
		}
		{ static float _itmp259[] = {-1.3237631e01,-7.8221509e-04,
		  3.4802717e-01,6.7101200e-03,-3.0670589e-01,-3.7678619e-04,
		  3.7747409e-02,1.8477956e-04,-4.9046990e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[259][j] = _itmp259[j];
			}
		}
		{ static float _itmp260[] = {-1.3239866e01,-7.7661587e-04,
		  3.4649777e-01,6.6615213e-03,-3.0577818e-01,-3.7015782e-04,
		  3.7638478e-02,1.8455279e-04,-4.9021095e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[260][j] = _itmp260[j];
			}
		}
		{ static float _itmp261[] = {-1.3242065e01,-7.7587500e-04,
		  3.4494504e-01,6.6431137e-03,-3.0516601e-01,-3.6851389e-04,
		  3.7567075e-02,1.8428953e-04,-4.8961658e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[261][j] = _itmp261[j];
			}
		}
		{ static float _itmp262[] = {-1.3244277e01,-7.7473716e-04,
		  3.4343606e-01,6.6277492e-03,-3.0462205e-01,-3.6600183e-04,
		  3.7491769e-02,1.8407908e-04,-4.8919981e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[262][j] = _itmp262[j];
			}
		}
		{ static float _itmp263[] = {-1.3246514e01,-7.6982356e-04,
		  3.4197462e-01,6.6093961e-03,-3.0411428e-01,-3.5832656e-04,
		  3.7374150e-02,1.8397857e-04,-4.8931879e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[263][j] = _itmp263[j];
			}
		}
		{ static float _itmp264[] = {-1.3248674e01,-7.6786662e-04,
		  3.4035367e-01,6.5990705e-03,-3.0355665e-01,-3.5436792e-04,
		  3.7258577e-02,1.8353549e-04,-4.8844139e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[264][j] = _itmp264[j];
			}
		}
		{ static float _itmp265[] = {-1.3250866e01,-7.6698826e-04,
		  3.3884928e-01,6.5965806e-03,-3.0316821e-01,-3.5148970e-04,
		  3.7173793e-02,1.8331234e-04,-4.8802532e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[265][j] = _itmp265[j];
			}
		}
		{ static float _itmp266[] = {-1.3253060e01,-7.6706550e-04,
		  3.3737850e-01,6.6036237e-03,-3.0291149e-01,-3.4998709e-04,
		  3.7106588e-02,1.8314648e-04,-4.8767943e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[266][j] = _itmp266[j];
			}
		}
		{ static float _itmp267[] = {-1.3255239e01,-7.6635060e-04,
		  3.3587417e-01,6.6003860e-03,-3.0250448e-01,-3.4785736e-04,
		  3.7028316e-02,1.8290862e-04,-4.8718541e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[267][j] = _itmp267[j];
			}
		}
		{ static float _itmp268[] = {-1.3257406e01,-7.6270301e-04,
		  3.3429962e-01,6.6470760e-03,-3.0270508e-01,-3.4295832e-04,
		  3.6897313e-02,1.8258177e-04,-4.8669172e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[268][j] = _itmp268[j];
			}
		}
		{ static float _itmp269[] = {-1.3259604e01,-7.6197751e-04,
		  3.3289754e-01,6.6299546e-03,-3.0219194e-01,-3.4094765e-04,
		  3.6842607e-02,1.8252357e-04,-4.8659579e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[269][j] = _itmp269[j];
			}
		}
		{ static float _itmp270[] = {-1.3261773e01,-7.6131657e-04,
		  3.3141935e-01,6.6127926e-03,-3.0161223e-01,-3.3772172e-04,
		  3.6753304e-02,1.8223225e-04,-4.8602796e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[270][j] = _itmp270[j];
			}
		}
		{ static float _itmp271[] = {-1.3263946e01,-7.6055585e-04,
		  3.2997060e-01,6.6239107e-03,-3.0141404e-01,-3.3696077e-04,
		  3.6695331e-02,1.8209773e-04,-4.8571369e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[271][j] = _itmp271[j];
			}
		}
		{ static float _itmp272[] = {-1.3266154e01,-7.4938894e-04,
		  3.2846478e-01,6.6200942e-03,-3.0106264e-01,-3.3477889e-04,
		  3.6628854e-02,1.8193998e-04,-4.8542800e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[272][j] = _itmp272[j];
			}
		}
		{ static float _itmp273[] = {-1.3268272e01,-7.4989401e-04,
		  3.2694817e-01,6.6437419e-03,-3.0097267e-01,-3.3262209e-04,
		  3.6539260e-02,1.8170584e-04,-4.8495396e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[273][j] = _itmp273[j];
			}
		}
		{ static float _itmp274[] = {-1.3270416e01,-7.4682286e-04,
		  3.2544985e-01,6.6162404e-03,-3.0025589e-01,-3.3200128e-04,
		  3.6482181e-02,1.8143585e-04,-4.8427745e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[274][j] = _itmp274[j];
			}
		}
		{ static float _itmp275[] = {-1.3272562e01,-7.4775046e-04,
		  3.2402435e-01,6.6079204e-03,-2.9979685e-01,-3.3100939e-04,
		  3.6420766e-02,1.8122388e-04,-4.8377509e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[275][j] = _itmp275[j];
			}
		}
		{ static float _itmp276[] = {-1.3274715e01,-7.4409379e-04,
		  3.2256514e-01,6.6108876e-03,-2.9949516e-01,-3.2935388e-04,
		  3.6351468e-02,1.8102983e-04,-4.8336703e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[276][j] = _itmp276[j];
			}
		}
		{ static float _itmp277[] = {-1.3276862e01,-7.3208753e-04,
		  3.2100445e-01,6.5953392e-03,-2.9896069e-01,-3.2560749e-04,
		  3.6260229e-02,1.8078032e-04,-4.8292326e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[277][j] = _itmp277[j];
			}
		}
		{ static float _itmp278[] = {-1.3279005e01,-7.2463247e-04,
		  3.1950697e-01,6.5992936e-03,-2.9867467e-01,-3.2317292e-04,
		  3.6181431e-02,1.8057832e-04,-4.8253858e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[278][j] = _itmp278[j];
			}
		}
		{ static float _itmp279[] = {-1.3281138e01,-7.2414626e-04,
		  3.1809390e-01,6.5852907e-03,-2.9814985e-01,-3.2238773e-04,
		  3.6124296e-02,1.8036173e-04,-4.8200982e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[279][j] = _itmp279[j];
			}
		}
		{ static float _itmp280[] = {-1.3283271e01,-7.2541944e-04,
		  3.1670111e-01,6.6498742e-03,-2.9862520e-01,-3.2082084e-04,
		  3.6040924e-02,1.8021531e-04,-4.8176800e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[280][j] = _itmp280[j];
			}
		}
		{ static float _itmp281[] = {-1.3285397e01,-7.2328729e-04,
		  3.1528217e-01,6.6386028e-03,-2.9814252e-01,-3.1824748e-04,
		  3.5962638e-02,1.7998186e-04,-4.8130103e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[281][j] = _itmp281[j];
			}
		}
		{ static float _itmp282[] = {-1.3287514e01,-7.2373607e-04,
		  3.1389114e-01,6.6279173e-03,-2.9765782e-01,-3.1813583e-04,
		  3.5912123e-02,1.7976992e-04,-4.8074829e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[282][j] = _itmp282[j];
			}
		}
		{ static float _itmp283[] = {-1.3289632e01,-7.2179240e-04,
		  3.1248084e-01,6.6279927e-03,-2.9731724e-01,-3.1635899e-04,
		  3.5840303e-02,1.7955374e-04,-4.8028813e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[283][j] = _itmp283[j];
			}
		}
}
void HRFitData6(void)
{
	int j;
		{ static float _itmp284[] = {-1.3291748e01,-7.1921229e-04,
		  3.1107983e-01,6.5994109e-03,-2.9661700e-01,-3.1426287e-04,
		  3.5771307e-02,1.7929859e-04,-4.7972943e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[284][j] = _itmp284[j];
			}
		}
		{ static float _itmp285[] = {-1.3293864e01,-7.1916747e-04,
		  3.0971345e-01,6.6430373e-03,-2.9684189e-01,-3.1236492e-04,
		  3.5691325e-02,1.7914933e-04,-4.7947778e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[285][j] = _itmp285[j];
			}
		}
		{ static float _itmp286[] = {-1.3295978e01,-7.1638305e-04,
		  3.0832469e-01,6.6466858e-03,-2.9655865e-01,-3.1014753e-04,
		  3.5615344e-02,1.7894122e-04,-4.7906698e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[286][j] = _itmp286[j];
			}
		}
		{ static float _itmp287[] = {-1.3298111e01,-7.1157719e-04,
		  3.0695164e-01,6.7487522e-03,-2.9757187e-01,-2.9744394e-04,
		  3.5399422e-02,1.7874957e-04,-4.7937366e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[287][j] = _itmp287[j];
			}
		}
		{ static float _itmp288[] = {-1.3300211e01,-7.0996478e-04,
		  3.0558297e-01,6.7332764e-03,-2.9704145e-01,-2.9655869e-04,
		  3.5342693e-02,1.7853112e-04,-4.7884570e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[288][j] = _itmp288[j];
			}
		}
		{ static float _itmp289[] = {-1.3302302e01,-7.0314389e-04,
		  3.0414376e-01,6.7103403e-03,-2.9641232e-01,-2.9556340e-04,
		  3.5285067e-02,1.7829708e-04,-4.7827670e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[289][j] = _itmp289[j];
			}
		}
		{ static float _itmp290[] = {-1.3304389e01,-7.0238649e-04,
		  3.0278865e-01,6.6829543e-03,-2.9572523e-01,-2.9509258e-04,
		  3.5234705e-02,1.7806100e-04,-4.7767097e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[290][j] = _itmp290[j];
			}
		}
		{ static float _itmp291[] = {-1.3306476e01,-7.0066412e-04,
		  3.0143383e-01,6.6583427e-03,-2.9507849e-01,-2.9492422e-04,
		  3.5188518e-02,1.7784169e-04,-4.7709127e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[291][j] = _itmp291[j];
			}
		}
		{ static float _itmp292[] = {-1.3308562e01,-6.9839024e-04,
		  3.0007389e-01,6.6698603e-03,-2.9489484e-01,-2.9427942e-04,
		  3.5129134e-02,1.7765714e-04,-4.7665513e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[292][j] = _itmp292[j];
			}
		}
		{ static float _itmp293[] = {-1.3310640e01,-6.8606721e-04,
		  2.9858753e-01,6.6589457e-03,-2.9442561e-01,-2.9244725e-04,
		  3.5059165e-02,1.7742314e-04,-4.7614411e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[293][j] = _itmp293[j];
			}
		}
		{ static float _itmp294[] = {-1.3312718e01,-6.8443635e-04,
		  2.9724762e-01,6.6695767e-03,-2.9423219e-01,-2.9134200e-04,
		  3.4994066e-02,1.7722219e-04,-4.7569526e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[294][j] = _itmp294[j];
			}
		}
		{ static float _itmp295[] = {-1.3314791e01,-6.8401627e-04,
		  2.9592329e-01,6.6776946e-03,-2.9400373e-01,-2.9097329e-04,
		  3.4937967e-02,1.7702823e-04,-4.7521754e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[295][j] = _itmp295[j];
			}
		}
		{ static float _itmp296[] = {-1.3316862e01,-6.8413402e-04,
		  2.9462621e-01,6.6809035e-03,-2.9372609e-01,-2.8897583e-04,
		  3.4866169e-02,1.7682857e-04,-4.7481353e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[296][j] = _itmp296[j];
			}
		}
		{ static float _itmp297[] = {-1.3318931e01,-6.8457774e-04,
		  2.9333210e-01,6.6764248e-03,-2.9334196e-01,-2.8796814e-04,
		  3.4805600e-02,1.7660984e-04,-4.7430275e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[297][j] = _itmp297[j];
			}
		}
		{ static float _itmp298[] = {-1.3321024e01,-6.9219188e-04,
		  2.9218215e-01,6.7424779e-03,-2.9388982e-01,-2.8830193e-04,
		  3.4753792e-02,1.7654235e-04,-4.7415416e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[298][j] = _itmp298[j];
			}
		}
		{ static float _itmp299[] = {-1.3323061e01,-6.9039548e-04,
		  2.9080269e-01,6.8318467e-03,-2.9467437e-01,-2.8822370e-04,
		  3.4676809e-02,1.7638467e-04,-4.7381804e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[299][j] = _itmp299[j];
			}
		}
		{ static float _itmp300[] = {-1.3325114e01,-6.8570761e-04,
		  2.8946334e-01,6.8117632e-03,-2.9410237e-01,-2.8629441e-04,
		  3.4610856e-02,1.7616407e-04,-4.7333441e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[300][j] = _itmp300[j];
			}
		}
		{ static float _itmp301[] = {-1.3327168e01,-6.8333244e-04,
		  2.8816110e-01,6.7891153e-03,-2.9349571e-01,-2.8545308e-04,
		  3.4558099e-02,1.7594516e-04,-4.7279513e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[301][j] = _itmp301[j];
			}
		}
		{ static float _itmp302[] = {-1.3329213e01,-6.8286801e-04,
		  2.8687060e-01,6.8020425e-03,-2.9333314e-01,-2.8617826e-04,
		  3.4514062e-02,1.7576176e-04,-4.7228807e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[302][j] = _itmp302[j];
			}
		}
		{ static float _itmp303[] = {-1.3331262e01,-6.8112818e-04,
		  2.8558660e-01,6.8088411e-03,-2.9310653e-01,-2.8490837e-04,
		  3.4450054e-02,1.7556766e-04,-4.7186059e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[303][j] = _itmp303[j];
			}
		}
		{ static float _itmp304[] = {-1.3333305e01,-6.8009045e-04,
		  2.8431413e-01,6.8148398e-03,-2.9286990e-01,-2.8349922e-04,
		  3.4384597e-02,1.7537260e-04,-4.7143647e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[304][j] = _itmp304[j];
			}
		}
		{ static float _itmp305[] = {-1.3335341e01,-6.7798776e-04,
		  2.8301808e-01,6.8475646e-03,-2.9296634e-01,-2.8363758e-04,
		  3.4329690e-02,1.7520291e-04,-4.7101439e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[305][j] = _itmp305[j];
			}
		}
		{ static float _itmp306[] = {-1.3337382e01,-6.7299634e-04,
		  2.8171086e-01,6.8343580e-03,-2.9248318e-01,-2.8325382e-04,
		  3.4279492e-02,1.7498007e-04,-4.7045299e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[306][j] = _itmp306[j];
			}
		}
		{ static float _itmp307[] = {-1.3339415e01,-6.6342711e-04,
		  2.8036222e-01,6.8071792e-03,-2.9184675e-01,-2.7421783e-04,
		  3.4132324e-02,1.7467278e-04,-4.7014994e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[307][j] = _itmp307[j];
			}
		}
		{ static float _itmp308[] = {-1.3341439e01,-6.6059188e-04,
		  2.7907839e-01,6.7957086e-03,-2.9138327e-01,-2.7309640e-04,
		  3.4072679e-02,1.7444849e-04,-4.6962206e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[308][j] = _itmp308[j];
			}
		}
		{ static float _itmp309[] = {-1.3343461e01,-6.5792468e-04,
		  2.7781174e-01,6.7545613e-03,-2.9054502e-01,-2.7160440e-04,
		  3.4016114e-02,1.7419273e-04,-4.6901372e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[309][j] = _itmp309[j];
			}
		}
		{ static float _itmp310[] = {-1.3345473e01,-6.5459608e-04,
		  2.7652240e-01,6.7398851e-03,-2.9003680e-01,-2.7052639e-04,
		  3.3956595e-02,1.7395485e-04,-4.6844743e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[310][j] = _itmp310[j];
			}
		}
		{ static float _itmp311[] = {-1.3347496e01,-6.5362640e-04,
		  2.7530217e-01,6.7212498e-03,-2.8949887e-01,-2.6946960e-04,
		  3.3903334e-02,1.7375243e-04,-4.6796510e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[311][j] = _itmp311[j];
			}
		}
		{ static float _itmp312[] = {-1.3349520e01,-6.4906600e-04,
		  2.7405357e-01,6.7125200e-03,-2.8910217e-01,-2.6574897e-04,
		  3.3818189e-02,1.7354758e-04,-4.6763285e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[312][j] = _itmp312[j];
			}
		}
		{ static float _itmp313[] = {-1.3351519e01,-6.4685661e-04,
		  2.7279213e-01,6.6730855e-03,-2.8827655e-01,-2.6251690e-04,
		  3.3736717e-02,1.7323349e-04,-4.6698241e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[313][j] = _itmp313[j];
			}
		}
		{ static float _itmp314[] = {-1.3353528e01,-6.4423436e-04,
		  2.7156013e-01,6.6452129e-03,-2.8762120e-01,-2.5995274e-04,
		  3.3666797e-02,1.7299831e-04,-4.6649375e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[314][j] = _itmp314[j];
			}
		}
		{ static float _itmp315[] = {-1.3355539e01,-6.4398092e-04,
		  2.7036965e-01,6.6402685e-03,-2.8726271e-01,-2.5887135e-04,
		  3.3610597e-02,1.7280941e-04,-4.6605817e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[315][j] = _itmp315[j];
			}
		}
		{ static float _itmp316[] = {-1.3357540e01,-6.4337399e-04,
		  2.6916251e-01,6.6606235e-03,-2.8722435e-01,-2.5581112e-04,
		  3.3523131e-02,1.7260655e-04,-4.6572061e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[316][j] = _itmp316[j];
			}
		}
		{ static float _itmp317[] = {-1.3359543e01,-6.4163830e-04,
		  2.6795366e-01,6.6622980e-03,-2.8694680e-01,-2.5500418e-04,
		  3.3467337e-02,1.7241320e-04,-4.6526566e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[317][j] = _itmp317[j];
			}
		}
		{ static float _itmp318[] = {-1.3361534e01,-6.3993782e-04,
		  2.6674256e-01,6.6507608e-03,-2.8649792e-01,-2.5374995e-04,
		  3.3408754e-02,1.7220073e-04,-4.6477490e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[318][j] = _itmp318[j];
			}
		}
		{ static float _itmp319[] = {-1.3363512e01,-6.3811243e-04,
		  2.6550519e-01,6.6600107e-03,-2.8629789e-01,-2.5408546e-04,
		  3.3360671e-02,1.7199689e-04,-4.6423408e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[319][j] = _itmp319[j];
			}
		}
		{ static float _itmp320[] = {-1.3365514e01,-6.3873112e-04,
		  2.6436511e-01,6.6588847e-03,-2.8600225e-01,-2.5240026e-04,
		  3.3298705e-02,1.7181796e-04,-4.6386141e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[320][j] = _itmp320[j];
			}
		}
		{ static float _itmp321[] = {-1.3367488e01,-6.3719816e-04,
		  2.6315734e-01,6.6418066e-03,-2.8547776e-01,-2.4957670e-04,
		  3.3220313e-02,1.7155833e-04,-4.6333820e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[321][j] = _itmp321[j];
			}
		}
		{ static float _itmp322[] = {-1.3369480e01,-6.3675677e-04,
		  2.6201126e-01,6.6172159e-03,-2.8488338e-01,-2.4635284e-04,
		  3.3145152e-02,1.7133885e-04,-4.6292916e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[322][j] = _itmp322[j];
			}
		}
		{ static float _itmp323[] = {-1.3371452e01,-6.3471007e-04,
		  2.6081765e-01,6.5979501e-03,-2.8433919e-01,-2.4487005e-04,
		  3.3085931e-02,1.7111596e-04,-4.6241856e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[323][j] = _itmp323[j];
			}
		}
		{ static float _itmp324[] = {-1.3373438e01,-6.3167274e-04,
		  2.5964290e-01,6.5956870e-03,-2.8403071e-01,-2.4289235e-04,
		  3.3020388e-02,1.7092931e-04,-4.6204114e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[324][j] = _itmp324[j];
			}
		}
		{ static float _itmp325[] = {-1.3375402e01,-6.2763580e-04,
		  2.5842378e-01,6.5851444e-03,-2.8359541e-01,-2.3895565e-04,
		  3.2927018e-02,1.7065652e-04,-4.6155471e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[325][j] = _itmp325[j];
			}
		}
		{ static float _itmp326[] = {-1.3377335e01,-6.0663017e-04,
		  2.5691289e-01,6.5815612e-03,-2.8320050e-01,-2.3779477e-04,
		  3.2852635e-02,1.7031492e-04,-4.6075252e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[326][j] = _itmp326[j];
			}
		}
		{ static float _itmp327[] = {-1.3379336e01,-6.0534716e-04,
		  2.5584093e-01,6.5581608e-03,-2.8266579e-01,-2.3623426e-04,
		  3.2808758e-02,1.7021863e-04,-4.6055024e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[327][j] = _itmp327[j];
			}
		}
		{ static float _itmp328[] = {-1.3381303e01,-5.8850710e-04,
		  2.5449637e-01,6.5356535e-03,-2.8209597e-01,-2.3512293e-04,
		  3.2757048e-02,1.7001665e-04,-4.6006567e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[328][j] = _itmp328[j];
			}
		}
		{ static float _itmp329[] = {-1.3383256e01,-5.8878894e-04,
		  2.5334999e-01,6.5559577e-03,-2.8205806e-01,-2.3489298e-04,
		  3.2702781e-02,1.6983029e-04,-4.5961211e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[329][j] = _itmp329[j];
			}
		}
		{ static float _itmp330[] = {-1.3385196e01,-5.8889500e-04,
		  2.5217858e-01,6.5703699e-03,-2.8192601e-01,-2.3572835e-04,
		  3.2658398e-02,1.6961477e-04,-4.5902194e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[330][j] = _itmp330[j];
			}
		}
}
void HRFitData7(void)
{
	int j;
		{ static float _itmp331[] = {-1.3387143e01,-5.8821723e-04,
		  2.5103953e-01,6.5495595e-03,-2.8136694e-01,-2.3554920e-04,
		  3.2616150e-02,1.6940940e-04,-4.5847944e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[331][j] = _itmp331[j];
			}
		}
		{ static float _itmp332[] = {-1.3389093e01,-5.9045834e-04,
		  2.4995309e-01,6.5237428e-03,-2.8074950e-01,-2.3367975e-04,
		  3.2555372e-02,1.6917952e-04,-4.5796856e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[332][j] = _itmp332[j];
			}
		}
		{ static float _itmp333[] = {-1.3391018e01,-5.8387528e-04,
		  2.4870461e-01,6.5301615e-03,-2.8051826e-01,-2.3303642e-04,
		  3.2494403e-02,1.6893038e-04,-4.5737163e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[333][j] = _itmp333[j];
			}
		}
		{ static float _itmp334[] = {-1.3392981e01,-5.8193255e-04,
		  2.4761587e-01,6.5082568e-03,-2.7998090e-01,-2.3276437e-04,
		  3.2459758e-02,1.6879344e-04,-4.5700083e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[334][j] = _itmp334[j];
			}
		}
		{ static float _itmp335[] = {-1.3394905e01,-5.8407109e-04,
		  2.4651144e-01,6.4598406e-03,-2.7905416e-01,-2.3189146e-04,
		  3.2411881e-02,1.6851618e-04,-4.5629749e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[335][j] = _itmp335[j];
			}
		}
		{ static float _itmp336[] = {-1.3396842e01,-5.8410002e-04,
		  2.4541464e-01,6.4404900e-03,-2.7852464e-01,-2.3123690e-04,
		  3.2365590e-02,1.6831883e-04,-4.5580305e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[336][j] = _itmp336[j];
			}
		}
		{ static float _itmp337[] = {-1.3398787e01,-5.8474776e-04,
		  2.4434231e-01,6.4675990e-03,-2.7860382e-01,-2.3116139e-04,
		  3.2318000e-02,1.6819101e-04,-4.5548943e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[337][j] = _itmp337[j];
			}
		}
		{ static float _itmp338[] = {-1.3400708e01,-5.8316148e-04,
		  2.4321960e-01,6.4258650e-03,-2.7778167e-01,-2.2463693e-04,
		  3.2201663e-02,1.6785174e-04,-4.5495923e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[338][j] = _itmp338[j];
			}
		}
		{ static float _itmp339[] = {-1.3402897e01,-5.9998850e-04,
		  2.4322040e-01,5.7196333e-03,-2.6906997e-01,-5.5409688e-05,
		  3.0373884e-02,1.6561423e-04,-4.5829816e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[339][j] = _itmp339[j];
			}
		}
		{ static float _itmp340[] = {-1.3404808e01,-5.9558300e-04,
		  2.4205256e-01,5.7007922e-03,-2.6852867e-01,-5.5509412e-05,
		  3.0331738e-02,1.6539171e-04,-4.5769759e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[340][j] = _itmp340[j];
			}
		}
		{ static float _itmp341[] = {-1.3406730e01,-5.9095293e-04,
		  2.4091353e-01,5.6736534e-03,-2.6789662e-01,-5.5455457e-05,
		  3.0293126e-02,1.6518046e-04,-4.5713070e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[341][j] = _itmp341[j];
			}
		}
		{ static float _itmp342[] = {-1.3408638e01,-5.8064883e-04,
		  2.3968893e-01,5.6302799e-03,-2.6704735e-01,-5.5424814e-05,
		  3.0256612e-02,1.6494616e-04,-4.5648599e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[342][j] = _itmp342[j];
			}
		}
		{ static float _itmp343[] = {-1.3410547e01,-5.7923497e-04,
		  2.3859352e-01,5.5925809e-03,-2.6627776e-01,-5.5313612e-05,
		  3.0220067e-02,1.6473178e-04,-4.5590117e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[343][j] = _itmp343[j];
			}
		}
		{ static float _itmp344[] = {-1.3412461e01,-5.7871698e-04,
		  2.3752292e-01,5.5659139e-03,-2.6565379e-01,-5.4856999e-05,
		  3.0176738e-02,1.6451425e-04,-4.5534219e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[344][j] = _itmp344[j];
			}
		}
		{ static float _itmp345[] = {-1.3414354e01,-5.7856692e-04,
		  2.3643082e-01,5.5261557e-03,-2.6484543e-01,-5.4251090e-05,
		  3.0131448e-02,1.6426267e-04,-4.5469161e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[345][j] = _itmp345[j];
			}
		}
		{ static float _itmp346[] = {-1.3416247e01,-5.7491352e-04,
		  2.3530848e-01,5.4815803e-03,-2.6398301e-01,-5.3759650e-05,
		  3.0090371e-02,1.6402134e-04,-4.5405612e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[346][j] = _itmp346[j];
			}
		}
		{ static float _itmp347[] = {-1.3418148e01,-5.7412992e-04,
		  2.3423953e-01,5.4626339e-03,-2.6345900e-01,-5.3616612e-05,
		  3.0049451e-02,1.6382139e-04,-4.5352685e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[347][j] = _itmp347[j];
			}
		}
		{ static float _itmp348[] = {-1.3420036e01,-5.6813494e-04,
		  2.3308247e-01,5.4314230e-03,-2.6276055e-01,-5.3533895e-05,
		  3.0007513e-02,1.6357032e-04,-4.5285807e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[348][j] = _itmp348[j];
			}
		}
		{ static float _itmp349[] = {-1.3421936e01,-5.6752085e-04,
		  2.3203650e-01,5.4174578e-03,-2.6231018e-01,-5.3495733e-05,
		  2.9968787e-02,1.6339066e-04,-4.5237849e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[349][j] = _itmp349[j];
			}
		}
		{ static float _itmp350[] = {-1.3423819e01,-5.6533731e-04,
		  2.3094353e-01,5.4152338e-03,-2.6199707e-01,-5.3282838e-05,
		  2.9921787e-02,1.6319643e-04,-4.5187967e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[350][j] = _itmp350[j];
			}
		}
		{ static float _itmp351[] = {-1.3425708e01,-5.6360610e-04,
		  2.2987735e-01,5.4042363e-03,-2.6157799e-01,-5.3312058e-05,
		  2.9881123e-02,1.6300114e-04,-4.5136078e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[351][j] = _itmp351[j];
			}
		}
		{ static float _itmp352[] = {-1.3427589e01,-5.5677549e-04,
		  2.2874257e-01,5.3794845e-03,-2.6098007e-01,-5.3177209e-05,
		  2.9841229e-02,1.6279097e-04,-4.5079994e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[352][j] = _itmp352[j];
			}
		}
		{ static float _itmp353[] = {-1.3429470e01,-5.5960112e-04,
		  2.2774051e-01,5.3585768e-03,-2.6043138e-01,-5.3392061e-05,
		  2.9805183e-02,1.6258843e-04,-4.5024343e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[353][j] = _itmp353[j];
			}
		}
		{ static float _itmp354[] = {-1.3431344e01,-5.6105340e-04,
		  2.2672038e-01,5.3373934e-03,-2.5987861e-01,-5.3465988e-05,
		  2.9767314e-02,1.6238568e-04,-4.4969148e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[354][j] = _itmp354[j];
			}
		}
		{ static float _itmp355[] = {-1.3433219e01,-5.6162011e-04,
		  2.2569683e-01,5.3206407e-03,-2.5938579e-01,-5.3238109e-05,
		  2.9724862e-02,1.6217942e-04,-4.4915434e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[355][j] = _itmp355[j];
			}
		}
		{ static float _itmp356[] = {-1.3435085e01,-5.4627762e-04,
		  2.2445939e-01,5.2893148e-03,-2.5870210e-01,-5.3644475e-05,
		  2.9692220e-02,1.6196092e-04,-4.4853692e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[356][j] = _itmp356[j];
			}
		}
		{ static float _itmp357[] = {-1.3436959e01,-5.4430892e-04,
		  2.2341958e-01,5.2698567e-03,-2.5818163e-01,-5.3604777e-05,
		  2.9654274e-02,1.6176738e-04,-4.4801794e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[357][j] = _itmp357[j];
			}
		}
		{ static float _itmp358[] = {-1.3438821e01,-5.4146082e-04,
		  2.2235101e-01,5.2518644e-03,-2.5766891e-01,-5.3449523e-05,
		  2.9611800e-02,1.6155244e-04,-4.4745221e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[358][j] = _itmp358[j];
			}
		}
		{ static float _itmp359[] = {-1.3440679e01,-5.4033770e-04,
		  2.2131585e-01,5.2389847e-03,-2.5722888e-01,-5.3312062e-05,
		  2.9570058e-02,1.6135514e-04,-4.4693430e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[359][j] = _itmp359[j];
			}
		}
		{ static float _itmp360[] = {-1.3442543e01,-5.3498452e-04,
		  2.2024029e-01,5.2287355e-03,-2.5682974e-01,-5.3366177e-05,
		  2.9531173e-02,1.6117093e-04,-4.4644084e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[360][j] = _itmp360[j];
			}
		}
		{ static float _itmp361[] = {-1.3444395e01,-5.3274946e-04,
		  2.1919201e-01,5.2028238e-03,-2.5621736e-01,-5.3193740e-05,
		  2.9490998e-02,1.6095283e-04,-4.4586072e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[361][j] = _itmp361[j];
			}
		}
		{ static float _itmp362[] = {-1.3446251e01,-5.3106959e-04,
		  2.1817052e-01,5.1822658e-03,-2.5568423e-01,-5.2986397e-05,
		  2.9451221e-02,1.6075018e-04,-4.4532917e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[362][j] = _itmp362[j];
			}
		}
		{ static float _itmp363[] = {-1.3448102e01,-5.2851939e-04,
		  2.1713461e-01,5.1668147e-03,-2.5521386e-01,-5.3021464e-05,
		  2.9412365e-02,1.6055124e-04,-4.4479524e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[363][j] = _itmp363[j];
			}
		}
		{ static float _itmp364[] = {-1.3449940e01,-5.2841200e-04,
		  2.1612079e-01,5.1480713e-03,-2.5469479e-01,-5.2943651e-05,
		  2.9371742e-02,1.6033644e-04,-4.4422662e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[364][j] = _itmp364[j];
			}
		}
		{ static float _itmp365[] = {-1.3451788e01,-5.2516325e-04,
		  2.1508823e-01,5.1349015e-03,-2.5425920e-01,-5.2455973e-05,
		  2.9326860e-02,1.6013913e-04,-4.4372799e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[365][j] = _itmp365[j];
			}
		}
		{ static float _itmp366[] = {-1.3453640e01,-5.2437029e-04,
		  2.1410584e-01,5.1179705e-03,-2.5378299e-01,-5.2259096e-05,
		  2.9288430e-02,1.5995430e-04,-4.4324310e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[366][j] = _itmp366[j];
			}
		}
		{ static float _itmp367[] = {-1.3455469e01,-5.2365393e-04,
		  2.1309257e-01,5.1024701e-03,-2.5330791e-01,-5.2052397e-05,
		  2.9245827e-02,1.5974564e-04,-4.4269739e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[367][j] = _itmp367[j];
			}
		}
		{ static float _itmp368[] = {-1.3457305e01,-5.2595424e-04,
		  2.1214049e-01,5.0808676e-03,-2.5276288e-01,-5.1406751e-05,
		  2.9201243e-02,1.5953783e-04,-4.4217459e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[368][j] = _itmp368[j];
			}
		}
		{ static float _itmp369[] = {-1.3459147e01,-5.2699505e-04,
		  2.1118838e-01,5.0700386e-03,-2.5236568e-01,-5.1184801e-05,
		  2.9160999e-02,1.5935594e-04,-4.4170367e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[369][j] = _itmp369[j];
			}
		}
		{ static float _itmp370[] = {-1.3460976e01,-5.2605453e-04,
		  2.1019137e-01,5.0870380e-03,-2.5231907e-01,-5.1113886e-05,
		  2.9113211e-02,1.5917842e-04,-4.4125891e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[370][j] = _itmp370[j];
			}
		}
		{ static float _itmp371[] = {-1.3462813e01,-5.2376068e-04,
		  2.0920347e-01,5.1070182e-03,-2.5232932e-01,-5.0638311e-05,
		  2.9064225e-02,1.5904113e-04,-4.4093542e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[371][j] = _itmp371[j];
			}
		}
		{ static float _itmp372[] = {-1.3464632e01,-5.2468892e-04,
		  2.0824027e-01,5.0774552e-03,-2.5167766e-01,-5.0429298e-05,
		  2.9025577e-02,1.5881509e-04,-4.4033835e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[372][j] = _itmp372[j];
			}
		}
		{ static float _itmp373[] = {-1.3466458e01,-5.2301114e-04,
		  2.0725718e-01,5.0661573e-03,-2.5127259e-01,-5.0377919e-05,
		  2.8986817e-02,1.5863118e-04,-4.3984973e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[373][j] = _itmp373[j];
			}
		}
		{ static float _itmp374[] = {-1.3468272e01,-5.2184315e-04,
		  2.0627180e-01,5.0417604e-03,-2.5069001e-01,-5.0612744e-05,
		  2.8953008e-02,1.5842298e-04,-4.3927473e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[374][j] = _itmp374[j];
			}
		}
		{ static float _itmp375[] = {-1.3470092e01,-5.2075420e-04,
		  2.0530306e-01,5.0208061e-03,-2.5015959e-01,-5.0410046e-05,
		  2.8914483e-02,1.5822354e-04,-4.3874881e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[375][j] = _itmp375[j];
			}
		}
}
void HRFitData8(void)
{
	int j;
		{ static float _itmp376[] = {-1.3471908e01,-5.2073761e-04,
		  2.0435268e-01,5.0184177e-03,-2.4987309e-01,-5.0300125e-05,
		  2.8873259e-02,1.5804353e-04,-4.3828399e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[376][j] = _itmp376[j];
			}
		}
		{ static float _itmp377[] = {-1.3473718e01,-5.1884778e-04,
		  2.0337488e-01,5.0073978e-03,-2.4947239e-01,-5.0142218e-05,
		  2.8833093e-02,1.5785587e-04,-4.3779253e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[377][j] = _itmp377[j];
			}
		}
		{ static float _itmp378[] = {-1.3475526e01,-5.1867502e-04,
		  2.0242590e-01,4.9930606e-03,-2.4903008e-01,-4.9979852e-05,
		  2.8793992e-02,1.5766635e-04,-4.3729418e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[378][j] = _itmp378[j];
			}
		}
		{ static float _itmp379[] = {-1.3477336e01,-5.1798020e-04,
		  2.0147698e-01,4.9705892e-03,-2.4848221e-01,-4.9767252e-05,
		  2.8755860e-02,1.5746131e-04,-4.3675555e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[379][j] = _itmp379[j];
			}
		}
		{ static float _itmp380[] = {-1.3479135e01,-5.1879266e-04,
		  2.0054440e-01,4.9597467e-03,-2.4808612e-01,-4.9424652e-05,
		  2.8713973e-02,1.5727508e-04,-4.3627773e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[380][j] = _itmp380[j];
			}
		}
		{ static float _itmp381[] = {-1.3480955e01,-5.0645816e-04,
		  1.9946223e-01,5.0437427e-03,-2.4894129e-01,-4.9719700e-05,
		  2.8659787e-02,1.5719853e-04,-4.3612192e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[381][j] = _itmp381[j];
			}
		}
		{ static float _itmp382[] = {-1.3482750e01,-5.0782535e-04,
		  1.9854429e-01,5.0322996e-03,-2.4853832e-01,-4.9550417e-05,
		  2.8620133e-02,1.5701079e-04,-4.3563116e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[382][j] = _itmp382[j];
			}
		}
		{ static float _itmp383[] = {-1.3484550e01,-5.0496479e-04,
		  1.9758213e-01,5.0149821e-03,-2.4806277e-01,-4.9406801e-05,
		  2.8582441e-02,1.5681628e-04,-4.3511917e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[383][j] = _itmp383[j];
			}
		}
		{ static float _itmp384[] = {-1.3486340e01,-5.0413748e-04,
		  1.9663975e-01,5.0094528e-03,-2.4773665e-01,-4.9586164e-05,
		  2.8545359e-02,1.5663556e-04,-4.3463088e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[384][j] = _itmp384[j];
			}
		}
		{ static float _itmp385[] = {-1.3488131e01,-5.0580583e-04,
		  1.9574098e-01,4.9940459e-03,-2.4728586e-01,-4.9316877e-05,
		  2.8505942e-02,1.5644348e-04,-4.3413262e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[385][j] = _itmp385[j];
			}
		}
		{ static float _itmp386[] = {-1.3489913e01,-5.0525734e-04,
		  1.9480501e-01,4.9842638e-03,-2.4690475e-01,-4.9221489e-05,
		  2.8466556e-02,1.5625518e-04,-4.3363725e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[386][j] = _itmp386[j];
			}
		}
		{ static float _itmp387[] = {-1.3491702e01,-5.0535059e-04,
		  1.9390297e-01,4.9720565e-03,-2.4650504e-01,-4.9007154e-05,
		  2.8429180e-02,1.5608415e-04,-4.3319063e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[387][j] = _itmp387[j];
			}
		}
		{ static float _itmp388[] = {-1.3493484e01,-5.0564634e-04,
		  1.9299126e-01,4.9644280e-03,-2.4615569e-01,-4.8738988e-05,
		  2.8387567e-02,1.5589329e-04,-4.3270304e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[388][j] = _itmp388[j];
			}
		}
		{ static float _itmp389[] = {-1.3495264e01,-5.0470192e-04,
		  1.9206928e-01,4.9464586e-03,-2.4567330e-01,-4.8603750e-05,
		  2.8350485e-02,1.5570037e-04,-4.3219239e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[389][j] = _itmp389[j];
			}
		}
		{ static float _itmp390[] = {-1.3497040e01,-5.0395192e-04,
		  1.9115102e-01,4.9323053e-03,-2.4524111e-01,-4.8578215e-05,
		  2.8314052e-02,1.5551514e-04,-4.3169698e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[390][j] = _itmp390[j];
			}
		}
		{ static float _itmp391[] = {-1.3498812e01,-5.0287205e-04,
		  1.9023225e-01,4.9182768e-03,-2.4481200e-01,-4.8453687e-05,
		  2.8276579e-02,1.5532854e-04,-4.3120477e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[391][j] = _itmp391[j];
			}
		}
		{ static float _itmp392[] = {-1.3500591e01,-5.0118362e-04,
		  1.8932080e-01,4.9065398e-03,-2.4441838e-01,-4.8540009e-05,
		  2.8242085e-02,1.5514973e-04,-4.3072458e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[392][j] = _itmp392[j];
			}
		}
		{ static float _itmp393[] = {-1.3502354e01,-4.9850869e-04,
		  1.8837827e-01,4.9015228e-03,-2.4410313e-01,-4.8614202e-05,
		  2.8203795e-02,1.5496442e-04,-4.3023173e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[393][j] = _itmp393[j];
			}
		}
		{ static float _itmp394[] = {-1.3504123e01,-4.9585995e-04,
		  1.8745469e-01,4.8734988e-03,-2.4349482e-01,-4.8805134e-05,
		  2.8173974e-02,1.5476868e-04,-4.2968853e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[394][j] = _itmp394[j];
			}
		}
		{ static float _itmp395[] = {-1.3505879e01,-4.9636659e-04,
		  1.8656057e-01,4.8640827e-03,-2.4312454e-01,-4.8810489e-05,
		  2.8136833e-02,1.5458697e-04,-4.2920378e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[395][j] = _itmp395[j];
			}
		}
		{ static float _itmp396[] = {-1.3507652e01,-4.9601414e-04,
		  1.8568815e-01,4.8620077e-03,-2.4286601e-01,-4.8830294e-05,
		  2.8101277e-02,1.5443259e-04,-4.2879535e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[396][j] = _itmp396[j];
			}
		}
		{ static float _itmp397[] = {-1.3509407e01,-4.9635331e-04,
		  1.8480478e-01,4.8415638e-03,-2.4235551e-01,-4.8553691e-05,
		  2.8063804e-02,1.5423905e-04,-4.2828801e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[397][j] = _itmp397[j];
			}
		}
		{ static float _itmp398[] = {-1.3511166e01,-5.0095358e-04,
		  1.8398423e-01,4.8632743e-03,-2.4239895e-01,-4.8677954e-05,
		  2.8022015e-02,1.5408915e-04,-4.2790673e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[398][j] = _itmp398[j];
			}
		}
		{ static float _itmp399[] = {-1.3512916e01,-5.0014578e-04,
		  1.8309142e-01,4.8452988e-03,-2.4192156e-01,-4.8541115e-05,
		  2.7985662e-02,1.5389876e-04,-4.2740130e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[399][j] = _itmp399[j];
			}
		}
		{ static float _itmp400[] = {-1.3514669e01,-5.0193211e-04,
		  1.8224095e-01,4.8482865e-03,-2.4172197e-01,-4.8310252e-05,
		  2.7943892e-02,1.5372732e-04,-4.2696916e-03};
		for( j=0; j < 9; j++ )
		{
			HRF[400][j] = _itmp400[j];
			}
		}


#	ifdef DEBUG_FUN
	fputs( " <->HRFitData()\n", debug_fp );
#	endif
}
#ifdef _MSC_VER
#pragma warning( default : 4305 )/* enable warning that conditional expression*/
#endif
/*lint +e736 loss of precision 36 to 32 bits */

