/* This file is part of Cloudy and is copyright (C) 1978-2004 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*atmdat_2phot_shapefunction two photon emission function for hydrogen */
/*atmdat_2phot_rate hydrogen two photon emission, including induced emission */
#include "cddefines.h"
#include "taulines.h"
#include "interpolate.h"
#include "trace.h"
#include "rfield.h"
#include "iso.h"
#include "atmdat.h"
#define NCRS_H	9
#define NCRS_HE	51

/* A grid of energies for hydrogenic two photon, in steps of 1/16 E2Nu */
static double Hydro2NuEnergies[NCRS_H] = {  0., 0.0625, 0.125, 0.1875, 
	0.25, 0.3125, 0.375, 0.4375, 0.5};

/* Goldman and Drake only give values for 1, 20, 40, and up.  The other numbers 
 * (Z=2-19 and 21-30) are calculated from simple linear interpolation. */
/*>>refer	Hydro	2nu	Goldman,S.P. & Drake,G.W.F. 1981 Phys. Rev. A 24, 183.	*/
static double Hydro2NuShapeFunc[30][NCRS_H]={
	{0.0,2.03239,3.15792,3.84452,4.28435,4.56958,4.74855,4.84732,4.87892},
	{0.0,2.02777,3.15409,3.84145,4.28193,4.56766,4.74699,4.84597,4.87764},
	{0.0,2.02316,3.15026,3.83839,4.27950,4.56574,4.74542,4.84462,4.87636},
	{0.0,2.01854,3.14643,3.83532,4.27708,4.56382,4.74386,4.84327,4.87509},
	{0.0,2.01392,3.14260,3.83226,4.27466,4.56190,4.74230,4.84192,4.87381},
	{0.0,2.00931,3.13877,3.82919,4.27223,4.55997,4.74073,4.84057,4.87253},
	{0.0,2.00469,3.13494,3.82613,4.26981,4.55805,4.73917,4.83922,4.87125},
	{0.0,2.00007,3.13111,3.82306,4.26739,4.55613,4.73761,4.83787,4.86997},
	{0.0,1.99546,3.12728,3.82000,4.26496,4.55421,4.73604,4.83652,4.86870},
	{0.0,1.99084,3.12345,3.81693,4.26254,4.55229,4.73448,4.83517,4.86742},
	{0.0,1.98622,3.11962,3.81387,4.26012,4.55037,4.73292,4.83383,4.86614},
	{0.0,1.98160,3.11579,3.81080,4.25770,4.54845,4.73136,4.83248,4.86486},
	{0.0,1.97699,3.11196,3.80774,4.25527,4.54653,4.72979,4.83113,4.86359},
	{0.0,1.97237,3.10813,3.80467,4.25285,4.54461,4.72823,4.82978,4.86231},
	{0.0,1.96775,3.10430,3.80161,4.25043,4.54269,4.72667,4.82843,4.86103},
	{0.0,1.96314,3.10047,3.79854,4.24800,4.54076,4.72510,4.82708,4.85975},
	{0.0,1.95852,3.09664,3.79548,4.24558,4.53884,4.72354,4.82573,4.85847},
	{0.0,1.95390,3.09281,3.79241,4.24316,4.53692,4.72198,4.82438,4.85720},
	{0.0,1.94929,3.08898,3.78935,4.24073,4.53500,4.72041,4.82303,4.85592},
	{0.0,1.94467,3.08515,3.78628,4.23831,4.53308,4.71885,4.82168,4.85464},
	{0.0,1.93317,3.07480,3.77774,4.23144,4.52756,4.71432,4.81774,4.85090},
	{0.0,1.92167,3.06444,3.76919,4.22456,4.52205,4.70979,4.81381,4.84717},
	{0.0,1.91016,3.05409,3.76065,4.21769,4.51653,4.70525,4.80987,4.84343},
	{0.0,1.89866,3.04374,3.75210,4.21082,4.51101,4.70072,4.80594,4.83969},
	{0.0,1.88716,3.03338,3.74356,4.20394,4.50549,4.69619,4.80200,4.83596},
	{0.0,1.87566,3.02303,3.73501,4.19707,4.49998,4.69166,4.79807,4.83222},
	{0.0,1.86416,3.01268,3.72647,4.19020,4.49446,4.68713,4.79413,4.82848},
	{0.0,1.85265,3.00232,3.71792,4.18332,4.48894,4.68259,4.79020,4.82474},
	{0.0,1.84115,2.99197,3.70938,4.17645,4.48342,4.67806,4.78626,4.82101},
	{0.0,1.82965,2.98162,3.70083,4.16958,4.47791,4.67353,4.78233,4.81727}};

/* A grid of energies, in 1% of E2Nu steps */
static double He2NuEnergies[NCRS_HE] = {  0., 0.01, 0.02, 0.03, 0.04, 0.05, 0.06, 0.07, 0.08, 0.09, 0.1,
	0.11, 0.12, 0.13, 0.14, 0.15, 0.16, 0.17, 0.18, 0.19, 0.2,  0.21, 0.22, 0.23, 0.24, 0.25,
	0.26, 0.27, 0.28, 0.29, 0.3,  0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4,
	0.41, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5};

/* These numbers come directly from Walter Johnson.	*/
/*>>refer HeI	2nu	Johnson,W. 2002, private communication	*/
static double He2NuShapeFunc[29][NCRS_HE]={
	{0.0,0.02034,0.09681,0.21242,0.34804,0.49162,0.63613,0.77764,0.91399,1.0441 ,1.16747,
		1.28401,1.39382,1.49713,1.59422,1.68542,1.77106,1.85147,1.92697,1.99784,2.06438,
		2.12685,2.18549,2.24053,2.29218,2.34064,2.38607,2.42864,2.46851,2.50581,2.54066,
		2.5732 ,2.60351,2.6317 ,2.65786,2.68207,2.7044 ,2.72492,2.7437 ,2.76078,2.77622,
		2.79006,2.80235,2.8131 ,2.82237,2.83017,2.83652,2.84144,2.84494,2.84704,2.84774},
	{0.0,0.02988,0.12882,0.26547,0.41655,0.57004,0.71996,0.86344,0.99923,1.12691,1.24653,
		1.35836,1.46281,1.56031,1.65134,1.73632,1.81569,1.88984,1.95915,2.02395,2.08456,
		2.14126,2.19431,2.24396,2.29041,2.33388,2.37454,2.41255,2.44807,2.48124,2.51218,
		2.541  ,2.56781,2.59271,2.61578,2.63711,2.65675,2.67479,2.69127,2.70625,2.71978,
		2.7319 ,2.74265,2.75206,2.76016,2.76697,2.77252,2.77681,2.77987,2.7817 ,2.78231},
	{0.0,0.03916,0.15567,0.3059 ,0.46528,0.62293,0.77411,0.91688,1.05064,1.17543,1.29163,
		1.39971,1.50024,1.59375,1.68078,1.76183,1.83735,1.90777,1.97346,2.03479,2.09206,
		2.14558,2.19559,2.24234,2.28605,2.3269 ,2.36509,2.40076,2.43407,2.46515,2.49412,
		2.5211 ,2.54619,2.56947,2.59103,2.61095,2.6293 ,2.64614,2.66152,2.6755 ,2.68812,
		2.69942,2.70945,2.71822,2.72576,2.73211,2.73728,2.74128,2.74413,2.74584,2.74641},
	{0.0,0.04796,0.17833,0.33778,0.50199,0.66147,0.81255,0.95401,1.08572,1.20803,1.32148,
		1.4267 ,1.52433,1.61497,1.69919,1.77751,1.8504 ,1.91828,1.98157,2.04059,2.09568,
		2.14711,2.19515,2.24004,2.28197,2.32116,2.35777,2.39196,2.42388,2.45365,2.48139,
		2.50722,2.53123,2.55351,2.57414,2.5932 ,2.61074,2.62684,2.64155,2.65491,2.66698,
		2.67778,2.68735,2.69574,2.70295,2.70901,2.71395,2.71777,2.7205 ,2.72213,2.72267},
	{0.0,0.05615,0.19754,0.36344,0.53056,0.69076,0.84124,0.98133,1.11121,1.23145,1.34271,
		1.44572,1.54115,1.62964,1.71177,1.78808,1.85906,1.92512,1.98667,2.04405,2.09758,
		2.14754,2.1942 ,2.23777,2.27848,2.3165 ,2.35202,2.38518,2.41613,2.44499,2.47189,
		2.49693,2.5202 ,2.54179,2.56178,2.58024,2.59724,2.61284,2.62708,2.64003,2.65171,
		2.66217,2.67145,2.67956,2.68655,2.69242,2.6972 ,2.7009 ,2.70354,2.70512,2.70565},
	{0.0,0.0637 ,0.21393,0.38444,0.55335,0.7137 ,0.8634 ,1.00219,1.1305 ,1.24903,1.35853,
		1.45978,1.55349,1.64031,1.72085,1.79564,1.86516,1.92985,1.9901 ,2.04625,2.09862,
		2.14749,2.19312,2.23573,2.27552,2.31269,2.3474 ,2.37981,2.41006,2.43826,2.46454,
		2.489  ,2.51173,2.53282,2.55235,2.57038,2.58699,2.60222,2.61613,2.62877,2.64018,
		2.6504 ,2.65945,2.66738,2.6742 ,2.67993,2.6846 ,2.68821,2.69079,2.69233,2.69284},
	{0.0,0.07061,0.22794,0.4018 ,0.5718 ,0.732  ,0.88089,1.01851,1.14548,1.26258,1.37065,
		1.47049,1.56282,1.64833,1.72762,1.80121,1.86961,1.93324,1.99248,2.04769,2.09917,
		2.14721,2.19205,2.23391,2.27302,2.30954,2.34364,2.37548,2.40519,2.4329 ,2.45871,
		2.48273,2.50506,2.52577,2.54495,2.56266,2.57897,2.59393,2.60759,2.62   ,2.63121,
		2.64124,2.65013,2.65791,2.66461,2.67024,2.67483,2.67838,2.6809 ,2.68242,2.68292},
	{0.0,0.07687,0.23997,0.41628,0.58691,0.74682,0.89492,1.03151,1.15733,1.27324,1.38014,
		1.47882,1.57005,1.6545 ,1.73278,1.80543,1.87294,1.93572,1.99418,2.04864,2.09943,
		2.14681,2.19104,2.23233,2.27089,2.30691,2.34054,2.37194,2.40124,2.42856,2.45401,
		2.4777 ,2.49972,2.52014,2.53905,2.55652,2.57259,2.58734,2.60081,2.61305,2.6241 ,
		2.63399,2.64276,2.65043,2.65704,2.66259,2.66711,2.67061,2.6731 ,2.67459,2.67509},
	{0.0,0.08253,0.2503 ,0.42842,0.59941,0.75894,0.90631,1.042  ,1.16683,1.28176,1.38767,
		1.4854 ,1.57573,1.65932,1.73679,1.80868,1.87546,1.93758,1.9954 ,2.04927,2.09951,
		2.14637,2.19011,2.23095,2.26909,2.30471,2.33797,2.36902,2.398  ,2.42502,2.45019,
		2.47362,2.49539,2.51559,2.53429,2.55156,2.56746,2.58205,2.59537,2.60748,2.6184 ,
		2.62819,2.63686,2.64445,2.65098,2.65647,2.66094,2.6644 ,2.66686,2.66834,2.66883},
	{0.0,0.08761,0.25918,0.43865,0.6098 ,0.76892,0.91562,1.05051,1.17451,1.2886 ,1.39369,
		1.49064,1.58022,1.66311,1.73992,1.81118,1.87738,1.93895,1.99626,2.04966,2.09945,
		2.1459 ,2.18925,2.22973,2.26753,2.30284,2.33581,2.36658,2.3953 ,2.42208,2.44703,
		2.47026,2.49184,2.51186,2.5304 ,2.54752,2.56328,2.57774,2.59095,2.60295,2.61378,
		2.62348,2.63207,2.6396 ,2.64607,2.65151,2.65594,2.65938,2.66182,2.66328,2.66377},
	{0.0,0.09214,0.2668 ,0.44726,0.61845,0.77717,0.92327,1.05746,1.18075,1.29412,1.39853,
		1.49482,1.58379,1.66609,1.74236,1.81311,1.87884,1.93997,1.99687,2.04988,2.09931,
		2.14543,2.18847,2.22866,2.26619,2.30125,2.33398,2.36454,2.39306,2.41965,2.44443,
		2.46749,2.48892,2.5088 ,2.52721,2.54421,2.55987,2.57423,2.58735,2.59926,2.61002,
		2.61965,2.62819,2.63566,2.64209,2.6475 ,2.6519 ,2.65531,2.65774,2.65919,2.65968},
	{0.0,0.09615,0.27332,0.45452,0.62566,0.78399,0.92954,1.06313,1.1858 ,1.29858,1.40241,
		1.49816,1.58661,1.66844,1.74426,1.8146 ,1.87994,1.94071,1.99728,2.04998,2.09912,
		2.14497,2.18777,2.22773,2.26504,2.2999 ,2.33245,2.36284,2.39119,2.41764,2.44228,
		2.46521,2.48653,2.5063 ,2.52461,2.54152,2.55709,2.57138,2.58443,2.59628,2.60698,
		2.61656,2.62506,2.63249,2.63889,2.64427,2.64864,2.65204,2.65445,2.6559 ,2.65638},
	{0.0,0.09967,0.27888,0.4606 ,0.63164,0.7896 ,0.93467,1.06774,1.18989,1.30216,1.4055 ,
		1.50081,1.58883,1.67027,1.74572,1.81572,1.88075,1.94123,1.99753,2.04998,2.0989 ,
		2.14453,2.18713,2.22691,2.26406,2.29876,2.33116,2.36142,2.38965,2.41598,2.44052,
		2.46335,2.48458,2.50427,2.52251,2.53935,2.55486,2.56908,2.58208,2.59389,2.60454,
		2.61409,2.62255,2.62995,2.63632,2.64168,2.64604,2.64942,2.65183,2.65327,2.65375},
	{0.0,0.10275,0.28358,0.46567,0.63656,0.79418,0.93882,1.07145,1.19315,1.30499,1.40793,
		1.50286,1.59053,1.67165,1.7468 ,1.81653,1.88131,1.94155,1.99764,2.0499 ,2.09863,
		2.1441 ,2.18654,2.22618,2.2632 ,2.29778,2.33007,2.36023,2.38837,2.41461,2.43907,
		2.46183,2.48299,2.50262,2.5208 ,2.53759,2.55305,2.56724,2.5802 ,2.59197,2.6026 ,
		2.61211,2.62055,2.62793,2.63429,2.63963,2.64398,2.64735,2.64975,2.65119,2.65166},
	{0.0,0.1054 ,0.28752,0.46984,0.64058,0.79788,0.94214,1.07438,1.19571,1.30719,1.4098 ,
		1.50442,1.59181,1.67266,1.74758,1.81709,1.88167,1.94173,1.99765,2.04976,2.09835,
		2.14369,2.18602,2.22555,2.26247,2.29696,2.32918,2.35926,2.38733,2.41352,2.43792,
		2.46063,2.48174,2.50133,2.51947,2.53623,2.55166,2.56581,2.57874,2.5905 ,2.6011 ,
		2.6106 ,2.61902,2.62639,2.63273,2.63807,2.64241,2.64577,2.64817,2.6496 ,2.65008},
	{0.0,0.10766,0.29077,0.47323,0.6438 ,0.8008 ,0.94474,1.07665,1.19766,1.30884,1.41118,
		1.50555,1.59271,1.67336,1.74809,1.81743,1.88185,1.94177,1.99757,2.04956,2.09805,
		2.1433 ,2.18554,2.225  ,2.26185,2.29628,2.32844,2.35847,2.3865 ,2.41264,2.43701,
		2.45969,2.48077,2.50034,2.51845,2.53519,2.5506 ,2.56474,2.57766,2.58939,2.59999,
		2.60948,2.61789,2.62525,2.63159,2.63692,2.64126,2.64462,2.64701,2.64844,2.64892},
	{0.0,0.10955,0.29339,0.47592,0.6463 ,0.80304,0.9467 ,1.07833,1.19907,1.31001,1.41213,
		1.5063 ,1.59328,1.67377,1.74835,1.81756,1.88187,1.94169,1.9974 ,2.04931,2.09773,
		2.14292,2.18511,2.22451,2.26133,2.29572,2.32784,2.35784,2.38585,2.41197,2.43631,
		2.45898,2.48005,2.4996 ,2.51771,2.53443,2.54984,2.56397,2.57688,2.58862,2.59921,
		2.6087 ,2.61711,2.62447,2.6308 ,2.63613,2.64047,2.64383,2.64622,2.64765,2.64813},
	{0.0,0.1111 ,0.29546,0.47797,0.64817,0.80468,0.94808,1.07948,1.20001,1.31076,1.4127 ,
		1.50671,1.59356,1.67393,1.74841,1.81753,1.88176,1.94151,1.99716,2.04902,2.0974 ,
		2.14255,2.18471,2.22409,2.26088,2.29526,2.32737,2.35736,2.38536,2.41147,2.43582,
		2.45848,2.47955,2.4991 ,2.51721,2.53394,2.54934,2.56348,2.5764 ,2.58814,2.59873,
		2.60822,2.61664,2.624  ,2.63034,2.63567,2.64001,2.64337,2.64576,2.6472 ,2.64767},
	{0.0,0.11234,0.29702,0.47946,0.64947,0.80576,0.94896,1.08016,1.20052,1.31111,1.41293,
		1.50683,1.59358,1.67386,1.74827,1.81734,1.88152,1.94124,1.99685,2.0487 ,2.09705,
		2.14219,2.18435,2.22372,2.26052,2.29489,2.32701,2.35701,2.38501,2.41114,2.43549,
		2.45817,2.47925,2.49881,2.51694,2.53368,2.5491 ,2.56325,2.57617,2.58792,2.59853,
		2.60803,2.61645,2.62382,2.63017,2.6355 ,2.63985,2.64321,2.64561,2.64704,2.64752},
	{0.0,0.11328,0.29812,0.48043,0.65025,0.80635,0.94937,1.08042,1.20064,1.31112,1.41284,
		1.50666,1.59336,1.67359,1.74797,1.81701,1.88117,1.94088,1.99649,2.04833,2.0967 ,
		2.14185,2.18401,2.22341,2.26022,2.29462,2.32675,2.35678,2.3848 ,2.41095,2.43533,
		2.45803,2.47913,2.49872,2.51687,2.53363,2.54907,2.56324,2.57619,2.58795,2.59858,
		2.60809,2.61653,2.62391,2.63027,2.63561,2.63996,2.64333,2.64574,2.64717,2.64765},
	{0.0,0.11396,0.29878,0.48093,0.65056,0.80649,0.94936,1.08028,1.2004 ,1.31081,1.41247,
		1.50625,1.59291,1.67313,1.74749,1.81653,1.88071,1.94043,1.99606,2.04793,2.09632,
		2.1415 ,2.1837 ,2.22312,2.25997,2.2944 ,2.32658,2.35664,2.3847 ,2.41089,2.4353 ,
		2.45803,2.47917,2.4988 ,2.51697,2.53376,2.54923,2.56343,2.5764 ,2.58819,2.59884,
		2.60837,2.61683,2.62423,2.6306 ,2.63595,2.64031,2.64369,2.6461 ,2.64754,2.64802},
	{0.0,0.11439,0.29907,0.481  ,0.65045,0.80621,0.94896,1.07979,1.19985,1.31021,1.41184,
		1.50561,1.59227,1.6725 ,1.74688,1.81595,1.88016,1.93992,1.99559,2.0475 ,2.09594,
		2.14117,2.18342,2.2229 ,2.25979,2.29428,2.32651,2.35661,2.38473,2.41096,2.43542,
		2.4582 ,2.47939,2.49905,2.51727,2.5341 ,2.5496 ,2.56383,2.57684,2.58866,2.59933,
		2.60889,2.61737,2.62479,2.63117,2.63654,2.64092,2.64431,2.64672,2.64816,2.64864},
	{0.0,0.1146 ,0.29899,0.48067,0.64993,0.80556,0.9482 ,1.07897,1.19898,1.30933,1.41096,
		1.50474,1.59143,1.67169,1.74612,1.81524,1.8795 ,1.93932,1.99505,2.04703,2.09554,
		2.14083,2.18315,2.22269,2.25965,2.2942 ,2.32649,2.35667,2.38484,2.41114,2.43565,
		2.45849,2.47972,2.49944,2.5177 ,2.53458,2.55013,2.5644 ,2.57744,2.5893 ,2.6	   ,
		2.60959,2.61809,2.62554,2.63194,2.63733,2.64172,2.64512,2.64754,2.64899,2.64947},
	{0.0,0.11461,0.29859,0.47999,0.64905,0.80455,0.94712,1.07784,1.19785,1.3082 ,1.40986,
		1.50368,1.59042,1.67074,1.74524,1.81443,1.87877,1.93866,1.99448,2.04654,2.09512,
		2.1405 ,2.1829 ,2.22252,2.25957,2.2942 ,2.32656,2.35681,2.38506,2.41142,2.43601,
		2.4589 ,2.4802 ,2.49998,2.5183 ,2.53523,2.55083,2.56514,2.57823,2.59012,2.60086,
		2.61049,2.61902,2.62649,2.63292,2.63833,2.64273,2.64614,2.64857,2.65003,2.65051},
	{0.0,0.11442,0.2979 ,0.47897,0.64783,0.80322,0.94573,1.07643,1.19645,1.30683,1.40854,
		1.50242,1.58924,1.66965,1.74423,1.81351,1.87795,1.93794,1.99385,2.04601,2.09469,
		2.14017,2.18266,2.22238,2.25952,2.29424,2.32669,2.35703,2.38536,2.4118 ,2.43646,
		2.45943,2.4808 ,2.50064,2.51903,2.53602,2.55167,2.56604,2.57918,2.59112,2.6019 ,
		2.61156,2.62012,2.62762,2.63408,2.63951,2.64393,2.64735,2.64979,2.65125,2.65174},
	{0.0,0.11408,0.29693,0.47764,0.6463 ,0.80158,0.94405,1.07475,1.1948 ,1.30523,1.40702,
		1.50099,1.5879 ,1.66841,1.7431 ,1.81249,1.87704,1.93715,1.99317,2.04544,2.09424,
		2.13983,2.18243,2.22226,2.2595 ,2.29433,2.32688,2.35731,2.38573,2.41227,2.43701,
		2.46007,2.48151,2.50143,2.51988,2.53694,2.55266,2.56708,2.58027,2.59226,2.60309,
		2.61279,2.62139,2.62892,2.63541,2.64086,2.6453 ,2.64874,2.65119,2.65266,2.65315},
	{0.0,0.11358,0.29571,0.47603,0.64448,0.79966,0.94209,1.07281,1.19291,1.30343,1.4053 ,
		1.49938,1.58641,1.66704,1.74186,1.81138,1.87605,1.93629,1.99245,2.04485,2.09378,
		2.13949,2.18222,2.22217,2.25952,2.29446,2.32712,2.35766,2.38618,2.41282,2.43766,
		2.4608 ,2.48233,2.50233,2.52086,2.53799,2.55378,2.56827,2.58151,2.59356,2.60443,
		2.61418,2.62282,2.63039,2.6369 ,2.64238,2.64685,2.6503 ,2.65277,2.65424,2.65473},
	{0.0,0.11294,0.29426,0.47415,0.64238,0.79747,0.93989,1.07064,1.19081,1.30142,1.40341,
		1.49761,1.58477,1.66554,1.7405 ,1.81017,1.87499,1.93538,1.99168,2.04423,2.0933 ,
		2.13915,2.18201,2.22209,2.25958,2.29464,2.32742,2.35807,2.38671,2.41345,2.43839,
		2.46163,2.48326,2.50334,2.52196,2.53917,2.55502,2.56958,2.58289,2.595  ,2.60593,
		2.61572,2.62441,2.63201,2.63856,2.64407,2.64855,2.65203,2.6545 ,2.65599,2.65648},
	{0.0,0.11219,0.2926 ,0.47203,0.64004,0.79503,0.93744,1.06825,1.1885 ,1.29922,1.40134,
		1.49568,1.58299,1.66392,1.73904,1.80887,1.87385,1.9344 ,1.99086,2.04356,2.09279,
		2.13879,2.1818 ,2.22202,2.25965,2.29484,2.32776,2.35853,2.38729,2.41415,2.4392 ,
		2.46255,2.48428,2.50446,2.52316,2.54045,2.55639,2.57102,2.5844 ,2.59656,2.60755,
		2.6174 ,2.62613,2.63378,2.64036,2.6459 ,2.65041,2.6539 ,2.65639,2.65788,2.65838}};

/*atmdat_2phot_shapefunction two photon emission function */
double atmdat_2phot_shapefunction( double EbyE2nu, long ipISO, long nelem )
{
	double result,y;
	static double y2[MAX2(NCRS_HE,NCRS_H)];

#if	0
#define	NCRS	21
	/* this is the photon energy in Rydbergs, energy Table 1 */
	/* >>refer	HI	2nu	Spitzer, L., & Greenstein, J., 1951, ApJ, 114, 407 */
	static double ener[NCRS]={
	  0.0,    0.05,  0.10,  0.15,  0.20,  0.25,
	  0.30,	  0.35,  0.40,  0.45,  0.50,  0.55,
	  0.60,   0.65,  0.70,  0.75,  0.80,  0.85,
	  0.90,   0.95,  1.00};

	  /* probability \psi(y), from col 3 of Table 1 of SG 51 
	   * but normalized so that the integral from 0 to 1 equals 1	*/
	static double psi[NCRS]={
		0.000,	0.458,	0.738,	0.923,	1.051,	1.142,	1.206,
		1.250,	1.280,	1.297,	1.302,	1.297,	1.280,	1.250,	
		1.206,	1.142,	1.051,	0.923,	0.738,	0.458,	0.000};
#undef	NCRS
#endif
	
#	ifdef DEBUG_FUN
	fputs( "<+>atmdat_2phot_shapefunction()\n", debug_fp );
#	endif

	/* must reset spline coefficients every time...otherwise every one but
	 * the first is wrong!	*/
	
	if( ipISO == ipH_LIKE )
	{
		/*  set up coefficients for spline */
		spline(Hydro2NuEnergies,Hydro2NuShapeFunc[nelem],NCRS_H,2e31,2e31,y2);

		/* EbyE2nu is analogous to the "y" in SG51	*/
		if( EbyE2nu < 0. || EbyE2nu >= 1.0 )
		{
			result = 0.;
		}
		else
		{
			/* H-like table only goes up to 0.5, 
			 * but other half is symmetric.	*/
			if( EbyE2nu > 0.5 )
				EbyE2nu = 1. - EbyE2nu;

			splint(Hydro2NuEnergies,Hydro2NuShapeFunc[nelem],y2,NCRS_H,EbyE2nu,&y);
			result = y;
		}
	}
	else
	{
		ASSERT( ipISO == ipHE_LIKE );

		/*  set up coefficients for spline */
		spline(He2NuEnergies,He2NuShapeFunc[nelem-1],NCRS_HE,2e31,2e31,y2);

		/* EbyE2nu is analogous to the "y" in SG51	*/
		if( EbyE2nu < 0. || EbyE2nu >= 1.0 )
		{
			result = 0.;
		}
		else
		{
			/* Helike table only goes up to 0.5, 
			 * but other half is symmetric.	*/
			if( EbyE2nu > 0.5 )
				EbyE2nu = 1. - EbyE2nu;

			splint(He2NuEnergies,He2NuShapeFunc[nelem-1],y2,NCRS_HE,EbyE2nu,&y);
			/* divide by two here because Johnson gave us numbers that integrate up
			 * to two on the range [0,1].	*/
			result = y/2.;
		}
	}

	ASSERT( result > 0. );

#	ifdef DEBUG_FUN
	fputs( " <->atmdat_2phot_shapefunction()\n", debug_fp );
#	endif

	return result;
}

/*atmdat_2phot_rate evaluate rate of induced hydrogen two photon emission */
/* A Two photon constant is not needed in this routine at all!  Everything here depends 
 * upon transition probability per energy bin, which is exactly what As2nu gives	
 * Also, everything in this routine is INDUCED except SponRate, which is only used for sanity
 * checking (redundant) and optional output.  	*/
void atmdat_2phot_rate(
	/* the iso sequence, must be either 0 (hydrogen) or 1 (he-like) */
	long ipISO  , 
	/* atomic number on c scale */
	long nelem	)
{
	long int i, 
	  limit;
	double SponRate;

	double s1, s2, t1, t2, t3, p;

#	ifdef DEBUG_FUN
	fputs( "<+>atmdat_2phot_rate()\n", debug_fp );
#	endif

	/* ipISO must be either 0 or 1 */
	/* This routine should be reviewed VERY carefully if other ISO's are added
	 * since logic to identify 2s-1s levels is important.	*/
	ASSERT( ipISO==ipH_LIKE || ipISO==ipHE_LIKE );

	/* initialize rates to zero now, we bail if fluorescense is turned off */

	s1 = s2 = 0;

	/* return if induced processes are turned off */
	if( !rfield.lgInducProcess )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->atmdat_2phot_rate()\n", debug_fp );
#		endif
		return;
	}

	SponRate = 0.;
	limit = iso.ipHalfTwoPhoE[ipISO][nelem];
	for( i=0; i <= limit ; i++ )
	{
		
		ASSERT( iso.ipSym2nu[ipISO][nelem][i] <= iso.ipTwoPhoE[ipISO][nelem] );
		/* the integral over the product of the two occupation numbers and the
		 * effecive spontaneous A.  As2nu already has width of cell included */

		/* this is the induced downward rate, DOES NOT INCLUDE SPONTANEOUS */
		t1 = iso.As2nu[ipISO][nelem][i];
		t2 = rfield.SummedOcc[i];
		t3 = rfield.SummedOcc[iso.ipSym2nu[ipISO][nelem][i]-1];
		p = t1*t2*t3;
		s1 += t1*(t2+t3)+p;

		s2 += p; 

		SponRate += t1;
	}
	/* above was only half the integral - other half is symmetric */
	s1 *= 2;

	iso.TwoNu_induc_dn[ipISO][nelem] = s1;
	/* absorption should almost certainly be multiplied by 1/2 here, to account for 
	 * the fact that this absorption can only happen if the two photons are oppositely
	 * polarized, so that the sum of their ang mom is 0, rather than 2.	Also, why do we
	 * not subtract one from each occupation number for absorption, as we add one to 
	 * each for emission?  With these two things taken together, should we have 
	 * s2 = t1*(t2-1)*(t3-1) = t1*t2*t3 + t1 - t1*t2- t1*t3 instead of s2 = 2*t1*t2*t3?	*/
	/*TODO	2	How should these affect locally emitted continuum calculated in rtdiffuse?
	 * Just subtract these rates from the emitted ones?	*/
	iso.TwoNu_induc_up[ipISO][nelem] = s2;
	SponRate *= 2.;

	/* remember maximum value of induced downward rate on per species basis	*/
	if (s1 > iso.TwoNu_induc_dn_max[ipISO][nelem])
		iso.TwoNu_induc_dn_max[ipISO][nelem] = s1;

#	if 0
/****************************************************************************/
/********* the original treatment - can probably be discarded.	*************/

#	ifdef DEBUG_FUN
	fputs( "<+>atmdat_2phot_rate()\n", debug_fp );
#	endif

	/* ipISO must be either 0 or 1 */
	/* This routine should be reviewed VERY carefully if other ISO's are added
	 * since logic to identify 2s-1s levels is important.	*/
	ASSERT( ipISO==ipH_LIKE || ipISO==ipHE_LIKE );

	/* initialize rates to zero now, we bail if fluorescense is turned off */

	iso.TwoNu_induc_dn[ipISO][nelem] = 0.;
	iso.TwoNu_induc_up[ipISO][nelem] = 0.;

	/* return if induced processes are turned off */
	if( !rfield.lgInducProcess )
	{ 
		iso.TwoNu_induc_dn[ipISO][nelem] = 0.;
		iso.TwoNu_induc_up[ipISO][nelem] = 0.;
#		ifdef DEBUG_FUN
		fputs( " <->atmdat_2phot_rate()\n", debug_fp );
#		endif
		return;
	}

	SponRate = 0.;
	for( i=0; i <= iso.ipHalfTwoPhoE[ipISO][nelem] ; i++ )
	{
		double rate1;
		
		ASSERT( iso.ipSym2nu[ipISO][nelem][i] <= iso.ipTwoPhoE[ipISO][nelem] );
		/* the integral over the product of the two occupation numbers and the
		 * effecive spontaneous A.  As2nu already has width of cell included */

		/* this is the induced downward rate, DOES NOT INCLUDE SPONTANEOUS */
		rate1 = iso.As2nu[ipISO][nelem][i]*(rfield.SummedOcc[i] 
			+rfield.SummedOcc[iso.ipSym2nu[ipISO][nelem][i]-1]
			+rfield.SummedOcc[iso.ipSym2nu[ipISO][nelem][i]-1]*rfield.SummedOcc[i]);
		iso.TwoNu_induc_dn[ipISO][nelem] += rate1; 
		/* this is the upward rate */
		rate1 = iso.As2nu[ipISO][nelem][i]*rfield.SummedOcc[i]*
			rfield.SummedOcc[iso.ipSym2nu[ipISO][nelem][i]-1];
		iso.TwoNu_induc_up[ipISO][nelem] += rate1;

		SponRate += iso.As2nu[ipISO][nelem][i];
	}
	/* above was only half the integral - other half is symmetric */
	iso.TwoNu_induc_dn[ipISO][nelem] *= 2.;
	iso.TwoNu_induc_up[ipISO][nelem] *= 2.;
	SponRate *= 2.;

	/* remember maximum value of induced downward rate on per species basis	*/
	iso.TwoNu_induc_dn_max[ipISO][nelem] =  
		MAX2( iso.TwoNu_induc_dn_max[ipISO][nelem] , iso.TwoNu_induc_dn[ipISO][nelem] );

#	endif

	/* >>chng 01 jan 26, from 8.23 to 8.226 */
	/* The 1+ipISO is not rigorous, it just works. Must most likely change if other 
	 * ISO's are added.	*/
	ASSERT( fabs(SponRate-EmisLines[ipISO][nelem][1+ipISO][0].Aul) / SponRate < 0.05 );

	if( trace.lgTrace && trace.lgBug2nu )
	{
		limit = iso.ipTwoPhoE[ipISO][nelem];
		for( i=1; i < limit; i += 10 )
		{
			fprintf( ioQQQ, "%12.4e%12.4e", rfield.anu[i], 
			  rfield.OccNumbIncidCont[i] );
		}
		fprintf( ioQQQ, "\n" );

		/* sum two-nu A's to check OK */
		SponRate = 0.;
		for( i=0; i < limit; i++ )
		{
			SponRate += iso.As2nu[ipISO][nelem][i];
		}

		fprintf( ioQQQ, "       TWOPHT finds induced two photon rate =%12.3e spontaneous A=%12.3e\n", 
		  iso.TwoNu_induc_dn[ipISO][nelem], SponRate );
	}

#	ifdef DEBUG_FUN
	fputs( " <->atmdat_2phot_rate()\n", debug_fp );
#	endif
	return;
}

