/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*RTMake do escape and destruction probs for all lines in code.  
 * Called with FALSE arg by ionize, to only redo destruction probabilites,
 * and with TRUE by cloudy to do both escape and destruction */
#include "cddefines.h"
#include "physconst.h"
#include "taulines.h"
#include "rtescprob.h"
#include "atomfeii.h"
#include "strk.h"
#include "dense.h"
#include "doppvel.h"
#include "fe2ovr.h"
#include "atom_oi.h"
#include "rfield.h"
#include "iso.h"
#include "wind.h"
#include "h2.h"
#include "opacity.h"
#include "trace.h"
#include "lines_service.h"
#include "ionrange.h"
#include "converge.h"
#include "twophoton.h"
#include "hydrogenic.h"
#include "rt.h"
/*HydroPesc evaluate escape and destruction probabilities for hydrogen lines,
 * second parameter is flag saying whether to also do escape prob (TRUE) or
 * just the deEXTERN struction probabilites (FALSE) */
static void HydroPesc(
	/* nelem is on C scale, 0 for H, 1 for He, etc */
	long int nelem ,
	/* flag saying whether we should do escape prob  and reevaluate A and opac (TRUE) or
	 * only dest probs (FALSE).  This is called with TRUE one time per zone,
	 * when optical depths are updated.  It is called FALSE many times, within ionzie,
	 * each time the opacity changes */
	int lgDoEsc );

void RTMake(
	/* this is TRUE if we want to do both escape and destruction probs,
	 * and FALSE if only destruction probabilities are needed */
	int lgDoEsc ,
	/* flag saying whether to update fine opacities */
	int lgUpdateFineOpac )
{
	long int i,
		nelem;
	long ipHi , ipLo;
	double factor;

#	ifdef DEBUG_FUN
	fputs( "<+>RTMake()\n", debug_fp );
#	endif

	rfield.lgFine_opac_update = lgUpdateFineOpac;
	if( rfield.lgFine_opac_update )
	{
		/*fprintf(ioQQQ," fine opacity update zone %li\n", nzone );*/
		memset(rfield.fine_opac , 0 , (unsigned long)rfield.nfine*sizeof(float) );
	}

	/*this is loop over h-like species - this is special case since 2s-2p are
	 * resolved but degenerate so must bring balmer line optical depths together */
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* note that nelem scale is totally on c not physical scale, so 0 is h */
		/* evaluate hydrogenic balance if ionization reaches this high */

		if( (IonRange.IonHigh[nelem] == nelem + 1)  )
		{
			/* evaluate branch As, escape probabilities, and destruction rates */
			HydroPesc(nelem , lgDoEsc);
		}
	}

	/* now main sets of lines */
	/* loop over he-like species */
	for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
	{
		/* do not consider elements that have been turned off */
		if( dense.lgElmtOn[nelem] )
		{
			if( dense.xIonDense[nelem][nelem] > 1e-30 )
			{
				factor = dense.xIonDense[nelem][nelem];
			}
			else
			{
				/* case where almost no parent ion - this will make
					* very large line opacity, so background dest small */
				factor = 1.;
			}
			for( ipHi=1; ipHi < iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				for( ipLo=0; ipLo < ipHi; ipLo++ )
				{
					/* >>chng 01 aug 18, do not work on fake he-like lines (majority) */
					/* >>chng 02 feb 10, test had been on A, change to ipCont */
					if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ipCont < 1 ) 
						continue;

					/* must temporarily make ipLnPopOpc physical */
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc *= 
						(float)factor;

					/* generate escape prob, pumping rate, destruction prob, 
						* inward outward fracs 
					fprintf(ioQQQ,"%li %li %li %li\n", nelem, ipHi, ipLo,EmisLines[ipHE_LIKE]);
					fflush(ioQQQ); */

					RTMakeLine(&EmisLines[ipHE_LIKE][nelem][ipHi][ipLo] , lgDoEsc );
					/* >>chng 03 feb 02, moved into rtstat not needed here too */
					/*EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Pelec_esc = 
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Pelec_esc*0.25f + 0.75f*eesc ;*/

					/* go back to original units so that final correction ok */
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc /= 
						(float)factor;
				}
			}
			/* now update two photon induced rates */
			twopht(ipHE_LIKE , nelem);

			/* >>chng 02 nov 18, add this cap to reduce outward Lya */
			if( dense.lgElmtOn[nelem] )
			{
				/* don't let too much Lya escape outward since so important */
				EmisLines[ipHE_LIKE][nelem][ipHe2p1P][ipHe1s1S].Pesc *= 
					opac.ExpmTau[EmisLines[ipHE_LIKE][nelem][ipHe2p1P][ipHe1s1S].ipCont-1];
			}

		}
	}
	{
		/* following should be set true to print ots contributors for he-like lines*/
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && iteration==2 )
		{
			/* this is 4 1F to 3 1D */
			/*fprintf(ioQQQ,"helike otsss aver Pdest\t%g\n",
				EmisLines[ipHE_LIKE][ipHELIUM][19][11].Pdest);*/
			DumpLine(&EmisLines[ipHE_LIKE][ipHELIUM][19][11] );
		}
	}
	/*fprintf(ioQQQ,"new opc, dst, redis\t%.2e\t%.2e\t%i\n",
	EmisLines[ipHE_LIKE][1][ipHe2p1P][0].opacity, EmisLines[ipHE_LIKE][1][ipHe2p1P][0].Pdest, EmisLines[ipHE_LIKE][1][ipHe2p1P][0].iRedisFun );*/	
	/* static model - level 1 lines */
	for( i=1; i <= nLevel1; i++ )
	{
		RTMakeLine(&TauLines[i] , lgDoEsc );
	}
	/* co carbon monoxide */
	for( i=0; i < nCORotate; i++ )
	{
		RTMakeLine(&C12O16Rotate[i] , lgDoEsc );
		RTMakeLine(&C13O16Rotate[i] , lgDoEsc );
	}
	for( i=0; i < nHFLines; i++ )
	{
		RTMakeLine(&HFLines[i] , lgDoEsc );
	}
	for( i=0; i < nUTA; i++ )
	{
		/* these are not defined in cooling routines */
		UTALines[i].PopOpc = dense.xIonDense[UTALines[i].nelem-1][UTALines[i].IonStg-1];
		UTALines[i].PopLo = dense.xIonDense[UTALines[i].nelem-1][UTALines[i].IonStg-1];
		UTALines[i].PopHi = 0.;
		RTMakeLine(&UTALines[i] , lgDoEsc );
	}

	/* >>chng 01 aug 11, the level 2 lines were only updated when lgDoEsc was true,
		* this meant that dest probs were not updated but once, causing very high-Z models
		* to develop numerical oscialltions.  removed if, now evaluate level 2 always */
	/* only update their dest/esc probs one time per zone */
	/* lgLevel2_OTS_Imp set true in dimacool if ots rates were significant */
	/* level 2 heavy element lines in cooling with only g-bar,*/
	if( lgDoEsc + conv.lgLevel2_OTS_Imp )
	{
		for( i=0; i < nWindLine; i++ )
		{
			/* >>chng 02 sug 11, do not include he-like in sum */
			if( TauLine2[i].IonStg <= TauLine2[i].nelem-1 )
			{
				/*float dest , eesc;
				dest = TauLine2[i].Pdest;
				eesc = TauLine2[i].Pelec_esc;*/
				RTMakeLine(&TauLine2[i] , lgDoEsc );
				/* >>chng 99 sep 17, following to prevent oscillations
					* where level 2 lines overlap He+ ionizing continuum */
				/* dest probs for these level 2 lines should not matter
					* since they should be weak ots sources.  there can be
					* problems with the zoning when they high H or He edges,
					* so use average */
				/* >>chng 03 feb 02, both are in RTMakeLine not needed here */
				/*TauLine2[i].Pdest = TauLine2[i].Pdest*0.25f + 0.75f*dest ;
				TauLine2[i].Pelec_esc = TauLine2[i].Pelec_esc*0.25f + 0.75f*eesc ;*/
			}
		}
	}

	/* the large H2 molecule */
	H2_RTMake( lgDoEsc );

	/* The large model FeII atom
	 * do not call if Netzer model used, or it Fe(2) is zero
	 * exception is when code is searching for first soln */
	if( FeII.lgFeIION )
	{
		FeIIRTMake( lgDoEsc );
	}

#	ifdef DEBUG_FUN
	fputs( " <->RTMake()\n", debug_fp );
#	endif
	return;
}


static void HydroPesc(
	/* nelem is on C scale, 0 for H, 1 for He, etc */
	long int nelem ,
	/* flag saying whether we should do escape prob  and reevaluate A and opac (TRUE) or
	 * only dest probs (FALSE).  This is called with TRUE one time per zone,
	 * when optical depths are updated.  It is called FALSE many times, within ionzie,
	 * each time the opacity changes */
	int lgDoEsc )
{
	static int lgTOnSave;
	long int ipHi, 
	  ipLo, 
	  limit;
	double abundan=0., 
	  coloi, 
	  factor, 
	  tout,
	  z4;/* physical scale z to the 4th power, used often below */
	float dest=0.f, 
	  esin;
	/* this will be used to save old vals of destruction prob in case
	 * we need to use mean of old and new, as is necessary when ots oscillations
	 * take place */
	float DespSave;
	float eEscSave;

#	ifdef DEBUG_FUN
	fputs( "<+>HydroPesc()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );

	/* will need this for some scaling laws - physical Z to the 4th power*/
	z4 = POW2(nelem+1.);
	z4 *= z4;

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
			"          HydroPesc nelem=%ld called\n", 
		  nelem );
	}

	/*always update for hydrogen */
	if( nelem == ipHYDROGEN )
	{
		/* find Stark escape probabilities */
		strk();
	}

	if( lgDoEsc )
	{
		/* hydrogen lines opacity is function of A's
		 * above 15 just use actual A's for high density limit
		 * only change opacities for levels between 4 and 15 */
		/* NB! must never set opacity for 2s-1s two-photon transition here, since
		 * the "line" can become optically thick if treated as a line.  It is a 
		 * continuum and the opacity is not related to the A by the usual expression */
		limit = MIN2(16,iso.numLevels[ipH_LIKE][nelem]);
		/* do Paschen and higher lines first, 
		 * do balmer below since must separate 2s and 2p */
		for( ipHi=4; ipHi < limit; ipHi++ )
		{
			for( ipLo=3; ipLo < ipHi; ipLo++ )
			{
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul = 
					(float)(hydro.HyLife[ipHi]*
					HydroBranch(ipHi,ipLo,nelem+1)*z4);
				ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul > 0.);

				/* make self-consistent opacity, convert new As back into opacities */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity = 
					(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul*
				  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
				  iso.stat[ipH_LIKE][nelem][ipLo]*
				  POW3(RYDLAM/(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN * WAVNRYD)));

				/* check that results are ok */
				ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity > 0.);
			}
		}

		/* the actual branching ratio from high levels down to
		 * 2s and 2p depend on the density.  the code goes to
		 * the correct low and high density limits - I know of
		 * nothing better than can be done. */
		factor = MAX2( 0.25 , 0.32 - 0.07*dense.eden/(dense.eden + 1e7) );

		/* treat 2s = to 2p for HydroBranch, which returns total to 2s+2p */
		for( ipHi=4; ipHi < limit; ipHi++ )
		{
			/* get new effective A for this density and temperature */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul = 
				(float)(hydro.HyLife[ipHi]*
				factor*HydroBranch(ipHi,2,nelem+1)*z4);

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul > 0.);

			/* do 2p by scaling relative to 2s */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul = (float)(
				EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul / factor *( 1. - factor )); 

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul > 0.);

			/* make self-consistent opaciity for 2s, from A */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].opacity = 
				(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].Aul*
			  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
			  iso.stat[ipH_LIKE][nelem][ipH2s]*
			  POW3(RYDLAM/(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].EnergyWN * WAVNRYD)));

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].opacity > 0.);

			/* make self-consistent opaciity for 2p, from A */
			EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].opacity = 
				(float)(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].Aul*
			  2.2448e-26*iso.stat[ipH_LIKE][nelem][ipHi]/
			  iso.stat[ipH_LIKE][nelem][ipH2p]*
			  POW3(RYDLAM/ (EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].EnergyWN * WAVNRYD)));

				/* check that results are ok */
			ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].opacity > 0.);
		}
	}
	/* end test branch lgDoEsc */

	/* now update escape and destruction prob */

	if( dense.xIonDense[nelem][nelem+1] > 1e-30 )
	{
		factor = dense.xIonDense[nelem][nelem+1];
	}
	else
	{
		/* case where almost no parent ion - this will make
		 * very large line opacity, so background dest small */
		factor = 1.;
	}

	/* save destruction probs for Lyman alphs */
	DespSave = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest;
	eEscSave = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc;

	/* hydrogenic lyman lines special since outward optical depths always set,
		* trick routines for Lyman lines only */
	lgTOnSave = opac.lgTauOutOn;
	opac.lgTauOutOn = TRUE;

	/* first do lyman alpha lya la, but only if optical depths not overrun */
	/* >>chng 01 apr 01, from 0.9 to 0.99 since outer edge better defined */
	tout = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot*0.99 - 
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn;

	/* must temporarily make ipLnPopOpc physical */
	EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc *= 
		(float)factor;

	/* generate escape prob, pumping rate, destruction prob, 
	 * inward outward fracs  */
	RTMakeLine(&EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s] , lgDoEsc );

	/* go back to original units so that final correction ok */
	EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc /= 
		(float)factor;

	/* >>>chng 99 dec 18, repair dest prob that got clobbered in call to
	 * RTMakeLine, since will not be evaluated when tout not positive */
	EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest = DespSave;
	ASSERT( EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest>= 0. );
	EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc = eEscSave;

	/* only update La dest prob if we have good outward optical depths */
	/* Lya is special since escape prob is always updated */
	/* only in case of static soln do we update Lya as special case */
	if( tout > 0. && wind.windv==0. )
	{
		double opac_line , eesc;

		tout = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot - 
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn;

		abundan = EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].PopOpc*dense.xIonDense[nelem][nelem+1];

		/* the descruction prob comes back as dest */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc = 
			(float)(RTesc_lya(&esin, &dest,abundan,nelem));

		/* this is current destruction rate */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest = dest;
		ASSERT( dest >= 0. );

		/* elec scat escape prob */
		opac_line = abundan * EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].opacity/DoppVel.doppler[nelem];

		if( opac_line > SMALLFLOAT )
		{
			double es = dense.eden*6.65e-25;
			/* this is equation 5 of 
			*>>refer	line	desp	Netzer, H., Elitzur, M., & Ferland, G. J. 1985, ApJ, 299, 752*/
			eesc = es/(es+opac_line) * MAX2(0.,1.-EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc - EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest);;
		}
		else
			eesc = 0.;

		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc = (float)eesc;

		if( nelem == ipHYDROGEN )
		{
			/* add on destruction of hydrogen Lya by FeII
				* now add in FeII deexcitation via overlap,
				* but only as loss, not photoionization, source
				* dstfe2lya is Ly-alpha deexcit by FeII overlap - conversion into FeII em */
			/* find FeII overlap destruction rate, 
			 * this does NOTHING when large FeII atom is turned on */
			fe2ovr();

			/* this was introduced in the fort - c conversion, and is a bug since
				* dest feii added to ots dest */
			/*EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest = MIN2(1.f,
				EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest + hydro.dstfe2lya);*/
			/* >>chng 00 jan 06, let dest be large than 1 to desaturate the atom */
			/* >>chng 01 apr 01, add test for tout >= 0., 
				* must not add to Pdest when it has not been refreshed here */
			if( tout >= 0. )
				EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest += hydro.dstfe2lya;

			/* now add on stark broadening term */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc = (float)(MIN2(1.,
				EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc+
				hydro.pestrk[ipH1s][ipH2p]));
		}

		{
			/* Lya debugging */
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && nelem==ipHYDROGEN )
			{
				fprintf(ioQQQ,
					"Lya eval nz%li abund\t%.2g\tc opac\t%.2e\tPdest\t%g\tPesc\t%g\tt tot\t%g\t t in\t%g\t Fe2 dest\t%g\n",
					nzone ,
					abundan , 
					opac.opacity_abs[EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].ipCont-1],
					EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest, 
					EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc, 
					EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauTot,
					EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].TauIn,
					hydro.dstfe2lya);
			}
		}

		/*  this is fraction of line which is inward escaping */
		EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].FracInwd = rt.fracin;
		/*if(nelem==1)
			fprintf(ioQQQ," desp=%g\n", dest );*/
	}

	/* now do remainder of Lyman lines, skipping 2s */
	ipLo=ipH1s;
	for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
	{
		/* must temporarily make ipLnPopOpc physical */
		EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc *= 
			(float)factor;

		/* generate escape prob, pumping rate, destruction prob, 
			* inward outward fracs  */
		RTMakeLine(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo] , lgDoEsc );

		{
			/* Lyman line debugging */
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && nelem==0 && iteration == 2)
			{
				fprintf(ioQQQ,
					"z%3li Ly3 eval Pdest popl\t%g\tconopac\t%g\tPdest\t%g\tPesc\t%g\t tot in\t%g\t%g\t ots\t%g\t conv\t%i\n",
					nzone ,
					abundan , 
					opac.opacity_abs[EmisLines[ipH_LIKE][nelem][3][ipH1s].ipCont-1],
					EmisLines[ipH_LIKE][nelem][3][ipH1s].Pdest, 
					EmisLines[ipH_LIKE][nelem][3][ipH1s].Pesc, 
					EmisLines[ipH_LIKE][nelem][3][ipH1s].TauTot,
					EmisLines[ipH_LIKE][nelem][3][ipH1s].TauIn,
					EmisLines[ipH_LIKE][nelem][3][ipH1s].ots,
					conv.lgSearch
					);
			}
		}

		/* go back to original units so that final correction ok */
		EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc /= 
			(float)factor;
	}

	/* reset the flag, so only Lyman lines forced to include outward */
	opac.lgTauOutOn = lgTOnSave;

	/* this loop for Balmer lines which are special,
		* because must bring 2s and 2p together */
	/* now do 2s and 2p, must bring optical depths together */
	for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
	{
		/* these are used for saving current state of two lines */
		float tauin2s , tauin2p , tauout2s , tauout2p;
		double opac2s , opac2p;

		/* 2s inward and total optical depths */
		tauin2s =  EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn;
		tauout2s = EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot;

		/* 2p inward and total optical depths */
		tauin2p =  EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn;
		tauout2p = EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot;

		/* add inward optical depths together */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn += 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn;

		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn = 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn;

		/* add both total optical depths together */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot += 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot;

		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot = 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot;

		/* current 2s and 2p opacities */
		opac2s = EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc;
		opac2p = EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc;

		/* add opacities together */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc += 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc;

		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc = 
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc;

		/* must temporarily make ipLnPopOpc physical */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc *= 
			(float)factor;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc *= 
			(float)factor;

		/* generate escape prob, pumping rate, destruction prob, 
			* inward outward fracs  */
		RTMakeLine(&EmisLines[ipH_LIKE][nelem][ipHi][ipH2s] , lgDoEsc );
		RTMakeLine(&EmisLines[ipH_LIKE][nelem][ipHi][ipH2p] , lgDoEsc );

		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].PopOpc =   opac2s;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauIn =   tauin2s;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2s].TauTot = tauout2s;

		/* 2p inward and total optical depths */
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].PopOpc =   opac2p;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauIn =   tauin2p;
		EmisLines[ipH_LIKE][nelem][ipHi][ipH2p].TauTot = tauout2p;
	}

	/* now do Paschen and higher lines */
	for( ipLo=3; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
	{
		for( ipHi=ipLo+1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
		{
			/* must temporarily make ipLnPopOpc physical */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc *= 
				(float)factor;

			/* generate escape prob, pumping rate, destruction prob, 
				* inward outward fracs  */
			RTMakeLine(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo] , lgDoEsc );

			/* go back to original units so that final correction ok */
			EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc /= 
				(float)factor;

		}
	}

	/* 666    C90 had the following code and warning about oscillations
		*  highest level is a mess, set escape prob to one, actually a blend
		*  if this is removed, then oscillations in totally optically thick
		*  blr occur, for cases where all lines are optically thick
		*  see model badbugs/bug0.in */
	/* >>chng 02 may 08, only update Pesc if it was updated in RTMakeLine */
	if( lgDoEsc )
		EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipH1s].Pesc = (float)(
		MAX2( EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipH1s].Pesc , 0.1));

	/* set esp prob from very highest level (a fake level) to all lower
	 *levels to unity */
	for( ipLo=ipH2s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
	{
		EmisLines[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem]-1][ipLo].Pesc = 1.;
	}

	{
		/* following should be set true to print ots contributors */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nelem==0 )
		{
			fprintf(ioQQQ,"Lya aver DespSave\t%g\tPdest\t%g\n",
				DespSave,
				EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest);
		}
	}
	/* >>chng 03 feb 02, rm whole block - mean done in rtstat */

	{
		/* following should be set true to print ots contributors */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			if( nelem<0 && iteration == 2)
			{
				fprintf(ioQQQ,
					"z%3li Bala popopc\t%g\tconopac\t%g\tPdest\t%g\tPup\t%g\tPlo\t%g\t%g pesc%g\n",
					nzone ,
					EmisLines[ipH_LIKE][nelem][3][4].PopOpc , 
					opac.opacity_abs[EmisLines[ipH_LIKE][nelem][3][2].ipCont-1],
					EmisLines[ipH_LIKE][nelem][3][2].Pdest, 
					EmisLines[ipH_LIKE][nelem][3][2].PopHi, 
					EmisLines[ipH_LIKE][nelem][3][2].PopLo,
					EmisLines[ipH_LIKE][nelem][3][2].TauIn,
					EmisLines[ipH_LIKE][nelem][3][2].Pesc
					);
			}
			else if (nelem==1 && iteration == 2 )
			{
				fprintf(ioQQQ,
					"z%3li he62a popopc\t%g\tconopac\t%g\tPdest\t%g\tPup\t%g\tPlo\t%g\t%g pesc%g\n",
					nzone ,
					EmisLines[ipH_LIKE][nelem][5][4].PopOpc , 
					opac.opacity_abs[EmisLines[ipH_LIKE][nelem][5][4].ipCont-1],
					EmisLines[ipH_LIKE][nelem][5][4].Pdest, 
					EmisLines[ipH_LIKE][nelem][5][4].PopHi, 
					EmisLines[ipH_LIKE][nelem][5][4].PopLo,
					EmisLines[ipH_LIKE][nelem][5][4].TauIn,
					EmisLines[ipH_LIKE][nelem][5][4].Pesc
					);
			}
		}
	}

	/* reset the flag */
	opac.lgTauOutOn = lgTOnSave;

	/* find the rate of  induced upward and downward two photon, save in
	 * iso.TwoNu_induc_dn[ipISO][nelem], iso.TwoNu_induc_up[ipISO][nelem] */
	twopht(ipH_LIKE , nelem);

	/* >>chng 02 jul 15, do not include pump upward rate */
	EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].pump = 0.;

	/* >>chng 01 jan 18, us derived two-photon, the sum of spontaneous and induced */
	/* >>chng 02 jun 26, by Ryan...this should be set else where, and is purely spontaneous	
	EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].Aul = ri2s1s;*/

	if( nelem == ipHYDROGEN )
	{
		/* do the 8446 problem */
		atom_oi_calc(&coloi);
		EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH1s].Pesc = atom_oi.pmph31/
		  EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH1s].Aul;
	}

	/* hydrogen-only solutions */
	/* >>chng 02 may 08, add test on lgDoEsc since adding to Pesc, and this is only
	 * evaulated above if lgDoEsc is true */
	if( nelem == ipHYDROGEN && lgDoEsc )
	{
		/* find Stark escape probabilities */
		/* >>chng 02 jun 12, move evaluation to above since need for Lya */
		/*strk();*/
		for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
		{
			/* >>chng 02 jun 12, do not do Lya here, since done above */
			for( ipHi=MAX2((long)3,ipLo+1); ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				/* >>chng 02 may 08, moved elec scat esc to rtmakestat */
				EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc = (float)(MIN2(1.,
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Pesc+
					hydro.pestrk[ipLo][ipHi]));
			}
		}

		if( trace.lgTrace && trace.lgIsoTraceFull[ipH_LIKE] )
		{
			fprintf( ioQQQ, "       HydroPesc calls P8446 who found pmph31=%10.2e\n", 
			  atom_oi.pmph31 );
		}

		{
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC )
			{
				fprintf(ioQQQ,"hydropescdebugg\t%li\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					nzone, 
					EmisLines[ipH_LIKE][nelem][2][0].pump,
					EmisLines[ipH_LIKE][nelem][4][3].TauIn,
					EmisLines[ipH_LIKE][nelem][4][3].TauCon,
					EmisLines[ipH_LIKE][nelem][4][3].pump,
					rfield.flux[EmisLines[ipH_LIKE][nelem][4][3].ipCont-1],
					rfield.OccNumbIncidCont[EmisLines[ipH_LIKE][nelem][4][3].ipCont-1]
					);
			}
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			if( nelem==0&& nzone>164 )
			{
				fprintf(ioQQQ,"hydropescdebugg\t%li\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n", 
					nzone, 
					EmisLines[ipH_LIKE][nelem][2][0].TauIn,
					EmisLines[ipH_LIKE][nelem][2][0].TauTot,
					EmisLines[ipH_LIKE][nelem][2][0].Pdest,
					EmisLines[ipH_LIKE][nelem][2][0].Pesc,
					tout);
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroPesc()\n", debug_fp );
#	endif
	return;
}

