/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*RadInc do work associated with geometry increments of this zone, called before RTOptDepthIncre */
/*pnegopc punch negative opacities on io unit, iff 'set negopc' command was given */
#include "cddefines.h"
#include "physconst.h"
#include "iso.h"
#include "atomcwgt.h"
#include "hydrogenic.h"
#include "rfield.h"
#include "jmin.h"
#include "plasnu.h"
#include "colden.h"
#include "geometry.h"
#include "opacity.h"
#include "thermal.h"
#include "dense.h"
#include "timesc.h"
#include "crsnut.h"
#include "neutrn.h"
#include "hevmolec.h"
#include "hmi.h"
#include "trace.h"
#include "wind.h"
#include "phycon.h"
#include "radius.h"
#include "pressure.h"
#include "aver.h"
#include "colmas.h"
#include "grainvar.h"
#include "molcol.h"
#include "h21cm.h"
#include "mean.h"
#include "heat.h"
#include "struc.h"
#include "forlin.h"
#include "cmshft.h"
#include "radinc.h"

#if !defined(NDEBUG)
/*pnegopc punch negative opacities on io unit, iff 'set negopc' command was given */
/* function only used in debug mode */
static void pnegopc(void);
#endif

void RadInc(void)
{
#	if !defined(NDEBUG)
	int lgFlxNeg;
#	endif

	long int nelem,
		i, 
		ion,
	  nd,
		mol;
	double avWeight,
	  DilutionHere ,
	  escatc, 
	  fmol,
	  opfac, 
	  relec, 
	  rforce, 
	  t;

	double ajmass, 
	  rjeans;

#	ifdef DEBUG_FUN
	fputs( "<+>RadInc()\n", debug_fp );
#	endif

	/* when this sub is called RADIUS is the outer edge of zone */

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
			" RadInc called; radius=%10.3e rinner=%10.3e DRAD=%10.3e drNext=%10.3e ROUTER=%10.3e DEPTH=%10.3e\n", 
		  radius.Radius, radius.rinner, radius.drad, radius.drNext, 
		  radius.router[iteration-1], radius.depth );
	}

	/* check cooling time for this zone, remember longest */
	timesc.ttherm = MAX2(timesc.ttherm,1.5*phycon.pden*BOLTZMANN*phycon.te/
	  thermal.ctot);

	/* fmol is fraction of hydrogen in form of any molecule or ion */
	/* if N_H_MOLEC != 8, we probably need to change this line... */
	assert (N_H_MOLEC == 8);
	fmol = (hmi.Molec[ipMHm] + 2.*(hmi.Molec[ipMH2] + hmi.Molec[ipMH2p]))/dense.gas_phase[ipHYDROGEN];

	/* remember the largest fraction that occurs in the model */
	hmi.BiggestH2 = MAX2(hmi.BiggestH2,(float)fmol);

	/* H 21 cm equilibrium timescale, H21cm returns H (not e) collisional deexcit rate (not cs) */
	t = H21cm_H_atom( phycon.te )* dense.xIonDense[ipHELIUM][ipHYDROGEN] +
		/* >>chng 02 feb 14, add electron term as per discussion in */
		/* >>refer	H1	21cm	Liszt, H., 2001, A&A, 371, 698 */
		H21cm_electron( phycon.te )*dense.eden;

	if( t > SMALLFLOAT )
	{
		timesc.TimeH21cm = MAX2( 1./t, timesc.TimeH21cm );
	}

	/* remember longest CO timescale */
	if( dense.xIonDense[ipCARBON][0]*dense.xIonDense[ipOXYGEN][0] > SMALLFLOAT )
	{
		double timemin;
		timemin = MIN2(timesc.AgeCOMoleDest ,
			timesc.AgeCOMoleDest* hevmolec.hevmol[ipCO]/
			MAX2(SMALLFLOAT,dense.xIonDense[ipCARBON][0]*dense.xIonDense[ipOXYGEN][0]) );
		/* this is rate CO is destroyed, equal to formation rate in equilibrium */
		timesc.BigCOMoleForm = MAX2(  timesc.BigCOMoleForm , timemin );
	}

	/* remember longest H2  destruction timescale timescale */
	timesc.time_H2_Dest_longest = MAX2(timesc.time_H2_Dest_longest, timesc.time_H2_Dest_here );

	/* remember longest H2 formation timescale timescale */
	timesc.time_H2_Form_longest = MAX2( timesc.time_H2_Form_longest , timesc.time_H2_Form_here );

	/* increment counter if this zone possibly thermally unstable
	 * this flag was set in ionte, deriv of heating and cooling negative */
	if( thermal.lgUnstable )
	{
		thermal.nUnstable += 1;
	}

	/* remember Stromgren radius - where hydrogen ionization falls below half */
	if( !rfield.lgUSphON && dense.xIonDense[ipHYDROGEN][0]/dense.gas_phase[ipHYDROGEN] > 0.49 )
	{
		rfield.rstrom = (float)radius.Radius;
		rfield.lgUSphON = TRUE;
	}

	/* ratio of inner to outer radii, at this point
	 * radius is the outer radius of this zone */
	DilutionHere = POW2((radius.Radius - radius.drad*radius.dRadSign)/
	  radius.Radius);

	relec = 0.;
	rforce = 0.;

	if( trace.lgTrace && trace.lgConBug )
	{
		fprintf( ioQQQ, " Energy, flux, OTS:\n" );
		for( i=0; i < rfield.nflux; i++ )
		{
			fprintf( ioQQQ, "%6ld%10.2e%10.2e%10.2e", i, rfield.anu[i], 
			  rfield.flux[i] + rfield.outlin[i] + rfield.ConInterOut[i], 
			  rfield.otscon[i] + rfield.otslin[i] );
		}
		fprintf( ioQQQ, "\n" );
	}

	/* diffuse continua factor
	 * if lgSphere then all diffuse radiation is outward (COVRT=1)
	 * lgSphere not set then COVRT=0.0 */

	/* begin sanity check */
	/* only do this test in debug mode */
#	if !defined(NDEBUG)
	lgFlxNeg = FALSE;
	for( i=0; i < rfield.nflux; i++ )
	{
		if( rfield.flux[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative intensity in flux.\n" );
			fprintf( ioQQQ, " Intensity, frequency, pointer=%11.3e%11.3e%6ld\n", 
			  rfield.flux[i], rfield.anu[i], i );
			lgFlxNeg = TRUE;
		}
		if( rfield.otscon[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative intensity in otscon.\n" );
			fprintf( ioQQQ, " Intensity, frequency, pointer=%11.3e%11.3e%6ld\n", 
			  rfield.otscon[i], rfield.anu[i], i );
			lgFlxNeg = TRUE;
		}
		if( opac.tmn[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative tmn.\n" );
			fprintf( ioQQQ, " value, frequency, pointer=%11.3e%11.3e%6ld %4.4s\n", 
			  opac.tmn[i], rfield.anu[i], i, rfield.chLineLabel[i] );
			lgFlxNeg = TRUE;
		}
		if( rfield.otslin[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative intensity in otslin.\n" );
			fprintf( ioQQQ, " Intensity, frequency, pointer=%11.3e%11.3e%6ld %4.4s\n", 
			  rfield.otslin[i], rfield.anu[i], i, rfield.chLineLabel[i]  );
			lgFlxNeg = TRUE;
		}
		if( rfield.outlin[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative intensity in outlin.\n" );
			fprintf( ioQQQ, " Intensity, frequency, pointer=%11.3e%11.3e%6ld %4.4s\n", 
			  rfield.outlin[i], rfield.anu[i], i, rfield.chLineLabel[i]  );
			lgFlxNeg = TRUE;
		}
		if( rfield.ConInterOut[i] < 0. )
		{
			fprintf( ioQQQ, " RadInc finds negative intensity in ConInterOut.\n" );
			fprintf( ioQQQ, " Intensity, frequency, pointer=%11.3e%11.3e%6ld %4.4s\n", 
			  rfield.ConInterOut[i], rfield.anu[i], i, rfield.chContLabel[i]  );
			lgFlxNeg = TRUE;
		}
		if( opac.opacity_abs[i] < 0. )
		{
			opac.lgOpacNeg = TRUE;
			/* this sub will punch negative opacities on io unit,
			 * iff 'set negopc' command was given */
			pnegopc();
		}
	}
	if( lgFlxNeg )
	{
		fprintf( ioQQQ, " Insanity has occurred, this is zone%4ld\n", 
		  nzone );
		ShowMe();
		puts( "[Stop in RadInc]" );
		cdEXIT(EXIT_FAILURE);
	}
	/*end sanity check*/
#	endif

	/* attenuate the flux array after doing radiative acceleration */
	escatc = 6.65e-25*dense.eden;

	/* this loop should not be to <= nflux since we not want to clobber the
	 * continuum unit integration */
	for( i=0; i < rfield.nflux; i++ )
	{
		/* sum total continuous optical depths */
		opac.TauAbsGeo[0][i] += (float)(opac.opacity_abs[i]*radius.drad_x_fillfac);
		opac.TauScatGeo[0][i] += (float)(opac.opacity_sct[i]*radius.drad_x_fillfac);

		/* following only optical depth to illuminated face */
		opac.TauAbsFace[i] += (float)(opac.opacity_abs[i]*radius.drad_x_fillfac);
		opac.TauScatFace[i] += (float)(opac.opacity_sct[i]*radius.drad_x_fillfac);

		/* these are total in inward direction, large if sherical */
		opac.TauTotalGeo[0][i] = opac.TauAbsGeo[0][i] + opac.TauScatGeo[0][i];

		/* TMN is array of scale factors which account for attenuation
		 * of continuum across the zone (necessary to conserve energy
		 * at the 1% - 5% level.) sphere effects in
		 * drNext was set by NEXTDR and will be next dr */
		/* compute both total and thompson scat rad acceleration */
		rforce += (rfield.flux[i] + rfield.outlin[i] + rfield.ConInterOut[i])*
			rfield.anu[i]*(opac.opacity_abs[i] + 
		  opac.opacity_sct[i]);

		relec += ((rfield.flux[i] + rfield.outlin[i] + rfield.ConInterOut[i])*
			escatc)*rfield.anu[i];

		/* attenuation of flux by optical depths IN THIS ZONE 
		 * AngleIllum is 1/COS(theta), is usually 1, reset with illuminate command,
		 * option for illumination of slab at an angle */
		opac.ExpZone[i] = sexp(opac.opacity_abs[i]*radius.drad_x_fillfac*geometry.AngleIllum);

		/* e(-tau) in inward direction, up to illuminated face */
		opac.ExpmTau[i] *= (float)opac.ExpZone[i];

		/* e2(tau) in inward direction, up to illuminated face */
		opac.e2TauAbs[i] = (float)e2(opac.TauAbsFace[i],opac.ExpmTau[i]);
		ASSERT( opac.e2TauAbs[i] <= 1. && opac.e2TauAbs[i] >= 0. );

		/* DilutionHere is square of ratio of inner to outer radius */
		opfac = opac.ExpZone[i]*DilutionHere;

		rfield.flux[i] *= (float)opfac;

		/* outward lines and continua */
		rfield.ConInterOut[i] *= (float)opfac;
		rfield.outlin[i] *= (float)opfac;

		/* this simulates the behavior in C90 and before,
		 * first attenuate flux acculumated so far, then
		 * add on flux from this zone 
		rfield.ConEmitOut[i] *= (float)opfac;*/

		rfield.ConEmitOut[i] *= (float)opfac;
		rfield.ConEmitOut[i] += rfield.ConEmitLocal[i]*(float)radius.dVolOutwrd*opac.tmn[i]/**/;

		/* set occupation numbers, first attenuated incident continuum */
		rfield.OccNumbIncidCont[i] = rfield.flux[i]*rfield.convoc[i];

		/* >>chng 00 oct 03, add diffuse continua */
		/* local diffuse continua */
		rfield.OccNumbDiffCont[i] = rfield.ConEmitLocal[i]*rfield.convoc[i];

	}

	/* begin sanity check, compare total Lyman continuum optical depth 
	 * with amount of extinction there */

	/* this is amount continuum attenuated to illuminated face, 
	 * but only do test if flux positive, not counting scattering opacity,
	 * and correction for spherical dilution not important */
	/* >>chng 02 dec 05, add test for small float, protect against models where we have
	 * gone below smallfloat, and so float is ragged */
	if( rfield.flux[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]>SMALLFLOAT &&
		(rfield.flux[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]/MAX2(SMALLFLOAT,rfield.FluxSave[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]) ) > SMALLFLOAT && 
		!opac.lgScatON &&
		radius.depth/radius.Radius < 1e-4 )
	{
		/* ratio of current to incident continuum, converted to optical depth */
		/* >>chng 99 apr 23, this crashed on alpha due to underflow to zero in argy
		 * to log.  defended two tways - above checks that ratio of fluxes is large enough, 
		 * and here convert to double.
		 * error found by Peter van Hoof */
		double tau_effec = -log((double)rfield.flux[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]/
			(double)opac.tmn[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]/
			(double)rfield.FluxSave[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]);

		/* this is computed absorption optical depth to illuminated face */
		double tau_true = opac.TauAbsFace[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1] ;

		/* first test is relative error, second is to absolute error and comes
		 * in for very small tau, where differences are in the round-off */
		if( fabs( tau_effec - tau_true ) / t > 0.01 && 
			/* for very small inner optical depths, the tmn correction is major,
			 * and this test is not precise */
			fabs(tau_effec-tau_true)>MAX2(0.001,1.-opac.tmn[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][0]-1]) )
		{
			/* in print below must add extra HI col den since this will later be
			 * incremented in RTOptDepthIncre */
			fprintf( ioQQQ,
				" PROBLEM RadInc lyman continuum insanity zone %li, effective tau=%g, atomic tau=%g simple tau=%g\n",
				nzone, 
				tau_effec , 
				tau_true ,
				6.327e-18*(dense.xIonDense[ipHYDROGEN][0]*radius.drad_x_fillfac+colden.colden[ipCHI]) );
			TotalInsanity();
		}
	}
	/* end sanity check */

	/* do scattering opacity, intensity goes as 1/(1+tau) */
	if( opac.lgScatON )
	{
		for( i=0; i < rfield.nflux; i++ )
		{
			/* assume half forward scattering
			 * opfac = 1. / ( 1. + dReff*0.5 * scatop(i) )
			 * >>chng 97 apr 25, remove .5 to get agreement with
			 * Lightman and White formula */
			opfac = 1./(1. + radius.drad_x_fillfac*opac.opacity_sct[i]);
			rfield.flux[i] *= (float)opfac;
			rfield.ConInterOut[i] *= (float)opfac;
			rfield.ConEmitOut[i] *= (float)opfac;
			rfield.outlin[i] *= (float)opfac;
		}
	}

	/* now do slight reshuffling of energy due to compton scattering */
	cmshft();

	relec *= EN1RYD;

	/* radiative acceleration; xMassDensity is gm per cc, eval when PTOT called */
	t = 1./SPEEDLIGHT/phycon.xMassDensity;
	/* >>chng 03 feb 01, lgLineRadPresOn was missing */
	wind.AccelLine = (float)(forlin()*t)*pressure.lgLineRadPresOn;
	/* >>chng 01 aug 03, add lgConPress to kill AccelCont here */
	wind.AccelCont = (float)(rforce*EN1RYD*t)*pressure.lgContRadPresOn;
	/* this is numerically unstable */
	wind.AccelPres = 0.;
	/* total acceleration */
	wind.AccelTot = wind.AccelCont + wind.AccelLine + wind.AccelPres;
	/* remember the largest value */
	wind.AccelMax = (float)MAX2(wind.AccelMax,wind.AccelTot);
	/* this shows that AccelLine becomes unphysicsl, presumaby due to forlin
	 * fprintf(ioQQQ," wind te %e  %e %e %e\n", phycon.te,	wind.AccelLine ,
		wind.AccelCont ,wind.AccelTot );*/

	/* force multiplier; RELEC*T can be zero for very low densities */
	if( relec*t > SMALLFLOAT )
	{
		wind.fmul = (float)((wind.AccelLine + wind.AccelCont)/(relec*t));
	}
	else
	{
		wind.fmul = 0.;
	}

	/* following is integral of radiative force */
	pressure.pinzon = (float)(wind.AccelTot*phycon.xMassDensity*radius.drad_x_fillfac*geometry.AngleIllum);
	pressure.PresInteg += pressure.pinzon;

	/* sound is sound travel time, sqrt term is sound speed */
	timesc.sound_speed_isothermal = sqrt(pressure.PresGasCurr/phycon.xMassDensity);
	/* adiabatic sound speed assuming monatomic gas - gamnma is 5/3*/
	timesc.sound_speed_adiabatic = sqrt(5.*pressure.PresGasCurr/(3.*phycon.xMassDensity) );
	timesc.sound += radius.drad_x_fillfac / timesc.sound_speed_isothermal;

	/* attenuate neutrons if they are present */
	if( neutrn.lgNeutrnHeatOn )
	{
		/* correct for optical depth effects */
		neutrn.totneu *= (float)sexp(crsnut.CrsSecNeutron*dense.gas_phase[ipHYDROGEN]*radius.drad_x_fillfac*geometry.AngleIllum);
		/* correct for spherical effects */
		neutrn.totneu *= (float)DilutionHere;
	}

	/* following radiation factors are extinguished by 1/r**2 and e- opacity
	 * also bound electrons */
	
	/* do all emergent spectrum from illuminated face if model is NOT spherical */
	if( !geometry.lgSphere )
	{
		double Reflec_Diffuse_Cont;

		/* >>chng 01 jul 14, from lower limit of 0 to plasma frequency -
		 * never should have added diffuse emission from below the plasma frequency */
		for( i=plasnu.ipPlasma-1; i < rfield.nflux; i++ )
		{
			if( opac.TauAbsGeo[0][i] < 30. )
			{
				/* ConEmitLocal is diffuse emission per unit vol, fill fac
				 * the 1/2 comes from isotropic emission
				 * scatsv(i) = (flux(i)+ConInterOut(i))*0.5*scatop(i)
				 * >>chng 97 apr 25, remove fac of 2 to
				 * get agreement wth lightman&white
				 * Reflec_Diffuse_Cont = (ConEmitLocal(i)+scatsv(i)) * dReff *e2TauAbs(i)
				 * >>chng 97 may 08, needed 1/2 in front of ConEmitLocal 
				 * had been there, but also with scatsv, and lost both
				 * when /2 removed to get agreement with lightman and white */
				/* >>chng 99 dec 21, add ConLocNoInter to account for rec continua
				 * >>chng 01 jul 01, remove ConLocNoInter from code - only had thick brems */
				Reflec_Diffuse_Cont = rfield.ConEmitLocal[i]/2.*
					radius.drad_x_fillfac * opac.e2TauAbs[i]*radius.r1r0sq;

				/* Reflec_Diffuse_Cont = (ConEmitLocal(i)+scatsv(i)) * dReff/2. *e2TauAbs(i)
				 * ConEmitReflec - reflected diffuse continuum */
				rfield.ConEmitReflec[i] += (float)(Reflec_Diffuse_Cont);

				/* the reflected part of the incident continuum */
				rfield.ConRefIncid[i] += (float)(rfield.flux[i]*opac.opacity_sct[i]*
				  radius.drad_x_fillfac*opac.e2TauAbs[i]*radius.r1r0sq);

				/* >>chng 97 apr 25, remove fac of 2 to get
				 * agreement wth lightman&white */
				rfield.reflin[i] += (float)(rfield.outlin[i]*opac.opacity_sct[i]*
				  radius.drad_x_fillfac*opac.e2TauAbs[i]*radius.r1r0sq);
			}
		}
	}

	/* following is general method to find means weighted by various functions
	 * called in StartIter to initialize to zero, called here to put in numbers
	 * results will be weighted by radius and volumn
	 * this is the only place things must be entered to create averages */
	aver("zone",1.,1.,"          ");
	aver("doit",phycon.te,1.,"    Te    ");
	aver("doit",phycon.te,dense.eden,"  Te(Ne)  ");
	aver("doit",phycon.te,dense.eden*dense.xIonDense[ipHYDROGEN][1]," Te(NeNp) ");
	aver("doit",phycon.te,dense.eden*dense.xIonDense[ipHELIUM][1]," Te(NeHe+)");
	aver("doit",phycon.te,dense.eden*dense.xIonDense[ipHELIUM][2],"Te(NeHe2+)");
	aver("doit",phycon.te,dense.eden*dense.xIonDense[ipOXYGEN][1]," Te(NeO+) " );
	aver("doit",phycon.te,dense.eden*dense.xIonDense[ipOXYGEN][2]," Te(NeO2+)");
	aver("doit",phycon.te,hmi.Molec[ipMH2],"  Te(H2)  ");
	aver("doit",dense.gas_phase[ipHYDROGEN],1.,"   N(H)   ");
	aver("doit",dense.eden,dense.xIonDense[ipOXYGEN][2],"  Ne(O2+) ");
	aver("doit",dense.eden,dense.xIonDense[ipHYDROGEN][1],"  Ne(Np)  ");

	/* save information about structure of model, now used to get t^2 */
	/*nd = MIN2(nzone , NZLIM)-1;*/
	nd = nzone - 1;
	struc.testr[nd] = phycon.te;
	/* number of particles per unit vol */
	struc.DenParticles[nd] = phycon.pden;
	/* >>chng 02 May 2001 rjrw: add hden for dilution */
	struc.hden[nd] = (float)dense.gas_phase[ipHYDROGEN];
	/* total grams per unit vol */
	struc.DenMass[nd] = phycon.xMassDensity;
	struc.heatstr[nd] = heat.htot;
	struc.coolstr[nd] = thermal.ctot;
	struc.volstr[nd] = (float)radius.dVeff;
	struc.drad[nd] = (float)radius.drad;
	struc.radstr[nd] = (float)radius.drad_x_fillfac;
	struc.histr[nd] = dense.xIonDense[ipHYDROGEN][0];
	struc.hiistr[nd] = dense.xIonDense[ipHYDROGEN][1];
	struc.ednstr[nd] = (float)dense.eden;
	struc.o3str[nd] = dense.xIonDense[ipOXYGEN][2];
	struc.pressure[nd] = (float)pressure.PresTotlCurr;
	struc.PresRadCurr[nd] = (float)pressure.PresRadCurr;
	struc.GasPressure[nd] = (float)pressure.PresGasCurr;
	struc.depth[nd] = (float)radius.depth;
	/* save absorption optical depth from illuminated face to current position */
	struc.xLyman_depth[nd] = opac.TauAbsFace[iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s]];
	for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
	{
		struc.gas_phase[nd][nelem] = dense.gas_phase[nelem];
		for( ion=0; ion<nelem+2 ; ++ion )
		{
			struc.xIonDense[nd][nelem][ion] = dense.xIonDense[nelem][ion];
		}
	}

	/* the hydrogen molecules */
	for (mol=0;mol<N_H_MOLEC;mol++) 
	{
		struc.Molec[mol][nd] = hmi.Molec[mol];
	}

	colden.dlnenp += dense.eden*(double)(dense.xIonDense[ipHYDROGEN][1])*radius.drad_x_fillfac;
	colden.dlnenHep += dense.eden*(double)(dense.xIonDense[ipHELIUM][1])*radius.drad_x_fillfac;
	colden.dlnenHepp += dense.eden*(double)(dense.xIonDense[ipHELIUM][2])*radius.drad_x_fillfac;

	/* counter for number of times Lya excit temp hotter than gas */
	if( hydro.TexcLya > phycon.te )
	{
		hydro.nLyaHot += 1;
		if( hydro.TexcLya > hydro.TLyaMax )
		{
			hydro.TLyaMax = hydro.TexcLya;
			hydro.TeLyaMax = phycon.te;
			hydro.nZTLaMax = nzone;
		}
	}

	/* column densities in various species */
	colden.colden[ipCOLUMNDENSITY] += (float)(dense.gas_phase[ipHYDROGEN]*radius.drad_x_fillfac);
	colden.colden[ipCHMIN] += hmi.Molec[ipMHm]*(float)radius.drad_x_fillfac;
	/* >>chng 02 sep 20, from htwo to htwo_total */
	colden.colden[ipCOLH2] += hmi.htwo_total*(float)radius.drad_x_fillfac;
	colden.colden[ipCHEHP] += hmi.Molec[ipMHeHp]*(float)radius.drad_x_fillfac;
	colden.colden[ipCH2PLS] += hmi.Molec[ipMH2p]*(float)radius.drad_x_fillfac;
	colden.colden[ipCH3PLS] += hmi.Molec[ipMH3p]*(float)radius.drad_x_fillfac;
	colden.colden[ipCHII] += dense.xIonDense[ipHYDROGEN][1]*(float)radius.drad_x_fillfac;
	colden.colden[ipCHI] += dense.xIonDense[ipHYDROGEN][0]*(float)radius.drad_x_fillfac;

	/* He I t2S column density*/
	colden.He123S += dense.xIonDense[ipHELIUM][1]*
		iso.Pop2Ion[ipHE_LIKE][ipHELIUM][ipHe2s3S]*(float)radius.drad_x_fillfac;

	/* the CII atom is a special case */
	for( i=0; i < 5; i++ )
	{
		/* pops and column density for C II atom */
		colden.C2Colden[i] += colden.C2Pops[i]*(float)radius.drad_x_fillfac;
		/* pops and column density for SiII atom */
		colden.Si2Colden[i] += colden.Si2Pops[i]*(float)radius.drad_x_fillfac;
	}
	for( i=0; i < 3; i++ )
	{
		/* pops and column density for CI atom */
		colden.C1Colden[i] += colden.C1Pops[i]*(float)radius.drad_x_fillfac;
		/* pops and column density for OI atom */
		colden.O1Colden[i] += colden.O1Pops[i]*(float)radius.drad_x_fillfac;
	}

	/* now add total molecular column densities */
	molcol("ADD ");

	/* increment forming the mean ionizatin and temperature */
	MeanInc();

	/*-----------------------------------------------------------------------*/

	/* calculate average atomic weight per hydrogen of the plasma */
	avWeight = 0.;
	for( nelem=0; nelem < LIMELM; nelem++ ) {
		avWeight += dense.gas_phase[nelem]*AtomcWgt.AtomicWeight[nelem];
	}
	avWeight /= dense.gas_phase[ipHYDROGEN];

	for( nd=0; nd < gv.nBin; nd++ )
	{
		gv.bin[nd]->avdust += gv.bin[nd]->tedust*(float)radius.drad_x_fillfac;
		gv.bin[nd]->avdft += gv.bin[nd]->DustDftVel*(float)radius.drad_x_fillfac;
		gv.bin[nd]->avdpot += (float)(gv.bin[nd]->dstpot*EVRYD*radius.drad_x_fillfac);
		gv.bin[nd]->avDGRatio += (float)(gv.bin[nd]->dustp[1]*gv.bin[nd]->dustp[2]*
			gv.bin[nd]->dustp[3]*gv.bin[nd]->dustp[4]*gv.bin[nd]->dstAbund/avWeight*
			radius.drad_x_fillfac);

		/* this is total extinction at V and B, - total absorption and scattering,
		 * does not discount forward scattering to be similar to extinction
		 * measurements made within ism */
		rfield.extin_B += (gv.bin[nd]->dstab1[rfield.ipB_filter] +
			gv.bin[nd]->pure_sc1[rfield.ipB_filter])*gv.bin[nd]->dstAbund*
			radius.drad_x_fillfac;

		rfield.extin_V += (gv.bin[nd]->dstab1[rfield.ipV_filter] +
			gv.bin[nd]->pure_sc1[rfield.ipV_filter])*gv.bin[nd]->dstAbund*
			radius.drad_x_fillfac;
	}

	colmas.TotMassColl += phycon.xMassDensity*(float)radius.drad_x_fillfac;
	colmas.tmas += phycon.te*phycon.xMassDensity*(float)radius.drad_x_fillfac;
	colmas.wmas += phycon.wmole*phycon.xMassDensity*(float)radius.drad_x_fillfac;

	/* now find minimum Jeans length and mass; length in cm */
	rjeans = 7.79637 + (phycon.alogte - log10(phycon.wmole) - log10(phycon.xMassDensity*
	  geometry.FillFac))/2.;

	/* minimum Jeans mass in gm */
	ajmass = 3.*(rjeans - 0.30103) + log10(4.*PI/3.*phycon.xMassDensity*
	  geometry.FillFac);

	/* now remember smallest */
	jmin.rjnmin = MIN2(jmin.rjnmin,(float)rjeans);
	jmin.ajmmin = MIN2(jmin.ajmmin,(float)ajmass);

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, " RadInc returns\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->RadInc()\n", debug_fp );
#	endif
	return;
}

#if !defined(NDEBUG)
/*pnegopc punch negative opacities on io unit, iff 'set negopc' command was given */
static void pnegopc(void)
{
	long int i;
	FILE *ioFile;

#	ifdef DEBUG_FUN
	fputs( "<+>pnegopc()\n", debug_fp );
#	endif

	if( opac.lgNegOpacIO )
	{
		/* option to punch negative opacities */
		if( NULL==(ioFile=fopen("negopc.txt","w")) )
		{
			fprintf( ioQQQ,"pnegopc could not open negopc.txt for writing\n");
			puts( "[Stop in opac0]" );
			cdEXIT(EXIT_FAILURE);
		}
		for( i=0; i < rfield.nflux; i++ )
		{
			fprintf( ioFile, "%10.2e %10.2e \n", rfield.anu[i], 
			  opac.opacity_abs[i] );
		}
		fclose( ioFile);
	}


#	ifdef DEBUG_FUN
	fputs( " <->pnegopc()\n", debug_fp );
#	endif
	return;
}
#endif
/* Note - when this file is compiled in fast optimized mode,
 * a good compiler will complain that the routine pnegopc is a
 * static, local routine, but that it has not been used.  It is
 * indeed used , but only when NDEBUG is not set, so it is only
 * used when the code is compiled in debug mode.  So this is
 * not a problem. */
