/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/* PressureChange called by ConvPresTempEdenIoniz
 * evaluate the current pressure, change needed to get it to converge,
 * the global static variable pressure_change_factor
 * aplies this correction factor to all gas constituents,
 * sets conv.lgConvPres true if good pressure, false if pressure change capped
 */
/*lgConvPres returns true if pressure is converged */
#include "cddefines.h"
#include "physconst.h"
#include "abund.h"
#include "hmi.h"
#include "trace.h"
#include "wind.h"
#include "fluct.h"
#include "geometry.h"
#include "radius.h"
#include "dense.h"
#include "phycon.h"
#include "dynamics.h"
#include "pressure.h"
#include "colden.h"
#include "dndt.h"
#include "hevmolec.h"
#include "tfidle.h"
#include "radacl.h"
#include "tabden.h"
#include "fabden.h"
#include "converge.h"

/* this is the pressure change pressure_change_factor, needed to move current pressure to correct pressure */
static double pressure_change_factor;
/*PressureChange evaluate the current pressure, and change needed to get it to PresTotlInit,
 * lgPresOscil is true if pressure oscillations have occurred, as when we are near solution,
 * return value is true is density was changed, false if no changes were necessary */
int PressureChange( 
	/* this is true if change in pressure has changed sign, showing near solution */
	int lgPresOscil ,
	/* this is change in pressure as result of change in density */
	double slope )
{
	long int ion, 
	  nelem,
		lgChange,
		mol;

	double abun, 
	  edensave, 
	  hold;

	/* biggest multiplicative change in the pressure that we will allow */
	/* allowed error in the pressure is conv.PressureErrorAllowed*/
	float pdelta;

	static double FacAbun, 
	  FacAbunSav, 
	  OldAbun;

#	ifdef DEBUG_FUN
	fputs( "<+>PressureChange()\n", debug_fp );
#	endif

	edensave = dense.eden;

	lgChange = FALSE;

	/* this code fragment is totally bogus, since lgPresOscil and slope are not
	 * used in this routine - shuts the lint up - may be used in future */
	if( lgPresOscil )
		abun = slope;

	/* first evaluate total pressure for this location, and current conditions
	 * CurrentPressure is just sum of gas plus local line radiation pressure */
	/* this sets values of pressure.PresTotlCurr, also calls tfidle */
	PresTotCurrent();

	/* this is special case where we are working on first zone, at
	 * illuminated face of the cloud.  Here we want to remember the
	 * initial pressure in case constant pressure is needed */
	if( nzone == 1 )
	{
		/* this is first zone, lock onto pressure */
		pressure.PresTotlInit = pressure.PresTotlCurr;
		pressure.PresRamInit = pressure.PresRamCurr;
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, 
			  " PressureChange called, this is first zone, so reseting pressure to%10.3e\n", 
			  pressure.PresTotlInit );
		}
	}

	/* remember old hydrogen density */
	hold = dense.gas_phase[ipHYDROGEN];

	/* this evaluates pressure_change_factor and pressure.PresTotlCorrect, 
	 * and returns TRUE if pressure has converged */
	conv.lgConvPres = lgConvPres();

	/* Short-circuit if convergence is OK at present state, so no change reqd */
	if (conv.lgConvPres)
		return lgChange;

	/* >> chng 02 dec 13 rjrw: short-circuit if nothing changes */
	if (pressure_change_factor != 1.)
	{
		lgChange = TRUE;
	}

	/* >>chng 01 nov 08 pass logical var saying whether sign of change in pressure has
	 * changed, indicating we are close to soln - use bigger steps if 
	 * change in sign has not occurred */
#if 0
	if( lgPresOscil )
	{
		/* near solution since change in pressure has changed sign */

		/* make sure that change is not too extreme */
		pressure_change_factor = MIN2(pressure_change_factor,1.+2.*conv.PressureErrorAllowed/slope);
		pressure_change_factor = MAX2(pressure_change_factor,1.-2.*conv.PressureErrorAllowed/slope);
	}
	else
	{
		/* this is rel change allowed */
		TestCode();/* allow 2percent changes */
		pdelta = 0.02f;

		/* make sure that change is not too extreme */
		pressure_change_factor = MIN2(pressure_change_factor,1.+pdelta);
		pressure_change_factor = MAX2(pressure_change_factor,1.-pdelta);
	}
#endif

	TestCode();/* allow 2percent changes - kill slope */
	pdelta = 0.03f;

	/* make sure that change is not too extreme */
	/* pressure_change_factor = MIN2(pressure_change_factor,1.+2.*conv.PressureErrorAllowed/slope); */
	/* pressure_change_factor = MAX2(pressure_change_factor,1.-2.*conv.PressureErrorAllowed/slope); */

	/* make sure that change is not too extreme */
	pressure_change_factor = MIN2(pressure_change_factor,1.+pdelta);
	pressure_change_factor = MAX2(pressure_change_factor,1.-pdelta);

	{
		/*@-redef@*/
		enum{DEBUG_LOC=FALSE};
		static long int nsave=-1;
		/*@+redef@*/
		if( DEBUG_LOC /*&& nzone > 150 && iteration > 1*/ )
		{
			if( nsave-nzone ) fprintf(ioQQQ,"\n");
			nsave = nzone;
			fprintf(ioQQQ,"nnzzone\t%li\t%.2f%%\t%.3f\t%.2e\t%.2e\t%.2e\t%.2e\n", 
				nzone,
				pressure_change_factor,
				/* when this is negative we need to raise the density */
				(pressure.PresTotlCorrect-pressure.PresTotlCurr)*100./pressure.PresTotlCorrect, 
				pressure.PresTotlCorrect,
			    pressure.PresTotlCurr, 
				pressure.PresGasCurr,
				pressure.PresRadCurr ) ;
		}
	}

	/* >>chng 97 jun 03, added variable abundances for element table command
	 * option to get abundances off a table with element read command */
	if( abund.lgAbTaON )
	{
		lgChange = TRUE;
		for( nelem=1; nelem < LIMELM; nelem++ )
		{
			if( abund.lgAbunTabl[nelem] )
			{
				abun = AbundancesTable(radius.Radius,radius.depth,nelem+1)*
				  dense.gas_phase[ipHYDROGEN];

				hold = abun/dense.gas_phase[nelem];
				dense.gas_phase[nelem] = (float)abun;

				for( ion=0; ion < (nelem + 2); ion++ )
				{
					dense.xIonDense[nelem][ion] *= (float)hold;
				}
			}
		}
	}

	/* this happens if fluctuations abundances command entered */
	if( !fluct.lgDenFluc )
	{
		lgChange = TRUE;
		if( nzone <= 1 )
		{
			OldAbun = 1.;
			FacAbun = 1.;
			FacAbunSav = fluct.cfirst*cos(radius.depth*fluct.flong+
			  fluct.flcPhase) + fluct.csecnd;
		}

		else
		{
			OldAbun = FacAbunSav;
			/* rapid abundances fluctuation */
			FacAbunSav = fluct.cfirst*cos(radius.depth*fluct.flong+
			  fluct.flcPhase) + fluct.csecnd;
			FacAbun = FacAbunSav/OldAbun;
		}
	}

	else
	{
		FacAbun = 1.;
	}

	/* chng 02 dec 11 rjrw -- remove test, saves marginal time could generate nasty intermittent bug when 
		 ( pressure_change_factor*FacAbun == 1 ) && (pressure_change_factor != 1) */
	if( lgChange )
	{
		/* H, He not affected by abundance fluctuations */
		for( nelem=0; nelem < 2; nelem++ )
		{
			dense.xMolecules[nelem] *= (float)pressure_change_factor;
			dense.gas_phase[nelem] *= (float)pressure_change_factor;
			for( ion=0; ion < (nelem + 2); ion++ )
			{
				/* >>chng 96 jul 12 had only multiplied total abun, not ions */
				dense.xIonDense[nelem][ion] *= (float)pressure_change_factor;
			}
		}

		/* Lit on up is, so with FacAbun */
		for( nelem=2; nelem < LIMELM; nelem++ )
		{
			dense.xMolecules[nelem] *= (float)(pressure_change_factor*FacAbun);
			dense.gas_phase[nelem] *= (float)(pressure_change_factor*FacAbun);
			for( ion=0; ion < (nelem + 2); ion++ )
			{
				/* >>chng 96 jul 12 had only multiplied total abun, not ions */
				dense.xIonDense[nelem][ion] *= (float)(pressure_change_factor*FacAbun);
			}
		}

		dense.eden *= pressure_change_factor;

		/* >>chng 96 oct 25 update some te-ne variables */
		tfidle(FALSE);
		/*dense.hden *= pressure_change_factor;*/
		/*TODO when this is confirmed good, start removing dense.hden and replace 
		 * with gas_phase[ipH] - rm above */
		/*ASSERT( fabs(dense.hden-dense.gas_phase[ipHYDROGEN])/dense.hden < 3e-5 );*/

		/* molecules done in hmole */
		for (mol = 0; mol < N_H_MOLEC; mol++) 
			hmi.Molec[mol] *= (float)pressure_change_factor;
		hmi.htwo_total *= (float)pressure_change_factor;

		/* molecules done in comole */
		for( ion=0; ion < NHEVML; ion++ )
		{
			hevmolec.hevmol[ion] *= (float)FacAbun;
		}
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
		  " PressureChange called, changing HDEN from %10.3e to %10.3e Set fill fac to %10.3e\n", 
		  hold, dense.gas_phase[ipHYDROGEN], geometry.FillFac );

		if( trace.lgNeBug )
		{
			fprintf( ioQQQ, " EDEN change PressureChange from to %10.3e %10.3e %10.3e\n", 
			  edensave, dense.eden, edensave/dense.eden );
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (nzone>215)/**/ )
		{
			fprintf( ioQQQ, 
				"%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%c\n", 
			  radius.depth, 
			  pressure.PresTotlCurr, 
			  pressure.PresTotlInit + pressure.PresInteg, 
			  pressure.PresTotlInit, 
			  pressure.PresGasCurr, 
			  pressure.PresRamCurr, 
			  pressure.PresRadCurr, 
			  /* subtract continuum rad pres which has already been added on */
			  pressure.PresInteg - pressure.pinzon, 
			  wind.windv/1e5,
			  sqrt(5.*pressure.PresGasCurr/3./phycon.xMassDensity)/1e5,
			  TorF(conv.lgConvPres) );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->PressureChange()\n", debug_fp );
#	endif
	return lgChange;
}

/* returns true if pressure is converged */
int lgConvPres(void)
{

	static double pold = 0.;

	double windnw,
	  dnew, 
	  term;
	int lgRet;

	/* make sure this is set by one of the following branches - set to zero here, 
	 * then assert that it is greater than zero at end */
	pressure.PresTotlCorrect = 0.;

	/* evaluate a series of possible pressure options, and set the file static variable
	 * pressure_change_factor */
	/* inside out globule */
	if( strcmp(dense.chDenseLaw,"GLOB") == 0 )
	{
		/* GLBDST is distance from globule, or glbrad-DEPTH */
		if( radius.glbdst < 0. )
		{
			fprintf( ioQQQ, " Globule distance is negative, internal overflow has occured,  sorry.\n" );
			fprintf( ioQQQ, " This is routine PressureChange, GLBDST is%10.2e\n", 
			  radius.glbdst );
			puts( "[Stop in PressureChange]" );
			cdEXIT(EXIT_FAILURE);
		}
		pressure_change_factor = (radius.glbden*pow(radius.glbrad/(radius.glbdst),radius.glbpow))/
		  dense.gas_phase[ipHYDROGEN];
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	/* this is positive wind velocity the outflowing wind beyond sonic point */
	else if( (strcmp(dense.chDenseLaw,"WIND") == 0) && wind.windv > 0. )
	{

		static double rsave = 0.;
		/* this is logic for "old" wind solution,
		 * which assumes positive velocity, well above sonic point.
		 * following makes sure wind v only updated once per zone */
		/*lint -e777 float test equality */
		if( radius.depth != rsave )
		/*lint +e777 float test equality */
		{
			rsave = radius.depth;
			/* evaluate radiative acceleration */
			radacl();
			/* Wind model
			 * G, COMASS = mass of star in solar masses */
			wind.agrav = (float)((6.67e-8*wind.comass)*(SOLAR_MASS/radius.Radius)/
			  radius.Radius);

			/* is this the first zone? */
			if( nzone > 2 )
			{
				/* acceleration due to grad P(rad), xMassDensity is gm per cc */
				wind.AccelPres = (float)(-(pressure.PresRadCurr - pold)/radius.drad/
				  phycon.xMassDensity);
			}
			else
			{
				wind.AccelPres = 0.;
			}

			/* this is numerically unstable */
			wind.AccelPres = 0.;

			/* AccelXXX is computed in radinc, is continuum and line acceleration
			 * AccelCont is continuum accel 
			 * AccelLine is line accel 
			 * AccelPres is gradient of local gas pressure */
			wind.AccelTot = wind.AccelCont + wind.AccelLine + wind.AccelPres;

			/* remember largest radiative acceleration */
			wind.AccelMax = (float)MAX2(wind.AccelMax,wind.AccelTot);

			/* keep track of average acceleration */
			wind.AccelAver += wind.AccelTot*(float)radius.drad_x_fillfac;
			wind.acldr += (float)radius.drad_x_fillfac;

			/* following is form of energy equation, only used to check if neg vel */
			term = POW2(wind.windv) + 2.*(wind.AccelTot - wind.agrav)*
			  radius.drad;

			/* remember the current wind velocity
			vold = wind.windv; */
			if( term <= 1e3 )
			{
				/* wind vel is way below sonic point, give up, do not change vel */
				wind.lgVelPos = FALSE;
			}
			else
			{
				/* square of new wind velocity for outer edge of this zone */
				windnw = (double)(wind.windv*wind.windv) + (double)(2.*
				  (wind.AccelTot-wind.agrav))*radius.drad;

				/* WINDV is velocity at OUTER edge of this zone */
				wind.windv = (float)sqrt(windnw);
				wind.lgVelPos = TRUE;
			}

			/* following needed for expansion cooling */
			dndt.dDensityDT = (float)(wind.AccelTot/wind.windv + 2.*wind.windv/
			  radius.Radius);

			/* following used to control new DRAD
			wind.dVeldRad = (float)(fabs(wind.windv-vold)/wind.windv/radius.drad); */

		}
		else
		{
			pressure_change_factor = 1.;
		}

		pressure_change_factor = wind.emdot/(wind.windv*dense.gas_phase[ipHYDROGEN])/radius.r1r0sq;
		pold = pressure.PresRadCurr;
		pressure.PresTotlCorrect = pressure.PresTotlCurr * pressure_change_factor;
	}

	/* this is negative wind velocity the new dynamics */
	else if( (strcmp(dense.chDenseLaw,"WIND") == 0) && wind.windv < 0. )
	{
		/* sets pressure.PresTotlCorrect  */
		pressure_change_factor = DynaPresChngFactor();
	}

	/* this is impossible - wind with identically zero velocity */
	else if( (strcmp(dense.chDenseLaw,"WIND") == 0) && wind.windv == 0. )
	{
		/* declare insanity  */
		fprintf( ioQQQ, " PROBLEM WIND called with zero velocity - this is impossible.\n Sorry.\n" );
		/* TotalInsanity announces fatal problem, ShowMe, then cdEXIT with failure */
		TotalInsanity();
	}

	else if( strcmp(dense.chDenseLaw,"SINE") == 0 )
	{
		/* rapid density fluctuation */
		pressure_change_factor = (fluct.cfirst*cos(radius.depth*fluct.flong+fluct.flcPhase) + 
		  fluct.csecnd)/dense.gas_phase[ipHYDROGEN];
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	else if( strcmp(dense.chDenseLaw,"POWR") == 0 )
	{
		/* power law function of radius */
		dnew = dense.den0*pow(radius.Radius/radius.rinner,(double)dense.DensityPower);
		pressure_change_factor = dnew/dense.gas_phase[ipHYDROGEN];
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	else if( strcmp(dense.chDenseLaw,"POWD") == 0 )
	{
		/* power law function of depth */
		dnew = dense.den0*pow(1. + radius.depth/dense.rscale,(double)dense.DensityPower);
		pressure_change_factor = dnew/dense.gas_phase[ipHYDROGEN];
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	else if( strcmp(dense.chDenseLaw,"POWC") == 0 )
	{
		/* power law function of column density */
		dnew = dense.den0*pow(1.f + colden.colden[ipCOLUMNDENSITY]/
		  dense.rscale,dense.DensityPower);
		pressure_change_factor = dnew/dense.gas_phase[ipHYDROGEN];
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	else if( strcmp(dense.chDenseLaw,"CPRE") == 0 )
	{
		/* constant pressure */
		if( pressure.lgContRadPresOn )
		{
			/* >>chng 01 oct 31, replace pneed with CorretPressure */
			/* this has pressure due to incident continuum */
			pressure.PresTotlCorrect = pressure.PresTotlInit + pressure.PresInteg;
		}
		else
		{
			/* this does not have pressure due to incident continuum*/
			pressure.PresTotlCorrect = pressure.PresTotlInit*
				/* following term normally unity, power law set with option par on cmmnd*/
				pow(radius.Radius/radius.rinner,(double)pressure.PresPowerlaw);
		}

		/* ratio of correct to current pressures */
		pressure_change_factor = pressure.PresTotlCorrect / pressure.PresTotlCurr;
	}

	else if( strncmp( dense.chDenseLaw ,"DLW" , 3) == 0 )
	{
		if( strcmp(dense.chDenseLaw,"DLW1") == 0 )
		{
			/* call ACF sub */
			pressure_change_factor = fabden(radius.Radius,radius.depth)/dense.gas_phase[ipHYDROGEN];
		}
		else if( strcmp(dense.chDenseLaw,"DLW2") == 0 )
		{
			/* call table interpolation subroutine
			 * >>chng 96 nov 29, added tabden */
			pressure_change_factor = tabden(radius.Radius,radius.depth)/dense.gas_phase[ipHYDROGEN];
		}
		else
		{
			fprintf( ioQQQ, " Insanity, PressureChange gets chCPres=%4.4s\n", 
			  dense.chDenseLaw );
			puts( "[Stop in PressureChange]" );
			cdEXIT(EXIT_FAILURE);
		}
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
		if( pressure.PresTotlCorrect > 3. || pressure.PresTotlCorrect< 1./3 )
		{
			static int lgWARN2BIG=FALSE;
			if( !lgWARN2BIG )
			{
				lgWARN2BIG = TRUE;
				fprintf(ioQQQ,"\n\n >========== Warning!  The tabulated or functional change in density as a function of depth was VERY large. This is zone %li.\n",nzone);
				fprintf(ioQQQ," >========== Warning!  This will cause convergence problems. \n");
				fprintf(ioQQQ," >========== Warning!  The current radius is %.3e. \n",radius.Radius);
				fprintf(ioQQQ," >========== Warning!  The current depth is %.3e. \n",radius.depth);
				fprintf(ioQQQ," >========== Warning!  Consider using a more modest change in density vs radius. \n\n\n");
			}
		}
	}

	else if( strcmp(dense.chDenseLaw,"CDEN") == 0 )
	{
		/* this is the default, constant density */
		pressure_change_factor = 1.;
		pressure.PresTotlCorrect = pressure.PresTotlCurr*pressure_change_factor;
	}

	else
	{
		fprintf( ioQQQ, " Unknown pressure law=%s= This is a major internal error.\n", 
		  dense.chDenseLaw );
		puts( "[Stop in PressureChange]" );
		ShowMe();
		cdEXIT(EXIT_FAILURE);
	}

	/* one of the branches above must have reset this variable,
	 * and it was init to 0 at start.  Confirm that non-zero */
	ASSERT( pressure.PresTotlCorrect > FLT_MIN );

	/* now see whether current pressure is within error bounds */
	if( pressure_change_factor > 1. + conv.PressureErrorAllowed || pressure_change_factor < 1. - conv.PressureErrorAllowed )
	{
		lgRet = FALSE;
		conv.lgConvPres = FALSE;
	}
	else
	{
		lgRet = TRUE;
		conv.lgConvPres = TRUE;
	}

	return lgRet;
}
