/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*
 * physical constants used by Cloudy, taken from
 * >>refer	phys	const	Mohr P.J. & Taylor B.N., 1998 Codata, see Reviews of Modern Physics, Vol. 72, No. 2, 2000
 * >>refercon	or http://www.physics.nist.gov/constants
 *
 * NB - these are all printed with the "print constants" command, 
 * which is in parseprint.c, so any new constants
 * added here must also be added to the prt_constants routine  
 */
/*#include "physconst.h"*/

/*********************************************************************
 * first come math constants                                         *
 *********************************************************************/

/* the number zero */
# define ZerO		0.0

/* the number e */
#define E	2.7182818284590452353602874

/* pi */
#define	PI	3.141592653589793238462643

/* 2*pi */
#define	PI2	6.283185307179586476925287

/* 4*pi */
#define	PI4	12.56637061435917295385057

/* 8*pi */
#define	PI8	25.13274122871834590770115

/* sqrt(pi) */
#define	SQRTPI	1.772453850905516027298167

/* ln(2) */
#define LN_TWO  0.6931471805599453094172321

/* ln(10) */
#define LN_TEN  2.3025850929940456840179915

/* factor that converts optical depth into extinction in mags,
 * 2.5 log e */
#define OPTDEP2EXTIN	1.0857362048

/*********************************************************************
 * astronomical constants go here                                    *
 *********************************************************************/

/* solar mass in grams */
#define	SOLAR_MASS	1.9892e33

/* parsec in cm */
#define	PARSEC	        3.085678e18

/*********************************************************************
 * fundamental constants go next, eventually rest should be defined  *
 * in terms of these                                                 *
 *********************************************************************/

/* atomic mass unit, grams */
#define	ATOMIC_MASS_UNIT    1.66053873e-24

/* electron mass, grams */
#define	ELECTRON_MASS	    9.10938188e-28

/* proton mass, grams */
#define	PROTON_MASS	    1.67262158e-24

/* this is the Boltzmann factor, erg/K */
#define	BOLTZMANN	    1.3806503e-16

/* speed of light, cm/s */
#define	SPEEDLIGHT	    2.99792458e10

/* Planck's constant */
#define	HPLANCK	            6.62606876e-27

/* elementary charge, in C in SI units, to use this must convert to cgs */
#define	ELEM_CHARGE	         1.602176462e-19

/* infinite mass rydberg constant, in cm^-1 */
#define	RYD_INF	             1.0973731568549e5

/* ionization potential of real hydrogen atom, in inf mass ryd, based on Codata 98,
 * uncertainty 24e-12,  calculated by Peter van Hoof */
#define	HIONPOT	            0.999466508219

/*********************************************************************
 * below here should be derived constants                            *
 *                                                                   *
 * NB - explicit values in comments are approximate                  *
 *      and are not maintained !                                     *
 *********************************************************************/

/* h/2pi = 1.05457e-27 */
#define	H_BAR	               (HPLANCK/(2.*PI))

/* elementary charge, in ESU */
#define	ELEM_CHARGE_ESU	         (ELEM_CHARGE*SPEEDLIGHT/10.)

/* electric constant, in F/m, 8.854e-12 */
#define	ELECTRIC_CONST	      (1.e11/PI4/SPEEDLIGHT/SPEEDLIGHT)

/* this is the factor that appears in front of boltzmann factor to get
 * LTE level populations for hydrogenic ions.  It is given in the
 * first parts of section 5 of part 2 of hazy, and is actually
 * ( planck^2 / (2 pi m_e k ) )^3/2, but cannot evaluate powers here,
 * must raise this to 3/2 when used, HION_LTE_POP = 5.556e-11  */
#define	HION_LTE_POP	    (HPLANCK*HPLANCK/2./PI/BOLTZMANN/ELECTRON_MASS)

/* SAHA2 is ( h^2/2/pi/m/k )^3, is correct constant for free electron
 * NB - the exponent should actually be 3/2, so take sqrt(SAHA2) before use !!
 * SAHA = 4.141e-16, SAHA2 = 1.7151e-31 */
#define	SAHA2	               (HION_LTE_POP*HION_LTE_POP*HION_LTE_POP)

/* number of ergs per wavenumber, 1.9864e-16 */
#define	ERG1CM	            (HPLANCK*SPEEDLIGHT)

/* degrees kelvin per unit wavenumber, 1.4388 */
#define	T1CM	            (HPLANCK*SPEEDLIGHT/BOLTZMANN)

/* number of Ryd per wavenumber, 9.11267e-6 */
#define	WAVNRYD	            (1./RYD_INF)

/* Angstrom per infinite mass Ryd, 911.2671 */
#define	RYDLAM	            (1.e8/RYD_INF)

/* ergs per inf mass Ryd, 2.180e-11 */
#define	EN1RYD	            (HPLANCK*SPEEDLIGHT*RYD_INF)

/* the temperature of 1 Rydberg */
/* te1ryd is h/k is temp of 1 Rydberg, 1.579e5 */
#define	TE1RYD	            (HPLANCK*SPEEDLIGHT*RYD_INF/BOLTZMANN)

/* Kelvins per eV, 1.1604e4 */
#define	EVDEGK	            (ELEM_CHARGE*1.e7/BOLTZMANN)

/* eV per inf mass Ryd, 13.606 */
#define	EVRYD	            (HPLANCK*SPEEDLIGHT*RYD_INF/ELEM_CHARGE*1.e-7)

/* ergs per eV, 1.602176e-012 */
#define	EN1EV				(EN1RYD/EVRYD)

/* frequency of one Ryd for infinite mass nuclei, 3.289842e15 */
#define	FR1RYD	            (SPEEDLIGHT*RYD_INF)

/*2 h FR1RYD^3 / c^2 for infinite mass nucleus, 0.5250 */
#define	HNU3C2	            (2.*HPLANCK*SPEEDLIGHT*RYD_INF*RYD_INF*RYD_INF)

/* frequency of ionization potential of H (not inf mass), 3.288087e15 - never used */
#define	FR1RYDHYD	           (SPEEDLIGHT*RYD_INF*HIONPOT)

/* H_BAR in eV sec */
#define HBAReV					(H_BAR/EN1EV)       

/* wavelength (A) of ionization potential of Hydrogen, 911.7535 - never used */
#define	RYDLAMHYD	           (RYDLAM/HIONPOT)

/* Stefan-Boltzmann constant, 5.6704e-5,
 * do not use STEFAN_BOLTZMANN to avoid clash with BOLTZMANN above ! */
#define	STEFAN_BOLTZ	        (PI*PI/60.*BOLTZMANN*BOLTZMANN*BOLTZMANN*BOLTZMANN/H_BAR/H_BAR/H_BAR/SPEEDLIGHT/SPEEDLIGHT)

/* the frequency of one eV */
#define	FREQ_1EV	            ( SPEEDLIGHT*RYD_INF/EVRYD)

/* the fine-structure constant a= 2pi e^2/hc 7.297 352 533 x 10-3 */
#define FINE_STRUCTURE	(ELEM_CHARGE_ESU*ELEM_CHARGE_ESU/SPEEDLIGHT/H_BAR)

/* the square of the fine-structure constant */
#define FINE_STRUCTURE2	(FINE_STRUCTURE*FINE_STRUCTURE)

/* the two photon constant as defined by Breit & Teller, as in equation 4 of Spitzer & Greenstein 51 */
#define	TWO_PHOT_CONST	(9.*FINE_STRUCTURE2*FINE_STRUCTURE2*FINE_STRUCTURE2*FR1RYD/(2.*2.*2.*2.*2.*2.*2.*2.*2.*2.*2.))

