/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*ParseGrain parse parameters on grains command */
#include "cddefines.h"
#include "grainvar.h"
#include "physok.h"
#include "input.h"
#include "varypar.h"
#include "parse.h"
#include "mie.h"

void ParseGrain(char *chCard, 
		int *lgDSet)
{
	int lgEOL,
	  lgLinSet, 
	  lgLogLinSet, 
	  lgQuoteFound,
	  lgSizeDistribution;
	GrainPar gp;

	/*possible name of input file with opacities */
	char chFile[FILENAME_PATH_LENGTH_2];
	char *chOption;
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>ParseGrain()\n", debug_fp );
#	endif

	*lgDSet = TRUE;

	/* >>chng 00 dec 20, initialize chFile to empty string..., PvH */
	chFile[0] = '\0';
	chOption = "";
	/* >>chng 01 jan 17, read filename first, it may contain digits that would upset FFmtRead,
	 * as well as other tests that will follow. GetQuote will erase the filename from chCard, PvH */
	lgQuoteFound = strchr( input.chOrgCard,'\"' ) != NULL;
	if( lgQuoteFound )
	{
		/* 
		 * this will both scan in whatever label is inside the quotes in OrgCard, 
		 * but also remove the contents there and in chCard,
		 * so that following keywords will not trigger off it
		 */
		GetQuote( chFile , chCard , TRUE );
	}

	gp.lgGreyGrain = lgMatch("GREY",chCard) || lgMatch("GRAY",chCard) || lgMatch("grey_",chFile) == 1;
		
	/* now check for the keyword "function" - this is a special case,
	 * where grain abundance varies with depth */
	gp.lgAbunVsDepth = lgMatch("FUNC",chCard);

	/* check for the keyword "single bin" - 
	 * use resolved grain size distributions if not present */
	lgSizeDistribution = ! lgMatch("SING",chCard);

	/* check for the keyword "qheating" - 
	 * quantum heating should be turned on */
	gp.lgForbidQHeating = FALSE;
	gp.lgRequestQHeating = lgMatch("QHEA",chCard);

	/* the keyword "no qheat" always takes precedence */
	if( lgMatch("O QH",chCard) ) 
	{
		gp.lgForbidQHeating = TRUE;
		gp.lgRequestQHeating = FALSE;
		physok.lgPhysOK = FALSE;
	}
	
	/* option to force constant reevaluation of grain physics - reevaluate */
	gv.lgReevaluate = lgMatch("REEV",chCard);

	/* option to turn off photoelectric heating by grain, NO HEATING */
	if( lgMatch("O HE",chCard) )
	{
		physok.lgPhysOK = FALSE;
		gv.lgDHetOn = FALSE;
	}

	/* option to turn off gas cooling by grain, NO COOLING */
	if( lgMatch("O CO",chCard) )
	{
		physok.lgPhysOK = FALSE;
		gv.lgDColOn = FALSE;
	}


	/* log - linear option for grain abundance */
	lgLogLinSet = FALSE;
	lgLinSet = FALSE;
	if( lgMatch(" LOG",chCard) )
	{
		lgLogLinSet = TRUE;
		lgLinSet = FALSE;
	}
	else if( lgMatch("LINE",chCard) )
	{
		lgLogLinSet = TRUE;
		lgLinSet = TRUE;
	}

	/* get the grain abundance as the first parameter,
	 * returns 0 if no number, ok since interpreted as log, or unity*/
	i = 5;
	gp.dep = FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* was keyword log or linear on the line? */
	if( lgLogLinSet )
	{
		/* log or linear was specified, which was it */
		if( lgLinSet )
		{
			/* linear quantity entered, check it */
			if( gp.dep <= 0. )
			{
				fprintf( ioQQQ, " Impossible value for linear abundance.\n" );
				fprintf( ioQQQ, " Abundance entered was%10.2e\n", gp.dep );
				fprintf( ioQQQ, " Sorry.\n" );
				puts( "[Stop in ParseGrain]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
		else
		{
			gp.dep = pow(10.,gp.dep);
		}
	}
	else
	{
		/* neither log nor linear specified, check sign
		 * force it to be a log - linear if greater than 0 */
		if( gp.dep <= 0. )
		{
			gp.dep = pow(10.,gp.dep);
		}
	}

	if( gp.dep < FLT_MIN )
	{
		fprintf( ioQQQ, " Grain abundance entered here (%f) is impossible.\n", gp.dep );
		puts( "[Stop in ParseGrain]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* it is possible that there is a file name on the command line - 
	 * if so then we want to call correct reoutine, and not look for keywords 
	 * the signature of a keyword is a pair of quotes - is one present?
	 */
	if( lgQuoteFound )
	{
		/* this is the new code for many-sized grains */
		mie_read_opc(chFile,gp);
	}
	else
	{
		if( lgMatch("ORIO",chCard) )
		{
			/* optional keyword ORION to use orion curves for large R grains */
			/* only turn on one if graphite or silicate is on line, both if not */
			if( lgMatch("GRAP",chCard) )
			{
				/* only turn on orion graphite */
				chOption = "ORION GRAPHITE ";
				if( lgSizeDistribution )
				{
					mie_read_opc("graphite_orion_10.opc",gp);
				}
				else
				{
					mie_read_opc("graphite_orion_01.opc",gp);
				}
			}
			else if( lgMatch("SILI",chCard) )
			{
				/* only turn on orion silicate */
				chOption = "ORION SILICATE ";
				if( lgSizeDistribution )
				{
					mie_read_opc("silicate_orion_10.opc",gp);
				}
				else
				{
					mie_read_opc("silicate_orion_01.opc",gp);
				}
			}
			else
			{
				/* turn both on */
				chOption = "ORION ";
				if( lgSizeDistribution )
				{
					mie_read_opc("graphite_orion_10.opc",gp);
					mie_read_opc("silicate_orion_10.opc",gp);
				}
				else
				{
					mie_read_opc("graphite_orion_01.opc",gp);
					mie_read_opc("silicate_orion_01.opc",gp);
				}
			}
		}

		else if( lgMatch(" PAH",chCard) )
		{
			/* there is a small and large pah */
			chOption = "PAH ";
			mie_read_opc("pah1_0n341.opc",gp);
			mie_read_opc("pah1_0n682.opc",gp);
		}

		else if( lgMatch("GREY",chCard) || lgMatch("GRAY",chCard) )
		{
			/* grey grains */
			chOption = "GREY ";
			if( lgSizeDistribution )
			{
				mie_read_opc("grey_ism_10.opc",gp);
			}
			else
			{
				mie_read_opc("grey_ism_01.opc",gp);
			}
		}

		else if( lgMatch(" ISM",chCard) )
		{
			if( lgMatch("GRAP",chCard) )
			{
				/* only turn on ism graphite */
				chOption = "ISM GRAPHITE ";
				if( lgSizeDistribution )
				{
					mie_read_opc("graphite_ism_10.opc",gp);
				}
				else
				{
					mie_read_opc("graphite_ism_01.opc",gp);
				}
			}
			else if( lgMatch("SILI",chCard) )
			{
				/* only turn on orion silicate */
				chOption = "ISM SILICATE ";
				if( lgSizeDistribution )
				{
					mie_read_opc("silicate_ism_10.opc",gp);
				}
				else
				{
					mie_read_opc("silicate_ism_01.opc",gp);
				}
			}
			else
			{
				/* turn both ISM graphite and silicate on */
				chOption = "ISM ";
				if( lgSizeDistribution )
				{
					mie_read_opc("graphite_ism_10.opc",gp);
					mie_read_opc("silicate_ism_10.opc",gp);
				}
				else
				{
					mie_read_opc("graphite_ism_01.opc",gp);
					mie_read_opc("silicate_ism_01.opc",gp);
				}
			}
		}


		/* default case */
		else
		{
			/* turn both ISM graphite and silicate on */
			if( lgSizeDistribution )
			{
				mie_read_opc("graphite_ism_10.opc",gp);
				mie_read_opc("silicate_ism_10.opc",gp);
			}
			else
			{
				mie_read_opc("graphite_ism_01.opc",gp);
				mie_read_opc("silicate_ism_01.opc",gp);
			}
		}
	}

	/* vary option */
	if( VaryPar.lgVarOn )
	{
		char chHold[INPUT_LINE_LENGTH];
		/*  pointer to where to write */
		VaryPar.nvfpnt[VaryPar.nparm] = input.nRead;
		VaryPar.vparm[0][VaryPar.nparm] = (float)log10(gp.dep);
		VaryPar.vincr[VaryPar.nparm] = 1.;

		/* these are varios options for density laws, 
		 * first is constant density or pressre*/
		strcpy( VaryPar.chVarFmt[VaryPar.nparm] , "GRAIN ABUND=%f LOG ");
		/* >>chng 00 dec 20, chFile may not be defined..., PvH */
		if( chFile[0] != '\0' )
		{
			sprintf(chHold , "\"%s\" ", chFile  );
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , chHold );
		}
		strcat( VaryPar.chVarFmt[VaryPar.nparm] , chOption );
		if( gp.lgAbunVsDepth )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "FUNCTION " );
		if( !lgSizeDistribution )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "SINGLE " );
		if( gp.lgForbidQHeating )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "NO QHEAT " );
		else if( gp.lgRequestQHeating )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "QHEAT " );
		if( gv.lgReevaluate )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "REEVALUATE " );
		if( !gv.lgDHetOn )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "NO HEATING " );
		if( !gv.lgDColOn )
			strcat( VaryPar.chVarFmt[VaryPar.nparm] , "NO COOLING " );

		VaryPar.nvarxt[VaryPar.nparm] = 1;
		++VaryPar.nparm;
	}

#	ifdef DEBUG_FUN
	fputs( " <->ParseGrain()\n", debug_fp );
#	endif
	return;
}

