/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*HydroT2Low called to do hydrogenic level populations when temp too low for matrix */
#include "cddefines.h"
#include "taulines.h"
#include "hydrogenic.h"
#include "iso.h"
#include "secondaries.h"
#include "ionrec.h"
#include "dense.h"
#include "trace.h"
#include "phycon.h"
#include "chargtran.h"
/*lint -e662 out of bounds pointer */

void HydroT2Low(long int nelem, 
  double rfac)
{
	long int iup, 
	  level, 
	  low;
	double above, 
	  below, 
	  Recom3Body,
	  TotIonizRate;

#	ifdef DEBUG_FUN
	fputs( "<+>HydroT2Low()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	ASSERT( nelem >= 0);
	ASSERT( nelem < LIMELM );

	/* changed to EdenHCorr
	 * edhi = eden + hi * 1.7e-4
	 * */
	if( trace.lgHBug && trace.lgTrace )
	{
		if( rfac < 1e-7 )
		{
			fprintf( ioQQQ, " H%3ld matrix not used, H too neutral\n", nelem );
		}
		if( !iso.lgPopLTE_OK[ipH_LIKE][nelem] )
		{
			fprintf( ioQQQ, " H%3ld matrix not used, Hlte not ok\n", nelem );
		}
	}

	/* do radiative only cascades */
	Recom3Body = 0.;
	ionrec.TotRecomRate[nelem][nelem] = 0.;
	for( level=iso.numLevels[ipH_LIKE][nelem]-1; level >= ipH2s; level-- )
	{
		/* radiative rec plus three body rec */
		above = (iso.RadRecomb[ipH_LIKE][nelem][level][ipRecRad]*
			iso.RadRecomb[ipH_LIKE][nelem][level][ipRecNetEsc] + 
		   iso.ColIoniz[ipH_LIKE][nelem][level]*iso.PopLTE[ipH_LIKE][nelem][level]*phycon.EdenHCorr)*
	  	   dense.eden;

		/* keep track of three body recom */
		Recom3Body += iso.ColIoniz[ipH_LIKE][nelem][level]*iso.PopLTE[ipH_LIKE][nelem][level]*phycon.EdenHCorr;

		ionrec.TotRecomRate[nelem][nelem] += above;

		for( iup=level + 1; iup < iso.numLevels[ipH_LIKE][nelem]; iup++ )
		{
			double RadDecay;
			RadDecay =
				EmisLines[ipH_LIKE][nelem][iup][level].Aul*
			  (EmisLines[ipH_LIKE][nelem][iup][level].Pesc + 
			  EmisLines[ipH_LIKE][nelem][iup][level].Pelec_esc + 
			  EmisLines[ipH_LIKE][nelem][iup][level].Pdest);

			/* hlev(level,low) is A * esc prob */
			above += iso.Pop2Ion[ipH_LIKE][nelem][iup]*(RadDecay + 
			  EmisLines[ipH_LIKE][nelem][iup][level].ColUL*phycon.EdenHCorr);
		}

		below = 0.;
		for( low=ipH1s; low <= (level - 1); low++ )
		{
			double RadDecay;
			RadDecay =
				EmisLines[ipH_LIKE][nelem][level][low].Aul*
			  (EmisLines[ipH_LIKE][nelem][level][low].Pesc + 
			  EmisLines[ipH_LIKE][nelem][level][low].Pelec_esc + 
			  EmisLines[ipH_LIKE][nelem][level][low].Pdest);

			below += RadDecay + EmisLines[ipH_LIKE][nelem][level][low].ColUL*
			  phycon.EdenHCorr;
		}
		if( below > 1e-25 )
		{
			iso.Pop2Ion[ipH_LIKE][nelem][level] = (float)(above/below);
		}
		else
		{
			iso.Pop2Ion[ipH_LIKE][nelem][level] = 0.;
		}

		if( iso.PopLTE[ipH_LIKE][nelem][level] > 1e-25 )
		{
			iso.DepartCoef[ipH_LIKE][nelem][level] = 
				iso.Pop2Ion[ipH_LIKE][nelem][level]/(iso.PopLTE[ipH_LIKE][nelem][level]*
			  dense.eden);
		}
		else
		{
			iso.DepartCoef[ipH_LIKE][nelem][level] = 0.;
		}
	}
	/* fraction of total recombs due to three body - when most are due to this
	 * small changes in temperature can produce large changes in rec coef,
	 * and so in ionization */
	iso.RecomCollisFrac[ipH_LIKE][nelem] = (float)(Recom3Body* dense.eden / ionrec.TotRecomRate[nelem][nelem] );

	/* total ionization rate */
	TotIonizRate = iso.gamnc[ipH_LIKE][nelem][ipH1s] + Secondaries.csupra + iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*
	  phycon.EdenHCorr;

	/* following is total rec coef */
	ionrec.TotRecomRate[nelem][nelem] = dense.eden * (
		iso.RadRecomb[ipH_LIKE][nelem][ipH1s][ipRecRad]*
	  iso.RadRecomb[ipH_LIKE][nelem][ipH1s][ipRecNetEsc] + iso.RadRec_caseB[ipH_LIKE][nelem] + 
	  ionrec.CotaRate[nelem]);

	/* >>chng 01 may 05, add this in */
	/* add on charge transfer ionization of hydrogen */
	if( nelem == ipHYDROGEN )
	{
		long int ipCollide , ion;
		/* add on charge transfer ionization of hydrogen,
		 * recombination for other element is ionization of hydrogen */
		ChargTran.HCharExcIonTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			for( ion=2; ion<=ipCollide+2; ++ion )
			{
				/* charge transfer ionization of H, recombination for other species */
				ChargTran.HCharExcIonTotal += 
					ChargTran.HCharExcRec[ipCollide][ion-2]*dense.xIonDense[ipCollide][ion-1];
			}
		}
		TotIonizRate += ChargTran.HCharExcIonTotal;

		/* >>chng 01 may 07,  add this in */
		/* charge transfer recombination of hydrogen,
		 * which is ionization of the heavy element */
		ChargTran.HCharExcRecTotal = 0.;
		for( ipCollide=1; ipCollide<LIMELM; ++ipCollide)
		{
			/* this is ion on the abundances scale, 1 is atom, so goes up to ipCollide+1,
			 * for helium ipCollide=1, ion must go up to 3 */
			for( ion=1; ion<=ipCollide+1; ++ion )
			{
				/* charge transfer ionization of H, recombination for other species */
				ChargTran.HCharExcRecTotal += 
					ChargTran.HCharExcIon[ipCollide][ion-1]*dense.xIonDense[ipCollide][ion-1];
			}
		}
	}
	ionrec.TotRecomRate[nelem][nelem] += ChargTran.HCharExcRecTotal;

	/* =================================================================== 
	 *
	 * at this point all destruction processes have been established 
	 *
	 * ==================================================================== */

	/* save total ground state destruction rate for possible later use */
	hydro.DestRate[nelem][ipH1s] = TotIonizRate;

	iso.xLevel2Cont[ipH_LIKE][nelem][0] = (float)TotIonizRate;
	for( level=1; level < iso.numLevels[ipH_LIKE][nelem]; level++ )
	{
		iso.xLevel2Cont[ipH_LIKE][nelem][level] = 0.;
	}

	/* do simple two level for hydrogen, then fill in HN(i) for rest
	 * above in below is tot rec */
	iso.Pop2Ion[ipH_LIKE][nelem][ipH1s] = (float)(ionrec.TotRecomRate[nelem][nelem]/TotIonizRate);

	if( iso.PopLTE[ipH_LIKE][nelem][ipH1s] > 1e-25 )
	{
		iso.DepartCoef[ipH_LIKE][nelem][ipH1s] = 
			iso.Pop2Ion[ipH_LIKE][nelem][ipH1s]/(iso.PopLTE[ipH_LIKE][nelem][ipH1s]*
		  dense.eden);
	}
	else
	{
		iso.DepartCoef[ipH_LIKE][nelem][ipH1s] = 0.;
	}

	iso.Pop2Ion[ipH_LIKE][nelem][ipH2s] = 
		(float)((iso.RadRec_caseB[ipH_LIKE][nelem]*0.33*dense.eden + 
	  iso.Pop2Ion[ipH_LIKE][nelem][ipH1s]*Secondaries.Hx12[MIN2(nelem+1,2)-1][ipH2s] + 
	  iso.Pop2Ion[ipH_LIKE][nelem][ipH2p]*
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].ColUL*phycon.EdenHCorr)/
	  (EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].Aul + 
	  EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].ColUL*phycon.EdenHCorr + 
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].ColUL*
	  phycon.EdenHCorr*6.));

	iso.Pop2Ion[ipH_LIKE][nelem][ipH2p] = (float)((iso.RadRec_caseB[ipH_LIKE][nelem]*0.67*dense.eden + 
	  iso.Pop2Ion[ipH_LIKE][nelem][ipH1s]*Secondaries.Hx12[MIN2(nelem+1,2)-1][ipH2p] + 
	  iso.Pop2Ion[ipH_LIKE][nelem][ipH2s]*
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].ColUL*phycon.EdenHCorr*6.)/
	  (
	   EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Aul*
	  (EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pesc + 
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pelec_esc +
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].Pdest)
	  + 
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].ColUL * phycon.EdenHCorr + 
	  EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].ColUL * phycon.EdenHCorr));

	/* were upper n levels set? */
	/* >>chng 02 aug 28, do not set here since set in hydrolevel */
	/*iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 
		1./(iso.Pop2Ion[ipH_LIKE][nelem][ipH1s] + 
		iso.Pop2Ion[ipH_LIKE][nelem][ipH2p] + 
	  iso.Pop2Ion[ipH_LIKE][nelem][ipH2s] + 
	  iso.Pop2Ion[ipH_LIKE][nelem][3] + iso.Pop2Ion[ipH_LIKE][nelem][4] + 
	  iso.Pop2Ion[ipH_LIKE][nelem][5] + iso.Pop2Ion[ipH_LIKE][nelem][6]);*/

	if( trace.lgHBug && trace.lgTrace )
	{
		fprintf( ioQQQ, "       LOW TE,=%10.3e HN(1)=%10.3e rec=%10.3e Hgamnc(1s)=%10.3e\n", 
		  phycon.te, iso.Pop2Ion[ipH_LIKE][nelem][ipH1s], ionrec.TotRecomRate[nelem][nelem], 
		  iso.gamnc[ipH_LIKE][nelem][ipH1s] );
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, 
			"       HydroT2Low return, LOW TE used, HII/HI=%10.3e simple=%10.3e IonRate=%10.2e RecCo=%10.2e\n", 
		  iso.pop_ion_ov_neut[ipH_LIKE][nelem], iso.pop_ion_ov_neut[ipH_LIKE][nelem], 
		  iso.gamnc[ipH_LIKE][nelem][ipH1s] + Secondaries.csupra + iso.ColIoniz[ipH_LIKE][nelem][ipH1s]*
		  phycon.EdenHCorr, ionrec.TotRecomRate[nelem][nelem] );
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroT2Low()\n", debug_fp );
#	endif
	return;
}
/*lint +e662 out of bounds pointer */

