/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*Hydrogenic main routine to call HydroLevel and determine model hydrogen atom level balance */
#include "cddefines.h"
#include "taulines.h"
#include "hydrogenic.h"
#include "iso.h"
#include "trace.h"
#include "secondaries.h"
#include "phycon.h"
#include "bidiag.h"
#include "hmi.h"
#include "h2.h"
#include "chargtran.h"
#include "dense.h"
#include "ionrange.h"
#include "hmole.h"

void Hydrogenic(void)
{
	long int ipLo ,
		ipHi , 
		nelem,
		mol;
	double  coltot, 
	  gamtot;
	double sum , error;

#	ifdef DEBUG_FUN
	fputs( "<+>Hydrogenic()\n", debug_fp );
#	endif

	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* do not consider elements that have been turned off */
		if( dense.lgElmtOn[nelem] )
		{
			/* note that nelem scale is totally on c not physical scale, so 0 is h */
			/* evaluate hydrogenic balance if ionization reaches this high */
			if( (IonRange.IonHigh[nelem] == nelem + 1)  )
			{
				/* evaluate collisional rates */
				HydroCollid(nelem);

				/* evaluate photoionization rates  */
				iso_photo( ipHYDROGEN , nelem );

				/* evaluate recombination rates */
				HydroRecom(nelem);

				/* solve for the level populations */
				HydroLevel(nelem);

				if( trace.lgTrace )
				{
					fprintf( ioQQQ, "     Hydrogenic Z=%2ld H2OVH1=",nelem);
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.pop_ion_ov_neut[ipH_LIKE][nelem]));
					fprintf( ioQQQ, " simple=");
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.pop_ion_ov_neut[ipH_LIKE][nelem]));
					/* fprintf( ioQQQ, " H-ovH1:");
						 fprintf( ioQQQ,PrintEfmt("%10.2e", hmi.hmovh1 ) ); */
					fprintf( ioQQQ, "\n"); 
				}

			}
			else
			{
				iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 0.;
				/* iso.pop_ion_ov_neut[ipH_LIKE][nelem] = 0.; */ /* Duplicate line, why ??? */

				/* zero it out since no population*/
				for( ipHi=ipH2s; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
				{
					for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
					{
						/* population of lower level rel to ion */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopLo = 0.;

						/* population of upper level rel to ion */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi =  0.;

						/* population of lower level rel to ion, corrected for stim em */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc =  0.;
					}
				}
			}
		}
	}

	/* ============================================================================== */
	/* rest is for hydrogen only */

	/* this block appears redundant and could be removed? */
	/* >> 02 nov 21 rjrw -- xIonDense is used in hmole but only set in bidiag, 
	 * so we need this at least for the first iteration. */

#if 0
	/* update neutral and ion densities for current pop ratio and molecules */
	fprintf(ioQQQ,"<<%g %g %g\n",
					(float)(dense.gas_phase[ipHYDROGEN]/(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN])),
					dense.gas_phase[ipHYDROGEN]*iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN]/
					(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN]),
					hmi.hmovh1);
	if(1 || conv.nPres2Ioniz > 0 )
	{
		dense.xIonDense[ipHYDROGEN][1] = (float)(dense.gas_phase[ipHYDROGEN]*iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN]/
			(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + hmi.hmovh1));

		dense.xIonDense[ipHYDROGEN][0] = (float)(dense.gas_phase[ipHYDROGEN]/(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + 
			hmi.hmovh1));
	}
	fprintf(ioQQQ,">>%g %g\n",dense.xIonDense[ipHYDROGEN][0],dense.xIonDense[ipHYDROGEN][1]);
#endif

	/* do molecular balance 
	 * hmovh1 will be ratio of molecular to atomic hydrogen
	 * HIonFrac is fraction of H that is ionized, ratio of ion to atom */
	
	if (0) 
	{
		fprintf(ioQQQ,"Ions into Hmole %g %g\n",dense.xIonDense[ipHYDROGEN][0],dense.xIonDense[ipHYDROGEN][1]); 
	}

	hmole();

	/* >>chng 01 may 09, add option to force abundance, with element name ioniz cmmnd */
	if( dense.lgSetIoniz[ipHYDROGEN] )
	{
		dense.xIonDense[ipHYDROGEN][1] = dense.SetIoniz[ipHYDROGEN][1]*dense.gas_phase[ipHYDROGEN];
		dense.xIonDense[ipHYDROGEN][0] = dense.SetIoniz[ipHYDROGEN][0]*dense.gas_phase[ipHYDROGEN];
	}
	else
	{
		/*float 
		save_gasphase = dense.gas_phase[ipHYDROGEN],
		save_moles = dense.xMolecules[ipHYDROGEN];*/

		/* >> chng 03 jan 15 rjrw: switch this off as now we have updated the atom/ion densities. */
#if 0 
		{
			/* at this stage we have updated the molecular abundances by calling
			 * hmole above, but have not updated the atom/ion fractions, and
			 * these are no longer self-consistent.  The following code
			 * gets sum rule right - all H will be somewhere */
			/* get ionization balance - iso.pop_ion_ov_neut is ratio of H+ to H0 */
			
			dense.xIonDense[ipHYDROGEN][1] = (float)(dense.gas_phase[ipHYDROGEN]*iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN]/
				(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + hmi.hmovh1));
			dense.xIonDense[ipHYDROGEN][0] = (float)(dense.gas_phase[ipHYDROGEN]/(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + 
				hmi.hmovh1));
			
			/*dense.gas_phase[ipHYDROGEN] = dense.xIonDense[ipHYDROGEN][1] + dense.xIonDense[ipHYDROGEN][0];
				dense.xMolecules[ipHYDROGEN] = 0.;*/
		}
#endif

		/* 
		 * >> chng 03 jan 15 rjrw:- terms are now in bidiag, to allow for
		 * molecular sources and sinks of H and H+.  Bidiag renormalizes
		 * to keep the total H abundance correct -- only the molecular
		 * network is allowed to change this.
		 *
		 * */

		BiDiag( ipHYDROGEN , FALSE );

#		if 0
		fprintf(ioQQQ," newww %.3e %.3e ", 
			dense.xIonDense[ipHYDROGEN][0],
			dense.xIonDense[ipHYDROGEN][1] );
#		endif
		/*dense.gas_phase[ipHYDROGEN] = save_gasphase;
		dense.xMolecules[ipHYDROGEN] = save_moles;*/

#if 0
		/* this is redundant with above - surely an oversight since it must
		 * get the same answer */
		/* get ionization balance - iso.pop_ion_ov_neut is ratio of H+ to H0 */
		iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] = dense.xIonDense[ipHYDROGEN][1]/dense.xIonDense[ipHYDROGEN][0];
		dense.xIonDense[ipHYDROGEN][1] = (float)(dense.gas_phase[ipHYDROGEN]*iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN]/
			(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + hmi.hmovh1));

		dense.xIonDense[ipHYDROGEN][0] = (float)(dense.gas_phase[ipHYDROGEN]/(1. + iso.pop_ion_ov_neut[ipH_LIKE][ipHYDROGEN] + 
			hmi.hmovh1));
#endif
	}

	/* confirm that species still add up correctly */
	/* this exposes a "leak" that occurs somewhere, almost certainly in hmole
	 * if DEBUG_LOC is set TRUE in the following there will be comments printing
	 * due to a loss of some protons */

	sum = dense.xIonDense[ipHYDROGEN][0] + dense.xIonDense[ipHYDROGEN][1];
	for (mol=0;mol<N_H_MOLEC;mol++) 
	{
		sum += hmi.Molec[mol]*hmi.protons[mol];
	}
	sum -=  hmi.Molec[ipMH]+hmi.Molec[ipMHp];

	/*ASSERT( fabs(dense.hden-dense.gas_phase[ipHYDROGEN])/dense.hden < 3e-5 );*/
	error = ( dense.gas_phase[ipHYDROGEN] - sum )/dense.gas_phase[ipHYDROGEN];
	{
		/*@-redef@*/
		/* often the H- route is the most efficient formation mechanism for H2,
		 * will be through rate called ratach
		 * this debug print statement is to trace h2 oscillations */
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if(DEBUG_LOC && (fabs(error) > 1e-4) )
			fprintf(ioQQQ,"PROBLEM hydrogenic zone %li hden %.4e, sum %.4e (h-s)/h %.3e \n", nzone, dense.gas_phase[ipHYDROGEN] , sum , 
				error );
	}

	/* now do level populations for H2 */
	H2_LevelPops();

	/* this is test whether collisions are important first define ratio of exits
	 * from ground due to coll, rel to everthing else, then flag is large */
	if( iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH1s] > 1e-30 )
	{
		coltot = 
			EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*
			iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH2p]/iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH1s]*
		  dense.eden*(iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH2p] + iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH2p] + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH2p].ColUL*iso.PopLTE[ipH_LIKE][ipHYDROGEN][3]/iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH2p])/
		  (EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*dense.eden + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Aul*
		  (EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pesc + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pelec_esc +
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest) );

		/* caution that collisions are important (and so only small changes in
		 * temperature should happen) if more than 20% of the total */
		if( coltot > 0.2 )
		{
			hydro.lgHColionImp = TRUE;
		}
	}
	else
	{
		hydro.lgHColionImp = FALSE;
	}

	/* remember the ratio of pops of 2p to 1s for possible printout in prtComment
	 * and to obtain Lya excitation temps.  the pop of ground is not defined if
	 * NO ionization at all since these pops are relative to ion */
	/* >>chng 99 jun 03, added MAX2 to protect against totally neutral gas */
	if( iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p]/MAX2(SMALLFLOAT,iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]) > 0.1 &&
		iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s] > SMALLFLOAT )
	{
		hydro.lgHiPop2 = TRUE;
		hydro.pop2mx = (float)MAX2(iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p]/iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s],
		  hydro.pop2mx);
	}

	gamtot = iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s] + Secondaries.csupra;

	coltot = iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH1s] + 
		EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*4.*
	  iso.Boltzmann[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s];

	/* if ground state destruction rate is significant, recall different dest procceses */
	if( hydro.DestRate[ipHYDROGEN][ipH1s] > 0. )
	{
		hydro.photn2 = 
			(float)(iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s]/hydro.DestRate[ipHYDROGEN][ipH1s] );

		/* fraction of ionizations of H from ground, due to thermal collisions */
		hydro.HCollIonFrac = 
			(float)(iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH1s]*dense.eden/hydro.DestRate[ipHYDROGEN][ipH1s]);

		/* this flag is used in ConvIonizeOpacityDo to decide whether we
		 * really need to converge the secondary ionization rates */
		Secondaries.sec2total = 
			(float)(Secondaries.csupra / hydro.DestRate[ipHYDROGEN][ipH1s]) ;

		/* frac of ionizations due to ct */
		ChargTran.HIonFrac = ChargTran.HCharExcIonTotal / hydro.DestRate[ipHYDROGEN][ipH1s];
	}
	else
	{
		hydro.HCollIonFrac = 0.;
		hydro.photn2 = 0.;
		Secondaries.sec2total = 0. ;
		ChargTran.HIonFrac = 0.;
	}
	/* >>chng 03 mar 15, use hyperfine only */
#if 0
	/* now do 21 cm, will define populations here */
	TauLines[ipH21cm].PopHi = dense.xIonDense[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0]*0.75;	
	TauLines[ipH21cm].PopLo = dense.xIonDense[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0]*0.25;
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipH21cm].PopOpc = dense.xIonDense[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0] * 
		0.25f * TauLines[ipH21cm].EnergyK / phycon.te;

	/* now do 3 cm, the HeII equivalent of H 21 cm will define populations here */
	TauLines[ipHe3cm].PopHi = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*0.75*hyperfine.HFLabundance[2];	
	TauLines[ipHe3cm].PopLo = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*0.25*hyperfine.HFLabundance[2];
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipHe3cm].PopOpc = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*hyperfine.HFLabundance[2] * 
		0.25f * TauLines[ipHe3cm].EnergyK / phycon.te;
#endif

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "       Hydrogenic retrn ");
		fprintf(ioQQQ,"H0:%.2e ", dense.xIonDense[ipHYDROGEN][0]);
		fprintf(ioQQQ,"H+:%.2e ", dense.xIonDense[ipHYDROGEN][1]);
		fprintf(ioQQQ,"H2:%.2e ", hmi.Molec[ipMH2]);
		fprintf(ioQQQ,"ne:%.2e ", dense.eden);
		fprintf( ioQQQ, " REC, COL, GAMT= ");
		/* recomb rate coef, cm^3 s-1 */
		fprintf(ioQQQ,"%.2e ", iso.RadRec_effec[ipH_LIKE][ipHYDROGEN] );
		fprintf(ioQQQ,"%.2e ", coltot);
		fprintf(ioQQQ,"%.2e ", gamtot);
		fprintf( ioQQQ, " CSUP=");
		PrintE82( ioQQQ, Secondaries.csupra);
		fprintf( ioQQQ, " Te=");
		fprintf(ioQQQ,"%.2e ", phycon.te );
		fprintf( ioQQQ, "\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->Hydrogenic()\n", debug_fp );
#	endif
	return;
}
