/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*hpfit state specific photoionization cross sections for model hydrogen atom
 * for any level up through NHYDRO_MAX_LEVEL defined 401L in cddefines.h */
#include "cddefines.h"
#include "hpfit.h"
#include "ph1com.h"
/* NHYDRO_MAX_LEVEL defined 401L in cddefines.h */
static float PHH[NHYDRO_MAX_LEVEL][5];
void hpfitData(void);
void hpfitData1(void);
void hpfitData2(void);
void hpfitData3(void);
void hpfitData4(void);
void hpfitData5(void);
void hpfitData6(void);
void hpfitData7(void);

/* same as hpfit, but energy is relative to threshold */
double hpfit_rel(long int iz, 
	  long int n, 
	  double e)
{
	long m;
	double crs , ex , eth;

	if( n == 0 )
	{
		m = 1;
	}
	else
	{
		if( n == 1 )
		{
			m = 2;
		}
		else
		{
			m = n;
		}
	}

	eth = PH1COM.PH1[0][0][iz-1][0]/POW2((double)m);
	ex = MAX2(1. , e/eth );

	crs = hpfit( iz , n , ex );
	ASSERT( crs > 0. );

	return crs;
}

double hpfit(long int iz, 
	  long int n, 
	  double e)
{
	/* flag saying first call so need to generate data */
	static int lgFIRST=TRUE;
	long int l, 
	  m;
	double cs,
	  eth, 
	  ex, 
	  q, 
	  x;

#	ifdef DEBUG_FUN
	fputs( "<+>hpfit()\n", debug_fp );
#	endif

	/*state specific photoionization cross sections for model hydrogen atom
	 * Version 1, September 23, 1997
	 ******************************************************************************
	 *** This subroutine calculates state-specific photoionization cross sections
	 *** for hydrogen and hydrogen-like ions.
	 *** Input parameters:  iz - atomic number 
	 ***          n  - shell number, from 0 to 400:
	 ***                                    0 - 1s
	 ***                                    1 - 2s
	 ***                                    2 - 2p
	 ***                                    3 - 3 
	 ***                                    ......
	 ***          e  - photon energy, eV
	 *** return value - cross section, cm^(-2)     
	 *******************************************************************************/

	/* on very first call to this routine we need to fill the data arrays
	 * with former contents of the block datas */
	if( lgFIRST )
	{
		lgFIRST = FALSE;
		/* copy the needed data int the arrays */
		hpfitData();
		hpfitData1();
		hpfitData2();
		hpfitData3();
		hpfitData4();
		hpfitData5();
		hpfitData6();
		hpfitData7();
	}

	cs = 0.0;

	ASSERT( iz > 0 && e>0. );
	
	if( n >= NHYDRO_MAX_LEVEL )
	{ 
		fprintf( ioQQQ, " hpfit called with too large n, =%li\n" , n );
		puts( "[Stop in hpfit]" );
		cdEXIT(EXIT_FAILURE);
	}

	l = 0;
	if( n == 2 )
	{
		l = 1;
	}
	q = 3.5 + l - 0.5*PHH[n][1];

	if( n == 0 )
	{
		m = 1;
	}
	else
	{
		if( n == 1 )
		{
			m = 2;
		}
		else
		{
			m = n;
		}
	}

	eth = PH1COM.PH1[0][0][iz-1][0]/POW2((double)m);
	ex = MAX2(1. , e/eth );

	/* Don't just force to be at least one...make sure e/eth is close to one or greater.	*/
	ASSERT( e/eth > 0.95 );

	if( ex < 1.0 )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->hpfit()\n", debug_fp );
#		endif
		return(0.);
	}

	x = ex/PHH[n][0];
	cs = (PHH[n][4]*pow(1.0 + ((double)PHH[n][2]/x),(double)PHH[n][3])/
	  pow(x,q)/pow(1.0 + sqrt(x),(double)PHH[n][1])*8.79737e-17/
	  POW2((double)iz));
	
#	ifdef DEBUG_FUN
	fputs( " <->hpfit()\n", debug_fp );
#	endif
	return(cs);
}

#ifdef _MSC_VER
#	pragma warning( disable : 4305 )/* disable truncation from double to float*/
#endif

/********************************************* */
void hpfitData(void)
{
	int j;
	  { static float _itmp0[] = {3.07626e-01,5.60964e00,9.95690e00,
															2.24237e-01,3.85291e01};
		for(  j = 0; j < 5; j++ )
		{
			PHH[0][j] = _itmp0[j];
			}
		}
		{ static float _itmp1[] = {1.07629e00,6.02640e00,2.38222e00,
		  1.63081e-01,7.69680e00};
		for(  j = 0; j < 5; j++ )
		{
			PHH[1][j] = _itmp1[j];
			}
		}
		{ static float _itmp2[] = {7.59605e-01,7.15690e00,1.36471e00,
		  4.46372e-01,3.42940e01};
		for(  j = 0; j < 5; j++ )
		{
			PHH[2][j] = _itmp2[j];
			}
		}
		{ static float _itmp3[] = {8.03881e00,3.38786e00,8.30480e-01,
		  6.96195e-01,3.81896e-03};
		for(  j = 0; j < 5; j++ )
		{
			PHH[3][j] = _itmp3[j];
			}
		}
		{ static float _itmp4[] = {2.24160e01,2.76959e00,4.91676e-01,
		  5.73670e-01,1.88765e-04};
		for(  j = 0; j < 5; j++ )
		{
			PHH[4][j] = _itmp4[j];
			}
		}
		{ static float _itmp5[] = {4.89715e01,2.40666e00,3.29170e-01,
		  4.83482e-01,1.93150e-05};
		for(  j = 0; j < 5; j++ )
		{
			PHH[5][j] = _itmp5[j];
			}
		}
		{ static float _itmp6[] = {9.00516e01,2.18635e00,2.42373e-01,
		  4.23331e-01,3.33215e-06};
		for(  j = 0; j < 5; j++ )
		{
			PHH[6][j] = _itmp6[j];
			}
		}
		{ static float _itmp7[] = {1.49785e02,2.03097e00,1.87576e-01,
		  3.77563e-01,7.72796e-07};
		for(  j = 0; j < 5; j++ )
		{
			PHH[7][j] = _itmp7[j];
			}
		}
		{ static float _itmp8[] = {2.36282e02,1.90366e00,1.47477e-01,
		  3.36978e-01,2.07509e-07};
		for(  j = 0; j < 5; j++ )
		{
			PHH[8][j] = _itmp8[j];
			}
		}
		{ static float _itmp9[] = {3.51115e02,1.80795e00,1.19663e-01,
		  3.05468e-01,6.65714e-08};
		for(  j = 0; j < 5; j++ )
		{
			PHH[9][j] = _itmp9[j];
			}
		}
		{ static float _itmp10[] = {5.05155e02,1.72688e00,9.81424e-02,
		  2.77588e-01,2.33701e-08};
		for(  j = 0; j < 5; j++ )
		{
			PHH[10][j] = _itmp10[j];
			}
		}
		{ static float _itmp11[] = {6.88866e02,1.66895e00,8.33773e-02,
		  2.57807e-01,9.67881e-09};
		for(  j = 0; j < 5; j++ )
		{
			PHH[11][j] = _itmp11[j];
			}
		}
		{ static float _itmp12[] = {9.24778e02,1.61588e00,7.09266e-02,
		  2.38951e-01,4.17212e-09};
		for(  j = 0; j < 5; j++ )
		{
			PHH[12][j] = _itmp12[j];
			}
		}
		{ static float _itmp13[] = {1.22140e03,1.56826e00,6.05304e-02,
		  2.21567e-01,1.87985e-09};
		for(  j = 0; j < 5; j++ )
		{
			PHH[13][j] = _itmp13[j];
			}
		}
		{ static float _itmp14[] = {1.55181e03,1.53342e00,5.30777e-02,
		  2.09108e-01,9.55475e-10};
		for(  j = 0; j < 5; j++ )
		{
			PHH[14][j] = _itmp14[j];
			}
		}
		{ static float _itmp15[] = {1.91695e03,1.50651e00,4.74662e-02,
		  1.99626e-01,5.29159e-10};
		for(  j = 0; j < 5; j++ )
		{
			PHH[15][j] = _itmp15[j];
			}
		}
		{ static float _itmp16[] = {2.25281e03,1.49288e00,4.42374e-02,
		  1.95646e-01,3.43234e-10};
		for(  j = 0; j < 5; j++ )
		{
			PHH[16][j] = _itmp16[j];
			}
		}
		{ static float _itmp17[] = {2.88811e03,1.45352e00,3.75427e-02,
		  1.79942e-01,1.66104e-10};
		for(  j = 0; j < 5; j++ )
		{
			PHH[17][j] = _itmp17[j];
			}
		}
		{ static float _itmp18[] = {3.36382e03,1.43914e00,3.46821e-02,
		  1.75102e-01,1.09654e-10};
		for(  j = 0; j < 5; j++ )
		{
			PHH[18][j] = _itmp18[j];
			}
		}
		{ static float _itmp19[] = {4.02076e03,1.41813e00,3.11794e-02,
		  1.67104e-01,6.62254e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[19][j] = _itmp19[j];
			}
		}
		{ static float _itmp20[] = {4.34720e03,1.41875e00,3.05827e-02,
		  1.68655e-01,5.52354e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[20][j] = _itmp20[j];
			}
		}
		{ static float _itmp21[] = {5.10349e03,1.39946e00,2.75666e-02,
		  1.61110e-01,3.50689e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[21][j] = _itmp21[j];
			}
		}
		{ static float _itmp22[] = {5.73713e03,1.38962e00,2.58979e-02,
		  1.57673e-01,2.55703e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[22][j] = _itmp22[j];
			}
		}
		{ static float _itmp23[] = {6.32447e03,1.38308e00,2.46845e-02,
		  1.55657e-01,1.98061e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[23][j] = _itmp23[j];
			}
		}
		{ static float _itmp24[] = {6.75642e03,1.38112e00,2.40425e-02,
		  1.55622e-01,1.69170e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[24][j] = _itmp24[j];
			}
		}
		{ static float _itmp25[] = {7.33346e03,1.37528e00,2.30102e-02,
		  1.53756e-01,1.36861e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[25][j] = _itmp25[j];
			}
		}
		{ static float _itmp26[] = {7.96918e03,1.36796e00,2.18658e-02,
		  1.51139e-01,1.09935e-11};
		for(  j = 0; j < 5; j++ )
		{
			PHH[26][j] = _itmp26[j];
			}
		}
		{ static float _itmp27[] = {8.38737e03,1.36654e00,2.14220e-02,
		  1.51153e-01,9.77583e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[27][j] = _itmp27[j];
			}
		}
		{ static float _itmp28[] = {9.30284e03,1.35578e00,2.00133e-02,
		  1.46881e-01,7.33042e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[28][j] = _itmp28[j];
			}
		}
		{ static float _itmp29[] = {1.09155e04,1.33508e00,1.76264e-02,
		  1.38015e-01,4.57661e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[29][j] = _itmp29[j];
			}
		}
		{ static float _itmp30[] = {1.24027e04,1.32101e00,1.60384e-02,
		  1.32113e-01,3.16776e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[30][j] = _itmp30[j];
			}
		}
		{ static float _itmp31[] = {1.29082e04,1.31991e00,1.57688e-02,
		  1.32097e-01,2.89890e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[31][j] = _itmp31[j];
			}
		}
		{ static float _itmp32[] = {1.33691e04,1.31869e00,1.55004e-02,
		  1.31988e-01,2.68842e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[32][j] = _itmp32[j];
			}
		}
		{ static float _itmp33[] = {1.39502e04,1.31601e00,1.51099e-02,
		  1.31186e-01,2.43068e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[33][j] = _itmp33[j];
			}
		}
		{ static float _itmp34[] = {1.51495e04,1.30713e00,1.41522e-02,
		  1.27516e-01,1.93140e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[34][j] = _itmp34[j];
			}
		}
		{ static float _itmp35[] = {1.53501e04,1.30790e00,1.41167e-02,
		  1.28245e-01,1.91250e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[35][j] = _itmp35[j];
			}
		}
		{ static float _itmp36[] = {1.61357e04,1.30466e00,1.37357e-02,
		  1.27105e-01,1.68549e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[36][j] = _itmp36[j];
			}
		}
		{ static float _itmp37[] = {1.80085e04,1.29316e00,1.26205e-02,
		  1.22162e-01,1.22606e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[37][j] = _itmp37[j];
			}
		}
		{ static float _itmp38[] = {1.81497e04,1.29424e00,1.26401e-02,
		  1.22980e-01,1.23130e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[38][j] = _itmp38[j];
			}
		}
		{ static float _itmp39[] = {1.85509e04,1.29336e00,1.24952e-02,
		  1.22879e-01,1.18145e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[39][j] = _itmp39[j];
			}
		}
		{ static float _itmp40[] = {1.91380e04,1.29059e00,1.22009e-02,
		  1.21878e-01,1.09869e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[40][j] = _itmp40[j];
			}
		}
		{ static float _itmp41[] = {1.92786e04,1.29140e00,1.22162e-02,
		  1.22522e-01,1.10236e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[41][j] = _itmp41[j];
			}
		}
		{ static float _itmp42[] = {1.92483e04,1.29283e00,1.22967e-02,
		  1.23437e-01,1.13619e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[42][j] = _itmp42[j];
			}
		}
		{ static float _itmp43[] = {1.93915e04,1.29339e00,1.22994e-02,
		  1.23948e-01,1.13792e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[43][j] = _itmp43[j];
			}
		}
		{ static float _itmp44[] = {1.94684e04,1.29398e00,1.23176e-02,
		  1.24447e-01,1.15087e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[44][j] = _itmp44[j];
			}
		}
		{ static float _itmp45[] = {1.96903e04,1.29314e00,1.22120e-02,
		  1.24285e-01,1.13559e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[45][j] = _itmp45[j];
			}
		}
		{ static float _itmp46[] = {1.98262e04,1.29341e00,1.22096e-02,
		  1.24622e-01,1.13686e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[46][j] = _itmp46[j];
			}
		}
		{ static float _itmp47[] = {2.00316e04,1.29273e00,1.21296e-02,
		  1.24512e-01,1.12439e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[47][j] = _itmp47[j];
			}
		}
		{ static float _itmp48[] = {2.03953e04,1.29120e00,1.19684e-02,
		  1.24016e-01,1.08500e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[48][j] = _itmp48[j];
			}
		}
		{ static float _itmp49[] = {2.08006e04,1.28917e00,1.17763e-02,
		  1.23269e-01,1.04047e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[49][j] = _itmp49[j];
			}
		}
		{ static float _itmp50[] = {2.10323e04,1.28832e00,1.16853e-02,
		  1.23061e-01,1.02511e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[50][j] = _itmp50[j];
			}
		}
		{ static float _itmp51[] = {2.11387e04,1.28828e00,1.16787e-02,
		  1.23207e-01,1.02916e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[51][j] = _itmp51[j];
			}
		}
		{ static float _itmp52[] = {2.12366e04,1.28823e00,1.16635e-02,
		  1.23350e-01,1.03416e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[52][j] = _itmp52[j];
			}
		}
		{ static float _itmp53[] = {2.06752e04,1.29167e00,1.19534e-02,
		  1.25091e-01,1.14697e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[53][j] = _itmp53[j];
			}
		}
		{ static float _itmp54[] = {2.05738e04,1.29240e00,1.20104e-02,
		  1.25581e-01,1.18637e-12};
		for(  j = 0; j < 5; j++ )
		{
			PHH[54][j] = _itmp54[j];
			}
		}
}
void hpfitData1(void)
{
	int j;
		{ static float _itmp55[] = {2.21989e04,1.28392e00,1.12798e-02,
		  1.21830e-01,9.50039e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[55][j] = _itmp55[j];
			}
		}
		{ static float _itmp56[] = {2.27207e04,1.28159e00,1.10767e-02,
		  1.20908e-01,8.98683e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[56][j] = _itmp56[j];
			}
		}
		{ static float _itmp57[] = {2.37584e04,1.27644e00,1.06472e-02,
		  1.18675e-01,7.93691e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[57][j] = _itmp57[j];
			}
		}
		{ static float _itmp58[] = {2.51633e04,1.27083e00,1.01830e-02,
		  1.16227e-01,6.73954e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[58][j] = _itmp58[j];
			}
		}
		{ static float _itmp59[] = {2.53424e04,1.27087e00,1.01888e-02,
		  1.16376e-01,6.70823e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[59][j] = _itmp59[j];
			}
		}
		{ static float _itmp60[] = {2.54713e04,1.27027e00,1.01408e-02,
		  1.16222e-01,6.70873e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[60][j] = _itmp60[j];
			}
		}
		{ static float _itmp61[] = {2.57923e04,1.26934e00,1.00661e-02,
		  1.15921e-01,6.55652e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[61][j] = _itmp61[j];
			}
		}
		{ static float _itmp62[] = {2.73005e04,1.26305e00,9.56598e-03,
		  1.13140e-01,5.56481e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[62][j] = _itmp62[j];
			}
		}
		{ static float _itmp63[] = {2.76820e04,1.26175e00,9.46526e-03,
		  1.12655e-01,5.41061e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[63][j] = _itmp63[j];
			}
		}
		{ static float _itmp64[] = {2.89190e04,1.25739e00,9.12933e-03,
		  1.10756e-01,4.78741e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[64][j] = _itmp64[j];
			}
		}
		{ static float _itmp65[] = {2.93302e04,1.25591e00,9.01763e-03,
		  1.10185e-01,4.64756e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[65][j] = _itmp65[j];
			}
		}
		{ static float _itmp66[] = {2.96601e04,1.25503e00,8.95437e-03,
		  1.09884e-01,4.55491e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[66][j] = _itmp66[j];
			}
		}
		{ static float _itmp67[] = {3.00211e04,1.25411e00,8.88879e-03,
		  1.09562e-01,4.45060e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[67][j] = _itmp67[j];
			}
		}
		{ static float _itmp68[] = {3.00215e04,1.25356e00,8.85608e-03,
		  1.09407e-01,4.51017e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[68][j] = _itmp68[j];
			}
		}
		{ static float _itmp69[] = {3.04895e04,1.25209e00,8.75018e-03,
		  1.08828e-01,4.35708e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[69][j] = _itmp69[j];
			}
		}
		{ static float _itmp70[] = {3.06108e04,1.25067e00,8.65586e-03,
		  1.08259e-01,4.35540e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[70][j] = _itmp70[j];
			}
		}
		{ static float _itmp71[] = {3.08893e04,1.24984e00,8.60245e-03,
		  1.07960e-01,4.29142e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[71][j] = _itmp71[j];
			}
		}
		{ static float _itmp72[] = {3.12948e04,1.24857e00,8.51112e-03,
		  1.07471e-01,4.17474e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[72][j] = _itmp72[j];
			}
		}
		{ static float _itmp73[] = {3.15102e04,1.24770e00,8.45464e-03,
		  1.07156e-01,4.13874e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[73][j] = _itmp73[j];
			}
		}
		{ static float _itmp74[] = {3.15318e04,1.24767e00,8.46726e-03,
		  1.07220e-01,4.18426e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[74][j] = _itmp74[j];
			}
		}
		{ static float _itmp75[] = {3.25503e04,1.24505e00,8.26816e-03,
		  1.06109e-01,3.83836e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[75][j] = _itmp75[j];
			}
		}
		{ static float _itmp76[] = {3.36744e04,1.24043e00,7.95003e-03,
		  1.04027e-01,3.48558e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[76][j] = _itmp76[j];
			}
		}
		{ static float _itmp77[] = {3.60942e04,1.23555e00,7.58706e-03,
		  1.01869e-01,2.84588e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[77][j] = _itmp77[j];
			}
		}
		{ static float _itmp78[] = {3.67949e04,1.23353e00,7.45286e-03,
		  1.01013e-01,2.71133e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[78][j] = _itmp78[j];
			}
		}
		{ static float _itmp79[] = {3.69918e04,1.23311e00,7.43040e-03,
		  1.00895e-01,2.69941e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[79][j] = _itmp79[j];
			}
		}
		{ static float _itmp80[] = {3.82903e04,1.22973e00,7.19264e-03,
		  9.94116e-02,2.45070e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[80][j] = _itmp80[j];
			}
		}
		{ static float _itmp81[] = {4.04734e04,1.22488e00,6.85977e-03,
		  9.72484e-02,2.08427e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[81][j] = _itmp81[j];
			}
		}
		{ static float _itmp82[] = {4.38950e04,1.21802e00,6.39805e-03,
		  9.41438e-02,1.63548e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[82][j] = _itmp82[j];
			}
		}
		{ static float _itmp83[] = {4.47467e04,1.21625e00,6.28883e-03,
		  9.33897e-02,1.55748e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[83][j] = _itmp83[j];
			}
		}
		{ static float _itmp84[] = {4.53623e04,1.21504e00,6.21636e-03,
		  9.28992e-02,1.50929e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[84][j] = _itmp84[j];
			}
		}
		{ static float _itmp85[] = {4.57412e04,1.21425e00,6.17286e-03,
		  9.25980e-02,1.48701e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[85][j] = _itmp85[j];
			}
		}
		{ static float _itmp86[] = {4.58669e04,1.21368e00,6.14706e-03,
		  9.23910e-02,1.48998e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[86][j] = _itmp86[j];
			}
		}
		{ static float _itmp87[] = {4.64746e04,1.21260e00,6.08280e-03,
		  9.19582e-02,1.44578e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[87][j] = _itmp87[j];
			}
		}
		{ static float _itmp88[] = {4.67313e04,1.21198e00,6.05299e-03,
		  9.17324e-02,1.43630e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[88][j] = _itmp88[j];
			}
		}
		{ static float _itmp89[] = {4.72504e04,1.21100e00,5.99627e-03,
		  9.13374e-02,1.40241e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[89][j] = _itmp89[j];
			}
		}
		{ static float _itmp90[] = {4.75397e04,1.21076e00,5.98892e-03,
		  9.12843e-02,1.39133e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[90][j] = _itmp90[j];
			}
		}
		{ static float _itmp91[] = {4.78916e04,1.20989e00,5.94172e-03,
		  9.09401e-02,1.37340e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[91][j] = _itmp91[j];
			}
		}
		{ static float _itmp92[] = {4.84595e04,1.20887e00,5.88514e-03,
		  9.05207e-02,1.33742e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[92][j] = _itmp92[j];
			}
		}
		{ static float _itmp93[] = {4.98872e04,1.20647e00,5.74072e-03,
		  8.94596e-02,1.23380e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[93][j] = _itmp93[j];
			}
		}
		{ static float _itmp94[] = {5.02178e04,1.20578e00,5.70418e-03,
		  8.91986e-02,1.22067e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[94][j] = _itmp94[j];
			}
		}
		{ static float _itmp95[] = {5.05910e04,1.20509e00,5.66983e-03,
		  8.89294e-02,1.20466e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[95][j] = _itmp95[j];
			}
		}
		{ static float _itmp96[] = {5.08355e04,1.20460e00,5.64584e-03,
		  8.87566e-02,1.19839e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[96][j] = _itmp96[j];
			}
		}
		{ static float _itmp97[] = {5.10768e04,1.20412e00,5.62479e-03,
		  8.85841e-02,1.19234e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[97][j] = _itmp97[j];
			}
		}
		{ static float _itmp98[] = {5.13650e04,1.20348e00,5.59185e-03,
		  8.83416e-02,1.18272e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[98][j] = _itmp98[j];
			}
		}
		{ static float _itmp99[] = {5.15616e04,1.20280e00,5.56150e-03,
		  8.80741e-02,1.17928e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[99][j] = _itmp99[j];
			}
		}
		{ static float _itmp100[] = {5.15049e04,1.20288e00,5.57236e-03,
		  8.81638e-02,1.19484e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[100][j] = _itmp100[j];
			}
		}
		{ static float _itmp101[] = {5.16203e04,1.20239e00,5.55256e-03,
		  8.79774e-02,1.19718e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[101][j] = _itmp101[j];
			}
		}
		{ static float _itmp102[] = {5.20700e04,1.20183e00,5.52369e-03,
		  8.77696e-02,1.17647e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[102][j] = _itmp102[j];
			}
		}
		{ static float _itmp103[] = {5.22801e04,1.20111e00,5.48987e-03,
		  8.74815e-02,1.17181e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[103][j] = _itmp103[j];
			}
		}
		{ static float _itmp104[] = {5.26918e04,1.20047e00,5.45422e-03,
		  8.72364e-02,1.15403e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[104][j] = _itmp104[j];
			}
		}
		{ static float _itmp105[] = {5.30097e04,1.19987e00,5.42558e-03,
		  8.69977e-02,1.14269e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[105][j] = _itmp105[j];
			}
		}
		{ static float _itmp106[] = {5.31625e04,1.19949e00,5.40908e-03,
		  8.68723e-02,1.14250e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[106][j] = _itmp106[j];
			}
		}
		{ static float _itmp107[] = {5.32084e04,1.19910e00,5.39659e-03,
		  8.67286e-02,1.14906e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[107][j] = _itmp107[j];
			}
		}
		{ static float _itmp108[] = {5.39327e04,1.19789e00,5.32780e-03,
		  8.62162e-02,1.11101e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[108][j] = _itmp108[j];
			}
		}
}
void hpfitData2(void)
{
	int j;
		{ static float _itmp109[] = {5.42985e04,1.19747e00,5.31012e-03,
		  8.60574e-02,1.09764e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[109][j] = _itmp109[j];
			}
		}
		{ static float _itmp110[] = {5.47482e04,1.19671e00,5.26994e-03,
		  8.57517e-02,1.07888e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[110][j] = _itmp110[j];
			}
		}
		{ static float _itmp111[] = {5.53598e04,1.19566e00,5.21107e-03,
		  8.53086e-02,1.05076e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[111][j] = _itmp111[j];
			}
		}
		{ static float _itmp112[] = {5.56526e04,1.19515e00,5.19132e-03,
		  8.51014e-02,1.04227e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[112][j] = _itmp112[j];
			}
		}
		{ static float _itmp113[] = {5.62626e04,1.19423e00,5.14124e-03,
		  8.47205e-02,1.01584e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[113][j] = _itmp113[j];
			}
		}
		{ static float _itmp114[] = {5.64509e04,1.19374e00,5.12138e-03,
		  8.45269e-02,1.01336e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[114][j] = _itmp114[j];
			}
		}
		{ static float _itmp115[] = {5.65048e04,1.19347e00,5.11202e-03,
		  8.44478e-02,1.01850e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[115][j] = _itmp115[j];
			}
		}
		{ static float _itmp116[] = {5.63001e04,1.19365e00,5.13057e-03,
		  8.45656e-02,1.03855e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[116][j] = _itmp116[j];
			}
		}
		{ static float _itmp117[] = {5.62106e04,1.19367e00,5.13907e-03,
		  8.46105e-02,1.05220e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[117][j] = _itmp117[j];
			}
		}
		{ static float _itmp118[] = {5.63240e04,1.19328e00,5.12788e-03,
		  8.44597e-02,1.05369e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[118][j] = _itmp118[j];
			}
		}
		{ static float _itmp119[] = {5.63976e04,1.19293e00,5.11407e-03,
		  8.43346e-02,1.05744e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[119][j] = _itmp119[j];
			}
		}
		{ static float _itmp120[] = {5.59892e04,1.19323e00,5.14229e-03,
		  8.45086e-02,1.08999e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[120][j] = _itmp120[j];
			}
		}
		{ static float _itmp121[] = {5.61317e04,1.19317e00,5.14126e-03,
		  8.45229e-02,1.09032e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[121][j] = _itmp121[j];
			}
		}
		{ static float _itmp122[] = {5.64118e04,1.19274e00,5.12409e-03,
		  8.43517e-02,1.08190e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[122][j] = _itmp122[j];
			}
		}
		{ static float _itmp123[] = {5.63712e04,1.19258e00,5.12406e-03,
		  8.43088e-02,1.09242e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[123][j] = _itmp123[j];
			}
		}
		{ static float _itmp124[] = {5.66924e04,1.19188e00,5.09079e-03,
		  8.40173e-02,1.08103e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[124][j] = _itmp124[j];
			}
		}
		{ static float _itmp125[] = {5.67500e04,1.19178e00,5.08880e-03,
		  8.40072e-02,1.08594e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[125][j] = _itmp125[j];
			}
		}
		{ static float _itmp126[] = {5.71390e04,1.19113e00,5.05839e-03,
		  8.37378e-02,1.07089e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[126][j] = _itmp126[j];
			}
		}
		{ static float _itmp127[] = {5.70679e04,1.19104e00,5.06027e-03,
		  8.37273e-02,1.08292e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[127][j] = _itmp127[j];
			}
		}
		{ static float _itmp128[] = {5.70957e04,1.19074e00,5.05193e-03,
		  8.36150e-02,1.08889e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[128][j] = _itmp128[j];
			}
		}
		{ static float _itmp129[] = {5.72020e04,1.19060e00,5.04686e-03,
		  8.35863e-02,1.09074e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[129][j] = _itmp129[j];
			}
		}
		{ static float _itmp130[] = {5.71903e04,1.19039e00,5.04318e-03,
		  8.35158e-02,1.09911e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[130][j] = _itmp130[j];
			}
		}
		{ static float _itmp131[] = {5.72388e04,1.19028e00,5.04478e-03,
		  8.34939e-02,1.10422e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[131][j] = _itmp131[j];
			}
		}
		{ static float _itmp132[] = {5.71377e04,1.19007e00,5.04055e-03,
		  8.34292e-02,1.11778e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[132][j] = _itmp132[j];
			}
		}
		{ static float _itmp133[] = {5.68854e04,1.19047e00,5.07016e-03,
		  8.36427e-02,1.14177e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[133][j] = _itmp133[j];
			}
		}
		{ static float _itmp134[] = {5.67902e04,1.19026e00,5.06625e-03,
		  8.35736e-02,1.15534e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[134][j] = _itmp134[j];
			}
		}
		{ static float _itmp135[] = {5.66644e04,1.19047e00,5.08405e-03,
		  8.37017e-02,1.17186e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[135][j] = _itmp135[j];
			}
		}
		{ static float _itmp136[] = {5.64886e04,1.19041e00,5.09217e-03,
		  8.36938e-02,1.19105e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[136][j] = _itmp136[j];
			}
		}
		{ static float _itmp137[] = {5.85530e04,1.18765e00,4.93064e-03,
		  8.24445e-02,1.07198e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[137][j] = _itmp137[j];
			}
		}
		{ static float _itmp138[] = {5.86855e04,1.18747e00,4.92364e-03,
		  8.23949e-02,1.07190e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[138][j] = _itmp138[j];
			}
		}
		{ static float _itmp139[] = {5.88453e04,1.18691e00,4.90199e-03,
		  8.21526e-02,1.06948e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[139][j] = _itmp139[j];
			}
		}
		{ static float _itmp140[] = {5.88638e04,1.18704e00,4.91314e-03,
		  8.22433e-02,1.07616e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[140][j] = _itmp140[j];
			}
		}
		{ static float _itmp141[] = {5.87851e04,1.18691e00,4.91251e-03,
		  8.22120e-02,1.08765e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[141][j] = _itmp141[j];
			}
		}
		{ static float _itmp142[] = {5.88130e04,1.18659e00,4.90079e-03,
		  8.20866e-02,1.09285e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[142][j] = _itmp142[j];
			}
		}
		{ static float _itmp143[] = {5.93471e04,1.18588e00,4.86743e-03,
		  8.17790e-02,1.06950e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[143][j] = _itmp143[j];
			}
		}
		{ static float _itmp144[] = {5.95908e04,1.18557e00,4.85105e-03,
		  8.16661e-02,1.06301e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[144][j] = _itmp144[j];
			}
		}
		{ static float _itmp145[] = {5.97129e04,1.18536e00,4.84487e-03,
		  8.15949e-02,1.06318e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[145][j] = _itmp145[j];
			}
		}
		{ static float _itmp146[] = {5.96874e04,1.18516e00,4.84112e-03,
		  8.15297e-02,1.07120e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[146][j] = _itmp146[j];
			}
		}
		{ static float _itmp147[] = {5.98537e04,1.18475e00,4.82531e-03,
		  8.13542e-02,1.06848e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[147][j] = _itmp147[j];
			}
		}
		{ static float _itmp148[] = {6.01163e04,1.18433e00,4.80679e-03,
		  8.11877e-02,1.06068e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[148][j] = _itmp148[j];
			}
		}
		{ static float _itmp149[] = {6.02535e04,1.18400e00,4.79354e-03,
		  8.10538e-02,1.05967e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[149][j] = _itmp149[j];
			}
		}
		{ static float _itmp150[] = {6.12993e04,1.18263e00,4.71801e-03,
		  8.04429e-02,1.01044e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[150][j] = _itmp150[j];
			}
		}
		{ static float _itmp151[] = {6.15881e04,1.18211e00,4.69647e-03,
		  8.02230e-02,1.00175e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[151][j] = _itmp151[j];
			}
		}
		{ static float _itmp152[] = {6.15834e04,1.18205e00,4.69907e-03,
		  8.02166e-02,1.00826e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[152][j] = _itmp152[j];
			}
		}
		{ static float _itmp153[] = {6.21590e04,1.18146e00,4.66635e-03,
		  7.99722e-02,9.85752e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[153][j] = _itmp153[j];
			}
		}
		{ static float _itmp154[] = {6.20649e04,1.18144e00,4.67359e-03,
		  7.99807e-02,9.96399e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[154][j] = _itmp154[j];
			}
		}
		{ static float _itmp155[] = {6.20998e04,1.18126e00,4.66945e-03,
		  7.99154e-02,1.00061e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[155][j] = _itmp155[j];
			}
		}
		{ static float _itmp156[] = {6.21969e04,1.18093e00,4.65326e-03,
		  7.97913e-02,1.00153e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[156][j] = _itmp156[j];
			}
		}
		{ static float _itmp157[] = {6.26209e04,1.18051e00,4.63736e-03,
		  7.96109e-02,9.86650e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[157][j] = _itmp157[j];
			}
		}
		{ static float _itmp158[] = {6.26851e04,1.18013e00,4.62024e-03,
		  7.94646e-02,9.88984e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[158][j] = _itmp158[j];
			}
		}
		{ static float _itmp159[] = {6.27897e04,1.17991e00,4.61297e-03,
		  7.93820e-02,9.89649e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[159][j] = _itmp159[j];
			}
		}
		{ static float _itmp160[] = {6.32154e04,1.17946e00,4.59261e-03,
		  7.91949e-02,9.74916e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[160][j] = _itmp160[j];
			}
		}
		{ static float _itmp161[] = {6.28296e04,1.17960e00,4.60748e-03,
		  7.92777e-02,9.99157e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[161][j] = _itmp161[j];
			}
		}
		{ static float _itmp162[] = {6.27666e04,1.17960e00,4.61281e-03,
		  7.93002e-02,1.00817e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[162][j] = _itmp162[j];
			}
		}
}
void hpfitData3(void)
{
	int j;
		{ static float _itmp163[] = {6.28188e04,1.17926e00,4.60014e-03,
		  7.91606e-02,1.01098e-13};
		for(  j = 0; j < 5; j++ )
		{
			PHH[163][j] = _itmp163[j];
			}
		}
		{ static float _itmp164[] = {6.49777e04,1.17700e00,4.47186e-03,
		  7.81305e-02,9.15301e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[164][j] = _itmp164[j];
			}
		}
		{ static float _itmp165[] = {6.49998e04,1.17682e00,4.46764e-03,
		  7.80687e-02,9.19442e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[165][j] = _itmp165[j];
			}
		}
		{ static float _itmp166[] = {6.50971e04,1.17673e00,4.46685e-03,
		  7.80477e-02,9.20540e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[166][j] = _itmp166[j];
			}
		}
		{ static float _itmp167[] = {6.51001e04,1.17668e00,4.47044e-03,
		  7.80443e-02,9.25666e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[167][j] = _itmp167[j];
			}
		}
		{ static float _itmp168[] = {6.53712e04,1.17590e00,4.43566e-03,
		  7.76892e-02,9.18101e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[168][j] = _itmp168[j];
			}
		}
		{ static float _itmp169[] = {6.59554e04,1.17541e00,4.40940e-03,
		  7.74885e-02,8.98314e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[169][j] = _itmp169[j];
			}
		}
		{ static float _itmp170[] = {6.60573e04,1.17525e00,4.40404e-03,
		  7.74358e-02,8.99013e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[170][j] = _itmp170[j];
			}
		}
		{ static float _itmp171[] = {6.61940e04,1.17513e00,4.40154e-03,
		  7.73997e-02,8.98355e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[171][j] = _itmp171[j];
			}
		}
		{ static float _itmp172[] = {6.63293e04,1.17456e00,4.37606e-03,
		  7.71503e-02,8.96919e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[172][j] = _itmp172[j];
			}
		}
		{ static float _itmp173[] = {6.65814e04,1.17430e00,4.36866e-03,
		  7.70420e-02,8.91325e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[173][j] = _itmp173[j];
			}
		}
		{ static float _itmp174[] = {6.65895e04,1.17427e00,4.36925e-03,
		  7.70586e-02,8.95946e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[174][j] = _itmp174[j];
			}
		}
		{ static float _itmp175[] = {6.68073e04,1.17394e00,4.35579e-03,
		  7.69209e-02,8.91590e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[175][j] = _itmp175[j];
			}
		}
		{ static float _itmp176[] = {6.68303e04,1.17387e00,4.35616e-03,
		  7.69057e-02,8.95459e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[176][j] = _itmp176[j];
			}
		}
		{ static float _itmp177[] = {6.74621e04,1.17325e00,4.32576e-03,
		  7.66363e-02,8.74358e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[177][j] = _itmp177[j];
			}
		}
		{ static float _itmp178[] = {6.83886e04,1.17218e00,4.26888e-03,
		  7.61559e-02,8.42239e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[178][j] = _itmp178[j];
			}
		}
		{ static float _itmp179[] = {6.92863e04,1.17117e00,4.21869e-03,
		  7.57032e-02,8.12833e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[179][j] = _itmp179[j];
			}
		}
		{ static float _itmp180[] = {6.93488e04,1.17128e00,4.22583e-03,
		  7.57736e-02,8.15203e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[180][j] = _itmp180[j];
			}
		}
		{ static float _itmp181[] = {6.95462e04,1.17057e00,4.19544e-03,
		  7.54550e-02,8.11497e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[181][j] = _itmp181[j];
			}
		}
		{ static float _itmp182[] = {7.01551e04,1.16996e00,4.16792e-03,
		  7.51812e-02,7.93871e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[182][j] = _itmp182[j];
			}
		}
		{ static float _itmp183[] = {7.11655e04,1.16926e00,4.12696e-03,
		  7.48849e-02,7.63674e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[183][j] = _itmp183[j];
			}
		}
		{ static float _itmp184[] = {7.26336e04,1.16728e00,4.02543e-03,
		  7.39746e-02,7.19420e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[184][j] = _itmp184[j];
			}
		}
		{ static float _itmp185[] = {7.28369e04,1.16733e00,4.03171e-03,
		  7.40156e-02,7.17252e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[185][j] = _itmp185[j];
			}
		}
		{ static float _itmp186[] = {7.33426e04,1.16671e00,4.00143e-03,
		  7.37486e-02,7.05385e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[186][j] = _itmp186[j];
			}
		}
		{ static float _itmp187[] = {7.36828e04,1.16643e00,3.99036e-03,
		  7.36310e-02,6.98917e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[187][j] = _itmp187[j];
			}
		}
		{ static float _itmp188[] = {7.39666e04,1.16614e00,3.97808e-03,
		  7.35127e-02,6.94100e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[188][j] = _itmp188[j];
			}
		}
		{ static float _itmp189[] = {7.40107e04,1.16601e00,3.97690e-03,
		  7.34663e-02,6.96236e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[189][j] = _itmp189[j];
			}
		}
		{ static float _itmp190[] = {7.39172e04,1.16586e00,3.97482e-03,
		  7.34083e-02,7.02226e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[190][j] = _itmp190[j];
			}
		}
		{ static float _itmp191[] = {7.45995e04,1.16527e00,3.94497e-03,
		  7.31551e-02,6.85885e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[191][j] = _itmp191[j];
			}
		}
		{ static float _itmp192[] = {7.50173e04,1.16479e00,3.92397e-03,
		  7.29436e-02,6.77286e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[192][j] = _itmp192[j];
			}
		}
		{ static float _itmp193[] = {7.58009e04,1.16410e00,3.88972e-03,
		  7.26377e-02,6.58975e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[193][j] = _itmp193[j];
			}
		}
		{ static float _itmp194[] = {7.65273e04,1.16333e00,3.85286e-03,
		  7.22975e-02,6.42674e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[194][j] = _itmp194[j];
			}
		}
		{ static float _itmp195[] = {7.67465e04,1.16309e00,3.84380e-03,
		  7.22001e-02,6.40069e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[195][j] = _itmp195[j];
			}
		}
		{ static float _itmp196[] = {7.83220e04,1.16165e00,3.76989e-03,
		  7.15471e-02,6.03590e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[196][j] = _itmp196[j];
			}
		}
		{ static float _itmp197[] = {7.86456e04,1.16134e00,3.75780e-03,
		  7.14128e-02,5.98753e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[197][j] = _itmp197[j];
			}
		}
		{ static float _itmp198[] = {7.88621e04,1.16124e00,3.75643e-03,
		  7.13818e-02,5.96646e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[198][j] = _itmp198[j];
			}
		}
		{ static float _itmp199[] = {7.94154e04,1.16056e00,3.72562e-03,
		  7.10786e-02,5.86356e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[199][j] = _itmp199[j];
			}
		}
		{ static float _itmp200[] = {7.96269e04,1.16016e00,3.70716e-03,
		  7.09092e-02,5.84067e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[200][j] = _itmp200[j];
			}
		}
		{ static float _itmp201[] = {8.00661e04,1.15983e00,3.69397e-03,
		  7.07664e-02,5.76932e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[201][j] = _itmp201[j];
			}
		}
		{ static float _itmp202[] = {8.10129e04,1.15905e00,3.65865e-03,
		  7.04134e-02,5.58797e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[202][j] = _itmp202[j];
			}
		}
		{ static float _itmp203[] = {8.13573e04,1.15870e00,3.64328e-03,
		  7.02705e-02,5.54012e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[203][j] = _itmp203[j];
			}
		}
		{ static float _itmp204[] = {8.15364e04,1.15849e00,3.63595e-03,
		  7.01817e-02,5.52756e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[204][j] = _itmp204[j];
			}
		}
		{ static float _itmp205[] = {8.16732e04,1.15816e00,3.62227e-03,
		  7.00475e-02,5.52242e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[205][j] = _itmp205[j];
			}
		}
		{ static float _itmp206[] = {8.20184e04,1.15784e00,3.61158e-03,
		  6.99051e-02,5.47526e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[206][j] = _itmp206[j];
			}
		}
		{ static float _itmp207[] = {8.23027e04,1.15760e00,3.60165e-03,
		  6.98110e-02,5.44176e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[207][j] = _itmp207[j];
			}
		}
		{ static float _itmp208[] = {8.26037e04,1.15719e00,3.58450e-03,
		  6.96305e-02,5.40326e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[208][j] = _itmp208[j];
			}
		}
		{ static float _itmp209[] = {8.28891e04,1.15698e00,3.57542e-03,
		  6.95479e-02,5.37045e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[209][j] = _itmp209[j];
			}
		}
		{ static float _itmp210[] = {8.31632e04,1.15664e00,3.56328e-03,
		  6.94017e-02,5.33854e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[210][j] = _itmp210[j];
			}
		}
		{ static float _itmp211[] = {8.38334e04,1.15620e00,3.54204e-03,
		  6.92124e-02,5.23135e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[211][j] = _itmp211[j];
			}
		}
		{ static float _itmp212[] = {8.45296e04,1.15556e00,3.51339e-03,
		  6.89287e-02,5.12031e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[212][j] = _itmp212[j];
			}
		}
		{ static float _itmp213[] = {8.51244e04,1.15504e00,3.49037e-03,
		  6.86951e-02,5.03153e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[213][j] = _itmp213[j];
			}
		}
		{ static float _itmp214[] = {8.55104e04,1.15472e00,3.47542e-03,
		  6.85645e-02,4.98329e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[214][j] = _itmp214[j];
			}
		}
		{ static float _itmp215[] = {8.59027e04,1.15445e00,3.46470e-03,
		  6.84513e-02,4.93502e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[215][j] = _itmp215[j];
			}
		}
		{ static float _itmp216[] = {8.63232e04,1.15394e00,3.44238e-03,
		  6.82267e-02,4.88031e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[216][j] = _itmp216[j];
			}
		}
}
void hpfitData4(void)
{
	int j;
		{ static float _itmp217[] = {8.67658e04,1.15358e00,3.42754e-03,
		  6.80725e-02,4.82421e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[217][j] = _itmp217[j];
			}
		}
		{ static float _itmp218[] = {8.71436e04,1.15326e00,3.41489e-03,
		  6.79327e-02,4.77995e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[218][j] = _itmp218[j];
			}
		}
		{ static float _itmp219[] = {8.75238e04,1.15294e00,3.40150e-03,
		  6.77952e-02,4.73598e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[219][j] = _itmp219[j];
			}
		}
		{ static float _itmp220[] = {8.78433e04,1.15270e00,3.39224e-03,
		  6.76996e-02,4.70311e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[220][j] = _itmp220[j];
			}
		}
		{ static float _itmp221[] = {8.83514e04,1.15228e00,3.37388e-03,
		  6.75155e-02,4.63900e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[221][j] = _itmp221[j];
			}
		}
		{ static float _itmp222[] = {8.87785e04,1.15172e00,3.34931e-03,
		  6.72688e-02,4.58735e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[222][j] = _itmp222[j];
			}
		}
		{ static float _itmp223[] = {8.91574e04,1.15145e00,3.33919e-03,
		  6.71508e-02,4.54645e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[223][j] = _itmp223[j];
			}
		}
		{ static float _itmp224[] = {8.94967e04,1.15121e00,3.32838e-03,
		  6.70588e-02,4.51244e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[224][j] = _itmp224[j];
			}
		}
		{ static float _itmp225[] = {9.12270e04,1.14991e00,3.26653e-03,
		  6.64589e-02,4.26817e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[225][j] = _itmp225[j];
			}
		}
		{ static float _itmp226[] = {9.13531e04,1.14982e00,3.26608e-03,
		  6.64247e-02,4.26803e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[226][j] = _itmp226[j];
			}
		}
		{ static float _itmp227[] = {9.15646e04,1.14979e00,3.26820e-03,
		  6.64228e-02,4.25650e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[227][j] = _itmp227[j];
			}
		}
		{ static float _itmp228[] = {9.17676e04,1.14936e00,3.25112e-03,
		  6.62332e-02,4.24261e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[228][j] = _itmp228[j];
			}
		}
		{ static float _itmp229[] = {9.19852e04,1.14917e00,3.24418e-03,
		  6.61558e-02,4.22891e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[229][j] = _itmp229[j];
			}
		}
		{ static float _itmp230[] = {9.23665e04,1.14871e00,3.22490e-03,
		  6.59510e-02,4.19049e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[230][j] = _itmp230[j];
			}
		}
		{ static float _itmp231[] = {9.27334e04,1.14853e00,3.21863e-03,
		  6.58796e-02,4.15699e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[231][j] = _itmp231[j];
			}
		}
		{ static float _itmp232[] = {9.31345e04,1.14835e00,3.21068e-03,
		  6.58105e-02,4.11939e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[232][j] = _itmp232[j];
			}
		}
		{ static float _itmp233[] = {9.32638e04,1.14820e00,3.20672e-03,
		  6.57554e-02,4.11821e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[233][j] = _itmp233[j];
			}
		}
		{ static float _itmp234[] = {9.34470e04,1.14799e00,3.20042e-03,
		  6.56619e-02,4.10922e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[234][j] = _itmp234[j];
			}
		}
		{ static float _itmp235[] = {9.36383e04,1.14773e00,3.19123e-03,
		  6.55532e-02,4.09887e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[235][j] = _itmp235[j];
			}
		}
		{ static float _itmp236[] = {9.39122e04,1.14741e00,3.17906e-03,
		  6.54152e-02,4.07725e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[236][j] = _itmp236[j];
			}
		}
		{ static float _itmp237[] = {9.46940e04,1.14701e00,3.16010e-03,
		  6.52396e-02,3.99037e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[237][j] = _itmp237[j];
			}
		}
		{ static float _itmp238[] = {9.55356e04,1.14632e00,3.12968e-03,
		  6.49305e-02,3.89646e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[238][j] = _itmp238[j];
			}
		}
		{ static float _itmp239[] = {9.58083e04,1.14612e00,3.12136e-03,
		  6.48497e-02,3.87748e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[239][j] = _itmp239[j];
			}
		}
		{ static float _itmp240[] = {9.59772e04,1.14581e00,3.10892e-03,
		  6.47174e-02,3.87024e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[240][j] = _itmp240[j];
			}
		}
		{ static float _itmp241[] = {9.60382e04,1.14590e00,3.11656e-03,
		  6.47660e-02,3.87914e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[241][j] = _itmp241[j];
			}
		}
		{ static float _itmp242[] = {9.61351e04,1.14571e00,3.11176e-03,
		  6.46857e-02,3.88142e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[242][j] = _itmp242[j];
			}
		}
		{ static float _itmp243[] = {9.62523e04,1.14556e00,3.10646e-03,
		  6.46315e-02,3.88161e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[243][j] = _itmp243[j];
			}
		}
		{ static float _itmp244[] = {9.64684e04,1.14534e00,3.09921e-03,
		  6.45360e-02,3.86919e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[244][j] = _itmp244[j];
			}
		}
		{ static float _itmp245[] = {9.65454e04,1.14519e00,3.09579e-03,
		  6.44746e-02,3.87404e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[245][j] = _itmp245[j];
			}
		}
		{ static float _itmp246[] = {9.66749e04,1.14507e00,3.09092e-03,
		  6.44320e-02,3.87282e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[246][j] = _itmp246[j];
			}
		}
		{ static float _itmp247[] = {9.68937e04,1.14485e00,3.08242e-03,
		  6.43444e-02,3.86016e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[247][j] = _itmp247[j];
			}
		}
		{ static float _itmp248[] = {9.70653e04,1.14474e00,3.08065e-03,
		  6.42944e-02,3.85386e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[248][j] = _itmp248[j];
			}
		}
		{ static float _itmp249[] = {9.71949e04,1.14465e00,3.07937e-03,
		  6.42618e-02,3.85276e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[249][j] = _itmp249[j];
			}
		}
		{ static float _itmp250[] = {9.74727e04,1.14443e00,3.07107e-03,
		  6.41735e-02,3.83312e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[250][j] = _itmp250[j];
			}
		}
		{ static float _itmp251[] = {9.77813e04,1.14427e00,3.06446e-03,
		  6.41107e-02,3.81055e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[251][j] = _itmp251[j];
			}
		}
		{ static float _itmp252[] = {9.80282e04,1.14378e00,3.04592e-03,
		  6.38878e-02,3.79248e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[252][j] = _itmp252[j];
			}
		}
		{ static float _itmp253[] = {9.83343e04,1.14363e00,3.04135e-03,
		  6.38269e-02,3.77052e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[253][j] = _itmp253[j];
			}
		}
		{ static float _itmp254[] = {9.85896e04,1.14345e00,3.03740e-03,
		  6.37486e-02,3.75425e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[254][j] = _itmp254[j];
			}
		}
		{ static float _itmp255[] = {9.87936e04,1.14322e00,3.02670e-03,
		  6.36560e-02,3.74358e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[255][j] = _itmp255[j];
			}
		}
		{ static float _itmp256[] = {9.91176e04,1.14305e00,3.02231e-03,
		  6.35790e-02,3.71976e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[256][j] = _itmp256[j];
			}
		}
		{ static float _itmp257[] = {9.94444e04,1.14271e00,3.00897e-03,
		  6.34309e-02,3.69463e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[257][j] = _itmp257[j];
			}
		}
		{ static float _itmp258[] = {9.96930e04,1.14241e00,2.99691e-03,
		  6.33062e-02,3.67862e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[258][j] = _itmp258[j];
			}
		}
		{ static float _itmp259[] = {9.99557e04,1.14230e00,2.99342e-03,
		  6.32636e-02,3.66264e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[259][j] = _itmp259[j];
			}
		}
		{ static float _itmp260[] = {1.00226e05,1.14209e00,2.98556e-03,
		  6.31740e-02,3.64523e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[260][j] = _itmp260[j];
			}
		}
		{ static float _itmp261[] = {1.00559e05,1.14201e00,2.98204e-03,
		  6.31493e-02,3.62206e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[261][j] = _itmp261[j];
			}
		}
		{ static float _itmp262[] = {1.00877e05,1.14176e00,2.97389e-03,
		  6.30401e-02,3.59942e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[262][j] = _itmp262[j];
			}
		}
		{ static float _itmp263[] = {1.01298e05,1.14139e00,2.96004e-03,
		  6.28730e-02,3.56510e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[263][j] = _itmp263[j];
			}
		}
		{ static float _itmp264[] = {1.01553e05,1.14121e00,2.95435e-03,
		  6.27955e-02,3.55004e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[264][j] = _itmp264[j];
			}
		}
		{ static float _itmp265[] = {1.01811e05,1.14107e00,2.95073e-03,
		  6.27426e-02,3.53516e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[265][j] = _itmp265[j];
			}
		}
		{ static float _itmp266[] = {1.02152e05,1.14069e00,2.93733e-03,
		  6.25658e-02,3.50987e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[266][j] = _itmp266[j];
			}
		}
		{ static float _itmp267[] = {1.02392e05,1.14052e00,2.92838e-03,
		  6.25039e-02,3.49685e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[267][j] = _itmp267[j];
			}
		}
		{ static float _itmp268[] = {1.02798e05,1.14020e00,2.91748e-03,
		  6.23580e-02,3.46574e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[268][j] = _itmp268[j];
			}
		}
		{ static float _itmp269[] = {1.03037e05,1.13993e00,2.90616e-03,
		  6.22441e-02,3.45226e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[269][j] = _itmp269[j];
			}
		}
		{ static float _itmp270[] = {1.03260e05,1.13977e00,2.90035e-03,
		  6.21773e-02,3.44118e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[270][j] = _itmp270[j];
			}
		}
}
void hpfitData5(void)
{
	int j;
		{ static float _itmp271[] = {1.03571e05,1.13972e00,2.90042e-03,
		  6.21642e-02,3.42222e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[271][j] = _itmp271[j];
			}
		}
		{ static float _itmp272[] = {1.03788e05,1.13947e00,2.89246e-03,
		  6.20533e-02,3.41121e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[272][j] = _itmp272[j];
			}
		}
		{ static float _itmp273[] = {1.04065e05,1.13929e00,2.88545e-03,
		  6.19798e-02,3.39492e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[273][j] = _itmp273[j];
			}
		}
		{ static float _itmp274[] = {1.04366e05,1.13882e00,2.86790e-03,
		  6.17639e-02,3.37421e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[274][j] = _itmp274[j];
			}
		}
		{ static float _itmp275[] = {1.04670e05,1.13872e00,2.86443e-03,
		  6.17290e-02,3.35615e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[275][j] = _itmp275[j];
			}
		}
		{ static float _itmp276[] = {1.04889e05,1.13852e00,2.85746e-03,
		  6.16407e-02,3.34543e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[276][j] = _itmp276[j];
			}
		}
		{ static float _itmp277[] = {1.05114e05,1.13832e00,2.84916e-03,
		  6.15632e-02,3.33441e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[277][j] = _itmp277[j];
			}
		}
		{ static float _itmp278[] = {1.05329e05,1.13825e00,2.84732e-03,
		  6.15349e-02,3.32519e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[278][j] = _itmp278[j];
			}
		}
		{ static float _itmp279[] = {1.05553e05,1.13815e00,2.84555e-03,
		  6.14957e-02,3.31488e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[279][j] = _itmp279[j];
			}
		}
		{ static float _itmp280[] = {1.05817e05,1.13778e00,2.83203e-03,
		  6.13320e-02,3.29906e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[280][j] = _itmp280[j];
			}
		}
		{ static float _itmp281[] = {1.06043e05,1.13754e00,2.82361e-03,
		  6.12226e-02,3.28763e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[281][j] = _itmp281[j];
			}
		}
		{ static float _itmp282[] = {1.06282e05,1.13739e00,2.81893e-03,
		  6.11609e-02,3.27574e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[282][j] = _itmp282[j];
			}
		}
		{ static float _itmp283[] = {1.06474e05,1.13727e00,2.81474e-03,
		  6.11173e-02,3.26853e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[283][j] = _itmp283[j];
			}
		}
		{ static float _itmp284[] = {1.06596e05,1.13737e00,2.81830e-03,
		  6.11813e-02,3.26931e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[284][j] = _itmp284[j];
			}
		}
		{ static float _itmp285[] = {1.06734e05,1.13707e00,2.80941e-03,
		  6.10364e-02,3.26558e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[285][j] = _itmp285[j];
			}
		}
		{ static float _itmp286[] = {1.06899e05,1.13691e00,2.80504e-03,
		  6.09718e-02,3.26044e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[286][j] = _itmp286[j];
			}
		}
		{ static float _itmp287[] = {1.07094e05,1.13677e00,2.80008e-03,
		  6.09126e-02,3.25265e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[287][j] = _itmp287[j];
			}
		}
		{ static float _itmp288[] = {1.07412e05,1.13658e00,2.79417e-03,
		  6.08281e-02,3.23350e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[288][j] = _itmp288[j];
			}
		}
		{ static float _itmp289[] = {1.07570e05,1.13636e00,2.78581e-03,
		  6.07358e-02,3.22865e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[289][j] = _itmp289[j];
			}
		}
		{ static float _itmp290[] = {1.07682e05,1.13628e00,2.78372e-03,
		  6.07066e-02,3.22886e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[290][j] = _itmp290[j];
			}
		}
		{ static float _itmp291[] = {1.07850e05,1.13601e00,2.77566e-03,
		  6.05877e-02,3.22272e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[291][j] = _itmp291[j];
			}
		}
		{ static float _itmp292[] = {1.08040e05,1.13590e00,2.77353e-03,
		  6.05401e-02,3.21560e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[292][j] = _itmp292[j];
			}
		}
		{ static float _itmp293[] = {1.08222e05,1.13577e00,2.76748e-03,
		  6.04890e-02,3.20926e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[293][j] = _itmp293[j];
			}
		}
		{ static float _itmp294[] = {1.08556e05,1.13579e00,2.76898e-03,
		  6.05083e-02,3.19048e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[294][j] = _itmp294[j];
			}
		}
		{ static float _itmp295[] = {1.08761e05,1.13541e00,2.75564e-03,
		  6.03343e-02,3.18031e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[295][j] = _itmp295[j];
			}
		}
		{ static float _itmp296[] = {1.09056e05,1.13522e00,2.74731e-03,
		  6.02573e-02,3.16373e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[296][j] = _itmp296[j];
			}
		}
		{ static float _itmp297[] = {1.09374e05,1.13506e00,2.74255e-03,
		  6.01859e-02,3.14542e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[297][j] = _itmp297[j];
			}
		}
		{ static float _itmp298[] = {1.09625e05,1.13502e00,2.74168e-03,
		  6.01773e-02,3.13390e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[298][j] = _itmp298[j];
			}
		}
		{ static float _itmp299[] = {1.09865e05,1.13479e00,2.73374e-03,
		  6.00805e-02,3.12204e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[299][j] = _itmp299[j];
			}
		}
		{ static float _itmp300[] = {1.10391e05,1.13460e00,2.72669e-03,
		  5.99970e-02,3.08649e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[300][j] = _itmp300[j];
			}
		}
		{ static float _itmp301[] = {1.10923e05,1.13432e00,2.71520e-03,
		  5.98730e-02,3.05034e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[301][j] = _itmp301[j];
			}
		}
		{ static float _itmp302[] = {1.11443e05,1.13406e00,2.70513e-03,
		  5.97586e-02,3.01597e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[302][j] = _itmp302[j];
			}
		}
		{ static float _itmp303[] = {1.11933e05,1.13369e00,2.69023e-03,
		  5.95971e-02,2.98384e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[303][j] = _itmp303[j];
			}
		}
		{ static float _itmp304[] = {1.12475e05,1.13364e00,2.68795e-03,
		  5.95804e-02,2.95013e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[304][j] = _itmp304[j];
			}
		}
		{ static float _itmp305[] = {1.13043e05,1.13328e00,2.67371e-03,
		  5.94170e-02,2.91297e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[305][j] = _itmp305[j];
			}
		}
		{ static float _itmp306[] = {1.13838e05,1.13282e00,2.65121e-03,
		  5.92218e-02,2.85892e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[306][j] = _itmp306[j];
			}
		}
		{ static float _itmp307[] = {1.14312e05,1.13236e00,2.63588e-03,
		  5.90055e-02,2.82961e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[307][j] = _itmp307[j];
			}
		}
		{ static float _itmp308[] = {1.14794e05,1.13205e00,2.62284e-03,
		  5.88698e-02,2.80115e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[308][j] = _itmp308[j];
			}
		}
		{ static float _itmp309[] = {1.15246e05,1.13181e00,2.61594e-03,
		  5.87597e-02,2.77564e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[309][j] = _itmp309[j];
			}
		}
		{ static float _itmp310[] = {1.15793e05,1.13141e00,2.60024e-03,
		  5.85782e-02,2.74275e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[310][j] = _itmp310[j];
			}
		}
		{ static float _itmp311[] = {1.16300e05,1.13105e00,2.58649e-03,
		  5.84129e-02,2.71347e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[311][j] = _itmp311[j];
			}
		}
		{ static float _itmp312[] = {1.16804e05,1.13095e00,2.58321e-03,
		  5.83762e-02,2.68636e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[312][j] = _itmp312[j];
			}
		}
		{ static float _itmp313[] = {1.17320e05,1.13056e00,2.56602e-03,
		  5.82069e-02,2.65719e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[313][j] = _itmp313[j];
			}
		}
		{ static float _itmp314[] = {1.17636e05,1.13059e00,2.56948e-03,
		  5.82217e-02,2.64415e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[314][j] = _itmp314[j];
			}
		}
		{ static float _itmp315[] = {1.18050e05,1.13032e00,2.55894e-03,
		  5.81028e-02,2.62306e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[315][j] = _itmp315[j];
			}
		}
		{ static float _itmp316[] = {1.18894e05,1.12981e00,2.53744e-03,
		  5.78760e-02,2.57281e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[316][j] = _itmp316[j];
			}
		}
		{ static float _itmp317[] = {1.19150e05,1.12967e00,2.53270e-03,
		  5.78189e-02,2.56337e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[317][j] = _itmp317[j];
			}
		}
		{ static float _itmp318[] = {1.19556e05,1.12962e00,2.53230e-03,
		  5.77995e-02,2.54488e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[318][j] = _itmp318[j];
			}
		}
		{ static float _itmp319[] = {1.19972e05,1.12914e00,2.51319e-03,
		  5.75804e-02,2.52355e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[319][j] = _itmp319[j];
			}
		}
		{ static float _itmp320[] = {1.20318e05,1.12868e00,2.49798e-03,
		  5.73674e-02,2.50694e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[320][j] = _itmp320[j];
			}
		}
		{ static float _itmp321[] = {1.20800e05,1.12863e00,2.49572e-03,
		  5.73535e-02,2.48435e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[321][j] = _itmp321[j];
			}
		}
		{ static float _itmp322[] = {1.21051e05,1.12890e00,2.50617e-03,
		  5.74894e-02,2.47794e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[322][j] = _itmp322[j];
			}
		}
		{ static float _itmp323[] = {1.22186e05,1.12812e00,2.47567e-03,
		  5.71245e-02,2.41275e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[323][j] = _itmp323[j];
			}
		}
		{ static float _itmp324[] = {1.22462e05,1.12820e00,2.47849e-03,
		  5.71775e-02,2.40426e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[324][j] = _itmp324[j];
			}
		}
}
void hpfitData6(void)
{
	int j;
		{ static float _itmp325[] = {1.22787e05,1.12778e00,2.46255e-03,
		  5.69852e-02,2.39018e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[325][j] = _itmp325[j];
			}
		}
		{ static float _itmp326[] = {1.23194e05,1.12772e00,2.46152e-03,
		  5.69624e-02,2.37334e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[326][j] = _itmp326[j];
			}
		}
		{ static float _itmp327[] = {1.23605e05,1.12738e00,2.44892e-03,
		  5.68079e-02,2.35501e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[327][j] = _itmp327[j];
			}
		}
		{ static float _itmp328[] = {1.23996e05,1.12716e00,2.44139e-03,
		  5.67088e-02,2.33863e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[328][j] = _itmp328[j];
			}
		}
		{ static float _itmp329[] = {1.24428e05,1.12704e00,2.43702e-03,
		  5.66622e-02,2.32064e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[329][j] = _itmp329[j];
			}
		}
		{ static float _itmp330[] = {1.24798e05,1.12680e00,2.42883e-03,
		  5.65549e-02,2.30569e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[330][j] = _itmp330[j];
			}
		}
		{ static float _itmp331[] = {1.25153e05,1.12656e00,2.42021e-03,
		  5.64523e-02,2.29168e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[331][j] = _itmp331[j];
			}
		}
		{ static float _itmp332[] = {1.25678e05,1.12620e00,2.40622e-03,
		  5.62834e-02,2.26796e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[332][j] = _itmp332[j];
			}
		}
		{ static float _itmp333[] = {1.26128e05,1.12597e00,2.39694e-03,
		  5.61878e-02,2.24925e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[333][j] = _itmp333[j];
			}
		}
		{ static float _itmp334[] = {1.27072e05,1.12567e00,2.38534e-03,
		  5.60563e-02,2.20459e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[334][j] = _itmp334[j];
			}
		}
		{ static float _itmp335[] = {1.27564e05,1.12562e00,2.38133e-03,
		  5.60437e-02,2.18532e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[335][j] = _itmp335[j];
			}
		}
		{ static float _itmp336[] = {1.28374e05,1.12534e00,2.37096e-03,
		  5.59177e-02,2.14923e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[336][j] = _itmp336[j];
			}
		}
		{ static float _itmp337[] = {1.29717e05,1.12455e00,2.34079e-03,
		  5.55498e-02,2.08563e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[337][j] = _itmp337[j];
			}
		}
		{ static float _itmp338[] = {1.30155e05,1.12440e00,2.33443e-03,
		  5.54879e-02,2.06993e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[338][j] = _itmp338[j];
			}
		}
		{ static float _itmp339[] = {1.30404e05,1.12429e00,2.33039e-03,
		  5.54440e-02,2.06357e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[339][j] = _itmp339[j];
			}
		}
		{ static float _itmp340[] = {1.30812e05,1.12419e00,2.32702e-03,
		  5.54081e-02,2.04981e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[340][j] = _itmp340[j];
			}
		}
		{ static float _itmp341[] = {1.31503e05,1.12394e00,2.31799e-03,
		  5.52939e-02,2.02235e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[341][j] = _itmp341[j];
			}
		}
		{ static float _itmp342[] = {1.34111e05,1.12274e00,2.26943e-03,
		  5.47373e-02,1.90725e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[342][j] = _itmp342[j];
			}
		}
		{ static float _itmp343[] = {1.34519e05,1.12263e00,2.26467e-03,
		  5.46931e-02,1.89491e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[343][j] = _itmp343[j];
			}
		}
		{ static float _itmp344[] = {1.34850e05,1.12250e00,2.26020e-03,
		  5.46412e-02,1.88579e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[344][j] = _itmp344[j];
			}
		}
		{ static float _itmp345[] = {1.35145e05,1.12245e00,2.25984e-03,
		  5.46190e-02,1.87852e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[345][j] = _itmp345[j];
			}
		}
		{ static float _itmp346[] = {1.35648e05,1.12207e00,2.24688e-03,
		  5.44442e-02,1.86139e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[346][j] = _itmp346[j];
			}
		}
		{ static float _itmp347[] = {1.36053e05,1.12216e00,2.25029e-03,
		  5.44993e-02,1.85041e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[347][j] = _itmp347[j];
			}
		}
		{ static float _itmp348[] = {1.36537e05,1.12178e00,2.23575e-03,
		  5.43243e-02,1.83440e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[348][j] = _itmp348[j];
			}
		}
		{ static float _itmp349[] = {1.36946e05,1.12172e00,2.23520e-03,
		  5.42987e-02,1.82286e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[349][j] = _itmp349[j];
			}
		}
		{ static float _itmp350[] = {1.37256e05,1.12130e00,2.21984e-03,
		  5.41103e-02,1.81388e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[350][j] = _itmp350[j];
			}
		}
		{ static float _itmp351[] = {1.37738e05,1.12120e00,2.21614e-03,
		  5.40724e-02,1.79956e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[351][j] = _itmp351[j];
			}
		}
		{ static float _itmp352[] = {1.38100e05,1.12094e00,2.20859e-03,
		  5.39509e-02,1.78931e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[352][j] = _itmp352[j];
			}
		}
		{ static float _itmp353[] = {1.38570e05,1.12082e00,2.20585e-03,
		  5.38944e-02,1.77558e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[353][j] = _itmp353[j];
			}
		}
		{ static float _itmp354[] = {1.39081e05,1.12049e00,2.19167e-03,
		  5.37511e-02,1.75969e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[354][j] = _itmp354[j];
			}
		}
		{ static float _itmp355[] = {1.39724e05,1.12050e00,2.19337e-03,
		  5.37621e-02,1.74034e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[355][j] = _itmp355[j];
			}
		}
		{ static float _itmp356[] = {1.40171e05,1.12010e00,2.17775e-03,
		  5.35842e-02,1.72697e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[356][j] = _itmp356[j];
			}
		}
		{ static float _itmp357[] = {1.40632e05,1.11991e00,2.17210e-03,
		  5.34985e-02,1.71400e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[357][j] = _itmp357[j];
			}
		}
		{ static float _itmp358[] = {1.41091e05,1.11980e00,2.16751e-03,
		  5.34529e-02,1.70155e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[358][j] = _itmp358[j];
			}
		}
		{ static float _itmp359[] = {1.41495e05,1.11962e00,2.16063e-03,
		  5.33793e-02,1.69096e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[359][j] = _itmp359[j];
			}
		}
		{ static float _itmp360[] = {1.41895e05,1.11942e00,2.15418e-03,
		  5.32886e-02,1.68048e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[360][j] = _itmp360[j];
			}
		}
		{ static float _itmp361[] = {1.42355e05,1.11912e00,2.14449e-03,
		  5.31519e-02,1.66763e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[361][j] = _itmp361[j];
			}
		}
		{ static float _itmp362[] = {1.42738e05,1.11907e00,2.14452e-03,
		  5.31304e-02,1.65852e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[362][j] = _itmp362[j];
			}
		}
		{ static float _itmp363[] = {1.43087e05,1.11871e00,2.13168e-03,
		  5.29652e-02,1.64951e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[363][j] = _itmp363[j];
			}
		}
		{ static float _itmp364[] = {1.43572e05,1.11863e00,2.12866e-03,
		  5.29320e-02,1.63693e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[364][j] = _itmp364[j];
			}
		}
		{ static float _itmp365[] = {1.43948e05,1.11845e00,2.12316e-03,
		  5.28519e-02,1.62782e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[365][j] = _itmp365[j];
			}
		}
		{ static float _itmp366[] = {1.44374e05,1.11823e00,2.11590e-03,
		  5.27563e-02,1.61697e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[366][j] = _itmp366[j];
			}
		}
		{ static float _itmp367[] = {1.44863e05,1.11800e00,2.10719e-03,
		  5.26528e-02,1.60406e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[367][j] = _itmp367[j];
			}
		}
		{ static float _itmp368[] = {1.45510e05,1.11776e00,2.09977e-03,
		  5.25456e-02,1.58617e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[368][j] = _itmp368[j];
			}
		}
		{ static float _itmp369[] = {1.45835e05,1.11765e00,2.09649e-03,
		  5.24969e-02,1.57934e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[369][j] = _itmp369[j];
			}
		}
		{ static float _itmp370[] = {1.46277e05,1.11749e00,2.09060e-03,
		  5.24318e-02,1.56865e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[370][j] = _itmp370[j];
			}
		}
		{ static float _itmp371[] = {1.46650e05,1.11735e00,2.08680e-03,
		  5.23694e-02,1.56032e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[371][j] = _itmp371[j];
			}
		}
		{ static float _itmp372[] = {1.46936e05,1.11707e00,2.07555e-03,
		  5.22466e-02,1.55434e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[372][j] = _itmp372[j];
			}
		}
}
void hpfitData7(void)
{
	int j;
		{ static float _itmp373[] = {1.47259e05,1.11709e00,2.07759e-03,
		  5.22580e-02,1.54823e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[373][j] = _itmp373[j];
			}
		}
		{ static float _itmp374[] = {1.47727e05,1.11685e00,2.07066e-03,
		  5.21501e-02,1.53672e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[374][j] = _itmp374[j];
			}
		}
		{ static float _itmp375[] = {1.47876e05,1.11688e00,2.07234e-03,
		  5.21669e-02,1.53618e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[375][j] = _itmp375[j];
			}
		}
		{ static float _itmp376[] = {1.48024e05,1.11674e00,2.06766e-03,
		  5.21101e-02,1.53509e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[376][j] = _itmp376[j];
			}
		}
		{ static float _itmp377[] = {1.48008e05,1.11660e00,2.06369e-03,
		  5.20502e-02,1.53905e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[377][j] = _itmp377[j];
			}
		}
		{ static float _itmp378[] = {1.48445e05,1.11638e00,2.05683e-03,
		  5.19494e-02,1.52867e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[378][j] = _itmp378[j];
			}
		}
		{ static float _itmp379[] = {1.48997e05,1.11631e00,2.05437e-03,
		  5.19214e-02,1.51541e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[379][j] = _itmp379[j];
			}
		}
		{ static float _itmp380[] = {1.49371e05,1.11605e00,2.04432e-03,
		  5.18076e-02,1.50704e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[380][j] = _itmp380[j];
			}
		}
		{ static float _itmp381[] = {1.49654e05,1.11597e00,2.04379e-03,
		  5.17688e-02,1.50204e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[381][j] = _itmp381[j];
			}
		}
		{ static float _itmp382[] = {1.49904e05,1.11588e00,2.03935e-03,
		  5.17409e-02,1.49810e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[382][j] = _itmp382[j];
			}
		}
		{ static float _itmp383[] = {1.50161e05,1.11571e00,2.03535e-03,
		  5.16593e-02,1.49361e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[383][j] = _itmp383[j];
			}
		}
		{ static float _itmp384[] = {1.50455e05,1.11566e00,2.03475e-03,
		  5.16429e-02,1.48849e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[384][j] = _itmp384[j];
			}
		}
		{ static float _itmp385[] = {1.50242e05,1.11567e00,2.03502e-03,
		  5.16517e-02,1.49861e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[385][j] = _itmp385[j];
			}
		}
		{ static float _itmp386[] = {1.50182e05,1.11556e00,2.03282e-03,
		  5.16038e-02,1.50379e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[386][j] = _itmp386[j];
			}
		}
		{ static float _itmp387[] = {1.50448e05,1.11538e00,2.02829e-03,
		  5.15247e-02,1.49901e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[387][j] = _itmp387[j];
			}
		}
		{ static float _itmp388[] = {1.50704e05,1.11536e00,2.02832e-03,
		  5.15188e-02,1.49505e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[388][j] = _itmp388[j];
			}
		}
		{ static float _itmp389[] = {1.50917e05,1.11531e00,2.02629e-03,
		  5.15064e-02,1.49232e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[389][j] = _itmp389[j];
			}
		}
		{ static float _itmp390[] = {1.51058e05,1.11514e00,2.02148e-03,
		  5.14281e-02,1.49126e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[390][j] = _itmp390[j];
			}
		}
		{ static float _itmp391[] = {1.51638e05,1.11496e00,2.01559e-03,
		  5.13484e-02,1.47731e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[391][j] = _itmp391[j];
			}
		}
		{ static float _itmp392[] = {1.52054e05,1.11489e00,2.01387e-03,
		  5.13197e-02,1.46865e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[392][j] = _itmp392[j];
			}
		}
		{ static float _itmp393[] = {1.52426e05,1.11462e00,2.00420e-03,
		  5.11978e-02,1.46066e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[393][j] = _itmp393[j];
			}
		}
		{ static float _itmp394[] = {1.52788e05,1.11463e00,2.00531e-03,
		  5.12052e-02,1.45392e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[394][j] = _itmp394[j];
			}
		}
		{ static float _itmp395[] = {1.53097e05,1.11446e00,1.99936e-03,
		  5.11318e-02,1.44815e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[395][j] = _itmp395[j];
			}
		}
		{ static float _itmp396[] = {1.53416e05,1.11434e00,1.99526e-03,
		  5.10830e-02,1.44231e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[396][j] = _itmp396[j];
			}
		}
		{ static float _itmp397[] = {1.53650e05,1.11430e00,1.99430e-03,
		  5.10689e-02,1.43914e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[397][j] = _itmp397[j];
			}
		}
		{ static float _itmp398[] = {1.54003e05,1.11402e00,1.98569e-03,
		  5.09380e-02,1.43181e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[398][j] = _itmp398[j];
			}
		}
		{ static float _itmp399[] = {1.54267e05,1.11390e00,1.98162e-03,
		  5.08885e-02,1.42757e-14};
		for(  j = 0; j < 5; j++ )
		{
			PHH[399][j] = _itmp399[j];
			}
		}
		{ static float _itmp400[] = {1.54505e05,1.11395e00,1.98405e-03,
		  5.09196e-02,1.42465e-14};
		for( j = 0; j < 5; j++ )
		{
			PHH[400][j] = _itmp400[j];
			}
		}

#	ifdef DEBUG_FUN
	fputs( "<+>blkdata0()\n", debug_fp );
#	endif
	/*block data for hydrogen atom photoionization cross sections */

#	ifdef DEBUG_FUN
	fputs( " <->blkdata0()\n", debug_fp );
#	endif
}
#ifdef _MSC_VER
#pragma warning( default : 4305 )/* enable warning that conditional expression*/
#endif

