/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*hmole determine populations of hydrogen molecules */
/*hmole_old determine populations of hydrogen molecules */
/*hmirat compute radiative association rate for H- */
/* >> chng 02 nov 7 rjrw, Mole Moreliano:
 *   changes to linearized iterative form */
/* from Robin Williams:
The process for these kind of problems seems to be pretty uniform:
switch printsol on to check which terms in the chemical matrix change,
and next to switch on the prints in the matrix assembly which apply to
the species involved to find what reactions are involved.  It's a bit
of a pain grepping down to find the 47th reaction, so I guess some
kind of naming scheme for the reactions may come in handy (I'd thought
about generating the in and out vectors from a text string, e.g. "H +
H => H2" ;-), but you'd have to verify uniqueness).
*/
#include "cddefines.h"
#include "physconst.h"
#include "iso.h"
#include "grainvar.h"
#include "dense.h"
#include "secondaries.h"
#include "colden.h"
#include "opacity.h"
#include "heavy.h"
#include "hevmolec.h"
#include "rfield.h"
#include "thermal.h"
#include "converge.h"
#include "ionrec.h"
#include "heat.h"
#include "punch.h"
#include "timesc.h"
#include "trace.h"
#include "nomole.h"
#include "phycon.h"
#include "doppvel.h"
#include "linpack.h"
#include "rtescprob.h"
#include "gammas.h"
#include "hmi.h"
#include "hmrate.h"
#include "h2.h"
#include "dynamics.h"
#include "radius.h"
#include "hmole.h"

/*s_ionmole ionmole;*/

/* Define to verify chemistry solution */
#if 0
#if !defined(NDEBUG)
/* >>chng 02 dec 21, line 14 changed from 
	if (fabs(total) > 1e-20 && fabs(total) > 1e-14*mtotal) { 
	to
	if (fabs(total) > 1e-20 && fabs(total) > 1e-8*mtotal) {  */
#define AUDIT(a)	{ \
		double total, mtotal; \
		for (i=0;i<N_H_MOLEC;i++) { \
			total = 0.; \
			for (j=0;j<N_H_MOLEC;j++) { \
				total += c[i][j]*nprot[j]; \
			} \
			if (fabs(total) > 1e-6*fabs(c[i][i]*nprot[i])) { \
					fprintf(ioQQQ,"PROBLEM Subtotal1 %c %.2e\n",a,fabs(total)/fabs(c[i][i]*nprot[i])); \
					fprintf(ioQQQ,"Species %li Total %g Diag %g\n",i,total,c[i][i]*nprot[i]); \
			} \
		} \
    total = mtotal = 0.;for (j=0;j<N_H_MOLEC;j++) { total += bvec[j]*nprot[j]; mtotal += fabs(bvec[j]*nprot[j]); }\
			if (fabs(total) > 1e-30 && fabs(total) > 1e-10*rtot) { \
					fprintf(ioQQQ,"PROBLEM Subtotal2 %c %.2e\n",a,fabs(total)/mtotal); \
					fprintf(ioQQQ,"RHS Total %g cf %g\n",total,mtotal); \
			} else if (a == '.' && fabs(total) > 1e-7*mtotal)  { \
					fprintf(ioQQQ,"WARNING zone %li Hmole RHS conservation error %.2e of %.2e\n",nzone,total,mtotal); \
					fprintf(ioQQQ,"(may be due to high rate equilibrium reactions)\n"); \
			} \
	}
#else
#define AUDIT /* nothing */
#endif
/* #define NDEBUG 0 */

#endif



/*fprintf(ioQQQ,"%c: %15.4g",a,bvec[0]+bvec[1]); for (i=2; i<N_H_MOLEC;i++) fprintf(ioQQQ,"%15.4g ",bvec[i]); fprintf(ioQQQ,"\n");\ */

/*hmirat computes radiative association rate for H- */
static double hmirat(double te);
/* Take one Newton step of the chemical network */
void hmole_step(int *nFixup, double *error);

void hmole( void )
{
	int nFixup, i;
	double error;

	enum {MAXFIX = 10};
	/* will be used to keep track of neg solns */
	nFixup = 0;
	error = 1.;

	for (i=0; i < 6 && error > 1e-4;i++) 
	{
		hmole_step(&nFixup, &error);
	}
	if (i == 6 || nFixup != 0)
		fprintf(ioQQQ,"Hmole, zone %li: %d iters, %d bad; final error: %g\n",nzone,i,nFixup,error);

	if (0 && nFixup >= MAXFIX)
	{
		ShowMe();
		puts( "[Stop in hmole]" );
		cdEXIT(EXIT_FAILURE);
	}
}

#define MAXREACTANTS 3
#define MAXRATES 3
#define MAXPRODUCTS  4
/* Calculate number of elements in an integer vector */
#define INTSZ(a)     (sizeof(a)/sizeof(int))

typedef struct reaction_s {
	int index;
	double ratek;
	int nreactants, nrates, nproducts;
	int reactants[MAXREACTANTS];
	int rates[MAXREACTANTS];
	int products[MAXPRODUCTS];
	struct reaction_s *next;
} reaction;

/* Generate new element for reaction list */
reaction *newreaction(int rindex, int *in, int nin, int *out, int nout, int *rate, int nrate)
{
	static reaction *list = NULL, *r;
	static int poolsize=1, index = 0;
	int i;

	/* fprintf(ioQQQ,"New reaction %d %d %d\n",rindex,nin,nout); */

	/* default assumption for chemical kinetics */
	if (rate == NULL) {
		rate=in;
		nrate=nin;
	}

	if (list == NULL || index == poolsize) 
		{
		poolsize <<=1;
		if( (list = ((reaction *)MALLOC( (size_t)poolsize*sizeof(reaction) ))) == NULL )
			BadMalloc();
		index = 0;
	}
	/* fprintf(ioQQQ,"Getting element %d+1 of %d\n",index,poolsize); */
	r = list+index;
	index++;
	r->next = NULL;
	r->index = rindex;
	assert (nin <= MAXREACTANTS && nout <= MAXPRODUCTS && nrate <= MAXRATES);
	r->nreactants =	nin;
	r->nrates =	nrate;
	r->nproducts  =	nout;
	for (i=0; i<r->nreactants; i++)
		r->reactants[i] = in[i];
	for (i=0; i<r->nrates; i++)
		r->rates[i] = rate[i];
	for (i=0; i<r->nproducts; i++)
		r->products[i] = out[i];
	return r;
}

#define ABSLIM  1e-12
void hmole_step(int *nFixup, double *error)
{
	enum {PRINTSOL = 0};

	char chLab[N_H_MOLEC][5];

	/* will punch debug output to this file */
	FILE*ioFile;

	long int i, 
	  ipConserve, 
	  ipiv[N_H_MOLEC], 
	  ipLo,
	  j, 
	  limit ,
	  merror, 
	  nd,
		mol;

	int nprot[N_H_MOLEC],
		printsol = PRINTSOL;

	int lgNegPop,
		iworst;

	double batach, 
	  bh2dis,
	  /*bh2h22hh2,*/
	  bhneut, 
	  c3bod,
	  cionhm, 
	  corr, 
	  H2star_deexcit,
	  damper,
	  deexc_htwo,
	  deexc_hneut,
	  desh2p, 
	  eh2hh, 
	  eh3_h2h,
	  etmp,
	  eh3p_3h,
	  ex3hp, 
	  excit,
	  exph2, 
	  exphmi, 
	  exphp, 
	  fac, 
	  faneut, 
	  fhneut, 
	  gamhd, 
	  gamheh, 
	  h1fnd, 
	  h1rat, 
	  h2pcin, 
	  h2ph3p, 
	  h2phhp, 
	  h2pion, 
	  H2star_excit ,
	  h32h2, 
	  h3petc, 
	  h3ph2p, 
	  ph3lte, 
	  radasc, 
	  radath, 
	  ratach, 
	  rh2h2p, 
	  saha,
	  h2hph3p,
	  hmihph2p,
	  h2phmh2h,
	  h2phmhhh,
	  h3phmh2hh,
	  /*h2h22hh2,*/
	  sum,
		fracneg,
		fracnegtmp,
		fracnegfac,
		hatom,
		rtot,
		conserve,
		rate,
		ratek,
		rated,
		rates[MAXREACTANTS],
		sinkrate[MAXREACTANTS];

	/* lte populations of H-, H2, and H2+ */
	static double phmlte, 
	  phplte,
	  h3pcop;

	static double HMinus_induc_rec_cooling, 
	  eh2hhm, 
	  eh2old, 
	  gamtwo, 
	  gh2exc_dissoc,
	  h2esc, 
	  h2phet, 
	  HMinus_photo_heat, 
	  protons, 
	  HMinus_induc_rec_rate, 
	  hminus_rad_attach, 
	  th2;
	static long int nzoneEval=9999;

	static double /*amat[N_H_MOLEC][N_H_MOLEC], */
	  b2pcin, 
	  *amat=NULL,
	  *bvec=NULL/*[N_H_MOLEC]*/, 
	  *bold=NULL/*[N_H_MOLEC]*/, 
	  **c=NULL/*[N_H_MOLEC+1][N_H_MOLEC+1]*/, 
	  plte;

	/* if this is still true then must create space for arrays */
	static int lgMustMalloc = TRUE;

	static reaction *rlist = NULL;
	reaction *r;
	long int rindex, ratei, ratej;

#	ifdef DEBUG_FUN
	fputs( "<+>hmole()\n", debug_fp );
#	endif

	if( lgMustMalloc )
	{
		/* on very first call must create space */
		lgMustMalloc = FALSE;

		if( (bvec = ((double*)MALLOC( (size_t)N_H_MOLEC*sizeof(double) ))) == NULL )
			BadMalloc();
		if( (bold = ((double*)MALLOC( (size_t)N_H_MOLEC*sizeof(double) ))) == NULL )
			BadMalloc();
		if( (amat = ((double*)MALLOC( (size_t)(N_H_MOLEC*N_H_MOLEC)*sizeof(double) ))) == NULL )
			BadMalloc();
		if( (c = ((double**)MALLOC( (size_t)N_H_MOLEC*sizeof(double *) ))) == NULL )
			BadMalloc();
		for( i=0; i<N_H_MOLEC; ++i )
		{
			if( (c[i] = ((double*)MALLOC( (size_t)N_H_MOLEC*sizeof(double) ))) == NULL )
				BadMalloc();
		}
	}

	/* Assume no error for cases with abundances set */
	*error = 0;

	/* there are two "no molecules" options, the no co, which turns off everything,
	 * and the no n2, which only turns off the h2.  in order to not kill the co
	 * part we still need to compute the hydrogen network here, and then set h2 to
	 * small values */
	/*TODO - this should have nomole.lgNoH2Mole */
	/* >> chng 03 jan 15 rjrw -- suddenly switching off molecules confuses the solvers... */
	if( nomole.lgNoH2Mole ) /*   || phycon.te > 1e5 ) */
	{
		/* thtmol = 0. */
		dense.xMolecules[ipHYDROGEN] = 0.;

		/* these are the molecular species */
		for (mol=0;mol<N_H_MOLEC;mol++) 
		{
			hmi.Molec[mol] = 0.;
		}
		hmi.Molec[ipMH] = dense.xIonDense[ipHYDROGEN][0];
		hmi.Molec[ipMHp] = dense.xIonDense[ipHYDROGEN][1];
		/* this is where the Emline struc expects to find the H2 abundance */
		dense.xIonDense[LIMELM+2][0] = 0.;
		hmi.hmihet = 0.;
		hmi.h2plus_heat = 0.;
		hmi.H2Opacity = 0.;
		hmi.hmicol = 0.;
		hmi.HeatH2Dish_TH85 = 0.;
		hmi.HeatH2Dexc_TH85 = 0.;
		hmi.deriv_HeatH2Dexc_TH85 = 0.;
		hmi.hmidep = 1.;
		hmi.rh2dis = 0.;
		hmi.HalphaHmin = 0.;

		ionrec.GrnIonRec = 0.;
		
#		ifdef DEBUG_FUN
		fputs( " <->hmole()\n", debug_fp );
#		endif
		return;
	}

	/* >>chng 02 may 27, add option to continue in full H2 limit */
	/* >>chng 02 jun 19, add option to force H2 abundance, for testing h2 molecules */
	if( hmi.lgAll_H2 || h2.frac_abund>0.)
	{
		for (mol=0;mol<N_H_MOLEC;mol++) 
		{
			hmi.Molec[mol] = 0.;
		}
		dense.xIonDense[ipHYDROGEN][0] = dense.xIonDense[ipHYDROGEN][1] = 0.;
		if( h2.frac_abund>0.)
		{
			hmi.Molec[ipMH2] = (float)(dense.gas_phase[ipHYDROGEN] * h2.frac_abund);
		}
		else
		{
			hmi.Molec[ipMH2] = (float)(dense.gas_phase[ipHYDROGEN]/2.);
		}

		hmi.hmihet = 0.;
		hmi.h2plus_heat = 0.;
		hmi.H2Opacity = 0.;
		hmi.hmicol = 0.;
		hmi.HeatH2Dish_TH85 = 0.;
		hmi.HeatH2Dexc_TH85 = 0.;
		hmi.deriv_HeatH2Dexc_TH85 = 0.;
		hmi.hmidep = 1.;
		hmi.HalphaHmin = 0.;
		ionrec.GrnIonRec = 0.;

		/* Not including H/H+ */
		for (mol=0;mol<N_H_MOLEC;mol++) 
		{
			dense.xMolecules[ipHYDROGEN] += hmi.Molec[mol]*hmi.protons[mol];
		}
		dense.xMolecules[ipHYDROGEN] -= (hmi.Molec[ipMH]+hmi.Molec[ipMHp]);

		dense.xMolecules[ipHYDROGEN] += hevmolec.hevmol[ipCH]+ hevmolec.hevmol[ipCHP]+ hevmolec.hevmol[ipOH]+
		hevmolec.hevmol[ipOHP]+ 2.f*hevmolec.hevmol[ipH2O]+ 2.f*hevmolec.hevmol[ipH2OP]+	
		3.f*hevmolec.hevmol[ipH3OP]+2.f*hevmolec.hevmol[ipCH2P]+
		2.f*hevmolec.hevmol[ipCH2]+ 3.f*hevmolec.hevmol[ipCH3];
	
#		ifdef DEBUG_FUN
		fputs( " <->hmole()\n", debug_fp );
#		endif
		return;
	}

	/* this checks whether very first call in this model */
	if( !conv.nTotalIoniz )
	{
		phmlte = 0.; 
		hmi.ph2lte = 0.;
		phplte = 0.;
	}

	hmi.Molec[ipMH] = dense.xIonDense[ipHYDROGEN][0];
	hmi.Molec[ipMHp] = dense.xIonDense[ipHYDROGEN][1];
	for (mol=0;mol<N_H_MOLEC;mol++) 
	{
		bold[mol] = hmi.Molec[mol];
	}

	/* this is the total density (cm-3) of hydrogen in the n=2 state
	hn2 = 
		(EmisLines[ipH_LIKE][ipHYDROGEN][ipH2s][ipH1s].PopHi +
		EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].PopHi ) *
			dense.xIonDense[ipHYDROGEN][1]; */

	/* population of H- in LTE
	 * IP is 0.754 eV
	 * LTE population of H minus - cm^3 */
	saha = sqrt(SAHA2);
	exphmi = sexp(8.745e3/phycon.te);
	if( exphmi > 0. )
	{
		/* these are ratio n*(H-)/[  n*(ne) n*(Ho)  ] */
		phmlte = saha/(phycon.te32*exphmi)*(1./(2.*2.));
	}
	else
	{
		phmlte = 0.;
	}

	/* population of H2 in LTE
	 * dissociation energy is 4.477eV */
	exph2 = sexp(5.195e4/phycon.te);

	/* Can the first clause be entirely removed now? */
  if (0) {
		if( exph2 > 0. )
		{
			if( nzone <= 0 )
			{
				damper = 0.;
			}
			else
			{
				damper = 0.5;
			}
			/* extra factor accounts for mass of H instead of e- in SAHA
			 * last factor was put in ver 85.23, missing before */
			hmi.ph2lte = (1. - damper)*saha/(phycon.te32*exph2)*
				(1./(2.*2.))*3.634e-5 + damper*hmi.ph2lte;
		}
		else
		{
			hmi.ph2lte = 0.;
		}
	}
	else
	{
		if( exph2 > 0. ) 
		{
			hmi.ph2lte = saha/(phycon.te32*exph2)*(1./(2.*2.))*3.634e-5;
		}
		else
		{
			hmi.ph2lte = 0.;
		}
	}
	
	{
		/*@-redef@*/
		/* often the H- route is the most efficient formation mechanism for H2,
		 * will be through rate called ratach
		 * this debug print statement is to trace h2 oscillations */
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nzone>187&& iteration > 1/**/)
		{
			/* rapid increase in H2 density caused by rapid increase in hmi.ph2lte */
			fprintf(ioQQQ,"ph2lteee\t%.2e\t%.1e\t%.1e\n", 
				hmi.ph2lte, 
				exph2,
				phycon.te);
		}
	}

	/* population of H2+ in LTE, phplte is H_2+/H / H+
	 * dissociation energy is 2.647 */
	exphp = sexp(3.072e4/phycon.te);
	if( exphp > 0. )
	{
		/* stat weight of H2+ is 4
		 * last factor was put in ver 85.23, missing before */
		phplte = saha/(phycon.te32*exphp)*(4./(2.*1.))*3.634e-5;
	}
	else
	{
		phplte = 0.;
	}

	/* population of H3+ in LTE, ph3lte is H_3+/( H2+ H+ )
	 * dissociation energy is 2.647 */
	ex3hp = sexp(1.882e4/phycon.te);
	if( ex3hp > 0. )
	{
		/* stat weight of H2+ is 4
		 * last factor was put in ver 85.23, missing before */
		ph3lte = saha/(phycon.te32*ex3hp)*(4./(2.*1.))*3.634e-5;
	}
	else
	{
		ph3lte = 0.;
	}

	hminus_rad_attach = hmirat(phycon.te);
	/* cooling due to radiative attachment */
	hmi.hmicol = hminus_rad_attach*EN1RYD*phycon.te*1.15e-5;

	/*fprintf(ioQQQ,"%.2e %.2e %.2e %.2e\n", phycon.te, hminus_rad_attach , hmi.hmicol,
		hmi.hmicol/(hminus_rad_attach*EN1RYD*phycon.te*1.15e-5) );*/

	/* get per unit vol */
	hminus_rad_attach *= dense.eden;
	hmi.hmicol *= dense.eden*bold[ipMH]; /* was dense.xIonDense[ipHYDROGEN][0]; */

	/* ================================================================= */
	/* evaluate H- photodissociation rate, induc rec and rec cooling rates */
	/* >>chng 00 dec 24, add test so that photo rate only reevaluated two times per zone.
	 * in grain-free models this was sometimes dominated by Lya and so oscillated.  
	 * especially bad in primal.in - change 2 to 4 and primal.in will stop due to Lya oscil */

	if( 1 || /**/(conv.nPres2Ioniz < 2) || (nzone==0) || (nzoneEval!=nzone) )
	{
		/* >>chng 02 feb 16, add damper on H- photo rate, wild oscillations in Lya photo rate in 
		 * grain free models */
		static double hm_damper=0.25;
		static double phot_new=0. , phot_old;
		nzoneEval = nzone;

		/*hmi.HMinus_photo_rate = GammaBn( hmi.iphmin-1 , nhe1Com.nhe1[0] , opac.iphmop ,
			0.055502 , &HMinus_induc_rec_rate , &HMinus_induc_rec_cooling );*/
		phot_old = phot_new;
		phot_new = GammaBn( hmi.iphmin-1 , iso.ipIsoLevNIonCon[ipHE_LIKE][ipHELIUM][0] , opac.iphmop ,
			0.055502 , &HMinus_induc_rec_rate , &HMinus_induc_rec_cooling );
		if( 1 || nzone==0 )
		{
			hmi.HMinus_photo_rate = phot_new;
		}
		else
		{
			hmi.HMinus_photo_rate = hm_damper*phot_new+ (1.-hm_damper)*phot_old;
		}

		{
			/* following should be set true to print populations */
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC)
			{
				fprintf(ioQQQ,"hminphoto\t%li\t%li\t%.2e\n", nzone, conv.nPres2Ioniz , hmi.HMinus_photo_rate );
			}
		}

		/* save H- photodissociation heating */
		HMinus_photo_heat = heat.HeatNet;

		/* induced recombination */
		HMinus_induc_rec_rate *= phmlte*dense.eden;

		/* induced recombination cooling per unit volume */
		HMinus_induc_rec_cooling *= phmlte*dense.eden*bold[ipMH]; /* dense.xIonDense[ipHYDROGEN][0]; */

		{
			/* following should be set true to debug H- photoionization rates */
			/*@-redef@*/
			enum {DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && nzone>187&& iteration > 1)
			{
				fprintf(ioQQQ,"hmoledebugg %li ",nzone);
				GammaPrt(
					hmi.iphmin-1 , iso.ipIsoLevNIonCon[ipHE_LIKE][ipHELIUM][0] , opac.iphmop ,
					/* io unit we will write to */
					ioQQQ, 
					/* total photo rate from previous call to GammaK */
					hmi.HMinus_photo_rate, 
					/* we will print contributors that are more than this rate */
					hmi.HMinus_photo_rate*0.05);
			}
		}
	}
	/* add on high energy ionization, assume hydrogen cross section
	 * n.b.; HGAMNC with secondaries */
	/* >>chng 00 dec 24, above goes to hei edge, no need for this, and was not important*/
	/*hmi.HMinus_photo_rate += iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s];*/

	/* ================================================================= */
	/* photodissociation by Lyman band absorption: esc prob treatment,
	* treatment based on 
	* >>refer	HI	abs	Tielens & Hollenbach 1985 ApJ 291, 722. */
	/* do up to carbon photo edge if carbon is turned on */
	/* >>>chng 00 apr 07, add test for whether element is turned on */
	if( dense.lgElmtOn[ipCARBON] )
	{
		/* carbon is turned on, use carbon 1 edge */
		ipLo = Heavy.ipHeavy[ipCARBON][0] - 1;
	}
	else
	{
		/* carbon truned off, use hydrogen balmer continuum */
		ipLo = iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH2s]-1;
	}
	hmi.GammaHabing = 0.;
	/* >>chng 00 apr 07 from explicit ipHeavy to ipLo */
	/* find total intensity over carbon-ionizing continuum */
	for( i=ipLo; i < iso.ipIsoLevNIonCon[ipH_LIKE][ipHYDROGEN][ipH1s]; i++ )
	{
		hmi.GammaHabing += (rfield.flux[i] + rfield.ConInterOut[i]+ 
		rfield.outlin[i])*rfield.anu[i];
	}

	/* now convert to Habing ISM units
	* GammaHabing is FUV continuum relative to Habing value */
	hmi.GammaHabing = (float)(hmi.GammaHabing*2.18e-11/1.6e-3);

	/* escape prob takes into account line shielding, 
	 * next is opacity then optical depth in H2 UV lines, using eqn A7 of TH85*/
	hmi.H2Opacity = (float)(1.2e-14*(1e5/DoppVel.doppler[ipHYDROGEN]));
	/* the typical Lyman -Werner H2 line optical depth eq A7 of TH85a */
	th2 = colden.colden[ipCOLH2]*hmi.H2Opacity;
	/* the escape probability - chance that continuum photon will penetrate to
	 * this depth to pump the Lyman Werner bands */
	h2esc = esc_PRD_1side(th2,1e-4);
	/*fprintf(ioQQQ," th2 %.3e h2esc %.3e\n", th2 , h2esc );*/

	/* cross section is eqn A8 of Tielens and Hollenbach 85a
	 * branching ratio of 10% is included, so 10x smaller than their number
	 * 10% lead to dissociation through H_2 + h nu => 2H */
	hmi.H2_Solomon_rate_TH85 = 3.4e-11 * hmi.GammaHabing * h2esc;

	/* this rate is equation 23 of
	 * Bertoldi, F., & Draine, B.T., 1996, 458, 222 */
	hmi.H2_Solomon_rate_BD96 = 4.6e-11 * hmi.GammaHabing * 
		pow( MAX2(1.,colden.colden[ipCOLH2]/1e14) , -0.75 );

	/* this is rate of photodissociation of H2 */
	hmi.H2_photodissoc_TH85 = hmi.GammaHabing*1e-11;

	/* at this point there are two or three independent estimates of the H2 dissociation rate.
	 * if the large H2 molecule is on, then hmi.H2_Solomon_rate_BigH2 has been defined in the last
	 * call to the large molecule.  Just above we have defined hmi.H2_Solomon_rate_TH85,
	 * the dissociation rate from Tielens & Hollenback 1985, and hmi.H2_Solomon_rate_BD96,
	 * the rate from Bertoldi & Draine 1996.  We can use any defined rate.  If the big H2
	 * molecule is on, use its rate.  If not, for now use the TH85 rate, since that is the
	 * rate we always have used in the past.
	 * The actual rate we will use is given by hmi.H2_Solomon_rate_used
	 */
	/* this is the Solomon process dissociation rate actually used */
	if( h2.lgH2ON  && hmi.lgBigH2_evaluated && hmi.lgH2_Chemistry_BigH2)
	{
		/* only update if this has been evaluated in this zone,
		 * on very first call this zone, leave at previous value */
		if( h2.nCallH2_this_zone )
		{
			/* the big molecule dissociation rate is true decays from each 
			 * excited electronic level into the continuum */ 
			hmi.H2_Solomon_rate_used = hmi.H2_Solomon_rate_BigH2;
			hmi.H2_photodissoc_used = hmi.H2_photodissoc_BigH2;
		}
	}
	else
	{
		/* the TH85 rate  */
		hmi.H2_Solomon_rate_used = hmi.H2_Solomon_rate_TH85;
		hmi.H2_photodissoc_used = hmi.H2_photodissoc_TH85;
	}

	{
		/*@-redef@*/
		/* often the H- route is the most efficient formation mechanism for H2,
		 * will be through rate called ratach
		 * this debug print statement is to trace h2 oscillations */
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && h2.lgH2ON )
		{
			fprintf(ioQQQ," Solomon H2 dest rates: TH85 %.2e BD96 %.2e Big %.2e \n",
				hmi.H2_Solomon_rate_TH85,
				hmi.H2_Solomon_rate_BD96,
				hmi.H2_Solomon_rate_BigH2 );
		}
	}
	/* collisional ionization of H-, rate from Janev, Langer et al. */
	if( phycon.te < 3074. )
	{
		cionhm = 1.46e-32*(powi(phycon.te,6))*phycon.sqrte*exphmi;
	}
	else if( phycon.te >= 3074. && phycon.te < 30000. )
	{

		/* >>chng 03 mar 07, from above to below */
		/*cionhm = 5.9e-19*phycon.te*phycon.te*phycon.sqrte*phycon.te03*
		  phycon.te01*phycon.te01;*/
		cionhm = 5.9e-19*phycon.tesqrd*phycon.sqrte*phycon.te05;
	}
	else
	{
		cionhm = 3e-7;
	}

	/* ion recomb on grain surfaces */
	ionrec.GrnIonRec = 0.;
	/* H2 formation on grains;
	 * rate from 
	 * >>refer	H2	grain formation	Hollenback, D., & McKee, C.F., 1979, ApJS, 41, 555 eq 3.4 3.8 */
	if( gv.lgDustOn )
	{
		/* first get ion grain recombination rate */
		ionrec.GrnIonRec = 0.;
		for( nd=0; nd < gv.nBin; nd++ )
		{
			/* rate ions recombine on grain surface - used elsewhere
			 * based on 
			 * >>refer	grain	rec	Draine and Sutin 1987 ApJ 320, 803 eqn 5.15
			 * ionrec.grecon is usually 1, option to turn this process off
			 * >>chng 97 feb 24, following had SQRT( tdlow ) not sqrte,
			 * caught by Jon Slavin */
			/*TODO understand how this relates to Draine & Sutin equation */
			ionrec.GrnIonRec +=(float)(5.8e-13/phycon.sqrte*gv.bin[nd]->cnv_H_pCM3*ionrec.grecon);
			/* >>chng 02 oct 03, from > to >=, failed when neut turned off,
			 * caught by Jon Slavin */
			ASSERT( ionrec.GrnIonRec >= 0. );
		}
		/* >>chng 03 mar 07, hack to get reasonable behavior from GrnIonRec which uses
		 * old style grain physics; we should be using gv.GrainRecom though!!!, PvH */
		ionrec.GrnIonRec /= gv.nBin;

		/* H2 grain interactions
		 * H2 ortho - para conversion on grain surface,
		 * rate (s-1) all v,J levels go to 0 or 1, preserving nuclear spin */
		hmi.rate_h2_allX_2_J0_grains = 0.;
		/* rate (s-1) all v,J levels go to 0, regardless of nuclear spin */
		hmi.rate_h2_ortho_para_conserve = 0. ;

		/* loop over all grain species */
		for( nd=0; nd < gv.nBin; nd++ )
		{
			/* >>chng 02 feb 15, removed check tedust > 1.01, change in InitGrains
			 * guarantees that all relevant parameters are initialized, PvH */

			/* sticking probability, 2H + grain equation 3.7 of
			 * >>refer	grain	phys	Hollenbach, D.J., & McKee, C.F., 1979, ApJS, 41, 555,
			 * fraction of H impacts on grain surface that stick */
			/* this sticking probability is used for both HM79 and CT02 */
			double sticking_probability_H = 1./(1. + 0.04*sqrt(gv.bin[nd]->tedust+phycon.te) + 
				0.002*phycon.te + 8e-6*phycon.te*phycon.te);

			{
				/* fraction of impacts that produce H2 before evaporation from grain surface.
				* this is equation 3.4 of
				* >>refer	grain	phys	Hollenbach, D.J., & McKee, C.F., 1979, ApJS, 41, 555
				* 1e4 is ratio of total absorption sites to apropriate sites 
				* 920 is D_H and chosen to get f_a = 0.5 at 100 K.
				* factor of 0.6252 needed to obtain std ism rate to be 3e-17 at 100 K,
				* the value deduced by
				* >>refer	H2	grain physics	Jura, M., 1974, ApJ, 197, 581 */
				double conversion_efficiency_HM79 = 1/(1. + 1e4*sexp(920./gv.bin[nd]->tedust));

				/* NB IntArea is total, not projected, area, must div by 4 */
				/* units s-1 */
				gv.bin[nd]->rate_h2_form_grains_HM79 = 0.5 * DoppVel.AveVel[ipHYDROGEN]* gv.bin[nd]->IntArea/4. * 
					gv.bin[nd]->cnv_H_pCM3 * sticking_probability_H * conversion_efficiency_HM79 ;
				ASSERT( gv.bin[nd]->rate_h2_form_grains_HM79 > 0. );
			}

			{
				/* H2 formation on grains from 
				 * >>refer	H2	form	Cazaux, S., & Tielens, A.G.G.M., 2002, ApJ, 575, L29 */
				/* number of monolayers per second - only affects efficienty at very low or high temperatures */
				double f = 1e-10;
				/* equation 17 
				double sqrt_term = POW2( 1. + sqrt( (10000.-200.)/(600.-200.) ) );*/
				double sqrt_term = 35.399494936611667;
				double beta_alpha = 0.25 * sqrt_term *sexp(200./gv.bin[nd]->tedust );
				/* equation 16 */
				double xi =  1./ (1. + 1.3e13*sexp(1.5*1e4/ gv.bin[nd]->tedust )*sqrt_term/(2.*f) ) ;
				/* expression for beta comes from just after equation 5 */
				double beta = 3e12 * sexp( 320. / gv.bin[nd]->tedust );
				/* recombination efficiency given by their equation 15, they call
				 * this epsilon_H2 */
				double recombination_efficiency_CT02 = xi / (1. + 0.005*f/2./MAX2(SMALLFLOAT,beta) + beta_alpha );

				/* gv.bin[nd]->IntArea integrated grain surface area Int(4pi*a^2), normalized per H, in cm^2/H,
				 * so x/4 is projected area of circle */
				/* gv.bin[nd]->cnv_H_pCM3 is H density [cm-3] times grain depletion factor */
				/* units s-1 */
				gv.bin[nd]->rate_h2_form_grains_CT02 = 0.5 * DoppVel.AveVel[ipHYDROGEN]* gv.bin[nd]->IntArea/4. * 
					gv.bin[nd]->cnv_H_pCM3 * sticking_probability_H * recombination_efficiency_CT02 ;
				ASSERT( gv.bin[nd]->rate_h2_form_grains_CT02 > 0. );
			}

			{
				/* ortho to para on grain surfaces, taken from 
				 * Le Bourlot, J., 2000, A&A, 360, 656-662 */
				/* For all grain temperatures, this process corresponds to high J going to
				 * either 0 or 1 preserving nuclear spin.  All ortho go to 1 and para go to 0.
				 * When the dust temperature is below Tcrit all 1 go to 0 and so all J go to 0.
				 */
				/* this temperature depends on grain composition, discussion left column of page 657,
				 * this is for a bare grain */
				/*TODO - put in actual composition dependent Tad - this is only valid 
				 * for bare surfaces - not ice - for ice Tad is 555K */
				double Tad = 800.;
				/* tau_nu the first equation in section 2.5
				double tnu = sqrt( BOLTZMANN / PROTON_MASS ) / PI / 2.6e-8; */
				/* equation one paragraph before equation 2 */
				/* ortho2para_final is ratio of ortho (J=1) to para (J=0) in final product,
				 * at low grain temperatures all end in para, J=1 */
				/* this is roughly 20 K */
				double T_ortho_para_crit = 2. * Tad / log( POW2(60. *1.1e11)*Tad); 

				/* AveVEl[LIMELM+2] is average speed of H2 molecules */
				/* for now assume that sticking probability for H2 on the grain is equal to
				 * that for H */
				if( gv.bin[nd]->tedust < T_ortho_para_crit )
				{
					/* rate (s-1) all v,J levels go to 0, regardless of nuclear spin */
					hmi.rate_h2_allX_2_J0_grains = DoppVel.AveVel[LIMELM+2]* gv.bin[nd]->IntArea/4. * 
						gv.bin[nd]->cnv_H_pCM3 * bold[ipMH2]/dense.gas_phase[ipHYDROGEN] * sticking_probability_H  ;

					/* rate (s-1) all v,J levels go to 0 or 1, preserving nuclear spin */
					hmi.rate_h2_ortho_para_conserve = 0. ;
				}
				else
				{
					/* rate (s-1) all v,J levels go to 0, regardless of nuclear spin */
					hmi.rate_h2_allX_2_J0_grains = 0.;

					/* rate (s-1) all v,J levels go to 0 or 1, preserving nuclear spin */
					hmi.rate_h2_ortho_para_conserve = DoppVel.AveVel[LIMELM+2]* gv.bin[nd]->IntArea/4. * 
						gv.bin[nd]->cnv_H_pCM3 * bold[ipMH2]/dense.gas_phase[ipHYDROGEN] * sticking_probability_H  ;
				}
			}
		}
		/*fprintf(ioQQQ," H2 grain form rate HM79 %.2e  %.2e CT02 %.2e  %.2e O-P grn %.2e %.2e\n", 
			gv.bin[nd]->rate_h2_form_grains_HM79[0] , 
			gv.bin[nd]->rate_h2_form_grains_HM79[1] ,
			gv.bin[nd]->rate_h2_form_grains_CT02[0] , 
			gv.bin[nd]->rate_h2_form_grains_CT02[1] , 
			hmi.rate_h2_allX_2_J0_grains,
			hmi.rate_h2_allX_2_J1_grains
			);*/
	}
	else
	{
		for( nd=0; nd < gv.nBin; nd++ )
		{
			gv.bin[nd]->rate_h2_form_grains_CT02 = 0.;
			gv.bin[nd]->rate_h2_form_grains_HM79 = 0.;
		}
		/* rate all H2 goes to either 0 or 1 depending on ortho/para */
		hmi.rate_h2_ortho_para_conserve = 0. ;
		/* at low temp, rate all H2 goes to J=0 */
		hmi.rate_h2_allX_2_J0_grains = 0.;
		ionrec.GrnIonRec = 0.;
	}

	/* this is the one that is actually used in calculations */
	gv.rate_h2_form_grains_used_total = 0.;
	for( nd=0; nd < gv.nBin; nd++ )
	{
		/* activate first one to use Hollenbach & McKee 1979 */
		/*gv.bin[nd]->rate_h2_form_grains_used = gv.bin[nd]->rate_h2_form_grains_HM79;*/
		/* use the new rate by Cazaux & Tielens */
		/* units are s-1*/
		gv.bin[nd]->rate_h2_form_grains_used = gv.bin[nd]->rate_h2_form_grains_CT02;
		gv.rate_h2_form_grains_used_total += gv.bin[nd]->rate_h2_form_grains_used;
	}
	/* print rate coefficient */
	/*fprintf(ioQQQ," total grain h2 form rate %.3e\n",gv.rate_h2_form_grains_used_total);*/

	/* collisional dissociation, rate from 
	 * >>refer	H2	collisional dissociation	Dove, J.E., and Mandy, M. E., 1986, ApJ, 311, L93.
	 * corr is correction for approach to high density limit
	 * H2 + H => 3H - rate very uncertain */
	corr = MIN2(6.,14.44-phycon.alogte*3.08);

	/* corr = pow(10.,corr/(1. + 1.6e4/dense.xIonDense[ipHYDROGEN][0])); */

	if (corr > 0.)
		corr = pow(10.,corr*bold[ipMH]/(bold[ipMH]+1.6e4));
	else
		corr = 1.;
	hmi.rh2dis = (float)(1.55e-8/phycon.sqrte*sexp(65107./phycon.te)* corr);

	/* old hminus rate Hollenbach & McKee 1979
	 *>>chng 98 jan 02, from 2.12e4 to 2.123e4 */
	/*hmi.bh2h2p = 1.8e-12f*phycon.sqrte*phycon.te10/phycon.te01*2.f/16.f;
	rh2h2p = 1.8e-12*phycon.sqrte*phycon.te10/phycon.te01*sexp(2.123e4/
	  phycon.te);*/

	/*>>chng 02 oct 25, update rate from above (Hollenbach & McKee 1979) to
	 * >>refer	H2	form	Karpas, Z., Anicich, V., & Huntress, W.T. 1979, J Chem Phys, 70, 2877 */
	/* H2+ + H => H2 + H+ */
	hmi.bh2h2p = 6.4e-10f;

	/* H2  +  H+  =>  H2+  +  H
	 * >>chng 02 oct 24, get back reaction from above */
	rh2h2p = hmi.bh2h2p*sexp(2.123e4/phycon.te) * 16.f / 2.f ;

	/* H2+  +  HNU  =>  H+  + H */
	gamtwo = GammaK(opac.ih2pnt[0],opac.ih2pnt[1],opac.ih2pof,1.);
	/*GammaPrt(opac.ih2pnt[0],opac.ih2pnt[1],opac.ih2pof,ioQQQ,gamtwo,0.01);*/

	h2phet = heat.HeatNet;

	/* >> chng 02 nov 15 rjrw: ionization fractions to multiply c[ipHo][*] terms
	 * as b[ipMHo] contains _both_ H0 and H+ */
	/* hatom = ((double)dense.xIonDense[ipHYDROGEN][0])+((double)dense.xIonDense[ipHYDROGEN][1]); */

	hatom = bold[ipMH]+bold[ipMHp];

	strcpy( chLab[ipMH], "HO  " );
	strcpy( chLab[ipMHp], "H/H+" );
	strcpy( chLab[ipMHm], "H-  " );
	strcpy( chLab[ipMH2], "H2  " );
	strcpy( chLab[ipMH2p], "H2+ " );
	strcpy( chLab[ipMH3p], "H3+ " );
	strcpy( chLab[ipMH2s], "H2* " );	
	strcpy( chLab[ipMHeHp], "HeH+" );	

	for (mol=0;mol<N_H_MOLEC;mol++)
		nprot[mol] = hmi.protons[mol];

	/* rtot is total absolute value of reaction rates, for
		 checking rate balance */

	rtot = 0.;
	rindex = 0;
	r = rlist;
	/* Special case, put null reaction at head of list */
	if (r == NULL) {
		int in[]={-1},out[]={-1};
		r = rlist = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	rindex++;

	/*-------------------------------------------------------------------- */

	/* H- H minus hminus balance equations
	 * (IHMI,IPMHO) == processes making H- from Ho =+sign
	 * radiative attachment: HI + NE => H-; */
	/* Use Newton-Raphson step to improve solution, so bvec[] contains reaction rates
	 * and c[][] components of the Jacobian of the rates */

	/* This block adds a reaction H => H- to the stack if it wasn't
	 * there already.
	 *
	 * >>>> ONLY CHANGE the elements of the in[] and out[] vectors and
	 * the rate constant, keep the rest fixed for all reactions 
	 * */
	if (r->next == NULL) {
		int in[]={ipMH},out[]={ipMHm};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = hminus_rad_attach + HMinus_induc_rec_rate;

	/* >>chng 02 oct 29, add these two chemical processes */
	/* H- + H+ => H2+ + e
	 * equation (H6) from 
	 * >>refer	H2	chemistry	Galli,D., & Palla, F. 1998, A&A, 335, 403-420 
	 * hmihph2p = 6.9e-9f*(Tg)^(-0.35) for Tg<=8000
	 * hmihph2p = 6.9e-9f*(Tg)^(-0.9) for Tg>=8000  */
	/* >>chng 02 nov 07 rjrw, include H+ ion density in rate constant */
	if (phycon.te <= 7891.)
	{
	    /*hmihph2p = 6.9e-9*pow(phycon.te , -0.35);*/
	    hmihph2p = 6.9e-9 / (phycon.te30 * phycon.te05) ;
	}
	else 
	{
		/* >>chng 02 nov 18, had typo for leading coef here */
        /*hmihph2p = 9.6e-7*pow(phycon.te , -0.9);*/
        hmihph2p = 9.6e-7 / phycon.te90;
	}

	if (r->next == NULL) {
		int in[]={ipMHm,ipMHp},out[]={ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = hmihph2p;

	/* >>chng 03 feb 6 */
    /* H2+ + H- => H2 + H
	 * equation (32) from
	 * >>refer  Stancil, P.C, & Lepp, S, & Dalgarno, A. 1998,ApJ, 509, 1-10
	 * h2phmh2h = 1.4e-7f*pow(phycon.te/300.0, -0.5) */
	h2phmh2h = 1.4e-7f*pow(phycon.te/300 , -0.5);
    if (r->next == NULL) {
		int in[]={ipMH2p,ipMHm},out[]={ipMH2,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2phmh2h;


    /* >>chng 03 feb 7 */
    /* H2+ + H- => H + H + H
	 * equation (33) from
	 * >>refer  Stancil, P.C, & Lepp, S, & Dalgarno, A. 1998,ApJ, 509, 1-10
	 * h2phmhhh = 1.4e-7f*pow(phycon.te/300.0, -0.5) */
     h2phmhhh = 1.4e-7f*pow(phycon.te/300 , -0.5);
    if (r->next == NULL) {
		int in[]={ipMH2p,ipMHm},out[]={ipMH,ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2phmhhh;


	/* (IHMI,IHMI) = processes destroying H- =-sign
	 * photodissociation, H- + H NU => H + NE */

	if (r->next == NULL) {
		int in[]={ipMHm},out[]={ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = hmi.HMinus_photo_rate;

	/* mutual neutralization with heavies, rate from Dalgarno and Mcray
	 * all charged ions contribute equally,
	 * H- + A+ => H + A */
	
	faneut = dense.eden-bold[ipMHp]-dense.xIonDense[ipHELIUM][1]-
		2.*dense.xIonDense[ipHELIUM][2];
	if (faneut > 0.)
		faneut *= 4e-6/phycon.sqrte;
	else
		faneut = 0.;

	/* faneut = 4e-6/phycon.sqrte*MAX2(0.,dense.eden-dense.xIonDense[ipHYDROGEN][1]-dense.xIonDense[ipHELIUM][1]-
		 2.*dense.xIonDense[ipHELIUM][2]); */

	if (r->next == NULL) {
		int in[]={ipMHm},out[]={ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = faneut;

	/* electron collisional ionization of H- */
	cionhm *= dense.eden;

	if (r->next == NULL) {
		int in[]={ipMHm},out[]={ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = cionhm;

	/* inverse process; three body rec */
	c3bod = cionhm*(phmlte*dense.eden);

	if (r->next == NULL) {
		int in[]={ipMH},out[]={ipMHm};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = c3bod;

	/* associative detachment:  H- + H => H2 + E */
	/* rate coefficient from 
	 * >>refer	H2	form	Launay, J.R., Le Dourneuf, M., & Zeippen, C.J., 
	 * >>refercon	1991, A&A, 252, 842-852*/
	{
		double y , x;
		x = MAX2(10., phycon.te );
		x = MIN2(1e4, x );
		y=545969508.1323510+x*71239.23653059864;
		hmi.assoc_detach = (float)(1./y);
	}

	/* >>chng 02 oct 17, temp dependent fit to rate, updated reference,
	 * about 40% larger than before */
	ratach = bold[ipMH]*((double)hmi.assoc_detach);
	/* ratach = dense.xIonDense[ipHYDROGEN][0]*((double)hmi.assoc_detach); */
	/*ratach = dense.xIonDense[ipHYDROGEN][0]*1.35e-9;*/
	/* make H2 from H- =+ sign
	 * associative detachment; H- + H => H2: 
	 * >>refer	H2	rates	Browne & Dalgarno J PHys B 2, 885 */
	/* >>chng 02 nov 7 rjrw, example case of 2-body process */

	if (r->next == NULL) {
		int in[]={ipMH, ipMHm},out[]={ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = hmi.assoc_detach;

	/* (1,IPMH2) convert H2 into H- = +sign
	 * the back reaction, H2 + e => H- + Ho */
	if( hmi.ph2lte > 0. )
	{ 
		batach = ((double)hmi.assoc_detach)*phmlte/hmi.ph2lte*dense.eden;
		/* batach = (ratach/dense.xIonDense[ipHYDROGEN][0])*phmlte/hmi.ph2lte*dense.eden; */
	}
	else
	{
		batach = 0.;
	}
	{
		/*@-redef@*/
		/* often the H- route is the most efficient formation mechanism for H2,
		 * will be through rate called ratach
		 * this debug print statement is to trace h2 oscillations */
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nzone>187&& iteration > 1/**/)
		{
			/* rapid increase in H2 density caused by rapid increase in hmi.ph2lte */
			fprintf(ioQQQ,"batach\t%.2e\t%.1e\t%.1e\t%.1e\t%.1e\t%.1e\t%.1e\n", 
				batach, 
				ratach,
				bold[ipMH],			/* dense.xIonDense[ipHYDROGEN][0], */
				phmlte,
				hmi.ph2lte,
				dense.eden,
				bold[ipMHm]
				);
		}
	}

	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH,ipMHm};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = batach;

#define HNEUT 7e-8
	/* mutual neut, mostly into n=3; rates from Janev et al
	 * H- + H+ => H + H(n=3) */
	fixit(); /* process is net source term for H(n=3) states */
	fhneut = bold[ipMHp]*HNEUT; /* dense.xIonDense[ipHYDROGEN][1]*7e-8; */

	if (r->next == NULL) {
		int in[]={ipMHm,ipMHp},out[]={ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = HNEUT;

	/* back reaction from excited state H */
	if( phycon.te > 1000. )
	{
		/* HBN(3,1) is defined; when <HydTempLimit then set to 1 */
		bhneut = (HNEUT*phmlte*dense.eden)*iso.DepartCoef[ipH_LIKE][ipHYDROGEN][3];
	}
	else
	{
		bhneut = 0.;
	}

	/* mutual neut, mostly into n=3; rates from Janev et al
	 * H + H(n=3) => H- + H+ */
	fixit(); /* process is net ionization term for H(n=3) states */
	/* this is the back reaction, forming H- from Ho */

	if (r->next == NULL) {
		int in[]={ipMH,ipMH},out[]={ipMHm,ipMHp}, ratesp[]={ipMH,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = bhneut; 
	bhneut *= bold[ipMHp];


	/* the processes H2(v>=4) + e- => H + H-,
	 * >>refer	H2	rates	Lenzuni et al. apj sup 76, 759
	 * >>refer	H2	rates	Janev et al.
	 * density dep is for non-lte, guess from dalgarno and roberge apl 233, 25
	 * extra expo factor added for low temps */
	if( nzone <= 1 )
	{
		/* this is initial setup of code, so set rate coef to actual val */
		eh2hhm = 2.7e-8*pow(10.,-0.7*POW2(phycon.alogte - 3.615))*
		  dense.eden*(dense.gas_phase[ipHYDROGEN]/(1e7 + dense.gas_phase[ipHYDROGEN]))*sexp(52000./
		  phycon.te);
		eh2old = eh2hhm;
	}
	else
	{
		/* this is deeper into the cloud, and there is danger of oscillation */
		eh2old = eh2hhm;
		eh2hhm = 2.7e-8*pow((double)10,-0.7*POW2(phycon.alogte - 3.615))*
		  dense.eden*(dense.gas_phase[ipHYDROGEN]/(1e7 + dense.gas_phase[ipHYDROGEN]))*sexp(52000./
		  phycon.te);
		fac = 0.5;
		eh2hhm = eh2hhm*fac + eh2old*(1. - fac);
	}

	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH,ipMHm};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = eh2hhm;

	/*--------------------------------------------------------------------
	 *
	 * molecular hydrogen H2 htwo balance equation
	 * (IPMH2,IPMHO)==create H2 from Ho =+ */

	/* H2 formation on grains */
	/* >>chng 01 jan 05, remove from matrix part and add hden to hmi.rate_h2_form_grains_used, */
	
	/* The reaction rate is only proportional to one of the ipMH, due to
	   surface saturation (?) */
	if (r->next == NULL) {
		int in[]={ipMH,ipMH},out[]={ipMH2},ratesp[]={ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = gv.rate_h2_form_grains_used_total;

	/* excited atom radiative association,
	 * H(n=2) + H(n=1) => H2 + hnu
	 * >>refer	H2	rates	Latter, W.B., & Black, J.H., 1991, Ap.J. 372, 161 */
	/* radasc = ((iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p] + iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2s])*dense.xIonDense[ipHYDROGEN][1])*3e-14; */
	
	radasc = ((iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p] + iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2s]))*3e-14;
	/* >>chng 02 nov 7 rjrw: correct for n^2 behaviour w.r.t. H 
	   >>chng 02 nov 7 rjrw, correct stoichiometry */
	fixit(); /* assume rate was for reaction... */

	/* Possible that changing to a rate proportional to ipMHp would be more consistent */
	if (r->next == NULL) {
		int in[]={ipMH,ipMH},out[]={ipMH2},ratesp[]={ipMH,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = radasc;  
	radasc *= bold[ipMHp];

	/* photo-destroy H2 */
	/* >>chng 00 nov 25 factor of 0.1, assume pump is total, and 10% distroy H2 */
	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	/* >>chng 03 mar 07, had factor of 0.1 for branching ratio to H+H, 
	 * but should not have since branching already included */
	/*r->ratek = hmi.H2_Solomon_rate_used*0.1;*/
	r->ratek = hmi.H2_Solomon_rate_used;

	/* the process H2 + e- => H + H + e
	 * >>refer	H2	rates	Lenzuni et al. apj sup 76, 759, quoted from Janev et al. 
	 * >>chng 02 nov 7 rjrw, correct stoichiometry */
	eh2hh = 1.3e-18*phycon.te*phycon.te*sexp(52000./phycon.te)*dense.eden;
	
	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = eh2hh;

	/* H2 + H+ => H3+ HNU
	 * equation H21 from 
	 * >>refer	H2	chemistry	Galli,D., & Palla, F. 1998, A&A, 335, 403-420 */
	/* >>chng 02 nov 07 rjrw, include H+ ion density in rate constant */
	h2hph3p = 1.0e-16f;

	if (r->next == NULL) {
		int in[]={ipMH2,ipMHp},out[]={ipMH3p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2hph3p;

	/* collisional dissociation, rate from 
	 * >>refer	H2	collisional dissociation	Dove, J.E., and Mandy, M. E., 1986, ApJ, 311, L93.
	 * H_2 + H => 2H + H 	 
	 * >>chng 02 nov 7 rjrw, correct stoichiometry */

	/* Rate is catalyzed by an additional H */
	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH,ipMH},ratesp[]={ipMH,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = hmi.rh2dis;
  
#	if 0
	/* >>chng 03 mar 19 */
	/* 2H + H2 => H2 + H2
	 * equation (5) from 
	 * >>refer	H2	chemistry Palla, F., Salpeter, E.E., & Stahler, S.W., 1983, ApJ,271, 632-641
	 * bh2h22hh2= 5.5e-29/(8*phycon.te) */
	bh2h22hh2 = 5.5e-29/(8.*phycon.te);

   if (r->next == NULL) {
	   int in[]={ipMH,ipMH,ipMH2},out[]={ipMH2,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = bh2h22hh2;

    /* >>chng 03 mar 19 */
	/* H2 + H2 => 2H + H2
	 * equation (5) from 
	 * >>refer	H2	chemistry Palla, F., Salpeter, E.E., & Stahler, S.W., 1983, ApJ,271, 632-641
	 * h2h22hh2 = bh2h22hh2/hmi.ph2lte */
	if( hmi.ph2lte > 0. )
	{
		h2h22hh2 = bh2h22hh2/hmi.ph2lte;
	}
	else
	{
		h2h22hh2 =0.;
	}

	if (r->next == NULL) {
		int in[]={ipMH2,ipMH2},out[]={ipMH,ipMH,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2h22hh2;
#	endif

	/* back rate, three body recombination, 2H + S => H_2 + S 	 
	 * >>chng 02 nov 7 rjrw: correct for n^2 behaviour w.r.t. H 
	 * >>chng 02 nov 7 rjrw, correct stoichiometry 
	 * >>chng 02 nov 7 rjrw, correct for n^3 behaviour w.r.t. H !! */
	/* bh2dis = hmi.rh2dis*hmi.ph2lte*dense.xIonDense[ipHYDROGEN][0]*dense.xIonDense[ipHYDROGEN][0]; */

	bh2dis = hmi.rh2dis*hmi.ph2lte*bold[ipMH]*bold[ipMH];	
	fixit(); /* assume rate was for reaction... */

	if (r->next == NULL) {
		int in[]={ipMH,ipMH},out[]={ipMH2},ratesp[]={ipMH,ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = hmi.rh2dis*hmi.ph2lte;

    

	/* H2 + HNU=>  H2+ + E
	 * photoionization by hard photons, crossection=3*HI */
	/* assume cosmic rays do the same thing */
	/* >>chng 00 nov 28, factor of 0.93 from
	 >>refer	cosmic ray	ionization rate	Tielens, A.G.G.M., & Hollenbach, D., 1985, ApJ, 291, 722
	 */
	/* make H2+ from H2 =+sign
	 * H2 + HNU => H2+ + E
	 * also cosmic rays */
	/* >>chng 00 nov 28, factor of 0.93 from
	 >>refer	cosmic ray	ionization rate	Maloney, P.R., Hollenbach, D., & Tielens, A. G. G. M., 1998, ApJ, 466, 561
	 */

	gamhd = iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s];

	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = gamhd + Secondaries.csupra*0.93;

	/* >> chng 02 nov 15 rjrw: multiply c[ipMHo][*] terms by ionization fraction
	 * as b[ipMHo] contains _both_ H0 and H+ */
	/* H2  +  H+  =>  H2+  +  H */
	if (r->next == NULL) {
		int in[]={ipMH2,ipMHp},out[]={ipMH,ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = rh2h2p;

	/* (3,IPMH2P) == destroy H2+ = -sign
	 * H + H2+ => H+ + H2 */
	/* >>chng 02 nov 7 rjrw, remove the destruction rate 
	 *	 c[ipMHo][ipMHo] += -hmi.bh2h2p*hmi.Molec[ipMH2p]; 
	 * twice -- reaction changes state of H within single [H0,H+] `species' */

	if (r->next == NULL) {
		int in[]={ipMH,ipMH2p},out[]={ipMHp,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = (double) hmi.bh2h2p;

	/*TODO this rate drives numerical instability in such models as secondary1 and 2.in */
	/* this rate couples H2+ and H3+, and tends to destabalize the matrix in both highly
	 * ionized and fully molecular conditions.  Setting this to zero had no effect - the th85
	 * predictions were identical.  
	 *
	 */
	/* H + H3+ => H2 + H2+ */

	h3ph2p = hmrate(2.08e-9,0.,1.88e4);

	if (r->next == NULL) {
		int in[]={ipMH,ipMH3p},out[]={ipMH2,ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h3ph2p;


	/* >>chng 03 feb 7 */
    /* H3+ + H- => H2 + H + H
	 * equation (50) from
	 * >>refer  Stancil, P.C, & Lepp, S, & Dalgarno, A. 1998,ApJ, 509, 1-10
	 * h3phmh2hh = 2.3e-7f*pow(phycon.te/300.0, -0.5) */
     h3phmh2hh = 2.3e-7f*pow(phycon.te/300 , -0.5);
    if (r->next == NULL) {
		int in[]={ipMH3p,ipMHm},out[]={ipMH2,ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h3phmh2hh;

	/* H2 + H3+ => H2 + H2+ + H */
	h3petc = hmrate(3.41e-11,0.5,7.16e4);

	if (r->next == NULL) {
		int in[]={ipMH2,ipMH3p},out[]={ipMH2,ipMH2p,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h3petc;

	/* H2 + H3+ => H2 + H+ + H2 */
	h32h2 = hmrate(3.41e-11,0.5,5.04e4);

	if (r->next == NULL) {
		int in[]={ipMH2,ipMH3p},out[]={ipMHp,ipMH2,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h32h2;

	/* e + H3+ => H2 + H */
	fixit(); /* e + H3+ => 3H was supposed to be included in this rate, 
						* and stoichiometric factor 2* on sink rate seemed wrong */
	eh3_h2h = hmrate(5.00e-9,-0.5,0.)*dense.eden;
	/* >>chng 03 feb 10, increase rate by factor of 13.6 to agree with
	 * >>refer	H3+	DR	McCall, B.J., et al. 2003, Nature, in press (astro-ph 0302106)*/
	/* >>chng 03 feb 13, extra 0.2 since 20% of these go to H2 + H, Stancil private comm */
	eh3_h2h *= 13.6 * 0.2;

	if (r->next == NULL) {
		int in[]={ipMH3p},out[]={ipMH,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = eh3_h2h;

    /* e + H3+ => 3H */
	eh3p_3h = hmrate(5.00e-9,-0.5,0.)*dense.eden;
	/* >>chng 03 feb 10, increase rate by factor of 13.6 to agree with
	 * >>refer	H3+	DR	McCall, B.J., et al. 2003, Nature, in press (astro-ph 0302106)*/
	/* >>chng 03 feb 13, extra 0.8 since 80% of these go to 3H, Stancil private comm */
	eh3p_3h *= 13.6 * 0.8;

	if (r->next == NULL) {
		int in[]={ipMH3p},out[]={ipMH,ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = eh3p_3h;


	if( (trace.lgTrace && trace.lgTrMole) || punch.lgPunH2 )
	{
		if( punch.lgPunH2 )
		{
			ioFile = punch.ipPunH2;
		}
		else
		{
			ioFile = ioQQQ;
		}

		if( c[ipMH2][ipMH2] != 0. )
		{
			rate = -c[ipMH2][ipMH2];
			fprintf( ioFile, 
			  " Destroy H2: rate=%.2e DIS;%.3f bat;%.3f h2dis;%.3f gamhd;%.3f h2h2p;%.3f E-h;%.3f E-h-;%.2f h2hph3p;%.3f sec;%.3f\n", 
			  rate, 
			  hmi.H2_Solomon_rate_used / rate, 
			  batach / rate, 
			  hmi.rh2dis*dense.xIonDense[ipHYDROGEN][0] / rate, 
			  gamhd / rate, 
			  rh2h2p*dense.xIonDense[ipHYDROGEN][1] / rate, 
			  eh2hh /rate, 
			  eh2hhm / rate ,
			  h2hph3p / rate ,
			  Secondaries.csupra*0.93 / rate
			  );
		}
		else
		{
			fprintf( ioFile, " Destroy H2: rate=0\n" );
		}
	}

	/*------------------------------------------------------------------- */

	/* h2plus H2+ balance equations */

	/*TODO must add process H2+ + H- => H2 + H, Dalgarno&Lepp 87 */
	/* >>refer	H2+	chemistry	Dalgarno, A., & Lepp, S., 1987, in Astrochemistry, eds. 
	 * >>refercon	M.S. Vardya & S.P. Tarafar, Reidel, Dordrecht, p 109 */
	/* rate = 5e-7 * sqrt(100. / phycon.te); */

	/*TODO put in H2+ + gamma => H + H+ */
	/* >>refer	H2+	chemistry	Stancil, P.C., 1994, ApJ, 430, 360 */
	/* cross section is log10( cs_25) = -1.6547717e6 + 1.8660333e5 ln(nu) - 7.8986431e3*ln(nu)^2
	 * 148.73693 * ln(nu)^3 - 1.0513032*ln(nu)^4 */

	/* make H2+ from Ho
	 * H+  + H  =>  H2+ + HNU
	 * approximation was from Kurucz thesis, not meant for hot gas 
	 * >>chng 02 nov 7 rjrw, stoichiometric factor */
	radath = MAX2(0.,2.325*MIN2(5000.,phycon.te)-1375.)*1e-20;

	if (r->next == NULL) {
		int in[]={ipMH,ipMHp},out[]={ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	/* Printf("O: %ld %g %g\n",rindex,radath*dense.xIonDense[ipHYDROGEN][1]*dense.xIonDense[ipHYDROGEN][0],
		 radath*dense.xIonDense[ipHYDROGEN][1]); */
	rindex++;
	r->ratek = radath;

	radath *= bold[ipMHp]; /* dense.xIonDense[ipHYDROGEN][1]; */

	/* H2+  +  p  => H + H+ + H+; Janev et al. 3.2.6 */
	/* >>chng 02 nov 7 rjrw, stoichiometric factor */
	h2pion = 2.4e-27*POW3(phycon.te);

	if (r->next == NULL) {
		int in[]={ipMH2p},out[]={ipMH,ipMHp},ratesp[]={ipMHp,ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = h2pion;

	/* H2+  +  E  => H + H+ + e-; Janev et al. */
	/* >>chng 02 nov 7 rjrw, stoichiometric factor */
	h2pcin = 2e-7*sexp(30720./phycon.te);

	if (r->next == NULL) {
		int in[]={ipMH2p},out[]={ipMH,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2pcin*dense.eden;

	/* back reaction, H + H+ + e => h2+ + e */
	b2pcin = h2pcin*phplte;
	/* this is the hot reaction at high densities */

	if (r->next == NULL) {
		int in[]={ipMH,ipMHp},out[]={ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = b2pcin*dense.eden;

	/* H2+  +  HNU  =>  H+  + H */

	if (r->next == NULL) {
		int in[]={ipMH2p},out[]={ipMH,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = gamtwo;

	/* photoionization by hard photons, crossection =2*HI (wild guess) */

	if (r->next == NULL) {
		int in[]={ipMH2p},out[]={ipMH,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 2.*iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s];

	/* H2 + H2+ => H + H3+ */
	h2ph3p = 1.40e-9*(1. - sexp(9940./phycon.te));

	if (r->next == NULL) {
		int in[]={ipMH2,ipMH2p},out[]={ipMH,ipMH3p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2ph3p;

	/* destroy H2+ via H2+ + H2 => H + H+ + H2 */
	h2phhp = 2.41e-12*phycon.sqrte*sexp(30720./phycon.te);

	if (r->next == NULL) {
		int in[]={ipMH2,ipMH2p},out[]={ipMH,ipMHp,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h2phhp;

	/*------------------------------------------------------------------ */

	/* H3+ balance equations*/

	/* CO + H3+ => H + H2 + CO+ */
	/* >>chng 02 nov 03, add this, which is important for CO dest */
	/* this also appears in hmole.c */
	h3pcop = 1.70e-9*hevmolec.hevmol[ipCO];

	if (r->next == NULL) {
		int in[]={ipMH3p},out[]={ipMH,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = h3pcop;

	/* photoionization by hard photons, crossection =2*HI (wild guess)
     -- rjrw: where do they go??? 
		 -- H3+ + hv => H2+ + H+ + e, best guess (P. Stancil, priv comm) */

	/* c[ipMH3p][ipMH3p] -= 0; fixit();*/ /* 2.*iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s]; */

	if (r->next == NULL) {
		int in[]={ipMH3p},out[]={ipMH2p,ipMHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 2.*iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s];

	/*------------------------------------------------------------------ */

	/* vib excited H2, called H2* balance equations, these closely follow
	 * >>refer	mh2	fits	Tielens, A.G.G.M., & Hollenbach, D., 1985a, ApJ 291, 722 */
	/* population of vib-excited H2, from discussion on pp 736-737 of TH85 */

	/* deexcitation rate from upper level, H2* => H2 */
	H2star_deexcit = bold[ipMH2]*1.4e-12*phycon.sqrte * sexp( 18100./(phycon.te + 1200.) ) +
		 bold[ipMH] * 1e-12*phycon.sqrte * sexp(1000./phycon.te );
	deexc_htwo = 1.4e-12*phycon.sqrte * sexp( 18100./(phycon.te + 1200.) );
	deexc_hneut =  1e-12*phycon.sqrte * sexp(1000./phycon.te );

	/* depopulate H2_star, 2e-7 is spontaneous deexcitation rate,
	 * which also appears in lines where intensity of vib lines is entered into line stack */

	if (r->next == NULL) {
		int in[]={ipMH2s},out[]={ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 2e-7;

	if (r->next == NULL) {
		int in[]={ipMH2s},out[]={ipMH2},ratesp[]={ipMH2s,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = deexc_htwo;

	if (r->next == NULL) {
		int in[]={ipMH2s},out[]={ipMH2},ratesp[]={ipMH2s,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = deexc_hneut;

	/* collisional excitation of vib from ground, 
	 * stat weight of ground 1, excit 6, as per TH discussion
	 * this must normally be zero */
	/* H2 producing H2_star */
	excit = 6. * sexp( 30172./phycon.te);
	/* total excitation rate to H2*, s-1, NB - this is also used in the cooling - heating
	 * rate below */
	H2star_excit = excit * H2star_deexcit;

	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH2s},ratesp[]={ipMH2,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = deexc_htwo*excit;

	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH2s},ratesp[]={ipMH,ipMH2};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),ratesp,INTSZ(ratesp));
	}
	r = r->next;
	rindex++;
	r->ratek = deexc_hneut*excit;

	/* assume that 0.9 of H2 dissociations lead to H2_star,
	 * H2 + 0.9*hmi.H2_Solomon_rate_used => h2_star */
	if (r->next == NULL) {
		int in[]={ipMH2},out[]={ipMH2s};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 0.9*hmi.H2_Solomon_rate_used;

	/* rate of photodissoc of vib-excit H2, A12 of TH85 */
	gh2exc_dissoc = hmi.GammaHabing*1e-11;

	if (r->next == NULL) {
		int in[]={ipMH2s},out[]={ipMH,ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = gh2exc_dissoc;

	/*---------------------------------------------------------------- */

	/* He H+ formation rates taken from Flower+Roueff, Black */

	/* He+ + H => HeH+
	 * radiative association from 
	 * >>refer	mheh+	rate	Zygelman, B., and Dalgarno, A. 1990, ApJ 365, 239 */

	if (r->next == NULL) {
		int in[]={ipMH},out[]={ipMHeHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 1e-15*dense.xIonDense[ipHELIUM][1];

	/* He + H+ => HeH+ */	
	if (r->next == NULL) {
		int in[]={ipMHp},out[]={ipMHeHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 1e-20*dense.xIonDense[ipHELIUM][0];

	/* H2+ + HE => HEH+ + H0 */
	if (r->next == NULL) {
		int in[]={ipMH2p},out[]={ipMH,ipMHeHp};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 3e-10*exp(-6717./phycon.te)*dense.xIonDense[ipHELIUM][0];

	/* photodissociation through 1.6->2.3 continuum */

	/* why is this in a look instead of GammaK?
	 * to fix must set opacities into stack */
	gamheh = 0.;
	limit = MIN2(hmi.iheh2-1 , rfield.nflux );
	for( i=hmi.iheh1-1; i < limit; i++ )
	{
		gamheh += rfield.flux[i] + rfield.ConInterOut[i]+ rfield.outlin[i];
	}
	gamheh *= 4e-18;

	/* hard radiation */
	gamheh += 3.*iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s];

	/* recombination, HeH+  +  e => He + H */
	gamheh += dense.eden*1e-9;

	if (r->next == NULL) {
		int in[]={ipMHeHp},out[]={ipMH};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = gamheh;

	/* HeH+  +  H => H2+  + He */
	if (r->next == NULL) {
		int in[]={ipMH,ipMHeHp},out[]={ipMH2p};
		r->next = newreaction(rindex,in,INTSZ(in),out,INTSZ(out),NULL,0);
	}
	r = r->next;
	rindex++;
	r->ratek = 1e-10;

	/******* END OF REACTIONS ********/

	/* Generate chemical error vector and Jacobian array from reaction list */
	r = rlist;
	rtot = 0.;
	for( i=0; i < N_H_MOLEC; i++ )
	{
		bvec[i] = 0.;
		for( j=0; j < N_H_MOLEC; j++ )
		{
			c[j][i] = 0.;
		}
	}
	/* Subtotal rates for H_0 and H^+ within ipMHo */
	for (i=0;i<2;i++) 
	{
		ionmole.source[i] = ionmole.sink[i] = 0.;
	}


	while (r->next != NULL)
	{
		r = r->next;
		if (r->index == rindex)
			break;
		ratek = r->ratek;
		/* There's an O(n) algorithm for this -- but it doesn't improve
       things unless nreactants is >= 4...!*/
		for (i=0;i<r->nrates;i++)
		{
			rates[i] = ratek;
			for (j=0;j<r->nrates;j++)
			{
				if (i!=j)
					rates[i] *= bold[r->rates[j]];
			}
		}

		rate = rates[0]*bold[r->rates[0]];
		rtot += rate;

		/* Get sink terms (i.e. rate/abundance) to include in ionization ladders */
		for (i=0;i<r->nreactants;i++)
		{
			int ok = 0;
			for (j=0;j<r->nrates && !ok;j++)
			{
				if (r->rates[j] == r->reactants[i]) 
				{
					sinkrate[i] = rates[j];
					ok = 1;
				}
			}
			if (!ok) 
			{
				/* Odd, the rate didn't depend on one of the species it used
				 * at all!  An alternative way of getting sinkrate is
				 *
				 * sinkrate[i] = rate/bold[r->reactants[i]]; 
				 *
				 * but this uses the possibly zero bold, and is prone to underflow of rate.
				 * */
				fprintf(ioQQQ,"A chemical rate in hmole was independent of the species it used\n");
				fprintf(ioQQQ,"This probably shouldn't happen (so you shouldn't see this message).\n");
				puts( "[Stop in hmole]" );
				cdEXIT(EXIT_FAILURE);
			}
		}


		/* if (nzone == 416)
			 fprintf(ioQQQ,"Adding reaction %d rate %g\n",r->index,rate); */

		{
			for (i=0;i<r->nreactants;i++)
			{
				ratei = r->reactants[i];
				bvec[ratei] -= rate;
				/*if ((nzone == 421 || nzone == 422) && ratei == ipMHm)
					 fprintf(ioQQQ,"snk %s %d %g\n",chLab[ratei],r->index,rate);*/
				if (ratei == ipMH || ratei == ipMHp)
					ionmole.sink[ratei] += sinkrate[i];
			}
			for (i=0;i<r->nproducts;i++)
			{
				ratei = r->products[i];
				bvec[ratei] += rate;
				/*if ((nzone == 421 || nzone == 422) && ratei == ipMHm)
					 fprintf(ioQQQ,"src %s %d %g\n",chLab[ratei],r->index,rate); */
				if (ratei == ipMH || ratei == ipMHp)
					ionmole.source[ratei] += rate;
			}			
			for (j=0;j<r->nrates;j++)
			{
				ratej = r->rates[j];
				rated = rates[j];
				for (i=0;i<r->nreactants;i++)
				{
					c[ratej][r->reactants[i]] -= rated;
				}
				for (i=0;i<r->nproducts;i++)
				{
					c[ratej][r->products[i]] += rated;
				}
			}
		}		
	}

	{
		/* following should be set true to print populations */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			if( DEBUG_LOC && (nzone == 421 || nzone == 422) ) 
			{
				printsol = 1;
				fprintf(ioQQQ,"Temperature %g\n",phycon.te);
				fprintf(ioQQQ," Net mol ion rate [%g %g] %g\n",ionmole.source[1],ionmole.sink[1],
								ionmole.source[1]-ionmole.sink[1]*bold[ipMHp]);
			}
		}
	}

	/* save total H2P destruction rate for possible later printout:
	 * NB this must come last */
	desh2p = -c[ipMH2p][ipMH2p]; 

	/* Check that matrix and vector generated in above loops make sense */
	/*if (!defined(NDEBUG))	*/
	/* in std C NDEBUG is a macro set at compile time */
/*#	if !defined(NDEBUG)*/
#	ifndef NDEBUG
	{
	double total, mtotal;
		for (i=0;i<N_H_MOLEC;i++) 
		{
			total = 0.;
			for (j=0;j<N_H_MOLEC;j++) 
			{
				total += c[i][j]*nprot[j];
			}
			if (fabs(total) > 1e-6*fabs(c[i][i]*nprot[i])) 
			{
				fprintf(ioQQQ,"PROBLEM Subtotal1 %.2e\n",fabs(total)/fabs(c[i][i]*nprot[i]));
				fprintf(ioQQQ,"Species %li Total %g Diag %g\n",i,total,c[i][i]*nprot[i]);
			}
		}
		total = mtotal = 0.;
		for (j=0;j<N_H_MOLEC;j++) 
		{ 
			total += bvec[j]*nprot[j]; 
			mtotal += fabs(bvec[j]*nprot[j]); 
		}
		if (fabs(total) > 1e-30 && fabs(total) > 1e-10*rtot) 
		{ 
			fprintf(ioQQQ,"PROBLEM Subtotal2 %.2e\n",fabs(total)/mtotal);
			fprintf(ioQQQ,"RHS Total %g cf %g\n",total,mtotal);
		} 
		else if (fabs(total) > 1e-7*mtotal)  
		{
			fprintf(ioQQQ,"WARNING zone %li Hmole RHS conservation error %.2e of %.2e\n",nzone,total,mtotal);
			fprintf(ioQQQ,"(may be due to high rate equilibrium reactions)\n");
		}
	}
#		endif


#define MOLMIN  1
#define N_H_MAT (N_H_MOLEC-MOLMIN)
	/* Will collapse ipMH and ipMHp into single species, as don't include
	 * all ionizations and recombinations here */
	/* last test - do not include advection if we have overrun the radius scale 
	 * of previous iteration */
	if( iteration >= 2 && dynamics.lgAdvection && radius.depth < dynamics.oldFullDepth ) 
	{
		ipConserve = -1; /* Don't use conservation form */
		fixit(); /* Check dynamics treatment update to new formalism */

		/* Add rate terms for dynamics to equilibrium, makes c[][] non-singular */
		for (i=0;i<N_H_MOLEC;i++) 
		{
			c[i][i] -= dynamics.Rate;
			bvec[i] -= (bold[i]*dynamics.Rate-dynamics.Molec[i]);
		}

		/* Dynamics implies conservation of advected material */
		protons = 0.;
		for (i=0; i<N_H_MOLEC;i++)
		{
			protons += nprot[i]*dynamics.Molec[i]/dynamics.Rate;
		}
		for (i=0;i<N_H_MOLEC;i++) {
			c[ipMHp][i] = (bold[ipMH]*c[ipMH][i]+bold[ipMHp]*c[ipMHp][i])/hatom;
			c[ipMH][i] = 0.;
		}
		for (i=1;i<N_H_MOLEC;i++) {
			c[i][ipMHp] += c[i][ipMH];
			c[i][ipMH] = 0.;
		}
		bvec[ipMHp] += bvec[ipMH];
		bvec[ipMH] = 0.;
		bold[ipMHp] += bold[ipMH];
		bold[ipMH] = 0.;
	}
	else
	{
		for (i=0;i<N_H_MOLEC;i++) {
			c[ipMHp][i] = (bold[ipMH]*c[ipMH][i]+bold[ipMHp]*c[ipMHp][i])/hatom;
			c[ipMH][i] = 0.;
		}
		bold[ipMHp] += bold[ipMH];
		bvec[ipMH] = bold[ipMH] = 0.;
		ipConserve = ipMHp;
		/* For Newton-Raphson method, want the change in populations to be zero,
		 * so the conserved component must also be zero */
		bvec[ipConserve] = 0.;  
		protons = 0.;
		for (i=MOLMIN;i<N_H_MOLEC;i++) 
		{
			c[i][ipConserve] = nprot[i];
			protons += nprot[i]*bold[i];
		}
	}

	{
		/* following should be set true to print populations */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			/* these are the raw results */
			fprintf( ioQQQ, " HMOLE h2 %.2e h2* %.2e\n" , bold[ipMH2] ,bold[ipMH2s] );
		}
	}
	
	/*------------------------------------------------------------------ */
	if(printsol || (trace.lgTrace && trace.lgTrMole ))
	{
		/* print the full matrix */
		fprintf( ioQQQ, "                ");
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			fprintf( ioQQQ, "      %s", chLab[i] );
		}
		fprintf( ioQQQ, " \n" );

		/*

		[0][0]  [0][1]  [0][2]  [0][3]  [0][4]  [0][5]
		[1][0]  [1][1]  [1][2]  [1][3]  [1][4]  [1][5]
		[2][0]  [2][1]  [2][2]  [2][3]  [2][4]  [2][5]
		[3][0]  [3][1]  [3][2]  [3][3]  [3][4]  [3][5]
		[4][0]  [4][1]  [4][2]  [4][3]  [4][4]  [4][5]
		[5][0]  [5][1]  [5][2]  [5][3]  [5][4]  [5][5]

		[ipMHo][ipMHo]  [ipMHo][ipMHm]  [ipMHo][ipMH2]  [ipMHo][ipMH2p]  [ipMHo][ipMH3p]  [ipMHo][ipMH2s]
		[ipMHm][ipMHo] [ipMHm][ipMHm] [ipMHm][ipMH2] [ipMHm][ipMH2p] [ipMHm][ipMH3p] [ipMHm][ipMH2s]
		[ipMH2][ipMHo]  [ipMH2][ipMHm]  [ipMH2][ipMH2]  [ipMH2][ipMH2p]  [ipMH2][ipMH3p]  [ipMH2][ipMH2s]
		[ipMH2p][ipMHo] [ipMH2p][ipMHm] [ipMH2p][ipMH2] [ipMH2p][ipMH2p] [ipMH2p][ipMH3p] [ipMH2p][ipMH2s]
		[ipMH3p][ipMHo] [ipMH3p][ipMHm] [ipMH3p][ipMH2] [ipMH3p][ipMH2p] [ipMH3p][ipMH3p] [ipMH3p][ipMH2s]
		[ipMH2s][ipMHo] [ipMH2s][ipMHm] [ipMH2s][ipMH2] [ipMH2s][ipMH2p] [ipMH2s][ipMH3p]  [ipMH2s][ipMH2s]

		*/

		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			fprintf( ioQQQ, "       MOLE%2ld %s", i-MOLMIN ,chLab[i] );
			for( j=MOLMIN; j < N_H_MOLEC; j++ )
			{
				fprintf( ioQQQ, "%10.2e", c[j][i] );
			}
			fprintf( ioQQQ, "%10.2e", bvec[i] );
			fprintf( ioQQQ, "\n" );
		}
	}

	/* establish local timescale for H2 molecule destruction */
	if( -c[ipMH2][ipMH2] > SMALLFLOAT )
	{
		/* units are now seconds */
		timesc.time_H2_Dest_here = -1./c[ipMH2][ipMH2];
	}
	else
	{
		timesc.time_H2_Dest_here = 0.;
	}

	/* local timescale for H2 formation 
	 * both grains and radiative attachment */
	timesc.time_H2_Form_here = gv.rate_h2_form_grains_used_total * 
		/* this corrects for fact that we still want to count fast timescale when H2 is
		 * fully formed.  The rate used is only for the local atomic hydrogen density */
		dense.gas_phase[ipHYDROGEN]/dense.xIonDense[ipHYDROGEN][0] + 
		hminus_rad_attach;
	/* timescale is inverse of this rate */
	if( timesc.time_H2_Form_here > SMALLFLOAT )
	{
		/* units are now seconds */
		timesc.time_H2_Form_here = 1./timesc.time_H2_Form_here;
	}
	else
	{
		timesc.time_H2_Form_here = 0.;
	}

#	ifdef MAT
#		undef MAT
#	endif
#	define MAT(a,I_,J_)	(*((a)+(I_)*(N_H_MAT)+(J_)))

	/* copy contents over to 1D array */
	for( j=0; j < N_H_MAT; j++ )
	{
		for( i=0; i < N_H_MAT; i++ )
		{
			MAT(amat,i,j) = c[i+MOLMIN][j+MOLMIN];
		}
	}
	
	if (PRINTSOL)
	{
		double total=0;
		fprintf(ioQQQ,"Zone %ld input:\n",nzone);
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
				fprintf(ioQQQ,"%15.7g\t",bold[i]);
				total += nprot[i]*bold[i];
		}
		fprintf(ioQQQ,"sum = %15.7g\n",total);
	}

	/* now invert the matrix */
	DGETRF(N_H_MAT,N_H_MAT,(double*)amat,N_H_MAT,ipiv, &merror);

	DGETRS('N',N_H_MAT,1,(double*)amat,N_H_MAT,ipiv,bvec+MOLMIN,N_H_MAT,&merror);

	if( merror != 0 )
	{
		fprintf( ioQQQ, " hmole dgetrs error\n" );
		puts( "[Stop in hmole]" );
		cdEXIT(EXIT_FAILURE);
	}

	if (PRINTSOL)
	{
		double total=0;
		fprintf(ioQQQ,"solution:\n");
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			fprintf(ioQQQ,"%15.7g\t",bvec[i]);
			total += nprot[i]*bvec[i];
		}
		fprintf(ioQQQ,"sum = %15.7g\n",total);
#if 0
		fprintf(ioQQQ,"Verify:\n");
		for (i=MOLMIN; i < N_H_MOLEC; i++ ) 
		{
			total = 0;
			for (j=MOLMIN; j<N_H_MOLEC; j++ ) 
			{
				total += c[j][i]*bvec[j];
				fprintf(ioQQQ,"%14.7g ",c[j][i]*bvec[j]);
			}
			fprintf(ioQQQ,"\t");
		  fprintf(ioQQQ,"%15.7g\n",total);
		}
		fprintf(ioQQQ,"\n");
#endif
	}

	/* For N-R iteration, output from matrix solution is now _change_ in solution */

	fixit(); /* This switches fixup off for dynamics -- can't just scale bvec in this case */

	if(0 && !( iteration >= 2 && dynamics.lgAdvection )) 
	{
		/* Limit the size of the step to up by 100%, down by 50% */
		double nrfac = 1.;
		static int zone = -1;

		/* Be careful on first call */
		if (nzone == 0 && zone != nzone) 
		{
			nrfac = 1e-3;
		  zone = nzone;
		}

		iworst = -1;
		for (i=MOLMIN; i< N_H_MOLEC; i++) 
		{
			if (bvec[i] > 0.) 
			{
				/* Only test `significant' species -- let others go -ve to be caught below */
				if (nrfac*bvec[i] > 0.5*bold[i] && bold[i] > 1e-6*protons) 
				{
					nrfac = 0.5*bold[i]/bvec[i];
					iworst = i;
				}
			} 
			else 
			{
				if (-nrfac*bvec[i] > bold[i] && bold[i] > 1e-6*protons) 
				{
					nrfac = -bold[i]/bvec[i];
					iworst = i;
				}
			}
		}
		/* fprintf(ioQQQ,"Step factor %g species %d\n",nrfac,iworst); */
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			bvec[i] *= nrfac;
		}
	}

	*error = 0;
	for( i=MOLMIN; i < N_H_MOLEC; i++ )
	{
		/* Smooth the error mode tailoff */
		etmp = bvec[i]/(ABSLIM+bold[i]);

		if (PRINTSOL)
			fprintf(ioQQQ,"%15.7g\t",etmp);

		*error += etmp*etmp;
		bvec[i] = bold[i]-bvec[i];
	}
	*error = sqrt(*error)/N_H_MAT;

	if (PRINTSOL)
	{
		double total=0;
		fprintf(ioQQQ,"err = %15.7g\n",*error);
		/* fprintf(ioQQQ,"derived:\n"); */
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
			{
				fprintf(ioQQQ,"%15.7g\t",bvec[i]);
				total += nprot[i]*bvec[i];
			}
		fprintf(ioQQQ,"sum = %15.7g\n",total);
	}

	sum = 0.;
	/* check for negative populations */
	lgNegPop = FALSE;
	fracneg = 0.;
	fracnegfac = 0.;
	iworst = -1;
	for( i=MOLMIN; i < N_H_MOLEC; i++ )
	{
		if( bvec[i] < 0. ) {
			lgNegPop = TRUE;
		}
		fracnegtmp = -bvec[i]/MAX2(bold[i],SMALLFLOAT);
		if (fracnegtmp > fracneg) {
			fracneg = fracnegtmp;
			fracnegfac = (0.5*bold[i]-bvec[i])/(bold[i]-bvec[i]);
			iworst = i;
		}
		sum += nprot[i] * bvec[i] ;
	}
	conserve = (protons - sum)/protons;
	if(fabs(conserve) > 1e-8 )
		fprintf(ioQQQ,"PROBLEM hmoleee protons %.4e, sum %.4e (h-s)/h %.3e \n", protons , sum , 
			conserve );

#	if 0
	/* NDEBUG is set by the compiler to indicate that a debugging mode
	 * has not been specified.  */
#	ifndef NDEBUG
	/*if (NDEBUG)*/
	{
		fprintf( ioQQQ, "Zone %li\n",nzone);
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			fprintf(ioQQQ," %s %.2e", chLab[i], bold[i]);
		}
		fprintf( ioQQQ, " =>\n" );
		for( i=MOLMIN; i < N_H_MOLEC; i++ )
		{
			fprintf(ioQQQ," %s %.2e", chLab[i], bvec[i]);
		}
		fprintf( ioQQQ, "\n" );
	}
#	endif
#	endif

	if(lgNegPop)
	{
#		ifndef NDEBUG
		/*if (NDEBUG)*/
		/* very common to obtain negative soln on very first try - 
		 * don't print in this case */
		if(*nFixup )
		{
			fprintf( ioQQQ, " PROBLEM hmole finds negative H molecule, in zone %li.\n",nzone );
			fprintf( ioQQQ, " Worst is species %d -ve by fraction %g.\n",iworst,fracneg );
			fprintf( ioQQQ, " The populations follow:\n");
			for( i=MOLMIN; i < N_H_MOLEC; i++ )
			{
				fprintf(ioQQQ," %s %.2e", chLab[i], bvec[i]);
			}
			fprintf( ioQQQ, "\n" );
		}
#		endif

		/* Fix negative abundance -- assume the new solution is better in some ways */
		{
			double total=0., ntotal=0., ratio;
			enum {FIXUPTYPE = 1};

			++*nFixup;

			if (FIXUPTYPE == 1) {
				for( i=MOLMIN; i < N_H_MOLEC; i++ )
				{
					total += nprot[i]*bvec[i];
					if (bvec[i] < 0) 
					{
						ntotal += nprot[i]*bvec[i];
						bvec[i] = 0.;
					}
				}
				ratio = total/(total-ntotal);
				for( i=MOLMIN; i < N_H_MOLEC; i++ )
				{
					bvec[i] *= ratio;
				}
			}
			else if (FIXUPTYPE == 2) 
			{
				for( i=MOLMIN; i < N_H_MOLEC; i++ )
				{
					bvec[i] = fracnegfac*bold[i]+(1-fracnegfac)*bvec[i];
				}
			}
				
#			ifndef NDEBUG
			/*if (NDEBUG)*/
			/* very common to obtain negative soln on very first try - 
			 * don't print in this case */
			if( *nFixup>1 )
			{
				fprintf(ioQQQ," FIXUP taken %i time%s.\n\n", *nFixup, (*nFixup == 1)?"":"s");
				fprintf( ioQQQ, " Initially:\n");
				for( i=MOLMIN; i < N_H_MOLEC; i++ )
				{
					fprintf(ioQQQ," %s %.2e", chLab[i], bold[i]);
				}
				fprintf( ioQQQ, "\n" );
				fprintf( ioQQQ, " Changed to:\n");
				for( i=MOLMIN; i < N_H_MOLEC; i++ )
				{
					fprintf(ioQQQ," %s %.2e", chLab[i], bvec[i]);
				}
				fprintf( ioQQQ, "\n" );
			}
#			endif
		}
	}

	/* put derived abundances back into appropriate molecular species */
	h1fnd = bvec[ipMHp];
	h1rat = h1fnd/hatom;
	bvec[ipMH] = dense.xIonDense[ipHYDROGEN][0] * h1rat;
	bvec[ipMHp] = dense.xIonDense[ipHYDROGEN][1] * h1rat;
	/* ASSERT (fabs(bvec[ipMH]+bvec[ipMHp]-h1fnd) < 1e-12 * h1fnd); */

	if (fabs(bvec[ipMH]+bvec[ipMHp]-h1fnd) >= 1e-12 * h1fnd) {
		fprintf(ioQQQ,"%g %g %g %g %g\n",bvec[ipMH],bvec[ipMHp],h1fnd,h1rat,bvec[ipMH]+bvec[ipMHp]-h1fnd);
	}

	for (mol=0;mol<N_H_MOLEC;mol++) 
		hmi.Molec[mol] = (float) bvec[mol];

	dense.xIonDense[ipHYDROGEN][0] = (float) bvec[ipMH];
	dense.xIonDense[ipHYDROGEN][1] = (float) bvec[ipMHp];

	/* total H2 - all forms */
	hmi.htwo_total = hmi.Molec[ipMH2s] + hmi.Molec[ipMH2];
	/* NB this must be kept parallel with nelem and ionstag in H2Lines EmLine struc,
	 * since that struc expects to find the abundances here */
	dense.xIonDense[LIMELM+2][0] = hmi.Molec[ipMH2];
	/* >>>chng 00 apr 04 add test for neg pops - had not been included before!! */

	/* identify dominant H2 formation process */
	{
		/* following should be set true to identify H- formation and destruction processes */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (nzone>200) )
		{
			double createsum ,create_from_Hn2 , create_3body_Ho, create_h2p, 
				create_h3p, create_h3pe, create_grains, create_hminus;
			double destroysum, destroy_hm ,destroy_hm2 ,destroy_soloman ,destroy_2h ,destroy_hp,
				destroy_h,destroy_hp2,destroy_h3p;

			/* H(n=2) + H(n=1) -> H2 */
			create_from_Hn2 = radasc*dense.xIonDense[ipHYDROGEN][0];
			/* 3H => H2 + H */
			create_3body_Ho = bh2dis*dense.xIonDense[ipHYDROGEN][0];
			/* H2+ + H => H2 + H+ */
			create_h2p = hmi.bh2h2p*dense.xIonDense[ipHYDROGEN][0]*bold[ipMH2p];
			/* H + H3+ => H2 + H2+ */
			create_h3p = h3ph2p*dense.xIonDense[ipHYDROGEN][0]*hmi.Molec[ipMH3p];
			/* e + H3+ => H2 + H */
			create_h3pe = eh3_h2h*dense.eden * hmi.Molec[ipMH3p];
			/* from grains */
			create_grains = gv.rate_h2_form_grains_used_total;
			/* from H- */
			create_hminus = ratach*hmi.Molec[ipMHm];

			createsum = create_from_Hn2 + create_3body_Ho + create_h2p +
				create_h3p + create_h3pe + create_grains + create_hminus;

			fprintf(ioQQQ,"createsum \t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
				createsum,
				create_hminus   / createsum,
				create_from_Hn2 / createsum,
				create_3body_Ho / createsum,
				create_h2p      / createsum,
				create_h3p      / createsum,
				create_h3pe     / createsum,
				create_grains   / createsum	);

			/* all h2 molecular hydrogen destruction processes */
			destroy_hm = batach;
			destroy_hm2 = eh2hhm;
			destroy_soloman = hmi.H2_Solomon_rate_used;
			destroy_2h = eh2hh;
			destroy_hp = h2hph3p*dense.xIonDense[ipHYDROGEN][1];
			destroy_h = hmi.rh2dis*dense.xIonDense[ipHYDROGEN][0];
			destroy_hp2 = rh2h2p*dense.xIonDense[ipHYDROGEN][1];
			destroy_h3p = h3petc * hmi.Molec[ipMH3p];
			destroysum = destroy_hm + destroy_hm2 + destroy_soloman + destroy_2h + 
				destroy_hp+ destroy_h+ destroy_hp2+ destroy_h3p;
			fprintf(ioQQQ,"destroysum\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
				destroysum,
				destroy_hm / destroysum ,
				destroy_hm2 / destroysum ,
				destroy_soloman / destroysum ,
				destroy_2h / destroysum ,
				destroy_hp / destroysum ,
				destroy_h / destroysum ,
				destroy_hp2 / destroysum ,
				destroy_h3p / destroysum );

		}
	}

	{
		/* following should be set true to identify H- formation and destruction processes */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (nzone>400) )
		{
			double create_from_Ho,create_3body_Ho,create_batach,create_eh2hhm,destroy_photo,
				destroy_coll_heavies,destroy_coll_electrons,destroy_Hattach,destroy_fhneut,
				destsum , createsum;

			create_from_Ho = (hminus_rad_attach + HMinus_induc_rec_rate);
			create_3body_Ho = c3bod;
			create_batach = batach;
			create_eh2hhm = eh2hhm;
			destroy_photo = hmi.HMinus_photo_rate;
			destroy_coll_heavies = faneut;
			destroy_coll_electrons = cionhm;
			destroy_Hattach = ratach;
			destroy_fhneut = fhneut;

			destsum = destroy_photo + destroy_coll_heavies + destroy_coll_electrons + 
				destroy_Hattach + destroy_fhneut;
			fprintf(ioQQQ,"hminusdebugg\t%.2e\t%.2f\t%.2f\t%.2f\t%.2f\t%.2f\n", 
			 destsum,
			 destroy_photo/destsum , 
			 destroy_coll_heavies/destsum,
			 destroy_coll_electrons/destsum, 
			 destroy_Hattach/destsum,
			 destroy_fhneut/destsum );

			createsum = create_from_Ho+create_3body_Ho+create_batach+create_eh2hhm;
			fprintf(ioQQQ,"createsum\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
				createsum,
				create_from_Ho/createsum,
				create_3body_Ho/createsum,
				create_batach/createsum,
				create_eh2hhm/createsum);
		}
	}

	/* rate H-alpha is created by H- ct */
	hmi.HalphaHmin = (float)(fhneut*hmi.Molec[ipMHm]);

	/* heating due to H2 dissociation */
	if( nomole.lgNoH2Mole )
	{
		hmi.HeatH2Dish_TH85 = 0.;
		hmi.HeatH2Dexc_TH85 = 0.;
	}
	else
	{
		/* H2 photodissociation heating, eqn A9 of Tielens & Hollenbach 1985a */
		hmi.HeatH2Dish_TH85 = (float)(1.36e-23*hmi.Molec[ipMH2]*h2esc*hmi.GammaHabing);

		/* >>chng 00 nov 25, explictly break out this heat agent */
		/* 2.6 eV of heat per deexcitation, consider difference
		 * between deexcitation (heat) and excitation (cooling) */
		hmi.HeatH2Dexc_TH85 = (float)( (hmi.Molec[ipMH2s]*H2star_deexcit - hmi.Molec[ipMH2]*H2star_excit) * 4.17e-12);
		/*fprintf(ioQQQ," hmole H2 deex heating:\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
			hmi.HeatH2Dexc_TH85,
			hmi.Molec[ipMH2s],
			H2star_deexcit, 
			hmi.Molec[ipMH2],
			H2star_excit,
			bold[ipMH2],
			bold[ipMH],
			phycon.te);*/
	}
	/* this is deriv wrt temperature, only if counted as a cooling source */
	hmi.deriv_HeatH2Dexc_TH85 = (float)(MAX2(0.,-hmi.HeatH2Dexc_TH85)* ( 30172. * thermal.tsq1 - thermal.halfte ) );


	/*dense.xMolecules[ipHYDROGEN] += hmi.hminus + 2.f*hmi.htwo + 2.f*hmi.h2plus + hmi.hehp + 
		2.f*hmi.htwo_star + 3.f*hmi.h3plus;

	dense.xMolecules[ipHYDROGEN] += hevmolec.hevmol[ipCH]+ hevmolec.hevmol[ipCHP]+ hevmolec.hevmol[ipOH]+
		hevmolec.hevmol[ipOHP]+ 2.f*hevmolec.hevmol[ipH2O]+ 2.f*hevmolec.hevmol[ipH2OP]+	
		3.f*hevmolec.hevmol[ipH3P_hev]+3.f*hevmolec.hevmol[ipH3OP]+2.f*hevmolec.hevmol[ipCH2P]+
		2.f*hevmolec.hevmol[ipCH2]+ 3.f*hevmolec.hevmol[ipCH3];*/

	{
		/* following should be set true to print populations */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			/* these are the raw results */
			fprintf( ioQQQ, " HMOLE raw; hi\t%.2e" , dense.xIonDense[ipHYDROGEN][0]);
			for( i=0; i < N_H_MOLEC; i++ )
			{
				fprintf( ioQQQ, "\t%s\t%.2e", chLab[i], bvec[i] );
			}
			fprintf( ioQQQ, " \n" );
		}
	}

	if( trace.lgTrace && trace.lgTrMole )
	{
		/* these are the raw results */
		fprintf( ioQQQ, " raw; " );
		for( i=0; i < N_H_MOLEC; i++ )
		{
			fprintf( ioQQQ, " %s:%.2e", chLab[i], bvec[i] );
		}
		fprintf( ioQQQ, " \n" );

	}

	/* option to print rate H2 forms */
	/* trace.lgTrMole is trace molecules option,
	 * punch htwo */
	if( (trace.lgTrace && trace.lgTrMole) || punch.lgPunH2 )
	{
		if( punch.lgPunH2 )
		{
			ioFile = punch.ipPunH2;
		}
		else
		{
			ioFile = ioQQQ;
		}

		rate = gv.rate_h2_form_grains_used_total + ratach*hmi.Molec[ipMHm] + bh2dis*dense.xIonDense[ipHYDROGEN][0] + 
			hmi.bh2h2p*dense.xIonDense[ipHYDROGEN][0]*hmi.Molec[ipMH2p] + 
			radasc*dense.xIonDense[ipHYDROGEN][0] + 
			h3ph2p*hmi.Molec[ipMH3p] + 
			h3petc*hmi.Molec[ipMH3p];

		if( rate > 0. )
		{
			fprintf( ioFile, 
				" Create H2, rate=%10.2e grain;%5.3f hmin;%5.3f bhedis;%5.3f h2+;%5.3f radasc:%5.3f h3ph2p:%5.3f h3petc:%5.3f\n", 
			  rate, 
			  gv.rate_h2_form_grains_used_total/rate, 
			  ratach*hmi.Molec[ipMHm]/rate, 
			  bh2dis/rate, 
			  hmi.bh2h2p*dense.xIonDense[ipHYDROGEN][0]*hmi.Molec[ipMH2p]/rate, 
			  radasc*dense.xIonDense[ipHYDROGEN][0]/rate, 
			  h3ph2p*hmi.Molec[ipMH3p]/rate, 
			  h3petc*hmi.Molec[ipMH3p]/rate );
		}
		else
		{
			fprintf( ioFile, " Create H2, rate=0\n" );
		}
	}

	/* this is H2+ */
	if( trace.lgTrace && trace.lgTrMole )
	{
		rate = rh2h2p*hmi.Molec[ipMH2]*dense.xIonDense[ipHYDROGEN][1] + b2pcin*dense.xIonDense[ipHYDROGEN][1]*dense.eden*dense.xIonDense[ipHYDROGEN][0] + 
		  h3ph2p*hmi.Molec[ipMH3p] + h3petc*hmi.Molec[ipMH3p];
		if( rate > 1e-25 )
		{
			fprintf( ioQQQ, "  Create H2+, rate=%10.2e rh2h2p;%5.3f b2pcin;%5.3f h3ph2p;%5.3f h3petc+;%5.3f\n", 
			  rate, rh2h2p*dense.xIonDense[ipHYDROGEN][1]*hmi.Molec[ipMH2]/rate, 
				b2pcin*dense.xIonDense[ipHYDROGEN][1]*dense.xIonDense[ipHYDROGEN][1]*
			  dense.eden/rate, h3ph2p*hmi.Molec[ipMH3p]/rate, h3petc*hmi.Molec[ipMH3p]/
			  rate );
		}
		else
		{
			fprintf( ioQQQ, "  Create H2+, rate=0\n" );
		}
	}

	if( hmi.Molec[ipMHm] > 0. && phmlte > 0. )
	{
		hmi.hmidep = (double)hmi.Molec[ipMHm]/((double)dense.xIonDense[ipHYDROGEN][0]*dense.eden*phmlte);
	}
	else
	{
		hmi.hmidep = 1.;
	}

	/* this will be net volume heating rate, photo heat - induc cool */
	hmi.hmihet = HMinus_photo_heat*hmi.Molec[ipMHm] - HMinus_induc_rec_cooling;
	hmi.h2plus_heat = h2phet*hmi.Molec[ipMH2p];

	/* THTMOL is total heating due to absorption of Balmer continuum
	 * thtmol = hmihet + h2phet * h2plus
	 * departure coef for H2 defined rel to N(1s)**2
	 * (see Littes and Mihalas Solar Phys 93, 23) */
	plte = (double)(dense.xIonDense[ipHYDROGEN][0]) * (double)(dense.xIonDense[ipHYDROGEN][0]) * hmi.ph2lte;
	if( plte > 0. )
	{
		hmi.h2dep = hmi.Molec[ipMH2]/plte;
	}
	else
	{
		hmi.h2dep = 1.;
	}

	/* departure coef of H2+ defined rel to N(1s) N(p)
	 * sec den was HI before 85.34 */
	plte = dense.xIonDense[ipHYDROGEN][0]*dense.xIonDense[ipHYDROGEN][1]*phplte;
	if( plte > 0. )
	{
		hmi.h2pdep = hmi.Molec[ipMH2p]/plte;
	}
	else
	{
		hmi.h2pdep = 1.;
	}

	/* departure coef of H3+ defined rel to N(H2+) N(p) */
	if( ph3lte > 0. )
	{
		hmi.h3pdep = hmi.Molec[ipMH3p]/ph3lte;
	}
	else
	{
		hmi.h3pdep = 1.;
	}


	if( trace.lgTrace && trace.lgTrMole )
	{
		fprintf( ioQQQ, " HMOLE, Dep Coef, H-:%10.2e H2:%10.2e H2+:%10.2e\n", 
		  hmi.hmidep, hmi.h2dep, hmi.h2pdep );
		fprintf( ioQQQ, "     H- creat: Rad atch%10.3e Induc%10.3e bHneut%10.2e 3bod%10.2e b=>H2%10.2e N(H-);%10.2e b(H-);%10.2e\n", 
		  hminus_rad_attach, HMinus_induc_rec_rate, bhneut, c3bod, batach, hmi.Molec[ipMHm], hmi.hmidep );

		fprintf( ioQQQ, "     H- destr: Photo;%10.3e mut neut%10.2e e- coll ion%10.2e =>H2%10.2e x-ray%10.2e p+H-%10.2e\n", 
		  hmi.HMinus_photo_rate, faneut, cionhm, ratach, iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s], 
		  fhneut );
		fprintf( ioQQQ, "     H- heating:%10.3e Ind cooling%10.2e Spon cooling%10.2e\n", 
		  hmi.hmihet, HMinus_induc_rec_cooling, hmi.hmicol );
	}

	/* identify creation and destruction processes for H2+ */
	if( trace.lgTrace && trace.lgTrMole )
	{
		rate = desh2p;
		if( rate != 0. )
		{
			fprintf( ioQQQ, 
				" Destroy H2+: rate=%10.2e e-;%5.3f phot;%5.3f hard gam;%5.3f H2col;%5.3f h2phhp;%5.3f pion;%5.3f bh2h2p:%5.3f\n", 
			  rate, h2pcin*dense.eden/rate, gamtwo/rate, 2.*iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s]/
			  rate, h2ph3p/rate, h2phhp/rate, h2pion/rate, hmi.bh2h2p*
			  dense.xIonDense[ipHYDROGEN][0]/rate );

			rate *= hmi.Molec[ipMH2p];
			if( rate > 0. )
			{
				fprintf( ioQQQ, 
					" Create  H2+: rate=%10.2e HII HI;%5.3f Col H2;%5.3f HII H2;%5.3f HI HI;%5.3f\n", 
				  rate, 
				  radath*dense.xIonDense[ipHYDROGEN][1]*dense.xIonDense[ipHYDROGEN][0]/rate, 
				  (gamhd + Secondaries.csupra*0.93)*hmi.Molec[ipMH2]/rate, 
				  rh2h2p*dense.xIonDense[ipHYDROGEN][1]*hmi.Molec[ipMH2]/rate, 
				  b2pcin*dense.xIonDense[ipHYDROGEN][0]*dense.xIonDense[ipHYDROGEN][1]*dense.eden/rate );
			}
			else
			{
				fprintf( ioQQQ, " Create  H2+: rate= is zero\n" );
			}
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->hmole()\n", debug_fp );
#	endif
	return;
}


/*hmirat compute radiative association rate for H- */
static double hmirat(double te)
{
	double hmirat_v;

#	ifdef DEBUG_FUN
	fputs( "<+>hmirat()\n", debug_fp );
#	endif


	/* fits to radiative association rate coefficients */
	if( te < 31.62 )
	{
		hmirat_v = 8.934e-18*phycon.sqrte*phycon.te003*phycon.te001*
		  phycon.te001;
	}
	else if( te < 90. )
	{
		hmirat_v = 5.159e-18*phycon.sqrte*phycon.te10*phycon.te03*
		  phycon.te03*phycon.te003*phycon.te001;
	}
	else if( te < 1200. )
	{
		hmirat_v = 2.042e-18*te/phycon.te10/phycon.te03;
	}
	else if( te < 3800. )
	{
		hmirat_v = 8.861e-18*phycon.te70/phycon.te03/phycon.te01*
		  phycon.te003;
	}
	else if( te <= 1.4e4 )
	{
		/* following really only optimized up to 1e4 */
		hmirat_v = 8.204e-17*phycon.sqrte/phycon.te10/phycon.te01*
		  phycon.te003;
	}
	else
	{
		/* >>chng 00 sep 28, add this branch */
		hmirat_v = 5.44e-16*phycon.te20/phycon.te01;
	}

#	ifdef DEBUG_FUN
	fputs( " <->hmirat()\n", debug_fp );
#	endif
	return( hmirat_v );
}

