/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*CS_l_mixing - find rate for l-mixing collisions by protons, for neutrals */
/*HeCSInterp interpolate on He1 collision strengths */
/*AtomCSInterp do the atom	*/
/*IonCSInterp do the ions	*/
#include "cddefines.h" 
#include "iso.h"
#include "abund.h"
#include "atomcwgt.h"
#include "dense.h"
#include "helike.h"
#include "helike_cs.h"
#include "phycon.h"
#include "physconst.h"
#include "taulines.h"
#include "hydro_bauman.h"
#include "hydro_vs_rates.h"

/*CS_l_mixing - find rate for l-mixing collisions by protons, for neutrals */
double CS_l_mixing(
	long nelem /* the chemical element, 1 for He */,
	long ipLo /* lower level, 0 for ground */,
	long ipHi /* upper level, 0 for ground */ )
{
	/* >>refer	He	l-mixing	Pengelly, R.M., & Seaton, M.J., 1964, MNRAS, 127, 165 */
	double cs, Dul,
		TwoLogDebye, TwoLogRc1, TwoLogRc2,
		factor1, factor2, factor3,
		bestfactor,	factorpart,
		reduced_mass, reduced_mass_2_emass,
		rate, radint, tau, RMSv, deltaE;
	/* only do proton collisions for now */
	const double ChargIncoming = 1.;

#	ifdef DEBUG_FUN
	fputs( "<+>CS_l_mixing()\n", debug_fp );
#	endif

	/* In this routine, three different cutoff radii are calculated, and as per PS64,
	 * the least of these is selected.  Must be careful to take least positive result!	*/
	
	/* This reduced mass is in grams.	*/
	reduced_mass = AtomcWgt.AtomicWeight[nelem]*AtomcWgt.AtomicWeight[ipHYDROGEN]/
		(AtomcWgt.AtomicWeight[nelem]+AtomcWgt.AtomicWeight[ipHYDROGEN])*ATOMIC_MASS_UNIT;
	/* reduced mass in gm, only do proton collisions for now
	 * this mass always appears relative to the electron mass, so define it that way */
	reduced_mass_2_emass = reduced_mass / ELECTRON_MASS;

	/* this is root mean squared velocity:	
	 * kt is in eV.  Divide by (1E6 * reduced_mass_MeVoverCsqrd)
	 * the factor of 0.001 is to convert grams to kilograms. */
	RMSv = sqrt( 2. * 8.61735E-5 * phycon.te * 1.6022E-19 / (0.001 * reduced_mass ) );
	/* This is the difference between energy levels, in eV.	*/
	deltaE = EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN * WAVNRYD * EVRYD;
	
	/* This is the lifetime of ipLo.	*/
	tau = helike.Lifetime[nelem][ipLo];
	
	/* equation 46 of PS64 */
	/* min on density added to prevent this from becoming large and negative
	 * at very high n_e - Pengelly & Seaton did not apply this above
	 * 1e12 cm^-3 */
	/* This is actually 2 times the log of the Debye radius.	*/
	TwoLogDebye = 1.68 + log10( phycon.te / MIN2(1e12 , dense.eden ) );

	/* This is cutoff = 0.72v(tau), where tau is the lifetime of the level nl.	*/
	TwoLogRc1 = 10.95 + log10( phycon.te * tau * tau / reduced_mass_2_emass );
	
	/* This is cutoff = 1.12hbar*v/deltaE.	*/
	/* If deltaE is zero or negative, set this cutoff to zero, so that it will be excluded below.	*/
	if ( deltaE > 0. )
		TwoLogRc2 = 2. * log10( 1.12 * HBAReV * RMSv / deltaE );
	else
		TwoLogRc2 = 0.;

	/* Calculate the hydrogenic radial integral */
	if (iso.quant_desig[ipHE_LIKE][nelem][ipHi].l > iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)
		radint = hri(iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,iso.quant_desig[ipHE_LIKE][nelem][ipHi].l,
			iso.quant_desig[ipHE_LIKE][nelem][ipLo].n, iso.quant_desig[ipHE_LIKE][nelem][ipLo].l, nelem);
	else if (iso.quant_desig[ipHE_LIKE][nelem][ipHi].l < iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)
		radint = hri(iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,iso.quant_desig[ipHE_LIKE][nelem][ipLo].l,
			iso.quant_desig[ipHE_LIKE][nelem][ipLo].n, iso.quant_desig[ipHE_LIKE][nelem][ipHi].l, nelem);
	else 
		radint = 0.;

	ASSERT( radint > 0. );

	Dul = 8./3. * radint * radint * ChargIncoming * ChargIncoming;

	factorpart = (11.54 + log10( phycon.te  / Dul / reduced_mass_2_emass ) );

	if( (factor1 = factorpart + TwoLogDebye) <= 0.)
		factor1 = BIGDOUBLE;
	if( (factor2 = factorpart + TwoLogRc1) <= 0.)
		factor2 = BIGDOUBLE;
	if( (TwoLogRc2 == 0.) || ( (factor3 = factorpart + TwoLogRc2) <= 0.) )
		factor3 = BIGDOUBLE;
	
	/* Now we find the least positive result.	*/
	/* It would be suspect if bestfactor gets above the double digit range,
	 * so let's make sure less than 100.	*/
	if( (bestfactor = MIN3(factor1,factor2,factor3)) > 100.)
	{
		fprintf( ioQQQ,"Must revise CS_l_mixing...all trials fail at nelem %li\tipLo %li\ttemp %e\teden %e\tbestfactor %e\n",
			nelem,
			ipLo,
			phycon.te,
			dense.eden,
			bestfactor);
		fprintf( ioQQQ,"Stop in Helike.c [CS_l_mixing]");
					cdEXIT(EXIT_FAILURE);
	}
	/* if the preferred method fails us this approximation	*/
	else
	{
		/* this calculates the TOTAL collision strength from nl to nl+/-1. Use as last resort.	*/
		/* equation 44 of PS64 */
		/* NB - this only works for helium-like ions, the nelem in pow2 is the residual nuclear charge */
		Dul = POW2( ChargIncoming / (nelem) ) * 6. * POW2( (double)iso.quant_desig[ipHE_LIKE][nelem][ipLo].n ) *
			( POW2((double)iso.quant_desig[ipHE_LIKE][nelem][ipLo].n) - 
			POW2((double)iso.quant_desig[ipHE_LIKE][nelem][ipLo].l) - iso.quant_desig[ipHE_LIKE][nelem][ipLo].l - 1);

		bestfactor = (11.54 + log10( phycon.te  / Dul / reduced_mass_2_emass ) );
	}

	ASSERT( bestfactor > 0.);

	rate = 9.93e-6 * sqrt( reduced_mass_2_emass  ) * Dul / phycon.sqrte * bestfactor;

	/* convert rate to cs */
	cs = rate / 8.629e-6 * phycon.sqrte * iso.stat[ipHE_LIKE][nelem][ipLo] ;

	ASSERT( cs > 0. );

#	ifdef DEBUG_FUN
	fputs( " <->CS_l_mixing()\n", debug_fp );
#	endif

	return cs ;
}

/* Choose either AtomCSInterp or IonCSInterp */
float HeCSInterp( long nelem, long ipHi , long ipLo )
{
	float cs, factor;

	/* This variable is for diagnostic purposes:
	 * a string used in the output to designate where each cs comes from.	*/	
	char *where = "      ";

	if( nelem == ipHELIUM )
		cs = AtomCSInterp( nelem, ipHi , ipLo, &factor, &where );
	else
		cs = IonCSInterp( nelem, ipHi , ipLo, &factor, &where );

	ASSERT( cs >= 0.f );
	
	/* in many cases the correction factor for split states has already been made,
	 * if not then factor is still negative */
	if( factor < 0.f )
	{
		ASSERT( helike.lgCS_Vriens );

		/* this branch both upper and lower are within collapsed levels */
		if( iso.quant_desig[ipHE_LIKE][nelem][ipLo].n > iso.n_HighestResolved[ipHE_LIKE][nelem] )
		{
			factor = 1.f;
		}
		/* this branch both upper and lower are within resolved levels */
		else if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n <= iso.n_HighestResolved[ipHE_LIKE][nelem] )
		{
			factor = iso.stat[ipHE_LIKE][nelem][ipHi]*iso.stat[ipHE_LIKE][nelem][ipLo]/
				((4.f*iso.quant_desig[ipHE_LIKE][nelem][ipHi].n)*(2.f*iso.quant_desig[ipHE_LIKE][nelem][ipHi].l+1))/
				((4.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].n)*(2.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].l+1));
		}
		/* this branch upper collapsed, lower resolved */
		else if( iso.quant_desig[ipHE_LIKE][nelem][ipLo].n <= iso.n_HighestResolved[ipHE_LIKE][nelem]
			&& iso.quant_desig[ipHE_LIKE][nelem][ipHi].n > iso.n_HighestResolved[ipHE_LIKE][nelem] )
		{
			factor = iso.stat[ipHE_LIKE][nelem][ipLo]/
				((4.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].n)*(2.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].l+1));
		}
		else
			TotalInsanity();
	}

	/* take factor into account, usually 1, ratio of stat weights if within 2 3P 
	 * and with collisions from collapsed to resolved levels */
	cs *= factor;

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		
		if( DEBUG_LOC && ( nelem==ipOXYGEN ) && (cs > 0.f) && (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == 2) 
			&& ( iso.quant_desig[ipHE_LIKE][nelem][ipLo].n <= 2 ) )
			fprintf(ioQQQ,"%li\t%li\t%li\t-\t%li\t%li\t%li\t%.2e\t%s\n", 
				iso.quant_desig[ipHE_LIKE][nelem][ipLo].n , iso.quant_desig[ipHE_LIKE][nelem][ipLo].s ,
				iso.quant_desig[ipHE_LIKE][nelem][ipLo].l ,	iso.quant_desig[ipHE_LIKE][nelem][ipHi].n ,
				iso.quant_desig[ipHE_LIKE][nelem][ipHi].s , iso.quant_desig[ipHE_LIKE][nelem][ipHi].l , cs,where);
	}

	return cs;
}

float AtomCSInterp( long nelem, long ipHi , long ipLo, float *factor, char **where )
{
	long ipArray;
	float cs, flow;

#	ifdef DEBUG_FUN
	fputs( "<+>AtomCSInterp()\n", debug_fp );
#	endif

	ASSERT( nelem == ipHELIUM );

	/* init values, better be positive when we exit */
	cs = -1.f; 

	/* this may be used for splitting up the collision strength within 2 3P when
	 * the lower level is withint 2 3P, and for collisions between resolved and collapsed levels.
	 * It may be set somewhere in routine, so set to negative value here as flag saying not set */
	*factor = -1.f;

	/* for most of the helium iso sequence, the order of the J levels within 2 3P 
	 * in increasing energy, is 0 1 2 - the exception is atomic helium itself,
	 * which is swapped, 2 1 0 */

	/* this branch is for upper and lower levels within 2p3P */
	if( ipLo >= ipHe2p3P0 && ipHi <= ipHe2p3P2 )
	{
		*factor = 1.f;
		/* atomic helium, use Berrington private comm cs */
		
		/* >>refer	he1	cs	Berrington, Keith, 2001, private communication - email follows
		> Dear Gary,
		> I could not find any literature on the He fine-structure
		> problem (but I didn't look very hard, so there may be 
		> something somewhere). However, I did a quick R-matrix run to 
		> see what the magnitude of the collision strengths are... At 
		> 1000K, I get the effective collision strength for 2^3P J=0-1, 
		>  0-2, 1-2 as 0.8,0.7,2.7; for 10000K I get 1.2, 2.1, 6.0
		*/
		if( ipLo == ipHe2p3P0 && ipHi == ipHe2p3P1 )
		{
			cs = 1.2f;
		}
		else if( ipLo == ipHe2p3P0 && ipHi == ipHe2p3P2 )
		{
			cs = 2.1f;
		}
		else if( ipLo == ipHe2p3P1 && ipHi == ipHe2p3P2 )
		{
			cs = 6.0f;
		}
		else
		{
			cs = 1.0f;
			TotalInsanity();
		}
		
		*where = "Berr  ";
	}

	/* >>chng 02 feb 25, Bray data should come first since it is the best we have.	*/
	/* this branch is the Bray et al data, for n <= 5, where quantal calcs exist 
	 * must exclude ipLo >= ipHe2p1P because they give no numbers for those	*/
	else if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n <= 5 && ipLo<ipHe2p1P )
	{
		ASSERT( *factor == -1.f );
		ASSERT( ipLo < ipHi );
		ASSERT( ipHe2p3P0 == 3 );

		/* ipLo is within 2^3P	*/
		if( ipLo >= ipHe2p3P0 && ipLo <= ipHe2p3P2 )
		{
			/* *factor is ratio of statistical weights of level to term */
			
			/* ipHe2p3P0, ipHe2p3P1, ipHe2p3P2 have indices 3,4,and 5, but j=0,1,and 2.	*/
			*factor = (2.f*((float)ipLo-3.f)+1.f) / 9.f;
			
			/* ipHi must be above ipHe2p3P2 since 2p3Pj->2p3Pk taken care of above	*/
			ASSERT( ipHi > ipHe2p3P2 );
		}
		/* ipHi is within 2^3P	*/
		else if( ipHi >= ipHe2p3P0 && ipHi <= ipHe2p3P2 )
		{
			ASSERT( ipLo < ipHe2p3P0 );

			*factor = (2.f*((float)ipHi-3.f)+1.f) / 9.f;
		}
		/* neither are within 2^3P...no splitting necessary	*/
		else 
		{
			*factor = 1.f;
		}

		/* SOME OF THESE ARE NOT N-CHANGING!	*/
		/* Must be careful about turning each one on or off.	*/
		
		/* this is the case where we have quantal calculations */
		/* >>refer	He1	cs	Bray, I., Burgess, A., Fursa, D.V., & Tully, J.A., 2000, A&AS, 146, 481-498 */
		/* check whether we are outside temperature array bounds,
		 * and use extreme value if we are */
		if( phycon.alogte <= helike.CSTemp[0] )
		{
			cs = helike.HeCS[nelem][ipHi][ipLo][0];
		}
		else if( phycon.alogte >= helike.CSTemp[helike.nCS-1] )
		{
			cs = helike.HeCS[nelem][ipHi][ipLo][helike.nCS-1];
		}
		else
		{
			/* find which array element within the cs vs temp array */
			ipArray = (long)((phycon.alogte - helike.CSTemp[0])/(helike.CSTemp[1]-helike.CSTemp[0]));
			ASSERT( ipArray < helike.nCS );
			ASSERT( ipArray >= 0 );

			/* when taking the average, this is the fraction from the lower temperature value */
			flow = (float)( (phycon.alogte - helike.CSTemp[ipArray])/
				(helike.CSTemp[ipArray+1]-helike.CSTemp[ipArray])) ;
			ASSERT( (flow >= 0.f) && (flow <= 1.f) );

			cs = helike.HeCS[nelem][ipHi][ipLo][ipArray] * (1.f-flow) +
				helike.HeCS[nelem][ipHi][ipLo][ipArray+1] * flow;
		}

		*where = "Bray ";

		/* options to kill collisional excitation and/or l-mixing	*/
		if ( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == iso.quant_desig[ipHE_LIKE][nelem][ipLo].n )
			/* iso.lgColl_l_mixing turned off with atom he-like l-mixing collisions off command */
			cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
		else
		{
			/* iso.lgColl_excite turned off with atom he-like collisional excitation off command */
			cs *= (float)iso.lgColl_excite[ipHE_LIKE];
			/* This kills collisions to n=5, for comparison to Benjamin et al 99.	*/
			if( ( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >= 5 ) && helike.lgSetBenjamin )
				cs = 0.f;
		}

		ASSERT( cs >= 0.f );
	}

	/* this branch, n-same, l-changing collision, and not case of He with quantal data */
	else if( (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == iso.quant_desig[ipHE_LIKE][nelem][ipLo].n ) 
		&& (iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == iso.quant_desig[ipHE_LIKE][nelem][ipLo].s ) )
	{
		ASSERT( *factor == -1.f );
		*factor = 1.f;
		
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >= 3 );
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n <= iso.n_HighestResolved[ipHE_LIKE][nelem] );

		/* this branch, l changing by one */
		if( abs(iso.quant_desig[ipHE_LIKE][nelem][ipHi].l - iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)== 1)
		{
			cs = (float)CS_l_mixing( nelem, ipLo, ipHi);

			/* add to existing collision strengths
			* this CS should be multiplied by the proton density,
			* but the code will use the electron density, 
			* so ratio of proton to electron densities precorrects for this. */
			cs *= (float)(dense.xIonDense[ipHYDROGEN][1]/phycon.EdenHCorr);
		}
		else
		{
			/* l changes by more than 1, but same-n collision */
			cs = 0.f;
		}

		*where = "lmix  ";
		cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
	}

	/* this is an atomic n-changing collision with no quantal calculations */
	else if ( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n != iso.quant_desig[ipHE_LIKE][nelem][ipLo].n )
	{
		ASSERT( *factor == -1.f );
		/* this is an atomic n-changing collision with no quantal calculations */
		/* gbar g-bar goes here */

		/* >>chng 02 jul 25, add option for fits to quantal cs data */
		if( helike.lgCS_Vriens )
		{
			/* this is Vriens & Smeets collision strength for neutral H */
			cs = (float)hydro_vs_deexcit( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n, iso.quant_desig[ipHE_LIKE][nelem][ipLo].n );
			*where = "Vriens";
		}
		else if( helike.lgCS_None )
		{
			cs = 0.f;
			*factor = 1.f;
			*where = "no gb";
		}
		else if( helike.lgCS_new )
		{
			*factor = 1.f;
			
			/* two different fits, allowed and forbidden */
			if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul > 1. )
			{
				/* permitted lines - large A */
				double x = 
					log10(MAX2(34.7,EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN));

				if( helike.lgCS_new == 1 )
				{
					/* this is the broken power law fit, passing through both quantal
					 * calcs at high energy and asymptotically goes to VS at low energies */
					if( x < 4.5 )
					{
						/* low energy fit for permitted transitions */
						cs = (float)pow( 10. , -1.45*x + 6.75);
					}
					else
					{
						/* higher energy fit for permitted transitions */
						cs = (float)pow( 10. , -3.33*x+15.15);
					}
				}
				else if( helike.lgCS_new == 2 )
				{
					/* single parallel fit for permitted transitions, runs parallel to VS */
					cs = (float)pow( 10. , -2.3*x+10.3);
				}
				else
					TotalInsanity();
			}
			else
			{
				/* fit for forbidden transitions */
				if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN < 25119.f )
				{
					cs = 0.631f; 
				}
				else
				{
					cs = (float)pow(10., 
						-3.*log10(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN)+12.8);
				}
			}

			*where = "newgb";

			/* option to print the resulting collision strengths
			if( nelem==1 )
				fprintf(ioQQQ,"%.3e\t%.3e\t%.3e\n", 
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul ,
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
				cs ); */
		}
		else
			TotalInsanity();

		/* options to turn off collisions */
		cs *= (float)iso.lgColl_excite[ipHE_LIKE];

		/* This kills collisions to n=5, for comparison to Benjamin et al 99.	*/
		if( ( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >= 5 ) && helike.lgSetBenjamin )
			cs = 0.f;
	}
	else
	{
		/* If spin changing collisions are prohibited in the l-mixing routine,
		 * they will fall here, and will have been assigned no collision strength.	
		 * Assign zero for now.	*/
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].s != iso.quant_desig[ipHE_LIKE][nelem][ipLo].s );
		cs = 0.f;
		*factor = 1.f;
	}

	ASSERT( cs >= 0.f );

	if( helike.lgSetBenjamin )
	{
		if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >=5 && 
			iso.quant_desig[ipHE_LIKE][nelem][ipLo].n != iso.quant_desig[ipHE_LIKE][nelem][ipHi].n )
		{
			ASSERT( cs == 0.f );
		}
		else if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >=5 &&
			iso.quant_desig[ipHE_LIKE][nelem][ipLo].s != iso.quant_desig[ipHE_LIKE][nelem][ipHi].s )
		{
			cs = 0.f;
		}
	}
	
	return(cs);

#	ifdef DEBUG_FUN
	fputs( " <->AtomCSInterp()\n", debug_fp );
#	endif

}

/* IonCSInterp interpolate on collision strengths for element nelem */
float IonCSInterp( long nelem , long ipHi , long ipLo, float *factor, char **where  )
{
	float cs;

#	ifdef DEBUG_FUN
	fputs( "<+>IonCSInterp()\n", debug_fp );
#	endif

	ASSERT( nelem > ipHELIUM );
	ASSERT( nelem < LIMELM );

	/* init values, better be positive when we exit */
	cs = -1.f; 

	/* this may be used for splitting up the collision strength for collisions between
	 * resolved and collapsed levels.  It may be set somewhere in routine, so set to 
	 * negative value here as flag saying not set */
	*factor = -1.f;


	/* >>chng 02 mar 04,  the approximation here for transitions within 2p3P was not needed,
	 * because the Zhang data give these transitions.  They are of the same order, but are 
	 * specific to the three transitions	*/

	/* this branch is ground to n=2 or from n=2 to n=2, for ions only	*/
	/*>>refer Helike	CS	Zhang, Honglin, & Sampson, Douglas H. 1987, ApJS 63, 487	*/
	if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n==2 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].n<=2 )
	{
		*where = "Zhang";
		*factor = 1.;
		
		/* Collisions from gound	*/
		if( ipLo == ipHe1s1S )
		{
			switch( ipHi )
			{
			case 1:	/* to 2tripS	*/
				cs = 0.25f/(float)POW2(nelem+1.);
				break;
			case 2: /* to 2singS	*/
				cs = 0.4f/(float)POW2(nelem+1.);
				break;
			case 3: /* to 2tripP0	*/
				cs = 0.15f/(float)POW2(nelem+1.);
				break;
			case 4: /* to 2tripP1	*/
				cs = 0.45f/(float)POW2(nelem+1.);
				break;
			case 5: /* to 2tripP2	*/
				cs = 0.75f/(float)POW2(nelem+1.);
				break;
			case 6: /* to 2singP	*/
				cs = 1.3f/(float)POW2(nelem+1.);
				break;
			default:
				TotalInsanity();
				break;
			}
			cs *= (float)iso.lgColl_excite[ipHE_LIKE];
		}
		/* collisions from 2tripS to n=2	*/
		else if( ipLo == ipHe2s3S )
		{
			switch( ipHi )
			{
			case 2: /* to 2singS	*/
				cs = 2.75f/(float)POW2(nelem+1.);
				break;
			case 3: /* to 2tripP0	*/
				cs = 60.f/(float)POW2(nelem+1.);
				break;
			case 4: /* to 2tripP1	*/
				cs = 180.f/(float)POW2(nelem+1.);
				break;
			case 5: /* to 2tripP2	*/
				cs = 300.f/(float)POW2(nelem+1.);
				break;
			case 6: /* to 2singP	*/
				cs = 5.8f/(float)POW2(nelem+1.);
				break;
			default:
				TotalInsanity();
				break;
			}
			cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
		}
		/* collisions from 2singS to n=2	*/
		else if( ipLo == ipHe2s1S )
		{
			switch( ipHi )
			{
			case 3: /* to 2tripP0	*/
				cs = 0.56f/(float)POW2(nelem+1.);
				break;
			case 4: /* to 2tripP1	*/
				cs = 1.74f/(float)POW2(nelem+1.);
				break;
			case 5: /* to 2tripP2	*/
				cs = 2.81f/(float)POW2(nelem+1.);
				break;
			case 6: /* to 2singP	*/
				cs = 190.f/(float)POW2(nelem+1.);
				break;
			default:
				TotalInsanity();
				break;
			}
			cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
		}
		/* collisions from 2tripP0 to n=2	*/
		else if( ipLo == ipHe2p3P0 )
		{
			switch( ipHi )
			{
			case 4: /* to 2tripP1	*/
				cs = 8.1f/(float)POW2(nelem+1.);
				break;
			case 5: /* to 2tripP2	*/
				cs = 8.2f/(float)POW2(nelem+1.);
				break;
			case 6: /* to 2singP	*/
				cs = 3.9f/(float)POW2(nelem+1.);
				break;
			default:
				TotalInsanity();
				break;
			}
			cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
		}
		/* collisions from 2tripP1 to n=2	*/
		else if( ipLo == ipHe2p3P1 )
		{
			switch( ipHi )
			{
			case 5: /* to 2tripP2	*/
				cs = 30.f/(float)POW2(nelem+1.);
				break;
			case 6: /* to 2singP	*/
				cs = 11.7f/(float)POW2(nelem+1.);
				break;
			default:
				TotalInsanity();
				break;
			}
			cs *= (float)iso.lgColl_l_mixing[ipHE_LIKE];
		}
		/* collisions from 2tripP2 to n=2	*/
		else if( ipLo == ipHe2p3P2 )
		{
			/* to 2singP	*/
			cs = 19.5f/(float)POW2(nelem+1.) * (float)iso.lgColl_l_mixing[ipHE_LIKE];
		}
		else
			TotalInsanity();
	}

	/* this branch, n-same, l-changing collisions */
	else if( (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == iso.quant_desig[ipHE_LIKE][nelem][ipLo].n ) &&
		(iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == iso.quant_desig[ipHE_LIKE][nelem][ipLo].s ) )
	{
		*factor = 1.f;
		
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n <= iso.n_HighestResolved[ipHE_LIKE][nelem] );

		/* this branch, l changing by one, regardless of spin */
		if( abs(iso.quant_desig[ipHE_LIKE][nelem][ipHi].l - iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)== 1)
		{
			/* this includes spin changing */
			cs = (float)CS_l_mixing( nelem, ipLo, ipHi);

			/* add to existing collision strengths
			* this CS should be multiplied by the proton density,
			* but the code will use the electron density, 
			* so ratio of proton to electron densities precorrects for this. */
			cs *= (float)(dense.xIonDense[ipHYDROGEN][1]/phycon.EdenHCorr*iso.lgColl_l_mixing[ipHE_LIKE]);
		}
		else
		{
			/* l changes by more than 1, but same-n collision */
			cs = 0.f;
		}
		*where = "lmix  ";
	}

	/* this branch, n changing collisions for ions */
	else if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n != iso.quant_desig[ipHE_LIKE][nelem][ipLo].n )
	{
		long int nlo = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n;
		
		/* TODO Even spin-changing collisions are included here.  Is this okay?	*/

		/* for this routine 2s is 1, grnd is 0 */
		if( nlo == 1)
			nlo = 0;

		/* ionic n-changing collision */
		cs = (float)Hion_colldeexc_cs(iso.quant_desig[ipHE_LIKE][nelem][ipHi].n, nlo, nelem, ipHE_LIKE );
		cs *= iso.lgColl_excite[ipHE_LIKE];

		*where = "hydro";
	}
	else
	{
		/* what's left are deltaN=0, spin changing collisions.
		 * These have not been accounted for.	*/
		/* Make sure what comes here is what we think it is.	*/
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == iso.quant_desig[ipHE_LIKE][nelem][ipLo].n );
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].s != iso.quant_desig[ipHE_LIKE][nelem][ipLo].s );
		cs = 0.f;
		*where = "spin ";
	}

	ASSERT( cs >= 0.f );

#	ifdef DEBUG_FUN
	fputs( " <->IonCSInterp()\n", debug_fp );
#	endif

	return(cs);

}
