/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*prt_He_like_DeparCoef routine to print departure coefficients for he-like species */
/*prt_He_like_Pops routine to print level populations for he-like species */
/*HeRecom - do recomb coef for He, called by HeLike */
/*HeCollid evaluate collisional rates */
/*He_ParseAtom parse parameters off the helium command */
/*HeLevel level populations  */
/*HeCreate create he-like series, called by ContCreatePointers */
/*he_1trans compute Aul for given line	*/
/*AGN_He1_CS routine to punch table needed for AGN3 - collision strengths of HeI */
/*defect - calculate quantum defect. */
/*ritoa - converts the square of the radial integral for a transition 
 * (calculated by scqdri) to the transition probability, Aul.	*/
/*he_energy - calculates energy of a given level.	*/
/*ForbiddenAuls calculates transition probabilities for forbidden transitions.	*/
/*orbit - produces orbit designations for output.*/
/*printstuff - a few lines below, printflag determines what will be printed.*/
/*cross_section - calculates the photoionization cross_section for a given level and photon energy*/
/*radrecomb - calculates radiative recombination coefficients. */
/*RecomInt - Integral in milne relation.  Called by qg32.	*/
/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * nelem is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
/*HelikeCheckRecomb - called by SanityCheck to confirm that recombination coef are ok,
 * return value is relative error between new calculation of recom, and interp value */
/*ContinuumLowering - limit max prin. quan. no. due to continuum lowering processes	*/
/* Recomb_Seaton59 - find recombination for given n,using Seaton 59 approximation.
 * The following three are needed by Recomb_Seaton59:
 *     ExponentialInt
 *     X1Int
 *     X2Int	*/

#include "cddefines.h" 

/* various useful print statements occur throughout this file - search on PRINTIT */
/* 
  
  Energy order within 2 3P

  The order of the levels within the 2 3P level of atomic helium is opposite
  from the order within astrophysically abundant ions.  The indices below
  consistently point to the correct level, and the energies are correct,
  so the J levels within 2 3P are not in increasing energy order for helium itself. 
  This is ok since the atomic data is correct, and the difference in energies is so small.

*/

/* 
the following block of defines occurs in cddefines.c, and echoed in cddefines.h 
const int ipHe1s1S = 0;
const int ipHe2s3S = 1;
const int ipHe2s1S = 2;
const int ipHe2p3P0 = 3;
const int ipHe2p3P1 = 4;
const int ipHe2p3P2 = 5;
const int ipHe2p1P = 6;

level 3
const int ipHe3s3S = 7;
const int ipHe3s1S = 8;
const int ipHe3p3P = 9;
const int ipHe3d3D = 10;
const int ipHe3d1D = 11;
const int ipHe3p1P = 12;

level 4
const int ipHe4s3S = 13;
const int ipHe4s1S = 14;
const int ipHe4p3P = 15;
const int ipHe4d3D = 16;
const int ipHe4d1D = 17;
const int ipHe4f3F = 18;
const int ipHe4f1F = 19;
const int ipHe4p1P = 20;
*/
#include "physconst.h" 
#include "lines_service.h"
#include "tfidle.h"
#include "elementnames.h"
#include "hydro_vs_rates.h"
#include "taulines.h"
#include "linpack.h"
#include "hevmolec.h"
#include "receff.h"
#include "path.h"
#include "rfield.h"
#include "atomcwgt.h"
#include "negcon.h"
#include "typmatrx.h"
#include "physok.h"
#include "trace.h"
#include "heavy.h"
#include "dynamics.h"
#include "drfe.h"
#include "cfit.h"
#include "opacity.h"
#include "secondaries.h"
#include "chargtran.h"
#include "phycon.h"
#include "ionrec.h"
#include "dense.h"
#include "ionrange.h"
#include "hydro_bauman.h"
#include "hrfit.h"
#include "hydroeinsta.h"
#include "iso.h"
#include "helike.h"
#include "helike_cs.h"
#include "plasnu.h"
/*lint -e662 creation of  out of bound pointer */
/*lint -e661 creation of  out of bound pointer */

/*	The Elevels data type, and then the iso.quant_desig[ipHE_LIKE] structure,
 *  which contain the quantum numbers n,l, and s for a given
 *  energy level, are defined in iso.h, for use in multiple subroutines. */

/*typedef struct { 
	long n;
	long s;
	long l;
} Elevels; iso.h */ 

/* An array of structures each containing for a given element, n,l, and s
 * s=0 for singlets, s=1 for triplets
 * in helike.h
static Elevels **iso.quant_desig[ipHE_LIKE]; */

/* This array does the reverse of iso.quant_desig[ipHE_LIKE]
 * Since separate j levels within a triplet term are only resolved in the case of 2tripP,
 * allocating memory for a j dimension is unwarranted.  Instead 
 * QuantumNumbers2Index[nelem][2][1][1] will point to 2^3P2, with 2^3P0 and 2^3P1
 * easily accessed by subtracting 2 and 1 respectively from the returned index.	*/
static long ****QuantumNumbers2Index;

/* this becomes the most number of levels in any member of the he-like iso sequence */
static long int max_num_levels, max_n;

/* this will save log of radiative recombination rate coefficients at N_HE_TE_RECOMB temperatures.
 * there will be NumHeLevRecomb[nelem] levels saved in RRCoef[nelem][level][temp] */
static double ***RRCoef/*[LIMELM][NumHeLevRecomb[nelem]][N_HE_TE_RECOMB]*/;

static double **TotalRecomb;	/*[nelem][i]*/

/* the designation of the levels, chLevel[n][string] */
static char **chLevel;

/* the array of logs of temperatures at which RRCoef was defined */
static double TeRRCoef[N_HE_TE_RECOMB];

static double He_RRCoef_Te( long nelem , long n );

/* This computes that amount of the total recombination into an element (at a
 * specific temperature) that goes into levels having n greater than that otherwise
 * accounted for.	The reference is Seaton 59, main equation is 29. */
static double Recomb_Seaton59( long nelem, double temp, long n );

/* The three of these are used in the computation of Recomb_Seaton59	*/
static double ExponentialInt( double v );
static double X1Int( double u );
static double X2Int( double u );

/* compute quantum defect */
static double defect( long nelem, long ipLo );
 
static double ritoa( long li, long lf, long nelem, double k, double RI2 );

static double he_energy( double Eff_n, long nelem, long ipLo );

static double ForbiddenAuls( long ipHi, long ipLo, long nelem );

static void orbit( long index, long nelem );

static void printstuff(long flag);

static double cross_section(double EgammaRyd);
	
static double radrecomb(double temp, long nelem, long ipLo);

static double RecomInt(double EE);

/* compute energy diffference in wn and Aul for given line
 * return is 0 for success, 1 for failure */
static void he_1trans(  
			  /* charge on the C scale, 1 is helium */
			  long nelem , 
			  /* energy difference in wavenumber */
			  double Enerwn , 
			  /* transition probability */
			  double *Aul ,
			  /* upper and lower quantum numbers	*/
			  double Eff_nupper, long lHi, long sHi, long jHi,
			  double Eff_nlower, long lLo, long sLo, long jLo);

/* Function he_assign assigns, in order of increasing energy, 	*/
/* quantum numbers n,l, and s to each element of vector iso.quant_desig[ipHE_LIKE].	*/ 
static void he_assign( long nelem );

static double CalculateRadialIntegral( long nelem, long ipLo, long ipHi );

static double RadialIntegrand( double r12 );

static double WaveFunction( double r12 );

static double kTRyd,EthRyd,Xn_S59; 
static long ipLev,globalZ;
static long	RI_ipHi, RI_ipLo, RI_ipLev;
static void HeRecom( long nelem );
/* evaluate collisional rates */
static void HeCollid( long nelem);
/* evaluate photoionization rates  */
static void HeLevel( long nelem);
/*ContinuumLowering - limit max prin. quan. no. due to continuum lowering processes	*/
static void ContinuumLowering( long nelem );
/* Kill radiative decays that have a frequency below the plasma frequency.	*/
static double CheckIfBelowPlasFreq( long nelem, long ipHi, long ipLo );

static FILE *ofp;  

/* experimental energies, in wavenumbers, for atomic helium */
#define NHE1LEVELS 111
static double He1Energies[NHE1LEVELS] =
{0,		  159850.318,  166271.7,169082.185,169081.189,169081.111,171129.148,
183231.08, 184859.06,  185559.2,  186095.9, 186099.22, 186203.62, 190292.46,
 190934.5, 191211.42, 191438.83, 191440.71, 191446.61, 191447.24, 191486.96,
193341.33, 193657.78, 193795.07, 193911.48, 193912.54, 193915.79, 193914.31,
193916.26, 193916.26, 193936.76, 194930.46, 195109.17, 195187.21, 195254.37,
195255.02, 195256.82, 195256.7,  195257.26, 195257.26, 195257.26, 195257.26,
195269.17, 195862.63, 195973.19, 196021.72,    196064,  196064.3, 196065.51, 
 196065.4, 196065.85, 196065.85, 196065.85, 196065.85, 196065.85, 196065.85, 
196073.41, 196455.79, 196529.03, 196561.08, 196589.42, 196589.73, 196590.42, 
 196590.3, 196590.65, 196590.65, 196590.65, 196590.65, 196590.65, 196590.65, 
196590.65, 196590.65, 196595.56, 196856.37, 196907.13, 196929.68, 196949.63, 
196949.49, 196950.36, 196950.3,  196950.45, 196950.45, 196950.45, 196950.45, 
196950.45, 196950.45, 196950.45, 196950.45, 196950.45, 196950.45, 196953.95, 
197139.76, 197176.36, 197192.63, 197207.30, 197207.08,    197208,    197208,
197208,		  197208,	 197208,	197208,	   197208,    197208,	 197208,
197208,		  197208,	 197208,	197208,	   197208, 197210.41}; 
/* Last energy is 10^1P.	*/

/* First in this refuse pile is 11^3S.	*/
/* 
197347.05, 197386.98, 197397.75, 197397.62,  197398.6,
197400.18, 197503.69, 197524.26, 197534.44, 197542.54, 197542.67, 
197544.56, 197624.98, 197649.07, 197649.78, 197655.19, 197655.47, 197656.95, 
197721.13, 197739.9, 197744.918, 197744.94, 197746.15, 197796.63, 197813.11, 
197817.05, 197818.12, 197872.95, 197876.41, 197877.04, 197922.51, 197925.33, 
197925.87, 197964.02, 197966.75, 197966.8, 197999.12, 198001.43, 198001.44, 
198029.07, 198031.02, 198031.41, 198054.83, 198056.5, 198077.15};
*/

	/* Ionization potentials (in wavenumber) for each ion in the iso-seq up to Z=30.		*/
	/* These are exactly what you get if you take EionRYD below and multiply by RYD_INF.	*/
static double EionWN[29] =       
	{198307.729760353,610003.839889137,1241136.72201499,2091948.45665631,3162116.52584231,
	 4452446.95015668,5962133.81875305,7692790.05069734,9645221.44709864,11814589.7994457,
	 14209766.0528639,16822685.5022862,19661412.9625169,22717883.6187518,26000162.0663204,
	 29508248.5246975,33234078.1790787,37185715.7345311,41363161.0813172,45766414.4389118,
	 50395475.4781030,55258409.0136949,60339085.8550283,65653635.1927626,71202056.8074231,
	 76976286.4328920,82984388.3352872,89194104.5722390,95726403.3055320};
	
	/* Ionization potentials (in Rydbergs) for each ion in the iso-seq up to Z=30.	*/
	/* These are exactly what you get if you take EionEV below and divide by EVRYD.	*/
static double EionRYD[29] =       
	{1.807113,5.558764,11.310070,19.063237,28.815326,40.573682,54.330961,70.101861,
	87.893725,107.662464,129.488916,153.299590,179.167978,207.020588,236.930910,
	268.898946,302.851204,338.861175,376.928858,417.054255,459.237363,503.551674,
	549.850208,598.279945,648.840883,701.459535,756.209388,812.796486,872.323172};

	/* Ionization potentials (in eV) for each ion in the iso-seq up to Z=30. These are 
	 * exactly what you get if you take Verner's numbers in PH1COM.PH1[0][1][nelem][0] 
	 * and multiply by 0.9998787, exactly as is done elsewhere in Cloudy.	*/
double EionEV[29] =       
	{24.587017,75.630824,153.881326,259.368529,392.052444,552.033006,739.210311,
	953.784316,1195.854925,1464.822296,1761.786269,2085.746968,2437.704271,
	2816.658298,3223.608929,3658.556163,4120.500123,4610.440686,5128.377852,
	5674.311623,6248.241996,6851.168852,7481.092433,8140.012497,8827.929042,
	9543.842191,10288.751823,11058.658422,11868.560169};

static int HeFsIndex;

/* used as parameters in qg32 integration */
static double vJint , zJint; 

static double Jint( double theta )
{
	/*	[ cos[vx - z sin[x]] ]  */
	double 
		d0 = ( 1.0 / PI ),
		d1 = vJint * theta,
		d2 = zJint * sin(theta),
		d3 = (d1 - d2),
		d4 = cos(d3),
		d5 = (d0 * d4);

	return( d5 );
}

static double AngerJ( double vv, double zz )
{
	long int rep = 0, ddiv, divsor;

	double y = 0.0;

	/* Estimate number of peaks in integrand.  */                            
	/* Divide region of integration by number  */
	/*  peaks in region.                       */
	if( (fabs(vv)) - (int)(fabs(vv)) > 0.5 )
		ddiv = (int)(fabs(vv)) + 1;
	else 
		ddiv = (int)(fabs(vv));

	divsor  = ((ddiv == 0) ? 1 : ddiv);     
	vJint = vv;
	zJint = zz;

	for( rep = 0; rep < divsor; rep++ )
	{
		double
		rl = (((double) rep)/((double) divsor)),
		ru = (((double) (rep+1))/((double) divsor)),
		x_low = (PI * rl),
		x_up  = (PI * ru);      

		y += qg32( x_low, x_up, Jint );
	}

	return( y );
}

/******************************************************************************/
/******************************************************************************/
/*                                                                            */
/*    Semi-Classical Quantum Defect Radial Integral                           */      
/*                                                                            */
/*   See for example                                                          */
/*     Atomic, Molecular & Optical Physics Handbook                           */
/*     Gordon W. F. Drake; Editor                                             */
/*     AIP Press                                                              */
/*     Woddbury, New York.                                                    */
/*     1996                                                                   */
/*                                                                            */
/* NOTE:: we do not include the Bohr Radius a_o in the                        */
/*           definition of of  R(n,L;n'L') as per Drake.                      */
/*                                                                            */
/*                                                                            */
/*                   1  (n_c)^2 | {      D_l max(L,L') }                      */
/*    R(n,L;n'L') = --- ------- | { 1 -  ------------- } J( D_n-1 ; -x )  -   */
/*                   Z   2 D_n  | {          n_c       }                      */
/*                                                                            */
/*                                                                            */
/*                    {      D_L max(L,L') }                                  */
/*                -   { 1 +  ------------- } J( D_n+1 ; -x )                  */
/*                    {          n_c       }                                  */
/*                                                                            */
/*                                                                            */
/*                     2                    |                                 */
/*                  + ---  sin(Pi D_n)(1-e) |                                 */
/*                     Pi                   |                                 */
/*                                                                            */
/*  where                                                                     */
/*        n_c = (2n*'n*)/(n*'+n*)                                             */
/*                                                                            */
/*       Here is the quantity Drake gives...                                  */
/*            n_c = ( 2.0 * nstar * npstar ) / ( nstar + npstar ) ;           */
/*                                                                            */
/*       while V.A. Davidkin uses                                             */
/*            n_c = sqrt(  nstar * npstar  );                                 */
/*                                                                            */
/*        D_n = n*' - n*                                                      */
/*                                                                            */      
/*        D_L = L*' - L*                                                      */
/*                                                                            */
/*        x = e D_n                                                           */
/*                                                                            */
/*        Lmx  = max(L',L)                                                    */
/*                                                                            */
/*        e = sqrt( 1 - {Lmx/n_c}^2 )                                         */
/*                                                                            */
/*                                                                            */
/*       Here n* = n - qd where qd is the quantum defect                      */
/*                                                                            */
/******************************************************************************/
/******************************************************************************/
double scqdri(/* upper and lower quantum numbers...n's are effective	*/
			  double nstar, long int l,
			  double npstar, long int lp,
              double iz
              )
{
	double n_c = ((2.0 * nstar * npstar ) / ( nstar + npstar ));
	double D_n = (nstar - npstar);
	double D_l = (double) ( l - lp );
	double lg  = (double) ( (lp > l) ? lp : l);

	double h = (lg/n_c);
	double g = h*h;
	double f = ( 1.0 - g );
	double e = (( f >= 0.0) ? sqrt( f ) : 0.0 );

	double x  = (e * D_n);
	double z  = (-1.0 * x);
	double v1 = (D_n + 1.0);
	double v2 = (D_n - 1.0); 

	double d1,d2,d7,d8,d9,d34,d56,d6_1; 

	if ( iz == 0.0 )
		iz += 1.0;

	if ( D_n == 0.0 )
	{
	  return( -1.0 );
	}

	if ( D_n < 0.0 )
	{
	  return( -1.0 );
	}

	if ( f < 0.0 )
	{
	  /* This can happen for certain  quantum defects   */
	  /* in the lower n=1:l=0 state. In which case you  */
	  /* probably should be using some other alogrithm  */
	  /* or theory to calculate the dipole moment.      */
	  return( -1.0 );
	}

	d1 = ( 1.0 / iz );

	d2 = (n_c * n_c)/(2.0 * D_n);

	d34 = (1.0 - ((D_l * lg)/n_c)) * AngerJ( v1, z );

	d56 = (1.0 + ((D_l * lg)/n_c)) * AngerJ( v2, z );

	d6_1 = PI * D_n;

	d7 = (2./PI) * sin( d6_1 ) * (1.0 - e);

	d8 = d1 * d2 * ( (d34) - (d56) + d7 );

	d9 = d8 * d8;

	ASSERT( D_n  > 0.0 );
	ASSERT( l  >= 0  );
	ASSERT( lp >= 0 );
	ASSERT( (l == lp + 1) || ( l == lp - 1) );
	ASSERT( n_c != 0.0 );
	ASSERT( f >= 0.0 );
	ASSERT( d9  > 0.0 );

	return( d9 );
}

/*He-like main routine to call HeLevel and determine model he-like species atom level balance,
 * called by ionize */
void HeLike(void)
{
	int lowsav , ihisav;
	
	long int ipLo, ipHi, nelem;

#	ifdef DEBUG_FUN
	fputs( "<+>HeLike()\n", debug_fp );
#	endif

	/* save state of he-like low and high ionziation, since must not change here,
	 * since there is a parallel helium atom that must decrement he */
	/*TODO remove this when this routine really controls helium itself */
	lowsav = IonRange.IonLow[ipHELIUM];
	ihisav = IonRange.IonHigh[ipHELIUM];
	for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* do not consider elements that have been turned off */
		if( dense.lgElmtOn[nelem] )
		{
			/* note that nelem scale is totally on c not physical scale, so 0 is h */
			/* evaluate helium-like balance if ionization reaches this high */
			if( (IonRange.IonHigh[nelem] >= nelem)  )
			{
				ContinuumLowering( nelem );
			
				/* evaluate recombination rates */
				HeRecom(nelem);

				/* evaluate collisional rates */
				HeCollid(nelem);

				/* >>chng 02 jan 18, move to unified photo rate routine */
				/* evaluate photoionization rates */
				iso_photo(ipHELIUM , nelem );

				/* solve for the level populations */
 				HeLevel(nelem);

			}
			else
			{
				iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 0.;
				iso.xIonSimple[ipHE_LIKE][nelem] = 0.;

				/* zero it out since no population*/
				iso.Pop2Ion[ipHE_LIKE][nelem][0] = 0.;
				for( ipHi=ipHe2s3S; ipHi < iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
				{
					iso.Pop2Ion[ipHE_LIKE][nelem][ipHi] = 0.;
					for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
					{
						/* population of lower and upper levels rel to ion */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopLo = 0.;
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi =  0.;

						/* population of lower level rel to ion, corrected for stim em */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc =  0.;
					}
				}
			}
		}
	}
	IonRange.IonLow[ipHELIUM] = lowsav;
	IonRange.IonHigh[ipHELIUM] = ihisav;

	return;
}

/*defect - calculate quantum defect for a given level and nuclear charge. */
static double defect(long nelem, long ipLo)  
{  
	/* The quantum defect, and parameters a,b, and c	*/
	double qd,a,b,c;	

	/* Parameters for fits to quantum defects for	*/
	/* P triplet and S orbitals. The dimensions are	*/
	/*	first: l									*/
	/*	second: n									*/
	/* 	third: parameters a,b,and c.				*/
	double param[3][4][3]=  
	{ 
		{{0.6451941,0.3119437,-1.2722842},		/* ^3S	*/
			{0.7664874,0.3455675,-1.3976462}, 
			{0.8247101,0.3603131,-1.4520500}, 
			{0.8878402,0.3714450,-1.4995732}}, 

		{{1.4203514,0.5311096,-2.6728087},		/* ^1S	*/
			{1.5733513,0.5997339,-2.9253834}, 
			{1.4531025,0.5924751,-2.8662756}, 
			{1.6038999,0.6342552,-3.0298071}}, 

		{{-2.2323488,0.0890840,-0.5166053},		/* ^3P	*/
			{-2.0463691,0.1222081,-0.6672983}, 
			{-1.9904104,0.1328918,-0.7150879}, 
			{-1.9500974,0.1452111,-0.7649031}} 
	};   

	/* Because they cannot be fit to a funtion of the same form as the other orbitals,
	 * the P singlets are fit to a different func, with these parameters and dimensions */
	/*	first: n									*/
	/*	second: parameters a and b.					*/
	double P1[4][2]=
	{
		{-56.65245,-3.661923},
		{-52.03411,-4.941075},
		{-50.43744,-5.525750},
		{-49.45137,-5.908615}
	};

	/* The P singlets of He do not conform to the above, 
	 * so the defects themselves are listed, for n = 2,3,4, and 5.	*/
	double P1He[4]=
	{-0.009492,-0.011351,-0.011904,-0.01218};

	long n = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n;
	long lqn = iso.quant_desig[ipHE_LIKE][nelem][ipLo].l;
	long s = iso.quant_desig[ipHE_LIKE][nelem][ipLo].s;
	
	ASSERT(n >= 1L);
	ASSERT(n > lqn);
	/* Only Helium and up, and only those turned on.	*/
	ASSERT((nelem >= ipHELIUM) && (nelem <= LIMELM));       

	if ( ipLo == ipHe1s1S )
	{
		/* Quantum defects for ground state are found from the rydberg
		 * equation, and the ionization potential of the ion.	*/
		ASSERT(nelem>ipHYDROGEN );
		qd = 1.0 - nelem * sqrt(1/EionRYD[nelem-1]);
		return qd;  
	}
	else
	{
		/* For levels with n > 5, the quantum defect	*/
		/* is approximately the same as if n equaled 5.	*/
		if ( n > 5L )
		{
			n = 5L;
		}
		/* For P singlets	*/
		if ( lqn==1L && s==0L )
		{
			if ( nelem == ipHELIUM )
				qd = P1He[n-2];
			else
				qd = 1./(P1[n-2][0] + P1[n-2][1] * (nelem+1) * log((double)nelem+1.) );
			return qd;  
		}
		/* Defects for orbitals with l>2 are approximately equal to zero.	*/
		else if ( lqn < 2L )
		{
			a = param[2*(lqn+1)-s-1][n-2][0];  
			b = param[2*(lqn+1)-s-1][n-2][1];  
			c = param[2*(lqn+1)-s-1][n-2][2];  
			qd = exp((a+c*(nelem+1))/(1.0+b*(nelem+1)));  
			return qd;  
		}
		/* This is d...somewhat arbitrary.	*/
		else if ( lqn == 2L )
		{
			return 0.002 / (double)nelem; 
		}
		/* even more arbitrary, but will allow some f-d and g-f transitions
		 * to have quasi-real energies.  This should affect which decays are zeroed
		 * due to being below the plasma frequency.	*/
		else if ( lqn == 3L )
		{
			return 0.001 / (double)nelem;
		}
		else if ( lqn > 3L )
		{
			return 0.;
		}
		else
		{
			cdEXIT(9);  
		}  
		/* it is impossible to reach this point */
	}   
}

/*he_energy calculates energy of a given level.	*/
static double he_energy(double Eff_n,long nelem, long ipLo )
{
	double Ef,gap2tripP;
	
	/* for atomic helium use experimental energies */
	if( nelem == ipHELIUM && ipLo < NHE1LEVELS )
	{
		Ef = EionWN[ipHELIUM-1] - He1Energies[ipLo];
	} 
	else
	/* Calculate energy difference for all other trans.	*/
	{
		Ef = RYD_INF * (nelem/Eff_n) * (nelem/Eff_n);  
	}
	
	if (nelem > 6)
	{
	/* gap2tripP represents the amount to add or subtract from the median 
	 * 2p3P term in order to find energies of individual levels of the term.	
	 * There's an unexpected complication here.  According to data from Porquet and Dubau 
	 * (2000), the separate J levels of the term are NOT well ordered along the 
	 * sequence.  In CV, for example, the J=1 term is closest to ground, while 
	 * in FeXXV, the order is J=0, J=1, J=2.  And HeI has the exact opposite, 
	 * J=2 is closest to ground (name source of He energies).  Our solution is 
	 * to treat the energies for the three levels as degenerate for Z=3..7(?),
	 * since according to Porquet and Dubau, the levels don't become well-
	 * ordered until around Z=8 (nelem=7). */
		gap2tripP = 0.007*pow((double)nelem+1., 4.5);
		if ( ipLo == ipHe2p3P0 )
		{
			Ef += (3*gap2tripP);
		}
		else if ( ipLo == ipHe2p3P1 )
		{
			Ef += (2*gap2tripP);
		}
		else if ( ipLo == ipHe2p3P2 )
		{
			Ef -= (5*gap2tripP);
		}
	}

	ASSERT(Ef > 0.);
	return Ef;
}

static double ForbiddenAuls( long ipHi, long ipLo, long nelem )
{
	double A;
	/* >>refer	Helike	2pho	Derevianko, A., & Johnson, W.R. 1997, Phys. Rev. A 56, 1288
	 * numbers are not explicitly given in this paper for Z=21-24,26,27,and 29.
	 * So numbers given here are interpolated.	*/
	double As2nuFrom1S[28] = {1940.,1.82E+04,9.21E+04,3.30E+05,9.44E+05,2.31E+06,5.03E+06,1.00E+07,
		1.86E+07,3.25E+07,5.42E+07,8.69E+07,1.34E+08,2.02E+08,2.96E+08,4.23E+08,5.93E+08,8.16E+08,
		1.08E+09,1.43E+09,1.88E+09,2.43E+09,3.25E+09,3.95E+09,4.96E+09,6.52E+09,7.62E+09,9.94E+09};
	/* Important clarification, according to Derevianko & Johnson (see ref above), 2^3S can decay
	 * to ground in one of two ways: through a two-photon process, or through a single-photon M1 decay,
	 * but the M1 rates are about 10^4 greater that the two-photon decays throughout the entire
	 * sequence.  Thus these numbers, are much weaker than the effective decay rate, but should probably
	 * be treated in as a two-photon decay at some point	
	double As2nuFrom3S[28] = {1.25E-06,5.53E-05,8.93E-04,8.05E-03,4.95E-02,2.33E-01,8.94E-01,2.95E+00,
		8.59E+00,2.26E+01,5.49E+01,1.24E+02,2.64E+02,5.33E+02,1.03E+03,1.91E+03,3.41E+03,5.91E+03,
		9.20E+03,1.50E+04,2.39E+04,3.72E+04,6.27E+04,8.57E+04,1.27E+05,2.04E+05,2.66E+05,4.17E+05};*/
				
	if ( (ipLo == ipHe1s1S) && (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == 2) )
	{
		if( nelem == ipHELIUM )
		{
			/* All of these but the second and third one (values 51.02 and 1E-20) are from
			 * >>refer	HeI	As	Lach, G, & Pachucki, K, 2001, Phys. Rev. A 64, 042510
			,* 1E-20 is made up
			 * 51.3 is from the Derevianko & Johnson paper cited above.	*/
			double ForbiddenHe[5] = { 1.272E-4,	51.02,	1E-20,	177.58,	0.327 };

			return ForbiddenHe[ipHi - 1];
		}
		else
		{
			switch ( (int)ipHi )
			{
			case 1: /* Parameters for 2^3S to ground transition.	*/
				/* >>refer	Helike	As	Lin, C.D., Johnson, W.R., and Dalgarno, A. 1977, 
				 * >>refercon	Phys. Rev. A 15, 1, 010015	*/
				return (3.9061E-7) * pow( (double)nelem+1., 10.419 );
			case 2: /* Parameters for 2^1S to ground transition.	*/
				return As2nuFrom1S[nelem-2];
			case 3: /* Parameters for 2^3P0 to ground transition.	*/
				return iso.SmallA;
			case 4: /* Parameters for 2^3P1 to ground transition.	*/
				return ( 11.431 * pow((double)nelem, 9.091) );
			case 5: /* Parameters for 2^3P2 to ground transition.	*/
				/* According to the trend in Porquet & Dubau (2000), these are about 
				 * 1E-3 times the rate of the 2^3P1 to ground transition.	*/	
				return ( 0.011431 * pow((double)nelem, 9.091) );
			default:
				TotalInsanity();
			}
		}
	}
	
	/* The next two cases are fits to probabilities given in 
	 * >>refer	He-like	As	Johnson, W.R., Savukov, I.M., Safronova, U.I., & 
	 * >>refercon	Dalgarno, A., 2002, ApJS 141, 543J	*/
	/* originally astro.ph. 0201454 */
	/* The involve Triplet P and Singlet S.  Rates for Triplet S to Singlet P 
	 * do not seem to be available.	*/
	
	/* Triplet P to Singlet S...Delta n not equal to zero!	*/
	else if( nelem > ipHELIUM && iso.quant_desig[ipHE_LIKE][nelem][ipHi].l == 1 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == 1 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].l == 0 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].s == 0 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].n < iso.quant_desig[ipHE_LIKE][nelem][ipHi].n )
	{
		A = 8.0E-3 * exp(9.283/sqrt((double)iso.quant_desig[ipHE_LIKE][nelem][ipLo].n)) 
			* pow((double)nelem,9.091) / pow((double)iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,2.877);
	}
	
	/* Singlet S to Triplet P...Delta n not equal to zero!	*/
	else if( nelem > ipHELIUM && iso.quant_desig[ipHE_LIKE][nelem][ipHi].l == 0 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == 0 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].l == 1 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].s == 1 
		&& iso.quant_desig[ipHE_LIKE][nelem][ipLo].n < iso.quant_desig[ipHE_LIKE][nelem][ipHi].n )
	{
		A = 2.416 * exp(-0.256 * iso.quant_desig[ipHE_LIKE][nelem][ipLo].n) * pow((double)nelem,9.159) 
			/ pow((double)iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,3.111);
		
		if ( ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P2) ) )
		{
			/* This is divided by 3 instead of 9, because value calculated is specifically for 2^3P1.
			 * Here we assume statistical population of the other two.	*/
			A *= (2.*(ipLo-3)+1.0)/3.0;
		}
	}
	
	else if( ( ipLo == ipHe2s3S ) && ( ipHi == ipHe2p1P ) )
	{
		/* This transition,1.549 , given by Lach and Pachucki, 2001 for the atom 
		 * assume Z^4 dependence for the ions...no sources.	*/
		A = 1.549/16.*pow((double)nelem+1., 4.);
	}

	else
	{
		/* Current transition is not supported.	*/
		A = iso.SmallA;
	}

	ASSERT( A > 0.);
	return A;   
}


/*orbit generates labels for each level and is only necessary for output */
static void orbit(long index, long nelem)
{
	char *c = "     ";

	/*  A string containing the orbit name is output.		*/

	if ( index < (iso.numLevels[ipHE_LIKE][nelem] - iso.nCollapsed[ipHE_LIKE][nelem]) )
	{
		switch ( (int)iso.quant_desig[ipHE_LIKE][nelem][index].l )
		{
		case 0:
			c = "S";
			break;
		case 1:
			c = "P";
			break;
		case 2:
			c = "D";
			break;
		default:
			fprintf(ofp,"%-2c ",67+(int)iso.quant_desig[ipHE_LIKE][nelem][index].l);
			return;
		}
		fprintf(ofp,"%-1.1s",c);
		if ( (index == 3) || (index == 4) || (index == 5) )
		{
			fprintf(ofp,"%1ld ",(index-3));
		}
		else
		{
			fprintf(ofp,"  ");
		}
	}
	return;
}

static double CalculateRadialIntegral( long nelem, long ipLo, long ipHi )
{
	double RadInt = 0.;

	RI_ipHi = ipHi;
	RI_ipLo = ipLo;

	/* Z is needed also, but cannot be passed to integrand directly.	*/
	globalZ = nelem;
	
	if( ( nelem == ipHELIUM) )
	{
		/* Evaluate radial integral	out to 10 Bohr radii.	*/
		RadInt = qg32( 0., 0.2, RadialIntegrand );
		RadInt += qg32( 0.2, 0.4, RadialIntegrand );
		RadInt += qg32( 0.4, 0.6, RadialIntegrand );
		RadInt += qg32( 0.8, 1.0, RadialIntegrand );
		RadInt += qg32( 1.0, 2.0, RadialIntegrand );
		RadInt += qg32( 2.0, 4.0, RadialIntegrand );
		RadInt += qg32( 4.0, 10., RadialIntegrand );
	}
	else
		fprintf(ioQQQ, "No can do in CalculateRadialIntegral, nelem %li, iplo %li, ipHi %li\n",
			nelem, ipLo, ipHi);
	
	/* mulitply by normalization constants */
	RadInt *= 1.;

	return RadInt;
}

static double RadialIntegrand( double r12 )
{
	double result;

	RI_ipLev = RI_ipHi;

	result = r12 * WaveFunction( r12 ); 
	
	RI_ipLev = RI_ipLo;

	result *= WaveFunction( r12 );

	return result;
}

static double WaveFunction( double r12 )
{
	double value,PsiSlow2,alphaOfR2, PsiFastOneTwo;
	long Z = globalZ;
	/* use classical radius of electron for R1.	*/
	/* Should it be multiplied by Z^2?	*/
	double R1 = POW2((double)Z) * 5.3E-5;
	double R2 = r12 + R1;
	double a=0.,b=0.,mu=0.,nu=0.,Beta=0.;
	long l = iso.quant_desig[ipHE_LIKE][Z][RI_ipLev].l;
	/* long s = iso.quant_desig[ipHE_LIKE][Z][RI_ipLev].s;	*/


	/*  This wavefunction from Koga and Matsuhashi 89.
	 *  It is a product of two wavefunctions...one for the 1s ground electron,
	 *  and the other for the excited state.	*/

	/* Excited function	*/
	if( l == 0 )
	{
		Beta = 0.411;
		PsiSlow2 = sqrt( POW3(Beta) / PI ) * exp( -1. * Beta * R2 );
		/* parameters for alphaOfR2, for S orbitals.	*/
		a = -0.1;
		b = 0.14;
		mu = 1.34;
		nu = 2.86;
	}
	else if( l == 1 )
	{
		Beta = 0.5;
		/* The leading number is sqrt( POW3(2/sqrt(3)) / PI )	*/
		PsiSlow2 = 1.1547 * pow( Beta, 2.5 ) * R2 * exp( -1. * Beta * R2 );
		/* parameters for alphaOfR2, for P orbitals.	*/
		a = -1;
		b = 0.2;
		mu = 0.1;
		nu = 4.0;
	}
	else
	{
		/* for now only do S and P	*/
		PsiSlow2 = 0.;
	}

	alphaOfR2 = 2 - (1. + a * pow( R2, mu ) )/ ( 1. + b * pow( R2, nu ) );
	
	/* Ground function	*/
	PsiFastOneTwo = sqrt( POW3(alphaOfR2) / PI ) * exp( -1. * alphaOfR2 * R1 );

	value = PsiFastOneTwo * PsiSlow2;

	return value;
}

/* Calculates Einstein A for a given transition.	*/
static void he_1trans( 
			   /* charge on c scale, Energy is wavenumbers, Einstein A	*/
			   long nelem , double Enerwn , double *Aul ,
			   /* quantum numbers of upper level:	*/
			   double Eff_nupper, long lHi, long sHi, long jHi,
			   /* and of lower level: */
			   double Eff_nlower, long lLo, long sLo, long jLo )
			   /* Note j is only necessary for 2 triplet P...for all other n,l,s,
			    * j is completely ignored.	*/
{
	/* this will be A returned for totally forbidden transition 
#	define SMALLA 1e-20 - 
	replaced with iso.SmallA */

	double RI2, MyRI, HeF, RatioEHetoEHyd;
	long nHi, nLo, ipHi, ipLo, DoHydrogenic = FALSE;

	/* >>refer He	As	Kono, A., and Hattori, S. 1984, Phys Rev A, 29, 2981	*/
	/*	The dimensions are:[nHi - 2][ipHi - 1 - (nHi-1)*nHi][index],
		where index is the order of the lower level for all allowed
		transitions from the upper level.	*/
	double KonoHattoriF[8][6][16]	=	{

	/* nupper = 2	*/
		{{0.539086,	0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.539086,	0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.539086,	0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.27616,	0.37644	,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},
		{0,			0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.27616,	0.37644	,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	}},
	/* nupper = 3	*/
		{{0.02317,	0.06951	,	0.11585	,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.14547,	0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.064461,	0.8909	,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},		
		{0.067802,	0.203407,	0.33901	,	0.1121	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.71016,	0		,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},					
		{0.07344,	0.15134	,	0.6262	,	0.02114	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	}},		
	/* nupper = 4	*/
		{{0.003524,	0.01057	,	0.017622,	0.4357	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.02587,	0.3075	,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},			
		{0.025769,	0.05008	,	0.037	,	1.2153	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},		
		{0.01365,	0.04095	,	0.06825	,	0.4476	,	0.2009	,	0		,	0		,	0		,	0,	
					0		,	0		,	0		,	0		,	0		,	0		,	0	},		
		{0.12026,	0.6481	,	0		,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.029863,	0.04916	,	0.1439	,	0.01531	,	0.858	,	0.0401	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	}},
	/* nupper = 5	*/
		{{0.00126,	0.00378,	0.0063	,	0.06761	,	0.668	,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},			
		{0.009652,	0.05548	,	0.476	,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.012491,	0.02292	,	0.0069	,	0.04423	,	0.0883	,	1.531	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},	
		{0.0052233,	0.01567	,	0.02612	,	0.1246	,	0.439	,	0.2801	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.04326,	0.1413	,	0.648	,	0		,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.015039,	0.02234	,	0.0505	,	0.00311	,	0.1463	,	0.0393	,	1.083	,	0.0573	,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	}},
	/* nupper = 6	*/
		{{0.000609,	0.001829,	0.003049,	0.0247	,	0.104	,	0.903	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.004769,	0.02111	,	0.0862	,	0.647	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},											
		{0.006982,	0.01199	,	0.00258	,	0.02163	,	0.017	,	0.0415	,	0.147	,	1.842	,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},						
		{0.002608,	0.007823,	0.013039,	0.05301	,	0.124	,	0.43	,	0.354	,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.02095 ,	0.05629	,	0.1529	,	0.67	,	0		,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.008628,	0.01213	,	0.02419	,	0.001188,	0.0528	,	0.00838	,	0.1526	,	0.0684	,	1.305,
					0.074	,	0		,	0		,	0		,	0		,	0		,	0	}},				
	/* nupper = 7	*/
		{{0.000346,	0.001038,	0.001729,	0.01221	,	0.0382	,	0.1403	,	1.139	,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.002739,	0.01064	,	0.03305	,	0.1172	,	0.819	,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.004299,	0.00707	,	0.001284,	0.01178	,	0.0065	,	0.02111	,	0.0289	,	0.0403	,	0.21,
					2.151	,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.001507,	0.00452 ,	0.007533,	0.02816	,	0.0552	,	0.1254	,	0.434	,	0.426	,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.0119  ,	0.0289	,	0.0636	,	0.1632	,	0.703	,	0		,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.005405,	0.00736	,	0.01368	,	0.000594,	0.02588	,	0.00327	,	0.0556	,	0.01501	,	0.1609,
					0.1008	,	1.53	,	0.089	,	0		,	0		,	0		,	0	}},		
	/* nupper = 8	*/
		{{0.000217,	0.000650,	0.001083,	0.00707	,	0.0191	,	0.0517	,	0.176	,	1.376	,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.00173 ,	0.00623	,	0.01681	,	0.0451	,	0.1482	,	0.993	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.002836,	0.00454	,	0.000743,	0.00716	,	0.00327	,	0.01174	,	0.01116	,	0.021	,	0.042,
					0.0399	,	0.276	,	2.458	,	0		,	0		,	0		,	0	},			
		{0.000956,	0.002868,	0.004779,	0.01698	,	0.0303	,	0.0571	,	0.128	,	0.445	,	0.496,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.007464,	0.01708	,	0.0336	,	0.0693	,	0.1732	,	0.741	,	0		,	0		,	0,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.00361 ,	0.004813,	0.00857	,	0.000345,	0.01495	,	0.001664,	0.0276	,	0.00595	,	0.0588,
					0.0226	,	0.1705	,	0.135	,	1.745	,	0.105	,	0		,	0	}},	
	/* nupper = 9	*/
		{{0.000145,	0.000435,	0.000726,	0.00451	,	0.01117	,	0.026	,	0.065	,	0.212	,	1.613	,
					0,			0		,	0		,	0		,	0		,	0		,	0	},								
		{0.001167,	0.003999,	0.00994	,	0.02305	,	0.0571	,	0.1793	,	1.166	,	0		,	0		,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.00197 ,	0.00309	,	0.000474,	0.00471	,	0.001919,	0.00727	,	0.00567	,	0.0118	,	0.01627	,
					0.0212	,	0.0555	,	0.04	,	0.343	,	2.764	,	0		,	0	},
		{0.000647,	0.001941,	0.003235,	0.01112	,	0.01871	,	0.0318	,	0.0589	,	0.132	,	0.46	,
					0.564	,	0		,	0		,	0		,	0		,	0		,	0	},							
		{0.005015,	0.01104	,	0.02034	,	0.0373	,	0.0745	,	0.1837	,	0.784	,	0		,	0		,
					0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.00253 ,	0.003325,	0.00576	,	0.00022	,	0.00954	,	0.000979,	0.01613	,	0.00306	,	0.0293	,
					0.00903	,	0.0623	,	0.0307	,	0.181	,	0.171	,	1.963	,	0.12}}};
	
	ASSERT(nelem > ipHYDROGEN);

	/* Since 0.4 is bigger than any defect, adding that to the effective principle quantum number,
	 * and truncating to an integer will produce the principal quantum number.	*/
	nHi = (int)(Eff_nupper + 0.4);
	nLo = (int)(Eff_nlower + 0.4);
	
	/* Make sure this worked correctly.	*/
	ASSERT( fabs(Eff_nupper-(double)nHi) < 0.4 );
	ASSERT( fabs(Eff_nlower-(double)nLo) < 0.4 );

	ipHi = QuantumNumbers2Index[nelem][nHi][lHi][sHi];
	if( (nHi==2) && (lHi==1) && (sHi==1) )
	{
		ASSERT( (jHi>=0) && (jHi<=2) );
		ipHi -= (2 - jHi);
	}

	ipLo = QuantumNumbers2Index[nelem][nLo][lLo][sLo];
	if( (nLo==2) && (lLo==1) && (sLo==1) )
	{
		ASSERT( (jLo>=0) && (jLo<=2) );
		ipLo -= (2 - jLo);
	}

	ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == nHi );
	if( nHi <= iso.n_HighestResolved[ipHE_LIKE][nelem] )
	{
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].l == lHi );
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == sHi );
	}
	ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipLo].n == nLo );
	if( nLo <= iso.n_HighestResolved[ipHE_LIKE][nelem] )
	{
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipLo].l == lLo );
		ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ipLo].s == sLo );
	}

	/* First do allowed transitions	*/
	if ( (sHi == sLo) && (abs((int)(lHi - lLo)) == 1) )
	{
		/* For clarity, let's do this in two separate chunks...one for helium, one for everything else.	*/
		if ( nelem == ipHELIUM )
		{
			{
				/* Experiment with calculating radial integrals myself.	*/
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
		
				if( ( DEBUG_LOC ) && (nLo<=2) && (nHi==2) )
				{
					MyRI = CalculateRadialIntegral( ipHELIUM, ipLo, ipHi );
					fprintf(ioQQQ,"ipHi\t%li\tipLo\t%li\tMyRI^2\t%.2e\n",
						ipHi,
						ipLo,
						MyRI*MyRI);
				}
			}

			/* Retrieve Kono & Hattori (1984) oscillator strengths for Helium.	*/
			/* Above table includes all allowed s,p, and d transitions with nu <= 9.	*/
			if ( (nHi < 10) && (lHi < 3) && (lLo < 3) && (HeFsIndex < 16) 
				&& (iso.quant_desig[ipHE_LIKE][ipHELIUM][ipHi].n <= iso.n_HighestResolved[ipHE_LIKE][ipHELIUM]) )
			{
				/*Must not be accessed by collapsed levels!	*/
				ASSERT( ipHi < ( iso.numLevels[ipHE_LIKE][ipHELIUM] - iso.nCollapsed[ipHE_LIKE][ipHELIUM] ) );
				ASSERT( ipLo < ( iso.numLevels[ipHE_LIKE][ipHELIUM] - iso.nCollapsed[ipHE_LIKE][ipHELIUM] ) );
				ASSERT( ipHi > 2 );

				if ( (lHi == 1) && (sHi == 0) )
					/* Special second dimension index for singlet P.	*/
					HeF = KonoHattoriF[nHi - 2][5][HeFsIndex];
				else
					HeF = KonoHattoriF[nHi - 2][ipHi-1-(nHi-1)*nHi][HeFsIndex];
				
				*Aul = 0.66702 * HeF * Enerwn * Enerwn;
				
				/* Oscillator strengths are already weighted correctly for delta l = +1.
				 * Must fix for delta l = -1.	*/
				if ( lHi > lLo )
					*Aul *= ( (2. * lLo + 1.) / (2. * lHi + 1.) );
				
				HeFsIndex++;
				ASSERT( *Aul > 0. );
			}

			/* same-n transitions that are not yet covered.	*/
			else if ( nLo == nHi )
			{
				/* This case should only be entered if not done by Kono and Hattori numbers.	*/
				ASSERT( (lLo > 2) || (lHi > 2) || (nHi > 9) || (nHi > iso.n_HighestResolved[ipHE_LIKE][ipHELIUM]) );

				/* Triplet P to triplet S, delta n = 0	*/
				if ( (lHi == 1) && (sHi == 1) && (lLo == 0) && (sLo == 1))
				{
					*Aul = 0.4 * 3.85E8 /pow((double)nHi,5.328);
				}
				/* Singlet P to singlet D, delta n = 0	*/
				else if ( (lHi == 1) && (sHi == 0) && (lLo == 2) && (sLo == 0))
				{
					*Aul = 1.95E4 / pow((double)nHi, 4.269);
				}
				/* Singlet P to singlet S, delta n = 0	*/
				else if ( (lHi == 1) && (sHi == 0) && (lLo == 0) )
				{
					*Aul = 6.646E7 / pow((double)nHi, 5.077);
				}
				else /*if ( (lHi == 2) && (sHi == 1)  && (lLo == 1) ) */
				{
					/* Triplet D to Triplet P	*/
					*Aul = 3.9E6 / pow((double)nHi, 4.9);
					/* Transitions involving some higher l, these don't much
					 * matter, so just make up something simple	*/
					if ( lHi>2 )
						*Aul *= (2./lHi);
				}
				ASSERT( *Aul > 0.);
			}

			else if ( DoHydrogenic )
			{
				ASSERT(nHi > nLo);

				RatioEHetoEHyd = POW2( ((double)nHi *(double)nLo)/(Eff_nupper * Eff_nlower) ) 
						* ( POW2(Eff_nupper) - POW2(Eff_nlower) )
						/ ( POW2((double)nHi) - POW2((double)nLo) );
					
				ASSERT( RatioEHetoEHyd>0. );
			
				/*lint -e790 integral to float */
				*Aul = H_Einstein_A(nHi ,lHi , nLo , lLo , ipHELIUM);
				/*lint +e790 integral to float */
				
				*Aul *= POW2( RatioEHetoEHyd );

				ASSERT( *Aul > 0.);
			}

			else
			{
				/* The following three cases are of great importance, but the below radial integral 
				 * routine fails to achieve desirable accuracy, so these are fits as produced 
				 * from He A's for nupper through 9.  They are transitions to ground;  
				 * 2 and 3 singlet S; and 2, 3, and 4 triplet S.	*/
				
				/* Here are the Lyman transitions.	*/
				if ( ipLo == ipHe1s1S )
				{
					ASSERT( (lHi == 1) && (sHi == 0) );

					/* this fit calculated from transitions with upper level 4^1P through 9^1P,
					 * from Kono and Hattori (1984).	*/
					*Aul = (1.508e10) / pow((double)nHi,2.975);
					ASSERT( *Aul > 0.);
				}
				
				/* Here are the Balmer transitions.	*/
				else if ( ipLo == ipHe2s1S )
				{
					ASSERT( (lHi == 1) && (sHi == 0) );

					/* this fit calculated from transitions with upper level 5^1P through 9^1P,
					 * from Kono and Hattori (1984).	*/
					*Aul = (4.00e8) / pow((double)nHi, 2.889);
					ASSERT( *Aul > 0.);
				}

				/* Here are the Paschen transitions.	*/
				else if ( ipLo == ipHe3s1S )
				{
					ASSERT( (lHi == 1) && (sHi == 0) );

					/* this fit calculated from transitions with upper level 5^1P through 9^1P,
					 * from Kono and Hattori (1984).	*/
					*Aul = (7.532e7) / pow((double)nHi, 2.722);
					ASSERT( *Aul > 0.);
				}
				
				/* Here are transitions down to triplet S, for nLo=2,3, and 4.	*/
				else if ( ( ipLo == ipHe2s3S ) || ( ipLo == ipHe3s3S ) || ( ipLo == ipHe4s3S ) )
				{
					ASSERT( (lHi == 1) && (sHi == 1) );

					if ( nLo == 2 )
						*Aul = 3.405E8 / pow((double)nHi, 2.883);
					else if ( nLo == 3 )
						*Aul = 4.613E7 / pow((double)nHi, 2.672);
					else 
						*Aul = 1.436E7 / pow((double)nHi, 2.617);

					ASSERT( *Aul > 0.);
				}

				/* Here are transitions down to singlet P, for nLo=2,3, and 4.	*/
				else if ( ( ipLo == ipHe2p1P ) || ( ipLo == ipHe3p1P ) || ( ipLo == ipHe4p1P ) )
				{
					ASSERT( ((lHi == 0)||(lHi == 2)) && (sHi == 0) );
					ASSERT( (nLo == 2) || (nLo == 3) || (nLo == 4) );

					if( lHi == 0 )
					{
						/* These ipHi are singlet s	*/
						if ( nLo == 2 )
							*Aul = 5.055E8 / pow((double)nHi, 3.135);
						else if ( nLo == 3 )
							*Aul = 3.749E8 / pow((double)nHi, 3.241);
						else 
							*Aul = 2.974E8 / pow((double)nHi, 3.340);
					}
					else if( lHi == 2 )
					{
						/* These ipHi are singlet d	*/
						if ( nLo == 2 )
							*Aul = 1.631E9 / pow((double)nHi, 3.239);
						else if ( nLo == 3 )
							*Aul = 5.809E8 / pow((double)nHi, 3.206);
						else 
							*Aul = 2.429E8 / pow((double)nHi, 3.145);
					}

					ASSERT( *Aul > 0.);
				}

				/* Here are transitions down to 2tripletP, j=0,1, or 2.	*/
				else if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
				{
					if( lHi == 0 )
					{
						/* this fit calculated from transitions with upper level 5^3S through 9^3S,
						 * from Kono and Hattori (1984).	*/
						*Aul = 7.845E8 / pow((double)nHi, 3.218);
					}
					else if( lHi == 2 )
					{
						/* this fit calculated from transitions with upper level 5^3D through 9^3D,
						 * from Kono and Hattori (1984).	*/
						*Aul = 1.872E9 / pow((double)nHi, 3.163);
					}
					else
					{
						fprintf(ioQQQ,"WHOA!  Stop in Helike.c");
						cdEXIT(EXIT_FAILURE);
					}	

					*Aul *= (2.*(ipLo-3)+1.0)/9.0;
				}

				else if ( ((lHi > 2) || (lLo > 2)) )
				{
					/*lint -e790 integral to float */
					*Aul = H_Einstein_A(nHi ,lHi , nLo , lLo , ipHELIUM);
					/*lint +e790 integral to float */
					ASSERT( *Aul > 0.);
				}
				
				/* last resort for transitions involving significant defects.	*/
				else
				{
					/* Calculate the radial integral from the quantum defects.	*/
					RI2 = scqdri(Eff_nupper,lHi,Eff_nlower,lLo,(double)(ipHELIUM));
					ASSERT( RI2 > 0. );
					/* Convert radial integral to Aul.	*/
					*Aul = ritoa(lHi,lLo,ipHELIUM,Enerwn,RI2);
					/* radial integral routine does not recognize hyperfine structure.
					 * Here we split 2^3P.	*/
					if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
					{
						*Aul *= (2.*(ipLo-3)+1.0)/9.0;
					}
					ASSERT( *Aul > 0. );
				}
			}
		}

		/* Heavier species	*/
		else
		{
			/* There is probably no need to go to this detail for the ions.
			 * These are remnants of the original version in which Helium was done
			 * in the same loops as the ions.  But of course, since the ions are 
			 * more hydrogenic for any given level, (not to mention far less abundant)
			 * this detail is not necessary, and the algorithm for ions should be simplified.	*/
			
			/* Do same-n transitions. */
			if( nLo == nHi )
			{
				/* These are 2p3Pj to 2s3S fits to (low Z) Porquet & Dubau (2000) & 
				 * (high Z) NIST Atomic Spectra Database.	*/
				if ( ipLo == ipHe2s3S )
				{
					if (ipHi == ipHe2p3P0)
						*Aul = 3.31E7 + 1.13E6 * pow((double)nelem+1.,1.76);
					else if (ipHi == ipHe2p3P1)
						*Aul = 2.73E7 + 1.31E6 * pow((double)nelem+1.,1.76);
					else if (ipHi == ipHe2p3P2)
						*Aul = 3.68E7 + 1.04E7 * exp(((double)nelem+1.)/5.29);
					else
					{
						fprintf(ioQQQ,"WHOA!  Stop in Helike.c");
						cdEXIT(EXIT_FAILURE);
					}
				}
				
				/* These are 2p1P to 2s1S fits to data from TOPbase.	*/
				else if ( ( ipLo == ipHe2s1S ) && ( ipHi == ipHe2p1P) )
				{
					*Aul = 5.53E6 * exp( 0.171*(nelem+1.) );
				}
				
				else 
				{
					/* This case should only be entered if n > 2.  Those cases were done above.	*/
					ASSERT( nLo > 2 );

					/* Triplet P to triplet S, delta n = 0	*/
					if ( (lHi == 1) && (sHi == 1) && (lLo == 0) && (sLo == 1))
					{
						*Aul = 0.4 * 3.85E8 * pow((double)nelem,1.6)/pow((double)nHi,5.328);
					}
					/* Singlet P to singlet D, delta n = 0	*/
					else if ( (lHi == 1) && (sHi == 0) && (lLo == 2) && (sLo == 0))
					{
						*Aul = 1.95E4 * pow((double)nelem,1.6) / pow((double)nHi, 4.269);
					}
					/* Singlet P to singlet S, delta n = 0	*/
					else if ( (lHi == 1) && (sHi == 0) && (lLo == 0) )
					{
						*Aul = 6.646E7 * pow((double)nelem,1.5) / pow((double)nHi, 5.077);
					}
					else /*if ( (lHi == 2) && (sHi == 1)  && (lLo == 1) ) */
					{
						*Aul = 3.9E6 * pow((double)nelem,1.6) / pow((double)nHi, 4.9);
						if ( (lHi >2) || (lLo > 2) )
							*Aul *= (lHi/2.);
						if (lLo > 2)
							*Aul *= (1./9.);
					}
				}
				ASSERT( *Aul > 0.);
			}

			/* assume transitions involving F and higher orbitals are hydrogenic.	*/
			else if ( ((lHi > 2) || (lLo > 2)) )
			{
				ASSERT( nHi > nLo );
				/*lint -e790 integral to float */
				*Aul = H_Einstein_A(nHi ,lHi , nLo , lLo , nelem);
				/*lint +e790 integral to float */
				ASSERT( *Aul > 0.);
			}
			
			/* These transitions are of great importance, but the below radial integral 
			 * routine fails to achieve desirable accuracy, so these are fits as produced 
			 * from He A's for nupper through 9.  They are transitions to ground and 
			 * 2, 3, and 4 triplet S.	*/
			else if ( ( ipLo == 0 ) || ( ipLo == ipHe2s1S ) || ( ipLo == ipHe2s3S ) 
				|| ( ipLo == ipHe3s3S ) || ( ipLo == ipHe4s3S ) )
			{
				/* Here are the Lyman transitions.	*/
				if ( ipLo == 0 )
				{
					ASSERT( (lHi == 1) && (sHi == 0) );

					/* In theory, this Z dependence should be Z^4, but values from TOPbase 
					 * suggest 3.9 is a more accurate exponent.	Values from 
					 * >>refer	He-like	As	Johnson, W.R., Savukov, I.M., Safronova, U.I., & 
					 * >>refercon	Dalgarno, A., 2002, ApJS 141, 543J	*/
					/* originally astro.ph. 0201454  */
					*Aul = 1.375E10 * pow((double)nelem, 3.9) / pow((double)nHi,3.1);
				}

				/* Here are the Balmer transitions.	*/
				else if ( ipLo == ipHe2s1S )
				{
					ASSERT( (lHi == 1) && (sHi == 0) );

					*Aul = 5.0e8 * pow((double)nelem,4.) / pow((double)nHi, 2.889);
				}

				/* Here are transitions down to triplet S	*/
				else
				{
					ASSERT( (lHi == 1) && (sHi == 1) );

					if ( nLo == 2 )
						*Aul = 1.5 * 3.405E8 * pow((double)nelem,4.) / pow((double)nHi, 2.883);
					else if ( nLo == 3 )
						*Aul = 2.5 * 4.613E7 * pow((double)nelem,4.) / pow((double)nHi, 2.672);
					else 
						*Aul = 3.0 * 1.436E7 * pow((double)nelem,4.) / pow((double)nHi, 2.617);
				}

				ASSERT( *Aul > 0.);
			}
			
			/* Every other allowed transition is calculated as follows.	*/
			else
			{
				/* Calculate the radial integral from the quantum defects.	*/
				RI2 = scqdri(Eff_nupper,lHi,Eff_nlower,lLo,(double)(nelem));
				/* Convert radial integral to Aul.	*/
				*Aul = ritoa(lHi,lLo,nelem,Enerwn,RI2);
				/* radial integral routine does not recognize hyperfine structure.
				 * Here we split 2^3P.	*/
				if ( ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) ) && (*Aul > 1e-20) )
				{
					*Aul *= (2.*(ipLo-3)+1.0)/9.0;
				}
				ASSERT( *Aul > 0. );
			}
		}
	}

	/* Now do forbidden transitions from 2-1 ... */
	/* and those given by  
	 * >>refer	He-like	As	Johnson, W.R., Savukov, I.M., Safronova, U.I., & 
	 * >>refercon	Dalgarno, A., 2002, ApJS 141, 543J	*/
	/* originally astro.ph. 0201454  
	 * for heavy elements. These are triplet P to singlet S, 
	 * going either up or down...Triplet S to Singlet P are not included, as they are far weaker.	*/
	else 
	{
		ASSERT( (sHi != sLo) || (abs((int)(lHi - lLo)) != 1) );
		*Aul = ForbiddenAuls(ipHi, ipLo, nelem);
		ASSERT( *Aul > 0. );
	}
	
	ASSERT( *Aul >= iso.SmallA );

	/* negative energy for a transition with substantial transition probability
	 * would be major logical error - but is ok for same-n l transitions */
	if( Enerwn < 0. && *Aul > iso.SmallA )
	{
		fprintf( ioQQQ," he_1trans hit negative energy, nelem=%li, val was %f \n", nelem ,Enerwn );
	}

	return;
}

static void DoFSMixing( long nelem, long ipLoSing, long ipHiSing )
{
	int nHi, lHi, nLo, lLo, ipHiTrip, ipLoTrip;
	float Ass, Att, Ast, Ats;
	float Sin2Hi, Sin2Lo, Cos2Hi, Cos2Lo;
	double HiMixingAngle, LoMixingAngle , f;
	
	nHi = iso.quant_desig[ipHE_LIKE][nelem][ipHiSing].n;
	lHi = iso.quant_desig[ipHE_LIKE][nelem][ipHiSing].l;
	nLo = iso.quant_desig[ipHE_LIKE][nelem][ipLoSing].n;
	lLo = iso.quant_desig[ipHE_LIKE][nelem][ipLoSing].l;
	ASSERT( (lHi > 1) && (lLo > 1) );
	
	ipHiTrip = QuantumNumbers2Index[nelem][nHi][lHi][1];
	ipLoTrip = QuantumNumbers2Index[nelem][nLo][lLo][1];
	
	if( lHi == 2 )
	{
		HiMixingAngle = 0.01;
	}
	else if( lHi == 3 )
	{
		HiMixingAngle = 0.5;
	}
	else
	{
		HiMixingAngle = PI/4.;
	}
	
	if( lLo == 2 )
	{
		LoMixingAngle = 0.01;
	}
	else if( lLo == 3 )
	{
		LoMixingAngle = 0.5;
	}
	else
	{
		LoMixingAngle = PI/4.;
	}

	/* 
	HiMixingAngle = 0.;
	LoMixingAngle = 0.;*/

	/* These would not work correctly if l<=1 were included in this treatment!	*/
	ASSERT( ipHiTrip > ipLoTrip );
	ASSERT( ipHiTrip > ipLoSing );
	ASSERT( ipHiSing > ipLoTrip );
	ASSERT( ipHiSing > ipLoSing );

	f = sin( HiMixingAngle );
	Sin2Hi = POW2( (float)f );

	f = sin( LoMixingAngle );
	Sin2Lo = POW2( (float)f );

	f = cos( HiMixingAngle );
	Cos2Hi = POW2( (float)f );

	f = cos( LoMixingAngle );
	Cos2Lo = POW2( (float)f );
	
	/* First fix decay and de-excitation processes...	*/
	Ass = 
		Cos2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul + 
		Sin2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul*(1.f/3.f)+ 
		Cos2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul*(1.f/3.f)+ 
		Sin2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul*(1.f/9.f);
	Ast = (2.f/3.f)*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul+
		Sin2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul + 
		Cos2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul*(1.f/3.f)+ 
		Sin2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul*(1.f/3.f)+ 
		Cos2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul*(1.f/9.f);
	Ats = (2.f/3.f)*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul+
		Cos2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul + 
		Sin2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul*(1.f/3.f)+ 
		Cos2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul*(1.f/3.f)+ 
		Sin2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul*(1.f/9.f);
	Att = (8.f/9.f)*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul+
		Sin2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul + 
		Cos2Lo*Sin2Hi*EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul*(1.f/3.f)+ 
		Sin2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul*(1.f/3.f)+ 
		Cos2Lo*Cos2Hi*EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul*(1.f/9.f);

	ASSERT( fabs( ( EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul+
		EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul+
		EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul+
		EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul )/(Ass+Ast+Ats+Att) - 1.f ) < 0.001 );

	EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoSing].Aul = Ass;
	EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoTrip].Aul = Att;
	EmisLines[ipHE_LIKE][nelem][ipHiSing][ipLoTrip].Aul = Ast;
	EmisLines[ipHE_LIKE][nelem][ipHiTrip][ipLoSing].Aul = Ats;

	return;
}

/* Function he_assign assigns, in order of increasing energy, 				*/
/* quantum numbers n,l, and s to each element of vector iso.quant_desig[ipHE_LIKE].			*/
static void he_assign( long nelem)
{
	/*	in, il, and is are dummies for n, l, and s.		*/
	/*	i is the index in the vector iso.quant_desig[ipHE_LIKE].			*/
	long in, il, is, ij, i = 0 , level;
	
	/* this loop is over quantum number n */
	for ( in = 1L; in <= iso.n_HighestResolved[ipHE_LIKE][nelem] ; ++in )
	{
		for ( il = 0L; il < in; ++il )
		{
			for ( is = 1L; is >= 0 ; --is )
			{
				/* All levels except singlet P follow the ordering scheme:	*/
				/*	lower l's have lower energy	*/
				/* 	triplets have lower energy	*/
				if ( (il == 1L) && (is == 0L) )
					continue;
				/* n = 1 has no triplet, of course.	*/
				if ( (in == 1L) && (is == 1L) )
					continue;
				ij = 0;
				do 
				{
					iso.quant_desig[ipHE_LIKE][nelem][i].n = in;
					iso.quant_desig[ipHE_LIKE][nelem][i].s = is;
					iso.quant_desig[ipHE_LIKE][nelem][i].l = il;
					QuantumNumbers2Index[nelem][in][il][is] = i;
					++i;
					++ij;
				}	while ( (in == 2) && (il == 1) && (is == 1) && (ij < 3) );
			}
		}
		/*	Insert singlet P at the end of every sequence for a given n.	*/
		if ( in > 1L )
		{
			iso.quant_desig[ipHE_LIKE][nelem][i].n = in;
			iso.quant_desig[ipHE_LIKE][nelem][i].s = 0L;
			iso.quant_desig[ipHE_LIKE][nelem][i].l = 1L;
			QuantumNumbers2Index[nelem][in][1][0] = i;
			++i;
		}
	}
	/* now do the collapsed levels */
	in = iso.n_HighestResolved[ipHE_LIKE][nelem] + 1;
	for( level = i; level< iso.numLevels[ipHE_LIKE][nelem]; ++level)
	{
		iso.quant_desig[ipHE_LIKE][nelem][level].n = in;
		iso.quant_desig[ipHE_LIKE][nelem][level].s = -LONG_MAX;
		iso.quant_desig[ipHE_LIKE][nelem][level].l = -LONG_MAX;
		/* Point every l and s to same index for collapsed levels.	*/
		for( il = 0; il < in; ++il )
		{
			for( is = 0; is <= 1; ++is )
			{
				QuantumNumbers2Index[nelem][in][il][is] = level;
			}
		}
		++in;
	}
	--in;

	/* confirm that we did not overrun the array */
	ASSERT( i <= iso.numLevels[ipHE_LIKE][nelem] );

	ASSERT( (in > 0) && (in < (iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) );


	/* Verify iso.quant_desig[ipHE_LIKE] and QuantumNumbers2Index agree in all cases	*/
	for ( in = 2L; in <= iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem]  ; ++in )
	{
		for ( il = 0L; il < in; ++il )
		{
			for ( is = 1L; is >= 0 ; --is )
			{
				/* Ground state is not triplicate.	*/
				if ( (in == 1L) && (is == 1L) )
					continue;
				
				ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ QuantumNumbers2Index[nelem][in][il][is] ].n == in );
				if( in <= iso.n_HighestResolved[ipHE_LIKE][nelem] )
				{
					/* Must only check these for resolved levels...
					 * collapsed levels in iso.quant_desig[ipHE_LIKE] have pointers for l and s that will blow if used.	*/
					ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ QuantumNumbers2Index[nelem][in][il][is] ].l == il );
					ASSERT( iso.quant_desig[ipHE_LIKE][nelem][ QuantumNumbers2Index[nelem][in][il][is] ].s == is );
				}
			}
		}
	}
	
	return;
}

/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * nelem is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
double He_cross_section( double EgammaRyd , long ipLevel , long nelem )
{
	double cs ;
	
	globalZ = nelem;

	ipLev = ipLevel;

	EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLev];
	
	/* make sure this routine not called within collapsed high levels */
	ASSERT( ipLevel < iso.numLevels[ipHE_LIKE][nelem] - iso.nCollapsed[ipHE_LIKE][nelem] );

	/* this is a resolved level */
	cs = (1.e-18)*cross_section( EgammaRyd );
	
	return cs;
}

/*cross_section calculates the photoionization cross_section for a given level and photon energy*/
static double cross_section(double EgammaRyd)
{
	double E0[29] = {
	1.36E+01,2.01E+01,1.76E+01,3.34E+01,4.62E+01,6.94E+01,8.71E+01,1.13E+02,1.59E+02,2.27E+02,
	2.04E+02,2.74E+02,2.75E+02,3.38E+02,4.39E+02,4.17E+02,4.47E+02,5.18E+02,6.30E+02,6.27E+02,
	8.66E+02,7.67E+02,9.70E+02,9.66E+02,1.06E+03,1.25E+03,1.35E+03,1.43E+03,1.56E+03};
	double sigma[29] = {
	9.49E+02,3.20E+02,5.46E+02,2.85E+02,2.34E+02,1.52E+02,1.33E+02,1.04E+02,6.70E+01,4.00E+01,
	6.14E+01,4.04E+01,4.75E+01,3.65E+01,2.45E+01,3.14E+01,3.11E+01,2.59E+01,1.94E+01,2.18E+01,
	1.23E+01,1.76E+01,1.19E+01,1.31E+01,1.20E+01,9.05E+00,8.38E+00,8.06E+00,7.17E+00};
	double y_a[29] = {
	1.47E+00,7.39E+00,1.72E+01,2.16E+01,2.18E+01,2.63E+01,2.54E+01,2.66E+01,3.35E+01,5.32E+01,
	2.78E+01,3.57E+01,2.85E+01,3.25E+01,4.41E+01,3.16E+01,3.04E+01,3.28E+01,3.92E+01,3.45E+01,
	5.89E+01,3.88E+01,5.35E+01,4.83E+01,5.77E+01,6.79E+01,7.43E+01,7.91E+01,9.10E+01};
	double P[29] = {
	3.19E+00,2.92E+00,3.16E+00,2.62E+00,2.58E+00,2.32E+00,2.34E+00,2.26E+00,2.00E+00,1.68E+00,
	2.16E+00,1.92E+00,2.14E+00,2.00E+00,1.77E+00,2.04E+00,2.09E+00,2.02E+00,1.86E+00,2.00E+00,
	1.62E+00,1.93E+00,1.70E+00,1.79E+00,1.72E+00,1.61E+00,1.59E+00,1.58E+00,1.54E+00};
	double y_w[29] = 
	{2.039,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	double yzero[29] =
	{0.4434,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	double yone[29] =
	{2.136,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	
	/* These numbers are for the calculation of S cross-sections for n = 2 to n = 10. */
	/* indices are [n-2][s][a,b]	*/
	double heliumSparameters[9][2][2] = {
		{{0.0242024555,0.9702079},	{0.095921966,0.7337243}},

		{{0.0141220876,3.5176515},	{0.062805390,3.4047836}},

		{{0.0091014916,8.4545141},	{0.044113776,9.5657406}},
		
		{{0.0059539267,16.705745},	{0.032224774,20.933772}},
	
		{{0.0039485621,28.762086},	{0.024926753,38.464649}},
	
		{{0.0026053536,45.352333},	{0.020215452,63.043223}},
		
		{{0.0016632439,67.296132},	{0.016490176,96.296457}},
		
		{{0.0009867019,95.189826},	{0.013689704,138.82733}},
		
		{{0.0005155132,129.46774},	{0.011708145,191.73963}}};

	/* These are for the calculation of P cross-sections for n = 2 and n = 3. */
	/* indices are [n-2][a,b]	*/
	double heliumPparameters[2][2] = {
		{0.00489878046,3.0995085}, {0.01284809745,14.191429}};
	
	/* These corrections ensure helium cross-sections agree with the highly accurate 
	 * values given by Hummer and Storey (1998).  The index is [ipLev - 1]	*/
	/* >>refer He	PCS	Hummer, D.G., & Storey, P.J. 1998, MNRAS 297, 1073	*/  
	double HeThHummerStorey98[21] = {
		9.960E-01,1.020E+00,9.890E-01,1.014E+00,1.014E+00,1.014E+00,1.010E+00,
		1.021E+00,9.705E-01,1.000E+00,1.020E+00,1.000E+00,1.046E+00,1.025E+00,
		9.654E-01,1.020E+00,1.025E+00,1.000E+00,1.000E+00,1.000E+00,1.065E+00};

	double pcs,Egamma,lnE,a,b,y,F,x;
	long n,l,s,nelem = globalZ;

	n = iso.quant_desig[ipHE_LIKE][nelem][ipLev].n;
	l = iso.quant_desig[ipHE_LIKE][nelem][ipLev].l;
	s = iso.quant_desig[ipHE_LIKE][nelem][ipLev].s;
	Egamma = EgammaRyd * EVRYD;
	

	/* >>refer Helike	PCS	Verner, D.A., Ferland, G.J., Korista, K.T., & Yakovlev, D.G.
	 * >>refercon	1996a, ApJ 465,487	*/
	/* All ground state cross-sections calculated here.	*/
	if (ipLev == ipHe1s1S)
	{
		x = Egamma/E0[nelem-1] - yzero[nelem-1];
		y = sqrt(x*x + yone[nelem-1]*yone[nelem-1]);
		F = ((x-1)*(x-1)+y_w[nelem-1]*y_w[nelem-1])
			* pow(y,0.5*P[nelem-1]-5.5) * pow((1+sqrt(y/y_a[nelem-1])),-P[nelem-1]);
		pcs = sigma[nelem-1]*F;
		ASSERT( pcs > 0. );
	}
	/* Due to the highly non-hydrogenic nature of S orbitals, both singlet and triplet,
	 * a special formula needs to be applied to them.  The hydrogenic formula is 
	 * unsatisfactory since its errors get larger with increasing n.	*/
	
	/* Fix a better one for n = 2 as well?	*/
	else if ( ( l == 0 ) && ( n > 2 ) && ( nelem >= ipCARBON ) )
	{
		lnE = log(EgammaRyd * POW2(1.667*n/nelem));
		pcs = POW2(5./(double)nelem) * exp(1.790166293 - 0.127439504 * POW2(lnE) - 1.660569112*lnE);
		pcs *= pow(n/3.,1.365);
		ASSERT( pcs > 0. );
	}
	/* In addion to the above, the S orbitals of helium need to be handled separately.	*/
	else if ( ( nelem == ipHELIUM ) && ( l == 0 ) )
	{
		if ( n<=10 )
		{
			/* These are fits to TOPbase data	*/
			a = heliumSparameters[n-2][s][0];
			b = heliumSparameters[n-2][s][1];
		}
		/* These are fits to the parameters of the above fits!	*/
		else if ( s == 0 )
		{
			a = 4.94 / pow((double)n,3.9);
			b = 0.147 * pow((double)n, 2.947);
		}
		else
		{
			a = 0.364/ pow((double)n, 1.49);
			b = 0.138 * pow((double)n, 3.146);
		}
			
		/* Put E^-3 tail on helium at energies beyond first resonance.	*/
		if ( EgammaRyd > 2.75 + EthRyd)
		{
			/* The value of the above fit at the point of the first resonance is approximately 
			 * equal to 1/(2.75^2 * b).  An E^-3 tail has a value of about 1/(2.75^3).  
			 * (EthRyd is much smaller than 2.75 even at n=2, just as parameter a is dwarved by b.)
			 * So the scale factor is 2.75/b.	*/
			 pcs = 2.75 * POW3(1./EgammaRyd) / b;
		}
		else
			pcs = 1./(a+b*POW2(EgammaRyd));
		
		ASSERT( pcs > 0. );
	}
	/* Fit to TOPbase data for triplet P...only n=2 and 3 are available.	*/
	else if ( ( nelem == ipHELIUM ) && ( l == 1 ) && (s == 1) && ( n <= 3 ) )
	{	
		a = heliumPparameters[n-2][0];
		b = heliumPparameters[n-2][1];
		pcs = 1./(a+b*POW3(EgammaRyd));

		ASSERT( pcs > 0. );
	}
	/* To everything else we apply a hydrogenic routine.	*/
	else 
	{
		double rel_photon_energy;
		
		/* >>chng 02 apr 24, more protection against calling with too small an energy */
		/* evaluating H-like photo cs at He energies, may be below threshold -
		 * prevent this from happening */
		rel_photon_energy = EgammaRyd / EthRyd;
		rel_photon_energy = MAX2( rel_photon_energy , 1. + FLT_EPSILON*2. );

		/* >>chng 02 apr 25, only one routine exposed, chooses which to evaluate */
		pcs = (1.e18)*H_photo_cs(rel_photon_energy , n, l, nelem);

		/* Rescale by ratio of real energy to hydrogenic energy.	*/
		pcs *= EthRyd * POW2( (double)n/(double)nelem );
					
		ASSERT( pcs > 0. );
	}

	/* Rescale helium to expected threshold values.	*/
	if ( ( nelem == ipHELIUM ) && ( ipLev <= 20 ) && !helike.lgSetBenjamin)
		pcs *= HeThHummerStorey98[ipLev];
	
	return pcs;
}

/*radrecomb calculates radiative recombination coefficients. */
static double radrecomb(double temp, long nelem, long ipLo)
{
	double alpha,RecomIntegral=0.,b,E1,E2,step,OldRecomIntegral,TotChangeLastFive;
	double change[5] = {0.,0.,0.,0.,0.};
	long s,l;
		
	s = iso.quant_desig[ipHE_LIKE][nelem][ipLo].s;
	l = iso.quant_desig[ipHE_LIKE][nelem][ipLo].l;

	/* Factors outside integral in Milne relation.	*/
	 b = 4.12373E+11 * (2.*l+1.) * (2.*s+1.) * pow(temp,-1.5) / 4;
	/* kT in Rydbergs.	*/
	kTRyd = temp / TE1RYD;
	globalZ = nelem;
	ipLev = ipLo;

	/* Begin integration.	*/
	/* First define characteristic step as the minimum of kTRyd and 3EthRyd.	*/
	E1 = EthRyd;
	step = MIN( kTRyd, 3.*E1 );
	E2 = E1 + step;
	/* Perform initial integration, from threshold to threshold + step.	*/
	RecomIntegral = qg32( E1, E2, RecomInt);
	/* Repeat the integration, adding each new result to the total, 
	 * except that the step size is doubled every time, since values away from 
	 * threshold tend to fall off more slowly.	*/
	do
	{
		OldRecomIntegral = RecomIntegral;
		E1 = E2;
		step *= 2;
		E2 = E1 + step;
		RecomIntegral += qg32( E1, E2, RecomInt);
		change[4] = change[3];
		change[3] = change[2];
		change[2] = change[1];
		change[1] = change[0];
		change[0] = (RecomIntegral - OldRecomIntegral)/RecomIntegral;
		TotChangeLastFive = change[0] + change[1] + change[2] + change[3] + change[4];
	/* Continue integration until the upper limit exceeds 100kTRyd, an arbitrary
	 * point at which the integrand component exp(electron energy/kT) is very small,
	 * making neglible cross-sections at photon energies beyond that point,
	 * OR when the last five steps resulted in less than a 1 percent change.	*/
	} while ( (E2 < 100.*kTRyd) && ( TotChangeLastFive > 0.01) );

	/* Calculate recombination coefficient.	*/
	alpha = b * RecomIntegral;

	if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
	{
		/* Split 2 trip P term.	*/
		alpha *= (2.*(ipLo-3L)+1.) / 9.;
	}

	if ( alpha < SMALLFLOAT )
		alpha = SMALLFLOAT;

	return alpha;
}

/*RecomInt, used in comput milne relation for he rec - the energy is photon Rydbergs.	*/
static double RecomInt(double ERyd)
{
	double x1;

	/* Milne relation integrand	*/
	x1 = ERyd * ERyd * exp(-1.0 * ( ERyd - EthRyd ) / kTRyd);
	x1 *= He_cross_section( ERyd , ipLev, globalZ );
		
	return x1;
}

/* >>refer	He-like	RR	Seaton, M.J. 1959, MNRAS 119, 81S */
static double Recomb_Seaton59( long nelem, double temp, long n)
{
	double lambda = TE1RYD * nelem * nelem / temp;
	/* smallest x ever used here should be lowest Z, highest T, highest n...
	 * using helium, logt = 10., and n = 100, we get xmin = 1.5789E-9.	*/
	double x = lambda / n / n;
	double AlphaN;
	double SzeroOfX = 0.;
	double SoneOfX = 0.;
	double StwoOfX = 0.;
	double SnOfLambda = 0.;
	double lowerlimit, upperlimit, step;

	Xn_S59 = x;

	/* Equation 12	*/
	lowerlimit = x;
	step = 3. * x;
	upperlimit = lowerlimit + step;
	SzeroOfX = qg32( lowerlimit, upperlimit, ExponentialInt);

	do
	{
		lowerlimit = upperlimit;
		step *= 2;
		upperlimit = lowerlimit + step;
		SzeroOfX += qg32( lowerlimit, upperlimit, ExponentialInt);
	} while ( upperlimit < 20. );

	/* This must be placed inside integral...too big to be 
	 * handled separately.	
	SzeroOfX *= exp( x );	*/

	/* Equations 13 and 14 */
	lowerlimit = 0.;
	step = 0.5;
	upperlimit = lowerlimit + step;
	SoneOfX = qg32( lowerlimit, upperlimit, X1Int);
	StwoOfX = qg32( lowerlimit, upperlimit, X2Int);

	do
	{
		lowerlimit = upperlimit;
		step *= 2;
		upperlimit = lowerlimit + step;
		SoneOfX += qg32( lowerlimit, upperlimit, X1Int);
		SoneOfX += qg32( lowerlimit, upperlimit, X2Int);
	} while ( upperlimit < 200. );

	SoneOfX *= 0.1728 * pow( x, 1./3. );
	StwoOfX *= -0.0496 * pow( x, 2./3. );

	/* Equation 11	*/
	SnOfLambda = SzeroOfX + pow(1./lambda, 1./3.)*SoneOfX + pow(1./lambda, 2./3.)*StwoOfX;

	AlphaN = 5.197E-14 * nelem * pow(x, 1.5) * SnOfLambda;

	return AlphaN;

}

static double ExponentialInt( double v )
{
	double Integrand;

	Integrand = exp( -1. * v + Xn_S59) / v;

	return Integrand;
}

static double X1Int( double u )
{

	double Integrand;

	Integrand = pow(1./(u + 1.), 5./3.) * (u - 1.) * exp( -1. * Xn_S59 * u );

	return Integrand;
}

static double X2Int( double u )
{

	double Integrand;

	Integrand = pow(1./(u + 1.), 7./3.) * (u*u + 4./3.*u + 1.) * exp( -1. * Xn_S59 * u );

	return Integrand;
}

/*printstuff printflag determines what will be printed.*/
static void printstuff(long flag)
{
	long int i,ic,nelem,ipLo,ipHi;
	int lgHeader = 0;
	double thresh_energy;

	switch(flag)
	{
	case 1:
		if ( (ofp = fopen("HelikeAs.txt","w")) == NULL )
			return;
		fprintf(ofp," Z\tni^JL\tnf^JL\tipLo\tipHi\tA\n");
		break;
	case 2:
		if ( (ofp = fopen("HelikefAs.txt","w")) ==NULL )
			return;
		fprintf(ofp,"FORBIDDEN	TRANSITIONS	to 1^1S\n");
		fprintf(ofp,"Z\t2^3S\t2^1S\t2^3P1\n");
		break;
	case 3:
		if ( (ofp = fopen("HelikeEs.txt","w")) == NULL )
			return;
		break;
	case 4:
		if ( ( ofp = fopen("HelikeThPCS.txt","w")) == NULL )
			return;
		break;
	case 5:
		if ( ( ofp = fopen("HelikeRR.txt","w")) ==NULL )
			return;
		break;
	case 6:
		if ( ( ofp = fopen("HelikePCSgrid.txt","w")) == NULL )
			return;
		break;
	default:
		return;
	}
	
	for (i=1; i<N_HE_TE_RECOMB; i++)
	{
		if (flag == 5)
		{
			fprintf(ofp,"temp= %2.1f \n",0.5*i);
		}
		else if (lgHeader)
		{
			return;
		}
		if ( (flag > 2) && (flag < 6) && (!lgHeader) )
		{
			fprintf(ofp,"WARNING: These levels are as resolved for He...not all elements necessarily resolved in same way!\n");
			fprintf(ofp,"Z   ");
			for( ipLo=ipHe1s1S; ipLo < iso.numLevels[ipHE_LIKE][ipHELIUM]; ipLo++ )
			{
				fprintf(ofp,"n%1lds%1ldl%1ld      ",
					iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].n,
					iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].s,
					iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].l);
			}
			fprintf(ofp,"\n");
		}
		lgHeader = TRUE;

		for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
		{
			if( nelem == ipHELIUM || dense.lgElmtOn[nelem] ) 
			{
				if ( (flag > 1) && (flag < 6) )
				{
					fprintf(ofp,"%2ld ",nelem+1L);
				}
				
				globalZ = nelem;
				
				for( ipLo=ipHe1s1S; ipLo < MIN2(iso.numLevels[ipHE_LIKE][ipHELIUM],
					iso.numLevels[ipHE_LIKE][nelem]); ipLo++ )
				{
					if (ipLo == ipHe1s1S)
					{
						thresh_energy = EionRYD[nelem-1L];
					}
					else
					{
						thresh_energy = EionRYD[nelem-1L] - EmisLines[ipHE_LIKE][nelem][ipLo][ipHe1s1S].EnergyWN * WAVNRYD;
					}
					
					ipLev = ipLo;
					EthRyd = thresh_energy;
				
					switch(flag)
					{
					case 1:
						for( ipHi=ipLo+1L; ipHi < iso.numLevels[ipHE_LIKE][nelem]; ipHi++)
						{
							if (EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul > (1e-20) )
							{
								fprintf(ofp,"%2ld\t%2ld^%1ld",nelem+1L,
									iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
									2*(iso.quant_desig[ipHE_LIKE][nelem][ipLo].s)+1);
								orbit(ipLo,nelem);
								fprintf(ofp,"\t%2ld^%1ld",
									iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,
									2*(iso.quant_desig[ipHE_LIKE][nelem][ipHi].s)+1);
								orbit(ipHi,nelem);
								fprintf(ofp,"\t%3ld\t%3ld\t%2.2e\n",ipLo,ipHi,EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul);
							}
						}
						break;
					case 2:
						if( (ipLo == ipHe2s3S) || (ipLo == ipHe2s1S) || (ipLo == ipHe2p3P1) )
						{
							fprintf(ofp,"\t%2.2e",EmisLines[ipHE_LIKE][nelem][ipLo][0].Aul);
						}
						break;
					case 3:
						fprintf(ofp,"%2.4e ",thresh_energy);
						break;
					case 4:
						fprintf(ofp,"%2.4e ",cross_section(thresh_energy)); 
						break;
					case 5:
						fprintf(ofp,"%2.4e ",pow(10.,RRCoef[nelem][ipLo][i]));
						break;
					case 6:
						fprintf(ofp,"Z=%ld  %2ld^%1ld",nelem+1,
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
							2*(iso.quant_desig[ipHE_LIKE][nelem][ipLo].s)+1);
						orbit(ipLo,nelem);
						fprintf(ofp,"\n");
						for (ic=0; ic<100; ic++)
						{
							fprintf(ofp,"%2.2e %2.2e\n",thresh_energy*(1+0.1*pow((double)ic,1.5)),
									(1E-18)*cross_section(thresh_energy*(1+0.1*ic)));
						}
						break;
					default:
						fprintf(ofp,"\n");
					}
				}
				fprintf(ofp,"\n");
			}
		}
	}
	fclose(ofp);
	return;
}

void He1Autoionize(void)
{

/* Autoionizing levels are as follows:	*/
	const int ipHe2s2p1P = 0;
/*	const int ipHe2s2p3P = 1;	*/
/*	const int ipHe2p2_1D = 2;	*/
 	const int ipHe2p2_3P = 3;		

	double a[4] = {75827., 72116., 74853., 74010.};
	double b[4] = {1.8956, 1.9084,    1.9, 1.9021};

	long nelem, ipHi;

	for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* only set elements that are turned on */
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
		{
			for ( ipHi = ipHe2s2p1P; ipHi <= ipHe2p2_3P; ipHi++ )
			{
				He1AutoLines[nelem][ipHi].EnergyWN = (float)(a[ipHi]*pow((double)(nelem+1),b[ipHi]));
			}
		}
	}

	return;
	/* TODO say where these come from...and do something with them!	*/	
	/* For double-ionization discussions, see Lindsay, Rejoub, & Stebbings 2002	*/
	/* Also read Itza-Ortiz, Godunov, Wang, and McGuire 2001.	*/
}

/*ritoa converts the square of the radial integral for a transition 
 * (calculated by scqdri) to the transition probability, Aul.	*/
static double ritoa(long li, long lf, long nelem, double k, double RI2)
{  
	/*	Variables are as follows:				*/
	/*	lg = larger of li and lf				*/
	/*	fmean = mean oscillator strength		*/
	/*		for a given level.					*/
	/*	mu = reduced mass of optical electron.	*/
	/*	EinsteinA = Einstein emission coef.		*/
	/*	w = angular frequency of transition.	*/
	/*	RI2_cm = square of rad. int. in cm^2.	*/
	long lg;
	double fmean,mu,EinsteinA,w,RI2_cm;
	
	mu = ELECTRON_MASS/(1+ELECTRON_MASS/(AtomcWgt.AtomicWeight[nelem]*ATOMIC_MASS_UNIT));

	/* 8065.541 converts cm-1 to eV.  HBAReV then converts to rad/sec 	*/

	w = k /(8065.541*HBAReV);

	RI2_cm = RI2 * (2.8002856e-17);

	lg = (lf > li) ? lf : li;

	fmean = 2.0*mu*w*lg*RI2_cm/((3.0*H_BAR) * (2.0*li+1.0));

	EinsteinA = (.66702)*k*k*fmean;

	ASSERT( EinsteinA > SMALLFLOAT );

	return EinsteinA;
}

#undef DEBUG_LOC 

/* create he-like series, called by ContCreatePointers */
void HeCreate(void)
{
	double **energies, *n_effs, **SumAPerN, **RobbinsP, **RobbinsC;
	double SumAul, RadRecombReturn;

	long int i, i1, i2, i3, i4, j, ipLo, ipHi, ipFirstCollapsed, nelem, *NumHeLevRecomb;

	static int nCalled = 0;

#	define chLine_LENGTH 1000
	char chLine[chLine_LENGTH] , 
		/* this must be longer than chDataPath, set in path.h */
		chFilename[FILENAME_PATH_LENGTH_2];

	FILE *ioDATA;
	int lgEOL, lgHIT;

	char chSpin[2]={'1','3'};
	char chL[6]={'S','P','D','F','G','H'};

#	ifdef DEBUG_FUN
	fputs( "<+>HeCreate()\n", debug_fp );
#	endif

	/* now MALLOC the space for the helium-like lines, 
	 * but this must be done exactly one time per coreload */
	if( nCalled > 0 )
	{
#		ifdef DEBUG_FUN
		fputs( " <->HeCreate()\n", debug_fp );
#		endif
		return;
	}
	++nCalled;

	if( helike.lgSetBenjamin )
	{
		max_num_levels = iso.numLevels[ipHE_LIKE][ipHELIUM];
		max_n = iso.n_HighestResolved[ipHE_LIKE][ipHELIUM];

		if( helike.lgCompileRecomb )
		{
			fprintf( ioQQQ, " Not compiling He recomb file...can not do with Benjamin command set.");
			helike.lgCompileRecomb = FALSE;
		}
		if( helike.lgFSM )
		{
			fprintf( ioQQQ, " You are including fine-structure mixing with the Benjamin command!");
			helike.lgFSM = FALSE;
		}
	}
	else
	{
		/* first find the largest number of levels in any element in the he-like sequence */
		max_num_levels = 0;
		max_n = 0;
		for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
		{
			/* only check elements that are turned on */
			if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
			{
				max_num_levels = MAX2( max_num_levels, iso.numLevels[ipHE_LIKE][nelem] );
				max_n = MAX2( max_n, iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] );
			}
		}
	}

	if( (helike.n_HighestResolved = (long*)MALLOC(sizeof(long)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();
	
	if( (helike.nCollapsed = (long*)MALLOC(sizeof(long)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();
	
	if( (helike.numLevels = (long*)MALLOC(sizeof(long)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	if( (helike.lgFullSize = (int*)MALLOC(sizeof(int)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/*double ***RRCoef[LIMELM][MaxLevels][N_HE_TE_RECOMB]*/
	if( (RRCoef = (double ***)MALLOC(sizeof(double **)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/*double **TotalRecomb[LIMELM][N_HE_TE_RECOMB]*/
	if( (TotalRecomb = (double **)MALLOC(sizeof(double *)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/*char ***helike chLevel[lev][lev][str]*/
	if( (chLevel = (char **)MALLOC(sizeof(char *)*(unsigned)(max_num_levels) ) )==NULL )
		BadMalloc();

	/* In this array are stored the P values described in Robbins 68.	*/
	if( (RobbinsP = (double**)MALLOC(sizeof(double*)*(unsigned)(max_num_levels) ))==NULL )
		BadMalloc();

	/* In this array are stored the C values described in Robbins 68.	*/
	if( (RobbinsC = (double**)MALLOC(sizeof(double*)*(unsigned)(max_num_levels) ))==NULL )
		BadMalloc();

	/* Wavenumber level energies for each element are stored here.	*/
	if( (energies = (double**)MALLOC(sizeof(double*)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/* The lifetime of individual levels.  helike.Lifetime[LIMELM][iso.numLevels[ipHE_LIKE][nelem]]	*/
	if( (helike.Lifetime = (double**)MALLOC(sizeof(double*)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/* The sum of all A's coming out of a given n...used to make sure trend is monotonic	*/
	if( (SumAPerN = (double**)MALLOC(sizeof(double*)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/* The number of recombination coefficients to be read from file for each element.	*/
	if( (NumHeLevRecomb = (long*)MALLOC(sizeof(long)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	/* the reverse of iso.quant_desig[ipHE_LIKE]...QuantumNumbers2Index[nelem][n][l][s] 
	 * gives corresponding index in iso.quant_desig[ipHE_LIKE].	*/
	if( (QuantumNumbers2Index = (long****)MALLOC(sizeof(double***)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();
	
	for( ipLo=ipHe1s1S; ipLo < iso.numLevels[ipHE_LIKE][ipHELIUM] ;++ipLo )
	{
		if( (chLevel[ipLo] = (char*)MALLOC(sizeof(char)*(unsigned)10 ))==NULL )
			BadMalloc();
	}

	/* >>refer He triplets	Robbins, R.R. 1968, ApJ 151, 497R	*/
	/* >>refer He triplets	Robbins, R.R. 1968a, ApJ 151, 511R	*/
	for( ipLo=ipHe1s1S; ipLo < max_num_levels ;++ipLo )
	{
		if( (RobbinsP[ipLo] = (double*)MALLOC(sizeof(double)*(unsigned)(ipLo+1) ))==NULL )
			BadMalloc();

		if( (RobbinsC[ipLo] = (double*)MALLOC(sizeof(double)*(unsigned)(ipLo+1) ))==NULL )
    		BadMalloc();
	}
	
	for( nelem=ipHELIUM; nelem < LIMELM; ++nelem )
	{

		long int MaxLevels;

		if( nelem == ipHELIUM )
			NumHeLevRecomb[nelem]= ( ( 1 + HE_RREC_MAXN ) * HE_RREC_MAXN + 1 );
		else
			NumHeLevRecomb[nelem]= ( ( 1 + HE_LIKE_RREC_MAXN ) * HE_LIKE_RREC_MAXN + 1 );

		/* must always have at least NumHeLevRecomb[nelem] levels since that is number 
		 * that will be read in from he rec data file, but possible to require more */
		MaxLevels = MAX2( NumHeLevRecomb[nelem] , iso.numLevels[ipHE_LIKE][nelem] );

		/* always define this */
		/* >>chng 02 jan 24, RRCoef will be iso.numLevels[ipHE_LIKE][nelem] levels, no iso.numLevels,
		 * code will stop if more than this is requested */
		if( (RRCoef[nelem] = (double**)MALLOC(sizeof(double*)*(unsigned)(MaxLevels) ))==NULL )
			BadMalloc();

		/*double ***TotalRecomb[LIMELM][N_HE_TE_RECOMB]*/
		if( (TotalRecomb[nelem] = (double*)MALLOC(sizeof(double)*(unsigned)N_HE_TE_RECOMB ))==NULL )
			BadMalloc();

		for( ipLo=ipHe1s1S; ipLo < MaxLevels ;++ipLo )
		{
			if( (RRCoef[nelem][ipLo] = (double*)MALLOC(sizeof(double)*(unsigned)N_HE_TE_RECOMB ))==NULL )
				BadMalloc();
		}
		
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
		{
			if( (energies[nelem] = (double*)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipHE_LIKE][nelem]) ))==NULL )
				BadMalloc();
			
			if( (helike.Lifetime[nelem] = (double*)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipHE_LIKE][nelem]) ))==NULL )
				BadMalloc();

			/* The sum of all A's coming out of a given n...used to make sure trend is monotonic	*/
			if( (SumAPerN[nelem] = (double*)MALLOC(sizeof(double)*
				(unsigned)(iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) )==NULL )
				BadMalloc();
			
			/* Allocate proper number of principal quantum number.	*/
			if( (QuantumNumbers2Index[nelem] = (long***)MALLOC(sizeof(double**)*
				(unsigned)(iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) )==NULL )
				BadMalloc();

			for( i = 0; i <= (iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem]); ++i)
			{
				/* Allocate proper number of angular momentum quantum number.	*/
				if( (QuantumNumbers2Index[nelem][i] = (long**)MALLOC(sizeof(double*)*(unsigned)MAX2(1,i) ) )==NULL )
					BadMalloc();

				for( i1 = 0; i1 < i; ++i1 )
				{
					/* Allocate 2 spaces for spin.	*/
					if( (QuantumNumbers2Index[nelem][i][i1] = (long*)MALLOC(sizeof(double)*(unsigned)2 ) )==NULL )
						BadMalloc();
				}
			}
		}
	}

	for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* only check elements that are turned on */
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
		{
			/* Function he_assign assigns, in order of increasing energy, 
			 * quantum numbers n,l,s, and j to each element of vector iso.quant_desig[ipHE_LIKE]. */
			he_assign(nelem);
	    }
	}

	/* >>allocate one more than needed to allow space for end of vector sentinal
	 * in case where no collapsed levels */
	if ( (n_effs= (double *)MALLOC((unsigned)(max_num_levels+1)*sizeof(double)) ) == NULL )
		BadMalloc();
	{
		/* option to print particulars of some line when called
		 * a prettier print statement is near where chSpin is defined below*/
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC )
		{
			for( nelem = ipHELIUM; nelem <LIMELM; ++nelem )
			{
				/* first print internal representation */
				for( i=0; i < iso.numLevels[ipHE_LIKE][nelem]; ++i )
				{
					fprintf(ioQQQ,"nelem %li i %li n %li s %li l %li\n", 
						nelem,
						i, 
						iso.quant_desig[ipHE_LIKE][nelem][i].n,
						iso.quant_desig[ipHE_LIKE][nelem][i].s,
						iso.quant_desig[ipHE_LIKE][nelem][i].l );
				}
				if( DEBUG_LOC )
					cdEXIT(EXIT_SUCCESS);
			}
		}
	}

	/* main helium-like arrays, fill with sane values */
	for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
	{
		double Atot;
		/* charge to 4th power, needed for scaling laws for As*/
		double z4 = POW2((double)nelem);
		z4 *= z4;

		/* must always do helium even if turned off */
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem] ) 
		{
			for( i=0; i <= ( iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] ); ++i)
			{
				SumAPerN[nelem][i] = 0.;
			}

			/* this is the number of resolved levels, so first collapsed level is [ipFirstCollapsed] */
			ipFirstCollapsed = iso.numLevels[ipHE_LIKE][nelem]-iso.nCollapsed[ipHE_LIKE][nelem]; 
			/* loop to obtain energies for current ion - this is only over all resolved levels 
			 * this is supposed to be one beyond the last valid datum - 
			 * when nTopOff is zero this is actually beyond the end of the atom */
			n_effs[ipFirstCollapsed] = LONG_MAX;

			for( ipLo=ipHe1s1S; ipLo < ipFirstCollapsed; ipLo++ )
			{
				/* the effective quantum number */
				n_effs[ipLo] = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n - defect(nelem,ipLo);
				
				/* energies (in wavenumbers) that correspond to quantum defect */
				if (ipLo>ipHe1s1S)
					energies[nelem][ipLo] = he_energy(n_effs[ipLo],nelem,ipLo);
				else
					energies[nelem][ipLo] = EionWN[nelem-1];
				
				/* convert wavenumbers to Ryd.	*/
				EthRyd = energies[nelem][ipLo] * WAVNRYD;
				
				/* Store threshold energy in larger cloudy array	*/
				iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo] = EthRyd;
				ASSERT( iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo] > 0. );
			}

			/* Don't calculate As (or collapsed level energies) if just compiling recomb file.	*/
			if( !helike.lgCompileRecomb )
			{
				/* loop to obtain energies for current ion - 
				 * both upper and lower are collapsed levels */
				for( ipLo=ipFirstCollapsed; ipLo < iso.numLevels[ipHE_LIKE][nelem]; ipLo++ )
				{
					/* this is the n in the collapsed level */
					long int nCollapsed = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n;

					n_effs[ipLo] = (double)nCollapsed; 
					energies[nelem][ipLo] = RYD_INF * POW2((double)nelem/(double)nCollapsed);  
					/* convert wavenumbers to Ryd.	*/
					EthRyd = energies[nelem][ipLo] * WAVNRYD;

					/* Store threshold energy in larger cloudy array	*/
					iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo] = EthRyd;
					ASSERT( iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo] > 0. );
				}

				RobbinsC[ipHe1s1S][ipHe1s1S] = 1.;
				RobbinsP[ipHe1s1S][ipHe1s1S] = 0.;
				helike.Lifetime[nelem][ipHe1s1S] = 0.;
		
				/****************************************************************************
				 * First just do A's and energies.  These loops must be nested in this order!
				 * HeFsIndex is an index to retrieve Kono & Hattori 98 F's from a table.	
				 ****************************************************************************/
				for( ipHi=ipHe2s3S; ipHi<iso.numLevels[ipHE_LIKE][nelem] ; ipHi++ )
				{
					HeFsIndex = 0;

					for( ipLo=ipHe1s1S; ipLo<ipHi ; ipLo++ )
					{
						double Enerwn, Aul, Aul1;
						long nHi,lHi,sHi,nLo,lLo,sLo;

						/* pointers to transitions, ipLo, ipHi
						 * actual quantum numbers, nLo nHi
						 * atomic number or charge and stage: */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].nelem = (int)(nelem+1);
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].IonStg = (int)(nelem);

						nHi = iso.quant_desig[ipHE_LIKE][nelem][ipHi].n;
						lHi = iso.quant_desig[ipHE_LIKE][nelem][ipHi].l;
						sHi = iso.quant_desig[ipHE_LIKE][nelem][ipHi].s;
						nLo = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n;
						lLo = iso.quant_desig[ipHE_LIKE][nelem][ipLo].l;
						sLo = iso.quant_desig[ipHE_LIKE][nelem][ipLo].s;
												
						/* Energies are w.r.t the continuum, so must subtract 
						 * upper lever from lower level.	*/
						Enerwn = energies[nelem][ipLo] - energies[nelem][ipHi];
						/* if both of these levels are hydrogenic and the same n,
						 * their energies will be exactly the same, the Enerwn zero */
						/* Set this difference to 1 wavenumber.	*/
						if ( Enerwn == 0 ) 
							Enerwn += 1.;

						/* Passed to function he_1trans are the upper	*/
						/* and lower levels	of a transition.  			*/
						/* Aul is calculated.							*/
						if ( ipHi >= ipFirstCollapsed )
						{
							if ( ipLo >= ipFirstCollapsed )
							{
								/* Neither upper nor lower is resolved...Aul's are easy.	*/
								Aul = HydroEinstA(iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
									iso.quant_desig[ipHE_LIKE][nelem][ipHi].n)*z4;
								ASSERT( Aul > 0.);
								{
									ASSERT( (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n > 0) && 
										(iso.quant_desig[ipHE_LIKE][nelem][ipHi].n < (iso.n_HighestResolved[ipHE_LIKE][nelem] 
										+ iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) );
									SumAPerN[nelem][iso.quant_desig[ipHE_LIKE][nelem][ipHi].n] += Aul;
								}
							}
							else 
							{
								/* Lower level resolved, upper not. 
								 * First calculate Aul from upper level with ang mom one higher...
								 * this is applicable to all sublevels of nLo.	*/
								he_1trans( nelem, Enerwn, &Aul,
									(double)nHi, lLo+1, sLo, -1, (double)nLo, lLo, sLo, ipLo-3);
								Aul *= (2.*lLo+3.) * (2.*sLo+1.) / (4.*(double)nHi*(double)nHi);
								if ( iso.quant_desig[ipHE_LIKE][nelem][ipLo].l != 0 )
								{
									/* For all l>0, add in transitions from upper level with ang mom one lower.	*/
									he_1trans( nelem, Enerwn, &Aul1, (double)nHi, lLo-1, sLo, -1, (double)nLo, lLo, sLo, ipLo-3);
									/* now add in this part after multiplying by stat weight for lHi = lLo-1.	*/
									Aul += Aul1*(2.*lLo-1.) * (2.*sLo+1.) / (4.*(double)nHi*(double)nHi);
								}
								ASSERT( Aul > 0.);
								{
									ASSERT( (nHi > 0) && (nHi < (iso.n_HighestResolved[ipHE_LIKE][nelem] 
										+ iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) );
									SumAPerN[nelem][nHi] += Aul;
								}
							}
							
						}
						else
						{
							/* Both levels are resolved...do the normal bit.	*/
							he_1trans( nelem, Enerwn, &Aul,  n_effs[ipHi], lHi, sHi, ipHi-3, n_effs[ipLo], lLo, sLo, ipLo-3);
							{
								ASSERT( (nHi > 0) && (nHi < (iso.n_HighestResolved[ipHE_LIKE][nelem] 
									+ iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) );
								SumAPerN[nelem][nHi] += Aul*(2.*lHi + 1.)*(2.*sHi + 1.)/(4. * (double)nHi * (double)nHi) ;
							}
						}

						/* set the transition probability */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul = (float)(Aul);
						ASSERT(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul > 0.);

						/* the energy of the transition in wavenumbers, this can be negative
						 * for highly forbidden upper levels where energy switches */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN = (float)Enerwn;

					}
				}
						
				/* Produce stat. weights, modify Auls for FSM, stuff gf, opacity, etc.	*/
				for( ipHi=ipHe2s3S; ipHi<iso.numLevels[ipHE_LIKE][nelem] ; ipHi++ )
				{
					Atot = 0;
					RobbinsC[ipHi][ipHi] = 1.;
					RobbinsP[ipHi][ipHi] = 0.;
					helike.Lifetime[nelem][ipHi] = 0.;

					for( ipLo=ipHe1s1S; ipLo<ipHi ; ipLo++ )
					{
						float gHi , gLo;
						
						RobbinsC[ipHi][ipLo] = 0.;
						RobbinsP[ipHi][ipLo] = 0.;
						
						/* pointers to transitions, ipLo, ipHi
						 * actual quantum numbers, nLo nHi
						 * atomic number or charge and stage: */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].nelem = (int)(nelem+1);
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].IonStg = (int)(nelem);
						
						/* stat. weights, s is 0 (singlets) or 1 (triplets)   */
						/* the stat. weights must be computed differently
						 * for the 2^3P levels.	*/
						if ( ipHi >= ipFirstCollapsed )
						{
							gHi = 4.f*POW2((float)iso.quant_desig[ipHE_LIKE][nelem][ipHi].n);
						}
						else if ( ipHi == ipHe2p3P0 )
						{
							gHi = 1.;
						}
						else if ( ipHi == ipHe2p3P1 )
						{
							gHi = 3.;
						}
						else if ( ipHi == ipHe2p3P2 )
						{
							gHi = 5.;
						}
						else
						{
							gHi = (2.f*iso.quant_desig[ipHE_LIKE][nelem][ipHi].l + 1.f)
								* (2.f*iso.quant_desig[ipHE_LIKE][nelem][ipHi].s + 1.f);
						}

						if ( ipLo >= ipFirstCollapsed )
						{
							gLo = 4.f*POW2((float)iso.quant_desig[ipHE_LIKE][nelem][ipLo].n);
						}
						else if ( ipLo == ipHe2p3P0 )
						{
							gLo = 1.;
						}
						else if ( ipLo == ipHe2p3P1 )
						{
							gLo = 3.;
						}
						else if ( ipLo == ipHe2p3P2 )
						{
							gLo = 5.;
						}
						else
						{
							gLo = (2.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].l + 1.f)
								* (2.f*iso.quant_desig[ipHE_LIKE][nelem][ipLo].s + 1.f);
						}
						
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gLo = gLo;
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gHi = gHi;
						/* save statistical weights */
						iso.stat[ipHE_LIKE][nelem][ipLo] = gLo;
						iso.stat[ipHE_LIKE][nelem][ipHi] = gHi;
						ASSERT( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gHi> 0.);
						ASSERT( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gLo> 0.);

						/*********************************************************
						 * FSM - Here is where the fine-structure mixing magic happens.
						 * Please buckle your seat belt.						 
						 * Do all four transitions n'l'->nl at once...skip if either upper
						 * or lower level is triplet.	*/
						if( helike.lgFSM==2 && (iso.quant_desig[ipHE_LIKE][nelem][ipHi].s==0) 
							&& (iso.quant_desig[ipHE_LIKE][nelem][ipLo].s==0)
							&& ( abs(iso.quant_desig[ipHE_LIKE][nelem][ipHi].l -
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)==1 )
							&& (iso.quant_desig[ipHE_LIKE][nelem][ipLo].l>1) 
							&& (iso.quant_desig[ipHE_LIKE][nelem][ipHi].l>1) 
							&& ( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n >
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].n ) )
						{
							DoFSMixing( nelem, ipLo, ipHi );
						}
						
						/* This is not actually the lifetime...must invert when all A's added.	*/
						/* No stat. weight is needed because this is the lifetime of an individual level,
						 * Collapsed or resolved is irrelevant.	*/
						helike.Lifetime[nelem][ipHi] += EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul;

						if ( (iso.quant_desig[ipHE_LIKE][nelem][ipLo].s == iso.quant_desig[ipHE_LIKE][nelem][ipHi].s) && 
							(abs((int)(iso.quant_desig[ipHE_LIKE][nelem][ipLo].l - iso.quant_desig[ipHE_LIKE][nelem][ipHi].l)) == 1) )
							Atot += EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul;
						
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyErg = 
							(float)(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN * WAVNRYD*
						  EN1RYD);

						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyK = 
							(float)(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN * WAVNRYD*
						  TE1RYD );

						/* there are some negative energy transitions, where the order
						 * has changed, but these are not optically allowed, these are
						 * same n, different L, forbidden transitions */
						ASSERT( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN > 0. ||
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul <= iso.SmallA );

						/* very forbidden transitions may have negative energies */
						if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul < iso.SmallA ||
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN <= 0.)
						{
							
							/* this branch - bogus line */
							/* transition energy in various units:*/

							/* take abs value since some have order switched */
							/* make following an air wavelength */
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].WLAng = 1e6f;

							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gf = 1e-20f;

							/* derive the abs coef, call to function is gf, wl (A), g_low */
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].opacity = 0.;
						}
						else
						{

							/* transition energy in various units:*/

							/* take abs value since some have order switched */
							/* make following an air wavelength */
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].WLAng = 
								(float)fabs(1.0e8/
							  EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN/
							  RefIndex(&EmisLines[ipHE_LIKE][nelem][ipHi][ipLo]));
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gf = 
								(float)(GetGF(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul,
							  EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
							  EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gHi));
							ASSERT(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gf > 0.);

							/* derive the abs coef, call to function is gf, wl (A), g_low */
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].opacity = 
								(float)(abscf(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gf,
							  EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
							  EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gLo));
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].opacity = 
								MAX2(0.f, EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].opacity );
						}
						{
							/* option to print particulars of some line when called
							 * a prettier print statement is near where chSpin is defined below
							 * search for "pretty print" */
							/*@-redef@*/
							enum {DEBUG_LOC=FALSE};
							/*@+redef@*/
							if( DEBUG_LOC  )
							{
								if( (nelem == ipHELIUM) /*&& (EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul> 1e-19)*/ )
								{
									fprintf(ioQQQ,"Z %li Lo %li n %li s %li l %li \t ", 
										nelem+1,
										ipLo, 
										iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
										iso.quant_desig[ipHE_LIKE][nelem][ipLo].s,
										iso.quant_desig[ipHE_LIKE][nelem][ipLo].l );
									fprintf(ioQQQ," Hi %li n %li s %li l %li \t", 
										ipHi, 
										iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,
										iso.quant_desig[ipHE_LIKE][nelem][ipHi].s,
										iso.quant_desig[ipHE_LIKE][nelem][ipHi].l );
									fprintf(ioQQQ,"%.4e\t%.4e\tcs\t%.4e\n",
										/*EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,*/
										1.e8/EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
										EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul ,
										EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs);
								}
							}
						}

						/* redistribution function = partial or complete? */
						if( ipLo == ipHe1s1S )
						{
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].iRedisFun = ipPRD;
						}
						else
						{
							/* >>chng 01 feb 27, had been -1, crd with core only,
							 * change to crd with wings as per discussion with Ivan Hubeny */
							/*EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].iRedisFun = -1;*/
							EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].iRedisFun = ipCRDW;
						}
						/* create array index that will blow up */
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ipCont = INT_MIN;
						iso.Pop2Ion[ipHE_LIKE][nelem][ipLo] = 0.;

					}

					/* convert sum of As into proper lifetime in seconds */
					helike.Lifetime[nelem][ipHi] = 1./helike.Lifetime[nelem][ipHi];

					if ( ipHi < ipFirstCollapsed )
					{
						for( ipLo=ipHe1s1S; ipLo<ipHi ; ipLo++ )
						{
							if ( (Atot > 0.) && (iso.quant_desig[ipHE_LIKE][nelem][ipLo].s == iso.quant_desig[ipHE_LIKE][nelem][ipHi].s) && 
								(abs((int)(iso.quant_desig[ipHE_LIKE][nelem][ipLo].l - iso.quant_desig[ipHE_LIKE][nelem][ipHi].l)) == 1) )
								RobbinsP[ipHi][ipLo] = (double)EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul/Atot;
							else 
								RobbinsP[ipHi][ipLo] = 0.;
						}
						
						for( ipLo=ipHe1s1S; ipLo<=ipHi ; ipLo++ )
						{
							for ( i=ipLo /*+ 1*/; i<=ipHi; i++ )
							{
								RobbinsC[ipHi][ipLo] += RobbinsP[ipHi][i] * RobbinsC[i][ipLo];
							}
						}
					}
				}

				{
					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
						
					if( DEBUG_LOC && (nelem == ipHELIUM ) )
					{
						/* This bit is for the calculation of conversion probabilities, as in Robbins68b, Table 1.
						 * To output Bm(n,l; ipLo), set ipLo, lo_l, and lo_s accordingly.	*/
						long hi_l,hi_s;
						double Bm;
						
						/* tripS to 2^3P	*/
						hi_l = 0;
						hi_s = 1;
						ipLo = ipHe2p3P0;

						/* tripD to 2^3P	
						hi_l = 2;
						hi_s = 1;
						ipLo = ipHe2p3P0;*/

						/* tripP to 2^3S	*/
						/*hi_l = 1;
						hi_s = 1;
						ipLo = ipHe2s3S;	*/
						
						ASSERT( hi_l != iso.quant_desig[ipHE_LIKE][nelem][ipLo].l );
						
						fprintf(ioQQQ,"Bm(n,%ld,%ld;%ld)\n",hi_l,hi_s,ipLo);
						fprintf(ioQQQ,"m\t2\t\t3\t\t4\t\t5\t\t6\n");

						for( ipHi=ipHe2p3P2; ipHi<iso.numLevels[ipHE_LIKE][nelem]-iso.nCollapsed[ipHE_LIKE][nelem] ; ipHi++ )
						{
							/* Pick out excitations from metastable 2tripS to ntripP.	*/
							if ( (iso.quant_desig[ipHE_LIKE][nelem][ipHi].l == 1) && (iso.quant_desig[ipHE_LIKE][nelem][ipHi].s == 1) )
							{
								fprintf(ioQQQ,"\n%ld\t",iso.quant_desig[ipHE_LIKE][nelem][ipHi].n);
								j = 0;
								Bm = 0;
								for ( i = ipLo; i<=ipHi; i++)
								{
									if ( (iso.quant_desig[ipHE_LIKE][nelem][i].l == hi_l) && (iso.quant_desig[ipHE_LIKE][nelem][i].s == hi_s)  )
									{
										if ( (ipLo == ipHe2p3P0) && (i > ipHe2p3P2) )
										{
											Bm += RobbinsC[ipHi][i] * ( RobbinsP[i][ipHe2p3P0] + 
												RobbinsP[i][ipHe2p3P1] + RobbinsP[i][ipHe2p3P2] );
										}
										else
											Bm += RobbinsC[ipHi][i] * RobbinsP[i][ipLo];
											
										if ( (i == ipHe2p3P0) || (i == ipHe2p3P1) || (i == ipHe2p3P2) )
										{
											j++;
											if (j == 3)
											{
												fprintf(ioQQQ,"%2.4e\t",Bm);
												Bm = 0;
											}
										}
										else
										{
											fprintf(ioQQQ,"%2.4e\t",Bm);
											Bm = 0;
										}
									}
								}
							}
						}
						fprintf(ioQQQ,"\n\n");
					}
				}
				
				
				ipLo = ipHe1s1S;
				/* This ipHi does NOT represent ipHe2s1S, but the n in n^1P. */
				for( ipHi=2; ipHi <iso.nLyman[ipHE_LIKE]; ipHi++ )
				{
					double Enerwn , Aul /*,RI2,npstar */ ;
					float gHi , gLo;

					/* pointers to transitions, ipLo, ipHi
						* actual quantum numbers, nLo nHi
						* atomic number or charge and stage: */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].nelem = (int)(nelem+1);
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].IonStg = (int)(nelem);

					/* stat weight of all n ^1P is 3 */
					gHi = 3.f;

					gLo = 1.f;

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gLo = gLo;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gHi = gHi;

					/* hydrogenic energy, first ryd, convert to wn */
					Enerwn = EionWN[nelem-1] - RYD_INF*POW2((double)nelem)/POW2((double)ipHi);
					
					if ( nelem == ipHELIUM )
					{
						/* A simple fit for the calculation of Helium lyman Aul's.	*/
						Aul = (1.508e10) / pow((double)ipHi,2.975);
					}
					else 
					{
						/* Fit to values given in 
					 	 * >>refer	He-like	As	Johnson, W.R., Savukov, I.M., Safronova, U.I., & 
						 * >>refercon	Dalgarno, A., 2002, ApJS 141, 543J	*/
						/* originally astro.ph. 0201454  */
						Aul = 1.375E10 * pow((double)nelem, 3.9) / pow((double)ipHi,3.1);

					}

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Aul = (float)(Aul);

					/* the energy of the transition in wavenumbers, this can be negative
						* for highly forbidden upper levels where energy switches */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN = (float)Enerwn;

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyErg = 
						(float)(iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN * WAVNRYD*
						EN1RYD);

					/* the damping constant times the velocity */
					/*TODO code does keep track of lifetime - use that for damping constant */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].dampXvel = (float)(Aul/
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN/PI4);

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyK = 
						(float)(iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN * WAVNRYD*
						TE1RYD );

					/* transition energy in various units:*/

					/* take abs value since some have order switched */
					/* make following an air wavelength */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].WLAng = 
						(float)fabs(1.0e8/
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN/
						RefIndex(&iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi]));

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gf = 
						(float)(GetGF(iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Aul,
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN,
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gHi));

					/* derive the abs coef, call to function is gf, wl (A), g_low */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].opacity = 
						(float)(abscf(iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gf,
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].EnergyWN,
						iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].gLo));

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].iRedisFun = ipPRD;

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Pesc = 1.0;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Pelec_esc = 0.0;

					/* destruction probability*/
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Pdest = 0.0;

					/* upper and lower level populations */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].PopHi = 0.0;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].PopLo = 0.0;

					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].xIntensity = 0.0;

					/* following three will eventually be set by model atoms, but
						* must have reasonable value at start */
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].pump = 0.;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].cool = 0.;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].heat = 0.;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].ColOvTot = 0.;
					iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].PopOpc = 0.;
					{
						/* option to print particulars of some line when called
							* a prettier print statement is near where chSpin is defined below
							* search for "pretty print" */
						/*@-redef@*/
						enum {DEBUG_LOC=FALSE};
						/*@+redef@*/
						if( DEBUG_LOC )
						{
							fprintf(ioQQQ,"%li\t%li\t%.2e\t%.2e\n",
								nelem+1,
								ipHi,
								iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].Aul , 	
								iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi].opacity
								);
						}
					}
				}

				{
					/* This is an option to print statistically weighted sum of Auls out of a given n.	*/

					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
					if( DEBUG_LOC && (nelem == ipHELIUM) )
					{
						fprintf(ioQQQ,"nu \tnl \tll \tsl \tsum \n");
						for( j = 15; j <= 20; ++j)
						{
							for ( i = 3; i < j; ++i)
							{
								for ( ipLo = i*(i-1) + 1; ipLo <= i*(i+1); ++ipLo )
								{
									SumAul = 0.;
									
									for ( ipHi = j*(j-1) + 1; ipHi <= j*(j+1); ++ipHi )
									{
										/*SumAul += (double)EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul; */
										if (abs(iso.quant_desig[ipHE_LIKE][nelem][ipHi].l - iso.quant_desig[ipHE_LIKE][nelem][ipLo].l)==1)
										{
											SumAul += H_Einstein_A(iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,
												iso.quant_desig[ipHE_LIKE][nelem][ipHi].l,
												iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
												iso.quant_desig[ipHE_LIKE][nelem][ipLo].l,
												nelem); 
										}
			 						}
									fprintf(ioQQQ,"%ld \t%ld \t%ld \t%ld \t%.4e\n",
											iso.quant_desig[ipHE_LIKE][nelem][ipHi-1].n,
											iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
											iso.quant_desig[ipHE_LIKE][nelem][ipLo].l,
											iso.quant_desig[ipHE_LIKE][nelem][ipLo].s,
											SumAul);
								}	
							}
						}
					}
				}

				{
					for( i=2; i < iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem]; ++i)
					{
						/* Lifetimes should increase monotonically with increasing n...
						 * Make sure A's decrease.	*/
						ASSERT( SumAPerN[nelem][i] > SumAPerN[nelem][i+1] );
					}
				}

				{
					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
					if( DEBUG_LOC /* && (nelem == ipHELIUM) */)
					{
						for( i = 2; i<= (iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem]); ++i)
						{
							fprintf(ioQQQ,"n %ld\t lifetime %.4e\n", i, 1./SumAPerN[nelem][i]);
						}
					}
				}

				/* this is the two photon transition, designated by wl of 0 */
				EmisLines[ipHE_LIKE][nelem][ipHe2s1S][ipHe1s1S].WLAng = 0.;

				/* opacity in two-photon continuum is not correct since treated as line,
				 * fix now */
				EmisLines[ipHE_LIKE][nelem][ipHe2s1S][ipHe1s1S].opacity /= 1e4f;

				/* Lya has special redistribution function */
				EmisLines[ipHE_LIKE][nelem][ipHe2p1P][ipHe1s1S].iRedisFun = ipPRD;

			}
			/* we now have lifetimes for everything - set up true damping constants */
			for( ipHi=ipHe2s3S; ipHi<iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				for( ipLo=ipHe1s1S; ipLo<ipHi; ipLo++ )
				{
					/* the damping constant times the velocity */
					/*TODO code does keep track of lifetime - use that for damping constant */
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].dampXvel = (float)(
						1./helike.Lifetime[nelem][ipHi]/
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN/PI4);
				}
			}
		}
	}

	
	/* now establish radiative recombination rate coefficients 
	 * this is the vector or temperatures */
	for (i = 0; i < N_HE_TE_RECOMB; i++)
	{
		TeRRCoef[i] = 0.5*(i);
	}

	/* read in recombination coefficients */
	if ( !helike.lgCompileRecomb)
	{
		/* check on path if file not here and path set */
		/* path was parsed in getset */
		if( lgDataPathSet == TRUE )
		{
			/*path set, so look only there */
			strcpy( chFilename , chDataPath );
			strcat( chFilename , "he_iso_recomb.dat" );
		}
		else
		{
			/* path not set, check local space only */
			strcpy( chFilename , "he_iso_recomb.dat" );
		}

		if( trace.lgTrace )
			fprintf( ioQQQ," HeCreate opening he_iso_recomb.dat:");

		if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n" );
			if( lgDataPathSet == TRUE )
				fprintf( ioQQQ, " even tried path\n" );

			if( lgDataPathSet == TRUE )
			{
				fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n");
				fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
				fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
				fprintf( ioQQQ, " Defaulting to on-the-fly computation, ");
				fprintf( ioQQQ, "but the code runs much faster if you compile he_iso_recomb.dat!\n");
			}
			
			/* Do on the fly computation of R.R. Coef's instead.	*/
			for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
			{
				if( dense.lgElmtOn[nelem] )
				{
					/* Zero out the recombination sum array.	*/
					for(i = 0; i < N_HE_TE_RECOMB; i++)
					{
						TotalRecomb[nelem][i] = 0.;
					}
					
					/* NumHeLevRecomb[nelem] corresponds to n = 40 for He and 10 for ions, at present	*/
					/* There is no need to fill in values for collapsed levels, because we do not need to
					 * interpolate for a given temperature, just calculate it directly with a hydrogenic routine.	*/
					for( ipLo=ipHe1s1S; ipLo < iso.numLevels[ipHE_LIKE][nelem]-iso.nCollapsed[ipHE_LIKE][nelem]; ipLo++ )
					{
						EthRyd = energies[nelem][ipLo] * WAVNRYD;

						/* loop over temperatures to produce array of recombination coefficients	*/
						for (i = 0; i < N_HE_TE_RECOMB; i++)
						{
							/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
							RadRecombReturn = radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo);
							TotalRecomb[nelem][i] += RadRecombReturn;
							RRCoef[nelem][ipLo][i] = log10(RadRecombReturn);
						}
					}
					for (i = 0; i < N_HE_TE_RECOMB; i++)
					{
						for( i1 = ( (nelem == ipHELIUM) ? (HE_RREC_MAXN + 1) : (HE_LIKE_RREC_MAXN + 1) ); i1<=SumUpToThisN; i1++ )
						{
							TotalRecomb[nelem][i] += Recomb_Seaton59( nelem, pow(10.,TeRRCoef[i]), i1 );
						}
						TotalRecomb[nelem][i] = log10( TotalRecomb[nelem][i] );
					}
				}
			}
		}
		else 
		{
			/* check that magic number is ok */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			{
				fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}
			i = 1;
			i1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i4 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( i1 !=RECOMBMAGIC || i2 !=NumHeLevRecomb[ipHELIUM] || i3 !=NumHeLevRecomb[ipLITHIUM] || i4 !=N_HE_TE_RECOMB )
			{
				fprintf( ioQQQ, 
					" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
				fprintf( ioQQQ, 
					" HeCreate: I expected to find the numbers  %i %li %li %i and got %li %li %li %li instead.\n" ,
					RECOMBMAGIC ,
					NumHeLevRecomb[ipHELIUM],
					NumHeLevRecomb[ipLITHIUM],
					N_HE_TE_RECOMB,
					i1 , i2 , i3, i4 );
				fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
				fprintf( ioQQQ, 
					" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}

			/* now read in the data */
			for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
			{
				for( ipLo=ipHe1s1S; ipLo <= NumHeLevRecomb[nelem]; ipLo++ )
				{
					/* get next line image */
					if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					{
						fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
						puts( "[Stop in HeCreate]" );
						cdEXIT(EXIT_FAILURE);
					}
					/* each line starts with element and level number */
					i3 = 1;
					i1 = (long)FFmtRead(chLine,&i3,INPUT_LINE_LENGTH,&lgEOL);
					i2 = (long)FFmtRead(chLine,&i3,INPUT_LINE_LENGTH,&lgEOL);
					/* check that these number are correct */
					if( i1!=nelem || i2!=ipLo )
					{
						fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
						fprintf( ioQQQ, 
							" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
						puts( "[Stop in HeCreate]" );
						cdEXIT(EXIT_FAILURE);
					}

					/* loop over temperatures to produce array of recombination coefficients	*/
					for (i = 0; i < N_HE_TE_RECOMB; i++)
					{
						/* The last line for each element is the total recombination for each temp.	*/
						if( ipLo == NumHeLevRecomb[nelem] )
						{
							TotalRecomb[nelem][i] = FFmtRead(chLine,&i3,chLine_LENGTH,&lgEOL);
						}
						else
							RRCoef[nelem][ipLo][i] = FFmtRead(chLine,&i3,chLine_LENGTH,&lgEOL);

						if( lgEOL )
						{
							fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
							fprintf( ioQQQ, 
								" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
							puts( "[Stop in HeCreate]" );
							cdEXIT(EXIT_FAILURE);
						}
					}
				}

				/* following loop only executed if we need more levels than are
				 * stored in the recom coef data set
				 * do not do collapsed levels since will use H-like recom there */
				if( nelem == ipHELIUM || dense.lgElmtOn[nelem] ) 
				{
					for( ipLo=NumHeLevRecomb[nelem]; ipLo < iso.numLevels[ipHE_LIKE][nelem]-iso.nCollapsed[ipHE_LIKE][nelem]; ipLo++ )
				    {
					     EthRyd = energies[nelem][ipLo] * WAVNRYD;
					     for (i = 0; i < N_HE_TE_RECOMB; i++)
					     {
						     /* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
						     RRCoef[nelem][ipLo][i] = log10(radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo));
					     }
				     }
			     }
			}

			/* check that ending magic number is ok */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			{
				fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}
			i = 1;
			i1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i4 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			
			if( i1 !=RECOMBMAGIC || i2 !=NumHeLevRecomb[ipHELIUM] || i3 !=NumHeLevRecomb[ipLITHIUM] || i4 !=N_HE_TE_RECOMB )
			{
				fprintf( ioQQQ, 
					" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
				fprintf( ioQQQ, 
					" HeCreate: I expected to find the numbers  %i %li %li %i and got %li %li %li %li instead.\n" ,
					RECOMBMAGIC ,
					NumHeLevRecomb[ipHELIUM],
					NumHeLevRecomb[ipLITHIUM],
					N_HE_TE_RECOMB,
					i1 , i2 , i3, i4 );
				fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
				fprintf( ioQQQ, 
					" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}

			/* close the data file */
			fclose( ioDATA );
		}
	}
	else
	{
		/* option to create table of recombination coefficients,
		 * executed with the compile he-like command */
		FILE *ioRECOMB;
		
		/*RECOMBMAGIC the magic number for the table of recombination coefficients */
		/*NumHeLevRecomb[nelem] the number of levels that will be done */
		/* create recombination coefficients  */
		if( ( ioRECOMB = fopen( "he_iso_recomb.dat" , "w" ) ) == NULL )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat for writing.\n" );
			puts( "[Stop in HeCreate]" );
			cdEXIT(EXIT_FAILURE);
		}
		fprintf(ioRECOMB,"%i\t%li\t%li\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %li He levels, %li ion levels, and %i temperatures.\n",
			RECOMBMAGIC ,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB );

		for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
		{
			/* this must pass since compile he-like command reset numLevels to the macro */
			ASSERT( NumHeLevRecomb[nelem] <= iso.numLevels[ipHE_LIKE][nelem] );

			/* Zero out the recombination sum array.	*/
			for(i = 0; i < N_HE_TE_RECOMB; i++)
			{
				TotalRecomb[nelem][i] = 0.;
			}
			
			for( ipLo=ipHe1s1S; ipLo < NumHeLevRecomb[nelem] ; ipLo++ )
			{
				EthRyd = energies[nelem][ipLo] * WAVNRYD;

				fprintf(ioRECOMB, "%li\t%li", nelem, ipLo );
				/* loop over temperatures to produce array of recombination coefficients	*/
				for (i = 0; i < N_HE_TE_RECOMB; i++)
				{
					/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
					RadRecombReturn = radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo);
					TotalRecomb[nelem][i] += RadRecombReturn;
					RRCoef[nelem][ipLo][i] = log10(RadRecombReturn);
					fprintf(ioRECOMB, "\t%f", RRCoef[nelem][ipLo][i] );
				}
				fprintf(ioRECOMB, "\n" );
			}
			
			/* Store one additional line in he_iso_recomb.dat that gives the total recombination,
			 * as computed by the sum so far, plus levels up to SumUpToThisN using Seaton 59, 
			 * for each element and each temperature.	*/
			fprintf(ioRECOMB, "%li\t%li", nelem, NumHeLevRecomb[nelem] );
			for (i = 0; i < N_HE_TE_RECOMB; i++)
			{
				for( i1 = ( (nelem == ipHELIUM) ? (HE_RREC_MAXN + 1) : (HE_LIKE_RREC_MAXN + 1) ); i1<=SumUpToThisN; i1++ )
				{
					TotalRecomb[nelem][i] += Recomb_Seaton59( nelem, pow(10.,TeRRCoef[i]), i1 );
				}
				fprintf(ioRECOMB, "\t%f", log10( TotalRecomb[nelem][i] ) );
			}
			fprintf(ioRECOMB, "\n" );
		}
		/* end the file with the same information */
		fprintf(ioRECOMB,"%i\t%li\t%li\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %li He levels, %li ion levels, and %i temperatures.\n",
			RECOMBMAGIC ,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB );

		fclose( ioRECOMB );

		fprintf( ioQQQ, "HeCreate: compilation complete, he_iso_recomb.dat created.\n" );
		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}

	printstuff(printflag);

	/* free the memory	*/
	free(n_effs );

	/* create spectroscopic designation of labels */
	for( ipLo=ipHe1s1S; ipLo < iso.numLevels[ipHE_LIKE][ipHELIUM]-iso.nCollapsed[ipHE_LIKE][ipHELIUM]; ++ipLo )
	{
		sprintf( chLevel[ipLo] , "%li %c%c", 
				iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].n,
				chSpin[iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].s],
				chL[MIN2(5,iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].l)] );
	}

	/* option to create trace printout */
	if( (trace.lgTrace && trace.lgHeBug) )
	{

		/* "pretty print" only helium, O, and Fe 
		 * comments elsewhere say to search for "pretty print" */
		/* next print designation of levels */
		fprintf(ioQQQ,"These indices may not represent the same levels for each element!\n");
		fprintf(ioQQQ,"index\tdesig\tE(wn,He)\tE(wn,O)\tE(wn,Fe)\n");
		for( ipHi=1; ipHi<( MAX3(iso.numLevels[ipHE_LIKE][ipHELIUM],
			iso.numLevels[ipHE_LIKE][ipIRON],iso.numLevels[ipHE_LIKE][ipOXYGEN]) - iso.nCollapsed[ipHE_LIKE][ipOXYGEN] ); ++ipHi )
		{
			float enHe=0. , enO=0., enFe=0.;
			long nel = -1;
			if( dense.lgElmtOn[ipHELIUM] &&(ipHi < iso.numLevels[ipHE_LIKE][ipHELIUM]) )
			{
				enHe = EmisLines[ipHE_LIKE][ipHELIUM][ipHi][0].EnergyWN;
				nel = ipHELIUM;
			}
			else
			{
				enHe = 0.;
			}
			if( dense.lgElmtOn[ipOXYGEN] &&(ipHi < iso.numLevels[ipHE_LIKE][ipOXYGEN]) )
			{
				enO = EmisLines[ipHE_LIKE][ipOXYGEN][ipHi][0].EnergyWN;
				nel = ipOXYGEN;
			}
			else
			{
				enO = 0.;
			}
			if( dense.lgElmtOn[ipIRON] &&(ipHi < iso.numLevels[ipHE_LIKE][ipIRON]) )
			{
				enFe = EmisLines[ipHE_LIKE][ipIRON][ipHi][0].EnergyWN;
				nel = ipIRON;
			}
			else
			{
				enFe = 0.;
			}
			ASSERT( nel >0 && nel < LIMELM );
			fprintf(ioQQQ,"%li\t%li %c%c\t%.5e\t%.5e\t%.5e\n", 
				ipHi, 
				iso.quant_desig[ipHE_LIKE][nel][ipHi].n,
				chSpin[iso.quant_desig[ipHE_LIKE][nel][ipHi].s],
				chL[MIN2(5,iso.quant_desig[ipHE_LIKE][nel][ipHi].l)] , 
				enHe ,
				enO ,
				enFe 
				);
		
		}

		fprintf(ioQQQ,"lo\thi\t He states \n");
		for( ipHi=ipHe2s3S; ipHi<iso.numLevels[ipHE_LIKE][ipHELIUM] - iso.nCollapsed[ipHE_LIKE][ipHELIUM] ; ++ipHi )
		{
			for( ipLo=ipHe1s1S; ipLo<ipHi ; ++ipLo )
			{
				fprintf(ioQQQ," %li\t%li\t%li %c%c\t%li %c%c\n",
					ipLo , ipHi ,
					iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].n,
					chSpin[iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].s],
					chL[MIN2(5,iso.quant_desig[ipHE_LIKE][ipHELIUM][ipLo].l)] , 
					iso.quant_desig[ipHE_LIKE][ipHELIUM][ipHi].n,
					chSpin[iso.quant_desig[ipHE_LIKE][ipHELIUM][ipHi].s],
					chL[MIN2(5,iso.quant_desig[ipHE_LIKE][ipHELIUM][ipHi].l)]
					);
			}
		}
	}

	if( iso.numLevels[ipHE_LIKE][ipHELIUM] > ipHe2p1P )
	{
		/* some wavelengths of HeI are expected to be exact */
		EmisLines[ipHE_LIKE][ipHELIUM][ipHe2p3P0][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHE_LIKE][ipHELIUM][ipHe2p3P1][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHE_LIKE][ipHELIUM][ipHe2p3P2][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHE_LIKE][ipHELIUM][ipHe2p1P][ipHe2s1S].WLAng = 20600.;

		if(  iso.numLevels[ipHE_LIKE][ipHELIUM] > ipHe3d1D )
		{
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3s3S][ipHe2p3P0].WLAng = 7065.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3s3S][ipHe2p3P1].WLAng = 7065.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3s3S][ipHe2p3P2].WLAng = 7065.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3p3P][ipHe2s3S].WLAng = 3889.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3d3D][ipHe2p3P0].WLAng = 5876.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3d3D][ipHe2p3P1].WLAng = 5876.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3d3D][ipHe2p3P2].WLAng = 5876.;
			EmisLines[ipHE_LIKE][ipHELIUM][ipHe3d1D][ipHe2p1P].WLAng = 6678.;

			if(  iso.numLevels[ipHE_LIKE][ipHELIUM] > ipHe4d3D )
			{
				EmisLines[ipHE_LIKE][ipHELIUM][ipHe4d3D][ipHe2p3P0].WLAng = 4471.;
				EmisLines[ipHE_LIKE][ipHELIUM][ipHe4d3D][ipHe2p3P1].WLAng = 4471.;
				EmisLines[ipHE_LIKE][ipHELIUM][ipHe4d3D][ipHe2p3P2].WLAng = 4471.;
			}
		}
	}

	{
		/* prints table of wavelengths of lines to ground */
		/*@-redef@*/
		enum {PRINTIT=FALSE};
		/*@+redef@*/
		if( PRINTIT )
		{
			for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
			{
				if( dense.lgElmtOn[nelem] )
				{
					fprintf(ioQQQ,"%li\t%s\t",nelem+1,elementnames.chElementSym[nelem] );

					ipLo = ipHe1s1S;
					for( ipHi=ipLo+1; ipHi<=ipHe2p1P; ++ipHi )
					{
						if( ipHi==ipHe2s1S || ipHi==ipHe2p3P0 ) continue;
						fprintf(ioQQQ,"%.4f\t", EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].WLAng);
					}
					fprintf(ioQQQ,"\n");
				}
			}
			cdEXIT(EXIT_SUCCESS);
		}
	}

	/* get the collision strength data for the He 1 lines */

	/* check on path if file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , "he1_cs.dat" );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , "he1_cs.dat" );
	}

	if( trace.lgTrace )
		fprintf( ioQQQ," HeCreate opening he1_cs.dat:");

	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not open he1_cs.dat\n" );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " HeCreate could not open he1_cs.dat\n");
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not read first line of he1_cs.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}
	i = 1;
	i1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	i2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
	i3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* the following is the set of numbers that appear at the start of level1.dat 01 02 20 */
	if( ( i1 != 02 ) || ( i2 != 02 ) || ( i3 != 25 ) )
	{
		fprintf( ioQQQ, 
			" HeCreate: the version of he1_cs.dat is not the current version.\n" );
		fprintf( ioQQQ, 
			" I expected to find the number 02 02 25 and got %li %li %li instead.\n" ,
			i1 , i2 , i3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* get the array of temperatures */
	lgHIT = FALSE;
	while( fgets( chLine , (int)sizeof(chLine) , ioDATA ) != NULL )
	{
		/* only look at lines without '#' in first col */
		if( chLine[0] != '#')
		{
			lgHIT = TRUE;
			helike.nCS = 0;
			lgEOL = FALSE;
			i = 1;
			while( !lgEOL && helike.nCS < HE1CSARRAY)
			{
				helike.CSTemp[helike.nCS] = FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
				++helike.nCS;
			}
			break;
		}
	}
	--helike.nCS;
	if( !lgHIT )
	{
		fprintf( ioQQQ, " HeCreate could not find line in CS temperatures in he1_cs.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* create space for array of CS values, if not already done */
	if( (helike.HeCS = (float ****)MALLOC(sizeof(float ***)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
	{
		/* TODO - this structure is currently only used for helium itself...
		 * stuff numbers in for other elements, or drop the [nelem] dimension off
		 * of helike.HeCS	*/
		if( nelem != ipHELIUM )
			continue;

		/* only grab core for elements that are turned on */
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem])
		{
			if( (helike.HeCS[nelem] = (float***)MALLOC(sizeof(float**)*(unsigned)(iso.numLevels[ipHE_LIKE][nelem]) ))==NULL )
				BadMalloc();

			/* avoid allocating 0 bytes, some OS return NULL pointer, PvH */
			helike.HeCS[nelem][0] = NULL;
			for( ipHi=ipHe2s3S; ipHi < iso.numLevels[ipHE_LIKE][nelem] ;++ipHi )
			{
				if( (helike.HeCS[nelem][ipHi] = (float**)MALLOC(sizeof(float*)*(unsigned)ipHi ))==NULL )
					BadMalloc();

				for( ipLo=ipHe1s1S; ipLo<ipHi; ++ipLo )
				{
					if( (helike.HeCS[nelem][ipHi][ipLo] = (float*)MALLOC(sizeof(float)*(unsigned)helike.nCS ))==NULL )
						BadMalloc();

					for( i=0; i<helike.nCS; ++i )
					{
						helike.HeCS[nelem][ipHi][ipLo][i] = 0.f;
					}
				}
			}
		}
	}

	/* now read in the CS data */
	lgHIT = FALSE;
	nelem = ipHELIUM;
	while( fgets( chLine , (int)sizeof(chLine) , ioDATA ) != NULL )
	{
		char *chTemp;
		/* only look at lines without '#' in first col */
		if( (chLine[0] == ' ') || (chLine[0]=='\n') )
			break;
		if( chLine[0] != '#')
		{
			lgHIT = TRUE;

			/* get lower and upper level index */
			j = 1;
			ipLo = (long)FFmtRead(chLine,&j,INPUT_LINE_LENGTH,&lgEOL);
			ipHi = (long)FFmtRead(chLine,&j,INPUT_LINE_LENGTH,&lgEOL);
			ASSERT( ipLo < ipHi );
			if( ipHi >= iso.numLevels[ipHE_LIKE][nelem] )
				continue;

			chTemp = chLine;
			/* skip over 4 tabs to start of cs data */
			for( i=0; i<3; ++i )
			{
				if( (chTemp = strchr( chTemp, '\t' )) == NULL )
				{
					fprintf( ioQQQ, " HeCreate no init cs\n" );
					puts( "[Stop in HeCreate]" );
					cdEXIT(EXIT_FAILURE);
				}
				++chTemp;
			}

			for( i=0; i<helike.nCS; ++i )
			{
				float a;
				if( (chTemp = strchr( chTemp, '\t' )) == NULL )
				{
					fprintf( ioQQQ, " HeCreate not scan cs\n" );
					puts( "[Stop in HeCreate]" );
					cdEXIT(EXIT_FAILURE);
				}
				++chTemp;
				sscanf( chTemp , "%e" , &a );
				helike.HeCS[nelem][ipHi][ipLo][i] = a;
			}
		}
	}

	/* close the data file */
	fclose( ioDATA );

	/* option to print cs data for AGN */
	/* create spectroscopic designation of labels */
	{
		/* option to print particulars of some line when called
		 * a prettier print statement is near where chSpin is defined below*/
		/*@-redef@*/
		enum {AGN=FALSE};
		/*@+redef@*/
		if( AGN )
		{
#			define NTEMP 6
			double te[NTEMP]={6000.,8000.,10000.,15000.,20000.,25000. };
			double telog[NTEMP] ,
				cs ,
				ratecoef;
			nelem = ipHELIUM;
			fprintf( ioQQQ,"trans");
			for(i=0; i<NTEMP; ++i)
			{
				telog[i] = log10( te[i] );
				fprintf( ioQQQ,"\t%.3e",te[i]);
			}
			for(i=0; i<NTEMP; ++i)
			{
				fprintf( ioQQQ,"\t%.3e",te[i]);
			}
			fprintf(ioQQQ,"\n");

			for( ipHi=ipHe2s3S; ipHi< iso.numLevels[ipHE_LIKE][ipHELIUM]; ++ipHi )
			{
				for( ipLo=ipHe1s1S; ipLo < ipHi; ++ipLo )
				{
					/* print the designations of the lower and upper levels */
					fprintf( ioQQQ,"%s - %s",
						chLevel[ipLo] , chLevel[ipHi] );

					/* print the interpolated collision strengths */
					for(i=0; i<NTEMP; ++i)
					{
						phycon.alogte = (float)telog[i];
						/* print cs */
						cs = HeCSInterp( nelem , ipHi , ipLo );
						fprintf(ioQQQ,"\t%.2e", cs );
					}

					/* print the rate coefficients */
					for(i=0; i<NTEMP; ++i)
					{
						phycon.alogte = (float)telog[i];
						phycon.te = (float)pow(10.,telog[i] );
						tfidle(FALSE);
						cs = HeCSInterp( nelem , ipHi , ipLo );
						/* collisional deexcitation rate */
						ratecoef = cs/sqrt(phycon.te) *8.629e-6/iso.stat[ipHE_LIKE][nelem][ipLo] *
							sexp( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyK / phycon.te );
						fprintf(ioQQQ,"\t%.2e", ratecoef );
					}
					fprintf(ioQQQ,"\n");
				}
			}
			cdEXIT(EXIT_FAILURE);
		}
	}

	for( ipLo=ipHe1s1S; ipLo < max_num_levels ;++ipLo )
	{
		free( RobbinsC[ipLo] );
		free( RobbinsP[ipLo] );
	}
	for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
	{
		if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
		{
			free( energies[nelem] );
			free( SumAPerN[nelem] );
		}
	}
	free( energies );
	free( SumAPerN );
	
	free( RobbinsC );
	free( RobbinsP );

	free( NumHeLevRecomb );

#	ifdef DEBUG_FUN
	fputs( " <->HeCreate()\n", debug_fp );
#	endif
	return;
}

/*===================================================================================*/
/* HeRecom do recomb coef for He, called by HeLike */
static void HeRecom( 
			/* nelem on the c scale, He is 1 */
			long nelem )
{
	static double ditcrt[LIMELM]= {0.,5e4,1.20E+04,1.10E+04,4.40E+05,7.00E+05,1.00E+06,1.50E+06,
	  1.50E+06,3.80E+06,3.40E+04,4.00E+06,5.50E+06,6.30E+06,5.00E+06,9.00E+06,1.40E+07,
	  1.40E+07,2.00E+05,2.00E+05,2.00E+05,1.00E+05,7.00E+04,4.00E+04,6.00E+06,1.00E+11,
	  1.00E+20,1.00E+20,1.00E+20,1.00E+20};

	static double dicoef0[LIMELM]= {0.,1.9e-3,6.15E-03,1.62E-03,4.78E-02,3.22E-02,4.61E-02,6.23E-02,
	  2.52E-01,1.44E-01,1.80E-01,1.44E-01,3.40E+00,1.89E-01,4.02E-01,2.41E-01,1.10E+01,2.97E-01,
	  5.49E-01,3.55E-01,2.30E-01,1.40E-01,1.10E-01,4.10E-02,7.47E-01,5.19E-01,5.75E-01,5.75E-01,
	  5.75E-01,5.75E-01};

	static double dite0[LIMELM]= {0.,4.7e5,1.41E+05,8.19E+04,3.44E+06,4.06E+06,5.44E+06,7.01E+06,1.40E+07,
	  1.50E+07,1.30E+07,1.50E+07,1.70E+07,1.99E+07,2.41E+07,2.54E+07,2.80E+07,3.13E+07,3.65E+07,
	  3.78E+07,1.09E+06,9.62E+05,7.23E+05,4.23E+05,5.84E+07,6.01E+07,6.81E+07,6.81E+07,6.81E+07,
	  6.81E+07};

	static double dicoef1[LIMELM]= {0.,0.3,5.88E-02,3.43E-01,3.62E-01,3.15E-01,3.09E-01,3.04E-01,
	  3.04E-01,2.96E-01,0.00E+00,2.91E-01,0.00E+00,2.86E-01,2.98E-01,2.81E-01,0.00E+00,2.77E-01,
	  2.92E-01,2.75E-01,3.60E+00,4.90E+00,1.60E+00,4.20E+00,2.84E-01,2.79E-01,0.00E+00,2.86E-01,
	  2.86E-01,2.86E-01};

	static double dite1[LIMELM]= {0.,9.4e4,1.41E+05,1.59E+05,5.87E+05,8.31E+05,1.14E+06,1.47E+06,
	  1.80E+06,2.24E+06,0.00E+00,3.09E+06,0.00E+00,4.14E+06,4.67E+06,5.30E+06,0.00E+00,6.54E+06,
	  7.25E+06,7.68E+06,1.10E+07,8.34E+06,1.01E+07,1.07E+07,1.17E+07,9.97E+06,0.00E+00,9.08E+06,
	  9.08E+06,9.08E+06};

	static long RUNONCE = FALSE;
	long n, ipFirstCollapsed, LastN=0L, ThisN=1L; 
	double RecUnitVolTotal, topoff1, topoff3, TotMinusExplicitResolved,
		TotRRThisN=0., TotRRLastN=0.;
	static double RecExplictLevels1[LIMELM], RecExplictLevels3[LIMELM],
		TotalRadRecomb[LIMELM], RecCollapsed[LIMELM];

#	ifdef DEBUG_FUN
	fputs( "<+>HeRecom()\n", debug_fp );
#	endif

	/* evaluate recombination escape probability for all levels */

	/* define radiative recombination rates for all levels */ 
	/* this will be the sum of all levels explicitly included in the model atom */
	RecExplictLevels1[nelem] = 0.;
	RecExplictLevels3[nelem] = 0.;

	/* this is the number of resolved levels, so first collapsed level is [ipFirstCollapsed] */
	ipFirstCollapsed = iso.numLevels[ipHE_LIKE][nelem]-iso.nCollapsed[ipHE_LIKE][nelem]; 

	if( helike.lgSetBenjamin && (nelem == ipHELIUM) )
        /* total topoff to Benjamin numbers */
		TotalRadRecomb[nelem] = 4.27E-13 * pow(10000./phycon.te, 0.678);
	else
		TotalRadRecomb[nelem] = He_RRCoef_Te( nelem , ipFirstCollapsed );

	for( n=0; n<ipFirstCollapsed; ++n )
	{
		double BenDirectRadRecHe[31] = {
		1.54E-13,1.49E-14,5.55E-15,5.61E-14,5.61E-14,5.61E-14,1.26E-14,3.72E-15,1.62E-15,
		1.95E-14,1.33E-14,4.31E-15,5.01E-15,1.50E-15,7.00E-16,9.04E-15,8.29E-15,2.67E-15,
		4.15E-15,1.38E-15,2.43E-15,7.51E-16,3.66E-16,4.84E-15,4.99E-15,1.60E-15,3.65E-15,
		1.22E-15,1.34E-15,4.46E-16,1.35E-15};

		double BenDirectRadExpHe[31] = {
		0.486,0.381,0.451,0.639,0.639,0.639,0.695,0.344,0.444,0.632,0.868,0.872,0.700,0.328,0.445,
		0.632,0.867,0.871,1.046,1.046,0.708,0.327,0.454,0.636,0.870,0.873,1.048,1.048,1.183,1.183,0.718};

		double BenIndirectRadRecHe[31] = {
		0.      ,7.07E-15,2.25E-15,1.01E-14,1.01E-14,1.01E-14,3.11E-15,1.51E-15,6.09E-16,
		3.63E-15,8.19E-15,2.64E-15,1.30E-15,5.16E-16,2.44E-16,1.62E-15,5.19E-15,1.64E-15,
		6.94E-15,2.31E-15,6.50E-16,2.09E-16,1.16E-16,8.63E-16,3.21E-15,9.86E-16,5.83E-15,
		1.94E-15,8.39E-15,2.80E-15,3.82E-16};

		double BenIndirectRadExpHe[31] = {
		0.000,0.675,0.766,0.901,0.901,0.901,0.879,0.696,0.782,0.873,1.097,1.114,0.876,0.701,0.783,
		0.845,1.078,1.106,1.271,1.271,0.854,0.711,0.786,0.813,1.055,1.096,1.267,1.267,1.375,1.375,0.832};

		double BenIndirectDensExpHe[31] = {
		0.000,-0.001,0.000,0.009,0.009,0.009,0.008,-0.001,0.000,0.010,0.002,0.003,0.008,
		-0.001,0.000,0.010,0.002,0.003,-0.001,-0.001,0.008,-0.001,0.000,0.010,0.002,0.003,
		-0.001,-0.001,-0.007,-0.007,0.008};

		if( helike.lgSetBenjamin && (nelem == ipHELIUM) && ( n <= 30 ) 
			&& ( phycon.te < 1000000.1f ) && ( phycon.te > 99.9f ) )
		{
			iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] = 
				(float)BenDirectRadRecHe[n] * (float)pow( 10000./phycon.te, BenDirectRadExpHe[n] ) +
				(float)BenIndirectRadRecHe[n] * (float)pow( 10000./phycon.te, BenIndirectRadExpHe[n] )
				* (float)pow( dense.eden/100., BenIndirectDensExpHe[n] );

			if ( (n == ipHe2p3P0) || (n == ipHe2p3P1) || (n == ipHe2p3P2) )
			{
				/* Split 2 trip P term.	*/
				iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] *= (2.f*(n-3L)+1.f) / 9.f;
			}
		}
		else
		{
			/* this is radiative recombination rate coefficient */
			iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] = (float)He_RRCoef_Te( nelem , n );
		}

		/* do not include ground in this sum	*/
		if( (iso.quant_desig[ipHE_LIKE][nelem][n].s == 0) && (n>0) )
		{
			RecExplictLevels1[nelem] += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad];
		}
		else if( iso.quant_desig[ipHE_LIKE][nelem][n].s == 1 )
		{
			RecExplictLevels3[nelem] += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad];
		}
		
		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] > 0. );
	}

	RecCollapsed[nelem] = 0.;
	for( n=ipFirstCollapsed; n<iso.numLevels[ipHE_LIKE][nelem]; ++n )
	{
		long int nup = iso.quant_desig[ipHE_LIKE][nelem][n].n;

		/* this is radiative recombination rate coefficient */
		iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] = (float)hrfit(nelem,nup,phycon.te);
		RecCollapsed[nelem] += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad];

		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad] > 0. );
	}

	for( n = 0; n<iso.numLevels[ipHE_LIKE][nelem]; n++ )
	{

		if( iso.quant_desig[ipHE_LIKE][nelem][n].n == ThisN )
		{
			TotRRThisN += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad];
		}
		else
		{
			ASSERT( (TotRRThisN<TotRRLastN) || (ThisN<=2L) || (phycon.te>3E7) || (nelem!=ipHELIUM) );
			LastN = ThisN;
			ThisN = iso.quant_desig[ipHE_LIKE][nelem][n].n;
			TotRRLastN = TotRRThisN;
			TotRRThisN = iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad];

			{
				/* Print the sum of recombination coefficients per n at current temp.	*/
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				
				if( DEBUG_LOC && !RUNONCE )
				{
					static long FIRSTTIME = TRUE;

					if( FIRSTTIME )
					{
						fprintf( ioQQQ,"Sum of Radiative Recombination at current temp, T=%.2f\n", phycon.te);
						FIRSTTIME= FALSE;
					}
					
					fprintf( ioQQQ,"%li\t%.2e\n",LastN,TotRRLastN );
				}
			}
		}
	}

	RUNONCE = TRUE;

	/* this is case B recombination, sum without the ground included */
	iso.RadRec_caseB[ipHE_LIKE][nelem] = (float)TotalRadRecomb[nelem] - iso.RadRecomb[ipHE_LIKE][nelem][0][ipRecRad];
	ASSERT( iso.RadRec_caseB[ipHE_LIKE][nelem] > 0.);

	/* this is used for radiative recombination cooling in coolr, this does not include
	 * optical depth effects - may be reevaluated in level solver with opt dep included */
	ionrec.RadRecomRateCoef[nelem][nelem-1] = TotalRadRecomb[nelem];

	/* >>chng 02 jan 07, do not add three body, complete model atom will automatically include it
	 * problem is that most elements will not go to high n */
	/* add three-body recombination, convert to per unit vol 
	RecUnitVolTotal = dense.eden*(TotalRadRecomb[nelem] + ionrec.CotaRate[nelem-1]);*/
	/* convert to per unit vol */
	RecUnitVolTotal = dense.eden*TotalRadRecomb[nelem];

	/* recombinatinon on grain surface */
	RecUnitVolTotal += ionrec.GrnIonRec;

	/* add on DR */
	if( nelem==ipIRON )
	{
		/* for case of iron, add on Dima's fits to diel part, above was zero */
		float DieRate;
		drfe(25,phycon.te,&DieRate);
		RecUnitVolTotal += DieRate*dense.eden;
	}
	else if( phycon.te > (float)(ditcrt[nelem]*0.1) )
	{
		/* Burgess dielectronic recombination */
		double diburg , t;
		/* the suppression factor is on a charge scale, so rec to neutral
		 * is always 0, hence the nelem-1 */
		if( nelem == ipHELIUM )
		{

			/* this is the dr rate that was in the original He singlets coding, and has been
			 * part of cloudy since the beginning - based on Aldrovandi & Pequignot */
			/* >>refer He	DR	Aldrovandi, S.M.V., & Pequignot, D. 1973, A&A 25, 137	*/
			if( phycon.te > 5e4 )
			{
				t = (float)(1.9e-3/phycon.te32*sexp(4.7e5/phycon.te)*(1. + 0.3*sexp(9.4e4/phycon.te)));
				/* >>chng 96 oct 22, first coef had been 4.5 below, change to 4.7 as
				*in aldrovandi and pequignot
				*  1    1.9e-3/te32*sexp(4.5e5/te)*(1.+0.3*sexp(9.4e4/te))*0. */
			}
			else
				t = 0;
			diburg = t;
		}
		else
		{
			diburg = ionrec.DielSupprs[0][nelem-1]/phycon.te32*dicoef0[nelem]*
				exp(-dite0[nelem]/phycon.te)*(1. + dicoef1[nelem]*sexp(dite1[nelem]/phycon.te));
		}
		RecUnitVolTotal += diburg* dense.eden;
	}

	/* >>chng 02 may 05, move this to here, had been before DR block, and so did not include it */
	/* save the rate */
	/*ionrec.RecombinRate[nelem][nelem-1] = (float)RecUnitVolTotal;*/

	/* at this point we have two numbers, RecExplictLevels1[nelem], RecExplictLevels3[nelem], 
	 * the sum of radiative recombination to all levels explicitly included in the model atom, 
	 * and RecUnitVolTotal, the total recombination rate.  The difference is the amount of 
	 * "topoff" we will need to do */

	/* Don't topoff if using Benjamin command, or if all levels below continuum are being considered.	*/
	if( !helike.lgSetBenjamin && !helike.lgFullSize[nelem] )
	{
		/* According to Hummer & Storey 98, the ratio of total case b singlets to triplets (at 10000K) is 1./3.37...
		 * but they also show in table 3, that at n=25, the values to l=3 (and presumably beyond) are (respectively)
		 * .25 or .75 of the corresponding H values.  So it would be more accurate to take .25 and .75 of the remainder
		 * rather than assuming .25 and .75 of the total fall into singlets and triplets.	*/
		TotMinusExplicitResolved = RecUnitVolTotal/dense.eden - iso.RadRecomb[ipHE_LIKE][nelem][0][ipRecRad] - RecExplictLevels1[nelem] 
			- RecExplictLevels3[nelem];
		topoff1 = TotMinusExplicitResolved*0.25;
		topoff3 = TotMinusExplicitResolved*0.75;
		/* the rrfit fits are too small at high temperatures, so this atom is
		 * better than the topoff.  Only a problem for helium itself, at high temperatures,
		 * when physics disabled with the atom he-like benjamin command.
		 * complain if the negative topoff is not for this case */
		/* >>chng 02 jul 30, do not print if benjamin is set - this causes several known problems */
		if( MIN2(topoff1,topoff3) < 0. && (nelem!=ipHELIUM || phycon.te < 1e5 ) )
		{
			fprintf(ioQQQ," PROBLEM: negative topoff for %li, rel error was %.2e %.2e\n", 
				nelem, 
				topoff1/RecExplictLevels1[nelem] ,
				topoff3/RecExplictLevels3[nelem] 
				);
		}
		
		/* >>chng 02 feb 24...now always have at least one collapsed level...so
		 * this bit greatly simplifies.	*/
		iso.RadRecomb[ipHE_LIKE][nelem][iso.numLevels[ipHE_LIKE][nelem]-1][ipRecRad] += 
			(float)MAX2(0., TotMinusExplicitResolved - RecCollapsed[nelem] );
	}

	/*fprintf(ioQQQ,"He1 topoff / case b = %.2e\n", sumtopoff / iso.RadRec_caseB[ipHE_LIKE][nelem] );*/
	
	/* total continuum effective escape prob for all levels  */
	/* RadRec_effec is total effective radiative recombination */
	iso.RadRec_effec[ipHE_LIKE][nelem] = 0.;

	for( n=0; n<iso.numLevels[ipHE_LIKE][nelem]; ++n )
	{
		/* this is escape probability */
		iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc] = 
			(float)receff(iso.ipIsoLevNIonCon[ipHE_LIKE][nelem][n]);

		/* correct above value for possible otsmin value
		 * otsmin set to zero in zerologic, set to 1 with 
		 * the NO ON THE SPOT command */
		iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc] = 
			(float)MAX2(iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc], opac.otsmin);

		/* net escape prob includes dest by background opacity */
		iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecNetEsc] = (float)
			MIN2(1.,iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc]+
		  (1.-iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc])*iso.ConOpacRatio[ipHE_LIKE][nelem][n]);

		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecEsc] >= 0. );
		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecNetEsc] >= 0. );

		/* sum of all effective rad rec */
		iso.RadRec_effec[ipHE_LIKE][nelem] += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad]*
		  iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecNetEsc];
	}
#if 0
	{
#include "radius.h"
	if( nelem==1 )
	fprintf(ioQQQ,"newwww\t%.5e\t%.3e\t%.3e\n", 
		radius.depth,
		iso.RadRecomb[ipHE_LIKE][nelem][0][ipRecEsc],
		iso.RadRecomb[ipHE_LIKE][nelem][0][ipRecNetEsc]);
	}
#endif

	/* option for case b conditions, kill ground state recombination */
	if( opac.lgCaseB )
	{
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecEsc] = 1e-10f;
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecNetEsc] = 1e-10f;
	}

	return;

#	ifdef DEBUG_FUN
	fputs( " <->HeRecom()\n", debug_fp );
#	endif
}

/*===================================================================================*/
/* evaluate collisional rates */
static void HeCollid( long nelem)
{

	double factor , ConvLTEPOP, *ColIonizPerN;
	long int ipLo , ipHi, n, i;
	float DimaRate, crate;
	static float TeUsed[LIMELM]={0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
		0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0};

#	ifdef DEBUG_FUN
	fputs( "<+>HeCollid()\n", debug_fp );
#	endif

	if( (ColIonizPerN = (double*)MALLOC(sizeof(double)*
		(unsigned)(iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] + 1) ) )==NULL )
		BadMalloc();

	for( n=0; n <= ( iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] ); ++n)
	{
		ColIonizPerN[n] = 0.;
	}

	/*lint -e777 */
	if( TeUsed[nelem] != phycon.te )
	{
	/*lint +e777 */
		TeUsed[nelem] = phycon.te;

		/* following factor is actually 4.1412957e-16 (old=4.14158E-16), 
		 * but e- stat weight is included */
		/* ConvLTEPOP = 2.0708e-16/phycon.te32;*/
		/* >>chng 99 jun 02, use codata and infinite mass nuc 
		 * actually correct reduced mass appears below */
		/* HION_LTE_POP	is planck^2 / (2 pi m_e k ), must raise this to 3/2 when used */
		factor = HION_LTE_POP*AtomcWgt.AtomicWeight[nelem]/
			(AtomcWgt.AtomicWeight[nelem]+ELECTRON_MASS/ATOMIC_MASS_UNIT) ;

		/* term in () is stat weight of electron * ion */
		ConvLTEPOP = pow(factor,1.5)/(2.*iso.stat_ion[ipHE_LIKE])/phycon.te32;

		iso.lgPopLTE_OK[ipHE_LIKE][nelem] = TRUE;
		/* fully define Boltzmann factors to continuum for model levels */
		for( ipLo=ipHe1s1S; ipLo < (iso.numLevels[ipHE_LIKE][nelem]); ipLo++ )
		{
			/* this boltzmann factor is exp( +ioniz energy / Te ) */
			iso.ConBoltz[ipHE_LIKE][nelem][ipLo] = 
				sexp(iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo]/phycon.te_ryd);

			if( iso.ConBoltz[ipHE_LIKE][nelem][ipLo] < SMALLDOUBLE )
			{
				iso.ConBoltz[ipHE_LIKE][nelem][ipLo] = 0.;
			}

			/***********************************************************************
			 *                                                                     *
			 * LTE abundances for all levels, ionization and                       *
			 * exciation between levels                                            *
			 *                                                                     *
			 ***********************************************************************/

			if( iso.ConBoltz[ipHE_LIKE][nelem][ipLo] > 1e-100 )
			{
				iso.PopLTE[ipHE_LIKE][nelem][ipLo] = 
					iso.stat[ipHE_LIKE][nelem][ipLo] / iso.ConBoltz[ipHE_LIKE][nelem][ipLo]* ConvLTEPOP;
			}
			else
			{
				iso.PopLTE[ipHE_LIKE][nelem][ipLo] = 0.;
			}

			/* now check for any zeros - possible above underflowed to zero */
			if( iso.PopLTE[ipHE_LIKE][nelem][ipLo] <= 0. )
			{
				iso.lgPopLTE_OK[ipHE_LIKE][nelem] = FALSE;
			}
		}

		for( ipHi=ipHe2s3S; ipHi <iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
		{
			for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
			{
				/* get collision strength */
  				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs = HeCSInterp( nelem , ipHi , ipLo );
				{
					/* option to print particulars of some line when called
					 * a prettier print statement is near where chSpin is defined below*/
					/*@-redef@*/
					enum {DEBUG_LOC=FALSE};
					/*@+redef@*/
					if( DEBUG_LOC && (nelem==ipHELIUM) /*&& (iso.quant_desig[ipHE_LIKE][nelem][ipHi].n == 15) 
						&& (iso.quant_desig[ipHE_LIKE][nelem][ipLo].n >= 14)*/ )
					{
						fprintf(ioQQQ,"Lo %li n %li s %li l %li \t ", 
							ipLo, 
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].s,
							iso.quant_desig[ipHE_LIKE][nelem][ipLo].l );
						fprintf(ioQQQ," Hi %li n %li s %li l %li \t", 
							ipHi, 
							iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,
							iso.quant_desig[ipHE_LIKE][nelem][ipHi].s,
							iso.quant_desig[ipHE_LIKE][nelem][ipHi].l );
						fprintf(ioQQQ,"%.4e\t%.0f\t%.0f\t%.4e\t%.4e\t%.4e\n",
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
							iso.stat[ipHE_LIKE][nelem][ipHi],
							iso.stat[ipHE_LIKE][nelem][ipLo],
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul,
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gf ,
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs );
					}
				}

				/* collisional deexcitation rate, can be turned off with 
				 * atom he-like collisions off command */
				/*This is downward collision rate */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ColUL = 
					(float)(EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs/
					phycon.sqrte*8.629e-6/(double)iso.stat[ipHE_LIKE][nelem][ipHi] );

				/* define line Boltzmann factors 
				 * some energy-degenerate collisions have very small negative energy,
				 * when different l, same n, levels flip over */
				if( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyK > 0. )
				{
					iso.Boltzmann[ipHE_LIKE][nelem][ipHi][ipLo] = 
						sexp( EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyK / phycon.te );
				}
				else
				{
					/* negative energy, assume h nu/kT is very small */
					iso.Boltzmann[ipHE_LIKE][nelem][ipHi][ipLo] = 1.;
				}
			}
			/* collisional ionization
			 * this is rate coefficient, not rate, for hydrogenic species */
			/*TODO find ioniz rates for helium like species, these are hydrogenic rates */
			ASSERT( ipHi > 0 && ipHi < iso.numLevels[ipHE_LIKE][nelem]);
			if( nelem==ipHELIUM && helike.lgSetBenjamin )
			{
				/* atoms */
				
					double BenCollIonParamA[31] = {
						3.36E-09,9.36E-08,1.33E-07,1.58E-07,1.58E-07,1.58E-07,1.81E-07,5.36E-07,
						6.58E-07,7.23E-07,7.81E-07,7.81E-07,7.93E-07,1.62E-06,1.87E-06,2.00E-06,
						2.11E-06,2.11E-06,2.11E-06,2.11E-06,2.13E-06,3.61E-06,4.02E-06,4.21E-06,
						4.39E-06,4.40E-06,4.40E-06,4.40E-06,4.40E-06,4.40E-06,4.44E-06};

					double BenCollIonParamB[31] = {
						0.499,0.280,0.253,0.240,0.240,0.240,0.229,0.138,0.120,0.111,0.104,0.104,
						0.103,0.035,0.021,0.014,0.008,0.008,0.008,0.008,0.007,-0.047,-0.059,-0.064,
						-0.068,-0.068,-0.068,-0.068,-0.068,-0.068,-0.069};

					double BenCollIonParamC[31] = {
						28.625,5.655,4.733,4.331,4.331,4.331,4.037,2.304,2.072,1.972,1.895,
						1.895,1.880,1.298,1.207,1.167,1.135,1.135,1.134,1.134,1.128,0.866,
						0.821,0.802,0.785,0.785,0.784,0.784,0.784,0.784,0.781};

					ASSERT( ipHi < 31 );

					iso.ColIoniz[ipHE_LIKE][nelem][ipHi] = BenCollIonParamA[ipHi] * 
						pow( (double)phycon.te/10000., BenCollIonParamB[ipHi] ) *
						sexp( 10000.*BenCollIonParamC[ipHi]/phycon.te );
			}
			else 
			{
				if( nelem == ipHELIUM )
				{
					iso.ColIoniz[ipHE_LIKE][nelem][ipHi] = hydro_vs_ioniz( ipHE_LIKE, nelem, ipHi );
				}
				else
				{
					/* ions */
					iso.ColIoniz[ipHE_LIKE][nelem][ipHi] = Hion_coll_ioniz_ratecoef(ipHE_LIKE , nelem , ipHi);
				}

				if( iso.quant_desig[ipHE_LIKE][nelem][ipHi].n <= iso.n_HighestResolved[ipHE_LIKE][nelem] )
				{
					iso.ColIoniz[ipHE_LIKE][nelem][ipHi] *= iso.stat[ipHE_LIKE][nelem][ipHi] /
						(4.*iso.quant_desig[ipHE_LIKE][nelem][ipHi].n)/
						(2.*iso.quant_desig[ipHE_LIKE][nelem][ipHi].l+1);
				}

				/* Always leave top levels coupled to the continuum, but zero other levels if 
				 * collisional ionization is turned off.	*/
				if( ipHi < iso.numLevels[ipHE_LIKE][nelem] - 1 )
				{
					iso.ColIoniz[ipHE_LIKE][nelem][ipHi] *= iso.lgColl_ionize[ipHE_LIKE];
				}
			}

			ColIonizPerN[iso.quant_desig[ipHE_LIKE][nelem][ipHi].n] += iso.ColIoniz[ipHE_LIKE][nelem][ipHi];
		}

		ipLo = iso.numLevels[ipHE_LIKE][nelem]-1;
		n = iso.n_HighestResolved[ipHE_LIKE][nelem]+iso.nCollapsed[ipHE_LIKE][nelem];

		/* To make the top level as realistic as possible, we need to add in a few collisional 
		 * delta n > 0 upward transitions.  But since we are considering no higher level, it has
		 * to be added to the collisional ionization rate.  */
		if( nelem == ipHELIUM && !helike.lgSetBenjamin && !helike.lgFullSize[nelem])
		{
			/* will add on 5 levels above top level - somewhat arbitrary */
			for( i=1; i<=5; i++ )
			{
				iso.ColIoniz[ipHE_LIKE][nelem][ipLo] +=
					hydro_vs_deexcit( n+i, n )/phycon.sqrte*8.629e-6/
					(double)iso.stat[ipHE_LIKE][nelem][ipLo]*
					sexp( (1./n/n - 1./(n+i)/(n+i)) * TE1RYD/phycon.te );
			}
		}
		else if( nelem != ipHELIUM && !helike.lgFullSize[nelem])
		{
			for( i=1; i<=5; i++ )
			{
				iso.ColIoniz[ipHE_LIKE][nelem][ipLo] +=
					Hion_colldeexc_cs( n+i, n, nelem, ipHE_LIKE )/phycon.sqrte*8.629e-6/
					(double)iso.stat[ipHE_LIKE][nelem][ipLo]*
					sexp( POW2((double)nelem) * (1./n/n - 1./(n+i)/(n+i)) * TE1RYD/phycon.te );
			}
		}

		/* Assert that total coll. ioniz. per n increases monotonically with increasing n.	*/
		if( iso.lgColl_ionize[ipHE_LIKE] && (nelem!=ipHELIUM || !helike.lgSetBenjamin ) )
		{
			for( n = 2; n < iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem]; n++  )
			{
				ASSERT( ColIonizPerN[n] < ColIonizPerN[n+1] || ColIonizPerN[n]==0.);
			}
		}

		/* this is to kill collisional excitations from n=1 and n=2 in the
		 * case b hummer and storey option */
		if( opac.lgCaseB_HummerStorey )
		{

			/* this kills collisional excitation and ionization from the ground
			 * 1s 1S level, and the two singlet n=2 levels, 2s 1S and 2p 1P */
			ipLo = ipHe1s1S;
			iso.ColIoniz[ipHE_LIKE][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ColUL = 0.;
			}
			ipLo = ipHe2s1S;
			iso.ColIoniz[ipHE_LIKE][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ColUL = 0.;
			}
			ipLo = ipHe2p1P;
			iso.ColIoniz[ipHE_LIKE][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
			{
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ColUL = 0.;
			}
		}
	}

	/* cfit( Atom num (H=1), Num elec, >=0 */
	DimaRate = (float)cfit( nelem+1, 2 , phycon.te );

	crate = DimaRate*(float)phycon.EdenHCorr;
	/* NB hecolnc does not include secondaries since not total, but CollidRate does,
	 * since is total */
	iso.ColIoniz[ipHE_LIKE][nelem][0] = DimaRate*iso.lgColl_ionize[ipHE_LIKE];

	/* total collisional ionization rate with both thermal and suprathermal electrons */
	/* there is no attempt to use scaled csupra here, since this is mostly 
	 * ionic and csupra will be zero.  For He itself, other routine is used.
	 * when this routine becomes fundamental for He a scale factor should
	 * use used.  TH85 Tab 10 suggests should be 0.5, but this is for
	 * their background rate - the rate derived here is self-consistent with unity */
	ionrec.CollIonRate_Ground[nelem][nelem-1][0] = crate + Secondaries.csupra;

	/* cooling due to collisional ionization, which only includes thermal */
	ionrec.CollIonRate_Ground[nelem][nelem-1][1] = (crate*
		rfield.anu[Heavy.ipHeavy[nelem][nelem-1]-1]* EN1RYD);

	/* collisional ionizaiton of atomic helium */
	if( nelem == ipHELIUM && !helike.lgSetBenjamin )
	{
		double c2sion , c2pion;
		/* >>refer	he1	cs	Seaton, M.S. 1964, Plan Sp Sci 12, 55. */
		/* TODO verify this division of collisional ionization within 3P is OK */
		c2sion = 8.8e-10*(double)phycon.sqrte*iso.ConBoltz[ipHE_LIKE][nelem][ipHe2s3S];
		c2pion = 3.*8.8e-10*(double)phycon.sqrte*iso.ConBoltz[ipHE_LIKE][nelem][ipHe2p3P0];
		iso.ColIoniz[ipHE_LIKE][nelem][ipHe2s3S] = c2sion*iso.lgColl_ionize[ipHE_LIKE];
		iso.ColIoniz[ipHE_LIKE][nelem][ipHe2p3P0] = c2pion*iso.lgColl_ionize[ipHE_LIKE]/3.;
		iso.ColIoniz[ipHE_LIKE][nelem][ipHe2p3P1] = c2pion*iso.lgColl_ionize[ipHE_LIKE]/3.;
		iso.ColIoniz[ipHE_LIKE][nelem][ipHe2p3P2] = c2pion*iso.lgColl_ionize[ipHE_LIKE]/3.;
	}
	{
		/* prints table of wavelengths of lines to ground */
		/*@-redef@*/
		enum {PRINTIT=FALSE};
		/*@+redef@*/
		if( PRINTIT )
		{

			nelem = ipHELIUM;
			for( ipHi=1; ipHi<iso.numLevels[ipHE_LIKE][ipHELIUM]; ++ipHi )
			{
				for( ipLo=0; ipLo<ipHi; ++ipLo )
				{
					fprintf(ioQQQ,"Z %li Lo %li n %li s %li l %li \t ", 
						nelem+1,
						ipLo, 
						iso.quant_desig[ipHE_LIKE][nelem][ipLo].n,
						iso.quant_desig[ipHE_LIKE][nelem][ipLo].s,
						iso.quant_desig[ipHE_LIKE][nelem][ipLo].l );
					fprintf(ioQQQ," Hi %li n %li s %li l %li \t", 
						ipHi, 
						iso.quant_desig[ipHE_LIKE][nelem][ipHi].n,
						iso.quant_desig[ipHE_LIKE][nelem][ipHi].s,
						iso.quant_desig[ipHE_LIKE][nelem][ipHi].l );
					fprintf(ioQQQ,"%.4e\t%.4e\tcs\t%.4e\n",
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN,
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].Aul ,
						EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].cs);
				}
			}
			exit(1);
		}
	}

	free( ColIonizPerN );

#	ifdef DEBUG_FUN
	fputs( " <->HeCollid()\n", debug_fp );
#	endif
}

/*===================================================================================*/
/* solve for level populations  */
static void HeLevel( long nelem)
{
	long int n, i, ipHi, ipLo, j, level, nerror1, nerror2;
	double create, destroy, HighestPColOut[2] = {0.,0.}, sum_popn_ov_ion, killAul;
	int lgNegPop;
	static long int *ipiv ; /* malloc out to [helike.numLevels[nelem]] */
	/* this block of variables will be obtained and freed here */
	static double
		*creation ,
		*error/*[helike.numLevels[nelem]]*/, 
		*work/*[helike.numLevels[nelem]]*/, 
		**SaveZ/*[helike.numLevels[nelem]+1][iso.numLevels[ipHE_LIKE]]*/, 
		**z/*[helike.numLevels[nelem]][iso.numLevels[ipHE_LIKE]]*/;
	static double
		*CollisionsGoingUp,	*CollisionsGoingDown,
		*OutOfNGoingUp,	*OutOfNGoingDown;
	static int lgSpaceMalloc=FALSE;
	static FILE *matrix;

	/* this flag true means to malloc and free all space each time */
#	define PARALLEL	FALSE
	
#	ifdef DEBUG_FUN
	fputs( "<+>HeLevel()\n", debug_fp );
#	endif

	ASSERT(nelem < LIMELM );

	/* malloc some scratch space */

	if( PARALLEL )
	{
		if( (ipiv = (long int *)MALLOC(sizeof(long int)*(unsigned)helike.numLevels[nelem] ) )==NULL )
			BadMalloc();

		if( (creation = (double *)MALLOC(sizeof(double)*(unsigned)helike.numLevels[nelem] ) )==NULL )
			BadMalloc();

		if( (CollisionsGoingUp = (double *)MALLOC(sizeof(double)*(unsigned)max_num_levels ) )==NULL )
			BadMalloc();

		if( (CollisionsGoingDown = (double *)MALLOC(sizeof(double)*(unsigned)max_num_levels ) )==NULL )
			BadMalloc();

		if( (OutOfNGoingUp = (double *)MALLOC(sizeof(double)*(unsigned)(max_n + 1) ) )==NULL )
			BadMalloc();

		if( (OutOfNGoingDown = (double *)MALLOC(sizeof(double)*(unsigned)(max_n + 1) ) )==NULL )
			BadMalloc();

		if( (error = (double *)MALLOC(sizeof(double)*(unsigned)helike.numLevels[nelem] ) )==NULL )
			BadMalloc();

		if( (work = (double *)MALLOC(sizeof(double)*(unsigned)helike.numLevels[nelem] ) )==NULL )
			BadMalloc();

		/* now do the 2D arrays */
		if( (SaveZ = (double **)MALLOC(sizeof(double *)*(unsigned)helike.numLevels[nelem] ) ) ==NULL )
			BadMalloc();

		if(  (z = (double **)MALLOC(sizeof(double *)*(unsigned)helike.numLevels[nelem] ))==NULL  )
			BadMalloc();
		
		/* now do the second dimension */
		for( i=0; i<(helike.numLevels[nelem]); ++i )
		{
			if( (SaveZ[i] = (double *)MALLOC(sizeof(double)*(unsigned)helike.numLevels[nelem] ) )==NULL )
				BadMalloc();

			if( (z[i] = (double *)MALLOC(sizeof(double)*(unsigned)helike.numLevels[nelem] ) )==NULL )
				BadMalloc();
		}
	}
	else if( !lgSpaceMalloc )
	{
		/* space has not been malloced yet, but will only do it one time */
		lgSpaceMalloc = TRUE;

		if( (ipiv = (long int *)MALLOC(sizeof(long int)*(unsigned)(max_num_levels) ) )==NULL )
			BadMalloc();

		if( (creation = (double *)MALLOC(sizeof(double)*(unsigned)(max_num_levels) ) )==NULL )
			BadMalloc();

		if( (CollisionsGoingUp = (double *)MALLOC(sizeof(double)*(unsigned)
			(max_num_levels) ) )==NULL )
			BadMalloc();

		if( (CollisionsGoingDown = (double *)MALLOC(sizeof(double)*(unsigned)
			(max_num_levels) ) )==NULL )
			BadMalloc();

		if( (OutOfNGoingUp = (double *)MALLOC(sizeof(double)*(unsigned)(max_n + 1) ) )==NULL )
			BadMalloc();

		if( (OutOfNGoingDown = (double *)MALLOC(sizeof(double)*(unsigned)(max_n + 1) ) )==NULL )
			BadMalloc();

		if( (error = (double *)MALLOC(sizeof(double)*(unsigned)(max_num_levels) ) )==NULL )
			BadMalloc();

		if( (work = (double *)MALLOC(sizeof(double)*(unsigned)(max_num_levels) ) )==NULL )
			BadMalloc();

		/* now do the 2D arrays */
		if( (SaveZ = (double **)MALLOC(sizeof(double *)*(unsigned)max_num_levels ) ) ==NULL )
			BadMalloc();

		if(  (z = (double **)MALLOC(sizeof(double *)*(unsigned)max_num_levels ))==NULL  )
			BadMalloc();

		/* now do the second dimension */
		for( i=0; i<max_num_levels; ++i )
		{
			if( (SaveZ[i] = (double *)MALLOC(sizeof(double)*(unsigned)(max_num_levels+2) ) )==NULL )
				BadMalloc();

			if( (z[i] = (double *)MALLOC(sizeof(double)*(unsigned)(max_num_levels+2) ) )==NULL )
				BadMalloc();
		}
	}

	for( i=0; i < max_num_levels; ++i )
	{
		CollisionsGoingUp[i] = 0.;
		CollisionsGoingDown[i] = 0.;
	}
	sum_popn_ov_ion = 0.;

	for( i=0; i <= max_n; ++i )
	{
		OutOfNGoingUp[i] = 0.;
		OutOfNGoingDown[i] = 0.;
	}

	/* this rad rec coef will include optical depth effects */
	ionrec.RadRecomRateCoef[nelem][nelem-1] = 0.;

	/* fill in creation/recombination vector, rate levels are populated from the continuum,
	 * rates s^-1 */
	for( n=0; n < helike.numLevels[nelem]; n++ )
	{
		double rec_radiative, rec_induced, CollisRecomb;

		/* radiative recombination rate [s-1]*/
		rec_radiative = iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad]*
			iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecNetEsc]*dense.eden;
		ASSERT( rec_radiative >= 0. );

		/* part of rad rec that escapes */
		ionrec.RadRecomRateCoef[nelem][nelem-1] += iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecRad]*
			iso.RadRecomb[ipHE_LIKE][nelem][n][ipRecNetEsc];

		/* induced recombination rate [s-1] */
		rec_induced = iso.RecomInducRate[ipHE_LIKE][nelem][n]*
			iso.PopLTE[ipHE_LIKE][nelem][n]*dense.eden;

		/* collisional recombination from collisional ionization,
		 * rate [s-1] */
		CollisRecomb = iso.ColIoniz[ipHE_LIKE][nelem][n]* 
			iso.PopLTE[ipHE_LIKE][nelem][n]*POW2(phycon.EdenHCorr);
		ASSERT( CollisRecomb>= 0. );

		/* total recombination to level n rate [s-1] */
		creation[n] = rec_radiative + rec_induced + CollisRecomb;
		ASSERT( creation[n] >= 0. );

	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nelem==5 )
		{
			fprintf(ioQQQ,"recomrate");
			for( n=0; n < helike.numLevels[nelem]; n++ )
			{
				fprintf(ioQQQ,"\t%e", creation[n] );
			}
			fprintf(ioQQQ,"\n"); 
		}
	}
	/* total destruction rate, photo and collis ioniz first, CollidRate already
	 * includes suprathermals */
	destroy = iso.gamnc[ipHE_LIKE][nelem][0] + ionrec.CollIonRate_Ground[nelem][nelem-1][0];


	/* >>chng 01 may 22, added CO to this treatment, (had always been included
	 * for old he1-he3 atom) */
	if( nelem == ipHELIUM )
	{
		/* add CO here, only for he itself, 
		 * destruction of CO but recombination for he */

		/* >>chng 01 nov 23, increase first index by one */
		/*z[helike.numLevels[nelem]][ipHe1s1S] += hevmolec.hevmol[ipCO]*1.10e-9 ;*/
		creation[ipHe1s1S] += hevmolec.hevmol[ipCO]*1.10e-9 ;
	}

	/* ratio of ion to atom, MUST be greater than zero since it is 
	 * the recombination rate iso.RadRec_effec[ipHE_LIKE][nelem]*/
	create = dense.eden*(iso.RadRec_effec[ipHE_LIKE][nelem] + 
	  ionrec.CotaRate[nelem]) + hevmolec.hevmol[ipCO]*1.10e-9;
	if( create > 0. )
	{
		iso.xIonSimple[ipHE_LIKE][nelem] = (float)(destroy / create);
	}
	else
		/* rec rate can't be zero */
		TotalInsanity();

#	define TOOSMALL 1e-10
	if( iso.xIonSimple[ipHE_LIKE][nelem] < TOOSMALL )
	{
		/* >>chng 01 nov 23, increase upper limit by 2 */
		for( n=0; n < helike.numLevels[nelem]; n++ )
		{
			iso.Pop2Ion[ipHE_LIKE][nelem][n] =  0.;
		}
		lgNegPop = FALSE;
		/* >>chng 02 apr 10, had set to zero,
		 * we will leave the level populations at zero but define an ion ratio,
		 * since He+ abundance cannot go to zero in a molecualr cloud,
		 * - He+ charge transfer is the main CO destruction mechanism 
		iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 0.;*/
		iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = iso.xIonSimple[ipHE_LIKE][nelem];
		ionrec.TotRecomRate[nelem][nelem-1] = create;
		ionrec.TotIonizRate[nelem][nelem-1] = destroy;
	}
	else
	{
		/* this will become the summed rec rate */
		ionrec.TotRecomRate[nelem][nelem-1] = 0.;

		/* master balance equation, use when significant population */
		for( level=ipHe1s1S; level < helike.numLevels[nelem]; level++ )
		{

			/* save the final set of creation rates - units are s-1 */
			ionrec.TotRecomRate[nelem][nelem-1] += creation[level];

			/* 
			 * all process depopulating level
			 */
			/* there is no attempt to use scaled csupra here, since this is mostly 
			 * ionic and csupra will be zero.  For He itself, other routine is used.
			 * when this routine becomes fundamental for He a scale factor should
			 * use used.  TH85 Tab 10 suggests should be 0.5, but this is for
			 * their background rate - the rate derived here is self-consistent with unity */
			/* NB collisional ionization rate already includes suprathermals */
			z[level][level] = iso.gamnc[ipHE_LIKE][nelem][level] + 
			  iso.ColIoniz[ipHE_LIKE][nelem][level]* phycon.EdenHCorr;

			/* this saves rate level goes to continuum */
			iso.xLevel2Cont[ipHE_LIKE][nelem][level] = (float)z[level][level];

			/* fixit(); in final version move this up a line, add to all levels 
			 * for now only add to ground to keep parallel to bidiag
			 * NB - comment in previous statement says this already included - which is wrong? */
			if( level==ipHe1s1S ) 
			{
				z[level][level] +=Secondaries.csupra;
				iso.xLevel2Cont[ipHE_LIKE][nelem][level] +=Secondaries.csupra;
			}

			/* all processes populating level from below */
			for( ipLo=ipHe1s1S; ipLo < level; ipLo++ )
			{
				double coll_down = EmisLines[ipHE_LIKE][nelem][level][ipLo].ColUL * phycon.EdenHCorr;

				z[ipLo][level] = 
					-coll_down * 
				  (double)EmisLines[ipHE_LIKE][nelem][level][ipLo].gHi/
				  (double)EmisLines[ipHE_LIKE][nelem][level][ipLo].gLo*
				  iso.Boltzmann[ipHE_LIKE][nelem][level][ipLo]  -
				  (double)EmisLines[ipHE_LIKE][nelem][level][ipLo].pump ;

				/* pumping out of here to lower level */
				/* >>chng 02 feb 16, add casts to double */
				z[level][level] += (double)EmisLines[ipHE_LIKE][nelem][level][ipLo].pump *
				  (double)EmisLines[ipHE_LIKE][nelem][level][ipLo].gLo/(double)EmisLines[ipHE_LIKE][nelem][level][ipLo].gHi;

				/* collisions out of here to lower level */
				z[level][level] += coll_down;

				if( iso.quant_desig[ipHE_LIKE][nelem][level].n > 
					iso.quant_desig[ipHE_LIKE][nelem][ipLo].n )
					CollisionsGoingDown[level] += coll_down;

				if( plasnu.lgPlasNu )
					killAul = CheckIfBelowPlasFreq( nelem, level, ipLo );
				else
					killAul = 1.;

				/* radiative decays out of here to lower level */
				z[level][level] += 
					killAul*EmisLines[ipHE_LIKE][nelem][level][ipLo].Aul*
				  (EmisLines[ipHE_LIKE][nelem][level][ipLo].Pesc + 
				   EmisLines[ipHE_LIKE][nelem][level][ipLo].Pelec_esc + 
				   EmisLines[ipHE_LIKE][nelem][level][ipLo].Pdest);

				if( (nelem == ipHELIUM) && iso.quant_desig[ipHE_LIKE][nelem][level].n == helike.n_HighestResolved[nelem]
					&& iso.quant_desig[ipHE_LIKE][nelem][level].l == 1 )
				{
					HighestPColOut[iso.quant_desig[ipHE_LIKE][nelem][level].s] += coll_down;
				}
			}

			/* all processes populating level from above */
			/* >>chng 01 nov 23, increase upper limit by two */
			for( ipHi=level + 1; ipHi < helike.numLevels[nelem]; ipHi++ )
			{
				/*double coll_down = EmisLines[ipHE_LIKE][nelem][ipHi][level].ColUL * phycon.EdenHCorr;*/
				double coll_down = EmisLines[ipHE_LIKE][nelem][ipHi][level].ColUL * phycon.EdenHCorr;

				double RadDecay =
					EmisLines[ipHE_LIKE][nelem][ipHi][level].Aul*
				  (EmisLines[ipHE_LIKE][nelem][ipHi][level].Pesc + 
				  EmisLines[ipHE_LIKE][nelem][ipHi][level].Pelec_esc +
				  EmisLines[ipHE_LIKE][nelem][ipHi][level].Pdest);

				if( plasnu.lgPlasNu )
					killAul = CheckIfBelowPlasFreq( nelem, ipHi, level );
				else
					killAul = 1.;

				z[ipHi][level] = 
					-(RadDecay*killAul + 
				  (double)EmisLines[ipHE_LIKE][nelem][ipHi][level].pump*
				  (double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gLo/
				  (double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gHi + 
				  coll_down);

				/* pumping out of here to upper level */
				z[level][level] += (double)EmisLines[ipHE_LIKE][nelem][ipHi][level].pump ;

				/* collisions out of here to upper level */
				z[level][level] += coll_down *
					(double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gHi / 
					(double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gLo *
					iso.Boltzmann[ipHE_LIKE][nelem][ipHi][level];

				if( iso.quant_desig[ipHE_LIKE][nelem][level].n <
					iso.quant_desig[ipHE_LIKE][nelem][ipHi].n )
				{
					CollisionsGoingUp[level] += coll_down *
						EmisLines[ipHE_LIKE][nelem][ipHi][level].gHi / EmisLines[ipHE_LIKE][nelem][ipHi][level].gLo *
						iso.Boltzmann[ipHE_LIKE][nelem][ipHi][level];
				}
				
				if( (nelem == ipHELIUM) && iso.quant_desig[ipHE_LIKE][nelem][level].n == helike.n_HighestResolved[nelem]
					&& iso.quant_desig[ipHE_LIKE][nelem][level].l == 1 )
				{
					HighestPColOut[iso.quant_desig[ipHE_LIKE][nelem][level].s] += coll_down *
					(double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gHi / 
					(double)EmisLines[ipHE_LIKE][nelem][ipHi][level].gLo *
					iso.Boltzmann[ipHE_LIKE][nelem][ipHi][level];
				}
			}
		}

		/* >>chng 02 jul 22, add induced 2-nu to level populations */
		/* induced two photon emission - special because upward and downward are
		 * not related by ratio of statistical weights */
		/* iso.lgInd2nu_On controls on or off, set with SET IND2 ON/OFF command */
		z[ipHe2s1S][ipHe1s1S] -= iso.TwoNu_induc_dn[ipHE_LIKE][nelem]*iso.lgInd2nu_On ;
		z[ipHe1s1S][ipHe2s1S] -= iso.TwoNu_induc_up[ipHE_LIKE][nelem]*iso.lgInd2nu_On ;

		/* rates out of 1s, and out of 2s */
		z[ipHe1s1S][ipHe1s1S] += iso.TwoNu_induc_up[ipHE_LIKE][nelem]*iso.lgInd2nu_On ;
		z[ipHe2s1S][ipHe2s1S] += iso.TwoNu_induc_dn[ipHE_LIKE][nelem]*iso.lgInd2nu_On ;

		/* add on charge transfer recombination*/
		/* add it to 1s for now, can't really do any better */
		/* >>chng 01 apr 28, had been added to ground, energy conservation off?? */
		/* in current data set (june, 01) next four ct reactions are all zero */
		/* >>chng 01 nov 23, upper limit increased by 2 */
		/*z[helike.numLevels[nelem]][ipHe1s1S] += */
		/* HCharExcIon[nelem][ion]*hii  is ion => ion+1 for nelem */
		/* HCharExcRec[nelem][ion]*hi is ion+1 => ion of nelem */
		/* >>chng 02 may 02, several logical errors in which ct elements are added */
		creation[ipHe1s1S] += 
			ChargTran.HeCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHELIUM][0] + 
			ChargTran.HCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][0] ;
		/* rates are s-1 */
		ionrec.TotRecomRate[nelem][nelem-1] += 
			ChargTran.HeCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHELIUM][0] + 
			ChargTran.HCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][0] ;

		/* ionization of this species by ct with H+ and He+ */
		z[ipHe1s1S][ipHe1s1S] +=  
			ChargTran.HeCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHELIUM][1] + 
			ChargTran.HCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][1] ;

		iso.xLevel2Cont[ipHE_LIKE][nelem][ipHe1s1S] += (float)(
			ChargTran.HeCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHELIUM][1] + 
			ChargTran.HCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][1] );

		/* >>chng 01 jun 21, add */
		/* charge transfer with helium itself, onto everything heavier */
		if( nelem == ipHELIUM )
		{
			int nelem_col , ion;

			/* >>logic checked 02 may 02, think it's right */
			/* add on charge transfer ionization of helium,
			 * recombination for other element is ionization of helium */
			ChargTran.HeCharExcIonTotal = 0.;
			/* loop up from Li, the next heavier element */
			for( nelem_col=ipLITHIUM; nelem_col<LIMELM; ++nelem_col)
			{
				double hold_one = 0.;
				/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
				 * for helium nelem=1, ion must go up to 3 */
				/* check that array bounds not exceeded */
				ASSERT( nelem_col < (LIMELM+1) );
				for( ion=2; ion<=nelem_col+2; ++ion )
				{
					/* charge transfer ionization of He, recombination for other species */
					hold_one = ChargTran.HeCharExcRec[nelem_col][ion-2]*dense.xIonDense[nelem_col][ion-1];
					ChargTran.HeCharExcIonTotal += hold_one;
				}
			}
			z[ipHe1s1S][ipHe1s1S] += ChargTran.HeCharExcIonTotal;
			iso.xLevel2Cont[ipHE_LIKE][nelem][ipHe1s1S] += (float)ChargTran.HeCharExcIonTotal;

			/* charge transfer recombination of helium,
			 * which is ionization of the heavy element */
			ChargTran.HeCharExcRecTotal = 0.;
			for( nelem_col=3; nelem_col<LIMELM; ++nelem_col)
			{
				/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
				 * for helium nelem=1, ion must go up to 3 */
				/* check that array bounds not exceeded */
				ASSERT( (nelem_col) < (LIMELM+1) );
				for( ion=1; ion<=nelem_col+1; ++ion )
				{
					/* charge transfer ionization of H, recombination for other species */
					ChargTran.HeCharExcRecTotal += 
						ChargTran.HeCharExcIon[nelem_col][ion-1]*dense.xIonDense[nelem_col][ion-1];
				}
			}
			/* >>chng 01 nov 23, increase first limit by 2 */
			/*z[helike.numLevels[nelem]][ipHe1s1S] += ChargTran.HeCharExcRecTotal;*/
			creation[ipHe1s1S] += ChargTran.HeCharExcRecTotal;
			/* rates are s-1 */
			ionrec.TotRecomRate[nelem][nelem-1] += ChargTran.HeCharExcRecTotal;
		}

		/* add on advection terms, normally zero */
		/* incoming material is in ground state */
		/* >>chng 01 nov 23, increase first limit by 2 */
		/*z[helike.numLevels[nelem]][ipHe1s1S] += dynamics.Source[nelem][nelem-1];
		creation[ipHe1s1S] += dynamics.Source[nelem][nelem-1]*dynamics.lgISO[ipHE_LIKE];

		z[ipHe1s1S][ipHe1s1S] += dynamics.Rate*dynamics.lgISO[ipHE_LIKE];*/

		fixit();/* neglect secondary ionization for now - csupra occurs in above look [level][level] */
		fixit();/* error in how Hx12 is allocated, sec dim has only LIMELM elements,
				 * will crash if following code ever used */

		/* >>chng 02 Sep 06 rjrw -- all elements have these terms */
		/*>>>chng 02 oct 01, only include if lgAdvection is set */
		if( dynamics.lgAdvection )
		{
			/* add in advection - these terms normally zero */
			/* assume for present that all advection is into ground state */
			creation[ipHe1s1S] += dynamics.Source[nelem][nelem-1]/MAX2(SMALLFLOAT,dense.xIonDense[nelem][nelem])*
				dynamics.lgISO[ipHE_LIKE];
			/* >>chng 02 Sep 06 rjrw -- advective term not recombination */
			/* can sink from all components (must do, for conservation) */
			for( ipLo=ipHe1s1S; ipLo<helike.numLevels[nelem]; ++ipLo )
			{
				z[ipLo][ipLo] += dynamics.Rate*dynamics.lgISO[ipHE_LIKE];
			}
		}

#		if 0
		if( Secondaries.Hx12[0][1] > 0. )
		{
			/* now add on supra thermal excitation */
			/* >>chng 01 nov 23, increase upper limit by 2 */
			for( level=1; level < helike.numLevels[nelem]; level++ )
			{
				double RateUp , RateDown;

				RateUp = Secondaries.Hx12[MIN2(nelem,1)][level];
				RateDown = RateUp * (double)EmisLines[ipHE_LIKE][nelem][level][ipH1s].gLo /
					(double)EmisLines[ipHE_LIKE][nelem][level][ipH1s].gHi;

				/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for ion */
				/* total rate out of lower level */
				fixit();/* these are all wrong */
				z[ipH1s][ipH1s] += RateUp;

				/* rate from the upper level to ground */
				z[level][ipH1s] -= RateDown ;

				/* rate from ground to upper level */
				z[ipH1s][level] -= RateUp ;

				z[level][level] += RateDown;  
			}
		}
#		endif

		if( trace.lgTrace && 
			trace.lgIsoTraceFull[ipHE_LIKE] && (nelem == trace.ipIsoTrace[ipHE_LIKE]) )
		{
			fprintf( ioQQQ, "  pop level     others => (HeLevel)\n" );
			/* >>chng 01 nov 23 increase upper limit by 2 */
			for( n=0; n < helike.numLevels[nelem]; n++ )
			{
				fprintf( ioQQQ, "       HII%2ld", n );
				/* limit is <= since last has rec coef */
				/* >>chng 01 nov 23 increase upper limit by 2 */
				for( j=0; j < helike.numLevels[nelem]; j++ )
				{
					fprintf( ioQQQ,"\t%.9e", z[j][n] );
				}
				fprintf( ioQQQ, "\n" );
			}
			fprintf(ioQQQ," recomb          ");
			for( n=0; n < helike.numLevels[nelem]; n++ )
			{
				fprintf( ioQQQ,"\t%.9e", creation[n] );
			}
			fprintf( ioQQQ, "\n" );
			fprintf(ioQQQ," recomb ct %.2e co %.2e hectr %.2e hctr %.2e\n",
				ChargTran.HeCharExcRecTotal,
				hevmolec.hevmol[ipCO] ,
				ChargTran.HeCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHELIUM][0] ,
				ChargTran.HCharExcRec[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][0] );
		}

		/* save matrix */
		for( ipLo=ipHe1s1S; ipLo < helike.numLevels[nelem]; ipLo++ )
		{
			for( n=ipHe1s1S; n < helike.numLevels[nelem]; n++ )
			{
				SaveZ[n][ipLo] = z[n][ipLo];
			}
		}

		if( helike.lgFSM==1 )
		{
			double HiMixingAngle, LoMixingAngle, f;
			float Sin2Hi, Sin2Lo, Cos2Hi, Cos2Lo;
			long ipLoSing, ipLoTrip, ipHiSing, ipHiTrip, nLo, nHi, lLo, lHi;

			for( nLo=1; nLo <= helike.n_HighestResolved[nelem]; nLo++ )
			{
				for( lLo = 4; lLo < nLo; lLo++ )
				{
					ipLoSing = QuantumNumbers2Index[ipHELIUM][nLo][lLo][0];
					ipLoTrip = QuantumNumbers2Index[ipHELIUM][nLo][lLo][1];

					if( ipLoTrip < 0. )
						ipLoTrip = ipLoSing;
							
					if( lLo == 0 )
					{
						LoMixingAngle = 0.;
					}
					else if( lLo == 1 )
					{
						LoMixingAngle = 0.001;
					}
					else if( lLo == 2 )
					{
						LoMixingAngle = 0.01;
					}
					else if( lLo == 3 )
					{
						LoMixingAngle = 0.5;
					}
					else
					{
						LoMixingAngle = PI/4.;
					}

					f = sin( LoMixingAngle );
					Sin2Lo = POW2( (float)f );
					
					f = cos( LoMixingAngle );
					Cos2Lo = POW2( (float)f );

					for( nHi=nLo; nHi <= helike.n_HighestResolved[nelem]; nHi++ )
					{
						for( lHi = 4; lHi < nHi; lHi++ )
						{
							if( (nHi==nLo) && ((lHi==lLo) || (lHi<2) || (lLo<2)) )
								continue;
							
							ipHiSing = QuantumNumbers2Index[ipHELIUM][nHi][lHi][0];
							ipHiTrip = QuantumNumbers2Index[ipHELIUM][nHi][lHi][1];

							if( lHi == 0 )
							{
								HiMixingAngle = 0.;
							}
							else if( lHi == 1 )
							{
								HiMixingAngle = 0.001;
							}
							else if( lHi == 2 )
							{
								HiMixingAngle = 0.01;
							}
							else if( lHi == 3 )
							{
								HiMixingAngle = 0.5;
							}
							else
							{
								HiMixingAngle = PI/4.;
							}

							f = sin( HiMixingAngle );
							Sin2Hi = POW2( (float)f );

							f = cos( HiMixingAngle );
							Cos2Hi = POW2( (float)f );
							
							/* First fix decay and de-excitation processes...	*/
							z[ipHiSing][ipLoSing] = 
								Cos2Lo*Cos2Hi*SaveZ[ipHiSing][ipLoSing] + 
								Sin2Lo*Cos2Hi*SaveZ[ipHiSing][ipLoTrip]*(1./3.)+ 
								Cos2Lo*Sin2Hi*SaveZ[ipHiTrip][ipLoSing]*(1./3.)+ 
								Sin2Lo*Sin2Hi*SaveZ[ipHiTrip][ipLoTrip]*(1./9.);
							z[ipHiSing][ipLoTrip] = (2./3.)*SaveZ[ipHiSing][ipLoTrip]+
								Sin2Lo*Cos2Hi*SaveZ[ipHiSing][ipLoSing] + 
								Cos2Lo*Cos2Hi*SaveZ[ipHiSing][ipLoTrip]*(1./3.)+ 
								Sin2Lo*Sin2Hi*SaveZ[ipHiTrip][ipLoSing]*(1./3.)+ 
								Cos2Lo*Sin2Hi*SaveZ[ipHiTrip][ipLoTrip]*(1./9.);
							z[ipHiTrip][ipLoSing] = (2./3.)*SaveZ[ipHiTrip][ipLoSing]+
								Cos2Lo*Sin2Hi*SaveZ[ipHiSing][ipLoSing] + 
								Sin2Lo*Sin2Hi*SaveZ[ipHiSing][ipLoTrip]*(1./3.)+ 
								Cos2Lo*Cos2Hi*SaveZ[ipHiTrip][ipLoSing]*(1./3.)+ 
								Sin2Lo*Cos2Hi*SaveZ[ipHiTrip][ipLoTrip]*(1./9.);
							z[ipHiTrip][ipLoTrip] = (8./9.)*SaveZ[ipHiTrip][ipLoTrip]+
								Sin2Lo*Sin2Hi*SaveZ[ipHiSing][ipLoSing] + 
								Cos2Lo*Sin2Hi*SaveZ[ipHiSing][ipLoTrip]*(1./3.)+ 
								Sin2Lo*Cos2Hi*SaveZ[ipHiTrip][ipLoSing]*(1./3.)+ 
								Cos2Lo*Cos2Hi*SaveZ[ipHiTrip][ipLoTrip]*(1./9.);

							/* then absorption and excitation...	*/
							z[ipLoSing][ipHiSing] = 
								Cos2Hi*Cos2Lo*SaveZ[ipLoSing][ipHiSing] + 
								Sin2Hi*Cos2Lo*SaveZ[ipLoSing][ipHiTrip]*(1./3.)+ 
								Cos2Hi*Sin2Lo*SaveZ[ipLoTrip][ipHiSing]*(1./3.)+ 
								Sin2Hi*Sin2Lo*SaveZ[ipLoTrip][ipHiTrip]*(1./9.);
							z[ipLoSing][ipHiTrip] = (2./3.)*SaveZ[ipLoSing][ipHiTrip]+
								Sin2Hi*Cos2Lo*SaveZ[ipLoSing][ipHiSing] + 
								Cos2Hi*Cos2Lo*SaveZ[ipLoSing][ipHiTrip]*(1./3.)+ 
								Sin2Hi*Sin2Lo*SaveZ[ipLoTrip][ipHiSing]*(1./3.)+ 
								Cos2Hi*Sin2Lo*SaveZ[ipLoTrip][ipHiTrip]*(1./9.);
							z[ipLoTrip][ipHiSing] = (2./3.)*SaveZ[ipLoTrip][ipHiSing]+
								Cos2Hi*Sin2Lo*SaveZ[ipLoSing][ipHiSing] + 
								Sin2Hi*Sin2Lo*SaveZ[ipLoSing][ipHiTrip]*(1./3.)+ 
								Cos2Hi*Cos2Lo*SaveZ[ipLoTrip][ipHiSing]*(1./3.)+ 
								Sin2Hi*Cos2Lo*SaveZ[ipLoTrip][ipHiTrip]*(1./9.);
							z[ipLoTrip][ipHiTrip] = (8./9.)*SaveZ[ipLoTrip][ipHiTrip]+
								Sin2Hi*Sin2Lo*SaveZ[ipLoSing][ipHiSing] + 
								Cos2Hi*Sin2Lo*SaveZ[ipLoSing][ipHiTrip]*(1./3.)+ 
								Sin2Hi*Cos2Lo*SaveZ[ipLoTrip][ipHiSing]*(1./3.)+ 
								Cos2Hi*Cos2Lo*SaveZ[ipLoTrip][ipHiTrip]*(1./9.);

							if( SaveZ[ipHiSing][ipLoSing] + SaveZ[ipHiSing][ipLoTrip] + 
								SaveZ[ipHiTrip][ipLoSing] + SaveZ[ipHiTrip][ipLoTrip] > 0. )
							{
								ASSERT( fabs( (z[ipHiSing][ipLoSing] + z[ipHiSing][ipLoTrip] + 
									z[ipHiTrip][ipLoSing] + z[ipHiTrip][ipLoTrip])/
									(SaveZ[ipHiSing][ipLoSing] + SaveZ[ipHiSing][ipLoTrip] + 
									SaveZ[ipHiTrip][ipLoSing] + SaveZ[ipHiTrip][ipLoTrip]) - 1.f ) < 0.001 );

								ASSERT( fabs( (z[ipLoSing][ipHiSing] + z[ipLoSing][ipHiTrip] + 
									z[ipLoTrip][ipHiSing] + z[ipLoTrip][ipHiTrip])/
									(SaveZ[ipLoSing][ipHiSing] + SaveZ[ipLoSing][ipHiTrip] + 
									SaveZ[ipLoTrip][ipHiSing] + SaveZ[ipLoTrip][ipHiTrip]) - 1.f ) < 0.001 );
							}

							/* then diagonal (destruction) elements. */
							z[ipLoSing][ipLoSing] = SaveZ[ipLoSing][ipLoSing] - SaveZ[ipLoSing][ipHiSing] 
								- SaveZ[ipLoSing][ipHiTrip] + z[ipLoSing][ipHiSing] + z[ipLoSing][ipHiTrip];
							z[ipLoTrip][ipLoTrip] = SaveZ[ipLoTrip][ipLoTrip] - SaveZ[ipLoTrip][ipHiSing]
								- SaveZ[ipLoTrip][ipHiTrip] + z[ipLoTrip][ipHiSing] + z[ipLoTrip][ipHiTrip];

							
							z[ipHiSing][ipHiSing] = SaveZ[ipHiSing][ipHiSing] - SaveZ[ipHiSing][ipLoSing]
								- SaveZ[ipHiSing][ipLoTrip] + z[ipHiSing][ipLoSing] + z[ipHiSing][ipLoTrip];
							z[ipHiTrip][ipHiTrip] = SaveZ[ipHiTrip][ipHiTrip] - SaveZ[ipHiTrip][ipLoSing]
								- SaveZ[ipHiTrip][ipLoTrip] + z[ipHiTrip][ipLoSing] + z[ipHiTrip][ipLoTrip];

							if( SaveZ[ipLoSing][ipLoSing] + SaveZ[ipLoTrip][ipLoTrip] != 0. )
							{
								ASSERT( fabs( (z[ipLoSing][ipLoSing] + z[ipLoTrip][ipLoTrip])/
									(SaveZ[ipLoSing][ipLoSing] + SaveZ[ipLoTrip][ipLoTrip]) - 1.f ) < 0.001 );
							}

							if( SaveZ[ipHiSing][ipHiSing] + SaveZ[ipHiTrip][ipHiTrip] != 0. )
							{
								ASSERT( fabs( (z[ipHiSing][ipHiSing] + z[ipHiTrip][ipHiTrip])/
								(SaveZ[ipHiSing][ipHiSing] + SaveZ[ipHiTrip][ipHiTrip]) - 1.f ) < 0.001 );
							}
						}
					}

					for( nHi=helike.n_HighestResolved[nelem]+1;nHi<=helike.n_HighestResolved[nelem]
						+helike.nCollapsed[nelem]; nHi++ )
					{
						ipHi = QuantumNumbers2Index[ipHELIUM][nHi][0][0];
						
						/* First fix decay and de-excitation processes...	*/
						z[ipHi][ipLoSing] = 
							Cos2Lo*SaveZ[ipHi][ipLoSing] + Sin2Lo*SaveZ[ipHi][ipLoTrip]*(1./9.);
						z[ipHi][ipLoTrip] = (8./9.)*SaveZ[ipHi][ipLoTrip]+
							Sin2Lo*SaveZ[ipHi][ipLoSing] + Cos2Lo*SaveZ[ipHi][ipLoTrip]*(1./9.);
						
						if( SaveZ[ipHi][ipLoSing] + SaveZ[ipHi][ipLoTrip] > 0. )
						{
							ASSERT( fabs( (z[ipHi][ipLoSing] + z[ipHi][ipLoTrip])/
								(SaveZ[ipHi][ipLoSing] + SaveZ[ipHi][ipLoTrip]) - 1.f ) < 0.001 );
						}

						/* then absorption and excitation...	*/
						z[ipLoSing][ipHi] = 
							Cos2Lo*SaveZ[ipLoSing][ipHi] + Sin2Lo*SaveZ[ipLoTrip][ipHi]*(1./9.);
						z[ipLoTrip][ipHi] = (8./9.)*SaveZ[ipLoTrip][ipHi]+
							Sin2Lo*SaveZ[ipLoSing][ipHi] + Cos2Lo*SaveZ[ipLoTrip][ipHi]*(1./9.);

						if( SaveZ[ipLoSing][ipHi] + SaveZ[ipLoTrip][ipHi] > 0. )
						{
							ASSERT( fabs( (z[ipLoSing][ipHi] + z[ipLoTrip][ipHi])/
								(SaveZ[ipLoSing][ipHi] + SaveZ[ipLoTrip][ipHi]) - 1.f ) < 0.001 );
						}

						/* then diagonal (destruction) elements. */
						z[ipLoSing][ipLoSing] = SaveZ[ipLoSing][ipLoSing] - Sin2Lo * 
							( SaveZ[ipLoTrip][ipHi] - SaveZ[ipLoSing][ipHi] );
						z[ipLoTrip][ipLoTrip] = SaveZ[ipLoTrip][ipLoTrip] + Sin2Lo * 
							( SaveZ[ipLoTrip][ipHi] - SaveZ[ipLoSing][ipHi] );

						z[ipLoSing][ipLoSing] = SaveZ[ipLoSing][ipLoSing] - SaveZ[ipLoSing][ipHi]
							+ z[ipLoSing][ipHi];
						z[ipLoTrip][ipLoTrip] = SaveZ[ipLoTrip][ipLoTrip] - SaveZ[ipLoTrip][ipHi] 
							+ z[ipLoTrip][ipHi];
					
						ASSERT( fabs( (z[ipLoSing][ipLoSing] + z[ipLoTrip][ipLoTrip])/
							(SaveZ[ipLoSing][ipLoSing] + SaveZ[ipLoTrip][ipLoTrip]) - 1.f ) < 0.001 );

						/* The collapsed level doesn't change*/
						z[ipHi][ipHi] = SaveZ[ipHi][ipHi] - SaveZ[ipHi][ipLoTrip] - SaveZ[ipHi][ipLoSing]
							+ z[ipHi][ipLoSing] + z[ipHi][ipLoTrip];

						ASSERT( fabs( z[ipHi][ipHi]/SaveZ[ipHi][ipHi] - 1.f ) < 0.001 );
					}
				}
			}
		}

		/* this is the default matrix solver */
		if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
		{
			static int lgAMAT_alloc=FALSE;
			static double *amat;
#			ifdef AMAT
#				undef AMAT
#			endif
			/* >>chng 01 nov 23, increase upper limit by 2 */
#			define AMAT(I_,J_)	(*(amat+(I_)*(helike.numLevels[nelem])+(J_)))
			/* MALLOC space for the  1-d array */
			/* >>chng 02 jun 19, do not always malloc */
			if( PARALLEL )
			{
				if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)(helike.numLevels[nelem]*helike.numLevels[nelem]) ))) == NULL )
					BadMalloc();
			}
			else if( !lgAMAT_alloc )
			{
				/* this branch - only malloc one time */
				lgAMAT_alloc = TRUE;
				if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)((max_num_levels)*(max_num_levels)) ))) == NULL )
					BadMalloc();
			}

			/* this is the default */
			for( j=0; j < helike.numLevels[nelem]; j++ )
			{
				for( n=0; n < helike.numLevels[nelem]; n++ )
				{
					AMAT(n,j) = z[n][j];
				}
			}
			DGETRF(helike.numLevels[nelem],helike.numLevels[nelem],
			  amat,helike.numLevels[nelem],ipiv,&nerror1);

			DGETRS('N',helike.numLevels[nelem],1,amat,helike.numLevels[nelem],ipiv,
			  creation,helike.numLevels[nelem],&nerror2);

			if( nerror1 != 0 || nerror2 != 0 )
			{
				fprintf( ioQQQ, " helike dgetrs error\n" );
				puts( "[Stop in helike]" );
				cdEXIT(EXIT_FAILURE);
			}
			if( PARALLEL )
				free( amat);
		}
		else
		{
			fprintf( ioQQQ, " chMatrix type insane in helike, was%7.7s\n", 
			  TypMatrx.chMatrix );
			puts( "[Stop in helike]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* end of branches for which matrix solution, now check if valid */

		/* this will eventually become the ratio of ionized to neutral hydrogenic 
		 * species, create sum of level pops per ion first */
		iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 0.;
		sum_popn_ov_ion = 0.;

		for( level=0; level < helike.numLevels[nelem]; level++ )
		{
			iso.Pop2Ion[ipHE_LIKE][nelem][level] = (float)creation[level];
		}

		/* zero populations of unused levels.	*/
		for( level=helike.numLevels[nelem]; level < iso.numLevels[ipHE_LIKE][nelem]; level++ )
		{
			iso.Pop2Ion[ipHE_LIKE][nelem][level] = 0.f;
		}

		lgNegPop = FALSE;
		/* >>chng 02 nov 06, have all levels included in TotIonizRate */
		/* >>chng 02 nov 03, add ct to this term */
		/* this is charge transfer ionization of this species by hydrogen and helium */
		ipHi = 0;
		ionrec.TotIonizRate[nelem][nelem-1] = 0.;
		/*iso.Pop2Ion[ipHE_LIKE][nelem][ipHi] * (Secondaries.csupra +
			ChargTran.HeCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHELIUM][1]+ 
			ChargTran.HCharExcIon[nelem][nelem-1]*dense.xIonDense[ipHYDROGEN][1]);*/

		/* >>chng 01 nov 23, increase upper limit by 2 */
		for( level=0; level < helike.numLevels[nelem]; level++ )
		{

			/* sum of all ionization processes from this atom to ion */
			ionrec.TotIonizRate[nelem][nelem-1] += 
				iso.Pop2Ion[ipHE_LIKE][nelem][level] * iso.xLevel2Cont[ipHE_LIKE][nelem][level];

			/* check for negative level populations */
			if( iso.Pop2Ion[ipHE_LIKE][nelem][level] < 0. )
				lgNegPop = TRUE;

			/* create sum of populations,
			 * iso.Pop2Ion is actually populations relative to ion */
			/*iso.pop_ion_ov_neut[ipHE_LIKE][nelem] += (float)pops[level];*/
			sum_popn_ov_ion += (float)iso.Pop2Ion[ipHE_LIKE][nelem][level];

			if( iso.PopLTE[ipHE_LIKE][nelem][level] > 0. )
			{
				/* the NLTE LTE departure coefficients */
				iso.DepartCoef[ipHE_LIKE][nelem][level] = 
					(iso.Pop2Ion[ipHE_LIKE][nelem][level]/
					(iso.PopLTE[ipHE_LIKE][nelem][level]* dense.eden) );
			}
			else
			{
				iso.DepartCoef[ipHE_LIKE][nelem][level] = 0.;
			}
		}
		/* convert back to scaled from ground */
		ionrec.TotIonizRate[nelem][nelem-1] /= MAX2(SMALLFLOAT , sum_popn_ov_ion);

		/* >>chng 02 jan 22, actually use real ratios, not just ground */
		/* this is sum of all populations in model, div by ion 
		 * create ratio of ion pops to total atom */
		if( sum_popn_ov_ion > SMALLFLOAT )
		{
			iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 1.f/sum_popn_ov_ion;
		}
		else
		{
			iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 0.;
		}
	}

	/* >> chng 02 Sep 20 rjrw: TotRecomRate should be from total not ratio, switched off for time being */
	/* >>chng 02 oct 01, back to old way when dynamics not being done.  Then results of large
	 * atom will feed back into solution in bidiag */
	ASSERT( helike.lgFSM || dynamics.lgAdvection ||
		fabs(iso.pop_ion_ov_neut[ipHE_LIKE][nelem] - 
		ionrec.TotIonizRate[nelem][nelem-1] / 
		MAX2(SMALLFLOAT,ionrec.TotRecomRate[nelem][nelem-1]) )/
		MAX2(SMALLFLOAT,iso.pop_ion_ov_neut[ipHE_LIKE][nelem]) < 0.001);

	/* check on the sum of the populations */
	if( lgNegPop || iso.pop_ion_ov_neut[ipHE_LIKE][nelem] < 0.)
	{
		fprintf( ioQQQ, 
			" HeLevel finds negative He-like ion fraction for nelem=%ld using routine HeLevel, IonFrac= %10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
		  nelem, 
		  iso.pop_ion_ov_neut[ipHE_LIKE][nelem], 
		  iso.xIonSimple[ipHE_LIKE][nelem], 
		  phycon.te, 
		  nzone );

		fprintf( ioQQQ, " level pop are:\n" );
		for( i=0; i < helike.numLevels[nelem]; i++ )
		{
			fprintf( ioQQQ," %8.1e;", iso.Pop2Ion[ipHE_LIKE][nelem][i] );
			if( (i!=0) && !(i%10) ) fprintf( ioQQQ,"\n" );
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in helike]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* get level populations, two cases, 
	 * first, may be zero since all cases drop down to here, 
	 * this branch is for trivial abundance, so zero out species */
	/* >>chng 01 apr 21, add second test on high stage being he-like -
	 * there are tests where this is hydrogen like, and jump would be over
	 * two stages of ionization - do not do this */
	if( iso.pop_ion_ov_neut[ipHE_LIKE][nelem] >= 0. && iso.pop_ion_ov_neut[ipHE_LIKE][nelem] < TOOSMALL  &&
		IonRange.IonHigh[nelem] == nelem )
	{

		/* >>chng 02 apr 10, if actually zero then set to zero, but 
		 * do not kill a very small ion abundance since must let He+
		 * remain despite very low abundance, since important CO destruction
		 * process in molecular clouds */
		if( iso.pop_ion_ov_neut[ipHE_LIKE][nelem] == 0. && !dense.lgSetIoniz[nelem] )
		{
			iso.pop_ion_ov_neut[ipHE_LIKE][nelem] = 0.;
			/* decrement highest stage of ionization, must not
			 * go up to He-like being ionized again */
			IonRange.IonHigh[nelem] = nelem-1;
			/* >>chng 01 apr 21, added second drop on IonLow -
			 * there was a model where light atoms where stripped and he-like test triggered,
			 * which resulted in insanity downstream since IonHigh and IonLow were equal */
			IonRange.IonLow[nelem] = MIN2( IonRange.IonLow[nelem] , IonRange.IonHigh[nelem]-1 );
		}

		/* now zero out high stages we just cleared out */
		for( ipHi=ipHe2s3S; ipHi < helike.numLevels[nelem]; ipHi++ )
		{
			iso.Pop2Ion[ipHE_LIKE][nelem][ipHi] = 0.;
			for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
			{
				/* population of lower level rel to ion */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopLo = 0.;

				/* population of upper level rel to ion */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi =  0.;

				/* population of lower level rel to ion, corrected for stim em */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc =  0.;

				/* local ots destruction, cm^-3 s^-1 */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].ots =  0.;
			}
		}

		ionrec.TotIonizRate[nelem][nelem-1] = 0.;

		if( trace.lgTrace && trace.lgHeBug )
		{
			fprintf(ioQQQ,"  HeLike for Z=%li finds small population set to zerozero.\n",nelem);
		}
	}
	/* this is the main branch that applies when we have non-trivial abundance */
	else
	{
		/* level populations 
		 * this brance we have significant population as sum of levels per ion,
		 * store inverse, ion per atom, here.  For non-H species this is ratio
		 * of highest to next highest stage of ionization */

		/* now find emission in each line */
		for( ipHi=0; ipHi<helike.numLevels[nelem]; ++ipHi )
		{
			for( ipLo=0; ipLo<ipHi; ++ipLo )
			{
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopLo = iso.Pop2Ion[ipHE_LIKE][nelem][ipLo];
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopHi = iso.Pop2Ion[ipHE_LIKE][nelem][ipHi];
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc = (iso.Pop2Ion[ipHE_LIKE][nelem][ipLo] - iso.Pop2Ion[ipHE_LIKE][nelem][ipHi]*
					EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gLo/EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].gHi);
				/*fixit(); some lines had strong masers, quench for now */
				EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc = MAX2(SMALLFLOAT,EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].PopOpc );
			}
		}

	}

	/* helike.lgCritDensLMix is a flag used to print warning if density is
	 * too low for first collapsed level to be l-mixed.  Check is if l-mixing collisions
	 * out of highest resolved singlet P are greater than sum of transition probs out.	*/
	/* TODO only done for helium currently...leave or apply to ions?	*/
 	if( (HighestPColOut[0] > 1./helike.Lifetime[ipHELIUM][QuantumNumbers2Index[ipHELIUM][iso.n_HighestResolved[ipHE_LIKE][ipHELIUM]][1][0]] )
		&& (HighestPColOut[1] > 1./helike.Lifetime[ipHELIUM][QuantumNumbers2Index[ipHELIUM][iso.n_HighestResolved[ipHE_LIKE][ipHELIUM]][1][1]] )
		&& nelem==ipHELIUM ) 
	{
		helike.lgCritDensLMix = TRUE;
	}
	
	for( n=0; n < helike.numLevels[nelem]; n++ )
	{
		OutOfNGoingUp[iso.quant_desig[ipHE_LIKE][nelem][n].n] += 
			CollisionsGoingUp[n]*iso.Pop2Ion[ipHE_LIKE][nelem][n];
		OutOfNGoingDown[iso.quant_desig[ipHE_LIKE][nelem][n].n] += 
			CollisionsGoingDown[n]*iso.Pop2Ion[ipHE_LIKE][nelem][n];
	}

	/* TODO at the moment can only assert these for the atom, and for Vriens gbar.	*/
	if( iso.lgColl_excite[ipHE_LIKE] && !helike.lgSetBenjamin && helike.lgCS_Vriens /*&& nelem==ipHELIUM*/ )
	{
		/* TODO try to assert this for n=3 and possibly n=2 as well	*/
		for( n = 5; n < helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem] - 2; n++  )
		{
			float DecayN = iso.Pop2Ion[ipHE_LIKE][nelem][n]/(float)helike.Lifetime[nelem][n];
			float DecayNPlusOne = iso.Pop2Ion[ipHE_LIKE][nelem][n+1]/(float)helike.Lifetime[nelem][n+1];
			/* This is not monotonic for the highest two levels...the highest because you can't go up from there!
			 * and the second highest because there is only one level it can go up to, which is too few to make 
			 * the total greater than the third highest!	*/
			/* This doesn't work well for very low density or temperature plasmas, but we don't
			 * particularly care since those have very weak collisional rates.  Only check
			 * these trends if the rates are not much smaller than the corresponding decay rate.	*/
			if( (OutOfNGoingUp[n]>(0.01*DecayN)) && (OutOfNGoingUp[n+1]>(0.01*DecayNPlusOne)) && 
				n!=helike.n_HighestResolved[nelem] )
			{
				float DecayNMinusOne = iso.Pop2Ion[ipHE_LIKE][nelem][n-1]/(float)helike.Lifetime[nelem][n-1];
			
				ASSERT( (OutOfNGoingUp[n]<OutOfNGoingUp[n+1]) || (OutOfNGoingUp[n+1]==0.) ||
					( fabs(1. - OutOfNGoingUp[n]/OutOfNGoingUp[n+1]) < 0.05 ) );
				ASSERT( (OutOfNGoingDown[n]<OutOfNGoingDown[n+1]) || (OutOfNGoingDown[n+1]==0.) ||
					( fabs(1. - OutOfNGoingDown[n]/OutOfNGoingDown[n+1]) < 0.05 ) );
			
				/* Also assert a monotonic trend in the derivative.  */
				/* There is a small discontinuity at the collapsed/resolved interface which is 
				 * probably due to an inaccuracy in assuming all collisions are hydrogenic	*/
				if( (nelem != ipHELIUM || helike.lgCritDensLMix  ) && 
					n < (helike.n_HighestResolved[nelem] + MAX2(0, (int)(0.75*helike.nCollapsed[nelem] - 1) ) ) &&
					OutOfNGoingUp[n-1]>(0.01*DecayNMinusOne) && OutOfNGoingDown[n-1]>(0.01*DecayNMinusOne))
				{
					ASSERT( (OutOfNGoingUp[n]-OutOfNGoingUp[n-1]) < (OutOfNGoingUp[n+1]-OutOfNGoingUp[n]) );
					ASSERT( (OutOfNGoingDown[n]-OutOfNGoingDown[n-1]) < (OutOfNGoingDown[n+1]-OutOfNGoingDown[n]) );
					/*fprintf( ioQQQ, "n\t%li\tOutOfNGoingUp[n]\t%.3e\tOutOfNGoingDown[n]\t%.3e\tDecayNMinusOne\t%.3e\n",
						n, OutOfNGoingUp[n], OutOfNGoingDown[n], DecayNMinusOne );*/
				}
			}
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nelem == ipHELIUM )
		{
			/* limit output to one run.	*/
			static long RUNTHISONCE = FALSE;

			if( !RUNTHISONCE )
			{
#if	0				
				fprintf( ioQQQ,"n\tl\ts\tCollisionsGoingUp\tCollisionsGoingDown\n");
				
				for( n=0; n < helike.numLevels[nelem]; n++ )
				{
					fprintf( ioQQQ,"%2ld\t%2ld\t%2ld\t%.9e\t%.9e\n",
						iso.quant_desig[ipHE_LIKE][nelem][n].n,
						iso.quant_desig[ipHE_LIKE][nelem][n].l,
						iso.quant_desig[ipHE_LIKE][nelem][n].s,
						CollisionsGoingUp[n]*iso.Pop2Ion[ipHE_LIKE][nelem][n],
						CollisionsGoingDown[n]*iso.Pop2Ion[ipHE_LIKE][nelem][n]	);
				}
#endif

				fprintf( ioQQQ,"n\tOutOfNGoingUp\tOutOfNGoingDown\n");
				
				for( n=1; n <= helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem]; n++ )
				{
					fprintf( ioQQQ,"%2ld\t%.9e\t%.9e\n",
						n, OutOfNGoingUp[n], OutOfNGoingDown[n] );
				}

				fprintf( ioQQQ, "\n" );
			}

			RUNTHISONCE = TRUE;
		}
	}

	{
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && nelem == ipOXYGEN )
		{

			/* limit output to one run.	*/
			static long RUNONCE = FALSE, nlo, nhi;
			static double **zPerN;

			if( !RUNONCE )
			{
				if( (zPerN = (double **)MALLOC(sizeof(double*)*(unsigned)
					(helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem] + 1) ) )==NULL )
					BadMalloc();

				for( i=0; i <= (helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem]); ++i )
				{
					if( (zPerN[i] = (double *)MALLOC(sizeof(double)*(unsigned)
						(helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem] + 1) ) )==NULL )
						BadMalloc();
				}

				if ( (matrix = fopen("OxyMatrix.txt","w")) == NULL )
					return;

				/* first zero array zPerN	*/
				for( nlo = 1; nlo <= helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem]; nlo++ )
				{
					for( nhi = 1; nhi <= iso.n_HighestResolved[ipHE_LIKE][ipHELIUM] + iso.nCollapsed[ipHE_LIKE][ipHELIUM]; nhi++ )
					{		
						zPerN[nlo][nhi] = 0.;
					}
				}

				/* then stuff it with values from original z matrix	*/
				for( level=ipHe1s1S; level < helike.numLevels[nelem]; level++ )
				{
					for( ipLo=ipHe1s1S; ipLo < level; ipLo++ )
					{
						zPerN[iso.quant_desig[ipHE_LIKE][nelem][ipLo].n][iso.quant_desig[ipHE_LIKE][nelem][level].n] 
							+= z[ipLo][level]*iso.Pop2Ion[ipHE_LIKE][nelem][ipLo];
					}

					for( ipHi=level; ipHi < helike.numLevels[nelem]; ipHi++ )
					{
						zPerN[iso.quant_desig[ipHE_LIKE][nelem][ipHi].n][iso.quant_desig[ipHE_LIKE][nelem][level].n] 
							+= z[ipHi][level]*iso.Pop2Ion[ipHE_LIKE][nelem][ipHi];
					}
				}


				/* then print it!	*/
				for( nlo = 1; nlo <= helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem]; nlo++ )
				{
					for( nhi = 1; nhi <= helike.n_HighestResolved[nelem] + helike.nCollapsed[nelem]; nhi++ )
					{		
						double f=fabs(zPerN[nlo][nhi]);
						fprintf( matrix, "%.2e\t", log10(MAX2(SMALLFLOAT,f)) );
					}
					fprintf( matrix, "\n" );
				}

#if	0
				/* then print it!	*/
				for( nlo = 0; nlo < helike.numLevels[nelem]; nlo++ )
				{
					for( nhi = 0; nhi < helike.numLevels[nelem]; nhi++ )
					{		
						fprintf( matrix, "%.2e\t", log10(MAX2(SMALLFLOAT,fabs(z[nlo][nhi]))) );
					}
					fprintf( matrix, "\n" );
				}
#endif			
				fprintf( matrix, "\n" );
				fclose( matrix );
			}
			
			RUNONCE = TRUE;
		}
	}

	{
		/* following should be set true to print information */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC /*&& (iteration==2)*/ )
		{
			if( nelem==ipHELIUM )
				fprintf(ioQQQ,"\n");
			fprintf(ioQQQ,"%li\t%.4e", nelem,
				iso.pop_ion_ov_neut[ipHE_LIKE][nelem] );
			if( dense.xIonDense[nelem][nelem]>SMALLFLOAT )
			{
				float ratio = dense.xIonDense[nelem][nelem]/ dense.xIonDense[nelem][nelem-1];

				fprintf(ioQQQ,"\t%.4e\t%.4e", ratio ,
					(ratio-iso.pop_ion_ov_neut[ipHE_LIKE][nelem])/ratio );
			}
			fprintf(ioQQQ,"\n");
		}

	}
	{
		/* following should be set true to print information */
		/*@-redef@*/
		enum {DEBUG_LOC=FALSE};
		/*@+redef@*/
		if( DEBUG_LOC && (nelem==ipCARBON)/**/ )
		{
			{
				fprintf( ioQQQ, "     He-like Z=%2ld H2OVH1=",nelem);
				fprintf( ioQQQ,"%10.3e", iso.pop_ion_ov_neut[ipHE_LIKE][nelem]);
				fprintf( ioQQQ, " simple=");
				fprintf( ioQQQ,"%10.3e", iso.xIonSimple[ipHE_LIKE][nelem]);
				fprintf( ioQQQ," dest, creat rates %.2e %.2e", destroy , ionrec.TotRecomRate[nelem][nelem-1]);
				fprintf( ioQQQ, "\n"); 
			}
		}
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "     He-like Z=%2ld H2OVH1=",nelem);
		fprintf( ioQQQ,"%10.3e", iso.pop_ion_ov_neut[ipHE_LIKE][nelem]);
		fprintf( ioQQQ, " simple=");
		fprintf( ioQQQ,"%10.3e", iso.xIonSimple[ipHE_LIKE][nelem]);
		fprintf( ioQQQ," dest, creat rates %.2e %.2e", SaveZ[0][0] , ionrec.TotRecomRate[nelem][nelem-1]);
		fprintf( ioQQQ, "\n"); 
	}

	/* only release this if option for parallel processing */
	if( PARALLEL )
	{
		free( ipiv );
		free( creation );
		free( OutOfNGoingUp );
		free( OutOfNGoingDown );
		free( CollisionsGoingUp );
		free( CollisionsGoingDown );
		free( error );
		free( work );

		/* now do the second dimension */
		for( i=0; i<helike.numLevels[nelem]; ++i )
		{
			free( SaveZ[i] );

			free( z[i] );
		}

		free( SaveZ );
		free( z );
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeLevel()\n", debug_fp );
#	endif
}

/* routine to punch table needed for AGN3 - collision strengths of HeI */
void AGN_He1_CS( FILE *ioPun )
{

	long int i;

	/* list of temperatures where cs will be printed */
#	define NTE 5 
	double TeList[NTE] = {6000.,10000.,15000.,20000.,25000.};
	float TempSave;

#	ifdef DEBUG_FUN
	fputs( "<+>AGN_He1_CS()\n", debug_fp );
#	endif

	/* put on a header */
	fprintf(ioPun, "Te\t2 3s 33s\n");

	/* Restore the original temp when this routine done.	*/
	TempSave = phycon.te;

	for( i=0; i<NTE; ++i )
	{
		phycon.te = (float)TeList[i];
		tfidle(FALSE);

		fprintf(ioPun , "%.0f\t", 
			TeList[i] );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3s3S , ipHe2s3S ) );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3p3P , ipHe2s3S ) );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3d3D , ipHe2s3S ) );
		fprintf(ioPun , "%.3f\t", 
			HeCSInterp( 1 , ipHe3d1D , ipHe2s3S ) );
		/*fprintf(ioPun , "%.1f\t%.1f\t%.1f\n", */
		fprintf(ioPun , "%.1f\n", 
			HeCSInterp( 1 , ipHe2p3P0 , ipHe2s3S ) +
			HeCSInterp( 1 , ipHe2p3P1 , ipHe2s3S ) +
			HeCSInterp( 1 , ipHe2p3P2 , ipHe2s3S ));
	}

	phycon.te = TempSave;
	/* no need to force update since didn't do above	*/
	tfidle(FALSE);

#	ifdef DEBUG_FUN
	fputs( " <->AGN_He1_CS()\n", debug_fp );
#	endif

	return;
#	undef NTE
}

/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
static double He_RRCoef_Te( long nelem , long n )
{
	static long int ipTe=-1;
	double rate = 0.;
	long i;

#	ifdef DEBUG_FUN
	fputs( "<+>He_RRCoef_Te()\n", debug_fp );
#	endif

	ASSERT( fabs( 1. - (double)phycon.alogte/log10((double)phycon.te) ) < 0.0001 );

	if( ipTe < 0 )
	{
		/* te totally unknown */
		if( ( phycon.alogte < (float)TeRRCoef[0] ) || 
			( phycon.alogte > (float)TeRRCoef[N_HE_TE_RECOMB-1] ) )
		{
			fprintf(ioQQQ," He_RRCoef_Te called with te out of bounds \n");
			puts( "[Stop in He_RRCoef_Te]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* now search for temperature */
		for( i=0; i<N_HE_TE_RECOMB-1; ++i )
		{
			if( ( phycon.alogte > (float)TeRRCoef[i] ) && 
				( phycon.alogte <= (float)TeRRCoef[i+1] ) )
			{
				/* found the temperature - use it */
				ipTe = i;
				break;
			}
		}
		ASSERT( (ipTe >=0) && (ipTe < N_HE_TE_RECOMB-1)  );
			
	}
	else if( phycon.alogte < (float)TeRRCoef[ipTe] )
	{
		/* temp is too low, must also lower ipTe */
		ASSERT( phycon.alogte > (float)TeRRCoef[0] );
		/* decrement ipTe until it is correct */
		while( ( phycon.alogte < (float)TeRRCoef[ipTe] ) && ipTe > 0)
			--ipTe;
	}
	else if( phycon.alogte > (float)TeRRCoef[ipTe+1] )
	{
		/* temp is too high */
		ASSERT( phycon.alogte <= (float)TeRRCoef[N_HE_TE_RECOMB-1] );
		/* increment ipTe until it is correct */
		while( ( phycon.alogte > (float)TeRRCoef[ipTe+1] ) && ipTe < N_HE_TE_RECOMB-1)
			++ipTe;
	}

	ASSERT( (ipTe >=0) && (ipTe < N_HE_TE_RECOMB-1)  );

	/* ipTe should now be valid */
	ASSERT( ( phycon.alogte >= (float)TeRRCoef[ipTe] )
		&& ( phycon.alogte <= (float)TeRRCoef[ipTe+1] ) && ( ipTe < N_HE_TE_RECOMB-1 ) );

	/* if n is equal to the number of levels, return the total recomb, else recomb for given level.	*/
	if( n == iso.numLevels[ipHE_LIKE][nelem] - iso.nCollapsed[ipHE_LIKE][nelem] )
	{
		rate = ((double)phycon.alogte - TeRRCoef[ipTe]) / (TeRRCoef[ipTe+1] - TeRRCoef[ipTe]) *
			(TotalRecomb[nelem][ipTe+1] - TotalRecomb[nelem][ipTe]) + TotalRecomb[nelem][ipTe] ;
	}
	else
	{
		rate = ((double)phycon.alogte - TeRRCoef[ipTe]) / (TeRRCoef[ipTe+1] - TeRRCoef[ipTe]) *
			(RRCoef[nelem][n][ipTe+1] - RRCoef[nelem][n][ipTe]) + RRCoef[nelem][n][ipTe] ;
	}

	/* that was the log, now make linear */
	rate = pow( 10. , rate );

#	ifdef DEBUG_FUN
	fputs( " <->He_RRCoef_Te()\n", debug_fp );
#	endif

	return rate;
}

/*HelikeCheckRecomb called by SanityCheck to confirm that recombination coef are ok
 * return value is relative error between new calculation of recom, and interp value */
double HelikeCheckRecomb(
	/* the chemical element, 1 for He */
	long nelem ,
	/* the level, 0 for ground */
	long level ,
	/* the temperature to be used */
	double temperature )
{
	double RecombRelError ,
		RecombInterp,
		RecombCalc,
		SaveTemp;

#	ifdef DEBUG_FUN
	fputs( "<+>HelikeCheckRecomb()\n", debug_fp );
#	endif

	/* this global variable is needed by radrecomb */
	EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][level];

	/* first set temp todesired value */
	SaveTemp = phycon.te;
	phycon.te = (float)temperature;
	tfidle(TRUE);

	/* actually calculate the recomb coef from the Milne relation,
	 * normally only done due to compile he-like command */
	RecombCalc = radrecomb( temperature , nelem , level );

	/* interpolate the recomb coef, this is the usual method */
	RecombInterp = He_RRCoef_Te( nelem , level );

	/* reset temp */
	phycon.te = (float)SaveTemp;
	tfidle(TRUE);

	RecombRelError = ( RecombInterp - RecombCalc ) / MAX2( RecombInterp , RecombCalc );

#	ifdef DEBUG_FUN
	fputs( " <->HelikeCheckRecomb()\n", debug_fp );
#	endif

	return RecombRelError ;
}

static void ContinuumLowering( long nelem )
{
	double a;
	long np, nd, ns, nc;
	
	/* size of rate matrices will be defined according to the n calculated here	*/
	
	ASSERT( nelem < LIMELM );

	/* Particle packing - the density here should be density of all nuclei in the plasma */
	a = sqrt( 1.8887E8 * (nelem+1.) / pow(phycon.TotalNuclei, 0.333) );
	if( a > (double)iso.n_HighestResolved[ipHE_LIKE][nelem]+(double)iso.nCollapsed[ipHE_LIKE][nelem] )
	{
		np = iso.n_HighestResolved[ipHE_LIKE][nelem]+iso.nCollapsed[ipHE_LIKE][nelem] + 1;
	}
	else
		np = (long)a;

	/* Debye shielding - the density here is electron density	*/
	a = 2.6E7 * nelem * nelem * pow( phycon.te/dense.eden, 0.25);
	if( a > (double)iso.n_HighestResolved[ipHE_LIKE][nelem]+(double)iso.nCollapsed[ipHE_LIKE][nelem] )
	{
		nd = iso.n_HighestResolved[ipHE_LIKE][nelem]+iso.nCollapsed[ipHE_LIKE][nelem] + 1;
	}
	else
		nd = (long)a;
	
	/* Stark broadening - this should be the density of singly charged ions, 
	 * both positive and negative.  The sum of protons, electrons, and HeII should be
	 * good enough.	*/
	a = 3171. * pow( (double)nelem, 0.8 ) * pow( dense.eden + (double)dense.xIonDense[ipHYDROGEN][1]
		+ (double)dense.xIonDense[ipHELIUM][1], -0.1333);
	if( a > (double)iso.n_HighestResolved[ipHE_LIKE][nelem]+(double)iso.nCollapsed[ipHE_LIKE][nelem] )
	{
		ns = iso.n_HighestResolved[ipHE_LIKE][nelem]+iso.nCollapsed[ipHE_LIKE][nelem] + 1;
	}
	else
		ns = (long)a;
	
	ASSERT( np > 3 );
	ASSERT( nd > 3 );
	ASSERT( ns > 3 );

	nc = MIN3(np, nd, ns);
	
	/* I assert greater than three because the code depends upon having at least up to n=3, and
	 * because it would take EXTREMELY dense conditions to lower the continuum that much, and
	 * the code would probably get a wrong answer then anyway.  */
	ASSERT( nc > 3 );

	if( nc < iso.n_HighestResolved[ipHE_LIKE][nelem])
	{
		helike.n_HighestResolved[nelem] = nc;
		helike.nCollapsed[nelem] = 0;
		helike.numLevels[nelem] = nc*nc + nc + 1;
		helike.lgFullSize[nelem] = TRUE;
		iso.numPrintLevels[ipHE_LIKE][nelem] = helike.numLevels[nelem];
	}
	else if( nc <= iso.n_HighestResolved[ipHE_LIKE][nelem] + iso.nCollapsed[ipHE_LIKE][nelem] )
	{
		helike.n_HighestResolved[nelem] = iso.n_HighestResolved[ipHE_LIKE][nelem];
		helike.nCollapsed[nelem] = nc - helike.n_HighestResolved[nelem];
		helike.numLevels[nelem] = helike.n_HighestResolved[nelem]*helike.n_HighestResolved[nelem] +
			helike.n_HighestResolved[nelem] + 1 + helike.nCollapsed[nelem];
		helike.lgFullSize[nelem] = TRUE;
		iso.numPrintLevels[ipHE_LIKE][nelem] = helike.numLevels[nelem] - helike.nCollapsed[nelem];
	}
	else
	{
		helike.numLevels[nelem] = iso.numLevels[ipHE_LIKE][nelem];
		helike.nCollapsed[nelem] = iso.nCollapsed[ipHE_LIKE][nelem];
		helike.n_HighestResolved[nelem] = iso.n_HighestResolved[ipHE_LIKE][nelem];
		helike.lgFullSize[nelem] = FALSE;
	}

	ASSERT( helike.numLevels[nelem] <= iso.numLevels[ipHE_LIKE][nelem] );
	ASSERT( helike.nCollapsed[nelem] <= iso.nCollapsed[ipHE_LIKE][nelem] );
	ASSERT( helike.n_HighestResolved[nelem] <= iso.n_HighestResolved[ipHE_LIKE][nelem] );

	return;
}

static double CheckIfBelowPlasFreq( long nelem, long ipHi, long ipLo )
{
	double energy, Kill = 1.;
	long nHi, nLo;

	energy = EmisLines[ipHE_LIKE][nelem][ipHi][ipLo].EnergyWN;
	nHi = iso.quant_desig[ipHE_LIKE][nelem][ipHi].n;
	nLo = iso.quant_desig[ipHE_LIKE][nelem][ipLo].n;

	if( energy <= 0. )
	{
		Kill = 0.;
	}
	/* Must zero decays with frequencies below the plasma frequency.	*/
	else if( energy*SPEEDLIGHT<plasnu.plsfrq ) 
	{
		Kill = 0.;
	}
	else if( nHi==nLo && plasnu.plsfrq > 
		( POW2((double)nelem) * (1./POW2(nHi-1.)-1./POW2(nHi)) * TE1RYD/phycon.te ) ) 
	{
		Kill = 0.;
	}
	else
	{
		Kill = 1.;
	}

	return Kill;
}

/*prt_He_like_DeparCoef routine to print departure coefficients for he-like species */
void prt_He_like_DeparCoef(long int nelem )
{
	long in, il, is, i, ipLo, nResolved, ipFirstCollapsed=LONG_MIN;
	
	for( is = 0; is<=1; ++is)
	{
		char chSpin[2][9]={"singlets","triplets"};

		ipFirstCollapsed= helike.numLevels[nelem]-helike.nCollapsed[nelem];
		nResolved = iso.quant_desig[ipHE_LIKE][nelem][ipFirstCollapsed-1].n;
		ASSERT( nResolved == helike.n_HighestResolved[nelem] );
		ASSERT(nResolved > 0 );
		
		/* give element number and spin */
		fprintf(ioQQQ," He-like %s  %s departure\n",
			elementnames.chElementSym[nelem],
			chSpin[is]);

		/* header with the l states */
		fprintf(ioQQQ," n\\l=>         ");
		for ( i =0; i < nResolved ; ++i)
		{
			fprintf(ioQQQ,"%2ld       ",i);
		}
		fprintf(ioQQQ,"\n");
	
		/* loop over prin quant numbers, one per line, with l across */
		for( in = is+1; in <= nResolved; ++in)
		{
			fprintf(ioQQQ," %2ld           ",in);
			for( il = 0; il < in; ++il)
			{
				if ( (in==2) && (il==1) && (is==1) )
				{
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHE_LIKE][nelem][ipHe2p3P0]));
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHE_LIKE][nelem][ipHe2p3P1]));
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHE_LIKE][nelem][ipHe2p3P2]));
				}
				else
				{
					ipLo = QuantumNumbers2Index[nelem][in][il][is];
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHE_LIKE][nelem][ipLo]));
				}
			}
			fprintf(ioQQQ,"\n");
		}
	}		
	/* above loop was over spin, now do collapsed levels, no spin or ang momen */
	for( il = ipFirstCollapsed; il < helike.numLevels[nelem]; ++il)
	{
		in = iso.quant_desig[ipHE_LIKE][nelem][il].n;

		/* prin quan number of collapsed levels */
		fprintf(ioQQQ," %2ld           ",in);
		fprintf( ioQQQ,PrintEfmt("%9.2e", iso.DepartCoef[ipHE_LIKE][nelem][il]));
		fprintf(ioQQQ,"\n");
	}		
	
	return;
}

/*prt_He_like_Pops routine to print level populations for he-like species */
void prt_He_like_Pops(long int nelem )
{
	long in, il, is, i, ipLo, nResolved, ipFirstCollapsed=LONG_MIN;

	for( is = 0; is<=1; ++is)
	{
		char chSpin[2][9]={"singlets","triplets"};

		ipFirstCollapsed= helike.numLevels[nelem]-helike.nCollapsed[nelem];
		nResolved = iso.quant_desig[ipHE_LIKE][nelem][ipFirstCollapsed-1].n;
		ASSERT( nResolved == helike.n_HighestResolved[nelem] );
		ASSERT(nResolved > 0 );
	
		/* give element number and spin */
		fprintf(ioQQQ," He-like %s  %s populations\n",
			elementnames.chElementSym[nelem],
			chSpin[is]);

		/* header with the l states */
		fprintf(ioQQQ," n\\l=>         ");
		for ( i =0; i < nResolved ; ++i)
		{
			fprintf(ioQQQ,"%2ld       ",i);
		}
		fprintf(ioQQQ,"\n");

		/* loop over prin quant numbers, one per line, with l across */
		for( in = is+1; in <= nResolved; ++in)
		{
			fprintf(ioQQQ," %2ld           ",in);

			for( il = 0; il < in; ++il)
			{
				if ( (in==2) && (il==1) && (is==1) )
				{
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHE_LIKE][nelem][ipHe2p3P0]));
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHE_LIKE][nelem][ipHe2p3P1]));
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHE_LIKE][nelem][ipHe2p3P2]));
				}
				else
				{
					ipLo = QuantumNumbers2Index[nelem][in][il][is];
					fprintf( ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHE_LIKE][nelem][ipLo]));
				}
			}
			fprintf(ioQQQ,"\n");
		}
	}
	/* above loop was over spin, now do collapsed levels, no spin or ang momen */
	for( il = ipFirstCollapsed; il < helike.numLevels[nelem]; ++il)
	{
		in = iso.quant_desig[ipHE_LIKE][nelem][il].n;
		/* prin quan number of collapsed levels */
		fprintf(ioQQQ," %2ld           ",in);
		fprintf( ioQQQ,PrintEfmt("%9.2e", iso.Pop2Ion[ipHE_LIKE][nelem][il]));
		fprintf(ioQQQ,"\n");
	}

	return;
}

/*He_ParseAtom parse parameters off the helium command */
void He_ParseAtom(char *chCard )
{
	long int i, 
		nHe,
		nelem;
	int lgEOL;

#	ifdef DEBUG_FUN
	fputs( "<+>He_ParseAtom()\n", debug_fp );
#	endif

	/* collisions - don't pick up the levels collapsed command */
	if( lgMatch("COLL",chCard) && !lgMatch("LEVE" , chCard ) )
	{
		if( lgMatch("EXCI",chCard) )
		{
			/* turn off collisional excitation */
			iso.lgColl_excite[ipHE_LIKE] = FALSE;
			physok.lgPhysOK = FALSE;
		}
		else if( lgMatch("IONI",chCard) )
		{
			/* turn off collisional ionization */
			iso.lgColl_ionize[ipHE_LIKE] = FALSE;
			physok.lgPhysOK = FALSE;
		}
		else if( lgMatch("L-MI",chCard) )
		{
			/* >>chng 02 feb 07, change from 2s2p to l-mixing */
			/* this is the atom he-like collision l-mixing command */
			/* turn off same-n collisions */
			iso.lgColl_l_mixing[ipHE_LIKE] = FALSE;
			physok.lgPhysOK = FALSE;
		}
		else
		{
			/* turn everything off, since no keyword given */
			iso.lgColl_excite[ipHE_LIKE] = FALSE;
			iso.lgColl_ionize[ipHE_LIKE] = FALSE;
			iso.lgColl_l_mixing[ipHE_LIKE] = FALSE;
			physok.lgPhysOK = FALSE;
		}
	}

	else if( lgMatch("FSM ",chCard) )
	{
		/* turn on fine structure mixing of spontaneous decays.  
		 * >>refer	Helike	FSM	Bauman, R., MacAdams, K., and Ferland, G. (2003).	*/
		i = 5;
		helike.lgFSM = (int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( helike.lgFSM > 2 )
		{
			fprintf( ioQQQ, "   FSM number must be 0, 1, or 2.\n");
		}
	}

	else if( lgMatch("GBAR",chCard) )
	{
		/* the HEGBAR command - to change cs of higher levels */
		/* first turn all off, one will be turned back on */
		helike.lgCS_Vriens = FALSE;
		helike.lgCS_None = FALSE;
		helike.lgCS_new = FALSE;

		/* now turn one on */
		if( lgMatch("VRIE",chCard) )
		{
			/* option to change how collisions for unknown levels affect things */
			helike.lgCS_Vriens = TRUE;
		}
		else if( lgMatch(" NEW",chCard) )
		{
			/* option to change how collisions for unknown levels affect things */
			i = 5;
			helike.lgCS_new = (int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

			/* there are two options, 1 and 2, for which fit - default (no number)
			 * will be 1, the broken power law fit */
			if( lgEOL )
				helike.lgCS_new = 1;

			ASSERT( helike.lgCS_new );
		}
		else if( lgMatch(" OFF",chCard) )
		{
			/* option to change how collisions for unknown levels affect things */
			helike.lgCS_None = TRUE;
		}
		else
		{
			fprintf( ioQQQ, " needs parameter\n" );
			puts( "[Stop in ParseSet]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	else if( lgMatch("BENJ",chCard) )
	{
		/* use set of physics to replicate Benjamin et al HeI papers */
		/* affects are:
				use his recombination coefficients, rather than Milne relation results
				do not use fake 0.1 collisions for transitions with no data
				no collisional excitation to n=5 
				forces max n to 5	*/

		helike.lgSetBenjamin = TRUE;
	}

	else if( lgMatch("LYMA",chCard) )
	{
		/* option to set number of "extra" lyman lines, used for optical depths only */
		i = 5;
		iso.nLyman[ipHE_LIKE] = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
			NoNumb(chCard);
	}

	else if( lgMatch("LEVE",chCard) )
	{
		/* the number of levels read in is n, the principal quantum number
		 * only lines with upper levels less than n will be printed */

		/* number of levels for helium iso-sequence */
		/* there are two options here,
		 * when keyword ELEMENT appears, scan off element name and change levels only for
		 * that one.
		 * when there is no ELEMENT then set all he-like levels to same number */

		/* lgHydroMalloc is FALSE at start of calculation, set true when space 
		 * allocated for the hydrogen and helium lines.  Once done we must ignore all 
		 * future changes in the number of levels */
		if( !lgHydroMalloc )
		{
			i = 5;
			nHe = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

			if( lgEOL )
			{
				/* must be a number, or a key, either large (or limit) or small
				 * if no number but LARGER or SMALL then set to preset number */
				if( lgMatch("LARG",chCard) )
				{
					/* there is no real limit, but this includes all levels with tabulated rec coef
					 * set to 20 in helike.h*/
					nHe = HE_RREC_MAXN;
				}

				/* this is small or compact keyword */
				else if( lgMatch("SMAL",chCard) || lgMatch("COMP",chCard) )
				{
					/* a small atom only has n=3, 13 levels */
					nHe = 3;
				}

				/* this is limit keyword, sets to largest possible value */
				else
				{
					/* punch out if no number */
					NoNumb(chCard);
				}
			}
			else if( ( nHe < 3 ) && !lgMatch("COLL",chCard) )
			{
				/* only enforce this limit if not working on collapsed levels */
				fprintf( ioQQQ, " cannot have fewer than 3 levels, the requested number was %li\n" ,
					nHe  );
				fprintf( ioQQQ, " Sorry.\n" );
				puts( "[Stop in He_ParseAtom]" );
				cdEXIT(EXIT_FAILURE);
			}

			if( lgMatch("COLL",chCard) )
			{
				if( (nHe<1) && !helike.lgSetBenjamin )
				{
					fprintf( ioQQQ, "He-like: there must be at least one collapsed level.\n");
					puts( "[Stop in He_ParseAtom]" );
					cdEXIT(EXIT_FAILURE);
				}

				if( lgEOL )
					NoNumb(chCard);

				/* set collapsed levels for entire sequence or just one element */
				if( lgMatch("ELEM",chCard) )
				{
					/* check which element is on the line, nelem is atomic number on C scale */
					nelem = GetElem(chCard);
					/* number of collapsed levels for the he atoms,
					* these are the highest levels and are unresolved in l and s */
					iso.nCollapsed[ipHE_LIKE][nelem] = nHe;
				}
				else
				{
					/* keyword element did not appear, so set all he-like species to this number */
					for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
					{
						iso.nCollapsed[ipHE_LIKE][nelem] = nHe;
					}
				}
			}

			/* not collapsed, not this is total number of levels */
			/* we now have the desired number of levels, and it has been fully qualified. 
			 * now check whether ELEMENT keyword is on line, if not then set all to
			 * the number, if so then only element in use */
			else if( lgMatch("ELEM",chCard) )
			{
				/* check which element is on the line, nelem is atomic number on C scale */
				nelem = GetElem(chCard);
				iso.n_HighestResolved[ipHE_LIKE][nelem] = nHe;
				iso.numLevels[ipHE_LIKE][nelem] = nHe*nHe + nHe + 1;
			}
			else
			{
				/* keyword element did not appear, so set all he-like species to this number */
				for( i=ipHELIUM; i<LIMELM; ++i )
				{
					iso.n_HighestResolved[ipHE_LIKE][i] = nHe;
					iso.numLevels[ipHE_LIKE][i] = nHe*nHe + nHe + 1;
				}
			}
		}
	}

	else if( lgMatch("TOPO",chCard) )
	{
		fprintf( ioQQQ, " topoff has no effect on helike sequence...did you mean collapsed?" );			
	}

	else
	{
		fprintf( ioQQQ, " There should have been a keyword - STOP\n" );
		puts( "[Stop in He_ParseAtom]" );
		cdEXIT(EXIT_FAILURE);
	}

#	ifdef DEBUG_FUN
	fputs( " <->He_ParseAtom()\n", debug_fp );
#	endif
	return;
}
/*lint +e662 creation of  out of bound pointer */

#undef PARALLEL
/*lint +e662 creation of  out of bound pointer */
/*lint +e661 creation of  out of bound pointer */

