/* This file is part of Cloudy and is copyright (C) 1978-2003 by Gary J. Ferland.
 * For conditions of distribution and use, see copyright notice in license.txt */
/*ConvPresTempEdenIoniz solve for current pressure, calls PressureChange, ConvTempEdenIonize,
 * called by cloudy */
/*ConvFail handle conergece failure */
#include "cddefines.h"
#include "phycon.h"
#include "prt.h"
#include "called.h"
#include "rt.h"
#include "dense.h"
#include "map.h"
#include "wind.h"
#include "thermal.h"
#include "heat.h"
#include "pressure.h"
#include "trace.h"
#include "converge.h"

/* the limit to the number of loops */
/* >>chng 02 jun 13, from 40 to 50 */
#define LOOPMAX 50

/* ----------------------------------------------------------------------------- */

/*ConvFail handle conergece failure */
static void ConvFail(char chMode[]); 

void ConvPresTempEdenIoniz(void)
{
	long int loop,
		LoopMax=LOOPMAX;
	double Error,
		hden_old ,
		hden_chng_old ,
		hden_chng,
		pres_old ,
		pres_chng_old ,
		pres_chng,
		old_slope,
		rel_slope,
		slope;
	int lgPresOscil;
	float TemperatureInitial;

#	ifdef DEBUG_FUN
	fputs( "<+>ConvPresTempEdenIoniz()\n", debug_fp );
#	endif

	/* this will count number of times we call ConvIonizeOpacityDo in this zone,
	 * counter is incremented there */
	conv.nPres2Ioniz = 0;
	loop = 0;

	/* this will be the limit, which we will increase if no oscillations occur */
	LoopMax = LOOPMAX;
	/* set the initial temperature to the current value, so we will know
	 * if we are trying to jump over a thermal front */
	TemperatureInitial = phycon.te;

	/* this will be flag to check for pressure oscillations */
	lgPresOscil = FALSE;

	/* this could be set true somewhere, if we implode */
	conv.lgAbort = FALSE;

	/* we will use these to check whether hden oscillating - would need to decrease step size */
	hden_old = dense.gas_phase[ipHYDROGEN];
	hden_chng = 0.;
	pres_old = pressure.PresTotlCurr;
	pres_chng = 0.;

	/* this is slope of dP / dN relation, see how to change n to get P */
	slope = 1.;
	rel_slope = 0.;

	if( trace.lgTrConvg>=1  )
	{
		fprintf( ioQQQ, 
			" ConvPresTempEdenIoniz1 entered, will call ConvIoniz to initialize\n");
	}

	/* converge the ionization first, so that we know where we are, and have
	 * a valid foundation to begin the search */
	/* the true electron density phycon.EdenTrue is set in esum called by ConvIonizeOpacityDo */

	/* chng 02 dec 11 rjrw -- ConvIoniz() => ConvTempEdenIoniz() here for consistency with inside loop */
	/* ConvIoniz(); */
	ConvTempEdenIoniz();

	/* this evaluates current pressure, and returns whether or not 
	 * it is within tolerance of correct pressure */
	conv.lgConvPres = FALSE; /* lgConvPres(); */

	/* convergence trace at this level */
	if( trace.lgTrConvg>=1  )
	{
		fprintf( ioQQQ, 
			" ConvPresTempEdenIoniz1 ConvIoniz found following converged: Pres;%c, Eden;%c, Temp;%c, Ion:%c\n", 
			TorF(conv.lgConvPres) , 
			TorF(lgConvEden() ),
			TorF(lgConvTemp() ) ,
			TorF(conv.lgConvIoniz) );
	}

	/* >>chng 01 apr 01, add test for at least 2 loops to get better pressure convergence */
	/* >>chng 01 oct 31, add test for number of times converged, for constant
	 * pressure will get two valid solutions */
	/*while( (loop < LoopMax) && !(conv.lgConvPres  &&nPresConverged > 1) &&  !conv.lgAbort )*/
	/* >>chng 02 dec 12, do not demand two constant pressure being valid - should not be
	 * necessary if first one really is valid, and this caused unneeded second evaluation
	 * in the constant density cases */
	while( (loop < LoopMax) && !conv.lgConvPres &&  !conv.lgAbort )
	{
		/* >>chng 01 aug 24, if large change in temperature allow lots more loops */
		if( fabs( TemperatureInitial - phycon.te )/phycon.te > 0.3 )
			LoopMax = 2*LOOPMAX;

		/* change current densities of all constituents if necessary, 
		 * PressureChange evaluates lgPresOK, true if pressure is now ok
		 * sets CurrentPressure and CorrectPressure */
		hden_old = dense.gas_phase[ipHYDROGEN];
		pres_old = pressure.PresTotlCurr;

		/* convergence trace at this level */
		if( trace.lgTrConvg>=1  )
		{
			fprintf( ioQQQ, 
				" ConvPresTempEdenIoniz1 loop %li, will call PressureChange then ConvTempEdenIoniz.\n",
				loop );
		}
		/* this will update the densities and set conv.lgConvPres */
		if (PressureChange( lgPresOscil , slope )) 
		{

		/* heating cooling balance while doing ionization,
		 * this is where the heavy lifting is done, this calls PresTotCurrent,
		 * which sets pressure.PresTotlCurr */
			ConvTempEdenIoniz();
		}

		/* if product of these two is negative then hden is oscillating */
		hden_chng_old = hden_chng;
		pres_chng_old = pres_chng;
		hden_chng = dense.gas_phase[ipHYDROGEN] - hden_old;
		if( fabs(hden_chng)<SMALLFLOAT)
			hden_chng = sign(SMALLFLOAT , hden_chng );
		pres_chng = pressure.PresTotlCurr - pres_old;
		old_slope = rel_slope;
		rel_slope = (pres_chng/pressure.PresTotlCurr) / (hden_chng/dense.gas_phase[ipHYDROGEN]);

		{
			/*@-redef@*/
			enum{DEBUG_LOC=FALSE};
			/*@+redef@*/
			if( DEBUG_LOC && nzone > 150 && iteration > 1 )
			{
				fprintf(ioQQQ,"%li\t%.2e\t%.2e\t%.2e\n", 
					nzone,
					pressure.PresTotlCurr, 
					pressure.PresTotlCorrect,
					(pressure.PresTotlCorrect - pressure.PresTotlCurr)*100./pressure.PresTotlCorrect
					) ;
			}
		}

		/* check whether pressure is oscillating */
		/* >>chng 02 may 31, add check on sign on hden changes */
		/* >>chng 02 jun 05, add loop > 1 so that don't trigger off old vals that have
		 * not stabilized yet */
		if( ( ( pres_chng*pres_chng_old < 0. )
			||( hden_chng*hden_chng_old < 0. ) ) && loop > 1)
		{
			/* the sign of the change in pressure has changed, so things
			 * are oscillating.  This would be a problem */
			lgPresOscil = TRUE;
			/* slope is how pressure changes with density - pass this to
			 * changing routine if it is stable */
			if( loop > 4 && old_slope*rel_slope > 0. )
				slope = rel_slope;
			/*fprintf(ioQQQ,"oscilll %li %.2e %.2e %.2e %.2e slope %.2e\n", 
				loop ,
				pres_chng, 
				pres_chng_old,
				hden_chng , 
				hden_chng_old ,
				rel_slope);*/
		}

		/* convergence trace at this level */
		if( trace.lgTrConvg>=1  )
		{
			fprintf( ioQQQ, 
				" ConvPresTempEdenIoniz1 %4ld Curr Pres:%.4e Corr Pres:%.4e err:%6.3f%% Hden:%.4e Te:%.4e Osc:%c\n", 
			  loop, 
			  pressure.PresTotlCurr, 
			  pressure.PresTotlCorrect, 
			  /* this is percentage error */
			  100.*(pressure.PresTotlCurr - pressure.PresTotlCorrect )/pressure.PresTotlCorrect,
			  dense.gas_phase[ipHYDROGEN], 
			  phycon.te,
			  TorF(lgPresOscil)  );
		}

		/* increment loop counter */
		++loop;

		/* if we hit limit of loop, but no oscillations have happened, then we are
		 * making progress, and can keep going */
		if( loop == LoopMax && !lgPresOscil )
		{
			LoopMax = MIN2( 100 , LoopMax*2 );
		}
	}

	/* >>chng 01 mar 14, only call ConvFail one time, no matter how
	 * many failures occured below.  Had been series of if, so multiple
	 * calls per failure possible.  This change means that logic in ConvFail
	 * for abort is simpler. This may be set true in PressureChange */
	if( !conv.lgConvPres )
	{
		/* announce the pressure failure */
		ConvFail("pres");
	}

	else if( !conv.lgConvTemp )
	{
		/* announce the temperature failure */
		ConvFail("temp");
	}

	/* say something if we did not converge */
	else if( !conv.lgConvEden )
	{
		ConvFail("eden") ;
	}

	/* say something if we did not converge */
	else if( !conv.lgConvIoniz )
	{
		ConvFail("ioni") ;
	}

	/* remember mean and largest errors on electron density */
	Error = fabs(dense.eden - phycon.EdenTrue)/phycon.EdenTrue;
	if( Error > conv.BigEdenError )
	{
		conv.BigEdenError = (float)Error;
		phycon.nzEdenBad = nzone;
	}
	conv.AverEdenError += (float)Error;

	/* remember mean and largest errors between heating and cooling */
	Error = fabs(thermal.ctot - heat.htot) / thermal.ctot;
	conv.BigHeatCoolError = MAX2((float)Error , conv.BigHeatCoolError );
	conv.AverHeatCoolError += (float)Error;

	/* remember mean and largest pressure errors */
	Error = fabs(pressure.PresTotlCurr - pressure.PresTotlCorrect) / pressure.PresTotlCorrect;
	conv.BigPressError = MAX2((float)Error , conv.BigPressError );
	conv.AverPressError += (float)Error;

	/* this is only a sanity check that the summed continua accurately reflect
	 * all of the individual components.  Only include this when NDEBUG is not set,
	 * we are in not debug compile */
#	if !defined(NDEBUG)
	RT_OTS_ChkSum(0);
#	endif

#	ifdef DEBUG_FUN
	fputs( " <->ConvPresTempEdenIoniz()\n", debug_fp );
#	endif
	return;
}

/* ----------------------------------------------------------------------------- */

/*ConvFail handle conergece failure */
static void ConvFail(char chMode[]) /* chMode[5] */
{
	double relerror ;

#	ifdef DEBUG_FUN
	fputs( "<+>ConvFail()\n", debug_fp );
#	endif

	/* >>chng 02 jun 15, add this branch */
	/* do not announce a convergence failure - this was an abort before
		* convergence was acheived */
	if( conv.lgAbort )
	{
		/* an abort is not one of the failures we deal with - simply return and
		 * let something else handle this */
		/*fprintf( ioQQQ, " ConvFail - abort has been set.\n");*/
#		ifdef DEBUG_FUN
		fputs( " <->ConvFail()\n", debug_fp );
#		endif
		return;
	}

	/* pressure failure */
	if( strcmp( chMode , "pres" )==0 )
	{
		/* record number of pressure failures */
		++conv.nPreFail;
		if( called.lgTalk )
		{
			fprintf( ioQQQ, 
				" PROBLEM: pressure not converged; zone%4ld Te:%11.2e Hden:%10.2e curr Pres:%10.2e Corr Pres%10.2e\n", 
			  nzone, 
			  phycon.te, 
			  dense.gas_phase[ipHYDROGEN], 
			  pressure.PresTotlCurr, 
			  pressure.PresTotlCorrect );

			/* this identifies new dynamics that failed near the sonic point */
			if( fabs(pressure.PresGasCurr - pressure.PresRamCurr)/pressure.PresGasCurr < 0.1 &&
				((strcmp(dense.chDenseLaw,"WIND") == 0) && wind.windv < 0. ) )
			{
				fprintf( ioQQQ, 
					"\n PROBLEM: pressure not converged; we are stuck at the sonic point.\n\n");
				pressure.lgSonicPoint = TRUE;
			}
		}
	}

	/* electron density failure */
	else if( strcmp( chMode, "eden" ) == 0 )
	{
		/* record number of electron density failures */
		++conv.nNeFail;

		if( called.lgTalk )
		{
			fprintf( ioQQQ, 
				" PROBLEM: electron density fails, correct=%11.3e "
				" assumed=%11.3e zone=%3ld.", 
			  phycon.EdenTrue, dense.eden, nzone );

			/* some extra information that may be printed */
			/* heating cooling failure */
			if( !conv.lgConvTemp )
			{
				fprintf( ioQQQ, "  Temperature failure also." );
			}

			/* heating cooling failure */
			if( !conv.lgConvIoniz )
			{
				fprintf( ioQQQ, "  Ionization failure also." );
			}

			/* electron density is oscillating */
			if( conv.lgEdenOscl )
			{
				fprintf( ioQQQ, "  Electron density oscillating." );
			}

			/* heating cooling oscillating */
			if( conv.lgCmHOsc )
			{
				fprintf( ioQQQ, "  Heating-cooling oscillating." );
			}
		}
		fprintf( ioQQQ, " \n");
	}

	else if( strcmp( chMode, "ioni" ) == 0 )
	{
		/* ionization failure */
		++conv.nIonFail;
		if( called.lgTalk )
		{
			fprintf( ioQQQ, " PROBLEM: ionization not converged %3ld Z=%4ld \n", 
			  conv.nIonFail, nzone );
		}
	}

	/* rest of routine is temperature failure */
	else if( strcmp( chMode, "temp" ) == 0 )
	{
		++conv.nTeFail;
		if( called.lgTalk )
		{
			fprintf( ioQQQ, 
				" PROBLEM: te fail %2ld Z=%4ld TE=%10.4e ERROR=%10.2e"
				" HTOT=%10.2e CTOT=%10.2e DTe=%10.2e IonDone%1c\n", 
			  conv.nTeFail, 
			  nzone, 
			  phycon.te, 
			  (heat.htot - thermal.ctot)/ heat.htot, 
			  heat.htot, 
			  thermal.ctot, 
			  thermal.dTemper, 
			  TorF(conv.lgConvIoniz) );

			if( conv.lgTOscl && conv.lgCmHOsc )
			{
				fprintf( ioQQQ, " Temperature and d(Cool-Heat)/dT were both oscillating.\n" );
			}

			else if( conv.lgTOscl )
			{
				fprintf( ioQQQ, " Temperature was oscillating.\n" );
			}

			else if( conv.lgCmHOsc )
			{
				fprintf( ioQQQ, " d(Cool-Heat)/dT was oscillating.\n" );
			}

			/* not really a temperature failure, but something else */
			if( !conv.lgConvIoniz )
			{
				fprintf( ioQQQ, " Solution not converged due to %10.10s\n", 
				  conv.chConvIoniz );
			}
		}
	}
	else
	{
		fprintf( ioQQQ, " ConvFail called with insane mode %s \n", 
		  chMode );
		ShowMe();
		puts( "[Stop in ConvFail]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* increment total number of failures */
	++conv.nTotalFailures;

	/* now see how many total failures we have, and if it is time to abort */
	/* remember which zone this is */
	conv.ifailz[MIN2(conv.nTotalFailures,10)-1] = nzone;

	/* remember the relative error
	 * convert to single precision for following max, abs (vax failed here) */
	relerror = fabs((heat.htot-thermal.ctot)/ heat.htot);

	conv.failmx = MAX2(conv.failmx,(float)relerror);
	/*conv.failmx = (float)fmax(conv.failmx,fabs((float)((heat.htot-thermal.ctot)/
	  heat.htot)));*/


	/* this branch is non-abort exit - we have not exceeded the limit to the number of failures */
	if( conv.nTotalFailures < conv.LimFail )
	{ 
#		ifdef DEBUG_FUN
		fputs( " <->ConvFail()\n", debug_fp );
#		endif
		return;
	}

	/* punt */
	if( called.lgTalk )
	{
		fprintf( ioQQQ, " Stop after%3ld convergence failures.\n", 
		  conv.LimFail );
		fprintf( ioQQQ, " This limit can be reset with the FAILURES command.\n" );

		/* only do map if requested */
		/* adjust range of punting map */
		map.RangeMap[0] = (float)(phycon.te/100.);
		map.RangeMap[1] = (float)MIN2(phycon.te*100.,9e9);

		if( conv.lgMap )
		{
			/* need to make printout out now, before disturbing soln with map */
			PrtZone();
			map_do(ioQQQ,"punt");
		}
	}

	/* return out from here and let lgEndFun catch lgBusted set,
	 * and generate normal output there */
	called.lgBusted = TRUE;
	if( called.lgTalk )
	{
		fprintf( ioQQQ, " ConvFail sets lgBusted since nTeFail>=LimFail%5ld%5ld\n", 
		  conv.nTeFail, conv.LimFail );
	}

#	ifdef DEBUG_FUN
	fputs( " <->ConvFail()\n", debug_fp );
#	endif
	return;
}
#undef LOOPMAX 
