/*ParseOptimize parse the optimize command line */
/*GetOptColDen read observed column densities & errors for optimizer */
/*GetOptLineInt parse observed line intensites for optimization routines */
#include "cddefines.h"
#include "trace.h"
#include "readar.h"
#include "varypar.h"
#include "input.h"
#include "prt.h"
#include "parse.h"
#ifdef DEFERR
#	undef DEFERR
#endif
#define	DEFERR	0.05f

/*GetOptLineInt parse observed line intensites for optimization routines */
static void GetOptLineInt(char *chCard );

/*GetOptColDen read observed column densities & errors for optimizer */
static void GetOptColDen(char *chCard );

void ParseOptimize(char *chCard)
{
	int lgEOL;
	long int i;
#	ifdef __unix
	long int dum;
#	endif

#	ifdef DEBUG_FUN
	fputs( "<+>ParseOptimize()\n", debug_fp );
#	endif

	/* this must come first so that contents of filename do not trigger wrong command */
	if( lgMatch("FILE",chCard) )
	{
		/* option to send final set of parameters to an input file 
		 * get name within double quotes, 
		 * and set to blanks in chCard and OrgCard */
		/* file handle set NULL in DoOptimize when first set up,
		 * this routine called many times during optimization process,
		 * only want to open file one time */
		if( VaryPar.ioOptim == NULL )
		{
			GetQuote( chOptimFileName , chCard );

			/* open the file */
			VaryPar.ioOptim = fopen( chOptimFileName , "w" ) ;
			if( VaryPar.ioOptim == NULL )
			{
				fprintf(ioQQQ," error opening file %s\n", chOptimFileName );
				cdEXIT(EXIT_FAILURE);
			}
		}

	}
	else if( lgMatch("AMOE",chCard) )
	{
		/* use Amoeba to optimize parameters */
		strcpy( VaryPar.chOptRtn, "AMOE" );

	}
	else if( lgMatch("COLU",chCard) )
	{
		/* optimize column density */
		VaryPar.lgOptCol = TRUE;

		/* read lcolumn densities to match */
		GetOptColDen(chCard);

	}
	else if( lgMatch("CONT",chCard) )
	{
		/* set flag saying that optimization should start from continue file */
		VaryPar.lgOptCont = TRUE;

	}
	else if( lgMatch("INCR",chCard) )
	{
		/* scan off increments for the previously selected paramter */
		if( VaryPar.nparm > 0 )
		{
			/* also called during optimization process, ignore then */
			i = 5;
			VaryPar.OptIncrm[VaryPar.nparm-1] = 
				(float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		}

	}
	else if( lgMatch("LUMI",chCard) || lgMatch("INTE",chCard) )
	{
		/* scan off intensity or luminosity of normalization line */
		i = 5;
		VaryPar.optint = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		VaryPar.optier = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
		{
			VaryPar.optier = DEFERR;
		}

		/* set flag to say that intentity or luminosity of line set */
		VaryPar.lgOptLum = TRUE;

	}
	else if( lgMatch("ITER",chCard) )
	{
		/* scan off number of iterations */
		i = 5;
		VaryPar.nIterOptim = (long)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

	}
	else if( lgMatch("LINE",chCard) )
	{
		/* read lines to match */
		GetOptLineInt(chCard);

		/* set flag saying that something has been set */
		VaryPar.lgOptLin = TRUE;

	}
	else if( lgMatch("PHYM",chCard) )
	{
		/* use PHYMIR to optimize parameters */
		strcpy( VaryPar.chOptRtn, "PHYM" );
#		ifdef __unix
		VaryPar.lgParallel = ! lgMatch("SEQU",chCard);
		if( VaryPar.lgParallel ) {
			i = 5;
			dum = (long)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
			/* default has already been set in cdinit ! */
			if( ! lgEOL ) {
				VaryPar.maxCPU = dum;
			}
		}
		else {
			VaryPar.maxCPU = 1;
		}
#		else
		VaryPar.lgParallel = FALSE;
		VaryPar.maxCPU = 1;
#		endif

	}
	else if( lgMatch("POWE",chCard) )
	{
		/* use Powell to optimize parameters */
		strcpy( VaryPar.chOptRtn, "POWE" );

	}
	else if( lgMatch("RANG",chCard) )
	{
		/* scan off range for the previously selected variable */
		if( VaryPar.nparm > 0 )
		{
			i = 5;
			VaryPar.varang[VaryPar.nparm-1][0] = 
				(float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( lgEOL )
			{
				VaryPar.varang[VaryPar.nparm-1][0] = -1e38f;
			}
			VaryPar.varang[VaryPar.nparm-1][1] = 
				(float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( lgEOL )
			{
				VaryPar.varang[VaryPar.nparm-1][1] = 1e38f;
			}
		}

	}
	else if( lgMatch("SUBP",chCard) )
	{
		/* use subplex to optimize parameters */
		strcpy( VaryPar.chOptRtn, "SUBP" );

	}
	else if( lgMatch("TOLE",chCard) )
	{
		/* scan off tolerance of fit, sum of residuals must be smaller than this
		 * default is 0.10 */
		i = 5;
		VaryPar.OptGlobalErr = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

	}
	else if( lgMatch("TRAC",chCard) )
	{
		if( lgMatch("STAR",chCard) )
		{
			/* trace start iteration number
			 * turn on trace printout starting on nth call to cloudy */
			i = 5;
			VaryPar.nTrOpt = (long)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( lgEOL )
			{
				fprintf( ioQQQ, " optimize trace start command:\n" );
				fprintf( ioQQQ, " The iteration number must appear.\n" );
				puts( "[Stop in ParseOptimize]" );
				cdEXIT(EXIT_FAILURE);
			}
			VaryPar.lgTrOpt = TRUE;
		}
		else if( lgMatch("FLOW",chCard) )
		{
			/* trace flow
			 * follow logical flow within code */
			VaryPar.lgOptimFlow = TRUE;
		}
		else
		{
			fprintf( ioQQQ, " optimize trace flow command:\n" );
			fprintf( ioQQQ, " One of the sub keys START or FLOW must appear.\n" );
			puts( "[Stop in ParseOptimize]" );
			cdEXIT(EXIT_FAILURE);
		}

	}
	else
	{
		fprintf( ioQQQ, " Unrecognized keyword, consult HAZY.\n" );
		puts( "[Stop in ParseOptimize]" );
		cdEXIT(EXIT_FAILURE);
	}

#	ifdef DEBUG_FUN
	fputs( " <->ParseOptimize()\n", debug_fp );
#	endif
	return;
}

/*GetOptColDen read observed column densities & errors for optimizer */
static void GetOptColDen(char *chCard )
{
	char chCap[INPUT_LINE_LENGTH];
	int lgEOF, 
	  lgEOL;
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>GetOptColDen()\n", debug_fp );
#	endif

	/* read observed column densities & errors */
	VaryPar.ncobs = 0;

	/* get first line */
	readar(chCard,&lgEOF);
	if( lgEOF )
	{
		fprintf( ioQQQ, " Hit EOF while reading column density list; use END to end list.\n" );
		puts( "[Stop in GetOptColDen]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* now copy this line to chCap, then convert to caps */
	strcpy( chCap, chCard );
	caps(chCap);

	while( !lgEOF )
	{
		if( VaryPar.ncobs > NCOLLM )
		{
			fprintf( ioQQQ, " Too many column densities have been entered; the limit is%4ld.  Increase variable NCOLLM.\n", 
			  NCOLLM );
			puts( "[Stop in GetOptColDen]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* order on line is label (col 1-4), ionization stage, column density, err */
		/* copy cap'd version of first 4 char of chCard to chColmnLab */
		cap4((char*)VaryPar.chColmnLab[VaryPar.ncobs] , chCard );

		/* the original translated code has a mess going into cap4, and the following */
		/*f_subscpy( chCard, 0, 3, chCard_s - 1, _p0 );*/
		i = 5;
		VaryPar.ionam[VaryPar.ncobs] = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
		{
			fprintf( ioQQQ, " %80.80s\n", chCard );
			fprintf( ioQQQ, " The ionization stage MUST appear on this line.  Sorry.\n" );
			puts( "[Stop in GetOptColDen]" );
			cdEXIT(EXIT_FAILURE);
		}
		else if( VaryPar.ionam[VaryPar.ncobs] <= 0 )
		{
			fprintf( ioQQQ, " %80.80s\n", chCard );
			fprintf( ioQQQ, " An ionization stage of%4ld does not make sense.  Sorry.\n", 
			  VaryPar.ionam[VaryPar.ncobs] );
			puts( "[Stop in GetOptColDen]" );
			cdEXIT(EXIT_FAILURE);
		}

		VaryPar.amcol[VaryPar.ncobs] = (float)pow(10.,FFmtRead(chCard,&i,
		  INPUT_LINE_LENGTH,&lgEOL));
		if( lgEOL )
		{
			fprintf( ioQQQ, " %80.80s\n", chCard );
			fprintf( ioQQQ, " An observed column density MUST be entered.  Sorry.\n" );
			puts( "[Stop in GetOptColDen]" );
			cdEXIT(EXIT_FAILURE);
		}

		VaryPar.amcerr[VaryPar.ncobs] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( VaryPar.amcerr[VaryPar.ncobs] <= 0.0 )
		{
			/* this is the relative error allowed */
			VaryPar.amcerr[VaryPar.ncobs] = (float)DEFERR;
		}

		/* check if number is a limit - if '<' appears on the line then it is */
		if( strchr( chCard , '<' ) != NULL )
		{
			/* value is an upper limit only, use negative error to flag */
			VaryPar.amcerr[VaryPar.ncobs] = -VaryPar.amcerr[VaryPar.ncobs];
		}

		readar(chCard,&lgEOF);
		strcpy( chCap, chCard );
		caps(chCap);
		if( lgEOF )
		{
			fprintf( ioQQQ, " Hit EOF while reading column density list; use END to end list.\n" );
			puts( "[Stop in GetOptColDen]" );
			cdEXIT(EXIT_FAILURE);
		}

		if( strncmp( chCap , "END" , 3) == 0 )
		{
			lgEOF = TRUE;
		}

		/* now increment the number of columns we have entered */
		VaryPar.ncobs += 1;
	}

	if( trace.lgTrace && VaryPar.lgTrOpt )
	{
		fprintf( ioQQQ, "%4ld colunms were entered, they were;\n", 
		  VaryPar.ncobs );
		for( i=0; i < VaryPar.ncobs; i++ )
		{
			fprintf( ioQQQ, " %4.4s ion=%5ld%10.2e%10.2e\n", 
			  VaryPar.chColmnLab[i], VaryPar.ionam[i], VaryPar.amcol[i], 
			  VaryPar.amcerr[i] );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GetOptColDen()\n", debug_fp );
#	endif
	return;
}

/*GetOptLineInt parse observed line intensites for optimization routines */
static void GetOptLineInt(char *chCard )
{
	char chCap[INPUT_LINE_LENGTH];

	int lgEOF, 
	  lgEOL;
	long int i;
	double a;

#	ifdef DEBUG_FUN
	fputs( "<+>GetOptLineInt()\n", debug_fp );
#	endif

	/* read observed line fluxes & errors */
	VaryPar.nlobs = 0;

	readar(chCard,&lgEOF);
	if( lgEOF )
	{
		fprintf( ioQQQ, " Hit EOF while reading line list; use END to end list.\n" );
		puts( "[Stop in GetOptLineInt]" );
		cdEXIT(EXIT_FAILURE);
	}

	strcpy( chCap, chCard );
	caps(chCap);

	while( !lgEOF )
	{
		if( VaryPar.nlobs >= NOBSLM )
		{
			fprintf( ioQQQ, 
				" Too many lines have been entered; the limit is%4ld.  Increase variable NOBSLM.\n", 
			  NOBSLM );
			puts( "[Stop in GetOptLineInt]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* order on line is label (col 1-4), wavelength, flux, error */
		strncpy( VaryPar.chAmLab[VaryPar.nlobs], chCard , 4 );
		/* null terminate the label*/
		VaryPar.chAmLab[VaryPar.nlobs][4] = 0;

		i = 5;
		/* next get the wavelength */
		VaryPar.wavelength[VaryPar.nlobs] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

		/* now convert wavelength to angstroms */
		/* line was entered, look for possible micron or cm label */
		if( input.chCARDCAPS[i-1] == 'M' )
		{
			/* microns */
			VaryPar.wavelength[VaryPar.nlobs] *= 1e4;
		}
		else if( input.chCARDCAPS[i-1] == 'C' )
		{
			/* microns */
			VaryPar.wavelength[VaryPar.nlobs] *= 1e8;
		}

		/* get the error assocated with 4 significant figures */
		if( VaryPar.wavelength[VaryPar.nlobs] > 0. )
		{
			a = log10( VaryPar.wavelength[VaryPar.nlobs] );
			if( VaryPar.wavelength[VaryPar.nlobs]>=1. )
			{
				a -= floor(a);
			}
			else
			{
				a += floor(a);
			}
			a = pow(10.,a);
			/* a is now 1 - 10, so this corresponds to 4 sig fig */
			VaryPar.errorwave[VaryPar.nlobs] = (float)(0.00105/a);
		}
		else
		{
			VaryPar.errorwave[VaryPar.nlobs] = 1e-4f;
		}

		/* next get the observed intensity */
		VaryPar.amint[VaryPar.nlobs] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		if( lgEOL )
		{
			fprintf( ioQQQ, " %80.80s\n", chCard );
			fprintf( ioQQQ, " The wavelength and relative intensity MUST be entered on this line.  Sorry.\n" );
			puts( "[Stop in GetOptLineInt]" );
			cdEXIT(EXIT_FAILURE);
		}

		if( VaryPar.amint[VaryPar.nlobs] <= 0. )
		{
			fprintf( ioQQQ, " An observed intensity of%10.2e is not allowed.  Sorry.\n", 
			  VaryPar.amint[VaryPar.nlobs] );
			puts( "[Stop in GetOptLineInt]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* finally the optional error */
		VaryPar.amierr[VaryPar.nlobs] = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		/* most often will use the default */
		if( VaryPar.amierr[VaryPar.nlobs] <= 0.0 )
		{
			/* this is the relative error allowed */
			VaryPar.amierr[VaryPar.nlobs] = (float)DEFERR;
		}

		/* check if number is a limit - if '<' appears on the line then it is */
		if( strchr( chCard , '<' ) != NULL )
		{
			/* value is an upper limit only, use negative error to flag */
			VaryPar.amierr[VaryPar.nlobs] = -VaryPar.amierr[VaryPar.nlobs];
		}

		/* get next line */
		readar(chCard,&lgEOF);
		if( lgEOF )
		{
			fprintf( ioQQQ, " Hit EOF while reading line list; use END to end list.\n" );
			puts( "[Stop in GetOptLineInt]" );
			cdEXIT(EXIT_FAILURE);
		}

		strcpy( chCap, chCard );
		caps(chCap);
		if( strncmp( chCap ,"END" , 3 ) == 0 )
			lgEOF = TRUE;

		/* finally increment the number of observed lines */
		++VaryPar.nlobs;
	}

	if( trace.lgTrace && trace.lgTrOptm )
	{
		fprintf( ioQQQ, "%4ld lines were entered, they were;\n", 
		  VaryPar.nlobs );

		for( i=0; i < VaryPar.nlobs; i++ )
		{
			fprintf( ioQQQ, " %4.4s ", VaryPar.chAmLab[i] );
			prt_wl( ioQQQ, VaryPar.wavelength[i] );

			fprintf( ioQQQ, " %10.2e%10.2e\n", 
				VaryPar.amint[i], 
				VaryPar.amierr[i] );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->GetOptLineInt()\n", debug_fp );
#	endif
	return;
}

