/*ParseGrain parse parameters on grains command */
#include "cddefines.h"
#include "grain.h"
#include "grainvar.h"
#include "physok.h"
#include "input.h"
#include "atomcwgt.h"
#include "physconst.h"
#include "parse.h"

#ifdef INCLUDE_OLD_GRAINS

/*rdfile read grains parameters from a file. called by getgrains when grain number  >10 */
static void rdfile(long int igr ,  /* integer counter for grain type we will read in */
		   char chFile[]); /* name of file with data */

#define NINT(X) ((long)((X) < 0. ? (X)-0.5 : (X)+0.5))

void ParseGrain(char *chCard, 
		int *lgDSet)
{
	int lgEOL, 
	  lgEOL1,
	  lgFoundFileName,
	  lgLinSet, 
	  lgLogOn,
	  lgQHeatOn,
	  lgVaryAbun;

	char chFile[FILENAME_PATH_LENGTH_2];/*possible name of input file with opacities */
	long int i,
	  nd;
	double dep, 
	  dep1;

#	ifdef DEBUG_FUN
	fputs( "<+>ParseGrain()\n", debug_fp );
#	endif

	/* grain model, numbers are logs of grain abundance rel to std */
	gv.lgDustOn = TRUE;

	*lgDSet = TRUE;

	/* log - linear option for grains */
	lgLogOn = FALSE;
	lgLinSet = FALSE;
	if( lgMatch(" LOG",chCard) )
	{
		lgLogOn = TRUE;
		lgLinSet = FALSE;
	}

	else if( lgMatch("LINE",chCard) )
	{
		lgLogOn = TRUE;
		lgLinSet = TRUE;
	}

	/* check for the keyword "function" - this is a special case */
	lgVaryAbun = lgMatch("FUNC",chCard);

	/* check for the keyword "qheat" - this is a special case */
	lgQHeatOn = lgMatch("QHEA",chCard);

	/* the keyword "no qheat" always takes precedence */
	if( lgMatch("O QH",chCard) ) {
		lgQHeatOn = FALSE;
		physok.lgPhysOK = FALSE;
	}

	/* get the grain abundance as the first parameter */
	i = 5;
	dep = FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

	/* was keyword log on the line? */
	if( lgLogOn )
	{
		/* log or linear was specified, which was it */
		if( lgLinSet )
		{
			/* linear quantity enterd, and we want a log */
			if( dep > 0. )
			{
				dep = log10(dep);
			}
			else
			{
				fprintf( ioQQQ, " Impossible for linear negative abundance.\n" );
				fprintf( ioQQQ, " Abundance entered was%10.2e\n", dep );
				fprintf( ioQQQ, " Sorry.\n" );
				puts( "[Stop in getgrain]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
	}

	else
	{
		/* neither log nor linear specified, check sign
		 * force it to be a log - linear if greater than 0 */
		if( dep > 0. )
		{
			dep = log10(dep);
		}
	}

	/* now change to linear since will be used throughout code */
	dep = pow(10.,dep);
	if( dep < FLT_MIN )
	{
		fprintf(ioQQQ," Grain abundance entered here (%f) is impossible.\n", dep);
		puts( "[Stop in getgrain]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* it is possible that there is a file name on the command line - 
	 * if so then we want to call correct reoutine, and not look for keywords 
	 * the signature of a eyword is a pair of quotes - is one present?
	 */
	lgFoundFileName = ( strchr( input.chOrgCard,'\"' ) != NULL );
	if( lgFoundFileName )
	{
		/* 
		 * this will both scan in whatever label is inside the quotes in OrgCard, 
		 * but also remove the contents there and in chCard,
		 * so that following keywords will not trigger off it
		 */
		GetQuote( chFile , chCard );
	}
	else
	{
		/* set end of line sentinel so we know file not on line */
		chFile[0] = '\0';
	}

	/* 
	 * NB! lgDustOn1(nd) turns on grains for some species.  None of these
	 * commands turn off other grains, so successive commands will turn on
	 * more and more grains.  This is intended
	 * 
	 */
	if( lgMatch("ORIO",chCard) )
	{
		/* optional keyword ORION to use orion curves for large R grains
		 * >>chng 96 nov 30, into loop for extra K Volk grain types */
		gv.bin[2]->lgDustOn1 = TRUE;
		gv.bin[3]->lgDustOn1 = TRUE;
		gv.bin[2]->dstfactor = (float)dep;
		gv.bin[3]->dstfactor = (float)dep;
		gv.bin[2]->lgQHeat = lgQHeatOn;
		gv.bin[3]->lgQHeat = lgQHeatOn;
		/* option to have grain abundance depend on depth */
		if( lgVaryAbun )
		{
			gv.bin[2]->lgDustVary = TRUE;
			gv.bin[3]->lgDustVary = TRUE;
		}
	}

	else if( lgMatch(" PAH",chCard) )
	{
		/* >>chng 97 mar 01, added this keyworkd
		 * optional keyword  PAH to use two PAH grain species */
		gv.bin[8]->lgDustOn1 = TRUE;
		gv.bin[9]->lgDustOn1 = TRUE;
		gv.bin[8]->dstfactor = (float)dep;
		gv.bin[9]->dstfactor = (float)dep;

		/* make quantum heating the default for PAH grains */
		gv.bin[8]->lgQHeat = TRUE;
		gv.bin[9]->lgQHeat = TRUE;

		/* for PAH grains (only) there is a "no qheat" option for turning off quantum heating */
		if( lgMatch("O QH",chCard) )
		{
			gv.bin[8]->lgQHeat = FALSE;
			gv.bin[9]->lgQHeat = FALSE;
		}

		/* option to have grain abundance depend on depth */
		if( lgVaryAbun )
		{
			gv.bin[8]->lgDustVary = TRUE;
			gv.bin[9]->lgDustVary = TRUE;
		}
	}

	else if( lgMatch("GRAY",chCard) || lgMatch("GREY",chCard) )
	{
		/* >>chng 97 mar 01, added this keyworkd, 97 july 18, second spelling
		 * optional keyword GRAY to use gray grain, 0.1 micron */
		gv.bin[7]->lgDustOn1 = TRUE;
		gv.bin[7]->dstfactor = (float)dep;
		gv.bin[7]->lgQHeat = lgQHeatOn;

		/* option to have grain abundance depend on depth */
		if( lgVaryAbun )
		{
			gv.bin[7]->lgDustVary = TRUE;
		}
	}

	else if( lgMatch(" AGB",chCard) || lgMatch("AGB ",chCard) || lgMatch("PLAN",chCard) )
	{
		/* key used to be planetary nebula, should be agb
		 * planetary nebula - post agb, from Kevin Volk
		 * Try to read the second depletion factor (K Volk) */
		dep1 = FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL1);
		if( lgEOL1 )
		{
			dep1 = dep;
		}

		/* >>chng 97 jul 17, linear dep1
		 * if(dep1.gt.0.) dep1 = LOG10(dep1) */
		if( dep1 <= 0. )
		{
			dep1 = pow(10.,dep1);
		}

		/* End of changes here (K Volk) */
		gv.bin[0]->lgDustOn1 = TRUE;
		gv.bin[6]->lgDustOn1 = TRUE;
		gv.bin[0]->dstfactor = (float)dep;
		/* >>chng 96 nov 30, as per K Volk second parameter */
		gv.bin[6]->dstfactor = (float)dep1;
		gv.bin[0]->lgQHeat = lgQHeatOn;
		gv.bin[6]->lgQHeat = lgQHeatOn;

		/* option to have grain abundance depend on depth */
		if( lgVaryAbun )
		{
			gv.bin[0]->lgDustVary = TRUE;
			gv.bin[6]->lgDustVary = TRUE;
		}
	}

	else
	{
		/* turn one type of dust on if second number present */
		nd = (long int)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);
		/* leave documented behavior or grain pointers as they were,
		 * so this is one greater than array element.  decrement here 
		 * to bring onto C scale */
		--nd;

		/* lgEOL if no second number was present */
		/* >>chng 00 may 12, if file name was given, grain type number is obsolete, PvH */
		if( lgEOL && !lgFoundFileName )
		{
			/* no second number, use default ISM dust with optional depletion
			 * Mathis et al. ISM grains */
			gv.bin[0]->lgDustOn1 = TRUE;
			gv.bin[1]->lgDustOn1 = TRUE;
			gv.bin[0]->dstfactor = (float)dep;
			gv.bin[1]->dstfactor = (float)dep;
			gv.bin[0]->lgQHeat = lgQHeatOn;
			gv.bin[1]->lgQHeat = lgQHeatOn;

			/* option to have grain abundance depend on depth */
			if( lgVaryAbun )
			{
				gv.bin[0]->lgDustVary = TRUE;
				gv.bin[1]->lgDustVary = TRUE;
			}
		}

		else
		{
			if( lgFoundFileName )
			{
				nd = NewGrainBin();
			}

			/* this branch if grain number on line */
			gv.bin[nd]->lgDustOn1 = TRUE;
			gv.bin[nd]->dstfactor = (float)dep;
			/* >>chng 01 dec 04, make quantum heating default for single sized grains, PvH */
			if( nd == 4 || nd == 5 || nd == 8 || nd == 9 || nd == 10 ) {
				/* special case for PAH grains, single sized grains */
				gv.bin[nd]->lgQHeat = TRUE;
				if( lgMatch("O QH",chCard) ) {
					gv.bin[nd]->lgQHeat = FALSE;
				}
			}
			else {
				gv.bin[nd]->lgQHeat = lgQHeatOn;
			}

			/* option to have grain abundance depend on depth */
			if( lgVaryAbun )
			{
				gv.bin[nd]->lgDustVary = TRUE;
			}

			/* Changes to dust for other grains (K Volk)
			 * this branch if grain species greater than 10 - we will
			 * read in optical properties from ancillary file */
			if( lgFoundFileName )
			{
				rdfile( nd , chFile  );
			}

			/* End of changes here (K Volk) */
		}
	}

	/* option to turn off photoelectric heating by grain, NO HEATING */
	if( lgMatch("O HE",chCard) )
	{
		physok.lgPhysOK = FALSE;
		gv.lgDHetOn = FALSE;
	}

	/* option to turn off gas cooling by grain, NO COOLING */
	if( lgMatch("O CO",chCard) )
	{
		physok.lgPhysOK = FALSE;
		gv.lgDColOn = FALSE;
	}

#	ifdef DEBUG_FUN
	fputs( " <->ParseGrain()\n", debug_fp );
#	endif
	return;
}

/*rdfile read grains parameters from a file. called by getgrain when grain number  >10 */

static void rdfile(long int nd,
		   char chLine[]) /* the name of the file */
{
	FILE * ioFILE ;
	long int  i,
	  nelem,
	  itype,
	  nfield;
	float anumkv,
	  s1,
	  dstela[4],
	  dstelz[4];
	char chCard[INPUT_LINE_LENGTH];

#	ifdef DEBUG_FUN
	fputs( "<+>rdfile()\n", debug_fp );
#	endif

	/* read grains parameters from a file.  It is called when the grain number  >11
	 * The sublimation temperature is a new addition for c90.
	 * This subroutine added by (K Volk) 27 August 1993 */

	/* first try to open the file, 
	 * its name was read in with GetQuote in GetGrain, but only it quotes were entered */

	/* if first char is null then we know name was not properly entered */
	if( chLine[0]=='\0' )
	{
		/* big problem - we got here but double quotes were not picked up -
		 * not enough information to proceed */
		fprintf( ioQQQ , " rdfile got grain species that needed input file, but none specified.\n");
		fprintf( ioQQQ , " file name must occur between pair of double quotes.\n");
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	ioFILE = fopen( chLine , "r" );
	if( ioFILE == NULL )
	{
		fprintf( ioQQQ , "rdfile could not open grain opacity file named %s\n",chLine);
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	s1 = gv.bin[nd]->Tsublimat;

	/* read first line */
	if( fgets( chCard , (int)sizeof(chCard) , ioFILE ) == NULL )
	{
		fprintf( ioQQQ, " rdfile error getting first line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	nfield = sscanf( chCard ,  " %12s%f %1li %12e %12le %7f",
	  gv.bin[nd]->chDstLab, 
	  &gv.bin[nd]->DustWorkFcn, 
	  &itype, 
	  &anumkv, 
	  &gv.bin[nd]->cnv_H_pGR, 
	  &gv.bin[nd]->Tsublimat );
	gv.bin[nd]->cnv_GR_pH = 1./gv.bin[nd]->cnv_H_pGR;

	/* mak sure we got 6 values  */
	if( nfield != 6 )
	{
		fprintf( ioQQQ, " rdfile error parsing first line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* The "old" dust input file does not have the sublimation temperature 
	 * so it is reset to the initial value if it is zero, or if the value is 
	 * absurdly small.
	 */
	if( gv.bin[nd]->Tsublimat <= 50. )
	{
		gv.bin[nd]->Tsublimat = s1;
	}

	/* read second line */
	if( fgets( chCard , (int)sizeof(chCard) , ioFILE ) == NULL )
	{
		fprintf( ioQQQ, " rdfile error getting second line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	nfield = sscanf( chCard ,"%9e %9e %9e  %9e %3li  %9e  %9e",
		&gv.bin[nd]->dustp[0] ,
		&gv.bin[nd]->dustp[1] ,
		&gv.bin[nd]->dustp[2] ,
		&gv.bin[nd]->dustp[3] ,
		&gv.bin[nd]->ndpts, 
		&gv.bin[nd]->IntArea, 
	   &gv.bin[nd]->AvRadius
		);
	/* make sure we got 7 values */
	if( nfield != 7 )
	{
		fprintf( ioQQQ, " rdfile error parsing second line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* these data are never size resolved -> all mass is in this bin */
	gv.bin[nd]->dustp[4] = 1.;

	if( gv.bin[nd]->ndpts > LIMCRS )
		gv.bin[nd]->ndpts = LIMCRS;

	/* read third line */
	if( fgets( chCard , (int)sizeof(chCard) , ioFILE ) == NULL )
	{
		fprintf( ioQQQ, " rdfile error getting third line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

/*		original fortran
      read(89,'(4(f5.1,1x,e12.4))', err=585,end=585) (dstelz(j,nd),
     1                              dstela(j,nd),j=1,4)
	  */
	nfield = sscanf( chCard ,"%5f %12e%5f %12e%5f %12e%5f %12e",
		&dstelz[0], 
		&dstela[0],
		&dstelz[1], 
		&dstela[1],
		&dstelz[2], 
		&dstela[2],
		&dstelz[3], 
		&dstela[3]
		);
	/* make sure we got 8 values */
	if( nfield != 8 )
	{
		fprintf( ioQQQ, " rdfile error parsing third line\n" );
		puts( "[Stop in rdfile]" );
		cdEXIT(EXIT_FAILURE);
	}

	for( i=0; i < gv.bin[nd]->ndpts; i++ )
	{
		/* read the data */
		if( fgets( chCard , (int)sizeof(chCard) , ioFILE ) == NULL )
		{
			fprintf( ioQQQ, " rdfile error getting data, line=%li\n",i );
			puts( "[Stop in rdfile]" );
			cdEXIT(EXIT_FAILURE);
		}

		nfield = sscanf( chCard ,"%11e%11e%11e",
			&gv.bin[nd]->eev[i], 
			&gv.bin[nd]->sab[i], 
			&gv.bin[nd]->sse[i]
			);
		/* make sure we got 8 values */
		if( nfield != 3 )
		{
			fprintf( ioQQQ, " rdfile error parsing data line, %li=\n",i );
			puts( "[Stop in rdfile]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* indicate old style grain data */
	gv.bin[nd]->lgOldGrain1 = TRUE;

	/* initialize some other grain data */
	gv.bin[nd]->AvArea = (float)(gv.bin[nd]->IntArea*gv.bin[nd]->cnv_H_pGR);
	gv.bin[nd]->AvVol = (float)(gv.bin[nd]->AvRadius*gv.bin[nd]->AvArea/3.);
	gv.bin[nd]->IntVol = (float)(gv.bin[nd]->AvVol*gv.bin[nd]->cnv_GR_pH);
	/* this next statement is not exact, but it is the best we can do... */
	gv.bin[nd]->IntRadius = (float)(gv.bin[nd]->AvRadius*gv.bin[nd]->cnv_GR_pH);
	gv.bin[nd]->matType = ( itype == 0 ) ? MAT_CAR : MAT_SIL;
	gv.bin[nd]->atomWeight = (float)(gv.bin[nd]->AvVol*gv.bin[nd]->dustp[0]/ATOMIC_MASS_UNIT/anumkv);

	/* copy elemental abundances to new structure, initialize accomodation coef */
	for( nelem=0; nelem < LIMELM; nelem++ ) 
	{
		gv.bin[nd]->elmAbund[nelem] = 0.;
		/* this coefficient is defined at the end of appendix A.10 of BFM */
		gv.bin[nd]->AccomCoef[nelem] = 2.*gv.bin[nd]->atomWeight*AtomcWgt.AtomicWeight[nelem]/
			POW2(gv.bin[nd]->atomWeight+AtomcWgt.AtomicWeight[nelem]);
	}
	for( i=0; i < 4; i++ ) 
	{
		nelem = NINT(dstelz[i]) - 1;
		if( nelem >= 0 && nelem < LIMELM ) 
		{
			gv.bin[nd]->elmAbund[nelem] = dstela[i];
		}
	}
	/* guesstimate */
	gv.bin[nd]->ThermEff = 0.5;
	if( gv.bin[nd]->matType == MAT_SIL ) 
	{
		gv.bin[nd]->BandGap = 0.3676f;
	}
	else 
	{
		gv.bin[nd]->BandGap = 0.f;
	}
	/* this is capacity per grain, in Farad per grain */
	gv.bin[nd]->Capacity = PI4*ELECTRIC_CONST*gv.bin[nd]->IntRadius/100.*gv.bin[nd]->cnv_H_pGR;

	fclose( ioFILE );

#	ifdef DEBUG_FUN
	fputs( " <->rdfile()\n", debug_fp );
#	endif
	return;
}

#endif
