/*MakeCS compute collision strength by g-bar approximations */
#include "cddefines.h"
#include "physconst.h"
#include "ionfracs.h"
#include "phycon.h"
#include "makecs.h"

/*gbar0 compute g-bar gaunt factor for neutrals */
static void gbar0(double ex, 
	  float *g);

/*gbar1 compute g-bar collision strength using Mewe approximations */
static void gbar1(double ex, 
	  float *g);

void MakeCS(EmLine * t)
{
	long int ion;
	double Abun, 
	  cs;
	float
	  gbar;

#	ifdef DEBUG_FUN
	fputs( "<+>MakeCS()\n", debug_fp );
#	endif

	/* routine to get cs from various approximations */

	/* check if abundance greater than 0 */
	ion = t->IonStg;

	Abun = xIonFracs[ t->nelem -1 ][ ion-1 ];
	if( Abun <= 0. )
	{
		gbar = 1.;
	}
	else
	{
		/* check if neutral or ion */
		if( ion == 1 )
		{
			/* neutral - compute gbar for evantual cs */
			gbar0(t->EnergyK,&gbar);
		}
		else
		{
			/* ion - compute gbar for evantual cs */
			gbar1(t->EnergyK,&gbar);
		}
	}

	/* above was g-bar, convert to cs */
	cs = gbar*(14.5104/WAVNRYD)*t->gf/t->EnergyWN;

	/* stuff the cs in place */
	t->cs = (float)cs;

#	ifdef DEBUG_FUN
	fputs( " <->MakeCS()\n", debug_fp );
#	endif
	return;
}

/*gbar0 compute g-bar gaunt factor for neutrals */
static void gbar0(double ex, 
	  float *g)
{
	double a, 
	  b, 
	  c, 
	  d, 
	  y;

#	ifdef DEBUG_FUN
	fputs( "<+>gbar0()\n", debug_fp );
#	endif

	/* written by Dima Verner
	 *
	 * Calculation of the effective Gaunt-factor by use of 
	 * >>refer	all	cs	Van Regemorter, 1962, ApJ 136, 906
	 * fits for neutrals
	 *  Input parameters: 
	 * ex - energy ryd - now K
	 * t  - temperature in K
	 *  Output parameter:
	 * g  - effective Gaunt factor
	 * */

	/* y = ex*157813.7/te */
	y = ex/phycon.te;
	if( y < 0.01 )
	{
		*g = (float)(0.29*(log(1.0+1.0/y) - 0.4/POW2(y + 1.0))/exp(y));
	}
	else
	{
		if( y > 10.0 )
		{
			*g = (float)(0.066/sqrt(y));
		}
		else
		{
			a = 1.5819068e-02;
			b = 1.3018207e00;
			c = 2.6896230e-03;
			d = 2.5486007e00;
			d = log(y/c)/d;
			*g = (float)(a + b*exp(-0.5*POW2(d)));
		}
	}


#	ifdef DEBUG_FUN
	fputs( " <->gbar0()\n", debug_fp );
#	endif
	return;
}

/*gbar1 compute g-bar collision strength using Mewe approximations */
static void gbar1(double ex, 
	  float *g)
{
	double y;

#	ifdef DEBUG_FUN
	fputs( "<+>gbar1()\n", debug_fp );
#	endif

	/*	*written by Dima Verner
	 *
	 * Calculation of the effective Gaunt-factor by use of Mewe (1972, A&A 20, 215)
	 * fits for permitted transitions in ions MgII, CaII, FeII (delta n = 0)
	 * Input parameters: 
	 * ex - excitation energy in Ryd - now K
	 * te  - temperature in K
	 * Output parameter:
	 * g  - effective Gaunt factor
	 */

	/* y = ex*157813.7/te */
	y = ex/phycon.te;
	*g = (float)(0.6 + 0.28*(log(1.0+1.0/y) - 0.4/POW2(y + 1.0)));


#	ifdef DEBUG_FUN
	fputs( " <->gbar1()\n", debug_fp );
#	endif
	return;
}

