/*iso_create create data for hydrogen and helium, 1 per coreload, called by ContCreatePointers 
 * in turn called after commands parsed */
/*iso_zero zero data for hydrogen and helium */
#include "cddefines.h"
#include "physconst.h"
#include "taulines.h"
#include "hydrogenic.h"
#include "nhe1lvl.h"
#include "he3tau.h"/* needed for NHE3TAU */
#include "he1dmp.h"
#include "opacity.h"
#include "heopfr.h"
#include "he1as.h"
#include "assav.h"
#include "he1stat.h"
#include "he1tau.h"
#include "he1nxt.h"
#include "phe1lv.h"
#include "he1ind.h"
#include "he1deltat.h"
#include "refindex.h"
#include "he1nionryd.h"
#include "hydroeinsta.h"
#include "abundances.h"
#include "abscf.h"
#include "getgf.h"
#include "ph1com.h"
#include "iso.h"

/*iso_zero zero data for hydrogen and helium */
static void iso_zero(void);

void iso_create(void)
{
	long int i, 
	  ipHi, 
	  ipISO ,
	  ipLo, 
	  nelem, 
	  j, 
	  lo, 
	  n ,
	  need,
	  nHi, 
	  n2 ,
	  nLo;

	static int nCalled = 0;

	double HIonPoten, 
	  Potential, 
	  gHi, 
	  gLo, 
	  *hdamp/*[LMHLVL+1]*/, 
	  scrhe1, 
	  test, 
	  z4;
	double He1EnerRyd[NHE1LVL][NHE1LVL];

#	ifdef DEBUG_FUN
	fputs( "<+>iso_create()\n", debug_fp );
#	endif

	/* > 1 if not first call, then just zero arrays out */
	if( nCalled > 0 )
	{
		iso_zero();
#		ifdef DEBUG_FUN
		fputs( " <->iso_create()\n", debug_fp );
#		endif
		return;
	}

	/* this is first call, increment the nCalled counterso never do this again */
	++nCalled;

	/* >>chng 02 feb 21, the entire He-like iso-sequence will have nTopOff n-grouped
	 * levels in addition to the number of l-resolved levels 
	 * atom he-like levels adjusts number of l-resolved levels 
	 * atom he-like topoff adjusts nTopOff */
	for( nelem=ipHELIUM; nelem < LIMELM; ++nelem )
	{
		/* only grab core for elements that are turned on */
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			iso.numLevels[ipHE_LIKE][nelem] += iso.nTopOff[ipHE_LIKE];
		}
	}

	/* hdamp is freed at end of sub */
	if( (hdamp=(double*)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][ipHYDROGEN])) )==NULL )
		bad_malloc();

	/* these are the statistical weights of the ions */
	iso.stat_ion[ipH_LIKE] = 1.f;
	iso.stat_ion[ipHE_LIKE] = 2.f;

	/* create space for arrays of indices for ionization edges within continuum array */
	/* iso.ipOpac[NISO][LIMELM][NHPLPHOT] */
	if( (iso.ipOpac = (long int ***)MALLOC(sizeof(long **)*NISO ))==NULL )
		bad_malloc();

	/* Master array of levels...iso_quant_desig[ipHE_LIKE][nelem][ipLevel].n, .l, and .s	*/
	if( (iso_quant_desig = (Elevels***)MALLOC(sizeof(Elevels**)*(unsigned)NISO ) )==NULL )
		bad_malloc();

	if( (iso.ipIsoLevNIonCon = (long int ***)MALLOC(sizeof(long **)*NISO ))==NULL )
		bad_malloc();

	if( (iso.xIsoLevNIonRyd = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.stat = (float ***)MALLOC(sizeof(float **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.RadRecomb = (float ****)MALLOC(sizeof(float ***)*NISO )) == NULL )
		bad_malloc();

	if( (iso.ExtraLymanLines = (EmLine ***)MALLOC(sizeof(EmLine **)*NISO )) ==NULL)
		bad_malloc();

	if( (EmisLines = (EmLine ****)MALLOC(sizeof(EmLine **)*NISO )) ==NULL)
		bad_malloc();

	if( (iso.Boltzmann = (double ****)MALLOC(sizeof(double ***)*NISO )) == NULL )
		bad_malloc();

	if( (iso.ConBoltz = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.Pop2Ion = (float ***)MALLOC(sizeof(float **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.ConOpacRatio = (float ***)MALLOC(sizeof(float **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.DepartCoef = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.gamnc = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.RecomInducRate = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.RecomInducCool_Coef = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.PhotoHeat = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.PopLTE = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	if( (iso.ColIoniz = (double ***)MALLOC(sizeof(double **)*NISO )) == NULL )
		bad_malloc();

	/* the hydrogen and helium like iso-sequences */
	for( ipISO=0; ipISO<NISO; ++ipISO )
	{
		long int nmax;

		/* Master array of levels...iso_quant_desig[ipHE_LIKE][nelem][ipLevel].n, .l, and .s	*/
		if( (iso_quant_desig[ipISO] = (Elevels**)MALLOC(sizeof(Elevels*)*(unsigned)LIMELM ) )==NULL )
			bad_malloc();

		if( (iso.ExtraLymanLines[ipISO] = (EmLine **)MALLOC(sizeof(EmLine *)*LIMELM )) ==NULL)
			bad_malloc();

		if( (iso.ipOpac[ipISO] = (long int **)MALLOC(sizeof(long *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.ipIsoLevNIonCon[ipISO] = (long int **)MALLOC(sizeof(long *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.xIsoLevNIonRyd[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.stat[ipISO] = (float **)MALLOC(sizeof(float*)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.RadRecomb[ipISO] = (float ***)MALLOC(sizeof(float **)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.Boltzmann[ipISO] = (double ***)MALLOC(sizeof(double **)*LIMELM ))==NULL )
			bad_malloc();

		if( (EmisLines[ipISO] = (EmLine ***)MALLOC(sizeof(EmLine **)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.ConBoltz[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.Pop2Ion[ipISO] = (float **)MALLOC(sizeof(float *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.ConOpacRatio[ipISO] = (float **)MALLOC(sizeof(float *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.DepartCoef[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM ))==NULL )
			bad_malloc();

		if( (iso.gamnc[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		if( (iso.RecomInducRate[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		if( (iso.RecomInducCool_Coef[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		if( (iso.PhotoHeat[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		if( (iso.PopLTE[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		if( (iso.ColIoniz[ipISO] = (double **)MALLOC(sizeof(double *)*LIMELM )) == NULL )
			bad_malloc();

		/* this will remember the greatest number of levels for each iso sequence,
		 * and is used to allocate the stat weight array after this loop */
		nmax = 0;
		for( nelem=ipISO; nelem < LIMELM; ++nelem )
		{
			/* only grab core for elements that are turned on */
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				nmax = MAX2( nmax , iso.numLevels[ipISO][nelem] );

				/*fprintf(ioQQQ,"assert failll %li\t%li\t%li\n", ipISO, nelem , iso.numLevels[ipISO][nelem] );*/
				ASSERT( iso.numLevels[ipISO][nelem] > 0 );
			
				/* Allocate space so that array iso_quant_desig[ipHE_LIKE] contains proper number of elements for each element. */
				if( (iso_quant_desig[ipISO][nelem] = 
					(Elevels*)MALLOC((unsigned)iso.numLevels[ipISO][nelem]*sizeof(Elevels)) ) == NULL )
					bad_malloc();

				if( (iso.ExtraLymanLines[ipISO][nelem] = 
					(EmLine *)MALLOC(sizeof(EmLine )*(unsigned)iso.nLyman[ipISO] )) ==NULL)
					bad_malloc();

				if( (iso.ipOpac[ipISO][nelem] = 
					(long int *)MALLOC(sizeof(long)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.ipIsoLevNIonCon[ipISO][nelem] = 
					(long int *)MALLOC(sizeof(long)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.xIsoLevNIonRyd[ipISO][nelem] = 
					(double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.stat[ipISO][nelem] = (float *)MALLOC(sizeof(float)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
					bad_malloc();

				if( (iso.RadRecomb[ipISO][nelem] = 
					(float **)MALLOC(sizeof(float *)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.Boltzmann[ipISO][nelem] = 
					(double **)MALLOC(sizeof(double *)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (EmisLines[ipISO][nelem] = 
					(EmLine **)MALLOC(sizeof(EmLine *)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.ConBoltz[ipISO][nelem] = 
					(double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.DepartCoef[ipISO][nelem] = 
					(double *)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.Pop2Ion[ipISO][nelem] = 
					(float *)MALLOC(sizeof(float)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.ConOpacRatio[ipISO][nelem] = 
					(float *)MALLOC(sizeof(float)*(unsigned)(iso.numLevels[ipISO][nelem]) ))==NULL )
						bad_malloc();

				if( (iso.gamnc[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				if( (iso.RecomInducRate[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				if( (iso.RecomInducCool_Coef[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				if( (iso.PhotoHeat[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				if( (iso.PopLTE[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				if( (iso.ColIoniz[ipISO][nelem] = (double *)MALLOC(sizeof(double )*(unsigned)(iso.numLevels[ipISO][nelem]) )) == NULL )
					bad_malloc();

				n = 0;
				/* this needs n = 0, but RadRecomb will crash since malloc(0), not used */
				if( (iso.RadRecomb[ipISO][nelem][n] = (float *)MALLOC(sizeof(float)*(unsigned)3 ))==NULL )
					bad_malloc();
				for( n=1; n<iso.numLevels[ipISO][nelem]; ++n )
				{
					if( (iso.RadRecomb[ipISO][nelem][n] = (float *)MALLOC(sizeof(float)*(unsigned)3 ))==NULL )
						bad_malloc();

					/* sec to last dim is upper level n,
					 * last dim of this array is lower level, will go from 0 to n-1 */
					if( (iso.Boltzmann[ipISO][nelem][n] = (double *)MALLOC(sizeof(double)*(unsigned)n ))==NULL )
						bad_malloc();

					if( (EmisLines[ipISO][nelem][n] = (EmLine *)MALLOC(sizeof(EmLine)*(unsigned)n ))==NULL )
						bad_malloc();
				}
			}
		}
	}

	/* now MALLOC the space for the hydrogenic lines, 
	 * but this must be done exactly one time per coreload */
	if( !lgHydroMalloc )
	{

		if( (He1AutoLines = (EmLine **)MALLOC(sizeof(EmLine *)*LIMELM )) ==NULL)
			bad_malloc();

		/* the helium like iso-sequence */
		for( nelem=1; nelem < LIMELM; ++nelem )
		{
			/* only allocate space for elements that are turned on */
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				if( (He1AutoLines[nelem]=(EmLine *)MALLOC(sizeof(EmLine )*(unsigned)(4))) ==NULL)
					bad_malloc();
				
			}
		}

		/*double **DestRate[LIMELM][LMHLVL+1]*/
		if( (hydro.DestRate = (double **)MALLOC(sizeof(double *)*(unsigned)LIMELM ) )==NULL )
		{
			fprintf( ioQQQ, 
				" iso_create could not MALLOC hydro.DestRate 1\n" );
			puts( "[Stop in iso_create]" );
			cdEXIT(EXIT_FAILURE);
		}
		/*float **strkar strkar[LMHLVL+1][LMHLVL+1]*/
		if( (hydro.strkar = (float **)MALLOC(sizeof(float *)*(unsigned)(iso.numLevels[ipH_LIKE][ipHYDROGEN]) ) )==NULL )
		{
			fprintf( ioQQQ, 
				" iso_create could not MALLOC hydro.strkar 1\n" );
			puts( "[Stop in iso_create]" );
			cdEXIT(EXIT_FAILURE);
		}
		/*float **pestrk pestrk[LMHLVL+1][LMHLVL+1]*/
		if( (hydro.pestrk = (float **)MALLOC(sizeof(float *)*(unsigned)(iso.numLevels[ipH_LIKE][ipHYDROGEN]) ) )==NULL )
		{
			fprintf( ioQQQ, 
				" iso_create could not MALLOC hydro.pestrk 1\n" );
			puts( "[Stop in iso_create]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* now do second dimension of previous two */
		for( ipLo=0; ipLo< iso.numLevels[ipH_LIKE][ipHYDROGEN] ;++ipLo )
		{
			if( (hydro.strkar[ipLo] = 
				(float*)MALLOC(sizeof(float)*(unsigned)(iso.numLevels[ipH_LIKE][ipHYDROGEN]) ))==NULL )
			{
				fprintf( ioQQQ, 
					" iso_create could not second MALLOC hydro.strkar 1\n" );
				puts( "[Stop in iso_create]" );
				cdEXIT(EXIT_FAILURE);
			}
			if( (hydro.pestrk[ipLo] = 
				(float*)MALLOC(sizeof(float)*(unsigned)(iso.numLevels[ipH_LIKE][ipHYDROGEN]) ))==NULL )
			{
				fprintf( ioQQQ, 
					" iso_create could not second MALLOC hydro.pestrk 1\n" );
				puts( "[Stop in iso_create]" );
				cdEXIT(EXIT_FAILURE);
			}
		}

		for( nelem=0; nelem < LIMELM; ++nelem )
		{
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{

				if( (hydro.DestRate[nelem] = 
					(double*)MALLOC(sizeof(double)*(unsigned)(iso.numLevels[ipH_LIKE][nelem]) ))==NULL )
				{
					fprintf( ioQQQ, 
						" iso_create could not second MALLOC hydro.DestRate 1\n" );
					puts( "[Stop in iso_create]" );
					cdEXIT(EXIT_FAILURE);
				}
			}
		}

		/* we will never do this again, in this coreload,
		 * following says never change number of levels in hydrogen atom again,
		 * future hydrogenic levels commands will be ignored*/
		lgHydroMalloc = TRUE;	

		/* main hydrogenic arrays, initialize so they will blow up if misused
		 * only do this when array created, since pointers to continuum energies
		 * will not need to be recreated */
		for( nelem=0; nelem < LIMELM; nelem++ )
		{
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				/* arrays are dim'd iso.numLevels[ipH_LIKE][nelem]+1 */
				for( ipLo=ipH1s; ipLo < iso.numLevels[ipH_LIKE][nelem]-1; ipLo++ )
				{
					/* set ENTIRE array to impossible values, in case of bad pointer */
					for( ipHi=ipLo+1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
					{
						EmLineJunk( &EmisLines[ipH_LIKE][nelem][ipHi][ipLo] );
					}
				}
				/* these are the extra Lyman lines */
				for( ipHi=2; ipHi < iso.nLyman[ipH_LIKE]; ipHi++ )
				{
					EmLineJunk( &iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi] );
				}
			}
		}
		/* the main he-like arrays */
		for( nelem=1; nelem < LIMELM; nelem++ )
		{
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				/* set ENTIRE array to impossible values, in case of bad pointer */
				for( ipHi=1; ipHi < iso.numLevels[ipHE_LIKE][nelem]; ipHi++ )
				{
					/* arrays are dim'd iso.numLevels[ipHE_LIKE][nelem] */
					for( ipLo=0; ipLo < ipHi; ipLo++ )
					{
						EmLineJunk( &EmisLines[ipHE_LIKE][nelem][ipHi][ipLo] );
					}
				}
				/* these are the extra Lyman lines */
				for( ipHi=2; ipHi < iso.nLyman[ipHE_LIKE]; ipHi++ )
				{
					EmLineJunk( &iso.ExtraLymanLines[ipHE_LIKE][nelem][ipHi] );
				}
			}
		}
		for( nelem=1; nelem < LIMELM; nelem++ )
		{
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				/* There are four autoionization levels to be considered. */
				for( ipHi = 0; ipHi < 4 ; ipHi++ )
				{
					/* set ENTIRE array to impossible values, in case of bad pointer */
					EmLineJunk( &He1AutoLines[nelem][ipHi] );
				}
			}
		}
	}

	/* upper level 4 and above, going to lower level 2 and above,
	 * come from the HyLife and branch product formed in routine pesc
	 * all other values come from here*/

	/* main hydrogenic arrays, fill with sane values */
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* must always do helium even if turned off */
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			/* charge to 4th power, needed for scaling laws */
			z4 = POW2(nelem+1.);
			z4 *= z4;

			/* Dima's array has ionization potentials in eV, but not on same
				* scale as cloudy itself*/
			/* extra factor accounts for this */
			HIonPoten = PH1COM.PH1[0][0][nelem][0]/EVRYD* 0.9998787;
			ASSERT(HIonPoten > 0.);

			/*TODO this will not work if highest level is resolved */
			iso_quant_desig[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem] - 1].n = iso.numLevels[ipH_LIKE][nelem] - 1;
			iso_quant_desig[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem] - 1].l = -1;
			iso_quant_desig[ipH_LIKE][nelem][iso.numLevels[ipH_LIKE][nelem] - 1].s = -1;

			for( ipLo=ipH1s; ipLo < (iso.numLevels[ipH_LIKE][nelem] - 1); ipLo++ )
			{
				/* test code to see what happens with lyman lines in case a */
				if( ipLo == ipH1s )
				{
					/* ground */
					nLo = 1;
					gLo = 2.;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].n = 1;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].l = 0;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].s = 0;
				}
				else if( ipLo == ipH2s )
				{
					/* 2s level */
					nLo = 2;
					gLo = 2.;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].n = 2;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].l = 0;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].s = 0;
				}
				else if( ipLo == ipH2p )
				{
					/* 2p level */
					nLo = 2;
					gLo = 6.;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].n = 2;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].l = 1;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].s = 0;
				}
				else
				{
					nLo = ipLo;
					gLo = 2.*nLo*nLo;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].n = ipLo;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].l = -1;
					iso_quant_desig[ipH_LIKE][nelem][ipLo].s = -1;
				}
				iso.stat[ipH_LIKE][nelem][ipLo] = (float)gLo;
				for( ipHi=ipLo + 1; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
				{
					double EnergyRyd;

					if( ipHi == ipH2s )
					{
						nHi = 2;
						gHi = 2.;
					}
					else if( ipHi == ipH2p )
					{
						nHi = 2;
						gHi = 6.;
					}
					else
					{
						nHi = ipHi;
						gHi = 2.*nHi*nHi;
					}
					iso.stat[ipH_LIKE][nelem][ipHi] = (float)gHi;
					/* pointers to transitions, ipLo, ipHi
					 * actual quantum numbers, nLo nHi
					 * atomic number or charge and stage: */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].nelem = (int)(nelem+1);
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].IonStg = (int)(nelem+1);

					/* transition energy in various units:
					 * 1e-10 is to stop 2s-2p from having zero energy */
					/* >>chng02 feb 09, use zero energy as sentinal of non-existant transition */
					if( ipLo==ipH2s && ipHi==ipH2p )
					{
						/*EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyRyd = 1e-10f;*/
						EnergyRyd = 0.;
					}
					else
					{
						EnergyRyd = (HIonPoten*
						  (  1./POW2((double)nLo) -1./POW2((double)nHi) ) );
					}
					/* >>chng 02 feb 09, change test to >= 0 since we now use 0 for 2s-2p */
					ASSERT(EnergyRyd >= 0.);

					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyErg = 
						(float)(EnergyRyd*
					  EN1RYD);
					ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyErg >= 0.);

					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyK = 
						(float)(EnergyRyd*
					  TE1RYD);
					ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyK >= 0.);

					/* the energy of the transition in wavenumbers */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN = 
						(float)(EnergyRyd/ WAVNRYD);
					ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN >= 0.);

					if( ipLo==ipH2s && ipHi==ipH2p )
					{
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].WLAng = 0.;
					}
					else
					{
						/* make following an air wavelength */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].WLAng = 
							(float)(1.0e8/
						  EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN/
						  RefIndex(&EmisLines[ipH_LIKE][nelem][ipHi][ipLo]));
						ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].WLAng > 0.);
					}

					/* stat. weights and Einstein As & gfs: */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gLo = (float)gLo;

					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gHi = (float)gHi;

					/* transition prob, EinstA uses current H atom indices */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul = 
						(float)(HydroEinstA(ipLo,ipHi)*z4);
					ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul > 0.);

					/* >>chng 01 feb 27, had been following, for all lines except Lya */
					/* redistribution function = complete 
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].iRedisFun = -1; - this is same as ipCRD */
					if( ipLo == ipH1s && ipHi == ipH2p )
					{
						/* this is La, ipPRD is 1 = ipPRD */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].iRedisFun = hydro.ipLya;
					}
					else if( ipLo == ipH1s && ipHi != ipH2p )
					{
						/* these are rest of Lyman lines, 
						 * complete redistribution, doppler core only, K2 core, ipCRD */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].iRedisFun = hydro.ipReso;
					}
					else
					{
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].iRedisFun = hydro.ipSub;
					}
					if( EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN <= 0. )
					{
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gf = 0.;
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity = 0.;
					}
					else
					{
						/* all subordinate lines, CRD with wings */
						/* >>chng 01 feb 27, had been -1, crd with core only,
						* change to crd with wings as per discussion with Ivan Hubeny */
						/*EmisLines[ipH_LIKE][nelem][ipHi][ipLo].iRedisFun = -1; = ipCRD*/

						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gf = 
							(float)(GetGF(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].Aul,
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN,
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gHi));
						ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gf > 0.);

						/* derive the abs coef, call to function is gf, wl (A), g_low */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity = 
							(float)(abscf(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gf,
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN,
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].gLo));
						ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].opacity > 0.);
					}

					/* create array index that will blow up */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].ipCont = INT_MIN;
				}
			}
			/* these are the extra Lyman lines */
			nLo = 1;
			for( ipHi=2; ipHi < iso.nLyman[ipH_LIKE]; ipHi++ )
			{
				double EnergyRyd;
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].nelem = (int)(nelem+1);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].IonStg = (int)(nelem+1);
				EnergyRyd = 
							(float)(HIonPoten*
						  (  1./POW2((double)nLo) -1./POW2((double)ipHi) ) );
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyErg = 
						(float)(EnergyRyd*
					  EN1RYD);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyK = 
						(float)(EnergyRyd* TE1RYD);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyWN = 
						(float)(EnergyRyd/ WAVNRYD);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].WLAng = 
							(float)(1.0e8/
						  iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyWN/
						  RefIndex(&iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi]));
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gLo = 2.L;
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gHi = (float)(2.*ipHi*ipHi);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].Aul = 
						(float)(HydroEinstA(0,ipHi)*z4);
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].iRedisFun = hydro.ipReso;
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gf = 
						(float)(GetGF(iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].Aul,
					  iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyWN,
					  iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gHi));
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].opacity = 
						(float)(abscf(iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gf,
					  iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyWN,
					  iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].gLo));

				/* create array index that will blow up */
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].ipCont = INT_MIN;
			}

			/* set in abscf.h */
#			if LOWDEN_LYMAN
			/* for fix Lyman lines, in terms of alpha transition */
			for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
					float Ratio_lyman_alpha_Z1[25]={
						0.,0.,0.,
5.52E-01f,/* this mult by 1.33 since Ha split into two */
3.30E-01f,
2.96E-01f,
2.80E-01f,
2.74E-01f,
2.72E-01f,
2.72E-01f,
2.74E-01f,
2.76E-01f,
2.78E-01f,
2.81E-01f,
2.84E-01f,
2.86E-01f,
2.89E-01f,
2.92E-01f,
2.95E-01f,
2.98E-01f,
3.01E-01f,
3.05E-01f,
3.09E-01f,
3.13E-01f,
3.18E-01f};
					float Ratio_lyman_alpha_Z2[25]={
						0.,0.,0.,
4.52E-01f,
2.38E-01f,
1.98E-01f,
1.80E-01f,
1.71E-01f,
1.66E-01f,
1.64E-01f,
1.63E-01f,
1.63E-01f,
1.64E-01f,
1.65E-01f,
1.66E-01f,
1.68E-01f,
1.69E-01f,
1.71E-01f,
1.72E-01f,
1.73E-01f,
1.75E-01f,
1.76E-01f,
1.78E-01f,
1.79E-01f,
1.80E-01f};

						if( nelem==0 )
						{
							EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Aul = EmisLines[ipH_LIKE][nelem][ipHi][ipHi-1].Aul *
							Ratio_lyman_alpha_Z1[MIN2(23,ipHi) ];
						}
						else
						{
							EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Aul = EmisLines[ipH_LIKE][nelem][ipHi][ipHi-1].Aul *
							Ratio_lyman_alpha_Z2[MIN2(23,ipHi) ];
						}

						/* derive the abs coef, call to function is gf, wl (A), g_low */
						EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].opacity = 
							(float)(abscf(
						  EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].gf,
						  EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].EnergyWN,
						  EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].gLo));
			}
#			endif
			/* total toptical depth in 1s - 2s */
			EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].TauTot = 0.;

			/* opacity in two-photon transition is incorrect - actually a continuum,
			 * so divide by typical width*/
			/* >>chng 00 nov 27, do this correction, when considered as line sometimes thick */
			EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].opacity /= 1e4f;

			/* wavelength of 0 is sentinal for two-photon emission */
			EmisLines[ipH_LIKE][nelem][ipH2s][ipH1s].WLAng = 0.;

			/* 2s - 2p cannot have zero opacity */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].opacity = 1e-30f;

			/* Lya has special redistribution function */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH1s].iRedisFun = ipPRD;
		}
	}

	/* following comes out very slightly off, correct here */
	EmisLines[ipH_LIKE][ipHELIUM][3][ipH2s].WLAng = 1640.f;
	EmisLines[ipH_LIKE][ipHELIUM][3][ipH2p].WLAng = 1640.f;

	/**************************************************************************
	 * now set HyLife for hydrogen itself, does not include Z^4 or Lyman lines 
	 * HyLife is used to get lifetimes of full H iso sequence, so we must find
	 * largest number of levels that will ever be needed 
	 **************************************************************************/
	need = 0;
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			need = MAX2( need , iso.numLevels[ipH_LIKE][nelem] );
		}
	}
	need = MAX2(need , iso.nLyman[ipH_LIKE]);

	/* >>chng 01 jan 20, had been La transition prob */
	/* does not include Lyman lines so this is zero,
	 * will be added to line itself below */
	hydro.HyLife[ipH2p] = 0.;

	/* >>chng 00 dec 16, from iso.numLevels[ipH_LIKE] for hydrogen, to most ever needed */
	for( ipHi=3; ipHi < need; ipHi++ )
	{
		hydro.HyLife[ipHi] = 0.;

		/* HyLife does not include trans to ground since case b branch used later */
		for( ipLo=ipH2s; ipLo < ipHi; ipLo++ )
		{
			/* array of damping constants for hydrogen itself */
			/* >>chng 00 dec 16, had used hydrogen line structure to get lifetime,
			 * but failed in case where H itself had few levels, and other elements
			 * had more */
			double ea = HydroEinstA(ipLo,ipHi);
			ASSERT(ea>0.);
			hydro.HyLife[ipHi] += (float)ea;
		}
	}

	/* now fill in rest of array with just to make sure never used */
	for( ipHi=need; ipHi<NHYDRO_MAX_LEVEL; ++ipHi )
	{
		hydro.HyLife[ipHi] = -FLT_MAX;
	}

	/* this are not defined and must never be used */
	hydro.HyLife[ipH1s] = -FLT_MAX;
	/* >>chng 01 jan 26, from 8.23 to 8.226 */
	hydro.HyLife[ipH2s] = 8.226f;

	/**************************************************************************
	 *  now fill in full array of damping constants 
	 **************************************************************************/
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			/* charge to 4th power, needed for scaling laws */
			z4 = POW2(nelem+1.);
			z4 *= z4;
			/* put something in for 2s - 2p */
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].damprel = 1e-30f;
			EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].damp = EmisLines[ipH_LIKE][nelem][ipH2p][ipH2s].damprel / 1e5f;

			for( ipHi=ipH2p; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
				{
					if( EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN <= 0. )
						continue;
					/* get A's from H itself, scaling by charge */
					/* ipLnDampRel is related to the damping constant.  The number stored
					 * is the ratio Gamma lambda / 4 pi where lambda is the wavelength in
					 * cm (obtained by dividing by the energy in wavenumbers) and Gamma is
					 * the inverse lifetime, or sum of A's out of that level */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].damprel = 
						(float)(
						/* this is sum of A's to n=2 for H by itself */
						(hydro.HyLife[ipHi]*z4 + 
						/* in following must add on the Lyman lines */
					  EmisLines[ipH_LIKE][nelem][ipHi][ipH1s].Aul)/
					  PI4/EmisLines[ipH_LIKE][nelem][ipHi][ipLo].EnergyWN);

					/* >>chng 01 oct 19, define here */
					/* the actual damping constant */
					EmisLines[ipH_LIKE][nelem][ipHi][ipLo].damp = EmisLines[ipH_LIKE][nelem][ipHi][ipLo].damprel/1e5f;

					ASSERT(EmisLines[ipH_LIKE][nelem][ipHi][ipLo].damprel> 0.);
				}
			}

			/* now do extra Lyman lines */
			ipLo = 0;
			for( ipHi=2; ipHi < iso.nLyman[ipH_LIKE]; ipHi++ )
			{
				iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].damprel = 
					(float)(
					/* this is sum of A's to n=2 for H by itself */
					(hydro.HyLife[ipHi]*z4 + 
					/* in following must add on the Lyman lines */
					iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].Aul)/
					PI4/iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].EnergyWN);
					/* the actual damping constant */
					iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].damp = iso.ExtraLymanLines[ipH_LIKE][nelem][ipHi].damprel/1e5f;
			}
		}
	}

	/* 2p 1s is special since do not assume 2s 2p mixed */
	/* the constant 7.958e-6 in the damping constant is 1/4pi * conversion fractor for microns */
	hdamp[ipH2p] = EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Aul*7.958e-6/0.75;

	/* NB hdamp is only dimensioned up through lmhlvl, so 2s-1s has no damp cnst */
	hydro.HyLife[ipH2p] = EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Aul;

	for( ipHi=3; ipHi < iso.numLevels[ipH_LIKE][ipHYDROGEN]; ipHi++ )
	{
		hydro.HyLife[ipHi] = 0.;
		for( ipLo=ipH2s; ipLo < ipHi; ipLo++ )
		{
			/* array of lifetimes for hydrogen, scaled values used in HydroPesc to
			 * generate density dependent A's, Lyman lines are not included in this loop */
			hydro.HyLife[ipHi] += EmisLines[ipH_LIKE][ipHYDROGEN][ipHi][ipLo].Aul;
		}
		/* lyman lines not included in HyLife, since branching ratios only defined
		 * to get case B right.  for damp now add on lyman lines */
		hdamp[ipHi] = (hydro.HyLife[ipHi] + EmisLines[ipH_LIKE][ipHYDROGEN][ipHi][ipH1s].Aul)*7.958e-6/
		  (1. - 1./POW2( (float)ipHi));
	}

	/* levels not possible */
	hdamp[ipH1s] = 0.;
	hdamp[ipH2s] = 0.;

	/* define excitation and ionization temperatures for hydrogen only,
	 * helium is done helike where non-hydrogenic energies are assigned */
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* define these for H and He always */
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{

			/* get ground ionization threshold from Dima's fits */
			Potential = PH1COM.PH1[0][0][nelem][0]/EVRYD* 0.9998787;

			for( ipHi=ipH2p; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
			{
				iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipHi] = 
					Potential/POW2((double)ipHi);
				/*HdeltaT.HCionTe[nelem][ipHi] = 
					(float)(TE1RYD*iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipHi]);*/
			}

			/* 2s done here */
			iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipH2s] = iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipH2p];
			/*HdeltaT.HCionTe[nelem][ipH2s] = 
				(float)(TE1RYD*iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipH2p]);*/

			/* 1s done here */
			iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipH1s] = Potential;
			/*HdeltaT.HCionTe[nelem][ipH1s] = 
				(float)(TE1RYD*iso.xIsoLevNIonRyd[ipH_LIKE][nelem][ipH1s]);*/
		}
	}

	/* statistical weights for helium singlet atom and ion */
	for( ipHi=0; ipHi < NHE1LVL; ipHi++ )
	{
		n2 = (ipHi+1)*(ipHi+1);
		He1NIonRyd.He1IonRyd[ipHi] = 1.f/n2;
		He1deltaT.He1CionTe[ipHi] = (float)(TE1RYD*He1NIonRyd.He1IonRyd[ipHi]);
		he1statCOM.he1stat[ipHi] = (float)n2;
	}

	/* ground and n=2 not hydrogenic */
	He1NIonRyd.He1IonRyd[0] = 1.80802f;
	He1NIonRyd.He1IonRyd[1] = 0.2478f;
	He1NIonRyd.He1IonRyd[NHE1LVL] = He1NIonRyd.He1IonRyd[1];
	he1statCOM.he1stat[NHE1LVL] = 1.;
	he1statCOM.he1stat[1] = 3.;

	for( ipHi=1; ipHi < NHE1LVL; ipHi++ )
	{
		for( lo=0; lo <= (ipHi - 1); lo++ )
		{
			He1EnerRyd[lo][ipHi] = He1NIonRyd.He1IonRyd[lo] - 
			  He1NIonRyd.He1IonRyd[ipHi];
			He1deltaT.He1deltaTe[lo][ipHi] = 
				(float)(TE1RYD*He1EnerRyd[lo][ipHi]);
		}
	}

	for( i=0; i < (NHE1LVL + 1); i++ )
	{
		he1ind.he1qin[i] = 0.;
		he1ind.he1rin[i] = 0.;
	}

	/* these are all the helium triplet lines */
	for( i=0; i < NHE3TAU; i++ )
	{
		he3tau[i].TauIn = opac.taumin;
		he3tau[i].TauTot = 1e20f;
		he3tau[i].Pesc = 1.;
		he3tau[i].Pelec_esc = 0.;
		he3tau[i].FracInwd = 1.;
		/* 2 is atomic number of helium */
		he3tau[i].nelem = 2;
		he3tau[i].IonStg = 1;
	}
	/* now set wavelengths */
	/*lint -e778 expression evaluates to 0 */
	he3tau[IPT10830-1].WLAng = 10830.;
	he3tau[IPT3889-1].WLAng = 3889.;
	he3tau[IPT5876-1].WLAng = 5876.;
	he3tau[IPT7065-1].WLAng = 7065.;
	/* now set A's */
	he3tau[IPT10830-1].Aul = 1.02e7f;
	he3tau[IPT3889-1].Aul = 9.48e6f;
	he3tau[IPT5876-1].Aul = 7.06e7f;
	he3tau[IPT7065-1].Aul = 2.78e7f;
	/*lint +e778 expression evaluates to 0 */

	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			for( i=ipH1s; i < iso.numLevels[ipH_LIKE][nelem]; i++ )
			{
				hydro.DestRate[nelem][i] = -FLT_MAX;
			}
		}
	}

	for( ipISO=ipHYDROGEN; ipISO<NISO; ++ipISO )
	{
		for( nelem=0; nelem < LIMELM; nelem++ )
		{
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				for( i=0; i < iso.numLevels[ipISO][nelem]; i++ )
				{
					iso.gamnc[ipISO][nelem][i] = -DBL_MAX;
					iso.RecomInducRate[ipISO][nelem][i] = -DBL_MAX;
					iso.DepartCoef[ipISO][nelem][i] = -FLT_MAX;
					iso.Pop2Ion[ipISO][nelem][i] = 0.;
					iso.ConOpacRatio[ipISO][nelem][i] = 1.f;
					iso.RadRecomb[ipISO][nelem][i][ipRecRad] = -FLT_MAX;
					iso.RadRecomb[ipISO][nelem][i][ipRecNetEsc] = -FLT_MAX;
					iso.RadRecomb[ipISO][nelem][i][ipRecEsc] = -FLT_MAX;
				}
			}
		}
	}

	/* ground state of H and He is different since totally determine
	 * their own opacities */
	iso.ConOpacRatio[ipH_LIKE][ipHYDROGEN][0] = 1e-5f;
	iso.ConOpacRatio[ipH_LIKE][ipHELIUM][0] = 1e-5f;
	iso.ConOpacRatio[ipHE_LIKE][ipHELIUM][0] = 1e-5f;

	/* this is hydrogenic screening factor for HeI singlet ground state */
	scrhe1 = 1.808;
	for( i=0; i < (NHE1LVL - 1); i++ )
	{
		double glo , ghi;
		/* damping constant scales as A*lambda, or z */
		/* for hydrogen, hdamp[1] is 1s, [2] is 2,
		 * so in following changed i_ to i in hdamp */
		he1dmpCOM.he1dmp[i] = (float)hdamp[i+1];
		phe1lv.he1gam[i] = 0.;
		phe1lv.he1n[i] = 1e-35f;
		phe1lv.he1clc[i] = 0.;
		phe1lv.he1rec[0][i] = 0.;
		phe1lv.he1rec[ipRecNetEsc][i] = 1.;
		phe1lv.he1rec[ipRecEsc][i] = 1.;
		/* these are hydro but ok since only comes in ratio */
		if( i==0 )
		{
			glo = 2.;
		}
		else if( i==1 )
		{
			glo = 2.;
		}
		else if( i==2 )
		{
			glo = 6.;
		}
		else
		{
			glo = 2.*POW2(i+1.);
		}
		/* NHE1LVL is 9 in nhe1lvl.h */
		for( j=i + 1; j < NHE1LVL; j++ )
		{
			/* these are hydro but ok since only comes in ratio */
			if( j==0 )
			{
				ghi = 2.;
			}
			else if( j==1 )
			{
				ghi = 2.;
			}
			else if( j==2 )
			{
				ghi = 6.;
			}
			else
			{
				ghi = 2.*POW2(j+1.);
			}
			phe1lv.he1cll[i][j] = 0.;

			/* scale HeII A's from hydrogenic, HeAsSav are set in a series of =
			 * in zerologic */
			he1as.He1EinA[i][j] = (float)(scrhe1*AsSav.HeAsSav[i][j]);

			he1tauCOM.he1opc[i][j] = 
				(float)(he1as.He1EinA[i][j]*2.249e-26*
			  ghi/glo*
			  /* HEnerRyd has 1 for 1s, so is offset by one in the c version,
			   * I changed the i_ and j_ to i,j */
				/*>>>>chng 99 jan 26, crashed when H atom smaller than He atom*/
			  /*powi(RYDLAM/HEnrRyd.HEnerRyd[i+1][j+1],3));*/
			  powi(RYDLAM/(HIONPOT*(1./((i+1.)*(i+1.)) - 1./((j+1.)*(j+1.)) ) ),3));
			/* destruction probabilities */
			phe1lv.he1dst[i][j] = 0.;
			phe1lv.he1mis[i][j] = 0.;
		}
		scrhe1 = 1.;
	}

	for( i=0; i < NHE1LVL; i++ )
	{
		heopfr.ophe1f[i] = 0.;
	}

	for( lo=0; lo < (NHE1LVL - 1); lo++ )
	{
		if( lo == 1 )
		{
			test = 8.;
		}
		else
		{
			test = he1statCOM.he1stat[lo];
		}
		for( ipHi=lo + 1; ipHi < NHE1LVL; ipHi++ )
		{
			/*lint -e771 He1EnerRyd possibley not init */
			he1tauCOM.he1opc[lo][ipHi] = (float)(he1as.He1EinA[lo][ipHi]*
			  2.24484e-26*he1statCOM.he1stat[ipHi]/test*powi(RYDLAM/
			  He1EnerRyd[lo][ipHi],3));
			/*lint +e771 He1EnerRyd possibley not init */
		}
	}

	/* HeI ground state is special */
	he1tauCOM.he1opc[0][1] = 2.42e-8f;
	he1as.He1EinA[0][1] = 17.99e8f;
	he1dmpCOM.he1dmp[1] *= 1.37f;
	phe1lv.he1n[NHE1LVL-1] = 1e-30f;
	phe1lv.he1gam[NHE1LVL-1] = 0.;
	phe1lv.he1clc[NHE1LVL-1] = 0.;
	phe1lv.he1rec[0][NHE1LVL-1] = 0.;
	phe1lv.he1rec[ipRecNetEsc][NHE1LVL-1] = 1.;
	phe1lv.he1rec[ipRecEsc][NHE1LVL-1] = 1.;
	he1dmpCOM.he1dmp[NHE1LVL-1] = (float)hdamp[8];
	phe1lv.he12s = 1e-30f;
	phe1lv.he12p = 1e-30f;

	/* zero out inner optical depths */
	for( i=0; i < (NHE1LVL - 1); i++ )
	{
		for( j=0; j < NHE1LVL; j++ )
		{
			he1nxtCOM.he1nxt[i][j] = opac.taumin;
			he1tauCOM.he1tau[i][j] = opac.taumin;
		}
	}
	t206.TauIn = opac.taumin;
	t206.TauTot = opac.taumin;
	t206.Pesc = 1.;
	t206.Pelec_esc = 0.;
	t206.FracInwd = 0.5;
	t206.Aul = 1.976e6;

	/* now free the local space malloced above */
	free(hdamp);

	/* zero out some line information */
	iso_zero();

#	ifdef DEBUG_FUN
	fputs( " <->iso_create()\n", debug_fp );
#	endif
	return;
}

/* ============================================================================== */
static void iso_zero(void)
{
	long int i, 
	  j ,
	  ipHi, 
	  ipISO,
	  ipLo, 
	  nelem;

#	ifdef DEBUG_FUN
	fputs( "<+>iso_zero()\n", debug_fp );
#	endif

	hydro.HLineWidth = 0.;

	/* main isoelectronic arrays, fill with sane values */
	for( ipISO=ipHYDROGEN; ipISO < NISO; ++ipISO )
	{
		for( nelem=ipISO; nelem < LIMELM; nelem++ )
		{
			/* must always do helium even if turned off */
			if( nelem < 2 || abundances.lgElmtOn[nelem] )
			{
				for( ipLo=0; ipLo < (iso.numLevels[ipISO][nelem] - 1); ipLo++ )
				{
					for( ipHi=ipLo + 1; ipHi < iso.numLevels[ipISO][nelem]; ipHi++ )
					{
						EmisLines[ipISO][nelem][ipHi][ipLo].Pesc = 1.0;

						/* destruction probability*/
						EmisLines[ipISO][nelem][ipHi][ipLo].Pdest = 0.0;
						EmisLines[ipISO][nelem][ipHi][ipLo].Pelec_esc = 0.0;

						/* upper and lower level populations */
						EmisLines[ipISO][nelem][ipHi][ipLo].PopHi = 0.0;
						EmisLines[ipISO][nelem][ipHi][ipLo].PopLo = 0.0;

						EmisLines[ipISO][nelem][ipHi][ipLo].xIntensity = 0.0;

						/* basic collision Info:
						* these are not Mewe lines, they are high quality: */
						EmisLines[ipISO][nelem][ipHi][ipLo].cs1 = 0.0;

						/* following three will eventually be set by model atoms, but
						* must have reasonable value at start */
						EmisLines[ipISO][nelem][ipHi][ipLo].pump = 0.;
						EmisLines[ipISO][nelem][ipHi][ipLo].AovTot = 0.;
						EmisLines[ipISO][nelem][ipHi][ipLo].cool = 0.;
						EmisLines[ipISO][nelem][ipHi][ipLo].heat = 0.;
						EmisLines[ipISO][nelem][ipHi][ipLo].ColOvTot = 0.;
						EmisLines[ipISO][nelem][ipHi][ipLo].PopOpc = 0.;
					}
				}
			}
		}
	}

	/* heliumlike autoionization array, fill with sane values */
	for( nelem=1; nelem < LIMELM; nelem++ )
	{
		/* must always do helium even if turned off */
		if( nelem < 2 || abundances.lgElmtOn[nelem] )
		{
			for( ipHi = 0; ipHi < 4 ; ipHi++ )
			{
				He1AutoLines[nelem][ipHi].Pesc = 1.0;

				/* destruction probability*/
				He1AutoLines[nelem][ipHi].Pdest = 0.0;
				He1AutoLines[nelem][ipHi].Pelec_esc = 0.0;

				/* upper and lower level populations */
				He1AutoLines[nelem][ipHi].PopHi = 0.0;
				He1AutoLines[nelem][ipHi].PopLo = 0.0;

				He1AutoLines[nelem][ipHi].xIntensity = 0.0;

				/* basic collision Info:
				 * these are not Mewe lines, they are high quality: */
				He1AutoLines[nelem][ipHi].cs1 = 0.0;

				/* following three will eventually be set by model atoms, but
				 * must have reasonable value at start */
				He1AutoLines[nelem][ipHi].pump = 0.;
				He1AutoLines[nelem][ipHi].AovTot = 0.;
				He1AutoLines[nelem][ipHi].cool = 0.;
				He1AutoLines[nelem][ipHi].heat = 0.;
				He1AutoLines[nelem][ipHi].ColOvTot = 0.;
				He1AutoLines[nelem][ipHi].PopOpc = 0.;
			}
		}
	}

	for( i=0; i < (NHE1LVL + 1); i++ )
	{
		he1ind.he1qin[i] = 0.;
		he1ind.he1rin[i] = 0.;
	}

	t206.TauIn = opac.taumin;
	t206.TauTot = opac.taumin;
	t206.Pesc = 1.;
	t206.Pelec_esc = 0.;
	t206.FracInwd = 0.5;
	t206.Aul = 1.976e6;

	/* these are all the helium triplet lines */
	for( i=0; i < NHE3TAU; i++ )
	{
		he3tau[i].TauIn = opac.taumin;
		he3tau[i].TauTot = 1e20f;
		he3tau[i].Pesc = 1.;
		he3tau[i].Pelec_esc = 0.;
		he3tau[i].FracInwd = 1.;
		/* 2 is atomic number of helium */
		he3tau[i].nelem = 2;
		he3tau[i].IonStg = 1;
	}
	
	for( i=0; i < NHE1LVL; i++ )
	{
		heopfr.ophe1f[i] = 0.;
	}
	for( i=0; i < (NHE1LVL - 1); i++ )
	{
		phe1lv.he1gam[i] = 0.;
		for( j=0; j < NHE1LVL; j++ )
		{
			he1nxtCOM.he1nxt[i][j] = opac.taumin;
			he1tauCOM.he1tau[i][j] = opac.taumin;
		}
	}

	/* initialize heavy element line array */
	for( i=1; i <= nLevel1; i++ )
	{
		TauLines[i].TauIn = opac.taumin;
		TauLines[i].TauCon = opac.taumin;
		TauLines[i].ColOvTot = 0.;
		/* outward optical depth */
		TauLines[i].TauTot = 1e30f;
		/* escape probability */
		TauLines[i].Pesc = 1.;
		/* inward part of line */
		TauLines[i].FracInwd = 1.;
		/* destruction probability */
		TauLines[i].Pdest = 0.;
		TauLines[i].Pelec_esc = 0.;
		/* line pumping rate */
		TauLines[i].pump = 0.;
		/* population of lower level */
		TauLines[i].PopLo = 0.;
		/* >>chng 97 jul 21, added following zero
		 * population of upper level */
		TauLines[i].PopHi = 0.;
		/* population of lower level with correction for stim emission */
		TauLines[i].PopOpc = 0.;
		/* following two heat exchange excitation, deexcitation */
		TauLines[i].cool = 0.;
		TauLines[i].heat = 0.;
		/* intensity of line */
		TauLines[i].xIntensity = 0.;
		/* number of photons emitted in line */
		TauLines[i].phots = 0.;
		/* opacity in line */
		TauLines[i].dTau = 0.;
		/* indicates that this is a high quality transition */
		TauLines[i].cs1 = 0.;
	}

	for( i=0; i < nWindLine; i++ )
	{
		TauLine2[i].PopLo = 0.;
		TauLine2[i].PopHi = 0.;

		/* these are line optical depth arrays
		 * inward optical depth */
		TauLine2[i].TauIn = opac.taumin;
		TauLine2[i].TauCon = opac.taumin;
		TauLine2[i].ColOvTot = 0.;
		/* outward optical depth */
		TauLine2[i].TauTot = 1e20f;
		/* escape probability */
		TauLine2[i].Pesc = 1.;
		/* inward part of line */
		TauLine2[i].FracInwd = 1.;
		/* destruction probability */
		TauLine2[i].Pdest = 0.;
		TauLine2[i].Pelec_esc = 0.;
		/* line pumping rate */
		TauLine2[i].pump = 0.;
		/* population of lower level */
		TauLine2[i].PopLo = 0.;
		/* population of upper level */
		TauLine2[i].PopHi = 0.;
		/* population of lower level with correction for stim emission */
		TauLine2[i].PopOpc = 0.;
		/* following two heat exchange excitation, deexcitation */
		TauLine2[i].cool = 0.;
		TauLine2[i].heat = 0.;
		/* intensity of line */
		TauLine2[i].xIntensity = 0.;
		/* number of photons emitted in line */
		TauLine2[i].phots = 0.;
		/* opacity in line */
		TauLine2[i].dTau = 0.;
	}

	for( i=0; i < nHFLines; i++ )
	{
		HFLines[i].PopLo = 0.;
		HFLines[i].PopHi = 0.;

		/* these are line optical depth arrays
		 * inward optical depth */
		HFLines[i].TauIn = opac.taumin;
		HFLines[i].TauCon = opac.taumin;
		HFLines[i].ColOvTot = 0.;
		/* outward optical depth */
		HFLines[i].TauTot = 1e20f;
		/* escape probability */
		HFLines[i].Pesc = 1.;
		/* inward part of line */
		HFLines[i].FracInwd = 1.;
		/* destruction probability */
		HFLines[i].Pdest = 0.;
		HFLines[i].Pelec_esc = 0.;
		/* line pumping rate */
		HFLines[i].pump = 0.;
		/* population of lower level */
		HFLines[i].PopLo = 0.;
		/* population of upper level */
		HFLines[i].PopHi = 0.;
		/* population of lower level with correction for stim emission */
		HFLines[i].PopOpc = 0.;
		/* following two heat exchange excitation, deexcitation */
		HFLines[i].cool = 0.;
		HFLines[i].heat = 0.;
		/* intensity of line */
		HFLines[i].xIntensity = 0.;
		/* number of photons emitted in line */
		HFLines[i].phots = 0.;
		/* opacity in line */
		HFLines[i].dTau = 0.;
	}

	for( i=0; i < nCORotate; i++ )
	{
		C12O16Rotate[i].PopLo = 0.;
		C13O16Rotate[i].PopLo = 0.;
		C12O16Rotate[i].PopHi = 0.;
		C13O16Rotate[i].PopHi = 0.;

		/* these are line optical depth arrays
		 * inward optical depth */
		C12O16Rotate[i].TauIn = opac.taumin;
		C13O16Rotate[i].TauIn = opac.taumin;
		C12O16Rotate[i].TauCon = opac.taumin;
		C13O16Rotate[i].TauCon = opac.taumin;
		C12O16Rotate[i].ColOvTot = 0.;
		C13O16Rotate[i].ColOvTot = 0.;
		/* outward optical depth */
		C12O16Rotate[i].TauTot = 1e20f;
		C13O16Rotate[i].TauTot = 1e20f;
		/* escape probability */
		C12O16Rotate[i].Pesc = 1.;
		C13O16Rotate[i].Pesc = 1.;
		/* inward part of line */
		C12O16Rotate[i].FracInwd = 1.;
		C13O16Rotate[i].FracInwd = 1.;
		/* destruction probability */
		C12O16Rotate[i].Pdest = 0.;
		C13O16Rotate[i].Pdest = 0.;
		C12O16Rotate[i].Pelec_esc = 0.;
		C13O16Rotate[i].Pelec_esc = 0.;
		/* line pumping rate */
		C12O16Rotate[i].pump = 0.;
		C13O16Rotate[i].pump = 0.;
		/* population of lower level */
		C12O16Rotate[i].PopLo = 0.;
		C13O16Rotate[i].PopLo = 0.;
		/* population of upper level */
		C12O16Rotate[i].PopHi = 0.;
		C13O16Rotate[i].PopHi = 0.;
		/* population of lower level with correction for stim emission */
		C12O16Rotate[i].PopOpc = 0.;
		C13O16Rotate[i].PopOpc = 0.;
		/* following two heat exchange excitation, deexcitation */
		C12O16Rotate[i].cool = 0.;
		C13O16Rotate[i].cool = 0.;
		C12O16Rotate[i].heat = 0.;
		C13O16Rotate[i].heat = 0.;
		/* intensity of line */
		C12O16Rotate[i].xIntensity = 0.;
		C13O16Rotate[i].xIntensity = 0.;
		/* number of photons emitted in line */
		C12O16Rotate[i].phots = 0.;
		C13O16Rotate[i].phots = 0.;
		/* opacity in line */
		C12O16Rotate[i].dTau = 0.;
		C13O16Rotate[i].dTau = 0.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->iso_zero()\n", debug_fp );
#	endif
	return;
}

