/*Hydrogenic main routine to call HydroLevel and determine model hydrogen atom level balance */
#include "cddefines.h"
#include "taulines.h"
#include "hydrogenic.h"
#include "iso.h"
#include "trace.h"
#include "ionfracs.h"
#include "secondaries.h"
#include "phycon.h"
#include "hmi.h"
#include "h2.h"
#include "hyperfine.h"
#include "chargtran.h"
#include "abundances.h"
#include "ionrange.h"
#include "hmole.h"

void Hydrogenic(void)
{
	long int ipLo ,
		ipHi , 
		nelem;
	double  coltot, 
	  gamtot;
	float hmovh1 = 0.; 

#	ifdef DEBUG_FUN
	fputs( "<+>Hydrogenic()\n", debug_fp );
#	endif

	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* do not consider elements that have been turned off */
		if( abundances.lgElmtOn[nelem] )
		{
			/* note that nelem scale is totally on c not physical scale, so 0 is h */
			/* evaluate hydrogenic balance if ionization reaches this high */
			if( (IonRange.IonHigh[nelem] == nelem + 1)  )
			{
				/* evaluate collisional rates */
				HydroCollid(nelem);

				/* evaluate photoionization rates  */
				iso_photo( ipHYDROGEN , nelem );

				/* evaluate recombination rates */
				HydroRecom(nelem);

				/* solve for the level populations */
				HydroLevel(nelem);

				if( trace.lgTrace )
				{
					fprintf( ioQQQ, "     Hydrogenic Z=%2ld H2OVH1=",nelem);
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.xIonRatio[ipH_LIKE][nelem]));
					fprintf( ioQQQ, " simple=");
					fprintf( ioQQQ,PrintEfmt("%10.3e", iso.xIonRatio[ipH_LIKE][nelem]));
					fprintf( ioQQQ, " H-ovH1:");
					fprintf( ioQQQ,PrintEfmt("%10.2e", hmovh1 ) );
					fprintf( ioQQQ, "\n"); 
				}

			}
			else
			{
				iso.xIonRatio[ipH_LIKE][nelem] = 0.;
				iso.xIonRatio[ipH_LIKE][nelem] = 0.;

				/* zero it out since no population*/
				for( ipHi=ipH2s; ipHi < iso.numLevels[ipH_LIKE][nelem]; ipHi++ )
				{
					for( ipLo=ipH1s; ipLo < ipHi; ipLo++ )
					{
						/* population of lower level rel to ion */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopLo = 0.;

						/* population of upper level rel to ion */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopHi =  0.;

						/* population of lower level rel to ion, corrected for stim em */
						EmisLines[ipH_LIKE][nelem][ipHi][ipLo].PopOpc =  0.;
					}
				}
			}
		}
	}

	/* rest is for hydrogen only */

	/* do molecular balance 
	 * hmovh1 will be ratio of molecular to atomic hydrogen
	 * HIonFrac is fration of H that is ionized, ratio of ion to atom */
	hmole(&hmovh1,iso.xIonRatio[ipH_LIKE][0]);

	/* >>chng 01 may 09, add option to force abundance, with element name ioniz cmmnd */
	if( abundances.lgSetIoniz[ipHYDROGEN] )
	{
		xIonFracs[ipHYDROGEN][1] = abundances.SetIoniz[ipHYDROGEN][1]*abundances.gas_phase[ipHYDROGEN];
		xIonFracs[ipHYDROGEN][0] = abundances.SetIoniz[ipHYDROGEN][0]*abundances.gas_phase[ipHYDROGEN];
	}
	else
	{
		/* get ionization balance */
		xIonFracs[ipHYDROGEN][1] = (float)(phycon.hden*iso.xIonRatio[ipH_LIKE][0]/
			(1. + iso.xIonRatio[ipH_LIKE][0] + hmovh1));

		xIonFracs[ipHYDROGEN][0] = (float)(phycon.hden/(1. + iso.xIonRatio[ipH_LIKE][0] + hmovh1));
	}

	/* now do level populations for H2 */
	H2_LevelPops();

	/* is hydrogen atom in condition of collisional rec? if so then
	 * only very small dT's can be tolerated in ConvTempEdenIoniz
	 * RadRec_effec is sum of radiative recombination */
	if( iso.RecomCollis[ipH_LIKE][ipHYDROGEN]/iso.RadRec_effec[ipH_LIKE][ipHYDROGEN] > 100. )
	{
		hydro.lgHColRec = TRUE;
	}
	else
	{
		hydro.lgHColRec = FALSE;
	}

	/*fprintf(ioQQQ,"ratio %.1e\n", iso.gamnc[ipH_LIKE][ipHELIUM][ipH2s]/EmisLines[ipH_LIKE][ipHELIUM][ipH2s][ipH1s].Aul);*/

	/* this is test whether collisions are important first define ratio of exits
	 * from ground due to coll, rel to everthing else, then flag is large */
	if( iso.PopLTE[ipH_LIKE][0][ipH1s] > 1e-30 )
	{
		coltot = 
			EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*
			iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH2p]/iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH1s]*
		  phycon.eden*(iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH2p] + iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH2p] + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][3][ipH2p].ColUL*iso.PopLTE[ipH_LIKE][ipHYDROGEN][3]/iso.PopLTE[ipH_LIKE][ipHYDROGEN][ipH2p])/
		  (EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*phycon.eden + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Aul*
		  (EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pesc + 
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pelec_esc +
		  EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].Pdest) );

		/* caution that collisions are important (and so only small changes in
		 * temperature should happen) if more than 20% of the total */
		if( coltot > 0.2 )
		{
			hydro.lgHColionImp = TRUE;
		}
	}
	else
	{
		hydro.lgHColionImp = FALSE;
	}

	/* remember the ratio of pops of 2p to 1s for possible printout in prtComment
	 * and to obtain Lya excitation temps.  the pop of ground is not defined if
	 * NO ionization at all since these pops are relative to ion */
	/* >>chng 99 jun 03, added MAX2 to protect against totally neutral gas */
	if( iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p]/MAX2(SMALLFLOAT,iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]) > 0.1 &&
		iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s] > SMALLFLOAT )
	{
		hydro.lgHiPop2 = TRUE;
		hydro.pop2mx = (float)MAX2(iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH2p]/iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s],
		  hydro.pop2mx);
	}

	gamtot = iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s] + Secondaries.csupra;

	coltot = iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH1s] + 
		EmisLines[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s].ColUL*4.*
	  iso.Boltzmann[ipH_LIKE][ipHYDROGEN][ipH2p][ipH1s];

	/* if ground state destruction rate is significant, recall different dest procceses */
	if( hydro.DestRate[ipHYDROGEN][ipH1s] > 0. )
	{
		hydro.photn2 = 
			(float)(iso.gamnc[ipH_LIKE][ipHYDROGEN][ipH1s]/hydro.DestRate[ipHYDROGEN][ipH1s] );

		/* fraction of ionizations of H from ground, due to thermal collisions */
		hydro.HCollIonFrac = 
			(float)(iso.ColIoniz[ipH_LIKE][ipHYDROGEN][ipH1s]*phycon.eden/hydro.DestRate[ipHYDROGEN][ipH1s]);

		/* this flag is used in ConvIonizeOpacityDo to decide whether we
		 * really need to converge the secondary ionization rates */
		Secondaries.sec2total = 
			(float)(Secondaries.csupra / hydro.DestRate[ipHYDROGEN][ipH1s]) ;

		/* frac of ionizations due to ct */
		ChargTran.HIonFrac = ChargTran.HCharExcIonTotal / hydro.DestRate[ipHYDROGEN][ipH1s];
	}
	else
	{
		hydro.HCollIonFrac = 0.;
		hydro.photn2 = 0.;
		Secondaries.sec2total = 0. ;
		ChargTran.HIonFrac = 0.;
	}

	/* now do 21 cm, will define populations here */
	TauLines[ipH21cm].PopHi = xIonFracs[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0]*0.75;	
	TauLines[ipH21cm].PopLo = xIonFracs[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0]*0.25;
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipH21cm].PopOpc = xIonFracs[ipHYDROGEN][1]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][0] * 
		0.25f * TauLines[ipH21cm].EnergyK / phycon.te;

	/* now do 3 cm, the HeII equivalent of H 21 cm will define populations here */
	TauLines[ipHe3cm].PopHi = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*0.75*hyperfine.HFLabundance[2];	
	TauLines[ipHe3cm].PopLo = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*0.25*hyperfine.HFLabundance[2];
	/* assume spin temperature is gas kinetic temperature */
	TauLines[ipHe3cm].PopOpc = EmisLines[ipH_LIKE][ipHELIUM][ipH2p][ipH1s].PopLo*hyperfine.HFLabundance[2] * 
		0.25f * TauLines[ipHe3cm].EnergyK / phycon.te;


	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "       Hydrogenic retrn H,2,NE=");
		fprintf(ioQQQ,"%.2e ", xIonFracs[ipHYDROGEN][0]);
		fprintf(ioQQQ,"%.2e ", xIonFracs[ipHYDROGEN][1]);
		fprintf(ioQQQ,"%.2e ", phycon.eden);
		fprintf( ioQQQ, " REC, COL, GAMT= ");
		/* recomb rate coef, cm^3 s-1 */
		fprintf(ioQQQ,"%.2e ", iso.RadRec_effec[ipH_LIKE][0] );
		fprintf(ioQQQ,"%.2e ", coltot);
		fprintf(ioQQQ,"%.2e ", gamtot);
		fprintf( ioQQQ, " CSUP=");
		PrintE82( ioQQQ, Secondaries.csupra);
		fprintf( ioQQQ, " Htwo=");
		fprintf(ioQQQ,"%.2e ", hmi.htwo);
		fprintf( ioQQQ, " Te=");
		fprintf(ioQQQ,"%.2e ", phycon.te );
		fprintf( ioQQQ, "\n");
	}

#	ifdef DEBUG_FUN
	fputs( " <->Hydrogenic()\n", debug_fp );
#	endif
	return;
}
