/*HydroBranch compute branching ratios for Jason Ferguson's model hydrogen atom */
#include "cddefines.h"
#include "hydrogenic.h"
#include "phycon.h"
#include "atomcwgt.h"

double HydroBranch(
	/* upper level quantum number, must be <= 15 */
	long int nHi, 
	/* lower quantum number, 2 or greater */
	long int nLo, 
	/* physical atomic number, H is 1 */
	long int ipZPhys)
{

	/* function to evaluate branching ratios for hydrogen lines
	 * as a function of density
	 * Jason Ferguson, 1994-95,
	 * defined for lower 2 and above, and ipHi from 4 to 15 */

	long int i, 
	  nelem ,
	  ipHi,
	  ipLo,
	  k, 
	  _r;
	double DenScale, 
	  RedMass, 
	  answer, 
	  branch_v, 
	  den, 
	  denhigh, 
	  denint, 
	  denlow, 
	  denzero, 
	  t, 
	  teuse, 
	  tt, 
	  z7;
	static double 
	  /* Low density limit: */
	  Alow[14][15], 
	  Blow[14][15], 
	  Clow[14][15], 

	  /* High density limit: */
	  Ahigh[14][15], 

	  /* Intermediate limit: */
	  Aint[14][15], 
	  Bint[14][15], 
	  Cint[14][15], 

	  /* Zero density limit: */
	  Azero[14][15], 
	  Bzero[14][15], 
	  Czero[14][15], 

	  DenCut[15];

	/*
	 * Arrays A,B,C satisfy the equation y=A+B*x+C*x**2, x=LOG10(temp)
	 * for the low density branching ratio.
	 * Each line or pair of lines are for an upper level shown
	 * by the continue number on the left.  Reading from left to 
	 * right the fit for the ratio to 2, then 3, etc.. */
	static int _aini = 1;
	if( _aini ){ /* Do 1 TIME INITIALIZATIONS! */
		{ static double _itmp0[] = {0.24586785,0.80264270,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.19051158,0.09085006,0.79271698,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.16235442,0.06181727,
		  0.07010160,0.78640544,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.14329052,0.06178869,0.01691143,0.09711918,0.76783526,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.12881671,0.06023609,0.01400870,
		  0.01189304,0.13314129,0.74124563,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.11800103,0.05861854,0.01869008,-.00136751,0.02648726,
		  0.16735578,0.70279408,0.0,0.0,0.0,0.0,0.0,0.0,0.10937667,
		  0.05715862,0.02125686,-.00113247,0.00146353,0.04791027,0.19134593,
		  0.66540849,0.0,0.0,0.0,0.0,0.0,0.10216851,0.05506275,0.02260461,
		  0.00098223,-.00544246,0.01218124,0.06891189,0.20716748,0.62612939,
		  0.0,0.0,0.0,0.0,0.09678371,0.05294095,0.02397135,0.00386226,
		  -.00551451,-.00000937,0.02652743,0.08618066,0.21521707,0.58427924,
		  0.0,0.0,0.0,0.09139322,0.05099193,0.02426896,0.00527421,
		  -.00318780,-.00382327,0.00930019,0.04048099,0.09946553,0.22109380,
		  0.54895425,0.0,0.0,0.08803733,0.04954891,0.02397056,0.00616667,
		  -.00376645,-.00508919,0.00228244,0.02005925,0.05219151,0.10910100,
		  0.21981779,0.51764560,0.0,0.08430315,0.04793394,0.02469692,
		  0.00804711,-.00171838,-.00521967,-.00099487,0.01011098,0.02994304,
		  0.06201692,0.11499117,0.21663782,0.48468909};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Alow[k-1][i-1] = _itmp0[_r++];
				}
			}
		}
		{ static double _itmp1[] = {-.06965577,0.03657217,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.10774879,0.06099085,-.00347499,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.10866447,0.00230383,
		  0.09698921,-.04478357,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  -.10299814,-.02240677,0.05148011,0.09099282,-.07518020,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.09617072,-.03170714,0.02470934,
		  0.06817564,0.07148378,-.09601037,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,-.09021704,-.03582631,0.00840594,0.04625041,0.06688988,
		  0.04854436,-.10440453,0.0,0.0,0.0,0.0,0.0,0.0,-.08485942,
		  -.03768479,-.00040299,0.03113073,0.05237036,0.05748083,0.02920597,
		  -.10917129,0.0,0.0,0.0,0.0,0.0,-.08013427,-.03791236,-.00548895,
		  0.02134862,0.04124308,0.05070819,0.04562783,0.01378860,-.10889098,
		  0.0,0.0,0.0,0.0,-.07644645,-.03753002,-.00905632,0.01432663,
		  0.03206793,0.04263915,0.04437728,0.03431035,0.00249642,-.10357222,
		  0.0,0.0,0.0,-.07260653,-.03679784,-.01098852,0.01001109,
		  0.02469400,0.03564836,0.03989327,0.03670710,0.02434047,-.00755424,
		  -.09944646,0.0,0.0,-.07012866,-.03617226,-.01189039,0.00719885,
		  0.02125553,0.03027839,0.03484540,0.03476696,0.02935039,0.01596028,
		  -.01302458,-.09566928,0.0,-.06721935,-.03524328,-.01314037,
		  0.00436710,0.01722575,0.02618851,0.03065444,0.03176235,0.02918975,
		  0.02242342,0.00957746,-.01702143,-.08896416};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Blow[k-1][i-1] = _itmp1[_r++];
				}
			}
		}
		{ static double _itmp2[] = {0.02894746,-.02370636,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.03216739,-.00333852,-.02091632,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02987297,0.00741434,
		  -.01419180,-.01465622,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.02730905,0.01092300,-.00400422,-.01624613,-.00893700,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02502816,0.01180543,0.00070299,
		  -.00901673,-.01497620,-.00433506,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.02321886,0.01194935,0.00321121,-.00459220,-.01045978,
		  -.01243754,-.00154914,0.0,0.0,0.0,0.0,0.0,0.0,0.02167892,
		  0.01181686,0.00443314,-.00193959,-.00693192,-.01012886,-.00995642,
		  0.00061426,0.0,0.0,0.0,0.0,0.0,0.02039435,0.01149679,0.00505063,
		  -.00035140,-.00472313,-.00770657,-.00904245,-.00779705,0.00189308,
		  0.0,0.0,0.0,0.0,0.01938171,0.01114673,0.00542739,0.00071200,
		  -.00308134,-.00583235,-.00744778,-.00776871,-.00607435,0.00227978,
		  0.0,0.0,0.0,0.01840467,0.01077421,0.00556475,0.00132076,
		  -.00185743,-.00445269,-.00608454,-.00676714,-.00651325,-.00442778,
		  0.00269725,0.0,0.0,0.01771163,0.01045784,0.00554454,0.00166741,
		  -.00130716,-.00346459,-.00494351,-.00577684,-.00597581,-.00537135,
		  -.00341156,0.00307518,0.0,0.01697192,0.01011908,0.00560253,
		  0.00202628,-.00069935,-.00275550,-.00409237,-.00492361,-.00526458,
		  -.00513330,-.00444688,-.00258177,0.00290569};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Clow[k-1][i-1] = _itmp2[_r++];
				}
			}
		}
		{ static double _itmp3[] = {0.48369980,0.51630020,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.34058446,0.29616642,0.36324912,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.27441120,0.21932639,
		  0.21726075,0.28900164,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.23590802,0.18054493,0.16349038,0.17483552,0.24522109,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.21071367,0.15709351,0.13551381,
		  0.13202147,0.14847758,0.21617998,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.19277872,0.14122622,0.11830245,0.10954082,0.11204237,
		  0.13061658,0.19549286,0.0,0.0,0.0,0.0,0.0,0.0,0.17927188,
		  0.12975664,0.10657505,0.09563817,0.09280899,0.09828085,0.11770198,
		  0.17996643,0.0,0.0,0.0,0.0,0.0,0.16869755,0.12102546,0.09807262,
		  0.08613688,0.08090513,0.08119519,0.08824348,0.10786394,0.16785973,
		  0.0,0.0,0.0,0.0,0.16005717,0.11414084,0.09155979,0.07921857,
		  0.07277973,0.07063557,0.07270862,0.08059834,0.10020255,0.15809883,
		  0.0,0.0,0.0,0.15296075,0.10851952,0.08635456,0.07393567,
		  0.06685564,0.06340194,0.06307726,0.06620434,0.07459931,0.09401251,
		  0.15007855,0.0,0.0,0.14694592,0.10389631,0.08218327,0.06973679,
		  0.06231181,0.05814345,0.05647485,0.05728300,0.06108717,0.06974826,
		  0.08890329,0.14328589,0.0,0.14177141,0.09992123,0.07867678,
		  0.06632221,0.05872055,0.05413064,0.05170280,0.05118973,0.05271383,
		  0.05695079,0.06573306,0.08462565,0.13754131};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				/* The array Ahigh is the high density branching ratio */
				Ahigh[k-1][i-1] = _itmp3[_r++];
				}
			}
		}

		/* Arrays Aint,Bint,Cint are the linear fits A + B*x + C*x^2, 
		 * x=LOG10(Te) to the branching ratios at the density of DenCut
		 * below. */
		{ static double _itmp4[] = {0.66434872,0.32487863,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.52922803,0.24920899,0.22371560,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.47442847,0.22981724,
		  0.17536080,0.14932327,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.39508852,0.18620653,0.10003041,0.09467793,0.21150076,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.38539791,0.19879374,0.11116324,
		  0.08889906,0.05501413,0.14879820,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.28505614,0.12071127,0.06209489,0.05181919,0.07183042,
		  0.13235646,0.31531373,0.0,0.0,0.0,0.0,0.0,0.0,0.30299437,
		  0.13807319,0.07255217,0.04947615,0.03823176,0.05706942,0.11060041,
		  0.24197698,0.0,0.0,0.0,0.0,0.0,0.29553533,0.14984956,0.08494213,
		  0.05616220,0.03994378,0.04310994,0.05726811,0.08381837,0.17914248,
		  0.0,0.0,0.0,0.0,0.21065815,0.08030247,0.02607192,0.00796368,
		  0.00902981,0.01663509,0.05050347,0.09376139,0.17327742,0.37573284,
		  0.0,0.0,0.0,0.22357875,0.09268114,0.03927113,0.01676973,
		  0.01239286,0.01458975,0.10078439,0.05129088,0.08720516,0.15031278,
		  0.31912005,0.0,0.0,0.23536813,0.10608470,0.05187503,0.02720319,
		  0.01810590,0.02021686,0.02915859,0.03929549,0.05029994,0.07638058,
		  0.12673533,0.26089033,0.0,0.24545340,0.11870858,0.06288756,
		  0.03675172,0.02513174,0.02162876,0.02628436,0.02496247,0.03612624,
		  0.04682715,0.06872973,0.10501918,0.21104985};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Aint[k-1][i-1] = _itmp4[_r++];
				}
			}
		}
		{ static double _itmp5[] = {-.13038647,0.13880141,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.17014815,0.02383933,0.14412016,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.15722525,-.01605760,
		  0.03496768,0.12344342,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  -.16813646,-.03379565,0.03514176,0.07409590,0.09916054,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.15753850,-.04857516,0.00787171,
		  0.03282380,0.07508625,0.09556210,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,-.14428063,-.03378010,0.01326779,0.03393491,0.04253088,
		  0.04162927,0.02468526,0.0,0.0,0.0,0.0,0.0,0.0,-.14761159,
		  -.04340817,0.00149218,0.02232939,0.03857816,0.04230095,0.04012249,
		  0.03920417,0.0,0.0,0.0,0.0,0.0,-.13405739,-.04789433,-.00817234,
		  0.01151119,0.02540556,0.03051347,0.03396039,0.04259263,0.05142863,
		  0.0,0.0,0.0,0.0,-.12201415,-.03252991,0.00844759,0.02694654,
		  0.03404603,0.03830293,0.02953922,0.02094501,0.00495875,-.03406398,
		  0.0,0.0,0.0,-.12438774,-.03745292,0.00045989,0.01905439,
		  0.02676029,0.03065167,-.01522608,0.02562980,0.01880628,0.00887335,
		  -.01870824,0.0,0.0,-.12493016,-.04218918,-.00624778,0.01152544,
		  0.02001382,0.02204354,0.02116373,0.02040166,0.02162306,0.01910837,
		  0.01364339,-.00082072,0.0,-.12333725,-.04594492,-.01151305,
		  0.00531293,0.01373485,0.01759274,0.01735237,0.02141650,0.01891622,
		  0.01949276,0.01783597,0.01810829,0.01366899};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Bint[k-1][i-1] = _itmp5[_r++];
				}
			}
		}
		{ static double _itmp6[] = {0.02175534,-.02309002,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02976573,-.00343827,-.02586198,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02581227,0.00289695,
		  -.00593932,-.02089735,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.03001541,0.00705758,-.00484505,-.01262115,-.02040019,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02673706,0.00862823,-.00069219,
		  -.00525714,-.01205728,-.01788146,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.02758587,0.00827859,-.00030501,-.00471771,-.00742926,
		  -.00921960,-.01115843,0.0,0.0,0.0,0.0,0.0,0.0,0.02677808,
		  0.00902097,0.00120367,-.00274426,-.00589243,-.00730726,-.00845389,
		  -.01155141,0.0,0.0,0.0,0.0,0.0,0.02356059,0.00903905,0.00233088,
		  -.00114740,-.00365081,-.00487307,-.00594484,-.00818613,-.01179175,
		  0.0,0.0,0.0,0.0,0.02442109,0.00861272,0.00120121,-.00252505,
		  -.00441173,-.00580961,-.00537214,-.00517138,-.00451686,-.00284681,
		  0.0,0.0,0.0,0.02400008,0.00882706,0.00206227,-.00148540,
		  -.00327287,-.00436406,0.00176029,-.00487489,-.00467927,-.00457737,
		  -.00388879,0.0,0.0,0.02328219,0.00900932,0.00272242,-.00054353,
		  -.00229477,-.00301244,-.00331316,-.00357001,-.00418877,-.00446308,
		  -.00474152,-.00535384,0.0,0.02217958,0.00904923,0.00318304,
		  0.00019224,-.00143247,-.00230601,-.00257126,-.00343121,-.00332350,
		  -.00375437,-.00400222,-.00489069,-.00642249};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Cint[k-1][i-1] = _itmp6[_r++];
				}
			}
		}

		/* The zero density BR for He+ fit to a1 + a2*x + a3*x^2, LOG10(Te) */
		{ static double _itmp7[] = {0.17650783,0.79178172,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.11867613,0.05929772,0.74253058,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.09456554,0.06829530,
		  0.01239460,0.83593869,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.07854890,0.06024549,0.02647738,-.01424192,0.88956189,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.06812134,0.05505072,0.03572085,
		  -.00518158,-.02103810,0.87689340,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.06009094,0.04981204,0.03593512,0.01130077,-.02871113,
		  -.00077989,0.88960916,0.0,0.0,0.0,0.0,0.0,0.0,0.05477042,
		  0.04528970,0.03588740,0.01903958,-.01054250,-.04243176,0.01183166,
		  0.88795990,0.0,0.0,0.0,0.0,0.0,0.05026724,0.04242069,0.03366005,
		  0.02245793,0.00133515,-.02691272,-.04733379,0.04473624,0.89264274,
		  0.0,0.0,0.0,0.0,0.04690169,0.03938944,0.03376045,0.02355297,
		  0.00789906,-.01436680,-.03800366,-.04426146,0.07616858,0.88815314,
		  0.0,0.0,0.0,0.04397374,0.03698078,0.03201684,0.02334986,
		  0.01122027,-.00562193,-.02602815,-.04370789,-.03201467,0.06795727,
		  0.86584538,0.0,0.0,0.04160265,0.03513645,0.03060758,0.02355473,
		  0.01325112,-.00045402,-.01711418,-.03386206,-.04390364,-.02696792,
		  0.12224789,0.83731639,0.0,0.03962459,0.03353527,0.02942983,
		  0.02213968,0.01444045,0.00276285,-.01075065,-.02564916,-.03826793,
		  -.04084002,-.00175745,0.17509449,0.83548069};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Azero[k-1][i-1] = _itmp7[_r++];
				}
			}
		}
		{ static double _itmp8[] = {-.07035338,0.09562425,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.08158211,0.03164004,0.11208197,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.07511979,-.03725602,
		  0.08876404,0.01319402,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  -.06665008,-.04495577,0.00507675,0.12884131,-.05716589,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,-.06014463,-.04579195,-.02074505,
		  0.04024187,0.15064797,-.07466140,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,-.05446626,-.04373614,-.02764152,0.00453248,0.06907025,
		  0.14825842,-.10831214,0.0,0.0,0.0,0.0,0.0,0.0,-.05060443,
		  -.04108966,-.03038551,-.01036674,0.02829686,0.08961807,0.15077420,
		  -.13541403,0.0,0.0,0.0,0.0,0.0,-.04707611,-.03928460,-.03006442,
		  -.01712540,0.00814572,0.04789762,0.10238425,0.13016707,-.16754462,
		  0.0,0.0,0.0,0.0,-.04442082,-.03708148,-.03108100,-.02011352,
		  -.00227110,0.02537041,0.06300265,0.10766625,0.11173646,-.18835720,
		  0.0,0.0,0.0,-.04199021,-.03523231,-.03014371,-.02108224,
		  -.00771431,0.01209119,0.03932492,0.07343302,0.10473152,0.12190442,
		  -.19509342,0.0,0.0,-.04000892,-.03375297,-.02924086,-.02199272,
		  -.01099745,0.00441046,0.02488632,0.05008875,0.07892339,0.10631334,
		  0.08286211,-.19638860,0.0,-.03830726,-.03245678,-.02841859,
		  -.02117853,-.01298174,-.00027526,0.01561246,0.03534120,0.05785085,
		  0.08128670,0.09122626,0.04216800,-.21794444};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Bzero[k-1][i-1] = _itmp8[_r++];
				}
			}
		}
		{ static double _itmp9[] = {0.03442191,-.03955488,0.0,0.0,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02967211,0.00815910,-.04957449,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.02477524,0.01836980,
		  -.00648346,-.03452190,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.02109816,0.01736323,0.00891404,-.01742835,-.02286775,0.0,
		  0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.01858165,0.01595490,0.01213738,
		  0.00084739,-.02460505,-.02045348,0.0,0.0,0.0,0.0,0.0,0.0,
		  0.0,0.01661396,0.01454392,0.01216815,0.00665049,-.00613984,
		  -.02685568,-.01496959,0.0,0.0,0.0,0.0,0.0,0.0,0.01526155,
		  0.01331509,0.01175279,0.00852281,0.00129271,-.01158901,-.02992533,
		  -.00906113,0.0,0.0,0.0,0.0,0.0,0.01410287,0.01244706,0.01103599,
		  0.00904981,0.00447271,-.00339200,-.01552273,-.02747463,-.00206192,
		  0.0,0.0,0.0,0.0,0.01323370,0.01162088,0.01078594,0.00903940,
		  0.00586641,0.00054976,-.00724106,-.01790333,-.02571587,0.00274676,
		  0.0,0.0,0.0,0.01246430,0.01094577,0.01022211,0.00875643,
		  0.00641093,0.00266276,-.00279207,-.01019288,-.01863403,-.02885537,
		  0.00516807,0.0,0.0,0.01184059,0.01040109,0.00974126,0.00857950,
		  0.00664601,0.00375395,-.00028953,-.00555012,-.01214473,-.02005664,
		  -.02277730,0.00666500,0.0,0.01130772,0.00994360,0.00933425,
		  0.00811075,0.00670637,0.00431554,0.00120072,-.00283961,-.00772052,
		  -.01344394,-.01817991,-.01575024,0.01235058};
		for( i=4, _r = 0; i <= 15; i++ )
		{
			for( k=2; k <= 14; k++ )
			{
				Czero[k-1][i-1] = _itmp9[_r++];
				}
			}
		}

		/* the intermediate density for cutting the 
		 * interpolation.  Only used for ipHi=4,5,6,7,8. */
		{ static float _itmp10[] = {9.0,8.0,8.0,7.0,7.0,6.0,6.0,6.0,
		  5.0,5.0,5.0,5.0};
		for( k=4, _r = 0; k <= 15; k++ )
		{
			DenCut[k-1] = _itmp10[_r++];
			}
		}
		_aini = 0;
	}

#	ifdef DEBUG_FUN
	fputs( "<+>HydroBranch()\n", debug_fp );
#	endif

	/* this allows quantum levels to be in any order and converts to C scale */
	ipLo = MIN2( nLo , nHi ) - 1;
	ipHi = MAX2( nLo , nHi ) - 1;

	/* only works for Balmer and higher lines since HS tables only had this */
	ASSERT(ipLo >= 1 );
	/* only works for Hbeta and higher */
	ASSERT( ipHi >= 3 );
	/* only tabulated up to 15 */
	ASSERT( ipHi <= 14);

	/* check on charge */
	ASSERT( ipZPhys > 0 );
	/* we will work on c scale for part of this */
	nelem = ipZPhys - 1;

	RedMass = AtomcWgt.AtomicWeight[nelem]*1./
		(AtomcWgt.AtomicWeight[nelem] + 1.);

	z7 = powi((double)ipZPhys,7);

	DenScale = log10(sqrt(2.*RedMass)/z7);

	den = log10(phycon.eden) + DenScale;

	/* the fits only work down to den = log10( 0.001 ) - below that we are in ldl */
	den = MAX2( -2. , den );

	/* tt is used for the intermediate density fit
	 * at low densities the fit is good down to 100K
	 * at int. densities only 500K */
	t = phycon.te/POW2((double)ipZPhys);
	teuse = MAX2(100.,t);
	teuse = MIN2(3e4,teuse);
	t = log10(teuse);

	tt = phycon.te/POW2((double)ipZPhys);
	teuse = MAX2(500.,tt);
	teuse = MIN2(3e4,teuse);
	tt = log10(teuse);

	/* Compute the low and high density limits. */
	denlow = Alow[ipLo][ipHi] + Blow[ipLo][ipHi]*t + 
		Clow[ipLo][ipHi]* t*t;
	denhigh = Ahigh[ipLo][ipHi];

	/* Compute the really low den limit  */
	denzero = Azero[ipLo][ipHi] + Bzero[ipLo][ipHi]*t + 
	  Czero[ipLo][ipHi]*t*t;

	/* the branching ratio at intermediate density: */
	denint = Aint[ipLo][ipHi] + Bint[ipLo][ipHi]*tt + Cint[ipLo][ipHi]*
	  tt*tt;

	if( den >= (DenCut[ipHi] + 2.) )
	{
		/* if the den>11.0 (for ipHi=4) then use just the high limit... */
		answer = denhigh;
	}
	else if( den >= DenCut[ipHi] )
	{
		/* between limits interpolate */
		answer = denint + (denhigh - denint)*(den - DenCut[ipHi])/
		  2.0;
	}
	else if( den <= 4.0 )
	{
		/* if the den<4.0 then use the low limit... */
		if( ipZPhys != 1 && t <= 3.3 )
		{
			/* really low Te need an extra factor... */
			answer = denzero + (denlow - denzero)*(den - 2.0 - DenScale)/
			  (2.0 - DenScale);
			if( answer <= 0. )
			{
				fprintf(ioQQQ," low den limit ans %g %g %g %g %g\n",
					answer , denzero , denlow , den , DenScale);
				insane();
				answer = denzero;
			}
		}
		else
		{
			answer = denlow;
		}
	}
	else
	{
		/* else linerly interpolate between two limits... */
		answer = denlow + (denint - denlow)*(den - 4.0)/5.0;
	}

	/* the answer... */
	if( answer <= 0. )
	{
		fprintf(ioQQQ,
			" HydroBranch finds insane branching ratio, Z=%ld lo=%ld up=%ld val=%g\n" ,
			ipZPhys, ipLo+1, ipHi+1, answer);
		puts( "[Stop in HydroBranch]" );
		cdEXIT(EXIT_FAILURE);
	}

	branch_v = answer;

#	ifdef DEBUG_FUN
	fputs( " <->HydroBranch()\n", debug_fp );
#	endif
	return( branch_v );
}
