/*H21cm computes rate for H 21 cm from upper to lower excitation by atomic hydrogen */ 
/*h21_t_ge_20 computes rate for H 21 cm from upper to lower excitation by atomic hydrogen */ 
/*h21_t_lt_20 computes rate for H 21 cm from upper to lower excitation by atomic hydrogen */ 
/*H21cm_electron computes rate for H 21 cm from upper to lower excitation by electrons 
 * >>refer	H1	cs	Smith, F.J., 1966, Planet. Space Sci 14, 929 */
#include "cddefines.h"
#include "h21cm.h"

/*H21cm_electron computes rate for H 21 cm from upper to lower excitation by electrons 
 * >>refer	H1	cs	Smith, F.J., 1966, Planet. Space Sci 14, 929 */
double H21cm_electron( double temp )
{
	double hold;
	temp = MIN2(1e4 , temp );
	/* following fit is from */
	/* >>refer	H1	21cm	Liszt, H., 2001, A&A, 371, 698 */

	hold = -9.607 + log10( sqrt(temp)) * sexp( pow(log10(temp) , 4.5 ) / 1800. );
	hold = pow(10.,hold );
	return( hold );
}

/* computes rate for H 21 cm from upper to lower excitation by atomic hydrogen 
 * from 
 * >>refer	H1	cs	Allison, A.C., & Dalgarno A., 1969, ApJ 158, 423 */
/* the following is the best current survey of 21 cm excitation */
/* >>refer	H1	21cm	Liszt, H., 2001, A&A, 371, 698 */
static double h21_t_ge_20( double temp )
{
	double y;
	double x1,
		teorginal = temp;
	/* data go up to 1,000K must not go above this */
	temp = MIN2( 1000.,temp );
	x1 =1.0/sqrt(temp);
	y =-21.70880995483007-13.76259674006133*x1;
	y = exp(y);

	/* >>chng 02 feb 14, extrapolate above 1e3 K as per Liszt 2001 recommendation 
	 * page 699 of */
	/* >>refer	H1	21cm	Liszt, H., 2001, A&A, 371, 698 */
	if( teorginal > 1e3 )
	{
		y *= pow(teorginal/1e3 , 0.33 );
	}

	return( y );
}

/* this branch for T < 20K, data go down to 1 K */
static double h21_t_lt_20( double temp )
{
	double y;
	double x1;

	/* must not go below 1K */
	temp = MAX2(1., temp );
	x1 =temp*log(temp);
	y =9.720710314268267E-08+6.325515312006680E-08*x1;
	return(y*y);
}

double H21cm_H_atom( double temp )
{
	double hold ;
	if( temp >= 20. )
	{
		hold = h21_t_ge_20( temp );
	}
	else
	{
		hold = h21_t_lt_20( temp );
	}

	return hold;
}
