/* DynaEndIter called at end of iteration when advection is turned on */
/* DynaStartZone called at end of iteration when advection is turned on */
/* DynaIonize, called from ionize to evaluate advective terms for current conditions */
/* DynaChangeDensity, called from PressureChange to evaluate new density needed for
 * current conditions and wind solution, returns ratio of new to old density */
/* DynaZero zero some dynamics variables, called from zero.c */
/* DynaCreateArrays allocate some space needed to save the dynamics structure variables, 
 * called from DynaCreateArrays */
#include "cddefines.h"
#include "struc.h"
#include "radius.h"
#include "heat.h"
#include "converge.h"
#include "nomole.h"
#include "physok.h"
#include "abundances.h"
#include "pressure.h"
#include "ionrange.h"
#include "ionfracs.h"
#include "phycon.h"
#include "radacl.h"
#include "wind.h"
#include "dynamics.h"
static int ipUpstream=0,iphUpstream=0,ipyUpstream=0;

/* 
 * >>chng 01 mar 16, incorporate advection within dynamical solutions
 * this file contains the routines that incorporeate effects of dynamics and advection upon the
 * thermal and ionization solutions.  
 *
 * This code was originally developed in March 2001 during
 * a visit to UNAM Morelia, in collaboration with Robin Williams, Jane Arthur, and Will Henney.
 * Development was continued by email, and in a meeting in July/Aug 2001 at U Cardiff
 */

/* turn this on for dynamics printout */
#define DIAG_PRINT FALSE
#define MAINPRINT FALSE

/* the interpolated upstream densities of all ionization stages,
 * [element][ion] */
static double UpstreamIon[LIMELM][LIMELM+2];

/* the recombination term due to advection */
static double HRecom;

/* lookahead distance, as derived in DynaEndIter */
static double Dyn_dr;

/* advected work */
static double AdvecWork;

/* upstream HI density */
static double HAtom;

/* the upstream hydrogen atoms per unit vol*/
static double Dyn_hden;

/* HI ionization structure from previous iteration */
static float *Old_histr/*[NZLIM]*/ ,
	/* Lyman continuum optical depth from previous iteration */
	*Old_xLyman_depth/*[NZLIM]*/,
	/* depth of position in previous iteration */
	*Old_depth/*[NZLIM]*/,
	/* old n_p density from previous iteration */
	*Old_hiistr/*[NZLIM]*/,
	/* old pressure from previous iteration */
	*Old_pressure/*[NZLIM]*/,
	/* H density - particles per unit vol */
	*Old_hden/*[NZLIM]*/ ,
	/* density - total grams per unit vol */
	*Old_DenMass/*[NZLIM]*/ ,
	/* sum of enthalpy and kinetic energy per gram */
	*Dyn_work/*[NZLIM]*/,
	/* old electron density from previous iteration */
	*Old_ednstr/*[NZLIM]*/,
	/* sum of enthalpy and kinetic energy per gram */
	*Old_work/*[NZLIM]*/;
/* the ionization fractions from the previous iteration */
static float ***Old_xIonFracs;

/* the gas phase abundances from the previous iteration */
static float **Old_gas_phase;

/* the number of zones that were saved in the previous iteration */
static long int nOld_zone;

/* epsilon1 and epsilon2 give estimates of convergence: 
	 :: epsilon1 -- change between the last iterations;
	 :: epsilon2 -- error in the upstream interpolation.
	 When (and if) epsilon2 >> epsilon1, the interpolation length should be decreased.

	 They should both be based on the same norm of the models, but what norm may be
	 experimented with -- at present, it's H+/Htot just weighted by cell number, 
	 which makes the estimates sensitive to the structure of the primary ionization front.
*/
static double epsilon2=0;

/* ============================================================================== */
/* DynaChangeDensity, called from PressureChange to evaluate factor needed
 * to find new density needed for 
 * current conditions and wind solution, returns ratio of new to old density,
 * called when wind velocity is negative */

/* object is to get the local ram pressure
 * RamNeeded = pressure.PressureInit + pressure.PressureGas + pressure.PresInteg;
 * to equal the sum of the inital pressur at the illuminated face, the local gas pressure,
 * and the integrate radiative acceleration from the incident continuum 
 *
 * the local gas pressure is linear in density if the temperature is constant,
 *
 * the local ram pressure is inversely linear in density because of the relationship
 * between velocity and density introduced by the mass flux conservation 
 */
double DynaChangeDensity(void)
{
	double factor;
	static double rsave = -1.;
	static double vold = -1.;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaChangeDensity()\n", debug_fp );
#	endif

	/* following makes sure old wind v only updated once per zone,
	 * difference between old and new velocity is used to control zoning in nextdr */
	if( radius.depth != rsave )
	{
		rsave = radius.depth;
		/* evaluate radiative acceleration */
		radacl();

		/* remember the current wind velocity, only update once per zone since
		 * used to get change in velocity across zone, to adjust dr, below */
		vold = wind.windv;
	}

	/* update the current desired pressure */
	pressure.PressureCorrect = pressure.PressureInit + pressure.PresInteg;

	if( pressure.PressureGas > pressure.PressureRam )
	{
		/* when gas pressure dominated need to increase density to increase pressure */
		factor = pressure.PressureCorrect / pressure.PressureCurrent;
	}
	else
	{
		/* when ram pressure dominated need to decrease density to increase pressure */
		factor = pressure.PressureCurrent / pressure.PressureCorrect;
	}

	/* update the wind velocity from mass conservation */
	/* Robin or Will, this is the updated wind velocity but does not include factor,
	 * the change in the density.  Should it? */
	wind.windv = wind.windv0*struc.DenMass[0]/(phycon.xMassDensity);

	{
		/*@-redef@*/
		enum{DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{
			fprintf(ioQQQ,
				"nz\t%li\tfac\t%.3f\tPcor\t%.2e\tPcrnt\t%.2e\tPerr\t%.2e\tInteg\t%.2e\n",
				nzone,
				factor,
				pressure.PressureCorrect , 
				pressure.PressureCurrent, 
				(pressure.PressureCorrect-pressure.PressureCurrent)*100./pressure.PressureCorrect,
				pressure.PresInteg
				);
		}
	}

	if( DIAG_PRINT )
	{
		fprintf(ioQQQ,
			"vold %.2e vnew %.2e\n", vold , wind.windv );
	}

#	ifdef DEBUG_FUN
	fputs( " <->DynaChangeDensity()\n", debug_fp );
#	endif
	return factor ;
}

/* ============================================================================== */
/* DynaIonize, called from ionize to evaluate advective terms for current conditions,
 * calculates terms to add to ionization balance equation */
void DynaIonize(void)
{
	long int nelem, ion;
	double  timestep, velocity;
	double work , EnergyInternal;
	/*static double SaveHden;*/
	static long nzoneUsed = -1;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaIonize()\n", debug_fp );
#	endif

	/* >>chng 01 aug 02, move up to here since always must be done
	n = MIN2(nzone, NZLIM-1); */

	/* true, both advection and wind solution */
	velocity = wind.windv;
	timestep = -Dyn_dr/velocity;

	/* >>chng 01 aug 02, advection is now always wind */
#	if 0
	if (lgAdvecWind ) 
	{
	}
	else 
	{
		/* false, only advection, no dynamical wind soln */
		timestep = dynamics.DynTimestep;
		velocity = dynamics.Velocity;
	} 
#	endif

	/* >>chng 01 apr 30, moved work here from radinc. */
	/* this is the sum of the enthalpy and the kinetic energy per cm^3 */
	EnergyInternal = phycon.EnergyExcitation + phycon.EnergyIonization + phycon.EnergyBinding;
	work = 	EnergyInternal +
		0.5*POW2(velocity)*phycon.xMassDensity +	/* KE */
		5./2.*pressure.PressureGas;						/* thermal plus PdV work */

	ASSERT(nzone<struc.nzlim );
	Dyn_work[nzone] = (float)work;

	/* do nothing on first iteration */
	if( iteration < 2 )
	{
		/* first iteration, return zero */
		HRecom = 0. ;
		dynamics.CoolHeat = 0.;
		dynamics.dCoolHeatdT = 0.;
		dynamics.Photo = 0.;
		
		for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
		{
			for( ion=0; ion<nelem+1 ; ++ion )
			{
				dynamics.Recomb[nelem][ion] = 0.;
			}
		}

#		ifdef DEBUG_FUN
		fputs( " <->DynaIonize()\n", debug_fp );
#		endif
		return;
	}

	if( MAINPRINT )
	{
		fprintf(ioQQQ,"workwork\t%li\t%.3e\t%.3e\t%.3e\t%.3e\n",
			nzone,
			work,
			EnergyInternal ,
			0.5*POW2(velocity)*phycon.xMassDensity ,
			5./2.*pressure.PressureGas
			); /**/
	}

	/* net cooling due to advection */
	/* >>chng 01 aug 01, removed hden from dilution, put back into here */
	/* >>chng 01 sep 15, do not update this variable the very first time this routine
	 * is called at the new zone. */
	if( nzone==nzoneUsed )
	{
		dynamics.CoolHeat =
			(work - AdvecWork*phycon.hden)/timestep;
		nzoneUsed = nzone;
	}
	
#ifdef FOO
	dynamics.CoolHeat =	dynamics.dCoolHeatdT = 0;broken();
#endif

	if( MAINPRINT )
	{
		fprintf(ioQQQ,
			"dynamics cool-heat\t%li\t%.3e\t%.3e\t%.3e\t%.3e\t %.3e\t%.3e\t%.3e\t%.3e\t%.3e\n",
			nzone,
			phycon.te, 
			dynamics.CoolHeat,
			heat.htot,
			work/timestep,
			AdvecWork*phycon.hden/timestep,
			work,
			AdvecWork*phycon.hden,
			phycon.hden,
			timestep);
	}

	/* temp deriv of cooling minus heating */
	dynamics.dCoolHeatdT = 5./2.*pressure.PressureGas/phycon.te/timestep;

	/* second or greater iteration, have advective terms */
	/* this will evaluate advective terms for current physical conditions */
	ASSERT( HAtom > 0.);

	/* the rate (s^-1) atoms drift in from upstream, a source term for the ground */

	/* dynamics.Hatom/timestep is the source (cm^-3 s^-1) of neutrals,
		 normalized to (s^-1) by the next higher ionization state as for all 
		 other recombination terms */

	/* >>01 aug 01, add hden here, removed from diluation above */
	HRecom = 
		HAtom / timestep/xIonFracs[ipHYDROGEN][1]*phycon.hden;

	/* dynamics.xIonFracs[ipHYDROGEN][1]/timestep is the sink (cm^-3 s^-1) of 
		 ions, normalized to (s^-1) by the ionization state as for all other
		 ionization terms */

	dynamics.Photo = 1./ timestep;

	for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
	{
		for( ion=IonRange.IonLow[nelem]; ion<IonRange.IonHigh[nelem]; ++ion )
		{
			ASSERT( xIonFracs[nelem][ion+1] > 0. );
			/* >>chng 01 aug 01, removed hden from dilution, put back into here */
			dynamics.Recomb[nelem][ion] = 
				/* the +1 in xIonFracs is due to offset from 0, which has abundance */
				UpstreamIon[nelem][ion]*phycon.hden/ timestep/xIonFracs[nelem][ion+1];
		}
	}

	ASSERT( fabs(HRecom - dynamics.Recomb[ipHYDROGEN][0])/HRecom < 0.001 );

	if( MAINPRINT )
	{
		fprintf(ioQQQ,"    DynaIonize, %4li photo=%.2e , H recom= %.2e \n",
			nzone,dynamics.Photo , dynamics.Recomb[0][0]  );
	}
	
	/*phycon.hden = SaveHden;*/

#	ifdef DEBUG_FUN
	fputs( " <->DynaIonize()\n", debug_fp );
#	endif
	return;
}

/* ============================================================================== */
/* DynaStartZone called at start of zone calculation when advection is turned on */
void DynaStartZone(void)
{
	/* this routine is called at the start of a zone calculation, by StartZone:
	 *
	 * it sets deduced variables to zero if first iteration,
	 *
	 * if upstream depth is is outside the computed structure on previous iteration,
	 * return value at shielded face 
	 *
	 * Also calculates epsilon2, an estimate of the accuracy of the source terms.
	 *
	 * */

	/* this is index of upstream cell in structure stored from previous iteration */
	double upstream, /*timestep, velocity,*/ dilution, nextfrac;

	/* Properties for cell half as far upstream, used to converge timestep */
	double hupstream, hnextfrac, hhistr, hhden;

	/* Properties for cell at present depth, used to converge timestep */
	double ynextfrac, yhistr, yhden;

	long int nelem , ion;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaStartZone()\n", debug_fp );
#	endif

	/* do nothing on first iteration */
	if( iteration < 2 )
	{
		Dyn_hden = 0.;
		HAtom = 0.;
		AdvecWork = 0.;
		for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
		{
			for( ion=0; ion<nelem+1 ; ++ion )
			{
				UpstreamIon[nelem][ion] = 0.;
			}
		}
		ipUpstream = 0;
		iphUpstream = 0;
		ipyUpstream = 0;

#		ifdef DEBUG_FUN
		fputs( " <->DynaStartZone()\n", debug_fp );
#		endif
		return;
	}

	/* radius.depth is distance from illuminated face of cloud to outer edge of
	 * current zone, which has thickness radius.drad */

	/* find where the down stream point is, in previous iteration,
	 * NB neg sign since velocity is negative, we are looking for point
	 * where current gas cell was, in previous iteration */

	/* true, both advection and wind solution */
	/*velocity = wind.windv;
	timestep = -Dyn_dr/velocity;*/
	upstream = radius.depth + Dyn_dr;
	hupstream = radius.depth + 0.5*Dyn_dr;

	/* >>chng 01 aug 02, advection is always wind */
#	if 0
	if (lgAdvecWind ) 
	{
	}
	else 
	{
		/* false, only advection, no dynamical wind soln */
		timestep = dynamics.DynTimestep;
		velocity = dynamics.Velocity;
		upstream = radius.depth - velocity*timestep;
		hupstream = radius.depth - 0.5*velocity*timestep;
	} 
#	endif

	/* now locate upstream point in previous stored structure,
	 * will be at the same point or higher than we found previously */
	while( (Old_depth[ipUpstream+1] < upstream ) && 
		( ipUpstream < nOld_zone-1 ) )
	{
		++ipUpstream;
	}
	ASSERT( ipUpstream <= nOld_zone-1 );
	
	/* above loop will give ipUpstream == nOld_zone-1 if computed structure has been overrun */

	if (ipUpstream != nOld_zone-1)
	{
		nextfrac = ( upstream - Old_depth[ipUpstream])/
			(Old_depth[ipUpstream+1] - Old_depth[ipUpstream]);
		Dyn_hden = Old_hden[ipUpstream] +
			(Old_hden[ipUpstream+1] - Old_hden[ipUpstream])*
			nextfrac;

		/* fractional changes in density from passive advection */
		/* >>chng 02 May 2001 rjrw: use hden for dilution */
		/* >>chng 01 aug 01, remove hden here, put back into vars when used in DynaIonize */
		dilution = /*phycon.hden*/1./Dyn_hden;
		/*OldHden = phycon.hden;*/
		HAtom = (Old_histr[ipUpstream] +
			(Old_histr[ipUpstream+1] - Old_histr[ipUpstream])*
			nextfrac)*dilution;
		/* the advected work */
		AdvecWork = (Old_work[ipUpstream] +
			(Old_work[ipUpstream+1] - Old_work[ipUpstream])*
			nextfrac)*dilution;
		for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
		{
			for( ion=0; ion<nelem+2 ; ++ion )
			{
				UpstreamIon[nelem][ion] = 
					(Old_xIonFracs[ipUpstream][nelem][ion] +
					(Old_xIonFracs[ipUpstream+1][nelem][ion] - 
					Old_xIonFracs[ipUpstream][nelem][ion])*
					nextfrac)*dilution;
			}
		}
	}
	else
	{
		Dyn_hden = Old_hden[ipUpstream];
		/* fractional changes in density from passive advection */
		/* >>chng 01 aug 01, remove hden here, put back into vars when used in DynaIonize */
		dilution = /*phycon.hden*/1./Dyn_hden;
		/*OldHden = phycon.hden;*/
		HAtom = Old_histr[ipUpstream]*dilution;
		AdvecWork = Old_work[ipUpstream]*dilution;
		for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
		{
			for( ion=0; ion<nelem+2 ; ++ion )
			{
				UpstreamIon[nelem][ion] = 
					Old_xIonFracs[ipUpstream][nelem][ion]*dilution;
			}
		}
	}
	ASSERT( fabs(HAtom - UpstreamIon[ipHYDROGEN][0])/HAtom < 0.001 );

	/* Repeat enough of the above for half-step and no-step to judge convergence:
		 the result of this code is the increment of epsilon2 */

	while( (Old_depth[iphUpstream+1] < hupstream ) && 
		( iphUpstream < nOld_zone-1 ) )
	{
		++iphUpstream;
	}
	ASSERT( iphUpstream <= nOld_zone-1 );
	if (iphUpstream != nOld_zone-1)
	{
		hnextfrac = ( hupstream - Old_depth[iphUpstream])/
			(Old_depth[iphUpstream+1] - Old_depth[iphUpstream]);
		hhden = Old_hden[iphUpstream] +
			(Old_hden[iphUpstream+1] - Old_hden[iphUpstream])*
			hnextfrac;
		hhistr = (Old_histr[iphUpstream] +
			(Old_histr[iphUpstream+1] - Old_histr[iphUpstream])*
			hnextfrac);
	}
	else
	{
		hhden = Old_hden[iphUpstream];
		hhistr = Old_histr[iphUpstream];
	}

	while( (Old_depth[ipyUpstream+1] < radius.depth ) && 
		( ipyUpstream < nOld_zone-1 ) )
	{
		++ipyUpstream;
	}
	ASSERT( ipyUpstream <= nOld_zone-1 );
	if (ipyUpstream != nOld_zone-1)
	{
		ynextfrac = ( radius.depth - Old_depth[ipyUpstream])/
			(Old_depth[ipyUpstream+1] - Old_depth[ipyUpstream]);
		yhden = Old_hden[ipyUpstream] +
			(Old_hden[ipyUpstream+1] - Old_hden[ipyUpstream])*
			ynextfrac;
		yhistr = (Old_histr[ipyUpstream] +
			(Old_histr[ipyUpstream+1] - Old_histr[ipyUpstream])*
			ynextfrac);
	}
	else
	{
		yhden = Old_hden[ipyUpstream];
		yhistr = Old_histr[ipyUpstream];
	}
	
	/* Must be consistent with epsilon1 below */
	/* >>chngf 01 aug 01, remove phycon.hden from HAtom */
	/* >>chngf 02 aug 01, multiply by cell width */
	epsilon2 += POW2(yhistr/yhden+hhistr/hhden-2.*HAtom) /* *radius.drad */; 

	if( MAINPRINT )
	{
		fprintf(ioQQQ," DynaStartZone, %4li photo=%.2e , H recom= %.2e dil %.2e \n",
			nzone,dynamics.Photo , dynamics.Recomb[0][0] , dilution*phycon.hden );
	}

#	ifdef DEBUG_FUN
	fputs( " <->DynaStartZone()\n", debug_fp );
#	endif
	return;
}

/* ============================================================================== */
/* DynaEndIter called at end of iteration when advection is turned on */
void DynaEndIter(void)
{
	/* this routine is called by RestartIter at the end of an iteration 
	 * when advection is turned on.  currently it only derives a 
	 * timestep by looking at the spatial derivative of
	 * some stored quantities */

	/* largest desired fractional change in quantities */
	const double FRAC = 0.1;
	long i;
	double BigDeriv,
		BigRate,
		rec_time;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaEndIter()\n", debug_fp );
#	endif

	/* This routine is only called if advection is turned on at the end of
	 * each iteration.  The test 
	 * is to check whether wind velocity is also set by dynamics code */
	/* >>chng 01 aug 02, advection is always wind */
#	if 0
	if (strcmp(pressure.chCPres,"WIND") == 0)
	{
		lgAdvecWind = TRUE;
	}
	else
	{
		/* advection only, no wind solution */
		lgAdvecWind = FALSE;
	}
#	endif

	if (iteration == 2)
	{
		/* look for largest change in proton density, wrt depth, to
		 * get limit to timestep */
		BigDeriv = 0.;
		BigRate = 0.;
		for( i=0; i<nzone-1; ++i )
		{
			double val = fabs(struc.hiistr[i+1]-struc.hiistr[i])/MAX2(SMALLFLOAT,struc.hiistr[i])/
				MAX2(SMALLFLOAT , (struc.depth[i+1]-struc.depth[i]) ) ;
			BigDeriv = MAX2( BigDeriv , val );

			/* >>chng 01 aug 02, from dynamics.Velocity to windv */
			BigRate  = MAX2( BigRate , val * -wind.windv );

			/* >>chng 01 aug 02, advection is always wind */
#			if 0
			if (lgAdvecWind)
			{
			}
			else
			{
				BigRate  = MAX2( BigRate , val * -dynamics.Velocity );
			}
#			endif
		}
		
		/* the physical thickness (cm) over which quantity varied by this amount */
		if( BigDeriv < SMALLFLOAT )
		{
			Dyn_dr = 0.;
		}
		else
		{
			Dyn_dr = FRAC / BigDeriv;
		}

		/* the physical timescale (s) over which quantity varied by this amount */
		if( BigRate < SMALLFLOAT )
		{
			dynamics.DynTimestep = 0.;
		}
		else
		{
			dynamics.DynTimestep = FRAC / BigRate;
		}
		
		/* recombination timescale for first zone */
		rec_time = 1./(struc.ednstr[0]*2.90e-10*pow(struc.testr[0],-0.77f));
		
		/* hack in rec timescale for now */
		dynamics.DynTimestep = rec_time;
		Dyn_dr = dynamics.DynTimestep * -wind.windv0 ;
		
		if( MAINPRINT )
		{
			fprintf(ioQQQ," DynaEndIter, BigDeriv=%.2e rec_time=%.2e\n",
				BigDeriv , rec_time );
			fprintf(ioQQQ," DynaEndIter, dr=%.2e timestep=%.2e\n",
				Dyn_dr , dynamics.DynTimestep);
		}
	}
	else
	{
		DynaNewStep();
	}

	ASSERT( Dyn_dr > 0. );
	ASSERT( dynamics.DynTimestep > 0. );

	/* reset the upstream counters */
	ipUpstream = iphUpstream = ipyUpstream = 0;
	epsilon2 = 0.;

	DynaSaveLast();

#	ifdef DEBUG_FUN
	fputs( " <->DynaEndIter()\n", debug_fp );
#	endif
	return;
}

void DynaNewStep(void)
{
	long int ilast = 0,
		i;

	double nextfrac,
		Oldi_hden,
		Oldi_histr,
		epsilon1;

#	ifdef DEBUG_FUN
	fputs( " <->DynaNewStep()\n", debug_fp );
#	endif

	/*n = MIN2(nzone, NZLIM-1);*/
	epsilon1 = 0;

	ASSERT( nzone < struc.nzlim);
	for (i=0;i<nzone;++i) 
	{
		/* Interpolate for present position in previous solution */
		while( (Old_depth[ilast] < struc.depth[i] ) && 
			( ilast < nOld_zone-1 ) )
		{
			++ilast;
		}
		ASSERT( ilast <= nOld_zone-1 );
			
		if (ilast != nOld_zone-1)
		{
			nextfrac = ( struc.depth[i] - Old_depth[ilast])/
				(Old_depth[ilast+1] - Old_depth[ilast]);
			Oldi_hden = Old_hden[ilast] +
				(Old_hden[ilast+1] - Old_hden[ilast])*
				nextfrac;
			Oldi_histr = (Old_histr[ilast] +
				(Old_histr[ilast+1] - Old_histr[ilast])*
				nextfrac);
		}
		else
		{
			Oldi_hden = Old_hden[ilast];
			Oldi_histr = Old_histr[ilast];
		}
		/* Must be consistent with epsilon2 above */
		/* >>chngf 02 aug 01, multiply by cell width */
		epsilon1 += POW2(Oldi_histr/Oldi_hden-struc.histr[i]/struc.hden[i]) /* *struc.dr[i] */ ;  
	}

	/* epsilon1 is an estimate of the convergence of the solution from its change during the last iteration,
		 epsilon2 is an estimate of the accuracy of the advective terms, calculated in DynaStartZone above:
		 if dominant error is from the advective terms, need to make them more accurate.
	*/
	if ( epsilon1 < 0.1*epsilon2 ) 
	{
		dynamics.DynTimestep /= 1.5;
		Dyn_dr       /= 1.5;
	}
	
	if( MAINPRINT )
	{
		fprintf(ioQQQ," DynaNewStep, eps1=%.2e eps2=%.2e => new dr=%.2e timestep=%.2e\n",
			epsilon1 , epsilon2 , Dyn_dr , dynamics.DynTimestep );
	}

#	ifdef DEBUG_FUN
	fputs( " <->DynaNewStep()\n", debug_fp );
#	endif
	return;
}


void DynaSaveLast(void)
{
	long int i,
		ion,
		nelem;

#	ifdef DEBUG_FUN
	fputs( " <->DynaSaveLast()\n", debug_fp );
#	endif

	/*n = MIN2(nzone, NZLIM-1);*/

	/* Save results from previous iteration */
	nOld_zone = nzone;
	ASSERT( nzone < struc.nzlim );
	for( i=0; i<nzone ; ++i )
	{
		Old_histr[i] = struc.histr[i];
		Old_depth[i] = struc.depth[i];
		Old_xLyman_depth[i] = struc.xLyman_depth[i];
		/* old n_p density from previous iteration */
		Old_hiistr[i] = struc.hiistr[i];
		/* old pressure from previous iteration */
		Old_pressure[i] = struc.pressure[i];
		/* old electron density from previous iteration */
		Old_ednstr[i] = struc.ednstr[i];
		/* energy term */
		Old_work[i] = Dyn_work[i];
		/* >>chng 02 May 2001 rjrw: add hden for dilution */
		Old_hden[i] = struc.hden[i];
		Old_DenMass[i] = struc.DenMass[i];
		for( nelem=ipHYDROGEN; nelem<LIMELM; ++nelem)
		{
			Old_gas_phase[i][nelem] = abundances.gas_phase[nelem];
			for( ion=0; ion<nelem+2 ; ++ion )
			{
				Old_xIonFracs[i][nelem][ion] = struc.xIonFracs[i][nelem][ion];
			}
		}
	}
#	ifdef DEBUG_FUN
	fputs( " <->DynaSaveLast()\n", debug_fp );
#	endif
	return;
}

/* ============================================================================== */
/*DynaZero zero some dynamics variables, called from zero.c */
void DynaZero( void )
{

#	ifdef DEBUG_FUN
	fputs( "<+>DynaZero()\n", debug_fp );
#	endif

	/* the number of zones in the previous iteration */
	nOld_zone = 0;

	/* by default advection is turned off */
	dynamics.lgAdvection = FALSE;
	/*dynamics.Velocity = 0.;*/
	HRecom = 0.;
	dynamics.Photo = 0.;
	AdvecWork = 0.;
	dynamics.CoolHeat = 0.;
	dynamics.dCoolHeatdT = 0.;
	dynamics.HeatMax = 0.;
	dynamics.CoolMax = 0.;

#	ifdef DEBUG_FUN
	fputs( " <->DynaZero()\n", debug_fp );
#	endif
	return;
}


/* ============================================================================== */
/* DynaCreateArrays allocate some space needed to save the dynamics structure variables, 
 * called from DynaCreateArrays */
void DynaCreateArrays( void )
{
	long int nelem,
		ns,
		i,j;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaCreateArrays()\n", debug_fp );
#	endif

	if( (Old_work = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_ednstr = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Dyn_work = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_DenMass = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_hden = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_pressure = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_histr = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_hiistr = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_depth = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_xLyman_depth = ((float*)MALLOC( (size_t)(struc.nzlim)*sizeof(float )))) == NULL )
		bad_malloc();

	if( (Old_xIonFracs = (float ***)MALLOC(sizeof(float **)*(unsigned)(struc.nzlim) ) )==NULL )
		bad_malloc();

	if( (Old_gas_phase = (float **)MALLOC(sizeof(float **)*(unsigned)(struc.nzlim) ) )==NULL )
		bad_malloc();

	/* now create diagonal of space for ionization arrays */
	for( ns=0; ns < struc.nzlim; ++ns )
	{
		if( (Old_xIonFracs[ns] = 
			(float**)MALLOC(sizeof(float*)*(unsigned)(LIMELM+3) ))==NULL )
			bad_malloc();

		if( (Old_gas_phase[ns] = 
			(float*)MALLOC(sizeof(float*)*(unsigned)(LIMELM+3) ))==NULL )
			bad_malloc();

		for( nelem=0; nelem< (LIMELM+3) ;++nelem )
		{
			if( (Old_xIonFracs[ns][nelem] = 
				(float*)MALLOC(sizeof(float)*(unsigned)(LIMELM+1) ))==NULL )
				bad_malloc();
		}
	}

	for( i=0; i < struc.nzlim; i++ )
	{
		/* these are values if H0 and tau_912 from previous iteration */
		Old_histr[i] = 0.;
		Old_xLyman_depth[i] = 0.;
		Old_depth[i] = 0.;
		/* old n_p density from previous iteration */
		Old_hiistr[i] = 0.;
		/* old pressure from previous iteration */
		Old_pressure[i] = 0.;
		/* old electron density from previous iteration */
		Old_ednstr[i] = 0.;
	}

	for( i=0; i<LIMELM+2; ++i )
	{
		for( j=0; j<LIMELM; ++j )
		{
			dynamics.Recomb[j][i] = 0.;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->DynaCreateArrays()\n", debug_fp );
#	endif
	return;
}

/* ============================================================================== */
/* ParseDynaWind parse the wind command, called from ParseCommands */
void ParseDynaWind( char *chCard )
{
	long int i;
	int lgEOL;

#	ifdef DEBUG_FUN
	fputs( "<+>DynaCreateArrays()\n", debug_fp );
#	endif

	/* wind structure, parameters are initial velocity and optional mass
	 *  v in km/sec, mass in solar masses */
	i = 5;
	wind.windv0 = (float)(FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL)*1e5);
	if( lgEOL ) 
	{
		NoNumb(chCard);
	}

	/* wind d is flow from blister */
	if( wind.windv0 < 0. )
	{
		/* gravity is not included in hydro flow*/
		wind.comass = 0.;

		/* option to include advection */
		if( lgMatch( "ADVE" , chCard ) )
		{
			/* turn on advection */
			dynamics.lgAdvection = TRUE;

			/* Set to default mode, no wind, only advection (constant velocity) 
			lgAdvecWind = FALSE;*/

			/* turn off prediction of next zone's temperature, as guessed in StartZone */
			phycon.lgPredNextTe = FALSE;

			/* turn off both CO and H2 networks since advection not included, also say physicsl
			 * conditions are not ok*/
			nomole.lgNoH2Mole = TRUE;
			nomole.lgNoCOMole = TRUE;
			physok.lgPhysOK = FALSE;

			/* use the new temperature solver */
			strcpy( conv.chSolverEden , "new" );

			/* increse precision of solution */
			phycon.EdenError = 1e-3f;
		}
	}

	/* this is usual hypersonic outflow */
	else
	{
		if( wind.windv0 <= 1.e6 )
		{
			/* speed of sound roughly 10 km/s */
			fprintf( ioQQQ, " >>>>Initial wind velocity should be greater than speed of sound; calculation only valid above sonic point.\n" );
			wind.lgWindOK = FALSE;
		}

		/* set the central object mass, in solar masses */
		wind.comass = (float)FFmtRead(chCard,&i,INPUT_LINE_LENGTH,&lgEOL);

		if( lgEOL )
		{
			wind.comass = 1.;
		}

	}

	wind.windv = wind.windv0;

	/* this is used in convpres to say wind solution - both cases use this*/
	strcpy( pressure.chCPres, "WIND" );

	/*  option to turn off continuum radiatrive acceleration */
	if( lgMatch("O CO",chCard) )
	{
		pressure.lgConPres = FALSE;
	}
	else
	{
		pressure.lgConPres = TRUE;
	}

#	ifdef DEBUG_FUN
	fputs( " <->DynaCreateArrays()\n", debug_fp );
#	endif
	return;
}
