/*CoolSulf compute sulphur cooling */
/*S2cs compute [sii] collision strengths
 * compute collision strengths for [SII] transitions 
 * w/in S II ground term. From 
 *>>refer	s2	cs	Ramsbottom, C.A., Bell, K.L., Stafford, R.P. 1996, At. Data Nucl. Data Tables, 63, 57 */
#include "cddefines.h"
#include "coolheavy.h"
#include "sii.h"
#include "taulines.h"
#include "phycon.h"
#include "embesq.h"
#include "poplevls.h"
#include "ionfracs.h"
#include "ligbar.h"
#include "cooling.h"
#include "level2.h"
#include "putcs.h"
#include "atomseqberyllium.h"
#include "level3.h"
#include "pop3.h"
#include "popexc.h"
#include "expion.h"
#include "atomseqboron.h"
#include "fivel.h"

/*S2cs compute [sii] collision strengths
 * compute collision strengths for [SII] transitions 
 * w/in S II ground term. From 
 *>>refer	s2	cs	Ramsbottom, C.A., Bell, K.L., Stafford, R.P. 1996, At. Data Nucl. Data Tables, 63, 57 */
static void S2cs(double telog, 
	  double *cs12, 
	  double *cs13, 
	  double *cs14, 
	  double *cs15, 
	  double *cs23, 
	  double *cs24, 
	  double *cs25, 
	  double *cs34, 
	  double *cs35, 
	  double *cs45);

void CoolSulf()
{
	double cs, 
	  cs2s2p, 
	  cs2s3p, 
	  cs01, 
	  cs02, 
	  cs12, 
	  cs13, 
	  cs14, 
	  cs15, 
	  cs23, 
	  cs24, 
	  cs25, 
	  cs34, 
	  cs35, 
	  cs45, 
	  p[5];
	float 
	  pop2;
	static double gS2[5]={4.,4.,6.,2.,4.};
	static double exS2[4]={14851.9,31.5,9640.8,48.6};
	

#	ifdef DEBUG_FUN
	fputs( "<+>CoolSulf()\n", debug_fp );
#	endif

	/* Sulphur II [S II] 6731 + 6716, A 
	 * >>refer	s2	as	trans Mendoza, C., & Zeippen, C.J., 1982, MNRAS, 198, 127
	 * collision strength from 
	 *>>refer	s2	cs Ramsbottom, C.A., Bell, K.L., Stafford, R.P. 1996,
	 *>>refercon	At. Data Nucl. Data Tables, 63, 57 
	 * this agrees very well with 
	 * >>refer	s2	cs	Tayal, S., 1997, ApJS, 111, 459
	 * */
	S2cs(phycon.alogte,&cs12,&cs13,&cs14,&cs15,&cs23,&cs24,&cs25,&cs34,
	  &cs35,&cs45);
	/* cs13 = MIN(4.95 , 3.2640828 + 1.9369099*sexp(te/30337.498) )
	 * cs12 = cs13 / 1.5
	 * cs15 = MIN( 1.59 , 0.10*te10*te10*te10/te01/te003/te001/te001)
	 * cs14 = cs15 * 0.5
	 * FIVEL( G(1-5) , ex(wn,1-5), cs12,cs13,14,15,23,24,25,34,35,45,
	 *  A21,31,41,51,32,42,52,43,53,54, pop(1-5), abund) */
	fivel(gS2,exS2,cs12,cs13,cs14,cs15,cs23,cs24,cs25,cs34,cs35,cs45,
	  8.82e-4,2.60e-4,9.06e-2,0.225,3.35e-7,0.163,0.133,0.078,0.179,
	  1.03e-6,p,xIonFracs[ipSULPHUR][1]);
	sii.S6733 = (float)(p[1]*8.82e-4*2.96e-12);
	sii.S6718 = (float)(p[2]*2.60e-4*2.962e-12);
	sii.S4070 = (float)(p[4]*0.225*4.89e-12);
	sii.S4078 = (float)(p[3]*0.0906*4.88e-12);
	sii.S10323 = (float)(p[4]*0.179*1.93e-12);
	sii.S10289 = (float)(p[4]*0.133*1.93e-12);
	sii.S10373 = (float)(p[3]*0.078*1.92e-12);
	sii.S10339 = (float)(p[3]*0.163*1.92e-12);
	CoolHeavy.c6731 = sii.S6733 + sii.S6718;
	CoolHeavy.c10330 = sii.S10323 + sii.S10289 + sii.S10373 + sii.S10339;
	cooling.dCooldT += CoolHeavy.c6731*(2.14e4*cooling.tsq1 - cooling.halfte);
	cooling.dCooldT += CoolHeavy.c10330*2.58*(3.53e4*cooling.tsq1 - cooling.halfte);
	coladd("S  2",6731,CoolHeavy.c6731);
	coladd("S  2",10330,CoolHeavy.c10330);
	coladd("S  2",4070,CoolHeavy.c10330*1.58);

	/* SII 1256 */
	cs = MIN2(8.46,-4.9416304+47.01064/phycon.alogte);
	cs = MAX2(4.466,cs);
	PutCS(cs,&TauLines[ipT1256]);
	level2(&TauLines[ipT1256]);

	/* S III 18.7M, 33.6M, A 
	 * >>refer	s3	as	Mendoza, C. 1982, in Planetary Nebulae, IAU Symp No. 103,
	 * >>refercon	ed by D.R. Flower, (D. Reidel: Holland), 143
	 * 
	 * >>refer	s3	cs	Galavis, M.E., Mendoza, C., & Zeippen, C.J. 1995, A&AS, 111, 347 
	 * >>chng 99 dec 22, cs changed from above to
	 * >>refer	s3	cs	Tayal, S.S., and Gupta, G.P. 1999 ApJ 526, 544 */
	/* the 1-2 transition */
	if( phycon.te < 5000. )
	{
		cs12 = 4.44;
		cs13 = 1.41;
		cs14 = 0.802;
		cs15 = 0.129;
		cs23 = 8.72;
		cs24 = 2.41;
		cs25 = 0.388;
		cs34 = 4.01;
		cs35 = 0.646;
		cs45 = 1.31;
	}
	else if( phycon.te > 1e5 )
	{
		cs12 = 1.9;
		cs13 = 1.24;
		cs14 = 0.664;
		cs15 = 0.136;
		cs23 = 5.13;
		cs24 = 1.99;
		cs25 = 0.407;
		cs34 = 3.32;
		cs35 = 0.679;
		cs45 = 1.84;
	}
	else
	{
		cs12 = 52.47/(phycon.te30/phycon.te02);
		cs13 = 1.894/(phycon.te02*phycon.te02);
		cs14 = 1.34/(phycon.te05*phycon.te01);
		cs15 = 0.109*phycon.te02;
		cs23 = 41.3/(phycon.te20/phycon.te02);
		cs24 = 4.03/(phycon.te05*phycon.te01);
		cs25 = 0.327*phycon.te02;
		cs34 = 6.708/(phycon.te05*phycon.te01);
		cs35 = 0.545*phycon.te02;
		cs45 = 0.501*phycon.te10*phycon.te01;
	}
	/*cs = MIN2(2.331,7.935*phycon.te/(phycon.te10*phycon.te03*phycon.te003));*/
	PutCS(cs12,&TauLines[ipTS34]);

	/* the 2-3 transition 
	if( phycon.te <= 39811. )
	{
		cs = MIN2(5.78,3.114*phycon.te03*phycon.te03);
	}
	else
	{
		cs = 24.93/(phycon.te10*phycon.te03*phycon.te01/phycon.te001/
		  phycon.te001);
	}*/
	PutCS(cs23,&TauLines[ipTS19]);
	/* the 1-3 transition 
	cs = MIN2(1.413,0.221*phycon.te*phycon.te20/phycon.te03*phycon.te005);*/
	PutCS(cs13,&TauDummy);
	level3(&TauLines[ipTS34],&TauLines[ipTS19],&TauDummy);

	/* S III O III-like lines, A from 
	 * >>refer	s3	as	Mendoza, C., & Zeippen, C.J. 1982, MNRAS, 199, 1025
	 * CS from 
	 * >>refer	s3	cs	Galavis, M.E., Mendoza, C., & Zeippen, C.J. 1995, A&AS, 111, 347
	 * >>chng 00 Sep 11, cs changed from above to
	 * >>refer	s3	cs	Tayal, S.S., and Gupta, G.P. 1999 ApJ 526, 544 */
	/*cs = MIN2(2.05,0.0821*phycon.te30);*/
	/* POP3(G1,G2,G3,O12,O13,O23,A21,A31,A32,E12,E23,P2,ABUND,GAM2) */
	/*CoolHeavy.c6312 = pop3(9.,5.,1.,7.98,1.14,cs,7.97e-2,0.807,2.22,*/
	CoolHeavy.c6312 = pop3(9.,5.,1.,cs14+cs24+cs34,cs15+cs25+cs35,cs45,7.97e-2,0.807,2.22,
	  1.55e4,2.28e4,&pop2,xIonFracs[ipSULPHUR][2],0.)*2.22*3.15e-12;
	/* folowing is 9532 + 9069 together (OIII-like) */
	CoolHeavy.c9532 = pop2*7.93e-2*2.11e-12;
	coladd("S  3",6312,CoolHeavy.c6312);
	coladd("S  3",9532,CoolHeavy.c9532);
	coladd("S  3",3722,CoolHeavy.c6312*0.59);

	/* S III 1194, data from 
	 * >>refer	s3	cs	Ho, Y.K., & Henry,R.J.W. 1984, ApJ, 282, 816
	 * >>chng 97 may 17, to, about 2x larger than above
	 * >>refer	s3	cs	Tayal, S.S. 1997, ApJ 481, 550 */
	if( phycon.te <= 3e4 )
	{
		cs = 12.04/(phycon.te02*phycon.te02);
	}
	else if( phycon.te > 3e4 && phycon.te <= 4e4 )
	{
		cs = 7.97;
	}
	else
	{
		cs = 55.42/(phycon.te20/phycon.te02*phycon.te003);
	}
	PutCS(cs,&TauLines[ipT1194]);
	level2(&TauLines[ipT1194]);

	/* S III] 1713.12, 1728.94, cs from 
	 * >>refer	s3	cs	Hayes, M.A., 1986, J Phys B 19, 1853.
	 * cs = MIN( 4.0 , 7.794 / (te10/te02/te001/te001)  )
	 * >>chng 97 may 17, about 20% smaller than before
	 * >>refer	s3	cs	Tayal, S.S. 1997, ApJ 481, 550 */
	if( phycon.te <= 3e4 )
	{
		cs = 1.786*phycon.te05*phycon.te01*phycon.te001;
	}
	else
	{
		cs = 9.392/phycon.te10;
	}
	PutCS(cs,&TauLines[ipTS1720]);
	level2(&TauLines[ipTS1720]);

	/* S IV 10.5MI, 
	 * >>refer	s4	as	Johnson, C.T., Kingston, A.E., Dufton, P.L. 1986, 220, 155 
	 * >>referold	s4	cs	Johnson, C.T., Kingston, A.E., Dufton, P.L. 1986, MNRAS, 220, 155
	 * >>chng 97 feb 14, error in cs below t = 10,000K
	 * >>chng 96 dec 19, to CS from 
	 * >>referold	s4	cs	Saraph, H.E., Storey, P.J., & Tully, J.A. 1995, 5th International
	 * >>referoldcon	Colloquium on Atomic Spectra and Oscillator Strengths, ed. by
	 * >>referoldcon	W.-U L.  Tchang-Brillet, J.-F. Wyart, C.J. Zeippen,
	 * >>referoldcon	(Meudon: Publications de l'Observaroire de Paris), p.110
	 * above said to be A&A in press */
	/* >>refer	s4	cs	Tayal, S.S., 2000, ApJ, 530, 1091 
	 */
	if( phycon.te < 1e4 )
	{
		cs = 3.71*phycon.te10/phycon.te01;
	}
	else
	{
		cs = MIN2(8.5,19.472/(phycon.te10/phycon.te01));
	}
	PutCS(cs,&TauLines[ipTS11]);
	/*level2(&TauLines[ipTS11]);*/
	/*AtomSeqBoron compute cooling from 5-level boron sequence model atom */
	/* >>refer	s4	cs	Tayal, S.S., 2000, ApJ 530, 1091*/
	AtomSeqBoron(&TauLines[ipTS11], 
	  &TauLines[ipS4_1405], 
	  &TauLines[ipS4_1398], 
	  &TauLines[ipS4_1424], 
	  &TauLines[ipS4_1417], 
	  &TauLines[ipS4_1407], 
	  1.168 , 3.366 , 2.924 , 7.233 , "S  4");

	/* S V Be-seq line, A=
	 * >>refer	s5	as	Mendoza, C. 1982, in Planetary Nebulae, IAU Symp No. 103,
	 * >>refercon	ed by D.R. Flower, (D. Reidel: Holland), 143
	 * all cs 
	 * >>refer	s5	as	Dufton, P.L., Hibbert, A., Keenan, F.P, Kingston, A.E., &
	 * >>refercon	Doschek, G.A. 1986, ApJ, 300, 448
	 * and 
	 * >>refer	s5	cs	Dufton, P.L., & Kingston, A.E. 1984, J.Phys. B, 17, 3321 */
	/* >>chng 01 sep 09, AtomSeqBeryllium will reset this to 1/3 so critical density correct */
	cs = MIN2(1.58,35.372/(phycon.te10*phycon.te10*phycon.te10));
	PutCS(cs,&TauLines[ipT1198]);
	/* cs01 = MIN(1.98, 29.625/(T**0.25)) */
	cs01 = MIN2(1.98,29.625/(phycon.te20*phycon.te05));
	/* cs02 = MIN(2.26, 13.477/(T**0.165)) */
	cs02 = MIN2(2.26,13.477/(phycon.te10*phycon.te03*phycon.te03*
	  phycon.te005));
	/* cs12 = MIN(7.59, 63.994/(T**0.197)) */
	cs12 = MIN2(7.59,63.994/(phycon.te20/phycon.te003));
	/* AtomSeqBeryllium(cs23,cs24,cs34,tarray,a41)
	 * call AtomSeqBeryllium( 1.98 , 2.26 , 7.59, t1198 ,.066) */
	AtomSeqBeryllium(cs01,cs02,cs12,&TauLines[ipT1198],.066);
	embesq.em1198 = (float)(PopLevls.PopLevels[3]*0.066*1.66e-11);

	/* S V 786
	 * cs from 
	 * >>refer	s5	cs	Dufton, P.L., & Kingston, A.E. 1984, J.Phys. B, 17, 3321 */
	PutCS(8.3,&TauLines[ipT786]);
	level2(&TauLines[ipT786]);

	/* S VI, Li sequence, A from
	 * >>refer	s6	as	Mendoza, C. 1982, in Planetary Nebulae, IAU Symp No. 103,
	 * >>refercon	ed by D.R. Flower, (D. Reidel: Holland), 143
	 * CS from 
	 * >>refer	s6	cs	Dufton, P.L., & Kingston, A.E. 1987, J.Phys. B, 20, 3899 */
	cs = 11.9;
	PutCS(cs*0.667,&TauLines[ipT933]);
	PutCS(cs*0.333,&TauLines[ipT944]);
	PutCS(1.0,&TauDummy);

	level3(&TauLines[ipT944],&TauDummy,&TauLines[ipT933]);

	/* S VIII 9913
	 * >>refer	s8	cs	Saraph, H.E. & Tully, J.A. 1994, A&AS, 107, 29 */
	cs = MIN2(0.291,0.0289*phycon.te20/phycon.te01* phycon.te001);
	cs = MAX2(0.192,cs);
	PutCS(cs,&TauLines[ipfsS810]);

	level2(&TauLines[ipfsS810]);

	/* S IX 1715, 1987, 
	 * >>refer	s9	all	Kafatos, M., & Lynch, J.P. 1980, ApJS, 42, 611 */
	CoolHeavy.c1715 = popexc(0.4,9.,5.,80.,8.39e4,xIonFracs[ipSULPHUR][8])*
	  1.16e-11;
	coladd("S  9",1715,CoolHeavy.c1715);

	/* S IX 1.25, 3.78 microns
	 * cs 
	 * >>refer	s9	cs	Butler, K., & Zeippen, C.J. 1994, A&AS, 108, 1 */
	cs = MIN2(3.10,277.2/(phycon.sqrte/phycon.te003));
	PutCS(cs,&TauLines[ipfsS912]);

	cs = MIN2(.50,7.940/(phycon.te10*phycon.te10*phycon.te10));
	PutCS(cs,&TauLines[ipfsS938]);

	cs = MIN2(1.1,195.72/(phycon.sqrte*phycon.te03*phycon.te03*
	  phycon.te01*phycon.te003*phycon.te001));
	PutCS(cs,&TauDummy);

	level3(&TauLines[ipfsS912],&TauLines[ipfsS938],&TauDummy);

	/* S X 1213, 1197, 
	 * >>refer	si10	cs	Bell, K.L., & Ramsbottom, C.A., 2000, ADNDT, 76, 176-190 */
	CoolHeavy.c1213 = popexc(0.131,4.,6.,6.,1.19e5,xIonFracs[ipSULPHUR][10-1])*
	  1.64e-11;
	coladd("S 10",1213,CoolHeavy.c1213);

	/* S XI 1.91, 1.40 microns, cs 
	 * >>refer	si11	cs	Lennon, D.J. Burke, V.M. 1994, A&AS, 103, 273 */
	cs = MIN2(0.22,0.0151*phycon.te20*phycon.te03*phycon.te001*
	  phycon.te001);
	PutCS(cs,&TauLines[ipfsS1119]);
	cs = MIN2(0.6,0.0168*phycon.te10*phycon.te10*phycon.te10*
	  phycon.te01);
	PutCS(cs,&TauLines[ipfsS1114]);
	cs = MIN2(0.15,1.5e-3*phycon.te30*phycon.te10*phycon.te005);
	PutCS(cs,&TauDummy);
	level3(&TauLines[ipfsS1119],&TauLines[ipfsS1114],&TauDummy);

	/* S XI 1615, 1826, 
	 * >>refer	si11	cs	Kafatos, M., & Lynch, J.P. 1980, ApJS, 42, 611 */
	CoolHeavy.c1826 = popexc(0.3,9.,5.,100.,8.92e4,xIonFracs[ipSULPHUR][11-1])*
	  1.23e-11;
	coladd("S 11",1615,CoolHeavy.c1826);

	/* S XII 7611.2, cs from 
	 * >>refer	s12	cs	Oliva, E., Pasquali, A., & Reconditi, M. 1996, A&A, 305, 210
	 * >>refer	s12	as	Froese Fischer, C. 1983, J.Phys. B, 16, 157
	 * >>refer	s12	as	Saha, H.P., & Trefftz, E. 1983, SoPh, 87, 233 */
	PutCS(0.135,&TauLines[ipfsS1207]);
	level2(&TauLines[ipfsS1207]);

	/* S XII 520, actually group of four intercombination lines
	 * all atomic data from 
	 * >>refer	s12	all	Saha, H.P., & Trefftz, E. 1983, SoPh, 87, 233 */
	CoolHeavy.c520 = 0.03*1.65e-16*expion(2.8e5,xIonFracs[ipSULPHUR][12-1]);
	coladd("S 12",520,CoolHeavy.c520);

	/* S XIII 488.4, 1909-like */
	CoolHeavy.c488 = popexc(0.11,1.,9.,3.3e5,2.948e5,xIonFracs[ipSULPHUR][13-1])*
	  4.07e-11;
	coladd("S 13",488,CoolHeavy.c488);

	/* li seq 2s2p and 2s3p, S XIV 418, 446
	 * >>refer	s14	cs	Cochrane, D.M., & McWhirter, R.W.P. 1983, PhyS, 28, 25 */
	ligbar(16,&TauLines[ipTSu418],&TauLines[ipTSu30],&cs2s2p,&cs2s3p);
	PutCS(cs2s2p,&TauLines[ipTSu418]);
	PutCS(cs2s2p*0.5,&TauLines[ipTSu446]);
	PutCS(1.0,&TauDummy);
	level3(&TauLines[ipTSu446],&TauDummy,&TauLines[ipTSu418]);

	PutCS(cs2s3p,&TauLines[ipTSu30]);
	level2(&TauLines[ipTSu30]);

#	ifdef DEBUG_FUN
	fputs( " <->CoolSulf()\n", debug_fp );
#	endif
	return;
}

/*S2cs compute [sii] collision strengths
 * compute collision strengths for [SII] transitions 
 * w/in S II ground term. From 
 *>>refer	s2	cs	Ramsbottom, C.A., Bell, K.L., Stafford, R.P. 1996, ADNDT, 63, 57 */
static void S2cs(double telog, 
	  double *cs12, 
	  double *cs13, 
	  double *cs14, 
	  double *cs15, 
	  double *cs23, 
	  double *cs24, 
	  double *cs25, 
	  double *cs34, 
	  double *cs35, 
	  double *cs45)
{
	/* written by Kirk Korista  */
	double a, 
	  b, 
	  c, 
	  telogn1;

#	ifdef DEBUG_FUN
	fputs( "<+>S2cs()\n", debug_fp );
#	endif

	/* limit to stop exceeding bounds */
	telogn1 = MAX2(3.5,telog);
	telogn1 = MIN2(telogn1,5.0);

	/* 2D5/2 - 2P3/2    S II  10320.4   A=0.179 (3-->5) */
	a = 18.335524;
	b = -5.1180248;
	c = 0.44482438;
	*cs35 = MIN2(5.82,a+b*telogn1+c*telogn1*telogn1);
	*cs35 = MAX2(3.87,*cs35);

	/* 2D3/2 - 2P3/2          10286.7   A=0.1335 (2-->5) */
	a = 6.690242;
	b = -1.061514;
	c = 0.034535506;
	*cs25 = MIN2(3.38,a+b*telogn1+c*telogn1*telogn1);
	*cs25 = MAX2(2.24,*cs25);

	/* 2D5/2 - 2P1/2          10373.3   A=0.0779 (3-->4) */
	a = 4.2250081;
	b = -0.46549935;
	c = -0.010172139;
	*cs34 = MIN2(2.46,a+b*telogn1+c*telogn1*telogn1);
	*cs34 = MAX2(1.64,*cs34);

	/* 2D3/2 - 2P1/2          10336.3   A=0.1626 (2-->4) */
	a = 8.274085;
	b = -2.6223732;
	c = 0.2502924;
	*cs24 = MIN2(2.14,a+b*telogn1+c*telogn1*telogn1);
	*cs24 = MAX2(1.42,*cs24);

	/* 2P1/2 - 2P3/2   */
	a = -5.1994665;
	b = 49.334586;
	c = -70.93344;
	*cs45 = MIN2(3.07,a+b/telogn1+c/(telogn1*telogn1));
	*cs45 = MAX2(1.85,*cs45);

	/* 2D3/2 - 2D5/2 */
	a = -27.497273;
	b = 247.27405;
	c = -429.9142;
	*cs23 = MIN2(8.01,a+b/telogn1+c/(telogn1*telogn1));
	*cs23 = MAX2(4.79,*cs23);

	/* 4S3/2 - 2P3/2           4068.6   A=0.220  */
	a = 2.6106784;
	b = -3.2766908e-05;
	c = 6.5105436;
	*cs15 = a+b*pow(telogn1,c);
	*cs15 = MIN2(2.46,*cs15);
	*cs15 = MAX2(1.45,*cs15);

	/* 4S3/2 - 2P1/2           4076.4   A=0.091       */
	*cs14 = 0.5**cs15;

	/* 4S3/2 - 2D5/2            6716.5   A=2.601e-04   */
	a = 8.1458628;
	b = -0.5389108;
	c = 1.4486586;
	*cs13 = a+b*pow(telogn1,c);
	*cs13 = MIN2(4.77,*cs13);
	*cs13 = MAX2(2.54,*cs13);

	/* 4S3/2 - 2D3/2           6730.8   A=8.82e-04    */
	*cs12 = *cs13/1.5;

	/* option to print values, usually commented out
	 * write(6,*) 10.**telog
	 * write(6,*) cs12
	 * write(6,*) cs13
	 * write(6,*) cs14
	 * write(6,*) cs15
	 * write(6,*) cs23
	 * write(6,*) cs24
	 * write(6,*) cs25
	 * write(6,*) cs34
	 * write(6,*) cs35
	 * write(6,*) cs45
	 * stop
	 * */

#	ifdef DEBUG_FUN
	fputs( " <->S2cs()\n", debug_fp );
#	endif
	return;
}

