/*conorm normalize continuum to proper intensity */
/*qintr integrates Q for any continuum between two limits, used for normalization */
/*pintr integrates L for any continuum between two limits, used for normalization */
/*pow1 evaluate luminosity of continuum */
#include "cddefines.h"
#include "ipoint.h"
#include "physconst.h"
#include "phycon.h"
#include "radius.h"
#include "trace.h"
#include "sphere.h"
#include "grainvar.h"
#include "printit.h"
#include "bit32.h"
#include "ffun.h"
#include "rfield.h"
#include "conorm.h"

#define	SKIP	2.

/*pintr integrates L for any continuum between two limits, used for normalization */
static double pintr(double penlo, 
  double penhi);

/*qintr integrates Q for any continuum between two limits, used for normalization */
static double qintr(double *qenlo, 
  double *qenhi);

/*pow1 evaluate luminosity of continuum */
static double pow1(double freq);

void conorm(void)
{
	long int i , nd;
	double xLog_radius_inner, 
	  diff, 
	  f, 
	  flx1, 
	  flx2, 
	  pentrd, 
	  qentrd;

#	ifdef DEBUG_FUN
	fputs( "<+>conorm()\n", debug_fp );
#	endif

	xLog_radius_inner = log10(radius.Radius);

	/* this is a sanity check, it can't happen */
	for( i=0; i < rfield.nspec; i++ )
	{
		if( strcmp(rfield.chRSpec[i],"UNKN") == 0 )
		{
			fprintf( ioQQQ, " UNKN spectral normalization cannot happen.\n" );
			fprintf( ioQQQ, " conorm punts.\n" );
			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}

		else if( strcmp(rfield.chRSpec[i],"SQCM") != 0 && 
			      strcmp(rfield.chRSpec[i],"4 PI") != 0 )
		{
			fprintf( ioQQQ, " chRSpec must be SQCM or 4 PI, and it was %4.4s.  This cannot happen.\n", 
			  rfield.chRSpec[i] );
			fprintf( ioQQQ, " conorm punts.\n" );
			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}


		/* this sanity check makes sure that atlas.mod or werner.mod grids
		 * are for the current version of the code */
		if( strcmp(rfield.chSpType[i],"VOLK ") == 0 )
		{
			/* check that wavelength scale is actually defined outside here */
			ASSERT( rfield.AnuOrg[rfield.nupper-1]>0. );

			diff = fabs(rfield.tNuRyd[rfield.nupper-1][i]-rfield.AnuOrg[rfield.nupper-1])/
			  rfield.AnuOrg[rfield.nupper-1];

			/* this was read from a binary file, so match should be precise */
			if( diff > 10.*FLT_EPSILON )
			{
				fprintf( ioQQQ, "%10.2e%10.2e\n", rfield.AnuOrg[rfield.nupper-1], 
				  rfield.tNuRyd[rfield.nupper-1][i] );

				fprintf( ioQQQ,"conorm: The energy grid of the stellar atmosphere file does not agree with the grid in this version of the code.\n" );
				fprintf( ioQQQ,"A stellar atmosphere grid from an old version of the code is probably in place.\n" );
				fprintf( ioQQQ,"A grid for the current version of Cloudy must be generated and used.\n" );
				fprintf( ioQQQ,"This is done with the COMPILE STARS command.\n" );
				fprintf( ioQQQ,"Sorry.\n" );

				puts( "[Stop in conorm]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
	}

	/* this sanity check is that the grains we have read in from opacity files agree
	 * with the energy grid in this version of cloudy */
	for( nd=0; nd < gv.nBin; nd++ )
	{
#ifdef INCLUDE_OLD_GRAINS
		/* check ok if turned on and we read the opacities in from a pgrains command  */
		if( gv.bin[nd]->lgDustOn1 && gv.bin[nd]->EnergyCheck>0. )
		{
			int lgErrorFound = FALSE;

			/* these better agree */
			if( gv.bin[nd]->NFPCheck != rfield.nupper )
			{
				fprintf( ioQQQ, "%ld %ld: number of frequency points do not match\n",
					 rfield.nupper, gv.bin[nd]->NFPCheck );
				lgErrorFound = TRUE;
			}

			/* check that wavelength scale is actually defined outside here */
			ASSERT( rfield.AnuOrg[rfield.nupper-1]>0. );

			diff = fabs( gv.bin[nd]->EnergyCheck - rfield.AnuOrg[rfield.nupper-1] )/
			  rfield.AnuOrg[rfield.nupper-1];

			/* this was read from an ascii file, so we have to be more lenient
			 * the last constant is determined by the number of decimal places in the .opc file */
			if( diff > MAX2(10.*FLT_EPSILON,3.e-6f) )
			{
				fprintf( ioQQQ, "%14.6e%14.6e: frequencies of last grid point do not match\n",
					 rfield.AnuOrg[rfield.nupper-1], gv.bin[nd]->EnergyCheck );
				lgErrorFound = TRUE;
			}

			if( lgErrorFound )
			{
				fprintf( ioQQQ,"conorm: The energy grid of the grain opacity file does not agree with the grid in this version of the code.\n" );
				fprintf( ioQQQ,"A compiled grain grid from an old version of the code is probably in place.\n" );
				fprintf( ioQQQ,"A grid for the current version of Cloudy must be generated and used.\n" );
				fprintf( ioQQQ,"This is done with the COMPILE ALL GRAINS command.\n" );
				fprintf( ioQQQ,"Sorry.\n" );

				puts( "[Stop in conorm]" );
				cdEXIT(EXIT_FAILURE);
			}
		}
#else
		int lgErrorFound = FALSE;

		/* these better agree */
		if( gv.bin[nd]->NFPCheck != rfield.nupper )
		{
			fprintf( ioQQQ, "%ld %ld: number of frequency points do not match\n",
				 rfield.nupper, gv.bin[nd]->NFPCheck );
			lgErrorFound = TRUE;
		}

		/* check that wavelength scale is actually defined outside here */
		ASSERT( rfield.AnuOrg[rfield.nupper-1]>0. );

		diff = fabs( gv.bin[nd]->EnergyCheck - rfield.AnuOrg[rfield.nupper-1] )/
		  rfield.AnuOrg[rfield.nupper-1];

		/* this was read from an ascii file, so we have to be more lenient
		 * the last constant is determined by the number of decimal places in the .opc file */
		if( diff > MAX2(10.*FLT_EPSILON,3.e-6f) )
		{
			fprintf( ioQQQ, "%14.6e%14.6e: frequencies of last grid point do not match\n",
				 rfield.AnuOrg[rfield.nupper-1], gv.bin[nd]->EnergyCheck );
			lgErrorFound = TRUE;
		}

		if( lgErrorFound )
		{
			fprintf( ioQQQ,"conorm: The energy grid of the grain opacity file does not agree with the grid in this version of the code.\n" );
			fprintf( ioQQQ,"A compiled grain grid from an old version of the code is probably in place.\n" );
			fprintf( ioQQQ,"A grid for the current version of Cloudy must be generated and used.\n" );
			fprintf( ioQQQ,"This is done with the COMPILE ALL GRAINS command.\n" );
			fprintf( ioQQQ,"Sorry.\n" );

			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}
#endif
	}

	/* default is is to predict line intensities, 
	 * but if any continuum specified as luminosity then would override this -
	 * following two values are correct for intensities */
	radius.pirsq = 0.;
	radius.lgPredLumin = FALSE;

	/* check whether ANY luminosities are present */
	for( i=0; i < rfield.nspec; i++ )
	{
		if( strcmp(rfield.chRSpec[i],"4 PI") == 0 )
		{
			radius.pirsq = (float)(1.0992099 + 2.*xLog_radius_inner);
			radius.lgPredLumin = TRUE;
			/* convert down to intensity */
			rfield.totpow[i] -= radius.pirsq;

			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm converts continuum%3ld from luminosity to intensity.\n", 
				  i );
			}
		}
	}

	/* convert ionization parameters to q(h) */
	for( i=0; i < rfield.nspec; i++ )
	{
		if( strcmp(rfield.chSpNorm[i],"IONI") == 0 )
		{
			rfield.totpow[i] += log10(phycon.hden) + log10(SPEEDLIGHT);
			strcpy( rfield.chSpNorm[i], "Q(H)" );
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm converts continuum%3ld from ionizat par to q(h).\n", 
				  i );
			}
		}
	}

	/* convert x-ray ionization parameter xi to intensity */
	for( i=0; i < rfield.nspec; i++ )
	{
		if( strcmp(rfield.chSpNorm[i],"IONX") == 0 )
		{
			/* this converts it to an intensity */
			rfield.totpow[i] += log10(phycon.hden) - log10(PI4);
			strcpy( rfield.chSpNorm[i], "LUMI" );
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm converts continuum%3ld from x-ray ionizat par to I.\n", 
				  i );
			}
		}
	}

	/* indicate whether we ended up with luminosity or intensity */
	if( trace.lgTrace )
	{
		if( radius.lgPredLumin )
		{
			fprintf( ioQQQ, " Cloudy will predict lumin into 4pi\n" );
		}
		else
		{
			fprintf( ioQQQ, " Cloudy will do surface flux for lumin\n" );
		}
	}

	/* if intensity per unit area is predicted then geometric
	 * covering factor must be unity
	 * variable can also be set elsewhere */
	if( !radius.lgPredLumin )
	{
		sphere.covgeo = 1.;
	}

	/* main loop to find continuum normalization starts here */
	for( i=0; i < rfield.nspec; i++ )
	{
		rfield.ipspec = i;
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, " conorm continuum number%3ld is shape %5.5s range is %11.2e%11.2e\n", 
			  i, rfield.chSpType[i], rfield.range[0][i], rfield.range[1][i] );
		}

		if( strcmp(rfield.chSpNorm[i],"RATI") == 0 )
		{
			/* option to scale relative to previous continua
			 * this must come first since otherwise may be too late
			 * BUT ratio cannot be the first continuum source */
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm this is ratio to 1st con\n" );
			}

			/* check that this is not the first continuum source, we must ratio */
			if( i == 0 )
			{
				fprintf( ioQQQ, " I cant form a ratio if continuum is first source.\n" );
				puts( "[Stop in conorm]" );
				cdEXIT(EXIT_FAILURE);
			}

			/* first find photon flux and Q of prevous continuum */
			rfield.ipspec -= 1;
			flx1 = ffun1(rfield.range[0][i])*rfield.spfac[rfield.ipspec]*
			  rfield.range[0][i];

			/* check that previous continua were not zero where ratio is formed */
			if( flx1 <= 0. )
			{
				fprintf( ioQQQ, " Previous continua were zero where ratio is desired.\n" );
				puts( "[Stop in conorm]" );
				cdEXIT(EXIT_FAILURE);
			}

			/* return pointer to previous (correct) value, find F, Q */
			rfield.ipspec += 1;

			/* we want a continuum totpow as powerful, flx is now desired flx */
			flx1 *= rfield.totpow[i];

			/*.        find flux of this new continuum at that point */
			flx2 = ffun1(rfield.range[1][i])*rfield.range[1][i];

			/* this is ratio of desired to actual */
			rfield.spfac[i] = flx1/flx2;
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm ratio will set scale fac to%10.3e at%10.2e Ryd.\n", 
				  rfield.totpow[i], rfield.range[0][i] );
			}
		}

		else if( strcmp(rfield.chSpNorm[i],"FLUX") == 0 )
		{
			/* specify flux density
			 * option to use arbitrary frequency or range */
			f = ffun1(rfield.range[0][i]); 
			f = MAX2(1e-37,f); 
			f = log10(f) + log10(rfield.range[0][i]*EN1RYD/FR1RYD);

			f = rfield.totpow[i] - f;
			/* >>chng 96 dec 31, added following test */
			if( bit32.lgBit32 )
			{
				if( f > 35. )
				{
					fprintf( ioQQQ, " Continuum source%3ld is too intense for this cpu - is it normalized correctly?\n", 
					  i );
					puts( "[Stop in conorm]" );
					cdEXIT(EXIT_FAILURE);
				}
			}

			rfield.spfac[i] = pow(10.,f);
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm will set log fnu to%10.3e at%10.2e Ryd.  Factor=%11.4e\n", 
				  rfield.totpow[i], rfield.range[0][i], rfield.spfac[i] );
			}
		}

		else if( strcmp(rfield.chSpNorm[i],"Q(H)") == 0 || 
			strcmp(rfield.chSpNorm[i],"PHI ") == 0 )
		{
			/* some type of photon density entered */
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm calling qintr range=%11.3e %11.3e desired val is %11.3e\n", 
				  rfield.range[0][i], 
				  rfield.range[1][i] ,
				  rfield.totpow[i]);
			}

			qentrd = qintr(&rfield.range[0][i],&rfield.range[1][i]);
			diff = rfield.totpow[i] - qentrd;

			if( bit32.lgBit32 && (diff < -25. || diff > 35.) )
			{
				fprintf( ioQQQ, " Continuum source specified is too extreme for a 32 bit cpu.\n" );
				puts( "[Stop in conorm]" );
				cdEXIT(EXIT_FAILURE);
			}

			else
			{
				rfield.spfac[i] = pow(10.,diff);
			}

			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm finds Q over range from%11.4e-%11.4e Ryd, integral= %10.4e Factor=%11.4e\n", 
				  rfield.range[0][i], 
				  rfield.range[1][i], 
				  qentrd ,
				  rfield.spfac[i] );
			}
		}

		else if( strcmp(rfield.chSpNorm[i],"LUMI") == 0 )
		{
			/* luminosity entered, special since default is TOTAL lumin */
			pentrd = pintr(rfield.range[0][i],rfield.range[1][i]) + 
			  log10(EN1RYD);
			f = rfield.totpow[i] - pentrd;

			/* >>chng 96 dec 31, added following test */
			if( bit32.lgBit32 )
			{
				if( f > 35. )
				{
					fprintf( ioQQQ, " Continuum source%3ld is too intense for this cpu - is it normalized correctly?\n", 
					  i );
					puts( "[Stop in conorm]" );
					cdEXIT(EXIT_FAILURE);
				}
			}

			rfield.spfac[i] = pow(10.,f);
			if( trace.lgTrace )
			{
				fprintf( ioQQQ, " conorm finds luminosity range is%10.3e to %9.3e Ryd, factor is%11.4e\n", 
				  rfield.range[0][i], rfield.range[1][i], 
				  rfield.spfac[i] );
			}
		}

		else
		{
			fprintf( ioQQQ, "What chSpNorm label is this? =%s=\n", rfield.chSpNorm[i]);
			fprintf( ioQQQ, "Insanity has been detected, I cant go on.\n");
			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}

		/* sec part after .or. added June 93 because sometimes spfac=0
		 * got past first test */
		if( 1./rfield.spfac[i] == 0. || rfield.spfac[i] == 0. )
		{
			fprintf( ioQQQ, "conorm finds infinite continuum scale factor.\n" );
			fprintf( ioQQQ, "The continuum is too intense to compute with this cpu.\n" );
			fprintf( ioQQQ, "Were the intensity and luminosity commandsswitched?\n" );
			fprintf( ioQQQ, "Cloudy punts.  Sorry.\n" );
			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}
	}

	/* this is conversion factor for final units of line intensities or luminosities in printout,
	 * will be intensities (==0) unless luminosity is to be printed, or flux at Earth 
	 * pirsq is the log of 4 pi r_in^2 */
	radius.Conv2PrtInten = radius.pirsq;

	/* >>chng 02 apr 25, add option for slit on aperture command */
	if( sphere.iEmissPower == 1  )
	{
		if( radius.lgPredLumin )
		{
			/* factor should be divided by 2 r_in */
			radius.Conv2PrtInten -= (log10(2.) + xLog_radius_inner);
		}
		else if( !radius.lgPredLumin )
		{
			/* this is an error - slit requested but radius is not known */
			fprintf( ioQQQ, "conorm: Aperture slit specified, but not predicting luminosity.\n" );
			fprintf( ioQQQ, "conorm: Please specify an inner radius to determine L.\nSorry\n" );
			puts( "[Stop in conorm]" );
			cdEXIT(EXIT_FAILURE);
		}
	}
	if( sphere.iEmissPower == 0 && radius.lgPredLumin)
	{
		/* leave Conv2PrtInten at zero if not predicting luminosity */
		radius.Conv2PrtInten = log10(2.);
	}

	/* this is option go give final absolute results as flux observed at Earth */
	if( radius.distance > 0. && radius.lgRadiusKnown && printit.lgPrintFluxEarth )
	{
		/*radius.Conv2PrtInten -= 2.*xLog_radius_inner - 2.*log10(radius.distance) ;*/
		/* div (in log) by 4 pi dist^2 */
		radius.Conv2PrtInten -= log10( 4.*PI*POW2(radius.distance) ) ;
	}

#	ifdef DEBUG_FUN
	fputs( " <->conorm()\n", debug_fp );
#	endif
	return;
}

/*qintr integrates Q for any continuum between two limits, used for normalization */
static double qintr(double *qenlo, 
  double *qenhi)
{
	int lgMore;
	long int i, 
	  ipHi, 
	  ipLo, 
	  j;
	double en1, 
	  en2, 
	  fsum, 
	  hold, 
	  qintr_v, 
	  sum, 
	  wanu, 
	  wfun;
	static double aweigh[4], 
	  fweigh[4];
	static int _aini = 1;
	if( _aini ){ /* Do 1 TIME INITIALIZATIONS! */
		aweigh[0] = -0.4305682;
		aweigh[1] = -0.1699905;
		aweigh[2] = 0.1699905;
		aweigh[3] = 0.4305682;
		fweigh[0] = 0.1739274;
		fweigh[1] = 0.3260726;
		fweigh[2] = 0.3260726;
		fweigh[3] = 0.1739274;
		_aini = 0;
	}

#	ifdef DEBUG_FUN
	fputs( "<+>qintr()\n", debug_fp );
#	endif

	/* returns LOG of number of photons over energy interval */
	sum = 0.;
	if( strcmp(rfield.chSpType[rfield.ipspec],"LASER") == 0 )
	{
		long int ip1 , ip2;
		/* laser is special since delta function, this is center of laser */
		/* >>chng 01 jul 01, was +-21 cells, change to call to ipoint with optional range */
		ip1 = ipoint( rfield.slope[rfield.ipspec]*
			(1.-rfield.cutoff[0][rfield.ipspec]*1.2) );

		ip2 = ipoint( rfield.slope[rfield.ipspec]*
			(1.+rfield.cutoff[0][rfield.ipspec]*1.2) );

		for( i=ip1; i < ip2; i++ )
		{
			fsum = 0.;
			for( j=0; j < 4; j++ )
			{
				wanu = rfield.anu[i] + rfield.widflx[i]*aweigh[j];
				wfun = fweigh[j]*ffun1(wanu);
				fsum += wfun;
			}
			sum += fsum*rfield.widflx[i];
		}
	}

	else if( strcmp(rfield.chSpType[rfield.ipspec],"INTER") == 0 )
	{
		/* interpolate is special since might be very chaotic */
		ipLo = ipoint(*qenlo);
		ipHi = ipoint(*qenhi);
		/* there are possible edge effects due to finite size of cells 
		 * do start and end as special cases */
		/* this is the lowest energy cell, but not below continuum array*/
		hold = MAX2(rfield.emm,rfield.anu[ipLo]);
		sum = ffun1(hold)*rfield.widflx[ipLo];
		if( ipLo != ipHi )
		{
			/* this is highest energy point, but not above limit */
			hold = MIN2(rfield.egamry,rfield.anu[ipHi-1]);
			sum += ffun1(hold)*rfield.widflx[ipHi-1];
		}
		for( i=ipLo; i < (ipHi - 1); i++ )
		{
			sum += ffun1(rfield.anu[i])*rfield.widflx[i];
		}
	}
	else
	{
		lgMore = TRUE;
		en1 = *qenlo/SKIP;
		while( lgMore )
		{
			en1 *= SKIP;
			en2 = en1*SKIP;
			if( en2 >= *qenhi )
			{
				en2 = *qenhi;
				lgMore = FALSE;
			}
			sum += qg32(en1,en2,ffun1);
		}
	}

	if( sum <= 0. )
	{
		fprintf( ioQQQ, " Photon number sum in QINTR is%10.3e\n", 
		  sum );
		fprintf( ioQQQ, " This source has no ionizing radiation, and the number of ionizing photons was specified.\n" );
		fprintf( ioQQQ, " This was continuum source number%3ld\n", 
		  rfield.ipspec );
		fprintf( ioQQQ, " Sorry, but I cannot go on.  ANU and FLUX arrays follow.  Enjoy.\n" );
		for( i=1; i <= rfield.nupper; i++ )
		{
			fprintf( ioQQQ, "%10.2e%10.2e", rfield.anu[i-1], 
			  rfield.flux[i-1] );
		}
		fprintf( ioQQQ, "\n" );
		puts( "[Stop in qintr]" );
		cdEXIT(EXIT_FAILURE);
	}
	else
	{
		qintr_v = log10(sum);
	}

#	ifdef DEBUG_FUN
	fputs( " <->qintr()\n", debug_fp );
#	endif
	return( qintr_v );
}

/*pintr integrates L for any continuum between two limits, used for normalization */
static double pintr(double penlo, 
  double penhi)
{
	int lgMore;
	long int i, 
	  j;
	double en1, 
	  en2, 
	  fsum, 
	  pintr_v, 
	  sum, 
	  wanu, 
	  wfun;
	static double aweigh[4], 
	  fweigh[4];

	static int _aini = 1;
	if( _aini ){ /* Do 1 TIME INITIALIZATIONS! */
		aweigh[0] = -0.4305682;
		aweigh[1] = -0.1699905;
		aweigh[2] = 0.1699905;
		aweigh[3] = 0.4305682;
		fweigh[0] = 0.1739274;
		fweigh[1] = 0.3260726;
		fweigh[2] = 0.3260726;
		fweigh[3] = 0.1739274;
		_aini = 0;
	}

#	ifdef DEBUG_FUN
	fputs( "<+>pintr()\n", debug_fp );
#	endif

	/* computes log of luminosity in radiation over some intergal
	 * answer is in Ryd per sec */

	sum = 0.;
	if( strcmp(rfield.chSpType[rfield.ipspec],"LASER") == 0 )
	{
		long int ip1 , ip2;
		/* laser is special since delta function, this is center of laser */
		/* >>chng 01 jul 01, was +-21 cells, change to call to ipoint */
		ip1 = ipoint( rfield.slope[rfield.ipspec]*
			(1.-rfield.cutoff[0][rfield.ipspec]*1.2) );

		ip2 = ipoint( rfield.slope[rfield.ipspec]*
			(1.+rfield.cutoff[0][rfield.ipspec]*1.2) );

		for( i=ip1; i < ip2; i++ )
		{
			fsum = 0.;
			for( j=0; j < 4; j++ )
			{
				wanu = rfield.anu[i] + rfield.widflx[i]*aweigh[j];
				wfun = fweigh[j]*ffun1(wanu)*wanu;
				fsum += wfun;
			}
			sum += fsum*rfield.widflx[i];
		}

	}
	else
	{
		lgMore = TRUE;
		en1 = penlo/SKIP;
		while( lgMore )
		{
			en1 *= SKIP;
			en2 = en1*SKIP;
			if( en2 >= penhi )
			{
				en2 = penhi;
				lgMore = FALSE;
			}
			sum += qg32(en1,en2,pow1);
		}
	}

	if( sum > 0. )
	{
		pintr_v = log10(sum);
	}
	else
	{
		pintr_v = -38.;
	}


#	ifdef DEBUG_FUN
	fputs( " <->pintr()\n", debug_fp );
#	endif
	return( pintr_v );
}

/*pow1 evaluate luminosity of continuum */
static double pow1(double freq)
{
	double pow1_v;

#	ifdef DEBUG_FUN
	fputs( "<+>pow1()\n", debug_fp );
#	endif

	pow1_v = ffun1(freq)*freq;

#	ifdef DEBUG_FUN
	fputs( " <->pow1()\n", debug_fp );
#	endif
	return( pow1_v );
}

