/*BiDiag solve the bi-diagonal matrix for ionization balance */
#include "cddefines.h"
#include "yield.h"
#include "printit.h"
#include "ionrange.h"
#include "recom.h"
#include "ionrec.h"
#include "ionfracs.h"
#include "iso.h"
#include "dynamics.h"
#include "heat.h"
#include "destcrt.h"
#include "chargtran.h"
#include "heavy.h"
#include "elementnames.h"
#include "abundances.h"
#include "converge.h"
#include "negcon.h"
#include "bidiag.h"

void BiDiag(
	/* this is element on the c scale, H is 0 */
	long int nelem, 
	/* option to print this element when called */
	int lgPrintIt)
{
	int lgDone, 
	  lgNegPop;
	long int LoopBig, 
	  ion, 
	  limit, 
	  MaxVal, 
	  nej, 
	  nelec, 
	  ns;
	float abundOld;
	double xIonRatio[LIMELM+2], 
	  ratio, 
	  sum;
	double AbundNew[LIMELM+1];
	double total_abund;

#	ifdef DEBUG_FUN
	fputs( "<+>BiDiag()\n", debug_fp );
#	endif

	/* this is on the c scale, so H is 0 */
	ASSERT( nelem >= 0);
	ASSERT( IonRange.IonLow[nelem] >= 0 );
	ASSERT( IonRange.IonHigh[nelem] >= 0 );

	/* >>chng 01 dec 07, define total_abund, total atom and ion abundance here, removing molecules */
	total_abund = MAX2( SMALLFLOAT , abundances.gas_phase[nelem] -  xMolFracs[nelem] );

	/* >>chng 00 dec 12, return if IonHigh is zero, since no ionization at all */
	if( IonRange.IonHigh[nelem] == 0 )
	{
		/* set the atom to the total gas phase abundance */
		xIonFracs[nelem][0] = (float)total_abund ;
#		ifdef DEBUG_FUN
		fputs( " <->BiDiag()\n", debug_fp );
#		endif
		return;
	}

	/* >>chng 01 may 09, add option to force ionization distribution with element name ioniz */
	if( abundances.lgSetIoniz[nelem] )
	{
		for( ion=0; ion<nelem+2; ++ion )
		{
			xIonFracs[nelem][ion] = abundances.SetIoniz[nelem][ion]*(float)total_abund;
		}
#		ifdef DEBUG_FUN
		fputs( " <->BiDiag()\n", debug_fp );
#		endif
		return;
	}

	/* impossible for HIonFrac[nelem] to be zero if IonHigh(nelem)=nelem+1
	 * HIonFrac(nelem) is stripped to hydrogen */
	/* >>chng 01 oct 30, to assert */
	ASSERT( (IonRange.IonHigh[nelem] < nelem + 1) || iso.xIonRatio[ipH_LIKE][nelem] > 0. );

#	if !defined(NDEBUG)
	/* set to negative value to make sure (with assert) reset before used below */
	for( ion=0; ion<nelem+2; ++ion )
	{
		xIonRatio[ion] = -1.;
	}
#	endif

	/* this will be a sanity check */
	lgNegPop = FALSE;

	LoopBig = 0;
	lgDone = FALSE;
	while( LoopBig < 5 && (!lgDone) )
	{
		/* zero out the ionization and recombination rates that we will modify here,
		 * but not he h-like and he-like stages which are done elsewhere,
		 * the nelem stage of ionization is he-like,
		 * the nelem+1 stage of ionization is h-like */

		/* sanity check */
		ASSERT( IonRange.IonLow[nelem] < IonRange.IonHigh[nelem] );

		/* loop over stages of ionization that we solve for here, 
		 * up through one less than he-like,
		 * neveral actually do very highest stage of ionization since it
		 * comes from the ionization ratio from the next lower stage */
		limit = MIN2(nelem-2,IonRange.IonHigh[nelem]-1);

		/* zero-out loop comes before main loop since there are off-diagonal
		 * elements in the main ionization loop, due to multi-electron processes */
		for( ion=0; ion <= limit; ion++ )
		{
			/* >>chng 01 mar 15, add advective term */
			/* add advection terms here, normally zero */
			DestCrt.xIonize[nelem][ion] = dynamics.Photo;
			DestCrt.Recombine[nelem][ion] = dynamics.Recomb[nelem][ion];
		}

		/* add on compton recoil ionization for atoms */
		/* >>chng 02 mar 24, moved this to photoionize */
		/* >>chng 01 dec 19, add on these terms for atoms */
		/* this must be moved to photoionize and have code parallel to iso_photo code*/
		/* DestCrt.x Ionize[nelem][0] += ionrec.CompRecoil IonRate[nelem];*/
		/* heat.heat ing[nelem][0] = ionrec.CompRecoil HeatRate[nelem]; */

		/* now get actual arrays of ionization and recombination processes */
		for( ion=IonRange.IonLow[nelem]; ion <= limit; ion++ )
		{
			/* get total destruction rates */
			/* collisional ionization */
			DestCrt.xIonize[nelem][ion] += ionrec.CollIonRate_Ground[nelem][ion][0];

			/* photoionization loop */
			for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
			{
				DestCrt.xIonize[nelem][ion] += ionrec.PhotoRate_Ground[nelem][ion][ns][0];

				/* following is most number of electrons freed */
				nelec = yield.nyield[nelem][ion][ns];

				/* following loop is over number of electrons that come out of shell
				 * with multiple electron ejection */
				for( nej=2; nej <= nelec; nej++ )
				{
					/* this is highest possible stage of ionization -
					 * do not want to ignore ionization that go beyond this */

					MaxVal = MIN2(ion+nej,IonRange.IonHigh[nelem]);
					/* if( xIonFracs(nelem,ion+nej-1).gt.1e-30 ) then */
					if( xIonFracs[nelem][MaxVal-1] > 1e-30 )
					{
						/* ion points to current stage of ionization */
						ratio = (double)(xIonFracs[nelem][ion])/
						  (double)(xIonFracs[nelem][MaxVal-1]);
					}
					else
					{
						ratio = 1.;
					}
					/* yield here is fraction removing ion electrons */
					DestCrt.xIonize[nelem][MaxVal-1] += ratio *
						ionrec.PhotoRate_Ground[nelem][ion][ns][0]*
					  yield.vyield[nelem][ion][ns][nej-1] *
					  /* >>chng 01 dec 18, number of electrons */
					  (double)nej;
				}
			}
			/* this is charge transfer ionization of this specieds by hydrogen and helium */
			DestCrt.xIonize[nelem][ion] += ChargTran.HeCharExcIon[nelem][ion]*xIonFracs[ipHELIUM][1]+ 
			  ChargTran.HCharExcIon[nelem][ion]*xIonFracs[ipHYDROGEN][1];

			/* recombination rates */
			/* >>chng 01 may 15, had been just = not +=, and so clobbered
			 * the advection term.  Robin Williams */
			DestCrt.Recombine[nelem][ion] += recom.RecombinRate[nelem][ion];

			/* charge transfer recombination of this species by ionizing hydrogen and helium */
			DestCrt.Recombine[nelem][ion] += 
				/* >>chng 02 may 05, should be just ground state, make it so */
				/*ChargTran.HeCharExcRec[nelem][ion]*xIonFracs[ipHELIUM][0] + */
				ChargTran.HeCharExcRec[nelem][ion]*iso.Pop2Ion[ipHE_LIKE][ipHELIUM][ipHe1s1S]*xIonFracs[ipHELIUM][1] + 
				/*ChargTran.HCharExcRec[nelem][ion]*xIonFracs[ipHYDROGEN][0];*/
				/* >>chng 01 06 01 should be ground state only, change to do so */
				ChargTran.HCharExcRec[nelem][ion]*iso.Pop2Ion[ipH_LIKE][ipHYDROGEN][ipH1s]*xIonFracs[ipHYDROGEN][1];
		}
		/* after this loop, xIonize and Recombine have been defined for the
		 * stages of ionization that are done with simple */

		/* invert and solve bidiagonal matrix */
		/* >>chng 99 may 1, this had been set to 1 for all previous versions of the code,
		 * and range of ionization was limited by logic dealing with ionization parameter.
		 * that logic was removed (caused other problems) and now comphi.in crashes with
		 * overflow in loop below.  dur to xIonRatio getting larger than biggest double.
		 * change 1 to 1e-200 to get more range, could cause problems with very low
		 * ionization solns??? */

		/* get sum of ratios for stages higher than this lowest stage of ionization,
		 * limit is the highest stage of ionization actually done with simple 2-level atom */
		for( ion=IonRange.IonLow[nelem]; ion <= limit; ion++ )
		{
			xIonRatio[ion] = DestCrt.xIonize[nelem][ion]/DestCrt.Recombine[nelem][ion];
		}

		/* at this point we have the sum of ratios for ionization stages that are done
		 * with simple two-level atom.  Now do he-like and h-like stages of ionization */

		/* test for h-like stages of ionization  */
		if( IonRange.IonHigh[nelem] == nelem + 1 )
		{
			/* extends up to hydrogenic - use hydrogenic solutions*/
			xIonRatio[nelem] = iso.xIonRatio[ipH_LIKE][nelem];
			if( xIonRatio[nelem] == 0. )
			{
				/* this is zero abundance, drop IonHigh */
				IonRange.IonHigh[nelem] = nelem;
			}
		}

		/* test for he-like stages of ionization */
		if( IonRange.IonHigh[nelem] >= nelem )
		{
			/* this will only work for li and above */
			ASSERT( nelem > 1 );
			xIonRatio[nelem-1] = iso.xIonRatio[ipHE_LIKE][nelem];
			if( xIonRatio[nelem-1] == 0. )
			{
				/* this is zero abundance, drop IonHigh */
				IonRange.IonHigh[nelem] = nelem-1;
			}
		}

		/* limit is the highest stage that has an ionization balance to the
		 * next higher stage.  If limit is not equal to IonHigh-1 
		 * then because IonHigh is too high, and will be trimed down later */
		/*ASSERT( limit == (IonRange.IonHigh[nelem]-1) );*/

		/* at this stage it is possible that some elements of xIonRatio will be
		 * zero, especially during search phase when ionizing continuum does
		 * not extend through range of ionization */
		/* >>chng 01 apr 21, first comp had been against 0, change to low+2 */
		/* >>chng 02 may 24, change limit on xIonRatio from SMALLFLOAT to 1e-20 */
		while( IonRange.IonHigh[nelem] > IonRange.IonLow[nelem] && 
			xIonRatio[IonRange.IonHigh[nelem]-1] < 1e-20 )
			/*xIonRatio[IonRange.IonHigh[nelem]-1] < SMALLFLOAT )*/
		{
			/* >>chng 01 oct 30, add this assert to go with preset to -1 above */
			ASSERT( xIonRatio[IonRange.IonHigh[nelem]-1]>= 0. );
			/* zero out abundance and heating due to stage of ionization we are about to zero out */
			xIonFracs[nelem][IonRange.IonHigh[nelem]] = 0.;
			heat.heating[nelem][IonRange.IonHigh[nelem]-1] = 0.;
			/* decrement counter */
			--IonRange.IonHigh[nelem];
		}

		/* sanity check, either offset stages of low and high ionization,
		 * or no ionization at all */
		ASSERT( (IonRange.IonLow[nelem] < IonRange.IonHigh[nelem]) ||
			(IonRange.IonLow[nelem]==0 && IonRange.IonHigh[nelem]==0 ) );

		/* >>chng 00 dec 12, return if IonHigh is zero, since no ionization at all */
		if( IonRange.IonHigh[nelem] == 0 )
		{
			/* set the atom to the total gas phase abundance */
			xIonFracs[nelem][0] = (float)total_abund;

#			ifdef DEBUG_FUN
			fputs( " <->BiDiag()\n", debug_fp );
#			endif
			return;
		}

		/* find dynamic range of abundances - necessary to make sure
		 * sum of abundances will not under or over flow*/
		sum = 0.;
		for( ion=IonRange.IonLow[nelem]; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			sum += log10( xIonRatio[ion] );
		}

		/* form inital sum of abundances such that they are evenly
		 * divided on either side of unity - this will be scale factor
		 * for this initial set of abundances */
		ratio = pow(10. , sum/2 );

		/* define set of new ionfractions, on arbitrary scale for now */
		AbundNew[IonRange.IonLow[nelem]] = 1./ratio;
		sum = AbundNew[IonRange.IonLow[nelem]];

		for( ion=IonRange.IonLow[nelem]; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			AbundNew[ion+1] = xIonRatio[ion] * AbundNew[ion];
			sum += AbundNew[ion+1];
		}

		/* now renormalize so that abundances are correct */
		ratio = total_abund/sum;
		sum = 0.;
		for( ion=IonRange.IonLow[nelem]; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			AbundNew[ion] *= ratio;
			sum += AbundNew[ion];
		}
		{
			/*@-redef@*/
			/* this is to debug following failed assert */
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG && lgPrintIt )
			{
				fprintf(ioQQQ,"sum was %.2e nelem is %li total abundance is %.2e\n", 
					sum , nelem , total_abund );
			}
		}
		/* check that total abundance is positive */
		ASSERT( sum > 0.);
		/* and that it adds up to the correct abundance */
		ASSERT( fabs( sum-total_abund)/total_abund < 1e-7 );

		/* zero out abundances of ions below lowest stage we will consider */
		for( ion=0; ion < IonRange.IonLow[nelem]; ion++ )
		{
			/* recall that xIonFracs ion scale offset by 1 since [0] is total gas phase abundance */
			xIonFracs[nelem][ion] = 0.;
		}

		/* zero out abundances of ions above highest stage we will consider */
		for( ion=IonRange.IonHigh[nelem]+1; ion < (nelem + 2); ion++ )
		{
			/* recall that xIonFracs ion scale offset by 1 since [0] is total gas phase abundance */
			xIonFracs[nelem][ion] = 0.;
		}

		lgDone = TRUE;
		/* we now have a guess at the new ionization fractions - now do something
		 * with this, keeping solution stable */
		for( ion=IonRange.IonLow[nelem]; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			/* scale for xIonFracs is offset by one, since 0 is the total gas phase abundance */
			if( conv.lgSearch && (xIonFracs[nelem][ion] > 0.) && (AbundNew[ion] > 0.) )
			{
				/* this branch used during search phase */
				/* >>chng 96 oct 27, sulphur and above oscillated, corrected by
				 * using log mean during search phase */
				abundOld = xIonFracs[nelem][ion];

				/* >>chng 99 Jul 02, the log in the following is protected against 
				 * non-pos value by if branch above */
				xIonFracs[nelem][ion] = 
					(float)sqrt(xIonFracs[nelem][ion]*AbundNew[ion]);

				/* >>chng 97 jul 29, set flag to redo if changes happen */
				if( xIonFracs[nelem][ion] > 1e-15 )
				{
					if( fabs(abundOld/xIonFracs[nelem][ion]- 1.) > 0.2 )
					{
						lgDone = FALSE;

						/* these are printed as the old and new values that forced 
						 * declaration of no convergence */
						conv.BadConvIoniz[0] = abundOld;
						conv.BadConvIoniz[1] = xIonFracs[nelem][ion];
					}
				}
			}
			else
			{
				/* this branch used for usual search for soln deep in model */
				abundOld = xIonFracs[nelem][ion];
				xIonFracs[nelem][ion] = (float)(AbundNew[ion]);
				if( xIonFracs[nelem][ion]/total_abund > 1e-10 )
				{
					if( fabs(abundOld/xIonFracs[nelem][ion]-1.) > 0.2 )
					{
						lgDone = FALSE;

						/* these are printed as the old and new values that forced 
						 * declaration of no convergence */
						conv.BadConvIoniz[0] = abundOld;
						conv.BadConvIoniz[1] = xIonFracs[nelem][ion];
					}
				}
			}
			if( xIonFracs[nelem][ion] < 0. )
			{
				lgNegPop = TRUE;
			}
		}

		if( printit.lgPrtArry || lgPrintIt )
		{
			/* this is the array that was inverted to produce abundances */
			fprintf( ioQQQ, " IonRat   " );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, " %8.1e", xIonRatio[ion] );
			}
			fprintf( ioQQQ, "\n" );
			/* these are the predicted abundances */
			fprintf( ioQQQ, " AbundX   " );
			for( ion=1; ion <= IonRange.IonHigh[nelem]+1; ion++ )
			{
				fprintf( ioQQQ, " %8.1e", xIonFracs[nelem][ion-1] );
			}
			fprintf( ioQQQ, "\n" );
		}
		++LoopBig;
	}

	if( !lgDone )
	{
		conv.lgConvIoniz = FALSE;
		strcpy( conv.chConvIoniz,  "BiDiag:" );

		/* this is two-char short form of element name */
		strcat( conv.chConvIoniz, elementnames.chElementSym[nelem] );
	}

	/* this can`t possibly happen */
	if( lgNegPop )
	{
		fprintf( ioQQQ, " Negative population found for abundance of ionization stage of element %4.4s, ZONE=%4ld\n", 
		  elementnames.chElementNameShort[nelem], nzone );

		fprintf( ioQQQ, " Populations were" );
		for( ion=1; ion <= IonRange.IonHigh[nelem]+1; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", xIonFracs[nelem][ion-1] );
		}
		fprintf( ioQQQ, "\n" );

		fprintf( ioQQQ, " destroy vector =" );
		for( ion=1; ion <= IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", DestCrt.xIonize[nelem][ion-1] );
		}
		fprintf( ioQQQ, "\n" );

		/* check whether any negative level populations occured */
		lgNegPop = FALSE;
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			if( DestCrt.xIonize[nelem][ion] < 0. )
				lgNegPop = TRUE;
		}

		/* print some extra stuff if destroy was negative */
		if( lgNegPop )
		{
			fprintf( ioQQQ, " CTHeavy  vector =" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ChargTran.HeCharExcIon[nelem][ion] );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, " HCharExcIon vtr=" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ChargTran.HCharExcIon[nelem][ion] );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, " CollidRate  vtr=" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%9.1e", ionrec.CollIonRate_Ground[nelem][ion][0] );
			}
			fprintf( ioQQQ, "\n" );

			/* photo rates per subshell */
			fprintf( ioQQQ, " photo rates per subshell, ion\n" );
			for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
			{
				fprintf( ioQQQ, "%3ld", ion );
				for( ns=0; ns < Heavy.nsShells[nelem][ion]; ns++ )
				{
					fprintf( ioQQQ, "%9.1e", ionrec.PhotoRate_Ground[nelem][ion][ns][0] );
				}
				fprintf( ioQQQ, "\n" );
			}
		}

		/* now check out creation vector */
		fprintf( ioQQQ, " create  vector =" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, "%9.1e", DestCrt.Recombine[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		negcon();
		ShowMe();
		puts( "[Stop in bidiag]" );
		cdEXIT(EXIT_FAILURE);
	}

	/* option to print ionization and recombination arrays
	 * printit flag set with "print array" command */
	if( printit.lgPrtArry || lgPrintIt )
	{
		/* total ionization rate, all processes */
		fprintf( ioQQQ, "\n %4.4s", elementnames.chElementNameShort[nelem] );
		fprintf( ioQQQ, " Itot" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", DestCrt.xIonize[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		/* collisional ionization */
		fprintf( ioQQQ, " %4.4s", elementnames.chElementNameShort[nelem] );
		fprintf( ioQQQ, " Coll" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", ionrec.CollIonRate_Ground[nelem][ion][0] );
		}
		fprintf( ioQQQ, "\n" );

		/* photo ionization */
		fprintf( ioQQQ, " %4.4s", elementnames.chElementNameShort[nelem] );
		fprintf( ioQQQ, " Phot" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", 
				ionrec.PhotoRate_Ground[nelem][ion][Heavy.nsShells[nelem][ion]-1][0] );
		}
		fprintf( ioQQQ, "\n" );

		/* sum of all creation processes */
		fprintf( ioQQQ, " %4.4s", elementnames.chElementNameShort[nelem]);
		fprintf( ioQQQ, " Rtot" );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", DestCrt.Recombine[nelem][ion] );
		}
		fprintf( ioQQQ, "\n" );

		/* charge exchange ionization */
		fprintf( ioQQQ, " char T I " );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", ChargTran.HCharExcIon[nelem][ion]*
			  xIonFracs[ipHYDROGEN][1] );
		}
		fprintf( ioQQQ, "\n" );

		/* charge exchange recombination */
		fprintf( ioQQQ, " char T R " );
		for( ion=0; ion < IonRange.IonHigh[nelem]; ion++ )
		{
			fprintf( ioQQQ, " %8.1e", ChargTran.HCharExcRec[nelem][ion]*
			  xIonFracs[ipHYDROGEN][0] );
		}
		fprintf( ioQQQ, "\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->BiDiag()\n", debug_fp );
#	endif
	return;
}
