#include <float.h>
#include "cddefines.h"
#include "physconst.h"
#include "radius.h"
#include "heat.h"
#include "rfield.h"
#include "trace.h"
#include "interpolate.h"
#include "itercnt.h"
#include "grainvar.h"
#include "qheat.h"


/* set to 0 to revert to original quantum heating code (used in Cloudy 96 beta 1 and 2) */
#define NEW_CODE 1

#if !NEW_CODE
#include "ipoint.h"
#endif


#define NINT(X) ((long)((X) < 0. ? (X)-0.5 : (X)+0.5))

/* NB NB -- the sequence below has been carefully chosen and should NEVER be
 *          altered unless you really know what you are doing !! */
typedef enum {
	/* the following are OK */
	/* 0        1              2             3    */
	QH_OK, QH_ANALYTIC, QH_ANALYTIC_LOW, QH_DELTA, 
	/* the following are mild errors we already recovered from */
	/*   4              5               6         */
	QH_LOOP_FAIL, QH_NEGRATE_FAIL, QH_ARRAY_FAIL,
	/* any of these errors will prompt qheat to do another iteration */
	/*  7          8             9        */
	QH_RETRY, QH_CONV_FAIL, QH_BOUND_FAIL,
	/* any error larger than QH_NO_REBIN will cause GetProbDistr_LowLimit to return
	 * before even attempting to rebin the results; we may be able to recover though... */
	/*   10           11            12            13     */
	QH_NO_REBIN, QH_DELTA_FAIL, QH_LOW_FAIL, QH_HIGH_FAIL,
	/* any case larger than QH_FATAL is truly pathological; there is no hope of recovery */
	/* 14          15           16             17      */
	QH_FATAL, QH_STEP_FAIL, QH_NBIN_FAIL, QH_REBIN_FAIL
} QH_Code;

/*================================================================================*/
/* definitions relating to quantum heating routines */

/* this is the minimum number of bins for quantum heating calculation to be valid */
static const long NQMIN = 10L;

/* this is the lowest value for dPdlnT that should be included in the modeling */
static const double PROB_CUTOFF_LO = 1.e-15;
static const double PROB_CUTOFF_HI = 1.e-15;
static const double SAFETY = 1.e+8;

/* during the first NSTARTUP steps, use special algorithm to calculate stepsize */
static const long NSTARTUP = 5L;

/* if the average number of multiple events is above this number
 * don't try full quantum heating treatment. */
static const double MAX_EVENTS = 150.;
/* if the average number of multiple events is above this number
 * don't even try analytic approximation, use delta function instead */
/* static const double MAX_EVENTS2 = 10000.; */

/* maximum number of tries for quantum heating routine */
/* >>chng 02 jan 30, changed LOOP_MAX from 10 -> 20, PvH */
static const long LOOP_MAX = 20L;

/* if all else fails, divide temp estimate by this factor */
static const double DEF_FAC = 3.;

/* total probability for all bins should not deviate more than this from 1. */
static const double PROB_TOL = 0.02;

#if NEW_CODE
/* after NQTEST steps, make an estimate if prob distr will converge in NQGRID steps */
/* >>chng 02 jan 30, change NQTEST from 1000 -> 500, PvH */
static const long NQTEST = 500L;

/* if the ratio fwhm/Umax is lower than this number
 * don't try full quantum heating treatment. */
static const double FWHM_RATIO = 0.1;
/* if the ratio fwhm/Umax is lower than this number
 * don't even try analytic approximation, use delta function instead */
static const double FWHM_RATIO2 = 0.007;

/* maximum number of steps for integrating quantum heating probability distribution */
static const long MAX_LOOP = 2*NQGRID;

/* this is the tolerance used while integrating the temperature distribution of the grains */
static const double QHEAT_TOL = 5.e-3;

/* when rebinning quantum heating results, make ln(qtemp[i]/qtemp[i-1]) == QT_RATIO */
/* this constant determines the accuracy with which the Wien tail of the grain emission
 * is calculated; if x = h*nu/k*T_gr and d = QT_RATIO-1., then the relative accuracy of
 * the flux in the Wien tail is Acc = fabs(x*d^2/12 - x^2*d^2/24). A typical value of x
 * would be x = 15, so QT_RATIO = 1.03 should converge the spectrum to better than 1% */
static const double QT_RATIO = 1.03;

#else

/* minimum fractional temperature step after rebinning */
static const double QT_RATIO = 1.01;
#endif


/*================================================================================*/
/* global variables */

/* these data define the enthalpy function for silicates
 * derived from:
 * >>refer Guhathakurta & Draine, 1989, ApJ, 345, 230
 * coefficients converted to rydberg energy units, per unit atom
 * assuming a density of 3.3 g/cm^3 and pure MgSiFeO4.
 * this is not right, but the result is correct because number
 * of atoms will be calculated using the same assumption. */

/* this is the specific density of silicate in g/cm^3 */
#define DEN_SIL     3.30

/* these are the mean molecular weights per atom for MgSiFeO4 and SiO2 in amu */
#define MW_SIL      24.6051
/*#define MW_SIO2     20.0283*/

static const double tlim[5]={0.,50.,150.,500.,DBL_MAX};
static const double power[4]={3.00,2.30,1.68,1.00};
static const double cval[4]={
	1.40e3/3.00/DEN_SIL*ATOMIC_MASS_UNIT*MW_SIL/EN1RYD,
	2.20e4/2.30/DEN_SIL*ATOMIC_MASS_UNIT*MW_SIL/EN1RYD,
	4.80e5/1.68/DEN_SIL*ATOMIC_MASS_UNIT*MW_SIL/EN1RYD,
	3.41e7/1.00/DEN_SIL*ATOMIC_MASS_UNIT*MW_SIL/EN1RYD};


/* main routine for quantum heating */
static void qheat(/*@out@*/double[],/*@out@*/double[],/*@out@*/long*,long);
#if NEW_CODE
/* worker routine, this implements the algorithm of Guhathakurtha & Draine */
static void GetProbDistr_LowLimit(long,double,double,/*@in@*/double[],/*@in@*/double[],
				  /*@out@*/double[],/*@out@*/double[],/*@out@*/double[],
				  /*@out@*/long*,/*@out@*/double*,/*@out@*/QH_Code*);
/* try two consecutive integration steps using stepsize "step/2." (yielding p[k]),
 * and also one double integration step using stepsize "step" (yielding p2k). */
static double TryDoubleStep(double[],double[],double[],double[],double[],double[],
			    double[],double,/*@out@*/double*,long,long);
/* calculate logarithmic integral from (x1,y1) to (x2,y2) */
static double log_integral(double,double,double,double);
/* rebin the quantum heating results to speed up RTDiffuse */
static long RebinQHeatResults(long,long,double[],double[],double[],double[],double[],
			      double[],double[],QH_Code*);
/* calculate approximate probability distribution in high intensity limit */
static void GetProbDistr_HighLimit(long,double,double,/*@out@*/double[],/*@out@*/double[],
				   /*@out@*/double[],/*@out@*/long*,/*@out@*/double*,
				   /*@out@*/QH_Code*);
/* derivative of the enthalpy function dU/dT */
static double uderiv(double,long);
#else
/* worker routine, this implements the algorithm of Guhathakurtha & Draine */
static void qheat1(/*@out@*/double[],/*@out@*/double[],/*@out@*/double[],/*@out@*/long*,
		   long,/*@out@*/double*,/*@out@*/int*,/*@out@*/int*,/*@out@*/int*);
/* rebin the quantum heating results to speed up RTDiffuse */
static long RebinQHeatResults(long,long,double[],double[],double[],double[],double[],
			      double[],double[]);
#endif
/* enthalpy function */
static double ufunct(double,long);
/* inverse enthalpy function */
static double inv_ufunct(double,long);

/* >>chng 01 oct 29, introduced gv.bin[nd]->cnv_H_pGR, cnv_GR_pH, etc. PvH */


/* main routine for generating the grain diffuse emission */
void GrainMakeDiffuse()
{
	long i,
	  j,
	  nd;
	double bfac,
	  f,
	  factor,
	  flux,
	  x;

#	ifndef NDEBUG
	double BolFlux,
	  Comparison1,
	  Comparison2;
#	endif

#	ifdef DEBUG_FUN
	fputs( "<+>GrainMakeDiffuse()\n", debug_fp );
#	endif

	factor = 2.*PI4*POW2(FR1RYD/SPEEDLIGHT)*FR1RYD;
	/* >>chng 96 apr 26 upper limt chng from 15 to 75 Peter van Hoof  */
	/* >>chng 00 apr 10 use constants appropriate for double precision, by PvH */
	x = log(0.999*DBL_MAX);

	/* add grain dust emission in IR */
	for( i=0; i < rfield.nflux; i++ )
	{
		/* this is added to one of the local outward beams below, for only one
		 * of the diffuse transfer methods */
		gv.GrainEmission[i] = 0.;
	}

	for( nd=0; nd < gv.nBin; nd++ )
	{
		int lgLocalQHeat;
		long qnbin=-200;
		float *grn;
		/* >>chng 01 sep 11, replace array allocation on stack with
		 * MALLOC to avoid bug in gcc 3.0 on Sparc platforms, PvH */
		double *qtemp/*[NQGRID]*/, *qprob/*[NQGRID]*/;
		double xx;

		if( ( qtemp = (double*)MALLOC((size_t)(NQGRID*sizeof(double))) ) == NULL )
			bad_malloc();
		if( ( qprob = (double*)MALLOC((size_t)(NQGRID*sizeof(double))) ) == NULL )
			bad_malloc();
	
		/* save emission from this species */
		grn = ( gv.bin[nd]->matType == MAT_CAR ) ? gv.GraphiteEmission : gv.SilicateEmission;

		/* this local copy is necessary to keep lint happy */
		lgLocalQHeat = gv.bin[nd]->lgQHeat;
		if( lgLocalQHeat )
		{
			qheat(qtemp,qprob,&qnbin,nd);

			if( gv.bin[nd]->lgUseQHeat )
			{
				assert( qnbin > 0 );
			}
		}

		flux = 1.;
		/* flux can only become zero in the Wien tail */
		for( i=0; i < rfield.nflux && flux > 0.; i++ )
		{
			flux = 0.;
			if( lgLocalQHeat && gv.bin[nd]->lgUseQHeat )
			{
				xx = 1.;
				/* we start at high temperature end and work our way down
				 * until contribution becomes negligible */
				for( j=qnbin-1; j >= 0 && xx > flux*DBL_EPSILON; j-- )
				{
					f = TE1RYD/qtemp[j]*rfield.anu[i];
					if( f < x )
					{
						/* want the number exp(hnu/kT) - 1, two expansions */
						bfac = ( f < 1.e-5 ) ? f + 0.5*f*f : exp(f) - 1.;
						xx = qprob[j]*gv.bin[nd]->dstab1[i]*gv.bin[nd]->cnv_H_pCM3*
							factor*rfield.anu2[i]*rfield.widflx[i]/bfac;
						flux += xx;
					}
					else
					{
						xx = 0.;
					}
				}
			}
			else
			{
				f = TE1RYD/gv.bin[nd]->tedust*rfield.anu[i];
				if( f < x )
				{
					bfac = ( f < 1.e-5 ) ? f + 0.5*f*f : exp(f) - 1.;
					flux = gv.bin[nd]->dstab1[i]*gv.bin[nd]->cnv_H_pCM3*
						factor*rfield.anu2[i]*rfield.widflx[i]/bfac;
				}
			}
			
			/* >>chng 96 jul 13, save grain emission array
			 * >>chng 97 feb 28, GrainEmission was set to last evaluated grain,
			 * zeroed out above, and now increment grain instead of setting it */

			/* only local emission since zeroed out on each zone */
			gv.GrainEmission[i] += (float)flux;
			/* total emission */
			grn[i] += (float)(flux*radius.dVolOutwrd);
		}

		free( qprob );
		free( qtemp );
	}

#	ifndef NDEBUG
	/*********************************************************************************
	 *
	 * Following are three checks on energy and charge conservation by the grain code.
	 * Their primary function is as an internal consistency check, so that coding
	 * errors get caught as early as possible. This is why the program terminates as
	 * soon as any one of the checks fails.
	 *
	 * NB NB - despite appearances, these checks do NOT guarantee overall energy
	 *         conservation in the Cloudy model to the asserted tolerance, see note 1B !
	 *
	 * Note 1: there are two sources for energy imbalance in the grain code (see A & B).
	 *   A: Interpolation in dstems. The code calculates how much energy the grains
	 *      emit in thermal radiation (gv.bin[nd]->GrainHeat), and convert that into
	 *      an (average) grain temperature by reverse interpolation in dstems. If
	 *      quantum heating is not used, that temperature is used directly to generate
	 *      the local diffuse emission. Hence the finite resolution of the dstems grid
	 *      can lead to small errors in flux. This is tested in Check 1. The maximum
	 *      error of interpolating in dstems scales with NDEMS^-3. The same problem
	 *      can also occur when quantum heating is used, however, the fact that many
	 *      bins are used will probably lead to a cancellation effect of the errors.
	 *   B: RT_OTS_Update gets called AFTER grain() has completed, so the grain heating
	 *      was calculated using a different version of the OTS fields than the one
	 *      that gets fed into the RT routines (where the actual attenuation of the
	 *      radiation fields by the grain opacity is done). This can lead to an energy
	 *      imbalance, depending on how accurate the convergence of the OTS fields is.
	 *      This is outside the control of the grain code and is therefore NOT checked.
	 *      Rather, the grain code remembers the contribution from the old OTS fields
	 *      (through gv.bin[nd]->BolFlux) and uses that in Check 3. In most models the
	 *      difference will be well below 0.1%, but in AGN type models where OTS continua
	 *      are important, the energy imbalance can be of the order of 0.5% of the grain
	 *      heating (status nov 2001).
	 *   C: Energy conservation for collisional processes is guaranteed by adding in
	 *      (very small) correction terms. These corrections are needed to cover up
	 *      small imperfection in the theory, and cannot be avoided without making the
	 *      already very complex theory even more complex.
	 *   D: Photo-electric heating and collisional cooling can have an important effect
	 *      on the total heating balance of the gas. Both processes depend strongly on
	 *      the grain charge, so assuring proper charge balance is important as well.
	 *      This is tested in Check 2.
	 *
	 * Note 2: for quantum heating it is important to resolve the Maxwell distribution
	 *   of the electrons and ions far enough into the tail that the total amount of
	 *   energy contained in the remaining tail is negligible. If this is not the case,
	 *   the assert at the beginning of the qheat() routine will fail. This is because
	 *   the code filling up the phiTilde array in GrainCollHeating() assumes a value for
	 *   the average particle energy based on a Maxwell distribution going to infinity.
	 *   If the maximum energy used is too low, the assumed average energy would be
	 *   incorrect.
	 *
	 *********************************************************************************/

	/* CHECK 1: does the grain thermal emission conserve energy ? */
	BolFlux = 0.;
	for( i=0; i < rfield.nflux; i++ )
	{
		BolFlux += gv.GrainEmission[i]*rfield.anu[i]*EN1RYD;
	}
	Comparison1 = 0.;
	for( nd=0; nd < gv.nBin; nd++ )
	{
		if( gv.bin[nd]->tedust < gv.bin[nd]->Tsublimat )
			Comparison1 += CONSERV_TOL*gv.bin[nd]->GrainHeat;
		else
			/* for high temperatures the interpolation in dstems
			 * is less accurate, so we have to be more lenient */
			Comparison1 += 10.*CONSERV_TOL*gv.bin[nd]->GrainHeat;
	}
	assert( fabs(BolFlux-gv.GrainHeatSum) < Comparison1 );

	/* CHECK 2: assert charging balance */
	for( nd=0; nd < gv.nBin; nd++ )
	{
		if( gv.bin[nd]->lgChrgConverged )
		{
			double ave = 0.5*(gv.bin[nd]->RateDn+gv.bin[nd]->RateUp);
			assert( fabs(gv.bin[nd]->RateDn-gv.bin[nd]->RateUp) < CONSERV_TOL*ave );
		}
	}

	if( gv.lgDHetOn && gv.lgDColOn )
	{
		/* CHECK 3: calculate the total energy donated to grains, must be balanced by
		 * the energy emitted in thermal radiation plus various forms of gas heating */
		Comparison1 = 0.;
		for( nd=0; nd < gv.nBin; nd++ )
		{
			Comparison1 += gv.bin[nd]->BolFlux;
		}
		/* add in collisional heating of grains by plasma (if positive) */
		Comparison1 += MAX2(gv.GasCoolColl,0.);
		/* add in net amount of chemical energy donated by recombining ions */
		Comparison1 += gv.GrainHeatChem;

		/*              thermal emis        PE effect         gas heating by coll    thermionic emis */
		Comparison2 = gv.GrainHeatSum + heat.heating[0][13] + heat.heating[0][14] + heat.heating[0][25];

		assert( fabs(Comparison1-Comparison2)/Comparison2 < CONSERV_TOL );
	}
#	endif

#	ifdef DEBUG_FUN
	fputs( "<+>GrainMakeDiffuse()\n", debug_fp );
#	endif
	return;
}


/****************************************************************************
 *
 * qheat: driver routine for non-equilibrium heating of grains
 *
 * This routine calls GetProbDistr_LowLimit, GetProbDistr_HighLimit
 * (which do the actual non-equilibrium calculations), and does the
 * subsequent quality control.
 *
 * Written by Peter van Hoof (UK, CITA).
 *
 ****************************************************************************/

#if NEW_CODE

/* this is the new version of the quantum heating code, used starting Cloudy 96 beta 3 */

static void qheat(/*@out@*/ double qtemp[],  /* qtemp[NQGRID] */
		  /*@out@*/ double qprob[],  /* qprob[NQGRID] */
		  /*@out@*/ long int *qnbin,
		  long int nd)
{
	int lgDelta,
	  lgNegRate,
	  lgOK,
	  lgTried;
	long int i;
	QH_Code ErrorCode,
	  ErrorStart;
	double c0,
	  c1,
	  c2,
	  deriv,
	  *dPdlnT/*[NQGRID]*/,
	  fwhm,
	  FwhmRatio,
	  integral,
	  minBracket,
	  maxBracket,
	  new_tmin,
	  NumEvents,
	  oldy,
	  *Phi/*[NC_ELL]*/,
	  *PhiDrv/*[NC_ELL]*/,
	  Tmax,
	  Umax,
	  xx,
	  y;


#	ifdef DEBUG_FUN
	fputs( "<+>qheat()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( gv.bin[nd]->lgQHeat );
	assert( nd >= 0 && nd < gv.nBin );

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "\n >>>>qheat called for grain %s\n", gv.bin[nd]->chDstLab );
	}

	/* >>chng 01 aug 22, allocate space */
	if( (Phi = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)) ) ) == NULL )
		bad_malloc();
	if( (PhiDrv = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)) ) ) == NULL )
		bad_malloc();
	if( (dPdlnT = (double*)MALLOC((size_t)(NQGRID*sizeof(double)) ) ) == NULL )
		bad_malloc();

	xx = integral = 0.;
	c0 = c1 = c2 = 0.;
	lgNegRate = FALSE;
	oldy = 0.;

	/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
	for( i=gv.qnflux-1; i >= 0; i-- )
	{
		/* >>chng 97 jul 17, to summed continuum */
		/* >>chng 00 mar 30, to phiTilde, to keep track of photo-electric effect and collisions, by PvH */
		/* >>chng 01 oct 10, use trapezoidal rule for integrating Phi, reverse direction of integration.
		 * Phi[i] is now integral from exactly rfield.anu[i] to infinity to second-order precision, PvH */
		/* >>chng 01 oct 30, change normalization of Phi, PhiDrv from <unit>/cm^3 -> <unit>/grain, PvH */
		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		double fac = ( i < gv.qnflux-1 ) ? 1./rfield.widflx[i] : 0.;
		/* phiTilde has units events/H/s, y has units events/grain/s/Ryd */
		y = gv.bin[nd]->phiTilde[i]*gv.bin[nd]->cnv_H_pGR*fac;
		/* PhiDrv[i] = (Phi[i+1]-Phi[i])/(anu[i+1]-anu[i]), units events/grain/s/Ryd */
		PhiDrv[i] = -0.5*(y + oldy);
		/* there is a minus sign here because we are integrating from infinity downwards */
		xx -= PhiDrv[i]*(rfield.anu[i+1]-rfield.anu[i]);
		/* Phi has units events/grain/s */
		Phi[i] = xx;

#		ifndef NDEBUG
		/* trapezoidal rule is not needed for integral, this is also second-order correct */
		integral += gv.bin[nd]->phiTilde[i]*gv.bin[nd]->cnv_H_pCM3*rfield.anu[i]*EN1RYD;
#		endif

		/* c<n> has units Ryd^(n+1)/grain/s */
		c0 += Phi[i]*rfield.widflx[i];
		c1 += Phi[i]*rfield.anu[i]*rfield.widflx[i];
		c2 += Phi[i]*POW2(rfield.anu[i])*rfield.widflx[i];

		lgNegRate = lgNegRate || ( gv.bin[nd]->phiTilde[i] < 0. );

		oldy = y;
	}

#	if 0
	{
		char fnam[50];
		FILE *file;

		sprintf(fnam,"Lambda_%2.2ld.asc",nd);
		file = fopen(fnam,"w");
		for( i=0; i < NDEMS; ++i )
			fprintf(file,"%e %e %e\n",
				exp(gv.dsttmp[i]),
				ufunct(exp(gv.dsttmp[i]),nd),
				exp(gv.bin[nd]->dstems[i])*gv.bin[nd]->cnv_H_pGR/EN1RYD);
		fclose(file);

		sprintf(fnam,"Phi_%2.2ld.asc",nd);
		file = fopen(fnam,"w");
		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		for( i=0; i < gv.qnflux; ++i )
			fprintf(file,"%e %e\n", rfield.anu[i],Phi[i]);
		fclose(file);
	}
#	endif

	/* sanity check */
	assert( fabs(gv.bin[nd]->GrainHeat-integral)/gv.bin[nd]->GrainHeat <= CONSERV_TOL );

	/* Tmax is where p(U) will peak (at least in high intensity limit) */
	Tmax = gv.bin[nd]->tedust;
	/* grain enthalpy at peak of p(U), in Ryd */
	Umax = ufunct(Tmax,nd);
	/* y is dln(Lambda)/dlnT at Tmax */
	spldrv(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(Tmax),&y);
	/* deriv is dLambda/dU at Tmax, in 1/grain/s */
	deriv = y*c0/(uderiv(Tmax,nd)*Tmax);
	/* estimated FWHM of probability distribution, in Ryd */
	fwhm = sqrt(8.*LN_TWO*c1/deriv);

	NumEvents = POW2(fwhm)*c0/(4.*LN_TWO*c2);
	FwhmRatio = fwhm/Umax;

	/* >>chng 01 nov 15, change ( NumEvents > MAX_EVENTS2 ) --> ( FwhmRatio < FWHM_RATIO2 ), PvH */
	lgDelta = ( FwhmRatio < FWHM_RATIO2 );
	/* high intensity case is always OK since we will use equilibrium treatment */
	lgOK = lgDelta;

	ErrorStart = QH_OK;

	if( lgDelta ) 
	{
		/* in this case we ignore negative rates, equilibrium treatment is good enough */
		lgNegRate = FALSE;
		ErrorStart = MAX2(ErrorStart,QH_DELTA);
	}

	if( lgNegRate )
		ErrorStart = MAX2(ErrorStart,QH_NEGRATE_FAIL);

	ErrorCode = ErrorStart;

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "   grain heating: %.4e, integral %.4e, total rate %.4e\n",
			 gv.bin[nd]->GrainHeat,integral,Phi[0]);
		fprintf( ioQQQ, "   av grain temp %.4e av grain enthalpy (Ryd) %.4e\n",
			 gv.bin[nd]->tedust,Umax);
		fprintf( ioQQQ, "   fwhm^2/(4ln2*c2/c0): %.4e fwhm (Ryd) %.4e fwhm/Umax %.4e\n",
			 NumEvents,fwhm,FwhmRatio );
	}

	/* these two variables will bracket qtmin, they should only be needed during the initial search phase */
	minBracket = GRAIN_TMIN;
	maxBracket = gv.bin[nd]->tedust;

	/* >>chng 02 jan 30, introduced lgTried to avoid running GetProbDistr_HighLimit over and over..., PvH */
	lgTried = FALSE;

	/* if average number of multiple photon events is too high, lgOK is set to TRUE */
	/* if negative rate was found, bail out */
/* 	for( i=0; i < LOOP_MAX && !lgOK && !lgNegRate; i++ ) */
	for( i=0; i < LOOP_MAX && !lgOK; i++ )
	{
		if( gv.bin[nd]->qtmin >= gv.bin[nd]->tedust )
			gv.bin[nd]->qtmin = 0.7*gv.bin[nd]->tedust;
		gv.bin[nd]->qtmin = MAX2(gv.bin[nd]->qtmin,GRAIN_TMIN);

		assert( minBracket <= gv.bin[nd]->qtmin && gv.bin[nd]->qtmin <= maxBracket );

		ErrorCode = ErrorStart;

		/* >>chng 01 nov 15, add ( FwhmRatio >= FWHM_RATIO ), PvH */
		if( FwhmRatio >= FWHM_RATIO && NumEvents <= MAX_EVENTS )
		{
			GetProbDistr_LowLimit(nd,Umax,fwhm,Phi,PhiDrv,qtemp,qprob,dPdlnT,qnbin,&new_tmin,&ErrorCode);

			/* >>chng 02 jan 07, various changes to improve convergence for very cold grains, PvH */
			if( ErrorCode == QH_DELTA_FAIL && fwhm < Umax && !lgTried )
			{
				double dummy;

				/* this situation can mean two things: either the photon rate is so high that
				 * the code needs too many steps to integrate the probability distribution,
				 * or alternatively, tmin is far too low and the code needs too many steps
				 * to overcome the rising side of the probability distribution.
				 * So we call GetProbDistr_HighLimit first to determine if the former is the
				 * case; if that fails then the latter must be true and we reset QH_DELTA_FAIL */
				ErrorCode = MAX2(ErrorStart,QH_ANALYTIC_LOW);
				/* use dummy to avoid losing estimate for new_tmin from GetProbDistr_LowLimit */
				GetProbDistr_HighLimit(nd,Umax,fwhm,qtemp,qprob,dPdlnT,qnbin,&dummy,&ErrorCode);
				if( ErrorCode > QH_RETRY )
				{
					ErrorCode = QH_DELTA_FAIL;
					lgTried = TRUE;
				}
			}

			if( ErrorCode == QH_DELTA_FAIL || ErrorCode == QH_LOW_FAIL )
			{
				assert( new_tmin >= minBracket );
				/* >>chng 02 jan 30, set minBracket in both cases, PvH */
				/* if( ErrorCode == QH_LOW_FAIL && new_tmin < maxBracket ) */
				if( new_tmin < maxBracket )
					minBracket = new_tmin;
				new_tmin = MIN2(new_tmin*sqrt(DEF_FAC),sqrt(new_tmin*maxBracket));
			}
			else if( ErrorCode == QH_HIGH_FAIL )
			{
				assert( new_tmin <= maxBracket );
				maxBracket = new_tmin;
				new_tmin = MAX3(new_tmin/DEF_FAC,sqrt(new_tmin*minBracket),GRAIN_TMIN);
			}
			else
			{
				if( new_tmin <= minBracket )
					new_tmin = sqrt(gv.bin[nd]->qtmin*minBracket);
				if( new_tmin >= maxBracket )
					new_tmin = sqrt(gv.bin[nd]->qtmin*maxBracket);
			}
		}
		else
		{
			GetProbDistr_HighLimit(nd,Umax,fwhm,qtemp,qprob,dPdlnT,qnbin,&new_tmin,&ErrorCode);
		}

		gv.bin[nd]->qtmin = new_tmin;

		lgOK = ( ErrorCode < QH_RETRY );

		if( ErrorCode > QH_FATAL )
			break;

		if( trace.lgTrace && trace.lgDustBug ) 
		{
			fprintf( ioQQQ, "   GetProbDistr returns code %d\n", ErrorCode );
			if( !lgOK )
			{
				fprintf( ioQQQ, " >>>>qheat trying another iteration, qtmin bracket %.4e %.4e\n",
					 minBracket,maxBracket );
			}
		}
	}

	/* as a last resort try the analytic approximation */
	if( !lgOK && !lgTried && FwhmRatio >= FWHM_RATIO && NumEvents <= MAX_EVENTS )
	{
		ErrorCode = MAX2(ErrorStart,QH_ANALYTIC_LOW);
		GetProbDistr_HighLimit(nd,Umax,fwhm,qtemp,qprob,dPdlnT,qnbin,&new_tmin,&ErrorCode);

		gv.bin[nd]->qtmin = new_tmin;
		lgOK = ( ErrorCode < QH_RETRY );
	}

	if( nzone == 1 )
		gv.bin[nd]->qtmin_zone1 = gv.bin[nd]->qtmin;

	gv.bin[nd]->lgUseQHeat = ( lgOK && !lgDelta );
	gv.bin[nd]->lgEverQHeat = ( gv.bin[nd]->lgEverQHeat || gv.bin[nd]->lgUseQHeat );

	if( lgOK )
	{
		if( trace.lgTrace && trace.lgDustBug )
			fprintf( ioQQQ, " >>>>qheat converged with code: %d\n", ErrorCode );
	}
	else
	{
		*qnbin = 0;
		++gv.bin[nd]->QHeatFailures;
		fprintf( ioQQQ, " >>>>qheat did not converge grain %s in zone %ld, error code = %d\n",
			 gv.bin[nd]->chDstLab,nzone,ErrorCode );		
	}

	if( gv.QHPunchFile != NULL && ( IterCnt.lgLastIt || !gv.lgQHPunLast ) ) 
	{
		fprintf( gv.QHPunchFile, "\nDust Temperature Distribution: grain %s zone %ld\n",
			 gv.bin[nd]->chDstLab,nzone );

		fprintf( gv.QHPunchFile, "Equilibrium temperature: %.2f\n", gv.bin[nd]->tedust );

		if( gv.bin[nd]->lgUseQHeat ) 
		{
			/* >>chng 01 oct 09, remove qprob from output, it depends on step size, PvH */
			fprintf( gv.QHPunchFile, "Number of bins: %ld\n", *qnbin );
			fprintf( gv.QHPunchFile, "  Tgrain      dP/dlnT\n" );
			for( i=0; i < *qnbin; i++ ) 
			{
				fprintf( gv.QHPunchFile, "%.4e %.4e\n", qtemp[i],dPdlnT[i] );
			}
		}
		else 
		{
			fprintf( gv.QHPunchFile, "**** quantum heating was not used\n" );
		}
	}

	free ( Phi );
	free ( PhiDrv );
	free ( dPdlnT );
 
#	ifdef DEBUG_FUN
	fputs( " <->qheat()\n", debug_fp );
#	endif
	return;
}

/*******************************************************************************************
 *
 * GetProbDistr_LowLimit: main routine for calculating non-equilibrium heating of grains
 *
 * This routine implements the algorithm outlined in:
 * >>refer  Guhathakurtha & Draine, 1989, ApJ, 345, 230
 *
 * The original (fortran) version of the code was written by Kevin Volk.
 *
 * Heavily modified and adapted for new style grains by Peter van Hoof.
 *
 *******************************************************************************************/

static void GetProbDistr_LowLimit(long int nd,
				  double Umax,
				  double fwhm,
				  /*@in@*/ double Phi[],     /* Phi[NQGRID]      */
				  /*@in@*/ double PhiDrv[],  /* PhiDrv[NQGRID]   */
				  /*@out@*/ double qtemp[],  /* qtemp[NQGRID]    */
				  /*@out@*/ double qprob[],  /* qprob[NQGRID]    */
				  /*@out@*/ double dPdlnT[], /* dPdlnT[NQGRID]   */
				  /*@out@*/ long int *qnbin,
				  /*@out@*/ double *new_tmin,
				  /*@out@*/ QH_Code *ErrorCode)
{
	long int j,
	  k,
	  l,
	  nbad,
	  nbin,
	  nok;
	double dCool,
	  delta_x,
	  delta_y,
	  delu[NQGRID],
	  dlnLdlnT,
	  dlnpdlnU,
	  fac = 0.,
	  Lambda[NQGRID],
	  NextStep,
	  p[NQGRID],
	  /* >>chng 01 aug 22, from array to pointer with MALLOC */
	  qtmin1, 
	  RadCooling,
	  sum,
	  u1[NQGRID],
	  y;


#	ifdef DEBUG_FUN
	fputs( "<+>GetProbDistr_LowLimit()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( nd >= 0 && nd < gv.nBin );

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "   GetProbDistr_LowLimit called for grain %s\n", gv.bin[nd]->chDstLab );
		fprintf( ioQQQ, "    got qtmin1 %.4e\n", gv.bin[nd]->qtmin);
	}

	qtmin1 = gv.bin[nd]->qtmin;
	qtemp[0] = qtmin1;
	/* u1 holds enthalpy in Ryd/grain */
	u1[0] = ufunct(qtemp[0],nd);
	/* >>chng 00 mar 22, taken out factor 4, factored in hden and dstAbund
	 * interpolate in dstems array instead of integrating explicitly, by PvH */
	splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(qtemp[0]),&y);
	/* Lambda holds the radiated energy for grains in this bin, in Ryd/s/grain */
	Lambda[0] = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;
	/* set up first step of integration */
	/* >>chng 01 nov 14, set to 2.*Lambda[0]/Phi[0] instead of u1[0],
	 * this choice assures that p[1] doesn't make a large jump from p[0], PvH */
	delu[0] = 2.*Lambda[0]/Phi[0];
	p[0] = PROB_CUTOFF_LO;
	dPdlnT[0] = p[0]*qtemp[0]*uderiv(qtemp[0],nd);
	RadCooling = 0.5*p[0]*Lambda[0]*delu[0];
	NextStep = 0.01*Lambda[0]/Phi[0];
	nbad = 0;
	k = 0;

	*qnbin = 0;
	*new_tmin = qtmin1;

	spldrv(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(qtemp[0]),&dlnLdlnT);
	dlnpdlnU = u1[0]*Phi[0]/Lambda[0] - dlnLdlnT*u1[0]/(qtemp[0]*uderiv(qtemp[0],nd));
	if( dlnpdlnU < 0. )
	{
		/* if p[k] starts falling immediately, 
		 * qtmin1 was too high and convergence is impossible */
		*ErrorCode = MAX2(*ErrorCode,QH_HIGH_FAIL);

#		ifdef DEBUG_FUN
		fputs( " <->GetProbDistr_LowLimit()\n", debug_fp );
#		endif
		return;
	}

	/* NB NB -- every break in this loop should set *ErrorCode !! */
	for( l=0; l < MAX_LOOP; ++l )
	{
		/* on the rising slope the derivative method is more reliable,
		 * on the falling slope the regular methos id more reliable */
		double RelError = TryDoubleStep(u1,delu,p,qtemp,Lambda,Phi,PhiDrv,NextStep,&dCool,k,nd);

		/* estimate new stepsize */
		if( RelError > QHEAT_TOL )
		{
			nbad += 2;

			/* step is rejected, decrease stepsize and try again */
			NextStep *= sqrt(0.9*QHEAT_TOL/RelError);

			/* stepsize too small, this is a pathological condition */
			if( NextStep < 5.*DBL_EPSILON*u1[k] )
			{
				*ErrorCode = MAX2(*ErrorCode,QH_STEP_FAIL);
				break;
			}

			continue;
		}
		else
		{
			/* step was OK, adjust stepsize */
			k += 2;

			NextStep *= MIN2(pow(0.9*QHEAT_TOL/RelError,1./3.),4.);
			NextStep = MIN2(NextStep,Lambda[k]/Phi[0]);
		}

		dPdlnT[k-1] = p[k-1]*qtemp[k-1]*uderiv(qtemp[k-1],nd);
		dPdlnT[k] = p[k]*qtemp[k]*uderiv(qtemp[k],nd);

		RadCooling += dCool;

/*  		printf(" k %ld T[k] %.6e U[k] %.6e p[k] %.6e dPdlnT[k] %.6e\n",k-1,qtemp[k-1],u1[k-1],p[k-1],dPdlnT[k-1]); */
/*  		printf(" k %ld T[k] %.6e U[k] %.6e p[k] %.6e dPdlnT[k] %.6e\n",k,qtemp[k],u1[k],p[k],dPdlnT[k]); */

		/* if qtmin is far too low, p[k] can become extremely large, exceeding
		 * even double precision range. the following check should prevent overflows */
		/* >>chng 01 nov 07, sqrt(DBL_MAX) -> sqrt(DBL_MAX/100.) so that sqrt(p[k]*p[k+1]) is safe */
		if( p[k] > sqrt(DBL_MAX/100.) ) 
		{
			*ErrorCode = MAX2(*ErrorCode,QH_LOW_FAIL);
			break;

#if 0
			/* >>chng 01 apr 21, change DBL_MAX -> DBL_MAX/100. to work around
			 * a bug in the Compaq C compiler V6.3-025 when invoked with -fast, PvH */
			for( j=0; j <= k; j++ ) 
			{
				p[j] /= DBL_MAX/100.;
				dPdlnT[j] /= DBL_MAX/100.;
			}
			RadCooling /= DBL_MAX/100.;
#endif
		}

		/* this may catch a bug in the Compaq C compiler V6.3-025
		 * if this gets triggered, try compiling with -ieee */
		assert( p[k] > 0. && dPdlnT[k] > 0. && RadCooling > 0. );

		/* do a check if there will be enough room to store results */
		/* >>chng 02 jan 30, do this test only every NQTEST steps, PvH */
		/* if( k >= MIN2(NQTEST,NQGRID-2) ) */
		if( k > 0 && k%NQTEST == 0 )
		{
			/* this is a lower limit for the total width of the probability distr */
			/* >>chng 02 jan 30, corrected calculation of wid and avStep, PvH */
			double wid = (sqrt(-log(PROB_CUTOFF_LO)/(4.*LN_TWO)) +
				      sqrt(-log(PROB_CUTOFF_HI)/(4.*LN_TWO)))*fwhm/Umax;
			double avStep = log(u1[k]/u1[0])/(double)k;
			/* make an estimate for the number of steps needed */
			/* >>chng 02 jan 30, included factor 1.5 because stepsize increases near peak, PvH */
			if( wid/avStep > 1.5*(double)NQGRID )
			{
				*ErrorCode = MAX2(*ErrorCode,QH_DELTA_FAIL);
				break;
			}
		}

		/* if we run out of room to store results, do regular break
		 * the code below will sort out if integration is valid or not */
		if( k >= NQGRID-2 )
		{
			*ErrorCode = MAX2(*ErrorCode,QH_ARRAY_FAIL);
			break;
		}

		/* force thermal equilibrium of the grains */
		fac = RadCooling*gv.bin[nd]->cnv_GR_pCM3*EN1RYD/gv.bin[nd]->GrainHeat;

		/* this is regular stop criterion */
		if( dPdlnT[k] < dPdlnT[k-1] && dPdlnT[k]/fac < PROB_CUTOFF_HI )
		{
			break;
		}
	}

	if( l == MAX_LOOP )
		*ErrorCode = MAX2(*ErrorCode,QH_LOOP_FAIL);

	nok = k;
	nbin = k+1;

	/* there are insufficient bins to attempt rebinning */
	if( *ErrorCode < QH_NO_REBIN && nbin <= NQMIN ) 
		*ErrorCode = MAX2(*ErrorCode,QH_NBIN_FAIL);

	if( *ErrorCode > QH_NO_REBIN )
	{
#		ifdef DEBUG_FUN
		fputs( " <->GetProbDistr_LowLimit()\n", debug_fp );
#		endif
		return;
	}

	for( j=0; j < nbin; j++ )
	{
		p[j] /= fac;
		dPdlnT[j] /= fac;
	}
	RadCooling /= fac;

	/* >>chng 01 may 11, rebin the quantum heating results
	 *
	 * for grains in intense radiation fields, the code needs high resolution for stability
	 * and therefore produces lots of small bins, even though the grains never make large
	 * excurions from the equilibrium temperature; adding in the resulting spectra in RTDiffuse
	 * takes up an excessive amount of CPU time where most CPU is spent on grains for which
	 * the quantum treatment matters least, and moreover on temperature bins with very low
	 * probability; rebinning the results on a coarser grid should help reduce the overhead */

	nbin = RebinQHeatResults(nd,nbin,p,qtemp,qprob,dPdlnT,u1,delu,Lambda,ErrorCode);

	/* >>chng 01 jul 13, add fail-safe for failure in RebinQHeatResults */
	if( nbin == 0 )
	{
#		ifdef DEBUG_FUN
		fputs( " <->GetProbDistr_LowLimit()\n", debug_fp );
#		endif
		return;
	}

	*qnbin = nbin;

	*new_tmin = 0.;
	for( j=0; j < nbin; j++ ) 
	{
		if( dPdlnT[j] < PROB_CUTOFF_LO ) 
		{
			*new_tmin = qtemp[j];
		}
		else 
		{
			if( j == 0 ) 
			{
				if( dPdlnT[j] > SAFETY*PROB_CUTOFF_LO )
					*ErrorCode = MAX2(*ErrorCode,QH_BOUND_FAIL);

				/* this extrapolation can be unreliable, use something else ?? */

				if( dPdlnT[0] < 0.999*dPdlnT[NSTARTUP] ) 
				{
					delta_x = log(qtemp[NSTARTUP]/qtemp[0]);
					delta_y = log(dPdlnT[NSTARTUP]/dPdlnT[0]);
					delta_x *= log(PROB_CUTOFF_LO/dPdlnT[0])/delta_y;
					*new_tmin = qtemp[0]*exp(delta_x);
				}
			}
			break;
		}
	}
	*new_tmin = MAX3(*new_tmin,qtmin1/DEF_FAC,GRAIN_TMIN);

	/* >>chng 02 jan 30, prevent excessive looping when prob distribution simply won't fit, PvH */
	if( dPdlnT[nbin-1] > SAFETY*PROB_CUTOFF_HI )
	{
		if( *ErrorCode == QH_ARRAY_FAIL )
		{
			/* this indicates that low end was OK, but we ran out of room
			 * to store the high end -> try GetProbDistr_HighLimit instead */
			*ErrorCode = MAX2(*ErrorCode,QH_DELTA_FAIL);
		}
		else
		{
			*ErrorCode = MAX2(*ErrorCode,QH_BOUND_FAIL);
		}
	}

	sum = 0.;
	for( j=0; j < nbin; j++ )
	{
		sum += qprob[j];
	}

	/* the fact that the probability normalization fails probably indicates
	 * that the distribution is too close to a delta function to resolve */
	if( fabs(sum-1.) > PROB_TOL )
		*ErrorCode = MAX2(*ErrorCode,QH_CONV_FAIL);
		
	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    zone %ld %s nbin %ld nok %ld nbad %ld total prob %.4e new_tmin %.4e\n",
			 nzone,gv.bin[nd]->chDstLab,nbin,nok,nbad,sum,*new_tmin );
	}

#	ifdef DEBUG_FUN
	fputs( " <->GetProbDistr_LowLimit()\n", debug_fp );
#	endif
	return;
}


/* try two consecutive integration steps using stepsize "step/2." (yielding p[k]),
 * and also one double integration step using stepsize "step" (yielding p2k).
 * the difference fabs(p2k-p[k])/(3.*p[k]) can be used to estimate the relative
 * accuracy of p[k] and will be used to adapt the stepsize to an optimal value */
static double TryDoubleStep(double u1[],
			    double delu[],
			    double p[],
			    double qtemp[],
			    double Lambda[],
			    double Phi[],
			    double PhiDrv[],
			    double step,
			    /*@out@*/ double *cooling,
			    long index,
			    long nd)
{
	long i,
	  j,
	  jlo,
	  k=-1000;
	double bval_jk,
	  cooling2,
	  p2k,
	  p_max,
	  RelErrCool,
	  RelErrPk,
	  sum,
	  sum2 = -DBL_MAX,
	  trap1,
	  trap12 = -DBL_MAX,
	  trap2,
	  uhi,
	  ulo,
	  umin,
	  y;

#	ifdef DEBUG_FUN
	fputs( "<+>TryDoubleStep()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( index >= 0 && index < NQGRID-2 && nd >= 0 && nd < gv.nBin && step > 0. );

	ulo = rfield.anu[0];
	/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
	uhi = rfield.anu[gv.qnflux-1];

	/* >>chng 01 nov 21, skip initial bins if they have very low probability */
	p_max = 0.;
	for( i=0; i <= index; i++ )
		p_max = MAX2(p_max,p[i]);
	jlo = 0;
	while( p[jlo] < PROB_CUTOFF_LO*p_max )
		jlo++;

	for( i=1; i <= 2; i++ )
	{
		long ipLo = 0;
		/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
		long ipHi = gv.qnflux-1;
		k = index + i;
		delu[k] = step/2.;
		u1[k] = u1[k-1] + delu[k];
		qtemp[k] = inv_ufunct(u1[k],nd);
		splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(qtemp[k]),&y);
		Lambda[k] = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;

		sum = sum2 = 0.;
		trap1 = trap2 = trap12 = 0.;

		/* this loop uses up a large fraction of the total CPU time, it should be well optimized */
		for( j=jlo; j < k; j++ )
		{
			umin = u1[k] - u1[j];

			if( umin >= uhi ) 
			{
				/* for increasing j, umin will decrease monotonically. If ( umin > uhi ) holds for
				 * the current value of j, it must have held for the previous value as well. Hence
				 * both trap1 and trap2 are zero at this point and we would only be adding zero
				 * to sum. Therefore we skip this step to save CPU time */
				continue;
			}
			else if( umin > ulo )
			{
				/* do a bisection search such that rfield.anu[ipLo] <= umin < rfield.anu[ipHi]
				 * ipoint doesn't always give the correct index into anu (it works for AnuOrg)
				 * bisection search is also faster, which is important here to save CPU time */
				ipLo = 0;
				/* no need to reset ipHi, since umin is monotonically decreasing,
				 * the previous value is a valid upper limit, thus saving CPU time */
				while( ipHi-ipLo > 1 )
				{
					long ipMd = (ipLo+ipHi)/2;
					if( rfield.anu[ipMd] > (float)umin )
						ipHi = ipMd;
					else
						ipLo = ipMd;
				}
				bval_jk = Phi[ipLo] + (umin - rfield.anu[ipLo])*PhiDrv[ipLo];
			}
			else
			{
				bval_jk = Phi[0];
			}

			/* these two quantities are needed to take double step from index -> index+2 */
			trap12 = trap1;
			sum2 = sum;

			/* bval_jk*gv.bin[nd]->cnv_CM3_pGR is the total excitation rate from j to k and
			 * higher due to photon absorptions and particle collisions, it already implements
			 * Eq. 2.17 of Guhathakurtha & Draine, in events/grain/s */
			/* >>chng 00 mar 27, factored in hden (in Phi), by PvH */
			/* >>chng 00 mar 29, add in contribution due to particle collisions, by PvH */
			/* >>chng 01 mar 30, moved multiplication of bval_jk with gv.bin[nd]->cnv_CM3_pGR
			 *   out of loop, PvH */
			trap2 = p[j]*bval_jk;
			/* Trapezoidal rule, multiplication with factor 0.5 is done outside loop */
			sum += (trap1 + trap2)*delu[j];
			trap1 = trap2;
		}

		/* >>chng 00 mar 27, multiplied with delu, by PvH */
		/* >>chng 00 apr 05, taken out Lambda[0], improves convergence at low end dramatically!, by PvH */
		/* qprob[k] = sum*gv.bin[nd]->cnv_CM3_pGR*delu[k]/(Lambda[k] - Lambda[0]); */
		/* this expression includes factor 0.5 from trapezoidal rule above */
		/* p[k] = 0.5*(sum + (trap1 + p[k]*Phi[0])*delu[k])/Lambda[k] */
		p[k] = (sum + trap1*delu[k])/(2.*Lambda[k] - Phi[0]*delu[k]);

		assert( p[k] > 0. );
	}

	/* this is estimate for p[k] using one double step of size "step" */
	p2k = (sum2 + trap12*step)/(2.*Lambda[k] - Phi[0]*step);

	assert( p2k > 0. );

	RelErrPk = fabs(p2k-p[k])/p[k];

	/* this is radiative cooling due to the two probability bins we just added */
	/* simple trapezoidal rule will not do here, RelErrCool would never converge */
	*cooling = log_integral(u1[k-2],p[k-2]*Lambda[k-2],u1[k-1],p[k-1]*Lambda[k-1]);
	*cooling += log_integral(u1[k-1],p[k-1]*Lambda[k-1],u1[k],p[k]*Lambda[k]);

	/* same as cooling, but now for double step of size "step" */
	cooling2 = log_integral(u1[k-2],p[k-2]*Lambda[k-2],u1[k],p2k*Lambda[k]);

	/* p[0] is not reliable, so ignore convergence test on cooling on first step */
	RelErrCool = ( index > 0 ) ? fabs(cooling2-(*cooling))/(*cooling) : 0.;

/*  	printf( " TryDoubleStep p[k-1] %.4e p[k] %.4e p2k %.4e\n",p[k-1],p[k],p2k ); */

#	ifdef DEBUG_FUN
	fputs( " <->TryDoubleStep()\n", debug_fp );
#	endif
	/* error scales as O(step^3), so this is relative accuracy of p[k] or cooling */
	return MAX2(RelErrPk,RelErrCool)/3.;
}


/* calculate logarithmic integral from (x1,y1) to (x2,y2) */
static double log_integral(double x1,
			   double y1,
			   double x2,
			   double y2)
{
	double eps,
	  result,
	  xx;

#	ifdef DEBUG_FUN
	fputs( "<+>log_integral()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( x1 > 0. && y1 > 0. && x2 > 0. && y2 > 0. );

	xx = log(x2/x1);
	eps = log((x2/x1)*(y2/y1));
	if( fabs(eps) < 1.e-4 )
	{
		result = x1*y1*xx*((eps/6. + 0.5)*eps + 1.);
	}
	else
	{
		result = (x2*y2 - x1*y1)*xx/eps;
	}

#	ifdef DEBUG_FUN
	fputs( " <->log_integral()\n", debug_fp );
#	endif
	return result;
}


/* rebin the quantum heating results to speed up RTDiffuse */
static long RebinQHeatResults(long nd,
			      long oldnbin,
			      double p[],
			      double qtemp[],
			      double qprob[],
			      double dPdlnT[],
			      double u1[],
			      double delu[],
			      double Lambda[],
			      QH_Code *ErrorCode)
{
	long i,
	  newnbin;
	double fac,
	  mul_fac,
	  new_delu[NQGRID],
	  new_dPdlnT[NQGRID],
	  new_Lambda[NQGRID],
	  new_p[NQGRID],
	  new_qprob[NQGRID],
	  new_qtemp[NQGRID],
	  new_u1[NQGRID],
	  PP1,
	  PP2,
	  RadCooling,
	  T1,
	  T2,
	  Ucheck;

#	ifdef DEBUG_FUN
	fputs( "<+>RebinQHeatResults()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( nd >= 0 && nd < gv.nBin );
	assert( oldnbin > 0 && oldnbin <= NQGRID );

	/* leading entries may have become very small or zero -> skip */
	for( i=0; i < oldnbin && dPdlnT[i] < PROB_CUTOFF_LO; i++ ) {}

	newnbin = 0;

	T1 = qtemp[i];
	PP1 = p[i];

	mul_fac = MIN2(QT_RATIO,pow(qtemp[oldnbin-1]/qtemp[i],1./(2.*NQMIN)));

	Ucheck = u1[i];
	RadCooling = 0.;

	while( i < oldnbin-1 )
	{
		int lgDone= FALSE;
		double s0 = 0.;
		double s1 = 0.;
		double wid = 0.;
		double xx,y;

		T2 = T1*mul_fac;

		do 
		{
			double p1,p2,L1,L2,uu1,uu2,frac,slope;
			if( qtemp[i] <= T1 && T1 <= qtemp[i+1] )
			{
				uu1 = ufunct(T1,nd);
				frac = log(T1/qtemp[i]);
				slope = log(p[i+1]/p[i])/log(qtemp[i+1]/qtemp[i]);
				p1 = p[i]*exp(frac*slope);
				slope = log(Lambda[i+1]/Lambda[i])/log(qtemp[i+1]/qtemp[i]);
				L1 = Lambda[i]*exp(frac*slope);
			}
			else
			{
				uu1 = u1[i];
				p1 = p[i];
				L1 = Lambda[i];
			}
			if( qtemp[i] <= T2 && T2 <= qtemp[i+1] )
			{
				uu2 = ufunct(T2,nd);
				frac = log(T2/qtemp[i]);
				slope = log(p[i+1]/p[i])/log(qtemp[i+1]/qtemp[i]);
				p2 = p[i]*exp(frac*slope);
				slope = log(Lambda[i+1]/Lambda[i])/log(qtemp[i+1]/qtemp[i]);
				L2 = Lambda[i]*exp(frac*slope);
				lgDone = TRUE;
			}
			else
			{
				uu2 = u1[i+1];
				p2 = p[i+1];
				L2 = Lambda[i+1];
				/* >>chng 01 nov 15, this caps the range in p(U) integrated in one bin
				 * it helps avoid spurious QH_BOUND_FAIL's when flank is very steep, PvH */
				if( MAX2(p2,PP1)/MIN2(p2,PP1) > sqrt(SAFETY) )
				{
					lgDone = TRUE;
					T2 = qtemp[i+1];
				}
				++i;
			}
			PP2 = p2;
			wid += uu2 - uu1;
/*  			s0 += 0.5*(p1 + p2)*(uu2 - uu1); */
			s0 += log_integral(uu1,p1,uu2,p2);
			s1 += log_integral(uu1,p1*L1,uu2,p2*L2);

		} while( i < oldnbin-1 && ! lgDone );

		/* >>chng 01 nov 14, if T2 == qtemp[oldnbin-1], the code will try another iteration
		 * break here to avoid zero divide, the assert on Ucheck tests if we are really finished */
		/* >>chng 01 dec 04, change ( s0 == 0. ) to ( s0 <= 0. ), PvH */
		if( s0 <= 0. )
			break;

		new_qprob[newnbin] = s0;
		new_Lambda[newnbin] = s1/s0;
		xx = log(new_Lambda[newnbin]*EN1RYD*gv.bin[nd]->cnv_GR_pH);
		splint(gv.bin[nd]->dstems,gv.dsttmp,gv.bin[nd]->dstslp,NDEMS,xx,&y);
		new_qtemp[newnbin] = exp(y);
		new_u1[newnbin] = ufunct(new_qtemp[newnbin],nd);
		new_delu[newnbin] = wid;
		new_p[newnbin] = new_qprob[newnbin]/new_delu[newnbin];
		new_dPdlnT[newnbin] = new_p[newnbin]*new_qtemp[newnbin]*uderiv(new_qtemp[newnbin],nd);

		Ucheck += wid;
		RadCooling += new_qprob[newnbin]*new_Lambda[newnbin];

		T1 = T2;
		PP1 = PP2;
		++newnbin;
	}

	/* >>chng 01 jul 13, add fail-safe */
	if( newnbin < NQMIN )
	{
		*ErrorCode = MAX2(*ErrorCode,QH_REBIN_FAIL);

#		ifdef DEBUG_FUN
		fputs( " <->RebinQHeatResults()\n", debug_fp );
#		endif
		return 0;
	}
		
	fac = RadCooling*EN1RYD*gv.bin[nd]->cnv_GR_pCM3/gv.bin[nd]->GrainHeat;

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "     RebinQHeatResults found tol1 %.4e tol2 %.4e\n",
			 fabs(u1[oldnbin-1]/Ucheck-1.), fabs(fac-1.) );
	}

	/* do quality control */
	assert( fabs(u1[oldnbin-1]/Ucheck-1.) < 1.e-6 );

	if( fabs(fac-1.) > CONSERV_TOL )
		*ErrorCode = MAX2(*ErrorCode,QH_CONV_FAIL);

	for( i=0; i < newnbin; i++ )
	{
		/* renormalize the distribution to assure energy conservation */
		p[i] = new_p[i]/fac;
		qtemp[i] = new_qtemp[i];
		qprob[i] = new_qprob[i]/fac;
		dPdlnT[i] = new_dPdlnT[i]/fac;
		u1[i] = new_u1[i];
		delu[i] = new_delu[i];
		Lambda[i] = new_Lambda[i];

		/* sanity checks */
		assert( qtemp[i] > 0. && qprob[i] > 0. );
	}

#	ifdef DEBUG_FUN
	fputs( " <->RebinQHeatResults()\n", debug_fp );
#	endif
	return newnbin;
}


/* calculate approximate probability distribution in high intensity limit */
static void GetProbDistr_HighLimit(long nd,
				   double Umax,
				   double fwhm,
				   /*@out@*/double qtemp[],
				   /*@out@*/double qprob[],
				   /*@out@*/double dPdlnT[],
				   /*@out@*/long *qnbin,
				   /*@out@*/double *new_tmin,
				   /*@out@*/QH_Code *ErrorCode)
{
	long i,
	  nbin;
	double c1,
	  c2,
	  delu[NQGRID],
	  fac,
	  L1,
	  L2,
	  Lambda[NQGRID],
	  mul_fac,
	  p[NQGRID],
	  p1,
	  p2,
	  RadCooling,
	  sum,
	  u1[NQGRID],
	  T1,
	  T2,
	  Tlo,
	  Thi,
	  uu1,
	  uu2,
	  xx,
	  y;

#	ifdef DEBUG_FUN
	fputs( "<+>GetProbDistr_HighLimit()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "   GetProbDistr_HighLimit called for grain %s\n", gv.bin[nd]->chDstLab );
	}

	c1 = sqrt(4.*LN_TWO/PI)/fwhm*exp(-POW2(fwhm/Umax)/(16.*LN_TWO));
	c2 = 4.*LN_TWO*POW2(Umax/fwhm);

	Tlo = MAX2(inv_ufunct(Umax*exp(-fwhm/Umax*sqrt(-log(PROB_CUTOFF_LO)/(4.*LN_TWO))),nd),GRAIN_TMIN);
	Thi = inv_ufunct(Umax*exp(fwhm/Umax*sqrt(-log(PROB_CUTOFF_HI)/(4.*LN_TWO))),nd);

	nbin = 0;

	T1 = Tlo;
	uu1 = ufunct(T1,nd);
	p1 = c1*exp(-c2*POW2(log(uu1/Umax)));
	splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(T1),&y);
	L1 = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;

	mul_fac = MIN2(QT_RATIO,pow(Thi/Tlo,1./(2.*NQMIN)));

	sum = 0.;
	RadCooling = 0.;

	do 
	{
		double s0,s1,wid;

		T2 = T1*mul_fac;
		uu2 = ufunct(T2,nd);
		p2 = c1*exp(-c2*POW2(log(uu2/Umax)));
		splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,log(T2),&y);
		L2 = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;

		wid = uu2 - uu1;
		s0 = log_integral(uu1,p1,uu2,p2);
		s1 = log_integral(uu1,p1*L1,uu2,p2*L2);

		qprob[nbin] = s0;
		Lambda[nbin] = s1/s0;
		xx = log(Lambda[nbin]*EN1RYD*gv.bin[nd]->cnv_GR_pH);
		splint(gv.bin[nd]->dstems,gv.dsttmp,gv.bin[nd]->dstslp,NDEMS,xx,&y);
		qtemp[nbin] = exp(y);
		u1[nbin] = ufunct(qtemp[nbin],nd);
		delu[nbin] = wid;
		p[nbin] = qprob[nbin]/delu[nbin];
		dPdlnT[nbin] = p[nbin]*qtemp[nbin]*uderiv(qtemp[nbin],nd);

		sum += qprob[nbin];
		RadCooling += qprob[nbin]*Lambda[nbin];

		T1 = T2;
		uu1 = uu2;
		p1 = p2;
		L1 = L2;

		++nbin;

	} while( T2 < Thi && nbin < NQGRID );

	fac = RadCooling*EN1RYD*gv.bin[nd]->cnv_GR_pCM3/gv.bin[nd]->GrainHeat;

	for( i=0; i < nbin; ++i )
	{
		qprob[i] /= fac;
		dPdlnT[i] /= fac;
	}

	*qnbin = nbin;
	*new_tmin = qtemp[0];
	*ErrorCode = MAX2(*ErrorCode,QH_ANALYTIC);

	/* do quality control */
	if( fabs(sum/fac-1.) > PROB_TOL )
		*ErrorCode = MAX2(*ErrorCode,QH_CONV_FAIL);

	if( dPdlnT[0] > SAFETY*PROB_CUTOFF_LO || dPdlnT[nbin-1] > SAFETY*PROB_CUTOFF_HI )
		*ErrorCode = MAX2(*ErrorCode,QH_BOUND_FAIL);
		
	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "     GetProbDistr_HighLimit found tol1 %.4e tol2 %.4e\n",
			 fabs(sum-1.), fabs(sum/fac-1.) );
		fprintf( ioQQQ, "    zone %ld %s nbin %ld total prob %.4e new_tmin %.4e\n",
			 nzone,gv.bin[nd]->chDstLab,nbin,sum/fac,*new_tmin );
	}

#	ifdef DEBUG_FUN
	fputs( "<+>GetProbDistr_HighLimit()\n", debug_fp );
#	endif
	return;
}

#else

/* this is the old version of the quantum heating code, used in Cloudy 96 beta 1 and 2 */

static void qheat(/*@out@*/ double qtemp[],  /* qtemp[NQGRID] */
		  /*@out@*/ double qprob[],  /* qprob[NQGRID] */
		  /*@out@*/ long int *qnbin,
		  long int nd)
{
	int lgBigChange=TRUE,
	  lgDelta=TRUE,
	  lgFailed=TRUE,
	  lgNegRate=FALSE,
	  lgOK=FALSE;
	long int i;
	double dPdlnT[NQGRID],
	  new_tmin,
	  x1;

#	ifdef DEBUG_FUN
	fputs( "<+>qheat()\n", debug_fp );
#	endif

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, " >>>>qheat called grain %s\n", gv.bin[nd]->chDstLab );
	}

	assert( gv.bin[nd]->lgQHeat );

	for( i=0; i < LOOP_MAX && !lgOK; i++ ) 
	{
		qheat1(qtemp,qprob,dPdlnT,qnbin,nd,&new_tmin,&lgFailed,&lgDelta,&lgNegRate);

		x1 = MAX2(new_tmin,gv.bin[nd]->qtmin)/MIN2(new_tmin,gv.bin[nd]->qtmin);
		lgBigChange = (x1 > 2.0);

		gv.bin[nd]->qtmin = new_tmin;

		lgOK = !lgFailed && !lgDelta && !lgBigChange && !lgNegRate;

		if( lgDelta ) 
		{
			if( gv.bin[nd]->qpres > 1.01*QHEAT_HIGH_RES )
			{
				gv.bin[nd]->qpres /= 3.;
				gv.bin[nd]->qpres = MAX2(gv.bin[nd]->qpres,QHEAT_HIGH_RES);
			}
			else 
			{
				break;
			}
		}

		if( !lgOK && trace.lgTrace && trace.lgDustBug ) 
		{
			fprintf( ioQQQ, " >>>>qheat trying another iteration, lgFailed %c lgBigChange %c\n\n",
				 TorF(lgFailed), TorF(lgBigChange) );
		}
	}

	gv.bin[nd]->lgUseQHeat = lgOK;
	gv.bin[nd]->lgEverQHeat = gv.bin[nd]->lgEverQHeat || gv.bin[nd]->lgUseQHeat;

	if( !lgOK ) 
	{
		*qnbin = 0;
	}

	if( lgNegRate )
	{
		++gv.bin[nd]->QHeatFailures;
	}

	if( gv.QHPunchFile != NULL && ( IterCnt.lgLastIt || !gv.lgQHPunLast ) ) 
	{
		fprintf( gv.QHPunchFile, "\nDust Temperature Distribution: grain %s zone %ld\n",
			 gv.bin[nd]->chDstLab,nzone );

		fprintf( gv.QHPunchFile, "Equilibrium temperature: %.2f\n", gv.bin[nd]->tedust );

		if( lgOK ) 
		{
			/* >>chng 01 oct 09, remove qprob from output, it depends on step size, PvH */
			fprintf( gv.QHPunchFile, "Number of bins: %ld\n", *qnbin );
			fprintf( gv.QHPunchFile, "  Tgrain     dP/dlnT\n" );
			for( i=0; i < *qnbin; i++ ) 
			{
				fprintf( gv.QHPunchFile, "%-12.4e %-12.4e\n", qtemp[i],dPdlnT[i] );
			}
		}
		else 
		{
			fprintf( gv.QHPunchFile, "**** quantum heating was not used\n" );
		}
	}

	if( trace.lgTrace && trace.lgDustBug )
	{
		if( lgOK ) 
		{
			fprintf( ioQQQ, " >>>>qheat converged\n\n");
		}
		else if( lgDelta ) 
		{
			fprintf( ioQQQ, " >>>>qheat found delta function, resolution %.4e\n\n",
				 gv.bin[nd]->qpres);
		}
		else 
		{
			fprintf( ioQQQ, " >>>>qheat failed, lgFailed %c lgBigChange %c\n",
				 TorF(lgFailed), TorF(lgBigChange) );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->qheat()\n", debug_fp );
#	endif
	return;
}

/****************************************************************************
 *
 * qheat1: main routine for calculating non-equilibrium heating of grains
 *
 * This routine implements the algorithm outlined in:
 * >>refer  Guhathakurtha & Draine, 1989, ApJ, 345, 230
 *
 * The original (fortran) version of the code was written by Kevin Volk.
 *
 * Heavily modified and adapted for new style grains by Peter van Hoof.
 *
 ****************************************************************************/

static void qheat1(/*@out@*/ double qtemp[],  /* qtemp[NQGRID]  */
		   /*@out@*/ double qprob[],  /* qprob[NQGRID]  */
		   /*@out@*/ double dPdlnT[], /* dPdlnT[NQGRID] */
		   /*@out@*/ long int *qnbin,
		   long int nd,
		   /*@out@*/ double *new_tmin,
		   /*@out@*/ int *lgFailed,
		   /*@out@*/ int *lgDelta,
		   /*@out@*/ int *lgNegRate)
{
	long int ind,
	  j,
	  k, 
	  nbin;

	           long nok;

	double AveEnergyEvent,
	  bval_jk,
	  delta_x,
	  delta_y,
	  delu[NQGRID],
	  dlnpdU,
	  fac = 0.,
	  frac,
	  integral,
	  Lambda[NQGRID],
	  p[NQGRID],
	  *Phi/*[NC_ELL]*/,
	  NumEvents,
	  EventRate,
	  qtmin1, 
	  RadCooling,
	  sum,
	  sum2,
	  u1[NQGRID],
	  u1Half[NQGRID],
	  uhi,
	  ulo,
	  umin,
	  val,
	  val2,
	  x,
	  xx,
	  y;


#	ifdef DEBUG_FUN
	fputs( "<+>qheat1()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( nd >= 0 && nd < gv.nBin );

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "  qheat1 called grain %s\n", gv.bin[nd]->chDstLab );
	}

	/* >>chng 01 aug 22, allocate space */
	if( (Phi = (double*)MALLOC((size_t)(rfield.nupper*sizeof(double)) ) ) == NULL )
		bad_malloc();

	*lgNegRate = FALSE;
	integral = 0.;

	xx = 0.;
	/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
	for( j=0; j < gv.qnflux; j++ )
	{
		*lgNegRate = *lgNegRate || ( gv.bin[nd]->phiTilde[j] < 0. );
		/* >>chng 97 jul 17, to summed continuum */
		/* >>chng 00 mar 30, to phiTilde, to keep track of photo-electric effect and collisions, by PvH */
		xx += gv.bin[nd]->phiTilde[j]*gv.bin[nd]->cnv_H_pCM3;
		integral += gv.bin[nd]->phiTilde[j]*gv.bin[nd]->cnv_H_pCM3*rfield.anu[j]*EN1RYD;
		/* Phi has units events/cm^3/s at actual depletion */
		Phi[j] = xx;
	}

	/* sanity check */
	assert( fabs(gv.bin[nd]->GrainHeat-integral)/gv.bin[nd]->GrainHeat <= CONSERV_TOL );

	/* EventRate is the rate at which photons/particles hit grains, in events/cm^3/s */
	/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
	EventRate = Phi[gv.qnflux-1];
	AveEnergyEvent = gv.bin[nd]->GrainHeat/EventRate;
	x = ufunct((double)gv.bin[nd]->tedust,nd)*EN1RYD;
	NumEvents = x/AveEnergyEvent;

	qtmin1 = ( gv.bin[nd]->qtmin < gv.bin[nd]->tedust ) ? gv.bin[nd]->qtmin : 0.7*gv.bin[nd]->tedust;
	qtmin1 = MAX2(qtmin1,GRAIN_TMIN);

	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    grain heating: %.4e, integral %.4e, total rate %.4e\n",
			 gv.bin[nd]->GrainHeat,integral,EventRate*gv.bin[nd]->cnv_CM3_pGR);
		fprintf( ioQQQ, "    av grain temp %.4e av grain enthalpy %.4e\n",
			 gv.bin[nd]->tedust,x);
		fprintf( ioQQQ, "    got lowest grain temp %.4e used lowest: %.4e\n",gv.bin[nd]->qtmin,qtmin1);
		fprintf( ioQQQ, "    average number of multiple events: %.4e, resolution %.4f\n",
			 NumEvents,gv.bin[nd]->qpres );
	}

	/* if average number of multiple photon events is too high -> don't even try */
	/* if negative rate was found, bail out */
	if( NumEvents > MAX_EVENTS || *lgNegRate ) 
	{
		*qnbin = 0;
		*new_tmin = qtmin1;
		*lgFailed = TRUE;
		if( NumEvents > MAX_EVENTS )
		{
			*lgDelta = TRUE;
			/* in this case we ignore negative rates, equilibrium treatment is good enough */
			*lgNegRate = FALSE;
			if( trace.lgTrace && trace.lgDustBug )
				fprintf( ioQQQ, "  qheat1 returns, NumEvents exceeding MAX_EVENTS\n" );
		}
		else
		{
			*lgDelta = FALSE;
			if( trace.lgTrace && trace.lgDustBug )
				fprintf( ioQQQ, "  qheat1 returns, found negative rate in phiTilde\n" );
		}

		free ( Phi );

#		ifdef DEBUG_FUN
		fputs( " <->qheat1()\n", debug_fp );
#		endif
		return;
	}

	ulo = rfield.AnuOrg[0];
	/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
	uhi = rfield.AnuOrg[gv.qnflux-1];

	nbin = NQGRID;
	qtemp[0] = qtmin1;
	/* u1 holds enthalpy in Ryd/grain */
	u1[0] = ufunct(qtemp[0],nd);
	/* >>chng 00 mar 22, taken out factor 4, factored in hden and dstAbund
	 * interpolate in dstems array instead of integrating explicitly, by PvH */
	xx = log(qtemp[0]);
	splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,xx,&y);
	/* Lambda holds the radiated energy for grains in this bin, in Ryd/s/grain */
	Lambda[0] = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;
	/* set up first step of integration */
	delu[0] = gv.bin[nd]->qpres*u1[0]/100.;
	u1Half[0] = u1[0] - delu[0]/2.;
	p[0] = PROB_CUTOFF_LO;
	qprob[0] = p[0]*delu[0];
	dPdlnT[0] = 0.;
	RadCooling = qprob[0]*Lambda[0];
	delu[1] = delu[0];

	for( k=1; k < NQGRID; k++ )
	{
		u1[k] = u1[k-1] + delu[k];
		u1Half[k] = u1[k] - delu[k]/2.;
		qtemp[k] = inv_ufunct(u1[k],nd);
		xx = log(qtemp[k]);
		splint(gv.dsttmp,gv.bin[nd]->dstems,gv.bin[nd]->dstslp2,NDEMS,xx,&y);
		Lambda[k] = exp(y)*gv.bin[nd]->cnv_H_pGR/EN1RYD;

		sum = 0.;
		sum2 = 0.;
		/* >>chng 01 may 12, reorder loop so that umin increases with each step
		 * this allows us to break out of the loop as soon as umin > uhi, PvH */
		for( j=k-1; j >= 0; j-- )
		{
			umin = u1Half[k] - u1[j];

			/* val is the value of Phi at umin */
			/* val2 is the value of phiTilde at umin */
			if( umin < ulo ) 
			{
				val = 0.;
			}
			else if( umin <= uhi )
			{
				/* ind is on C scale */
				ind = ipoint(umin)-1;
				frac = (umin-(double)rfield.anu[ind])/(double)rfield.widflx[ind] + 0.5;
				y = ( ind == 0 ) ? 0. : Phi[ind-1];
				val = y + frac*(Phi[ind] - y);
				if( k <= NSTARTUP )
				{
					y = ( ind == 0 ) ? 0. : gv.bin[nd]->phiTilde[ind-1]/rfield.widflx[ind-1];
					val2 = y + frac*(gv.bin[nd]->phiTilde[ind]/rfield.widflx[ind] - y);
					sum2 += qprob[j]*val2;
				}
			}
			else
			{
				break;
			}

			/* bval_jk*gv.bin[nd]->cnv_CM3_pGR is the total excitation rate from j to k
			 * and higher due to photon absorptions and particle collisions, it already
			 * implements Eq. 2.17 of Guhathakurtha & Draine, in events/grain/s */
			/* >>chng 00 mar 27, factored in hden (in Phi), by PvH */
			/* >>chng 00 mar 29, add in contribution due to particle collisions, by PvH */
			/* >>chng 01 mar 30, moved multiplication of bval_jk with gv.bin[nd]->cnv_CM3_pGR
			 *   out of loop, PvH */
			/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
			bval_jk = Phi[gv.qnflux-1]-val;
			sum += qprob[j]*bval_jk;
		}

		/* >>chng 00 mar 27, multiplied with delu, by PvH */
 		/* qprob[k] = sum*gv.bin[nd]->cnv_CM3_pGR*delu[k]/(Lambda[k] - Lambda[0]); */
		/* >>chng 00 apr 5, taken out Lambda[0], improves convergence at low end dramatically!, by PvH */
		p[k] = sum*gv.bin[nd]->cnv_CM3_pGR/Lambda[k];
		qprob[k] = p[k]*delu[k];
		dPdlnT[k] = qprob[k]/log(qtemp[k]/qtemp[k-1]);

/*  		printf(" k %ld T[k] %.6e U[k] %.6e p[k] %.6e dPdlnT[k] %.6e\n",k,qtemp[k],u1[k],p[k],dPdlnT[k]); */

		/* calculate logarithmic derivative to estimate new stepsize */
		if( k+1 < NQGRID ) 
		{
			/* this expression is more stable for the first few points */
			if( k <= NSTARTUP ) 
			{
				/* >>chng 01 nov 29, rfield.nflux -> gv.qnflux, PvH */
				y = Phi[gv.qnflux-1]*gv.bin[nd]->cnv_CM3_pGR;
				y = y - (Lambda[k] - Lambda[k-1])/delu[k];
				dlnpdU = (y - sum2*gv.bin[nd]->cnv_H_pGR/p[k])/Lambda[k];
			}
			else 
			{
				if( p[k] > p[k-1] ) 
				{
					dlnpdU = log(p[k]/p[k-1])/delu[k-1];
				}
				else 
				{
					dlnpdU = log(p[k]/p[k-1])/delu[k];
				}
			}
			xx = fabs(dlnpdU);
			/* the second expression limits the stepsize near the maximum */
			delu[k+1] = ( xx > 4./u1[k] ) ? gv.bin[nd]->qpres/xx : gv.bin[nd]->qpres*u1[k]/4.;

			/* >>chng 01 may 10, for very intense radiation fields the integration can become
			 * wildly unstable if gv.bin[nd]->qpres is too big, this prevents catastrophic failure */
			/* >>chng 01 nov 29, rfield.nflux -> rfield.nflux-1, PvH */
			delu[k+1] = MIN2(delu[k+1],rfield.anu[rfield.nflux-1]/2.);

			/* prevent stepsize from becoming infinitesimally small */
			assert( delu[k+1] > 5.*DBL_EPSILON*u1[k] );
		}

		/* if qtmin is far too low, qprob can become extremely large, exceeding
		 * even double precision range. the following check should prevent overflows */
		if( qprob[k] > sqrt(DBL_MAX) ) 
		{
			/* >>chng 01 apr 21, change DBL_MAX -> DBL_MAX/10. to work around
			 * a bug in the Compaq C compiler V6.3-025 when invoked with -fast, PvH */
			for( j=0; j <= k; j++ ) 
			{
				p[j] /= DBL_MAX/10.;
				qprob[j] /= DBL_MAX/10.;
				dPdlnT[j] /= DBL_MAX/10.;
			}
			RadCooling /= DBL_MAX/10.;
		}
		RadCooling += qprob[k]*Lambda[k];

		/* this may catch a bug in the Compaq C compiler V6.3-025
		 * if this gets triggered, try compiling with -ieee */
		assert( p[k] > 0. && qprob[k] > 0. && dPdlnT[k] > 0. );

		/* force thermal equilibrium of the grains */
		fac = RadCooling/gv.bin[nd]->GrainHeat*EN1RYD/gv.bin[nd]->cnv_CM3_pGR;

		/* this is regular stop criterion */
		if( dPdlnT[k] < dPdlnT[k-1] && dPdlnT[k]/fac < PROB_CUTOFF_HI ) 
		{
			nbin = k+1;
			break;
		}
	}
/*  	printf(" qheat1 did %ld iterations\n",nbin); */

	/* >>chng 01 may 11, multiply with NINT(QHEAT_INIT_RES/gv.bin[nd]->qpres) */
	if( nbin <= NQMIN*NINT(QHEAT_INIT_RES/gv.bin[nd]->qpres) ) 
	{
		*qnbin = 0;
		*new_tmin = qtmin1;
		*lgFailed = TRUE;
		*lgDelta = TRUE;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "  qheat1 returns, nbin=%ld <= NQMIN\n", nbin );
		}

		free ( Phi );

#		ifdef DEBUG_FUN
		fputs( " <->qheat1()\n", debug_fp );
#		endif
		return;
	}

	dPdlnT[0] = qprob[0]/log(qtemp[1]/qtemp[0]);

	for( j=0; j < nbin; j++ )
	{
		p[j] /= fac;
		qprob[j] /= fac;
		dPdlnT[j] /= fac;
	}

	/* >>chng 01 may 11, rebin the quantum heating results
	 *
	 * for grains in intense radiation fields, the code needs high resolution for stability
	 * and therefore produces lots of small bins, even though the grains never make large
	 * excurions from the equilibrium temperature; adding in the resulting spectra in RTDiffuse
	 * takes up an excessive amount of CPU time where most CPU is spent on grains for which
	 * the quantum treatment matters least, and moreover on temperature bins with very low
	 * probability; rebinning the results on a coarser grid should help reduce the overhead */

	                nok = nbin;

	nbin = RebinQHeatResults(nd,nbin,p,qtemp,qprob,dPdlnT,u1,delu,Lambda);

	*qnbin = nbin;

	/* >>chng 01 jul 13, add fail-safe for failure in RebinQHeatResults */
	if( nbin == 0 )
	{
		*new_tmin = qtmin1;
		*lgFailed = TRUE;
		*lgDelta = TRUE;

		if( trace.lgTrace && trace.lgDustBug )
		{
			fprintf( ioQQQ, "  qheat1 returns, RebinQHeatResults failed\n" );
		}

		free ( Phi );

#		ifdef DEBUG_FUN
		fputs( " <->qheat1()\n", debug_fp );
#		endif
		return;
	}

	sum = 0.;
	for( j=0; j < nbin; j++ )
	{
		sum += qprob[j];
	}

	*lgFailed = ( dPdlnT[nbin-1]/sum > SAFETY*PROB_CUTOFF_HI );

	*new_tmin = 0.;
	for( j=1; j < nbin; j++ ) 
	{
		if( dPdlnT[j]/sum < PROB_CUTOFF_LO ) 
		{
			*new_tmin = qtemp[j];
		}
		else 
		{
			if( j == 1 ) 
			{
				*lgFailed = *lgFailed || ( dPdlnT[j]/sum > SAFETY*PROB_CUTOFF_LO );
				if( dPdlnT[1] < 0.999*dPdlnT[NSTARTUP] ) 
				{
					delta_x = log(qtemp[NSTARTUP]/qtemp[1]);
					delta_y = log(dPdlnT[NSTARTUP]/dPdlnT[1]);
					delta_x *= log(PROB_CUTOFF_LO*sum/dPdlnT[1])/delta_y;
					*new_tmin = exp(log(qtemp[1]) + delta_x);
				}
			}
			break;
		}
	}
	*new_tmin = MAX3(*new_tmin,qtmin1/DEF_FAC,GRAIN_TMIN);

	/* the fact that the probability normalization fails probably indicates
	 * that the distribution is too close to a delta function to resolve */
	*lgDelta = *lgFailed ? FALSE : (fabs(sum-1.) > PROB_TOL);
		
	if( trace.lgTrace && trace.lgDustBug )
	{
		fprintf( ioQQQ, "    nbin %ld total probability %.4e new_tmin %.4e lgFailed %c lgDelta %c\n",
			 nbin,sum,*new_tmin,TorF(*lgFailed),TorF(*lgDelta) );
		fprintf( ioQQQ, "  qheat1 returns\n\n" );
	}

	free ( Phi );

#	ifdef DEBUG_FUN
	fputs( " <->qheat1()\n", debug_fp );
#	endif
	return;
}


/* rebin the quantum heating results to speed up RTDiffuse */
static long RebinQHeatResults(long nd,
			      long oldnbin,
			      double p[],
			      double qtemp[],
			      double qprob[],
			      double dPdlnT[],
			      double u1[],
			      double delu[],
			      double Lambda[])
{
	long i,
	  newnbin;

#	ifdef DEBUG_FUN
	fputs( "<+>RebinQHeatResults()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( nd >= 0 && nd < gv.nBin );
	assert( oldnbin > 0 && oldnbin < NQGRID );

	newnbin = 0;

	/* leading entries may have become very small or zero -> skip */
	for( i=0; i < oldnbin && dPdlnT[i] < PROB_CUTOFF_LO; i++ ) {}
		
	while( i < oldnbin )
	{
		double reftemp = qtemp[i];
		double s0 = 0.;
		double s1 = 0.;
		double xx,y;
		do 
		{
			/* determine average Lambda */
			s0 += qprob[i];
			s1 += qprob[i]*Lambda[i];
			++i;
		} while( i < oldnbin && qtemp[i]/reftemp < QT_RATIO );

		qprob[newnbin] = s0;
		Lambda[newnbin] = s1/s0;
		xx = log(Lambda[newnbin]*EN1RYD/gv.bin[nd]->cnv_H_pGR);
		splint(gv.bin[nd]->dstems,gv.dsttmp,gv.bin[nd]->dstslp,NDEMS,xx,&y);
		qtemp[newnbin] = exp(y);
		u1[newnbin] = ufunct(qtemp[newnbin],nd);
		++newnbin;
	}

	/* >>chng 01 jul 13, add fail-safe */
	if( newnbin < 2 )
	{
#		ifdef DEBUG_FUN
		fputs( " <->RebinQHeatResults()\n", debug_fp );
#		endif
		return 0;
	}
		
	/* now do the numerical derivatives. despite appearances, these are only first-order correct */
	for( i=0; i < newnbin; i++ )
	{
		long lo = ( i == 0 ) ? i : i-1;
		long hi = ( i == newnbin-1 ) ? i : i+1;
		double fac = (double)(hi-lo);
		delu[i] = (u1[hi] - u1[lo])/fac;
		p[i] = qprob[i]/delu[i];
		dPdlnT[i] = fac*qprob[i]/log(qtemp[hi]/qtemp[lo]);
	}
	
	/* invalidate the rest of the arrays */
	for( i=newnbin; i < oldnbin; i++ )
	{
		p[i] = -DBL_MAX;
		qtemp[i] =  -DBL_MAX;
		qprob[i] =  -DBL_MAX;
		dPdlnT[i] =  -DBL_MAX;
		u1[i] =  -DBL_MAX;
		delu[i] =  -DBL_MAX;
		Lambda[i] =  -DBL_MAX;
	}

#	ifdef DEBUG_FUN
	fputs( " <->RebinQHeatResults()\n", debug_fp );
#	endif
	return newnbin;
}
#endif


/* calculate the enthalpy of a grain at a given temperature, in Ryd */
static double ufunct(double temp, 
		     long int nd)
{
	long int itype,
	  j;
	double atoms,
	  enthalpy;

#	ifdef DEBUG_FUN
	fputs( "<+>ufunct()\n", debug_fp );
#	endif

	if( temp <= 0. ) 
	{
		fprintf( ioQQQ, " ufunct called with non-positive temperature: %.6e\n" , temp );
		puts( "[Stop in ufunct]" );
		cdEXIT(1);
	}
	assert( nd >= 0 && nd < gv.nBin );

	itype = gv.bin[nd]->matType;
	atoms = gv.bin[nd]->AvVol*gv.bin[nd]->dustp[0]/ATOMIC_MASS_UNIT/gv.bin[nd]->atomWeight;

	enthalpy = 0.;
	if( itype == MAT_CAR )
	{
		/* enthalpy for pah/graphitic grains in Ryd
		 * derived from:
		 * >>refer Guhathakurta & Draine, 1989, ApJ, 345, 230 */
		enthalpy = (4.15e-22/EN1RYD)*pow(temp,3.3)/(1. + 6.51e-03*temp + 
		  1.5e-06*temp*temp + 8.3e-07*pow(temp,2.3));
	}
	else if( itype == MAT_SIL )
	{
		/* enthalpy for silicate grains (and grey grains) in Ryd */
		/* >>chgn 00 Mar 22, changed run of j, highest bin was missed, by PvH */
		/* limits of tlim set above, 0 and DBL_MAX, so always OK */
		for( j = 0; j < 4 && temp > tlim[j]; j++ )
		{
			enthalpy += cval[j]*(pow(MIN2(temp,tlim[j+1]),power[j]) - pow(tlim[j],power[j]));
		}
	}
	else 
	{
		fprintf( ioQQQ, " ufunct called with unknown type for enthalpy function: %ld\n" , itype );
		puts( "[Stop in ufunct]" );
		cdEXIT(1);
	}

	/* >>chng 00 mar 23, use formula 3.1 of Guhathakurtha & Draine, by PvH */
	enthalpy *= atoms-2.;

	if( enthalpy <= 0. ) 
	{
		fprintf( ioQQQ, " ufunct finds non-positive enthalpy: %.6e, what's up?\n" , enthalpy );
		puts( "[Stop in ufunct]" );
		cdEXIT(1);
	}

#	ifdef DEBUG_FUN
	fputs( " <->ufunct()\n", debug_fp );
#	endif
	return( enthalpy );
}


#if NEW_CODE
/* calculate derivative of the enthalpy function dU/dT (aka the specific heat) at a given temperature, in Ryd/K */
static double uderiv(double temp, 
		     long int nd)
{
	long int itype,
	  j;
	double atoms,
	  deriv = 0.;

#	ifdef DEBUG_FUN
	fputs( "<+>uderiv()\n", debug_fp );
#	endif

	if( temp <= 0. ) 
	{
		fprintf( ioQQQ, " uderiv called with non-positive temperature: %.6e\n" , temp );
		puts( "[Stop in uderiv]" );
		cdEXIT(1);
	}
	assert( nd >= 0 && nd < gv.nBin );

	itype = gv.bin[nd]->matType;
	atoms = gv.bin[nd]->AvVol*gv.bin[nd]->dustp[0]/ATOMIC_MASS_UNIT/gv.bin[nd]->atomWeight;

	if( itype == MAT_CAR )
	{
		double numer = (4.15e-22/EN1RYD)*pow(temp,3.3);
		double dnumer = (3.3*4.15e-22/EN1RYD)*pow(temp,2.3);
		double denom = 1. + 6.51e-03*temp + 1.5e-06*temp*temp + 8.3e-07*pow(temp,2.3);
		double ddenom = 6.51e-03 + 2.*1.5e-06*temp + 2.3*8.3e-07*pow(temp,1.3);
		/* dU/dT for pah/graphitic grains in Ryd/K
		 * derived from:
		 * >>refer Guhathakurta & Draine, 1989, ApJ, 345, 230 */
		deriv = (dnumer*denom - numer*ddenom)/POW2(denom);
	}
	else if( itype == MAT_SIL )
	{
		/* dU/dT for silicate grains (and grey grains) in Ryd/K */
		/* limits of tlim set above, 0 and DBL_MAX, so always OK */
		for( j = 0; j < 4; j++ )
		{
			if( temp > tlim[j] && temp <= tlim[j+1] )
			{
				deriv = power[j]*cval[j]*pow(temp,power[j]-1.);
				break;
			}
		}
	}
	else 
	{
		fprintf( ioQQQ, " uderiv called with unknown type for enthalpy function: %ld\n" , itype );
		puts( "[Stop in uderiv]" );
		cdEXIT(1);
	}

	/* >>chng 00 mar 23, use formula 3.1 of Guhathakurtha & Draine, by PvH */
	deriv *= atoms-2.;

	if( deriv <= 0. ) 
	{
		fprintf( ioQQQ, " uderiv finds non-positive derivative: %.6e, what's up?\n" , deriv );
		puts( "[Stop in uderiv]" );
		cdEXIT(1);
	}

#	ifdef DEBUG_FUN
	fputs( " <->uderiv()\n", debug_fp );
#	endif
	return( deriv );
}
#endif


/* this is the inverse of ufunct: determine grain temperature as a function of enthalpy */
static double inv_ufunct(double enthalpy, 
			 long int nd)
{
	long int itype;
	double atoms,
	  temp;

#	ifdef DEBUG_FUN
	fputs( "<+>inv_ufunct()\n", debug_fp );
#	endif

	if( enthalpy <= 0. ) 
	{
		fprintf( ioQQQ, " inv_ufunct called with non-positive enthalpy: %.6e\n" , enthalpy );
		puts( "[Stop in ufunct]" );
		cdEXIT(1);
	}
	assert( nd >= 0 && nd < gv.nBin );

	itype = gv.bin[nd]->matType;
	atoms = gv.bin[nd]->AvVol*gv.bin[nd]->dustp[0]/ATOMIC_MASS_UNIT/gv.bin[nd]->atomWeight;

	enthalpy /= atoms-2.;
	temp = 0.;
	if( itype == MAT_CAR )
	{
		double x0,x1,x2,denom,toler=1.e-6;

		/* invert enthalpy for pah/graphitic grains in Ryd
		 * derived from:
		 * >>refer Guhathakurta & Draine, 1989, ApJ, 345, 230 */
		x1 = pow(enthalpy*(EN1RYD/4.15e-22),1./3.3);
		x2 = pow(enthalpy*(EN1RYD/2.74e-19),1./1.9);
		x1 = MAX2(x1,x2);

		do 
		{
			x0 = x1;
			denom = 1. + 6.51e-03*x0 + 1.5e-06*x0*x0 + 8.3e-07*pow(x0,2.3);
			x1 = pow(enthalpy*denom*(EN1RYD/4.15e-22),1./3.3);
		} 
		while( fabs(x1-x0)/x1 > toler );

		temp = x1;
	}
	else if( itype == MAT_SIL )
	{
		double ulow;
		long int j;

		/* invert enthalpy for silicate grains (and grey grains) in Ryd */
		for( j = 0; j < 4; j++ )
		{
			ulow = cval[j]*(pow(tlim[j+1],power[j]) - pow(tlim[j],power[j]));
			if( ulow < enthalpy ) 
			{
				enthalpy -= ulow;
			}
			else 
			{
				break;
			}
		}
		enthalpy += cval[j]*pow(tlim[j],power[j]);
		temp = pow(enthalpy/cval[j],1./power[j]);
	}
	else 
	{
		fprintf( ioQQQ, " inv_ufunct called with unknown type for enthalpy function: %ld\n" , itype );
		puts( "[Stop in inv_ufunct]" );
		cdEXIT(1);
	}

	if( temp <= 0. ) 
	{
		fprintf( ioQQQ, " inv_ufunct finds non-positive temperature: %.6e, what's up?\n" , temp );
		puts( "[Stop in inv_ufunct]" );
		cdEXIT(1);
	}

#	ifdef DEBUG_FUN
	fputs( " <->inv_ufunct()\n", debug_fp );
#	endif
	return( temp );
}
