/*ParsePunch parse the punch command */
#include "cddefines.h"
#include "elementnames.h"
#include "input.h"
#include "zonecnt.h"
#include "opacity.h"
#include "mappar.h"
#include "pop371.h"
#include "version.h"
#include "grainvar.h"
#include "physconst.h"
#include "punch.h"
#include "parse.h"
/* check for keyword UNITS on line, then scan wavelength or energy units if present */
static void ChkUnits( char *chCard);

/* NB NB NB NB NB NB NB NB NB NB
 *
 * if any "special" punch commands are added (commands that copy the file pointer
 * into a separate variable, e.g. like PUNCH _DR_), be sure to add that file pointer
 * to PunchFilesInit and ClosePunchFiles !!!
 *
 * PUNCH FILE POINTERS SHOULD NEVER BE ALTERED BY ROUTINES OUTSIDE THIS MODULE !!!
 *
 * hence initializations of punch file pointers should never be included in zero() !!
 * the pointer might be lost without the file being closed
 * 
 * NB NB NB NB NB NB NB NB NB NB */


void ParsePunch(char *chCard )
{
	char chLabel[81];
	int lgEOL ;
	/* pointer to column across line image for free format number reader*/
	long int ipFFmt, 
	  i,
	  j,
	  nelem;

#	ifdef DEBUG_FUN
	fputs( "<+>ParsePunch()\n", debug_fp );
#	endif

	/* while optimizing, punch commands are not executed -> return */
	if( !punch.lgOpenUnits )
	{
#		ifdef DEBUG_FUN
		fputs( " <->ParsePunch()\n", debug_fp );
#		endif
		return;
	}

	/* check that limit not exceeded */
	if( punch.npunch > LIMPUN )
	{
		fprintf( ioQQQ, 
			"The limit to the number of PUNCH options is %ld.  Increase LIMPUN in punch.h if more are needed.\n", 
		  LIMPUN );
		puts( "[Stop in ParsePunch]" );
		cdEXIT(1);
	}

	/* LAST keyword is an option to punch only on last iteration */
	punch.lgPunLstIter[punch.npunch] = lgMatch("LAST",chCard);

	/* NOCLOBBER keyword is an option to append to a given file instead of overwriting it */
	punch.lgNoClobber[punch.npunch] = lgMatch("NOCL",chCard);

	/* 
	 * get file name for this punch output.
	 * GetQuote does the following -
	 * first copy original version of file name into chLabel, 
	 * string does include null termination.
	 * set filename in OrgCard and second parameter to spaces so 
	 * that not picked up below as keyword
	 * aborts if no quote found 
	 */
	GetQuote( chLabel , chCard );

	/* check that name is not same as opacity.opc, a special file */
	if( strcmp(chLabel , "opacity.opc") == 0 )
	{
		fprintf( ioQQQ, " ParsePunch will not allow punch file name %s, please choose another.\n",
			chLabel);
		puts( "[Stop in ParsePunch]" );
		cdEXIT(1);
	}

	/* only open if file has not already been opened during a previous call (should never happen!) */
	if( punch.ipPnunit[punch.npunch] == NULL )
	{
		/* punch.npunch is set to 0 in ZERO, 
		 * punch.ipPnunit is the file pointer  */

		/* open the file with the name generated above */
		if( punch.lgNoClobber[punch.npunch] ) 
		{
			punch.ipPnunit[punch.npunch] = fopen( chLabel,"a+" );
		}
		else 
		{
			punch.ipPnunit[punch.npunch] = fopen( chLabel,"w" );
		}
		if( NULL == punch.ipPnunit[punch.npunch] )
		{
			fprintf( ioQQQ, " ParsePunch could not open punch file %s for writing.\n",
				chLabel);
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	/* put version number and title of model on output file, but only if
	 * there is not a "no title" on the line*/
	if( !lgMatch("O TI",chCard) )
	{
		fprintf( punch.ipPnunit[punch.npunch], 
			"# %s %s\n", 
		  version.chVersion, input.chTitle );
	}

	/* usually results for each iteration are followed by a series of
	 * hash marks, ####, which fool excel.  This is an option to not print
	 * the line.  If the keyword NO HASH appears anywhere, on any punch command,
	 * the hash marks will never be produced */
	if( lgMatch("O HA",chCard) )
	{
		punch.lgHashEndIter = FALSE;
	}

	/* punch opacity must come first since elements appear as sub-keywords */
	if( lgMatch("OPAC",chCard) )
	{

		/* check for keyword UNITS on line, then scan wavelength or energy units if present,
		 * units are copied into punch.chConPunEnr */
		ChkUnits(chCard);

		strcpy( punch.chPunch[punch.npunch], "OPAC" );
		if( lgMatch("TOTA",chCard) )
		{
			/* DoPunch will call popac to parse the subcommands
			 * punch total opacity */
			strcpy( opac.chOpcTyp, "TOTL" );
			fprintf( punch.ipPnunit[punch.npunch], 
				"#nu\tTot opac\tAbs opac\tScat opac\tAlbedo\telem\n" );
		}

		else if( lgMatch("FIGU",chCard) )
		{
			/* do figure for hazy */
			strcpy( opac.chOpcTyp, "FIGU" );
			fprintf( punch.ipPnunit[punch.npunch], 
				"#nu, H, He, tot opac\n" );
		}

		else if( lgMatch("GRAI",chCard) )
		{
			/* punch grain command, give optical properties of gains in calculation */
			strcpy( punch.chPunch[punch.npunch], "DUSO" );
			/* punch grain opacity command in twice, here and above in opacity
			 * chDst = 'OPAC' */
			fprintf( punch.ipPnunit[punch.npunch], 
				"#grain nu,  total,  absorpt,  scat\n" );
		}

		else if( lgMatch("SHEL",chCard) )
		{
			/* punch shells, a form of the punch opacity command for showing subshell crossections*/
			strcpy( punch.chPunch[punch.npunch], "OPAC" );

			/* punch subshell cross sections */
			strcpy( opac.chOpcTyp, "SHEL" );

			/* this is element */
			ipFFmt = 3;
			punch.punarg[0][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,
			  LINELENGTH,&lgEOL);

			/* this is ion */
			punch.punarg[1][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,
			  LINELENGTH,&lgEOL);

			/* this is shell */
			punch.punarg[2][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,
			  LINELENGTH,&lgEOL);

			if( lgEOL )
			{
				fprintf( ioQQQ, " there must be io unit, atom wght, ion, shell\n" );
				puts( "[Stop in ParsePunch]" );
				cdEXIT(1);
			}
			fprintf( punch.ipPnunit[punch.npunch], "#sub shell cross section\n" );
		}

		else
		{
			/* punch elenent opacity, produces n name.n files, one for each stage of 
			 * ionization.  the name is the 4-char version of the element's name, and
			 * n is the stage of ionization.  the file nameon the card is ignored.
			 * The code Stop in getpunchs after these files are produced. */

			/* this will be used as check that we did find an element on the command lines */
			nelem = GetElem( chCard );

			/* lgHit was set true if an element was found, error if not */
			if( nelem < 0 )
			{
				fprintf( ioQQQ, " There must be a second keyword on the opacity command.  Sorry.\n" );
				fprintf( ioQQQ, " Options are total, figure, grain, or element name.\n" );
				puts( "[Stop in ParsePunch]" );
				cdEXIT(1);
			}

			/* copy string over */
			strcpy( opac.chOpcTyp, elementnames.chElementNameShort[nelem] );
		}
	}

	else if( lgMatch("ABUN",chCard) )
	{
		/* punch abundances */
		strcpy( punch.chPunch[punch.npunch], "ABUN" );
		fprintf( punch.ipPnunit[punch.npunch], "#abundances\n" );
	}

	else if( lgMatch("ADVE",chCard) )
	{
		/* punch information relating to advection */
		strcpy( punch.chPunch[punch.npunch], "ADVE");
		fprintf( punch.ipPnunit[punch.npunch], 
			"#avection depth\thtot\tCoolHeat\tdCoolHeatdT\tRecomb\tPhoto\ttimestp\n" );
	}

	else if( lgMatch("AGES",chCard) )
	{
		/* punch ages */
		strcpy( punch.chPunch[punch.npunch], "AGES" );
		fprintf( punch.ipPnunit[punch.npunch], "#ages\n" );
	}

	else if( lgMatch(" AGN",chCard) )
	{
		/* punch tables needed for AGN3 */
		strcpy( punch.chPunch[punch.npunch], " AGN" );
		/* this is the AGN option, to produce a table for AGN */

		/* charge exchange rate coefficients */
		if( lgMatch("CHAR",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "CHAG" );
			fprintf( punch.ipPnunit[punch.npunch], "#charge exchange rate coefficnt\n" );
		}

		else if( lgMatch("RECO",chCard) )
		{
			/* punch recombination rates for agn table */
			strcpy( punch.chPunch[punch.npunch], "RECA" );
			fprintf( punch.ipPnunit[punch.npunch], "#Recomrates for agn table\n" );
		}

		else if( lgMatch("OPAC",chCard) )
		{
			/* create table for appendix in AGN */
			strcpy( opac.chOpcTyp, " AGN" );
			strcpy( punch.chPunch[punch.npunch], "OPAC" );
		}

		else if( lgMatch("HECS",chCard) )
		{
			/* create table for appendix in AGN */
			strcpy( punch.chPunchArgs[punch.npunch], "HECS" );
			fprintf( punch.ipPnunit[punch.npunch], "#agn he cs \n" );
		}

		else if( lgMatch("HEMI",chCard) )
		{
			/* hemis - continuum emission needed for chap 4 */
			strcpy( punch.chPunchArgs[punch.npunch], "HEMI" );
			/* routine will make its own title */
			/*fprintf( punch.ipPnunit[punch.npunch], "#agn wl contin \n" );*/

			/* check for keyword UNITS on line, then scan wavelength or energy units if present,
			 * units are copied into punch.chConPunEnr */
			ChkUnits(chCard);
		}
		else
		{
			fprintf( ioQQQ, " I did not recognize this .\n" );
			fprintf( ioQQQ, " Sorry.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}

	}

	else if( lgMatch("ASSE",chCard) )
	{
		/* punch asserts */
		strcpy( punch.chPunch[punch.npunch], "ASSE" );
		/* no need to print this standard line of explanation*/
		/*fprintf( punch.ipPnunit[punch.npunch], " asserts\n" );*/
	}

	/* punch charge transfer */
	else if( lgMatch("CHAR",chCard) )
	{
		/* NB in dopunch only the first three characters are compared to find this option,
		 * search for "CHA" */
		/* punch charge transfer */
		strcpy( punch.chPunch[punch.npunch], "CHAR" );
		fprintf( punch.ipPnunit[punch.npunch], "#charge exchange rate coefficnt\n" );
	}

	else if( lgMatch("COLU",chCard) )
	{
		/* punch column density */
		strcpy( punch.chPunch[punch.npunch], "COLU" );
		fprintf( punch.ipPnunit[punch.npunch], "#column densities\n" );
	}

	else if( lgMatch("COMP",chCard) )
	{
		/* punch compton, details of the energy exchange problem */
		strcpy( punch.chPunch[punch.npunch], "COMP" );
		fprintf( punch.ipPnunit[punch.npunch], "#nu, comup, comdn\n" );
	}

	else if( lgMatch("COOL",chCard) )
	{
		/* cooling, actually done by routine */
		strcpy( punch.chPunch[punch.npunch], "COOL" );
		fprintf( punch.ipPnunit[punch.npunch], "#temp\theat\tcool\tfrac coolant \n" );
	}

	/* the punch continuum command, with many options,
	 * the first 3 char of the chPunch flag will always be "CON" 
	 * with the last indicating which one */
	else if( lgMatch("CONT",chCard) )
	{
		/* this flag is checked in PrtComment to generate a caution
		 * if continuum is punched but iterations not performed */
		punch.lgPunContinuum = TRUE;

		/* check for keyword UNITS on line, then scan wavelength or energy units if present,
		 * units are copied into punch.chConPunEnr */
		ChkUnits(chCard);

		if( lgMatch("BINS",chCard) )
		{
			/* continuum binning */
			strcpy( punch.chPunch[punch.npunch], "CONB" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Continuum binning, enrOrg, Enr, width of cells\n" );
		}

		else if( lgMatch("DIFF",chCard) )
		{
			/* diffuse continuum, the locally emitted continuum stored in rfield.ConEmitLocal */
			strcpy( punch.chPunch[punch.npunch], "COND" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#energy\t ConEmitLocal \n" );
		}

		else if( lgMatch("EMIT",chCard) )
		{
			/* continuum emitted by cloud */
			strcpy( punch.chPunch[punch.npunch], "CONE" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Energy\treflec\toutward\ttotal\tline\tcont\n" );
		}

		else if( lgMatch("FEII",chCard) )
		{
			/* FeII continuum, only valid if large atom is set */
			strcpy( punch.chPunch[punch.npunch], "CONF" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#FeII: wavelength(A)\tsummed intensity\n" );
		}

		else if( lgMatch("GRAI",chCard) )
		{
			/* grain continuum */
			strcpy( punch.chPunch[punch.npunch], "CONG" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#energy,     graphite,  rest \n" );
		}

		else if( lgMatch("INCI",chCard) )
		{
			/* incident continuum */
			strcpy( punch.chPunch[punch.npunch], "CONC" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Incident Continuum, Enr\tnFn \n" );
		}

		else if( lgMatch("INTE",chCard) )
		{
			/* continuum interactions */
			strcpy( punch.chPunch[punch.npunch], "CONi" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Continuum interactions, inc, otslin. otscon, ConInterOut, outlin \n" );
			/* this is option for lowest energy, if nothing then zero */
			ipFFmt = 3;
			punch.punarg[0][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,
			  LINELENGTH,&lgEOL);
		}

		else if( lgMatch("IONI",chCard) )
		{
			/* punch ionizing continuum*/
			strcpy( punch.chPunch[punch.npunch], "CONI" );

			/* this is option for lowest energy, if nothing then zero */
			ipFFmt = 3;
			punch.punarg[0][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,
			  LINELENGTH,&lgEOL);

			/* this is option for smallest interaction to punch, def is 1 percent */
			punch.punarg[1][punch.npunch] = (float)FFmtRead(chCard,&ipFFmt,LINELENGTH,&lgEOL);
			if( lgEOL )
			{
				punch.punarg[1][punch.npunch] = 0.01f;
			}
			/* put the header at the top of the file */
			fprintf( punch.ipPnunit[punch.npunch], 
				"#cell\tnu\tflux\tflx*cs\tFinc\totsl\totsc\tout\trate/tot\tintegral\tline\tcont\n" );
		}

		else if( lgMatch("OUTW",chCard) )
		{
			/* outward only continuum */
			if( lgMatch("LOCA",chCard) )
			{
				strcpy( punch.chPunch[punch.npunch], "CONo" );
				fprintf( punch.ipPnunit[punch.npunch], 
					"#Local Out   SavOutCon  ConInterOut+line SvOt*opc pass*opc\n" );
			}
			else
			{
				strcpy( punch.chPunch[punch.npunch], "CONO" );
				fprintf( punch.ipPnunit[punch.npunch], 
					"#Out Con      OutIncid  OutConD   OutLinD   OutConS\n" );
			}
		}

		else if( lgMatch("TRAN",chCard) )
		{
			/* transmitted continuum */
			strcpy( punch.chPunch[punch.npunch], "CONT" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Transmitted Continuum\n" );
		}

		else if( lgMatch(" RAW",chCard) )
		{
			/* "raw" continua */
			strcpy( punch.chPunch[punch.npunch], "CORA" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Raw Con anu\tflux\totslin\totscon\tConRefIncid\tConEmitReflec\tConInterOut\toutlin\tThrowOut\tConEmitOut\tline\tcont\n" );
		}

		else if( lgMatch("REFL",chCard) )
		{
			/* reflected continuum */
			strcpy( punch.chPunch[punch.npunch], "CONR" );

			fprintf( punch.ipPnunit[punch.npunch], 
				"#Reflected      cont       line       total        albedo    ConID\n" );
		}

		else
		{
			/* this is the usual punch continuum command */
			strcpy( punch.chPunch[punch.npunch], "CON " );
			fprintf( punch.ipPnunit[punch.npunch], 
				"#Cont nu\tincident\ttrans\tDiffOut\tnet trans\treflc\ttotal\tline\tcont\n" );
		}

	}

	/* punch information about convergence of this model 
	 * reason - why it did not converge an iteration
	 * error - zone by zone display of various convergence errors */
	else if( lgMatch("CONV",chCard) )
	{
		if( lgMatch("REAS",chCard) )
		{
			/* punch reason model declared not converged
			 * not a true punch command, since done elsewhere */
			punch.ipPunConv = punch.ipPnunit[punch.npunch];
			punch.lgPunConv = TRUE;
			fprintf( punch.ipPunConv, " reason for continued iterations\n" );
			/* this does not count as a punch option (really) */
			punch.ipPnunit[punch.npunch] = NULL;
			--punch.npunch;
		}
		else if( lgMatch("ERRO",chCard) )
		{
			/* punch zone by zone errors in pressure, electron density, and heating-cooling*/
			punch.ipPunConv = punch.ipPnunit[punch.npunch];
			punch.lgPunConv = TRUE;
			/* convergence error */
			strcpy( punch.chPunch[punch.npunch], "CNVE" );
			fprintf( punch.ipPunConv, "depth\tP(cor)\tP(cur)\tP%%error\tNE(cor)\tNE(cur)\tNE%%error\tHeat\tCool\tHC%%error\n" );
		}
		else
		{
			fprintf( ioQQQ, " There must be a second keyword on this command.\n" );
			fprintf( ioQQQ, " The ones I know about are reason and error.\n" );
			fprintf( ioQQQ, " Sorry.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch(" DR ",chCard) )
	{
		punch.ipDRout = punch.ipPnunit[punch.npunch];
		punch.lgDROn = TRUE;

		/* set punch last flag to whatever it was above */
		punch.lgDRPLst = punch.lgPunLstIter[punch.npunch];

		fprintf( punch.ipDRout, "zone\tdepth\tdr\tdr 2 go\treason \n" );

		/* this does not count as a punch option (really) */
		punch.ipPnunit[punch.npunch] = NULL;
		punch.npunch -= 1;
	}

	else if( lgMatch("ELEM",chCard) )
	{
		/* option to punch ionization structure of some element */
		strcpy( punch.chPunch[punch.npunch], "ELEM" );
		/* this returns element number on c scale */
		j = GetElem(chCard);
		++j;
		/* this is the atomic number on the physical scale */
		punch.punarg[0][punch.npunch] = (float)j;
		assert( j>0);
		fprintf( punch.ipPnunit[punch.npunch], "#depth");
		for(i=0; i<j+1;++i )
		{
			assert( j>0 );
			fprintf( punch.ipPnunit[punch.npunch], 
				"\t%2s%2li", elementnames.chElementSym[j-1],i+1);
		}
		fprintf( punch.ipPnunit[punch.npunch], "\n");
	}

	else if( lgMatch("FEII",chCard) )
	{
		if( lgMatch("FRED",chCard) )
		{
			/* punch out some stuff for Fred's FeII atom */
			fprintf( punch.ipPnunit[punch.npunch], "#FeII rates \n" );
			strcpy( punch.chPunch[punch.npunch], "FE2f" );
		}
		else if( lgMatch("DEPA",chCard) )
		{
			/* punch out departure coef for the large FeII atom */
			fprintf( punch.ipPnunit[punch.npunch], "#FeII departure coef \n" );
			/* optional keyword all means do all levels, if not then just do subset */
			if( lgMatch(" ALL",chCard) )
			{
				/* punch all levels, calls routine FeIIPunDepart */
				strcpy( punch.chPunch[punch.npunch], "FE2D" );
			}
			else
			{
				/* punch a very few selected levels, calls routine FeIIPunDepart */
				strcpy( punch.chPunch[punch.npunch], "FE2d" );
			}
		}
		else if( lgMatch("POPU",chCard) )
		{
			/* punch out level populations for the large FeII atom */
			fprintf( punch.ipPnunit[punch.npunch], "#FeII level populations \n" );
			/* optional keyword all means do all levels, if not then just do subset */
			if( lgMatch(" ALL",chCard) )
			{
				/* punch all levels, calls routine FeIIPunDepart */
				strcpy( punch.chPunch[punch.npunch], "FE2P" );
			}
			else
			{
				/* punch a very few selected levels, calls routine FeIIPunDepart */
				strcpy( punch.chPunch[punch.npunch], "FE2p" );
			}
		}
		else
		{
			fprintf( ioQQQ, " There must be a second keyword on this command.\n" );
			fprintf( ioQQQ, " The ones I know about are departure and fred.\n" );
			fprintf( ioQQQ, " Sorry.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch("GAMM",chCard) )
	{
		/* punch all photoionizaiton rates for all subshells */
		fprintf( punch.ipPnunit[punch.npunch], "#Photoionization rates \n" );
		strcpy( punch.chPunch[punch.npunch], "GAMM" );

	}
	else if( lgMatch("GRAI",chCard) )
	{
		/* punch grains
		 * chDst = 'NONE' */
		if( lgMatch("OPAC",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "DUSO" );
			/* punch grain opacity command in twice, here and above in opacity
			 * chDst = 'OPAC' */
			fprintf( punch.ipPnunit[punch.npunch], "#grain ext, abs, scat\n" );
		}
		else if( lgMatch("PHYS",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "DUSP" );
			/* chDst = 'PHYS' */
			fprintf( punch.ipPnunit[punch.npunch], "#grain depth temp, charge, drift, frc heat, cool\n" );
		}
		else if( lgMatch(" QS ",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "DUSQ" );
			fprintf( punch.ipPnunit[punch.npunch], "#grain abs, scat, qs\n" );
		}
		else
		{
			fprintf( ioQQQ, " There must be a second key; they are  OPAC, _QS_, and PHYS\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch("GAUN",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "GAUN" );
		fprintf( punch.ipPnunit[punch.npunch], "#Gaunt factors.\n" );
	}

	else if( lgMatch("HTWO",chCard) )
	{
		punch.ipPunH2 = punch.ipPnunit[punch.npunch];
		punch.lgPunH2 = TRUE;
		fprintf( punch.ipPnunit[punch.npunch], "#H2 creation, destruction. \n" );
		/* this does not count as a punch option (really) */
		punch.ipPnunit[punch.npunch] = NULL;
		punch.npunch -= 1;

	}

	/* QHEAT has to be in front of HEAT... */
	else if( lgMatch("QHEA",chCard) )
	{
		gv.QHPunchFile = punch.ipPnunit[punch.npunch];
		gv.lgQHPunLast = punch.lgPunLstIter[punch.npunch];
		fprintf( gv.QHPunchFile, "#Probability distributions from quantum heating routine.\n" );
		/* this does not count as a punch option (really) */
		punch.ipPnunit[punch.npunch] = NULL;
		punch.npunch -= 1;
	}

	else if( lgMatch("HEAT",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "HEAT" );
		fprintf( punch.ipPnunit[punch.npunch], "depth\thtot\tctot\theat fracs\n" );
	}

	/* information about the helium atom */
	else if( lgMatch("HELI",chCard) )
	{
		/* singlet populations and rates */
		if( lgMatch("SING",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "HEL1" );
			fprintf( punch.ipPnunit[punch.npunch], "#he1    level,   simple,  clhe2ov1  smhe2ov1   rec,     ioniz,    hei3 \n" );
		}

		/* triplet populations and rates actually debug new he ion */
		else if( lgMatch("TRIP",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "HEL3" );
			fprintf( punch.ipPnunit[punch.npunch], "#HeTrip old ion\tnew ion\told n 23S\tnew n 23S\to rec\tn rec\t o phot\t n phot \n" );
		}
		else
		{
			fprintf( ioQQQ, " did not find a keyword on this  line.  Keys are SING, _ION, and TRIP.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch("HUMM",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "HUMM" );
		fprintf( punch.ipPnunit[punch.npunch], "#input to DHs routine.\n" );
	}

	else if( lgMatch("HYDR",chCard) )
	{
		/* punch hydrogen physical conditions */
		if( lgMatch("COND",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "HYDc" );
			fprintf( punch.ipPnunit[punch.npunch], "#depth\tTe\tHDEN\tEDEN\tHI/H\tHII/H\tH2/H\tH2+/H\tH3+/H\tH-/H\n" );
			/* punch hydrogen ionization */
		}
		else if( lgMatch("IONI",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "HYDi" );
			fprintf( punch.ipPnunit[punch.npunch], "#  gam1    RecEff   HRecRad    hii/hi Shii/hi   hrec(13) dec grd  GamEff   ColEff\n" );
			/* punch hydrogen populations */
		}
		else if( lgMatch("POPU",chCard) )
		{
			strcpy( punch.chPunch[punch.npunch], "HYDp" );
			fprintf( punch.ipPnunit[punch.npunch], "#  depth      HI      HII       HN                                                   2s       2p\n" );
		}
		else if( lgMatch("RECC",chCard) )
		{
			/* recombination cooling, for AGN */
			strcpy( punch.chPunch[punch.npunch], "HYDr" );
			fprintf( punch.ipPnunit[punch.npunch], "#T\tbAS\tb1\tbB\n" );
		}
		else if( lgMatch("21CM",chCard) )
		{
			/* 21 cm optical depth and spin temperature  */
			strcpy( punch.chPunch[punch.npunch], "HYD2" );
			fprintf( punch.ipPnunit[punch.npunch], "#depth\ttau(21cm)\tT(spin)\n" );
		}
		else
		{
			fprintf( ioQQQ, " Punch hydrogen has options: CONDitions POPUlations and IONIzation.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch("IONI",chCard) )
	{
		/* punch table giving ionization means */
		strcpy( punch.chPunch[punch.npunch], "IONI" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#Mean ionization distribution\n" );
	}

	else if( lgMatch(" IP ",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], " IP " );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#ionization potentials, valence shell\n" );
	}

	else if( lgMatch("LINE",chCard) )
	{
		
		/* check for keyword UNITS on line, then scan wavelength or energy units, 
		 * sets punch.chConPunEnr*/
		/* >>chng 02 jan 28, following had been commented out, and punch lines
		 * failed with insane units.  What happened? */
		ChkUnits(chCard);

		/* options are punch line structure, line intensity, line array,
		 * and line data */
		if( lgMatch("STRU",chCard) )
		{
			/* this will be line structure, emissivity vs depth */
			strcpy( punch.chPunch[punch.npunch], "LINS" );
			fprintf( punch.ipPnunit[punch.npunch], 
				"#Emission line structure.\n" );
			/* read in the list of lines to examine */
			punlin(punch.ipPnunit[punch.npunch],"READ",FALSE);
		}

		else if( lgMatch("CUMU",chCard) )
		{
			/* this will be integrated line intensity, fcn of depth */
			strcpy( punch.chPunch[punch.npunch], "LINC" );
			/* option for either relative intensity or abs luminosity */
			if( lgMatch("RELA",chCard) )
			{
				lgEOL = TRUE;
				fprintf( punch.ipPnunit[punch.npunch], 
					"#Cumulative emission line relative intensity.\n" );
			}
			else
			{
				fprintf( punch.ipPnunit[punch.npunch], 
					"#Cumulative emission line absolute intensity.\n" );
				lgEOL = FALSE;
			}
			/* read in the list of lines to examine */
			punlin(punch.ipPnunit[punch.npunch],"READ",lgEOL);
		}

		else if( lgMatch("DATA",chCard) )
		{
			/* punch line data, done in PunchLineData */
			strcpy( punch.chPunch[punch.npunch], "LIND" );
			fprintf( punch.ipPnunit[punch.npunch], "#Emission line data.\n" );
		}

		else if( lgMatch("ARRA",chCard) )
		{
			/* punch energies and luminosities of transferred lines */
			strcpy( punch.chPunch[punch.npunch], "LINA" );
			fprintf( punch.ipPnunit[punch.npunch], "#enr\tintensity\tid\ttype\n" );

		}

		else if( lgMatch("LABE",chCard) )
		{
			/* punch line labels */
			strcpy( punch.chPunch[punch.npunch], "LINL" );
			fprintf( punch.ipPnunit[punch.npunch], "#Emission line labels\n" );
		}

		else if( lgMatch("OPTI",chCard) )
		{
			/* punch line optical depths, done in PunchLineStuff, located in punchdo.c */
			strcpy( punch.chPunch[punch.npunch], "LINO" );
			fprintf( punch.ipPnunit[punch.npunch], "#species\tenergy\tlog tau\n" );

			/* the default will be to make wavelengths line in the printout, called labels,
			 * if units appears then other units will be used instead */
			strcpy( punch.chConPunEnr[punch.npunch], "labl" );

			/* check for keyword UNITS on line, then scan wavelength or energy units if present,
			 * units are copied into punch.chConPunEnr */
			if( lgMatch("UNIT",chCard) )
				ChkUnits(chCard);

			/* this is optional limit to smallest optical depths */
			ipFFmt = 5;
			punch.punarg[0][punch.npunch] = (float)pow(10.,FFmtRead(chCard,&ipFFmt, LINELENGTH,&lgEOL));
			/* this is default of 0.1 napier */
			if( lgEOL )
			{
				punch.punarg[0][punch.npunch] = 0.1f;
			}
		}

		else if( lgMatch("POPU",chCard) )
		{
			/* punch level populations */
			strcpy( punch.chPunch[punch.npunch], "LINP" );
			fprintf( punch.ipPnunit[punch.npunch], "#population information\n" );
			/* this is optional limit to smallest population to punch */
			ipFFmt = 5;
			punch.punarg[0][punch.npunch] = (float)pow(10.,FFmtRead(chCard,&ipFFmt, LINELENGTH,&lgEOL));
			/* this is default of all populations */
			if( lgEOL )
			{
				punch.punarg[0][punch.npunch] = 0.f;
			}
		}

		else if( lgMatch("INTE",chCard) )
		{
			/* this will be full set of line intensities */
			strcpy( punch.chPunch[punch.npunch], "LINI" );
			fprintf( punch.ipPnunit[punch.npunch], 
				"#Emission line intensities per unit inner area\n" );
			if( lgMatch("COLU",chCard) )
			{
				/* column is key to punch single column */
				strcpy( punch.chPunRltType, "column" );
			}
			else
			{
				/* array is key to punch large array */
				strcpy( punch.chPunRltType, "array " );
			}
			if( lgMatch("EVER",chCard) )
			{
				ipFFmt = 3;
				punch.LinEvery = (long int)FFmtRead(chCard,&ipFFmt,LINELENGTH,&lgEOL);
				punch.lgLinEvery = TRUE;
				if( lgEOL )
				{
					fprintf( ioQQQ, 
						" There must be a second number, the number of zones to print.\n" );
					puts( "[Stop in ParsePunch]" );
					cdEXIT(1);
				}
			}
			else
			{
				punch.LinEvery = ZoneCnt.nend[0];
				punch.lgLinEvery = FALSE;
			}
		}
		else
		{
			fprintf( ioQQQ, 
				" This option for PUNCH LINE is something that I dont understand.  Sorry.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	else if( lgMatch("LYMA",chCard) )
	{
		/* lyman alpha lya details */
		strcpy( punch.chPunch[punch.npunch], "LYMA" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#depth\tTau\tN2/N1\tTexc\tTe\tTex/Te\tdest\tots\n" );
	}

	else if( lgMatch(" MAP",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "MAP " );
		fprintf( punch.ipPnunit[punch.npunch], "#te, heating, cooling.\n" );
		/* do cooling space map for specified zones
		 * if no number, or <0, do map and punch out without doing first zone
		 * does map by calling punt(" map") 
		 */
		i = 5;
		MapPar.MapZone = (long)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
		if( lgEOL )
		{
			MapPar.MapZone = 1;
		}

		/* say output goes to special punch */
		ioMAP = punch.ipPnunit[punch.npunch];

		if( lgMatch("RANG",chCard) )
		{
			int lgLogOn;
			MapPar.RangeMap[0] = (float)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
			if( MapPar.RangeMap[0] <= 10. )
			{
				MapPar.RangeMap[0] = (float)pow(10.f,MapPar.RangeMap[0]);
				lgLogOn = TRUE;
			}
			else
			{
				lgLogOn = FALSE;
			}

			MapPar.RangeMap[1] = (float)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
			if( lgLogOn )
				MapPar.RangeMap[1] = (float)pow(10.f,MapPar.RangeMap[1]);

			if( lgEOL )
			{
				fprintf( ioQQQ, " There must be a zone number, followed by two temperatures, on this line.  Sorry.\n" );
				puts( "[Stop in ParsePunch]" );
				cdEXIT(1);
			}
		
		}
	}

	else if( lgMatch("MOLE",chCard) )
	{
		/* molecules, especially for pdr calculations */
		strcpy( punch.chPunch[punch.npunch], "MOLE" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#dr\t2H2/H\t2H2*/H\tCO/C\tH2O/O\tOH/O\tCH/C\t2O2/O\n" );
	}

	else if( lgMatch("OPTI",chCard) )
	{
		/* check for keyword UNITS on line, then scan wavelength or energy units if present,
		 * units are copied into punch.chConPunEnr */
		ChkUnits(chCard);
		/* punch continuum optical depths */
		strcpy( punch.chPunch[punch.npunch], "OPTI" );
		fprintf( punch.ipPnunit[punch.npunch], "#energy\ttotal\tabsorp\tscat\n" );

	}
	else if( lgMatch(" OTS",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], " OTS" );
		fprintf( punch.ipPnunit[punch.npunch], "#otscon, lin, conOpac LinOpc\n" );
	}

	else if( lgMatch("OVER",chCard) && lgMatch(" OVE",chCard) )
	{
		/* punch overview of model results */
		strcpy( punch.chPunch[punch.npunch], "OVER" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#depth\tTe\tHtot\thden\teden\t2H_2/H\tHI\tHII\tHeI\tHeII\tHeIII\tCO/C\tC1\tC2\tC3\tC4\tO1\tO2\tO3\tO4\tO5\tO6\n" );
	}

	else if( lgMatch(" PDR",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], " PDR" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#depth\tH colden\tAV\tTe\tHI/HDEN\tH2/HDEN\tH2*/HDEN\tCI/C\tCO/C\tH2O/O\n" );
	}

	else if( lgMatch("PHYS",chCard) )
	{
		/* punch physical conditions */
		strcpy( punch.chPunch[punch.npunch], "PHYS" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#PhyC    Depth       Te,       HDEN,       Ne,       HTOT,      accel\n" );
	}

	else if( lgMatch("POIN",chCard) )
	{
		/* punch out the pointers */
		punch.ipPoint = punch.ipPnunit[punch.npunch];
		punch.lgPunPoint = TRUE;
		fprintf( punch.ipPnunit[punch.npunch], "#pointers. \n" );
		/* this does not count as a punch option (really) */
		punch.ipPnunit[punch.npunch] = NULL;
		punch.npunch -= 1;
	}

	else if( lgMatch("PRES",chCard) )
	{
		/* the punch pressure command */
		strcpy( punch.chPunch[punch.npunch], "PRES" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#P depth\tPcorrect\tPcurrent\tPIn+pintg\tpgas(r0)\tpgas\tpram\tprad\tinteg\twindv\tcad\tconv?\n" );
	}

	else if( lgMatch("RADI",chCard) )
	{
		/* the punch radius command */
		strcpy( punch.chPunch[punch.npunch], "RADI" );
		fprintf( punch.ipPnunit[punch.npunch], "#NZONE, radius, r-r0, dr.\n" );
	}

	else if( lgMatch("RECO",chCard) )
	{
		if( lgMatch("COEF",chCard) )
		{
			/* recombination coefficients for everything
			 * punch.ioRecom set to zero in routine zero, non-zero value
			 * is flag to punch recombination coefficients. the output is actually
			 * produced by a series of routines, as they generate the recombination
			 * coefficients.  these include 
			 * dielsupres, helium, hydrorecom, iibod, and makerecomb*/
			punch.ioRecom = punch.ipPnunit[punch.npunch];
			punch.lgioRecom = TRUE;
			fprintf( punch.ipPnunit[punch.npunch], 
				"#recombination coefficients cm3 s-1 for current density and temperature\n" );

			/* this does not count as a punch option (really) */
			punch.ipPnunit[punch.npunch] = NULL;
			punch.npunch -= 1;
		}

		else if( lgMatch("EFFI",chCard) )
		{
			/* punch recombination efficiency */
			strcpy( punch.chPunch[punch.npunch], "RECE" );
			fprintf( punch.ipPnunit[punch.npunch], "#Recomn effic H, Heo, He+\n" );
		}

		else
		{
			fprintf( ioQQQ, " No option recognized on this punch recombination command\n" );
			fprintf( ioQQQ, " Valid options are COEFFICIENTS, AGN, and EFFICIENCY\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}

	/* punch results command, either as single column or wide array */
	else if( lgMatch("RESU",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "RESU" );
		if( lgMatch("COLU",chCard) )
		{
			/* column is key to punch single column */
			strcpy( punch.chPunRltType, "column" );
		}
		else
		{
			/* array is key to punch large array */
			strcpy( punch.chPunRltType, "array " );
		}

		/* do not change following, is used as flag in getlines */
		fprintf( punch.ipPnunit[punch.npunch], "#results of calculation\n" );
	}

	else if( lgMatch("SOUR",chCard) )
	{
		if( lgMatch("DEPT",chCard) )
		{
			/* print continuum source function as function of depth */
			strcpy( punch.chPunch[punch.npunch], "SOUD" );
			fprintf( punch.ipPnunit[punch.npunch], "#continuum source function vs depth\n" );
		}
		else if( lgMatch("SPEC",chCard) )
		{
			/* print spectrum continuum source function at 1 depth */
			strcpy( punch.chPunch[punch.npunch], "SOUS" );
			fprintf( punch.ipPnunit[punch.npunch], "#continuum source function\n" );
		}
		else
		{
			fprintf( ioQQQ, " A second keyword must appear on this line.\n" );
			fprintf( ioQQQ, " They are DEPTH and SPECTRUM.\n" );
			fprintf( ioQQQ, " Sorry.\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}


	/* punch spectrum the new form of the punch continuum, will eventually replace the standard
	 * punch continuum command */
	else if( lgMatch("SPEC",chCard) && lgMatch("TRUM",chCard) )
	{
		/* this flag is checked in PrtComment to generate a caution
		 * if continuum is punched but iterations not performed */
		punch.lgPunContinuum = TRUE;

		/* set flag for spectrum */
		strcpy( punch.chPunch[punch.npunch], "CONN" );

		fprintf( punch.ipPnunit[punch.npunch], 
			"#Cont Enr\tincid nFn\ttrans\tdiff \n" );

		/* check for keyword UNITS on line, then scan wavelength or energy units if present,
		 * units are copied into punch.chConPunEnr */
		ChkUnits(chCard);

		/* this points to which punch new continuum this is -
		 * parameters were stored in previous set cpunch commands */
		punch.punarg[0][punch.npunch] = (float)punch.cp_npun;

		++punch.cp_npun;
		/* check that limit not exceeded */
		if( punch.cp_npun > LIMPUN )
		{
			fprintf( ioQQQ, 
				"The limit to the number of PUNCH options is %ld.  Increase LIMPUN in punch.h if more are needed.\n", 
			  LIMPUN );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}

	}

	else if( lgMatch("SPEC",chCard) && lgMatch("CIAL",chCard) )
	{
		/* punch special, will call routine PunSpec */
		strcpy( punch.chPunch[punch.npunch], "SPEC" );
		fprintf( punch.ipPnunit[punch.npunch], "#Special.\n" );
	}

	else if( lgMatch("TEGR",chCard) )
	{
		/* punch tegrid */
		strcpy( punch.chPunch[punch.npunch], "TEGR" );
		fprintf( punch.ipPnunit[punch.npunch], "#zone, te, heat, cool.\n" );
	}

	else if( lgMatch("TEMP",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "TEMP" );
		fprintf( punch.ipPnunit[punch.npunch], "#DEPTH, te, dt/dr.\n" );
	}
	else if( lgMatch("TIME",chCard) )
	{
		/* output the elapsed time per zone */
		strcpy( punch.chPunch[punch.npunch], "TIME" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#zone\tdTime\tElapsed t\n" );
	}


	else if( lgMatch("TPRE",chCard) )
	{
		/* debug output from the temperature predictor in zonestart, 
		 * set with punch tpred command */
		strcpy( punch.chPunch[punch.npunch], "TPRE" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#zone  old temp,  guess Tnew, new temp    delta \n" );
	}

	else if( lgMatch("VERN",chCard) )
	{
		/* punch verner command , three optional parameters, threshold for faintest
		 * line to print, lower and upper energy bounds */

		/* punch intensities from large FeII atom */
		strcpy( punch.chPunch[punch.npunch], "VERN" );

		/* short keyword makes punch half as big */
		if( lgMatch("SHOR",chCard) )
		{
			FeII.lgShortFe2 = TRUE;
		}
		else
		{
			FeII.lgShortFe2 = FALSE;
		}

		/* first optional number changes the theshold of weakest line to print*/
		i = 5;
		/* fe2thresh is intensity relative to normalization line,
		 * normally Hbeta, and is set to zero in zero.c */
		FeII.fe2thresh = (float)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
		if( lgEOL )
		{
			FeII.fe2thresh = 0.;
		}

		/* it is a log if negative */
		if( FeII.fe2thresh < 0. )
		{
			FeII.fe2thresh = (float)pow(10.f,FeII.fe2thresh);
		}

		/* check for energy range of lines to be punched,
		 * this is to limit size of output file */
		FeII.fe2ener[0] = (float)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
		if( lgEOL )
		{
			FeII.fe2ener[0] = 0.;
		}

		FeII.fe2ener[1] = (float)FFmtRead(chCard,&i,LINELENGTH,&lgEOL);
		if( lgEOL )
		{
			FeII.fe2ener[1] = 1e8;
		}
		/* if either is negative then both are logs */
		if( FeII.fe2ener[0] < 0. || FeII.fe2ener[1] < 0. )
		{
			FeII.fe2ener[0] = (float)pow(10.f,FeII.fe2ener[0]);
			FeII.fe2ener[1] = (float)pow(10.f,FeII.fe2ener[1]);
		}
		/* entered in Ryd in above, convert to wavenumbers */
		FeII.fe2ener[0] /= (float)WAVNRYD;
		FeII.fe2ener[1] /= (float)WAVNRYD;

		/* these results are actually created by the FeIIPunchLines routine
		 * that lives in the pop371 file */
		fprintf( punch.ipPnunit[punch.npunch], "#Results from Verner FeII atom.\n" );
	}

	else if( lgMatch("WIND",chCard) )
	{
		strcpy( punch.chPunch[punch.npunch], "WIND" );
		fprintf( punch.ipPnunit[punch.npunch], 
			"#radius\tdepth\tvelocity\tTot accel\tLin accel\tCon accel\tforce mult\n" );
	}

	else
	{
		fprintf( ioQQQ, 
			" ParsePunch cannot find a recognized keyword on this PUNCH command line.\n" );
		puts( "[Stop in ParsePunch]" );
		cdEXIT(1);
	}

	/* increment total number of punch commands, */
	++punch.npunch;

#	ifdef DEBUG_FUN
	fputs( " <->ParsePunch()\n", debug_fp );
#	endif
	return;
}

/* initialize punch file pointers. KEEP THIS ROUTINE SYNCHED UP WITH THE NEXT ONE */
void PunchFilesInit(void)
{
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>PunchFilesInit()\n", debug_fp );
#	endif

	for( i=0; i < LIMPUN; i++ )
	{
		punch.ipPnunit[i] = NULL;
		/* this sets energy range of continuum, zero says to do full continuum */
		punch.cp_range_min[i] = 0.;
		punch.cp_range_max[i] = 0.;
		/* this means to keep native resolution of code, reset to another
		 * resolution with set cpunch resolution command */
		punch.cp_resolving_power[i] = 0.;
	}
	punch.ipPunConv = NULL;
	punch.lgPunConv = FALSE;
	punch.ipDRout = NULL;
	punch.lgDROn = FALSE;
	punch.ipPunH2 = NULL;
	punch.lgPunH2 = FALSE;
	punch.ipPoint = NULL;
	punch.lgPunPoint = FALSE;
	gv.QHPunchFile = NULL;
	punch.ioRecom = NULL;
	punch.lgioRecom = FALSE;
	ioMAP = NULL;

#	ifdef DEBUG_FUN
	fputs( " <->PunchFilesInit()\n", debug_fp );
#	endif
	return;
}

/* close all punch files. KEEP THIS ROUTINE SYNCHED UP WITH THE PREVIOUS ONE */
void ClosePunchFiles(void)
{
	long int i;

#	ifdef DEBUG_FUN
	fputs( "<+>ClosePunchFiles()\n", debug_fp );
#	endif

	/* close all punch units cloudy opened with punch command */
	for( i=0; i < punch.npunch; i++ )
	{
		if( punch.ipPnunit[i] != NULL )
		{
			fclose( punch.ipPnunit[i] );
			punch.ipPnunit[i] = NULL;
		}
	}

	if( punch.ipPunConv != NULL )
	{
		fclose( punch.ipPunConv );
		punch.ipPunConv = NULL;
		punch.lgPunConv = FALSE;
	}
	if( punch.ipDRout != NULL )
	{
		fclose( punch.ipDRout );
		punch.ipDRout = NULL;
		punch.lgDROn = FALSE;
	}
	if( punch.ipPunH2 != NULL )
	{
		fclose( punch.ipPunH2 );
		punch.ipPunH2 = NULL;
		punch.lgPunH2 = FALSE;
	}
	if( punch.ipPoint != NULL )
	{
		fclose( punch.ipPoint );
		punch.ipPoint = NULL;
		punch.lgPunPoint = FALSE;
	}
	if( gv.QHPunchFile != NULL )
	{
		fclose( gv.QHPunchFile );
		gv.QHPunchFile = NULL;
	}
	if( punch.ioRecom != NULL )
	{
		fclose( punch.ioRecom );
		punch.ioRecom = NULL;
		punch.lgioRecom = FALSE;
	}
	/* this file was already closed as a punch.ipPnunit, erase copy of pointer as well */
	ioMAP = NULL;

#	ifdef DEBUG_FUN
	fputs( " <->ClosePunchFiles()\n", debug_fp );
#	endif
	return;
}

/* check for keyword UNITS on line, then scan wavelength or energy units if present,
 * units are copied into punch.chConPunEnr */
static void ChkUnits( char *chCard)
{

#	ifdef DEBUG_FUN
	fputs( "<+>ChkUnits()\n", debug_fp );
#	endif

	/* option to set units for continuum energy in punch output */
	if( lgMatch("NITS",chCard) )
	{
		if( lgMatch("MICR",chCard) )
		{
			strcpy( punch.chConPunEnr[punch.npunch], "micr" );
		}
		else if( lgMatch(" KEV",chCard) )
		{
			strcpy( punch.chConPunEnr[punch.npunch], " kev" );
		}
		else if( lgMatch(" EV ",chCard) )
		{
			strcpy( punch.chConPunEnr[punch.npunch], " ev " );
		}
		else if( lgMatch("ANGS",chCard) )
		{
			strcpy( punch.chConPunEnr[punch.npunch], "angs" );
		}
		else if( lgMatch("RYDB",chCard) )
		{
			/* >>chng 01 sep 02, had been rdyb unlike proper ryd */
			strcpy( punch.chConPunEnr[punch.npunch], "ryd " );
		}
		else
		{
			fprintf( ioQQQ, " I did not recognize units on this line.\n" );
			fprintf( ioQQQ, " Units are keV, eV, Angstroms, microns\n" );
			puts( "[Stop in ParsePunch]" );
			cdEXIT(1);
		}
	}
	else
	{
		strcpy( punch.chConPunEnr[punch.npunch], "ryd " );
	}

#	ifdef DEBUG_FUN
	fputs( " <->ChkUnits()\n", debug_fp );
#	endif
	return;
}
