#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include "cddefines.h"
#include "elementnames.h"
#include "physconst.h"
#include "path.h"
#include "atomcwgt.h"
#include "version.h"
#include "grain.h"
#include "grainvar.h"
#include "rfield.h"
#include "mie.h"

/*=======================================================*
 *
 * Mie code for spherical grains.
 *
 * Calculates <pi*a^2*Q_abs> and <pi*a^2*Q_sct>*(1-<g>)
 * for arbitrary grain species and size distribution.
 *
 * This code is derived from the program cmieuvx.f
 *
 * Original author: P.G. Martin (CITA)
 *
 * Adapted for Cloudy by Peter A.M. van Hoof (UK, CITA)
 *
 *=======================================================*/


/* these are the magic numbers for the .rfi, .szd and .opc files
 * the first digit is file type, the rest is date */
const long MAGIC_RFI = 1010802L;
const long MAGIC_SZD = 2010403L;
const long MAGIC_OPC = 3011204L;

#define NINT(X) ((long)((X) < 0. ? (X)-0.5 : (X)+0.5))
#define SIGN3(X) ((X) < 0. ? -1L : ((X) == 0. ? 0L : 1L))
#define POW4(X) (POW2(X)*POW2(X))

typedef struct {
	double re;
	double im;
} complex;

/* these are the absolute smallest and largest grain sizes we will
 * consider (in micron). the lower limit gives a grain with on the
 * order of one atom in it, so it is physically motivated. the upper
 * limit comes from the series expansions used in the mie theory,
 * they will have increasingly more problems converging for larger
 * grains, so this limit is numerically motivated */
const double SMALLEST_GRAIN = 0.0001*(1.-10.*DBL_EPSILON);
const double LARGEST_GRAIN = 10.*(1.+10.*DBL_EPSILON);

/* maximum no. of parameters for grain size distribution */
#define NSD 7

/* these are the indices into the parameter array a[NSD],
 * NB NB -- the numbers defined below should range from 0 to NSD-1 */
const int ipSize  = 0; /* single size */
const int ipBLo   = 0; /* lower bound */
const int ipBHi   = 1; /* upper bound */
const int ipExp   = 2; /* exponent for powerlaw */
const int ipBeta  = 3; /* beta parameter for powerlaw */
const int ipSLo   = 4; /* scale size for lower exp. cutoff */
const int ipSHi   = 5; /* scale size for upper exp. cutoff */
const int ipAlpha = 6; /* alpha parameter for exp. cutoff */
const int ipGCen  = 2; /* center of gaussian distribution */
const int ipGSig  = 3; /* 1-sigma width of gaussian distribution */

/* these are all the size distribution cases we support */
typedef enum {
	SD_ILLEGAL, SD_SINGLE_SIZE, SD_POWERLAW, SD_EXP_CUTOFF1, SD_EXP_CUTOFF2,
	SD_EXP_CUTOFF3, SD_LOG_NORMAL, SD_LIN_NORMAL, SD_TABLE
} sd_type;

typedef struct {
	double a[NSD];    /* parameters for size distribution */
	double lim[2];    /* holds lower and upper size limit for entire distribution */
	double clim[2];   /* holds lower and upper size limit for current bin */
	double *xx;       /* xx[nn]: abcissas for Gauss quadrature on [-1,1] */
	double *aa;       /* aa[nn]: weights for Gauss quadrature on [-1,1] */
	double *rr;       /* rr[nn]: abcissas for Gauss quadrature */
	double *ww;       /* ww[nn]: weights for Gauss quadrature */
	double unity;     /* normalization for integrals over size distribution */
	double unity_bin; /* normalization for integrals over size distribution */
	double radius;    /* average grain radius <a^3>/<a^2> */
	double area;      /* integrated grain surface area Int(4pi*a^2) */
	double vol;       /* integrated grain volume Int(4pi/3*a^3) */
	double *ln_a;     /* ln(a)[npts]: log of grain radii for user-supplied size distr */
	double *ln_a4dNda;/* ln(a^4*dN/da)[npts]: log of user-supplied size distr */
	sd_type sdCase;   /* SD_SINGLE_SIZE, SD_POWERLAW, ... */
	long int cPart;   /* current partition no. for size distribution */
	long int nPart;   /* total no. of partitions for size distribution */
	long int nmul;    /* multiplier for obtaining no. of abscissas in gaussian quadrature */
	long int nn;      /* no. of abscissas used in gaussian quadrature (per partition) */
	long int npts;    /* no. of points in user-supplied size distr */
	int lgLogScale;   /* use logarithmic mesh for integration over size ? */
} sd_data;

/* maximum no. of principal axes for crystalline grains */
#define NAX 3

typedef struct {
	double *wavlen[NAX];    /* wavelength grid for rfi for all axes (micron) */
	complex *n[NAX];        /* refractive index n for all axes */
	double *nr1[NAX];       /* re(n)-1 for all axes */
	double wt[NAX];         /* relative weight of each axis */
	double abun;            /* abundance of grain molecule rel. to hydrogen for max depletion */
	double depl;            /* depletion efficiency */
	double elmAbun[LIMELM]; /* abundances of constituent elements rel. to hydrogen */
	double mol_weight;      /* molecular weight of grain molecule (amu) */
	double atom_weight;     /* molecular weight of grain molecule per atom (amu) */
	double rho;             /* specific weight (g/cm^3) */
	double norm;            /* number of protons in plasma per average grain */
	double work;            /* work function (Ryd) */
	double bandgap;         /* gap between valence and conduction band (Ryd) */
	double therm_eff;       /* efficiency of thermionic emission, between 0 and 1 */
	double subl_temp;       /* sublimation temperature (K) */
	long int nAxes;         /* no. of principal axes for this grain */
	long int ndata[NAX];    /* no. of wavelength points for each axis */
	long int mat_type;      /* material type, determines enthalpy function */
} grain_data;

/* used in mie_read_rfi, mie_read_opc, mie_write_opc */
/* >>chng 01 oct 23, to DATA_PATH_LENGTH_2 the largest possible path */
#define LINELEN 140

/* maximum size for grain type labels */
#define LABELSUB1 3
#define LABELSUB2 5
#define LABELSIZE (LABELSUB1+LABELSUB2+4)

static void mie_auxiliary(/*@partial@*/sd_data*,/*@in@*/const char*);
static void mie_integrate(/*@partial@*/sd_data*,double,double,/*@out@*/double*,int);
static void mie_cs_size_distr(double,sd_data,grain_data,long int,/*@out@*/double*,
			      /*@out@*/double*,/*@out@*/double*,/*@out@*/int*);
static void mie_cs(double,double,double,double,double,/*@out@*/double*,/*@out@*/double*,
		   /*@out@*/double*,/*@out@*/int*);
static double size_distr(double,sd_data);
static double search_limit(double,double,double,sd_data);
static void mie_calc_ial(/*@in@*/grain_data,long,/*@out@*/double[],/*@in@*/const char*,/*@in@*/int*);
static void mie_repair(/*@in@*/const char*,long,int,int,/*@in@*/float[],double[],/*@in@*/int[],
		       int,/*@in@*/int*);
static double mie_find_slope(/*@in@*/const float[],/*@in@*/const double[],/*@in@*/const int[],
			     long,long,int,int,/*@in@*/int*);
static void mie_read_rfi(/*@in@*/const char*,/*@out@*/grain_data*);
static void mie_read_szd(/*@in@*/const char*,/*@out@*/sd_data*);
static void mie_read_long(/*@in@*/const char*,/*@in@*/const char[],/*@out@*/long int*,int,long int);
static void mie_read_float(/*@in@*/const char*,/*@in@*/const char[],/*@out@*/float*,int,long int);
static void mie_read_double(/*@in@*/const char*,/*@in@*/const char[],/*@out@*/double*,int,long int);
static void mie_read_form(/*@in@*/const char*,/*@out@*/double[],/*@out@*/double*,/*@out@*/double*);
static void mie_read_word(/*@in@*/const char[],/*@out@*/char[],long,int);
static void mie_next_data(/*@in@*/const char*,/*@in@*/FILE*,/*@out@*/char*,/*@in@*/long int*);
static void mie_next_line(/*@in@*/const char*,/*@in@*/FILE*,/*@out@*/char*,/*@in@*/long int*);

/*=======================================================*/
/* the following four routines are the core of the Mie code supplied by Peter Martin */

static void sinpar(double,double,double,/*@out@*/double*,/*@out@*/double*,/*@out@*/double*,
		   /*@out@*/double*,/*@out@*/double*,/*@out@*/long*);
static void anomal(double,/*@out@*/double*,/*@out@*/double*,/*@out@*/double*,/*@out@*/double*,double,double);
static void bigk(complex,/*@out@*/complex*);
static void dftori(/*@out@*/double*,/*@out@*/double*,double,double);

/*=======================================================*/
/* some things needed to handle complex variables */

static complex ftocm(double,double);
static complex cmneg(complex);
static double cmabs(complex);
static complex cmadd(complex,complex);
static complex cmsub(complex,complex);
static complex cmmul(complex,complex);
static complex cmdiv(complex,complex);
static complex cmexp(complex);


/* >>chng 01 oct 29, introduced gv.bin[nd]->cnv_H_pGR, cnv_GR_pH, PvH */

void mie_write_opc(/*@in@*/ const char *rfi_file,
		   /*@in@*/ const char *szd_file,
		   long int nbin)
{
	int Error = 0,
		/* >>chng 01 aug 01, MALLOC the space, no more ncell */
	  *ErrorIndex/*[NC_ELL]*/,
	  lgErr,
	  lgErrorOccured,
	  lgGreyGrains,
	  lgWarning;
	long int i,
	  j,
	  p;
	double **acs_abs,
	  **acs_sct,
	  **a1g,
	  *inv_att_len,
	  cosb,
	  cs_abs,
	  cs_sct,
	  wavlen;
	char chGrainLabel[LABELSIZE+1],
	  ext[3],
	  fnam[LINELEN],
	  fnam2[LINELEN],
	  hlp1[LABELSUB1+2],
	  hlp2[LABELSUB2+2],
	  *str,
	  string[LINELEN];
	sd_data sd;
	grain_data gd,
	  gd2;
	time_t timer;
	FILE *fdes;

	/* no. of logarithmic intervals in table printout of size distribution function */
	const long NPTS_TABLE = 100L;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_write_opc()\n", debug_fp );
#	endif

	/* >>chng 01 aug 22, MALLOC this space */
	if( (ErrorIndex = (int*)MALLOC((size_t)(rfield.nupper*sizeof(int)) ) ) == NULL )
		bad_malloc();

	gd.nAxes = 0;
	for( j=0; j < NAX; j++ ) 
	{
		gd.wavlen[j] = NULL;
		gd.n[j] = NULL;
		gd.nr1[j] = NULL;
	}
	gd2.nAxes = 0;
	for( j=0; j < NAX; j++ ) 
	{
		gd2.wavlen[j] = NULL;
		gd2.n[j] = NULL;
		gd2.nr1[j] = NULL;
	}
	sd.ln_a = NULL;
	sd.ln_a4dNda = NULL;

	mie_read_szd( szd_file , &sd );

	sd.nPart = ( sd.sdCase == SD_SINGLE_SIZE ) ? 1 : nbin;
	if( sd.nPart <= 0 || sd.nPart >= 100 ) 
	{
		fprintf( ioQQQ, " Illegal number of size distribution bins: %ld\n",sd.nPart );
		fprintf( ioQQQ, " The number should be between 1 and 99.\n" );
		puts( "[Stop in mie_write_opc]" );
		cdEXIT(1);
	}
	sd.lgLogScale = TRUE;

	mie_read_rfi( rfi_file , &gd );

	lgGreyGrains = (gd.wavlen[0] == NULL);

	lgWarning = FALSE;

	/* generate output file name from input file names */
	strcpy(fnam,rfi_file);
	str = strstr(fnam,".");

	if( str != NULL )
		*str = '\0';

	strcpy(fnam2,szd_file);
	str = strstr(fnam2,".");

	if( str != NULL )
		*str = '\0';

	if( sd.sdCase != SD_SINGLE_SIZE ) 
	{
		sprintf(ext,"%02ld",nbin);
		strcat(strcat(strcat(strcat(strcat(fnam,"_"),fnam2),"_"),ext),".opc");
	}
	else 
	{
		strcat(strcat(strcat(fnam,"_"),fnam2),".opc");
	}

	fprintf( ioQQQ, "\n Starting mie_write_opc, output will be written to %s\n\n",fnam );

	mie_auxiliary(&sd,"init");

	/* number of protons in plasma per average grain volume */
	gd.norm = sd.vol*gd.rho/(ATOMIC_MASS_UNIT*gd.mol_weight*gd.abun*gd.depl);

	acs_abs = (double **)MALLOC(sizeof(double *)*(unsigned)sd.nPart);
	acs_sct = (double **)MALLOC(sizeof(double *)*(unsigned)sd.nPart);
	a1g = (double **)MALLOC(sizeof(double *)*(unsigned)sd.nPart);
	inv_att_len = (double *)MALLOC(sizeof(double)*(unsigned)rfield.nupper);

	if( acs_abs == NULL || acs_sct == NULL || a1g == NULL || inv_att_len == NULL ) 
	{
		fprintf( ioQQQ, " Could not MALLOC arrays\n" );
		puts( "[Stop in mie_write_opc]" );
		cdEXIT(1);
	}
	for( p=0; p < sd.nPart; p++ ) 
	{
		acs_abs[p] = (double *)MALLOC(sizeof(double)*(unsigned)rfield.nupper);
		acs_sct[p] = (double *)MALLOC(sizeof(double)*(unsigned)rfield.nupper);
		a1g[p] = (double *)MALLOC(sizeof(double)*(unsigned)rfield.nupper);
		if( acs_abs[p] == NULL || acs_sct[p] == NULL || a1g[p] == NULL ) 
		{
			fprintf( ioQQQ, " Could not MALLOC arrays\n" );
			puts( "[Stop in mie_write_opc]" );
			cdEXIT(1);
		}
	}

	if( (fdes = fopen( fnam, "w" )) == NULL ) 
	{
		fprintf( ioQQQ, " Could not open %s for writing\n", fnam );
		puts( "[Stop in mie_write_opc]" );
		cdEXIT(1);
	}
	lgErr = FALSE;

	(void)time(&timer);
	lgErr = lgErr || ( fprintf(fdes,"# this file was created by Cloudy %s (%s) on %s",
				   version.chVersion,version.chDate,ctime(&timer)) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# ===========================================\n#\n") < 0 );
#ifdef IGNORE_ASYMMETRY
	lgErr = lgErr || ( fprintf(fdes,"%12ld # magic number opacity file\n",0L) < 0 );
#else
	lgErr = lgErr || ( fprintf(fdes,"%12ld # magic number opacity file\n",MAGIC_OPC) < 0 );
#endif
	lgErr = lgErr || ( fprintf(fdes,"%12ld # magic number refractive index file\n",MAGIC_RFI) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%12ld # magic number size distribution file\n",MAGIC_SZD) < 0 );

	/* generate grain label for Cloudy output
	 * adjust LABELSIZE in mie.h when the format defined below is changed ! */
	strncpy(hlp1,fnam,(size_t)(LABELSUB1+1));
	hlp1[LABELSUB1+1] = '\0';
	str = strstr(hlp1,"-");

	if( str != NULL )
		*str = '\0';

	strncpy(hlp2,fnam2,(size_t)(LABELSUB2+1));
	hlp2[LABELSUB2+1] = '\0';
	str = strstr(hlp2,"-");

	if( str != NULL )
		*str = '\0';

	strcpy(chGrainLabel," ");
	if( sd.nPart > 1 ) 
	{
		hlp1[LABELSUB1] = '\0';
		hlp2[LABELSUB2] = '\0';
		strcat(strcat(strcat(strcat(chGrainLabel,hlp1),"-"),hlp2),"xx");
		lgErr = lgErr || ( fprintf(fdes,"%-12.12s # grain type label, xx will be replaced by bin no.\n",
					   chGrainLabel) < 0 );
	}
	else 
	{
		strcat(strcat(strcat(chGrainLabel,hlp1),"-"),hlp2);
		lgErr = lgErr || ( fprintf(fdes,"%-12.12s # grain type label\n", chGrainLabel) < 0 );
	}

	lgErr = lgErr || ( fprintf(fdes,"%.6e # specific weight (g/cm^3)\n",gd.rho) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # molecular weight of grain molecule (amu)\n",gd.mol_weight) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # average molecular weight per atom (amu)\n", gd.atom_weight) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # abundance of grain molecule at max depletion\n",gd.abun) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # depletion efficiency\n",gd.depl) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # average grain radius <a^3>/<a^2>, full size distr (cm)\n",
			   3.*sd.vol/sd.area) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # average grain surface area <4pi*a^2>, full size distr (cm^2)\n",
			   sd.area) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # average grain volume <4/3pi*a^3>, full size distr (cm^3)\n",
			   sd.vol) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain radius Int(a) per H, full size distr (cm/H)\n",
			   sd.radius/gd.norm) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain area Int(4pi*a^2) per H, full size distr (cm^2/H)\n",
			   sd.area/gd.norm) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain vol Int(4/3pi*a^3) per H, full size distr (cm^3/H)\n",
			   sd.vol/gd.norm) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # work function (Ryd)\n",gd.work) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # gap between valence and conduction band (Ryd)\n",gd.bandgap) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # efficiency of thermionic emission\n",gd.therm_eff) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%.6e # sublimation temperature (K)\n",gd.subl_temp) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%12ld # material type, 1=carbonaceous, 2=silicate\n",gd.mat_type) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"#\n# abundances of constituent elements rel. to hydrogen\n#\n") < 0 );

	for( i=0; i < LIMELM; i++ ) 
	{
		lgErr = lgErr || ( fprintf(fdes,"%.6e # %s\n",gd.elmAbun[i],elementnames.chElementSym[i]) < 0 );
	}

	if( sd.sdCase != SD_SINGLE_SIZE )
	{
		lgErr = lgErr || ( fprintf(fdes,"#\n# entire size distribution, amin=%.5f amax=%.5f micron\n",
					   sd.lim[ipBLo],sd.lim[ipBHi]) < 0 );
		lgErr = lgErr || ( fprintf(fdes,"#\n%.6e # ratio a_max/a_min in each size bin\n",
					   pow(sd.lim[ipBHi]/sd.lim[ipBLo],1./(double)sd.nPart) ) < 0 );
		lgErr = lgErr || ( fprintf(fdes,"#\n# size distribution function\n#\n") < 0 );
		lgErr = lgErr || ( fprintf(fdes,"%12ld # number of table entries\n#\n",NPTS_TABLE+1) < 0 );
		lgErr = lgErr || ( fprintf(fdes,"# ============================\n") < 0 );
		lgErr = lgErr || ( fprintf(fdes,"# size (micr) a^4*dN/da (cm^3/H)\n#\n") < 0 );
		for( i=0; i <= NPTS_TABLE; i++ )
		{
			double radius, a4dNda;
			radius = sd.lim[ipBLo]*exp((double)i/(double)NPTS_TABLE*log(sd.lim[ipBHi]/sd.lim[ipBLo]));
			radius = MAX2(MIN2(radius,sd.lim[ipBHi]),sd.lim[ipBLo]);
			a4dNda = POW4(radius)*size_distr(radius,sd)/gd.norm*1.e-12/sd.unity;
			lgErr = lgErr || ( fprintf(fdes,"%.6e %.6e\n",radius,a4dNda) < 0 );
		}
	}
	else
	{
		lgErr = lgErr || ( fprintf(fdes,"#\n") < 0 );
		lgErr = lgErr || ( fprintf(fdes,"%.6e # a_max/a_min = 1 for single sized grain\n", 1. ) < 0 );
		lgErr = lgErr || ( fprintf(fdes,"%12ld # no size distribution table\n",0L) < 0 );
	}

	lgErr = lgErr || ( fprintf(fdes,"#\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%12ld # rfield.nupper\n",rfield.nupper) < 0 );
	lgErr = lgErr || ( fprintf(fdes,"%12ld # number of size distr. bins\n#\n",sd.nPart) < 0 );

	for( p=0; p < sd.nPart; p++ ) 
	{
		sd.cPart = p;

		mie_auxiliary(&sd,"step");

		if( sd.nPart > 1 ) 
		{
			/* >>chng 01 mar 20, creating mie_integrate introduced a change in the normalization
			 * of sd.radius, sd.area, and sd.vol; they now give average quantities for this bin.
			 * gd.norm converts average quanties to integrated quantities per H assuming the
			 * number of grains for the entire size distribution, hence multiplication by frac is
			 * needed to convert to the number of grains in this particular size bin, PvH */
			double frac = sd.unity_bin/sd.unity;
			fprintf( ioQQQ, " Starting size bin %ld, amin=%.5f amax=%.5f micron\n",
				 p+1,sd.clim[ipBLo],sd.clim[ipBHi] );
			lgErr = lgErr || ( fprintf(fdes,"# size bin %ld, amin=%.5f amax=%.5f micron\n",
						   p+1,sd.clim[ipBLo],sd.clim[ipBHi]) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # average grain ",3.*sd.vol/sd.area) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"radius <a^3>/<a^2>, this bin (cm)\n") < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # average ",sd.area) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"grain area <4pi*a^2>, this bin (cm^2)\n") < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # average ",sd.vol) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"grain volume <4/3pi*a^3>, this bin (cm^3)\n") < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain ",sd.radius*frac/gd.norm) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"radius Int(a) per H, this bin (cm/H)\n") < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain area ",sd.area*frac/gd.norm) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"Int(4pi*a^2) per H, this bin (cm^2/H)\n") < 0 );
			lgErr = lgErr || ( fprintf(fdes,"%.6e # total grain volume ",sd.vol*frac/gd.norm) < 0 );
			lgErr = lgErr || ( fprintf(fdes,"Int(4/3pi*a^3) per H, this bin (cm^3/H)\n#\n") < 0 );
		}

		lgErrorOccured = FALSE;

		for( i=0; i < rfield.nupper; i++ ) 
		{
			if( lgGreyGrains ) 
			{
				ErrorIndex[i] = 0;
				acs_abs[p][i] = 1.3121e-23*gd.norm;
				acs_sct[p][i] = 2.6242e-23*gd.norm;
				a1g[p][i] = 1.;
			}
			else 
			{
				wavlen = WAVNRYD/rfield.anu[i]*1.e4;

				ErrorIndex[i] = 0;
				acs_abs[p][i] = 0.;
				acs_sct[p][i] = 0.;
				a1g[p][i] = 0.;

				for( j=0; j < gd.nAxes; j++ ) 
				{
					mie_cs_size_distr(wavlen,sd,gd,j,&cs_abs,&cs_sct,&cosb,&Error);
					ErrorIndex[i] = MAX2(ErrorIndex[i],Error);
					acs_abs[p][i] += cs_abs*gd.wt[j];
					acs_sct[p][i] += cs_sct*gd.wt[j];
					a1g[p][i] += cs_sct*(1.-cosb)*gd.wt[j];
				}

				if( ErrorIndex[i] > 0 ) 
				{
					ErrorIndex[i] = MIN2(ErrorIndex[i],2);
					lgErrorOccured = TRUE;
				}

				switch( ErrorIndex[i] ) 
				{
				/*lint -e616 */
				case 2:
					acs_abs[p][i] = 0.;
					acs_sct[p][i] = 0.;
					/* controls is supposed to flow to the next case */
				case 1:
					a1g[p][i] = 0.;
					break;
				/*lint +e616 */
				case 0:
					a1g[p][i] /= acs_sct[p][i];
					break;
				default:
					fprintf( ioQQQ, " Insane value for ErrorIndex: %d\n", ErrorIndex[i] );
					ShowMe();
					puts( "[Stop in mie_write_opc]" );
					cdEXIT(1);
				}

				/* sanity checks */
				if( ErrorIndex[i] < 2 )
					assert( acs_abs[p][i] > 0. && acs_sct[p][i] > 0. );
				if( ErrorIndex[i] < 1 )
					assert( a1g[p][i] > 0. );
			}
		}

		/* extrapolate/interpolate for missing data */
		if( lgErrorOccured ) 
		{
			strcpy(string,"absorption cs");
			mie_repair(string,rfield.nupper,2,0,rfield.anu,acs_abs[p],ErrorIndex,FALSE,&lgWarning);
			strcpy(string,"scattering cs");
			mie_repair(string,rfield.nupper,2,1,rfield.anu,acs_sct[p],ErrorIndex,FALSE,&lgWarning);
			strcpy(string,"asymmetry parameter");
			mie_repair(string,rfield.nupper,1,1,rfield.anu,a1g[p],ErrorIndex,TRUE,&lgWarning);
		}

		for( i=0; i < rfield.nupper; i++ ) 
		{
			acs_abs[p][i] /= gd.norm;
#ifdef IGNORE_ASYMMETRY
			/* Do not multiply with (1-g); this is useful for calculating
			 * extinction curves, but should never be used in Cloudy! To avoid
			 * accidental use, the magic number has been set to zero above */
			acs_sct[p][i] /= gd.norm;
#else
			acs_sct[p][i] *= a1g[p][i]/gd.norm;
#endif
		}
#if 0
		{
			FILE* ftmp;
			char name[20] = "asymxx";
			sprintf(&name[4],"%2.2li",p+1);
			ftmp = fopen(name,"w");
			for( i=0; i < rfield.nupper; i++ ) 
			{
				fprintf( ftmp, "%.6e %.6e\n", rfield.anu[i],a1g[p][i]);
			}
			fclose(ftmp);
		}
#endif

		mie_auxiliary(&sd,"cleanup");
	}

	lgErr = lgErr || ( fprintf(fdes,"#\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# ===========================================\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# anu (Ryd) abs_cs_01 (cm^2/H) abs_cs_02.....\n#\n") < 0 );

	for( i=0; i < rfield.nupper; i++ ) 
	{
		lgErr = lgErr || ( fprintf(fdes,"%.6e ",rfield.anu[i]) < 0 );
		for( p=0; p < sd.nPart; p++ ) 
		{
			lgErr = lgErr || ( fprintf(fdes,"%.6e ",acs_abs[p][i]) < 0 );
		}
		lgErr = lgErr || ( fprintf(fdes,"\n") < 0 );
	}

	lgErr = lgErr || ( fprintf(fdes,"#\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# ===========================================\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# anu (Ryd) sct_cs_01 (cm^2/H) sct_cs_02.....\n#\n") < 0 );

	for( i=0; i < rfield.nupper; i++ ) 
	{
		lgErr = lgErr || ( fprintf(fdes,"%.6e ",rfield.anu[i]) < 0 );
		for( p=0; p < sd.nPart; p++ ) 
		{
			lgErr = lgErr || ( fprintf(fdes,"%.6e ",acs_sct[p][i]) < 0 );
		}
		lgErr = lgErr || ( fprintf(fdes,"\n") < 0 );
	}

	fprintf( ioQQQ, " Starting calculation of inverse attenuation length\n" );
	strcpy(string,"inverse attenuation length");
	if( lgGreyGrains ) 
	{
		mie_read_rfi("silicate.rfi",&gd2);
		mie_calc_ial(gd2,rfield.nupper,inv_att_len,string,&lgWarning);
	}
	else 
	{
		mie_calc_ial(gd,rfield.nupper,inv_att_len,string,&lgWarning);
	}

	lgErr = lgErr || ( fprintf(fdes,"#\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# ===========================================\n") < 0 );
	lgErr = lgErr || ( fprintf(fdes,"# anu (Ryd) inverse attenuation length (cm)\n#\n") < 0 );

	for( i=0; i < rfield.nupper; i++ ) 
	{
		lgErr = lgErr || ( fprintf(fdes,"%.6e %.6e\n",rfield.anu[i],inv_att_len[i]) < 0 );
	}

	fclose(fdes);

	if( lgErr ) 
	{
		fprintf( ioQQQ, "\n Error writing file: %s\n", fnam );
		if( remove(fnam) == 0 )
		{
			fprintf( ioQQQ, " The file has been removed\n" );
			puts( "[Stop in mie_write_opc]" );
			cdEXIT(1);
		}
	}
	else 
	{
		fprintf( ioQQQ, "\n Opacity file %s written succesfully\n\n", fnam );
		if( lgWarning )
		{
			fprintf( ioQQQ, "\n !!! Warnings were detected !!!\n\n" );
		}
	}

	for( p=0; p < sd.nPart; p++ ) 
	{
		free(a1g[p]);
		free(acs_sct[p]);
		free(acs_abs[p]);
	}

	free(inv_att_len);
	free(a1g);
	free(acs_sct);
	free(acs_abs);

	/* these arrays were allocated in mie_read_szd */
	if( sd.ln_a != NULL )
		free(sd.ln_a);
	if( sd.ln_a4dNda != NULL )
		free(sd.ln_a4dNda);

	/* these arrays were allocated in mie_read_rfi */
	for( j=0; j < gd2.nAxes; j++ ) 
	{
		if( gd2.nr1[j] != NULL )
			free(gd2.nr1[j]);
		if( gd2.n[j] != NULL )
			free(gd2.n[j]);
		if( gd2.wavlen[j] != NULL )
			free(gd2.wavlen[j]);
	}

	for( j=0; j < gd.nAxes; j++ ) 
	{
		if( gd.nr1[j] != NULL )
			free(gd.nr1[j]);
		if( gd.n[j] != NULL )
			free(gd.n[j]);
		if( gd.wavlen[j] != NULL )
			free(gd.wavlen[j]);
	}

	free( ErrorIndex );

#	ifdef DEBUG_FUN
	fputs( " <->mie_write_opc()\n", debug_fp );
#	endif
	return;
}

static void mie_auxiliary(/*@partial@*/ sd_data *sd,
			  /*@in@*/ const char *auxCase)
{
	double amin,
	  amax,
	  delta,
	  oldvol,
	  step;

	/* desired relative accuracy of integration over size distribution */
	const double TOLER = 1.e-3;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_auxiliary()\n", debug_fp );
#	endif
	if( strcmp(auxCase,"init") == 0 )
	{
		sd->xx = NULL;
		sd->aa = NULL;
		sd->rr = NULL;
		sd->ww = NULL;

		/* this is the initial estimate for the multiplier needed to get the
		 * number of abscissas in the gaussian quadrature, the correct value
		 * will be iterated below */
		sd->nmul = 1;

		/* calculate average grain surface area and volume over size distribution */
		switch( sd->sdCase ) 
		{
		case SD_SINGLE_SIZE:
			sd->radius = sd->a[ipSize]*1.e-4;
			sd->area = 4.*PI*POW2(sd->a[ipSize])*1.e-8;
			sd->vol = 4./3.*PI*POW3(sd->a[ipSize])*1.e-12;
			break;
		case SD_POWERLAW:
		case SD_EXP_CUTOFF1:
		case SD_EXP_CUTOFF2:
		case SD_EXP_CUTOFF3:
		case SD_LOG_NORMAL:
		case SD_LIN_NORMAL:
		case SD_TABLE:
			/* set up Gaussian quadrature for entire size range,
			 * first estimate no. of abscissas needed */
			amin = sd->lgLogScale ? log(sd->lim[ipBLo]) : sd->lim[ipBLo];
			amax = sd->lgLogScale ? log(sd->lim[ipBHi]) : sd->lim[ipBHi];

			sd->clim[ipBLo] = sd->lim[ipBLo];
			sd->clim[ipBHi] = sd->lim[ipBHi];

			/* iterate nmul until the integrated volume has converged sufficiently */
			oldvol= 0.;
			do 
			{
				sd->nmul *= 2;
				mie_integrate(sd,amin,amax,&sd->unity,TRUE);
				delta = fabs(sd->vol-oldvol)/sd->vol;
				oldvol = sd->vol;
			} while( sd->nmul <= 1024 && delta > TOLER );

			if( delta > TOLER )
			{
				fprintf( ioQQQ, " could not converge integration of size distribution\n" );
				puts( "[Stop in mie_auxiliary]" );
				cdEXIT(1);
			}

			/* we can safely reduce nmul by a factor of 2 and
			 * still reach a relative accuracy of TOLER */
			sd->nmul /= 2;
			mie_integrate(sd,amin,amax,&sd->unity,TRUE);
			break;
		default:
			fprintf( ioQQQ, " insane case for grain size distribution: %d\n" , sd->sdCase );
			ShowMe();
			puts( "[Stop in mie_auxiliary]" );
			cdEXIT(1);
		}
	}
	else if( strcmp(auxCase,"step") == 0 ) 
	{
		/* calculate average grain surface area and volume over size bin */
		switch( sd->sdCase ) 
		{
		case SD_SINGLE_SIZE:
			break;
		case SD_POWERLAW:
		case SD_EXP_CUTOFF1:
		case SD_EXP_CUTOFF2:
		case SD_EXP_CUTOFF3:
		case SD_LOG_NORMAL:
		case SD_LIN_NORMAL:
		case SD_TABLE:
			amin = sd->lgLogScale ? log(sd->lim[ipBLo]) : sd->lim[ipBLo];
			amax = sd->lgLogScale ? log(sd->lim[ipBHi]) : sd->lim[ipBHi];
			step = (amax - amin)/(double)sd->nPart;
			amin = amin + (double)sd->cPart*step;
			amax = MIN2(amax,amin + step);

			sd->clim[ipBLo] = sd->lgLogScale ? exp(amin) : amin;
			sd->clim[ipBHi] = sd->lgLogScale ? exp(amax) : amax;

			mie_integrate(sd,amin,amax,&sd->unity_bin,FALSE);

			break;
		default:
			fprintf( ioQQQ, " insane case for grain size distribution: %d\n" , sd->sdCase );
			ShowMe();
			puts( "[Stop in mie_auxiliary]" );
			cdEXIT(1);
		}
	}
	else if( strcmp(auxCase,"cleanup") == 0 ) 
	{
		if( sd->xx != NULL )
			free(sd->xx);
		sd->xx = NULL;
		if( sd->aa != NULL )
			free(sd->aa);
		sd->aa = NULL;
		if( sd->rr != NULL )
			free(sd->rr);
		sd->rr = NULL;
		if( sd->ww != NULL )
			free(sd->ww);
		sd->ww = NULL;
	}
	else 
	{
		fprintf( ioQQQ, " mie_auxiliary called with insane argument: %s\n", auxCase );
		ShowMe();
		puts( "[Stop in mie_auxiliary]" );
		cdEXIT(1);
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_auxiliary()\n", debug_fp );
#	endif
	return;
}

static void mie_integrate(/*@partial@*/ sd_data *sd,
			  double amin,
			  double amax,
			  /*@out@*/ double *normalization,
			  int lgFreeMem)
{
	long int j;
	double unity;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_integrate()\n", debug_fp );
#	endif

	/* set up Gaussian quadrature for size range,
	 * first estimate no. of abscissas needed */
	sd->nn = sd->nmul*((long)(2.*log(sd->clim[ipBHi]/sd->clim[ipBLo])) + 1);
	sd->nn = MIN2(MAX2(sd->nn,2*sd->nmul),4096);
	sd->xx = (double *)MALLOC(sizeof(double)*(unsigned)sd->nn);
	sd->aa = (double *)MALLOC(sizeof(double)*(unsigned)sd->nn);
	sd->rr = (double *)MALLOC(sizeof(double)*(unsigned)sd->nn);
	sd->ww = (double *)MALLOC(sizeof(double)*(unsigned)sd->nn);
	if( sd->xx == NULL || sd->aa == NULL || sd->rr == NULL || sd->ww == NULL ) 
	{
		fprintf( ioQQQ, " Could not MALLOC arrays\n" );
		puts( "[Stop in mie_integrate]" );
		cdEXIT(1);
	}
	gauss_legendre(sd->nn,sd->xx,sd->aa);
	gauss_init(sd->nn,amin,amax,sd->xx,sd->aa,sd->rr,sd->ww);

	/* now integrate surface area and volume */
	unity = 0.;
	sd->radius = 0.;
	sd->area = 0.;
	sd->vol = 0.;

	for( j=0; j < sd->nn; j++ ) 
	{
		double weight;

		/* use extra factor of size in weights when we use logarithmic mesh */
		if( sd->lgLogScale ) 
		{
			sd->rr[j] = exp(sd->rr[j]);
			sd->ww[j] *= sd->rr[j];
		}
		weight = sd->ww[j]*size_distr(sd->rr[j],*sd);
		unity += weight;
		sd->radius += weight*sd->rr[j];
		sd->area += weight*POW2(sd->rr[j]);
		sd->vol += weight*POW3(sd->rr[j]);
	}
	*normalization = unity;
	sd->radius *= 1.e-4/unity;
	sd->area *= 4.*PI*1.e-8/unity;
	sd->vol *= 4./3.*PI*1.e-12/unity;

	if( lgFreeMem )
	{
		if( sd->xx != NULL )
			free(sd->xx);
		sd->xx = NULL;
		if( sd->aa != NULL )
			free(sd->aa);
		sd->aa = NULL;
		if( sd->rr != NULL )
			free(sd->rr);
		sd->rr = NULL;
		if( sd->ww != NULL )
			free(sd->ww);
		sd->ww = NULL;
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_integrate()\n", debug_fp );
#	endif
	return;
}

/* read in the *.opc file with opacities and other relevant information */
void mie_read_opc(/*@in@*/const char *fnam,
		  GrainPar gp)
{
	int res;
	long int dl,
	  i,
	  ipZ,
	  j,
	  magic,
	  nbin,
	  nd,
	  nd2,
	  nup;
	float RefAbund[LIMELM],
	  VolTotal;
	double anu;
	double RadiusRatio;
	char chLine[DATA_PATH_LENGTH_2],
	  *str,
	  string[DATA_PATH_LENGTH_2];
	FILE *io2;

	/* if a_max/a_min in a single size bin is less than
	 * RATIO_MAX quantum heating will be turned on by default */
	const double RATIO_MAX = pow(100.,1./3.);

#	ifdef DEBUG_FUN
	fputs( "<+>mie_read_opc()\n", debug_fp );
#	endif

	/* true is special path set, false if file in same dir */
	if( lgDataPathSet )
	{
		/* path is set, generate full path name with file */
		strcpy( string , chDataPath );
		strcat( string,  fnam );
	}
	else
	{
		/* path not set, look here */
		strcpy( string , fnam );
	}

	if( (io2 = fopen(string,"r")) == NULL )
	{
		fprintf( ioQQQ, " Could not open %s for reading\n",string);
		puts( "[Stop in mie_read_opc]" );
		cdEXIT(1);
	}

	/* include the name of the file we are reading in the Cloudy output */
	sprintf( &chLine[0], "                       * >>>> mie_read_opc reading file -- " );
	i = (long)strlen(chLine);
	sprintf( &chLine[i], "                                         <<<< *\n" );
	if( strlen(fnam) <= 40 )
	{
		strncpy( &chLine[i], fnam, strlen(fnam) );
	}
	else
	{
		strncpy( &chLine[i], fnam, 37 );
		strncpy( &chLine[i+37], "...", 3 );
	}
	fprintf( ioQQQ, chLine );
                 
	/* >>chng 02 jan 30, check if file has already been read before, PvH */
	for( i=0; i < gv.ReadPtr; ++i )
	{
		if( strcmp(fnam,gv.ReadRecord[i]) == 0 )
		{
			fprintf( ioQQQ, " File %s has already been read before, was this intended ?\n", fnam );
			break;
		}
	}
	/* remember the name of the file we are reading now */
	if( gv.ReadPtr < MAX_READ_RECORDS )
	{
		strcpy(gv.ReadRecord[gv.ReadPtr],fnam);
		++gv.ReadPtr;
	}

	/* allocate memory for first bin */
	nd = NewGrainBin();

	dl = 0; /* line counter for input file */

	/* first read magic numbers */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&magic,TRUE,dl);
	if( magic != MAGIC_OPC ) 
	{
		fprintf( ioQQQ, " Opacity file %s has obsolete magic number\n",fnam );
		fprintf( ioQQQ, " I found %ld, but expected %ld on line #%ld\n",magic,MAGIC_OPC,dl );
		fprintf( ioQQQ, " Please recompile this file\n" );
		puts( "[Stop in mie_read_opc]" );
		cdEXIT(1);
	}

	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&magic,TRUE,dl);
	if( magic != MAGIC_RFI ) 
	{
		fprintf( ioQQQ, " Opacity file %s used obsolete refractive index file\n",fnam );
		fprintf( ioQQQ, " I found magic number %ld, but expected %ld on line #%ld\n",magic,MAGIC_RFI,dl );
		fprintf( ioQQQ, " Please recompile %s\n",fnam );
		puts( "[Stop in mie_read_opc]" );
		cdEXIT(1);
	}

	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&magic,TRUE,dl);
	if( magic != MAGIC_SZD ) 
	{
		fprintf( ioQQQ, " Opacity file %s used obsolete size distribution file\n",fnam );
		fprintf( ioQQQ, " I found magic number %ld, but expected %ld on line #%ld\n",magic,MAGIC_SZD,dl );
		fprintf( ioQQQ, " Please recompile %s\n",fnam );
		puts( "[Stop in mie_read_opc]" );
		cdEXIT(1);
	}

	gv.bin[nd]->dstfactor = (float)gp.dep;

	/* grain type label */
	mie_next_data(fnam,io2,chLine,&dl);
	strncpy(gv.bin[nd]->chDstLab,chLine,(size_t)LABELSIZE);
	gv.bin[nd]->chDstLab[LABELSIZE] = '\0';

	/* specific weight (g/cm^3) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->dustp[0],TRUE,dl);

	/* molecular weight of grain molecule (amu) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->dustp[1],TRUE,dl);

	/* average molecular weight per atom (amu) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->atomWeight,TRUE,dl);

	/* abundance of grain molecule for max depletion */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->dustp[2],TRUE,dl);

	/* depletion efficiency */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->dustp[3],TRUE,dl);

	/* average grain radius <a^3>/<a^2> for entire size distr (cm) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->AvRadius,TRUE,dl);

	/* average grain area <4pi*a^2> for entire size distr (cm^2) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->AvArea,TRUE,dl);

	/* average grain volume <4/3pi*a^3> for entire size distr (cm^3) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->AvVol,TRUE,dl);

	/* total grain radius Int(a) per H for entire size distr (cm/H) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->IntRadius,TRUE,dl);

	/* total grain area Int(4pi*a^2) per H for entire size distr (cm^2/H) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->IntArea,TRUE,dl);

	/* total grain vol Int(4/3pi*a^3) per H for entire size distr (cm^3/H) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->IntVol,TRUE,dl);

	/* work function, in Rydberg */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->DustWorkFcn,TRUE,dl);

	/* bandgap, in Rydberg */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->BandGap,FALSE,dl);

	/* efficiency of thermionic emissions, between 0 and 1 */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->ThermEff,TRUE,dl);

	/* sublimation temperature in K */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_float(fnam,chLine,&gv.bin[nd]->Tsublimat,TRUE,dl);

	/* material type, determines enthalpy function */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&gv.bin[nd]->matType,TRUE,dl);

	for( ipZ=0; ipZ < LIMELM; ipZ++ ) 
	{
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_float(fnam,chLine,&RefAbund[ipZ],FALSE,dl);

		/* this coefficient is defined at the end of appendix A.10 of BFM */
		gv.bin[nd]->AccomCoef[ipZ] = 2.*gv.bin[nd]->atomWeight*AtomcWgt.AtomicWeight[ipZ]/
			POW2(gv.bin[nd]->atomWeight+AtomcWgt.AtomicWeight[ipZ]);
	}

	/* ratio a_max/a_min for grains in a single size bin */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&RadiusRatio,TRUE,dl);

	gv.bin[nd]->lgDustVary = gp.lgAbunVsDepth;
	gv.bin[nd]->lgQHeat = ( gp.lgForbidQHeating ) ? FALSE : ( gp.lgRequestQHeating || RadiusRatio < RATIO_MAX );
	gv.bin[nd]->cnv_H_pGR = gv.bin[nd]->AvVol/gv.bin[nd]->IntVol;
	gv.bin[nd]->cnv_GR_pH = 1./gv.bin[nd]->cnv_H_pGR;

	/* this is capacity per grain, in Farad per grain */
	gv.bin[nd]->Capacity = PI4*ELECTRIC_CONST*gv.bin[nd]->IntRadius/100.*gv.bin[nd]->cnv_H_pGR;

#	ifdef INCLUDE_OLD_GRAINS
	gv.bin[nd]->lgDustOn1 = TRUE;
	gv.bin[nd]->ndpts = -1;
#	endif

	/* skip the table of the size distribution function (if present) */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&nup,FALSE,dl);
	for( i=0; i < nup; i++ )
		mie_next_data(fnam,io2,chLine,&dl);

	/* nup is number of frequency bins stored in file, this should match rfield.nupper */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&nup,TRUE,dl);

	gv.bin[nd]->NFPCheck = nup;

	/* no. of size distribution bins */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&nbin,TRUE,dl);

	/* these data are the same for all bins, so we allocate the memory here and copy the pointer below */
	assert( gv.bin[nd]->inv_att_len == NULL ); /* prevent memory leaks */
	if( ( gv.bin[nd]->inv_att_len = (float*)MALLOC((size_t)(nup*sizeof(float))) ) == NULL ) 
		bad_malloc();

	if( nbin == 1 )
	{
		/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
		assert( gv.bin[nd]->dstab1 == NULL ); /* prevent memory leaks */
		if( ( gv.bin[nd]->dstab1 = (double*)MALLOC((size_t)(nup*sizeof(double))) ) == NULL ) 
			bad_malloc();
		assert( gv.bin[nd]->dstsc1 == NULL ); /* prevent memory leaks */
		if( ( gv.bin[nd]->dstsc1 = (double*)MALLOC((size_t)(nup*sizeof(double))) ) == NULL ) 
			bad_malloc();

		gv.bin[nd]->dustp[4] = 1.;
		for( ipZ=0; ipZ < LIMELM; ipZ++ )
		{
			gv.bin[nd]->elmAbund[ipZ] = RefAbund[ipZ];
		}
	}
	else if( nbin > 1 )
	{
		/* remember this number since it will be overwritten below */
		VolTotal = gv.bin[nd]->IntVol;

		for( i=0; i < nbin; i++ ) 
		{
			/* allocate memory for remaining bins */
			nd2 = ( i == 0 ) ? nd : NewGrainBin();

			/* >>chng 01 sep 12, allocate/free [rfield.nupper] arrays dynamically */
			assert( gv.bin[nd2]->dstab1 == NULL ); /* prevent memory leaks */
			if( ( gv.bin[nd2]->dstab1 = (double*)MALLOC((size_t)(nup*sizeof(double))) ) == NULL ) 
				bad_malloc();
			assert( gv.bin[nd2]->dstsc1 == NULL ); /* prevent memory leaks */
			if( ( gv.bin[nd2]->dstsc1 = (double*)MALLOC((size_t)(nup*sizeof(double))) ) == NULL ) 
				bad_malloc();

			/* average grain radius <a^3>/<a^2> for this bin (cm) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->AvRadius,TRUE,dl);

			/* average grain area in this bin (cm^2) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->AvArea,TRUE,dl);

			/* average grain volume in this bin (cm^3) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->AvVol,TRUE,dl);

			/* total grain radius Int(a) per H for this bin (cm/H) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->IntRadius,TRUE,dl);

			/* total grain area Int(4pi*a^2) per H for this bin (cm^2/H) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->IntArea,TRUE,dl);

			/* total grain vol Int(4/3pi*a^3) per H for this bin (cm^3/H) */
			mie_next_data(fnam,io2,chLine,&dl);
			mie_read_float(fnam,chLine,&gv.bin[nd2]->IntVol,TRUE,dl);

			gv.bin[nd2]->cnv_H_pGR = gv.bin[nd2]->AvVol/gv.bin[nd2]->IntVol;
			gv.bin[nd2]->cnv_GR_pH = 1./gv.bin[nd2]->cnv_H_pGR;

			/* this is capacity per grain, in Farad per grain */
			gv.bin[nd2]->Capacity =
				PI4*ELECTRIC_CONST*gv.bin[nd2]->IntRadius/100.*gv.bin[nd2]->cnv_H_pGR;

			/* dustp[4] gives the fraction of the grain abundance that is
			 * contained in a particular bin. for unresolved distributions it is
			 * by definition 1, for resolved distributions it is smaller than 1. */
			gv.bin[nd2]->dustp[4] = gv.bin[nd2]->IntVol/VolTotal;
			for( ipZ=0; ipZ < LIMELM; ipZ++ )
			{
				gv.bin[nd2]->elmAbund[ipZ] = RefAbund[ipZ]*gv.bin[nd2]->dustp[4];
			}

			if( i > 0 ) 
			{
				/* inv_att_len is the same for all bins, so we allocate the memory only once */
				gv.bin[nd2]->inv_att_len = gv.bin[nd]->inv_att_len;
				gv.bin[nd2]->dstfactor = gv.bin[nd]->dstfactor;
				strcpy(gv.bin[nd2]->chDstLab,gv.bin[nd]->chDstLab);
				gv.bin[nd2]->atomWeight = gv.bin[nd]->atomWeight;
				for( j=0; j < 4; j++ ) 
				{
					gv.bin[nd2]->dustp[j] = gv.bin[nd]->dustp[j];
				}
				gv.bin[nd2]->DustWorkFcn = gv.bin[nd]->DustWorkFcn;
				gv.bin[nd2]->BandGap = gv.bin[nd]->BandGap;
				gv.bin[nd2]->ThermEff = gv.bin[nd]->ThermEff;
				gv.bin[nd2]->Tsublimat = gv.bin[nd]->Tsublimat;
				gv.bin[nd2]->matType = gv.bin[nd]->matType;
				gv.bin[nd2]->lgDustVary = gv.bin[nd]->lgDustVary;
				gv.bin[nd2]->lgQHeat = gv.bin[nd]->lgQHeat;
#				ifdef INCLUDE_OLD_GRAINS
				gv.bin[nd2]->lgDustOn1 = TRUE;
				gv.bin[nd2]->ndpts = gv.bin[nd]->ndpts;
#				endif
				gv.bin[nd2]->NFPCheck = gv.bin[nd]->NFPCheck;
				for( ipZ=0; ipZ < LIMELM; ipZ++ )
				{
					gv.bin[nd2]->AccomCoef[ipZ] = gv.bin[nd]->AccomCoef[ipZ];
				}
			}
		}
		for( i=0; i < nbin; i++ ) 
		{
			nd2 = nd + i;
			/* modify grain labels */
			str = strstr(gv.bin[nd2]->chDstLab,"xx");
			if( str != NULL )
				sprintf(str,"%02ld",i+1);
		}
	}

	/* skip the next 5 lines */
	for( i=0; i < 5; i++ )
		mie_next_line(fnam,io2,chLine,&dl);

	/* now read absorption opacities */
	for( i=0; i < nup; i++ ) 
	{
		/* read in energy scale and then opacities */
		if( (res = fscanf(io2,"%le",&anu)) != 1 ) 
		{
			fprintf( ioQQQ, " Read failed on %s\n",fnam );
			if( res == EOF )
				fprintf( ioQQQ, " EOF reached prematurely\n" );
			puts( "[Stop in mie_read_opc]" );
			cdEXIT(1);
		}
		for( j=0; j < nbin; j++ ) 
		{
			nd2 = nd + j;
			if( (res = fscanf(io2,"%le",&gv.bin[nd2]->dstab1[i])) != 1 ) 
			{
				fprintf( ioQQQ, " Read failed on %s\n",fnam );
				if( res == EOF )
					fprintf( ioQQQ, " EOF reached prematurely\n" );
				puts( "[Stop in mie_read_opc]" );
				cdEXIT(1);
			}
			assert( gv.bin[nd2]->dstab1[i] > 0. );
		}
	}

	/* skip to end-of-line and then skip next 4 lines */
	for( i=0; i < 5; i++ )
		mie_next_line(fnam,io2,chLine,&dl);

	/* now read scattering opacities */
	for( i=0; i < nup; i++ ) 
	{
		if( (res = fscanf(io2,"%le",&anu)) != 1 ) 
		{
			fprintf( ioQQQ, " Read failed on %s\n",fnam );
			if( res == EOF )
				fprintf( ioQQQ, " EOF reached prematurely\n" );
			puts( "[Stop in mie_read_opc]" );
			cdEXIT(1);
		}
		for( j=0; j < nbin; j++ ) 
		{
			nd2 = nd + j;
			if( (res = fscanf(io2,"%le",&gv.bin[nd2]->dstsc1[i])) != 1 ) 
			{
				fprintf( ioQQQ, " Read failed on %s\n",fnam );
				if( res == EOF )
					fprintf( ioQQQ, " EOF reached prematurely\n" );
				puts( "[Stop in mie_read_opc]" );
				cdEXIT(1);
			}
			assert( gv.bin[nd2]->dstsc1[i] > 0. );
		}
	}

	/* skip to end-of-line and then skip next 4 lines */
	for( i=0; i < 5; i++ )
		mie_next_line(fnam,io2,chLine,&dl);

	/* now read inverse attenuation length */
	for( i=0; i < nup; i++ ) 
	{
		if( (res = fscanf(io2,"%le %e",&anu,&gv.bin[nd]->inv_att_len[i])) != 2 ) 
		{
			fprintf( ioQQQ, " Read failed on %s\n",fnam );
			if( res == EOF )
				fprintf( ioQQQ, " EOF reached prematurely\n" );
			puts( "[Stop in mie_read_opc]" );
			cdEXIT(1);
		}
		assert( gv.bin[nd]->inv_att_len[i] > 0. );
		/* save energy for later tests of energy grid */
		gv.bin[nd]->EnergyCheck = (float)anu;

		for( j=1; j < nbin; j++ ) 
		{
			nd2 = nd + j;
			/* gv.bin[nd]->inv_att_len and gv.bin[nd2]->inv_att_len
			 * now point to the same physical memory -> no copy needed */
			/* gv.bin[nd2]->inv_att_len[i] = gv.bin[nd]->inv_att_len[i]; */
			gv.bin[nd2]->EnergyCheck = gv.bin[nd]->EnergyCheck;
		}
	}

	fclose(io2);

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_opc()\n", debug_fp );
#	endif
	return;
}


/* calculate average absorption, scattering cross section (i.e. pi a^2 Q) and
 * average asymmetry parameter g for an arbitrary grain size distribution */
static void mie_cs_size_distr(double wavlen, /* micron */
			      sd_data sd,
			      grain_data gd,
			      long int axis,
			      /*@out@*/ double *cs_abs, /* cm^2, average */
			      /*@out@*/ double *cs_sct, /* cm^2, average */
			      /*@out@*/ double *cosb,
			      /*@out@*/ int *error)
{
	int lgADLused,
	  lgOutOfBounds;
	long int i,
	  ind;
	double absval,
	  frac,
	  g,
	  nim,
	  nr1,
	  nre,
	  sctval,
	  weight;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_cs_size_distr()\n", debug_fp );
#	endif

	/* sanity checks */
	assert( wavlen > 0. );
	assert( axis >= 0 && axis < NAX );

	/* first interpolate optical constants */
	find_arr(wavlen,gd.wavlen[axis],gd.ndata[axis],&ind,&lgOutOfBounds);

	if( lgOutOfBounds ) 
	{
		*error = 3;
		*cs_abs = -1.;
		*cs_sct = -1.;
		*cosb = -2.;

#		ifdef DEBUG_FUN
		fputs( " <->mie_cs_size_distr()\n", debug_fp );
#		endif
		return;
	}

	frac = (wavlen-gd.wavlen[axis][ind])/(gd.wavlen[axis][ind+1]-gd.wavlen[axis][ind]);
	assert( frac > 0.-10.*DBL_EPSILON && frac < 1.+10.*DBL_EPSILON );
	nre = (1.-frac)*gd.n[axis][ind].re + frac*gd.n[axis][ind+1].re;
	assert( nre > 0. );
	nim = (1.-frac)*gd.n[axis][ind].im + frac*gd.n[axis][ind+1].im;
	assert( nim > 0. );
	nr1 = (1.-frac)*gd.nr1[axis][ind] + frac*gd.nr1[axis][ind+1];
	assert( fabs(nre-1.-nr1) < 10.*nre*DBL_EPSILON );

	switch( sd.sdCase ) 
	{
	case SD_SINGLE_SIZE:
		/* do single sized grain */
		assert( sd.a[ipSize] > 0. );
		mie_cs(wavlen,sd.a[ipSize],nre,nim,nr1,cs_abs,cs_sct,cosb,error);
		break;
	case SD_POWERLAW:
		/* simple powerlaw distribution */
	case SD_EXP_CUTOFF1:
	case SD_EXP_CUTOFF2:
	case SD_EXP_CUTOFF3:
		/* powerlaw distribution with exponential cutoff */
	case SD_LOG_NORMAL:
		/* gaussian distribution in ln(a) */
	case SD_LIN_NORMAL:
		/* gaussian distribution in a */
	case SD_TABLE:
		/* user supplied table of a^4*dN/da */
		assert( sd.lim[ipBLo] > 0. && sd.lim[ipBHi] > 0. && sd.lim[ipBHi] > sd.lim[ipBLo] );
		lgADLused = FALSE;
		*cs_abs = 0.;
		*cs_sct = 0.;
		*cosb = 0.;
		for( i=0; i < sd.nn; i++ ) 
		{
			mie_cs(wavlen,sd.rr[i],nre,nim,nr1,&absval,&sctval,&g,error);
			if( *error == 2 ) 
			{
				/* mie_cs failed to converge -> integration is invalid */
				*cs_abs = -1.;
				*cs_sct = -1.;
				*cosb = -2.;

#				ifdef DEBUG_FUN
				fputs( " <->mie_cs_size_distr()\n", debug_fp );
#				endif
				return;
			}
			else if( *error == 1 ) 
			{
				/* anomalous diffraction limit used -> g is not valid */
				lgADLused = TRUE;
			}
			weight = sd.ww[i]*size_distr(sd.rr[i],sd);
			*cs_abs += weight*absval;
			*cs_sct += weight*sctval;
			*cosb += weight*sctval*g;
		}
		if( lgADLused ) 
		{
			*error = 1;
			*cosb = -2.;
		}
		else 
		{
			*error = 0;
			*cosb /= *cs_sct;
		}
		*cs_abs /= sd.unity;
		*cs_sct /= sd.unity;
		break;
	default:
		fprintf( ioQQQ, " insane case for grain size distribution: %d\n" , sd.sdCase );
		ShowMe();
		puts( "[Stop in mie_cs_size_distr]" );
		cdEXIT(1);
	}
	/* sanity checks */
	if( *error < 2 )
		assert( *cs_abs > 0. && *cs_sct > 0. );
	if( *error < 1 )
		assert( fabs(*cosb) <= 1.+10.*DBL_EPSILON );

#	ifdef DEBUG_FUN
	fputs( " <->mie_cs_size_distr()\n", debug_fp );
#	endif
	return;
}


/* calculate absorption, scattering cross section (i.e. pi a^2 Q) and
 * asymmetry parameter g (=cosb) for a single sized grain defined by gd */
static void mie_cs(double wavlen,  /* micron */
		   double size,    /* micron */
		   double nre,
		   double nim,
		   double nr1,
		   /*@out@*/ double *cs_abs, /* cm^2 */
		   /*@out@*/ double *cs_sct, /* cm^2 */
		   /*@out@*/ double *cosb,
		   /*@out@*/ int *error)
{
	long int iflag;
	double area,
	  aqabs,
	  aqext,
	  aqphas,
	  beta,
	  ctbrqs,
	  delta,
	  qback,
	  qext,
	  qphase,
	  qscatt,
	  x,
	  xistar;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_cs()\n", debug_fp );
#	endif

	/* sanity checks, should already have been checked further upstream */
	assert( wavlen > 0. );
	assert( size > 0. );
	assert( nre > 0. && nim >= 0. );
	assert( fabs(nre-1.-nr1) < 10.*nre*DBL_EPSILON );

	/* size in micron, area in cm^2 */
	area = PI*POW2(size)*1.e-8;

	x = size/wavlen*2.*PI;

	/* note that in the following, only nre, nim are used in sinpar
	 * and also nr1 in anomalous diffraction limit */

	sinpar(nre,nim,x,&qext,&qphase,&qscatt,&ctbrqs,&qback,&iflag);

	/* iflag=0 normal exit, 1 failure to converge, 2 not even tried
	 * for exit 1,2, see whether anomalous diffraction is available */

	if( iflag == 0 ) 
	{
		*error = 0;
		*cs_abs = area*(qext - qscatt);
		*cs_sct = area*qscatt;
		*cosb = ctbrqs/qscatt;
	}
	else 
	{
		/* anomalous diffraction -- x >> 1 and |m-1| << 1 but any phase shift */
		if( x >= 100. && sqrt(nr1*nr1+nim*nim) <= 0.001 ) 
		{
			delta = -nr1;
			beta = nim;

			anomal(x,&aqext,&aqabs,&aqphas,&xistar,delta,beta);

			/* cosb is invalid */
			*error = 1;
			*cs_abs = area*aqabs;
			*cs_sct = area*(aqext - aqabs);
			*cosb = -2.;
		}
		/* nothing works */
		else 
		{
			*error = 2;
			*cs_abs = -1.;
			*cs_sct = -1.;
			*cosb = -2.;
		}
	}
	if( *error < 2 ) 
	{
		if( *cs_abs <= 0. || *cs_sct <= 0. ) 
		{
			fprintf( ioQQQ, " illegal opacity found: wavl=%.4e micron," , wavlen );
			fprintf( ioQQQ, " abs_cs=%.2e, sct_cs=%.2e\n" , *cs_abs , *cs_sct );
			fprintf( ioQQQ, " please check refractive index file...\n" );
			puts( "[Stop in mie_cs]" );
			cdEXIT(1);
		}
	}
	if( *error < 1 ) 
	{
		if( fabs(*cosb) > 1.+10.*DBL_EPSILON ) 
		{
			fprintf( ioQQQ, " illegal asymmetry parameter found: wavl=%.4e micron," , wavlen );
			fprintf( ioQQQ, " cosb=%.2e\n" , *cosb );
			fprintf( ioQQQ, " please check refractive index file...\n" );
			puts( "[Stop in mie_cs]" );
			cdEXIT(1);
		}
	}
		

#	ifdef DEBUG_FUN
	fputs( " <->mie_cs()\n", debug_fp );
#	endif
	return;
}


static double size_distr(double size,
			 sd_data sd)
{
	int lgOutOfBounds;
	long ind;
	double frac,
	  res,
	  x;

#   ifdef DEBUG_FUN
	fputs( "<+>size_distr()\n", debug_fp );
#   endif

	if( size >= sd.lim[ipBLo] && size <= sd.lim[ipBHi] )
		switch( sd.sdCase ) 
		{
		case SD_SINGLE_SIZE:
			res = 1.; /* should really not be used in this case */
			break;
		case SD_POWERLAW:
			/* simple powerlaw */
		case SD_EXP_CUTOFF1:
		case SD_EXP_CUTOFF2:
		case SD_EXP_CUTOFF3:
			/* powerlaw with exponential cutoff, inspired by Greenberg (1978)
			 * Cosmic Dust, ed. J.A.M. McDonnell, Wiley, p. 187 */
			res = pow(size,sd.a[ipExp]);
			if( sd.a[ipBeta] < 0. )
				res /= (1. - sd.a[ipBeta]*size);
			else if( sd.a[ipBeta] > 0. )
				res *= (1. + sd.a[ipBeta]*size);
			if( size < sd.a[ipBLo] && sd.a[ipSLo] > 0. )
				res *= exp(-powi((sd.a[ipBLo]-size)/sd.a[ipSLo],NINT(sd.a[ipAlpha])));
			if( size > sd.a[ipBHi] && sd.a[ipSHi] > 0. )
				res *= exp(-powi((size-sd.a[ipBHi])/sd.a[ipSHi],NINT(sd.a[ipAlpha])));
			break;
		case SD_LOG_NORMAL:
			x = log(size/sd.a[ipGCen])/sd.a[ipGSig];
			res = exp(-0.5*POW2(x))/size;
			break;
		case SD_LIN_NORMAL:
			x = (size-sd.a[ipGCen])/sd.a[ipGSig];
			res = exp(-0.5*POW2(x))/size;
			break;
		case SD_TABLE:
			find_arr(log(size),sd.ln_a,sd.npts,&ind,&lgOutOfBounds);
			if( lgOutOfBounds )
			{
				fprintf( ioQQQ, " size distribution table has insufficient range\n" );
				fprintf( ioQQQ, " requested size: %.5f table range %.5f - %.5f\n",
					 size, exp(sd.ln_a[0]), exp(sd.ln_a[sd.npts-1]) );
				puts( "[Stop in size_distr]" );
				cdEXIT(1);
			}				
			frac = (log(size)-sd.ln_a[ind])/(sd.ln_a[ind+1]-sd.ln_a[ind]);
			assert( frac > 0.-10.*DBL_EPSILON && frac < 1.+10.*DBL_EPSILON );
			res = (1.-frac)*sd.ln_a4dNda[ind] + frac*sd.ln_a4dNda[ind+1];
			/* convert from a^4*dN/da to dN/da */
			res = exp(res)/POW4(size);
			break;
		default:
			fprintf( ioQQQ, " insane case for grain size distribution: %d\n" , sd.sdCase );
			ShowMe();
			puts( "[Stop in size_distr]" );
			cdEXIT(1);
		}
	else
		res = 0.;

#	ifdef DEBUG_FUN
	fputs( " <->size_distr()\n", debug_fp );
#	endif
	return res;
}


/* search for upper/lower limit lim of size distribution such that
 * lim^4 * dN/da(lim) < rel_cutoff * ref^4 * dN/da(ref)
 * the initial estimate of lim = ref + step
 * step may be both positive (upper limit) or negative (lower limit) */ 
static double search_limit(double ref,
			   double step,
			   double rel_cutoff,
			   sd_data sd)
{
	long i;
	double f1,
	  f2,
	  fmid,
	  renorm,
	  x1,
	  x2 = DBL_MAX,
	  xmid = DBL_MAX;

	/* TOLER is the relative accuracy with which lim is determined */
	const double TOLER = 1.e-6;

#	ifdef DEBUG_FUN
	fputs( "<+>search_limit()\n", debug_fp );
#	endif

	/* sanity check */
	assert( rel_cutoff > 0. && rel_cutoff < 1. );

	if( step == 0. )
	{
#		ifdef DEBUG_FUN
		fputs( " <->search_limit()\n", debug_fp );
#		endif
		return ref;
	}

	/* these need to be set in order for size_distr to work...
	 * NB - since this is a local copy of sd, it will not
	 * upset anything in the calling routine */
	sd.lim[ipBLo] = 0.;
	sd.lim[ipBHi] = DBL_MAX;

	x1 = ref;
	/* previous assert guarantees that f1 > 0. */
	f1 = -log(rel_cutoff);
	renorm = f1 - log(POW4(x1)*size_distr(x1,sd));

	/* bracket solution */
	f2 = 1.;
	for( i=0; i < 20 && f2 > 0.; ++i )
	{
		x2 = MAX2(ref + step,SMALLEST_GRAIN);
		f2 = log(POW4(x2)*size_distr(x2,sd)) + renorm;
		if( f2 >= 0. )
		{
			x1 = x2;
			f1 = f2;
		}
		step *= 2.;
	}
	if( f2 > 0. )
	{
		fprintf( ioQQQ, " Could not bracket solution\n" );
		puts( "[Stop in search_limit]" );
		cdEXIT(1);
	}

	/* do bisection search */
	while( 2.*fabs(x1-x2)/(x1+x2) > TOLER )
	{
		xmid = (x1+x2)/2.;
		fmid = log(POW4(xmid)*size_distr(xmid,sd)) + renorm;

		if( fmid == 0. )
			break;

		if( f1*fmid > 0. )
		{
			x1 = xmid;
			f1 = fmid;
		}
		else
		{
			x2 = xmid;
			f2 = fmid;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->search_limit()\n", debug_fp );
#	endif
	return (x1+x2)/2.;
}


#ifdef INCLUDE_OLD_GRAINS
/* calculate the inverse attenuation length for a given refractive index file */
void mie_read_ial(/*@in@*/ const char *fnam,  /* rfi file */
		  long int n,
		  /*@out@*/ double invlen[])  /* invlen[n] */
{
	int lgWarning = FALSE;
	long j;
	char string[DATA_PATH_LENGTH_2];
	grain_data gd;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_read_ial()\n", debug_fp );
#	endif

	gd.nAxes = 0;
	for( j=0; j < NAX; j++ ) 
	{
		gd.wavlen[j] = NULL;
		gd.n[j] = NULL;
		gd.nr1[j] = NULL;
	}

	/* true is special path set, false if file in same dir */
	if( lgDataPathSet )
	{
		/* path is set, generate full path name with file */
		strcpy( string , chDataPath );
		strcat( string,  fnam );
	}
	else
	{
		/* path not set, look here */
		strcpy( string , fnam );
	}
	mie_read_rfi(string,&gd);

	/* empty string indicates no output in mie_calc_ial */
	string[0] = '\0';
	mie_calc_ial(gd,n,invlen,string,&lgWarning);

	/* all arrays below were allocated in mie_read_rfi */
	for( j=0; j < gd.nAxes; j++ ) 
	{
		if( gd.nr1[j] != NULL )
			free(gd.nr1[j]);
		if( gd.n[j] != NULL )
			free(gd.n[j]);
		if( gd.wavlen[j] != NULL )
			free(gd.wavlen[j]);
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_ial()\n", debug_fp );
#	endif
	return;
}
#endif


/* calculate the inverse attenuation length for given refractive index data */
static void mie_calc_ial(/*@in@*/ grain_data gd,
			 long int n,
			 /*@out@*/ double invlen[],  /* invlen[n] */
			 /*@in@*/ const char *string,
			 /*@in@*/ int *lgWarning)
{
	/* >>chng 01 aug 22, MALLOC this space */
	int *ErrorIndex/*[NC_ELL]*/,
	  lgErrorOccured=TRUE,
	  lgOutOfBounds;
	long int i,
	  ind,
	  j;
	double frac,
	  InvDep,
	  nim,
	  wavlen;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_calc_ial()\n", debug_fp );
#	endif

	/* >>chng 01 aug 22, MALLOC this space */
	if( (ErrorIndex = (int*)MALLOC((size_t)(rfield.nupper*sizeof(int)) ) ) == NULL )
		bad_malloc();

	for( i=0; i < n; i++ ) 
	{
		wavlen = WAVNRYD/rfield.anu[i]*1.e4;

		ErrorIndex[i] = 0;
		lgErrorOccured = FALSE;
		invlen[i] = 0.;

		for( j=0; j < gd.nAxes; j++ ) 
		{
			/* first interpolate optical constants */
			find_arr(wavlen,gd.wavlen[j],gd.ndata[j],&ind,&lgOutOfBounds);
			if( lgOutOfBounds ) 
			{
				ErrorIndex[i] = 3;
				lgErrorOccured = TRUE;
				invlen[i] = 0.;
				break;
			}
			frac = (wavlen-gd.wavlen[j][ind])/(gd.wavlen[j][ind+1]-gd.wavlen[j][ind]);
			nim = (1.-frac)*gd.n[j][ind].im + frac*gd.n[j][ind+1].im;
			/* this is the inverse of the photon attenuation depth,
			 * >>refer Weingartner & Draine, 2000, ApJ, ... */
			InvDep = PI4*nim/wavlen*1.e4;
			assert( InvDep > 0. );

			invlen[i] += InvDep*gd.wt[j];
		}
	}

	if( lgErrorOccured ) 
	{
		mie_repair(string,n,3,3,rfield.anu,invlen,ErrorIndex,FALSE,lgWarning);
	}

	free( ErrorIndex );

#	ifdef DEBUG_FUN
	fputs( " <->mie_calc_ial()\n", debug_fp );
#	endif
	return;
}


/* this is the number of x-values we use for extrapolating functions */
#define NPTS_DERIV 8
#define NPTS_COMB  (NPTS_DERIV*(NPTS_DERIV-1)/2)

/* extrapolate/interpolate mie data to fill in the blanks */
static void mie_repair(/*@in@*/ const char *string,
		       long int n,
		       int val,
		       int del,
		       /*@in@*/ float anu[],      /* anu[n] */
		       double data[],             /* data[n] */
		       /*@in@*/ int ErrorIndex[], /* ErrorIndex[n] */
		       int lgRound,
		       /*@in@*/ int *lgWarning)
{
	int lgExtrapolate,
	  lgVerbose;
	long int i1,
	  i2,
	  ind1,
	  ind2,
	  j;
	double dx,
	  sgn,
	  slp1,
	  xlg1,
	  xlg2,
	  y1lg1,
	  y1lg2;

	/* interpolating over more that this number of points results in a warning */
	const long BIG_INTERPOLATION = 10;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_repair()\n", debug_fp );
#	endif

	lgVerbose = ( string[0] != '\0' );

	for( ind1=0; ind1 < n; ) 
	{
		if( ErrorIndex[ind1] == val ) 
		{
			/* search for region with identical error index */
			ind2 = ind1;
			while( ind2 < n && ErrorIndex[ind2] == val )
				ind2++;

			if( lgVerbose )
				fprintf( ioQQQ, "    %s", string );

			if( ind1 == 0 ) 
			{
				/* low energy extrapolation */
				i1 = ind2;
				i2 = ind2+NPTS_DERIV-1;
				lgExtrapolate = TRUE;
				sgn = +1.;
				if( lgVerbose ) 
				{
					fprintf( ioQQQ, " extrapolated below %.4e Ryd\n",anu[i1] );
				}
			}
			else if( ind2 == n ) 
			{
				/* high energy extrapolation */
				i1 = ind1-NPTS_DERIV;
				i2 = ind1-1;
				lgExtrapolate = TRUE;
				sgn = -1.;
				if( lgVerbose ) 
				{
					fprintf( ioQQQ, " extrapolated above %.4e Ryd\n",anu[i2] );
				}
			}
			else 
			{
				/* interpolation */
				i1 = ind1-1;
				i2 = ind2;
				lgExtrapolate = FALSE;
				sgn = 0.;
				if( lgVerbose ) 
				{
					fprintf( ioQQQ, " interpolated between %.4e and %.4e Ryd\n",
						 anu[i1],anu[i2] );
				}
				if( i2-i1-1 > BIG_INTERPOLATION )
				{
					if( lgVerbose )
					{
						fprintf( ioQQQ, " ***Warning: extensive interpolation used\n" );
					}
					*lgWarning = TRUE;
				}
			}

			if( i1 < 0 || i2 >= n ) 
			{
				fprintf( ioQQQ, " Insufficient data for extrapolation\n" );
				puts( "[Stop in mie_repair]" );
				cdEXIT(1);
			}

			xlg1 = log(anu[i1]);
			y1lg1 = log(data[i1]);
			/* >>chng 01 jul 30, replace simple-minded extrapolation with more robust routine, PvH */
			if( lgExtrapolate )
				slp1 = mie_find_slope(anu,data,ErrorIndex,i1,i2,val,lgVerbose,lgWarning);
			else
			{
				xlg2 = log(anu[i2]);
				y1lg2 = log(data[i2]);
				slp1 = (y1lg2-y1lg1)/(xlg2-xlg1);
			}
			if( lgRound && lgExtrapolate && sgn > 0. ) 
			{
				/* in low energy extrapolation, 1-g is very close to 1 and almost constant
				 * hence slp1 is very close to 0 and can even be slightly negative
				 * to prevent 1-g becoming greater than 1, the following is necessary */
				slp1 = MAX2(slp1,0.);
			}
			else if( lgExtrapolate && sgn*slp1 <= 0. ) 
			{
				fprintf( ioQQQ, " Illegal value for slope in extrapolation %.6e\n", slp1 );
				puts( "[Stop in mie_repair]" );
				cdEXIT(1);
			}

			for( j=ind1; j < ind2; j++ ) 
			{
				dx = log(anu[j]) - xlg1;
				data[j] = exp(y1lg1 + dx*slp1);
				ErrorIndex[j] -= del;
			}

			ind1 = ind2;
		}
		else 
		{
			ind1++;
		}
	}
	/* sanity check */
	for( j=0; j < n; j++ )
	{
		if( ErrorIndex[j] > val-del ) 
		{
			fprintf( ioQQQ, " Internal error in mie_repair, index=%ld, val=%d\n",j,ErrorIndex[j] );
			ShowMe();
			puts( "[Stop in mie_repair]" );
			cdEXIT(1);
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_repair()\n", debug_fp );
#	endif
	return;
}


static double mie_find_slope(/*@in@*/ const float anu[],
			     /*@in@*/ const double data[],
			     /*@in@*/ const int ErrorIndex[],
			     long i1,
			     long i2,
			     int val,
			     int lgVerbose,
			     /*@in@*/ int *lgWarning)
{
	long i,
	  j,
	  k;
	double s1,
	  s2,
	  slope,
	  slp1[NPTS_COMB],
	  stdev;

	/* threshold for standard deviation in the logarithmic derivative to generate warning,
	 * this corresponds to an uncertainty of a factor 10 for a typical extrapolation */
	const double LARGE_STDEV = 0.2;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_find_slope()\n", debug_fp );
#	endif

	/* sanity check */
	assert( i2-i1 == NPTS_DERIV-1 );
	for( i=i1; i <= i2; i++ )
	{
		assert( ErrorIndex[i] < val );
		assert( anu[i] > 0. && data[i] > 0. );
	}

	k = 0;
	/* calculate the logarithmic derivative for every possible combination of data points */
	for( i=i1; i < i2; i++ )
	{
		for( j=i+1; j <= i2; j++ )
		{
			slp1[k++] = log(data[j]/data[i])/log(anu[j]/anu[i]);
		}
	}
	/* sort the values; we want the median -> values for i > NPTS_COMB/2 are irrelevant */
	for( i=0; i <= NPTS_COMB/2; i++ )
	{
		for( j=i+1; j < NPTS_COMB; j++ )
		{
			if( slp1[i] > slp1[j] )
			{
				double xxx = slp1[i];
				slp1[i] = slp1[j];
				slp1[j] = xxx;
			}
		}
	}
	/* now calculate the median value */
	slope = ( NPTS_COMB%2 == 1 ) ? slp1[NPTS_COMB/2] : (slp1[NPTS_COMB/2-1] + slp1[NPTS_COMB/2])/2.;

	/* and finally calculate the standard deviation of all slopes */
	s1 = s2 = 0.;
	for( i=0; i < NPTS_COMB; i++ )
	{
		s1 += slp1[i];
		s2 += POW2(slp1[i]);
	}
	stdev = sqrt(s2/(double)NPTS_COMB - POW2(s1/(double)NPTS_COMB));

#if 0
	for( i=i1; i <= i2; i++ )
		printf("input: %ld %.4e %.4e\n",i,anu[i],data[i]);
	for( i=0; i < NPTS_COMB; i++ )
		printf("%.3f ",slp1[i]);
	printf("\n");
	printf("slope %.3f +/- %.3f\n",slope,stdev);
#endif
	
	/* print warning if standard deviation is large */
	if( stdev > LARGE_STDEV )
	{
		if( lgVerbose )
		{
			fprintf( ioQQQ, " ***Warning: slope for extrapolation may be unreliable\n" );
		}
		*lgWarning = TRUE;
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_find_slope()\n", debug_fp );
#	endif
	return slope;
}


/* read in the file with optical constants and other relevant information */
static void mie_read_rfi(/*@in@*/  const char *fnam,
			 /*@out@*/ grain_data *gd)
{
	long int dl,
	  i,
	  ipZ,
	  j,
	  magic,
	  nridf,
	  sgn = 0;
	double eps1,
	  eps2,
	  molw,
	  nAtoms,
	  nr,
	  ni,
	  total = 0.;
	char chLine[LINELEN],
	  chWord[LINELEN];
	FILE *io2;

#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_rfi()\n", debug_fp );
#       endif
	if( (io2 = fopen(fnam,"r")) == NULL )
	{
		fprintf( ioQQQ, " Could not open %s for reading\n",fnam);
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	dl = 0; /* line counter for input file */

	/* first read magic number */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&magic,TRUE,dl);
	if( magic != MAGIC_RFI ) 
	{
		fprintf( ioQQQ, " Refractive index file %s has obsolete magic number\n",fnam );
		fprintf( ioQQQ, " I found %ld, but expected %ld on line #%ld\n",magic,MAGIC_RFI,dl );
		fprintf( ioQQQ, " Please replace this file with an up to date version\n" );
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* get chemical formula of the grain, e.g., Mg0.4Fe0.6SiO3 */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_word(chLine,chWord,LINELEN,FALSE);
	mie_read_form(chWord,gd->elmAbun,&nAtoms,&molw);

	/* molecular weight, in atomic units */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->mol_weight,FALSE,dl);
	/* if zero is read from file -> use default */
	if( gd->mol_weight <= 0. )
		gd->mol_weight = molw;
	gd->atom_weight = gd->mol_weight/nAtoms;

	/* determine abundance of grain molecule assuming max depletion */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->abun,TRUE,dl);

	/* default depletion of grain molecule */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->depl,TRUE,dl);
	if( gd->depl > 1. ) 
	{
		fprintf( ioQQQ, " Illegal value for default depletion in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, depl=%14.6e\n",dl,gd->depl);
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	for( ipZ=0; ipZ < LIMELM; ipZ++ )
		gd->elmAbun[ipZ] *= gd->abun*gd->depl;

	/* material density, to get cross section per unit mass: rho in cgs */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->rho,TRUE,dl);

	/* material type, determines enthalpy function: 1 -- carbonaceous, 2 -- silicate */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&gd->mat_type,TRUE,dl);
	if( gd->mat_type > MAT_TOP ) 
	{
		fprintf( ioQQQ, " Illegal value for material type in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, type=%ld\n",dl,gd->mat_type);
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* work function, in Rydberg */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->work,TRUE,dl);

	/* bandgap, in Rydberg */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->bandgap,FALSE,dl);
	if( gd->bandgap >= gd->work ) 
	{
		fprintf( ioQQQ, " Illegal value for bandgap in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, bandgap=%.4e, work function=%.4e\n",dl,gd->bandgap,gd->work);
		fprintf( ioQQQ, " Bandgap should always be less than work function\n");
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* efficiency of thermionic emission, between 0 and 1 */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->therm_eff,TRUE,dl);
	if( gd->therm_eff > 1.f ) 
	{
		fprintf( ioQQQ, " Illegal value for thermionic efficiency in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, value=%.4e\n",dl,gd->therm_eff);
		fprintf( ioQQQ, " Allowed values are 0. < efficiency <= 1.\n");
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* sublimation temperature in K */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_double(fnam,chLine,&gd->subl_temp,TRUE,dl);

	/* nridf is for choosing ref index or diel function input
	 * case 2 allows greater accuracy reading in, when nr is close to 1. */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&nridf,TRUE,dl);
	if( nridf > 3 ) 
	{
		fprintf( ioQQQ, " Illegal data code in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, data code=%ld\n",dl,nridf);
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* no. of principal axes, always 1 for amorphous grains,
	 * maybe larger for crystalline grains */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&gd->nAxes,TRUE,dl);
	if( gd->nAxes > NAX ) 
	{
		fprintf( ioQQQ, " Illegal no. of axes in %s\n",fnam );
		fprintf( ioQQQ, " Line #%ld, number=%ld\n",dl,gd->nAxes);
		puts( "[Stop in mie_read_rfi]" );
		cdEXIT(1);
	}

	/* now get relative weights of axes */
	mie_next_data(fnam,io2,chLine,&dl);
	switch( gd->nAxes ) 
	{
	case 1:
		mie_read_double(fnam,chLine,&gd->wt[0],TRUE,dl);
		total = gd->wt[0];
		break;
	case 2:
		if( sscanf( chLine , "%lf %lf", &gd->wt[0], &gd->wt[1] ) != 2 ) 
		{
			fprintf( ioQQQ, " Syntax error in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}
		if( gd->wt[0] <= 0. || gd->wt[1] <= 0. ) 
		{
			fprintf( ioQQQ, " Illegal data in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}
		total = gd->wt[0] + gd->wt[1];
		break;
	case 3:		
		if( sscanf( chLine , "%lf %lf %lf", &gd->wt[0], &gd->wt[1], &gd->wt[2] ) != 3 ) 
		{
			fprintf( ioQQQ, " Syntax error in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}
		if( gd->wt[0] <= 0. || gd->wt[1] <= 0. || gd->wt[2] <= 0. ) 
		{
			fprintf( ioQQQ, " Illegal data in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}
		total = gd->wt[0] + gd->wt[1] + gd->wt[2];
	}
	for( j=0; j < gd->nAxes; j++ ) 
	{
		gd->wt[j] /= total;

		/* read in optical constants for each principal axis. */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_long(fnam,chLine,&gd->ndata[j],FALSE,dl);
		if( gd->nAxes == 1 && gd->ndata[j] == 0 ) 
		{
			/* special case for grey grains */
			gd->wavlen[j] = NULL;
			gd->n[j] = NULL;
			gd->nr1[j] = NULL;

#			ifdef DEBUG_FUN
			fputs( " <->mie_read_rfi()\n", debug_fp );
#			endif
			return;
		}
		else if ( gd->ndata[j] < 2 ) 
		{
			fprintf( ioQQQ, " Illegal number of data points in %s\n",fnam );
			fprintf( ioQQQ, " Line #%ld, number=%ld\n",dl,gd->ndata[j]);
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}

		/* allocate space for wavelength and optical constants arrays
		 * the memory is freed in mie_auxiliary */
		gd->wavlen[j] = (double *)MALLOC(sizeof(double)*(unsigned)gd->ndata[j]);
		gd->n[j] = (complex *)MALLOC(sizeof(complex)*(unsigned)gd->ndata[j]);
		gd->nr1[j] = (double *)MALLOC(sizeof(double)*(unsigned)gd->ndata[j]);

		if( gd->wavlen[j] == NULL || gd->n[j] == NULL || gd->nr1[j] == NULL ) 
		{
			fprintf( ioQQQ, " Could not MALLOC arrays\n" );
			puts( "[Stop in mie_read_rfi]" );
			cdEXIT(1);
		}

		for( i=0; i < gd->ndata[j]; i++ ) 
		{
			/*     read in the wavelength in microns
			 *     and the complex refractive index or dielectric function of material */
			mie_next_data(fnam,io2,chLine,&dl);
			if( sscanf( chLine , "%lf %lf %lf", gd->wavlen[j]+i, &nr, &ni ) != 3 ) 
			{
				fprintf( ioQQQ, " Syntax error in %s\n",fnam);
				fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
				puts( "[Stop in mie_read_rfi]" );
				cdEXIT(1);
			}
			if( gd->wavlen[j][i] <= 0. ) 
			{
				fprintf( ioQQQ, " Illegal value for wavelength in %s\n",fnam);
				fprintf( ioQQQ, " Line #%ld, wavl=%14.6e\n",dl,gd->wavlen[j][i]);
				puts( "[Stop in mie_read_rfi]" );
				cdEXIT(1);
			}
			/* the data in the input file should be sorted on wavelength, either
			 * strictly monotonically increasing or decreasing, check this here... */
			if( i == 1 ) 
			{
				sgn = SIGN3(gd->wavlen[j][1]-gd->wavlen[j][0]);
				if( sgn == 0 ) 
				{
					fprintf( ioQQQ, " Illegal value for wavelength in %s\n",fnam);
					fprintf( ioQQQ, " Line #%ld, wavl=%14.6e\n",dl,gd->wavlen[j][i]);
					puts( "[Stop in mie_read_rfi]" );
					cdEXIT(1);
				}
			}
			else if( i > 1 ) 
			{
				if( SIGN3(gd->wavlen[j][i]-gd->wavlen[j][i-1]) != sgn ) 
				{
					fprintf( ioQQQ, " Illegal value for wavelength in %s\n",fnam);
					fprintf( ioQQQ, " Line #%ld, wavl=%14.6e\n",dl,gd->wavlen[j][i]);
					puts( "[Stop in mie_read_rfi]" );
					cdEXIT(1);
				}
			}
			/*     this version reads in real and imaginary parts of the refractive
			 *     index, with imaginary part positive (nridf = 3) or nr-1 (nridf = 2) or
			 *     real and imaginary parts of the dielectric function (nridf = 1)
			 * */
			switch( nridf ) 
			{
			case 1:
				eps1 = nr;
				eps2 = ni;
				dftori(&nr,&ni,eps1,eps2);
				gd->nr1[j][i] = nr - 1.;
				break;
			case 2:
				gd->nr1[j][i] = nr;
				nr += 1.;
				break;
			case 3:
				gd->nr1[j][i] = nr - 1.;
			}
			gd->n[j][i] = ftocm(nr,ni);

			/* sanity checks */
			if( nr <= 0. || ni < 0. ) 
			{
				fprintf( ioQQQ, " Illegal value for refractive index in %s\n",fnam);
				fprintf( ioQQQ, " Line #%ld, (nr,ni)=(%14.6e,%14.6e)\n",dl,nr,ni);
				puts( "[Stop in mie_read_rfi]" );
				cdEXIT(1);
			}
			assert( fabs(nr-1.-gd->nr1[j][i]) < 10.*nr*DBL_EPSILON );
		}
	}
	fclose(io2);

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_rfi()\n", debug_fp );
#	endif
	return;
}

#define WORDLEN 5

/* read in the file with optical constants and other relevant information */
static void mie_read_szd(/*@in@*/  const char *fnam,
			 /*@out@*/ sd_data *sd)
{
	int lgTryOverride = FALSE;
	long int dl,
	  i,
	  magic;
	double mul = 0.,
	  ref_neg = DBL_MAX,
	  ref_pos = DBL_MAX,
	  step_neg = DBL_MAX,
	  step_pos = DBL_MAX;
	char chLine[LINELEN],
	  chWord[WORDLEN];
	FILE *io2;

	/* these constants are used to get initial estimates for the cutoffs (lim)
	 * in the SD_EXP_CUTOFFx and SD_xxx_NORMAL cases, they are iterated by
	 * search_limit such that
	 * lim^4 * dN/da(lim) == FRAC_CUTOFF * ref^4 * dN/da(ref)
	 * where ref as an appropriate reference point for each of the cases */
	const double FRAC_CUTOFF = 1.e-4;
	const double MUL_CO1 = -log(FRAC_CUTOFF);
	const double MUL_CO2 = sqrt(MUL_CO1);
	const double MUL_CO3 = pow(MUL_CO1,1./3.);
	const double MUL_LND = sqrt(-2.*log(FRAC_CUTOFF));
	const double MUL_NRM = MUL_LND;

#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_szd()\n", debug_fp );
#       endif
	if( (io2 = fopen(fnam,"r")) == NULL )
	{
		fprintf( ioQQQ, " Could not open %s for reading\n",fnam);
		puts( "[Stop in mie_read_szd]" );
		cdEXIT(1);
	}

	dl = 0; /* line counter for input file */

	/* first read magic number */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_long(fnam,chLine,&magic,TRUE,dl);
	if( magic != MAGIC_SZD ) 
	{
		fprintf( ioQQQ, " Size distribution file %s has obsolete magic number\n",fnam );
		fprintf( ioQQQ, " I found %ld, but expected %ld on line #%ld\n",magic,MAGIC_SZD,dl );
		fprintf( ioQQQ, " Please replace this file with an up to date version\n" );
		puts( "[Stop in mie_read_szd]" );
		cdEXIT(1);
	}

	/* size distribution case */
	mie_next_data(fnam,io2,chLine,&dl);
	mie_read_word(chLine,chWord,WORDLEN,TRUE);

	if( lgMatch( chWord, "SSIZ" ) )
	{
		sd->sdCase = SD_SINGLE_SIZE;
	}
	else if( lgMatch( chWord, "POWE" ) )
	{
		sd->sdCase = SD_POWERLAW;
	}
	else if( lgMatch( chWord, "EXP1" ) )
	{
		sd->sdCase = SD_EXP_CUTOFF1;
		sd->a[ipAlpha] = 1.;
		mul = MUL_CO1;
	}
	else if( lgMatch( chWord, "EXP2" ) )
	{
		sd->sdCase = SD_EXP_CUTOFF2;
		sd->a[ipAlpha] = 2.;
		mul = MUL_CO2;
	}
	else if( lgMatch( chWord, "EXP3" ) )
	{
		sd->sdCase = SD_EXP_CUTOFF3;
		sd->a[ipAlpha] = 3.;
		mul = MUL_CO3;
	}
	else if( lgMatch( chWord, "LOGN" ) )
	{
		sd->sdCase = SD_LOG_NORMAL;
		mul = MUL_LND;
	}
	/* this one must come after LOGNORMAL */
	else if( lgMatch( chWord, "NORM" ) )
	{
		sd->sdCase = SD_LIN_NORMAL;
		mul = MUL_NRM;
	}
	else if( lgMatch( chWord, "TABL" ) )
	{
		sd->sdCase = SD_TABLE;
	}
	else
	{
		sd->sdCase = SD_ILLEGAL;
	}

	switch( sd->sdCase ) 
	{
	case SD_SINGLE_SIZE:
		/* single sized grain */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipSize],TRUE,dl);
		if( sd->a[ipSize] < SMALLEST_GRAIN || sd->a[ipSize] > LARGEST_GRAIN ) 
		{
			fprintf( ioQQQ, " Illegal value for grain size\n" );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}
		break;
	case SD_POWERLAW:
		/* simple power law distribution, first get lower limit */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBLo],TRUE,dl);
		if( sd->a[ipBLo] < SMALLEST_GRAIN || sd->a[ipBLo] > LARGEST_GRAIN ) 
		{
			fprintf( ioQQQ, " Illegal value for grain size (lower limit)\n" );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* upper limit */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBHi],TRUE,dl);
		if( sd->a[ipBHi] < SMALLEST_GRAIN || sd->a[ipBHi] > LARGEST_GRAIN ||
		    sd->a[ipBHi] <= sd->a[ipBLo] ) 
		{
			fprintf( ioQQQ, " Illegal value for grain size (upper limit)\n" );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " and upper limit should be greater than lower limit\n" );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* slope */
		mie_next_data(fnam,io2,chLine,&dl);
		if( sscanf( chLine ,"%lf",&sd->a[ipExp] ) != 1 ) 
		{
			fprintf( ioQQQ, " Syntax error in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		sd->a[ipBeta] = 0.;
		sd->a[ipSLo] = 0.;
		sd->a[ipSHi] = 0.;

		sd->lim[ipBLo] = sd->a[ipBLo];
		sd->lim[ipBHi] = sd->a[ipBHi];
		break;
	case SD_EXP_CUTOFF1:
	case SD_EXP_CUTOFF2:
	case SD_EXP_CUTOFF3:
		/* powerlaw with first/second/third order exponential cutoff, inspired by
		 * Greenberg (1978), Cosmic Dust, ed. J.A.M. McDonnell, Wiley, p. 187 */
		/* "lower limit", below this the exponential cutoff sets in */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBLo],TRUE,dl);

		/* "upper" limit, above this the exponential cutoff sets in */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBHi],TRUE,dl);

		/* exponent for power law */
		mie_next_data(fnam,io2,chLine,&dl);
		if( sscanf( chLine ,"%lf",&sd->a[ipExp] ) != 1 ) 
		{
			fprintf( ioQQQ, " Syntax error in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* beta parameter, for extra curvature in the powerlaw region */
		mie_next_data(fnam,io2,chLine,&dl);
		if( sscanf( chLine ,"%lf",&sd->a[ipBeta] ) != 1 ) 
		{
			fprintf( ioQQQ, " Syntax error in %s\n",fnam);
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* scale size for lower exponential cutoff, zero indicates normal cutoff */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipSLo],FALSE,dl);

		/* scale size for upper exponential cutoff, zero indicates normal cutoff */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipSHi],FALSE,dl);

		ref_neg = sd->a[ipBLo];
		step_neg = -mul*sd->a[ipSLo];
		ref_pos = sd->a[ipBHi];
		step_pos = mul*sd->a[ipSHi];
		lgTryOverride = TRUE;
		break;
	case SD_LOG_NORMAL:
		/* log-normal distribution, first get center of gaussian */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipGCen],TRUE,dl);

		/* 1-sigma width */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipGSig],TRUE,dl);

		/* ref_pos, ref_neg is the grain radius at which a^4*dN/da peaks */
		ref_neg = ref_pos = sd->a[ipGCen]*exp(3.*POW2(sd->a[ipGSig]));
		step_neg = sd->a[ipGCen]*(exp(-mul*sd->a[ipGSig]) - 1.);
		step_pos = sd->a[ipGCen]*(exp(mul*sd->a[ipGSig]) - 1.);
		lgTryOverride = TRUE;
		break;
	case SD_LIN_NORMAL:
		/* normal gaussian distribution, first get center of gaussian */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipGCen],TRUE,dl);

		/* 1-sigma width */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipGSig],TRUE,dl);

		/* ref_pos, ref_neg is the grain radius at which a^4*dN/da peaks */
		ref_neg = ref_pos = (sd->a[ipGCen] + sqrt(POW2(sd->a[ipGCen]) + 12.*POW2(sd->a[ipGSig])))/2.;
		step_neg = -mul*sd->a[ipGSig];
		step_pos = mul*sd->a[ipGSig];
		lgTryOverride = TRUE;
		break;
	case SD_TABLE:
		/* user-supplied table of a^4*dN/da vs. a, first get lower limit on a */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBLo],TRUE,dl);
		if( sd->a[ipBLo] < SMALLEST_GRAIN || sd->a[ipBLo] > LARGEST_GRAIN ) 
		{
			fprintf( ioQQQ, " Illegal value for grain size (lower limit)\n" );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* upper limit */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&sd->a[ipBHi],TRUE,dl);
		if( sd->a[ipBHi] < SMALLEST_GRAIN || sd->a[ipBHi] > LARGEST_GRAIN ||
		    sd->a[ipBHi] <= sd->a[ipBLo] ) 
		{
			fprintf( ioQQQ, " Illegal value for grain size (upper limit)\n" );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " and upper limit should be greater than lower limit\n" );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* number of user supplied points */
		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_long(fnam,chLine,&sd->npts,TRUE,dl);
		if( sd->npts < 2 )
		{
			fprintf( ioQQQ, " Illegal value for no. of points in table\n" );
			fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* allocate space for the table */
		sd->ln_a = (double *)MALLOC(sizeof(double)*(unsigned)sd->npts);
		sd->ln_a4dNda = (double *)MALLOC(sizeof(double)*(unsigned)sd->npts);
		if( sd->ln_a == NULL || sd->ln_a4dNda == NULL )
		{
			fprintf( ioQQQ, " Could not MALLOC arrays\n" );
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}

		/* and read the table */
		for( i=0; i < sd->npts; ++i )
		{
			double help1, help2;

			mie_next_data(fnam,io2,chLine,&dl);
			/* read data pair: a (micron), a^4*dN/da (arbitrary units) */
			if( sscanf( chLine , "%le %le", &help1, &help2 ) != 2 ) 
			{
				fprintf( ioQQQ, " Syntax error in %s\n",fnam);
				fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
				puts( "[Stop in mie_read_szd]" );
				cdEXIT(1);
			}

			if( help1 <= 0. || help2 <= 0. )
			{
				fprintf( ioQQQ, " Reading table failed on line #%ld of %s\n",dl,fnam );
				fprintf( ioQQQ, " Illegal data value %.6e or %.6e\n", help1, help2 );
				puts( "[Stop in mie_read_szd]" );
				cdEXIT(1);
			}

			sd->ln_a[i] = log(help1);
			sd->ln_a4dNda[i] = log(help2);

			if( i > 0 && sd->ln_a[i] <= sd->ln_a[i-1] )
			{
				fprintf( ioQQQ, " Reading table failed on line #%ld of %s\n",dl,fnam );
				fprintf( ioQQQ, " Grain radii should be monotonically increasing\n" );
				puts( "[Stop in mie_read_szd]" );
				cdEXIT(1);
			}
		}

		sd->lim[ipBLo] = sd->a[ipBLo];
		sd->lim[ipBHi] = sd->a[ipBHi];
		break;
	default:
		fprintf( ioQQQ, " unimplemented case for grain size distribution in file %s\n" , fnam );
		fprintf( ioQQQ, " Line #%ld: value %s\n",dl,chWord);
		puts( "[Stop in mie_read_szd]" );
		cdEXIT(1);
	}

	/* >>chng 01 feb 12, use a^4*dN/da instead of dN/da to determine limits,
	 * this assures that upper limit gives negligible mass fraction, PvH */
	/* in all cases where search_limit is used to determine lim[ipBLo] and lim[ipBHi],
	 * the user can override these values in the last two lines of the size distribution
	 * file. these inputs are mandatory, and should be given in the sequence lower
	 * limit, upper limit. a value <= 0 indicates that search_limit should be used. */
	if( lgTryOverride )
	{
		double help;

		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&help,FALSE,dl);
		sd->lim[ipBLo] = ( help <= 0. ) ? search_limit(ref_neg,step_neg,FRAC_CUTOFF,*sd) : help;

		mie_next_data(fnam,io2,chLine,&dl);
		mie_read_double(fnam,chLine,&help,FALSE,dl);
		sd->lim[ipBHi] = ( help <= 0. ) ? search_limit(ref_pos,step_pos,FRAC_CUTOFF,*sd) : help;

		if( sd->lim[ipBLo] < SMALLEST_GRAIN || sd->lim[ipBHi] > LARGEST_GRAIN ||
		    sd->lim[ipBHi] <= sd->lim[ipBLo] ) 
		{
			fprintf( ioQQQ, " Illegal size limits: lower %.5f and/or upper %.5f\n",
				 sd->lim[ipBLo], sd->lim[ipBHi] );
			fprintf( ioQQQ, " Grain sizes should be between %.5f and %.0f micron\n",
				 SMALLEST_GRAIN, LARGEST_GRAIN );
			fprintf( ioQQQ, " and upper limit should be greater than lower limit\n" );
			fprintf( ioQQQ, " Please alter the size distribution file\n" );
			puts( "[Stop in mie_read_szd]" );
			cdEXIT(1);
		}
	}

	fclose(io2);

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_szd()\n", debug_fp );
#	endif
	return;
}


static void mie_read_long(/*@in@*/  const char *fnam,
			  /*@in@*/  const char chLine[], /* chLine[LINELEN] */
			  /*@out@*/ long int *data,
			  int lgZeroIllegal,
			  long int dl)
{
#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_long()\n", debug_fp );
#       endif
	if( sscanf( chLine ,"%ld",data ) != 1 )
	{
		fprintf( ioQQQ, " Syntax error in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
		puts( "[Stop in mie_read_long]" );
		cdEXIT(1);
	}
	if( *data < 0 || (*data == 0 && lgZeroIllegal) )
	{
		fprintf( ioQQQ, " Illegal data value in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %ld\n",dl,*data);
		puts( "[Stop in mie_read_long]" );
		cdEXIT(1);
	}
#	ifdef DEBUG_FUN
	fputs( " <->mie_read_long()\n", debug_fp );
#	endif
	return;
}


static void mie_read_float(/*@in@*/  const char *fnam,
			   /*@in@*/  const char chLine[], /* chLine[LINELEN] */
			   /*@out@*/ float *data,
			   int lgZeroIllegal,
			   long int dl)
{
#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_float()\n", debug_fp );
#       endif
	if( sscanf( chLine ,"%f",data ) != 1 )
	{
		fprintf( ioQQQ, " Syntax error in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
		puts( "[Stop in mie_read_float]" );
		cdEXIT(1);
	}
	if( *data < 0. || (*data == 0. && lgZeroIllegal) )
	{
		fprintf( ioQQQ, " Illegal data value in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %14.6e\n",dl,*data);
		puts( "[Stop in mie_read_float]" );
		cdEXIT(1);
	}
#	ifdef DEBUG_FUN
	fputs( " <->mie_read_float()\n", debug_fp );
#	endif
	return;
}


static void mie_read_double(/*@in@*/  const char *fnam,
			    /*@in@*/  const char chLine[], /* chLine[LINELEN] */
			    /*@out@*/ double *data,
			    int lgZeroIllegal,
			    long int dl)
{
#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_double()\n", debug_fp );
#       endif
	if( sscanf( chLine ,"%lf",data ) != 1 )
	{
		fprintf( ioQQQ, " Syntax error in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %s\n",dl,chLine);
		puts( "[Stop in mie_read_double]" );
		cdEXIT(1);
	}
	if( *data < 0. || (*data == 0. && lgZeroIllegal) )
	{
		fprintf( ioQQQ, " Illegal data value in %s\n",fnam);
		fprintf( ioQQQ, " Line #%ld: %14.6e\n",dl,*data);
		puts( "[Stop in mie_read_double]" );
		cdEXIT(1);
	}
#	ifdef DEBUG_FUN
	fputs( " <->mie_read_double()\n", debug_fp );
#	endif
	return;
}


static void mie_read_form(/*@in@*/  const char chWord[], /* chWord[LINELEN] */
			  /*@out@*/ double elmAbun[],    /* elmAbun[LIMELM] */
			  /*@out@*/ double *no_atoms,
			  /*@out@*/ double *mol_weight)
{
	long int ipZ,
	  len;
	char *str;
	char chElmName[3];
	double frac;

#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_form()\n", debug_fp );
#       endif
	*no_atoms = 0.;
	*mol_weight = 0.;
	for( ipZ=0; ipZ < LIMELM; ipZ++ ) 
	{
		frac = 0.;
		strcpy(chElmName,elementnames.chElementSym[ipZ]);
		if( chElmName[1] == ' ' )
			chElmName[1] = '\0';
		str = strstr(chWord,chElmName);
		if( str != NULL ) 
		{
			len = (long)strlen(chElmName);
			/* prevent spurious match, e.g. F matches Fe */
			if( !islower((unsigned char)str[len]) ) 
			{
				if( isdigit((unsigned char)str[len]) ) 
				{
					sscanf(str+len,"%lf",&frac);
				}
				else 
				{
					frac = 1.;
				}
			}
		}
		elmAbun[ipZ] = frac;
		*no_atoms += frac;
		*mol_weight += frac*AtomcWgt.AtomicWeight[ipZ];
	}
	/* prevent division by zero when no chemical formula was supplied */
	if( *no_atoms == 0. ) 
	{
		*no_atoms = 1.;
		*mol_weight = 1.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_form()\n", debug_fp );
#	endif
	return;
}


static void mie_read_word(/*@in@*/  const char chLine[], /* chLine[LINELEN] */
			  /*@out@*/ char chWord[],       /* chWord[n] */
			  long n,
			  int toUpper)
{
	long ip = 0,
	  op = 0;

#       ifdef DEBUG_FUN
	fputs( "<+>mie_read_word()\n", debug_fp );
#       endif

	/* skip leading spaces or double quotes */
	while( chLine[ip] == ' ' || chLine[ip] == '"' )
		ip++;
	/* now copy string until we hit next space or double quote */
	while( op < n-1 && chLine[ip] != ' ' && chLine[ip] != '"' )
		if( toUpper )
			chWord[op++] = (char)toupper(chLine[ip++]);
		else
			chWord[op++] = chLine[ip++];
	/* the output string is always zero terminated */
	chWord[op] = '\0';

#	ifdef DEBUG_FUN
	fputs( " <->mie_read_word()\n", debug_fp );
#	endif
	return;
}

/*=====================================================================*/
static void mie_next_data(/*@in@*/  const char *fnam,
			  /*@in@*/  FILE *io,
			  /*@out@*/ char chLine[], /* chLine[LINELEN] */
			  /*@in@*/  long int *dl)
{
	char *str;

#	ifdef DEBUG_FUN
	fputs( "<+>mie_next_data()\n", debug_fp );
#	endif

	/* lines starting with a pound sign are considered comments and are skipped,
	 * lines not starting with a pound sign are considered to contain useful data.
	 * however, comments may still be appended to the line and will be erased. */

	chLine[0] = '#';
	while( chLine[0] == '#' )
	{
		mie_next_line(fnam,io,chLine,dl);
	}

	/* erase comment part of the line */
	str = strstr(chLine,"#");
	if( str != NULL )
		*str = '\0';

#	ifdef DEBUG_FUN
	fputs( " <->mie_next_data()\n", debug_fp );
#	endif
	return;
}


/*=====================================================================*/
static void mie_next_line(/*@in@*/  const char *fnam,
			  /*@in@*/  FILE *io,
			  /*@out@*/ char chLine[], /* chLine[LINELEN] */
			  /*@in@*/  long int *dl)
{
#	ifdef DEBUG_FUN
	fputs( "<+>mie_next_line()\n", debug_fp );
#	endif

	if( fgets( chLine , LINELEN , io ) == NULL ) 
	{
		fprintf( ioQQQ, " Could not read from %s\n",fnam);
		if( feof(io) )
			fprintf( ioQQQ, " EOF reached\n");
		puts( "[Stop in mie_next_line]" );
		cdEXIT(1);
	}
	(*dl)++;

#	ifdef DEBUG_FUN
	fputs( " <->mie_next_line()\n", debug_fp );
#	endif
	return;
}

/* make next two gauss.c ??? */

/*=====================================================================*/
/* set up Gaussian quadrature for arbitrary interval */
void gauss_init(long int nn,
	    double xbot,
	    double xtop,
	    double x[],  /* x[nn]  */
	    double a[],  /* a[nn]  */
	    double rr[], /* rr[nn] */
	    double ww[]) /* ww[nn] */
{
	long int i;
	double bma,
	  bpa;

#	ifdef DEBUG_FUN
	fputs( "<+>gauss_init()\n", debug_fp );
#	endif

	bpa = (xtop+xbot)/2.;
	bma = (xtop-xbot)/2.;

	for( i=0; i < nn; i++ ) 
	{
		rr[i] = bpa + bma*x[nn-1-i];
		ww[i] = bma*a[i];
	}

#	ifdef DEBUG_FUN
	fputs( " <->gauss_init()\n", debug_fp );
#	endif
	return;
}


/*=====================================================================*/
/* set up abscissas and weights for Gauss-Legendre intergration of arbitrary even order */
void gauss_legendre(long int nn,
	     double x[], /* x[nn] */
	     double a[]) /* a[nn] */
{
	long int i,
	  iter,
	  j;
	double *c,
	  cc,
	  csa,
	  d,
	  dp1,
	  dpn = 0.,
	  dq,
	  fj,
	  fn,
	  pn,
	  pn1 = 0.,
	  q,
	  xt = 0.;

	const double SAFETY = 5.;


#	ifdef DEBUG_FUN
	fputs( "<+>gauss_legendre()\n", debug_fp );
#	endif

	if( nn%2 == 1 ) 
	{
		fprintf( ioQQQ, " Illegal number of abcissas\n" );
		puts( "[Stop in gauss_legendre]" );
		cdEXIT(1);
	}

	if( ( c = (double *)MALLOC((size_t)(nn*sizeof(double))) ) == NULL )
		bad_malloc();

	fn = (double)nn;
	csa = 0.;
	cc = 2.;
	for( j=1; j < nn; j++ ) 
	{
		fj = (double)j;
		/* >>chng 01 apr 10, prevent underflows in cc, pn, pn1, dpn and dp1 for large nn
		 * renormalize c[j] -> 4*c[j],  cc -> 4^(nn-1)*cc,  hence cc = O(1), etc...
		 * Old code: c[j] = POW2(fj)/(4.*(fj-0.5)*(fj+0.5)); */
		c[j] = POW2(fj)/((fj-0.5)*(fj+0.5));
		cc *= c[j];
	}

	for( i=0; i < nn/2; i++ ) 
	{
		switch( i ) 
		{
		case 0:
			xt = 1. - 2.78/(4. + POW2(fn));
			break;
		case 1:
			xt = xt - 4.1*(1. + 0.06*(1. - 8./fn))*(1. - xt);
			break;
		case 2:
			xt = xt - 1.67*(1. + 0.22*(1. - 8./fn))*(x[0] - xt);
			break;
		default:
			xt = 3.*(x[i-1] - x[i-2]) + x[i-3];
		}
		d = 1.;
		for( iter=1; (iter < 20) && (fabs(d) > DBL_EPSILON); iter++ ) 
		{
			/* >>chng 01 apr 10, renormalize pn -> 2^(nn-1)*pn, dpn -> 2^(nn-1)*dpn
			 * pn1 -> 2^(nn-2)*pn1, dp1 -> 2^(nn-2)*dp1
			 * Old code: pn1 = 1.; */
			pn1 = 0.5;
			pn = xt;
			dp1 = 0.;
			dpn = 1.;
			for( j=1; j < nn; j++ )
			{
				/* >>chng 01 apr 10, renormalize pn -> 2^(nn-1)*pn, dpn -> 2^(nn-1)*dpn
				 * Old code: q = xt*pn - c[j]*pn1;  dq = xt*dpn - c[j]*dp1 + pn; */
				q = 2.*xt*pn - c[j]*pn1;
				dq = 2.*xt*dpn - c[j]*dp1 + 2.*pn;
				pn1 = pn;
				pn = q;
				dp1 = dpn;
				dpn = dq;
			}
			d = pn/dpn;
			xt -= d;
		}
		x[i] = xt;
		x[nn-1-i] = -xt;
		/* >>chng 01 apr 10, renormalize dpn -> 2^(nn-1)*dpn, pn1 -> 2^(nn-2)*pn1
		 * Old code: a[i] = cc/(dpn*pn1); */
		a[i] = cc/(dpn*2.*pn1);
		a[nn-1-i] = a[i];
		csa += a[i];
	}

	/* this routine has been tested for every even nn between 2 and 4096
	 * it passed the test for each of those cases with SAFETY < 3.11 */
	if( fabs(1.-csa) > SAFETY*fn*DBL_EPSILON ) 
	{
		fprintf( ioQQQ, " gauss_legendre failed to converge: delta = %.4e\n",fabs(1.-csa) );
		puts( "[Stop in gauss_legendre]" );
		cdEXIT(1);
	}
	free(c);

#	ifdef DEBUG_FUN
	fputs( " <->gauss_legendre()\n", debug_fp );
#	endif
	return;
}


/* find index ind such that min(xa[ind],xa[ind+1]) <= x <= max(xa[ind],xa[ind+1]).
 * xa is assumed to be strictly monotically increasing or decreasing.
 * if x is outside the range spanned by xa, lgOutOfBounds is raised and ind is set to -1
 * n is the number of elements in xa. */
void find_arr(double x,
	      double xa[],
	      long int n,
	      /*@out@*/ long int *ind,
	      /*@out@*/ int *lgOutOfBounds)
{
	long int i1,
		i2,
		i3,
		sgn,
		sgn2;

#	ifdef DEBUG_FUN
	fputs( "<+>find_arr()\n", debug_fp );
#	endif
	/* this routine works for strictly monotically increasing
	 * and decreasing arrays, sgn indicates which case it is */
	if( n < 2 ) 
	{
		fprintf( ioQQQ, " Invalid array\n");
		puts( "[Stop in find_arr]" );
		cdEXIT(1);
	}

	i1 = 0;
	i3 = n-1;
	sgn = SIGN3(xa[i3]-xa[i1]);
	if( sgn == 0 ) 
	{
		fprintf( ioQQQ, " Ill-ordered array\n");
		puts( "[Stop in find_arr]" );
		cdEXIT(1);
	}

	*lgOutOfBounds = x < MIN2(xa[0],xa[n-1]) || x > MAX2(xa[0],xa[n-1]);
	if( *lgOutOfBounds ) 
	{
		*ind = -1;
#		ifdef DEBUG_FUN
		fputs( " <->find_arr()\n", debug_fp );
#		endif
		return;
	}

	i2 = (n-1)/2;
	while( (i3-i1) > 1 ) 
	{
		sgn2 = SIGN3(x-xa[i2]);
		if( sgn2 != 0 )
		{
			if( sgn == sgn2 ) 
			{
				i1 = i2;
			}
			else 
			{
				i3 = i2;
			}
			i2 = (i1+i3)/2;
		}
		else 
		{
			*ind = i2;
#			ifdef DEBUG_FUN
			fputs( " <->find_arr()\n", debug_fp );
#			endif
			return;
		}
	}
	*ind = i1;
#	ifdef DEBUG_FUN
	fputs( " <->find_arr()\n", debug_fp );
#	endif
	return;
}


/* Oct 1988 for UV - X-ray extinction, including anomalous diffraction check
 *     this version reads in real and imaginary parts of the refractive
 *     index, with imaginary part positive (nridf = 3) or nr-1 (nridf = 2) or
 *     real and imaginary parts of the dielectric function (nridf = 1)
 * Dec 1988: added qback; approximation for small x;
 * qphase, better convergence checking
 *
 * in anomalous diffraction: qext and qabs calculated - qscatt by subtraction
 * in rayleigh-gans:         qscatt and qabs calculated
 * in mie:                   qext and qscatt calculated
 * */

/* sinpar.f
 * consistency checks updated july 1999
 * t1 updated mildly 19 oct 1992
 * utility for mieuvx.f and mieuvxsd.f */
#define	NMXLIM	16000

static void sinpar(double nre,
		   double nim,
		   double x,
		   /*@out@*/ double *qext,
		   /*@out@*/ double *qphase,
		   /*@out@*/ double *qscat,
		   /*@out@*/ double *ctbrqs,
		   /*@out@*/ double *qback,
		   /*@out@*/ long int *iflag)
{
	long int n,
	  nmx1,
	  nmx2,
	  nn,
	  nsqbk;
	double ectb,
	  eqext,
	  eqpha,
	  eqscat,
	  error=0.,
	  error1=0.,
	  rx,
	  t1,
	  t2,
	  t3,
	  t4,
	  t5,
	  tx,
	  x3,
	  x5=0.,
	  xcut,
	  xrd;
	/* >>chng 01 sep 11, replace array allocation on stack with
	 * MALLOC to avoid bug in gcc 3.0 on Sparc platforms, PvH */
/*  	complex a[NMXLIM], */
	complex *a;
	complex cdum1,
	  cdum2,
	  ci,
	  eqb,
	  nc,
	  nc2,
	  nc212,
	  qbck,
	  rrf,
	  rrfx,
	  sman,
	  sman1,
	  smbn,
	  smbn1,
	  tc1,
	  tc2,
	  wn,
	  wn1,
	  wn2;

#	ifdef DEBUG_FUN
	fputs( "<+>sinpar()\n", debug_fp );
#	endif

	if( ( a = (complex*)MALLOC((size_t)(NMXLIM*sizeof(complex))) ) == NULL )
		bad_malloc();

	*iflag = 0;
	ci = ftocm(0.,1.);
	nc = ftocm(nre,-nim);
	nc2 = cmmul(nc,nc);
	rrf = cmdiv(ftocm(1.,0.),nc);
	rx = 1.0/x;
	rrfx = cmmul(rrf,ftocm(rx,0.));

	/*  t1 is the number of terms nmx2 that will be needed to obtain convergence
	 *  try to minimize this, because the a(n) downwards recursion has to
	 *  start at nmx1 larger than this
	 *
	 * major loop series is summed to nmx2, or less when converged
	 * nmx1 is used for a(n) only, n up to nmx2.
	 * must start evaluation sufficiently above nmx2 that a(nmx1)=(0.,0.)
	 * is a good approximation
	 *
	 *
	 *orig with slight modification for extreme UV and X-ray, n near 1., large x
	 *orig      t1=x*dmax1( 1.1d0,dsqrt(nr*nr+ni*ni) )*
	 *orig     1(1.d0+0.02d0*dmax1(dexp(-x/100.d0)*x/10.d0,dlog10(x)))
	 *
	 * rules like those of wiscombe 1980 are slightly more efficient */
	xrd = exp(log(x)/3.);
	/* the final number in t1 was 1., 2. for large x, and 3. is needed sometimes
	 * see also idnint use below */
	t1 = x + 4.*xrd + 3.;
	/*      t1=t1+0.05d0*xrd
	 * was 0., then 1., then 2., now 3. for intermediate x
	 * 19 oct 1992 */
	if( !(x <= 8. || x >= 4200.) )
		t1 += 0.05*xrd + 3.;
	t1 *= 1.01;

	/* the original rule of dave for starting the downwards recursion was
	 * to start at 1.1*|mx| + 1, i.e. with the original version of t1
	 *orig      nmx1=1.10d0*t1
	 *
	 * try a simpler, less costly one, as in bohren and huffman, p 478
	 * this is the form for use with wiscombe rules for t1
	 * tests: it produces the same results as the more costly version
	 * */
	t4 = x*sqrt(nre*nre+nim*nim);
	nmx1 = NINT(MAX2(t1,t4)) + 15;

	if( nmx1 < NMXLIM ) 
	{
		nmx2 = NINT(t1);
		/*orig      if ( nmx1  .gt. 150 ) go to 22
		 *orig      nmx1 = 150
		 *orig      nmx2 = 135
		 *
		 * try a more efficient scheme */
		if( nmx2 <= 4 ) 
		{
			nmx2 = 4;
			nmx1 = NINT(MAX2(4.,t4)) + 15;
		}

		/* downwards recursion for logarithmic derivative */
		a[nmx1] = ftocm(0.,0.);

		/* note that with the method of lentz 1976 (appl opt 15, 668), it would be
		 * possible to find a(nmx2) directly, and start the downwards recursion there
		 * however, there is not much in it with above form for nmx1 which uses just */
		for( n=0; n < nmx1; n++ )
		{
			nn = nmx1 - n;
			a[nn-1] = cmsub(cmmul(ftocm((double)(nn+1),0.),rrfx),cmdiv(ftocm(1.,0.),
			  (cmadd(cmmul(ftocm((double)(nn+1),0.),rrfx),a[nn]))));
		}

		t1 = cos(x);
		t2 = sin(x);
		wn2 = ftocm(t1,-t2);
		wn1 = ftocm(t2,t1);
		wn = cmsub(cmmul(ftocm(rx,0.),wn1),wn2);
		tc1 = cmadd(cmmul(a[0],rrf),ftocm(rx,0.));
		tc2 = cmadd(cmmul(a[0],nc),ftocm(rx,0.));
		sman = cmdiv((cmsub(cmmul(tc1,ftocm(wn.re,0.)),ftocm(wn1.re,0.))),
		  (cmsub(cmmul(tc1,wn),wn1)));
		smbn = cmdiv((cmsub(cmmul(tc2,ftocm(wn.re,0.)),ftocm(wn1.re,0.))),
		  (cmsub(cmmul(tc2,wn),wn1)));

		/* small x; above calculations subject to rounding errors
		 * see bohren and huffman p 131
		 * wiscombe 1980 appl opt 19, 1505 gives alternative formulation */
		xcut = 3.e-04;
		if( x < xcut ) 
		{
			nc212 = cmdiv((cmsub(nc2,ftocm(1.,0.))),(cmadd(nc2,ftocm(2.,0.))));
			x3 = POW3(x);
			x5 = x3*POW2(x);
			/* note change sign convention for m = n - ik here */
			sman = cmadd(cmmul(cmmul(cmmul(cmmul(ci,ftocm(2.,0.)),ftocm(x3,0.)),
			  nc212),(cmadd(ftocm(1./3.,0.),cmdiv(cmmul(ftocm(x*x*0.2,0.),
			  (cmsub(nc2,ftocm(2.,0.)))),(cmadd(nc2,ftocm(2.,0.))))))),
			  cmdiv(cmmul(cmmul(ftocm(4.*x5*x,0.),nc212),nc212),ftocm(9.,0.)));
			smbn = cmdiv(cmmul(cmmul(ci,ftocm(x5,0.)),(cmsub(nc2,ftocm(1.,0.)))),
			  ftocm(45.,0.));
		}

		sman1 = sman;
		smbn1 = smbn;
		t1 = 1.5;
		sman.re *= t1;
		sman.im *= t1;
		smbn.re *= t1;
		smbn.im *= t1;
		/* case n=1; note previous multiplication of sman and smbn by t1=1.5 */
		*qext = 2.*(sman.re + smbn.re);
		*qphase = 2.*(sman.im + smbn.im);
		nsqbk = -1;
		qbck.re = -2.*(sman.re - smbn.re);
		qbck.im = -2.*(sman.im - smbn.im);
		*qscat = (POW2(sman.re) + POW2(sman.im) + POW2(smbn.re) + POW2(smbn.im))/.75;
		  
		*ctbrqs = 0.0;
		n = 2;

		/************************* Major loop begins here ************************/
		while( TRUE ) 
		{
			t1 = (double)(2*n - 1);
			t3 = (double)(2*n + 1);
			wn2 = wn1;
			wn1 = wn;
			wn = cmsub(cmmul(ftocm(t1*rx,0.),wn1),wn2);
			cdum1 = a[n-1];
			cdum2 = ftocm(n*rx,0.);
			tc1 = cmadd(cmmul(cdum1,rrf),cdum2);
			tc2 = cmadd(cmmul(cdum1,nc),cdum2);
			sman = cmdiv((cmsub(cmmul(tc1,ftocm(wn.re,0.)),ftocm(wn1.re,0.))),
			  (cmsub(cmmul(tc1,wn),wn1)));
			smbn = cmdiv((cmsub(cmmul(tc2,ftocm(wn.re,0.)),ftocm(wn1.re,0.))),
			  (cmsub(cmmul(tc2,wn),wn1)));

			/* small x, n=2
			 * see bohren and huffman p 131 */
			if( x < xcut && n == 2 ) 
			{
				/* note change sign convention for m = n - ik here */
				sman = cmdiv(cmmul(cmmul(ci,ftocm(x5,0.)),(cmsub(nc2,ftocm(1.,0.)))),
					 (cmmul(ftocm(15.,0.),(cmadd(cmmul(ftocm(2.,0.),nc2),
					 ftocm(3.,0.))))));
				smbn = ftocm(0.,0.);
			}

			eqext = t3*(sman.re + smbn.re);
			*qext += eqext;
			eqpha = t3*(sman.im + smbn.im);
			*qphase += eqpha;
			nsqbk = -nsqbk;
			eqb.re = t3*(sman.re - smbn.re)*nsqbk;
			qbck.re += eqb.re;
			eqb.im = t3*(sman.im - smbn.im)*nsqbk;
			qbck.im += eqb.im;
			tx = POW2(sman.re) + POW2(sman.im) + POW2(smbn.re) + POW2(smbn.im);
			eqscat = t3*tx;
			*qscat += eqscat;
			t2 = (double)(n - 1);
			t5 = (double)n;
			t4 = t1/(t5*t2);
			t2 = (t2*(t5 + 1.))/t5;
			ectb = t2*(sman1.re*sman.re + sman1.im*sman.im + smbn1.re*smbn.re +
				 smbn1.im*smbn.im) + t4*(sman1.re*smbn1.re + sman1.im*smbn1.im);
			*ctbrqs += ectb;

			/* check convergence
			 * could decrease for large x and small m-1 in UV - X-ray; probably negligible */
			if( tx < 1.e-14 ) 
			{
				/* looks good but check relative convergence */
				eqext = fabs(eqext/ *qext);
				eqpha = fabs(eqpha/ *qphase);
				eqscat = fabs(eqscat/ *qscat);
				ectb = ( n == 2 ) ? 0. : fabs(ectb/ *ctbrqs);
				eqb.re = fabs(eqb.re/qbck.re);
				eqb.im = fabs(eqb.im/qbck.im);
				/* leave out eqb.re/im, which are sometimes least well converged */
				error = MAX4(eqext,eqpha,eqscat,ectb);
				/* put a milder constraint on eqb.re/im */
				error1 = MAX2(eqb.re,eqb.im);
				if( error < 1.e-07 && error1 < 1.e-04 )
					break;

				/* not sufficiently converged
				 *
				 * cut out after n=2 for small x, since approximation is being used */
				if( x < xcut )
					break;
			}

			smbn1 = smbn;
			sman1 = sman;
			n++;
			if( n > nmx2 ) 
			{
				*iflag = 1;
				break;
			}
		}
		/* renormalize */
		t1 = 2.*POW2(rx);
		*qext *= t1;
		*qphase *= t1;
		*qback = (POW2(qbck.re) + POW2(qbck.im))*POW2(rx);
		*qscat *= t1;
		*ctbrqs *= 2.*t1;
	}
	else 
	{
		*iflag = 2;
	}

	free( a );

#	ifdef DEBUG_FUN
	fputs( " <->sinpar()\n", debug_fp );
#	endif
	return;
}


static void anomal(double x,
		   /*@out@*/ double *qext,
		   /*@out@*/ double *qabs,
		   /*@out@*/ double *qphase,
		   /*@out@*/ double *xistar,
		   double delta,
		   double beta)
{
	/*
	 *
	 * in anomalous diffraction: qext and qabs calculated - qscatt by subtraction
	 * in rayleigh-gans:         qscatt and qabs calculated
	 * in mie:                   qext and qscatt calculated
	 *
	 */
	double xi,
	  xii;
	complex cbigk,
	  ci,
	  cw;

#	ifdef DEBUG_FUN
	fputs( "<+>anomal()\n", debug_fp );
#	endif
	/* anomalous diffraction: x>>1 and |m-1|<<1, any xi,xii
	 * original approach see Martin 1970. MN 149, 221 */
	xi = 2.*x*delta;
	xii = 2.*x*beta;
	/* xistar small is the basis for rayleigh-gans, any x, m-1 */
	*xistar = sqrt(POW2(xi)+POW2(xii));
	/* alternative approach see martin 1978 p 23 */
	ci = ftocm(0.,1.);
	cw = cmneg(cmmul(ftocm(xi,xii),ci));
	bigk(cw,&cbigk);
	*qext = 4.*cbigk.re;
	*qphase = 4.*cbigk.im;
	cw = ftocm(2.*xii,0.);
	bigk(cw,&cbigk);
	*qabs = 2.*cbigk.re;
	/* ?? put g in here - analytic version not known */

#	ifdef DEBUG_FUN
	fputs( " <->anomal()\n", debug_fp );
#	endif
	return;
}


static void bigk(complex cw,
		 /*@out@*/ complex *cbigk)
{
	/*
	 * see martin 1978 p 23
	 */

#	ifdef DEBUG_FUN
	fputs( "<+>bigk()\n", debug_fp );
#	endif
	/* non-vax; use generic function */
	if( cmabs(cw) < 1.e-2 ) 
	{
		/* avoid severe loss of precision for small cw; expand exponential
		 * coefficients are 1/n! - 1/(n+1)! = 1/(n+1)(n-1)! ;n=2,3,4,5,6,7
		 * accurate to  (1+ order cw**6) */
		*cbigk = cmmul(cw,(cmsub(ftocm(1./3.,0.),cmmul(cw,(cmsub(ftocm(1./
		  8.,0.),cmmul(cw,(cmsub(ftocm(1./30.,0.),cmmul(cw,(cmsub(ftocm(1./
		  144.,0.),cmmul(cw,(cmsub(ftocm(1./840.,0.),cmmul(cw,ftocm(1./
		  5760.,0.)))))))))))))))));
	}
	else 
	{
		*cbigk = cmadd(ftocm(0.5,0.),cmdiv((cmsub(cmmul(cmexp(cmneg(cw)),
		  (cmadd(ftocm(1.,0.),cw))),ftocm(1.,0.))),(cmmul(cw,cw))));
	}

#	ifdef DEBUG_FUN
	fputs( " <->bigk()\n", debug_fp );
#	endif
	return;
}


/* utility for use with mieuvx/sd */
static void dftori(/*@out@*/ double *nr,
		   /*@out@*/ double *ni,
		   double eps1,
		   double eps2)
{
	double eps;

#	ifdef DEBUG_FUN
	fputs( "<+>dftori()\n", debug_fp );
#	endif
	/* dielectric function to refractive index  */
	eps = sqrt(eps2*eps2+eps1*eps1);
	*nr = sqrt((eps+eps1)/2.);
	*ni = sqrt((eps-eps1)/2.);

#	ifdef DEBUG_FUN
	fputs( " <->dftori()\n", debug_fp );
#	endif
	return;
}


static complex ftocm(double r,double i )	/* convert floats to dp complex */
{
	complex dz;
	dz.re = r;
	dz.im = i;
	return( dz );
}

static complex cmneg(complex dz )	/* dp complex unary minus operation */
{
	dz.re = -dz.re;
	dz.im = -dz.im;
	return( dz );
}

static double cmabs(complex dz )	/* dp complex absolute value */
{
	double s;

	s = dz.re*dz.re + dz.im*dz.im;
	return( sqrt(s) );
}

static complex cmadd(complex l,complex r )	/* retn the sum of two dp complex nos. */
{
	l.re += r.re;
	l.im += r.im;
	return( l );
}

static complex cmsub(complex l,complex r )	/* retn the difference of two dp complex nos. */
{
	l.re -= r.re;
	l.im -= r.im;
	return( l );
}


static complex cmmul(complex l,complex r )	/* retn the product of two dp complex nos. */
{
	complex dz;
	dz.re = l.re*r.re - l.im*r.im;
	dz.im = l.re*r.im + l.im*r.re;
	return( dz );
}

static complex cmdiv(complex l,complex r )	/* retn the quotient of two dp complex nos. */
{
	complex dz;
	double den;

	if( r.re == 0. && r.im == 0. )
		fprintf( ioQQQ, " Complex division by 0." );
	den = r.re*r.re + r.im*r.im;

	dz.re = ( l.re*r.re + l.im*r.im )/den;
	dz.im = ( r.re*l.im - l.re*r.im )/den;
	return( dz );
}

static complex cmexp(complex dz )	/* dp complex exponential */
{
	double exp_re;

	if( dz.re == 0. )
		exp_re = 1.;
	else
		exp_re = exp(dz.re);

	if( dz.im == 0.0 ) {	/* real only complex no. */
		dz.re = exp_re;
		return( dz );
	}
	dz.re = exp_re*cos(dz.im);
	dz.im = exp_re*sin(dz.im);
	return( dz );
}
