/*HydroRecom photoionization, recombination, radiative rates for model hydrogen atom */
#include "cddefines.h"
#include "hydrogenic.h"
#include "iso.h"
#include "opacity.h"
#include "trace.h"
#include "phycon.h"
#include "recom.h"
#include "hcaseb.h"
#include "punch.h"
#include "elementnames.h"
#include "hrfit.h"
#include "receff.h"

void HydroRecom(
	/* ipZ is on the C scale, 0 for H */
	long int ipZ)
{
	long int n;
	int lgOK;
	double extra, 
	  SumCaseB ,
	  SumTopOff;
	/* will be used to save temperature where rec coef evaluated */
	static double TeUsed[LIMELM]={0.};

#	ifdef DEBUG_FUN
	fputs( "<+>HydroRecom()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	assert( ipZ >= 0);
	assert( ipZ < LIMELM );

	/* evaluate recombination escape probability for all levels */
	for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
	{
		/* this is escape probability */
		iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc] = (float)receff(iso.ipIsoLevNIonCon[ipHYDROGEN][ipZ][n]);

		/* otsmin set to zero in zerologic, set to 1 with 
		 * the NO ON THE SPOT command */
		iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc] = 
			(float)MAX2(iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc], opac.otsmin);
	}
#	if 0
	n = iso.ipIsoLevNIonCon[ipHYDROGEN][0][3];
	if( ipZ==0 ) fprintf(ioQQQ,"%i\thydrogrecom optical depths\t%.2e\t%.2e\t%.2e \n",
		nzone,
		opac.TauAbsGeo[0][n-1],
		opac.TauAbsGeo[1][n-1],
		opac.TauAbsGeo[1][n-1] - opac.TauAbsGeo[0][n-1]);
#	endif

	/* option for case b conditions, kill ground state recombination */
	if( opac.lgCaseB )
	{
		iso.RadRecomb[ipHYDROGEN][ipZ][ipH1s][ipRecEsc] = 1e-10f;
	}

	/* always reevaluate net escape probability, which depends on background opacities */
	for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
	{
		/* This is the one that multiplies the radiative recombination rate for level n
		 * following accounts for absorption of diffuse continuum
		 * by .not.hydrogen, includes modified OTS factor*/
		iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc] = 
			(float)MIN2(1.,iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc]+
		  (1.-iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc])*iso.ConOpacRatio[ipHYDROGEN][ipZ][n]);

		assert( iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc] >= 0. && 
			iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc] <= 1. );
	}
	/*if( ipZ==1 ) fprintf(ioQQQ," he ground %.2e %.2e %.2e \n",
		iso.RadRecomb[ipHYDROGEN][ipZ][0][ipRecNetEsc], iso.RadRecomb[ipHYDROGEN][ipZ][0][ipRecEsc],iso.ConOpacRatio[ipHYDROGEN][ipZ][0]);*/

	if( (trace.lgTrace && trace.lgHBugFull) && (ipZ == trace.ipZTrace) )
	{
		/* print continuum escape probability */
		fprintf( ioQQQ, "       HydroRecom recomb effic Z%li ",ipZ );
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			fprintf( ioQQQ,PrintEfmt("%10.3e", iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecEsc] ));
		}
		fprintf( ioQQQ, "\n" );

		/* net recombination efficiency factor, including background opacity*/
		fprintf( ioQQQ, "       HydroRecom recomb net effic" );
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			fprintf( ioQQQ,PrintEfmt("%10.3e", iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc]) );
		}
		fprintf( ioQQQ, "\n" );

		/* inward continuous optical depths */
		fprintf( ioQQQ, "       HydroRecom in optic dep%10.3e", 
		  opac.TauAbsGeo[0][iso.ipIsoLevNIonCon[ipHYDROGEN][ipZ][ipH1s]-1] );
		for( n=2; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			fprintf( ioQQQ,PrintEfmt("%10.3e",  opac.TauAbsGeo[0][iso.ipIsoLevNIonCon[ipHYDROGEN][ipZ][n]-1] ));
		}
		fprintf( ioQQQ, "\n" );

		/* outward continuous optical depths*/
		fprintf( ioQQQ, "       HydroRecom out op depth%10.3e", 
		  opac.TauAbsGeo[1][iso.ipIsoLevNIonCon[ipHYDROGEN][ipZ][ipH1s]-1] );
		for( n=2; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			fprintf( ioQQQ,PrintEfmt("%10.3e",  opac.TauAbsGeo[1][iso.ipIsoLevNIonCon[ipHYDROGEN][ipZ][n]-1] ));
		}
		fprintf( ioQQQ, "\n" );
	}

	if( fabs(1.-TeUsed[ipZ]/phycon.te)> 0.001 )
	{
		TeUsed[ipZ] = phycon.te;
		/* evaluate radiative recom coef */
		/* first the ground state with Dima's routine */
		iso.RadRecomb[ipHYDROGEN][ipZ][ipH1s][ipRecRad] = (float)hrfit(ipZ+1,ipH1s,phycon.te);

		/* evaluate excited state rec coef, also do sum over all excited states, 
		 * SumCaseB will be true case B sum */
		SumCaseB = 0.;
		for( n=ipH2s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad] = (float)hrfit(ipZ+1,n,phycon.te);
			SumCaseB += iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad];
		}

		/* following returns case A rec coef, then we will subtract ground */
		iso.RadRec_caseB[ipHYDROGEN][ipZ] = (float)hrfit(ipZ+1,-1,phycon.te);

		/* save the total recombination rate, only used for printouts in recombmake,
		 * at this point CaseBradrec actually has the total recombination */
		recom.RecombinRate[ipZ][ipZ] = iso.RadRec_caseB[ipHYDROGEN][ipZ]*(float)phycon.eden;

		/* subtract ground to get correct case b radiative sum */
		iso.RadRec_caseB[ipHYDROGEN][ipZ] -= iso.RadRecomb[ipHYDROGEN][ipZ][ipH1s][ipRecRad];

		/* extra is +/- part needed to be added on to get correct total */
		extra = iso.RadRec_caseB[ipHYDROGEN][ipZ] - SumCaseB;

		/* only top off if this extra is positive since for large atom, sum is better
		 * than case b fit/
		 * adjust recombinations to levels to get total sum correct 
		 * default topoff is " add"
		 * with very large atom and high temperatures, extra can be negative,
		 * in which case we want to scale to make sure nothing is negative*/
		if( extra > 0. )
		{
			if( strcmp(hydro.chHTopType,"scal") == 0 )
			{
				/* this is now scale factor to get correct sum, we will add for n=3 to end
				 * nHTopOff default is 10, change with atom h-like topoff */
				SumTopOff = 0;
				for( n=iso.nTopOff[ipHYDROGEN]; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
				{
					SumTopOff += iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad];
				}

				/* scale factor to get right sum */
				if( SumTopOff > 1e-30 )
				{
					extra = (1. + extra/SumTopOff);
				}
				else
				{
					extra = 1.;
				}

				/* this is the top-off top off of the h atom */
				for( n=iso.nTopOff[ipHYDROGEN]; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
				{
					iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad] *= (float)extra;
				}
			}
			else if( strcmp(hydro.chHTopType," add") == 0 )
			{
				/* spread it out by adding same to all top levels
				 * this is default */
				extra /= iso.nLevels[ipHYDROGEN][ipZ] - iso.nTopOff[ipHYDROGEN];
				for( n=iso.nTopOff[ipHYDROGEN]; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
				{
					iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad] += (float)extra;
				}
			}
			else
			{
				/* this is insanity */
				fprintf( ioQQQ, " HydroRecom has insane chHTopType=%4.4s\n", 
				  hydro.chHTopType );
				ShowMe();
				puts( "[Stop in hydrorecom]" );
				cdEXIT(1);
			}
		}

		/*begin sanity check */
		lgOK = TRUE;
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			if( iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad] <= 0. )
			{
				fprintf( ioQQQ, 
					" HydroRecom non-positive recombination coefficient for Z=%3ld lev n=%3ld rec=%11.2e te=%11.2e\n", 
				  ipZ, n, iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad] , phycon.te);
					lgOK = FALSE;
			}
		}
		/* bail if we found problems */
		if( !lgOK )
		{
			ShowMe();
			puts( "[Stop in HydroRecom]" );
			cdEXIT(1);
		}
		/*end sanity check */

		if( (trace.lgTrace && trace.lgHBugFull) && (ipZ == trace.ipZTrace) )
		{
			fprintf( ioQQQ, "       HydroRecom eval rec cof" );
			for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad]) );
			}
			fprintf( ioQQQ, "\n" );
		}
	} 
	/* end branch checking on change in temperature */

	/* confirm that we have good rec coef at bottom and top of h atom */
	assert( iso.RadRecomb[ipHYDROGEN][ipZ][ipH1s][ipRecRad] > 0. );
	assert( iso.RadRecomb[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1][ipRecRad] > 0. );

	/* RadRec_effec is total effective radiative recombination */
	iso.RadRec_effec[ipHYDROGEN][ipZ] = 0.;
	for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
	{
		iso.RadRec_effec[ipHYDROGEN][ipZ] += iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad]*
		  iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc];
	}

	/* set true when punch recombinatino coefficients command entered */
	if( punch.lgioRecom )
	{
		if( ipZ==0 )
		{
			fprintf( punch.ioRecom, "Hydrogenic species\n");
		}
		/* this prints Z on physical, not C, scale */
		fprintf( punch.ioRecom, "H-like %2li %s ", 
			ipZ+1 , elementnames.chElementSym[ipZ] );
		fprintf( punch.ioRecom,PrintEfmt("%9.2e", iso.RadRec_caseB[ipHYDROGEN][ipZ] ));
		fprintf( punch.ioRecom, "\n" );
	}

	if( trace.lgTrace && trace.lgHBug )
	{
		fprintf( ioQQQ, "     HydroRecom Z=%3ld total rec coef", ipZ );
		fprintf( ioQQQ,PrintEfmt("%10.3e", iso.RadRec_effec[ipHYDROGEN][ipZ] ));
		fprintf( ioQQQ, " case A=" );
		fprintf( ioQQQ,PrintEfmt("%10.3e", 
			iso.RadRec_caseB[ipHYDROGEN][ipZ] + iso.RadRecomb[ipHYDROGEN][ipZ][ipH1s][ipRecRad] ) );
		fprintf( ioQQQ, " caseB=");
		fprintf( ioQQQ,PrintEfmt("%10.3e", iso.RadRec_caseB[ipHYDROGEN][ipZ] ));
		fprintf( ioQQQ, "\n" );
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroRecom()\n", debug_fp );
#	endif
	return;
}

