/*HydroLevelPop solve for ionization balance level populations of model hydrogen atom */
#include "cddefines.h"
#include "taulines.h"
#include "iso.h"
#include "secondaries.h"
#include "dynamics.h"
#include "chargtran.h"
#include "trace.h"
#include "phycon.h"
#include "hcaseb.h"
#include "ionfracs.h"
#include "typmatrx.h"
#include "linpack.h"
#include "veclib.h"
#include "hydrogenic.h"

void HydroLevelPop(long int ipZ ,
	double **SaveZ/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/, 
	double *bvec/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	double *error/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	double *work/*[iso.nLevels[ipHYDROGEN][ipZ]+2]*/, 
	double **z/*[iso.nLevels[ipHYDROGEN][ipZ]+2][iso.nLevels[ipHYDROGEN]+2]*/ ,
	long int *ipiv , /* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1] */
	double *totcap/* MALLOC out to [iso.nLevels[ipHYDROGEN][ipZ]+1]*/)
{
	long int n, 
	  ipHi, 
	  ipLo, 
	  j, 
	  job, 
	  level, 
	  nerror1,
	  nerror2;

	double HRateDestGnd[8];

#	ifdef DEBUG_FUN
	fputs( "<+>HydroLevelPop()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	assert( ipZ >= 0);
	assert( ipZ < LIMELM );

	/* 
	 * following electron density has approximate correction for neutrals
	 * corr of hi*1.7e-4 accounts for col ion by HI; Drawin Zs Phys 225, 483.
	 * used EdenHCorr instead
	 * edhi = eden + hi * 1.7e-4
	 */

	hcaseb.HRecEffec[ipZ] = 0.;
	for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
	{

		/* swap around logic, to be simpler */
#		if 0
		/* total rate level n is populated from the continuum */
		/* PopLTE(n,ipZ) is only LTE pop when mult by Ne Nh */
		z[iso.nLevels[ipHYDROGEN][ipZ]][n] = (

			/* radiative recombination */
			(double)(iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad]*
			iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc]) + 

			/* induced recombination */
			iso.RecomInducRate[ipHYDROGEN][ipZ][n]*iso.PopLTE[ipHYDROGEN][ipZ][n] + 

			/* collisional recombination */
			iso.ColIoniz[ipHYDROGEN][ipZ][n]*phycon.EdenHCorr*iso.PopLTE[ipHYDROGEN][ipZ][n] )*phycon.eden;

		totcap[n] = z[iso.nLevels[ipHYDROGEN][ipZ]][n]/phycon.eden;
#		endif
		/* total rate level n is populated from the continuum */
		/* PopLTE(n,ipZ) is only LTE pop when mult by Ne Nh */
		totcap[n] = 
			/* radiative recombination */
			(double)(iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecRad]*
			iso.RadRecomb[ipHYDROGEN][ipZ][n][ipRecNetEsc]) + 

			/* induced recombination */
			iso.RecomInducRate[ipHYDROGEN][ipZ][n]*iso.PopLTE[ipHYDROGEN][ipZ][n] + 

			/* collisional recombination */
			iso.ColIoniz[ipHYDROGEN][ipZ][n]*phycon.EdenHCorr*iso.PopLTE[ipHYDROGEN][ipZ][n] ;

		z[iso.nLevels[ipHYDROGEN][ipZ]][n] = totcap[n] * phycon.eden;

		hcaseb.HRecEffec[ipZ] += (float)totcap[n];
	}

	/* master balance equation */
	for( level=ipH1s; level < iso.nLevels[ipHYDROGEN][ipZ]; level++ )
	{

		/* all process depopulating level */
		z[level][level] = iso.gamnc[ipHYDROGEN][ipZ][level] + 
		  iso.ColIoniz[ipHYDROGEN][ipZ][level]* phycon.EdenHCorr + 
		  Secondaries.csupra;

		/* all processes populating level from below */
		for( ipLo=ipH1s; ipLo < level; ipLo++ )
		{
			double CollRate = EmisLines[ipHYDROGEN][ipZ][level][ipLo].ColUL* phycon.EdenHCorr;

			z[ipLo][level] = 
				-CollRate * 
			  (double)iso.stat[ipHYDROGEN][level]/(double)iso.stat[ipHYDROGEN][ipLo]*
			  iso.Boltzmann[ipHYDROGEN][ipZ][level][ipLo]  -
			  EmisLines[ipHYDROGEN][ipZ][level][ipLo].pump ;

			/* pumping out of here to lower level */
			/* fixit(); for 1s-2s two photon induced processes, pump contains the
			 * 1s - 2s pump rate, while Aul has the 2s-1s induced rate added in,
			 * and so is the total.  This should have a branch testing for 1s-2s
			 * and included the reverse pump in that case.  should not matter since
			 * induced 2s-1s should be slow compated to spontaneous A */
			z[level][level] += EmisLines[ipHYDROGEN][ipZ][level][ipLo].pump *
			  iso.stat[ipHYDROGEN][ipLo]/iso.stat[ipHYDROGEN][level];

			/* collisions out of here to lower level */
			z[level][level] += CollRate;

			/* radiative decays out of here to lower level */
			z[level][level] += 
				EmisLines[ipHYDROGEN][ipZ][level][ipLo].Aul*
			  (EmisLines[ipHYDROGEN][ipZ][level][ipLo].Pesc + 
			  EmisLines[ipHYDROGEN][ipZ][level][ipLo].Pdest);
		}

		/* all processes populating level from above */
		for( ipHi=level + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
		{
			double RadDecay , CollRate ;
			RadDecay =
				EmisLines[ipHYDROGEN][ipZ][ipHi][level].Aul*
			  (EmisLines[ipHYDROGEN][ipZ][ipHi][level].Pesc + 
			  EmisLines[ipHYDROGEN][ipZ][ipHi][level].Pdest);

			/* >>chng 02 feb 06, define this and use in each direction below -
			 * attempt at solving roundoff problems on alphas */
			CollRate = EmisLines[ipHYDROGEN][ipZ][ipHi][level].ColUL *phycon.EdenHCorr;

			z[ipHi][level] = 
				-(RadDecay + 
			  EmisLines[ipHYDROGEN][ipZ][ipHi][level].pump*
			  iso.stat[ipHYDROGEN][level]/iso.stat[ipHYDROGEN][ipHi] + 
			  CollRate );

			/* pumping out of here to upper level */
			z[level][level] += EmisLines[ipHYDROGEN][ipZ][ipHi][level].pump ;

			/* collisions out of here to upper level */
			z[level][level] += (double)iso.stat[ipHYDROGEN][ipHi] / (double)iso.stat[ipHYDROGEN][level] *
				iso.Boltzmann[ipHYDROGEN][ipZ][ipHi][level]*CollRate;
		}
	}

	if( Secondaries.Hx12[0][1] > 0. )
	{
		/* now add on supra thermal excitation */
		for( level=ipH2s; level < iso.nLevels[ipHYDROGEN][ipZ]; level++ )
		{
#			if 0
			/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for atom */
			z[ipH1s][ipH1s] += Secondaries.Hx12[MIN2(ipZ,1)][level];

			z[level][ipH1s] -= 
				(Secondaries.Hx12[MIN2(ipZ,1)][level]*
				iso.stat[ipHYDROGEN][ipH1s]/ iso.stat[ipHYDROGEN][level]);

			z[ipH1s][level] -= 
				Secondaries.Hx12[MIN2(ipZ,1)][level];

			z[level][level] += Secondaries.Hx12[MIN2(ipZ,1)][level]*
				iso.stat[ipHYDROGEN][ipH1s]/ iso.stat[ipHYDROGEN][level];
#			endif
			double RateUp , RateDown;

			RateUp = Secondaries.Hx12[MIN2(ipZ,1)][level];
			RateDown = RateUp * (double)iso.stat[ipHYDROGEN][ipH1s] /
				(double)iso.stat[ipHYDROGEN][level];

			/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for ion */
			/* total rate out of lower level */
			z[ipH1s][ipH1s] += RateUp;

			/* rate from the upper level to ground */
			z[level][ipH1s] -= RateDown ;

			/* rate from ground to upper level */
			z[ipH1s][level] -= RateUp ;

			z[level][level] += RateDown;  
		}
	}

	/* charge transfer, hydrogen onto everything else */
	if( ipZ == 0 )
	{
		/* >>chng 01 apr 28, add this in */
		int nelem , ion;

		/* add on charge transfer ionization of hydrogen,
		 * recombination for other element is ionization of hydrogen */
		ChargTran.HCharExcIonTotal = 0.;
		for( nelem=1; nelem<LIMELM; ++nelem)
		{
			double hold_one = 0.;
			/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
			 * for helium nelem=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			assert( (nelem+2-2) < (LIMELM+1) );
			for( ion=2; ion<=nelem+2; ++ion )
			{
				/* we intentionally skip CT with O+ since this would overwhelm every
				 * other process, but does not actually amount to much since is in
				 * balance with opposite CT reaction */
				if( nelem == ipOXYGEN && ion == 2 ) continue;

				/* charge transfer ionization of H, recombination for other species */
				hold_one = ChargTran.HCharExcRec[nelem][ion-2]*xIonFracs[nelem][ion];
				ChargTran.HCharExcIonTotal += hold_one;
			}
		}

		z[ipH1s][ipH1s] += ChargTran.HCharExcIonTotal;

		/* >>chng 01 may 07,  add this in */
		/* charge transfer recombination of hydrogen,
		 * which is ionization of the heavy element */
		ChargTran.HCharExcRecTotal = 0.;
		for( nelem=1; nelem<LIMELM; ++nelem)
		{
			/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
			 * for helium nelem=1, ion must go up to 3 */
			/* check that array bounds not exceeded */
			assert( (nelem+1-1) < (LIMELM+1) );
			for( ion=1; ion<=nelem+1; ++ion )
			{
				/* skip Oo => O+ */
				if( nelem == ipOXYGEN && ion == 1 ) continue;
				/* charge transfer ionization of H, recombination for other species */
				ChargTran.HCharExcRecTotal += 
					ChargTran.HCharExcIon[nelem][ion-1]*xIonFracs[nelem][ion];
			}
		}
		/*fprintf(ioQQQ," %.2e %.2e %.2e \n", z[iso.nLevels[ipHYDROGEN][ipZ]+1][ipH1s] , ChargTran.HCharExcRecTotal, 
			z[iso.nLevels[ipHYDROGEN][ipZ]+1][ipH2p]);*/
		z[iso.nLevels[ipHYDROGEN][ipZ]][ipH1s] += ChargTran.HCharExcRecTotal;

#		if 0
		fprintf(ioQQQ," h net ion %.2e %.2e %.2e\n",
			ChargTran.HCharExcIonTotal,ChargTran.HCharExcRecTotal,
			ChargTran.HCharExcIonTotal-ChargTran.HCharExcRecTotal);
#		endif

		totcap[ipH1s] = z[iso.nLevels[ipHYDROGEN][ipZ]][ipH1s]/phycon.eden;
		/* nb, hcaseb.HRecEffec should be updated too */

		{
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG )
			{
				long int maxnelem=-1, maxion=-1;
				double ctmax = 0.;
				for( nelem=1; nelem<LIMELM; ++nelem)
				{
					/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
					 * for helium nelem=1, ion must go up to 3 */
					for( ion=1; ion<=nelem+1; ++ion )
					{
						/* charge transfer ionization of H, recombination for other species */
						/*z[iso.nLevels[ipHYDROGEN][ipZ]+1][ipH1s] += ChargTran.HCharExcIon[nelem][ion-1]*xIonFracs[nelem][ion];*/
						if(	ChargTran.HCharExcIon[nelem][ion-1]*xIonFracs[nelem][ion] > ctmax )
						{
							ctmax = ChargTran.HCharExcIon[nelem][ion-1]*xIonFracs[nelem][ion];
							maxnelem = nelem;
							maxion = ion;
						}
					}
				}
				fprintf(ioQQQ," max H ct recom rate%.2e frac tot %.2e nelem %li ion %li\n", 
					ctmax,
					ctmax/MAX2(SMALLFLOAT,ChargTran.HCharExcRecTotal),
					maxnelem,
					maxion);
			}
		}
	}

	/* add in advection - these terms normally zero */
	z[iso.nLevels[ipHYDROGEN][ipZ]][ipH1s] += dynamics.Recomb[ipZ][ipZ];
	z[ipH1s][ipH1s] += dynamics.Photo;

	/* =================================================================== 
	 *
	 * at this point all matrix elements have been established 
	 *
	 * ==================================================================== */

	/* save total ground state destruction rate for possible later use */
	for( ipLo=ipH1s; ipLo<iso.nLevels[ipHYDROGEN][ipZ]; ++ipLo )
	{
		hydro.DestRate[ipZ][ipLo] = z[ipLo][ipLo];
	}

	if( (trace.lgTrace && trace.lgHBugFull) && (ipZ == trace.ipZTrace) )
	{
		if( ipZ == 0 )
		{
			/* remember ground state destruction rate */
			HRateDestGnd[0] = z[ipH1s][ipH1s];
			HRateDestGnd[1] = iso.gamnc[ipHYDROGEN][ipZ][ipH1s]/
			  z[ipH1s][ipH1s];
			HRateDestGnd[2] = iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*
			  phycon.EdenHCorr/z[ipH1s][ipH1s];
			HRateDestGnd[3] = Secondaries.csupra/z[ipH1s][ipH1s];
			HRateDestGnd[4] = Secondaries.Hx12[MIN2(ipZ,1)][ipH2p]*
			  9./z[ipH1s][ipH1s];

			fprintf(ioQQQ," grnd dest fracs:");
			for( j=ipH1s; j < 5; j++ )
			{
				fprintf( ioQQQ," ");
				fprintf( ioQQQ,PrintEfmt("%8.1e", HRateDestGnd[j] ));
			}
			fprintf(ioQQQ," hcoldc %e\n", phycon.EdenHCorr );
			fprintf( ioQQQ, "\n" );
		}

		fprintf( ioQQQ, "  pop level     others => (HydroLevelPop)\n" );
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			fprintf( ioQQQ, "       HII%2ld", n );
			for( j=ipH1s; j < (iso.nLevels[ipHYDROGEN][ipZ] + 1); j++ )
			{
				fprintf( ioQQQ," ");
				/*fprintf( ioQQQ,PrintEfmt("%8.1e", z[j][n] ) );*/
				fprintf( ioQQQ,"%.9e\t", z[j][n] );
			}
			fprintf( ioQQQ, "\n" );
		}
	}

	/* save matrix */
	for( j=ipH1s; j < iso.nLevels[ipHYDROGEN][ipZ]; j++ )
	{
		for( n=ipH1s; n < (iso.nLevels[ipHYDROGEN][ipZ] + 1); n++ )
		{
			SaveZ[n][j] = z[n][j];
		}
	}

	/* which matrix solver? */
	if( strcmp(TypMatrx.chMatrix,"matin1 ") == 0 )
	{
		/* this is the usual matrix inversion method, slightly faster */
		/*matin1();*/
		nerror1 = 0;
		if( nerror1 != 0 )
		{
			fprintf( ioQQQ, " hydrogen matrix error, stop.\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(1);
		}
	}

	/* this is the default */
	else if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
	{
		/*double amat[LMHLVL+2][LMHLVL+2];*/
		double *amat;
	  /* iso.nLevels[ipHYDROGEN][ipZ]+1 is right dimension of matrix */
#		ifdef AMAT
#			undef AMAT
#		endif
#		define AMAT(I_,J_)	(*(amat+(I_)*(iso.nLevels[ipHYDROGEN][ipZ])+(J_)))
		/* MALLOC space for the  1-d array */
		if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)((iso.nLevels[ipHYDROGEN][ipZ])*(iso.nLevels[ipHYDROGEN][ipZ])) ))) == NULL )
		{
			fprintf( ioQQQ, " HydroLevelPop MALLOC amat error\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(1);
		}

		/* this one may be more robust */
		for( j=ipH1s; j < iso.nLevels[ipHYDROGEN][ipZ]; j++ )
		{
			for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
			{
				/*amat[n][j] = z[n][j];*/
				AMAT(n,j) = z[n][j];
			}
			bvec[j] = z[iso.nLevels[ipHYDROGEN][ipZ]][j];
		}
		/*DGETRF(iso.nLevels[ipHYDROGEN][ipZ]+1,iso.nLevels[ipHYDROGEN]+1,
		  (double*)amat,LMHLVL+2,ipiv,&nerror1);
		DGETRS('N',iso.nLevels[ipHYDROGEN][ipZ]+1,1,(double*)amat,LMHLVL+2,ipiv,
		  bvec,LMHLVL+1,&nerror2);*/

		DGETRF(iso.nLevels[ipHYDROGEN][ipZ],iso.nLevels[ipHYDROGEN][ipZ],
		  amat,(iso.nLevels[ipHYDROGEN][ipZ]),ipiv,&nerror1);

		DGETRS('N',iso.nLevels[ipHYDROGEN][ipZ],1,amat,(iso.nLevels[ipHYDROGEN][ipZ]),ipiv,
		  bvec,(iso.nLevels[ipHYDROGEN][ipZ]),&nerror2);

		if( nerror1 != 0 || nerror2 != 0 )
		{
			fprintf( ioQQQ, " HydroLevelPop dgetrs error\n" );
			puts( "[Stop in HydroLevelPop]" );
			cdEXIT(1);
		}
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			z[iso.nLevels[ipHYDROGEN][ipZ]][n] = bvec[n];
		}
		free( amat);
	}
	else if( strcmp(TypMatrx.chMatrix,"veclib ") == 0 )
	{
		double /*amat[LMHLVL+2][LMHLVL+2], */ rcond;
		/* Jason found this one on the Exemplar, distributed source just stops */
		fprintf( ioQQQ, " this has not been checked since H atom conv\n" );
		for( j=ipH1s; j < iso.nLevels[ipHYDROGEN][ipZ]; j++ )
		{
			/*for( n=ipH1s; n <= iso.nLevels[ipHYDROGEN][ipZ]; n++ )
			{
				amat[n][j] = z[n][j];
			}*/
			bvec[j] = z[iso.nLevels[ipHYDROGEN][ipZ]][j];
		}
		job = 0;
		rcond = 0.;
		/*dgeco((double*)amat,iso.nLevels[ipHYDROGEN][ipZ]+2,iso.nLevels[ipHYDROGEN]+1,ipiv,rcond, work);
		dgesl((double*)amat,iso.nLevels[ipHYDROGEN][ipZ]+2,iso.nLevels[ipHYDROGEN]+1,ipiv,bvec, job);*/
		dgeco((double*)z,iso.nLevels[ipHYDROGEN][ipZ],iso.nLevels[ipHYDROGEN][ipZ],ipiv,rcond, work);
		dgesl((double*)z,iso.nLevels[ipHYDROGEN][ipZ],iso.nLevels[ipHYDROGEN][ipZ],ipiv,bvec, job);
		/* now put results back into z so rest of code treates only
		 * one case - as if matin1 had been used */
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			z[iso.nLevels[ipHYDROGEN][ipZ]][n] = bvec[n];
		}
		puts( "[Stop in HydroLevelPop]" );
		cdEXIT(1);
	}
	else
	{
		fprintf( ioQQQ, " chMatrix type insane in HydroLevelPop, was%7.7s\n", 
		  TypMatrx.chMatrix );
		puts( "[Stop in HydroLevelPop]" );
		cdEXIT(1);
	}
	/* end of branches for which matrix solution, now check if valid */

	/* check whether solution is valid */
	for( level=ipH1s; level < iso.nLevels[ipHYDROGEN][ipZ]; level++ )
	{
		double BigSoln= 0.;
		error[level] = 0.;
		for( n=ipH1s; n < iso.nLevels[ipHYDROGEN][ipZ]; n++ )
		{
			/* remember the largest value of the soln matrix to div by below */
			if( fabs(SaveZ[n][level] ) > BigSoln )
				BigSoln = fabs(SaveZ[n][level]);

			error[level] += SaveZ[n][level]*z[iso.nLevels[ipHYDROGEN][ipZ]][n];
		}

		if( BigSoln > 0. )
		{
			error[level] = (error[level] - SaveZ[iso.nLevels[ipHYDROGEN][ipZ]][level])/ BigSoln;
		}
		else
		{
			error[level] = 0.;
		}
	}

	/* convert from departure coef in Z(I) to level pop rel to HII */

	/* put departure coefficients and level populations into master array */
	for( ipLo=ipH1s; ipLo < iso.nLevels[ipHYDROGEN][ipZ]; ipLo++ )
	{
		iso.Pop2Ion[ipHYDROGEN][ipZ][ipLo] = (float)z[iso.nLevels[ipHYDROGEN][ipZ]][ipLo];
		if( iso.PopLTE[ipHYDROGEN][ipZ][ipLo] > 0. )
		{
			iso.DepartCoef[ipHYDROGEN][ipZ][ipLo] = 
				(iso.Pop2Ion[ipHYDROGEN][ipZ][ipLo]/
				(iso.PopLTE[ipHYDROGEN][ipZ][ipLo]* phycon.eden) );
		}
		else
		{
			iso.DepartCoef[ipHYDROGEN][ipZ][ipLo] = 0.;
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroLevelPop()\n", debug_fp );
#	endif
	return;
}

