/*HydroCollid evaluate collision rate for model hydrogen atom */
/*HydColDwn collision strength for collisional de-excitation for any levels of hydrogenic Z*/
/* the dec/compac ccc compiler becomes confused with pointer arith - disable warning */
#ifdef __DECC
#    pragma message disable (BADBOUNDCHK)
#endif
#include "cddefines.h"
#include "physconst.h"
#include "taulines.h"
#include "iso.h"
#include "hydrogenic.h"
#include "trace.h"
#include "collionrate.h"
#include "phycon.h"
#include "ionfracs.h"
#include "rfield.h"
#include "heavy.h"
#include "secondaries.h"
#include "opacity.h"
#include "converge.h"
#include "hydrooscilstr.h"
#include "hdexct.h"
#include "cfit.h"
#include "smeets.h"
#include "atomcwgt.h"

static float HCSAR_interp( int ipLo , int ipHi );
/*HydColDwn collision strength for collisional de-excitation for any levels of hydrogenic Z*/
static double HydColDwn(long int ipHi, 
  long int ipLo, 
  long int iz);

/* >>refer	H1	cs	Anderson, H., Ballance, C.P., Badnell, N.R., & Summers, H.P., 
 * >>refercon	2000, J Phys B, 33, 1255 */
#define NHCSTE 8
#define NHCS 6
static float HCSTE[NHCSTE] = {5802.,11604.,34812.,58020.,116040.,174060.,232080.,290100.};

static float HCS[NHCSTE*NHCS*(NHCS-1)] =
{0., 0., 0., 0., 0., 0., 0., 0., /* 1s - 1s */
2.600E-01f, 2.960E-01f, 3.250E-01f, 3.370E-01f, 3.560E-01f, 3.680E-01f, 3.750E-01f, 3.800E-01f,/* 1s-2s*/
4.270E-01f, 5.360E-01f, 8.570E-01f, 1.150E+00f, 1.750E+00f, 2.130E+00f, 2.350E+00f, 2.460E+00f,/* 1s-2p*/
2.372E-01f, 2.605E-01f, 3.413E-01f, 4.104E-01f, 5.063E-01f, 5.351E-01f, 5.335E-01f, 5.236E-01f,/* 1s-3*/
9.990E-02f, 1.144E-01f, 1.564E-01f, 1.857E-01f, 2.263E-01f, 2.422E-01f, 2.445E-01f, 2.428E-01f,/* 1s-4*/
7.325E-02f, 8.263E-02f, 9.641E-02f, 1.064E-01f, 1.202E-01f, 1.233E-01f, 1.219E-01f, 1.191E-01f,/* 1s-5*/
0., 0., 0., 0., 0., 0., 0., 0., /* 2s - 1s */
0., 0., 0., 0., 0., 0., 0., 0., /* 2s - 2s */
0., 0., 0., 0., 0., 0., 0., 0., /* 2s - 2p */
5.860E+00f, 7.640E+00f, 1.419E+01f, 2.015E+01f, 3.156E+01f, 3.969E+01f, 4.598E+01f, 5.114E+01f,/* 2s - 3 */
2.276E+00f, 2.747E+00f, 4.365E+00f, 5.609E+00f, 7.803E+00f, 9.267E+00f, 1.038E+01f, 1.126E+01f,/* 2s - 4 */
1.610E+00f, 1.897E+00f, 2.421E+00f, 2.799E+00f, 3.512E+00f, 4.023E+00f, 4.450E+00f, 4.790E+00f,/* 2s - 5 */
0., 0., 0., 0., 0., 0., 0., 0., /* 2p - 1s */
0., 0., 0., 0., 0., 0., 0., 0., /* 2p - 2s */
0., 0., 0., 0., 0., 0., 0., 0., /* 2p - 2p */
2.228E+01f, 2.818E+01f, 5.026E+01f, 7.193E+01f, 1.178E+02f, 1.542E+02f, 1.828E+02f, 2.078E+02f,/* 2p - 3 */
8.772E+00f, 1.042E+01f, 1.600E+01f, 2.058E+01f, 2.898E+01f, 3.482E+01f, 3.923E+01f, 4.280E+01f,/* 2p - 4 */
6.161E+00f, 7.209E+00f, 9.091E+00f, 1.049E+01f, 1.308E+01f, 1.488E+01f, 1.626E+01f, 1.736E+01f,/* 2p - 5 */
0., 0., 0., 0., 0., 0., 0., 0., /* 3 - 1s */
0., 0., 0., 0., 0., 0., 0., 0., /* 3 - 2s */
0., 0., 0., 0., 0., 0., 0., 0., /* 3 - 2p */
0., 0., 0., 0., 0., 0., 0., 0., /* 3 - 3 */
8.648E+01f, 1.366E+02f, 3.252E+02f, 4.834E+02f, 7.715E+02f, 9.692E+02f, 1.123E+03f, 1.247E+03f,/* 3 - 4 */
6.229E+01f, 8.263E+01f, 1.255E+02f, 1.527E+02f, 1.951E+02f, 2.211E+02f, 2.400E+02f, 2.544E+02f,/* 3 - 5 */
0., 0., 0., 0., 0., 0., 0., 0., /* 4 - 1s */
0., 0., 0., 0., 0., 0., 0., 0., /* 4 - 2s */
0., 0., 0., 0., 0., 0., 0., 0., /* 4 - 2p */
0., 0., 0., 0., 0., 0., 0., 0., /* 4 - 3 */
0., 0., 0., 0., 0., 0., 0., 0., /* 4 - 4 */
4.625E+02f, 8.070E+02f, 1.707E+03f, 2.269E+03f, 3.122E+03f, 3.675E+03f, 4.083E+03f, 4.414E+03f /* 4 - 5 */
};

void HydroCollid(long int ipZ)
{
	long int i, 
	  ipHi, 
	  ipLo;
	double CStemp, 
	  factor ,
	  HCol2s2pp, 
	  ConvLTEPOP;
	/*lint -e785 too few initializers for aggregate, but it defaults to all zeros */
	static float told[LIMELM] = {0.};
	/*lint +e785 */

#	ifdef DEBUG_FUN
	fputs( "<+>HydroCollid()\n", debug_fp );
#	endif

	/* check that we were called with valid charge */
	assert( ipZ >= 0);
	assert( ipZ < LIMELM );

	/* evaluate collision rate for model hydrogen atom */
	/* ipZ is atomic number minus 1, so H is 0*/

	{
		/*@-redef@*/
		enum {AGN=FALSE};
		/*@+redef@*/
		if( AGN  )
		{
			float tear[3]={10000.f,15000.f,20000.f};
			for( i=0;i<3;++i)
			{
				phycon.te = tear[i];
				for( ipHi=1;ipHi<4;++ipHi )
				{
					for( ipLo=0; ipLo<ipHi; ++ipLo )
					{
						if( !(ipLo==1 && ipHi==2 ) )
						{
							fprintf(ioQQQ,"%.3f\t", HCSAR_interp(ipLo,ipHi) ) ;
						}
					}
				}
				fprintf(ioQQQ," \n");
			}
			exit(0);
		}
	}

	/***********************************************************************
	 *                                                                     *
	 * get secondary excitation by suprathermal electrons                  *
	 *                                                                     *
	 ***********************************************************************/

	if( ipZ == 0 )
	{
		/* following are for hydrogen ATOMS, ions are in next else */
		/* secondary excitation of Lyman lines */
		Secondaries.Hx12[0][ipH2p] = (float)(Secondaries.x12tot*2./3./1.261);
		Secondaries.Hx12[0][ipH2s] = (float)(Secondaries.x12tot*1./3./1.261);
		Secondaries.Hx12[0][3] = (float)(Secondaries.x12tot*0.165/1.261);
		Secondaries.Hx12[0][4] = (float)(Secondaries.x12tot*0.059/1.261);

		/* rest are just a wild guess */
		for( i=5; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			Secondaries.Hx12[0][i] = (float)(Secondaries.Hx12[0][i-1]/2.);
		}
	}
	else if( ipZ==1 )
	{
		/* following for ions, guess from Janev et al. 
		 * only evaluated for He, and there will be a problem if
		 * hydrogenic helium not present, but hydrogenic species
		 * of heavier elements are present.  I can see how this
		 * would ever happen */
		Secondaries.Hx12[1][ipH2p] = (float)(Secondaries.x12tot/3.*2./3.);
		Secondaries.Hx12[1][ipH2s] = (float)(Secondaries.x12tot/3./3.);
		/* this is just a wild guess */
		for( i=3; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			Secondaries.Hx12[1][i] = (float)(Secondaries.Hx12[1][i-1]/2.);
		}
	}

	/* skip most of this sub if temperature has not changed by much,
	 * the check on conv.nTotalIoniz is to make sure that we redo this
	 * on the very first call in a grid calc - it is 0 on the first call */
	/* loc1 bug 99 apr 08, tried commenting out first block so always reeval, no joy */
	if( fabs(phycon.te-told[ipZ])/phycon.te < 0.00002 && 
		EmisLines[ipHYDROGEN][ipZ][ipH2p][ipH1s].ColUL > 0. && conv.nTotalIoniz )
	{
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, 
				"       HydroCollid called ipZ %li - no reval Boltz fac, LTE dens\n",
				ipZ );
		}
		/* (but still eval col below, search for 
		 * following code executed even when temperature did not change" ) */
	}
	else
	{
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, 
				"       HydroCollid called ipZ %li - will reeval Boltz fac, LTE dens\n",
				ipZ );
		}
		told[ipZ] = phycon.te;

		/***********************************************************************
		 *                                                                     *
		 * Boltzmann factors for all levels, ionization and                    *
		 * exciation between levels                                            *
		 *                                                                     *
		 ***********************************************************************/

		/* define boltz facs between adjacent levels, will go from ipHi to one below */
		for( ipHi=ipH2s; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
		{
			/* HdeltaTe is array of line temps, upper to lower
			 * their boltzmann factors for this temp,
			 * the array addressing for hlbolt is correct, and backwards from
			 * other similar variables - [up][lo] */
			iso.Boltzmann[ipHYDROGEN][ipZ][ipHi][ipHi-1] = 
				exp(-(double)EmisLines[ipHYDROGEN][ipZ][ipHi][ipHi-1].EnergyK/(double)phycon.te);
		}

		/* the highest level, up to the continuum */
		/* >>chng 01 sep 23, got rid of HCionTe */
		iso.ConBoltz[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1] =
			exp(-iso.xIsoLevNIonRyd[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1]/(double)phycon.te_ryd);

		/* define remaining line Boltzmann factors in terms of existing factors between levels */
		for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ] - 2); ipLo++ )
		{
			for( ipHi=ipLo + 2; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
			{
				iso.Boltzmann[ipHYDROGEN][ipZ][ipHi][ipLo] = 
					iso.Boltzmann[ipHYDROGEN][ipZ][ipHi-1][ipLo]*
				  iso.Boltzmann[ipHYDROGEN][ipZ][ipHi][ipHi-1];
			}
		}

		for( i=ipH1s; i < (iso.nLevels[ipHYDROGEN][ipZ] - 1); i++ )
		{
			iso.ConBoltz[ipHYDROGEN][ipZ][i] = 
				iso.Boltzmann[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1][i]*
			  iso.ConBoltz[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1];
			/* this is to prevent close-to-overflow problems in model atom */
			if( iso.ConBoltz[ipHYDROGEN][ipZ][i] < SMALLDOUBLE )
			{
				iso.ConBoltz[ipHYDROGEN][ipZ][i] = 0.;
			}
		}

		/***********************************************************************
		 *                                                                     *
		 * LTE abundances for all levels, ionization and                       *
		 * exciation between levels                                            *
		 *                                                                     *
		 ***********************************************************************/

		/* following factor is actually 4.1412957e-16 (old=4.14158E-16), 
		 * but e- stat weight is in */
		/* ConvLTEPOP = 2.0708e-16/phycon.te32;*/
		/* >>chng 99 jun 02, use codata and infinite mass nuc */
		factor = HION_LTE_POP*AtomcWgt.AtomicWeight[ipZ]/
			(AtomcWgt.AtomicWeight[ipZ]+ELECTRON_MASS/ATOMIC_MASS_UNIT) ;

		ConvLTEPOP = pow(factor,1.5)/2./phycon.te32;

		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			/* >>>chng 99 mar 16, had gone to gt 0 previous week, but problems
			 * with very very small hcbolt, change back to 1e-100 */
			if( iso.ConBoltz[ipHYDROGEN][ipZ][i] > 1e-100 )
			{
				iso.PopLTE[ipHYDROGEN][ipZ][i] = 
					iso.stat[ipHYDROGEN][i] / iso.ConBoltz[ipHYDROGEN][ipZ][i]* ConvLTEPOP;
			}
			else
			{
				iso.PopLTE[ipHYDROGEN][ipZ][i] = 0.;
			}
		}

		/* now check for any zeros - if present then matrix cannot be used */
		iso.lgPopLTE_OK[ipHYDROGEN][ipZ] = TRUE;
		for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
		{
			if( iso.PopLTE[ipHYDROGEN][ipZ][i] <= 0. )
			{
				iso.lgPopLTE_OK[ipHYDROGEN][ipZ] = FALSE;
				break;
			}
		}

		/***********************************************************************
		 *                                                                     *
		 * collisional ionization, and three body recombination                *
		 * Vriens and Smeets collisional ionization data set                   *
		 * use for hydrogen itselt                                             *
		 *                                                                     *
		 ***********************************************************************/
		if( ipZ == 0 )
		{
			/* atomic hydrogen
			 * this is total to 2s and 2p, will be split below
			 * HColOn is set to 0 with hydrogen collisions off command, usually 1 */
			iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p] = smeets(2)*iso.lgColl_ionize[ipHYDROGEN];
			/* now do rest of levels, up to second highest */
			for( i=3; i < iso.nLevels[ipHYDROGEN][ipZ]-1; i++ )
			{
				/* HColOn is option to turn off collisions, "hydrogen collis off" comnd */
				iso.ColIoniz[ipHYDROGEN][ipZ][i] = smeets(i)*iso.lgColl_ionize[ipHYDROGEN];
			}
			/* always leave highest level coupled to continuum - HColIonOn not there*/
			iso.ColIoniz[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1] = smeets(iso.nLevels[ipHYDROGEN][ipZ]-1);
		}
		else
		{
			/* hydrogenic ions */
			/* this is total to 2s and 2p */
			iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p] = HydColIon(2,ipZ)*iso.lgColl_ionize[ipHYDROGEN];
			for( i=3; i < iso.nLevels[ipHYDROGEN][ipZ]-1; i++ )
			{
				/* HColOn is option to turn off collisions, "hydrogen collis off" comnd */
				iso.ColIoniz[ipHYDROGEN][ipZ][i] = HydColIon(i,ipZ)*iso.lgColl_ionize[ipHYDROGEN];
			}
			/* always leave highest level coupled to continuum - HColIonOn not there*/
			iso.ColIoniz[ipHYDROGEN][ipZ][iso.nLevels[ipHYDROGEN][ipZ]-1] = HydColIon(iso.nLevels[ipHYDROGEN][ipZ]-1,ipZ);
		}

		/* total 2s and 2p was set in previous branch, now split into 2s and 2p */
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p] *= 0.75;
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH2s] = iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p]/3.;

		/* collisional ionization from ground */
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s] = cfit(ipZ+1,1,phycon.te);

		/* option to kill collisional ionization with Hydro collis off */
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s] *= iso.lgColl_ionize[ipHYDROGEN];
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH2s] *= iso.lgColl_ionize[ipHYDROGEN];
		iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p] *= iso.lgColl_ionize[ipHYDROGEN];

		/* save into main array */
		CollIonRate.CollidRate[0][ipZ][ipZ] = 
			(float)(iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*phycon.eden) + Secondaries.csupra;

		/* cooling due to collisional ionization, which only includes thermal */
		CollIonRate.CollidRate[1][ipZ][ipZ] = (float)(iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s]*(float)phycon.eden*
			rfield.anu[Heavy.ipHeavy[ipZ][ipZ]-1]* EN1RYD);

		/* hummer and storey option for case b kills coll ioniz from 1 and 2 */
		if( opac.lgCaseB_HummerStorey )
		{
			iso.ColIoniz[ipHYDROGEN][ipZ][ipH1s] = 0.;
			iso.ColIoniz[ipHYDROGEN][ipZ][ipH2p] = 0.;
			iso.ColIoniz[ipHYDROGEN][ipZ][ipH2s] = 0.;
		}

		/***********************************************************************
		 * 
		 * collisional strengths for all hydrogen lines
		 * 
		 ************************************************************************/

		/* evaluate collisional deexcitation rates,
		 * Hydcs123 evaluates downward collision strength, which is stored in
		 * hydrogen lines vector
		 * Hydcs123, upon call 1=1s, 2=2s, 3=2p, 4=3 */
		for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ] - 1); ipLo++ )
		{
			for( ipHi=ipLo + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
			{
				if( ipHi == 1 )
				{
					/* 2s - 1s */
					if( ipZ==0 )
					{
						CStemp = HCSAR_interp(ipLo,ipHi);
					}
					else
					{
						CStemp = Hydcs123(1,2,ipZ,'e');
					}
				}
				else if( ipHi == 2 )
				{
					if( ipLo == 0 )
					{
						/* 2p - 1s */
						if( ipZ==0 )
						{
							CStemp = HCSAR_interp(ipLo,ipHi);
						}
						else
						{
							CStemp = Hydcs123(1,3,ipZ,'e');
						}
					}
					/* this is 2s - 2p */
					else if( ipLo == 1 )
					{
						/* option to kill 2s2p collisions */
						if( iso.lgColl_l_mixing[ipHYDROGEN] )
						{
							/* 2s - 2p for electrons ONLY */
							CStemp = Hydcs123(2,3,ipZ,'e');
							/* 2s - 2p for protons ONLY */
							HCol2s2pp = Hydcs123(2,3,ipZ,'p');
							/* trick, since will later be multiplied by electron,
							 * not proton density */
							/*CStemp += HCol2s2pp*xIonFracs[ipHYDROGEN][2]/phycon.eden;*/
							/* NB - all collision rates will be multiplied by this number */
							CStemp += HCol2s2pp*xIonFracs[ipHYDROGEN][2]/phycon.EdenHCorr;
						}
						else
						{
							CStemp = 0.;
						}
					}
					else
					{
						/* this is not possible */
						CStemp = -DBL_MAX;
					}
				}

				else if( ipHi == 3 )
				{
					/* 2p - 1s */
					if( ipZ==0 )
					{
						CStemp = HCSAR_interp(ipLo,ipHi);
					}
					else
					{
						if( ipLo == 0 )
						{
							/* 3 - 1 */
							CStemp = Hydcs123(1,4,ipZ,'e');
						}
						else if( ipLo == 1 )
						{
							/* 3 - 2s */
							CStemp = Hydcs123(2,4,ipZ,'e');
						}
						else if( ipLo == 2 )
						{
							/* 3 - 2p */
							CStemp = Hydcs123(3,4,ipZ,'e');
						}
						else
						{
							/* this is not possible */
							CStemp = -DBL_MAX;
						}
					}
				}
				else if( ipZ==0 && ipHi <= 5 )
				{
					fixit();/* is it an oversight that this is only used for 4 and 5? */
					CStemp = HCSAR_interp(ipLo,ipHi);
				}
				else
				{
					/* highly excited levels */
					CStemp = HydColDwn(ipHi,ipLo,ipZ+1);
				}
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].cs = (float)CStemp;
			}
		}

		/* save 2s-2p collisions since we always want to include, even when other
		 * collisions are turned off, since 2s-2p act as one level.  there is a
		 * separate no 2s2p command, which is implemented below */
		CStemp = EmisLines[ipHYDROGEN][ipZ][ipH2p][ipH2s].cs;

		/* kill all collisional excitation if this set with 
		 * atom h-like collisional excitation */
		if( iso.lgColl_excite[ipHYDROGEN] == 0 )
		{
			for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ]-1); ipLo++ )
			{
				for( ipHi=ipLo + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
				{
					EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].cs = 0.;
				}
			}
		}

		/* case b hummer&storey does not have any collisions from n=1 or 2 
		 * just kill those collisions here */
		if( opac.lgCaseB_HummerStorey )
		{
			for( ipLo=ipH1s; ipLo <= ipH2p; ipLo++ )
			{
				for( ipHi=3; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
				{
					EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].cs = 0.;
				}
			}
		}

		/* reset the 2s-2p collsions unless turned off with atom h-like collisions l-mixing
		 * command, since we usually want to include these even when line excit
		 * is turned off */
		if( iso.lgColl_l_mixing[ipHYDROGEN] )
		{
			EmisLines[ipHYDROGEN][ipZ][ipH2p][ipH2s].cs = (float)CStemp;
		}

		/***********************************************************************
		 *                                                                     *
		 * collisional deexcitation for all lines                              *
		 *                                                                     *
		 ***********************************************************************/

		for( ipLo=ipH1s; ipLo < (iso.nLevels[ipHYDROGEN][ipZ] - 1); ipLo++ )
		{
			for( ipHi=ipLo + 1; ipHi < iso.nLevels[ipHYDROGEN][ipZ]; ipHi++ )
			{
				/* collisional deexcitation rate */
				EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].ColUL = 
					(float)(EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].cs/
				  phycon.sqrte*8.629e-6/EmisLines[ipHYDROGEN][ipZ][ipHi][ipLo].gHi );
				/*This is downward collision rate */
			}
		}

		if( (trace.lgTrace && trace.lgHBugFull) && (ipZ == trace.ipZTrace) )
		{
			fprintf( ioQQQ, "       HydroCollid rate n->n-1:" );
			for( i=ipH1s; i < (iso.nLevels[ipHYDROGEN][ipZ] - 1); i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i+1][i].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid col ion cof:" );
			for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e",  iso.ColIoniz[ipHYDROGEN][ipZ][i] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dn2 1s:" );
			for( i=ipH2s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][ipH1s].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dn2 2s:" );
			for( i=ipH2p; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][ipH2s].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dn2 2p:" );
			for( i=3; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][ipH2p].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dwn2 3:" );
			for( i=4; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][3].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dwn2 4:" );
			for( i=5; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][4].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dwn2 5:" );
			for( i=6; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][5].ColUL) );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dwn2 6:" );
			for( i=7; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][6].ColUL ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid rate dwn2 7:" );
			for( i=8; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", EmisLines[ipHYDROGEN][ipZ][i][7].ColUL) );
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid boltz fac c:" );
			for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", iso.ConBoltz[ipHYDROGEN][ipZ][i] ));
			}
			fprintf( ioQQQ, "\n" );

			fprintf( ioQQQ, "       HydroCollid HLTE(i,Z):  " );
			for( i=ipH1s; i < iso.nLevels[ipHYDROGEN][ipZ]; i++ )
			{
				fprintf( ioQQQ,PrintEfmt("%10.3e", iso.PopLTE[ipHYDROGEN][ipZ][i] ));
			}
			fprintf( ioQQQ, "\n" );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HydroCollid()\n", debug_fp );
#	endif
	return;
}


#define HCSAR(ilo , ihi , nte ) (*(HCS+(nte)+(ilo)*48+(ihi)*8))

static float HCSAR_interp( int ipLo , int ipHi )
{

	static int ip=1;
	float cs;

#	ifdef DEBUG_FUN
	fputs( "<+>HCSAR_interp()\n", debug_fp );
#	endif

	if( ipLo==1 && ipHi==2 )
	{
		fprintf(ioQQQ,"HCSAR_interp was called for the 2s-2p transition, which it cannot do\n");
		exit(EXIT_FAILURE);
	}
	if( phycon.te <= HCSTE[0] )
	{
		cs = HCSAR( ipLo , ipHi , 0 );
	}
	else if( phycon.te >= HCSTE[NHCSTE-1] )
	{
		cs = HCSAR( ipLo , ipHi , NHCSTE-1 );
	}
	else
	{
		/* the ip index is most likely correct since it points to the last temperature */
		if( !(HCSTE[ip-1] < phycon.te ) && ( phycon.te <= HCSTE[ip]) )
		{
			/* we must find the temperature in the array */
			for( ip=1; ip<NHCSTE ; ++ip )
			{
				if( (HCSTE[ip-1] < phycon.te ) && ( phycon.te <= HCSTE[ip]) )
					break;
			}
		}
		/* we now have the index */
		cs = HCSAR( ipLo , ipHi , ip-1 ) + 
			(HCSAR( ipLo , ipHi , ip ) - HCSAR( ipLo , ipHi , ip-1 ) ) / (HCSTE[ip]-HCSTE[ip-1] ) *
			(phycon.te - HCSTE[ip-1] );
		if( cs <= 0.)
		{
			fprintf(ioQQQ," insane cs returned by HCSAR_interp, values are\n");
			fprintf(ioQQQ,"%.3f %.3f \n", HCSAR( ipLo , ipHi , ip-1 ),HCSAR( ipLo , ipHi , ip ) );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HCSAR_interp()\n", debug_fp );
#	endif
	return(cs) ;
}


/*HydColDwn collision strength for collisional de-excitation for any levels of hydrogenic Z*/
static double HydColDwn(long int ipHi, 
  long int ipLo, 
  long int iz)
{
	long int i, 
	  nHi, 
	  nLo, 
	  _r;
	double A, 
	  An, 
	  D, 
	  H, 
	  HydColDwn_v, 
	  R, 
	  c, 
	  e1, 
	  exp_int_2, 
	  gHi, 
	  gLo, 
	  rate, 
	  ryd, 
	  sexpy ,
	  tev, 
	  xn, 
	  xnp2, 
	  xp, 
	  y;
	static double arrAn[10], 
	  arrH[4], 
	  arrR[10];
	static int _aini = 1;
	if( _aini ){ /* Do 1 TIME INITIALIZATIONS! */
		{ static double _itmp0[] = {1.48,3.64,5.93,8.32};
		for( i=1, _r = 0; i <= 4; i++ )
		{
			arrH[i-1] = _itmp0[_r++];
			}
		}
		{ static double _itmp1[] = {1.30,0.59,0.38,0.286,0.229,0.192,
		  0.164,0.141,0.121,0.105};
		for( i=1, _r = 0; i <= 10; i++ )
		{
			arrAn[i-1] = _itmp1[_r++];
			}
		}
		{ static double _itmp2[] = {1.83,1.60,1.53,1.495,1.475,1.46,
		  1.45,1.45,1.46,1.47};
		for( i=1, _r = 0; i <= 10; i++ )
		{
			arrR[i-1] = _itmp2[_r++];
			}
		}
		_aini = 0;
	}

#	ifdef DEBUG_FUN
	fputs( "<+>HydColDwn()\n", debug_fp );
#	endif
	/*rate coefficient for collisional de-excitation for any levels of hydrogenic Z
	 *written by Jason Ferguson to compute the 
	 * >>refer Sampson and Zhang 1988, ApJ, 335, 516
	 * rate coeff. for collisional excitation between any two levels of 
	 * hydrogenic atoms charge Z.  Valid for all ipLo, ipHi.
	 * */

	/*begin sanity check */
	if( (iz < 1 || ipLo >= ipHi) || ipLo*ipHi < 0 )
	{
		fprintf( ioQQQ, "HydColDwn called with impossible parameters.\n" );
		fprintf( ioQQQ, "%5ld%5ld%5ld\n", ipHi, ipLo, iz );
		puts( "[Stop in hydrocoldwn]" );
		cdEXIT(1);
	}
	/*end sanity check
	 *
	 * we need to check to see what the gs and quantums are
	 * ipHi will always be 4 and greater. */
	nHi = ipHi;
	gHi = 2.*nHi*nHi;
	if( ipLo == 0 )
	{
		nLo = 1;
		gLo = 2.;
	}
	else if( ipLo == 1 )
	{
		nLo = 2;
		gLo = 2.;
	}
	else if( ipLo == 2 )
	{
		nLo = 2;
		gLo = 6.;
	}
	else
	{
		nLo = ipLo;
		gLo = 2.*nLo*nLo;
	}

	/* call hydrogen if iz = 1 */
	if( iz == 1 )
	{
		HydColDwn_v = hdexct(nHi,nLo)*gLo;
		
#		ifdef DEBUG_FUN
		fputs( " <->HydColDwn()\n", debug_fp );
#		endif
		return( HydColDwn_v );
	}

	if( nLo > 4 )
	{
		H = 2.15*nLo;
	}
	else
	{
		H = arrH[nLo-1];
	}
	if( nLo > 10 )
	{
		An = 1./(float)(nLo);
		R = 1.50;
	}
	else
	{
		An = arrAn[nLo-1];
		R = arrR[nLo-1];
	}

	xp = (double)(nHi);
	xn = (double)(nLo);
	xnp2 = POW2(xn/xp);
	tev = 8.61716e-5*phycon.te;
	ryd = 13.60583;

	y = POW2((double)iz)*ryd*(1./POW2(xn) - 1./POW2(xp))/tev;
	/*>>>chng 98 dec 17, redefine ee1 so always first exponential integral*/
	/*if( y < 1. )*/
	/*{*/
		/*e1 = ee1(y);*/
		/*exp_int_2 = sexp(y) - y*e1;*/
	/*}*/
	/*else*/
	/*{*/
		/*e1 = ee1(y);*/
		/*exp_int_2 = 1 - y*e1;*/
	/*}*/
	sexpy = sexp(y);
	/* >>>chng 99 apr 15, as per Peter van Hoof comment, e1 underflowed due to 
	 * exp, then div by 0.  just set downward rate to zero when exp is this large */
	if( sexpy <= 0. )
	{
		HydColDwn_v = 0.;
	}
	else
	{
		e1 = ee1(y);
		exp_int_2 = sexpy - y*e1;

		A = 4.*powi(xn,4)/(1 - xnp2)*HydroOscilStr(xn,xp);
		D = A*H*(pow(1 - xnp2,R) - An*xnp2);
		c = 1.12*xn*A*(1 - xnp2);
		if( xp - xn == 1. )
		{
			c *= sexp(0.006*powi(xn-1.,6)/iz);
		}

		rate = 8.63e-6/(POW2(xn*iz)*phycon.sqrte);
		rate *= (A + y*c)*e1 + D*exp_int_2;

		/* Convert from excitation to de-excitation */
		/* >>>chng 99 apr 09, had not changed following, so low temp
		 * had too small a rate, and there was a discontinuity for he ii
		 * at T=30700 */
		HydColDwn_v = rate*gLo/gHi/sexpy;
	}

	HydColDwn_v = MAX2(HydColDwn_v,1.e-38);

	/* convert deexcitation rate to collision strength */
	HydColDwn_v = HydColDwn_v*gHi*phycon.sqrte/8.629e-6;

#	ifdef DEBUG_FUN
	fputs( " <->HydColDwn()\n", debug_fp );
#	endif
	return( HydColDwn_v );
}


