#include "cddefines.h"  
#include "hsrate.h"  

double HSRate( 
	  /* upper and lower quantum numbers*/
	  long int iHi, 
	  long int iLo, 
	  /* charge, 1 or 2 for now */
	  long int ipZ, 
	  /* temperature to interpolate, for H between 500-30,000K*/
	  double TempIn, 
	  /* density to interpolate */
	  double DenIn,
	  /* case - 'a' or 'b' */
	  char chCase )

/* general utility to interpolate line emissivities from the Storey & Hummer tables
 of case B emissivities.  
 iHi, iLo, two long integers are the principal quantum
 numbers, and are upper and lower levels in any order.  
 ipZ nuclear charge, 1 or 2 have data now
 TempIn = temperature, and must lie within the range of the table, which depends on
 the ion charge, and is 500 - 30,000K for hydrogen.  
 DenIn is the density and must not exceed the high density limit to the table.  

 routine returns -1 if conditions outside temperature range, or
 above density range of tabulated case b results
 If desired density is low limit, lower limit is used instead 
*/

{
	long int 
		ipTemp, /*pointer to temperature*/
		ipDens, /*pointer to density*/
		ipDensHi ,
		ipTempHi ;
	int ipUp , ipLo , ip ;
	double x1 , x2 , yy1 , yy2 , z1 , z2 , za , zb ,z;
	int iCase;

	/*make sure ipZ is 1 or 2*/
	if( ipZ<1 || ipZ> HS_NZ )
	{
		printf("HSRate called with improper ipZ, was%li and must be 1 or 2",ipZ);
		cdEXIT(1);

	}
	/* now back ipZ back one, to be on c scale */
	--ipZ;

	/* case A or B? */
	if( chCase == 'a' || chCase=='A' )
	{
		iCase = 0;
	}
	else if(  chCase == 'b' || chCase=='B' )
	{
		iCase = 1;
	}
	else
	{
		iCase = FALSE;
		printf("HSRate called with improper case, was %c and must be A or B",chCase);
		cdEXIT(1);
	}

	/*===========================================================*/
	/* following is option to have principal quantum number given in either order, 
	 * final result is that ipUp and ipLo will be the upper and lower levels */
	if( iHi > iLo )
	{
		ipUp = (int)iHi;  ipLo = (int)iLo ;
	}
	else if( iHi < iLo )
	{
		ipUp = (int)iLo ; ipLo = (int)iHi ;
	}
	else
	{ 
		printf("HSRate called with indices equal, %ld  %ld  \n",iHi,iLo);
		cdEXIT(1);
	}

	/* now check that they are in range of the predicted levels of their model atom*/
	if( ipLo <1 ) 
	{
		printf(" HSRate called with lower quantum number less than 1, = %i\n",
			ipLo);
		cdEXIT(1);
	}

	if( ipUp >25 ) 
	{
		printf(" HSRate called with upper quantum number greater than 25, = %i\n",
			ipUp);
		cdEXIT(1);
	}

	/*===========================================================*/
	/*bail if above high density limit */
	if( DenIn > CaseBHS.Density[iCase][ipZ][CaseBHS.nDensity[iCase][ipZ]-1] ) 
	{
		/* this is flag saying bogus results */
		return -1;
	}

	if( DenIn <= CaseBHS.Density[iCase][ipZ][0] )
	{
		/* this case, desired density is below lower limit to table,
		 * just use the lower limit */
		ipDens = 0;
	}
	else
	{
		/* this case find where within table density lies */
		for( ipDens=0; ipDens < CaseBHS.nDensity[iCase][ipZ]-1; ipDens++ )
		{
			if( DenIn >= CaseBHS.Density[iCase][ipZ][ipDens] && 
				DenIn < CaseBHS.Density[iCase][ipZ][ipDens+1] ) break ;
		}
	}


	/*===========================================================*/
	/* confirm within temperature range*/
	if( TempIn < CaseBHS.ElecTemp[iCase][ipZ][0] || 
		TempIn > CaseBHS.ElecTemp[iCase][ipZ][CaseBHS.ntemp[iCase][ipZ]-1] ) 
	{
		/* this is flag saying bogus results */
		return -1;
	}

	/* find where within grid this temperature lies */ 
	for( ipTemp=0; ipTemp < CaseBHS.ntemp[iCase][ipZ]-1; ipTemp++ )
	{
		if( TempIn >= CaseBHS.ElecTemp[iCase][ipZ][ipTemp] && 
			TempIn < CaseBHS.ElecTemp[iCase][ipZ][ipTemp+1] ) break ;
	}

	/*===========================================================*/
	/*we now have the array indices within the temperature array*/

	if( ipDens+1 > CaseBHS.nDensity[iCase][ipZ]-1 )
	{ 
		/* special case, when cell is highest density point */
		ipDensHi = CaseBHS.nDensity[iCase][ipZ]-1;
	}
	else if( DenIn < CaseBHS.Density[iCase][ipZ][0])
	{
		 /* or density below lower limit to table, set both bounds to 0 */
		ipDensHi = 0;
	}
	else
	{ 
		ipDensHi = ipDens+1;
	}

	/*special case, if cell is highest temperature point*/
	if( ipTemp+1 > CaseBHS.ntemp[iCase][ipZ]-1 )
	{ 
		ipTempHi = CaseBHS.ntemp[iCase][ipZ]-1;
	}
	else
	{ 
		ipTempHi = ipTemp+1;
	}

	x1 = log10( CaseBHS.Density[iCase][ipZ][ipDens] );
	x2 = log10( CaseBHS.Density[iCase][ipZ][ipDensHi] );

	yy1 = log10( CaseBHS.ElecTemp[iCase][ipZ][ipTemp] );
	yy2 = log10( CaseBHS.ElecTemp[iCase][ipZ][ipTempHi] );

	/*now generate the pointer to the array, expression from storey code -1 for c*/
	ip = (int)((((CaseBHS.ncut[iCase][ipZ]-ipUp)*(CaseBHS.ncut[iCase][ipZ]+ipUp-1))/2)+ipLo - 1);

	/*pointer must lie within line array*/
	assert( ip<NLINEHS ); 
	assert( ip >= 0 );

	/* interpolate on emission rate*/
	z1 = log10( CaseBHS.Emiss[iCase][ipZ][ipTemp][ipDens][ip]) ;
	z2 = log10( CaseBHS.Emiss[iCase][ipZ][ipTemp][ipDensHi][ip]) ;

	if(x2 == x1 ) 
	{
		za = z2;
	}
	else 
	{
		za = z1 + log10( DenIn / CaseBHS.Density[iCase][ipZ][ipDens] ) * (z2-z1)/(x2-x1);
	}

	z1 = log10( CaseBHS.Emiss[iCase][ipZ][ipTempHi][ipDens][ip]) ;
	z2 = log10( CaseBHS.Emiss[iCase][ipZ][ipTempHi][ipDensHi][ip]) ;

	if(x2 == x1 ) 
	{
		zb = z2;
	}
	else 
	{
		zb = z1 + log10( DenIn / CaseBHS.Density[iCase][ipZ][ipDens] ) * (z2-z1)/(x2-x1);
	}

	if( yy2 == yy1 ) 
	{
		z = zb ;
	}
	else 
	{
		z = za + log10( TempIn / CaseBHS.ElecTemp[iCase][ipZ][ipTemp] ) * (zb-za)/(yy2-yy1);
	}

	return ( pow(10.,z) );
}

