/*highen do high energy radiation field - gas interaction, Compton scattering, etc */
#include "cddefines.h"
#include "physconst.h"
#include "trace.h"
#include "recoil.h"
#include "radius.h"
#include "hextra.h"
#include "ionfracs.h"
#include "abundances.h"
#include "hmi.h"
#include "neutrn.h"
#include "doppvel.h"
#include "photrate.h"
#include "rfield.h"
#include "heat.h"
#include "phycon.h"
#include "opacity.h"
#include "crsnut.h"
#include "gammas.h"
#include "highen.h"

void highen( void )
{
	long int i,
		nelem;

	double CosRayDen, 
	  CosRayIonRate, 
	  CosRayHeatRate,
	  crsphi, 
	  energy, 
	  heat_cold_electrons, 
	  heatin, 
	  sqthot;

	double hin;

#	ifdef DEBUG_FUN
	fputs( "<+>highen()\n", debug_fp );
#	endif


	/**********************************************************************
	 *
	 *						COMPTON RECOIL IONIZATION
	 *
	 * bound electron scattering of >2.3 kev photons if neutral
	 * comoff usually 1, 0 if "NO COMPTON EFFECT" command given
	 * lgCompRecoil usually t, f if "NO RECOIL IONIZATION" command given 
	 *
	 **********************************************************************/

	/* comoff turned off with no compton command,
	 * lgCompRecoil - no recoil ionization */
	if( rfield.comoff > .0 && recoil.lgCompRecoil )
	{
		for( nelem=0; nelem<LIMELM; ++nelem )
		{
			recoil.CompRecoilIonRate[nelem] = 0.;
			recoil.CompRecoilHeatRate[nelem] = 0.;
			if( abundances.lgElmtOn[nelem])
			{
				/* recoil ionization starts at 194 Ryd = 2.6 keV */
				for( i=recoil.ipCompRecoil[nelem]-1; i < rfield.nflux; ++i)
				{
					crsphi = opac.OpacStack[i-1+opac.iopcom] * rfield.SummedCon[i] *
						recoil.nCompRecoilElec[nelem];

					/* direct hydrogen ionization due to compton scattering, 
					 * does not yet include secondaries,
					 * last term accounts for number of valence electrons that contribute */
					recoil.CompRecoilIonRate[nelem] += crsphi ;

					/* recoil energy in Rydbergs
					 * heating modified for suprathermal secondaries below; ANU2=ANU**2 */
					energy = 2.66e-5*rfield.anu2[i] - 1.;

					/* heating is in rydbergs because efionz, exctef, heatef in ryd */
					recoil.CompRecoilHeatRate[nelem] += crsphi*energy;
				}
				/* net heating efficiency, convert ryd/sec/cm3 to ergs/sec/cm3 */
				recoil.CompRecoilHeatRate[nelem] *= EN1RYD;
			}
			else
			{
				recoil.CompRecoilIonRate[nelem] = 0.;
				recoil.CompRecoilHeatRate[nelem] = 0.;
			}
		}
	}
	else
	{
		for( nelem=0; nelem<LIMELM; ++nelem )
		{
			recoil.CompRecoilIonRate[nelem] = 0.;
			recoil.CompRecoilHeatRate[nelem] = 0.;
		}
	}

	/**********************************************************************
	 *
	 *                          COSMIC RAYS
	 *
	 * heating and ionization by cosmic rays, other relativistic particles
	 * CRYDEN=density (1/CM**3), neutral rate assumes 15ev total
	 * energy loss, 13.6 into ionization, 1.4 into heating
	 * units erg/sec/cm**3
	 * iff not specified, CRTEMP is 2.6E9K
	 *
	 **********************************************************************/

	if( hextra.cryden > 0. )
	{
		/* this is current cosmic ray density */
		CosRayDen = hextra.cryden*pow(radius.Radius/radius.rinner,(double)hextra.crpowr);

		/* related to current temperature, when thermal */
		sqthot = sqrt(hextra.crtemp);

		/* rate hot electrons heat cold ones, Balbus and McKee 1982
		 * erg sec^-1 cm^-3,
		 * in sumheat we will multipy this rate by sum of neturals, but for this
		 * term we actually want eden, so mult by eden over sum of neut */
		heat_cold_electrons = 5.5e-14/sqthot*CosRayDen*
			phycon.eden/(xIonFracs[ipHYDROGEN][1] + xIonFracs[ipHELIUM][1] + hmi.htwo);

		/* ionization rate; Balbus and McKee */
		CosRayIonRate = 1.22e-4/sqthot*
			log(2.72*pow(hextra.crtemp/1e8,0.097))*CosRayDen;

		/* option for thermal CRs, first is the usual (and default) relativistic case */
		if( hextra.crtemp > 2e9 )
		{
			/*  cosmic ray ionization rate s-1 cm-3; ext rel limit */
			CosRayIonRate *= 3.;

			/*  usual circumstance; relativistic cosmic rays, factor is 15 eV per ionization,
			 * 13.6 into ionization itself, rest into non-ionizing heat, but must later
			 * multiply by number of secondaries that occur */
			CosRayHeatRate = (CosRayIonRate*0.093*EN1RYD + heat_cold_electrons);
		}
		else
		{
			/*  option for thermal cosmic rays */
			CosRayIonRate *= 10.;
			CosRayHeatRate = (heat_cold_electrons + CosRayIonRate*0.093*EN1RYD);
		}
		if( trace.lgTrace )
		{
			fprintf( ioQQQ, "     HIGHEN: cosmic ray density;%10.2e CRion rate;%10.2e CR heat rate=;%10.2e CRtemp;%10.2e\n", 
			  CosRayDen, CosRayIonRate, CosRayHeatRate, hextra.crtemp );
		}
	}
	else
	{
		CosRayIonRate = 0.;
		CosRayHeatRate = 0.;
	}
	/* will save ionization and heating into this cell,
	 * first will be directly added to secondary rate */
	PhotRate.PhotoRate[0][0][12][0] = CosRayIonRate;

	/* will assume all energy is low energy since already result of secondaries */
	PhotRate.PhotoRate[1][0][12][0] = CosRayHeatRate;
	PhotRate.PhotoRate[2][0][12][0] = 0.;

	/**********************************************************************
	 *
	 * add on extra heating due to turbulence, goes into [1] of [x][0][11][0]
	 *
	 **********************************************************************/

	/* TurbHeat added with hextra command, DispScale added with turbulence dissipative */
	if( (hextra.TurbHeat+DoppVel.DispScale) != 0. )
	{
		/* turbulent heating only goes into the low-energy heat of this element */
		/* >>>>chng 00 apr 28, functional form of radius dependence had bee turrad/depth
		 * and so went to infinity at the illuminated face.  Changed to current form as
		 * per Ivan Hubeny comment */
		if( hextra.turrad > 0. )
		{
			/* if turrad is >0 then vary heating with depth */
			PhotRate.PhotoRate[1][0][11][0] = 
				(float)(hextra.TurbHeat*sexp(radius.depth /hextra.turrad));

			/* >>chng 00 aug 16, added option for heating from back side */
			if( hextra.turback != 0. )
			{
				PhotRate.PhotoRate[1][0][11][0] += 
					(float)(hextra.TurbHeat*sexp((hextra.turback - radius.depth) /hextra.turrad));
			}
		}
		/* this is turbulence dissipate command */
		else if( DoppVel.DispScale > 0. )
		{
			double turb = DoppVel.TurbVel * sexp( radius.depth / DoppVel.DispScale );
			/* if turrad is >0 then vary heating with depth */
			/* >>chng 01 may 10, add extra factor of length over 1e13 cm */
			PhotRate.PhotoRate[1][0][11][0] = 3.45e-28 / 2.82824 * turb * turb * turb
				/ (DoppVel.DispScale/1e13) ;
		}
		else
		{
			/* constant extra heating */
			PhotRate.PhotoRate[1][0][11][0] = (float)hextra.TurbHeat;
		}

	}
	else
	{
		PhotRate.PhotoRate[1][0][11][0] = 0.;
	}


	/**********************************************************************
	 *
	 * option to add on fast neutron heating, goes into [0] & [2] of [x][0][11][0]
	 *
	 **********************************************************************/
	if( neutrn.lgNeutrnHeatOn )
	{
		/* CrsSecNeutron is 4E-26 cm^-2, cross sec for stopping rel neutrons
		 * this is heating due to fast neutrons, assumed to secondary ionize */
		PhotRate.PhotoRate[2][0][11][0] = neutrn.totneu*crsnut.CrsSecNeutron;
	}
	else
	{
		PhotRate.PhotoRate[2][0][11][0] = 0.;
	}
	/* neutrons assumed to only secondary ionize */
	PhotRate.PhotoRate[0][0][11][0] = 0.;


	/**********************************************************************
	 *
	 * pair production in elec field of nucleus 
	 *
	 **********************************************************************/
	PhotRate.PhotoRate[0][0][10][0] = GammaK(opac.ippr,rfield.nflux,opac.ioppr,1.);
	PhotRate.PhotoRate[1][0][10][0] = heat.HeatLowEnr;
	PhotRate.PhotoRate[2][0][10][0] = heat.HeatHiEnr;


	/**********************************************************************
	 *
	 * Compton energy exchange 
	 *
	 **********************************************************************/
	rfield.cmcool = 0.;
	rfield.cmheat = 0.;
	heatin = 0.;
	/* comoff usually 1, turns off Compton */
	if( rfield.comoff >0.01 )
	{
		for( i=0; i < rfield.nflux; i++ )
		{

			/* Compton cooling
			 * CSIGC is Tarter expression times ANU(I)*3.858E-25
			 * 6.338E-6 is k in inf mass Rydbergs, still needs factor of TE */
			rfield.comup[i] = (double)(rfield.flux[i]+rfield.ConInterOut[i]+
			  rfield.outlin[i])*rfield.csigc[i]*(phycon.eden*4.e0*
			  6.338e-6*1e-15);

			rfield.cmcool += rfield.comup[i];

			/* Compton heating 
			 * CSIGH is Tarter expression times ANU(I)**2 * 3.858E-25
			 * CMHEAT is just spontaneous, HEATIN is just induced */
			rfield.comdn[i] = (double)(rfield.flux[i]+rfield.ConInterOut[i]+
			  rfield.outlin[i])*rfield.csigh[i]*phycon.eden*1e-15;

			/* induced Compton heating */
			hin = (double)(rfield.flux[i]+rfield.ConInterOut[i]+rfield.outlin[i])*
			  rfield.csigh[i]*rfield.OccNumbIncidCont[i]*phycon.eden*1e-15;
			rfield.comdn[i] += hin;
			heatin += hin;

			/* following is total compton heating */
			rfield.cmheat += rfield.comdn[i];
		}

		/* remember how important induced compton heating is */
		if( rfield.cmheat > 0. )
		{
			rfield.cinrat = MAX2(rfield.cinrat,heatin/rfield.cmheat);
		}

		if( trace.lgTrace && trace.lgComBug )
		{
			fprintf( ioQQQ, "     HIGHEN: COOL num=%8.2e HEAT num=%8.2e\n", 
			  rfield.cmcool, rfield.cmheat );
		}
	}

	/* save compton heating rate into main heating save array, 
	 * no secondary ionizations from compton heating cooling */
	heat.heating[0][19] = rfield.cmheat;

	if( trace.lgTrace && trace.lgComBug )
	{
		fprintf( ioQQQ, 
			"     HIGHEN finds heating fracs= frac(compt)=%10.2e "
			" f(pair)%10.2e totHeat=%10.2e\n", 
		  heat.heating[0][19]/heat.htot, 
		  heat.heating[0][21]/heat.htot,
		  heat.htot	);
	}

#	ifdef DEBUG_FUN
	fputs( " <->highen()\n", debug_fp );
#	endif
	return;
}

