/*prt_He_like_DeparCoef routine to print departure coefficients for he-like species */
/*prt_He_like_Pops routine to print level populations for he-like species */
/*HeRecom - do recomb coef for He, called by HeLike */
/*HeCollid evaluate collisional rates */
/*HeCSInterp interpolate on He1 collision strengths */
/*HeLevel level populations  */
/*HeCreate create he-like series, called by ContCreatePointers */
/*he_1trans compute Aul for given line	*/
/*AGN_He1_CS routine to punch table needed for AGN3 - collision strengths of HeI */
/*defect defect - calculate quantum defect. */
/*correct2 - small routine to correct inaccuracies of Aul's for lower
 * level n = 1 and 2. */
/*ritoa - converts the square of the radial integral for a transition 
 * (calculated by scqdri) to the transition probability, Aul.	*/
/*he_energy - calculates energy of a given level.	*/
/*forbidden calculates transition probabilities for forbidden transitions.	*/
/*orbit - produces orbit designations for output.*/
/*printstuff - a few lines below, printflag determines what will be printed.*/
/*cross_section - calculates the photoionization cross_section for a given level and photon energy*/
/*radrecomb - calculates radiative recombination coefficients. */
/*RecomInt - Integral in milne relation.  Called by qg32.	*/
/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * ipZ is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
/*HelikeCheckRecomb - called by SanityCheck to confirm that recombination coef are ok,
 * return value is relative error between new calculation of recom, and interp value */
/*CS_l_mixing - find rate for l-mixing collisions by protons, for neutrals */
/* routine to print departure coefficients for he-like species */
#include "cddefines.h" 

/* 
  
  Energy order within 2 3P

  The order of the levels within the 2 3P level of atomic helium is opposite
  from the order within astrophysically abundant ions.  The indices below
  consistently point to the correct level, and the energies are correct,
  so the J levels within 2 3P are not in increasing energy order for helium itself. 
  This is ok since the atomic data is correct, and the difference in energies is so small.

*/

/* 
the following block of defines occurs in cddefines.c, and echoed in cddefines.h 
const int ipHe1s1S = 0;
const int ipHe2s3S = 1;
const int ipHe2s1S = 2;
const int ipHe2p3P0 = 3;
const int ipHe2p3P1 = 4;
const int ipHe2p3P2 = 5;
const int ipHe2p1P = 6;

level 3
const int ipHe3s3S = 7;
const int ipHe3s1S = 8;
const int ipHe3p3P = 9;
const int ipHe3d3D = 10;
const int ipHe3d1D = 11;
const int ipHe3p1P = 12;

level 4
const int ipHe4s3S = 13;
const int ipHe4s1S = 14;
const int ipHe4p3P = 15;
const int ipHe4d3D = 16;
const int ipHe4d1D = 17;
const int ipHe4f3F = 18;
const int ipHe4f1F = 19;
const int ipHe4p1P = 20;
*/

/* there are iso.nLevels[ipHELIUM][nelem] levels total */

/* Set this flag to one of the following values
 *	0	don't print
 *	1	print As
 *	2	print only forbidden As
 *	3	print Es
 *	4	print Threshold Photoionizing Cross-Sections
 *	5	print radiative recombination coefficients.	
 *  6   print Photoionizing Cross-Section grids.	*/
/* Out files are "Helike___.txt" with the blank being, respectively,
 *	"As","fAs","Es","ThPCS","RR", and "PCSgrid."	*/
#define	printflag	(0)

/* temperatures at which CS for He1 are stored */
#define HE1CSARRAY 10
/* the magic number for the table of recombination coefficients */
#define RECOMBMAGIC		(20124)

/* next two are in helike.h */
/* Max n desired in RRCoef file.
 * also used in atom he-like levels large 
#define HE_RREC_MAXN		20	*/
/* the number of levels that will be done
#define N_HE_LEV_RECOMB		( ( 1 + HE_RREC_MAXN ) * HE_RREC_MAXN + 1 ) */
/* the number of temperature steps to get from lowest to highest temperature */
#define N_HE_TE_RECOMB		21

static double CSTemp[HE1CSARRAY];

/* the 4-d array of cs's */
static float ****HeCS;

#include "physconst.h" 
#include "dynamics.h"
#include "iso.h"
#include "tfidle.h"
#include "getgf.h"
#include "elementnames.h"
#include "refindex.h"
#include "taulines.h"
#include "linpack.h"
#include "hevmolec.h"
#include "receff.h"
#include "abscf.h"
#include "he3c.h"
#include "path.h"
#include "veclib.h"
#include "rfield.h"
#include "atomcwgt.h"
#include "negcon.h"
#include "typmatrx.h"
#include "trace.h"
#include "destcrt.h"
#include "heavy.h"
#include "drfe.h"
#include "cfit.h"
#include "collionrate.h"
#include "ionfracs.h"
#include "opacity.h"
#include "secondaries.h"
#include "chargtran.h"
#include "phycon.h"
#include "rrfit.h"
#include "recom.h"
#include "abundances.h"
#include "ionrange.h"
#include "h_bauman.h"
#include "helike.h"

/*	The Elevels data type, and then the he_levels structure,
 *  which contain the quantum numbers n,l, and s for a given
 *  energy level, are defined here, for use in multiple subroutines. 
 *  Moved this definition here from "helike.h"	*/

typedef struct {
	long n;
	long s;
	long l;
} Elevels;/* Main_Config.h */ 

/* A vector of structures each containing n,l, and s
 * s=0 for singlets, s=1 for triplets */
static Elevels *he_levels;

/* this will save log of radiative recombination rate coefficients at N_HE_TE_RECOMB temperatures.
 * there will be N_HE_LEV_RECOMB levels saved in RRCoef[nelem][level][temp] */
static double ***RRCoef/*[LIMELM][N_HE_LEV_RECOMB][N_HE_TE_RECOMB]*/;

/* the array of logs of temperatures at which RRCoef was defined */
static double TeRRCoef[N_HE_TE_RECOMB];

static double He_RRCoef_Te( long nelem , long n );

/*CS_l_mixing - find rate for l-mixing collisions by protons, for neutrals */
static double CS_l_mixing(
	/* the chemical element, 1 for He */
	long nelem ,
	/* the level, 0 for ground */
	long ipLo,
	long ipHi);

/* compute quantum defect */
static double defect(
			long ipZ,
			long ipLo
			);
 
static double correct2(
			long ipHi,
			long ipLo
			);

static double ritoa(
			long li,
			long lf, 
			long ipZ, 
			double k, 
			double RI2
			);

static double he_energy(
			double Eff_n,
			long ipZ,
			long ipLo 
			);

static double forbidden(
			long ipHi,
			long ipLo,
			long ipZ
			);

static void orbit(
				  long index
				  );

static void printstuff(long flag);

static double cross_section(double EgammaRyd);
	
static double radrecomb(double temp, long ipZ, long ipLo);

static double RecomInt(double E);

/*static double CoolInt(double ERyd);*/

/* compute energy diffference in wn and Aul for given line
 * return is 0 for success, 1 for failure */
static void he_1trans(  
			  /* charge on the C scale, 1 is helium */
			  long ipZ , 
			  /* energy difference in wavenumber */
			  double Enerwn , 
			  /* transition probability */
			  double *Aul ,
			  long ipHi , long ipLo, double Eff_nupper, double Eff_nlower );

/* Function he_assign assigns, in order of increasing energy, 	*/
/* quantum numbers n,l, and s to each element of vector he_levels.	
 * nmax is the upper quantum number, there will be many l levels within it.
 * ndim is dimension of vector, used to confirm that size not exceeded */ 
static void he_assign( long nmax , long ndim );

/* 
 *  Inclusive -- i.e. n = 1,2,...,MAX_N 
 *  MAX_N -- Quantum Number n 
 * 
 *  maximum principle quantum number considered in calculations (inclusive) 
 * (long int) 
*/ 
/*#define      MAX_N           2L */
/* Effective charge of H+ ion */ 
/* (long int) */ 
/* ndim -- Array Dimension 
 * 
 *  Sum of levels (n,l) for 1 <= n <= MAX_N is given by 
 * 
 *         MAX_N 
 *         --- 
 * NDIM =  >  n' = ( (MAX_N)^2 + MAX_N ) * 1/2 
 *         --- 
 *         n'=1 
 * 
 * Now subtract one (1) since Array's are (indexed) like 
 * 0,1,2,3,..., NDIM-1 
 * DANGER WILL ROBINSON -- Divide (even) integer by 2 below 
 * 
 *  number of states (n,l) between n=1 and n=MAX_N (inclusive) 
 */ 
/*#define NDIM            ((((MAX_N)*(MAX_N))+MAX_N)/2) */
/* I can't remember what this was */ 
/*#define SDIM            ((MAX_N)*(MAX_N)) */
/* 
    if non-zero use Brocklehurst for recombination co-efficients 
*/ 

#define hbar					6.5821220e-16       /* eV sec */

/* these are not used */
/*#define AA(I_,J_)       (*(A+(J_)*(NDIM)+(I_))) */
/*#define Y(I_)           (*(yVector+(I_))) */
/*#define SS(I_,J_)       (*(S+(J_)*(SDIM)+(I_))) */
/*#define ipivector(I_)   (*(ipiv+(I_))) */
 
/******************** Program Modifying Defines ***********************/ 
/*#ifndef  REPLACE_GROUND_STATE */
/*#define  REPLACE_GROUND_STATE 0 */
/*#endif *//* REPLACE_GROUND_STATE */ 
/**********************************************************************/ 

static double kTRyd,/*EthEV,*/EthRyd; 
static long ipLev,globalZ;
static void HeRecom( long nelem );
/* evaluate collisional rates */
static void HeCollid( long nelem);
/* evaluate photoionization rates  */
static void HeLevel( long nelem);
/* HeCSInterp interpolate on He1 collision strengths */
static float HeCSInterp( long nelem , long ipHi , long ipLo );

static FILE *ofp;  

/* experimental energies, in wavenumbers, for atomic helium */
#define NHE1LEVELS 98
static double He1Energies[NHE1LEVELS] =
{0,		  159850.318,  166271.7,169082.185,169081.189,169081.111,171129.148,
183231.08, 184859.06,  185559.2,  186095.9, 186099.22, 186203.62, 190292.46,
 190934.5, 191211.42, 191438.83, 191440.71, 191446.61, 191447.24, 191486.96,
193341.33, 193657.78, 193795.07, 193911.48, 193912.54, 193915.79, 193914.31,
193916.26, 193916.26, 193936.76, 194930.46, 195109.17, 195187.21, 195254.37,
195255.02, 195256.82, 195256.7,  195257.26, 195257.26, 195257.26, 195257.26,
195269.17, 195862.63, 195973.19, 196021.72,    196064,  196064.3, 196065.51, 
 196065.4, 196065.85, 196065.85, 196065.85, 196065.85, 196065.85, 196065.85, 
196073.41, 196455.79, 196529.03, 196561.08, 196589.42, 196589.73, 196590.42, 
 196590.3, 196590.65, 196590.65, 196590.65, 196590.65, 196590.65, 196590.65, 
196590.65, 196590.65, 196595.56, 196856.37, 196907.13, 196929.68, 196949.63, 
196949.49, 196950.36, 196950.3,  196950.45, 196950.45, 196950.45, 196950.45, 
196950.45, 196950.45, 196950.45, 196950.45, 196950.45, 196950.45, 196953.95, 
197139.76, 197176.36, 197192.63, 197207.30, 197207.08,    197208,    197208}; 
/* Last energy is 10^1F.	*/

/* First in this refuse pile is 10^1P.	*/
/* 
197210.41, 197347.05, 197386.98, 197397.75, 197397.62,  197398.6,
197400.18, 197503.69, 197524.26, 197534.44, 197542.54, 197542.67, 
197544.56, 197624.98, 197649.07, 197649.78, 197655.19, 197655.47, 197656.95, 
197721.13, 197739.9, 197744.918, 197744.94, 197746.15, 197796.63, 197813.11, 
197817.05, 197818.12, 197872.95, 197876.41, 197877.04, 197922.51, 197925.33, 
197925.87, 197964.02, 197966.75, 197966.8, 197999.12, 198001.43, 198001.44, 
198029.07, 198031.02, 198031.41, 198054.83, 198056.5, 198077.15};
*/

/* Old threshold energies...changed these to Verner's numbers for uniformity.	*/
/*
198310.76,610078.5,1241242,2091976,3162408,4452722,5963078,7693810,9645400,
11816996,14209908,16824910,19660800,22700000,26001513,29507951.12,33237172.57,
37188596.44,41366547,45770332.07,50401565.71,55259441.05,60343958.1,65658343.06,
71203000,76978329.86,82984738.24,89224079.45,95800000
*/
	/* Ionization potentials (in wavenumber) for 	*/
	/* each ion in the iso-seq up to Z=30.			*/
	/* These are exactly what you get if you take EionRYD below and multiply by RYD_INF.	*/
static double EionWN[29] =       
	{198307.729760353,610003.839889137,1241136.72201499,2091948.45665631,
	 3162116.52584231,4452446.95015668,5962133.81875305,7692790.05069734,
	 9645221.44709864,11814589.7994457,14209766.0528639,16822685.5022862,
	 19661412.9625169,22717883.6187518,26000162.0663204,29508248.5246975,
	 33234078.1790787,37185715.7345311,41363161.0813172,45766414.4389118,
	 50395475.4781030,55258409.0136949,60339085.8550283,65653635.1927626,
	 71202056.8074231,76976286.4328920,82984388.3352872,89194104.5722390,
	 95726403.3055320};
	
	/* Ionization potentials (in Rydbergs) for 		*/
	/* each ion in the iso-seq up to Z=30.			*/
	/* These are exactly what you get if you take EionEV below and 
	 * divide by EVRYD.	*/
static double EionRYD[29] =       
	{1.807113,5.558764,11.310070,19.063237,28.815326,40.573682,54.330961,70.101861,
	87.893725,107.662464,129.488916,153.299590,179.167978,207.020588,236.930910,
	268.898946,302.851204,338.861175,376.928858,417.054255,459.237363,503.551674,
	549.850208,598.279945,648.840883,701.459535,756.209388,812.796486,872.323172};

	/* Ionization potentials (in eV) for 		*/
	/* each ion in the iso-seq up to Z=30.		*/
	/* These are exactly what you get if you take Verner's numbers in 
	 * PH1COM.PH1[0][1][nelem][0] and multiply by 0.9998787, exactly as 
	 * is done elsewhere in Cloudy.	*/
double EionEV[29] =       
	{24.587017,75.630824,153.881326,259.368529,392.052444,552.033006,739.210311,
	953.784316,1195.854925,1464.822296,1761.786269,2085.746968,2437.704271,
	2816.658298,3223.608929,3658.556163,4120.500123,4610.440686,5128.377852,
	5674.311623,6248.241996,6851.168852,7481.092433,8140.012497,8827.929042,
	9543.842191,10288.751823,11058.658422,11868.560169};

#if 0
#define NHE1AS 7
long ipHe1AsLo[NHE1AS] = {0,2,1,3,3,1,3},
	 ipHe1AsHi[NHE1AS] = {4,4,3,8,14,7,5};
double He1AsLam[NHE1AS] = {584.,20600.,10830.,5876.,4471.,3889.,7065.};
double He1AsAul[NHE1AS] = {1.80E+09,1.97E+06,1.02E+07,7.07E+07,2.46E+07,
	9.47E+06,2.78E+07};
#endif

static int HeAsIndex,HeFsIndex;

/*He-like main routine to call HeLevel and determine model he-like species atom level balance,
 * called by ionize */
void HeLike(void)
{
	int lowsav , ihisav;
	
	long int
		ipLo ,
		ipHi , 
		nelem;

#	ifdef DEBUG_FUN
	fputs( "<+>HeLike()\n", debug_fp );
#	endif


	/* save state of he-like low and high ionziation, since must not change here,
	 * since there is a parallel helium atom that must decrement he */
	/*TODO remove this when this routine really controls helium itself */
	lowsav = IonRange.IonLow[ipHELIUM];
	ihisav = IonRange.IonHigh[ipHELIUM];
	for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* do not consider elements that have been turned off */
		if( abundances.lgElmtOn[nelem] )
		{
			/* note that nelem scale is totally on c not physical scale, so 0 is h */
			/* evaluate helium-like balance if ionization reaches this high */
			if( (IonRange.IonHigh[nelem] >= nelem)  )
			{

				/* evaluate recombination rates */
				HeRecom(nelem);

				/* evaluate collisional rates */
				HeCollid(nelem);

				/* >>chng 02 jan 18, move to unified photo rate routine */
				/* evaluate photoionization rates */
				iso_Photo(ipHELIUM , nelem );

				/* solve for the level populations */
				HeLevel(nelem);

			}
			else
			{
				iso.xIonRatio[ipHELIUM][nelem] = 0.;
				iso.xIonSimple[ipHELIUM][nelem] = 0.;

				/* zero it out since no population*/
				iso.Pop2Ion[ipHELIUM][nelem][0] = 0.;
				for( ipHi=ipHe2s3S; ipHi < iso.nLevels[ipHELIUM][nelem]; ipHi++ )
				{
					iso.Pop2Ion[ipHELIUM][nelem][ipHi] = 0.;
					for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
					{
						/* population of lower level rel to ion */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopLo = 0.;

						/* population of upper level rel to ion */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopHi =  0.;

						/* population of lower level rel to ion, corrected for stim em */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopOpc =  0.;
					}
				}
			}
		}
	}
	IonRange.IonLow[ipHELIUM] = lowsav;
	IonRange.IonHigh[ipHELIUM] = ihisav;

	return;
}

static double correct2(
			  long ipHi,
			  long ipLo)
{

	long ni = he_levels[ipHi].n;
	long nf = he_levels[ipLo].n;
	long li = he_levels[ipHi].l;
	/* long si = he_levels[ipHi].s;	*/
	
	if( ( nf == 2 ) && ( li == 2 ) )
	{
		return ( 0.15*ni+0.52 );	
	}
	
#if	0
	else if ( nf == 3 )
	{
		/*if ( ipHi == ipHe3d3D )
		{
			return 0.4;
		}
		else */
		if ( ( ipHi == ipHe3p1P ) && ( ipLo == ipHe3d1D ) )
		{
			return 1.0;
		}
		else if ( ( ipLo == ipHe3d3D ) && ( li == 1 ) && ( si == 1 ) )
		{
			return 0.95;
		}
	}
#endif
	
	return 1.0;
}

/*  Sep 10,01: Removed old correction routine.  No longer applicable!	*/

/* This routine calculates the quantum defect	*/ 
/* for a given level and nuclear charge.		*/
static double defect(long ipZ, long ipLo)  
{  
	double qd;	/* The quantum defect	*/

	/* Parameters for fits to quantum defects for	*/
	/* P triplet and S orbitals. The dimensions are	*/
	/*	first: l									*/
	/*	second: n									*/
	/* 	third: parameters a,b,and c.				*/
	double param[3][4][3]=  
	{ 

		{{0.6451941,0.3119437,-1.2722842},		/* ^3S	*/
			{0.7664874,0.3455675,-1.3976462}, 
			{0.8247101,0.3603131,-1.4520500}, 
			{0.8878402,0.3714450,-1.4995732}}, 

		{{1.4203514,0.5311096,-2.6728087},		/* ^1S	*/
			{1.5733513,0.5997339,-2.9253834}, 
			{1.4531025,0.5924751,-2.8662756}, 
			{1.6038999,0.6342552,-3.0298071}}, 

		{{-2.2323488,0.0890840,-0.5166053},		/* ^3P	*/
			{-2.0463691,0.1222081,-0.6672983}, 
			{-1.9904104,0.1328918,-0.7150879}, 
			{-1.9500974,0.1452111,-0.7649031}} 
	};   

	/* Because they cannot be fit to a funtion of	*/
	/* the same form as the other orbitals, the 	*/
	/* P singlets are fit to a different function.	*/
	/* These are the parameters, with dimensions	*/
	/*	first: n									*/
	/*	second: parameters a and b.					*/
	double P1[4][2]=
	{
		{-56.65245,-3.661923},
		{-52.03411,-4.941075},
		{-50.43744,-5.525750},
		{-49.45137,-5.908615}
	};

	/* The P singlets of He do not conform to the 	*/
	/* above, so the defects themselves are listed,	*/
	/* for n = 2,3,4, and 5.						*/
	double P1He[4]=
	{-0.009492,-0.011351,-0.011904,-0.01218};


	double a,b,c;   
	long n = he_levels[ipLo].n;
	long lqn = he_levels[ipLo].l;
	long s = he_levels[ipLo].s;
	long nold = n;  

	assert(n >= 1L);
	/* Only Helium and up, and only those turned on.	*/
	assert((ipZ >= ipHELIUM) && (ipZ <= LIMELM));       

	if ( ipLo == ipHe1s1S )
	{
		/* Quantum defects for ground state are found	*/
		/* from the rydberg equation, and the			*/
		/* ionization potential of the ion.				*/
		assert(ipZ>0 );
		qd = 1.0 - ipZ * sqrt(1/EionRYD[ipZ-1]);
		return qd;  
	}
	else
	{
		/* For levels with n > 5, the quantum defect	*/
		/* is approximately the same as if n equaled 5.	*/
		if ( n > 5L )
		{
			n = 5L;
		}
		/* For P singlets	*/
		if ( lqn==1L && s==0L )
		{
			if ( ipZ == ipHELIUM )
				qd = P1He[n-2];
			else
				qd = 1./(P1[n-2][0] + P1[n-2][1] * (ipZ+1) * log((double)ipZ+1.) );
				/* qd = (P1[n-2][0])*exp(-1*(ipZ+1)/(P1[n-2][1]));  */
			return qd;  
		}
		/* Defects for orbitals with l>2 	*/
		/* are approximately equal to zero.	*/
		else if ( lqn < 2L )
		{
			a = param[2*(lqn+1)-s-1][n-2][0];  
			b = param[2*(lqn+1)-s-1][n-2][1];  
			c = param[2*(lqn+1)-s-1][n-2][2];  
			qd = exp((a+c*(ipZ+1))/(1.0+b*(ipZ+1)));  
			return qd;  
		}
		else if ( lqn < nold )
		{
			return 0.0 /*0.008 / (ipZ * lqn * lqn) )*/; 
		}
		else
		{
			exit(9);  
		}  
		/* it is impossible to reach this point */
	}   
}

static double he_energy(double Eff_n,long ipZ, long ipLo )
{
	double Ef,gap2tripP;
	
	/* for atomic helium use experimental energies */
	if( ipZ == ipHELIUM && ipLo < NHE1LEVELS )
	{
		Ef = EionWN[ipHELIUM-1] - He1Energies[ipLo];
	} 
	else
	/* Calculate energy difference for all other trans.	*/
	{
		Ef = RYD_INF * (ipZ/Eff_n) * (ipZ/Eff_n);  
	}
	
	if (ipZ > 6)
	{
	/* gap2tripP represents the amount to add or subtract from the median 
	 * 2p3P term in order to find energies of individual levels of the term.	
	 * There's an unexpected complication here.  According to data from Porquet and Dubau 
	 * (2000), the separate J levels of the term are NOT well ordered along the 
	 * sequence.  In CV, for example, the J=1 term is closest to ground, while 
	 * in FeXXV, the order is J=0, J=1, J=2.  And HeI has the exact opposite, 
	 * J=2 is closest to ground (name source of He energies).  Our solution is 
	 * to treat the energies for the three levels as degenerate for Z=3..7(?),
	 * since according to Porquet and Dubau, the levels don't become well-
	 * ordered until around Z=8 (ipZ=7). */
		gap2tripP = 0.007*pow((double)ipZ+1., 4.5);
		if ( ipLo == ipHe2p3P0 )
		{
			Ef += (3*gap2tripP);
		}
		else if ( ipLo == ipHe2p3P1 )
		{
			Ef += (2*gap2tripP);
		}
		else if ( ipLo == ipHe2p3P2 )
		{
			Ef -= (5*gap2tripP);
		}
	}

	assert (Ef > 0.);
	return Ef;
}

static double forbidden(
				long ipHi,
				long ipLo,
				long ipZ
				)
{
	double A,b,c;

	if ( (ipLo == ipHe1s1S) && (he_levels[ipHi].n == 2) )
	{
		switch ( (int)ipHi )
		{
		/* Parameters for 2^3S to ground transition.	*/
		case 1:
			b = 4E-7;
			c = 10.39;
			break;
		/* Parameters for 2^1S to ground transition.	*/
		case 2:
			b = 3.93;
			c = 6.3856;
			break;
		/* Parameters for 2^3P0 to ground transition.	*/
		case 3:
			b = 1e-20f;
			c = 0;
			break;
		/* Parameters for 2^3P1 to ground transition.	*/
		case 4:
			b = 0.4218 /* 0.2105 */;
			c = 10.081 /* 10.314 */;
			break;
		/* Parameters for 2^3P2 to ground transition.	*/
		case 5:
			b = 0.0073;
			c = 8.4733;
			break;
		default:
			b = 0;
			c = 0;
			return (1e-20f);
		}
		/* Old values (before splitting of 3P term) are...
			 *	b = 0.9317; c = 9.7306;	*/
				
		A = b * pow((double)ipZ+1.,c);
	}
	/* Triplet P to Singlet S...Delta n not equal to zero!	*/
	else if( (ipZ > ipHELIUM) && (he_levels[ipHi].l == 1) && (he_levels[ipHi].s == 1) 
		&& (he_levels[ipLo].l == 0) && (he_levels[ipLo].s == 0) )
	{
		assert( he_levels[ipLo].n < he_levels[ipHi].n );

		A = 8.0E-3 * exp(9.283/sqrt((double)he_levels[ipLo].n)) * pow((double)ipZ,9.091) 
			/ pow((double)he_levels[ipHi].n,2.877);
	}
	/* Singlet S to Triplet P...Delta n not equal to zero!	*/
	else if( (ipZ > ipHELIUM) && (he_levels[ipHi].l == 0) && (he_levels[ipHi].s == 0) 
		&& (he_levels[ipLo].l == 1) && (he_levels[ipLo].s == 1) )
	{
		assert( he_levels[ipLo].n < he_levels[ipHi].n );

		A = 2.416 * exp(-0.256 * he_levels[ipLo].n) * pow((double)ipZ,9.159) 
			/ pow((double)he_levels[ipHi].n,3.111);
		
		if ( ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P2) ) )
		{
			/* This is divided by 3 instead of 9, because value calculated is specifically for 2^3P1.
			 * Here we assume statistical population of the other two.	*/
			A *= (2.*(ipLo-3)+1.0)/3.0;
		}
	}
	else 
		A = helike.SmallA;

	assert( A > 0.);
	return A;   
}


/*  Function orbit generates labels for each level and is only necessary for output */
static void orbit(long index)
{
	char *c = "     ";

	/*  A string containing the orbit name is output.		*/

	switch ( (int)he_levels[index].l )
	{
	case 0:
		c = "S";
		break;
	case 1:
		c = "P";
		break;
	case 2:
		c = "D";
		break;
	default:
		fprintf(ofp,"%-2c ",67+(int)he_levels[index].l);
		return;
	}
	fprintf(ofp,"%-1.1s",c);
	if ( (index == 3) || (index == 4) || (index == 5) )
	{
		fprintf(ofp,"%1ld ",(index-3));
	}
	else
	{
		fprintf(ofp,"  ");
	}
	
	return;
}

/* REMOVED FUNCTION "loop"..."he_1trans" is an improved version.	*/

/* Calculates Einstein A for a given transition.	*/
static void he_1trans( long ipZ , double Enerwn , double *Aul ,
			   /* indices for the upper and lower levels */
			   long ipHi , long ipLo, double Eff_nupper, double Eff_nlower )
{
	/* this will be A returned for totally forbidden transition 
#	define SMALLA 1e-20 - 
	replaced with helike.SmallA */
	/* Variables used are:							*/
	/*  ipZ is charge on the c scale				*/
	/*	Eff_nupper,Eff_nlower = initial and final	*/
	/*		principal quantum numbers.				*/
	/*	*Aul = Einstein A							*/
	/*	RI2 = square of radial integral.			*/
	
	/* structure giving n, l, s of upper level */
	Elevels he_upper = he_levels[ipHi], 
	/* structure giving n, l, s of lower level */
			he_lower = he_levels[ipLo];

	/* This transition probabilities are now indexed by [ipHi-1], from 1 to 30.  
	 * The individual values in each vector are for allowed transitions to lower 
	 * levels, in order of increasing ipLo.  */

	/* Note that in order to give real meaningful numbers for all allowed transitions here, it
	 * was necessary to "invent" numbers for six same-n transitions for n = 4 and 5.  These are
	 * designated by comments characters just before the number.	*/
	double BenjaminAs[30][8] = {
		{1.13E-04,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{5.13E+01,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},			
		{1.76E+02,	1.02E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{1.76E+02,	1.02E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{1.76E+02,	1.02E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{1.80E+09,	1.97E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{3.09E+06,	9.27E+06,	1.54E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},			
		{1.83E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{9.47E+06,	1.07E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{7.86E+06,	2.36E+07,	3.93E+07,	1.29E+04,	   1E-20,	   1E-20,	   1E-20,	   1E-20},		
		{6.37E+07,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{5.66E+08,	1.34E+07,	2.52E+05,	1.53E+02,	   1E-20,	   1E-20,	   1E-20,	   1E-20},		
		{1.06E+06,	3.17E+06,	5.29E+06,	6.51E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20},		
		{6.77E+06,	4.59E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{5.63E+06,	7.09E+05,	6.46E+05,	2.28E+05,	   1E-20,	   1E-20,	   1E-20,	   1E-20},		
		{2.73E+06,	8.20E+06,	1.37E+07,	6.19E+06,	4.18E+03,	   1E-20,	   1E-20,	   1E-20},		
		{1.99E+07,	7.12E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{1.38E+07, /**/4E+04,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},			
		{1.38E+07, /**/4E+04,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},	   					
		{2.43E+08,	6.95E+06,	1.41E+06,	2.97E+05,	5.83E+04,	5.66E+01,	   1E-20,	   1E-20},	
		{4.94E+05,	1.48E+06,	2.47E+06,	2.73E+06,	2.02E+06,	   1E-20,	   1E-20,	   1E-20},
		{3.27E+06,	2.06E+06,	1.50E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},				
		{3.20E+06,	5.69E+05,	2.73E+05,	1.21E+05,	3.27E+05,	7.02E+04,	   1E-20,	   1E-20},	
		{1.29E+06,	3.87E+06,	6.44E+06,	3.48E+06,	1.28E+06,	5.05E+04,	1.51E+03,	   1E-20},
		{8.99E+06,	3.36E+06,	5.05E+04,	1.52E+06,	   1E-20,	   1E-20,	   1E-20,	   1E-20},			
		{4.54E+06,	2.58E+06,  /**/3E+03,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},					
		{4.54E+06,	2.58E+06,  /**/3E+03,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{4.25E+06, /**/4E+03,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{4.25E+06, /**/4E+03,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20,	   1E-20},
		{1.26E+08,	3.80E+06,	9.25E+05,	1.27E+05,	2.93E+05,	1.63E+05,	1.87E+04,	2.39E+01}};

	/* These are He oscillator strengths from Kono & Hattori 84.  */
	/*	The dimensions are:[upper.n - 6][ipHi - 1 - (n-1)*n][index],
		where index is the order of the lower level for all allowed
		transitions from the upper level.	*/
	double KonoHattoriF[4][6][16]	=	{
		{{0.000609,	0.001829,	0.003049,	0.0247	,	0.104	,	0.903	,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.004769,	0.02111	,	0.0862	,	0.647	,	0		,	0		,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},											
		{0.006982,	0.01199	,	0.00258	,	0.02163	,	0.017	,	0.0415	,	0.147	,	1.842	,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},						
		{0.002608,	0.007823,	0.013039,	0.05301	,	0.124	,	0.43	,	0.354	,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.02095 ,	0.05629	,	0.1529	,	0.67	,	0		,	0		,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.008628,	0.01213	,	0.02419	,	0.001188,	0.0528	,	0.00838	,	0.1526	,	0.0684	,	1.305	,
			0.074	,	0		,	0		,	0		,	0		,	0		,	0	}},				
		
		{{0.000346,	0.001038,	0.001729,	0.01221	,	0.0382	,	0.1403	,	1.139	,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.002739,	0.01064	,	0.03305	,	0.1172	,	0.819	,	0		,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.004299,	0.00707	,	0.001284,	0.01178	,	0.0065	,	0.02111	,	0.0289	,	0.0403	,	0.21	,
			2.151	,	0		,	0		,	0		,	0		,	0		,	0	},				
		{0.001507,	0.00452 ,	0.007533,	0.02816	,	0.0552	,	0.1254	,	0.434	,	0.426	,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.0119  ,	0.0289	,	0.0636	,	0.1632	,	0.703	,	0		,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},										
		{0.005405,	0.00736	,	0.01368	,	0.000594,	0.02588	,	0.00327	,	0.0556	,	0.01501	,	0.1609	,
			0.1008	,	1.53	,	0.089	,	0		,	0		,	0		,	0	}},		
		
		{{0.000217,	0.000650,	0.001083,	0.00707	,	0.0191	,	0.0517	,	0.176	,	1.376	,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.00173 ,	0.00623	,	0.01681	,	0.0451	,	0.1482	,	0.993	,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.002836,	0.00454	,	0.000743,	0.00716	,	0.00327	,	0.01174	,	0.01116	,	0.021	,	0.042	,
			0.0399	,	0.276	,	2.458	,	0		,	0		,	0		,	0	},			
		{0.000956,	0.002868,	0.004779,	0.01698	,	0.0303	,	0.0571	,	0.128	,	0.445	,	0.496	,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.007464,	0.01708	,	0.0336	,	0.0693	,	0.1732	,	0.741	,	0		,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},									
		{0.00361 ,	0.004813,	0.00857	,	0.000345,	0.01495	,	0.001664,	0.0276	,	0.00595	,	0.0588	,
			0.0226	,	0.1705	,	0.135	,	1.745	,	0.105	,	0		,	0	}},	
		
		{{0.000145,	0.000435,	0.000726,	0.00451	,	0.01117	,	0.026	,	0.065	,	0.212	,	1.613	,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.001167,	0.003999,	0.00994	,	0.02305	,	0.0571	,	0.1793	,	1.166	,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.00197 ,	0.00309	,	0.000474,	0.00471	,	0.001919,	0.00727	,	0.00567	,	0.0118	,	0.01627	,
			0.0212	,	0.0555	,	0.04	,	0.343	,	2.764	,	0		,	0	},
		{0.000647,	0.001941,	0.003235,	0.01112	,	0.01871	,	0.0318	,	0.0589	,	0.132	,	0.46	,
			0.564	,	0		,	0		,	0		,	0		,	0		,	0	},							
		{0.005015,	0.01104	,	0.02034	,	0.0373	,	0.0745	,	0.1837	,	0.784	,	0		,	0		,
			0		,	0		,	0		,	0		,	0		,	0		,	0	},								
		{0.00253 ,	0.003325,	0.00576	,	0.00022	,	0.00954	,	0.000979,	0.01613	,	0.00306	,	0.0293	,
			0.00903	,	0.0623	,	0.0307	,	0.181	,	0.171	,	1.963	,	0.12}}};
	
	double RI2,HeF;
	assert(ipZ > ipHYDROGEN);

	/* check if this is an allowed transition, or any 2-1 transition */
	if ( ((he_upper.s == he_lower.s) && (abs((int)(he_upper.l - he_lower.l)) == 1)) || 
		 ((he_upper.n == 2L) && (he_lower.n == 1L)) || 
		 /* Include trip P to sing S transitions from Johnson et al 2002.	*/
		 ( ( ipZ > 1 ) && (he_upper.n > he_lower.n) &&
		 ( ((he_upper.l == 1L) && (he_upper.s == 1L) && (he_lower.l == 0L) && (he_lower.s == 0L)) ||
		 ((he_upper.l == 0L) && (he_upper.s == 0L) && (he_lower.l == 1L) && (he_lower.s == 1L)) ) ) )
	{
		/* TOPbase suggests that even a hydrogenic 10-10 transition has a transition
		 * probability on the order of 1.  Make that the base for allowed transitions.	*/
		*Aul = 1.;

		/* transition prob - use tabulated values for He itself, for n <=5 -
		 * these appear in a list above */
		if ( (ipZ == ipHELIUM) && (ipHi <= ipHe5p1P) && (ipLo <= ipHe5g1G) && (HeAsIndex < 8) )
		{
			*Aul = BenjaminAs[ipHi-1][HeAsIndex];
			HeAsIndex++;
			assert( *Aul > 0. );
		}
		/* Retrieve Kono & Hattori (1984) oscillator strengths.	*/
		else if ( (ipZ == ipHELIUM) && (he_upper.n < 10) && (he_upper.l < 3) && (he_lower.l < 3) && (HeFsIndex < 16) )
		{
			if ( (he_upper.l == 1) && (he_upper.s == 0) )
				/* Special second dimension index for singlet P.	*/
				HeF = KonoHattoriF[he_upper.n - 6][5][HeFsIndex];
			else
				HeF = KonoHattoriF[he_upper.n - 6][ipHi-1-(he_upper.n-1)*he_upper.n][HeFsIndex];
			*Aul = 0.66702 * HeF * Enerwn * Enerwn;
			/* Oscillator strengths are already weighted correctly for delta l = +1.
			 * Must fix for delta l = -1.	*/
			if ( he_upper.l > he_lower.l )
				*Aul *= ( (2. * he_lower.l + 1.) / (2. * he_upper.l + 1.) );
			HeFsIndex++;
			assert( *Aul > 0. );
		}
		/* Calculate forbidden transitions.	*/
		else if ( ((he_upper.l == he_lower.l) || (he_upper.s != he_lower.s)) )
		{
			*Aul = forbidden(ipHi, ipLo, ipZ);
			assert( *Aul > 0. );
		}
		/* Calculate high-l, delta n != 0 transitions.	*/
		else if ( ((he_upper.l > 2) || (he_lower.l > 2)) && (he_upper.n > he_lower.n) )
		{
			/*lint -e790 integral to float */
			*Aul = H_Einstein_A(he_upper.n ,he_upper.l , he_lower.n , he_lower.l , ipZ);
			/* *Aul *= pow(Enerwn/109737.315,3.) * pow((double)(he_upper.n*he_lower.n)/(double)ipZ, 6.) 
				* pow(1./(he_upper.n*he_upper.n-he_lower.n*he_lower.n),3.); */
			/*lint +e790 integral to float */
		}
		/* Calculate Lyman alphas.	*/
		else if ( he_lower.n == 1 )
		{
			*Aul = (1.52e10) / POW3((double)he_upper.n);
				
			if ( ipZ > ipHELIUM )
				/* Additional terms for higher Z.  In theory, this Z dependence
				 * should be Z^4, but values from TOPbase suggest 3.9 is a more
				 * accurate exponent.	*/
				*Aul *= 0.875 * pow((double)ipZ, 3.9) ;
		}
		/* All remaining same-n transitions.	*/
		else if ( (he_lower.n == he_upper.n) )
		{
			/* These are 2^3Pj to 2s3S	*/
			/* These are fits to (low Z) Porquet & Dubau (2000) & 
				(high Z) NIST Atomic Spectra Database.	*/
			if ( ipLo == ipHe2s3S )
			{
				
				if (ipHi == ipHe2p3P0)
					*Aul = 3.31E7 + 1.13E6 * pow((double)ipZ+1.,1.76);
				else if (ipHi == ipHe2p3P1)
					*Aul = 2.73E7 + 1.31E6 * pow((double)ipZ+1.,1.76);
				else if (ipHi == ipHe2p3P2)
					*Aul = 3.68E7 + 1.04E7 * exp(((double)ipZ+1.)/5.29);
				else
				{
					fprintf(ioQQQ,"WHOA!  Stop in Helike.c");
					exit(0);
				}
			}
			/* Fit to data from TOPbase.	*/
			else if ( ( ipLo == ipHe2s1S ) && ( ipHi == ipHe2p1P) )
				*Aul = 5.53E6 * exp( 0.171*(ipZ+1.) );
			else 
			{
				double arg;
				/* Triplet P to triplet S, delta n = 0	*/
				if ( (he_upper.l == 1) && (he_upper.s == 1) && (he_lower.l == 0) && (he_lower.s == 1))
				{
					arg = 2008. + 60.41 * pow((double)ipZ,1.5) ;
					*Aul = POW2( arg ) / pow((double)he_upper.n/3., 5.11);
				}
				/* Singlet P to singlet D, delta n = 0	*/
				else if ( (he_upper.l == 1) && (he_upper.s == 0) && (he_lower.l == 2) && (he_lower.s == 0))
					*Aul = 5.5E7 * pow((double)ipZ,4.) / pow((double)he_upper.n, 5.1);
				/* Singlet P to singlet S, delta n = 0	*/
				else if ( (he_upper.l == 1) && (he_upper.s == 0) && (he_lower.l == 0) )
				{
					arg = 1102. + 42.85 * pow((double)ipZ,1.5) ;
					*Aul = POW2( arg ) / pow((double)he_upper.n/3., 4.93);
				}
				
				else /*if ( (he_upper.l == 2) && (he_upper.s == 1)  && (he_lower.l == 1) ) */
				{
					*Aul = 3.9E6 * pow((double)ipZ,4.) / pow((double)he_upper.n, 4.9);
					if ( (he_upper.l >2) || (he_lower.l > 2) )
						*Aul *= (he_upper.l/2.);
					if (he_lower.l > 2)
						*Aul *= (1./9.);
				}
			}
			assert( *Aul > 0.);
		}
		/* These transitions are of great importance, but the below radial integral 
		 * routine fails to achieve desirable accuracy, so these are fits as produced 
		 * from He A's for nupper through 9.	*/
		else if ( ( ipLo == ipHe2s3S ) || ( ipLo == ipHe3s3S ) )
		{
			if ( he_lower.n == 2 )
				*Aul = 5.181E8 * pow((double)ipZ,4.) / pow((double)he_upper.n, 2.88);
			else
				*Aul = 1.568E8 * pow((double)ipZ,4.) / pow((double)he_upper.n, 2.88);
		}
		/* Every other allowed transition is calculated as follows.	*/
		else
		{
			/* Calculate the radial integral from the quantum defects.	*/
			RI2 = scqdri(Eff_nupper,he_upper.l,Eff_nlower,he_lower.l,(double)(ipZ));
			/* Convert radial integral to Aul.	*/
			*Aul = ritoa(he_upper.l,he_lower.l,ipZ,Enerwn,RI2);
			/* Small routine for corrections to higher nonhydrogenic transitions.	*/
			*Aul *= correct2( ipHi, ipLo );
			/* radial integral routine does not recognize hyperfine structure.
			 * Here we split 2^3P.	*/
			if ( ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) ) && (*Aul > 1e-20) )
			{
				*Aul *= (2.*(ipLo-3)+1.0)/9.0;
			}
			assert( *Aul > 0. );
		}
	}
	/* Transition is neither allowed nor originating at n=2.	*/
	else
	{
		/* if we can do nothing, return 1e-20 */
		*Aul = helike.SmallA;
	}

	/* negative energy for a transition with substantial transition probability
	 * would be major logical error - but is ok for same-n l transitions */
	if( Enerwn < 0. && *Aul > helike.SmallA )
	{
		fprintf( ioQQQ," he_1trans hit negative energy, ipZ=%li, val was %f \n", ipZ ,Enerwn );
	}

	return;
}


/* REMOVED FUNCTION "org_assign"..."he_assign" is improved version.	*/

/* Function he_assign assigns, in order of increasing energy, 				*/
/* quantum numbers n,l, and s to each element of vector he_levels.			*/
/* nmax is the upper quantum number, there will be many l levels within it.	*/
/* ndim is dimension of vector, used to confirm that size not exceeded		*/ 
static void he_assign( long nmax , long ndim )
{
	/*	in, il, and is are dummies for n, l, and s.		*/
	/*	i is the index in the vector he_levels.			*/
	long in, il, is, ij, i = 0;
	for ( in = 1L; in <= nmax ; ++in )
	{
		for ( il = 0L; il < in; ++il )
		{
			for ( is = 1L; is >= 0 ; --is )
			{
				/* All levels except singlet P follow	*/
				/* the ordering scheme:			*/
				/*	lower l's have lower energy	*/
				/* 	triplets have lower energy	*/
				if ( (il == 1L) && (is == 0L) )
					continue;
				/* Ground state is not triplicate.	*/
				if ( (in == 1L) && (is == 1L) )
					continue;
				ij = 0;
				do 
				{
					he_levels[i].n = in;
					he_levels[i].s = is;
					he_levels[i].l = il;
					++i;
					++ij;
				}	while ( (in == 2) && (il == 1) && (is == 1) && (ij < 3) );
			}
		}
		/*	Insert singlet P at the end of	*/
		/*	every sequence for a given n.	*/
		if ( in > 1L )
		{
			he_levels[i].n = in;
			he_levels[i].s = 0L;
			he_levels[i].l = 1L;
			++i;
		}
	}

	/* confirm that we did not overrun the array */
	assert( i < ndim );

	return;
}

/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * ipZ is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
double He_cross_section( double EgammaRyd , long ipLevel , long ipZ )
{
	double cs ;
	
	globalZ = ipZ;

	ipLev = ipLevel;

	EthRyd = iso.xIsoLevNIonRyd[ipHELIUM][ipZ][ipLev];
	
	/* Threshold energy in eV.
	EthEV = EthRyd * EVRYD;	*/
		
	/* 01 dec 23, change xIsoLevNIonRyd to double, removed following double cast */
	cs = (1.e-18)*cross_section( /*(double)(float)*/EgammaRyd );
	
	return cs;
}

static double cross_section(double EgammaRyd)
{
	double E0[29] = {
	1.36E+01,2.01E+01,1.76E+01,3.34E+01,4.62E+01,6.94E+01,
	8.71E+01,1.13E+02,1.59E+02,2.27E+02,2.04E+02,2.74E+02,
	2.75E+02,3.38E+02,4.39E+02,4.17E+02,4.47E+02,5.18E+02,
	6.30E+02,6.27E+02,8.66E+02,7.67E+02,9.70E+02,9.66E+02,
	1.06E+03,1.25E+03,1.35E+03,1.43E+03,1.56E+03};
	double sigma[29] = {
	9.49E+02,3.20E+02,5.46E+02,2.85E+02,2.34E+02,1.52E+02,
	1.33E+02,1.04E+02,6.70E+01,4.00E+01,6.14E+01,4.04E+01,
	4.75E+01,3.65E+01,2.45E+01,3.14E+01,3.11E+01,2.59E+01,
	1.94E+01,2.18E+01,1.23E+01,1.76E+01,1.19E+01,1.31E+01,
	1.20E+01,9.05E+00,8.38E+00,8.06E+00,7.17E+00};
	double y_a[29] = {
	1.47E+00,7.39E+00,1.72E+01,2.16E+01,2.18E+01,2.63E+01,
	2.54E+01,2.66E+01,3.35E+01,5.32E+01,2.78E+01,3.57E+01,
	2.85E+01,3.25E+01,4.41E+01,3.16E+01,3.04E+01,3.28E+01,
	3.92E+01,3.45E+01,5.89E+01,3.88E+01,5.35E+01,4.83E+01,
	5.77E+01,6.79E+01,7.43E+01,7.91E+01,9.10E+01};
	double P[29] = {
	3.19E+00,2.92E+00,3.16E+00,2.62E+00,2.58E+00,2.32E+00,
	2.34E+00,2.26E+00,2.00E+00,1.68E+00,2.16E+00,1.92E+00,
	2.14E+00,2.00E+00,1.77E+00,2.04E+00,2.09E+00,2.02E+00,
	1.86E+00,2.00E+00,1.62E+00,1.93E+00,1.70E+00,1.79E+00,
	1.72E+00,1.61E+00,1.59E+00,1.58E+00,1.54E+00};
	double y,F,x;
	double y_w[29] = 
	{2.039,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	/* the symbols y0 and y1 are used in the std header math.h and must not be redefined here */
	double yzero[29] =
	{0.4434,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	double yone[29] =
	{2.136,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	
	/* These numbers are for the calculation of S cross-sections for n = 2 to n = 5. */
	/* indices are [n][s][a,b]	*/
	double heliumparameters[5][2][2] = {
		{{0.024202455499820335,0.970207933907144639},
		{0.0959219658351617204,0.733724345317537174}},

		{{0.0141220876458141004,3.51765152401954129},
		{0.0628053897344138336,3.40478362746828901}},

		{{0.00910149155325056681,8.45451408475178406},
		{0.0441137758429214155,9.56574062306688065}},
		
		{{0.00595392674283822128,16.7057454415064428},
		{0.0322247737609894561,20.9337721928726302}},
	
		{{0.0039485621346665004,28.7620861314787974},
		{0.0249267534436805543,38.4646485400563079}}};

	/* These corrections ensure helium cross-sections are accurate at threshold.
	 * index is [ipLev - 1]	*/
	double Heliumcorrections[30] = {
	1.02E+00,9.90E-01,1.18E+00,1.18E+00,1.18E+00,1.00E+00,
	1.02E+00,9.70E-01,1.09E+00,1.03E+00,1.01E+00,1.04E+00,
	1.03E+00,9.70E-01,1.05E+00,1.03E+00,1.01E+00,1.00E+00,
	1.00E+00,1.06E+00,1.00E+00,1.01E+00,1.07E+00,1.00E+00,
	1.00E+00,1.00E+00,1.00E+00,1.00E+00,1.00E+00,1.08E+00};

	long n,l,s,ipZ;
	double pcs,Egamma,lnE,a,b,ejected_electron_energy;
	
	ipZ = globalZ;
	n = he_levels[ipLev].n;
	l = he_levels[ipLev].l;
	s = he_levels[ipLev].s;
	Egamma = EgammaRyd * EVRYD;

	/* From Verner et al 96.  All ground state cross-sections calculated here.	*/
	if (ipLev == ipHe1s1S)
	{
		x = Egamma/E0[ipZ-1] - yzero[ipZ-1];
		y = sqrt(x*x + yone[ipZ-1]*yone[ipZ-1]);
		F = ((x-1)*(x-1)+y_w[ipZ-1]*y_w[ipZ-1])
			* pow(y,0.5*P[ipZ-1]-5.5) * pow((1+sqrt(y/y_a[ipZ-1])),-P[ipZ-1]);
		pcs = sigma[ipZ-1]*F;
		assert( pcs > 0. );
		return pcs;
	}
	/* Due to the highly non-hydrogenic nature of S orbitals, both singlet and triplet,
	 * a special formula needs to be applied to them.  The hydrogenic formula is 
	 * unsatisfactory since its errors get larger with increasing n.	*/
	
	/* Fix a better one for n = 2 as well?	*/
	else if ( ( l == 0 ) && ( n > 2 ) && ( ipZ >= ipCARBON ) )
	{
		lnE = log(EgammaRyd * POW2(1.667*n/ipZ));
		pcs = POW2(5./(double)ipZ) * exp(1.790166293 - 0.127439504 * POW2(lnE) - 1.660569112*lnE);
		pcs *= pow(n/3.,1.365);
		assert( pcs > 0. );
	}
	/* In addion to the above, the S orbitals of helium need to be handled separately.	*/
	else if ( ( ipZ == ipHELIUM ) && ( l == 0 ) && ( n <= 6 ) )
	{
		a = heliumparameters[n-2][s][0];
		b = heliumparameters[n-2][s][1];
		pcs = 1./(a+b*POW2(EgammaRyd));
		if ( ipLev <= 30 )
			pcs *= Heliumcorrections[ipLev-1];
		/* Put E^-3 tail on helium at energies beyond first resonance.	*/
		if ( EgammaRyd > 2.75 + EthRyd)
		{
			pcs = 35.*POW3(1./EgammaRyd)/pow((double)n,3.4);
		}
		assert( pcs > 0. );
	}
	/* To everything else we apply a hydrogenic routine.	*/
	else 
	{
		/* Make sure energy passed to hydrogenic routine is
		 * at least SMALLFLOAT above hydrogenic threshold, as 
		 * required by H_photo_cs().		*/
		ejected_electron_energy = EgammaRyd - EthRyd + SMALLFLOAT;
		
		assert( ejected_electron_energy > 0. );

		pcs = (1.e18)*H_photo_cs(ejected_electron_energy, n, l, ipZ);
		
		/* Rescale helium to expected threshold values.	*/
		if ( ( ipZ == ipHELIUM ) && ( ipLev < 31 ) )
			pcs *= Heliumcorrections[ipLev-1];
		
		assert( pcs > 0. );
	}
	
	return pcs;
}

static double radrecomb(double temp, long ipZ, long ipLo)
{
	double alpha,RecomIntegral=0.,b,E1,E2,step;
	long s,l/*,n*/;
		
	/*n = he_levels[ipLo].n;*/
	s = he_levels[ipLo].s;
	l = he_levels[ipLo].l;

#if 0
	/* The following are coefficients and temperature exponents 
	 * from Benjamin et al. 99, Table 1	*/
	/* We do not use these because of the small temperature range
	 * in which they are expected to be valid.	*/
	double BenRadRecHe[31] = {
		1.54E-13,1.49E-14,5.55E-15,5.61E-14,5.61E-14,5.61E-14,
		1.26E-14,3.72E-15,1.62E-15,1.95E-14,1.33E-14,4.31E-15,
		5.01E-15,1.50E-15,7.00E-16,9.04E-15,8.29E-15,2.67E-15,
		4.15E-15,1.38E-15,2.43E-15,7.51E-16,3.66E-16,4.84E-15,
		4.99E-15,1.60E-15,3.65E-15,1.22E-15,1.34E-15,4.46E-16,
		1.35E-15};

	double BenRadHeExp[31] = {
		0.486,0.381,0.451,0.639,0.639,0.639,0.695,0.344,0.444,
		0.632,0.868,0.872,0.700,0.328,0.445,0.632,0.867,0.871,
		1.046,1.046,0.708,0.327,0.454,0.636,0.870,0.873,1.048,
		1.048,1.183,1.183,0.718};

	if ( ( ipZ == ipHELIUM ) && (ipLo < 31) )
	{
		/* Get number from Benjamin et al. */
		alpha = BenRadRecHe[ipLo] * pow(1E+04/temp,BenRadHeExp[ipLo]);

	}
#endif	
	
	/* Factors outside integral in Milne relation.	*/
	 b = 4.12373E+11 * (2.*l+1.) * (2.*s+1.) * pow(temp,-1.5) / 4;
	/* kT in Rydbergs.	*/
	kTRyd = 8.64E-5 * temp / EVRYD;
	globalZ = ipZ;
	ipLev = ipLo;

	/* Begin integration.	*/
	/* First define characteristic step as the minimum of kTRyd and 3EthRyd.	*/
	E1 = EthRyd;
	step = MIN( kTRyd, 3.*E1 );
	E2 = E1 + step;
	/* Perform initial integration, from threshold to threshold + step.	*/
	RecomIntegral = qg32( E1, E2, RecomInt);
	/* Repeat the integration, adding each new result to the total, 
	 * except that the step size is tripled every time, since values away from 
	 * threshold tend to fall off more slowly.	*/
	do
	{
		E1 = E2;
		step *= 2;
		E2 = E1 + step;
		RecomIntegral += qg32( E1, E2, RecomInt);
	/* Continue integration until the upper limit exceeds 10kTRyd, ar arbitrary
	 * point at which the integrand component exp(electron energy/kT) is very small,
	 * making neglible cross-sections at photon energies beyond that point. */
	} while ( E2 < 10.*kTRyd );

	/* Calculate recombination coefficient.	*/
	alpha = b * RecomIntegral;

	if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
	{
		/* Split 2 trip P term.	*/
		alpha *= (2.*(ipLo-3L)+1.) / 9.;
	}

	assert( alpha > SMALLFLOAT );

	return alpha;
}

/* For RecomInt, the energy is photon Rydbergs.	*/
static double RecomInt(double ERyd)
{
	double x1;

	/* Milne relation integrand	*/
	x1 = ERyd * ERyd * exp(-1.0 * ( ERyd - EthRyd ) / kTRyd);
	x1 *= He_cross_section( ERyd , ipLev, globalZ );
		
	return x1;
}

#if 0

static double radcool(double temp, long ipZ, long ipLo)
{
	double beta,CoolingIntegral=0.,b;
	long s,l,n;
		
	n = he_levels[ipLo].n;
	s = he_levels[ipLo].s;
	l = he_levels[ipLo].l;

	/* Factors outside integral	*/
	 b = 4.12373E+11 * (2.*l+1.) * (2.*s+1.) * pow(temp,-2.5) / 4;
	 b *= 157890.;
	/* Integration is performed for electron energies of 0 to 10 kT, but is split
	 * into two separate integrations for more accurate results.	*/
	kTRyd = 8.64E-5 * temp / EVRYD;
	globalZ = ipZ;
	ipLev = ipLo;

	/* Begin integration.	*/
	E1 = EthRyd;
	step = MIN( kTRyd, 3.*E1 );
	E2 = E1 + step;
	CoolingIntegral = qg32( E1, E2, CoolInt);
	do
	{
		E1 = E2;
		step *= 3;
		E2 = E1 + step;
		CoolingIntegral += qg32( E1, E2, CoolInt);
	} while ( E2 < 10.*kTRyd );

	/* Calculate cooling coefficient.	*/
	beta = (1.E-18) * b * CoolingIntegral;

	if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
	{
		beta *= (2.*(ipLo-3L)+1.) / 9.;
	}
	
	assert (beta > SMALLFLOAT);

	return beta;
}

/* For CoolInt, the energy is photon Rydbergs.	*/
static double CoolInt(double ERyd)
{
	double x2;

	/* Milne relation integrand	*/
	x2 = ( ERyd - EthRyd ) * ERyd * ERyd * exp(-1.0 * ( ERyd - EthRyd ) / kTRyd);
	x2 *= cross_section(ERyd);
		
	return x2;
}

#endif

static void printstuff(long flag)
{
	long i,ipZ,ipLo,ipHi;
	int lgHeader = 0;
	double thresh_energy/*,n_eff*/;

	
	switch(flag)
	{
	case 1:
		ofp = fopen("HelikeAs.txt","w");
		fprintf(ofp," Z\tni^JL\tnf^JL\tipLo\tipHi\tA\n");
		break;
	case 2:
		ofp = fopen("HelikefAs.txt","w");
		fprintf(ofp,"FORBIDDEN	TRANSITIONS	to 1^1S\n");
		fprintf(ofp,"Z\t2^3S\t2^1S\t2^3P1\n");
		break;
	case 3:
		ofp = fopen("HelikeEs.txt","w");
		break;
	case 4:
		ofp = fopen("HelikeThPCS.txt","w");
		break;
	case 5:
		ofp = fopen("HelikeRR.txt","w");
		break;
	case 6:
		ofp = fopen("HelikePCSgrid.txt","w");
		break;
	default:
		return;
	}
	
	for (i=1; i<N_HE_TE_RECOMB; i++)
	{
		if (flag == 5)
		{
			fprintf(ofp,"temp= %2.1f \n",0.5*(i+1.));
		}
		else if (lgHeader)
		{
			return;
		}
		if ( (flag > 2) && (flag < 6) && (!lgHeader) )
		{
			fprintf(ofp,"Z   ");
			for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][1]; ipLo++ )
			{
				fprintf(ofp,"n%1lds%1ldl%1ld      ",he_levels[ipLo].n,he_levels[ipLo].s,he_levels[ipLo].l);
			}
			fprintf(ofp,"\n");
		}
		lgHeader = TRUE;

		for( ipZ=ipHELIUM; ipZ < LIMELM; ipZ++ )
		{
			if( ipZ == ipHELIUM || abundances.lgElmtOn[ipZ] ) 
			{
				if ( (flag > 1) && (flag < 6) )
				{
					fprintf(ofp,"%2ld ",ipZ+1L);
				}
				
				globalZ = ipZ;
				
				for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipZ]; ipLo++ )
				{
					if (ipLo == ipHe1s1S)
					{
						thresh_energy = EionRYD[ipZ-1L];
					}
					else
					{
						thresh_energy = EionRYD[ipZ-1L] - EmisLines[ipHELIUM][ipZ][ipLo][0].EnergyRyd;
					}
					
					/*n_eff = ipZ * pow(thresh_energy, -0.5);*/
					ipLev = ipLo;
					EthRyd = thresh_energy;
					/*EthEV = EthRyd * EVRYD;*/

					switch(flag)
					{
					case 1:
						for( ipHi=ipLo+1L; ipHi < iso.nLevels[ipHELIUM][ipZ]; ipHi++)
						{
							if (EmisLines[ipHELIUM][ipZ][ipHi][ipLo].Aul > (1e-20) )
							{
								fprintf(ofp,"%2ld\t%2ld^%1ld",ipZ+1L,he_levels[ipLo].n,2*(he_levels[ipLo].s)+1);
								orbit(ipLo);
								fprintf(ofp,"\t%2ld^%1ld",he_levels[ipHi].n,2*(he_levels[ipHi].s)+1);
								orbit(ipHi);
								fprintf(ofp,"\t%3ld\t%3ld\t%2.2e\n",ipLo,ipHi,EmisLines[ipHELIUM][ipZ][ipHi][ipLo].Aul);
							}
						}
						break;
					case 2:
						if( (ipLo == ipHe2s3S) || (ipLo == ipHe2s1S) || (ipLo == ipHe2p3P1) )
						{
							fprintf(ofp,"\t%2.2e",EmisLines[ipHELIUM][ipZ][ipLo][0].Aul);
						}
						break;
					case 3:
						fprintf(ofp,"%2.4e ",thresh_energy);
						break;
					case 4:
						fprintf(ofp,"%2.4e ",cross_section(thresh_energy)); 
						break;
					case 5:
						fprintf(ofp,"%2.4e ",pow(10.,RRCoef[ipZ][ipLo][i]));
						break;
					case 6:
						fprintf(ofp,"Z=%ld  %2ld^%1ld",ipZ+1,he_levels[ipLo].n,2*(he_levels[ipLo].s)+1);
						orbit(ipLo);
						fprintf(ofp,"\n");
						for (i=0; i<100; i++)
						{
							fprintf(ofp,"%2.2e %2.2e\n",thresh_energy*(1+0.1*pow((double)i,1.5)),
									(1E-18)*cross_section(thresh_energy*(1+0.1*i)));
						}
						break;
					default:
						fprintf(ofp,"\n");
					}
				}
				fprintf(ofp,"\n");
			}
		}
	}
	fclose(ofp);
	return;
}

void He1Autoionize(void)
{

/* Autoionizing levels are as follows:	*/
	const int ipHe2s2p1P = 0;
/*	const int ipHe2s2p3P = 1;	*/
/*	const int ipHe2p2_1D = 2;	*/
 	const int ipHe2p2_3P = 3;		

	double a[4] = {75827., 72116., 74853., 74010.};
	double b[4] = {1.8956, 1.9084,    1.9, 1.9021};

	long ipZ, ipHi;

	for( ipZ = ipHELIUM; ipZ < LIMELM; ipZ++ )
	{
		/* only set elements that are turned on */
		if( ipZ == ipHELIUM || abundances.lgElmtOn[ipZ] )
		{
			for ( ipHi = ipHe2s2p1P; ipHi <= ipHe2p2_3P; ipHi++ )
			{
				He1AutoLines[ipZ][ipHi].EnergyWN = (float)(a[ipHi]*pow((double)(ipZ+1),b[ipHi]));
			}
		}
	}

	return;
		
}

static double ritoa(long li, long lf, long ipZ, double k, double RI2)
{  
	/*	Variables are as follows:				*/
	/*	lg = larger of li and lf				*/
	/*	fmean = mean oscillator strength		*/
	/*		for a given level.					*/
	/*	mu = reduced mass of optical electron.	*/
	/*	EinsteinA = Einstein emission coef.		*/
	/*	w = angular frequency of transition.	*/
	/*	RI2_cm = square of rad. int. in cm^2.	*/
	/*	A = mass of ion in atomic mass units.	*/
	long lg;
	double fmean,mu,EinsteinA,w,RI2_cm;
	double A[30] =
	{1.00,4.0026,6.941,9.01218,10.81,
		12.011,14.0067,15.9994,18.998403,20.179,
		22.98977,24.305,26.98154,28.0855,30.97376,
		32.06,35.453,39.948,39.0983,40.08,
		44.9559,47.9,50.9415,51.996,54.938,
		55.847,58.9332,58.7,63.546,65.38};

	mu = ELECTRON_MASS/(1+ELECTRON_MASS/(A[ipZ]*ATOMIC_MASS_UNIT));

	/* 8065.541 converts cm-1 to eV 	*/
	/* hbar then converts to rad/sec 	*/

	w = k /(8065.541*hbar);

	RI2_cm = RI2 * (2.8002856e-17);

	lg = (lf > li) ? lf : li;

	fmean = 2.0*mu*w*lg*RI2_cm/((3.0*H_BAR) * (2.0*li+1.0));

	/* fmean = -(2/3)*(mu/hbar)*w*(lg/(2*li+1))*RI		*/

	EinsteinA = (.66702)*k*k*fmean;

	assert( EinsteinA > SMALLFLOAT );

	return EinsteinA;

}

#undef DEBUG 

/* create he-like series, called by ContCreatePointers */
void HeCreate(void)
{
	double **energies, *n_effs;

	double **RobbinsP, **RobbinsC;
	long int nmax, nl , nhedim;
	long int ipLo ,
		ipHi,
		nelem,
		need;

	static int nCalled = 0;

#	define chLine_LENGTH 1000
	char chLine[chLine_LENGTH] , 
		/* this must be longer than chDataPath */
		chFilename[210];

	FILE *ioDATA;
	long int i, i1, i2, i3 ;
	int lgEOL,
		lgHIT;
	long int j;

	char chSpin[2]={'1','3'};
	char chL[6]={'S','P','D','F','G','H'};

#	ifdef DEBUG_FUN
	fputs( "<+>HeCreate()\n", debug_fp );
#	endif

	/* now MALLOC the space for the helium-like lines, 
	 * but this must be done exactly one time per coreload */
	if( nCalled > 0 )
	{
#		ifdef DEBUG_FUN
		fputs( " <->HeCreate()\n", debug_fp );
#		endif
		return;
	}
	++nCalled;

	/*double ***RRCoef[LIMELM][MaxLevels][N_HE_TE_RECOMB]*/
	if( (RRCoef = (double ***)MALLOC(sizeof(double **)*(unsigned)LIMELM ) )==NULL )
		bad_malloc();

	/*char ***helike chLevel[lev][lev][str]*/
	if( (helike.chLevel = (char **)MALLOC(sizeof(char *)*(unsigned)(iso.nLevels[ipHELIUM][ipHELIUM]) ) )==NULL )
		bad_malloc();

	if( (RobbinsP = 
		(double**)MALLOC(sizeof(double*)*(unsigned)(iso.nLevels[ipHELIUM][ipHELIUM]) ))==NULL )
		bad_malloc();

	if( (RobbinsC = 
		(double**)MALLOC(sizeof(double*)*(unsigned)(iso.nLevels[ipHELIUM][ipHELIUM]) ))==NULL )
		bad_malloc();

	if( (energies = (double**)MALLOC(sizeof(double*)*(unsigned)LIMELM ) )==NULL )
		bad_malloc();

	for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipHELIUM] ;++ipLo )
	{
		if( (helike.chLevel[ipLo] = 
			(char*)MALLOC(sizeof(char)*(unsigned)10 ))==NULL )
			bad_malloc();
	}

	for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipHELIUM] ;++ipLo )
	{
		if( (RobbinsP[ipLo] = 
			(double*)MALLOC(sizeof(double)*(unsigned)(ipLo+1) ))==NULL )
			bad_malloc();

		if( (RobbinsC[ipLo] = 
			(double*)MALLOC(sizeof(double)*(unsigned)(ipLo+1) ))==NULL )
    		bad_malloc();
	}
	
	for( nelem=ipHELIUM; nelem < LIMELM; ++nelem )
	{
		/* must always have at least N_HE_LEV_RECOMB levels since that is
		 * number that will be read in from he rec data file, but possible to
		 * require more 
		 * N_HE_LEV_RECOMB corresponds to n = 20 */
		long int MaxLevels = MAX2( N_HE_LEV_RECOMB , iso.nLevels[ipHELIUM][nelem] );

		/* always define this */
		/* >>chng 02 jan 24, RRCoef will be iso.nLevels[ipHELIUM][nelem] levels, no iso.nLevels,
		 * code will stop if more than this is requested */
		if( (RRCoef[nelem] = 
			(double**)MALLOC(sizeof(double*)*(unsigned)(MaxLevels) ))==NULL )
			bad_malloc();
			
		for( ipLo=ipHe1s1S; ipLo < MaxLevels ;++ipLo )
		{
			if( (RRCoef[nelem][ipLo] = 
				(double*)MALLOC(sizeof(double)*(unsigned)N_HE_TE_RECOMB ))==NULL )
				bad_malloc();
		}
		
		if( nelem == ipHELIUM || abundances.lgElmtOn[nelem] )
		{
			if( (energies[nelem] = 
				(double*)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ))==NULL )
				bad_malloc();
		}
	}

	/* first find the largest number of l states in any he-like ion */
	/* main helium-like arrays, fill with sane values */
	need = 0;
	for( nelem=ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* only check elements that are turned on */
		if( nelem == ipHELIUM || abundances.lgElmtOn[nelem] )
		{
			need = MAX2( need, iso.nLevels[ipHELIUM][nelem] );
	    }
	}
	
	/* we know the number of l levels, now find number of n to pass
	 * to routine */
	nl = 1;
	/* the max number of n levels that will be needed */
	nmax = 1;
	/* count number of levels, remembering that there are three autoionizing levels */
	while( nl <= need )
	{
		++nmax;
		nl += 2*nmax;
		/* Must make space for two more levels...splitting of 2^3P.	*/
		if (nmax == 2)
			nl += 2;
	}
	nhedim = 2* nl;

	/* Allocate space so that vector he_levels contains	*/
	/* nhedim number of elements. */
	if ( (he_levels= (Elevels*)MALLOC((unsigned)nhedim*sizeof(Elevels)) ) == NULL )
		bad_malloc();

	if ( (n_effs= (double *)MALLOC((unsigned)need*sizeof(double)) ) == NULL )
		bad_malloc();

	/* Function he_assign assigns, in order of increasing energy, 
	 * quantum numbers n,l,s, and j to each element of vector he_levels.
	 * first arg is number of n levels, sec is dim of structure he_levels (with type Elevels)
	 * in MALLOC above */ 
	he_assign( nmax , nhedim );
	{
		/* option to print particulars of some line when called
		 * a prettier print statement is near where chSpin is defined below*/
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{
			/* first print internal representation */
			for( i=0; i<need; ++i )
			{
				fprintf(ioQQQ,"i %li n %li s %li l %li\n", 
					i, 
					he_levels[i].n,
					he_levels[i].s,
					he_levels[i].l );
			}
			exit(0);
		}
	}

	/* main helium-like arrays, fill with sane values */
	for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
	{
		double Atot;

		/* must always do helium even if turned off */
		if( nelem == ipHELIUM || abundances.lgElmtOn[nelem] ) 
		{
			/* loop to obtain energies for current ion*/
			for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][nelem]; ipLo++ )
			{
				/* the effective quantum number */
				n_effs[ipLo] = he_levels[ipLo].n - defect(nelem,ipLo);
				
				/* energies (in wavenumbers) that correspond to quantum defect */
				/* Note: energies[] will be valid until next interation of charge loop.	*/
				if (ipLo>ipHe1s1S)
					energies[nelem][ipLo] = he_energy(n_effs[ipLo],nelem,ipLo);
				else
					energies[nelem][ipLo] = EionWN[nelem-1];
				
				/* convert wavenumbers to Ryd.	*/
				EthRyd = energies[nelem][ipLo] * WAVNRYD;
				
				/* Store threshold energy in larger cloudy array	*/
				iso.xIsoLevNIonRyd[ipHELIUM][nelem][ipLo] = EthRyd;
				assert( iso.xIsoLevNIonRyd[ipHELIUM][nelem][ipLo] > 0. );
			}

			RobbinsC[ipHe1s1S][ipHe1s1S] = 1.;
			RobbinsP[ipHe1s1S][ipHe1s1S] = 0.;
			
			/* PRODUCE RECOMBINATION COEFFICIENTS, Aul's, STAT. WEIGHTS, etc
			 * These loops must be nested in this order!
			 * HeAsIndex is an index to retrieve Benjamin et al 98 A's from a table.
			 * HeFsIndex is an index to retrieve Kono & Hattori 98 F's from a table.	*/
			for( ipHi=ipHe2s3S; ipHi<iso.nLevels[ipHELIUM][nelem] ; ipHi++ )
			{
				HeAsIndex = 0;
				HeFsIndex = 0;
				Atot = 0;
				RobbinsC[ipHi][ipHi] = 1.;
				RobbinsP[ipHi][ipHi] = 0.;

				for( ipLo=ipHe1s1S; ipLo<ipHi ; ipLo++ )
				{
					double Enerwn, Aul;
					float gHi , gLo;

					RobbinsC[ipHi][ipLo] = 0.;
					RobbinsP[ipHi][ipLo] = 0.;
			
					/* pointers to transitions, ipLo, ipHi
					 * actual quantum numbers, nLo nHi
					 * atomic number or charge and stage: */
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].nelem = (int)(nelem+1);
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].IonStg = (int)(nelem);

					/* stat. weights, s is 0 (singlets) or 1 (triplets)   */
					/* the stat. weights must be computed differently
					 * for the 2^3P levels.	*/
					if ( ipHi == ipHe2p3P0 )
					{
						gHi = 1.;
					}
					else if ( ipHi == ipHe2p3P1 )
					{
						gHi = 3.;
					}
					else if ( ipHi == ipHe2p3P2 )
					{
						gHi = 5.;
					}
					else
					{
						gHi = (2.f*he_levels[ipHi].l + 1.f)* (2.f*he_levels[ipHi].s + 1.f);
					}

					if ( ipLo == ipHe2p3P0 )
					{
						gLo = 1.;
					}
					else if ( ipLo == ipHe2p3P1 )
					{
						gLo = 3.;
					}
					else if ( ipLo == ipHe2p3P2 )
					{
						gLo = 5.;
					}
					else
					{
						gLo = (2.f*he_levels[ipLo].l + 1.f)* (2.f*he_levels[ipLo].s + 1.f);
					}

					EmisLines[ipHELIUM][nelem][ipHi][ipLo].gLo = gLo;
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].gHi = gHi;
					/* save statistical weights */
					iso.stat[ipHELIUM][ipLo] = gLo;
					iso.stat[ipHELIUM][ipHi] = gHi;
					assert( EmisLines[ipHELIUM][nelem][ipHi][ipLo].gHi> 0.);
					assert( EmisLines[ipHELIUM][nelem][ipHi][ipLo].gLo> 0.);

					/* Energies are w.r.t the continuum, so must subtract 
					 * upper lever from lower level.	*/
					Enerwn = energies[nelem][ipLo] - energies[nelem][ipHi];
					/* if both of these levels are hydrogenic and the same n,
					 * their energies will be exactly the same, the Enerwn zero */
					/* Set this difference to 1 wavenumber.	*/
					if ( Enerwn == 0 ) 
						Enerwn += 1.;
						
					/* Passed to function he_1trans are the upper	*/
					/* and lower levels	of a transition.  			*/
					/* Aul is calculated.							*/
					/* Changed he_1trans so that the line energy and effective principal
					 * quantum numbers are already available...thus *energies and *n_effs
					 * no longer need to be global.	*/
					he_1trans( nelem , Enerwn , &Aul , ipHi , ipLo, n_effs[ipHi], n_effs[ipLo] );
					
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul = (float)(Aul);
					assert(EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul > 0.);
					
					if ( (he_levels[ipLo].s == he_levels[ipHi].s) && 
						(abs((int)(he_levels[ipLo].l - he_levels[ipHi].l)) == 1) )
						Atot += Aul;
					
					/* the energy of the transition in wavenumbers, this can be negative
					 * for highly forbidden upper levels where energy switches */
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN = (float)Enerwn;
					
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyRyd = 
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN* (float)WAVNRYD;

					EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyErg = 
						(float)(EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyRyd*
					  EN1RYD);

					EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyK = 
						(float)(EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyRyd*
					  TE1RYD );

					/* there are some negative energy transitions, where the order
					 * has changed, but these are not optically allowed, these are
					 * same n, different L, forbidden transitions */
					assert( EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN > 0. ||
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul <= 1.0e-20f );

					/* very forbidden transitions may have negative energies */
					if( EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul < 1e-20f ||
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN <= 0.)
					{
						
						/* this branch - bogus line */
						/* transition energy in various units:*/

						/* take abs value since some have order switched */
						/* make following an air wavelength */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].WLAng = 1e6f;

						EmisLines[ipHELIUM][nelem][ipHi][ipLo].gf = 1e-20f;

						/* derive the abs coef, call to function is gf, wl (A), g_low */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].opacity = 0.;
					}
					else
					{

						/* transition energy in various units:*/

						/* take abs value since some have order switched */
						/* make following an air wavelength */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].WLAng = 
							(float)fabs(1.0e8/
						  EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN/
						  RefIndex(&EmisLines[ipHELIUM][nelem][ipHi][ipLo]));
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].gf = 
							(float)(GetGF(EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul,
						  EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN,
						  iso.stat[ipHELIUM][ipHi]));
						assert(EmisLines[ipHELIUM][nelem][ipHi][ipLo].gf > 0.);

						/* derive the abs coef, call to function is gf, wl (A), g_low */
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].opacity = 
							(float)(abscf(EmisLines[ipHELIUM][nelem][ipHi][ipLo].gf,
						  EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN,
						  iso.stat[ipHELIUM][ipLo]));
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].opacity = MAX2(0.f, EmisLines[ipHELIUM][nelem][ipHi][ipLo].opacity );
					}
					{
						/* option to print particulars of some line when called
						 * a prettier print statement is near where chSpin is defined below
						 * search for "pretty print" */
						/*@-redef@*/
						enum {DEBUG=FALSE};
						/*@+redef@*/
						if( DEBUG  )
						{
							if( (nelem == ipHELIUM) && (EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul> 1e-19) )
							{
								fprintf(ioQQQ,"Z %li Lo %li n %li s %li l %li \t ", 
									nelem+1,
									ipLo, 
									he_levels[ipLo].n,
									he_levels[ipLo].s,
									he_levels[ipLo].l );
								fprintf(ioQQQ," Hi %li n %li s %li l %li \t", 
									ipHi, 
									he_levels[ipHi].n,
									he_levels[ipHi].s,
									he_levels[ipHi].l );
								fprintf(ioQQQ,"%.4e\t%.4e\n",
									/*EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN,*/
									1.e8/EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN,
									EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul);
							}
						}
					}

					/* redistribution function = partial or complete? */
					if( ipLo == ipHe1s1S )
					{
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].iRedisFun = ipPRD;
					}
					else
					{
						/* >>chng 01 feb 27, had been -1, crd with core only,
						 * change to crd with wings as per discussion with Ivan Hubeny */
						/*EmisLines[ipHELIUM][nelem][ipHi][ipLo].iRedisFun = -1;*/
						EmisLines[ipHELIUM][nelem][ipHi][ipLo].iRedisFun = ipCRDW;
					}
					/* create array index that will blow up */
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].ipCont = INT_MIN;
					iso.Pop2Ion[ipHELIUM][nelem][ipLo] = 0.;

					EmisLines[ipHELIUM][nelem][ipHi][ipLo].damprel = 1e-30f;
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].damp = 1e-30f;
				}

				for( ipLo=ipHe1s1S; ipLo<ipHi ; ipLo++ )
				{
					if ( (Atot > 0.) && (he_levels[ipLo].s == he_levels[ipHi].s) && 
						(abs((int)(he_levels[ipLo].l - he_levels[ipHi].l)) == 1) )
						RobbinsP[ipHi][ipLo] = (double)EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul/Atot;
					else 
						RobbinsP[ipHi][ipLo] = 0.;
				}
				
				for( ipLo=ipHe1s1S; ipLo<=ipHi ; ipLo++ )
				{
					for ( i=ipLo /*+ 1*/; i<=ipHi; i++ )
					{
						RobbinsC[ipHi][ipLo] += RobbinsP[ipHi][i] * RobbinsC[i][ipLo];
					}
					
					/*if ( (nelem == ipHELIUM) && (he_levels[ipHi].l == 1 ) && (he_levels[ipHi].s == 1 ) )
						fprintf(ioQQQ,"ipHi %ld\tipLo %ld\tC %2.4e\tP %2.4e\n",ipHi,ipLo,
							RobbinsC[ipHi][ipLo],RobbinsP[ipHi][ipLo]);*/
				}
			}

#if	0
			if (nelem == ipHELIUM )
			{
				/* This bit is for the calculation of conversion probabilities, as in Robbins68b, Table 1.
				 * To output Bm(n,l; ipLo), set ipLo, lo_l, and lo_s accordingly.	*/
				long i,j;
				long hi_l,hi_s;
				double Bm;
				
				hi_l = 1;
				hi_s = 1;
				ipLo = 1;
				assert( hi_l != he_levels[ipLo].l );
				
				fprintf(ioQQQ,"Bm(n,%ld)\n",ipLo);
				fprintf(ioQQQ,"m\t2\t\t3\t\t4\t\t5\t\t6\n");

				for( ipHi=ipHe2p3P2; ipHi<iso.nLevels[ipHELIUM][nelem] ; ipHi++ )
				{
					if ( (he_levels[ipHi].l == 1) && (he_levels[ipHi].s == 1) )
					{
						fprintf(ioQQQ,"\n%ld\t",he_levels[ipHi].n);
						j = 0;
						Bm = 0;
						for ( i = ipLo; i<=ipHi; i++)
						{
							if ( (he_levels[i].l == hi_l) && (he_levels[i].s == hi_s)  )
							{
								if ( (ipLo == ipHe2p3P0) && (i > ipHe2p3P2) )
								{
									Bm += RobbinsC[ipHi][i] * ( RobbinsP[i][ipHe2p3P0] + 
										RobbinsP[i][ipHe2p3P1] + RobbinsP[i][ipHe2p3P2] );
								}
								else
									Bm += RobbinsC[ipHi][i] * RobbinsP[i][ipLo];
									
								if ( (i == ipHe2p3P0) || (i == ipHe2p3P1) || (i == ipHe2p3P2) )
								{
									j++;
									if (j == 3)
									{
										fprintf(ioQQQ,"%2.4e\t",Bm);
										Bm = 0;
									}
								}
								else
								{
									fprintf(ioQQQ,"%2.4e\t",Bm);
									Bm = 0;
								}
							}
						}
					}
				}
			}
#endif

			ipLo = ipHe1s1S;
			/* This ipHi does NOT represent ipHe2s1S, but the n in n^1P. */
			for( ipHi=2; ipHi <iso.nLyman[ipHELIUM]; ipHi++ )
			{
				double Enerwn , Aul /*,RI2,npstar */ ;
				float gHi , gLo;

				/* pointers to transitions, ipLo, ipHi
					* actual quantum numbers, nLo nHi
					* atomic number or charge and stage: */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].nelem = (int)(nelem+1);
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].IonStg = (int)(nelem);

				/* stat weight of all n ^1P is 3 */
				gHi = 3.f;

				gLo = 1.;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gLo = gLo;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gHi = gHi;

				/* Passed to function he_1trans are the upper	*/
				/* and lower levels	of a transition.  			*/
				/* ret is 0 if line ok, 1 if not				
				he_1trans( nelem , Enerwn , &Aul , ipHi , ipLo, n_effs[ipHi], n_effs[ipLo] );*/
				/* hydrogenic energy, first ryd, convert to wn */
				Enerwn = EionWN[nelem-1] - RYD_INF*POW2((double)nelem)/POW2((double)ipHi);
				/* scqdri double nstar,long int l, double npstar,long int lp,  double iz */
				fixit();/* understand units of scqdri, is nelem 1 for HeI */
#if	0
				npstar = (nelem)*sqrt(1/EionRYD[nelem-1]);
				RI2 = scqdri((double)ipHi,1,npstar,0,(double)(nelem));
				/* ritoa(long li, long lf, long nelem, double k, double RI2) */
				Aul = ritoa(1,0,nelem,Enerwn,RI2);
#endif
				/*Aul = 1.;
				Enerwn = 1.;*/
				fixit();/* above need to fill in Enerwn, Aul */
				/* A simple fit for the calculation of Helium lyman Aul's.	*/
				Aul = (1.52e10) / POW3((double)ipHi);
				
				if ( nelem > ipHELIUM )
					/* Additional terms for higher Z.  In theory, this Z dependence
					 * should be Z^4, but values from TOPbase suggest 3.9 is a more
					 * accurate exponent.	*/
					Aul *= 0.875 * pow((double)nelem, 3.9) ;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].Aul = (float)(Aul);

				/* the energy of the transition in wavenumbers, this can be negative
					* for highly forbidden upper levels where energy switches */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyWN = (float)Enerwn;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyRyd = 
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyWN* (float)WAVNRYD;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyErg = 
					(float)(iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyRyd*
					EN1RYD);

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyK = 
					(float)(iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyRyd*
					TE1RYD );

				/* transition energy in various units:*/

				/* take abs value since some have order switched */
				/* make following an air wavelength */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].WLAng = 
					(float)fabs(1.0e8/
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyWN/
					RefIndex(&iso.ExtraLymanLines[ipHELIUM][nelem][ipHi]));

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gf = 
					(float)(GetGF(iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].Aul,
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyWN,
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gHi));

				/* derive the abs coef, call to function is gf, wl (A), g_low */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].opacity = 
					(float)(abscf(iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gf,
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].EnergyWN,
					iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].gLo));

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].iRedisFun = ipPRD;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].Pesc = 1.0;

				/* destruction probability*/
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].Pdest = 0.0;

				/* upper and lower level populations */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].PopHi = 0.0;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].PopLo = 0.0;

				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].xIntensity = 0.0;

				/* basic collision Info:
					* these are not Mewe lines, they are high quality: */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].cs1 = 0.0;

				/* following three will eventually be set by model atoms, but
					* must have reasonable value at start */
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].pump = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].AovTot = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].cool = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].heat = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].ColOvTot = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].PopOpc = 0.;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].damprel = 1e-30f;
				iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].damp = 1e-30f;
				{
					/* option to print particulars of some line when called
						* a prettier print statement is near where chSpin is defined below
						* search for "pretty print" */
					/*@-redef@*/
					enum {DEBUG=FALSE};
					/*@+redef@*/
					if( DEBUG )
					{
						fprintf(ioQQQ,"%li\t%li\t%.2e\t%.2e\n",
							nelem+1,
							ipHi,
							iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].Aul , 	
							iso.ExtraLymanLines[ipHELIUM][nelem][ipHi].opacity
							);
					}
				}
			}

			/* this is the two photon transition, designated by wl of 0 */
			EmisLines[ipHELIUM][nelem][ipHe2s1S][ipHe1s1S].WLAng = 0.;

			/* opacity in two-photon continuum is not correct since treated as line,
			 * fix now */
			EmisLines[ipHELIUM][nelem][ipHe2s1S][ipHe1s1S].opacity /= 1e4f;

			/* Lya has special redistribution function */
			EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].iRedisFun = ipPRD;
			
			EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].damprel = 
				(float)(
				/* this is sum of A's to n=2 for H by itself */
				EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].Aul/
			  PI4/EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].EnergyWN);
			/* needs the velocity to become the damping constant itself */
			EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].damp = EmisLines[ipHELIUM][nelem][ipHe2p1P][ipHe1s1S].damprel/1e6f;
		}
	}

	/* now establish radiative recombination rate coefficients 
	 * this is the vector or temperatures */
	for (i = 0; i < N_HE_TE_RECOMB; i++)
	{
		TeRRCoef[i] = 0.5*(i);
	}

	/* option to create table of recombination coefficients,
	 * executted with the compile he-like command */
	if (helike.lgCompileRecomb)
	{
		FILE *ioRECOMB;
		/*RECOMBMAGIC the magic number for the table of recombination coefficients */
		/*N_HE_LEV_RECOMB the number of levels that will be done */
		/* create recombination coefficients 
		 * N_HE_LEV_RECOMB corresponds to n = 20 */
		if( ( ioRECOMB = fopen( "he_iso_recomb.dat" , "w" ) ) == NULL )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat for writing.\n" );
			puts( "[Stop in HeCreate]" );
			cdEXIT(1);
		}
		fprintf(ioRECOMB,"%i\t%i\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %i levels and %i temperatures.\n",
			RECOMBMAGIC ,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB );

		for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
		{
			/* this must pass since compile he-like command reset nLevels to the macro */
			assert( N_HE_LEV_RECOMB <= iso.nLevels[ipHELIUM][nelem] );

			/* N_HE_LEV_RECOMB corresponds to n = 20 */
			for( ipLo=ipHe1s1S; ipLo < N_HE_LEV_RECOMB ; ipLo++ )
				{
					EthRyd = energies[nelem][ipLo] * WAVNRYD;

				fprintf(ioRECOMB, "%li\t%li", nelem, ipLo );
				/* loop over temperatures to produce array of recombination coefficients	*/
				for (i = 0; i < N_HE_TE_RECOMB; i++)
				{
					/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
					RRCoef[nelem][ipLo][i] = log10(radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo));
					fprintf(ioRECOMB, "\t%f", RRCoef[nelem][ipLo][i] );
				}
				fprintf(ioRECOMB, "\n" );
			}
		}
		/* end the file with the same information */
		fprintf(ioRECOMB,"%i\t%i\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %i levels and %i temperatures.\n",
			RECOMBMAGIC ,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB );

		fclose( ioRECOMB );

		fprintf( ioQQQ, "HeCreate: compilation complete, he_iso_recomb.dat created.\n" );
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}
	
	/* read in recombination coefficients */
	/* get the collision strength data for the He 1 lines */

	/* check on path if file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , "he_iso_recomb.dat" );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , "he_iso_recomb.dat" );
	}

	if( trace.lgTrace )
		fprintf( ioQQQ," HeCreate opening he_iso_recomb.dat:");

	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n" );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n");
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}
	i = 1;
	i1 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i2 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i3 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	/* N_HE_LEV_RECOMB corresponds to n = 20 */
	if( i1 !=RECOMBMAGIC || i2 !=N_HE_LEV_RECOMB || i3 !=N_HE_TE_RECOMB )
	{
		fprintf( ioQQQ, 
			" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
		fprintf( ioQQQ, 
			" HeCreate: I expected to find the numbers  %i %i %i  and got %li %li %li instead.\n" ,
			RECOMBMAGIC ,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB,
			i1 , i2 , i3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		fprintf( ioQQQ, 
			" HeCreate: please recompile with the COMPILE HE-LIKE command.\n" );
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* now read in the data */
	for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
	{
		/* N_HE_LEV_RECOMB corresponds to n = 20 */
		for( ipLo=ipHe1s1S; ipLo < N_HE_LEV_RECOMB; ipLo++ )
		{
			/* get next line image */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			{
				fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
				puts( "[Stop in HeCreate]" );
				cdEXIT(1);
			}
			/* each line starts with element and level number */
			i3 = 1;
			i1 = (long)FFmtRead(chLine,&i3,LINELENGTH,&lgEOL);
			i2 = (long)FFmtRead(chLine,&i3,LINELENGTH,&lgEOL);
			/* check that these number are correct */
			if( i1!=nelem || i2!=ipLo )
			{
				fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
				fprintf( ioQQQ, 
					" HeCreate: please recompile with the COMPILE HE-LIKE command.\n" );
				puts( "[Stop in HeCreate]" );
				cdEXIT(1);
			}

			/* loop over temperatures to produce array of recombination coefficients	*/
			for (i = 0; i < N_HE_TE_RECOMB; i++)
			{
				RRCoef[nelem][ipLo][i] = FFmtRead(chLine,&i3,chLine_LENGTH,&lgEOL);
				if( lgEOL )
				{
					fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
					fprintf( ioQQQ, 
						" HeCreate: please recompile with the COMPILE HE-LIKE command.\n" );
					puts( "[Stop in HeCreate]" );
					cdEXIT(1);
				}
			}
		}
		/* following loop only executed if we need more levels than are
		 * stored in the recom coef data set */
		for( ipLo=N_HE_LEV_RECOMB; ipLo < iso.nLevels[ipHELIUM][nelem]; ipLo++ )
		{
			EthRyd = energies[nelem][ipLo] * WAVNRYD;
			for (i = 0; i < N_HE_TE_RECOMB; i++)
			{
				/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
				RRCoef[nelem][ipLo][i] = log10(radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo));
			}
		}
	}

	/* check that ending magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}
	i = 1;
	i1 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i2 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i3 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	if( i1 !=RECOMBMAGIC || i2 !=N_HE_LEV_RECOMB || i3 !=N_HE_TE_RECOMB )
	{
		fprintf( ioQQQ, 
			" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
		fprintf( ioQQQ, 
			" HeCreate: I expected to find the numbers  %i %i %i  and got %li %li %li instead.\n" ,
			RECOMBMAGIC ,
			N_HE_LEV_RECOMB,
			N_HE_TE_RECOMB,
			i1 , i2 , i3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		fprintf( ioQQQ, 
			" HeCreate: please recompile with the COMPILE HE-LIKE command.\n" );
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* close the data file */
	fclose( ioDATA );

	printstuff(printflag);

	/* free the memory	*/
	free(n_effs );

	/* create spectroscopic designation of labels */
	for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipHELIUM]; ++ipLo )
	{
		sprintf( helike.chLevel[ipLo] , "%li %c%c", 
				he_levels[ipLo].n,
				chSpin[he_levels[ipLo].s],
				chL[MIN2(5,he_levels[ipLo].l)] );
	}

	/* option to create trace printout */
	if( (trace.lgTrace && trace.lgHeBug) )
	{

		/* "pretty print" only helium, O, and Fe 
		 * comments elsewhere say to search for "pretty print" */
		/* next print designation of levels */
		fprintf(ioQQQ,"index\tdesig\tE(wn,He)\tE(wn,O)\tE(wn,Fe)\n");
		for( ipHi=1; ipHi<need; ++ipHi )
		{
			float enHe=0. , enO=0., enFe=0.;
			if( ipHi < iso.nLevels[ipHELIUM][ipHELIUM] )
			{
				enHe = EmisLines[ipHELIUM][ipHELIUM][ipHi][0].EnergyWN;
			}
			else
			{
				enHe = 0.;
			}
			if( ipHi < iso.nLevels[ipHELIUM][ipOXYGEN] )
			{
				enO = EmisLines[ipHELIUM][ipOXYGEN][ipHi][0].EnergyWN;
			}
			else
			{
				enO = 0.;
			}
			if( ipHi < iso.nLevels[ipHELIUM][ipIRON] )
			{
				enFe = EmisLines[ipHELIUM][ipIRON][ipHi][0].EnergyWN;
			}
			else
			{
				enFe = 0.;
			}
			fprintf(ioQQQ,"%li\t%li %c%c\t%.5e\t%.5e\t%.5e\n", 
				ipHi, 
				he_levels[ipHi].n,
				chSpin[he_levels[ipHi].s],
				chL[MIN2(5,he_levels[ipHi].l)] , 
				enHe ,
				enO ,
				enFe 
				);
		}
		for( ipHi=ipHe2s3S; ipHi<iso.nLevels[ipHELIUM][ipHELIUM] ; ++ipHi )
		{
			for( ipLo=ipHe1s1S; ipLo<ipHi ; ++ipLo )
			{
				fprintf(ioQQQ," %li\t%li\t%li %c%c\t%li %c%c\n",
					ipLo , ipHi ,
					he_levels[ipLo].n,
					chSpin[he_levels[ipLo].s],
					chL[MIN2(5,he_levels[ipLo].l)] , 
					he_levels[ipHi].n,
					chSpin[he_levels[ipHi].s],
					chL[MIN2(5,he_levels[ipHi].l)]
					);
			}
		}
	}

	if( iso.nLevels[ipHELIUM][ipHELIUM] > ipHe2p1P )
	{
		/* some wavelengths of HeI are expected to be exact */
		EmisLines[ipHELIUM][ipHELIUM][ipHe2p3P0][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHELIUM][ipHELIUM][ipHe2p3P1][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHELIUM][ipHELIUM][ipHe2p3P2][ipHe2s3S].WLAng = 10830.;
		EmisLines[ipHELIUM][ipHELIUM][ipHe2p1P][ipHe2s1S].WLAng = 20600.;

		if(  iso.nLevels[ipHELIUM][ipHELIUM] > ipHe3d1D )
		{
			EmisLines[ipHELIUM][ipHELIUM][ipHe3s3S][ipHe2p3P0].WLAng = 7065.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3s3S][ipHe2p3P1].WLAng = 7065.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3s3S][ipHe2p3P2].WLAng = 7065.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3p3P][ipHe2s3S].WLAng = 3889.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3d3D][ipHe2p3P0].WLAng = 5876.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3d3D][ipHe2p3P1].WLAng = 5876.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3d3D][ipHe2p3P2].WLAng = 5876.;
			EmisLines[ipHELIUM][ipHELIUM][ipHe3d1D][ipHe2p1P].WLAng = 6678.;

			if(  iso.nLevels[ipHELIUM][ipHELIUM] > ipHe4d3D )
			{
				EmisLines[ipHELIUM][ipHELIUM][ipHe4d3D][ipHe2p3P0].WLAng = 4471.;
				EmisLines[ipHELIUM][ipHELIUM][ipHe4d3D][ipHe2p3P1].WLAng = 4471.;
				EmisLines[ipHELIUM][ipHELIUM][ipHe4d3D][ipHe2p3P2].WLAng = 4471.;
			}
		}
	}

	{
		/* prints table of wavelengths of lines to ground */
		/*@-redef@*/
		enum {PRINTIT=FALSE};
		/*@+redef@*/
		if( PRINTIT )
		{
			for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
			{
				if( abundances.lgElmtOn[nelem] )
				{
					fprintf(ioQQQ,"%li\t%s\t",nelem+1,elementnames.chElementSym[nelem] );

					ipLo = ipHe1s1S;
					for( ipHi=ipLo+1; ipHi<=ipHe2p1P; ++ipHi )
					{
						if( ipHi==ipHe2s1S || ipHi==ipHe2p3P0 ) continue;
						fprintf(ioQQQ,"%.4f\t", EmisLines[ipHELIUM][nelem][ipHi][ipLo].WLAng);
					}
					fprintf(ioQQQ,"\n");
				}
			}
			exit(0);
		}
	}

	/* get the collision strength data for the He 1 lines */

	/* check on path if file not here and path set */
	/* path was parsed in getset */
	if( lgDataPathSet == TRUE )
	{
		/*path set, so look only there */
		strcpy( chFilename , chDataPath );
		strcat( chFilename , "he1_cs.dat" );
	}
	else
	{
		/* path not set, check local space only */
		strcpy( chFilename , "he1_cs.dat" );
	}

	if( trace.lgTrace )
		fprintf( ioQQQ," HeCreate opening he1_cs.dat:");

	if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not open he1_cs.dat\n" );
		if( lgDataPathSet == TRUE )
			fprintf( ioQQQ, " even tried path\n" );

		if( lgDataPathSet == TRUE )
		{
			fprintf( ioQQQ, " HeCreate could not open he1_cs.dat\n");
			fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
			fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
		}

		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* check that magic number is ok */
	if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
	{
		fprintf( ioQQQ, " HeCreate could not read first line of he1_cs.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}
	i = 1;
	i1 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i2 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
	i3 = (long)FFmtRead(chLine,&i,LINELENGTH,&lgEOL);

	/* the following is the set of numbers that appear at the start of level1.dat 00 10 26 */
	if( ( i1 != 01 ) || ( i2 != 02 ) || ( i3 != 20 ) )
	{
		fprintf( ioQQQ, 
			" HeCreate: the version of he1_cs.dat is not the current version.\n" );
		fprintf( ioQQQ, 
			" I expected to find the number 01 02 20 and got %li %li %li instead.\n" ,
			i1 , i2 , i3 );
		fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* get the array of temperatures */
	lgHIT = FALSE;
	while( fgets( chLine , (int)sizeof(chLine) , ioDATA ) != NULL )
	{
		/* only look at lines without '#' in first col */
		if( chLine[0] != '#')
		{
			lgHIT = TRUE;
			helike.nCS = 0;
			lgEOL = FALSE;
			i = 1;
			while( !lgEOL && helike.nCS < HE1CSARRAY)
			{
				CSTemp[helike.nCS] = FFmtRead(chLine,&i,LINELENGTH,&lgEOL);
				++helike.nCS;
			}
			break;
		}
	}
	--helike.nCS;
	if( !lgHIT )
	{
		fprintf( ioQQQ, " HeCreate could not find line in CS temperatures in he1_cs.dat.\n");
		puts( "[Stop in HeCreate]" );
		cdEXIT(1);
	}

	/* create space for array of CS values, if not already done */
	if( (HeCS = (float ****)MALLOC(sizeof(float ***)*(unsigned)LIMELM ) )==NULL )
		bad_malloc();

	for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
	{
		/* only grab core for elements that are turned on */
		if( nelem == ipHELIUM || abundances.lgElmtOn[nelem])
		{
			if( (HeCS[nelem] = 
				(float***)MALLOC(sizeof(float**)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ))==NULL )
				bad_malloc();

			/* avoid allocating 0 bytes, some OS return NULL pointer, PvH */
			HeCS[nelem][0] = NULL;
			for( ipHi=ipHe2s3S; ipHi < iso.nLevels[ipHELIUM][nelem] ;++ipHi )
			{
				if( (HeCS[nelem][ipHi] = 
					(float**)MALLOC(sizeof(float*)*(unsigned)ipHi ))==NULL )
					bad_malloc();

				for( ipLo=ipHe1s1S; ipLo<ipHi; ++ipLo )
				{
					if( (HeCS[nelem][ipHi][ipLo] = 
						(float*)MALLOC(sizeof(float)*(unsigned)helike.nCS ))==NULL )
						bad_malloc();
					/* fill them with the default value, in case no better comes along */
					/* helike.lgFakeCS is flag saying whether to make up 
					 * collision strenths for
					 * transitions with no calculations - default is 1, set to
					 * zero with ATOM HE-LIKE COLLISIONS FAKE command */
					for( j=0; j< helike.nCS; ++j )
					{
						HeCS[nelem][ipHi][ipLo][j] = 0.1f *helike.lgFakeCS;
					}
				}
			}
		}
	}

	/* now read in the CS data */
	lgHIT = FALSE;
	nelem = ipHELIUM;
	while( fgets( chLine , (int)sizeof(chLine) , ioDATA ) != NULL )
	{
		char *chTemp;
		/* only look at lines without '#' in first col */
		if( (chLine[0] == ' ') || (chLine[0]=='\n') )
			break;
		if( chLine[0] != '#')
		{
			lgHIT = TRUE;

			/* get lower and upper level index */
			j = 1;
			ipLo = (long)FFmtRead(chLine,&j,LINELENGTH,&lgEOL);
			ipHi = (long)FFmtRead(chLine,&j,LINELENGTH,&lgEOL);
			assert( ipLo < ipHi );
			if( ipHi >= iso.nLevels[ipHELIUM][nelem] )
				continue;

			chTemp = chLine;
			/* skip over 4 tabs to start of cs data */
			for( i=0; i<3; ++i )
			{
				if( (chTemp = strchr( chTemp, '\t' )) == NULL )
				{
					fprintf( ioQQQ, " HeCreate no init cs\n" );
					puts( "[Stop in HeCreate]" );
					cdEXIT(1);
				}
				++chTemp;
			}

			for( i=0; i<helike.nCS; ++i )
			{
				float a;
				if( (chTemp = strchr( chTemp, '\t' )) == NULL )
				{
					fprintf( ioQQQ, " HeCreate not scan cs\n" );
					puts( "[Stop in HeCreate]" );
					cdEXIT(1);
				}
				++chTemp;
				sscanf( chTemp , "%e" , &a );
				HeCS[nelem][ipHi][ipLo][i] = a;
			}
		}
	}

	/* close the data file */
	fclose( ioDATA );

	/* option to print cs data for AGN */
	/* create spectroscopic designation of labels */
	{
		/* option to print particulars of some line when called
		 * a prettier print statement is near where chSpin is defined below*/
		/*@-redef@*/
		enum {AGN=FALSE};
		/*@+redef@*/
		if( AGN )
		{
#			define NTEMP 6
			double te[NTEMP]={6000.,8000.,10000.,15000.,20000.,25000. };
			double telog[NTEMP] ,
				cs ,
				ratecoef;
			nelem = ipHELIUM;
			fprintf( ioQQQ,"trans");
			for(i=0; i<NTEMP; ++i)
			{
				telog[i] = log10( te[i] );
				fprintf( ioQQQ,"\t%.3e",te[i]);
			}
			for(i=0; i<NTEMP; ++i)
			{
				fprintf( ioQQQ,"\t%.3e",te[i]);
			}
			fprintf(ioQQQ,"\n");

			for( ipHi=ipHe2s3S; ipHi< iso.nLevels[ipHELIUM][ipHELIUM]; ++ipHi )
			{
				for( ipLo=ipHe1s1S; ipLo < ipHi; ++ipLo )
				{
					/* print the designations of the lower and upper levels */
					fprintf( ioQQQ,"%s - %s",
						helike.chLevel[ipLo] , helike.chLevel[ipHi] );

					/* print the interpolated collision strengths */
					for(i=0; i<NTEMP; ++i)
					{
						phycon.alogte = (float)telog[i];
						/* print cs */
						cs = HeCSInterp( nelem , ipHi , ipLo );
						fprintf(ioQQQ,"\t%.2e", cs );
					}

					/* print the rate coefficients */
					for(i=0; i<NTEMP; ++i)
					{
						phycon.alogte = (float)telog[i];
						phycon.te = (float)pow(10.,telog[i] );
						tfidle(FALSE);
						cs = HeCSInterp( nelem , ipHi , ipLo );
						/* collisional deexcitation rate */
						ratecoef = cs/sqrt(phycon.te) *8.629e-6/iso.stat[ipHELIUM][ipLo] *
							sexp( EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyK / phycon.te );
						fprintf(ioQQQ,"\t%.2e", ratecoef );
					}
					fprintf(ioQQQ,"\n");
				}
			}
			exit(1);
		}
	}

	for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][ipHELIUM] ;++ipLo )
	{
		free( RobbinsC[ipLo] );
		free( RobbinsP[ipLo] );
	}
	for( nelem=ipHELIUM; nelem<LIMELM; ++nelem )
	{
		if( nelem == ipHELIUM || abundances.lgElmtOn[nelem] )
			free( energies[nelem] );
	}
	free( energies );
	free( RobbinsC );
	free( RobbinsP );

#	ifdef DEBUG_FUN
	fputs( " <->HeCreate()\n", debug_fp );
#	endif
	return;
}

/*===================================================================================*/
/* HeRecom - do recomb coef for He, called by HeLike */
static void HeRecom( 
			/* nelem on the c scale, He is 1 */
			long nelem )
{
	static double ditcrt[LIMELM]= {0.,5e4,1.20E+04,1.10E+04,4.40E+05,7.00E+05,1.00E+06,1.50E+06,
	  1.50E+06,3.80E+06,3.40E+04,4.00E+06,5.50E+06,6.30E+06,5.00E+06,9.00E+06,1.40E+07,
	  1.40E+07,2.00E+05,2.00E+05,2.00E+05,1.00E+05,7.00E+04,4.00E+04,6.00E+06,1.00E+11,
	  1.00E+20,1.00E+20,1.00E+20,1.00E+20};

	static double dicoef0[LIMELM]= {0.,1.9e-3,6.15E-03,1.62E-03,4.78E-02,3.22E-02,4.61E-02,6.23E-02,
	  2.52E-01,1.44E-01,1.80E-01,1.44E-01,3.40E+00,1.89E-01,4.02E-01,2.41E-01,1.10E+01,2.97E-01,
	  5.49E-01,3.55E-01,2.30E-01,1.40E-01,1.10E-01,4.10E-02,7.47E-01,5.19E-01,5.75E-01,5.75E-01,
	  5.75E-01,5.75E-01};

	static double dite0[LIMELM]= {0.,4.7e5,1.41E+05,8.19E+04,3.44E+06,4.06E+06,5.44E+06,7.01E+06,1.40E+07,
	  1.50E+07,1.30E+07,1.50E+07,1.70E+07,1.99E+07,2.41E+07,2.54E+07,2.80E+07,3.13E+07,3.65E+07,
	  3.78E+07,1.09E+06,9.62E+05,7.23E+05,4.23E+05,5.84E+07,6.01E+07,6.81E+07,6.81E+07,6.81E+07,
	  6.81E+07};

	static double dicoef1[LIMELM]= {0.,0.3,5.88E-02,3.43E-01,3.62E-01,3.15E-01,3.09E-01,3.04E-01,
	  3.04E-01,2.96E-01,0.00E+00,2.91E-01,0.00E+00,2.86E-01,2.98E-01,2.81E-01,0.00E+00,2.77E-01,
	  2.92E-01,2.75E-01,3.60E+00,4.90E+00,1.60E+00,4.20E+00,2.84E-01,2.79E-01,0.00E+00,2.86E-01,
	  2.86E-01,2.86E-01};

	static double dite1[LIMELM]= {0.,9.4e4,1.41E+05,1.59E+05,5.87E+05,8.31E+05,1.14E+06,1.47E+06,
	  1.80E+06,2.24E+06,0.00E+00,3.09E+06,0.00E+00,4.14E+06,4.67E+06,5.30E+06,0.00E+00,6.54E+06,
	  7.25E+06,7.68E+06,1.10E+07,8.34E+06,1.01E+07,1.07E+07,1.17E+07,9.97E+06,0.00E+00,9.08E+06,
	  9.08E+06,9.08E+06};


	long lbig;
	double corl;

	long n,
		ntopoff;
	double 
		RecUnitVolTotal , 
		sumtopoff,
		topoff1,
		topoff3;
	static double 
		RecExplictLevels1[LIMELM],
		RecExplictLevels3[LIMELM],
		TotalRadRecomb[LIMELM];

#	ifdef DEBUG_FUN
	fputs( "<+>HeRecom()\n", debug_fp );
#	endif

	fixit(); /* self ionization correction included, but not ots */
	/* evaluate recombination escape probability for all levels */

	/* define radiative recombination rates for all levels */ 
	/* this will be the sum of all levels explicitly included in the model atom */
	RecExplictLevels1[nelem] = 0.;
	RecExplictLevels3[nelem] = 0.;
	/* this is radiative recombination rate coefficient 
	 * do ground first so as not to include in sum below */
	n = 0;
	iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad] = (float)He_RRCoef_Te( nelem , n );
	for( n=1; n<iso.nLevels[ipHELIUM][nelem]; ++n )
	{
		/* this is radiative recombination rate coefficient */
		iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad] = (float)He_RRCoef_Te( nelem , n );
		if( he_levels[n].s == 0 )
		{
			RecExplictLevels1[nelem] += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
		}
		else if( he_levels[n].s == 1 )
		{
			RecExplictLevels3[nelem] += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
		}
		else
			total_insanity();

		assert( iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad] > 0. );
	}
	/* get total recombination rate */
	/* this is total recombination rate, all levels, from pre-computed fits */
	/* >>chng 02 jan 04, use exact values for nebular conditions */
	if( nelem==1 && phycon.te < 1e6 )
	{
		double excit;
		if( phycon.te< 2500. || phycon.te >= 40000. )
		{
			/* helium had special fits that were more accurate, but only below 1e6 K */
			/* >>refer	He1	rec	Verner, D.A., & Ferland, G.J., 1996, ApJS, 103, 467 */
			double tt = sqrt(phycon.te/15.54);

			TotalRadRecomb[nelem] = 
			  3.294e-11/ (tt*pow(tt+1. , 1. - 0.6910) *
			  pow(1.0 + sqrt(phycon.te/3.676e7) , 1.0 + 0.6910 ));
			excit = 0.;
		}
		else
		{
			/* this branch is HeI under nebular conditions - want to do excellent job */
			double power[4] = {-0.88903,-0.77672,-0.80206,-0.45125};
			double coef[4] = { 9.055E-10,3.479E-10,4.394E-10,1.361E-11};
			if( phycon.te < 5000. )
			{
				excit = coef[0] * pow((double)phycon.te ,power[0] );
			}
			else if( phycon.te < 10000. )
			{
				excit = coef[1] * pow((double)phycon.te ,power[1] );
			}
			else if( phycon.te < 20000. )
			{
				excit = coef[2] * pow((double)phycon.te ,power[2] );
			}
			else if( phycon.te < 40000. )
			{
				excit = coef[3] * pow((double)phycon.te ,power[3] );
			}
			else 
				total_insanity();
			TotalRadRecomb[nelem] = iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad] + excit;
		}
		/*fprintf(ioQQQ," %.3e\t%.3e\t%.3e\n", phycon.te , TotalRadRecomb , rrfit( nelem+1 , 2 , phycon.te ));exit(1);*/
		/*fprintf(ioQQQ," %.3e\t%.3e\t%.3e\t%.3e\n", phycon.te , TotalRadRecomb , 
			rrfit( nelem+1 , 2 , phycon.te ) - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad] ,
			excit );exit(1);*/
	}
	else
	{
		TotalRadRecomb[nelem] = rrfit( nelem+1 , 2 , phycon.te );
	}

	/* this is case B recombination, sum without the ground included */
	iso.RadRec_caseB[ipHELIUM][nelem] = (float)TotalRadRecomb[nelem] - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad];
	assert( iso.RadRec_caseB[ipHELIUM][nelem] > 0.);
	/* >>chng 02 jan 07, do not add three body, complete model atom will automatically include it
	 * problem is that most elements will not go to high n */
	/* add three-body recombination, convert to per unit vol 
	RecUnitVolTotal = phycon.eden*(TotalRadRecomb[nelem] + recom.CotaRate[nelem-1]);*/
	/* convert to per unit vol */
	RecUnitVolTotal = phycon.eden*TotalRadRecomb[nelem];

	/* recombinatinon on grain surface */
	RecUnitVolTotal += recom.GrnIonRec;

	/* save the rate */
	fixit(); /* why is dr not included in this ?  what is it used for?
	          * dr is added later */
	recom.RecombinRate[nelem][nelem-1] = (float)RecUnitVolTotal;
	DestCrt.Recombine[nelem][nelem-1] = (float)RecUnitVolTotal;

	/* add on DR */
	if( nelem==ipIRON )
	{
		/* for case of iron, add on Dima's fits to diel part, above was zero */
		float DieRate;
		drfe(25,phycon.te,&DieRate);
		/*fprintf(ioQQQ,"%li %g %g\n", IonStage_C_Scale, recom.RecombinRate[NELEM-1][IonStage_C_Scale], DieRate*phycon.eden );*/
		RecUnitVolTotal += (float)(DieRate*phycon.eden);
	}
	else if( phycon.te > (float)(ditcrt[nelem]*0.1) )
	{
		/* Burgess dielectronic recombination */
		double diburg;
		/* the suppression factor is on a charge scale, so rec to neutral
		 * is always 0, hence the nelem-1 */
		diburg = recom.DielSupprs[0][nelem-1]/phycon.te32*
		  dicoef0[nelem]*exp(-dite0[nelem]/phycon.te)*
		  (1. + dicoef1[nelem]*
		  sexp(dite1[nelem]/phycon.te));
		RecUnitVolTotal += diburg* phycon.eden;
	}

	{
		/* following should be set true to print recombination rates */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG /*&& (iteration==2)*/ )
		{
			if( nelem==ipHELIUM )
				fprintf(ioQQQ,"\n");
			fprintf(ioQQQ,"%li rec\t%.4e", nelem,
				recom.RecombinRate[nelem][nelem-1] );

			fprintf(ioQQQ,"\t%.4e\t%.4e\n", RecUnitVolTotal ,
				(recom.RecombinRate[nelem][nelem-1]-RecUnitVolTotal)/recom.RecombinRate[nelem][nelem-1] );
			/*fprintf( ioQQQ," %li\t%.2e \n", nelem, recom.RecombinRate[nelem][nelem-1] );*/
		}
	}
	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		double sumtrip=0. , sumsing=0.;
		/*@+redef@*/
		if( DEBUG && (nelem==ipHELIUM)/**/ )
		{
			/* sum does not include the ground level */
			for( n=1; n<iso.nLevels[ipHELIUM][nelem]; ++n)
			{
				if( he_levels[n].s == 0 )
				{
					sumsing += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
				}
				else if( he_levels[n].s == 1 )
				{
					sumtrip += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
				}
				else
					total_insanity();
			}
			fprintf(ioQQQ,"radreccc\t%.4e\tsing caB\t%.4e\ttrip\t%.4e\tsum caB\t%.4e\tneeded\t%.4e\n", 
				phycon.te, 
				sumsing, 
				sumtrip ,
				sumsing +sumtrip ,
				TotalRadRecomb[nelem] - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad]
				);
		}
	}

	/* at this point we have two numbers, 
	 * RecExplictLevels1[nelem], RecExplictLevels3[nelem], 
	 * the sum of radiative recombination to all 
	 * levels explicitly included in the model atom, and
	 * RecUnitVolTotal, the total recombination rate 
	 * the difference is the amount of "topoff" we will need to do */
	topoff1 = (RecUnitVolTotal/phycon.eden - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad]) /4.37 - RecExplictLevels1[nelem];
	topoff3 = (RecUnitVolTotal/phycon.eden - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad])*3.37/4.37 - RecExplictLevels3[nelem];
	/* the rrfit fits are too small at high temperatures, so this atom is
	 * better than the topoff.  Only a problem for helium itself, at high temperatures.
	 * complain if the negative topoff is not for this case */
	if( MIN2(topoff1,topoff3) < 0. && (nelem!=ipHELIUM || phycon.te < 1e5 ) )
	{
		fprintf(ioQQQ," negative topoff for %li, rel error was %.2e %.2e\n", 
			nelem, 
			topoff1/RecExplictLevels1[nelem] ,
			topoff3/RecExplictLevels3[nelem] 
			);
	}
	/* make sure it is positive - slightly negative values can reach here due to 
	 * precision in total fits to radiative recombination rates */
	topoff1 = MAX2(0.,topoff1);
	topoff3 = MAX2(0.,topoff3);

	/* this is the actual quantum number n of next to highest level */
	ntopoff = iso.nHighest[ipHELIUM][nelem] - 1;

	/* this is the array index within model atom which is the lowest
	 * nls state that is within the highest n quantum level 
	 * this is the first level we will add topoff to */
	ntopoff = ntopoff*ntopoff + ntopoff + 1;

	/* this will be the largest l to which rec will be added */
	lbig = (long)(iso.nHighest[ipHELIUM][nelem] *0.8);
	/*lbig = (long)(iso.nHighest[ipHELIUM][nelem] *0.5);*/	
	corl = (double)iso.nHighest[ipHELIUM][nelem] / (double)(lbig+1);
	
	/* check that this actually points to correct n */
	assert( (he_levels[ntopoff].n   == iso.nHighest[ipHELIUM][nelem]) &&
		    (he_levels[ntopoff-1].n == iso.nHighest[ipHELIUM][nelem] - 1)	);

	/* actually add the extra recombination coefficient */
	sumtopoff = 0.;
	for( n=ntopoff; n<iso.nLevels[ipHELIUM][nelem]; ++n)
	{
		double one;
		/* only add if l is within lbig */
		if( he_levels[n].l <= lbig )
		{
			/* no l dependence on topoff */
			if( he_levels[n].s == 0 )
			{
				/* singlet */
				one = topoff1/iso.nHighest[ipHELIUM][nelem]*corl;
			}
			else if( he_levels[n].s == 1 )
			{
				/* triplets */
				/* >>chng 02 jan 05, use this as the default */
				one = topoff3/iso.nHighest[ipHELIUM][nelem]*corl;
			}
			else
				total_insanity();

			sumtopoff += one;
			iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad] += (float)one;
		}
	}
	assert( fabs(sumtopoff-topoff1-topoff3)/MAX2(SMALLFLOAT , sumtopoff ) < 0.01 );
	/*fprintf(ioQQQ,"He1 topoff / case b = %.2e\n", sumtopoff / iso.RadRec_caseB[ipHELIUM][nelem] );*/
	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		double sumtrip=0. , sumsing=0.;
		/*@+redef@*/
		if( DEBUG && (nelem==ipHELIUM)/**/ )
		{
			/* sum does not include the ground level */
			for( n=1; n<iso.nLevels[ipHELIUM][nelem]; ++n)
			{
				if( he_levels[n].s == 0 )
				{
					sumsing += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
				}
				else if( he_levels[n].s == 1 )
				{
					sumtrip += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad];
				}
				else
					total_insanity();
			}
			fprintf(ioQQQ,"radreccc\t%.4e\tsing caB\t%.4e\ttrip\t%.4e\tsum caB\t%.4e\trecrad\t%.4e\n", 
				phycon.te, 
				sumsing, 
				sumtrip ,
				sumsing +sumtrip ,
				TotalRadRecomb[nelem] - iso.RadRecomb[ipHELIUM][nelem][0][ipRecRad]
				);
		}
	}

	/* total continuum effective escape prob for all levels  */
	/* RadRec_effec is total effective radiative recombination */
	iso.RadRec_effec[ipHELIUM][nelem] = 0.;

	for( n=0; n<iso.nLevels[ipHELIUM][nelem]; ++n )
	{
		/* this is escape probability */
		iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc] = 
			(float)receff(iso.ipIsoLevNIonCon[ipHELIUM][nelem][n]);

		/* correct above value for possible otsmin value
		 * otsmin set to zero in zerologic, set to 1 with 
		 * the NO ON THE SPOT command */
		iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc] = 
			(float)MAX2(iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc], opac.otsmin);

		/* net escape prob includes dest by background opacity */
		iso.RadRecomb[ipHELIUM][nelem][n][ipRecNetEsc] = (float)MIN2(1.,iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc]+
		  (1.-iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc])*iso.ConOpacRatio[ipHELIUM][nelem][n]);

		assert( iso.RadRecomb[ipHELIUM][nelem][n][ipRecEsc] >= 0. );
		assert( iso.RadRecomb[ipHELIUM][nelem][n][ipRecNetEsc] >= 0. );

		/* sum of all effective rad rec */
		iso.RadRec_effec[ipHELIUM][nelem] += iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad]*
		  iso.RadRecomb[ipHELIUM][nelem][n][ipRecNetEsc];
	}

	/* option for case b conditions, kill ground state recombination */
	if( opac.lgCaseB )
	{
		iso.RadRecomb[ipHELIUM][nelem][ipHe1s1S][ipRecEsc] = 1e-10f;
		iso.RadRecomb[ipHELIUM][nelem][ipHe1s1S][ipRecNetEsc] = 1e-10f;
	}

	return;

#	ifdef DEBUG_FUN
	fputs( " <->HeRecom()\n", debug_fp );
#	endif
}

/*===================================================================================*/
/* evaluate collisional rates */
static void HeCollid( long nelem)
{

	double factor , ConvLTEPOP;
	long int ipLo , ipHi ;
	float DimaRate, crate;
	static float TeUsed[LIMELM]={0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
		0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0};

#	ifdef DEBUG_FUN
	fputs( "<+>HeCollid()\n", debug_fp );
#	endif

	/*lint -e777 */
	if( TeUsed[nelem] != phycon.te )
	{
	/*lint +e777 */
		TeUsed[nelem] = phycon.te;

		/* following factor is actually 4.1412957e-16 (old=4.14158E-16), 
		 * but e- stat weight is in */
		/* ConvLTEPOP = 2.0708e-16/phycon.te32;*/
		/* >>chng 99 jun 02, use codata and infinite mass nuc */
		factor = HION_LTE_POP*AtomcWgt.AtomicWeight[nelem]/
			(AtomcWgt.AtomicWeight[nelem]+ELECTRON_MASS/ATOMIC_MASS_UNIT) ;

		ConvLTEPOP = pow(factor,1.5)/1./phycon.te32;

		iso.lgPopLTE_OK[ipHELIUM][nelem] = TRUE;
		/* fully define Boltzmann factors to continuum for model levels */
		for( ipLo=ipHe1s1S; ipLo < (iso.nLevels[ipHELIUM][nelem]); ipLo++ )
		{
			iso.ConBoltz[ipHELIUM][nelem][ipLo] = 
				sexp(iso.xIsoLevNIonRyd[ipHELIUM][nelem][ipLo]/phycon.te_ryd);

			/***********************************************************************
			 *                                                                     *
			 * LTE abundances for all levels, ionization and                       *
			 * exciation between levels                                            *
			 *                                                                     *
			 ***********************************************************************/

			if( iso.ConBoltz[ipHELIUM][nelem][ipLo] > 1e-100 )
			{
				iso.PopLTE[ipHELIUM][nelem][ipLo] = 
					iso.stat[ipHELIUM][ipLo] / iso.ConBoltz[ipHELIUM][nelem][ipLo]* ConvLTEPOP;
			}
			else
			{
				iso.PopLTE[ipHELIUM][nelem][ipLo] = 0.;
			}

			/* now check for any zeros - possible above underflowed to zero */
			if( iso.PopLTE[ipHELIUM][nelem][ipLo] <= 0. )
			{
				iso.lgPopLTE_OK[ipHELIUM][nelem] = FALSE;
			}
		}

		for( ipHi=ipHe2s3S; ipHi <iso.nLevels[ipHELIUM][nelem]; ipHi++ )
		{
			for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
			{
				/* get collision strength */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs = HeCSInterp( nelem , ipHi , ipLo );
				{
					/* option to print particulars of some line when called
						* a prettier print statement is near where chSpin is defined below*/
					/*@-redef@*/
					enum {DEBUG=FALSE};
					/*@+redef@*/
					if( DEBUG && (nelem==ipHELIUM) && (he_levels[ipHi].n == 15) && (he_levels[ipLo].n >= 14)/**/ )
					{
						/* only print "real" cs - made-up one is 0.1 */
						if( fabs(EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs -0.1)> 1e-4 )
							{
								fprintf(ioQQQ,"Lo %li n %li s %li l %li \t ", 
									ipLo, 
									he_levels[ipLo].n,
									he_levels[ipLo].s,
									he_levels[ipLo].l );
								fprintf(ioQQQ," Hi %li n %li s %li l %li \t", 
									ipHi, 
									he_levels[ipHi].n,
									he_levels[ipHi].s,
									he_levels[ipHi].l );
								fprintf(ioQQQ,"%.4e\t%.0f\t%.0f\t%.4e\t%.4e\t%.4e\n",
								EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyWN,
									iso.stat[ipHELIUM][ipHi],
									iso.stat[ipHELIUM][ipLo],
								EmisLines[ipHELIUM][nelem][ipHi][ipLo].Aul,
								EmisLines[ipHELIUM][nelem][ipHi][ipLo].gf ,
								EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs );
							}
						}
					}

				/* collisional deexcitation rate, lgHeClON is normally TRUE (1),
					* and set FALSE (0) with atom he-like collisions off command */
				/*This is downward collision rate */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].ColUL = 
					(float)(EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs/
					/* >>chng 02 feb 11, lgHeClON not used, various flags in place
						* that work with cs itself, so, in case where off, cs is zero here */
					/*phycon.sqrte*8.629e-6/iso.stat[ipHELIUM][ipHi]*helike.lgHeClON );*/
					phycon.sqrte*8.629e-6/iso.stat[ipHELIUM][ipHi] );

				/* define line Boltzmann factors
					* some energy-degenerate collisions have very small negative energy,
					* when different l, same n, levels flip over */
				if( EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyK > 0. )
				{
					iso.Boltzmann[ipHELIUM][nelem][ipHi][ipLo] = 
						sexp( EmisLines[ipHELIUM][nelem][ipHi][ipLo].EnergyK / phycon.te );
				}
				else
				{
					/* negative energy, assume h nu/kT is very small */
					iso.Boltzmann[ipHELIUM][nelem][ipHi][ipLo] = 1.;
				}
			}
			fixit();/* must set collisional ionization to something */
			iso.ColIoniz[ipHELIUM][nelem][ipLo] = 0.;
		}
		fixit();/* must set collisional ionization to something */
		/* above loop left off highest level */
		iso.ColIoniz[ipHELIUM][nelem][iso.nLevels[ipHELIUM][nelem] - 1] = 0.;

		/* this is to kill collisional excitations from n=1 and n=2 in the
		 * case b hummer and storey option */
		if( opac.lgCaseB_HummerStorey )
		{

			/* this kills collisional excitation and ionization from the ground
			 * 1s 1S level, and the two singlet n=2 levels, 2s 1S and 2p 1P */
			ipLo = ipHe1s1S;
			iso.ColIoniz[ipHELIUM][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.nLevels[ipHELIUM][nelem]; ipHi++ )
			{
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].ColUL = 0.;
			}
			ipLo = ipHe2s1S;
			iso.ColIoniz[ipHELIUM][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.nLevels[ipHELIUM][nelem]; ipHi++ )
			{
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].ColUL = 0.;
			}
			ipLo = ipHe2p1P;
			iso.ColIoniz[ipHELIUM][nelem][ipLo] = 0.;
			for( ipHi=ipLo+1; ipHi<iso.nLevels[ipHELIUM][nelem]; ipHi++ )
			{
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].cs = 0.;
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].ColUL = 0.;
			}
		}
	}

	/* cfit( Atom num (H=1), Num elec, >=0 */
	DimaRate = (float)cfit( nelem+1, 2 , phycon.te );

	crate = DimaRate*(float)phycon.eden;
	/* NB hecolnc does not include secondaries since not total, but CollidRate does,
	 * since is total */
	iso.ColIoniz[ipHELIUM][nelem][0] = DimaRate*iso.lgColl_ionize[ipHELIUM];

	/* total collisional ionization rate 
	 * with both thermal and suprathermal electrons */
	/* there is no attempt to use scaled csupra here, since this is mostly 
	 * ionic and csupra will be zero.  For He itself, other routine is used.
	 * when this routine becomes fundamental for He a scale factor should
	 * use used.  TH85 Tab 10 suggests should be 0.5, but this is for
	 * their background rate - the rate derived here is self-consistent with unity */
	CollIonRate.CollidRate[0][nelem-1][nelem] = crate + Secondaries.csupra;

	/* cooling due to collisional ionization, which only includes thermal */
	CollIonRate.CollidRate[1][nelem-1][nelem] = (float)(crate*
		rfield.anu[Heavy.ipHeavy[nelem-1][nelem]-1]* EN1RYD);

	/* collisional ionizaiton of atomic helium */
	if( nelem == ipHELIUM )
	{
		iso.ColIoniz[ipHELIUM][nelem][ipHe2s3S] = he3c.c2sion/ phycon.eden*iso.lgColl_ionize[ipHELIUM];
		iso.ColIoniz[ipHELIUM][nelem][ipHe2p3P0] = he3c.c2pion/ phycon.eden*iso.lgColl_ionize[ipHELIUM];
		iso.ColIoniz[ipHELIUM][nelem][ipHe2p3P1] = he3c.c2pion/ phycon.eden*iso.lgColl_ionize[ipHELIUM];
		iso.ColIoniz[ipHELIUM][nelem][ipHe2p3P2] = he3c.c2pion/ phycon.eden*iso.lgColl_ionize[ipHELIUM];
		iso.ColIoniz[ipHELIUM][nelem][ipHe2s1S] = 0.;
		iso.ColIoniz[ipHELIUM][nelem][ipHe2p1P] = 0.;
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeCollid()\n", debug_fp );
#	endif
}
#if 0
/* this routine no longer used, replaced with iso_Photo */
/*===================================================================================*/
/* evaluate photoionization rates  */
static void HePhoto( long nelem)
{
	long int ipLo,
		ns,
		iplow , iphi , ipop;
	double funduc , cind;

#	ifdef DEBUG_FUN
	fputs( "<+>HePhoto()\n", debug_fp );
#	endif

	/* ground state will be only one for now */
	/* nelem-1 is stage of ionization */
	ns = 0;
	iplow = opac.ipElement[0][ns][nelem-1][nelem];
	iphi = opac.ipElement[1][ns][nelem-1][nelem];
	ipop = opac.ipElement[2][ns][nelem-1][nelem];

	iso.gamnc[ipHELIUM][nelem][ipHe1s1S] = 
		GammaBn(iplow,iphi,
	  ipop,iso.xIsoLevNIonRyd[ipHELIUM][nelem][0],&funduc , &cind)*PhotRate.PhotScaleOn;

	/* add on compton recoil ionization for atomic helium only */
	if( nelem==ipHELIUM )
	{
		/* >>chng 01 dec 19, add this heat, ion terms */
		iso.gamnc[ipHELIUM][nelem][ipHe1s1S] += recoil.CompRecoilIonRate[ipHELIUM];
		PhotRate.PhotoRate[2][ns][nelem-1][nelem] += recoil.CompRecoilHeatRate[ipHELIUM];
	}

	/* the heating rate */
	PhotRate.PhotoRate[0][ns][nelem-1][nelem] = iso.gamnc[ipHELIUM][nelem][0];
	PhotRate.PhotoRate[1][ns][nelem-1][nelem] = heat.HeatLowEnr*PhotRate.PhotScaleOn;
	PhotRate.PhotoRate[2][ns][nelem-1][nelem] = heat.HeatHiEnr*PhotRate.PhotScaleOn;

#	if 0
	if( nelem==ipHELIUM )
		fprintf(ioQQQ,"\n");
	fprintf(ioQQQ,"%li\t%.4e", nelem,
		PhotRate.PhotoRate[0][0][nelem-1][nelem] );

	fprintf(ioQQQ,"\t%.4e\t%.4e\n", iso.gamnc[ipHELIUM][nelem][ipHe1s1S] ,
		(PhotRate.PhotoRate[0][0][nelem-1][nelem]-iso.gamnc[ipHELIUM][nelem][ipHe1s1S])/PhotRate.PhotoRate[0][0][nelem-1][nelem] );
#	endif

	/* rest zero */
	fixit();/* add photoionzation from all excited levels */
	for( ipLo=ipHe2s3S; ipLo < iso.nLevels[ipHELIUM][nelem]; ipLo++ )
	{
		iplow = opac.ipElement[0][ns][nelem-1][nelem];
		iphi = opac.ipElement[1][ns][nelem-1][nelem];
		ipop = opac.ipElement[2][ns][nelem-1][nelem];

		iso.gamnc[ipHELIUM][nelem][ipLo] = 0.;
	}

	/* except for HeI itself, use he routines */
	if( nelem==ipHELIUM )
	{
		iso.gamnc[ipHELIUM][nelem][ipHe2s3S] = he3gams.he3gam[0];
		iso.gamnc[ipHELIUM][nelem][ipHe2p3P0] = he3gams.he3gam[1];
		iso.gamnc[ipHELIUM][nelem][ipHe2p3P1] = he3gams.he3gam[1];
		iso.gamnc[ipHELIUM][nelem][ipHe2p3P2] = he3gams.he3gam[1];
		iso.gamnc[ipHELIUM][nelem][ipHe2s1S] = phe1lv.he1gam[1];
		iso.gamnc[ipHELIUM][nelem][ipHe2p1P] = phe1lv.he1gam[1];
	}

	/* print old values the reevaluate whole thing */
	fprintf(ioQQQ,"oldddd\t%.2e\t%.2e\t%.2e\t%.2e\n",
		iso.gamnc[ipHELIUM][nelem][0],
		iso.gamnc[ipHELIUM][nelem][ipHe2s3S],
		iso.gamnc[ipHELIUM][nelem][ipHe2s1S],
		iso.gamnc[ipHELIUM][nelem][ipHe2p1P]);
	iso_Photo(ipHELIUM , nelem );
	fprintf(ioQQQ,"newwww\t%.2e\t%.2e\t%.2e\t%.2e\n",
		iso.gamnc[ipHELIUM][nelem][0],
		iso.gamnc[ipHELIUM][nelem][ipHe2s3S],
		iso.gamnc[ipHELIUM][nelem][ipHe2s1S],
		iso.gamnc[ipHELIUM][nelem][ipHe2p1P]);

#	ifdef DEBUG_FUN
	fputs( " <->HePhoto()\n", debug_fp );
#	endif
}
#	endif

/*===================================================================================*/
/* solve for level populations  */
static void HeLevel( long nelem)
{
	long int n,
		i,
		ipHi,
		ipLo,
		j,
		job,
		level,
		nerror1,
		nerror2;
	double create,
		destroy;
	int lgNegPop;
	double *pops;
	long int *ipiv ; /* MALLOC out to [iso.nLevels[ipHELIUM][nelem]] */
	/* this block of variables will be obtained and freed here */
	double
		*creation ,
		*error/*[iso.nLevels[ipHELIUM][nelem]]*/, 
		*work/*[iso.nLevels[ipHELIUM][nelem]]*/, 
		**SaveZ/*[iso.nLevels[ipHELIUM][nelem]+1][iso.nLevels[ipHELIUM]]*/, 
		**z/*[iso.nLevels[ipHELIUM][nelem]][iso.nLevels[ipHELIUM]]*/;

#	ifdef DEBUG_FUN
	fputs( "<+>HeLevel()\n", debug_fp );
#	endif

	assert(nelem < LIMELM );

	/* MALLOC some scratch space */

	if( (ipiv = (long int *)MALLOC(sizeof(long int)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ) )==NULL )
		bad_malloc();

	if( (creation = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ) )==NULL )
		bad_malloc();

	if( (error = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ) )==NULL )
		bad_malloc();

	if( (work = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ) )==NULL )
		bad_malloc();

	if( (pops = (double *)MALLOC(sizeof(double )*(unsigned)iso.nLevels[ipHELIUM][nelem] ) )==NULL )
		bad_malloc();

	/* now do the 2D arrays */
	if( (SaveZ = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ) ) ==NULL )
		bad_malloc();

	if(  (z = (double **)MALLOC(sizeof(double *)*(unsigned)(iso.nLevels[ipHELIUM][nelem]) ))==NULL  )
		bad_malloc();

	/* now do the second dimension */
	for( i=0; i<(iso.nLevels[ipHELIUM][nelem]); ++i )
	{
		if( (SaveZ[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]+2) ) )==NULL )
			bad_malloc();

		if( (z[i] = (double *)MALLOC(sizeof(double)*(unsigned)(iso.nLevels[ipHELIUM][nelem]+2) ) )==NULL )
			bad_malloc();
	}

	/* this will become the summed rec coef */
	create = 0.;
	/* fill in creation vector, rate levels are populated from the continuum */
	for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
	{
		double rec_radiative , rec_induced;

		/* radiative recombination */
		rec_radiative = iso.RadRecomb[ipHELIUM][nelem][n][ipRecRad]*
			iso.RadRecomb[ipHELIUM][nelem][n][ipRecNetEsc]*phycon.eden;
		assert( rec_radiative >= 0. );

		/* induced recombination */
		rec_induced = iso.RecomInducRate[ipHELIUM][nelem][n]*
			iso.PopLTE[ipHELIUM][nelem][n]*phycon.eden;

		/* total recombination to level n */
		creation[n] = rec_radiative + rec_induced;

		/* note that create will sum up to zero if helium cannot be ionized, 
		 * since cont esc prob 0 */
		create += creation[n];
	}
	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG && nelem==5 )
		{
			fprintf(ioQQQ,"recomrate");
			for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
			{
				fprintf(ioQQQ,"\t%e", creation[n] );
			}
			fprintf(ioQQQ,"\n"); 
		}
	}

	/* create is the total recombination rate, and this MUST be greater than zero */
	assert( create > 0. );

	/* total destruction rate, photo and collis ioniz first, CollidRate already
	 * includes suprathermals */
	destroy = iso.gamnc[ipHELIUM][nelem][0] + CollIonRate.CollidRate[0][nelem-1][nelem];
	DestCrt.xIonize[nelem][nelem-1] = destroy;

	/* >>chng 01 may 22, added CO to this treatment, (had always been included
	 * for old he1-he3 atom) */
	if( nelem == ipHELIUM )
	{
		/* add CO here, only for he itself, 
		 * destruction of CO but recombination for he */
		create += hevmolec.hevmol[ipCO]*1.10e-9 ;
		/* >>chng 01 nov 23, increase first index by one */
		/*z[iso.nLevels[ipHELIUM][nelem]][ipHe1s1S] += hevmolec.hevmol[ipCO]*1.10e-9 ;*/
		/*z[iso.nLevels[ipHELIUM][nelem]][ipHe1s1S] += hevmolec.hevmol[ipCO]*1.10e-9 ;*/
		creation[ipHe1s1S] += hevmolec.hevmol[ipCO]*1.10e-9 ;
	}

	{
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG /*&& (iteration==2)*/ )
		{
			if( nelem==ipHELIUM )
				fprintf(ioQQQ,"\n");
			fprintf(ioQQQ,"%li rec\tstore=%.4e", nelem,
				recom.RecombinRate[nelem][nelem-1] );

			fprintf(ioQQQ,"\there=%.4e\tratio=%.4e\n", create ,
				(recom.RecombinRate[nelem][nelem-1]-create)/create );
			/*fprintf( ioQQQ," %li\t%.2e \n", nelem, recom.RecombinRate[nelem][nelem-1] );*/
		}
	}

	/* ratio of ion to atom, create MUST be greater than zero since it is 
	 * the recombination rate */
	if( create > 0. )
	{
		iso.xIonSimple[ipHELIUM][nelem] = (float)(destroy / create);
	}
	else
	{
		/* rec rate can't be zero */
		total_insanity();
	}

#	define TOOSMALL 1e-10
	if( iso.xIonSimple[ipHELIUM][nelem] < TOOSMALL )
	{
		/* >>chng 01 nov 23, increase upper limit by 2 */
		/*for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )*/
		for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
		{
			/*iso.DepartCoef[ipHYDROGEN][nelem][i] = 1.;*/
			pops[n] =  0.;
		}
		lgNegPop = FALSE;
		iso.xIonRatio[ipHELIUM][nelem] = 0.;
	}
	else
	{
		/* master balance equation, use when significant population */
		/* >>chng 01 nov 23, increase upper limit by 2 */
		/*for( level=0; level < iso.nLevels[ipHELIUM][nelem]; level++ )*/
		for( level=ipHe1s1S; level < iso.nLevels[ipHELIUM][nelem]; level++ )
		{
			/* 
			 * all process depopulating level
			 */
			/* there is no attempt to use scaled csupra here, since this is mostly 
			 * ionic and csupra will be zero.  For He itself, other routine is used.
			 * when this routine becomes fundamental for He a scale factor should
			 * use used.  TH85 Tab 10 suggests should be 0.5, but this is for
			 * their background rate - the rate derived here is self-consistent with unity */
			/* NB collisional ionization rate already includes suprathermals */
			z[level][level] = iso.gamnc[ipHELIUM][nelem][level] + 
			  iso.ColIoniz[ipHELIUM][nelem][level]* phycon.eden ;
			/* fixit(); in final version move this up a line, add to all levels 
			 * for now only add to groud to keep parallel to bidiag
			 * NB - comment in previous statement says this already included - which is wrong? */
			if( level==ipHe1s1S ) z[level][level] +=Secondaries.csupra;

			/* all processes populating level from below */
			for( ipLo=ipHe1s1S; ipLo < level; ipLo++ )
			{
				z[ipLo][level] = 
					-EmisLines[ipHELIUM][nelem][level][ipLo].ColUL* phycon.EdenHCorr * 
				  EmisLines[ipHELIUM][nelem][level][ipLo].gHi/EmisLines[ipHELIUM][nelem][level][ipLo].gLo*
				  iso.Boltzmann[ipHELIUM][nelem][level][ipLo]  -
				  (double)EmisLines[ipHELIUM][nelem][level][ipLo].pump ;

				/* pumping out of here to lower level */
				/* >>chng 02 feb 16, add casts to double */
				z[level][level] += (double)EmisLines[ipHELIUM][nelem][level][ipLo].pump *
				  (double)EmisLines[ipHELIUM][nelem][level][ipLo].gLo/(double)EmisLines[ipHELIUM][nelem][level][ipLo].gHi;

				/* collisions out of here to lower level */
				z[level][level] += 
					EmisLines[ipHELIUM][nelem][level][ipLo].ColUL * phycon.EdenHCorr;

				/* radiative decays out of here to lower level */
				z[level][level] += 
					EmisLines[ipHELIUM][nelem][level][ipLo].Aul*
				  (EmisLines[ipHELIUM][nelem][level][ipLo].Pesc + 
				  EmisLines[ipHELIUM][nelem][level][ipLo].Pdest);
			}

			/* all processes populating level from above */
			/* >>chng 01 nov 23, increase upper limit by two */
			for( ipHi=level + 1; ipHi < iso.nLevels[ipHELIUM][nelem]; ipHi++ )
			{
				double RadDecay;
				RadDecay =
					EmisLines[ipHELIUM][nelem][ipHi][level].Aul*
				  (EmisLines[ipHELIUM][nelem][ipHi][level].Pesc + 
				  EmisLines[ipHELIUM][nelem][ipHi][level].Pdest);

				z[ipHi][level] = 
					-(RadDecay + 
				  (double)EmisLines[ipHELIUM][nelem][ipHi][level].pump*
				  (double)EmisLines[ipHELIUM][nelem][ipHi][level].gLo/(double)EmisLines[ipHELIUM][nelem][ipHi][level].gHi + 
				  EmisLines[ipHELIUM][nelem][ipHi][level].ColUL *phycon.EdenHCorr);

				/* pumping out of here to upper level */
				z[level][level] += (double)EmisLines[ipHELIUM][nelem][ipHi][level].pump ;

				/* collisions out of here to upper level */
				z[level][level] += EmisLines[ipHELIUM][nelem][ipHi][level].ColUL *
					EmisLines[ipHELIUM][nelem][ipHi][level].gHi / EmisLines[ipHELIUM][nelem][ipHi][level].gLo *
					iso.Boltzmann[ipHELIUM][nelem][ipHi][level]*phycon.EdenHCorr;
			}
		}

		/* here is charge transfer ionization of other species with */
		z[ipHe1s1S][ipHe1s1S] += ChargTran.HeCharExcRec[nelem][nelem-1]*xIonFracs[ipHELIUM][2] + 
		  ChargTran.HeCharExcRec[nelem][nelem-1]*xIonFracs[ipHYDROGEN][2];
		/*if( nelem==5) fprintf(ioQQQ,"pumppp %.2e\n", ChargTran.HeCharExcRec[nelem][nelem-1]*xIonFracs[ipHELIUM][2] + 
		  ChargTran.HeCharExcRec[nelem][nelem-1]*xIonFracs[ipHYDROGEN][2]);*/

		/* add on charge transfer recombination*/
		/* add it to 1s for now, can't really do any better */
		/* >>chng 01 apr 28, had been added to ground, energy conservation off?? */
		/* in current data set (june, 01) next four ct reactions are all zero */
		/* >>chng 01 nov 23, upper limit increased by 2 */
		/*z[iso.nLevels[ipHELIUM][nelem]][ipHe1s1S] += */
		creation[ipHe1s1S] += 
			ChargTran.HeCharExcIon[nelem][nelem-1]*xIonFracs[ipHELIUM][1] + 
			ChargTran.HeCharExcIon[nelem][nelem-1]*xIonFracs[ipHYDROGEN][1] ;

		/* >>chng 01 jun 21, add this in */
		/* charge transfer with helium itself, onto everything heavier */
		if( nelem == ipHELIUM )
		{
			int nel_col , ion;

			/* add on charge transfer ionization of helium,
			 * recombination for other element is ionization of helium */
			ChargTran.HeCharExcIonTotal = 0.;
			/* loop up from Li, the next heavier element */
			for( nel_col=ipLITHIUM; nel_col<LIMELM; ++nel_col)
			{
				double hold_one = 0.;
				/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
				 * for helium nelem=1, ion must go up to 3 */
				/* check that array bounds not exceeded */
				assert( (nel_col) < (LIMELM+1) );
				for( ion=2; ion<=nel_col+2; ++ion )
				{
					/* charge transfer ionization of He, recombination for other species */
					hold_one = ChargTran.HeCharExcRec[nel_col][ion-2]*xIonFracs[nel_col][ion];
					ChargTran.HeCharExcIonTotal += hold_one;
				}
			}
			z[ipHe1s1S][ipHe1s1S] += ChargTran.HeCharExcIonTotal;

			/* charge transfer recombination of helium,
			 * which is ionization of the heavy element */
			ChargTran.HeCharExcRecTotal = 0.;
			for( nel_col=3; nel_col<LIMELM; ++nel_col)
			{
				/* this is ion on the abundances scale, 1 is atom, so goes up to nelem+1,
				 * for helium nelem=1, ion must go up to 3 */
				/* check that array bounds not exceeded */
				assert( (nel_col) < (LIMELM+1) );
				for( ion=1; ion<=nel_col+1; ++ion )
				{
					/* charge transfer ionization of H, recombination for other species */
					ChargTran.HeCharExcRecTotal += 
						ChargTran.HeCharExcIon[nel_col][ion-1]*xIonFracs[nel_col][ion];
				}
			}
			/* >>chng 01 nov 23, increase first limit by 2 */
			/*z[iso.nLevels[ipHELIUM][nelem]][ipHe1s1S] += ChargTran.HeCharExcRecTotal;*/
			creation[ipHe1s1S] += ChargTran.HeCharExcRecTotal;
		}

		/* add on advection terms, normally zero */
		/* incoming material is in ground state */
		/* >>chng 01 nov 23, increase first limit by 2 */
		/*z[iso.nLevels[ipHELIUM][nelem]][ipHe1s1S] += dynamics.Recomb[nelem-1][nelem];*/
		creation[ipHe1s1S] += dynamics.Recomb[nelem-1][nelem];

		z[ipHe1s1S][ipHe1s1S] += dynamics.Photo;

		fixit();/* neglect secondary ionization for now - csupra occurs in above look [level][level] */
#		if 0
		if( Secondaries.Hx12[0][1] > 0. )
		{
			/* now add on supra thermal excitation */
			/* >>chng 01 nov 23, increase upper limit by 2 */
			for( level=1; level < iso.nLevels[ipHELIUM][nelem]; level++ )
			{
				double RateUp , RateDown;

				RateUp = Secondaries.Hx12[MIN2(nelem,1)][level];
				RateDown = RateUp * (double)EmisLines[ipHELIUM][nelem][level][ipH1s].gLo /
					(double)EmisLines[ipHELIUM][nelem][level][ipH1s].gHi;

				/* stuff in min after Hx12 evaluates to 0 for atom, or 1 for ion */
				/* total rate out of lower level */
				fixit();/* these are all wrong */
				z[ipH1s][ipH1s] += RateUp;

				/* rate from the upper level to ground */
				z[level][ipH1s] -= RateDown ;

				/* rate from ground to upper level */
				z[ipH1s][level] -= RateUp ;

				z[level][level] += RateDown;  
			}
		}
#		endif

		{
			/* following should be set true to print information */
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( (trace.lgTrace && trace.lgHeBugFull && (nelem == trace.ipZTrace)) || (nelem==5 &&DEBUG) )
			{
				fprintf( ioQQQ, "  pop level     others => (HeliumLevelPop)\n" );
				/* >>chng 01 nov 23 increase upper limit by 2 */
				for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
				{
					fprintf( ioQQQ, "       HII%2ld", n );
					/* limit is <= since last has rec coef */
					/* >>chng 01 nov 23 increase upper limit by 2 */
					for( j=0; j < iso.nLevels[ipHELIUM][nelem]; j++ )
					{
						fprintf( ioQQQ,"\t%.9e", z[j][n] );
					}
					fprintf( ioQQQ, "\n" );
				}
			}
		}

		/* save matrix */
		/* >>chng 01 nov 23, increase both upper limits by 2 */
		for( ipLo=ipHe1s1S; ipLo < iso.nLevels[ipHELIUM][nelem]; ipLo++ )
		{
			for( n=ipHe1s1S; n < (iso.nLevels[ipHELIUM][nelem]); n++ )
			{
				SaveZ[n][ipLo] = z[n][ipLo];
			}
		}

		/* which matrix solver? */
		if( strcmp(TypMatrx.chMatrix,"matin1 ") == 0 )
		{
			/* this is the usual matrix inversion method, slightly faster */
			/*matin1();*/
			nerror1 = 0;
			if( nerror1 != 0 )
			{
				fprintf( ioQQQ, " helike matrix error, stop.\n" );
				puts( "[Stop in helike]" );
				cdEXIT(1);
			}
		}

		/* this is the default */
		else if( strcmp(TypMatrx.chMatrix,"linpack") == 0 )
		{
			double *amat;
#			ifdef AMAT
#				undef AMAT
#			endif
			/* >>chng 01 nov 23, increase upper limit by 2 */
/*#			define AMAT(I_,J_)	(*(amat+(I_)*(iso.nLevels[ipHELIUM][nelem])+(J_)))*/
#			define AMAT(I_,J_)	(*(amat+(I_)*(iso.nLevels[ipHELIUM][nelem])+(J_)))
			/* MALLOC space for the  1-d array */
			/* >>chng 01 nov 23, increase upper limit by 2 */
			if( (amat=(double*)MALLOC( (sizeof(double)*(unsigned)((iso.nLevels[ipHELIUM][nelem])*(iso.nLevels[ipHELIUM][nelem])) ))) == NULL )
			{
				fprintf( ioQQQ, " helike MALLOC amat error\n" );
				puts( "[Stop in helike]" );
				cdEXIT(1);
			}

			/* this is the default */
			for( j=0; j < iso.nLevels[ipHELIUM][nelem]; j++ )
			{
				for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
				{
					AMAT(n,j) = z[n][j];
				}
				/*bvec[j] = z[iso.nLevels[ipHELIUM][nelem]][j];
				bvec[j] = creation[j];*/
			}
			DGETRF(iso.nLevels[ipHELIUM][nelem],iso.nLevels[ipHELIUM][nelem],
			  amat,iso.nLevels[ipHELIUM][nelem],ipiv,&nerror1);

			/*DGETRS('N',iso.nLevels[ipHELIUM][nelem],1,amat,iso.nLevels[ipHELIUM][nelem],ipiv,
			  bvec,iso.nLevels[ipHELIUM][nelem],&nerror2);*/
			DGETRS('N',iso.nLevels[ipHELIUM][nelem],1,amat,iso.nLevels[ipHELIUM][nelem],ipiv,
			  creation,iso.nLevels[ipHELIUM][nelem],&nerror2);

			if( nerror1 != 0 || nerror2 != 0 )
			{
				fprintf( ioQQQ, " helike dgetrs error\n" );
				puts( "[Stop in helike]" );
				cdEXIT(1);
			}
			/* now put results back into z so rest of code treates only
			 * one case - as if matin1 had been used */
			for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
			{
				/*z[iso.nLevels[ipHELIUM][nelem]][n] = bvec[n];
				creation[n] = bvec[n];*/
			}
			free( amat);
		}
		else if( strcmp(TypMatrx.chMatrix,"veclib ") == 0 )
		{
			double /*amat[LMHLVL+2][LMHLVL+2], */ rcond;
			/* Jason found this one on the Exemplar, distributed source just stops */
			fprintf( ioQQQ, " this has not been checked since H atom conv\n" );
			for( j=ipH1s; j < iso.nLevels[ipHELIUM][nelem]; j++ )
			{
				/*for( n=ipH1s; n < iso.nLevels[ipHELIUM][nelem]; n++ )
				{
					amat[n][j] = z[n][j];
				}*/
				/*bvec[j] = z[iso.nLevels[ipHELIUM][nelem]][j];
				bvec[j] = creation[j];*/
			}
			job = 0;
			rcond = 0.;
			/*lint -e740 */
			/*lint -e826 */
			/*dgeco((double*)amat,iso.nLevels[ipHYDROGEN][nelem]+2,iso.nLevels[ipHYDROGEN]+1,ipiv,rcond, work);
			dgesl((double*)amat,iso.nLevels[ipHYDROGEN][nelem]+2,iso.nLevels[ipHYDROGEN]+1,ipiv,bvec, job);*/
			dgeco((double*)z,iso.nLevels[ipHELIUM][nelem],iso.nLevels[ipHELIUM][nelem],ipiv,rcond, work);
			dgesl((double*)z,iso.nLevels[ipHELIUM][nelem],iso.nLevels[ipHELIUM][nelem],ipiv,creation, job);
			/*lint +e740 */
			/*lint +e826 */
			/* now put results back into z so rest of code treates only
			 * one case - as if matin1 had been used */
			for( n=0; n < iso.nLevels[ipHELIUM][nelem]; n++ )
			{
				/*z[iso.nLevels[ipHELIUM][nelem]][n] = bvec[n];
				creation[n] = bvec[n];*/
			}
			puts( "[Stop in helike]" );
			cdEXIT(1);
		}
		else
		{
			fprintf( ioQQQ, " chMatrix type insane in helike, was%7.7s\n", 
			  TypMatrx.chMatrix );
			puts( "[Stop in helike]" );
			cdEXIT(1);
		}
		/* end of branches for which matrix solution, now check if valid */

		/* this will eventually become the ratio of ionized to neutral hydrogenic 
		 * species, create sum of level pops per ion first */
		iso.xIonRatio[ipHELIUM][nelem] = 0.;
		lgNegPop = FALSE;
		/* >>chng 01 nov 23, increase upper limit by 2 */
		for( level=0; level < iso.nLevels[ipHELIUM][nelem]; level++ )
		{

			pops[level] =  creation[level];

			/* check for negative level populations */
			if( pops[level] < 0. )
				lgNegPop = TRUE;

			/* create sum of populations,
			 * pops are actually populations relative to ion */
			iso.xIonRatio[ipHELIUM][nelem] += (float)pops[level];

			if( iso.PopLTE[ipHELIUM][nelem][level] > 0. )
			{
				/* the NLTE LTE departure coefficients */
				iso.DepartCoef[ipHELIUM][nelem][level] = 
					(iso.Pop2Ion[ipHELIUM][nelem][level]/
					(iso.PopLTE[ipHELIUM][nelem][level]* phycon.eden) );
			}
			else
			{
				iso.DepartCoef[ipHELIUM][nelem][level] = 0.;
			}
		}

		/* >>chng 02 jan 22, actually use real ratios, not just ground */
		/* this is sum of all populations in model, div by ion 
		 * create ratio of ion pops to total atom */
		if( iso.xIonRatio[ipHELIUM][nelem] > SMALLFLOAT )
		{
			iso.xIonRatio[ipHELIUM][nelem] = 1.f/iso.xIonRatio[ipHELIUM][nelem];
		}
		else
		{
				iso.xIonRatio[ipHELIUM][nelem] = 0.;
		}
	}

	/* check on the sum of the populations */
	if( lgNegPop || iso.xIonRatio[ipHELIUM][nelem] < 0.)
	{
		fprintf( ioQQQ, 
			" helevel finds negative He-like ion fraction for nelem=%ld using routine HeLevel, IonFrac= %10.3e simple=%10.3e TE=%10.3e ZONE=%4ld\n", 
		  nelem, 
		  iso.xIonRatio[ipHELIUM][nelem], 
		  iso.xIonSimple[ipHELIUM][nelem], 
		  phycon.te, 
		  nzone );

		fprintf( ioQQQ, " level pop are:\n" );
		for( i=0; i < iso.nLevels[ipHELIUM][nelem]; i++ )
		{
			fprintf( ioQQQ," %8.1e;", pops[i] );
			if( (i!=0) && !(i%10) ) fprintf( ioQQQ,"\n" );
		}
		fprintf( ioQQQ, "\n" );
		negcon();
		ShowMe();
		puts( "[Stop in helike]" );
		cdEXIT(1);
	}

	/* get level populations, two cases, 
	 * first, may be zero since all cases drop down to here, 
	 * this branch of if is for trivial abundance, so zero out species */
	/* >>chng 01 apr 21, add second test on high stage being he-like -
	 * there are tests where this is hydrogen like, and jump would be over
	 * two stages of ionization - do not do this */
	if( iso.xIonRatio[ipHELIUM][nelem] >= 0. && iso.xIonRatio[ipHELIUM][nelem] < TOOSMALL  &&
		IonRange.IonHigh[nelem] == nelem )
	{
		iso.xIonRatio[ipHELIUM][nelem] = 0.;
		/* decrement highest stage of ionization, must not
		 * go up to He-like being ionized again */
		IonRange.IonHigh[nelem] = nelem-1;
		/* >>chng 01 apr 21, added second drop on IonLow -
		 * there was a model where light atoms where stripped and he-like test triggered,
		 * which resulted in insanity downstream since IonHigh and IonLow were equal */
		IonRange.IonLow[nelem] = MIN2( IonRange.IonLow[nelem] , IonRange.IonHigh[nelem]-1 );

		/* now zero out high stages we just cleared out */
		for( ipHi=ipHe2s3S; ipHi < iso.nLevels[ipHELIUM][nelem]; ipHi++ )
		{
			iso.Pop2Ion[ipHELIUM][nelem][ipHi] = 0.;
			for( ipLo=ipHe1s1S; ipLo < ipHi; ipLo++ )
			{
				/* population of lower level rel to ion */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopLo = 0.;

				/* population of upper level rel to ion */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopHi =  0.;

				/* population of lower level rel to ion, corrected for stim em */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopOpc =  0.;

				/* local ots destruction, cm^-3 s^-1 */
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].ots =  0.;
			}
		}

		if( trace.lgTrace && trace.lgHeBug )
		{
			fprintf(ioQQQ,"  HeLike for Z=%li finds small population set to zerozero.\n",nelem);
		}
	}
	/* this is the main branch that applies when we have non-trivial abundance */
	else
	{
		/* level populations 
		 * this brance we have significant population as sum of levels per ion,
		 * store inverse, ion per atom, here.  For non-H species this is ratio
		 * of highest to next highest stage of ionization */

		/* now find emission in each line */
		for( ipHi=0; ipHi<iso.nLevels[ipHELIUM][nelem]; ++ipHi )
		{
			iso.Pop2Ion[ipHELIUM][nelem][ipHi] = (float)pops[ipHi];

			for( ipLo=0; ipLo<ipHi; ++ipLo )
			{
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopLo = pops[ipLo];
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopHi = pops[ipHi];
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopOpc = (pops[ipLo] - pops[ipHi]*
					EmisLines[ipHELIUM][nelem][ipHi][ipLo].gLo/EmisLines[ipHELIUM][nelem][ipHi][ipLo].gHi);
				/* some lines had strong masers, quench for now */
				fixit();
				EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopOpc = MAX2(SMALLFLOAT,EmisLines[ipHELIUM][nelem][ipHi][ipLo].PopOpc );
			}
		}

		/* for now just do ground state, like bidiag will do */
#		if 0
		if( iso.xIonRatio[ipHELIUM][nelem]> SMALLFLOAT )
		{
			iso.xIonRatio[ipHELIUM][nelem] = 1.f/(float)pops[0];
		}
		else
		{
			iso.xIonRatio[ipHELIUM][nelem] = 0.;
		}
#		endif
	}

	{
		/* following should be set true to print information */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG /*&& (iteration==2)*/ )
		{
			if( nelem==ipHELIUM )
				fprintf(ioQQQ,"\n");
			fprintf(ioQQQ,"%li\t%.4e", nelem,
				iso.xIonRatio[ipHELIUM][nelem] );
			if( xIonFracs[nelem][nelem+1]>SMALLFLOAT )
			{
				float ratio = xIonFracs[nelem][nelem+1]/ xIonFracs[nelem][nelem];

				fprintf(ioQQQ,"\t%.4e\t%.4e", ratio ,
					(ratio-iso.xIonRatio[ipHELIUM][nelem])/ratio );
			}
			fprintf(ioQQQ,"\n");
		}

	}
	{
		/* following should be set true to print information */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG && (nelem==ipCARBON)/**/ )
		{
			{
				fprintf( ioQQQ, "     He-like Z=%2ld H2OVH1=",nelem);
				fprintf( ioQQQ,"%10.3e", iso.xIonRatio[ipHELIUM][nelem]);
				fprintf( ioQQQ, " simple=");
				fprintf( ioQQQ,"%10.3e", iso.xIonSimple[ipHELIUM][nelem]);
				fprintf( ioQQQ," dest, creat rates %.2e %.2e", destroy , create);
				fprintf( ioQQQ, "\n"); 
			}
		}
	}

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "     He-like Z=%2ld H2OVH1=",nelem);
		fprintf( ioQQQ,"%10.3e", iso.xIonRatio[ipHELIUM][nelem]);
		fprintf( ioQQQ, " simple=");
		fprintf( ioQQQ,"%10.3e", iso.xIonSimple[ipHELIUM][nelem]);
		fprintf( ioQQQ," dest, creat rates %.2e %.2e", destroy , create);
		fprintf( ioQQQ, "\n"); 
	}

	free( ipiv );
	free( creation );
	free( error );
	free( work );
	free( pops );

	/* now do the second dimension */
	for( i=0; i<iso.nLevels[ipHELIUM][nelem]; ++i )
	{
		free( SaveZ[i] );

		free( z[i] );
	}

	free( SaveZ );
	free( z );

#	ifdef DEBUG_FUN
	fputs( " <->HeLevel()\n", debug_fp );
#	endif
}

/* HeCSInterp interpolate on collision strengths for element nelem */
static float HeCSInterp( long nelem , long ipHi , long ipLo )
{
	long ipArray,
		/* save local copy of array indices, which will be altered below due to splitting of 2 3P */
		ipLoNew = ipLo,
		ipHiNew = ipHi;
	int lgColl_on;
	float cs,
		factor ,
		flow;

#	ifdef DEBUG_FUN
	fputs( "<+>HeCSInterp()\n", debug_fp );
#	endif

	assert( nelem > ipHYDROGEN );
	assert( nelem < LIMELM );

	/* this factor will turn off some collisions, both terms normally 1,
	 * set to zero with atom he-like collisions command */
	if( he_levels[ipLo].n == he_levels[ipHi].n )
	{
		lgColl_on = iso.lgColl_l_mixing[ipHELIUM];
	}
	else
	{
		lgColl_on = iso.lgColl_excite[ipHELIUM];
	}

#	if 0
	fixit();/* should not return here - can add proper l-mixing cs */
	if( nelem > ipHELIUM )
	{
		cs = 0.1f;
		fixit();/* upt l-mix collisions here too */
#		ifdef DEBUG_FUN
		fputs( " <->HeCSInterp()\n", debug_fp );
#		endif

		return(cs*lgColl_on);
	}
#	endif

	/* for most of the helium iso sequence, the order of the J levels within 2 3P 
	 * in increasing energy, is 0 1 2 - the exception is atomic helium itself,
	 * which is swapped, 2 1 0 */

	/* this branch is for upper and lower levels within 2p3P */
	/* the atomic data did not split up the 2 ^3P levels, so the array indices are off by 2 for levels higher than this.
	 * we will reset the index ipLoNew and ipHiNew to correct for this */
	if( ipLoNew >= ipHe2p3P0 && ipHiNew <= ipHe2p3P2 )
	{
		/* this branch, both upper and lower are within 2 ^3P */
		/* >>chng 01 jul 20, add branch for charge */
		if( nelem==ipHELIUM )
		{
			/* >>refer he1 cs	Berring, Keith, 2001, private communication - email follows
			> Dear Gary,
			> I could not find any literature on the He fine-structure
			> problem (but I didn't look very hard, so there may be 
			> something somewhere). However, I did a quick R-matrix run to 
			> see what the magnitude of the collision strengths are... At 
			> 1000K, I get the effective collision strength for 2^3P J=0-1, 
			>  0-2, 1-2 as 0.8,0.7,2.7; for 10000K I get 1.2, 2.1, 6.0
			*/
			if( ipLoNew == ipHe2p3P0 && ipHiNew == ipHe2p3P1 )
			{
				cs = 1.2f;
			}
			else if( ipLoNew == ipHe2p3P0 && ipHiNew == ipHe2p3P2 )
			{
				cs = 2.1f;
			}
			else if( ipLoNew == ipHe2p3P1 && ipHiNew == ipHe2p3P2 )
			{
				cs = 6.0f;
			}
			else
			{
				cs = 1.0f;
				total_insanity();
			}
		}
		else
		{
			/* >>chng 01 jul 20, had been 0.1, change to 1 to be more like Brocklehurst numbers*/
			cs = 1.0f;
		}

#		ifdef DEBUG_FUN
		fputs( " <->HeCSInterp()\n", debug_fp );
#		endif

		return(cs*lgColl_on);
	}

	/* the atomic data did not split up the 2 ^3P levels, so the array indices are off by 2 for levels higher than this.
	 * we will reset the index ipLoNew and ipHiNew to correct for this */

	/* this may be used for splitting up the collision strength within 2 3P when
	 * the lower level is withint 2 3P */
	factor = 1.;

	/* fix lower index if it lies above J=0 of 2 3P */
	if( ipLoNew > ipHe2p3P0 )
	{
		/* two cases, within 2 ^3P or above it */
		if( ipLoNew > ipHe2p3P2 )
		{
			/* above it, simply subtract 2 from index */
			ipLoNew -= 2;
		}
		else
		{
			/* within it, use index for ipHe2p3P0 */
			ipLoNew = ipHe2p3P0;
			/* factor is ratio of statistical weights of level to term */
			if( ipLoNew == ipHe2p3P0 )
			{
				factor = 1.f/9.f;
			}
			else if( ipLoNew == ipHe2p3P1 )
			{
				factor = 3.f/9.f;
			}
			else if( ipLoNew == ipHe2p3P2 )
			{
				factor = 5.f/9.f;
			}
		}
	}

	/* fix upper index */
	if( ipHiNew >= ipHe2p3P0 )
	{
		/* two cases, within 2 ^3P or above it */
		if( ipHiNew > ipHe2p3P2 )
		{
			/* above it, simply subtract 2 from index */
			ipHiNew -= 2;
		}
		else
		{
			/* this is a collision to 2 3P 
			 * first find how to scale the cs with stat weight */
			if( ipHiNew == ipHe2p3P0 )
			{
				factor = 1.f/9.f;
			}
			else if( ipHiNew == ipHe2p3P1 )
			{
				factor = 3.f/9.f;
			}
			else if( ipHiNew == ipHe2p3P2 )
			{
				factor = 5.f/9.f;
			}

			/* since we are within it, use index for ipHe2p3P0 */
			ipHiNew = ipHe2p3P0;
		}
	}

	fixit();/* CSTemp is array of temperatures where cs is defined, this
			 * should be function of nelem for case when we have cs data
			 * for elements other than He. */
			/* currently HeCS is filled with default values of 0.1 for
			 * all cs for elements heavier than He,
			 * and for all transitions with no quantal data */
	/* check whether we are outside temperature array bounds,
	 * and use extreme value if we are */
	if( phycon.alogte <= CSTemp[0] )
	{
		cs = HeCS[nelem][ipHiNew][ipLoNew][0];
	}
	else if( phycon.alogte >= CSTemp[helike.nCS-1] )
	{
		cs = HeCS[nelem][ipHiNew][ipLoNew][helike.nCS-1];
	}
	else
	{
		/* find which array element within the cs vs temp array */
		ipArray = (long)((phycon.alogte - CSTemp[0])/(CSTemp[1]-CSTemp[0]));
		assert( ipArray < helike.nCS );
		assert( ipArray >= 0 );

		/* when taking the average, this is the fraction from the lower temperature value */
		flow = (float)( (phycon.alogte - CSTemp[ipArray])/
			(CSTemp[ipArray+1]-CSTemp[ipArray])) ;
		assert( (flow >= 0.) && (flow <= 1.) );

		cs = HeCS[nelem][ipHiNew][ipLoNew][ipArray] * (1.f-flow) +
			HeCS[nelem][ipHiNew][ipLoNew][ipArray+1] * flow;
	}

	/* take factor into account, usually 1, ratio of stat weights if within 2 3P */
	/* *iso.lgColl_excite[ipHELIUM] factor kills excitations 
	 * with command atom he-like collisions excit off */
	cs *= factor*lgColl_on;

#if	0
	if( ( nelem==ipHELIUM ) && (cs > 0.) )
			fprintf(ioQQQ,"%li\t%li\t%li\t-\t%li\t%li\t%li\t%.2e\texcitation\n", 
			he_levels[ipLoNew].n , he_levels[ipLoNew].s , he_levels[ipLoNew].l ,
			he_levels[ipHiNew].n , he_levels[ipHiNew].s , he_levels[ipHiNew].l , cs);
#endif

	/* this branch, n, s the same, but l => +/- 1 
	 * and only for levels that are energy-degenerate */
	if( (he_levels[ipLo].n >2 ) && 
	    (he_levels[ipLo].n == he_levels[ipHi].n ) &&
	    (he_levels[ipLo].s == he_levels[ipHi].s ) &&
	 abs(he_levels[ipLo].l - he_levels[ipHi].l ) == 1 )
	{
		 double	cs_lmix = CS_l_mixing(
			/* the chemical element, 1 for He */
			nelem ,
			/* the lower level index within the He model, 0 for ground - used to get n, l */
			ipLo,
			/* the upper level index */
			ipHi)*
			/* this is 1 by default, set to zero with the
			 * ATOM HE-LIKE COLLISIONS L-MIXING OFF command */
			lgColl_on;

		/* add to existing collision strengths
		 * this CS should be multiplied by the proton density,
		 * but the code will use the electron density, 
		 * so ratio of proton to electron densities precorrects for this. */
		cs += (float)(cs_lmix * xIonFracs[ipHYDROGEN][2]/phycon.eden);
	}

	return(cs);

#	ifdef DEBUG_FUN
	fputs( " <->HeCSInterp()\n", debug_fp );
#	endif
}

/* routine to punch table needed for AGN3 - collision strengths of HeI */
void AGN_He1_CS( FILE *ioPun )
{

	long int i;

	/* list of temperatures where cs will be printed */
#	define NTE 5 
	double TeList[NTE] = {6000.,10000.,15000.,20000.,25000.};

#	ifdef DEBUG_FUN
	fputs( "<+>AGN_He1_CS()\n", debug_fp );
#	endif

	/* put on a header */
	fprintf(ioPun, "Te\t2 3s 33s\n");

	for( i=0; i<NTE; ++i )
	{
		phycon.te = (float)TeList[i];
		tfidle(FALSE);

		fprintf(ioPun , "%.0f\t", 
			TeList[i] );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3s3S , ipHe2s3S ) );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3p3P , ipHe2s3S ) );
		fprintf(ioPun , "%.2f\t", 
			HeCSInterp( 1 , ipHe3d3D , ipHe2s3S ) );
		fprintf(ioPun , "%.3f\t", 
			HeCSInterp( 1 , ipHe3d1D , ipHe2s3S ) );
		/*fprintf(ioPun , "%.1f\t%.1f\t%.1f\n", */
		fprintf(ioPun , "%.1f\n", 
			HeCSInterp( 1 , ipHe2p3P0 , ipHe2s3S ) +
			HeCSInterp( 1 , ipHe2p3P1 , ipHe2s3S ) +
			HeCSInterp( 1 , ipHe2p3P2 , ipHe2s3S ));
	}

#	ifdef DEBUG_FUN
	fputs( " <->AGN_He1_CS()\n", debug_fp );
#	endif

	return;
}

/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
static double He_RRCoef_Te( long nelem , long n )
{
	static long int ipTe=-1;
	double rate = 0.;
	long i;

#	ifdef DEBUG_FUN
	fputs( "<+>He_RRCoef_Te()\n", debug_fp );
#	endif

	if( ipTe < 0 )
	{
		/* te totally unknown */
		if( phycon.alogte < TeRRCoef[0] || phycon.alogte > TeRRCoef[N_HE_TE_RECOMB-1] )
		{
			fprintf(ioQQQ," He_RRCoef_Te called with te out of bounds \n");
			puts( "[Stop in He_RRCoef_Te]" );
			cdEXIT(1);
		}
		/* now search for temperature */
		for( i=0; i<N_HE_TE_RECOMB-1; ++i )
		{
			if( phycon.alogte > TeRRCoef[i] && phycon.alogte <= TeRRCoef[i+1] )
			{
				/* found the temperature - use it */
				ipTe = i;
				break;
			}
		}
		assert( (ipTe >=0) && (ipTe < N_HE_TE_RECOMB-1)  );
			
		}
	else if( phycon.alogte < TeRRCoef[ipTe] )
	{
		/* temp is too low, must also lower ipTe */
		assert( phycon.alogte > TeRRCoef[0] );
		/* decrement ipTe until it is correct */
		while( phycon.alogte < TeRRCoef[ipTe] && ipTe > 0)
			--ipTe;
	}
	else if( phycon.alogte > TeRRCoef[ipTe+1] )
	{
		/* temp is too high */
		assert( phycon.alogte <= TeRRCoef[N_HE_TE_RECOMB-1] );
		/* increment ipTe until it is correct */
		while( phycon.alogte > TeRRCoef[ipTe+1] && ipTe < N_HE_TE_RECOMB-1)
			++ipTe;
	}

	assert( (ipTe >=0) && (ipTe < N_HE_TE_RECOMB-1)  );

	/* ipTe should now be valid */
	assert( phycon.alogte >= TeRRCoef[ipTe] && phycon.alogte <= TeRRCoef[ipTe+1] && ipTe < N_HE_TE_RECOMB-1 );

	rate = (phycon.alogte - TeRRCoef[ipTe]) / (TeRRCoef[ipTe+1] - TeRRCoef[ipTe]) *
		(RRCoef[nelem][n][ipTe+1] - RRCoef[nelem][n][ipTe]) + RRCoef[nelem][n][ipTe] ;

	/* that was the log, now make linear */
	rate = pow( 10. , rate );

#	ifdef DEBUG_FUN
	fputs( " <->He_RRCoef_Te()\n", debug_fp );
#	endif

	return rate;
}

/*HelikeCheckRecomb - called by SanityCheck to confirm that recombination coef are ok,
 * return value is relative error between new calculation of recom, and interp value */
double HelikeCheckRecomb(
	/* the chemical element, 1 for He */
	long nelem ,
	/* the level, 0 for ground */
	long level ,
	/* the temperature to be used */
	double temperature )
{
	double RecombRelError ,
		RecombInterp,
		RecombCalc,
		SaveTemp;

#	ifdef DEBUG_FUN
	fputs( "<+>HelikeCheckRecomb()\n", debug_fp );
#	endif

	/* this global variable is needed by radrecomb */
	EthRyd = iso.xIsoLevNIonRyd[ipHELIUM][nelem][level];

	/* first set temp todesired value */
	SaveTemp = phycon.te;
	phycon.te = (float)temperature;
	tfidle(TRUE);

	/* actually calculate the recomb coef from the Milne relation,
	 * normally only done due to compile he-like command */
	RecombCalc = radrecomb( temperature , nelem , level );

	/* interpolate the recomb coef, this is the usual method */
	RecombInterp = He_RRCoef_Te( nelem , level );

	/* reset temp */
	phycon.te = (float)SaveTemp;
	tfidle(TRUE);

	RecombRelError = ( RecombInterp - RecombCalc ) / MAX2( RecombInterp , RecombCalc );

#	ifdef DEBUG_FUN
	fputs( " <->HelikeCheckRecomb()\n", debug_fp );
#	endif

	return RecombRelError ;
}

/*CS_l_mixing - find rate for l-mixing collisions by protons, for neutrals */
/* >>refer	He	l-mixing	Pengelly, R.M., & Seaton, M.J., 1964, MNRAS, 127, 165 */
static double CS_l_mixing(
	/* the chemical element, 1 for He */
	long nelem ,
	/* the level, 0 for ground */
	long ipLo,
	long ipHi)
{
	double cs,
		Debye,
		Dul,
		factor,
		rate ,
		radint,
		reduced_mass_2_emass;
	/* only do proton collisions for now */
	const double ChargIncoming = 1.;

#	ifdef DEBUG_FUN
	fputs( "<+>CS_l_mixing()\n", debug_fp );
#	endif

	/* equation 46 of PS64 */
	/* min on density added to prevent this from becoming large and negative
	 * at very high n_e - Pengelly & Seaton did not apply this above
	 * 1e12 cm^-3 */
	Debye = 1.68 + log10( phycon.te / MIN2(1e12 , phycon.eden ) );


	/* Calculate the hydrogenic radial integral */
	if (he_levels[ipHi].l > he_levels[ipLo].l)
		radint = hri(he_levels[ipHi].n,he_levels[ipHi].l, he_levels[ipLo].n, he_levels[ipLo].l, nelem);
	else if (he_levels[ipHi].l < he_levels[ipLo].l)
		radint = hri(he_levels[ipHi].n,he_levels[ipLo].l, he_levels[ipLo].n, he_levels[ipHi].l, nelem);
	else 
		radint = 0.;

	Dul = 8./3. * radint * radint * ChargIncoming * ChargIncoming;

#if	0
	/* equation 44 of PS64 */
	/* NB - this only works for helium-like ions, the nelem in pow2 is the residual nuclear charge */
	Dul = POW2( ChargIncoming / (nelem) ) * 6. * POW2( (double)he_levels[ipLo].n ) *
		( POW2((double)he_levels[ipLo].n) - POW2((double)he_levels[ipLo].l) - he_levels[ipLo].l - 1);
#endif

	/* reduced mass in gm, only do proton collisions for now
	 * this mass always appears relative to the electron mass, so define it that way */
	reduced_mass_2_emass = AtomcWgt.AtomicWeight[nelem]*AtomcWgt.AtomicWeight[ipHYDROGEN]/
		(AtomcWgt.AtomicWeight[nelem]+AtomcWgt.AtomicWeight[ipHYDROGEN])*ATOMIC_MASS_UNIT / ELECTRON_MASS;

	/* this has large negative terms, make sure result is positive */
	factor = (11.54 + log10( phycon.te  / Dul / reduced_mass_2_emass ) + Debye);
	assert( factor > 0.);

	rate = 9.93e-6 * sqrt( reduced_mass_2_emass  ) * Dul / phycon.sqrte * factor;

	/* convert rate to cs */
	cs = rate / 8.629e-6 * phycon.sqrte * iso.stat[ipHELIUM][ipLo] ;

	assert( cs > 0. );

#	ifdef DEBUG_FUN
	fputs( " <->CS_l_mixing()\n", debug_fp );
#	endif

	return cs ;
}

/*prt_He_like_DeparCoef routine to print departure coefficients for he-like species */
void prt_He_like_DeparCoef(long int nelem )
{
	long is,i,in,il,ipLo;

	
	for( is = 0; is<=1; ++is)
	{
		fprintf(ioQQQ,"\nipZ=%li\tspin=%li",nelem,is);
		fprintf(ioQQQ,"\nn,l=>\t");

		for ( i =0; i < he_levels[iso.nLevels[ipHELIUM][nelem]-1].n ; ++i)
		{
			fprintf(ioQQQ,"%ld\t\t",i);
		}
	
		for( in = is+1; in <= he_levels[iso.nLevels[ipHELIUM][nelem]-1].n; ++in)
		{
			fprintf(ioQQQ,"\n%ld\t",in);
				
			for( il = 0; il < in; ++il)
			{
				if (in == 1)
					ipLo = 0;
				else if( (il==1) && (is ==0) )
					ipLo = in * (in + 1);
				else if ( il < 2 )
				{
					ipLo = (in-1) * in + 2 + 2*il - is;
					if( ( in == 2 ) && (il == 0) )
						ipLo -= 2;
				}
				else 
					ipLo = (in-1) * in + 1 + 2*il - is; 
				
				assert( he_levels[ipLo].n == in );
				assert( he_levels[ipLo].s == is );
				assert( he_levels[ipLo].l == il );

				fprintf(ioQQQ,"%.4e\t",iso.DepartCoef[ipHELIUM][nelem][ipLo]);
			}
			fprintf(ioQQQ,"\n");
		}
		fprintf(ioQQQ,"\n");
	}		
	
	return;
}

/*prt_He_like_Pops routine to print level populations for he-like species */
void prt_He_like_Pops(long int nelem )
{
	fprintf(ioQQQ," write me!\nfor element number %li\n",nelem);
	return;
}

/* these are not used */
/*#undef  AA   */
/*#undef  Y */
/*#undef  SS */
/*#undef  ipivector */

