/*HeatPunch punch contributors to local heating, with punch heat command */
#include "cddefines.h"
#include "cooling.h"
#include "heat.h"
#include "radius.h"
#include "converge.h"
#include "taulines.h"
#include "elementnames.h"
#include "wkhtcl.h"
#include "fndlineht.h"
/* limit for number of heat agents that are saved */
#	ifdef IPNSAVE
#undef IPNSAVE
#	endif
#define	IPNSAVE	30
/* limit to number to print */
#	ifdef IPRINT
#undef IPRINT
#	endif
#define IPRINT 9

void HeatPunch(FILE* io)
{
	char chLabel[IPNSAVE][10], 
	  chLbl[11];
	int lgHeatLine,
		lgFail;
	long int i, 
	  ipStrong, 
	  ipnt, 
	  ipOrdered[IPNSAVE],
	  ipsave[IPNSAVE], 
	  j, 
	  jpsave[IPNSAVE], 
	  k, 
	  level;
	double CS, 
	  ColHeat, 
	  EscP, 
	  Pump, 
	  Strong, 
	  TauIn;
	float save[IPNSAVE];

#	ifdef DEBUG_FUN
	fputs( "<+>HeatPunch()\n", debug_fp );
#	endif

	for( i=0; i<IPNSAVE ; ++i )
	{
		ipsave[i] = INT_MIN;
		jpsave[i] = INT_MIN;
		save[i] = -FLT_MAX;
	}

	ipnt = 0;
	/* heat sources are saved in a 2d square array */
	for( i=0; i < LIMELM; i++ )
	{
		for( j=0; j < LIMELM; j++ )
		{
			/* WeakHeatCool set with 'set weakheatcool' command
			 * default is 0.05 */
			if( heat.heating[i][j]/heat.htot > WkHtCl.WeakHeatCool )
			{
				ipsave[ipnt] = i;
				jpsave[ipnt] = j;
				save[ipnt] = (float)(heat.heating[i][j]/heat.htot);
				ipnt = MIN2(IPNSAVE,ipnt+1);
			}
		}
	}

	/* now check for possible line heating not counted in 1,23
	 * there should not be any significant heating source heree
	 * since they would not be counted in deriv correctly */
	for( i=0; i < cooling.ncltot; i++ )
	{
		if( cooling.heatnt[i]/heat.htot > WkHtCl.WeakHeatCool )
		{
			float awl;
			awl = cooling.collam[i];
			/* force to save wavelength convention as printout */
			if( awl > 100000 )
				awl /= 10000;
			fprintf( io, " Negative coolant was %s %.2f %.2e\n", 
			  cooling.chClntLab[i], awl, cooling.heatnt[i]/heat.htot );
		}
	}

	if( !conv.lgConvTemp )
	{
		fprintf( io, " >>>>  Temperature notconverged.\n" );
	}
	else if( !conv.lgConvEden )
	{
		fprintf( io, " >>>>  Electron density not convergenced.\n" );
	}
	else if( !conv.lgConvIoniz )
	{
		fprintf( io, " >>>>  Ionization not converged.\n" );
	}
	else if( !conv.lgConvPres )
	{
		fprintf( io, " >>>>  Pressure not converged.\n" );
	}


	/* following loop tries to identify strongest agents and turn to labels */
	for( k=0; k < ipnt; k++ )
	{
		/* generate labels that make sense in printout 
		 * if not identified with a specific agent, will print indices as [i][j],
		 * heating is heat.heating[i][j] */
		i = ipsave[k];
		j = jpsave[k];
		if( i >= j )
		{
			/* this is case of photoionization of atom or ion */
			strcpy( chLabel[k], elementnames.chElementSym[i] );
			strcat( chLabel[k], elementnames.chIonStage[j]  );
		}
		/* notice that in test i and j are swapped from order in heating array */
		else if( i == 0 && j == 1 )
		{
			/* photoionization from all excited states of Hydrogenic species,
			 * heating[1][0] */
			strcpy( chLabel[k], "Hn=2" );
		}
		else if( i == 0 && j == 3 )
		{
			/* collisional ionization of H from ground */
			strcpy( chLabel[k], "Hion" );
		}
		else if( i == 0 && j == 8 )
		{
			/* H2* deexcitation */
			strcpy( chLabel[k], "H2vd" );
		}
		else if( i == 0 && j == 9 )
		{
			/* CO dissociation */
			strcpy( chLabel[k], "COds" );
		}
		else if( i == 0 && j == 20 )
		{
			/* extra heat */
			strcpy( chLabel[k], "extH" );
		}
		else if( i == 0 && j == 11 )
		{
			/* free free heating */
			strcpy( chLabel[k], "H FF" );
		}
		else if( i == 0 && j == 12 )
		{
			/* heating line, that was supposed to cool */
			strcpy( chLabel[k], "Clin" );
		}
		else if( i == 0 && j == 13 )
		{
			/* grain photoionization */
			strcpy( chLabel[k], "dust" );
		}
		else if( i == 0 && j == 15 )
		{
			/* H- heating */
			strcpy( chLabel[k], "H-  " );
		}
		else if( i == 0 && j == 16 )
		{
			/* H2+ heating */
			strcpy( chLabel[k], "H2+ " );
		}
		else if( i == 0 && j == 17 )
		{
			/* H2 dissociation */
			strcpy( chLabel[k], "H2ds" );
		}
		else if( i == 0 && j == 18 )
		{
			/* Compton recoil of bound electrons */
			strcpy( chLabel[k], "CBnd" );
		}
		else if( i == 0 && j == 19 )
		{
			/* Compton heating */
			strcpy( chLabel[k], "Comp" );
		}
		else if( i == 0 && j == 22 )
		{
			/* line heating */
			strcpy( chLabel[k], "line" );
		}
		else if( i == 0 && j == 23 )
		{
			/* hydrogen line heating */
			strcpy( chLabel[k], "Hlin" );
		}
		else if( i == 0 && j == 24 )
		{
			/* charge transfer heating */
			strcpy( chLabel[k], "ChaT" );
		}
		else if( i == 1 && j == 3 )
		{
			/* helium triplet line heating */
			strcpy( chLabel[k], "He3l" );
		}
		else if( i == 25 && j == 27 )
		{
			/* helium triplet line heating */
			strcpy( chLabel[k], "Fe 2" );
		}
		else
		{
			sprintf( chLabel[k], "[%ld][%ld]" , i , j );
		}
	}

	/* now sort by decreasing importance */
	/*spsort netlib routine to sort array returning sorted indices */
	spsort(
		  /* input array to be sorted */
		  save, 
		  /* number of values in x */
		  ipnt, 
		  /* permutation output array */
		  ipOrdered, 
		  /* flag saying what to do - 1 sorts into increasing order, not changing
		   * the original routine */
		  -1, 
		  /* error condition, should be 0 */
		  &lgFail);

	/* begin the print out with zone number, total heating and cooling */
	fprintf( io, "%.5e\t%.2e\t%.2e", radius.depth, heat.htot, 
	  cooling.ctot );

	/* we only want to print the IPRINT strongest of the group */
	ipnt = MIN2( ipnt , IPRINT );

	for( k=0; k < ipnt; k++ )
	{
		i = ipOrdered[k];
		fprintf( io, "\t%s\t%6.3f ", chLabel[i], save[i] );
	}
	fprintf( io, " \n" );

	/* a negative pointer in the heating array is probably a problem,
	 * indicating that some line has become a heat source */
	lgHeatLine = FALSE;

	/* check if any lines were major heat sources */
	for( i=0; i < ipnt; i++ )
	{
		/* heating[22][0] is line heating - identify line if important */
		if( ipsave[i] == 0 && jpsave[i] == 22 )
			lgHeatLine = TRUE;
	}

	if( lgHeatLine )
	{
		/* a line was a major heat source - identify it */
		FndLineHt(&level,&ipStrong,&Strong);
		if( Strong/heat.htot > 0.005 )
		{
			if( level == 1 )
			{
				strcpy( chLbl, chLineLbl(&TauLines[ipStrong] ) );
				TauIn = TauLines[ipStrong].TauIn;
				Pump = TauLines[ipStrong].pump;
				EscP = TauLines[ipStrong].Pesc;
				CS = TauLines[ipStrong].cs;
				/* ratio of line to total heating */
				ColHeat = TauLines[ipStrong].heat/
				  heat.htot;
			}
			else if( level == 2 )
			{
				strcpy( chLbl, chLineLbl(&TauLine2[ipStrong]) );
				TauIn = TauLine2[ipStrong].TauIn;
				Pump = TauLine2[ipStrong].pump;
				EscP = TauLine2[ipStrong].Pesc;
				CS = TauLine2[ipStrong].cs;
				ColHeat = TauLine2[ipStrong].heat/
				  heat.htot;
			}
			else if( level == 3 )
				/* C12O16 */
			{
				strcpy( chLbl, chLineLbl(&C12O16Rotate[ipStrong]) );
				TauIn = C12O16Rotate[ipStrong].TauIn;
				Pump = C12O16Rotate[ipStrong].pump;
				EscP = C12O16Rotate[ipStrong].Pesc;
				CS = C12O16Rotate[ipStrong].cs;
				ColHeat = C12O16Rotate[ipStrong].heat/
				  heat.htot;
			}
			else if( level == 4 )
				/* C13O16 */
			{
				strcpy( chLbl, chLineLbl(&C13O16Rotate[ipStrong]) );
				TauIn = C13O16Rotate[ipStrong].TauIn;
				Pump = C13O16Rotate[ipStrong].pump;
				EscP = C13O16Rotate[ipStrong].Pesc;
				CS = C13O16Rotate[ipStrong].cs;
				ColHeat = C13O16Rotate[ipStrong].heat/
				  heat.htot;
			}
			else
			{
				fprintf( ioQQQ, " HeatPunch insane level%4ld\n", 
				  level );
				puts( "[Stop in punheat]" );
				cdEXIT(1);
			}
			fprintf( io, "  LHeat lv%2ld %10.10s TIn%10.2e Pmp%9.1e EscP%9.1e CS%9.1e Hlin/tot%10.2e\n", 
			  level, chLbl, TauIn, Pump, EscP, CS, ColHeat );
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeatPunch()\n", debug_fp );
#	endif
	return;
}

