/*coolr main routine to call others, to evaluate total cooling */
#include "cddefines.h"
#include "physconst.h"
#include "hydrogenic.h"
#include "nhe1lvl.h"
#include "taulines.h"
#include "wind.h"
#include "coolheavy.h"
#include "thermal.h"
#include "radius.h"
#include "freeon.h"
#include "heat.h"
#include "trace.h"
#include "dynamics.h"
#include "rfield.h"
#include "grainvar.h"
#include "level2.h"
#include "h21cm.h"
#include "called.h"
#include "numderiv.h"
#include "nomole.h"
#include "cionhe.h"
#include "he3lines.h"
#include "hmi.h"
#include "he.h"
#include "cyclot.h"
#include "phycon.h"
#include "he1ex.h"
#include "dndt.h"
#include "he1deltat.h"
#include "phe1lv.h"
#include "he1stat.h"
#include "he1blt.h"
#include "ionfracs.h"
#include "bit32.h"
#include "he1crt.h"
#include "tfidle.h"
#include "cooling.h"
#include "putcs.h"
#include "hyperfine.h"
#include "coolr.h"
/*fndneg search cooling array to find negative values */
static void fndneg(void);
/*fndstr search cooling stack to find strongest values */
static void fndstr(double tot, 
  double dc);

void coolr(double *tot)
{
	long int ion,
	  i,
	  iu, 
	  l,
	  nelem;

	static long int nhit = 0, 
	  nzSave=0;

	static double TeEvalCS = 0.;

	double brems, 
	  BremsFactor ,
	  cs ,
	  deriv, 
	  dhe1l, 
	  factor, 
	  he1ion, 
	  he1line, 
	  he1lmax, 
	  he1lone, 
	  qn, 
	  recem, 
	  rothi, 
	  rotlow, 
	  tripcl, 
	  x;

	static double oltcool=0., 
	  oldtemp=0.;

#	ifdef DEBUG_FUN
	fputs( "<+>coolr()\n", debug_fp );
#	endif

	/* returns tot, the total cooling,
	 * and dc, the derivative of the cooling */

	/* routine level2( t10 )
	 * routine level3( abund , t10,t21,t20)
	 * tsq1 = 1. / (te**2)
	 * POPEXC( O12,g1,g2,A21,excit,abund) ; result already*a21
	 * POP3(G1,G2,G3,O12,O13,O23,A21,A31,A32,E12,E23,P2,ABUND,GAM2)
	 * AtomSeqBeryllium(cs23,cs24,cs34,tarray,a41)
	 * FIVEL( G(1-5) , ex(wn,1-5), cs12,cs13,14,15,23,24,25,34,35,45,
	 *  A21,31,41,51,32,42,52,43,53,54, pop(1-5), abund) */

	if( trace.lgTrace )
	{
		fprintf( ioQQQ, "   COOLR called, TE=%12.4e\n", phycon.te );
	}

	/* possibly update temperature variables
	 * edensqte = eden/sqrte
	 * cdsqte = 8.629e-6*eden/sqrte */
	tfidle(FALSE);

	/* now zero out the cooling stack */
	colzro();

	/* grain recombination cooling, evaluated elsewhere
	 * can either heat or cool the gas, do cooling here */
	coladd("dust",0,MAX2(0.,gv.GasCoolColl));

	/* grain cooling proportional to temperature ^3/2 */
	cooling.dCooldT += MAX2(0.,gv.GasCoolColl)*3./(2.*phycon.te);

	/* molecular cooling */
	if( nomole.lgNoH2Mole )
	{
		CoolHeavy.hmfb = 0.;
		CoolHeavy.h2line = 0.;
		CoolHeavy.h2dis = 0.;
		/*  H + H+ => H2+ cooling */
		CoolHeavy.H2PlsCool = 0.;
	}

	else
	{

		/*  H + H+ => H2+ cooling */
		CoolHeavy.H2PlsCool = (float)(MAX2((2.325*phycon.te-1875.)*1e-20,0.)*
		  xIonFracs[ipHYDROGEN][1]*xIonFracs[ipHYDROGEN][2]*1.66e-11);

		/*  H- FB */
		CoolHeavy.hmfb = hmi.hmicol;
		/*  H- FF is in with H ff
		 *  rate for rotation lines from 
		 *  >>refer	mh2	cool	Lepp, S., & Shull, J.M. 1983, ApJ, 270, 578 */
		x = phycon.alogte - 4.;
		if( phycon.te > 1087. )
		{
			rothi = 3.90e-19*sexp(6118./phycon.te);
		}
		else
		{
			rothi = pow(10.,-19.24 + 0.474*x - 1.247*x*x);
		}

		/*  low density rotation cooling */
		qn = pow(MAX2(hmi.htwo,1e-37),0.77) + 1.2*pow(MAX2(xIonFracs[ipHYDROGEN][1],1e-37),0.77);
		if( phycon.te > 4031. )
		{
			rotlow = 1.38e-22*sexp(9243./phycon.te)*qn;
		}
		else
		{
			rotlow = pow(10.,-22.90 - 0.553*x - 1.148*x*x)*qn;
		}

		if( rotlow > 0. )
		{
			CoolHeavy.h2line = hmi.htwo*rothi/(1. + rothi/rotlow);
		}
		else
		{
			CoolHeavy.h2line = 0.;
		}

		{
			/*@-redef@*/
			enum {DEBUG=FALSE};
			/*@+redef@*/
			if( DEBUG && nzone>187&& iteration > 1/**/)
			{
				fprintf(ioQQQ,"h2coolbug\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\t%.2e\n",
					phycon.te, 
					CoolHeavy.h2line, 
					hmi.htwo, 
					hmi.hminus , 
					hmi.HMinus_photo_rate,
					hmi.ph2lte,
					rothi,
					rotlow );
			}
		}

		CoolHeavy.h2dis = 
			(float)(hmi.rh2dis*xIonFracs[ipHYDROGEN][1]*hmi.htwo*7.19e-12* 0.);
	}

	coladd("H-fb",0,CoolHeavy.hmfb);
	/* >>chng 96 nov 15, fac of 2 in deriv to help convergence in very dense
	 * models where H- is important, this takes change in eden into
	 * partial account */
	cooling.dCooldT += 2.*CoolHeavy.hmfb*phycon.teinv;

	coladd("H2ln",0,CoolHeavy.h2line);
	/* >>chng 00 oct 21, added coef of 3.5, sign had been wrong */
	/*cooling.dCooldT += CoolHeavy.h2line*phycon.teinv;*/
	cooling.dCooldT += 3.5*CoolHeavy.h2line*phycon.teinv;/**/

	coladd("H2ds",0,CoolHeavy.h2dis);
	cooling.dCooldT += CoolHeavy.h2dis*phycon.teinv;

	coladd("H2+ ",0,CoolHeavy.H2PlsCool);
	cooling.dCooldT += CoolHeavy.H2PlsCool*phycon.teinv;

	/* carbon monoxide CO vib-rot cooling */
	/* >>chng 00 oct 16, from simple expression to full model molecule */
	/* coladd("CO C",0,CoolHeavy.COVibCool);
	cooling.dCooldT += CoolHeavy.dCOVdT; */

	/* cooling due to C12O16 */
	coladd("CO C",12,CoolHeavy.C12O16Rot);
	cooling.dCooldT += CoolHeavy.dC12O16Rot; 
	/* >>chng 00 oct 25, add C13O16 cooling */
	/* cooling due to C13O16 */
	coladd("CO C",13,CoolHeavy.C13O16Rot);
	cooling.dCooldT += CoolHeavy.dC13O16Rot; 

	/* heating due to three-body, will be incremented in HydroCool*/
	heat.heating[0][3] = 0.;
	/* heating due to hydrogen lines */
	heat.heating[0][23] = 0.;
	/* heating due to photoionization of all excited states of hydrogen species */
	heat.heating[0][1] = 0.;

	/* hydrogenic species cooling, mainly lines, and level ionization */
	for( nelem=0; nelem < LIMELM; nelem++ )
	{
		/* must always call HydroCool since we must zero those variables
		 * that would have been set had the species been present */
		HydroCool( ipHYDROGEN , nelem );
	}

	/* brems cooling */
	/* Hydrogen BremsFactor free free cooling, free-free heating */
	/* NB major change in code with extension of continuum down to 1e-8 ryd
	 * firstdr does not check opacity below 1e-5 - if free--free heating runs
	 * away could be strong fir continuum hitting free free opacity */
	factor = 0.9 + 1.2/(2.2 + fabs(phycon.alogte-5.398));

	/* >>chng 96 jan 8, for stability always use corrected free free */
	/* rfield.EnergyBremsThin is freq (Ryd) where gas becomes optically thin to absorption */
	BremsFactor = 1.42e-27*phycon.eden*phycon.sqrte*factor*sexp(rfield.EnergyBremsThin*
	  TE1RYD/phycon.te);

	/* charge squared is hydrogenic brems factor */
	hydro.FreeFreeCool = 0.;
	/* following set FALSE with no free-free command */
	if( freeon.lgFreeOn )
	{
		for( nelem=0; nelem<LIMELM; ++nelem)
		{
			hydro.FreeFreeCool += POW2(nelem+1.)*BremsFactor*xIonFracs[nelem][nelem+2];
		}
	}

	/* these two terms are both large, nearly canceling, near lte */
	hydro.HFFNet = hydro.FreeFreeCool - hydro.FreeFreeHeat;

	/* if positive count as free-free cooling */
	coladd( "HFFc" , 0, MAX2(0.,hydro.HFFNet) );

	/* now stuff into heating array if negative */
	heat.heating[0][11] = MAX2(0.,-hydro.HFFNet);

	/* >>chng 96 oct 30, from HFFNet to just FreeFreeCool,
	 * since HeatSum picks up FreeFreeHeat */
	cooling.dCooldT += hydro.FreeFreeCool*cooling.halfte;

	/* net cooling for helium, HeI
	 * HE1X evaluated in HE1LEV */
	he1ion = he1exCOM.he1ex*xIonFracs[ipHELIUM][2];
	coladd("He1i",0,he1ion);
	cooling.dCooldT += he1ion*3.9e4*cooling.tsq1;

	/* all cooling due to hei lines */
	if( xIonFracs[ipHELIUM][2] > 0. )
	{
		he1line = 0.;
		he1lmax = 0.;
		dhe1l = 0.;
		/* first do 1s-2s and 1s-2p collisions */
		he1line = phycon.eden*xIonFracs[ipHELIUM][3]*BOLTZMANN*He1deltaT.He1deltaTe[0][1]*
		  (phe1lv.he1n[0]*he1blt.he1bol[1][0]*(he1crt.he1c2s1s + 3.*
		  phe1lv.he1cll[0][1]) - phe1lv.he12s*he1crt.he1c2s1s - phe1lv.he12p*
		  phe1lv.he1cll[0][1]);
		he1lmax = he1line;
		he1lone = he1line;
		if( he1lone > 0. )
		{
			/* line is coolant, note derivative counted from ground */
			dhe1l = he1lone*(He1deltaT.He1deltaTe[0][1]*cooling.tsq1 - 
			  cooling.halfte);
		}
		else
		{
			/* line is a heating agent */
			dhe1l = -he1lone*cooling.halfte;
		}

		/* upper limits changed to 5 and 6 due to strong cooling from
		 * highest psuedo state */
		for( l=0; l < 5; l++ )
		{
			for( iu=MAX2(2,l+1); iu < 6; iu++ )
			{
				he1lone = phe1lv.he1cll[l][iu]*(phe1lv.he1n[l]*
				  he1blt.he1bol[iu][l]*he1statCOM.he1stat[iu]/he1statCOM.he1stat[l] - 
				  phe1lv.he1n[iu])*phycon.eden*xIonFracs[ipHELIUM][2]*(BOLTZMANN*
				  He1deltaT.He1deltaTe[l][iu]);

				/*  3      eden*heii* (1.384e-16 * HdeltaT(iu,l)*4. )
				 * >>chng 97 mar 4, above heii was heiii */
				if( fabs(he1lone) > fabs(he1lmax) )
				{
					he1lmax = he1lone;
				}

				he1line += he1lone;
				/* >>chng 97 jul 1, add in cooling derivative */
				if( he1lone > 0. )
				{
					/*   line is coolant, note derivative counted from ground */
					dhe1l += he1lone*(He1deltaT.He1deltaTe[0][iu]*
					  cooling.tsq1 - cooling.halfte);
				}
				else
				{
					/*   line is a heating agent */
					dhe1l += -he1lone*cooling.halfte;
				}
			}
		}
	}

	else
	{
		dhe1l = 0.;
		he1line = 0.;
	}

	/* all HeI line collisional excitation */
	coladd("He1l",0,he1line);
	cooling.dCooldT += dhe1l;

	/* >>chng 01 mar 16, add advective cooling */
	coladd("adve",0,MAX2(0.,dynamics.CoolHeat) );
	cooling.dCooldT += 8.*dynamics.dCoolHeatdT;
	heat.heating[1][5] = MAX2(0.,-dynamics.CoolHeat);

	/* total Compton cooling */
	CoolHeavy.tccool = rfield.cmcool*phycon.te;
	coladd("Comp",0,CoolHeavy.tccool);
	cooling.dCooldT += rfield.cmcool;

	/* option for "extra" cooling, expressed as power-law */
	if( cooling.lgCExtraOn )
	{
		CoolHeavy.cextxx = 
			(float)(cooling.CoolExtra*pow(phycon.te/1e4f,cooling.cextpw));
	}
	else
	{
		CoolHeavy.cextxx = 0.;
	}
	coladd("Extr",0,CoolHeavy.cextxx);

	/* cooling due to wind expansion, only for winds expansion cooling */
	if( wind.windv > 0. )
	{
		dndt.dDensityDT = (float)(wind.AccelTot/wind.windv + 2.*wind.windv/
		  radius.Radius);
		CoolHeavy.expans = phycon.pden*phycon.te*BOLTZMANN*dndt.dDensityDT;
	}
	else
	{
		dndt.dDensityDT = 0.;
		CoolHeavy.expans = 0.;
	}
	coladd("Expn",0,CoolHeavy.expans);
	cooling.dCooldT += CoolHeavy.expans/phycon.te;

	/* cyclotron cooling */
	CoolHeavy.cyntrn = cyclot.CycloCoolCoef*phycon.eden*phycon.te;
	coladd("Cycl",0,CoolHeavy.cyntrn);
	cooling.dCooldT += cyclot.CycloCoolCoef*phycon.eden;

	/* electron-electron brems, approx form from 
	 * >>refer	ee	brems	Stepney and Guilbert, MNRAS 204, 1269 (1983)
	 * ok for T<10**9 */
	CoolHeavy.eebrm = POW2(phycon.eden*phycon.te*1.84e-21);

	/* >>chng 97 mar 12, added deriv */
	cooling.dCooldT += CoolHeavy.eebrm*cooling.halfte;
	coladd("eeff",0,CoolHeavy.eebrm);

	/* collisional ionization of triplet helium  */
	coladd("HeI3",0,MAX2(0.,cionhe.He3IonCool));
	cooling.dCooldT += MAX2(0.,cionhe.He3IonCool)*5.53e4*cooling.tsq1;

	/* net three body heating
	 * >>chng 96 jul 15, had always been cooling */
	heat.heating[1][4] = MAX2(0.,-cionhe.He3IonCool);

	/* collisional excitation of all triplet lines, done in he3lev
	 * can heat or cool */
	if( he.hei3 > 0. )
	{
		tripcl = he3lines.he3clx;
	}
	else
	{
		tripcl = 0.;
	}
	coladd("Hetr",0,MAX2(0.,tripcl));
	heat.heating[1][3] = MAX2(0.,-tripcl);

	/* >>chng 96 nov 19, to following expression for HeI Lya cooling
	 *666 error! break out 2s and 2p, really should write entire he cooling rout */
	CoolHeavy.c584 = phycon.eden*(double)xIonFracs[ipHELIUM][2]*3.41e-11*phe1lv.he1cll[0][1]*
	  (he1blt.he1bol[1][0]*phe1lv.he1n[0]*4. - phe1lv.he1n[1]);
	cooling.dCooldT += CoolHeavy.c584*2.46e5*cooling.tsq1;
	coladd("He I",584,CoolHeavy.c584);

	/* He I brems (free-free) cooling */
	factor = 0.9 + 1.2/(2.2 + fabs(phycon.alogte-5.398));

	/* >>chng 95 dec 23 always do ots for brems for stability
	 * ots continuum transport - do not count optically thick part */
	brems = 1.42e-27*phycon.eden*phycon.sqrte*factor*sexp(rfield.EnergyBremsThin*
	  TE1RYD/phycon.te);

	/* do not include ion since was done in hydrogenic above */
	/* following set FALSE with no free-free command */
	if( freeon.lgFreeOn )
	{
		CoolHeavy.bhe = brems*xIonFracs[ipHELIUM][2];
	}
	else
	{
		CoolHeavy.bhe = 0.;
	}
	coladd("Heff",0,CoolHeavy.bhe);

	/* recombination cooling for He+ to Heo */
	recem = 9.52e-26*phycon.te10*phycon.eden;
	CoolHeavy.herec = recem*xIonFracs[ipHELIUM][2];
	cooling.dCooldT += CoolHeavy.bhe*cooling.halfte + CoolHeavy.herec*
	  0.27*phycon.teinv;
	coladd("Hefb",0,CoolHeavy.herec);

	/* heavy element recombination cooling, do not count hydrogenic since
	 * already done above, also helium singlets have been done
	 * EdenFFSum set in ESUM, given by Harrington et al. 82
	 * 666 error! put in charge dependent rec term */
	CoolHeavy.heavfb = 0.;
	for( nelem=2; nelem < LIMELM; nelem++ )
	{
		for( ion=2; ion <= nelem ; ion++ )
		{
			CoolHeavy.heavfb += xIonFracs[nelem][ion];
		}
	}
	CoolHeavy.heavfb *= recem;
	coladd("hvFB",0,CoolHeavy.heavfb);

	/* heavy element brems (free-free) cooling */
	if( freeon.lgFreeOn )
	{
		CoolHeavy.heavff = 
			phycon.EdenFFSum*1.44e-27*phycon.sqrte*phycon.eden*
		  factor;
	}
	else
	{
		CoolHeavy.heavff = 0.;
	}

	cooling.dCooldT += CoolHeavy.heavfb*.113*phycon.teinv;
	coladd("Hvff",0,CoolHeavy.heavff);

	/* heavy element collisional ionization
	 * derivative should be zero since increased coll ion rate
	 * decreases neutral fraction by proportional amount */
	coladd("Hvin",0,CoolHeavy.colmet);

	/* all the fine structure lines added by Will Goddard */
	if( fabs(phycon.te-TeEvalCS)/phycon.te > 0.05 )
	{
		for( i=1; i < nHFLines; i++ )
		{
			cs = HyperfineCS( i );
			/* now generate the collision strength and put into the line array */
			PutCS(  cs , &HFLines[i] );
		}
		TeEvalCS = phycon.te;
	}

	/*TODO code review - check this logic */
	/* H 21 cm emission/population,
	 * cs will be sum of e cs and H cs converted from rate */
	cs = (H21cm_electron( phycon.te )*phycon.eden + H21cm_H_atom( phycon.te )* xIonFracs[ipHYDROGEN][1]) * 3./	phycon.cdsqte;

	/* add on H collisions of H 21 cm */
	for( i=0; i < nHFLines; i++ )
	{
		/* remember current gas-phase abundances */
		float save = xIonFracs[HFLines[i].nelem-1][HFLines[i].IonStg];

		/* bail if no abundance */
		if( save<=0. ) continue;

		/* set gas-phase abundance to total times isotope ratio */
		xIonFracs[HFLines[i].nelem-1][HFLines[i].IonStg] *= hyperfine.HFLabundance[i] ;

		/* use the collision strength generated above and find pops and cooling */
		level2( &HFLines[i] );

		/* put the correct gas-phase abundance back in the array */
		xIonFracs[HFLines[i].nelem-1][HFLines[i].IonStg] = save;
	}
	{
		/* this is an option to print out one of the two photon continua */
		/*@-redef@*/
		enum {DEBUG=FALSE};
		/*@+redef@*/
		if( DEBUG )
		{	
			fprintf(ioQQQ,"isotope pops\t%.3e\t%.3e\t%.3e\t%.3e\n",
				HFLines[0].PopHi , HFLines[0].PopLo,
				TauLines[ipH21cm].PopHi , TauLines[ipH21cm].PopLo);
		}
	}

	/* Carbon cooling */
	CoolCarb();

	/* Nitrogen cooling */
	CoolNitr();

	/* Oxygen cooling */
	CoolOxyg();

	/* Fluorine cooling */
	CoolFluo();

	/* Neon cooling */
	CoolNeon();

	/* Magnesium cooling */
	CoolMagn();

	/* Sodium cooling */
	CoolSodi();

	/* Aluminum cooling */
	CoolAlum();

	/* Silicon cooling */
	CoolSili();

	/* Phosphorus */
	CoolPhos();

	/* Sulphur cooling */
	CoolSulf();

	/* Chlorine cooling */
	CoolChlo();

	/* Argon cooling */
	CoolArgo();

	/* Potasium cooling */
	CoolPota();

	/* Calcium cooling */
	CoolCalc();

	/* Scandium cooling */
	CoolScan();

	/* Titanium cooling */
	CoolTita();

	/* Vanadium cooling */
	CoolVana();

	/* Chromium cooling */
	CoolChro();

	/* Iron cooling */
	CoolIron();

	/* Manganese cooling */
	CoolMang();

	/* Cobalt cooling */
	CoolCoba();

	/* Nickel cooling */
	CoolNick();

	/* do all the thousands of lines Dima added with g-bar approximation */
	CoolDima();

	/* now add up all the coolants */
	CoolSum(tot);

	/* negative cooling */
	if( *tot <= 0. )
	{
		fprintf( ioQQQ, " COOLR; cooling is <=0, this is impossible.\n" );
		ShowMe();
		puts( "[Stop in coolr]" );
		cdEXIT(1);
	}

	/* bad derivative */
	if( cooling.dCooldT == 0. )
	{
		fprintf( ioQQQ, " COOLR; cooling slope <=0, this is impossible.\n" );
		if( (*tot > 0. && bit32.lgBit32) && phycon.hden < 1e-4 )
		{
			fprintf( ioQQQ, " Probably due to very low density.\n" );
		}
		ShowMe();
		puts( "[Stop in coolr]" );
		cdEXIT(1);
	}

	if( trace.lgTrace )
	{
		fndstr(*tot,cooling.dCooldT);
	}

	/* lgTSetOn true for constant temperature model */
	if( (((((!thermal.lgTSetOn) && *tot < 0.) && called.lgTalk) && 
	  cooling.lgColNeg) && cooling.lgCNegChk) && nzone > 0 )
	{
		fprintf( ioQQQ, 
			" Negative cooling, zone%4ld, =%10.2e coola=%10.2e CHION=%10.2e Te=%10.2e\n", 
		  nzone, 
		  *tot, 
		  hydro.coola[0], 
		  hydro.chion[0], 
		  phycon.te );
		fndneg();
	}

	/* possibility of getting emperical cooling derivative
	 * normally false, set true with 'set numerical derivatives' command */
	if( NumDeriv.lgNumDeriv )
	{
		/*lint -e777 */
		if( ((nzone > 2 && nzone == nzSave) && oldtemp != phycon.te) && nhit > 4 )
		/*lint +e777 */
		{
			/* hnit is number of trys on this zone - use to stop numerical problems
			 * do not evaluate numerical deriv until well into solution */
			deriv = (oltcool - *tot)/(oldtemp - phycon.te);
			cooling.dCooldT = deriv;
		}
		else
		{
			deriv = cooling.dCooldT;
		}
		if( nzone != nzSave )
			nhit = 0;

		nzSave = nzone;
		nhit += 1;
		oltcool = *tot;
		oldtemp = phycon.te;
	}

#	ifdef DEBUG_FUN
	fputs( " <->coolr()\n", debug_fp );
#	endif
	return;
}

/*  */
#ifdef EPS
#	undef EPS
#endif
#define	EPS	0.01

/*fndneg search cooling array to find negative values */
static void fndneg(void)
{
	long int i;
	double trig;

#	ifdef DEBUG_FUN
	fputs( "<+>fndneg()\n", debug_fp );
#	endif

	trig = fabs(heat.htot*EPS);
	for( i=0; i < cooling.ncltot; i++ )
	{
		if( cooling.cooling[i] < 0. && fabs(cooling.cooling[i]) > trig )
		{
			fprintf( ioQQQ, " negative line=%s %.2f fraction of heating=%.3f\n", 
			  cooling.chClntLab[i], cooling.collam[i], cooling.cooling[i]/
			  heat.htot );
		}

		if( cooling.heatnt[i] > trig )
		{
			fprintf( ioQQQ, " heating line=%s %.2f fraction of heating=%.3f\n", 
			  cooling.chClntLab[i], cooling.collam[i], cooling.heatnt[i]/
			  heat.htot );
		}
	}


#	ifdef DEBUG_FUN
	fputs( " <->fndneg()\n", debug_fp );
#	endif
	return;
}

#include "coolpr.h"
/*fndstr search cooling stack to find strongest values */
static void fndstr(double tot, 
  double dc)
{
	char chStrngLab[5];
	long int i;
	float wl;
	double str, 
	  strong;

#	ifdef DEBUG_FUN
	fputs( "<+>fndstr()\n", debug_fp );
#	endif

	strong = 0.;
	wl = -FLT_MAX;
	for( i=0; i < cooling.ncltot; i++ )
	{
		if( fabs(cooling.cooling[i]) > strong )
		{
			/* this is the wavelength of the coolant, 0 for a continuum*/
			wl = cooling.collam[i];
			/* make sure labels are all valid*/
			assert( strlen( cooling.chClntLab[i] ) == 4 );
			strcpy( chStrngLab, cooling.chClntLab[i] );
			strong = fabs(cooling.cooling[i]);
		}
	}

	str = strong;

	fprintf( ioQQQ, 
		"   FndStr cool: TE=%10.4e Ne=%10.4e C=%10.3e H=%10.3e dC/dT=%10.2e ABS(%s %.1f)=%.2e nz=%ld\n", 
	  phycon.te, phycon.eden, tot, heat.htot, dc, chStrngLab
	  , wl, str, nzone );

	/* option for extensive printout of lines */
	if( trace.lgCoolTr )
	{
		float ratio;
		/*fprintf( ioQQQ, 
			"     All coolants greater than%6.2f%% of the total will be printed\n", 
		  EPS*100. );*/

		/* flag all significant coolants, first zero out the array */
		coolpr((char*)cooling.chClntLab[0],1,0.,"ZERO");

		/* push all coolants onto the stack */
		for( i=0; i < cooling.ncltot; i++ )
		{
			/* usually positive, although can be neg for coolants that heats, 
			 * only do positive here */
			ratio = (float)(cooling.cooling[i]/cooling.ctot);
			if( ratio >= EPS )
			{
				/*>>chng 99 jan 27, only cal when ratio is significant */
				coolpr((char*)cooling.chClntLab[i],cooling.collam[i], ratio,"DOIT");
			}
		}

		/* complete the printout for positive coolants */
		coolpr("DONE",1,0.,"DONE");

		/* now do cooling that was counted as a heat source if significant */
		if( heat.heating[0][22]/cooling.ctot > 0.05 )
		{
			fprintf( ioQQQ, 
				"     All coolant heat greater than%6.2f%% of the total will be printed.\n", 
			  EPS*100. );

			coolpr("ZERO",1,0.,"ZERO");
			for( i=0; i < cooling.ncltot; i++ )
			{
				ratio = (float)(cooling.heatnt[i]/cooling.ctot);
				if( fabs(ratio) >=EPS )
				{
					coolpr((char*)cooling.chClntLab[i],cooling.collam[i],
					  ratio,"DOIT");
				}
			}
			coolpr("DONE",1,0.,"DONE");
		}
	}

#	ifdef DEBUG_FUN
	fputs( " <->fndstr()\n", debug_fp );
#	endif
	return;
}

