/* This file contains routines (perhaps in modified form) written by third parties.
 * Use and distribution of these works are determined by their respective copyrights. */
#include "cddefines.h"
#include "math_complex.h"
#include "thirdparty.h"
#include "physconst.h"

static void coulomb_CF1( double lambda, double eta, double x,
					   double *fcl_sign, double *result, int *count );
static void coulomb_F_recur(double lam_min, int kmax, double eta, double x,
                double F_lam_max, double Fp_lam_max,
                double *F_lam_min, double *Fp_lam_min );
static void coulomb_FGmhalf_series(const double eta, const double x,
                       double *F, double *G);
static void coulomb_FG0_series(const double eta, const double x,
                   double *F, double *G);
static void coulomb_FG_series(const double lam, const double eta, const double x,
                  double *F, double *G);
static void coulomb_G_recur(const double lam_min, const int kmax,
                const double eta, const double x,
                const double G_lam_min, const double Gp_lam_min,
                double *G_lam_max, double *Gp_lam_max );
static void coulomb_CF2(const double lambda, const double eta, const double x,
            double *result_P, double *result_Q, int *count );
static double gsl_sf_psi_1piy_e(const double y);
static double cheb_eval_e(const cheb_series *cs, const double x );

double gsl_sf_bessel_J0_e(const double x);
double gsl_sf_bessel_J1_e(const double x);
double gsl_sf_bessel_Y0_e(const double x);
double gsl_sf_bessel_Y1_e(const double x);
double gsl_sf_bessel_asymp_Mnu_e(const double nu, const double x);
double gsl_sf_bessel_asymp_thetanu_corr_e(const double nu, const double x);
double gsl_sf_bessel_cos_pi4_e(double y, double eps);
double gsl_sf_bessel_sin_pi4_e(double y, double eps);
double gsl_sf_bessel_Jnu_asympx_e(const double nu, const double x);
double gsl_sf_bessel_Ynu_asympx_e(const double nu, const double x);
double gsl_sf_bessel_J_CF1(const double nu, const double x, double *ratio);
double gsl_sf_bessel_JY_steed_CF2(const double nu, const double x, double *P);
double gsl_sf_bessel_IJ_taylor_e(const double nu, const double x, const int sign,
                             const int kmax, const double threshold );
static double bessel_Yn_small_x(const int n, const double x);
double gsl_sf_psi_int_e(const int n);
double gsl_sf_fact_e(const unsigned int n);
double gsl_sf_lnfact_e(const unsigned int n);
static void lngamma_complex_stirling(complex z, double *lg_r, double *arg);
double gsl_sf_bessel_I0_scaled_e(const double x);
double gsl_sf_bessel_I0_e(const double x);
double gsl_sf_bessel_I1_scaled_e(const double x);
double gsl_sf_bessel_I1_e(const double x);


/*
static void coulomb_jwkb(const double lam, const double eta, const double x,
             double *fjwkb, double *gjwkb,
             double *exponent);
*/

/* Chebyshev fit for f(y) = Re(Psi(1+Iy)) + M_EULER - y^2/(1+y^2) - y^2/(2(4+y^2))
 * 1 < y < 10
 *   ==>
 * y(x) = (9x + 11)/2,  -1 < x < 1
 * x(y) = (2y - 11)/9
 *
 * g(x) := f(y(x))
 */
static double r1py_data[] = {
   1.59888328244976954803168395603,
   0.67905625353213463845115658455,
  -0.068485802980122530009506482524,
  -0.005788184183095866792008831182,
   0.008511258167108615980419855648,
  -0.004042656134699693434334556409,
   0.001352328406159402601778462956,
  -0.000311646563930660566674525382,
   0.000018507563785249135437219139,
   0.000028348705427529850296492146,
  -0.000019487536014574535567541960,
   8.0709788710834469408621587335e-06,
  -2.2983564321340518037060346561e-06,
   3.0506629599604749843855962658e-07,
   1.3042238632418364610774284846e-07,
  -1.2308657181048950589464690208e-07,
   5.7710855710682427240667414345e-08,
  -1.8275559342450963966092636354e-08,
   3.1020471300626589420759518930e-09,
   6.8989327480593812470039430640e-10,
  -8.7182290258923059852334818997e-10,
   4.4069147710243611798213548777e-10,
  -1.4727311099198535963467200277e-10,
   2.7589682523262644748825844248e-11,
   4.1871826756975856411554363568e-12,
  -6.5673460487260087541400767340e-12,
   3.4487900886723214020103638000e-12,
  -1.1807251417448690607973794078e-12,
   2.3798314343969589258709315574e-13,
   2.1663630410818831824259465821e-15
};

static cheb_series r1py_cs = {
  r1py_data,
  29,
  -1,1,
  18
};

static double bm0_data[21] = {
   0.09284961637381644,
  -0.00142987707403484,
   0.00002830579271257,
  -0.00000143300611424,
   0.00000012028628046,
  -0.00000001397113013,
   0.00000000204076188,
  -0.00000000035399669,
   0.00000000007024759,
  -0.00000000001554107,
   0.00000000000376226,
  -0.00000000000098282,
   0.00000000000027408,
  -0.00000000000008091,
   0.00000000000002511,
  -0.00000000000000814,
   0.00000000000000275,
  -0.00000000000000096,
   0.00000000000000034,
  -0.00000000000000012,
   0.00000000000000004
}; 
const cheb_series _gsl_sf_bessel_amp_phase_bm0_cs = {
  bm0_data,
  20,
  -1, 1,
  10
};
      
static double bth0_data[24] = {
  -0.24639163774300119,
   0.001737098307508963,
  -0.000062183633402968,
   0.000004368050165742,
  -0.000000456093019869,
   0.000000062197400101,
  -0.000000010300442889,
   0.000000001979526776,
  -0.000000000428198396,
   0.000000000102035840,
  -0.000000000026363898,
   0.000000000007297935,
  -0.000000000002144188,
   0.000000000000663693,
  -0.000000000000215126,
   0.000000000000072659,
  -0.000000000000025465,
   0.000000000000009229,
  -0.000000000000003448,
   0.000000000000001325,
  -0.000000000000000522,
   0.000000000000000210,
  -0.000000000000000087,
   0.000000000000000036
};
const cheb_series _gsl_sf_bessel_amp_phase_bth0_cs = {
  bth0_data,
  23,
  -1, 1,
  12
};


static double bm1_data[21] = {
   0.1047362510931285, 
   0.00442443893702345,
  -0.00005661639504035,
   0.00000231349417339,
  -0.00000017377182007,
   0.00000001893209930,
  -0.00000000265416023,
   0.00000000044740209,
  -0.00000000008691795,
   0.00000000001891492,
  -0.00000000000451884,
   0.00000000000116765,
  -0.00000000000032265,
   0.00000000000009450,
  -0.00000000000002913,
   0.00000000000000939,
  -0.00000000000000315,
   0.00000000000000109,
  -0.00000000000000039,
   0.00000000000000014,
  -0.00000000000000005,
}; 
const cheb_series _gsl_sf_bessel_amp_phase_bm1_cs = {
  bm1_data,
  20,
  -1, 1,
  10
};

static double bth1_data[24] = {
   0.74060141026313850, 
  -0.004571755659637690,
   0.000119818510964326,
  -0.000006964561891648,
   0.000000655495621447,
  -0.000000084066228945,
   0.000000013376886564,
  -0.000000002499565654,
   0.000000000529495100,
  -0.000000000124135944,
   0.000000000031656485,
  -0.000000000008668640,
   0.000000000002523758,
  -0.000000000000775085,
   0.000000000000249527,
  -0.000000000000083773,
   0.000000000000029205,
  -0.000000000000010534,
   0.000000000000003919,
  -0.000000000000001500,
   0.000000000000000589,
  -0.000000000000000237,
   0.000000000000000097,
  -0.000000000000000040,
};
const cheb_series _gsl_sf_bessel_amp_phase_bth1_cs = {
  bth1_data,
  23,
  -1, 1,
  12
};


#define PSI_TABLE_NMAX 100
static double psi_table[PSI_TABLE_NMAX+1] = {
  0.0,  /* Infinity */              /* psi(0) */
 -E,                                /* psi(1) */
  0.42278433509846713939348790992,  /* ...    */
  0.92278433509846713939348790992,
  1.25611766843180047272682124325,
  1.50611766843180047272682124325,
  1.70611766843180047272682124325,
  1.87278433509846713939348790992,
  2.01564147795560999653634505277,
  2.14064147795560999653634505277,
  2.25175258906672110764745616389,
  2.35175258906672110764745616389,
  2.44266167997581201673836525479,
  2.52599501330914535007169858813,
  2.60291809023222227314862166505,
  2.67434666166079370172005023648,
  2.74101332832746036838671690315,
  2.80351332832746036838671690315,
  2.86233685773922507426906984432,
  2.91789241329478062982462539988,
  2.97052399224214905087725697883,
  3.02052399224214905087725697883,
  3.06814303986119666992487602645,
  3.11359758531574212447033057190,
  3.15707584618530734186163491973,
  3.1987425128519740085283015864,
  3.2387425128519740085283015864,
  3.2772040513135124700667631249,
  3.3142410883505495071038001619,
  3.3499553740648352213895144476,
  3.3844381326855248765619282407,
  3.4177714660188582098952615740,
  3.4500295305349872421533260902,
  3.4812795305349872421533260902,
  3.5115825608380175451836291205,
  3.5409943255438998981248055911,
  3.5695657541153284695533770196,
  3.5973435318931062473311547974,
  3.6243705589201332743581818244,
  3.6506863483938174848844976139,
  3.6763273740348431259101386396,
  3.7013273740348431259101386396,
  3.7257176179372821503003825420,
  3.7495271417468059598241920658,
  3.7727829557002943319172153216,
  3.7955102284275670591899425943,
  3.8177324506497892814121648166,
  3.8394715810845718901078169905,
  3.8607481768292527411716467777,
  3.8815815101625860745049801110,
  3.9019896734278921969539597029,
  3.9219896734278921969539597029,
  3.9415975165651470989147440166,
  3.9608282857959163296839747858,
  3.9796962103242182164764276160,
  3.9982147288427367349949461345,
  4.0163965470245549168131279527,
  4.0342536898816977739559850956,
  4.0517975495308205809735289552,
  4.0690389288411654085597358518,
  4.0859880813835382899156680552,
  4.1026547480502049565823347218,
  4.1190481906731557762544658694,
  4.1351772229312202923834981274,
  4.1510502388042361653993711433,
  4.1666752388042361653993711433,
  4.1820598541888515500147557587,
  4.1972113693403667015299072739,
  4.2121367424746950597388624977,
  4.2268426248276362362094507330,
  4.2413353784508246420065521823,
  4.2556210927365389277208378966,
  4.2697055997787924488475984600,
  4.2835944886676813377364873489,
  4.2972931188046676391063503626,
  4.3108066323181811526198638761,
  4.3241399656515144859531972094,
  4.3372978603883565912163551041,
  4.3502848733753695782293421171,
  4.3631053861958823987421626300,
  4.3757636140439836645649474401,
  4.3882636140439836645649474401,
  4.4006092930563293435772931191,
  4.4128044150075488557724150703,
  4.4248526077786331931218126607,
  4.4367573696833950978837174226,
  4.4485220755657480390601880108,
  4.4601499825424922251066996387,
  4.4716442354160554434975042364,
  4.4830078717796918071338678728,
  4.4942438268358715824147667492,
  4.5053549379469826935258778603,
  4.5163439489359936825368668713,
  4.5272135141533849868846929582,
  4.5379662023254279976373811303,
  4.5486045001977684231692960239,
  4.5591308159872421073798223397,
  4.5695474826539087740464890064,
  4.5798567610044242379640147796,
  4.5900608426370772991885045755,
  4.6001618527380874001986055856
};

#define FACT_TABLE_MAX  170
#define FACT_TABLE_SIZE (FACT_TABLE_MAX+1)
static struct {int n; double f; long i; } fact_table[FACT_TABLE_SIZE] = {
    { 0,  1.0,     1L     },
    { 1,  1.0,     1L     },
    { 2,  2.0,     2L     },
    { 3,  6.0,     6L     },
    { 4,  24.0,    24L    },
    { 5,  120.0,   120L   },
    { 6,  720.0,   720L   },
    { 7,  5040.0,  5040L  },
    { 8,  40320.0, 40320L },

    { 9,  362880.0,     362880L    },
    { 10, 3628800.0,    3628800L   },
    { 11, 39916800.0,   39916800L  },
    { 12, 479001600.0,  479001600L },

    { 13, 6227020800.0,                               0 },
    { 14, 87178291200.0,                              0 },
    { 15, 1307674368000.0,                            0 },
    { 16, 20922789888000.0,                           0 },
    { 17, 355687428096000.0,                          0 },
    { 18, 6402373705728000.0,                         0 },
    { 19, 121645100408832000.0,                       0 },
    { 20, 2432902008176640000.0,                      0 },
    { 21, 51090942171709440000.0,                     0 },
    { 22, 1124000727777607680000.0,                   0 },
    { 23, 25852016738884976640000.0,                  0 },
    { 24, 620448401733239439360000.0,                 0 },
    { 25, 15511210043330985984000000.0,               0 },
    { 26, 403291461126605635584000000.0,              0 },
    { 27, 10888869450418352160768000000.0,            0 },
    { 28, 304888344611713860501504000000.0,           0 },
    { 29, 8841761993739701954543616000000.0,          0 },
    { 30, 265252859812191058636308480000000.0,        0 },
    { 31, 8222838654177922817725562880000000.0,       0 },
    { 32, 263130836933693530167218012160000000.0,     0 },
    { 33, 8683317618811886495518194401280000000.0,    0 },
    { 34, 2.95232799039604140847618609644e38,  0 },
    { 35, 1.03331479663861449296666513375e40,  0 },
    { 36, 3.71993326789901217467999448151e41,  0 },
    { 37, 1.37637530912263450463159795816e43,  0 },
    { 38, 5.23022617466601111760007224100e44,  0 },
    { 39, 2.03978820811974433586402817399e46,  0 },
    { 40, 8.15915283247897734345611269600e47,  0 },
    { 41, 3.34525266131638071081700620534e49,  0 },
    { 42, 1.40500611775287989854314260624e51,  0 },
    { 43, 6.04152630633738356373551320685e52,  0 },
    { 44, 2.65827157478844876804362581101e54,  0 },
    { 45, 1.19622220865480194561963161496e56,  0 },
    { 46, 5.50262215981208894985030542880e57,  0 },
    { 47, 2.58623241511168180642964355154e59,  0 },
    { 48, 1.24139155925360726708622890474e61,  0 },
    { 49, 6.08281864034267560872252163321e62,  0 },
    { 50, 3.04140932017133780436126081661e64,  0 },
    { 51, 1.55111875328738228022424301647e66,  0 },
    { 52, 8.06581751709438785716606368564e67,  0 },
    { 53, 4.27488328406002556429801375339e69,  0 },
    { 54, 2.30843697339241380472092742683e71,  0 },
    { 55, 1.26964033536582759259651008476e73,  0 },
    { 56, 7.10998587804863451854045647464e74,  0 },
    { 57, 4.05269195048772167556806019054e76,  0 },
    { 58, 2.35056133128287857182947491052e78,  0 },
    { 59, 1.38683118545689835737939019720e80,  0 },
    { 60, 8.32098711274139014427634118320e81,  0 },
    { 61, 5.07580213877224798800856812177e83,  0 },
    { 62, 3.14699732603879375256531223550e85,  0 },
    { 63, 1.982608315404440064116146708360e87,  0 },
    { 64, 1.268869321858841641034333893350e89,  0 },
    { 65, 8.247650592082470666723170306800e90,  0 },
    { 66, 5.443449390774430640037292402480e92,  0 },
    { 67, 3.647111091818868528824985909660e94,  0 },
    { 68, 2.480035542436830599600990418570e96,  0 },
    { 69, 1.711224524281413113724683388810e98,  0 },
    { 70, 1.197857166996989179607278372170e100,  0 },
    { 71, 8.504785885678623175211676442400e101,  0 },
    { 72, 6.123445837688608686152407038530e103,  0 },
    { 73, 4.470115461512684340891257138130e105,  0 },
    { 74, 3.307885441519386412259530282210e107,  0 },
    { 75, 2.480914081139539809194647711660e109,  0 },
    { 76, 1.885494701666050254987932260860e111,  0 },
    { 77, 1.451830920282858696340707840860e113,  0 },
    { 78, 1.132428117820629783145752115870e115,  0 },
    { 79, 8.946182130782975286851441715400e116,  0 },
    { 80, 7.156945704626380229481153372320e118,  0 },
    { 81, 5.797126020747367985879734231580e120,  0 },
    { 82, 4.753643337012841748421382069890e122,  0 },
    { 83, 3.945523969720658651189747118010e124,  0 },
    { 84, 3.314240134565353266999387579130e126,  0 },
    { 85, 2.817104114380550276949479442260e128,  0 },
    { 86, 2.422709538367273238176552320340e130,  0 },
    { 87, 2.107757298379527717213600518700e132,  0 },
    { 88, 1.854826422573984391147968456460e134,  0 },
    { 89, 1.650795516090846108121691926250e136,  0 },
    { 90, 1.485715964481761497309522733620e138,  0 },
    { 91, 1.352001527678402962551665687590e140,  0 },
    { 92, 1.243841405464130725547532432590e142,  0 },
    { 93, 1.156772507081641574759205162310e144,  0 },
    { 94, 1.087366156656743080273652852570e146,  0 },
    { 95, 1.032997848823905926259970209940e148,  0 },
    { 96, 9.916779348709496892095714015400e149,  0 },
    { 97, 9.619275968248211985332842594960e151,  0 },
    { 98, 9.426890448883247745626185743100e153,  0 },
    { 99, 9.332621544394415268169923885600e155,  0 },
    { 100, 9.33262154439441526816992388563e157,  0 },
    { 101, 9.42594775983835942085162312450e159,  0 },
    { 102, 9.61446671503512660926865558700e161,  0 },
    { 103, 9.90290071648618040754671525458e163,  0 },
    { 104, 1.02990167451456276238485838648e166,  0 },
    { 105, 1.08139675824029090050410130580e168,  0 },
    { 106, 1.146280563734708354534347384148e170,  0 },
    { 107, 1.226520203196137939351751701040e172,  0 },
    { 108, 1.324641819451828974499891837120e174,  0 },
    { 109, 1.443859583202493582204882102460e176,  0 },
    { 110, 1.588245541522742940425370312710e178,  0 },
    { 111, 1.762952551090244663872161047110e180,  0 },
    { 112, 1.974506857221074023536820372760e182,  0 },
    { 113, 2.231192748659813646596607021220e184,  0 },
    { 114, 2.543559733472187557120132004190e186,  0 },
    { 115, 2.925093693493015690688151804820e188,  0 },
    { 116, 3.393108684451898201198256093590e190,  0 },
    { 117, 3.96993716080872089540195962950e192,  0 },
    { 118, 4.68452584975429065657431236281e194,  0 },
    { 119, 5.57458576120760588132343171174e196,  0 },
    { 120, 6.68950291344912705758811805409e198,  0 },
    { 121, 8.09429852527344373968162284545e200,  0 },
    { 122, 9.87504420083360136241157987140e202,  0 },
    { 123, 1.21463043670253296757662432419e205,  0 },
    { 124, 1.50614174151114087979501416199e207,  0 },
    { 125, 1.88267717688892609974376770249e209,  0 },
    { 126, 2.37217324288004688567714730514e211,  0 },
    { 127, 3.01266001845765954480997707753e213,  0 },
    { 128, 3.85620482362580421735677065923e215,  0 },
    { 129, 4.97450422247728744039023415041e217,  0 },
    { 130, 6.46685548922047367250730439554e219,  0 },
    { 131, 8.47158069087882051098456875820e221,  0 },
    { 132, 1.11824865119600430744996307608e224,  0 },
    { 133, 1.48727070609068572890845089118e226,  0 },
    { 134, 1.99294274616151887673732419418e228,  0 },
    { 135, 2.69047270731805048359538766215e230,  0 },
    { 136, 3.65904288195254865768972722052e232,  0 },
    { 137, 5.01288874827499166103492629211e234,  0 },
    { 138, 6.91778647261948849222819828311e236,  0 },
    { 139, 9.61572319694108900419719561353e238,  0 },
    { 140, 1.34620124757175246058760738589e241,  0 },
    { 141, 1.89814375907617096942852641411e243,  0 },
    { 142, 2.69536413788816277658850750804e245,  0 },
    { 143, 3.85437071718007277052156573649e247,  0 },
    { 144, 5.55029383273930478955105466055e249,  0 },
    { 145, 8.04792605747199194484902925780e251,  0 },
    { 146, 1.17499720439091082394795827164e254,  0 },
    { 147, 1.72724589045463891120349865931e256,  0 },
    { 148, 2.55632391787286558858117801578e258,  0 },
    { 149, 3.80892263763056972698595524351e260,  0 },
    { 150, 5.71338395644585459047893286526e262,  0 },
    { 151, 8.62720977423324043162318862650e264,  0 },
    { 152, 1.31133588568345254560672467123e267,  0 },
    { 153, 2.00634390509568239477828874699e269,  0 },
    { 154, 3.08976961384735088795856467036e271,  0 },
    { 155, 4.78914290146339387633577523906e273,  0 },
    { 156, 7.47106292628289444708380937294e275,  0 },
    { 157, 1.17295687942641442819215807155e278,  0 },
    { 158, 1.85327186949373479654360975305e280,  0 },
    { 159, 2.94670227249503832650433950735e282,  0 },
    { 160, 4.71472363599206132240694321176e284,  0 },
    { 161, 7.59070505394721872907517857094e286,  0 },
    { 162, 1.22969421873944943411017892849e289,  0 },
    { 163, 2.00440157654530257759959165344e291,  0 },
    { 164, 3.28721858553429622726333031164e293,  0 },
    { 165, 5.42391066613158877498449501421e295,  0 },
    { 166, 9.00369170577843736647426172359e297,  0 },
    { 167, 1.50361651486499904020120170784e300,  0 },
    { 168, 2.52607574497319838753801886917e302,  0 },
    { 169, 4.26906800900470527493925188890e304,  0 },
    { 170, 7.25741561530799896739672821113e306,  0 },

    /*
    { 171, 1.24101807021766782342484052410e309,  0 },
    { 172, 2.13455108077438865629072570146e311,  0 },
    { 173, 3.69277336973969237538295546352e313,  0 },
    { 174, 6.42542566334706473316634250653e315,  0 },
    { 175, 1.12444949108573632830410993864e318,  0 },
    { 176, 1.97903110431089593781523349201e320,  0 },
    { 177, 3.50288505463028580993296328086e322,  0 },
    { 178, 6.23513539724190874168067463993e324,  0 },
    { 179, 1.11608923610630166476084076055e327,  0 },
    { 180, 2.00896062499134299656951336898e329,  0 },
    { 181, 3.63621873123433082379081919786e331,  0 },
    { 182, 6.61791809084648209929929094011e333,  0 },
    { 183, 1.21107901062490622417177024204e336,  0 },
    { 184, 2.22838537954982745247605724535e338,  0 },
    { 185, 4.12251295216718078708070590390e340,  0 },
    { 186, 7.66787409103095626397011298130e342,  0 },
    { 187, 1.43389245502278882136241112750e345,  0 },
    { 188, 2.69571781544284298416133291969e347,  0 },
    { 189, 5.09490667118697324006491921822e349,  0 },
    { 190, 9.68032267525524915612334651460e351,  0 },
    { 191, 1.84894163097375258881955918429e354,  0 },
    { 192, 3.54996793146960497053355363384e356,  0 },
    { 193, 6.85143810773633759312975851330e358,  0 },
    { 194, 1.32917899290084949306717315158e361,  0 },
    { 195, 2.59189903615665651148098764559e363,  0 },
    { 196, 5.08012211086704676250273578535e365,  0 },
    { 197, 1.00078405584080821221303894971e368,  0 },
    { 198, 1.98155243056480026018181712043e370,  0 },
    { 199, 3.94328933682395251776181606966e372,  0 },
    { 200, 7.88657867364790503552363213932e374,  0 }
    */
};

static double bk0_data[11] = {
  -0.03532739323390276872,
   0.3442898999246284869, 
   0.03597993651536150163,
   0.00126461541144692592,
   0.00002286212103119451,
   0.00000025347910790261,
   0.00000000190451637722,
   0.00000000001034969525,
   0.00000000000004259816,
   0.00000000000000013744,
   0.00000000000000000035
};
static cheb_series bk0_cs = {
  bk0_data,
  10,
  -1, 1,
  10
};

static double ak0_data[17] = {
  -0.07643947903327941,
  -0.02235652605699819,
   0.00077341811546938,
  -0.00004281006688886,
   0.00000308170017386,
  -0.00000026393672220,
   0.00000002563713036,
  -0.00000000274270554,
   0.00000000031694296,
  -0.00000000003902353,
   0.00000000000506804,
  -0.00000000000068895,
   0.00000000000009744,
  -0.00000000000001427,
   0.00000000000000215,
  -0.00000000000000033,
   0.00000000000000005
};
static cheb_series ak0_cs = {
  ak0_data,
  16,
  -1, 1,
  10
};

static double ak02_data[14] = {
  -0.01201869826307592,
  -0.00917485269102569,
   0.00014445509317750,
  -0.00000401361417543,
   0.00000015678318108,
  -0.00000000777011043,
   0.00000000046111825,
  -0.00000000003158592,
   0.00000000000243501,
  -0.00000000000020743,
   0.00000000000001925,
  -0.00000000000000192,
   0.00000000000000020,
  -0.00000000000000002
};
static cheb_series ak02_cs = {
  ak02_data,
  13,
  -1, 1,
  8
};

static double bk1_data[11] = {
   0.0253002273389477705,
  -0.3531559607765448760, 
  -0.1226111808226571480, 
  -0.0069757238596398643,
  -0.0001730288957513052,
  -0.0000024334061415659,
  -0.0000000221338763073,
  -0.0000000001411488392,
  -0.0000000000006666901,
  -0.0000000000000024274,
  -0.0000000000000000070
};

static cheb_series bk1_cs = {
  bk1_data,
  10,
  -1, 1,
  8
};

static double ak1_data[17] = {
   0.27443134069738830, 
   0.07571989953199368,
  -0.00144105155647540,
   0.00006650116955125,
  -0.00000436998470952,
   0.00000035402774997,
  -0.00000003311163779,
   0.00000000344597758,
  -0.00000000038989323,
   0.00000000004720819,
  -0.00000000000604783,
   0.00000000000081284,
  -0.00000000000011386,
   0.00000000000001654,
  -0.00000000000000248,
   0.00000000000000038,
  -0.00000000000000006
};
static cheb_series ak1_cs = {
  ak1_data,
  16,
  -1, 1,
  9
};

static double ak12_data[14] = {
   0.06379308343739001,
   0.02832887813049721,
  -0.00024753706739052,
   0.00000577197245160,
  -0.00000020689392195,
   0.00000000973998344,
  -0.00000000055853361,
   0.00000000003732996,
  -0.00000000000282505,
   0.00000000000023720,
  -0.00000000000002176,
   0.00000000000000215,
  -0.00000000000000022,
   0.00000000000000002
};
static cheb_series ak12_cs = {
  ak12_data,
  13,
  -1, 1,
  7
};

static double bi0_data[12] = {
  -.07660547252839144951,
  1.92733795399380827000,
   .22826445869203013390, 
   .01304891466707290428,
   .00043442709008164874,
   .00000942265768600193,
   .00000014340062895106,
   .00000000161384906966,
   .00000000001396650044,
   .00000000000009579451,
   .00000000000000053339,
   .00000000000000000245
};
static cheb_series bi0_cs = {
  bi0_data,
  11,
  -1, 1,
  11
};

static double ai0_data[21] = {
   .07575994494023796, 
   .00759138081082334,
   .00041531313389237,
   .00001070076463439,
  -.00000790117997921,
  -.00000078261435014,
   .00000027838499429,
   .00000000825247260,
  -.00000001204463945,
   .00000000155964859,
   .00000000022925563,
  -.00000000011916228,
   .00000000001757854,
   .00000000000112822,
  -.00000000000114684,
   .00000000000027155,
  -.00000000000002415,
  -.00000000000000608,
   .00000000000000314,
  -.00000000000000071,
   .00000000000000007
};
static cheb_series ai0_cs = {
  ai0_data,
  20,
  -1, 1,
  13
};

static double ai02_data[22] = {
   .05449041101410882,
   .00336911647825569,
   .00006889758346918,
   .00000289137052082,
   .00000020489185893,
   .00000002266668991,
   .00000000339623203,
   .00000000049406022,
   .00000000001188914,
  -.00000000003149915,
  -.00000000001321580,
  -.00000000000179419,
   .00000000000071801,
   .00000000000038529,
   .00000000000001539,
  -.00000000000004151,
  -.00000000000000954,
   .00000000000000382,
   .00000000000000176,
  -.00000000000000034,
  -.00000000000000027,
   .00000000000000003
};
static cheb_series ai02_cs = {
  ai02_data,
  21,
  -1, 1,
  11
};

static double bi1_data[11] = {
  -0.001971713261099859,
   0.407348876675464810,
   0.034838994299959456,
   0.001545394556300123,
   0.000041888521098377,
   0.000000764902676483,
   0.000000010042493924,
   0.000000000099322077,
   0.000000000000766380,
   0.000000000000004741,
   0.000000000000000024
};
static cheb_series bi1_cs = {
  bi1_data,
  10,
  -1, 1,
  10
};

static double ai1_data[21] = {
  -0.02846744181881479,
  -0.01922953231443221,
  -0.00061151858579437,
  -0.00002069971253350,
   0.00000858561914581,
   0.00000104949824671,
  -0.00000029183389184,
  -0.00000001559378146,
   0.00000001318012367,
  -0.00000000144842341,
  -0.00000000029085122,
   0.00000000012663889,
  -0.00000000001664947,
  -0.00000000000166665,
   0.00000000000124260,
  -0.00000000000027315,
   0.00000000000002023,
   0.00000000000000730,
  -0.00000000000000333,
   0.00000000000000071,
  -0.00000000000000006
};
static cheb_series ai1_cs = {
  ai1_data,
  20,
  -1, 1,
  11
};

static double ai12_data[22] = {
   0.02857623501828014,
  -0.00976109749136147,
  -0.00011058893876263,
  -0.00000388256480887,
  -0.00000025122362377,
  -0.00000002631468847,
  -0.00000000383538039,
  -0.00000000055897433,
  -0.00000000001897495,
   0.00000000003252602,
   0.00000000001412580,
   0.00000000000203564,
  -0.00000000000071985,
  -0.00000000000040836,
  -0.00000000000002101,
   0.00000000000004273,
   0.00000000000001041,
  -0.00000000000000382,
  -0.00000000000000186,
   0.00000000000000033,
   0.00000000000000028,
  -0.00000000000000003
};

static cheb_series ai12_cs = {
  ai12_data,
  21,
  -1, 1,
  9
};

/* This routine is taken from the GNU Science Library (GSL 1.1.1)...
 * what appears here is a modified form.	*/
complex cmpow( complex a, complex b )
{ 
	double logr, max, u, theta, rho, beta;
	volatile double y;
	complex Result;

	if (a.re >= a.im)
	{
		max = a.re;
		u = a.im / a.re;
	}
	else
	{
		max = a.im;
		u = a.re / a.im;
	}

	if ( u > SMALLFLOAT )
	{
		y = 1 + u*u;
		logr = log(max) + 0.5*( log(y) - log((y-1)-u*u)/y );
		theta = atan2 (a.im, a.re);
		rho = exp (logr * b.re - b.im * theta);
		beta = theta * b.re + b.im * logr;
	}
	else
	{
		ASSERT( a.im == 0. );
		logr = log(max);
		rho = exp (logr * b.re);
		beta = b.im * logr;
	}
	  
	Result.re = rho * cos(beta);
	Result.im = rho * sin(beta);

	return Result;
}

/* complex Gamma function in double precision */
/* this routine is a slightly modified version of the one found 
 * at http://momonga.t.u-tokyo.ac.jp/~ooura/gamerf.html	
 * The following copyright applies: 
    Copyright(C) 1996 Takuya OOURA (email: ooura@mmm.t.u-tokyo.ac.jp).
    You may use, copy, modify this code for any purpose and 
    without fee. You may distribute this ORIGINAL package.	*/
complex cdgamma(complex x)
{
    complex y;
    double xr, xi, wr, wi, ur, ui, vr, vi, yr, yi, t;

    xr = x.re;
    xi = x.im;
    if (xr < 0) {
        wr = 1 - xr;
        wi = -xi;
    } else {
        wr = xr;
        wi = xi;
    }
    ur = wr + 6.00009857740312429;
    vr = ur * (wr + 4.99999857982434025) - wi * wi;
    vi = wi * (wr + 4.99999857982434025) + ur * wi;
    yr = ur * 13.2280130755055088 + vr * 66.2756400966213521 + 
        0.293729529320536228;
    yi = wi * 13.2280130755055088 + vi * 66.2756400966213521;
    ur = vr * (wr + 4.00000003016801681) - vi * wi;
    ui = vi * (wr + 4.00000003016801681) + vr * wi;
    vr = ur * (wr + 2.99999999944915534) - ui * wi;
    vi = ui * (wr + 2.99999999944915534) + ur * wi;
    yr += ur * 91.1395751189899762 + vr * 47.3821439163096063;
    yi += ui * 91.1395751189899762 + vi * 47.3821439163096063;
    ur = vr * (wr + 2.00000000000603851) - vi * wi;
    ui = vi * (wr + 2.00000000000603851) + vr * wi;
    vr = ur * (wr + 0.999999999999975753) - ui * wi;
    vi = ui * (wr + 0.999999999999975753) + ur * wi;
    yr += ur * 10.5400280458730808 + vr;
    yi += ui * 10.5400280458730808 + vi;
    ur = vr * wr - vi * wi;
    ui = vi * wr + vr * wi;
    t = ur * ur + ui * ui;
    vr = yr * ur + yi * ui + t * 0.0327673720261526849;
    vi = yi * ur - yr * ui;
    yr = wr + 7.31790632447016203;
    ur = log(yr * yr + wi * wi) * 0.5 - 1;
    ui = atan2(wi, yr);
    yr = exp(ur * (wr - 0.5) - ui * wi - 3.48064577727581257) / t;
    yi = ui * (wr - 0.5) + ur * wi;
    ur = yr * cos(yi);
    ui = yr * sin(yi);
    yr = ur * vr - ui * vi;
    yi = ui * vr + ur * vi;
    if (xr < 0) {
        wr = xr * 3.14159265358979324;
        wi = exp(xi * 3.14159265358979324);
        vi = 1 / wi;
        ur = (vi + wi) * sin(wr);
        ui = (vi - wi) * cos(wr);
        vr = ur * yr + ui * yi;
        vi = ui * yr - ur * yi;
        ur = 6.2831853071795862 / (vr * vr + vi * vi);
        yr = ur * vr;
        yi = ur * vi;
    }
    y.re = yr;
	y.im = yi;

    return y;
}

/* From the GNU Science Library (GSL 1.4) */
/*-*-*-*-*-*-*-*-*-*-*-* Functions with Error Codes *-*-*-*-*-*-*-*-*-*-*-*/
void gsl_sf_coulomb_wave_FG_e(const double eta, const double x, 
                            const double lam_F,
                            const int  k_lam_G,      /* lam_G = lam_F - k_lam_G */
                            double *F, double *Fp,
                            double *G, double *Gp,
                            double *exp_F, double *exp_G)
{
	const double lam_G = lam_F - k_lam_G;

	if(x < 0.0 || lam_F <= -0.5 || lam_G <= -0.5)
	{
#if	0
		GSL_SF_RESULT_SET(F,  0.0, 0.0);
		GSL_SF_RESULT_SET(Fp, 0.0, 0.0);
		GSL_SF_RESULT_SET(G,  0.0, 0.0);
		GSL_SF_RESULT_SET(Gp, 0.0, 0.0);
		*exp_F = 0.0;
		*exp_G = 0.0;
		GSL_ERROR ("domain error", GSL_EDOM);
#endif
		puts( "[Stop in gsl_sf_coulomb_wave_FG_e (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	}
	else if(x == 0.0)
	{
#if	0
		gsl_sf_result C0;
		CLeta(0.0, eta, &C0);
		GSL_SF_RESULT_SET(F,  0.0, 0.0);
		GSL_SF_RESULT_SET(Fp, 0.0, 0.0);
		GSL_SF_RESULT_SET(G,  0.0, 0.0); /* FIXME: should be Inf */
		GSL_SF_RESULT_SET(Gp, 0.0, 0.0); /* FIXME: should be Inf */
		*exp_F = 0.0;
		*exp_G = 0.0;
		if(lam_F == 0.0)
		{
			GSL_SF_RESULT_SET(Fp, C0.val, C0.err);
		}
		if(lam_G == 0.0)
		{
			GSL_SF_RESULT_SET(Gp, 1.0/C0.val, fabs(C0.err/C0.val)/fabs(C0.val));
		}
		GSL_ERROR ("domain error", GSL_EDOM);
		/* After all, since we are asking for G, this is a domain error... */
#endif
		puts( "[Stop in gsl_sf_coulomb_wave_FG_e (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	}
	else if(x < 1.2 && 2.0*PI*eta < 0.9*(-DBL_MIN_10_EXP) && fabs(eta*x) < 10.0)
	{
		/* Reduce to a small lambda value and use the series
		 * representations for F and G. We cannot allow eta to
		 * be large and positive because the connection formula
		 * for G_lam is badly behaved due to an underflow in sin(phi_lam) 
		 * [see coulomb_FG_series() and coulomb_connection() above].
		 * Note that large negative eta is ok however.
		 */
		const double SMALL = sqrt(DBL_EPSILON);
		const int N    = (int)(lam_F + 0.5);
		const int span = MAX2(k_lam_G, N);
		const double lam_min = lam_F - N;    /* -1/2 <= lam_min < 1/2 */
		double F_lam_F, Fp_lam_F;
		double G_lam_G, Gp_lam_G;
		double Fp_over_F_lam_F;
		double F_sign_lam_F;
		double F_lam_min_unnorm, Fp_lam_min_unnorm;
		double Fp_over_F_lam_min;
		double F_lam_min;
		double G_lam_min, Gp_lam_min;
		double F_scale;

		/* Determine F'/F at lam_F. */
		int CF1_count;
		int stat_Fr;

		coulomb_CF1(lam_F, eta, x, &F_sign_lam_F, &Fp_over_F_lam_F, &CF1_count);

		/* Recurse down with unnormalized F,F' values. */
		F_lam_F  = SMALL;
		Fp_lam_F = Fp_over_F_lam_F * F_lam_F;
		if(span != 0)
		{
			coulomb_F_recur(lam_min, span, eta, x,
				 F_lam_F, Fp_lam_F, &F_lam_min_unnorm, &Fp_lam_min_unnorm );
		}
		else
		{
			F_lam_min_unnorm  =  F_lam_F;
			Fp_lam_min_unnorm = Fp_lam_F;
			stat_Fr = EXIT_SUCCESS;
		}

		/* Determine F and G at lam_min. */
		if(lam_min == -0.5)
		{
			coulomb_FGmhalf_series(eta, x, &F_lam_min, &G_lam_min);
		}
		else if(lam_min == 0.0)
		{
			coulomb_FG0_series(eta, x, &F_lam_min, &G_lam_min);
		}
		else if(lam_min == 0.5)
		{
			/* This cannot happen. 
			*F  = F_lam_F;
			*Fp = Fp_lam_F;
			*G  = G_lam_G;
			*Gp = Gp_lam_G;
			*exp_F = 0.0;
			*exp_G = 0.0;*/
			fprintf( ioQQQ, "Huh?!?!?!");
			puts( "[Stop in gsl_sf_coulomb_wave_FG_e (thirdparty.c)]" );
			ASSERT( 1==0 );
			cdEXIT( EXIT_FAILURE );
		}
		else
		{
			coulomb_FG_series(lam_min, eta, x, &F_lam_min, &G_lam_min);
		}

		/* Determine remaining quantities. */
		Fp_over_F_lam_min = Fp_lam_min_unnorm / F_lam_min_unnorm;
		Gp_lam_min  = Fp_over_F_lam_min*G_lam_min - 1.0/F_lam_min;
		F_scale     = F_lam_min / F_lam_min_unnorm;

		/* Apply scale to the original F,F' values. */
		F_lam_F     *= F_scale;
		Fp_lam_F    *= F_scale;
		
		/* Recurse up to get the required G,G' values. */
		coulomb_G_recur(lam_min, MAX2(N-k_lam_G,0), eta, x,
			G_lam_min, Gp_lam_min, &G_lam_G, &Gp_lam_G );

		*F  = F_lam_F;
		*Fp = Fp_lam_F;
		*G  = G_lam_G;
		*Gp = Gp_lam_G;

		*exp_F = 0.0;
		*exp_G = 0.0;

		return;
		/* GSL_ERROR_SELECT_4(stat_ser, stat_CF1, stat_Fr, stat_Gr); */
	}
#if	0
	else if(x < 2.0*eta)
	{
		/* Use WKB approximation to obtain F and G at the two
		 * lambda values, and use the Wronskian and the
		 * continued fractions for F'/F to obtain F' and G'.
		 */
		double F_lam_F, G_lam_F;
		double F_lam_G, G_lam_G;
		double exp_lam_F, exp_lam_G;
		int stat_lam_F;
		int stat_lam_G;
		int stat_CF1_lam_F;
		int stat_CF1_lam_G;
		int CF1_count;
		double Fp_over_F_lam_F;
		double Fp_over_F_lam_G;
		double F_sign_lam_F;
		double F_sign_lam_G;

		coulomb_jwkb(lam_F, eta, x, &F_lam_F, &G_lam_F, &exp_lam_F);
		if(k_lam_G == 0)
		{
			stat_lam_G = stat_lam_F;
			F_lam_G = F_lam_F;
			G_lam_G = G_lam_F;
			exp_lam_G = exp_lam_F;
		}
		else
		{
			coulomb_jwkb(lam_G, eta, x, &F_lam_G, &G_lam_G, &exp_lam_G);
		}

		coulomb_CF1(lam_F, eta, x, &F_sign_lam_F, &Fp_over_F_lam_F, &CF1_count);
		if(k_lam_G == 0)
		{
			stat_CF1_lam_G  = stat_CF1_lam_F;
			F_sign_lam_G    = F_sign_lam_F;
			Fp_over_F_lam_G = Fp_over_F_lam_F;
		}
		else
		{
			coulomb_CF1(lam_G, eta, x, &F_sign_lam_G, &Fp_over_F_lam_G, &CF1_count);
		}

		*F = F_lam_F;
		*G = G_lam_G;
		*Fp  = Fp_over_F_lam_F * F_lam_F;
		*Gp  = Fp_over_F_lam_G * G_lam_G - 1.0/F_lam_G;

		*exp_F = exp_lam_F;
		*exp_G = exp_lam_G;

		return;
	}
#endif
	else
	{
		/* x > 2 eta, so we know that we can find a lambda value such
		 * that x is above the turning point. We do this, evaluate
		 * using Steed's method at that oscillatory point, then
		 * use recursion on F and G to obtain the required values.
		 *
		 * lam_0   = a value of lambda such that x is below the turning point
		 * lam_min = minimum of lam_0 and the requested lam_G, since
		 *           we must go at least as low as lam_G
		 */
		const double SMALL = sqrt(DBL_EPSILON);
		const double C = sqrt(1.0 + 4.0*x*(x-2.0*eta));
		const int N = (int)(ceil(lam_F - C + 0.5));
		const double lam_0   = lam_F - MAX2(N, 0);
		const double lam_min = MIN2(lam_0, lam_G);
		double F_lam_F, Fp_lam_F;
		double G_lam_G, Gp_lam_G;
		double F_lam_min_unnorm, Fp_lam_min_unnorm;
		double F_lam_min, Fp_lam_min;
		double G_lam_min, Gp_lam_min;
		double Fp_over_F_lam_F;
		double Fp_over_F_lam_min;
		double F_sign_lam_F;
		double P_lam_min, Q_lam_min;
		double alpha;
		double gamma;
		double F_scale;

		int CF1_count;
		int CF2_count;

		int F_recur_count;
		int G_recur_count;

		double err_amplify;

		ASSERT( x > 2.0*eta );

		coulomb_CF1(lam_F, eta, x, &F_sign_lam_F, &Fp_over_F_lam_F, &CF1_count);


		F_lam_F  = SMALL;
		Fp_lam_F = Fp_over_F_lam_F * F_lam_F;

		/* Backward recurrence to get F,Fp at lam_min */
		F_recur_count = MAX2(k_lam_G, N);
		coulomb_F_recur(lam_min, F_recur_count, eta, x,
								  F_lam_F, Fp_lam_F,
								  &F_lam_min_unnorm, &Fp_lam_min_unnorm
								  );
		Fp_over_F_lam_min = Fp_lam_min_unnorm / F_lam_min_unnorm;

		/* Steed evaluation to complete evaluation of F,Fp,G,Gp at lam_min */
		coulomb_CF2(lam_min, eta, x, &P_lam_min, &Q_lam_min, &CF2_count);
		alpha = Fp_over_F_lam_min - P_lam_min;
		gamma = alpha/Q_lam_min;
		F_lam_min  = F_sign_lam_F / sqrt(alpha*alpha/Q_lam_min + Q_lam_min);
		Fp_lam_min = Fp_over_F_lam_min * F_lam_min;
		G_lam_min  = gamma * F_lam_min;
		Gp_lam_min = (P_lam_min * gamma - Q_lam_min) * F_lam_min;

		/* Apply scale to values of F,Fp at lam_F (the top). */
		F_scale = F_lam_min / F_lam_min_unnorm;    
		F_lam_F  *= F_scale;
		Fp_lam_F *= F_scale;

		/* Forward recurrence to get G,Gp at lam_G (the top). */
		G_recur_count = MAX2(N-k_lam_G,0);
		coulomb_G_recur(lam_min, G_recur_count, eta, x,
								  G_lam_min, Gp_lam_min,
								  &G_lam_G, &Gp_lam_G
								  );

		err_amplify = CF1_count + CF2_count + F_recur_count + G_recur_count + 1;

		*F  = F_lam_F;
		*Fp = Fp_lam_F;
		*G  = G_lam_G;
		*Gp = Gp_lam_G;
		
		*exp_F = 0.0;
		*exp_G = 0.0;

		return;
		/*GSL_ERROR_SELECT_4(stat_CF1, stat_CF2, stat_Fr, stat_Gr);*/
	}
}


/* the full definition of C_L(eta) for any valid L and eta
 * [Abramowitz and Stegun 14.1.7]
 * This depends on the complex gamma function. For large
 * arguments the phase of the complex gamma function is not
 * very accurately determined. However the modulus is, and that
 * is all that we need to calculate C_L.
 *
 * This is not valid for L <= -3/2  or  L = -1.
 */
static double CLeta(double L, double eta)
{
	double ln1; /* log of numerator Gamma function */
	double ln2; /* log of denominator Gamma function */
	double sgn = 1.0;
	/* double arg_val, arg_err; */
	complex cmplxNum;

	if(fabs(eta/(L+1.0)) < DBL_EPSILON)
	{
		ln1 = log( gammafun( L+1.0 ) );
	}
	else
	{
		/* phase of numerator Gamma -- not used */
		double p1;
		cmplxNum.re = L + 1.0;
		cmplxNum.im = eta;
		/* gsl_sf_lngamma_complex_e(L+1.0, eta, &ln1, &p1); */
		/* should be ok */
		lngamma_complex_stirling(cmplxNum, &ln1, &p1);
	}

	/* gsl_sf_lngamma_e(2.0*(L+1.0), &ln2); */
	ln2 = log( gammafun( 2.0*(L+1.0) ) );

	if(L < -1.0)
	{
		sgn = -sgn;
	}


	/*
	arg_val  = L*LN2 - 0.5*eta*PI + ln1.val - ln2.val;
	arg_err  = ln1.err + ln2.err;
	arg_err += GSL_DBL_EPSILON * (fabs(L*LN2) + fabs(0.5*eta*PI));
	return gsl_sf_exp_err_e(arg_val, arg_err, result);
	*/
	return exp( L*LN_TWO - 0.5*eta*PI + ln1 - ln2 );
}


double gsl_sf_coulomb_CL_e(double lam, double eta)
{
  /* CHECK_POINTER(result) */

	if(lam <= -1.0)
	{
		fprintf( ioQQQ, "Argh!!\n");
		puts( "[Stop in gsl_sf_coulomb_CL_e (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
		/* DOMAIN_ERROR(result);*/
	}
	/* this should not happen because we only want integral values of lam */
#if	0
	else if(fabs(lam) < DBL_EPSILON)
	{
		/* saves a calculation of complex_lngamma(), otherwise not necessary */
		return sqrt(C0sq(eta));
		cdEXIT( EXIT_SUCCESS );
	}
#endif
	else
	{
		return CLeta(lam, eta);
	}
}


/* cl[0] .. cl[kmax] = C_{lam_min}(eta) .. C_{lam_min+kmax}(eta)
 */
void gsl_sf_coulomb_CL_array(double lam_min, int kmax, double eta, double *cl)
{
  int k;
  double cl_0;
  cl_0 = gsl_sf_coulomb_CL_e(lam_min, eta);
  cl[0] = cl_0;

  for(k=1; k<=kmax; k++) {
    double L = lam_min + k;
    cl[k] = cl[k-1] * sqrt(L*L + eta*eta)/(L*(2.0*L+1.0));
  }

  return;
  /* GSL_SUCCESS; */
}


/* Evaluate the series for Phi_L(eta,x) and Phi_L*(eta,x)
 * [Abramowitz+Stegun 14.1.5]
 * [Abramowitz+Stegun 14.1.13]
 *
 * The sequence of coefficients A_k^L is
 * manifestly well-controlled for L >= -1/2
 * and eta < 10.
 *
 * This makes sense since this is the region
 * away from threshold, and you expect
 * the evaluation to become easier as you
 * get farther from threshold.
 *
 * Empirically, this is quite well-behaved for
 *   L >= -1/2
 *   eta < 10
 *   x   < 10
 */
#if 0
static
int
coulomb_Phi_series(const double lam, const double eta, const double x,
                   double * result, double * result_star)
{
  int kmin =   5;
  int kmax = 200;
  int k;
  double Akm2 = 1.0;
  double Akm1 = eta/(lam+1.0);
  double Ak;

  double xpow = x;
  double sum  = Akm2 + Akm1*x;
  double sump = (lam+1.0)*Akm2 + (lam+2.0)*Akm1*x;
  double prev_abs_del   = fabs(Akm1*x);
  double prev_abs_del_p = (lam+2.0) * prev_abs_del;

  for(k=2; k<kmax; k++) {
    double del;
    double del_p;
    double abs_del;
    double abs_del_p;

    Ak = (2.0*eta*Akm1 - Akm2)/(k*(2.0*lam + 1.0 + k));

    xpow *= x;
    del   = Ak*xpow;
    del_p = (k+lam+1.0)*del;
    sum  += del;
    sump += del_p;

    abs_del   = fabs(del);
    abs_del_p = fabs(del_p);

    if(          abs_del/(fabs(sum)+abs_del)          < GSL_DBL_EPSILON
       &&   prev_abs_del/(fabs(sum)+prev_abs_del)     < GSL_DBL_EPSILON
       &&      abs_del_p/(fabs(sump)+abs_del_p)       < GSL_DBL_EPSILON
       && prev_abs_del_p/(fabs(sump)+prev_abs_del_p)  < GSL_DBL_EPSILON
       && k > kmin
       ) break;

    /* We need to keep track of the previous delta because when
     * eta is near zero the odd terms of the sum are very small
     * and this could lead to premature termination.
     */
    prev_abs_del   = abs_del;
    prev_abs_del_p = abs_del_p;

    Akm2 = Akm1;
    Akm1 = Ak;
  }

  *result      = sum;
  *result_star = sump;

  if(k==kmax) {
    GSL_ERROR ("error", GSL_EMAXITER);
  }
  else {
    return GSL_SUCCESS;
  }
}
#endif /* 0 */


/* Determine the connection phase, phi_lambda.
 * See coulomb_FG_series() below. We have
 * to be careful about sin(phi)->0. Note that
 * there is an underflow condition for large 
 * positive eta in any case.
 */
static void coulomb_connection(const double lam, const double eta,
                   double *cos_phi, double *sin_phi)
{
  if(eta > -log( DBL_MIN )/2.0*PI-1.0) {
    *cos_phi = 1.0;
    *sin_phi = 0.0;
	fprintf( ioQQQ, "Argh!! v 2.0\n" );
	puts( "[Stop in coulomb_connection (thirdparty.c)]" );
	ASSERT( 1==0 );
	cdEXIT( EXIT_FAILURE );
    /* GSL_ERROR ("error", GSL_EUNDRFLW); */
  }
  else if(eta > -log( DBL_EPSILON )/(4.0*PI)) {
    const double eps = 2.0 * exp(-2.0*PI*eta);
    const double tpl = tan(PI * lam);
    const double dth = eps * tpl / (tpl*tpl + 1.0);
    *cos_phi = -1.0 + 0.5 * dth*dth;
    *sin_phi = -dth;
	return;
	/* GSL_SUCCESS; */
  }
  else {
    double X   = tanh(PI * eta) / tan(PI * lam);
    double phi = -atan(X) - (lam + 0.5) * PI;
    *cos_phi = cos(phi);
    *sin_phi = sin(phi);
    return;
	/* GSL_SUCCESS; */
  }
}


/* Evaluate the Frobenius series for F_lam(eta,x) and G_lam(eta,x).
 * Homegrown algebra. Evaluates the series for F_{lam} and
 * F_{-lam-1}, then uses
 *    G_{lam} = (F_{lam} cos(phi) - F_{-lam-1}) / sin(phi)
 * where
 *    phi = Arg[Gamma[1+lam+I eta]] - Arg[Gamma[-lam + I eta]] - (lam+1/2)Pi
 *        = Arg[Sin[Pi(-lam+I eta)] - (lam+1/2)Pi
 *        = atan2(-cos(lam Pi)sinh(eta Pi), -sin(lam Pi)cosh(eta Pi)) - (lam+1/2)Pi
 *
 *        = -atan(X) - (lam+1/2) Pi,  X = tanh(eta Pi)/tan(lam Pi)
 *
 * Not appropriate for lam <= -1/2, lam = 0, or lam >= 1/2.
 */
static void coulomb_FG_series(const double lam, const double eta, const double x,
                  double *F, double *G)
{
  const int max_iter = 800;
  double ClamA = CLeta(lam, eta);
  double ClamB = CLeta(-lam-1.0, eta);
  const double tlp1 = 2.0*lam + 1.0;
  const double pow_x = pow(x, lam);
  double cos_phi_lam;
  double sin_phi_lam;

  double uA_mm2 = 1.0;                  /* uA sum is for F_{lam} */
  double uA_mm1 = x*eta/(lam+1.0);
  double uA_m;
  double uB_mm2 = 1.0;                  /* uB sum is for F_{-lam-1} */
  double uB_mm1 = -x*eta/lam;
  double uB_m;
  double A_sum = uA_mm2 + uA_mm1;
  double B_sum = uB_mm2 + uB_mm1;
  double A_abs_del_prev = fabs(A_sum);
  double B_abs_del_prev = fabs(B_sum);
  double FA, FB;
  int m = 2;

  coulomb_connection(lam, eta, &cos_phi_lam, &sin_phi_lam);

  while(m < max_iter) {
    double abs_dA;
    double abs_dB;
    uA_m = x*(2.0*eta*uA_mm1 - x*uA_mm2)/(m*(m+tlp1));
    uB_m = x*(2.0*eta*uB_mm1 - x*uB_mm2)/(m*(m-tlp1));
    A_sum += uA_m;
    B_sum += uB_m;
    abs_dA = fabs(uA_m);
    abs_dB = fabs(uB_m);
    if(m > 15) {
      /* Don't bother checking until we have gone out a little ways;
       * a minor optimization. Also make sure to check both the
       * current and the previous increment because the odd and even
       * terms of the sum can have very different behaviour, depending
       * on the value of eta.
       */
      double max_abs_dA = MAX2(abs_dA, A_abs_del_prev);
      double max_abs_dB = MAX2(abs_dB, B_abs_del_prev);
      double abs_A = fabs(A_sum);
      double abs_B = fabs(B_sum);
      if(   max_abs_dA/(max_abs_dA + abs_A) < 4.0*DBL_EPSILON
         && max_abs_dB/(max_abs_dB + abs_B) < 4.0*DBL_EPSILON
         ) break;
    }
    A_abs_del_prev = abs_dA;
    B_abs_del_prev = abs_dB;
    uA_mm2 = uA_mm1;
    uA_mm1 = uA_m;
    uB_mm2 = uB_mm1;
    uB_mm1 = uB_m;
    m++;
  }

  FA = A_sum * ClamA * pow_x * x;
  FB = B_sum * ClamB / pow_x;

  *F = FA;
  *G = (FA * cos_phi_lam - FB)/sin_phi_lam;

	if(m >= max_iter)
	{
		fprintf( ioQQQ, "Argh!! v 3.0\n" );
		puts( "[Stop in coulomb_FG_series (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
		/* GSL_ERROR ("error", GSL_EMAXITER); */
	}
    return;
	/* GSL_ERROR_SELECT_2(stat_A, stat_B); */
}


/* Evaluate the Frobenius series for F_0(eta,x) and G_0(eta,x).
 * See [Bardin et al., CPC 3, 73 (1972), (14)-(17)];
 * note the misprint in (17): nu_0=1 is correct, not nu_0=0.
 */
static void coulomb_FG0_series(const double eta, const double x,
                   double *F, double *G)
{
  const int max_iter = 800;
  const double x2  = x*x;
  const double tex = 2.0*eta*x;
  double C0 = CLeta(0.0, eta);
  double r1pie = gsl_sf_psi_1piy_e(eta);
  double u_mm2 = 0.0;  /* u_0 */
  double u_mm1 = x;    /* u_1 */
  double u_m;
  double v_mm2 = 1.0;                               /* nu_0 */
  double v_mm1 = tex*(2.0*E-1.0+r1pie);   /* nu_1 */
  double v_m;
  double u_sum = u_mm2 + u_mm1;
  double v_sum = v_mm2 + v_mm1;
  double u_abs_del_prev = fabs(u_sum);
  double v_abs_del_prev = fabs(v_sum);
  int m = 2;
  double u_sum_err = 2.0 * DBL_EPSILON * fabs(u_sum);
  double v_sum_err = 2.0 * DBL_EPSILON * fabs(v_sum);
  double ln2x = log(2.0*x);

  while(m < max_iter) {
    double abs_du;
    double abs_dv;
    double m_mm1 = m*(m-1.0);
    u_m = (tex*u_mm1 - x2*u_mm2)/m_mm1;
    v_m = (tex*v_mm1 - x2*v_mm2 - 2.0*eta*(2*m-1)*u_m)/m_mm1;
    u_sum += u_m;
    v_sum += v_m;
    abs_du = fabs(u_m);
    abs_dv = fabs(v_m);
    u_sum_err += 2.0 * DBL_EPSILON * abs_du;
    v_sum_err += 2.0 * DBL_EPSILON * abs_dv;
    if(m > 15) {
      /* Don't bother checking until we have gone out a little ways;
       * a minor optimization. Also make sure to check both the
       * current and the previous increment because the odd and even
       * terms of the sum can have very different behaviour, depending
       * on the value of eta.
       */
      double max_abs_du = MAX2(abs_du, u_abs_del_prev);
      double max_abs_dv = MAX2(abs_dv, v_abs_del_prev);
      double abs_u = fabs(u_sum);
      double abs_v = fabs(v_sum);
      if(   max_abs_du/(max_abs_du + abs_u) < 40.0*DBL_EPSILON
         && max_abs_dv/(max_abs_dv + abs_v) < 40.0*DBL_EPSILON
         ) break;
    }
    u_abs_del_prev = abs_du;
    v_abs_del_prev = abs_dv;
    u_mm2 = u_mm1;
    u_mm1 = u_m;
    v_mm2 = v_mm1;
    v_mm1 = v_m;
    m++;
  }

  *F  = C0 * u_sum;
  *G  = (v_sum + 2.0*eta*u_sum * ln2x) / C0;

	if(m == max_iter)
	{
		fprintf( ioQQQ, "Argh!! v 4.0\n" );
		puts( "[Stop in coulomb_FG0_series (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
		/* GSL_ERROR ("error", GSL_EMAXITER); */
	}
	return;
	/* GSL_ERROR_SELECT_2(psi_stat, stat_CL); */
}


/* Evaluate the Frobenius series for F_{-1/2}(eta,x) and G_{-1/2}(eta,x).
 * Homegrown algebra.
 */
static void coulomb_FGmhalf_series(const double eta, const double x,
                       double *F, double *G)
{
  const int max_iter = 800;
  const double rx  = sqrt(x);
  const double x2  = x*x;
  const double tex = 2.0*eta*x;
  double Cmhalf = CLeta(-0.5, eta);
  double u_mm2 = 1.0;                      /* u_0 */
  double u_mm1 = tex * u_mm2;              /* u_1 */
  double u_m;
  double v_mm2, v_mm1, v_m;
  double f_sum, g_sum;
  double tmp1;
  double rpsi_1pe;
  double rpsi_1p2e;
  int m = 2;

  rpsi_1pe  = gsl_sf_psi_1piy_e(eta);
  rpsi_1p2e = gsl_sf_psi_1piy_e(2.0*eta);

  v_mm2 = 2.0*E - LN_TWO - rpsi_1pe + 2.0*rpsi_1p2e;
  v_mm1 = tex*(v_mm2 - 2.0*u_mm2);

  f_sum = u_mm2 + u_mm1;
  g_sum = v_mm2 + v_mm1;

  while(m < max_iter) {
    double m2 = m*m;
    u_m = (tex*u_mm1 - x2*u_mm2)/m2;
    v_m = (tex*v_mm1 - x2*v_mm2 - 2.0*m*u_m)/m2;
    f_sum += u_m;
    g_sum += v_m;
    if(   f_sum != 0.0
       && g_sum != 0.0
       && (fabs(u_m/f_sum) + fabs(v_m/g_sum) < 10.0*DBL_EPSILON)) break;
    u_mm2 = u_mm1;
    u_mm1 = u_m;
    v_mm2 = v_mm1;
    v_mm1 = v_m;
    m++;
  }
  
  *F = Cmhalf * rx * f_sum;
  
  tmp1 = f_sum*log(x);
  *G = -rx*(tmp1 + g_sum)/Cmhalf;

	if(m == max_iter)
	{
  		fprintf( ioQQQ, "Argh!! v 4.0\n" );
		puts( "[Stop in coulomb_FGmhalf_series (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
		/* GSL_ERROR ("error", GSL_EMAXITER); */
	}

	return;
  /* stat_CL; */
}


/* Evolve the backwards recurrence for F,F'.
 *
 *    F_{lam-1}  = (S_lam F_lam + F_lam') / R_lam
 *    F_{lam-1}' = (S_lam F_{lam-1} - R_lam F_lam)
 * where
 *    R_lam = sqrt(1 + (eta/lam)^2)
 *    S_lam = lam/x + eta/lam
 *
 */
static void coulomb_F_recur(double lam_min, int kmax,
                double eta, double x,
                double F_lam_max, double Fp_lam_max,
                double *F_lam_min, double *Fp_lam_min )
{
  double x_inv = 1.0/x;
  double fcl = F_lam_max;
  double fpl = Fp_lam_max;
  double lam_max = lam_min + kmax;
  double lam = lam_max;
  int k;

  for(k=kmax-1; k>=0; k--) {
    double el = eta/lam;
    double rl = sqrt(1.0 + el*el);
    double sl = el  + lam*x_inv;
    double fc_lm1;
    fc_lm1 = (fcl*sl + fpl)/rl;
    fpl    =  fc_lm1*sl - fcl*rl;
    fcl    =  fc_lm1;
    lam -= 1.0;
  }

  *F_lam_min  = fcl;
  *Fp_lam_min = fpl;  
  return;
  /*GSL_SUCCESS;*/
}


/* Evolve the forward recurrence for G,G'.
 *
 *   G_{lam+1}  = (S_lam G_lam - G_lam')/R_lam
 *   G_{lam+1}' = R_{lam+1} G_lam - S_lam G_{lam+1}
 *
 * where S_lam and R_lam are as above in the F recursion.
 */
static void coulomb_G_recur(const double lam_min, const int kmax,
                const double eta, const double x,
                const double G_lam_min, const double Gp_lam_min,
                double *G_lam_max, double *Gp_lam_max )
{
  double x_inv = 1.0/x;
  double gcl = G_lam_min;
  double gpl = Gp_lam_min;
  double lam = lam_min + 1.0;
  int k;

  for(k=1; k<=kmax; k++) {
    double el = eta/lam;
    double rl = sqrt(1.0 + el*el);
    double sl = el + lam*x_inv;
    double gcl1 = (sl*gcl - gpl)/rl;
    gpl   = rl*gcl - sl*gcl1;
    gcl   = gcl1;
    lam += 1.0;
  }
  
  *G_lam_max  = gcl;
  *Gp_lam_max = gpl;
  return;
  /* GSL_SUCCESS; */
}


/* Evaluate the first continued fraction, giving
 * the ratio F'/F at the upper lambda value.
 * We also determine the sign of F at that point,
 * since it is the sign of the last denominator
 * in the continued fraction.
 */
static void coulomb_CF1(double lambda,
            double eta, double x,
            double * fcl_sign,
            double * result,
            int * count
            )
{
  const double CF1_small = 1.e-30;
  const double CF1_abort = 1.0e+05;
  const double CF1_acc   = 2.0*DBL_EPSILON;
  const double x_inv     = 1.0/x;
  const double px        = lambda + 1.0 + CF1_abort;

  double pk = lambda + 1.0;
  double F  = eta/pk + pk*x_inv;
  double D, C;
  double df;

  *fcl_sign = 1.0;
  *count = 0;

  if(fabs(F) < CF1_small) F = CF1_small;
  D = 0.0;
  C = F;

  do {
    double pk1 = pk + 1.0;
    double ek  = eta / pk;
    double rk2 = 1.0 + ek*ek;
    double tk  = (pk + pk1)*(x_inv + ek/pk1);
    D   =  tk - rk2 * D;
    C   =  tk - rk2 / C;
    if(fabs(C) < CF1_small) C = CF1_small;
    if(fabs(D) < CF1_small) D = CF1_small;
    D = 1.0/D;
    df = D * C;
    F  = F * df;
    if(D < 0.0) {
      /* sign of result depends on sign of denominator */
      *fcl_sign = - *fcl_sign;
    }
    pk = pk1;
    if( pk > px )
	{

		/*
		*result = F;
		fprintf( ioQQQ, "Argh!! v 5.0\n");
		puts( "[Stop in coulomb_CF1 (thirdparty.c)]" );
		cdEXIT( EXIT_FAILURE );
		*/
		/* GSL_ERROR ("error", GSL_ERUNAWAY); */
		break;
    }
    ++(*count);
  }
  while(fabs(df-1.0) > CF1_acc);
  
  *result = F;
  return;
  /* GSL_SUCCESS; */
}

/* Evaluate the second continued fraction to 
 * obtain the ratio
 *    (G' + i F')/(G + i F) := P + i Q
 * at the specified lambda value.
 */
static void coulomb_CF2(const double lambda, const double eta, const double x,
            double *result_P, double *result_Q, int *count )
{
 /*   int status = GSL_SUCCESS; */

  const double CF2_acc   = 4.0*DBL_EPSILON;
  const double CF2_abort = 2.0e+05;

  const double wi    = 2.0*eta;
  const double x_inv = 1.0/x;
  const double e2mm1 = eta*eta + lambda*(lambda + 1.0);
  
  double ar = -e2mm1;
  double ai =  eta;

  double br =  2.0*(x - eta);
  double bi =  2.0;

  double dr =  br/(br*br + bi*bi);
  double di = -bi/(br*br + bi*bi);

  double dp = -x_inv*(ar*di + ai*dr);
  double dq =  x_inv*(ar*dr - ai*di);

  double A, B, C, D;

  double pk =  0.0;
  double P  =  0.0;
  double Q  =  1.0 - eta*x_inv;

  *count = 0;
 
  do {
    P += dp;
    Q += dq;
    pk += 2.0;
    ar += pk;
    ai += wi;
    bi += 2.0;
    D  = ar*dr - ai*di + br;
    di = ai*dr + ar*di + bi;
    C  = 1.0/(D*D + di*di);
    dr =  C*D;
    di = -C*di;
    A  = br*dr - bi*di - 1.;
    B  = bi*dr + br*di;
    C  = dp*A  - dq*B;
    dq = dp*B  + dq*A;
    dp = C;
    if(pk > CF2_abort)
	{
		/*
		fprintf( ioQQQ, "Argh!! v 6.0\n");
		puts( "[Stop in coulomb_CF2 (thirdparty.c)]" );
		cdEXIT( EXIT_FAILURE );
		*/
		/* status = GSL_ERUNAWAY; */
		break;
    }
    ++(*count);
  }
  while(fabs(dp)+fabs(dq) > (fabs(P)+fabs(Q))*CF2_acc);

  if(Q < CF2_abort*DBL_EPSILON*fabs(P))
  {
		fprintf( ioQQQ, "Argh!! v 7.0\n");
		puts( "[Stop in coulomb_CF2 (thirdparty.c)]" );
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	/* status = GSL_ELOSS; */
  }

  *result_P = P;
  *result_Q = Q;
  return;
  /* status; */
}

#if	0
static void coulomb_jwkb(const double lam, const double eta, const double x,
             double *fjwkb, double *gjwkb,
             double *exponent)
{
  const double llp1      = lam*(lam+1.0) + 6.0/35.0;
  const double llp1_eff  = MAX2(llp1, 0.0);
  const double rho_ghalf = sqrt(x*(2.0*eta - x) + llp1_eff);
  const double sinh_arg  = sqrt(llp1_eff/(eta*eta+llp1_eff)) * rho_ghalf / x;
  const double sinh_inv  = log(sinh_arg + sqrt(1.0 + sinh_arg*sinh_arg));

  const double phi = fabs(rho_ghalf - eta*atan2(rho_ghalf,x-eta) - sqrt(llp1_eff) * sinh_inv);

  const double zeta_half = pow(3.0*phi/2.0, 1.0/3.0);
  const double prefactor = sqrt(PI*phi*x/(6.0 * rho_ghalf));
  
  double F = prefactor * 3.0/zeta_half;
  double G = prefactor * 3.0/zeta_half; /* Note the sqrt(3) from Bi normalization */
  double F_exp;
  double G_exp;
  
  const double airy_scale_exp = phi;
  double ai;
  double bi;
  gsl_sf_airy_Ai_scaled_e(zeta_half*zeta_half, GSL_MODE_DEFAULT, &ai);
  gsl_sf_airy_Bi_scaled_e(zeta_half*zeta_half, GSL_MODE_DEFAULT, &bi);
  F *= ai;
  G *= bi;
  F_exp = log(F) - airy_scale_exp;
  G_exp = log(G) + airy_scale_exp;

  if(G_exp >= log( DBL_MAX ) ) {
    *fjwkb = F;
    *gjwkb = G;
    *exponent = airy_scale_exp;
	fprintf( ioQQQ, "Argh!! v 8.0");
	puts( "[Stop in coulomb_jwkb (thirdparty.c)]" );
	ASSERT( 1==0 );
	cdEXIT( EXIT_FAILURE );
    /* GSL_ERROR ("error", GSL_EOVRFLW); */
  }
  else {
    *fjwkb = exp(F_exp);
    *gjwkb = exp(G_exp);
    *exponent = 0.0;
    return;
	/* GSL_SUCCESS; */
  }
}
#endif

static double gsl_sf_psi_1piy_e(const double y)
{
  const double ay = fabs(y);
  double result;
  
  /* CHECK_POINTER(result) */

  if(ay > 1000.0) {
    /* [Abramowitz+Stegun, 6.3.19] */
    const double yi2 = 1.0/(ay*ay);
    const double lny = log(ay);
    const double sum = yi2 * (1.0/12.0 + 1.0/120.0 * yi2 + 1.0/252.0 * yi2*yi2);
    result = lny + sum;
    return result;
	/* GSL_SUCCESS; */
  }
  else if(ay > 10.0) {
    /* [Abramowitz+Stegun, 6.3.19] */
    const double yi2 = 1.0/(ay*ay);
    const double lny = log(ay);
    const double sum = yi2 * (1.0/12.0 +
                         yi2 * (1.0/120.0 +
                           yi2 * (1.0/252.0 +
                             yi2 * (1.0/240.0 +
                               yi2 * (1.0/132.0 + 691.0/32760.0 * yi2)))));
    result = lny + sum;
    return result;
	/* GSL_SUCCESS; */
  }
  else if(ay > 1.0){
    const double y2 = ay*ay;
    const double x  = (2.0*ay - 11.0)/9.0;
    const double v  = y2*(1.0/(1.0+y2) + 0.5/(4.0+y2));
    double result_c;
    result_c = cheb_eval_e(&r1py_cs, x);
    result  = result_c - E + v;
    return result;
	/* GSL_SUCCESS; */
  }
  else {
    /* [Abramowitz+Stegun, 6.3.17]
     *
     * -M_EULER + y^2 Sum[1/n 1/(n^2 + y^2), {n,1,M}]
     *   +     Sum[1/n^3, {n,M+1,Infinity}]
     *   - y^2 Sum[1/n^5, {n,M+1,Infinity}]
     *   + y^4 Sum[1/n^7, {n,M+1,Infinity}]
     *   - y^6 Sum[1/n^9, {n,M+1,Infinity}]
     *   + O(y^8)
     *
     * We take M=50 for at least 15 digit precision.
     */
    const int M = 50;
    const double y2 = y*y;
    const double c0 = 0.00019603999466879846570;
    const double c2 = 3.8426659205114376860e-08;
    const double c4 = 1.0041592839497643554e-11;
    const double c6 = 2.9516743763500191289e-15;
    const double p  = c0 + y2 *(-c2 + y2*(c4 - y2*c6));
    double sum = 0.0;
    double v;
    
    int n;
    for(n=1; n<=M; n++) {
      sum += 1.0/(n * (n*n + y*y));
    }

    v = y2 * (sum + p);
    result = -E + v;
    return result;
	/* GSL_SUCCESS; */
  }
}

static double cheb_eval_e(const cheb_series *cs, const double x )
{
  int j;
  double d  = 0.0;
  double dd = 0.0;
  double result;

  double y  = (2.0*x - cs->a - cs->b) / (cs->b - cs->a);
  double y2 = 2.0 * y;

  double e = 0.0;

  for(j = cs->order; j>=1; j--) {
    double temp = d;
    d = y2*d - dd + cs->c[j];
    e += fabs(y2*temp) + fabs(dd) + fabs(cs->c[j]);
    dd = temp;
  }

  { 
    double temp = d;
    d = y*d - dd + 0.5 * cs->c[0];
    e += fabs(y*temp) + fabs(dd) + 0.5 * fabs(cs->c[0]);
  }

  result = d;
  return result;
  /* GSL_SUCCESS; */
}

/* Bessel stuff starts here */

double gsl_sf_bessel_Jn_e(int n, double x)
{
	double result;
  int sign = 1;

  if(n < 0) {
    /* reduce to case n >= 0 */
    n = -n;
    if(IS_ODD(n)) sign = -sign;
  }  

  if(x < 0.0) {
    /* reduce to case x >= 0. */
    x = -x;
    if(IS_ODD(n)) sign = -sign;
  }

  /* CHECK_POINTER(result) */

  if(n == 0) {
    double b0 = gsl_sf_bessel_J0_e(x);
    result = sign * b0;
    return result;
  }
  else if(n == 1) {
    double b1 = gsl_sf_bessel_J1_e(x);
    result = sign * b1;
    return result;
  }
  else {
    if(x == 0.0) {
      result = 0.0;
      return result;
    }
    else if(x*x < 10.0*(n+1.0)* pow(DBL_EPSILON,0.2) ) {
      double b = gsl_sf_bessel_IJ_taylor_e((double)n, x, -1, 50, DBL_EPSILON);
      result = sign * b;
      return result;
    }
    else if( pow(DBL_EPSILON,0.25) * x > (n*n+1.0)) {
      result = gsl_sf_bessel_Jnu_asympx_e((double)n, x);
      return result;
    }
    else if(x > 1000.0)
    {
      /* We need this to avoid feeding large x to CF1; note that
       * due to the above check, we know that n <= 50.
       */
      result = gsl_sf_bessel_Jnu_asympx_e((double)n, x);
      return result;
    }

    else {
      double ans;
      double ratio;
      double sgn = gsl_sf_bessel_J_CF1((double)n, x, &ratio);

      /* backward recurrence */
      double Jkp1 = sqrt( DBL_MIN ) * ratio;
      double Jk   = sqrt( DBL_MIN );
      double Jkm1;
      int k;

      for(k=n; k>0; k--) {
        Jkm1 = 2.0*k/x * Jk - Jkp1;
        Jkp1 = Jk;
        Jk   = Jkm1;
      }

      if(fabs(Jkp1) > fabs(Jk)) {
        double b1 = gsl_sf_bessel_J1_e(x);
        ans = b1/Jkp1 * sqrt( DBL_MIN );
      }
      else {
        double b0 = gsl_sf_bessel_J0_e(x);
        ans = b0/Jk * sqrt( DBL_MIN );
      }

      result = sgn * ans;
      return result;
    }
  }
}

static double bj0_data[13] = {
   0.100254161968939137, 
  -0.665223007764405132, 
   0.248983703498281314, 
  -0.0332527231700357697,
   0.0023114179304694015,
  -0.0000991127741995080,
   0.0000028916708643998,
  -0.0000000612108586630,
   0.0000000009838650793,
  -0.0000000000124235515,
   0.0000000000001265433,
  -0.0000000000000010619,
   0.0000000000000000074,
};
static cheb_series bj0_cs = {
  bj0_data,
  12,
  -1, 1,
  9
};


/*-*-*-*-*-*-*-*-*-*-*-* Functions with Error Codes *-*-*-*-*-*-*-*-*-*-*-*/

double gsl_sf_bessel_J0_e(const double x)
{
	double result;
  double y = fabs(x);

  /* CHECK_POINTER(result) */

  if(y < 2.0* sqrt(DBL_EPSILON) ) {
    result = 1.0;
    return result;
  }
  else if(y <= 4.0) {
    return cheb_eval_e(&bj0_cs, 0.125*y*y - 1.0);
  }
  else {
    const double z = 32.0/(y*y) - 1.0;
    double ca = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bm0_cs,  z);
    double ct = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bth0_cs, z);
    double cp = gsl_sf_bessel_cos_pi4_e(y, ct/y);
    const double sqrty = sqrt(y);
    const double ampl  = (0.75 + ca) / sqrty;
    result = ampl * cp;
    return result;
  }
}

static double bj1_data[12] = {
  -0.11726141513332787,
  -0.25361521830790640,
   0.050127080984469569,
  -0.004631514809625081,
   0.000247996229415914,
  -0.000008678948686278,
   0.000000214293917143,
  -0.000000003936093079,
   0.000000000055911823,
  -0.000000000000632761,
   0.000000000000005840,
  -0.000000000000000044,
};
static cheb_series bj1_cs = {
  bj1_data,
  11,
  -1, 1,
  8
};


/*-*-*-*-*-*-*-*-*-*-*-* Functions with Error Codes *-*-*-*-*-*-*-*-*-*-*-*/

double gsl_sf_bessel_J1_e(const double x)
{
  double y = fabs(x);
  double result;

  /* CHECK_POINTER(result) */

  if(y == 0.0) {
    result = 0.0;
    return result;
  }
  else if(y < 2.0*DBL_MIN) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
    /* UNDERFLOW_ERROR(result); */
  }
  else if(y < sqrt(8.) * sqrt( DBL_EPSILON ) ) {
    result = 0.5*x;
    return result;
  }
  else if(y < 4.0) {
    double c = cheb_eval_e(&bj1_cs, 0.125*y*y-1.0);
    result = x * (0.25 + c);
    return result;
  }
  else {
    /* Because the leading term in the phase is y,
     * which we assume is exactly known, the error
     * in the cos() evaluation is bounded.
     */
    const double z  = 32.0/(y*y) - 1.0;
    double ca = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bm1_cs,  z);
    double ct = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bth1_cs, z);
    double sp = gsl_sf_bessel_sin_pi4_e(y, ct/y);
    const double sqrty = sqrt(y);
    const double ampl  = (0.75 + ca) / sqrty;
    result = (x < 0.0 ? -ampl : ampl) * sp;
    return result;
  }
}

double gsl_sf_bessel_Yn_e(int n, const double x)
{
	double result;
  int sign = 1;

  if(n < 0) {
    /* reduce to case n >= 0 */
    n = -n;
    if(IS_ODD(n)) sign = -1;
  }

  /* CHECK_POINTER(result) */

  if(n == 0) {
    result = gsl_sf_bessel_Y0_e(x);
    result *= sign;
    return result;
  }
  else if(n == 1) {
    result = gsl_sf_bessel_Y1_e(x);
    result *= sign;
    return result;
  }
  else {
    if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
		/*       DOMAIN_ERROR(result); */
    }
    if(x < 5.0) {
      result = bessel_Yn_small_x(n, x);
      result *= sign;
      return result;
    }
    else if( pow(DBL_EPSILON, 0.3333) * x > (n*n + 1.0)) {
      result = gsl_sf_bessel_Ynu_asympx_e((double)n, x);
      result *= sign;
      return result;
    }
	/*
    else if(n > 50) {
      result = gsl_sf_bessel_Ynu_asymp_Olver_e((double)n, x);
      result *= sign;
      return result;
    }
	*/
    else {
      double two_over_x = 2.0/x;
      double r_by = gsl_sf_bessel_Y1_e(x);
      double r_bym = gsl_sf_bessel_Y0_e(x);
      double bym = r_bym;
      double by  = r_by;
      double byp;
      int j;

      for(j=1; j<n; j++) { 
        byp = j*two_over_x*by - bym;
        bym = by;
        by  = byp;
      }
      result  = sign * by;
      return result;
    }
  }
}

static double by0_data[13] = {
  -0.011277839392865573,
  -0.128345237560420350,
  -0.104378847997942490,
   0.023662749183969695,
  -0.002090391647700486,
   0.000103975453939057,
  -0.000003369747162423,
   0.000000077293842676,
  -0.000000001324976772,
   0.000000000017648232,
  -0.000000000000188105,
   0.000000000000001641,
  -0.000000000000000011
};
static cheb_series by0_cs = {
  by0_data,
  12,
  -1, 1,
  8
};


/*-*-*-*-*-*-*-*-*-*-*-* Functions with Error Codes *-*-*-*-*-*-*-*-*-*-*-*/

double gsl_sf_bessel_Y0_e(const double x)
{
  const double two_over_pi = 2.0/PI;
  const double xmax        = 1.0/DBL_EPSILON;
  double result;

  /* CHECK_POINTER(result) */

  if (x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*     DOMAIN_ERROR(result); */
  }
  else if(x < 4.0) {
    double J0 = gsl_sf_bessel_J0_e(x);
    double c = cheb_eval_e(&by0_cs, 0.125*x*x-1.0);
    result = two_over_pi*(-LN_TWO + log(x))*J0 + 0.375 + c;
    return result;
  }
  else if(x < xmax) {
    /* Leading behaviour of phase is x, which is exact,
     * so the error is bounded.
     */
    const double z  = 32.0/(x*x) - 1.0;
    double c1 = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bm0_cs,  z);
    double c2 = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bth0_cs, z);
    double sp = gsl_sf_bessel_sin_pi4_e(x, c2/x);
    const double sqrtx = sqrt(x);
    const double ampl  = (0.75 + c1) / sqrtx;
    result  = ampl * sp;
    return result;
  }
  else {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*     DOMAIN_ERROR(result); */
  }
}

static double by1_data[14] = {
  0.03208047100611908629,
  1.262707897433500450,
  0.00649996189992317500,
 -0.08936164528860504117,
  0.01325088122175709545,
 -0.00089790591196483523,
  0.00003647361487958306,
 -0.00000100137438166600,
  0.00000001994539657390,
 -0.00000000030230656018,
  0.00000000000360987815,
 -0.00000000000003487488,
  0.00000000000000027838,
 -0.00000000000000000186
};
static cheb_series by1_cs = {
  by1_data,
  13,
  -1, 1,
  10
};


/*-*-*-*-*-*-*-*-*-*-*-* Functions with Error Codes *-*-*-*-*-*-*-*-*-*-*-*/

double gsl_sf_bessel_Y1_e(const double x)
{
  const double two_over_pi = 2.0/PI;
  const double xmin = 1.571*DBL_MIN; /*exp ( amax1(alog(r1mach(1)), -alog(r1mach(2)))+.01) */
  const double x_small = 2.0 * sqrt( DBL_EPSILON );
  const double xmax    = 1.0/DBL_EPSILON;
  double result;

  /* CHECK_POINTER(result) */

  if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*     DOMAIN_ERROR(result); */
  }
  else if(x < xmin) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*  OVERFLOW_ERROR(result); */
  }
  else if(x < x_small) {
    const double lnterm = log(0.5*x);
    double J1 = gsl_sf_bessel_J1_e(x);
    double c = cheb_eval_e(&by1_cs, -1.0);
    result = two_over_pi * lnterm * J1 + (0.5 + c)/x;
    return result;
  }
  else if(x < 4.0) {
    const double lnterm = log(0.5*x);
    double J1 = gsl_sf_bessel_J1_e(x);
    double c = cheb_eval_e(&by1_cs, 0.125*x*x-1.0);
    result = two_over_pi * lnterm * J1 + (0.5 + c)/x;
    return result;
  }
  else if(x < xmax) {
    const double z = 32.0/(x*x) - 1.0;
    double ca = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bm1_cs,  z);
    double ct = cheb_eval_e(&_gsl_sf_bessel_amp_phase_bth1_cs, z);
    double cp = gsl_sf_bessel_cos_pi4_e(x, ct/x);
    const double sqrtx = sqrt(x);
    const double ampl  = (0.75 + ca) / sqrtx;
    result = -ampl * cp;
    return result;
  }
  else {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*  UNDERFLOW_ERROR(result); */
  }
}

/* asymptotic Bessel solutions */


/* x >> nu*nu+1
 * error ~ O( ((nu*nu+1)/x)^4 )
 *
 * empirical error analysis:
 *   choose  GSL_ROOT4_MACH_EPS * x > (nu*nu + 1)
 *
 * This is not especially useful. When the argument gets
 * large enough for this to apply, the cos() and sin()
 * start loosing digits. However, this seems inevitable
 * for this particular method.
 *
 * Wed Jun 25 14:39:38 MDT 2003 [GJ]
 * This function was inconsistent since the Q term did not
 * go to relative order eps^2. That's why the error estimate
 * originally given was screwy (it didn't make sense that the
 * "empirical" error was coming out O(eps^3)).
 * With Q to proper order, the error is O(eps^4).
 */
double gsl_sf_bessel_Jnu_asympx_e(const double nu, const double x)
{
  double mu   = 4.0*nu*nu;
  double mum1 = mu-1.0;
  double mum9 = mu-9.0;
  double mum25 = mu-25.0;
  double chi = x - (0.5*nu + 0.25)*PI;
  double P   = 1.0 - mum1*mum9/(128.0*x*x);
  double Q   = mum1/(8.0*x) * (1.0 - mum9*mum25/(384.0*x*x));
  double pre = sqrt(2.0/(PI*x));
  double c   = cos(chi);
  double s   = sin(chi);
  double result = pre * (c*P - s*Q);
  return result;
}

/* x >> nu*nu+1
 */
double gsl_sf_bessel_Ynu_asympx_e(const double nu, const double x)
{
  double ampl = gsl_sf_bessel_asymp_Mnu_e(nu, x);
  double theta = gsl_sf_bessel_asymp_thetanu_corr_e(nu, x);
  double alpha = x;
  double beta  = -0.5*nu*PI;
  double sin_alpha = sin(alpha);
  double cos_alpha = cos(alpha);
  double sin_chi   = sin(beta + theta);
  double cos_chi   = cos(beta + theta);
  double sin_term     = sin_alpha * cos_chi + sin_chi * cos_alpha;
  /* double sin_term_mag = fabs(sin_alpha * cos_chi) + fabs(sin_chi * cos_alpha); */
  double result  = ampl * sin_term;
  return result;
}

double gsl_sf_bessel_asymp_Mnu_e(const double nu, const double x)
{
  const double r  = 2.0*nu/x;
  const double r2 = r*r;
  const double x2 = x*x;
  const double term1 = (r2-1.0/x2)/8.0;
  const double term2 = (r2-1.0/x2)*(r2-9.0/x2)*3.0/128.0;
  const double Mnu2_c = 2.0/(PI) * (1.0 + term1 + term2);
  double result = sqrt(Mnu2_c)/sqrt(x); /* will never underflow this way */
  return result;
}

double gsl_sf_bessel_asymp_thetanu_corr_e(const double nu, const double x)
{
  const double r  = 2.0*nu/x;
  const double r2 = r*r;
  const double x2 = x*x;
  const double term1 = x*(r2 - 1.0/x2)/8.0;
  const double term2 = x*(r2 - 1.0/x2)*(r2 - 25.0/x2)/384.0;
  double result = (-0.25*PI + term1 + term2);
  return result;
}


double gsl_sf_bessel_cos_pi4_e(double y, double eps)
{
  const double sy = sin(y);
  const double cy = cos(y);
  const double s = sy + cy;
  const double d = sy - cy;
  double seps;
  double ceps;
  double result;
  if(fabs(eps) < pow(DBL_EPSILON,0.2) ) {
    const double e2 = eps*eps;
    seps = eps * (1.0 - e2/6.0 * (1.0 - e2/20.0));
    ceps = 1.0 - e2/2.0 * (1.0 - e2/12.0);
  }
  else {
    seps = sin(eps);
    ceps = cos(eps);
  }
  result = (ceps * s - seps * d)/ sqrt(2.);
  return result;
}

double gsl_sf_bessel_sin_pi4_e(double y, double eps)
{
  const double sy = sin(y);
  const double cy = cos(y);
  const double s = sy + cy;
  const double d = sy - cy;
  double seps;
  double ceps;
  double result;
  if(fabs(eps) < pow(DBL_EPSILON,0.2) ) {
    const double e2 = eps*eps;
    seps = eps * (1.0 - e2/6.0 * (1.0 - e2/20.0));
    ceps = 1.0 - e2/2.0 * (1.0 - e2/12.0);
  }
  else {
    seps = sin(eps);
    ceps = cos(eps);
  }
  result = (ceps * d + seps * s)/ sqrt(2.);
  return result;
}

double gsl_sf_bessel_J_CF1(const double nu, const double x, double *ratio)
{
  const double RECUR_BIG = sqrt(DBL_MAX);
  const int maxiter = 10000;
  int n = 1;
  double Anm2 = 1.0;
  double Bnm2 = 0.0;
  double Anm1 = 0.0;
  double Bnm1 = 1.0;
  double a1 = x/(2.0*(nu+1.0));
  double An = Anm1 + a1*Anm2;
  double Bn = Bnm1 + a1*Bnm2;
  double an;
  double fn = An/Bn;
  double dn = a1;
  double s  = 1.0;
  double sgn;

  while(n < maxiter) {
    double old_fn;
    double del;
    n++;
    Anm2 = Anm1;
    Bnm2 = Bnm1;
    Anm1 = An;
    Bnm1 = Bn;
    an = -x*x/(4.0*(nu+n-1.0)*(nu+n));
    An = Anm1 + an*Anm2;
    Bn = Bnm1 + an*Bnm2;

    if(fabs(An) > RECUR_BIG || fabs(Bn) > RECUR_BIG) {
      An /= RECUR_BIG;
      Bn /= RECUR_BIG;
      Anm1 /= RECUR_BIG;
      Bnm1 /= RECUR_BIG;
      Anm2 /= RECUR_BIG;
      Bnm2 /= RECUR_BIG;
    }

    old_fn = fn;
    fn = An/Bn;
    del = old_fn/fn;

    dn = 1.0 / (2.0*(nu+n)/x - dn);
    if(dn < 0.0) s = -s;

    if(fabs(del - 1.0) < 2.0*DBL_EPSILON) break;
  }

  *ratio = fn;
  sgn   = s;

	return s;
}

double gsl_sf_bessel_JY_steed_CF2(const double nu, const double x, double *P)
{
  const int max_iter = 10000;
  const double SMALL = 1.0e-100;
  double Q;

  int i = 1;

  double x_inv = 1.0/x;
  double a = 0.25 - nu*nu;
  double p = -0.5*x_inv;
  double q = 1.0;
  double br = 2.0*x;
  double bi = 2.0;
  double fact = a*x_inv/(p*p + q*q);
  double cr = br + q*fact;
  double ci = bi + p*fact;
  double den = br*br + bi*bi;
  double dr = br/den;
  double di = -bi/den;
  double dlr = cr*dr - ci*di;
  double dli = cr*di + ci*dr;
  double temp = p*dlr - q*dli;
  q = p*dli + q*dlr;
  p = temp;
  for (i=2; i<=max_iter; i++) {
    a  += 2*(i-1);
    bi += 2.0;
    dr = a*dr + br;
    di = a*di + bi;
    if(fabs(dr)+fabs(di) < SMALL) dr = SMALL;
    fact = a/(cr*cr+ci*ci);
    cr = br + cr*fact;
    ci = bi - ci*fact;
    if(fabs(cr)+fabs(ci) < SMALL) cr = SMALL;
    den = dr*dr + di*di;
    dr /= den;
    di /= -den;
    dlr = cr*dr - ci*di;
    dli = cr*di + ci*dr;
    temp = p*dlr - q*dli;
    q = p*dli + q*dlr;
    p = temp;
    if(fabs(dlr-1.0)+fabs(dli) < DBL_EPSILON) break;
  }

  *P = p;
  Q = q;

    return Q;
}

double gsl_sf_bessel_IJ_taylor_e(const double nu, const double x, const int sign,
                             const int kmax, const double threshold )
{
  /* CHECK_POINTER(result) */
	double result = 0.;

	if(nu < 0.0 || x < 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* DOMAIN_ERROR(result); */
  }
  else if(x == 0.0) {
    if(nu == 0.0) {
      result = 1.0;
    }
    else {
      result = 0.0;
    }
    return result;
  }
  else {
    double prefactor;   /* (x/2)^nu / Gamma(nu+1) */
    double sum;

    if(nu == 0.0) {
      prefactor = 1.0;
    }
    else
	{

		ASSERT( nu < INT_MAX-1 );
      /* Separate the integer part and use
       * y^nu / Gamma(nu+1) = y^N /N! y^f / (N+1)_f,
       * to control the error.
       */
		/*
      const int    N = (int)floor(nu + 0.5);
      const double f = nu - N;
      double poch_factor = gsl_sf_poch_e(N+1.0, f);
      double tc_factor   = gsl_sf_taylorcoeff_e(N, 0.5*x);
      const double p = pow(0.5*x,f);
      prefactor  = tc_factor * p / poch_factor;
	*/
	  prefactor = pow( 0.5*x, nu ) / gammafun( nu + 1. );
    }


    /* Evaluate the sum.
     * [Abramowitz+Stegun, 9.1.10]
     * [Abramowitz+Stegun, 9.6.7]
     */
    {
      const double y = sign * 0.25 * x*x;
      double sumk = 1.0;
      double term = 1.0;
      int k;

      for(k=1; k<=kmax; k++) {
        term *= y/((nu+k)*k);
        sumk += term;
        if(fabs(term/sumk) < threshold) break;
      }

      sum = sumk;
    }

	result = prefactor * sum;

    return result;
  }

}

/* assumes n >= 1 */
static double bessel_Yn_small_x(const int n, const double x)
{
  double result;
	int k;
  double y = 0.25 * x * x;
  double ln_x_2 = log(0.5*x);
  double ln_nm1_fact;
  double k_term;
  double term1, sum1, ln_pre1;
  double term2, sum2, pre2;
  
  ln_nm1_fact = gsl_sf_lnfact_e((unsigned int)(n-1));

  ln_pre1 = -n*ln_x_2 + ln_nm1_fact;
  if(ln_pre1 > log10( DBL_MAX ) - 3.0)
  {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* GSL_ERROR ("error", GSL_EOVRFLW); */
  }

  sum1 = 1.0;
  k_term = 1.0;
  for(k=1; k<=n-1; k++) {
    k_term *= y/(k * (n-k));
    sum1 += k_term;
  }
  term1 = -exp(ln_pre1) * sum1 / PI;
  
  pre2 = -exp(n*ln_x_2) / PI;
  if(fabs(pre2) > 0.0) {
    const int KMAX = 20;
    double psi_n = gsl_sf_psi_int_e(n);
    double npk_fact = gsl_sf_fact_e((unsigned int)n);
    double yk = 1.0;
    double k_fact  = 1.0;
    double psi_kp1 = -E;
    double psi_npkp1;
    
    psi_npkp1 = psi_n + 1.0/n;
    sum2 = (psi_kp1 + psi_npkp1 - 2.0*ln_x_2)/npk_fact;
    for(k=1; k<KMAX; k++) {
      psi_kp1   += 1./k;
      psi_npkp1 += 1./(n+k);
      k_fact   *= k;
      npk_fact *= n+k;
      yk *= -y;
      k_term = yk*(psi_kp1 + psi_npkp1 - 2.0*ln_x_2)/(k_fact*npk_fact);
      sum2 += k_term;
    }
    term2 = pre2 * sum2;
  }
  else {
    term2 = 0.0;
  }

  result  = term1 + term2;

  return result;
}

double gsl_sf_psi_int_e(const int n)
{
  /* CHECK_POINTER(result) */
	double result;

  if(n <= 0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /*     DOMAIN_ERROR(result); */
  }
  else if(n <= PSI_TABLE_NMAX) {
    result = psi_table[n];
    return result;
  }
  else {
    /* Abramowitz+Stegun 6.3.18 */
    const double c2 = -1.0/12.0;
    const double c3 =  1.0/120.0;
    const double c4 = -1.0/252.0;
    const double c5 =  1.0/240.0;
    const double ni2 = (1.0/n)*(1.0/n);
    const double ser = ni2 * (c2 + ni2 * (c3 + ni2 * (c4 + ni2*c5)));
    result  = log((double)n) - 0.5/n + ser;
    return result;
  }
}

double gsl_sf_fact_e(const unsigned int n)
{
  /* CHECK_POINTER(result) */
	double result;

  if(n < 18) {
    result = fact_table[n].f;
    return result;
  }
  else if(n <= FACT_TABLE_MAX){
    result = fact_table[n].f;
    return result;
  }
  else {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* OVERFLOW_ERROR(result); */
  }
}

double gsl_sf_lnfact_e(const unsigned int n)
{
  /* CHECK_POINTER(result) */
	double result;

  if(n <= FACT_TABLE_MAX){
    result = log(fact_table[n].f);
    return result;
  }
  else {
    result = log( gammafun( n+1.0 ) );
    return result;
  }
}

/* This gets bad near the negative half axis. However, this
 * region can be avoided by use of the reflection formula, as usual.
 * Only the first two terms of the series are kept.
 */
static void lngamma_complex_stirling(complex z,
									   double *lg_r,
									   double *arg)
{
  double re_zinv,  im_zinv;
  double re_zinv2, im_zinv2;
  double re_zinv3, im_zinv3;
  double re_zhlnz, im_zhlnz;
  double r, lnr, theta, zr, zi;
  /*
  gsl_sf_complex_log_e(zr, zi, &lnr, &theta);
  r = exp(lnr);
  */
  zr = z.re;
  zi = z.im;
  
  if( zr==0 )
  {
	  theta = PI/2;
  }
  else
  {
	  theta = atan( zi/zr );
  }
  
  if( zr<0 && zi>=0 )
  {
	  theta = theta + PI;
  }
  else if( zr<0 && zi<=0 )
  {
	  theta = theta + PI;
  }

  r = cmabs( z );
  lnr = log( r );
  re_zinv =  (zr/r)/r;
  im_zinv = -(zi/r)/r;
  re_zinv2 = re_zinv*re_zinv - im_zinv*im_zinv;
  im_zinv2 = 2.0*re_zinv*im_zinv;
  re_zinv3 = re_zinv2*re_zinv - im_zinv2*im_zinv;
  im_zinv3 = re_zinv2*im_zinv + im_zinv2*re_zinv;
  re_zhlnz = (zr - 0.5)*lnr - zi*theta;
  im_zhlnz = zi*lnr + zr*theta;
  *lg_r = re_zhlnz - zr + 0.5*(LN_TWO+log(PI)) + re_zinv/12.0 - re_zinv3/360.0;
  *arg  = im_zhlnz - zi + 1.0/12.0*im_zinv - im_zinv3/360.0;
  return;
}


double gsl_sf_bessel_K0_scaled_e(const double x)
{
  /* CHECK_POINTER(result) */
	double result;

  if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
    /* DOMAIN_ERROR(result); */
  }
  else if(x <= 2.0) {
    const double lx = log(x);
    const double ex = exp(x);
    double I0 = gsl_sf_bessel_I0_e(x);
    double c = cheb_eval_e(&bk0_cs, 0.5*x*x-1.0);
    result = ex * ((-lx+LN_TWO)*I0 - 0.25 + c);
    return result;
  }
  else if(x <= 8.0) {
    const double sx = sqrt(x);
    double c = cheb_eval_e(&ak0_cs, (16.0/x-5.0)/3.0);
    result = (1.25 + c) / sx;
    return result;
  }
  else {
    const double sx = sqrt(x);
    double c = cheb_eval_e(&ak02_cs, 16.0/x-1.0);
    result = (1.25 + c) / sx;
    return result;
  } 
}

double gsl_sf_bessel_K0_e(const double x)
{
  /* CHECK_POINTER(result) */
	double result;

  if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* DOMAIN_ERROR(result); */
  }
  else if(x <= 2.0) {
    const double lx = log(x);
    double I0 = gsl_sf_bessel_I0_e(x);
    double c = cheb_eval_e(&bk0_cs, 0.5*x*x-1.0);
    result = (-lx+LN_TWO)*I0 - 0.25 + c;
    return result;
  }
  else {
    double K0_scaled = gsl_sf_bessel_K0_scaled_e(x);
    result = K0_scaled * sexp(x);
    return result;
  }
}

double gsl_sf_bessel_K1_scaled_e(const double x)
{
  /* CHECK_POINTER(result) */
	double result;

  if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* DOMAIN_ERROR(result); */
  }
  else if(x < 2.0*DBL_MIN) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* OVERFLOW_ERROR(result); */
  }
  else if(x <= 2.0) {
    const double lx = log(x);
    const double ex = exp(x);
    double I1 = gsl_sf_bessel_I1_e(x);
    double c = cheb_eval_e(&bk1_cs, 0.5*x*x-1.0);
    result  = ex * ((lx-LN_TWO)*I1 + (0.75 + c)/x);
    return result;
  }
  else if(x <= 8.0) {
    const double sx = sqrt(x);
    double c = cheb_eval_e(&ak1_cs, (16.0/x-5.0)/3.0);
    result = (1.25 + c) / sx;
    return result;
  }
  else {
    const double sx = sqrt(x);
    double c = cheb_eval_e(&ak12_cs, 16.0/x-1.0);
    result = (1.25 + c) / sx;
    return result;
  }
}

double gsl_sf_bessel_K1_e(const double x)
{
  /* CHECK_POINTER(result) */
	double result;

  if(x <= 0.0) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* DOMAIN_ERROR(result); */
  }
  else if(x < 2.0*DBL_MIN) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* OVERFLOW_ERROR(result); */
  }
  else if(x <= 2.0) {
    const double lx = log(x);
    double I1 = gsl_sf_bessel_I1_e(x);
    double c = cheb_eval_e(&bk1_cs, 0.5*x*x-1.0);
    result = (lx-LN_TWO)*I1 + (0.75 + c)/x;
    return result;
  }
  else {
    double K1_scaled = gsl_sf_bessel_K1_scaled_e(x);
    result = K1_scaled * sexp( x );
    return result;
  }
}

double gsl_sf_bessel_I0_scaled_e(const double x)
{
	double result;
  double y = fabs(x);

  /* CHECK_POINTER(result) */

  if(y < 2.0 * sqrt(DBL_EPSILON) ) {
    result = 1.0 - y;
    return result;
  }
  else if(y <= 3.0) {
    const double ey = exp(-y);
    double c = cheb_eval_e(&bi0_cs, y*y/4.5-1.0);
    result = ey * (2.75 + c);
    return result;
  }
  else if(y <= 8.0) {
    const double sy = sqrt(y);
    double c = cheb_eval_e(&ai0_cs, (48.0/y-11.0)/5.0);
    result = (0.375 + c) / sy;
    return result;
  }
  else {
    const double sy = sqrt(y);
    double c = cheb_eval_e(&ai02_cs, 16.0/y-1.0);
    result = (0.375 + c) / sy;
    return result;
  }
}


double gsl_sf_bessel_I0_e(const double x)
{
  double y = fabs(x);
  double result;

  /* CHECK_POINTER(result) */

  if(y < 2.0 * sqrt(DBL_EPSILON) ) {
    result = 1.0;
    return result;
  }
  else if(y <= 3.0) {
    double c = cheb_eval_e(&bi0_cs, y*y/4.5-1.0);
    result = 2.75 + c;
    return result;
  }
  else if(y < log(DBL_MAX) - 1.0) {
    const double ey = exp(y);
    double b_scaled = gsl_sf_bessel_I0_scaled_e(x);
    result = ey * b_scaled;
    return result;
  }
  else {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* OVERFLOW_ERROR(result); */
  }
}

double gsl_sf_bessel_I1_scaled_e(const double x)
{
  const double xmin    = 2.0 * DBL_MIN;
  const double x_small = sqrt(8.) * sqrt(DBL_EPSILON);
  const double y = fabs(x);
  double result;

  /* CHECK_POINTER(result) */

  if(y == 0.0) {
    result = 0.0;
    return result;
  }
  else if(y < xmin) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* UNDERFLOW_ERROR(result); */
  }
  else if(y < x_small) {
    result = 0.5*x;
    return result;
  }
  else if(y <= 3.0) {
    const double ey = exp(-y);
    double c = cheb_eval_e(&bi1_cs, y*y/4.5-1.0);
    result = x * ey * (0.875 + c);
    return result;
  }
  else if(y <= 8.0) {
    const double sy = sqrt(y);
    double c = cheb_eval_e(&ai1_cs, (48.0/y-11.0)/5.0);
    double b = (0.375 + c) / sy;
    double s = (x > 0.0 ? 1.0 : -1.0);
    result = s * b;
    return result;
  }
  else {
    const double sy = sqrt(y);
    double c = cheb_eval_e(&ai12_cs, 16.0/y-1.0);
    double b = (0.375 + c) / sy;
    double s = (x > 0.0 ? 1.0 : -1.0);
    result = s * b;
    return result;
  }
}

double gsl_sf_bessel_I1_e(const double x)
{
  const double xmin    = 2.0 * DBL_MIN;
  const double x_small = sqrt(8.) * sqrt(DBL_EPSILON);
  const double y = fabs(x);
  double result;

  /* CHECK_POINTER(result) */

  if(y == 0.0) {
    result = 0.0;
    return result;
  }
  else if(y < xmin) {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* UNDERFLOW_ERROR(result); */
  }
  else if(y < x_small) {
    result = 0.5*x;
    return result;
  }
  else if(y <= 3.0) {
    double c = cheb_eval_e(&bi1_cs, y*y/4.5-1.0);
    result = x * (0.875 + c);
    return result;
  }
  else if(y < log(DBL_MAX)) {
    const double ey = exp(y);
    double I1_scaled = gsl_sf_bessel_I1_scaled_e(x);
    result = ey * I1_scaled;
    return result;
  }
  else {
		ASSERT( 1==0 );
		cdEXIT( EXIT_FAILURE );
	  /* OVERFLOW_ERROR(result); */
  }
}

