/* This file is part of Cloudy and is copyright (C)1978-2006 by Gary J. Ferland
 * For conditions of distribution and use see copyright notice in license.txt */
/*HeRecom - do recomb coef for He, called by HeLike */
/*cross_section - calculates the photoionization cross_section for a given level and photon energy*/
/*radrecomb - calculates radiative recombination coefficients. */
/*RecomInt - Integral in milne relation.  Called by qg32.	*/
/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * nelem is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
/*HelikeCheckRecomb - called by SanityCheck to confirm that recombination coef are ok,
 * return value is relative error between new calculation of recom, and interp value */
/*Recomb_Seaton59 - find recombination for given n,using Seaton 59 approximation.
 * The following three are needed by Recomb_Seaton59:
 *     ExponentialInt
 *     X1Int
 *     X2Int	*/

#include "cddefines.h" 
#include "trace.h"
#include "rt.h"
#include "path.h"
#include "physconst.h" 
#include "hydro_bauman.h"
#include "iso.h"
#include "helike.h"
#include "thermal.h"
#include "helike_recom.h"
#include "interpolate.h"
#include "dense.h"
#include "phycon.h"
#include "opacity.h"
#include "ionbal.h"
#include "atmdat.h"
#include "bevington.h"
/* This will be needed if the Peach algorithms are turned on. */
/* #include "math_complex.h" */

/*lint -e662 creation of  out of bound pointer */
/*lint -e661 creation of  out of bound pointer */

/* this will save log of radiative recombination rate coefficients at N_HE_TE_RECOMB temperatures.
 * there will be NumHeLevRecomb[nelem] levels saved in RRCoef[nelem][level][temp] */
static double ***RRCoef/*[LIMELM][NumHeLevRecomb[nelem]][N_HE_TE_RECOMB]*/;

static double **TotalRecomb;	/*[nelem][i]*/

/* the designation of the levels, chLevel[n][string]
static char **chLevel; */

/* the array of logs of temperatures at which RRCoef was defined */
static double TeRRCoef[N_HE_TE_RECOMB];

static double TempInterp( double* TempArray , double* ValueArray, long NumElements );

static double He_RRCoef_Te( long nelem , long n );

/* This computes that amount of the total recombination into an element (at a
 * specific temperature) that goes into levels having n greater than that otherwise
 * accounted for.	The reference is Seaton 59, main equation is 29. */
static double Recomb_Seaton59( long nelem, double temp, long n );

/* The three of these are used in the computation of Recomb_Seaton59	*/
static double ExponentialInt( double v );
static double X1Int( double u );
static double X2Int( double u );

static double cross_section(double EgammaRyd);
static double radrecomb(double temp, long nelem, long ipLo);
static double RecomInt(double EE);

#if 0
/* all of this is Peach stuff and is commented out currently. */
static double PeachCrossSection( long int nelem, long int l, long int s, double electron_energy );
static double Get_mu_prime(long int s, long int lprime);
static double Get_Peach_g( double neff, long int l, double electron_energy, long int lprime );
static double g_integrand( double radius );
static double global_nu, global_k;
static long int global_l, global_lprime, global_s;
#endif

static double kTRyd,EthRyd,Xn_S59; 
static long int ipLev,globalZ;

/* Here are several structures containing fits to cross-sections calculated in the following: */
/* >>refer He	PCS	Peach, G. 1967 MmRAS 71, 13	*/  
double PeachE[460] ={
	0.0000E+00,1.3681E-07,7.1942E-07,1.7619E-06,3.2547E-06,5.1839E-06,7.5316E-06,1.0276E-05,
	1.3391E-05,1.6848E-05,2.0614E-05,2.4655E-05,2.8932E-05,3.3407E-05,3.8036E-05,4.2776E-05,
	4.7585E-05,5.2415E-05,5.7224E-05,6.1964E-05,6.6593E-05,7.1068E-05,7.5345E-05,7.9386E-05,
	8.3152E-05,8.6609E-05,8.9724E-05,9.2468E-05,9.4816E-05,9.6745E-05,9.8238E-05,9.9281E-05,
	9.9863E-05,1.0014E-04,1.0072E-04,1.0176E-04,1.0325E-04,1.0518E-04,1.0753E-04,1.1028E-04,
	1.1339E-04,1.1685E-04,1.2061E-04,1.2466E-04,1.2893E-04,1.3341E-04,1.3804E-04,1.4278E-04,
	1.4758E-04,1.5242E-04,1.5722E-04,1.6196E-04,1.6659E-04,1.7107E-04,1.7535E-04,1.7939E-04,
	1.8315E-04,1.8661E-04,1.8972E-04,1.9247E-04,1.9482E-04,1.9675E-04,1.9824E-04,1.9928E-04,
	1.9986E-04,2.0027E-04,2.0144E-04,2.0352E-04,2.0651E-04,2.1037E-04,2.1506E-04,2.2055E-04,
	2.2678E-04,2.3370E-04,2.4123E-04,2.4931E-04,2.5786E-04,2.6681E-04,2.7607E-04,2.8555E-04,
	2.9517E-04,3.0483E-04,3.1445E-04,3.2393E-04,3.3319E-04,3.4214E-04,3.5069E-04,3.5877E-04,
	3.6630E-04,3.7322E-04,3.7945E-04,3.8494E-04,3.8963E-04,3.9349E-04,3.9648E-04,3.9856E-04,
	3.9973E-04,4.0055E-04,4.0288E-04,4.0705E-04,4.1302E-04,4.2074E-04,4.3013E-04,4.4110E-04,
	4.5356E-04,4.6739E-04,4.8246E-04,4.9862E-04,5.1573E-04,5.3363E-04,5.5214E-04,5.7111E-04,
	5.9034E-04,6.0966E-04,6.2889E-04,6.4786E-04,6.6637E-04,6.8427E-04,7.0138E-04,7.1754E-04,
	7.3261E-04,7.4644E-04,7.5890E-04,7.6987E-04,7.7926E-04,7.8698E-04,7.9295E-04,7.9712E-04,
	7.9945E-04,8.0109E-04,8.0576E-04,8.1410E-04,8.2604E-04,8.4147E-04,8.6025E-04,8.8221E-04,
	9.0713E-04,9.3478E-04,9.6491E-04,9.9724E-04,1.0315E-03,1.0673E-03,1.1043E-03,1.1422E-03,
	1.1807E-03,1.2193E-03,1.2578E-03,1.2957E-03,1.3327E-03,1.3685E-03,1.4028E-03,1.4351E-03,
	1.4652E-03,1.4929E-03,1.5178E-03,1.5397E-03,1.5585E-03,1.5740E-03,1.5859E-03,1.5942E-03,
	1.5989E-03,1.6022E-03,1.6115E-03,1.6282E-03,1.6521E-03,1.6829E-03,1.7205E-03,1.7644E-03,
	1.8143E-03,1.8696E-03,1.9298E-03,1.9945E-03,2.0629E-03,2.1345E-03,2.2086E-03,2.2844E-03,
	2.3614E-03,2.4386E-03,2.5156E-03,2.5914E-03,2.6655E-03,2.7371E-03,2.8055E-03,2.8702E-03,
	2.9304E-03,2.9857E-03,3.0356E-03,3.0795E-03,3.1171E-03,3.1479E-03,3.1718E-03,3.1885E-03,
	3.1978E-03,3.2044E-03,3.2230E-03,3.2564E-03,3.3042E-03,3.3659E-03,3.4410E-03,3.5288E-03,
	3.6285E-03,3.7391E-03,3.8597E-03,3.9890E-03,4.1258E-03,4.2690E-03,4.4171E-03,4.5688E-03,
	4.7227E-03,4.8773E-03,5.0312E-03,5.1829E-03,5.3310E-03,5.4742E-03,5.6110E-03,5.7403E-03,
	5.8609E-03,5.9715E-03,6.0712E-03,6.1590E-03,6.2341E-03,6.2959E-03,6.3436E-03,6.3770E-03,
	6.3956E-03,6.4088E-03,6.4460E-03,6.5128E-03,6.6083E-03,6.7318E-03,6.8820E-03,7.0577E-03,
	7.2570E-03,7.4783E-03,7.7193E-03,7.9779E-03,8.2517E-03,8.5380E-03,8.8343E-03,9.1377E-03,
	9.4454E-03,9.7546E-03,1.0062E-02,1.0366E-02,1.0662E-02,1.0948E-02,1.1222E-02,1.1481E-02,
	1.1722E-02,1.1943E-02,1.2142E-02,1.2318E-02,1.2468E-02,1.2592E-02,1.2687E-02,1.2754E-02,
	1.2791E-02,1.2818E-02,1.2892E-02,1.3026E-02,1.3217E-02,1.3464E-02,1.3764E-02,1.4115E-02,
	1.4514E-02,1.4957E-02,1.5439E-02,1.5956E-02,1.6503E-02,1.7076E-02,1.7669E-02,1.8275E-02,
	1.8891E-02,1.9509E-02,2.0125E-02,2.0731E-02,2.1324E-02,2.1897E-02,2.2444E-02,2.2961E-02,
	2.3443E-02,2.3886E-02,2.4285E-02,2.4636E-02,2.4936E-02,2.5183E-02,2.5374E-02,2.5508E-02,
	2.5582E-02,2.5635E-02,2.5784E-02,2.6051E-02,2.6433E-02,2.6927E-02,2.7528E-02,2.8231E-02,
	2.9028E-02,2.9913E-02,3.0877E-02,3.1912E-02,3.3007E-02,3.4152E-02,3.5337E-02,3.6551E-02,
	3.7782E-02,3.9018E-02,4.0249E-02,4.1463E-02,4.2648E-02,4.3793E-02,4.4888E-02,4.5923E-02,
	4.6887E-02,4.7772E-02,4.8569E-02,4.9272E-02,4.9873E-02,5.0367E-02,5.0749E-02,5.1016E-02,
	5.1165E-02,5.1270E-02,5.1568E-02,5.2102E-02,5.2866E-02,5.3854E-02,5.5056E-02,5.6461E-02,
	5.8056E-02,5.9826E-02,6.1754E-02,6.3823E-02,6.6013E-02,6.8304E-02,7.0674E-02,7.3102E-02,
	7.5563E-02,7.8037E-02,8.0498E-02,8.2926E-02,8.5296E-02,8.7587E-02,8.9777E-02,9.1846E-02,
	9.3774E-02,9.5544E-02,9.7139E-02,9.8544E-02,9.9746E-02,1.0073E-01,1.0150E-01,1.0203E-01,
	1.0233E-01,1.0254E-01,1.0314E-01,1.0420E-01,1.0573E-01,1.0771E-01,1.1011E-01,1.1292E-01,
	1.1611E-01,1.1965E-01,1.2351E-01,1.2765E-01,1.3203E-01,1.3661E-01,1.4135E-01,1.4620E-01,
	1.5113E-01,1.5607E-01,1.6100E-01,1.6585E-01,1.7059E-01,1.7517E-01,1.7955E-01,1.8369E-01,
	1.8755E-01,1.9109E-01,1.9428E-01,1.9709E-01,1.9949E-01,2.0147E-01,2.0300E-01,2.0406E-01,
	2.0466E-01,2.0508E-01,2.0627E-01,2.0841E-01,2.1147E-01,2.1542E-01,2.2022E-01,2.2584E-01,
	2.3222E-01,2.3930E-01,2.4702E-01,2.5529E-01,2.6405E-01,2.7322E-01,2.8270E-01,2.9241E-01,
	3.0225E-01,3.1215E-01,3.2199E-01,3.3170E-01,3.4118E-01,3.5035E-01,3.5911E-01,3.6738E-01,
	3.7510E-01,3.8218E-01,3.8856E-01,3.9418E-01,3.9898E-01,4.0293E-01,4.0599E-01,4.0813E-01,
	4.0932E-01,4.1016E-01,4.1255E-01,4.1682E-01,4.2293E-01,4.3083E-01,4.4045E-01,4.5169E-01,
	4.6445E-01,4.7861E-01,4.9404E-01,5.1059E-01,5.2811E-01,5.4643E-01,5.6539E-01,5.8481E-01,
	6.0451E-01,6.2429E-01,6.4399E-01,6.6341E-01,6.8237E-01,7.0069E-01,7.1821E-01,7.3476E-01,
	7.5019E-01,7.6435E-01,7.7711E-01,7.8835E-01,7.9797E-01,8.0587E-01,8.1198E-01,8.1625E-01,
	8.1864E-01,8.2032E-01,8.2509E-01,8.3363E-01,8.4586E-01,8.6167E-01,8.8090E-01,9.0338E-01,
	9.2890E-01,9.5722E-01,9.8807E-01,1.0212E+00};

double TripP[7][460] ={
	/*4 111*/
	{
	1.4985E+00,1.4985E+00,1.4985E+00,1.4984E+00,1.4984E+00,1.4983E+00,1.4981E+00,1.4980E+00,
	1.4978E+00,1.4977E+00,1.4975E+00,1.4973E+00,1.4971E+00,1.4968E+00,1.4966E+00,1.4964E+00,
	1.4961E+00,1.4959E+00,1.4956E+00,1.4954E+00,1.4952E+00,1.4949E+00,1.4947E+00,1.4945E+00,
	1.4943E+00,1.4942E+00,1.4940E+00,1.4939E+00,1.4937E+00,1.4936E+00,1.4936E+00,1.4935E+00,
	1.4935E+00,1.4935E+00,1.4934E+00,1.4934E+00,1.4933E+00,1.4932E+00,1.4931E+00,1.4930E+00,
	1.4928E+00,1.4926E+00,1.4925E+00,1.4922E+00,1.4920E+00,1.4918E+00,1.4916E+00,1.4913E+00,
	1.4911E+00,1.4909E+00,1.4906E+00,1.4904E+00,1.4901E+00,1.4899E+00,1.4897E+00,1.4895E+00,
	1.4893E+00,1.4891E+00,1.4890E+00,1.4889E+00,1.4887E+00,1.4886E+00,1.4886E+00,1.4885E+00,
	1.4885E+00,1.4885E+00,1.4884E+00,1.4883E+00,1.4882E+00,1.4880E+00,1.4877E+00,1.4875E+00,
	1.4871E+00,1.4868E+00,1.4864E+00,1.4860E+00,1.4856E+00,1.4852E+00,1.4847E+00,1.4842E+00,
	1.4837E+00,1.4833E+00,1.4828E+00,1.4823E+00,1.4819E+00,1.4814E+00,1.4810E+00,1.4806E+00,
	1.4802E+00,1.4799E+00,1.4796E+00,1.4793E+00,1.4791E+00,1.4789E+00,1.4787E+00,1.4786E+00,
	1.4786E+00,1.4785E+00,1.4784E+00,1.4782E+00,1.4779E+00,1.4775E+00,1.4771E+00,1.4765E+00,
	1.4759E+00,1.4752E+00,1.4745E+00,1.4737E+00,1.4728E+00,1.4720E+00,1.4711E+00,1.4701E+00,
	1.4692E+00,1.4682E+00,1.4673E+00,1.4664E+00,1.4655E+00,1.4646E+00,1.4638E+00,1.4630E+00,
	1.4623E+00,1.4616E+00,1.4610E+00,1.4604E+00,1.4600E+00,1.4596E+00,1.4593E+00,1.4591E+00,
	1.4590E+00,1.4589E+00,1.4587E+00,1.4583E+00,1.4577E+00,1.4570E+00,1.4561E+00,1.4550E+00,
	1.4538E+00,1.4525E+00,1.4510E+00,1.4495E+00,1.4478E+00,1.4461E+00,1.4444E+00,1.4426E+00,
	1.4407E+00,1.4389E+00,1.4371E+00,1.4353E+00,1.4335E+00,1.4318E+00,1.4302E+00,1.4287E+00,
	1.4273E+00,1.4260E+00,1.4248E+00,1.4238E+00,1.4229E+00,1.4222E+00,1.4217E+00,1.4213E+00,
	1.4211E+00,1.4209E+00,1.4205E+00,1.4197E+00,1.4186E+00,1.4172E+00,1.4154E+00,1.4134E+00,
	1.4111E+00,1.4085E+00,1.4058E+00,1.4028E+00,1.3997E+00,1.3965E+00,1.3931E+00,1.3897E+00,
	1.3862E+00,1.3828E+00,1.3794E+00,1.3760E+00,1.3727E+00,1.3696E+00,1.3665E+00,1.3637E+00,
	1.3611E+00,1.3587E+00,1.3565E+00,1.3546E+00,1.3530E+00,1.3517E+00,1.3506E+00,1.3499E+00,
	1.3495E+00,1.3492E+00,1.3484E+00,1.3470E+00,1.3449E+00,1.3423E+00,1.3391E+00,1.3354E+00,
	1.3312E+00,1.3265E+00,1.3215E+00,1.3161E+00,1.3104E+00,1.3046E+00,1.2985E+00,1.2924E+00,
	1.2862E+00,1.2801E+00,1.2740E+00,1.2680E+00,1.2622E+00,1.2567E+00,1.2514E+00,1.2465E+00,
	1.2419E+00,1.2377E+00,1.2339E+00,1.2306E+00,1.2278E+00,1.2255E+00,1.2237E+00,1.2225E+00,
	1.2218E+00,1.2213E+00,1.2199E+00,1.2175E+00,1.2140E+00,1.2095E+00,1.2040E+00,1.1977E+00,
	1.1905E+00,1.1827E+00,1.1742E+00,1.1653E+00,1.1559E+00,1.1462E+00,1.1363E+00,1.1263E+00,
	1.1163E+00,1.1063E+00,1.0966E+00,1.0871E+00,1.0779E+00,1.0692E+00,1.0610E+00,1.0533E+00,
	1.0461E+00,1.0397E+00,1.0339E+00,1.0289E+00,1.0246E+00,1.0211E+00,1.0184E+00,1.0165E+00,
	1.0155E+00,1.0147E+00,1.0126E+00,1.0089E+00,1.0036E+00,9.9685E-01,9.8870E-01,9.7929E-01,
	9.6877E-01,9.5730E-01,9.4503E-01,9.3212E-01,9.1873E-01,9.0504E-01,8.9118E-01,8.7731E-01,
	8.6357E-01,8.5008E-01,8.3685E-01,8.2372E-01,8.1119E-01,7.9934E-01,7.8825E-01,7.7798E-01,
	7.6858E-01,7.6011E-01,7.5259E-01,7.4605E-01,7.4053E-01,7.3603E-01,7.3258E-01,7.3018E-01,
	7.2885E-01,7.2791E-01,7.2526E-01,7.2055E-01,7.1388E-01,7.0540E-01,6.9527E-01,6.8370E-01,
	6.7091E-01,6.5711E-01,6.4255E-01,6.2744E-01,6.1200E-01,5.9643E-01,5.8092E-01,5.6563E-01,
	5.5071E-01,5.3628E-01,5.2236E-01,5.0884E-01,4.9612E-01,4.8425E-01,4.7329E-01,4.6326E-01,
	4.5419E-01,4.4608E-01,4.3896E-01,4.3281E-01,4.2765E-01,4.2348E-01,4.2029E-01,4.1808E-01,
	4.1686E-01,4.1600E-01,4.1357E-01,4.0927E-01,4.0323E-01,3.9560E-01,3.8660E-01,3.7644E-01,
	3.6536E-01,3.5359E-01,3.4122E-01,3.2861E-01,3.1596E-01,3.0347E-01,2.9126E-01,2.7946E-01,
	2.6817E-01,2.5746E-01,2.4734E-01,2.3776E-01,2.2890E-01,2.2078E-01,2.1339E-01,2.0672E-01,
	2.0077E-01,1.9551E-01,1.9094E-01,1.8703E-01,1.8378E-01,1.8111E-01,1.7907E-01,1.7766E-01,
	1.7688E-01,1.7634E-01,1.7480E-01,1.7210E-01,1.6833E-01,1.6362E-01,1.5815E-01,1.5207E-01,
	1.4556E-01,1.3879E-01,1.3190E-01,1.2502E-01,1.1828E-01,1.1176E-01,1.0552E-01,9.9617E-02,
	9.4127E-02,8.9155E-02,8.4547E-02,8.0306E-02,7.6431E-02,7.2913E-02,6.9743E-02,6.6908E-02,
	6.4396E-02,6.2195E-02,6.0291E-02,5.8674E-02,5.7333E-02,5.6254E-02,5.5436E-02,5.4875E-02,
	5.4565E-02,5.4348E-02,5.3737E-02,5.2669E-02,5.1189E-02,4.9358E-02,4.7248E-02,4.4934E-02,
	4.2488E-02,3.9979E-02,3.7465E-02,3.4997E-02,3.2613E-02,3.0341E-02,2.8204E-02,2.6212E-02,
	2.4372E-02,2.2689E-02,2.1166E-02,1.9794E-02,1.8566E-02,1.7474E-02,1.6507E-02,1.5657E-02,
	1.4915E-02,1.4274E-02,1.3727E-02,1.3267E-02,1.2890E-02,1.2591E-02,1.2366E-02,1.2213E-02,
	1.2128E-02,1.2068E-02,1.1902E-02,1.1612E-02,1.1213E-02,1.0723E-02,1.0164E-02,9.5581E-03,
	8.9258E-03,8.2860E-03,7.6544E-03,7.0469E-03,6.4720E-03,5.9341E-03,5.4368E-03,4.9815E-03,
	4.5720E-03,4.2141E-03,3.8923E-03,3.6046E-03,3.3488E-03,3.1226E-03,2.9275E-03,2.7572E-03,
	2.6095E-03,2.4825E-03,2.3746E-03,2.2843E-03,2.2104E-03,2.1527E-03,2.1096E-03,2.0802E-03,
	2.0639E-03,2.0526E-03,2.0208E-03,1.9655E-03,1.8895E-03,1.7966E-03,1.6911E-03,1.5771E-03,
	1.4581E-03,1.3390E-03,1.2229E-03,1.6698E-03},
	/*5 111*/
	{
	1.9998E+00,1.9998E+00,1.9998E+00,1.9997E+00,1.9995E+00,1.9993E+00,1.9991E+00,1.9988E+00,
	1.9985E+00,1.9981E+00,1.9977E+00,1.9973E+00,1.9969E+00,1.9964E+00,1.9960E+00,1.9955E+00,
	1.9950E+00,1.9945E+00,1.9940E+00,1.9935E+00,1.9930E+00,1.9926E+00,1.9922E+00,1.9917E+00,
	1.9914E+00,1.9910E+00,1.9907E+00,1.9904E+00,1.9902E+00,1.9900E+00,1.9898E+00,1.9897E+00,
	1.9897E+00,1.9896E+00,1.9896E+00,1.9895E+00,1.9893E+00,1.9891E+00,1.9889E+00,1.9886E+00,
	1.9883E+00,1.9879E+00,1.9876E+00,1.9871E+00,1.9867E+00,1.9863E+00,1.9858E+00,1.9853E+00,
	1.9848E+00,1.9843E+00,1.9838E+00,1.9834E+00,1.9829E+00,1.9824E+00,1.9820E+00,1.9816E+00,
	1.9812E+00,1.9809E+00,1.9806E+00,1.9803E+00,1.9801E+00,1.9799E+00,1.9797E+00,1.9796E+00,
	1.9795E+00,1.9795E+00,1.9794E+00,1.9792E+00,1.9789E+00,1.9785E+00,1.9780E+00,1.9775E+00,
	1.9768E+00,1.9761E+00,1.9754E+00,1.9746E+00,1.9737E+00,1.9728E+00,1.9719E+00,1.9709E+00,
	1.9700E+00,1.9690E+00,1.9681E+00,1.9671E+00,1.9662E+00,1.9653E+00,1.9644E+00,1.9636E+00,
	1.9629E+00,1.9622E+00,1.9616E+00,1.9610E+00,1.9606E+00,1.9602E+00,1.9599E+00,1.9597E+00,
	1.9596E+00,1.9595E+00,1.9592E+00,1.9588E+00,1.9582E+00,1.9575E+00,1.9565E+00,1.9555E+00,
	1.9542E+00,1.9529E+00,1.9514E+00,1.9498E+00,1.9481E+00,1.9463E+00,1.9445E+00,1.9427E+00,
	1.9408E+00,1.9389E+00,1.9370E+00,1.9352E+00,1.9334E+00,1.9316E+00,1.9300E+00,1.9284E+00,
	1.9270E+00,1.9256E+00,1.9244E+00,1.9234E+00,1.9224E+00,1.9217E+00,1.9211E+00,1.9207E+00,
	1.9205E+00,1.9203E+00,1.9199E+00,1.9191E+00,1.9179E+00,1.9165E+00,1.9147E+00,1.9126E+00,
	1.9102E+00,1.9075E+00,1.9047E+00,1.9016E+00,1.8984E+00,1.8950E+00,1.8915E+00,1.8879E+00,
	1.8843E+00,1.8807E+00,1.8771E+00,1.8736E+00,1.8702E+00,1.8669E+00,1.8637E+00,1.8608E+00,
	1.8580E+00,1.8555E+00,1.8532E+00,1.8512E+00,1.8495E+00,1.8481E+00,1.8470E+00,1.8462E+00,
	1.8458E+00,1.8455E+00,1.8447E+00,1.8432E+00,1.8410E+00,1.8382E+00,1.8348E+00,1.8309E+00,
	1.8264E+00,1.8215E+00,1.8162E+00,1.8105E+00,1.8045E+00,1.7982E+00,1.7918E+00,1.7852E+00,
	1.7786E+00,1.7720E+00,1.7654E+00,1.7590E+00,1.7528E+00,1.7468E+00,1.7411E+00,1.7357E+00,
	1.7308E+00,1.7262E+00,1.7221E+00,1.7186E+00,1.7155E+00,1.7130E+00,1.7111E+00,1.7097E+00,
	1.7090E+00,1.7084E+00,1.7069E+00,1.7042E+00,1.7004E+00,1.6955E+00,1.6895E+00,1.6825E+00,
	1.6747E+00,1.6660E+00,1.6567E+00,1.6468E+00,1.6364E+00,1.6256E+00,1.6145E+00,1.6033E+00,
	1.5921E+00,1.5809E+00,1.5699E+00,1.5591E+00,1.5487E+00,1.5388E+00,1.5294E+00,1.5206E+00,
	1.5124E+00,1.5050E+00,1.4983E+00,1.4925E+00,1.4876E+00,1.4835E+00,1.4804E+00,1.4782E+00,
	1.4770E+00,1.4761E+00,1.4737E+00,1.4694E+00,1.4632E+00,1.4553E+00,1.4458E+00,1.4348E+00,
	1.4224E+00,1.4089E+00,1.3944E+00,1.3790E+00,1.3630E+00,1.3466E+00,1.3299E+00,1.3131E+00,
	1.2964E+00,1.2799E+00,1.2638E+00,1.2482E+00,1.2332E+00,1.2190E+00,1.2056E+00,1.1931E+00,
	1.1817E+00,1.1713E+00,1.1621E+00,1.1540E+00,1.1472E+00,1.1416E+00,1.1374E+00,1.1344E+00,
	1.1327E+00,1.1316E+00,1.1283E+00,1.1224E+00,1.1140E+00,1.1034E+00,1.0906E+00,1.0760E+00,
	1.0597E+00,1.0420E+00,1.0231E+00,1.0034E+00,9.8316E-01,9.6255E-01,9.4183E-01,9.2123E-01,
	9.0095E-01,8.8117E-01,8.6188E-01,8.4285E-01,8.2487E-01,8.0803E-01,7.9241E-01,7.7807E-01,
	7.6505E-01,7.5338E-01,7.4310E-01,7.3421E-01,7.2673E-01,7.2068E-01,7.1604E-01,7.1283E-01,
	7.1104E-01,7.0979E-01,7.0625E-01,6.9998E-01,6.9114E-01,6.7997E-01,6.6674E-01,6.5175E-01,
	6.3535E-01,6.1786E-01,5.9961E-01,5.8091E-01,5.6206E-01,5.4330E-01,5.2485E-01,5.0692E-01,
	4.8965E-01,4.7316E-01,4.5744E-01,4.4224E-01,4.2812E-01,4.1509E-01,4.0318E-01,3.9239E-01,
	3.8271E-01,3.7414E-01,3.6665E-01,3.6024E-01,3.5488E-01,3.5057E-01,3.4728E-01,3.4502E-01,
	3.4376E-01,3.4288E-01,3.4039E-01,3.3601E-01,3.2988E-01,3.2220E-01,3.1320E-01,3.0315E-01,
	2.9230E-01,2.8091E-01,2.6904E-01,2.5711E-01,2.4532E-01,2.3383E-01,2.2277E-01,2.1224E-01,
	2.0230E-01,1.9300E-01,1.8435E-01,1.7637E-01,1.6907E-01,1.6245E-01,1.5648E-01,1.5114E-01,
	1.4641E-01,1.4226E-01,1.3868E-01,1.3563E-01,1.3311E-01,1.3093E-01,1.2923E-01,1.2807E-01,
	1.2742E-01,1.2697E-01,1.2570E-01,1.2348E-01,1.2039E-01,1.1655E-01,1.1211E-01,1.0721E-01,
	1.0201E-01,9.6643E-02,9.1232E-02,8.5883E-02,8.0681E-02,7.5692E-02,7.0964E-02,6.6527E-02,
	6.2464E-02,5.8913E-02,5.5646E-02,5.2662E-02,4.9952E-02,4.7507E-02,4.5315E-02,4.3365E-02,
	4.1644E-02,4.0142E-02,3.8848E-02,3.7751E-02,3.6844E-02,3.6095E-02,3.5524E-02,3.5132E-02,
	3.4916E-02,3.4765E-02,3.4340E-02,3.3599E-02,3.2574E-02,3.1313E-02,2.9867E-02,2.8290E-02,
	2.6634E-02,2.4947E-02,2.3270E-02,2.1634E-02,2.0067E-02,1.8585E-02,1.7200E-02,1.5918E-02,
	1.4741E-02,1.3669E-02,1.2704E-02,1.1840E-02,1.1071E-02,1.0390E-02,9.7896E-03,9.2641E-03,
	8.8073E-03,8.4138E-03,8.0787E-03,7.7977E-03,7.5674E-03,7.3848E-03,7.2475E-03,7.1536E-03,
	7.1019E-03,7.0657E-03,6.9644E-03,6.7878E-03,6.5451E-03,6.2483E-03,5.9105E-03,5.5455E-03,
	5.1662E-03,4.7839E-03,4.4080E-03,4.0470E-03,3.7064E-03,3.3890E-03,3.0969E-03,2.8305E-03,
	2.5917E-03,2.3830E-03,2.1961E-03,2.0296E-03,1.8821E-03,1.7520E-03,1.6399E-03,1.5422E-03,
	1.4577E-03,1.3853E-03,1.3238E-03,1.2724E-03,1.2304E-03,1.1976E-03,1.1731E-03,1.1564E-03,
	1.1472E-03,1.1408E-03,1.1228E-03,1.0914E-03,1.0484E-03,9.9594E-04,9.3642E-04,8.7219E-04,
	8.0510E-04,7.3814E-04,6.7301E-04,6.2061E-04},
	/*6 111*/
	{
	2.5409E+00,2.5408E+00,2.5407E+00,2.5405E+00,2.5402E+00,2.5398E+00,2.5394E+00,2.5388E+00,
	2.5382E+00,2.5376E+00,2.5368E+00,2.5360E+00,2.5352E+00,2.5343E+00,2.5334E+00,2.5325E+00,
	2.5315E+00,2.5306E+00,2.5296E+00,2.5287E+00,2.5278E+00,2.5269E+00,2.5261E+00,2.5253E+00,
	2.5246E+00,2.5239E+00,2.5233E+00,2.5227E+00,2.5223E+00,2.5219E+00,2.5216E+00,2.5214E+00,
	2.5213E+00,2.5212E+00,2.5211E+00,2.5209E+00,2.5206E+00,2.5203E+00,2.5198E+00,2.5193E+00,
	2.5187E+00,2.5180E+00,2.5173E+00,2.5165E+00,2.5156E+00,2.5148E+00,2.5139E+00,2.5130E+00,
	2.5120E+00,2.5111E+00,2.5102E+00,2.5092E+00,2.5083E+00,2.5075E+00,2.5067E+00,2.5059E+00,
	2.5052E+00,2.5045E+00,2.5039E+00,2.5034E+00,2.5029E+00,2.5025E+00,2.5022E+00,2.5020E+00,
	2.5019E+00,2.5019E+00,2.5016E+00,2.5012E+00,2.5007E+00,2.4999E+00,2.4990E+00,2.4980E+00,
	2.4968E+00,2.4954E+00,2.4940E+00,2.4924E+00,2.4908E+00,2.4891E+00,2.4873E+00,2.4855E+00,
	2.4837E+00,2.4818E+00,2.4800E+00,2.4782E+00,2.4765E+00,2.4748E+00,2.4732E+00,2.4716E+00,
	2.4702E+00,2.4689E+00,2.4677E+00,2.4667E+00,2.4658E+00,2.4651E+00,2.4645E+00,2.4641E+00,
	2.4639E+00,2.4638E+00,2.4633E+00,2.4625E+00,2.4614E+00,2.4600E+00,2.4582E+00,2.4561E+00,
	2.4538E+00,2.4512E+00,2.4484E+00,2.4454E+00,2.4422E+00,2.4389E+00,2.4355E+00,2.4320E+00,
	2.4285E+00,2.4249E+00,2.4214E+00,2.4179E+00,2.4146E+00,2.4113E+00,2.4082E+00,2.4053E+00,
	2.4025E+00,2.4000E+00,2.3978E+00,2.3958E+00,2.3941E+00,2.3927E+00,2.3917E+00,2.3909E+00,
	2.3905E+00,2.3902E+00,2.3894E+00,2.3879E+00,2.3857E+00,2.3830E+00,2.3796E+00,2.3757E+00,
	2.3713E+00,2.3664E+00,2.3611E+00,2.3554E+00,2.3494E+00,2.3432E+00,2.3368E+00,2.3302E+00,
	2.3236E+00,2.3170E+00,2.3104E+00,2.3040E+00,2.2977E+00,2.2917E+00,2.2859E+00,2.2805E+00,
	2.2755E+00,2.2709E+00,2.2668E+00,2.2632E+00,2.2601E+00,2.2576E+00,2.2556E+00,2.2542E+00,
	2.2535E+00,2.2529E+00,2.2514E+00,2.2487E+00,2.2448E+00,2.2398E+00,2.2337E+00,2.2266E+00,
	2.2187E+00,2.2098E+00,2.2003E+00,2.1901E+00,2.1795E+00,2.1684E+00,2.1570E+00,2.1455E+00,
	2.1338E+00,2.1223E+00,2.1108E+00,2.0997E+00,2.0889E+00,2.0785E+00,2.0686E+00,2.0594E+00,
	2.0509E+00,2.0431E+00,2.0361E+00,2.0299E+00,2.0247E+00,2.0205E+00,2.0172E+00,2.0149E+00,
	2.0136E+00,2.0127E+00,2.0101E+00,2.0056E+00,1.9991E+00,1.9907E+00,1.9806E+00,1.9689E+00,
	1.9557E+00,1.9413E+00,1.9257E+00,1.9093E+00,1.8921E+00,1.8744E+00,1.8563E+00,1.8381E+00,
	1.8199E+00,1.8019E+00,1.7843E+00,1.7671E+00,1.7506E+00,1.7349E+00,1.7201E+00,1.7063E+00,
	1.6936E+00,1.6821E+00,1.6718E+00,1.6628E+00,1.6552E+00,1.6489E+00,1.6442E+00,1.6408E+00,
	1.6390E+00,1.6376E+00,1.6339E+00,1.6274E+00,1.6180E+00,1.6060E+00,1.5916E+00,1.5750E+00,
	1.5565E+00,1.5364E+00,1.5149E+00,1.4924E+00,1.4692E+00,1.4454E+00,1.4215E+00,1.3977E+00,
	1.3741E+00,1.3510E+00,1.3287E+00,1.3072E+00,1.2868E+00,1.2675E+00,1.2494E+00,1.2328E+00,
	1.2176E+00,1.2039E+00,1.1917E+00,1.1812E+00,1.1723E+00,1.1650E+00,1.1595E+00,1.1556E+00,
	1.1535E+00,1.1520E+00,1.1477E+00,1.1402E+00,1.1295E+00,1.1159E+00,1.0997E+00,1.0812E+00,
	1.0608E+00,1.0389E+00,1.0158E+00,9.9192E-01,9.6757E-01,9.4309E-01,9.1877E-01,8.9486E-01,
	8.7158E-01,8.4913E-01,8.2718E-01,8.0460E-01,7.8344E-01,7.6378E-01,7.4569E-01,7.2918E-01,
	7.1429E-01,7.0103E-01,6.8940E-01,6.7940E-01,6.7102E-01,6.6424E-01,6.5907E-01,6.5550E-01,
	6.5352E-01,6.5212E-01,6.4820E-01,6.4126E-01,6.3151E-01,6.1925E-01,6.0482E-01,5.8858E-01,
	5.7095E-01,5.5230E-01,5.3302E-01,5.1345E-01,4.9391E-01,4.7464E-01,4.5589E-01,4.3783E-01,
	4.2059E-01,4.0429E-01,3.8881E-01,3.7375E-01,3.5987E-01,3.4717E-01,3.3565E-01,3.2528E-01,
	3.1604E-01,3.0791E-01,3.0084E-01,2.9481E-01,2.8979E-01,2.8577E-01,2.8271E-01,2.8060E-01,
	2.7943E-01,2.7862E-01,2.7631E-01,2.7226E-01,2.6661E-01,2.5956E-01,2.5136E-01,2.4224E-01,
	2.3247E-01,2.2230E-01,2.1163E-01,2.0097E-01,1.9052E-01,1.8043E-01,1.7080E-01,1.6169E-01,
	1.5316E-01,1.4523E-01,1.3788E-01,1.3104E-01,1.2483E-01,1.1922E-01,1.1419E-01,1.0972E-01,
	1.0577E-01,1.0233E-01,9.9367E-02,9.6855E-02,9.4778E-02,9.3072E-02,9.1769E-02,9.0874E-02,
	9.0380E-02,9.0034E-02,8.9062E-02,8.7361E-02,8.5004E-02,8.2092E-02,7.8737E-02,7.5058E-02,
	7.1173E-02,6.7189E-02,6.3200E-02,5.9282E-02,5.5499E-02,5.1895E-02,4.8501E-02,4.5336E-02,
	4.2443E-02,3.9884E-02,3.7539E-02,3.5402E-02,3.3468E-02,3.1727E-02,3.0170E-02,2.8788E-02,
	2.7571E-02,2.6510E-02,2.5597E-02,2.4824E-02,2.4186E-02,2.3667E-02,2.3274E-02,2.3005E-02,
	2.2857E-02,2.2753E-02,2.2462E-02,2.1953E-02,2.1253E-02,2.0392E-02,1.9409E-02,1.8341E-02,
	1.7223E-02,1.6090E-02,1.4967E-02,1.3878E-02,1.2838E-02,1.1859E-02,1.0948E-02,1.0108E-02,
	9.3394E-03,8.6385E-03,8.0099E-03,7.4492E-03,6.9515E-03,6.5120E-03,6.1259E-03,5.7886E-03,
	5.4961E-03,5.2445E-03,5.0307E-03,4.8517E-03,4.7051E-03,4.5888E-03,4.5014E-03,4.4417E-03,
	4.4088E-03,4.3858E-03,4.3214E-03,4.2092E-03,4.0553E-03,3.8672E-03,3.6536E-03,3.4233E-03,
	3.1844E-03,2.9442E-03,2.7087E-03,2.4830E-03,2.2705E-03,2.0730E-03,1.8917E-03,1.7267E-03,
	1.5789E-03,1.4498E-03,1.3345E-03,1.2319E-03,1.1411E-03,1.0612E-03,9.9238E-04,9.3251E-04,
	8.8077E-04,8.3643E-04,7.9884E-04,7.6747E-04,7.4184E-04,7.2184E-04,7.0691E-04,6.9671E-04,
	6.9110E-04,6.8717E-04,6.7619E-04,6.5709E-04,6.3090E-04,5.9897E-04,5.6279E-04,5.2381E-04,
	4.8318E-04,4.4269E-04,4.0335E-04,3.9318E-04},
	/*7 111*/
	{
	3.1209E+00,3.1208E+00,3.1206E+00,3.1203E+00,3.1198E+00,3.1191E+00,3.1183E+00,3.1174E+00,
	3.1163E+00,3.1152E+00,3.1139E+00,3.1125E+00,3.1111E+00,3.1096E+00,3.1080E+00,3.1064E+00,
	3.1048E+00,3.1032E+00,3.1016E+00,3.1000E+00,3.0985E+00,3.0970E+00,3.0955E+00,3.0942E+00,
	3.0929E+00,3.0918E+00,3.0907E+00,3.0898E+00,3.0890E+00,3.0884E+00,3.0879E+00,3.0876E+00,
	3.0874E+00,3.0873E+00,3.0871E+00,3.0867E+00,3.0862E+00,3.0856E+00,3.0848E+00,3.0839E+00,
	3.0829E+00,3.0817E+00,3.0805E+00,3.0791E+00,3.0777E+00,3.0762E+00,3.0747E+00,3.0731E+00,
	3.0715E+00,3.0700E+00,3.0684E+00,3.0668E+00,3.0653E+00,3.0638E+00,3.0624E+00,3.0611E+00,
	3.0598E+00,3.0587E+00,3.0577E+00,3.0568E+00,3.0560E+00,3.0554E+00,3.0549E+00,3.0546E+00,
	3.0544E+00,3.0542E+00,3.0538E+00,3.0532E+00,3.0522E+00,3.0509E+00,3.0494E+00,3.0476E+00,
	3.0456E+00,3.0433E+00,3.0409E+00,3.0383E+00,3.0355E+00,3.0326E+00,3.0296E+00,3.0265E+00,
	3.0234E+00,3.0203E+00,3.0172E+00,3.0142E+00,3.0112E+00,3.0084E+00,3.0056E+00,3.0030E+00,
	3.0006E+00,2.9984E+00,2.9965E+00,2.9947E+00,2.9932E+00,2.9920E+00,2.9911E+00,2.9904E+00,
	2.9900E+00,2.9898E+00,2.9890E+00,2.9877E+00,2.9858E+00,2.9834E+00,2.9804E+00,2.9770E+00,
	2.9731E+00,2.9687E+00,2.9640E+00,2.9590E+00,2.9536E+00,2.9481E+00,2.9424E+00,2.9365E+00,
	2.9306E+00,2.9247E+00,2.9188E+00,2.9130E+00,2.9074E+00,2.9020E+00,2.8968E+00,2.8919E+00,
	2.8874E+00,2.8833E+00,2.8795E+00,2.8763E+00,2.8735E+00,2.8712E+00,2.8694E+00,2.8682E+00,
	2.8675E+00,2.8670E+00,2.8656E+00,2.8631E+00,2.8596E+00,2.8550E+00,2.8495E+00,2.8431E+00,
	2.8358E+00,2.8277E+00,2.8190E+00,2.8097E+00,2.7998E+00,2.7896E+00,2.7791E+00,2.7684E+00,
	2.7576E+00,2.7469E+00,2.7362E+00,2.7258E+00,2.7156E+00,2.7059E+00,2.6966E+00,2.6879E+00,
	2.6799E+00,2.6725E+00,2.6659E+00,2.6600E+00,2.6551E+00,2.6510E+00,2.6479E+00,2.6457E+00,
	2.6445E+00,2.6436E+00,2.6412E+00,2.6368E+00,2.6306E+00,2.6226E+00,2.6129E+00,2.6017E+00,
	2.5890E+00,2.5750E+00,2.5599E+00,2.5439E+00,2.5271E+00,2.5097E+00,2.4919E+00,2.4739E+00,
	2.4558E+00,2.4379E+00,2.4202E+00,2.4030E+00,2.3864E+00,2.3705E+00,2.3554E+00,2.3413E+00,
	2.3283E+00,2.3165E+00,2.3059E+00,2.2966E+00,2.2887E+00,2.2823E+00,2.2773E+00,2.2738E+00,
	2.2719E+00,2.2705E+00,2.2667E+00,2.2598E+00,2.2501E+00,2.2375E+00,2.2224E+00,2.2050E+00,
	2.1854E+00,2.1640E+00,2.1411E+00,2.1169E+00,2.0918E+00,2.0660E+00,2.0398E+00,2.0136E+00,
	1.9875E+00,1.9618E+00,1.9367E+00,1.9125E+00,1.8893E+00,1.8673E+00,1.8466E+00,1.8274E+00,
	1.8098E+00,1.7939E+00,1.7797E+00,1.7674E+00,1.7569E+00,1.7484E+00,1.7419E+00,1.7373E+00,
	1.7348E+00,1.7330E+00,1.7280E+00,1.7190E+00,1.7063E+00,1.6901E+00,1.6707E+00,1.6485E+00,
	1.6239E+00,1.5972E+00,1.5688E+00,1.5394E+00,1.5091E+00,1.4784E+00,1.4477E+00,1.4173E+00,
	1.3874E+00,1.3584E+00,1.3305E+00,1.3039E+00,1.2787E+00,1.2551E+00,1.2331E+00,1.2129E+00,
	1.1946E+00,1.1781E+00,1.1636E+00,1.1511E+00,1.1405E+00,1.1319E+00,1.1254E+00,1.1208E+00,
	1.1183E+00,1.1165E+00,1.1115E+00,1.1026E+00,1.0901E+00,1.0742E+00,1.0554E+00,1.0341E+00,
	1.0107E+00,9.8567E-01,9.5955E-01,9.3272E-01,9.0561E-01,8.7856E-01,8.5191E-01,8.2592E-01,
	8.0083E-01,7.7682E-01,7.5340E-01,7.2902E-01,7.0636E-01,6.8547E-01,6.6636E-01,6.4905E-01,
	6.3353E-01,6.1978E-01,6.0778E-01,5.9749E-01,5.8891E-01,5.8199E-01,5.7673E-01,5.7309E-01,
	5.7108E-01,5.6966E-01,5.6568E-01,5.5866E-01,5.4883E-01,5.3652E-01,5.2210E-01,5.0599E-01,
	4.8861E-01,4.7037E-01,4.5166E-01,4.3283E-01,4.1418E-01,3.9597E-01,3.7838E-01,3.6158E-01,
	3.4568E-01,3.3076E-01,3.1667E-01,3.0291E-01,2.9033E-01,2.7891E-01,2.6861E-01,2.5939E-01,
	2.5122E-01,2.4406E-01,2.3786E-01,2.3260E-01,2.2824E-01,2.2474E-01,2.2209E-01,2.2027E-01,
	2.1926E-01,2.1855E-01,2.1656E-01,2.1307E-01,2.0822E-01,2.0219E-01,1.9520E-01,1.8748E-01,
	1.7925E-01,1.7073E-01,1.6180E-01,1.5294E-01,1.4432E-01,1.3604E-01,1.2820E-01,1.2083E-01,
	1.1398E-01,1.0765E-01,1.0182E-01,9.6422E-02,9.1542E-02,8.7159E-02,8.3248E-02,7.9783E-02,
	7.6737E-02,7.4087E-02,7.1810E-02,6.9887E-02,6.8300E-02,6.6999E-02,6.6009E-02,6.5329E-02,
	6.4954E-02,6.4691E-02,6.3954E-02,6.2666E-02,6.0884E-02,5.8688E-02,5.6167E-02,5.3412E-02,
	5.0515E-02,4.7555E-02,4.4605E-02,4.1720E-02,3.8947E-02,3.6316E-02,3.3849E-02,3.1558E-02,
	2.9471E-02,2.7628E-02,2.5945E-02,2.4418E-02,2.3038E-02,2.1800E-02,2.0696E-02,1.9718E-02,
	1.8858E-02,1.8109E-02,1.7467E-02,1.6923E-02,1.6475E-02,1.6110E-02,1.5834E-02,1.5645E-02,
	1.5540E-02,1.5468E-02,1.5263E-02,1.4907E-02,1.4416E-02,1.3815E-02,1.3129E-02,1.2386E-02,
	1.1610E-02,1.0826E-02,1.0051E-02,9.3019E-03,8.5887E-03,7.9192E-03,7.2981E-03,6.7272E-03,
	6.2055E-03,5.7311E-03,5.3067E-03,4.9288E-03,4.5941E-03,4.2990E-03,4.0402E-03,3.8145E-03,
	3.6190E-03,3.4510E-03,3.3084E-03,3.1892E-03,3.0916E-03,3.0140E-03,2.9557E-03,2.9159E-03,
	2.8940E-03,2.8787E-03,2.8358E-03,2.7611E-03,2.6587E-03,2.5337E-03,2.3919E-03,2.2392E-03,
	2.0810E-03,1.9222E-03,1.7667E-03,1.6180E-03,1.4782E-03,1.3485E-03,1.2295E-03,1.1214E-03,
	1.0247E-03,9.4005E-04,8.6451E-04,7.9743E-04,7.3814E-04,6.8602E-04,6.4118E-04,6.0221E-04,
	5.6855E-04,5.3973E-04,5.1532E-04,4.9495E-04,4.7831E-04,4.6533E-04,4.5565E-04,4.4903E-04,
	4.4539E-04,4.4285E-04,4.3572E-04,4.2334E-04,4.0637E-04,3.8568E-04,3.6226E-04,3.3703E-04,
	3.1073E-04,2.8453E-04,2.5909E-04,2.5459E-04},
	/*8 111*/
	{
	3.7493E+00,3.7492E+00,3.7489E+00,3.7483E+00,3.7475E+00,3.7465E+00,3.7452E+00,3.7437E+00,
	3.7420E+00,3.7402E+00,3.7381E+00,3.7359E+00,3.7336E+00,3.7312E+00,3.7287E+00,3.7262E+00,
	3.7236E+00,3.7210E+00,3.7184E+00,3.7159E+00,3.7134E+00,3.7110E+00,3.7087E+00,3.7066E+00,
	3.7046E+00,3.7027E+00,3.7011E+00,3.6996E+00,3.6983E+00,3.6973E+00,3.6965E+00,3.6960E+00,
	3.6957E+00,3.6955E+00,3.6952E+00,3.6947E+00,3.6939E+00,3.6928E+00,3.6916E+00,3.6901E+00,
	3.6885E+00,3.6867E+00,3.6847E+00,3.6825E+00,3.6803E+00,3.6779E+00,3.6755E+00,3.6730E+00,
	3.6705E+00,3.6679E+00,3.6654E+00,3.6629E+00,3.6605E+00,3.6581E+00,3.6559E+00,3.6538E+00,
	3.6518E+00,3.6500E+00,3.6484E+00,3.6470E+00,3.6458E+00,3.6448E+00,3.6440E+00,3.6434E+00,
	3.6431E+00,3.6429E+00,3.6423E+00,3.6412E+00,3.6397E+00,3.6377E+00,3.6352E+00,3.6324E+00,
	3.6292E+00,3.6256E+00,3.6217E+00,3.6176E+00,3.6132E+00,3.6086E+00,3.6039E+00,3.5990E+00,
	3.5941E+00,3.5892E+00,3.5843E+00,3.5795E+00,3.5749E+00,3.5703E+00,3.5660E+00,3.5620E+00,
	3.5582E+00,3.5547E+00,3.5516E+00,3.5489E+00,3.5465E+00,3.5446E+00,3.5431E+00,3.5421E+00,
	3.5415E+00,3.5411E+00,3.5399E+00,3.5379E+00,3.5349E+00,3.5311E+00,3.5264E+00,3.5210E+00,
	3.5149E+00,3.5081E+00,3.5007E+00,3.4928E+00,3.4845E+00,3.4758E+00,3.4669E+00,3.4578E+00,
	3.4486E+00,3.4393E+00,3.4302E+00,3.4212E+00,3.4125E+00,3.4041E+00,3.3961E+00,3.3886E+00,
	3.3816E+00,3.3752E+00,3.3694E+00,3.3644E+00,3.3601E+00,3.3565E+00,3.3538E+00,3.3519E+00,
	3.3508E+00,3.3501E+00,3.3479E+00,3.3441E+00,3.3387E+00,3.3317E+00,3.3232E+00,3.3133E+00,
	3.3021E+00,3.2898E+00,3.2765E+00,3.2622E+00,3.2473E+00,3.2317E+00,3.2158E+00,3.1995E+00,
	3.1832E+00,3.1670E+00,3.1509E+00,3.1352E+00,3.1199E+00,3.1053E+00,3.0915E+00,3.0784E+00,
	3.0664E+00,3.0554E+00,3.0455E+00,3.0369E+00,3.0295E+00,3.0235E+00,3.0188E+00,3.0156E+00,
	3.0137E+00,3.0125E+00,3.0089E+00,3.0024E+00,2.9932E+00,2.9814E+00,2.9671E+00,2.9505E+00,
	2.9318E+00,2.9113E+00,2.8893E+00,2.8659E+00,2.8414E+00,2.8162E+00,2.7905E+00,2.7645E+00,
	2.7385E+00,2.7128E+00,2.6876E+00,2.6631E+00,2.6394E+00,2.6169E+00,2.5957E+00,2.5759E+00,
	2.5576E+00,2.5410E+00,2.5262E+00,2.5133E+00,2.5023E+00,2.4933E+00,2.4864E+00,2.4816E+00,
	2.4789E+00,2.4770E+00,2.4717E+00,2.4622E+00,2.4487E+00,2.4313E+00,2.4105E+00,2.3866E+00,
	2.3598E+00,2.3307E+00,2.2996E+00,2.2669E+00,2.2331E+00,2.1985E+00,2.1637E+00,2.1289E+00,
	2.0944E+00,2.0607E+00,2.0280E+00,1.9965E+00,1.9665E+00,1.9382E+00,1.9117E+00,1.8872E+00,
	1.8648E+00,1.8446E+00,1.8267E+00,1.8112E+00,1.7980E+00,1.7874E+00,1.7792E+00,1.7735E+00,
	1.7703E+00,1.7681E+00,1.7618E+00,1.7506E+00,1.7348E+00,1.7147E+00,1.6907E+00,1.6634E+00,
	1.6332E+00,1.6007E+00,1.5664E+00,1.5309E+00,1.4947E+00,1.4582E+00,1.4220E+00,1.3863E+00,
	1.3516E+00,1.3181E+00,1.2860E+00,1.2556E+00,1.2270E+00,1.2003E+00,1.1757E+00,1.1531E+00,
	1.1327E+00,1.1145E+00,1.0985E+00,1.0847E+00,1.0731E+00,1.0637E+00,1.0566E+00,1.0516E+00,
	1.0489E+00,1.0469E+00,1.0415E+00,1.0318E+00,1.0182E+00,1.0011E+00,9.8090E-01,9.5808E-01,
	9.3320E-01,9.0678E-01,8.7934E-01,8.5136E-01,8.2328E-01,7.9546E-01,7.6824E-01,7.4189E-01,
	7.1662E-01,6.9260E-01,6.6934E-01,6.4537E-01,6.2322E-01,6.0291E-01,5.8444E-01,5.6778E-01,
	5.5291E-01,5.3979E-01,5.2837E-01,5.1862E-01,5.1050E-01,5.0398E-01,4.9902E-01,4.9560E-01,
	4.9370E-01,4.9238E-01,4.8864E-01,4.8205E-01,4.7286E-01,4.6139E-01,4.4800E-01,4.3310E-01,
	4.1710E-01,4.0041E-01,3.8338E-01,3.6634E-01,3.4955E-01,3.3325E-01,3.1760E-01,3.0273E-01,
	2.8873E-01,2.7566E-01,2.6323E-01,2.5063E-01,2.3919E-01,2.2885E-01,2.1959E-01,2.1134E-01,
	2.0407E-01,1.9772E-01,1.9225E-01,1.8761E-01,1.8378E-01,1.8072E-01,1.7840E-01,1.7681E-01,
	1.7593E-01,1.7531E-01,1.7358E-01,1.7054E-01,1.6633E-01,1.6111E-01,1.5509E-01,1.4847E-01,
	1.4145E-01,1.3422E-01,1.2596E-01,1.1780E-01,1.0997E-01,1.0256E-01,9.5629E-02,8.9215E-02,
	8.3326E-02,7.7960E-02,7.3244E-02,6.9515E-02,6.6150E-02,6.3130E-02,6.0438E-02,5.8055E-02,
	5.5961E-02,5.4140E-02,5.2575E-02,5.1254E-02,5.0164E-02,4.9196E-02,4.8439E-02,4.7921E-02,
	4.7634E-02,4.7434E-02,4.6872E-02,4.5891E-02,4.4537E-02,4.2870E-02,4.0961E-02,3.8881E-02,
	3.6698E-02,3.4476E-02,3.2267E-02,3.0115E-02,2.8052E-02,2.6101E-02,2.4277E-02,2.2588E-02,
	2.1055E-02,1.9709E-02,1.8482E-02,1.7370E-02,1.6368E-02,1.5470E-02,1.4670E-02,1.3962E-02,
	1.3340E-02,1.2800E-02,1.2335E-02,1.1944E-02,1.1620E-02,1.1358E-02,1.1159E-02,1.1023E-02,
	1.0948E-02,1.0896E-02,1.0749E-02,1.0493E-02,1.0142E-02,9.7109E-03,9.2203E-03,8.6893E-03,
	8.1364E-03,7.5780E-03,7.0278E-03,6.4963E-03,5.9915E-03,5.5186E-03,5.0805E-03,4.6785E-03,
	4.3116E-03,3.9776E-03,3.6792E-03,3.4141E-03,3.1795E-03,2.9730E-03,2.7920E-03,2.6344E-03,
	2.4980E-03,2.3809E-03,2.2815E-03,2.1985E-03,2.1306E-03,2.0767E-03,2.0362E-03,2.0086E-03,
	1.9933E-03,1.9827E-03,1.9529E-03,1.9011E-03,1.8301E-03,1.7434E-03,1.6451E-03,1.5394E-03,
	1.4300E-03,1.3202E-03,1.2129E-03,1.1100E-03,1.0133E-03,9.2373E-04,8.4163E-04,7.6712E-04,
	7.0051E-04,6.4235E-04,5.9047E-04,5.4443E-04,5.0377E-04,4.6803E-04,4.3725E-04,4.1052E-04,
	3.8744E-04,3.6769E-04,3.5097E-04,3.3702E-04,3.2563E-04,3.1675E-04,3.1012E-04,3.0559E-04,
	3.0310E-04,3.0136E-04,2.9649E-04,2.8801E-04,2.7641E-04,2.6227E-04,2.4627E-04,2.2904E-04,
	2.1109E-04,1.9323E-04,1.7589E-04,1.7336E-04},
	/*9 111*/
	{
	4.4150E+00,4.4149E+00,4.4144E+00,4.4136E+00,4.4124E+00,4.4108E+00,4.4088E+00,4.4066E+00,
	4.4040E+00,4.4012E+00,4.3981E+00,4.3947E+00,4.3912E+00,4.3876E+00,4.3838E+00,4.3799E+00,
	4.3760E+00,4.3721E+00,4.3681E+00,4.3643E+00,4.3605E+00,4.3569E+00,4.3535E+00,4.3502E+00,
	4.3471E+00,4.3444E+00,4.3418E+00,4.3396E+00,4.3377E+00,4.3362E+00,4.3350E+00,4.3342E+00,
	4.3337E+00,4.3335E+00,4.3330E+00,4.3322E+00,4.3310E+00,4.3294E+00,4.3275E+00,4.3253E+00,
	4.3228E+00,4.3201E+00,4.3171E+00,4.3138E+00,4.3104E+00,4.3069E+00,4.3032E+00,4.2994E+00,
	4.2956E+00,4.2918E+00,4.2880E+00,4.2842E+00,4.2806E+00,4.2771E+00,4.2737E+00,4.2705E+00,
	4.2676E+00,4.2648E+00,4.2624E+00,4.2603E+00,4.2584E+00,4.2569E+00,4.2557E+00,4.2549E+00,
	4.2545E+00,4.2542E+00,4.2532E+00,4.2516E+00,4.2493E+00,4.2463E+00,4.2426E+00,4.2384E+00,
	4.2335E+00,4.2282E+00,4.2223E+00,4.2161E+00,4.2095E+00,4.2027E+00,4.1956E+00,4.1883E+00,
	4.1810E+00,4.1737E+00,4.1664E+00,4.1592E+00,4.1522E+00,4.1455E+00,4.1391E+00,4.1330E+00,
	4.1274E+00,4.1223E+00,4.1176E+00,4.1135E+00,4.1101E+00,4.1072E+00,4.1050E+00,4.1034E+00,
	4.1026E+00,4.1020E+00,4.1003E+00,4.0972E+00,4.0928E+00,4.0871E+00,4.0802E+00,4.0722E+00,
	4.0631E+00,4.0530E+00,4.0421E+00,4.0305E+00,4.0182E+00,4.0054E+00,3.9922E+00,3.9788E+00,
	3.9653E+00,3.9517E+00,3.9383E+00,3.9252E+00,3.9124E+00,3.9001E+00,3.8884E+00,3.8775E+00,
	3.8673E+00,3.8579E+00,3.8495E+00,3.8422E+00,3.8359E+00,3.8308E+00,3.8268E+00,3.8240E+00,
	3.8225E+00,3.8214E+00,3.8183E+00,3.8128E+00,3.8049E+00,3.7947E+00,3.7824E+00,3.7681E+00,
	3.7520E+00,3.7342E+00,3.7150E+00,3.6945E+00,3.6730E+00,3.6507E+00,3.6279E+00,3.6048E+00,
	3.5815E+00,3.5584E+00,3.5356E+00,3.5134E+00,3.4918E+00,3.4712E+00,3.4517E+00,3.4334E+00,
	3.4165E+00,3.4011E+00,3.3872E+00,3.3752E+00,3.3649E+00,3.3564E+00,3.3500E+00,3.3454E+00,
	3.3429E+00,3.3411E+00,3.3361E+00,3.3271E+00,3.3143E+00,3.2979E+00,3.2781E+00,3.2551E+00,
	3.2294E+00,3.2012E+00,3.1709E+00,3.1389E+00,3.1055E+00,3.0712E+00,3.0363E+00,3.0012E+00,
	2.9662E+00,2.9317E+00,2.8980E+00,2.8653E+00,2.8339E+00,2.8041E+00,2.7760E+00,2.7499E+00,
	2.7259E+00,2.7042E+00,2.6848E+00,2.6679E+00,2.6536E+00,2.6420E+00,2.6330E+00,2.6267E+00,
	2.6233E+00,2.6208E+00,2.6139E+00,2.6016E+00,2.5841E+00,2.5617E+00,2.5350E+00,2.5042E+00,
	2.4700E+00,2.4329E+00,2.3934E+00,2.3522E+00,2.3097E+00,2.2665E+00,2.2232E+00,2.1801E+00,
	2.1377E+00,2.0965E+00,2.0566E+00,2.0184E+00,1.9822E+00,1.9482E+00,1.9165E+00,1.8873E+00,
	1.8607E+00,1.8368E+00,1.8157E+00,1.7974E+00,1.7820E+00,1.7695E+00,1.7599E+00,1.7533E+00,
	1.7496E+00,1.7470E+00,1.7396E+00,1.7266E+00,1.7083E+00,1.6850E+00,1.6573E+00,1.6259E+00,
	1.5913E+00,1.5543E+00,1.5155E+00,1.4755E+00,1.4351E+00,1.3946E+00,1.3546E+00,1.3155E+00,
	1.2776E+00,1.2413E+00,1.2068E+00,1.1742E+00,1.1438E+00,1.1156E+00,1.0896E+00,1.0659E+00,
	1.0446E+00,1.0257E+00,1.0091E+00,9.9478E-01,9.8283E-01,9.7317E-01,9.6581E-01,9.6072E-01,
	9.5789E-01,9.5591E-01,9.5032E-01,9.4044E-01,9.2659E-01,9.0916E-01,8.8867E-01,8.6564E-01,
	8.4066E-01,8.1427E-01,7.8702E-01,7.5940E-01,7.3184E-01,7.0471E-01,6.7832E-01,6.5292E-01,
	6.2871E-01,6.0582E-01,5.8353E-01,5.5979E-01,5.3800E-01,5.1815E-01,5.0020E-01,4.8411E-01,
	4.6981E-01,4.5725E-01,4.4637E-01,4.3711E-01,4.2942E-01,4.2326E-01,4.1858E-01,4.1536E-01,
	4.1358E-01,4.1233E-01,4.0882E-01,4.0265E-01,3.9405E-01,3.8336E-01,3.7094E-01,3.5720E-01,
	3.4253E-01,3.2731E-01,3.1189E-01,2.9656E-01,2.8157E-01,2.6711E-01,2.5333E-01,2.4032E-01,
	2.2816E-01,2.1687E-01,2.0628E-01,1.9593E-01,1.8655E-01,1.7812E-01,1.7059E-01,1.6390E-01,
	1.5802E-01,1.5290E-01,1.4850E-01,1.4478E-01,1.4170E-01,1.3925E-01,1.3740E-01,1.3612E-01,
	1.3542E-01,1.3493E-01,1.3354E-01,1.3112E-01,1.2777E-01,1.2362E-01,1.1884E-01,1.1360E-01,
	1.0806E-01,1.0237E-01,9.6367E-02,9.0462E-02,8.4776E-02,7.9378E-02,7.4311E-02,6.9602E-02,
	6.5261E-02,6.1290E-02,5.7674E-02,5.4385E-02,5.1432E-02,4.8796E-02,4.6457E-02,4.4395E-02,
	4.2591E-02,4.1027E-02,3.9689E-02,3.8561E-02,3.7633E-02,3.6869E-02,3.6285E-02,3.5886E-02,
	3.5666E-02,3.5512E-02,3.5079E-02,3.4325E-02,3.3284E-02,3.2006E-02,3.0544E-02,2.8955E-02,
	2.7290E-02,2.5600E-02,2.3924E-02,2.2294E-02,2.0736E-02,1.9266E-02,1.7895E-02,1.6629E-02,
	1.5481E-02,1.4476E-02,1.3562E-02,1.2734E-02,1.1990E-02,1.1323E-02,1.0731E-02,1.0206E-02,
	9.7468E-03,9.3475E-03,9.0049E-03,8.7158E-03,8.4774E-03,8.2833E-03,8.1362E-03,8.0356E-03,
	7.9802E-03,7.9415E-03,7.8329E-03,7.6438E-03,7.3838E-03,7.0658E-03,6.7040E-03,6.3128E-03,
	5.9061E-03,5.4958E-03,5.0921E-03,4.7027E-03,4.3334E-03,3.9879E-03,3.6682E-03,3.3753E-03,
	3.1082E-03,2.8652E-03,2.6484E-03,2.4559E-03,2.2857E-03,2.1361E-03,2.0051E-03,1.8910E-03,
	1.7924E-03,1.7077E-03,1.6360E-03,1.5760E-03,1.5270E-03,1.4882E-03,1.4590E-03,1.4390E-03,
	1.4280E-03,1.4204E-03,1.3989E-03,1.3615E-03,1.3104E-03,1.2479E-03,1.1772E-03,1.1011E-03,
	1.0224E-03,9.4358E-04,8.6648E-04,7.9264E-04,7.2326E-04,6.5900E-04,6.0016E-04,5.4680E-04,
	4.9913E-04,4.5751E-04,4.2041E-04,3.8750E-04,3.5845E-04,3.3294E-04,3.1096E-04,2.9188E-04,
	2.7542E-04,2.6133E-04,2.4941E-04,2.3946E-04,2.3135E-04,2.2501E-04,2.2029E-04,2.1706E-04,
	2.1528E-04,2.1404E-04,2.1057E-04,2.0453E-04,1.9626E-04,1.8619E-04,1.7479E-04,1.6253E-04,
	1.4977E-04,1.3708E-04,1.2476E-04,1.2305E-04},
	/*10 111*/
	{
	5.1196E+00,5.1195E+00,5.1188E+00,5.1175E+00,5.1157E+00,5.1134E+00,5.1106E+00,5.1073E+00,
	5.1035E+00,5.0994E+00,5.0949E+00,5.0900E+00,5.0849E+00,5.0796E+00,5.0741E+00,5.0684E+00,
	5.0627E+00,5.0570E+00,5.0513E+00,5.0457E+00,5.0403E+00,5.0350E+00,5.0300E+00,5.0252E+00,
	5.0208E+00,5.0168E+00,5.0131E+00,5.0099E+00,5.0072E+00,5.0049E+00,5.0032E+00,5.0020E+00,
	5.0013E+00,5.0010E+00,5.0003E+00,4.9991E+00,4.9974E+00,4.9951E+00,4.9924E+00,4.9892E+00,
	4.9856E+00,4.9816E+00,4.9772E+00,4.9726E+00,4.9676E+00,4.9625E+00,4.9572E+00,4.9517E+00,
	4.9462E+00,4.9407E+00,4.9352E+00,4.9298E+00,4.9245E+00,4.9194E+00,4.9146E+00,4.9100E+00,
	4.9057E+00,4.9018E+00,4.8983E+00,4.8952E+00,4.8926E+00,4.8904E+00,4.8887E+00,4.8875E+00,
	4.8869E+00,4.8864E+00,4.8851E+00,4.8828E+00,4.8794E+00,4.8751E+00,4.8698E+00,4.8637E+00,
	4.8567E+00,4.8491E+00,4.8407E+00,4.8317E+00,4.8223E+00,4.8125E+00,4.8023E+00,4.7919E+00,
	4.7814E+00,4.7709E+00,4.7605E+00,4.7503E+00,4.7403E+00,4.7307E+00,4.7216E+00,4.7130E+00,
	4.7050E+00,4.6976E+00,4.6910E+00,4.6852E+00,4.6803E+00,4.6762E+00,4.6731E+00,4.6709E+00,
	4.6697E+00,4.6688E+00,4.6663E+00,4.6620E+00,4.6557E+00,4.6477E+00,4.6379E+00,4.6265E+00,
	4.6136E+00,4.5994E+00,4.5840E+00,4.5675E+00,4.5502E+00,4.5321E+00,4.5136E+00,4.4947E+00,
	4.4757E+00,4.4568E+00,4.4380E+00,4.4196E+00,4.4017E+00,4.3846E+00,4.3683E+00,4.3530E+00,
	4.3388E+00,4.3258E+00,4.3142E+00,4.3040E+00,4.2953E+00,4.2881E+00,4.2826E+00,4.2788E+00,
	4.2766E+00,4.2751E+00,4.2708E+00,4.2632E+00,4.2523E+00,4.2382E+00,4.2212E+00,4.2015E+00,
	4.1793E+00,4.1548E+00,4.1284E+00,4.1003E+00,4.0710E+00,4.0406E+00,4.0095E+00,3.9780E+00,
	3.9465E+00,3.9152E+00,3.8844E+00,3.8544E+00,3.8255E+00,3.7978E+00,3.7717E+00,3.7472E+00,
	3.7246E+00,3.7040E+00,3.6857E+00,3.6696E+00,3.6559E+00,3.6448E+00,3.6362E+00,3.6302E+00,
	3.6268E+00,3.6245E+00,3.6178E+00,3.6059E+00,3.5890E+00,3.5673E+00,3.5412E+00,3.5111E+00,
	3.4773E+00,3.4404E+00,3.4009E+00,3.3593E+00,3.3161E+00,3.2718E+00,3.2269E+00,3.1819E+00,
	3.1372E+00,3.0933E+00,3.0505E+00,3.0092E+00,2.9697E+00,2.9322E+00,2.8971E+00,2.8645E+00,
	2.8346E+00,2.8076E+00,2.7837E+00,2.7628E+00,2.7451E+00,2.7307E+00,2.7197E+00,2.7120E+00,
	2.7077E+00,2.7047E+00,2.6962E+00,2.6811E+00,2.6597E+00,2.6324E+00,2.5997E+00,2.5623E+00,
	2.5209E+00,2.4762E+00,2.4288E+00,2.3795E+00,2.3290E+00,2.2779E+00,2.2268E+00,2.1764E+00,
	2.1270E+00,2.0791E+00,2.0332E+00,1.9893E+00,1.9480E+00,1.9093E+00,1.8734E+00,1.8404E+00,
	1.8105E+00,1.7837E+00,1.7601E+00,1.7397E+00,1.7226E+00,1.7087E+00,1.6981E+00,1.6907E+00,
	1.6866E+00,1.6837E+00,1.6756E+00,1.6613E+00,1.6410E+00,1.6154E+00,1.5852E+00,1.5509E+00,
	1.5134E+00,1.4734E+00,1.4317E+00,1.3891E+00,1.3461E+00,1.3033E+00,1.2613E+00,1.2205E+00,
	1.1813E+00,1.1438E+00,1.1084E+00,1.0752E+00,1.0443E+00,1.0158E+00,9.8965E-01,9.6596E-01,
	9.4470E-01,9.2584E-01,9.0936E-01,8.9524E-01,8.8343E-01,8.7392E-01,8.6668E-01,8.6168E-01,
	8.5890E-01,8.5696E-01,8.5147E-01,8.4179E-01,8.2825E-01,8.1126E-01,7.9134E-01,7.6905E-01,
	7.4497E-01,7.1965E-01,6.9364E-01,6.6740E-01,6.4135E-01,6.1584E-01,5.9116E-01,5.6752E-01,
	5.4509E-01,5.2398E-01,5.0342E-01,4.8124E-01,4.6099E-01,4.4264E-01,4.2611E-01,4.1135E-01,
	3.9829E-01,3.8685E-01,3.7697E-01,3.6858E-01,3.6163E-01,3.5607E-01,3.5186E-01,3.4896E-01,
	3.4736E-01,3.4624E-01,3.4308E-01,3.3754E-01,3.2985E-01,3.2030E-01,3.0924E-01,2.9705E-01,
	2.8408E-01,2.7070E-01,2.5719E-01,2.4383E-01,2.3082E-01,2.1833E-01,2.0649E-01,1.9535E-01,
	1.8499E-01,1.7541E-01,1.6645E-01,1.5770E-01,1.4981E-01,1.4273E-01,1.3643E-01,1.3086E-01,
	1.2596E-01,1.2171E-01,1.1807E-01,1.1499E-01,1.1245E-01,1.1043E-01,1.0890E-01,1.0785E-01,
	1.0727E-01,1.0687E-01,1.0573E-01,1.0374E-01,1.0099E-01,9.7591E-02,9.3685E-02,8.9411E-02,
	8.4907E-02,8.0297E-02,7.5475E-02,7.0745E-02,6.6203E-02,6.1901E-02,5.7873E-02,5.4138E-02,
	5.0703E-02,4.7566E-02,4.4707E-02,4.2084E-02,3.9735E-02,3.7642E-02,3.5788E-02,3.4157E-02,
	3.2732E-02,3.1499E-02,3.0444E-02,2.9557E-02,2.8827E-02,2.8228E-02,2.7773E-02,2.7461E-02,
	2.7289E-02,2.7168E-02,2.6831E-02,2.6242E-02,2.5431E-02,2.4436E-02,2.3299E-02,2.2065E-02,
	2.0775E-02,1.9467E-02,1.8172E-02,1.6915E-02,1.5716E-02,1.4587E-02,1.3535E-02,1.2565E-02,
	1.1687E-02,1.0917E-02,1.0217E-02,9.5848E-03,9.0169E-03,8.5092E-03,8.0581E-03,7.6597E-03,
	7.3107E-03,7.0078E-03,6.7481E-03,6.5291E-03,6.3486E-03,6.2016E-03,6.0903E-03,6.0142E-03,
	5.9722E-03,5.9429E-03,5.8608E-03,5.7178E-03,5.5213E-03,5.2811E-03,5.0080E-03,4.7130E-03,
	4.4065E-03,4.0977E-03,3.7941E-03,3.5017E-03,3.2245E-03,2.9655E-03,2.7261E-03,2.5070E-03,
	2.3074E-03,2.1261E-03,1.9644E-03,1.8209E-03,1.6942E-03,1.5828E-03,1.4853E-03,1.4004E-03,
	1.3271E-03,1.2642E-03,1.2109E-03,1.1664E-03,1.1300E-03,1.1011E-03,1.0794E-03,1.0646E-03,
	1.0564E-03,1.0507E-03,1.0348E-03,1.0070E-03,9.6897E-04,9.2260E-04,8.7009E-04,8.1363E-04,
	7.5526E-04,6.9679E-04,6.3965E-04,5.8494E-04,5.3353E-04,4.8595E-04,4.4241E-04,4.0294E-04,
	3.6769E-04,3.3691E-04,3.0949E-04,2.8518E-04,2.6373E-04,2.4489E-04,2.2870E-04,2.1463E-04,
	2.0250E-04,1.9212E-04,1.8334E-04,1.7602E-04,1.7004E-04,1.6538E-04,1.6190E-04,1.5952E-04,
	1.5822E-04,1.5730E-04,1.5475E-04,1.5031E-04,1.4422E-04,1.3681E-04,1.2843E-04,1.1941E-04,
	1.1001E-04,1.0065E-04,9.1587E-05,9.0447E-05}};

double TripD[8][460] ={
	/*3 212*/
	{
	7.1970E-01,7.1346E-01,7.0920E-01,7.0587E-01,7.0302E-01,7.0051E-01,6.9823E-01,6.9615E-01,
	6.9422E-01,6.9243E-01,6.9076E-01,6.8920E-01,6.8773E-01,6.8636E-01,6.8506E-01,6.8385E-01,
	6.8272E-01,6.8166E-01,6.8067E-01,6.7975E-01,6.7890E-01,6.7812E-01,6.7740E-01,6.7675E-01,
	6.7616E-01,6.7564E-01,6.7518E-01,6.7479E-01,6.7446E-01,6.7419E-01,6.7398E-01,6.7384E-01,
	6.7376E-01,6.7373E-01,6.7365E-01,6.7351E-01,6.7331E-01,6.7306E-01,6.7275E-01,6.7240E-01,
	6.7201E-01,6.7159E-01,6.7114E-01,6.7066E-01,6.7017E-01,6.6967E-01,6.6916E-01,6.6866E-01,
	6.6815E-01,6.6766E-01,6.6718E-01,6.6671E-01,6.6626E-01,6.6584E-01,6.6544E-01,6.6507E-01,
	6.6473E-01,6.6442E-01,6.6415E-01,6.6391E-01,6.6371E-01,6.6354E-01,6.6341E-01,6.6333E-01,
	6.6328E-01,6.6324E-01,6.6314E-01,6.6297E-01,6.6272E-01,6.6240E-01,6.6202E-01,6.6158E-01,
	6.6109E-01,6.6055E-01,6.5998E-01,6.5938E-01,6.5876E-01,6.5812E-01,6.5748E-01,6.5683E-01,
	6.5619E-01,6.5556E-01,6.5494E-01,6.5435E-01,6.5377E-01,6.5323E-01,6.5272E-01,6.5224E-01,
	6.5180E-01,6.5140E-01,6.5105E-01,6.5074E-01,6.5047E-01,6.5026E-01,6.5009E-01,6.4998E-01,
	6.4991E-01,6.4987E-01,6.4974E-01,6.4951E-01,6.4919E-01,6.4878E-01,6.4828E-01,6.4770E-01,
	6.4706E-01,6.4636E-01,6.4561E-01,6.4482E-01,6.4401E-01,6.4316E-01,6.4231E-01,6.4145E-01,
	6.4060E-01,6.3975E-01,6.3893E-01,6.3813E-01,6.3736E-01,6.3663E-01,6.3594E-01,6.3529E-01,
	6.3470E-01,6.3416E-01,6.3368E-01,6.3325E-01,6.3290E-01,6.3260E-01,6.3238E-01,6.3222E-01,
	6.3214E-01,6.3207E-01,6.3190E-01,6.3159E-01,6.3115E-01,6.3058E-01,6.2990E-01,6.2911E-01,
	6.2823E-01,6.2726E-01,6.2622E-01,6.2513E-01,6.2399E-01,6.2282E-01,6.2162E-01,6.2042E-01,
	6.1921E-01,6.1802E-01,6.1686E-01,6.1572E-01,6.1463E-01,6.1358E-01,6.1259E-01,6.1167E-01,
	6.1082E-01,6.1004E-01,6.0935E-01,6.0874E-01,6.0823E-01,6.0780E-01,6.0748E-01,6.0725E-01,
	6.0713E-01,6.0704E-01,6.0678E-01,6.0633E-01,6.0569E-01,6.0487E-01,6.0388E-01,6.0273E-01,
	6.0144E-01,6.0003E-01,5.9851E-01,5.9689E-01,5.9521E-01,5.9348E-01,5.9170E-01,5.8991E-01,
	5.8812E-01,5.8634E-01,5.8459E-01,5.8289E-01,5.8124E-01,5.7967E-01,5.7818E-01,5.7678E-01,
	5.7549E-01,5.7432E-01,5.7326E-01,5.7234E-01,5.7156E-01,5.7091E-01,5.7042E-01,5.7007E-01,
	5.6988E-01,5.6975E-01,5.6936E-01,5.6868E-01,5.6770E-01,5.6644E-01,5.6493E-01,5.6317E-01,
	5.6119E-01,5.5903E-01,5.5669E-01,5.5421E-01,5.5162E-01,5.4894E-01,5.4621E-01,5.4344E-01,
	5.4067E-01,5.3792E-01,5.3521E-01,5.3257E-01,5.3002E-01,5.2758E-01,5.2527E-01,5.2311E-01,
	5.2111E-01,5.1929E-01,5.1766E-01,5.1623E-01,5.1501E-01,5.1402E-01,5.1325E-01,5.1271E-01,
	5.1242E-01,5.1221E-01,5.1161E-01,5.1055E-01,5.0904E-01,5.0710E-01,5.0475E-01,5.0203E-01,
	4.9898E-01,4.9564E-01,4.9204E-01,4.8822E-01,4.8424E-01,4.8014E-01,4.7595E-01,4.7172E-01,
	4.6750E-01,4.6332E-01,4.5921E-01,4.5522E-01,4.5138E-01,4.4771E-01,4.4424E-01,4.4101E-01,
	4.3802E-01,4.3531E-01,4.3289E-01,4.3077E-01,4.2897E-01,4.2750E-01,4.2637E-01,4.2558E-01,
	4.2514E-01,4.2483E-01,4.2395E-01,4.2239E-01,4.2017E-01,4.1732E-01,4.1389E-01,4.0994E-01,
	4.0552E-01,4.0069E-01,3.9552E-01,3.9008E-01,3.8443E-01,3.7864E-01,3.7278E-01,3.6691E-01,
	3.6108E-01,3.5535E-01,3.4977E-01,3.4439E-01,3.3924E-01,3.3436E-01,3.2978E-01,3.2553E-01,
	3.2163E-01,3.1811E-01,3.1499E-01,3.1226E-01,3.0996E-01,3.0808E-01,3.0664E-01,3.0564E-01,
	3.0508E-01,3.0469E-01,3.0358E-01,3.0161E-01,2.9881E-01,2.9525E-01,2.9099E-01,2.8612E-01,
	2.8071E-01,2.7486E-01,2.6868E-01,2.6224E-01,2.5564E-01,2.4897E-01,2.4230E-01,2.3571E-01,
	2.2926E-01,2.2301E-01,2.1701E-01,2.1129E-01,2.0589E-01,2.0084E-01,1.9616E-01,1.9186E-01,
	1.8797E-01,1.8448E-01,1.8141E-01,1.7876E-01,1.7653E-01,1.7473E-01,1.7335E-01,1.7239E-01,
	1.7186E-01,1.7149E-01,1.7044E-01,1.6857E-01,1.6595E-01,1.6263E-01,1.5871E-01,1.5428E-01,
	1.4944E-01,1.4429E-01,1.3893E-01,1.3346E-01,1.2797E-01,1.2252E-01,1.1720E-01,1.1204E-01,
	1.0710E-01,1.0241E-01,9.7995E-02,9.3875E-02,9.0061E-02,8.6558E-02,8.3367E-02,8.0488E-02,
	7.7916E-02,7.5645E-02,7.3670E-02,7.1983E-02,7.0578E-02,6.9449E-02,6.8591E-02,6.7999E-02,
	6.7672E-02,6.7442E-02,6.6795E-02,6.5657E-02,6.4070E-02,6.2088E-02,5.9779E-02,5.7215E-02,
	5.4467E-02,5.1609E-02,4.8703E-02,4.5809E-02,4.2975E-02,4.0240E-02,3.7633E-02,3.5178E-02,
	3.2886E-02,3.0767E-02,2.8823E-02,2.7052E-02,2.5451E-02,2.4013E-02,2.2730E-02,2.1595E-02,
	2.0600E-02,1.9735E-02,1.8993E-02,1.8368E-02,1.7853E-02,1.7444E-02,1.7134E-02,1.6922E-02,
	1.6805E-02,1.6723E-02,1.6493E-02,1.6092E-02,1.5537E-02,1.4855E-02,1.4073E-02,1.3221E-02,
	1.2330E-02,1.1425E-02,1.0529E-02,9.6605E-03,8.8344E-03,8.0601E-03,7.3438E-03,6.6885E-03,
	6.0946E-03,5.5609E-03,5.0847E-03,4.6624E-03,4.2901E-03,3.9638E-03,3.6793E-03,3.4328E-03,
	3.2207E-03,3.0397E-03,2.8869E-03,2.7599E-03,2.6566E-03,2.5751E-03,2.5141E-03,2.4725E-03,
	2.4497E-03,2.4337E-03,2.3891E-03,2.3116E-03,2.2058E-03,2.0776E-03,1.9334E-03,1.7797E-03,
	1.6226E-03,1.4672E-03,1.3178E-03,1.1772E-03,1.0474E-03,9.2961E-04,8.2401E-04,7.3040E-04,
	6.4820E-04,5.7654E-04,5.1447E-04,4.6098E-04,4.1510E-04,3.7591E-04,3.4255E-04,3.1428E-04,
	2.9045E-04,2.7049E-04,2.5392E-04,2.4034E-04,2.2942E-04,2.2090E-04,2.1458E-04,2.1029E-04,
	2.0794E-04,2.0630E-04,2.0174E-04,1.9389E-04,1.8329E-04,1.7063E-04,1.5665E-04,1.4207E-04,
	1.2750E-04,1.1346E-04,1.0030E-04,8.8249E-05},
	/*4 212*/
	{
	1.3110E+00,1.3110E+00,1.3110E+00,1.3109E+00,1.3108E+00,1.3107E+00,1.3106E+00,1.3104E+00,
	1.3102E+00,1.3100E+00,1.3098E+00,1.3096E+00,1.3093E+00,1.3091E+00,1.3088E+00,1.3085E+00,
	1.3083E+00,1.3080E+00,1.3077E+00,1.3074E+00,1.3072E+00,1.3069E+00,1.3067E+00,1.3064E+00,
	1.3062E+00,1.3060E+00,1.3059E+00,1.3057E+00,1.3056E+00,1.3055E+00,1.3054E+00,1.3053E+00,
	1.3053E+00,1.3053E+00,1.3052E+00,1.3052E+00,1.3051E+00,1.3050E+00,1.3048E+00,1.3047E+00,
	1.3045E+00,1.3043E+00,1.3041E+00,1.3039E+00,1.3036E+00,1.3034E+00,1.3031E+00,1.3028E+00,
	1.3026E+00,1.3023E+00,1.3020E+00,1.3017E+00,1.3015E+00,1.3012E+00,1.3010E+00,1.3007E+00,
	1.3005E+00,1.3003E+00,1.3002E+00,1.3000E+00,1.2999E+00,1.2998E+00,1.2997E+00,1.2996E+00,
	1.2996E+00,1.2996E+00,1.2995E+00,1.2994E+00,1.2992E+00,1.2990E+00,1.2987E+00,1.2984E+00,
	1.2981E+00,1.2977E+00,1.2972E+00,1.2968E+00,1.2963E+00,1.2958E+00,1.2953E+00,1.2947E+00,
	1.2942E+00,1.2936E+00,1.2931E+00,1.2926E+00,1.2920E+00,1.2915E+00,1.2910E+00,1.2906E+00,
	1.2902E+00,1.2898E+00,1.2894E+00,1.2891E+00,1.2889E+00,1.2886E+00,1.2885E+00,1.2884E+00,
	1.2883E+00,1.2882E+00,1.2881E+00,1.2879E+00,1.2875E+00,1.2871E+00,1.2866E+00,1.2860E+00,
	1.2853E+00,1.2845E+00,1.2837E+00,1.2828E+00,1.2818E+00,1.2808E+00,1.2798E+00,1.2787E+00,
	1.2776E+00,1.2766E+00,1.2755E+00,1.2744E+00,1.2734E+00,1.2724E+00,1.2715E+00,1.2706E+00,
	1.2698E+00,1.2690E+00,1.2683E+00,1.2677E+00,1.2672E+00,1.2668E+00,1.2664E+00,1.2662E+00,
	1.2661E+00,1.2660E+00,1.2657E+00,1.2653E+00,1.2646E+00,1.2638E+00,1.2628E+00,1.2616E+00,
	1.2602E+00,1.2587E+00,1.2570E+00,1.2553E+00,1.2534E+00,1.2515E+00,1.2495E+00,1.2474E+00,
	1.2454E+00,1.2433E+00,1.2412E+00,1.2392E+00,1.2372E+00,1.2353E+00,1.2335E+00,1.2318E+00,
	1.2302E+00,1.2287E+00,1.2274E+00,1.2263E+00,1.2253E+00,1.2245E+00,1.2238E+00,1.2234E+00,
	1.2231E+00,1.2230E+00,1.2225E+00,1.2216E+00,1.2204E+00,1.2187E+00,1.2168E+00,1.2145E+00,
	1.2119E+00,1.2090E+00,1.2059E+00,1.2026E+00,1.1991E+00,1.1954E+00,1.1916E+00,1.1878E+00,
	1.1839E+00,1.1800E+00,1.1762E+00,1.1724E+00,1.1687E+00,1.1652E+00,1.1618E+00,1.1586E+00,
	1.1557E+00,1.1530E+00,1.1505E+00,1.1484E+00,1.1466E+00,1.1451E+00,1.1439E+00,1.1431E+00,
	1.1427E+00,1.1424E+00,1.1415E+00,1.1399E+00,1.1376E+00,1.1346E+00,1.1311E+00,1.1269E+00,
	1.1222E+00,1.1170E+00,1.1114E+00,1.1054E+00,1.0991E+00,1.0925E+00,1.0858E+00,1.0790E+00,
	1.0721E+00,1.0653E+00,1.0586E+00,1.0520E+00,1.0456E+00,1.0394E+00,1.0336E+00,1.0281E+00,
	1.0231E+00,1.0184E+00,1.0143E+00,1.0107E+00,1.0076E+00,1.0050E+00,1.0031E+00,1.0017E+00,
	1.0010E+00,1.0004E+00,9.9892E-01,9.9621E-01,9.9235E-01,9.8739E-01,9.8139E-01,9.7444E-01,
	9.6663E-01,9.5805E-01,9.4881E-01,9.3902E-01,9.2879E-01,9.1824E-01,9.0748E-01,8.9662E-01,
	8.8577E-01,8.7503E-01,8.6451E-01,8.5428E-01,8.4444E-01,8.3505E-01,8.2620E-01,8.1795E-01,
	8.1034E-01,8.0344E-01,7.9728E-01,7.9190E-01,7.8733E-01,7.8361E-01,7.8074E-01,7.7874E-01,
	7.7763E-01,7.7684E-01,7.7462E-01,7.7067E-01,7.6506E-01,7.5788E-01,7.4925E-01,7.3931E-01,
	7.2822E-01,7.1615E-01,7.0327E-01,6.8976E-01,6.7579E-01,6.6154E-01,6.4717E-01,6.3283E-01,
	6.1867E-01,6.0482E-01,5.9149E-01,5.7912E-01,5.6735E-01,5.5625E-01,5.4590E-01,5.3633E-01,
	5.2760E-01,5.1975E-01,5.1279E-01,5.0676E-01,5.0167E-01,4.9753E-01,4.9436E-01,4.9216E-01,
	4.9093E-01,4.9007E-01,4.8764E-01,4.8332E-01,4.7722E-01,4.6947E-01,4.6026E-01,4.4976E-01,
	4.3819E-01,4.2577E-01,4.1272E-01,3.9926E-01,3.8557E-01,3.7184E-01,3.5825E-01,3.4493E-01,
	3.3201E-01,3.1959E-01,3.0786E-01,2.9713E-01,2.8710E-01,2.7780E-01,2.6924E-01,2.6145E-01,
	2.5444E-01,2.4820E-01,2.4273E-01,2.3804E-01,2.3411E-01,2.3094E-01,2.2852E-01,2.2685E-01,
	2.2592E-01,2.2527E-01,2.2343E-01,2.2019E-01,2.1565E-01,2.0994E-01,2.0324E-01,1.9571E-01,
	1.8757E-01,1.7898E-01,1.7041E-01,1.6178E-01,1.5322E-01,1.4485E-01,1.3677E-01,1.2904E-01,
	1.2173E-01,1.1488E-01,1.0854E-01,1.0281E-01,9.7568E-02,9.2807E-02,8.8515E-02,8.4678E-02,
	8.1280E-02,7.8303E-02,7.5731E-02,7.3548E-02,7.1740E-02,7.0328E-02,6.9268E-02,6.8539E-02,
	6.8136E-02,6.7853E-02,6.7060E-02,6.5668E-02,6.3735E-02,6.1339E-02,5.8570E-02,5.5521E-02,
	5.2290E-02,4.8964E-02,4.5624E-02,4.2336E-02,3.9156E-02,3.6126E-02,3.3273E-02,3.0618E-02,
	2.8199E-02,2.6084E-02,2.4165E-02,2.2435E-02,2.0885E-02,1.9505E-02,1.8284E-02,1.7212E-02,
	1.6278E-02,1.5471E-02,1.4783E-02,1.4206E-02,1.3733E-02,1.3379E-02,1.3118E-02,1.2939E-02,
	1.2840E-02,1.2772E-02,1.2578E-02,1.2241E-02,1.1776E-02,1.1207E-02,1.0556E-02,9.8509E-03,
	9.1157E-03,8.3729E-03,7.6412E-03,6.9357E-03,6.2675E-03,5.6443E-03,5.0704E-03,4.5477E-03,
	4.0876E-03,3.7117E-03,3.3765E-03,3.0794E-03,2.8176E-03,2.5881E-03,2.3880E-03,2.2146E-03,
	2.0653E-03,1.9379E-03,1.8303E-03,1.7409E-03,1.6681E-03,1.6148E-03,1.5760E-03,1.5495E-03,
	1.5349E-03,1.5247E-03,1.4962E-03,1.4467E-03,1.3791E-03,1.2968E-03,1.2040E-03,1.1047E-03,
	1.0028E-03,9.0155E-04,8.0361E-04,7.1604E-04,6.3734E-04,5.6518E-04,4.9987E-04,4.4139E-04,
	3.9026E-04,3.4761E-04,3.1035E-04,2.7798E-04,2.4999E-04,2.2593E-04,2.0613E-04,1.8923E-04,
	1.7489E-04,1.6281E-04,1.5273E-04,1.4443E-04,1.3773E-04,1.3262E-04,1.2885E-04,1.2630E-04,
	1.2489E-04,1.2391E-04,1.2118E-04,1.1647E-04,1.1008E-04,1.0240E-04,9.3856E-05,8.4912E-05,
	7.6125E-05,6.7563E-05,5.9456E-05,6.3401E-05},
	/*5 212*/
	{
	1.9659E+00,1.9658E+00,1.9658E+00,1.9656E+00,1.9655E+00,1.9652E+00,1.9649E+00,1.9646E+00,
	1.9642E+00,1.9638E+00,1.9633E+00,1.9628E+00,1.9623E+00,1.9617E+00,1.9612E+00,1.9606E+00,
	1.9600E+00,1.9594E+00,1.9588E+00,1.9582E+00,1.9576E+00,1.9571E+00,1.9565E+00,1.9560E+00,
	1.9556E+00,1.9552E+00,1.9548E+00,1.9544E+00,1.9541E+00,1.9539E+00,1.9537E+00,1.9536E+00,
	1.9535E+00,1.9535E+00,1.9534E+00,1.9533E+00,1.9531E+00,1.9529E+00,1.9526E+00,1.9522E+00,
	1.9519E+00,1.9514E+00,1.9510E+00,1.9505E+00,1.9500E+00,1.9494E+00,1.9488E+00,1.9483E+00,
	1.9477E+00,1.9471E+00,1.9465E+00,1.9459E+00,1.9453E+00,1.9448E+00,1.9443E+00,1.9438E+00,
	1.9433E+00,1.9429E+00,1.9425E+00,1.9422E+00,1.9419E+00,1.9417E+00,1.9415E+00,1.9414E+00,
	1.9413E+00,1.9412E+00,1.9411E+00,1.9408E+00,1.9405E+00,1.9400E+00,1.9394E+00,1.9388E+00,
	1.9380E+00,1.9372E+00,1.9363E+00,1.9353E+00,1.9342E+00,1.9331E+00,1.9320E+00,1.9309E+00,
	1.9297E+00,1.9285E+00,1.9274E+00,1.9262E+00,1.9251E+00,1.9240E+00,1.9230E+00,1.9220E+00,
	1.9211E+00,1.9203E+00,1.9196E+00,1.9189E+00,1.9183E+00,1.9179E+00,1.9175E+00,1.9173E+00,
	1.9171E+00,1.9170E+00,1.9168E+00,1.9163E+00,1.9155E+00,1.9146E+00,1.9135E+00,1.9122E+00,
	1.9107E+00,1.9090E+00,1.9072E+00,1.9053E+00,1.9033E+00,1.9012E+00,1.8990E+00,1.8967E+00,
	1.8945E+00,1.8922E+00,1.8899E+00,1.8877E+00,1.8855E+00,1.8834E+00,1.8814E+00,1.8796E+00,
	1.8778E+00,1.8762E+00,1.8747E+00,1.8735E+00,1.8724E+00,1.8715E+00,1.8708E+00,1.8703E+00,
	1.8700E+00,1.8699E+00,1.8693E+00,1.8683E+00,1.8670E+00,1.8652E+00,1.8630E+00,1.8605E+00,
	1.8576E+00,1.8545E+00,1.8510E+00,1.8473E+00,1.8434E+00,1.8394E+00,1.8352E+00,1.8309E+00,
	1.8266E+00,1.8223E+00,1.8180E+00,1.8138E+00,1.8097E+00,1.8057E+00,1.8019E+00,1.7984E+00,
	1.7951E+00,1.7921E+00,1.7893E+00,1.7869E+00,1.7849E+00,1.7832E+00,1.7819E+00,1.7810E+00,
	1.7805E+00,1.7802E+00,1.7792E+00,1.7774E+00,1.7748E+00,1.7715E+00,1.7674E+00,1.7627E+00,
	1.7574E+00,1.7515E+00,1.7452E+00,1.7384E+00,1.7312E+00,1.7238E+00,1.7162E+00,1.7084E+00,
	1.7005E+00,1.6927E+00,1.6849E+00,1.6774E+00,1.6700E+00,1.6629E+00,1.6562E+00,1.6499E+00,
	1.6440E+00,1.6386E+00,1.6338E+00,1.6296E+00,1.6260E+00,1.6231E+00,1.6208E+00,1.6192E+00,
	1.6183E+00,1.6177E+00,1.6159E+00,1.6128E+00,1.6082E+00,1.6024E+00,1.5954E+00,1.5873E+00,
	1.5781E+00,1.5680E+00,1.5570E+00,1.5454E+00,1.5333E+00,1.5207E+00,1.5079E+00,1.4948E+00,
	1.4818E+00,1.4688E+00,1.4561E+00,1.4437E+00,1.4317E+00,1.4202E+00,1.4094E+00,1.3992E+00,
	1.3899E+00,1.3813E+00,1.3737E+00,1.3671E+00,1.3614E+00,1.3567E+00,1.3532E+00,1.3507E+00,
	1.3493E+00,1.3483E+00,1.3456E+00,1.3406E+00,1.3336E+00,1.3246E+00,1.3138E+00,1.3013E+00,
	1.2873E+00,1.2719E+00,1.2555E+00,1.2382E+00,1.2203E+00,1.2018E+00,1.1832E+00,1.1644E+00,
	1.1458E+00,1.1275E+00,1.1097E+00,1.0925E+00,1.0760E+00,1.0604E+00,1.0457E+00,1.0321E+00,
	1.0197E+00,1.0084E+00,9.9836E-01,9.8964E-01,9.8225E-01,9.7624E-01,9.7162E-01,9.6841E-01,
	9.6662E-01,9.6537E-01,9.6181E-01,9.5549E-01,9.4653E-01,9.3511E-01,9.2145E-01,9.0580E-01,
	8.8846E-01,8.6970E-01,8.4984E-01,8.2916E-01,8.0797E-01,7.8652E-01,7.6509E-01,7.4388E-01,
	7.2312E-01,7.0298E-01,6.8367E-01,6.6549E-01,6.4834E-01,6.3232E-01,6.1748E-01,6.0388E-01,
	5.9155E-01,5.8052E-01,5.7080E-01,5.6242E-01,5.5538E-01,5.4968E-01,5.4531E-01,5.4229E-01,
	5.4062E-01,5.3944E-01,5.3611E-01,5.3023E-01,5.2194E-01,5.1148E-01,4.9910E-01,4.8512E-01,
	4.6985E-01,4.5361E-01,4.3672E-01,4.1946E-01,4.0212E-01,3.8493E-01,3.6809E-01,3.5177E-01,
	3.3612E-01,3.2124E-01,3.0729E-01,2.9454E-01,2.8273E-01,2.7189E-01,2.6201E-01,2.5309E-01,
	2.4512E-01,2.3807E-01,2.3195E-01,2.2671E-01,2.2234E-01,2.1883E-01,2.1617E-01,2.1433E-01,
	2.1331E-01,2.1259E-01,2.1058E-01,2.0704E-01,2.0209E-01,1.9591E-01,1.8870E-01,1.8068E-01,
	1.7206E-01,1.6308E-01,1.5414E-01,1.4524E-01,1.3650E-01,1.2805E-01,1.1997E-01,1.1233E-01,
	1.0518E-01,9.8535E-02,9.2443E-02,8.6983E-02,8.2033E-02,7.7575E-02,7.3588E-02,7.0050E-02,
	6.6936E-02,6.4225E-02,6.1895E-02,5.9926E-02,5.8300E-02,5.7034E-02,5.6085E-02,5.5434E-02,
	5.5074E-02,5.4822E-02,5.4115E-02,5.2879E-02,5.1169E-02,4.9059E-02,4.6635E-02,4.3986E-02,
	4.1199E-02,3.8355E-02,3.5523E-02,3.2760E-02,3.0112E-02,2.7609E-02,2.5274E-02,2.3118E-02,
	2.1169E-02,1.9473E-02,1.7946E-02,1.6579E-02,1.5363E-02,1.4286E-02,1.3339E-02,1.2511E-02,
	1.1793E-02,1.1175E-02,1.0651E-02,1.0212E-02,9.8524E-03,9.5834E-03,9.3853E-03,9.2499E-03,
	9.1753E-03,9.1231E-03,8.9770E-03,8.7225E-03,8.3730E-03,7.9457E-03,7.4603E-03,6.9367E-03,
	6.3938E-03,5.8486E-03,5.3148E-03,4.8034E-03,4.3221E-03,3.8758E-03,3.4673E-03,3.0974E-03,
	2.7738E-03,2.5115E-03,2.2782E-03,2.0720E-03,1.8907E-03,1.7321E-03,1.5941E-03,1.4747E-03,
	1.3720E-03,1.2845E-03,1.2107E-03,1.1495E-03,1.0996E-03,1.0635E-03,1.0373E-03,1.0194E-03,
	1.0095E-03,1.0027E-03,9.8347E-04,9.5014E-04,9.0460E-04,8.4934E-04,7.8712E-04,7.2069E-04,
	6.5265E-04,5.8520E-04,5.2013E-04,4.6214E-04,4.1027E-04,3.6289E-04,3.2018E-04,2.8208E-04,
	2.4886E-04,2.2117E-04,1.9707E-04,1.7619E-04,1.5818E-04,1.4274E-04,1.3004E-04,1.1923E-04,
	1.1007E-04,1.0236E-04,9.5939E-05,9.0659E-05,8.6402E-05,8.3155E-05,8.0761E-05,7.9136E-05,
	7.8244E-05,7.7622E-05,7.5889E-05,7.2897E-05,6.8844E-05,6.3982E-05,5.8584E-05,5.2939E-05,
	4.7396E-05,4.2007E-05,3.6915E-05,4.1910E-05},
	/*6 212*/
	{
	2.6362E+00,2.6361E+00,2.6360E+00,2.6358E+00,2.6354E+00,2.6350E+00,2.6344E+00,2.6338E+00,
	2.6331E+00,2.6323E+00,2.6314E+00,2.6305E+00,2.6295E+00,2.6284E+00,2.6274E+00,2.6263E+00,
	2.6252E+00,2.6241E+00,2.6230E+00,2.6219E+00,2.6208E+00,2.6198E+00,2.6188E+00,2.6179E+00,
	2.6170E+00,2.6162E+00,2.6155E+00,2.6149E+00,2.6144E+00,2.6139E+00,2.6136E+00,2.6133E+00,
	2.6132E+00,2.6131E+00,2.6130E+00,2.6128E+00,2.6124E+00,2.6120E+00,2.6114E+00,2.6108E+00,
	2.6101E+00,2.6093E+00,2.6085E+00,2.6075E+00,2.6066E+00,2.6055E+00,2.6045E+00,2.6034E+00,
	2.6023E+00,2.6012E+00,2.6001E+00,2.5991E+00,2.5980E+00,2.5970E+00,2.5960E+00,2.5951E+00,
	2.5943E+00,2.5935E+00,2.5928E+00,2.5922E+00,2.5916E+00,2.5912E+00,2.5908E+00,2.5906E+00,
	2.5905E+00,2.5904E+00,2.5901E+00,2.5897E+00,2.5890E+00,2.5881E+00,2.5871E+00,2.5858E+00,
	2.5844E+00,2.5829E+00,2.5812E+00,2.5794E+00,2.5774E+00,2.5754E+00,2.5734E+00,2.5712E+00,
	2.5691E+00,2.5669E+00,2.5648E+00,2.5627E+00,2.5606E+00,2.5586E+00,2.5568E+00,2.5550E+00,
	2.5533E+00,2.5518E+00,2.5504E+00,2.5492E+00,2.5481E+00,2.5473E+00,2.5466E+00,2.5462E+00,
	2.5459E+00,2.5457E+00,2.5452E+00,2.5443E+00,2.5430E+00,2.5413E+00,2.5392E+00,2.5368E+00,
	2.5341E+00,2.5311E+00,2.5278E+00,2.5243E+00,2.5206E+00,2.5167E+00,2.5127E+00,2.5086E+00,
	2.5044E+00,2.5003E+00,2.4962E+00,2.4921E+00,2.4882E+00,2.4843E+00,2.4807E+00,2.4773E+00,
	2.4741E+00,2.4712E+00,2.4685E+00,2.4662E+00,2.4643E+00,2.4626E+00,2.4614E+00,2.4605E+00,
	2.4600E+00,2.4597E+00,2.4587E+00,2.4569E+00,2.4544E+00,2.4512E+00,2.4473E+00,2.4427E+00,
	2.4376E+00,2.4319E+00,2.4257E+00,2.4190E+00,2.4120E+00,2.4048E+00,2.3973E+00,2.3896E+00,
	2.3819E+00,2.3742E+00,2.3665E+00,2.3590E+00,2.3517E+00,2.3447E+00,2.3380E+00,2.3317E+00,
	2.3259E+00,2.3205E+00,2.3157E+00,2.3115E+00,2.3079E+00,2.3049E+00,2.3027E+00,2.3011E+00,
	2.3002E+00,2.2995E+00,2.2978E+00,2.2946E+00,2.2901E+00,2.2842E+00,2.2772E+00,2.2689E+00,
	2.2596E+00,2.2494E+00,2.2383E+00,2.2265E+00,2.2141E+00,2.2012E+00,2.1880E+00,2.1746E+00,
	2.1611E+00,2.1477E+00,2.1345E+00,2.1215E+00,2.1090E+00,2.0970E+00,2.0856E+00,2.0749E+00,
	2.0650E+00,2.0560E+00,2.0479E+00,2.0408E+00,2.0348E+00,2.0299E+00,2.0261E+00,2.0234E+00,
	2.0219E+00,2.0209E+00,2.0179E+00,2.0127E+00,2.0051E+00,1.9955E+00,1.9838E+00,1.9703E+00,
	1.9552E+00,1.9385E+00,1.9206E+00,1.9017E+00,1.8819E+00,1.8615E+00,1.8407E+00,1.8198E+00,
	1.7989E+00,1.7783E+00,1.7581E+00,1.7384E+00,1.7196E+00,1.7016E+00,1.6847E+00,1.6689E+00,
	1.6544E+00,1.6412E+00,1.6295E+00,1.6193E+00,1.6106E+00,1.6035E+00,1.5980E+00,1.5942E+00,
	1.5921E+00,1.5906E+00,1.5864E+00,1.5789E+00,1.5682E+00,1.5546E+00,1.5382E+00,1.5194E+00,
	1.4984E+00,1.4756E+00,1.4513E+00,1.4258E+00,1.3995E+00,1.3727E+00,1.3457E+00,1.3188E+00,
	1.2923E+00,1.2664E+00,1.2413E+00,1.2173E+00,1.1944E+00,1.1728E+00,1.1527E+00,1.1341E+00,
	1.1171E+00,1.1019E+00,1.0883E+00,1.0766E+00,1.0667E+00,1.0587E+00,1.0525E+00,1.0483E+00,
	1.0459E+00,1.0442E+00,1.0395E+00,1.0311E+00,1.0192E+00,1.0042E+00,9.8628E-01,9.6588E-01,
	9.4341E-01,9.1927E-01,8.9389E-01,8.6769E-01,8.4104E-01,8.1430E-01,7.8779E-01,7.6179E-01,
	7.3655E-01,7.1226E-01,6.8904E-01,6.6695E-01,6.4629E-01,6.2714E-01,6.0954E-01,5.9353E-01,
	5.7910E-01,5.6627E-01,5.5504E-01,5.4539E-01,5.3731E-01,5.3079E-01,5.2582E-01,5.2239E-01,
	5.2048E-01,5.1914E-01,5.1537E-01,5.0871E-01,4.9937E-01,4.8763E-01,4.7384E-01,4.5836E-01,
	4.4159E-01,4.2391E-01,4.0569E-01,3.8725E-01,3.6891E-01,3.5090E-01,3.3344E-01,3.1669E-01,
	3.0077E-01,2.8578E-01,2.7181E-01,2.5902E-01,2.4727E-01,2.3657E-01,2.2689E-01,2.1821E-01,
	2.1050E-01,2.0373E-01,1.9787E-01,1.9288E-01,1.8874E-01,1.8542E-01,1.8290E-01,1.8116E-01,
	1.8021E-01,1.7953E-01,1.7764E-01,1.7432E-01,1.6970E-01,1.6395E-01,1.5728E-01,1.4991E-01,
	1.4204E-01,1.3390E-01,1.2585E-01,1.1789E-01,1.1016E-01,1.0274E-01,9.5707E-02,8.9117E-02,
	8.2997E-02,7.7358E-02,7.2223E-02,6.7646E-02,6.3519E-02,5.9824E-02,5.6536E-02,5.3631E-02,
	5.1085E-02,4.8876E-02,4.6984E-02,4.5390E-02,4.4077E-02,4.3056E-02,4.2292E-02,4.1769E-02,
	4.1480E-02,4.1278E-02,4.0712E-02,3.9722E-02,3.8357E-02,3.6679E-02,3.4759E-02,3.2670E-02,
	3.0485E-02,2.8266E-02,2.6070E-02,2.3939E-02,2.1909E-02,2.0001E-02,1.8231E-02,1.6606E-02,
	1.5145E-02,1.3884E-02,1.2754E-02,1.1746E-02,1.0852E-02,1.0065E-02,9.3740E-03,8.7720E-03,
	8.2511E-03,7.8043E-03,7.4253E-03,7.1088E-03,6.8502E-03,6.6571E-03,6.5152E-03,6.4183E-03,
	6.3649E-03,6.3276E-03,6.2231E-03,6.0413E-03,5.7920E-03,5.4878E-03,5.1430E-03,4.7721E-03,
	4.3887E-03,4.0047E-03,3.6299E-03,3.2719E-03,2.9359E-03,2.6254E-03,2.3420E-03,2.0860E-03,
	1.8628E-03,1.6829E-03,1.5234E-03,1.3829E-03,1.2597E-03,1.1522E-03,1.0589E-03,9.7836E-04,
	9.0925E-04,8.5045E-04,8.0093E-04,7.5985E-04,7.2647E-04,7.0220E-04,6.8457E-04,6.7255E-04,
	6.6595E-04,6.6133E-04,6.4844E-04,6.2608E-04,5.9556E-04,5.5859E-04,5.1702E-04,4.7274E-04,
	4.2748E-04,3.8272E-04,3.3963E-04,3.0128E-04,2.6703E-04,2.3583E-04,2.0776E-04,1.8279E-04,
	1.6105E-04,1.4295E-04,1.2723E-04,1.1363E-04,1.0192E-04,9.1897E-05,8.3644E-05,7.6624E-05,
	7.0685E-05,6.5695E-05,6.1540E-05,5.8126E-05,5.5376E-05,5.3279E-05,5.1734E-05,5.0686E-05,
	5.0110E-05,4.9709E-05,4.8592E-05,4.6663E-05,4.4051E-05,4.0921E-05,3.7448E-05,3.3819E-05,
	3.0256E-05,2.6795E-05,2.3529E-05,2.7967E-05},
	/*7 212*/
	{
	3.3444E+00,3.3444E+00,3.3442E+00,3.3437E+00,3.3432E+00,3.3424E+00,3.3415E+00,3.3404E+00,
	3.3392E+00,3.3378E+00,3.3363E+00,3.3347E+00,3.3330E+00,3.3313E+00,3.3294E+00,3.3276E+00,
	3.3257E+00,3.3238E+00,3.3219E+00,3.3201E+00,3.3183E+00,3.3165E+00,3.3148E+00,3.3133E+00,
	3.3118E+00,3.3104E+00,3.3092E+00,3.3082E+00,3.3073E+00,3.3065E+00,3.3059E+00,3.3055E+00,
	3.3053E+00,3.3052E+00,3.3050E+00,3.3045E+00,3.3040E+00,3.3032E+00,3.3023E+00,3.3012E+00,
	3.3000E+00,3.2987E+00,3.2972E+00,3.2957E+00,3.2940E+00,3.2923E+00,3.2905E+00,3.2887E+00,
	3.2868E+00,3.2849E+00,3.2831E+00,3.2813E+00,3.2795E+00,3.2778E+00,3.2761E+00,3.2746E+00,
	3.2731E+00,3.2718E+00,3.2706E+00,3.2696E+00,3.2687E+00,3.2679E+00,3.2674E+00,3.2670E+00,
	3.2667E+00,3.2666E+00,3.2661E+00,3.2653E+00,3.2642E+00,3.2627E+00,3.2609E+00,3.2589E+00,
	3.2565E+00,3.2539E+00,3.2510E+00,3.2479E+00,3.2447E+00,3.2413E+00,3.2378E+00,3.2342E+00,
	3.2306E+00,3.2270E+00,3.2234E+00,3.2198E+00,3.2164E+00,3.2130E+00,3.2098E+00,3.2068E+00,
	3.2040E+00,3.2015E+00,3.1991E+00,3.1971E+00,3.1954E+00,3.1939E+00,3.1928E+00,3.1921E+00,
	3.1916E+00,3.1913E+00,3.1905E+00,3.1889E+00,3.1867E+00,3.1839E+00,3.1804E+00,3.1764E+00,
	3.1718E+00,3.1668E+00,3.1613E+00,3.1554E+00,3.1492E+00,3.1427E+00,3.1360E+00,3.1292E+00,
	3.1223E+00,3.1154E+00,3.1086E+00,3.1018E+00,3.0953E+00,3.0890E+00,3.0830E+00,3.0773E+00,
	3.0720E+00,3.0672E+00,3.0628E+00,3.0590E+00,3.0558E+00,3.0531E+00,3.0510E+00,3.0496E+00,
	3.0488E+00,3.0482E+00,3.0466E+00,3.0437E+00,3.0396E+00,3.0343E+00,3.0279E+00,3.0204E+00,
	3.0119E+00,3.0025E+00,2.9923E+00,2.9815E+00,2.9701E+00,2.9582E+00,2.9460E+00,2.9335E+00,
	2.9210E+00,2.9085E+00,2.8961E+00,2.8840E+00,2.8722E+00,2.8609E+00,2.8501E+00,2.8400E+00,
	2.8306E+00,2.8221E+00,2.8144E+00,2.8076E+00,2.8019E+00,2.7972E+00,2.7935E+00,2.7910E+00,
	2.7896E+00,2.7886E+00,2.7858E+00,2.7807E+00,2.7735E+00,2.7642E+00,2.7530E+00,2.7399E+00,
	2.7252E+00,2.7091E+00,2.6916E+00,2.6730E+00,2.6536E+00,2.6335E+00,2.6129E+00,2.5920E+00,
	2.5711E+00,2.5504E+00,2.5300E+00,2.5101E+00,2.4909E+00,2.4725E+00,2.4551E+00,2.4389E+00,
	2.4239E+00,2.4102E+00,2.3980E+00,2.3873E+00,2.3782E+00,2.3708E+00,2.3651E+00,2.3611E+00,
	2.3589E+00,2.3573E+00,2.3529E+00,2.3450E+00,2.3337E+00,2.3193E+00,2.3019E+00,2.2818E+00,
	2.2594E+00,2.2348E+00,2.2084E+00,2.1807E+00,2.1518E+00,2.1222E+00,2.0922E+00,2.0621E+00,
	2.0322E+00,2.0028E+00,1.9742E+00,1.9465E+00,1.9200E+00,1.8949E+00,1.8713E+00,1.8494E+00,
	1.8293E+00,1.8112E+00,1.7950E+00,1.7810E+00,1.7691E+00,1.7594E+00,1.7520E+00,1.7468E+00,
	1.7439E+00,1.7419E+00,1.7362E+00,1.7260E+00,1.7115E+00,1.6931E+00,1.6711E+00,1.6459E+00,
	1.6179E+00,1.5877E+00,1.5556E+00,1.5222E+00,1.4880E+00,1.4533E+00,1.4187E+00,1.3844E+00,
	1.3508E+00,1.3182E+00,1.2868E+00,1.2569E+00,1.2286E+00,1.2022E+00,1.1776E+00,1.1550E+00,
	1.1345E+00,1.1161E+00,1.0999E+00,1.0859E+00,1.0741E+00,1.0646E+00,1.0573E+00,1.0522E+00,
	1.0494E+00,1.0474E+00,1.0418E+00,1.0319E+00,1.0180E+00,1.0003E+00,9.7943E-01,9.5575E-01,
	9.2981E-01,9.0214E-01,8.7324E-01,8.4363E-01,8.1373E-01,7.8397E-01,7.5468E-01,7.2618E-01,
	6.9871E-01,6.7248E-01,6.4748E-01,6.2350E-01,6.0126E-01,5.8079E-01,5.6210E-01,5.4521E-01,
	5.3008E-01,5.1669E-01,5.0503E-01,4.9504E-01,4.8672E-01,4.8002E-01,4.7492E-01,4.7140E-01,
	4.6945E-01,4.6808E-01,4.6423E-01,4.5745E-01,4.4796E-01,4.3609E-01,4.2222E-01,4.0675E-01,
	3.9009E-01,3.7266E-01,3.5484E-01,3.3695E-01,3.1930E-01,3.0211E-01,2.8558E-01,2.6985E-01,
	2.5503E-01,2.4117E-01,2.2832E-01,2.1654E-01,2.0580E-01,1.9609E-01,1.8735E-01,1.7957E-01,
	1.7269E-01,1.6667E-01,1.6148E-01,1.5708E-01,1.5344E-01,1.5053E-01,1.4832E-01,1.4681E-01,
	1.4597E-01,1.4538E-01,1.4374E-01,1.4084E-01,1.3683E-01,1.3186E-01,1.2611E-01,1.1979E-01,
	1.1309E-01,1.0619E-01,9.9368E-02,9.2673E-02,8.6202E-02,8.0037E-02,7.4233E-02,6.8827E-02,
	6.3835E-02,5.9262E-02,5.5121E-02,5.1451E-02,4.8160E-02,4.5225E-02,4.2625E-02,4.0337E-02,
	3.8339E-02,3.6611E-02,3.5134E-02,3.3893E-02,3.2873E-02,3.2076E-02,3.1480E-02,3.1073E-02,
	3.0847E-02,3.0690E-02,3.0249E-02,2.9480E-02,2.8421E-02,2.7122E-02,2.5640E-02,2.4035E-02,
	2.2361E-02,2.0669E-02,1.9002E-02,1.7392E-02,1.5864E-02,1.4435E-02,1.3114E-02,1.1907E-02,
	1.0828E-02,9.9075E-03,9.0845E-03,8.3529E-03,7.7060E-03,7.1368E-03,6.6387E-03,6.2053E-03,
	5.8308E-03,5.5100E-03,5.2383E-03,5.0115E-03,4.8264E-03,4.6881E-03,4.5864E-03,4.5170E-03,
	4.4788E-03,4.4521E-03,4.3773E-03,4.2473E-03,4.0691E-03,3.8519E-03,3.6060E-03,3.3419E-03,
	3.0692E-03,2.7965E-03,2.5308E-03,2.2773E-03,2.0398E-03,1.8206E-03,1.6208E-03,1.4406E-03,
	1.2838E-03,1.1578E-03,1.0465E-03,9.4858E-04,8.6294E-04,7.8836E-04,7.2373E-04,6.6801E-04,
	6.2029E-04,5.7973E-04,5.4562E-04,5.1734E-04,4.9439E-04,4.7771E-04,4.6560E-04,4.5736E-04,
	4.5282E-04,4.4966E-04,4.4081E-04,4.2548E-04,4.0457E-04,3.7924E-04,3.5081E-04,3.2055E-04,
	2.8965E-04,2.5913E-04,2.2980E-04,2.0367E-04,1.8034E-04,1.5912E-04,1.4006E-04,1.2312E-04,
	1.0839E-04,9.6138E-05,8.5499E-05,7.6308E-05,6.8404E-05,6.1640E-05,5.6076E-05,5.1347E-05,
	4.7348E-05,4.3990E-05,4.1196E-05,3.8901E-05,3.7053E-05,3.5643E-05,3.4603E-05,3.3897E-05,
	3.3510E-05,3.3241E-05,3.2489E-05,3.1192E-05,2.9437E-05,2.7334E-05,2.5003E-05,2.2569E-05,
	2.0182E-05,1.7865E-05,1.5680E-05,1.9086E-05},
	/*8 212*/
	{
	4.0802E+00,4.0801E+00,4.0798E+00,4.0791E+00,4.0782E+00,4.0770E+00,4.0755E+00,4.0738E+00,
	4.0718E+00,4.0697E+00,4.0673E+00,4.0648E+00,4.0621E+00,4.0593E+00,4.0564E+00,4.0535E+00,
	4.0505E+00,4.0475E+00,4.0445E+00,4.0415E+00,4.0387E+00,4.0359E+00,4.0332E+00,4.0308E+00,
	4.0284E+00,4.0263E+00,4.0244E+00,4.0227E+00,4.0212E+00,4.0200E+00,4.0191E+00,4.0185E+00,
	4.0181E+00,4.0180E+00,4.0176E+00,4.0170E+00,4.0160E+00,4.0149E+00,4.0134E+00,4.0117E+00,
	4.0098E+00,4.0077E+00,4.0054E+00,4.0029E+00,4.0003E+00,3.9976E+00,3.9948E+00,3.9919E+00,
	3.9889E+00,3.9860E+00,3.9831E+00,3.9802E+00,3.9774E+00,3.9747E+00,3.9721E+00,3.9697E+00,
	3.9674E+00,3.9653E+00,3.9634E+00,3.9618E+00,3.9604E+00,3.9592E+00,3.9583E+00,3.9577E+00,
	3.9573E+00,3.9571E+00,3.9564E+00,3.9551E+00,3.9533E+00,3.9510E+00,3.9482E+00,3.9449E+00,
	3.9412E+00,3.9371E+00,3.9326E+00,3.9278E+00,3.9227E+00,3.9174E+00,3.9119E+00,3.9063E+00,
	3.9006E+00,3.8949E+00,3.8893E+00,3.8837E+00,3.8783E+00,3.8731E+00,3.8681E+00,3.8634E+00,
	3.8591E+00,3.8551E+00,3.8515E+00,3.8483E+00,3.8456E+00,3.8434E+00,3.8416E+00,3.8404E+00,
	3.8398E+00,3.8393E+00,3.8380E+00,3.8356E+00,3.8321E+00,3.8277E+00,3.8223E+00,3.8161E+00,
	3.8090E+00,3.8011E+00,3.7926E+00,3.7835E+00,3.7739E+00,3.7639E+00,3.7535E+00,3.7430E+00,
	3.7324E+00,3.7217E+00,3.7112E+00,3.7008E+00,3.6908E+00,3.6811E+00,3.6718E+00,3.6631E+00,
	3.6551E+00,3.6477E+00,3.6410E+00,3.6352E+00,3.6302E+00,3.6261E+00,3.6230E+00,3.6208E+00,
	3.6196E+00,3.6187E+00,3.6162E+00,3.6118E+00,3.6056E+00,3.5975E+00,3.5877E+00,3.5763E+00,
	3.5634E+00,3.5492E+00,3.5338E+00,3.5174E+00,3.5001E+00,3.4822E+00,3.4638E+00,3.4452E+00,
	3.4264E+00,3.4076E+00,3.3891E+00,3.3710E+00,3.3535E+00,3.3367E+00,3.3207E+00,3.3057E+00,
	3.2919E+00,3.2792E+00,3.2679E+00,3.2579E+00,3.2494E+00,3.2425E+00,3.2372E+00,3.2334E+00,
	3.2313E+00,3.2299E+00,3.2257E+00,3.2183E+00,3.2077E+00,3.1941E+00,3.1777E+00,3.1586E+00,
	3.1372E+00,3.1137E+00,3.0884E+00,3.0615E+00,3.0335E+00,3.0045E+00,2.9750E+00,2.9452E+00,
	2.9155E+00,2.8860E+00,2.8571E+00,2.8290E+00,2.8020E+00,2.7762E+00,2.7519E+00,2.7293E+00,
	2.7084E+00,2.6894E+00,2.6725E+00,2.6577E+00,2.6452E+00,2.6349E+00,2.6270E+00,2.6215E+00,
	2.6185E+00,2.6163E+00,2.6102E+00,2.5994E+00,2.5839E+00,2.5642E+00,2.5404E+00,2.5131E+00,
	2.4826E+00,2.4494E+00,2.4139E+00,2.3767E+00,2.3382E+00,2.2989E+00,2.2593E+00,2.2197E+00,
	2.1806E+00,2.1423E+00,2.1052E+00,2.0695E+00,2.0355E+00,2.0034E+00,1.9734E+00,1.9456E+00,
	1.9203E+00,1.8975E+00,1.8772E+00,1.8596E+00,1.8448E+00,1.8327E+00,1.8235E+00,1.8170E+00,
	1.8135E+00,1.8109E+00,1.8038E+00,1.7912E+00,1.7734E+00,1.7507E+00,1.7236E+00,1.6928E+00,
	1.6587E+00,1.6221E+00,1.5835E+00,1.5436E+00,1.5029E+00,1.4620E+00,1.4213E+00,1.3813E+00,
	1.3424E+00,1.3049E+00,1.2690E+00,1.2350E+00,1.2031E+00,1.1733E+00,1.1458E+00,1.1207E+00,
	1.0980E+00,1.0777E+00,1.0599E+00,1.0445E+00,1.0316E+00,1.0212E+00,1.0133E+00,1.0078E+00,
	1.0047E+00,1.0025E+00,9.9648E-01,9.8576E-01,9.7069E-01,9.5169E-01,9.2927E-01,9.0399E-01,
	8.7644E-01,8.4722E-01,8.1690E-01,7.8602E-01,7.5505E-01,7.2442E-01,6.9449E-01,6.6555E-01,
	6.3783E-01,6.1151E-01,5.8647E-01,5.6216E-01,5.3975E-01,5.1927E-01,5.0068E-01,4.8396E-01,
	4.6907E-01,4.5596E-01,4.4457E-01,4.3486E-01,4.2679E-01,4.2031E-01,4.1539E-01,4.1200E-01,
	4.1012E-01,4.0880E-01,4.0510E-01,3.9859E-01,3.8951E-01,3.7819E-01,3.6502E-01,3.5040E-01,
	3.3476E-01,3.1850E-01,3.0197E-01,2.8550E-01,2.6935E-01,2.5374E-01,2.3882E-01,2.2472E-01,
	2.1152E-01,1.9925E-01,1.8793E-01,1.7756E-01,1.6817E-01,1.5971E-01,1.5214E-01,1.4542E-01,
	1.3951E-01,1.3436E-01,1.2993E-01,1.2618E-01,1.2309E-01,1.2062E-01,1.1875E-01,1.1747E-01,
	1.1676E-01,1.1627E-01,1.1487E-01,1.1244E-01,1.0906E-01,1.0489E-01,1.0008E-01,9.4812E-02,
	8.9249E-02,8.3547E-02,7.7918E-02,7.2421E-02,6.7136E-02,6.2126E-02,5.7433E-02,5.3082E-02,
	4.9084E-02,4.5437E-02,4.2146E-02,3.9233E-02,3.6631E-02,3.4321E-02,3.2282E-02,3.0493E-02,
	2.8936E-02,2.7593E-02,2.6447E-02,2.5487E-02,2.4698E-02,2.4087E-02,2.3630E-02,2.3318E-02,
	2.3146E-02,2.3026E-02,2.2688E-02,2.2100E-02,2.1291E-02,2.0300E-02,1.9171E-02,1.7949E-02,
	1.6676E-02,1.5392E-02,1.4127E-02,1.2908E-02,1.1752E-02,1.0673E-02,9.6765E-03,8.7663E-03,
	7.9536E-03,7.2616E-03,6.6448E-03,6.0980E-03,5.6158E-03,5.1925E-03,4.8230E-03,4.5021E-03,
	4.2253E-03,3.9886E-03,3.7884E-03,3.6216E-03,3.4855E-03,3.3840E-03,3.3094E-03,3.2585E-03,
	3.2304E-03,3.2109E-03,3.1561E-03,3.0608E-03,2.9305E-03,2.7717E-03,2.5923E-03,2.3998E-03,
	2.2014E-03,2.0034E-03,1.8108E-03,1.6275E-03,1.4561E-03,1.2981E-03,1.1544E-03,1.0249E-03,
	9.1244E-04,8.2213E-04,7.4241E-04,6.7242E-04,6.1127E-04,5.5807E-04,5.1201E-04,4.7235E-04,
	4.3840E-04,4.0957E-04,3.8534E-04,3.6526E-04,3.4897E-04,3.3712E-04,3.2851E-04,3.2265E-04,
	3.1942E-04,3.1717E-04,3.1089E-04,3.0000E-04,2.8515E-04,2.6718E-04,2.4701E-04,2.2557E-04,
	2.0370E-04,1.8212E-04,1.6140E-04,1.4295E-04,1.2650E-04,1.1155E-04,9.8127E-05,8.6212E-05,
	7.5859E-05,6.7243E-05,5.9771E-05,5.3321E-05,4.7776E-05,4.3036E-05,3.9136E-05,3.5824E-05,
	3.3024E-05,3.0674E-05,2.8720E-05,2.7115E-05,2.5822E-05,2.4837E-05,2.4111E-05,2.3619E-05,
	2.3349E-05,2.3160E-05,2.2636E-05,2.1730E-05,2.0506E-05,1.9039E-05,1.7413E-05,1.5715E-05,
	1.4047E-05,1.2429E-05,1.0904E-05,1.3553E-05},
	/*9 212*/
	{
	4.8659E+00,4.8658E+00,4.8652E+00,4.8642E+00,4.8628E+00,4.8609E+00,4.8587E+00,4.8561E+00,
	4.8531E+00,4.8498E+00,4.8463E+00,4.8424E+00,4.8384E+00,4.8341E+00,4.8298E+00,4.8253E+00,
	4.8208E+00,4.8162E+00,4.8117E+00,4.8072E+00,4.8029E+00,4.7987E+00,4.7947E+00,4.7909E+00,
	4.7874E+00,4.7842E+00,4.7813E+00,4.7788E+00,4.7766E+00,4.7748E+00,4.7734E+00,4.7724E+00,
	4.7719E+00,4.7716E+00,4.7711E+00,4.7701E+00,4.7687E+00,4.7669E+00,4.7648E+00,4.7622E+00,
	4.7594E+00,4.7562E+00,4.7527E+00,4.7490E+00,4.7450E+00,4.7409E+00,4.7367E+00,4.7323E+00,
	4.7279E+00,4.7235E+00,4.7191E+00,4.7148E+00,4.7105E+00,4.7065E+00,4.7026E+00,4.6989E+00,
	4.6955E+00,4.6924E+00,4.6896E+00,4.6871E+00,4.6850E+00,4.6832E+00,4.6819E+00,4.6809E+00,
	4.6804E+00,4.6800E+00,4.6790E+00,4.6771E+00,4.6744E+00,4.6709E+00,4.6667E+00,4.6618E+00,
	4.6562E+00,4.6500E+00,4.6433E+00,4.6361E+00,4.6285E+00,4.6206E+00,4.6124E+00,4.6040E+00,
	4.5956E+00,4.5871E+00,4.5787E+00,4.5704E+00,4.5624E+00,4.5546E+00,4.5472E+00,4.5402E+00,
	4.5338E+00,4.5278E+00,4.5225E+00,4.5178E+00,4.5137E+00,4.5104E+00,4.5079E+00,4.5061E+00,
	4.5051E+00,4.5044E+00,4.5024E+00,4.4989E+00,4.4938E+00,4.4873E+00,4.4793E+00,4.4700E+00,
	4.4596E+00,4.4480E+00,4.4354E+00,4.4220E+00,4.4078E+00,4.3930E+00,4.3779E+00,4.3624E+00,
	4.3468E+00,4.3312E+00,4.3158E+00,4.3006E+00,4.2859E+00,4.2718E+00,4.2583E+00,4.2457E+00,
	4.2339E+00,4.2232E+00,4.2135E+00,4.2051E+00,4.1979E+00,4.1919E+00,4.1874E+00,4.1842E+00,
	4.1824E+00,4.1811E+00,4.1776E+00,4.1712E+00,4.1621E+00,4.1504E+00,4.1363E+00,4.1198E+00,
	4.1013E+00,4.0808E+00,4.0587E+00,4.0351E+00,4.0104E+00,3.9849E+00,3.9586E+00,3.9320E+00,
	3.9053E+00,3.8787E+00,3.8526E+00,3.8270E+00,3.8023E+00,3.7786E+00,3.7562E+00,3.7352E+00,
	3.7158E+00,3.6981E+00,3.6823E+00,3.6684E+00,3.6566E+00,3.6469E+00,3.6395E+00,3.6343E+00,
	3.6314E+00,3.6294E+00,3.6236E+00,3.6133E+00,3.5986E+00,3.5798E+00,3.5571E+00,3.5308E+00,
	3.5013E+00,3.4690E+00,3.4343E+00,3.3976E+00,3.3595E+00,3.3202E+00,3.2803E+00,3.2401E+00,
	3.2001E+00,3.1607E+00,3.1222E+00,3.0848E+00,3.0490E+00,3.0150E+00,2.9830E+00,2.9532E+00,
	2.9258E+00,2.9010E+00,2.8789E+00,2.8597E+00,2.8434E+00,2.8301E+00,2.8199E+00,2.8127E+00,
	2.8088E+00,2.8060E+00,2.7981E+00,2.7841E+00,2.7642E+00,2.7387E+00,2.7083E+00,2.6733E+00,
	2.6343E+00,2.5921E+00,2.5472E+00,2.5004E+00,2.4521E+00,2.4031E+00,2.3540E+00,2.3051E+00,
	2.2571E+00,2.2103E+00,2.1652E+00,2.1220E+00,2.0810E+00,2.0425E+00,2.0067E+00,1.9737E+00,
	1.9437E+00,1.9167E+00,1.8929E+00,1.8722E+00,1.8549E+00,1.8407E+00,1.8299E+00,1.8224E+00,
	1.8183E+00,1.8153E+00,1.8071E+00,1.7924E+00,1.7717E+00,1.7454E+00,1.7143E+00,1.6789E+00,
	1.6400E+00,1.5984E+00,1.5547E+00,1.5099E+00,1.4644E+00,1.4190E+00,1.3742E+00,1.3303E+00,
	1.2880E+00,1.2473E+00,1.2087E+00,1.1724E+00,1.1384E+00,1.1068E+00,1.0779E+00,1.0515E+00,
	1.0278E+00,1.0067E+00,9.8817E-01,9.7228E-01,9.5898E-01,9.4824E-01,9.4005E-01,9.3439E-01,
	9.3125E-01,9.2904E-01,9.2282E-01,9.1185E-01,8.9645E-01,8.7710E-01,8.5435E-01,8.2880E-01,
	8.0109E-01,7.7186E-01,7.4169E-01,7.1113E-01,6.8067E-01,6.5071E-01,6.2160E-01,5.9361E-01,
	5.6695E-01,5.4178E-01,5.1786E-01,4.9445E-01,4.7301E-01,4.5351E-01,4.3590E-01,4.2014E-01,
	4.0616E-01,3.9390E-01,3.8328E-01,3.7426E-01,3.6678E-01,3.6078E-01,3.5624E-01,3.5311E-01,
	3.5138E-01,3.5017E-01,3.4676E-01,3.4077E-01,3.3244E-01,3.2209E-01,3.1009E-01,2.9683E-01,
	2.8271E-01,2.6809E-01,2.5332E-01,2.3868E-01,2.2440E-01,2.1068E-01,1.9764E-01,1.8537E-01,
	1.7394E-01,1.6337E-01,1.5365E-01,1.4474E-01,1.3670E-01,1.2948E-01,1.2305E-01,1.1737E-01,
	1.1238E-01,1.0804E-01,1.0433E-01,1.0119E-01,9.8603E-02,9.6542E-02,9.4986E-02,9.3919E-02,
	9.3329E-02,9.2917E-02,9.1759E-02,8.9733E-02,8.6933E-02,8.3479E-02,7.9510E-02,7.5174E-02,
	7.0610E-02,6.5949E-02,6.1355E-02,5.6885E-02,5.2604E-02,4.8560E-02,4.4786E-02,4.1300E-02,
	3.8106E-02,3.5202E-02,3.2587E-02,3.0274E-02,2.8214E-02,2.6389E-02,2.4781E-02,2.3374E-02,
	2.2151E-02,2.1098E-02,2.0201E-02,1.9449E-02,1.8833E-02,1.8358E-02,1.8003E-02,1.7761E-02,
	1.7627E-02,1.7534E-02,1.7272E-02,1.6816E-02,1.6190E-02,1.5423E-02,1.4550E-02,1.3606E-02,
	1.2626E-02,1.1637E-02,1.0666E-02,9.7302E-03,8.8450E-03,8.0193E-03,7.2584E-03,6.5641E-03,
	5.9455E-03,5.4212E-03,4.9547E-03,4.5419E-03,4.1785E-03,3.8600E-03,3.5823E-03,3.3415E-03,
	3.1340E-03,2.9568E-03,2.8070E-03,2.6822E-03,2.5806E-03,2.5046E-03,2.4487E-03,2.4107E-03,
	2.3897E-03,2.3751E-03,2.3341E-03,2.2629E-03,2.1655E-03,2.0470E-03,1.9132E-03,1.7699E-03,
	1.6223E-03,1.4752E-03,1.3322E-03,1.1964E-03,1.0695E-03,9.5271E-04,8.4659E-04,7.5117E-04,
	6.6825E-04,6.0163E-04,5.4289E-04,4.9137E-04,4.4640E-04,4.0732E-04,3.7351E-04,3.4441E-04,
	3.1953E-04,2.9841E-04,2.8066E-04,2.6597E-04,2.5405E-04,2.4538E-04,2.3909E-04,2.3480E-04,
	2.3245E-04,2.3080E-04,2.2621E-04,2.1825E-04,2.0740E-04,1.9428E-04,1.7957E-04,1.6393E-04,
	1.4798E-04,1.3226E-04,1.1717E-04,1.0374E-04,9.1761E-05,8.0881E-05,7.1124E-05,6.2466E-05,
	5.4944E-05,4.8684E-05,4.3257E-05,3.8575E-05,3.4552E-05,3.1115E-05,2.8291E-05,2.5893E-05,
	2.3867E-05,2.2167E-05,2.0753E-05,1.9592E-05,1.8657E-05,1.7943E-05,1.7417E-05,1.7060E-05,
	1.6864E-05,1.6728E-05,1.6348E-05,1.5692E-05,1.4805E-05,1.3742E-05,1.2565E-05,1.1336E-05,
	1.0132E-05,8.9641E-06,7.8635E-06,9.9096E-06},
	/*10 212*/
	{
	5.6937E+00,5.6935E+00,5.6927E+00,5.6913E+00,5.6892E+00,5.6865E+00,5.6832E+00,5.6794E+00,
	5.6751E+00,5.6703E+00,5.6651E+00,5.6596E+00,5.6537E+00,5.6475E+00,5.6412E+00,5.6347E+00,
	5.6281E+00,5.6215E+00,5.6149E+00,5.6085E+00,5.6022E+00,5.5961E+00,5.5903E+00,5.5848E+00,
	5.5798E+00,5.5751E+00,5.5709E+00,5.5672E+00,5.5640E+00,5.5614E+00,5.5594E+00,5.5580E+00,
	5.5573E+00,5.5569E+00,5.5561E+00,5.5547E+00,5.5527E+00,5.5501E+00,5.5470E+00,5.5433E+00,
	5.5391E+00,5.5345E+00,5.5295E+00,5.5241E+00,5.5184E+00,5.5125E+00,5.5064E+00,5.5001E+00,
	5.4937E+00,5.4874E+00,5.4810E+00,5.4748E+00,5.4687E+00,5.4629E+00,5.4573E+00,5.4520E+00,
	5.4471E+00,5.4426E+00,5.4385E+00,5.4350E+00,5.4319E+00,5.4294E+00,5.4275E+00,5.4261E+00,
	5.4254E+00,5.4249E+00,5.4233E+00,5.4206E+00,5.4168E+00,5.4118E+00,5.4057E+00,5.3987E+00,
	5.3907E+00,5.3818E+00,5.3722E+00,5.3619E+00,5.3510E+00,5.3397E+00,5.3280E+00,5.3160E+00,
	5.3039E+00,5.2919E+00,5.2799E+00,5.2681E+00,5.2566E+00,5.2456E+00,5.2350E+00,5.2251E+00,
	5.2159E+00,5.2074E+00,5.1998E+00,5.1932E+00,5.1875E+00,5.1828E+00,5.1792E+00,5.1767E+00,
	5.1752E+00,5.1743E+00,5.1714E+00,5.1664E+00,5.1592E+00,5.1499E+00,5.1387E+00,5.1256E+00,
	5.1108E+00,5.0944E+00,5.0767E+00,5.0577E+00,5.0378E+00,5.0170E+00,4.9957E+00,4.9740E+00,
	4.9522E+00,4.9304E+00,4.9088E+00,4.8876E+00,4.8671E+00,4.8474E+00,4.8287E+00,4.8111E+00,
	4.7948E+00,4.7799E+00,4.7665E+00,4.7548E+00,4.7448E+00,4.7366E+00,4.7302E+00,4.7258E+00,
	4.7234E+00,4.7216E+00,4.7167E+00,4.7079E+00,4.6954E+00,4.6793E+00,4.6598E+00,4.6371E+00,
	4.6116E+00,4.5835E+00,4.5532E+00,4.5210E+00,4.4873E+00,4.4524E+00,4.4168E+00,4.3807E+00,
	4.3446E+00,4.3087E+00,4.2734E+00,4.2390E+00,4.2059E+00,4.1742E+00,4.1442E+00,4.1162E+00,
	4.0903E+00,4.0668E+00,4.0458E+00,4.0274E+00,4.0117E+00,3.9989E+00,3.9891E+00,3.9822E+00,
	3.9784E+00,3.9757E+00,3.9681E+00,3.9545E+00,3.9351E+00,3.9104E+00,3.8805E+00,3.8460E+00,
	3.8074E+00,3.7653E+00,3.7201E+00,3.6726E+00,3.6232E+00,3.5726E+00,3.5214E+00,3.4700E+00,
	3.4190E+00,3.3689E+00,3.3201E+00,3.2730E+00,3.2280E+00,3.1854E+00,3.1453E+00,3.1082E+00,
	3.0742E+00,3.0435E+00,3.0162E+00,2.9925E+00,2.9724E+00,2.9560E+00,2.9434E+00,2.9347E+00,
	2.9298E+00,2.9264E+00,2.9167E+00,2.8995E+00,2.8752E+00,2.8441E+00,2.8071E+00,2.7646E+00,
	2.7175E+00,2.6667E+00,2.6129E+00,2.5570E+00,2.4997E+00,2.4418E+00,2.3840E+00,2.3268E+00,
	2.2710E+00,2.2168E+00,2.1648E+00,2.1152E+00,2.0685E+00,2.0247E+00,1.9842E+00,1.9470E+00,
	1.9132E+00,1.8830E+00,1.8564E+00,1.8334E+00,1.8141E+00,1.7984E+00,1.7865E+00,1.7782E+00,
	1.7735E+00,1.7703E+00,1.7612E+00,1.7450E+00,1.7222E+00,1.6934E+00,1.6593E+00,1.6207E+00,
	1.5785E+00,1.5335E+00,1.4867E+00,1.4387E+00,1.3905E+00,1.3425E+00,1.2954E+00,1.2497E+00,
	1.2057E+00,1.1637E+00,1.1241E+00,1.0869E+00,1.0524E+00,1.0205E+00,9.9129E-01,9.6483E-01,
	9.4108E-01,9.2003E-01,9.0165E-01,8.8589E-01,8.7273E-01,8.6212E-01,8.5404E-01,8.4846E-01,
	8.4537E-01,8.4320E-01,8.3709E-01,8.2630E-01,8.1121E-01,7.9228E-01,7.7010E-01,7.4528E-01,
	7.1849E-01,6.9033E-01,6.6141E-01,6.3226E-01,6.0333E-01,5.7503E-01,5.4765E-01,5.2145E-01,
	4.9661E-01,4.7325E-01,4.5107E-01,4.2920E-01,4.0927E-01,3.9123E-01,3.7501E-01,3.6055E-01,
	3.4777E-01,3.3660E-01,3.2696E-01,3.1878E-01,3.1202E-01,3.0660E-01,3.0251E-01,2.9969E-01,
	2.9813E-01,2.9704E-01,2.9398E-01,2.8860E-01,2.8114E-01,2.7189E-01,2.6119E-01,2.4942E-01,
	2.3693E-01,2.2407E-01,2.1112E-01,1.9834E-01,1.8595E-01,1.7408E-01,1.6286E-01,1.5235E-01,
	1.4260E-01,1.3362E-01,1.2538E-01,1.1783E-01,1.1104E-01,1.0497E-01,9.9576E-02,9.4818E-02,
	9.0652E-02,8.7042E-02,8.3950E-02,8.1346E-02,7.9202E-02,7.7496E-02,7.6208E-02,7.5326E-02,
	7.4839E-02,7.4498E-02,7.3542E-02,7.1870E-02,6.9562E-02,6.6720E-02,6.3461E-02,5.9907E-02,
	5.6177E-02,5.2376E-02,4.8639E-02,4.5013E-02,4.1549E-02,3.8287E-02,3.5249E-02,3.2450E-02,
	2.9892E-02,2.7571E-02,2.5486E-02,2.3645E-02,2.2008E-02,2.0560E-02,1.9287E-02,1.8174E-02,
	1.7207E-02,1.6375E-02,1.5668E-02,1.5075E-02,1.4590E-02,1.4216E-02,1.3937E-02,1.3747E-02,
	1.3642E-02,1.3569E-02,1.3363E-02,1.3005E-02,1.2514E-02,1.1912E-02,1.1229E-02,1.0491E-02,
	9.7251E-03,8.9540E-03,8.1973E-03,7.4697E-03,6.7822E-03,6.1417E-03,5.5522E-03,5.0151E-03,
	4.5371E-03,4.1328E-03,3.7736E-03,3.4562E-03,3.1771E-03,2.9329E-03,2.7201E-03,2.5357E-03,
	2.3771E-03,2.2416E-03,2.1272E-03,2.0320E-03,1.9544E-03,1.8964E-03,1.8538E-03,1.8248E-03,
	1.8088E-03,1.7976E-03,1.7663E-03,1.7121E-03,1.6378E-03,1.5476E-03,1.4457E-03,1.3366E-03,
	1.2245E-03,1.1127E-03,1.0043E-03,9.0136E-04,8.0528E-04,7.1695E-04,6.3675E-04,5.6469E-04,
	5.0205E-04,4.5149E-04,4.0698E-04,3.6800E-04,3.3401E-04,3.0450E-04,2.7901E-04,2.5709E-04,
	2.3836E-04,2.2248E-04,2.0915E-04,1.9811E-04,1.8916E-04,1.8271E-04,1.7804E-04,1.7485E-04,
	1.7310E-04,1.7188E-04,1.6847E-04,1.6256E-04,1.5451E-04,1.4476E-04,1.3383E-04,1.2221E-04,
	1.1036E-04,9.8672E-05,8.7450E-05,7.7422E-05,6.8460E-05,6.0325E-05,5.3032E-05,4.6563E-05,
	4.0947E-05,3.6274E-05,3.2225E-05,2.8733E-05,2.5734E-05,2.3171E-05,2.1063E-05,1.9274E-05,
	1.7762E-05,1.6494E-05,1.5440E-05,1.4574E-05,1.3878E-05,1.3346E-05,1.2954E-05,1.2688E-05,
	1.2542E-05,1.2440E-05,1.2157E-05,1.1669E-05,1.1008E-05,1.0216E-05,9.3398E-06,8.4254E-06,
	7.5298E-06,6.6616E-06,5.8435E-06,7.4263E-06}};

double SingD[8][460] ={
	/*3 202*/
	{
	7.0680E-01,7.0048E-01,6.9617E-01,6.9280E-01,6.8992E-01,6.8737E-01,6.8507E-01,6.8296E-01,
	6.8101E-01,6.7919E-01,6.7750E-01,6.7592E-01,6.7444E-01,6.7305E-01,6.7174E-01,6.7052E-01,
	6.6937E-01,6.6830E-01,6.6730E-01,6.6637E-01,6.6551E-01,6.6472E-01,6.6400E-01,6.6334E-01,
	6.6274E-01,6.6222E-01,6.6175E-01,6.6135E-01,6.6102E-01,6.6075E-01,6.6054E-01,6.6040E-01,
	6.6032E-01,6.6028E-01,6.6020E-01,6.6006E-01,6.5986E-01,6.5961E-01,6.5930E-01,6.5895E-01,
	6.5856E-01,6.5813E-01,6.5767E-01,6.5719E-01,6.5670E-01,6.5619E-01,6.5568E-01,6.5517E-01,
	6.5466E-01,6.5416E-01,6.5368E-01,6.5321E-01,6.5276E-01,6.5233E-01,6.5193E-01,6.5156E-01,
	6.5121E-01,6.5090E-01,6.5063E-01,6.5038E-01,6.5018E-01,6.5001E-01,6.4988E-01,6.4980E-01,
	6.4975E-01,6.4971E-01,6.4961E-01,6.4944E-01,6.4918E-01,6.4886E-01,6.4848E-01,6.4804E-01,
	6.4754E-01,6.4700E-01,6.4643E-01,6.4583E-01,6.4520E-01,6.4456E-01,6.4391E-01,6.4326E-01,
	6.4262E-01,6.4198E-01,6.4136E-01,6.4076E-01,6.4018E-01,6.3964E-01,6.3912E-01,6.3864E-01,
	6.3820E-01,6.3780E-01,6.3744E-01,6.3713E-01,6.3687E-01,6.3665E-01,6.3649E-01,6.3637E-01,
	6.3631E-01,6.3626E-01,6.3613E-01,6.3591E-01,6.3558E-01,6.3517E-01,6.3467E-01,6.3409E-01,
	6.3345E-01,6.3274E-01,6.3199E-01,6.3120E-01,6.3038E-01,6.2954E-01,6.2868E-01,6.2782E-01,
	6.2697E-01,6.2612E-01,6.2530E-01,6.2449E-01,6.2372E-01,6.2299E-01,6.2230E-01,6.2166E-01,
	6.2106E-01,6.2052E-01,6.2004E-01,6.1962E-01,6.1926E-01,6.1897E-01,6.1874E-01,6.1859E-01,
	6.1850E-01,6.1844E-01,6.1826E-01,6.1795E-01,6.1751E-01,6.1694E-01,6.1626E-01,6.1548E-01,
	6.1459E-01,6.1363E-01,6.1259E-01,6.1150E-01,6.1036E-01,6.0919E-01,6.0800E-01,6.0680E-01,
	6.0560E-01,6.0442E-01,6.0325E-01,6.0212E-01,6.0103E-01,5.9999E-01,5.9901E-01,5.9809E-01,
	5.9724E-01,5.9647E-01,5.9578E-01,5.9518E-01,5.9466E-01,5.9424E-01,5.9392E-01,5.9369E-01,
	5.9357E-01,5.9348E-01,5.9323E-01,5.9278E-01,5.9214E-01,5.9133E-01,5.9034E-01,5.8920E-01,
	5.8792E-01,5.8651E-01,5.8500E-01,5.8340E-01,5.8173E-01,5.8001E-01,5.7825E-01,5.7647E-01,
	5.7470E-01,5.7293E-01,5.7120E-01,5.6951E-01,5.6788E-01,5.6632E-01,5.6484E-01,5.6346E-01,
	5.6218E-01,5.6102E-01,5.5998E-01,5.5907E-01,5.5829E-01,5.5765E-01,5.5716E-01,5.5682E-01,
	5.5663E-01,5.5650E-01,5.5612E-01,5.5544E-01,5.5447E-01,5.5323E-01,5.5173E-01,5.5000E-01,
	5.4804E-01,5.4590E-01,5.4359E-01,5.4114E-01,5.3858E-01,5.3594E-01,5.3324E-01,5.3051E-01,
	5.2777E-01,5.2506E-01,5.2239E-01,5.1978E-01,5.1727E-01,5.1486E-01,5.1259E-01,5.1045E-01,
	5.0848E-01,5.0669E-01,5.0508E-01,5.0367E-01,5.0247E-01,5.0149E-01,5.0074E-01,5.0021E-01,
	4.9992E-01,4.9971E-01,4.9913E-01,4.9808E-01,4.9659E-01,4.9468E-01,4.9237E-01,4.8970E-01,
	4.8669E-01,4.8340E-01,4.7986E-01,4.7610E-01,4.7219E-01,4.6815E-01,4.6403E-01,4.5988E-01,
	4.5573E-01,4.5162E-01,4.4758E-01,4.4366E-01,4.3989E-01,4.3628E-01,4.3288E-01,4.2970E-01,
	4.2677E-01,4.2411E-01,4.2173E-01,4.1965E-01,4.1788E-01,4.1644E-01,4.1533E-01,4.1455E-01,
	4.1412E-01,4.1382E-01,4.1296E-01,4.1143E-01,4.0925E-01,4.0645E-01,4.0309E-01,3.9921E-01,
	3.9487E-01,3.9014E-01,3.8507E-01,3.7973E-01,3.7420E-01,3.6853E-01,3.6278E-01,3.5703E-01,
	3.5132E-01,3.4571E-01,3.4024E-01,3.3497E-01,3.2993E-01,3.2515E-01,3.2067E-01,3.1651E-01,
	3.1270E-01,3.0925E-01,3.0619E-01,3.0353E-01,3.0127E-01,2.9944E-01,2.9803E-01,2.9705E-01,
	2.9650E-01,2.9612E-01,2.9503E-01,2.9310E-01,2.9037E-01,2.8688E-01,2.8272E-01,2.7795E-01,
	2.7266E-01,2.6695E-01,2.6090E-01,2.5461E-01,2.4816E-01,2.4165E-01,2.3513E-01,2.2870E-01,
	2.2240E-01,2.1630E-01,2.1044E-01,2.0485E-01,1.9959E-01,1.9466E-01,1.9009E-01,1.8590E-01,
	1.8210E-01,1.7871E-01,1.7571E-01,1.7313E-01,1.7095E-01,1.6919E-01,1.6785E-01,1.6692E-01,
	1.6640E-01,1.6604E-01,1.6501E-01,1.6319E-01,1.6063E-01,1.5740E-01,1.5358E-01,1.4927E-01,
	1.4455E-01,1.3954E-01,1.3432E-01,1.2900E-01,1.2365E-01,1.1835E-01,1.1317E-01,1.0815E-01,
	1.0335E-01,9.8789E-02,9.4499E-02,9.0497E-02,8.6791E-02,8.3389E-02,8.0292E-02,7.7497E-02,
	7.5000E-02,7.2797E-02,7.0881E-02,6.9244E-02,6.7881E-02,6.6787E-02,6.5955E-02,6.5381E-02,
	6.5063E-02,6.4841E-02,6.4214E-02,6.3110E-02,6.1571E-02,5.9651E-02,5.7414E-02,5.4929E-02,
	5.2268E-02,4.9501E-02,4.6689E-02,4.3889E-02,4.1149E-02,3.8505E-02,3.5986E-02,3.3615E-02,
	3.1403E-02,2.9359E-02,2.7484E-02,2.5777E-02,2.4234E-02,2.2850E-02,2.1615E-02,2.0523E-02,
	1.9565E-02,1.8734E-02,1.8021E-02,1.7421E-02,1.6926E-02,1.6533E-02,1.6236E-02,1.6032E-02,
	1.5920E-02,1.5841E-02,1.5621E-02,1.5235E-02,1.4703E-02,1.4049E-02,1.3299E-02,1.2483E-02,
	1.1630E-02,1.0764E-02,9.9072E-03,9.0779E-03,8.2895E-03,7.5513E-03,6.8691E-03,6.2457E-03,
	5.6813E-03,5.1747E-03,4.7231E-03,4.3232E-03,3.9710E-03,3.6627E-03,3.3942E-03,3.1618E-03,
	2.9620E-03,2.7916E-03,2.6480E-03,2.5287E-03,2.4317E-03,2.3553E-03,2.2980E-03,2.2591E-03,
	2.2376E-03,2.2227E-03,2.1808E-03,2.1083E-03,2.0093E-03,1.8894E-03,1.7547E-03,1.6114E-03,
	1.4652E-03,1.3208E-03,1.1822E-03,1.0522E-03,9.3251E-04,8.2412E-04,7.2727E-04,6.4170E-04,
	5.6679E-04,5.0172E-04,4.4556E-04,3.9733E-04,3.5611E-04,3.2102E-04,2.9127E-04,2.6614E-04,
	2.4502E-04,2.2738E-04,2.1278E-04,2.0085E-04,1.9128E-04,1.8382E-04,1.7829E-04,1.7455E-04,
	1.7250E-04,1.7108E-04,1.6711E-04,1.6028E-04,1.5108E-04,1.4013E-04,1.2808E-04,1.1557E-04,
	1.0315E-04,9.1258E-05,8.0190E-05,7.0136E-05},
	/*4 202*/
	{
	1.2766E+00,1.2766E+00,1.2766E+00,1.2765E+00,1.2764E+00,1.2763E+00,1.2762E+00,1.2760E+00,
	1.2758E+00,1.2756E+00,1.2754E+00,1.2752E+00,1.2750E+00,1.2747E+00,1.2745E+00,1.2742E+00,
	1.2739E+00,1.2737E+00,1.2734E+00,1.2731E+00,1.2729E+00,1.2726E+00,1.2724E+00,1.2722E+00,
	1.2719E+00,1.2717E+00,1.2716E+00,1.2714E+00,1.2713E+00,1.2712E+00,1.2711E+00,1.2710E+00,
	1.2710E+00,1.2710E+00,1.2710E+00,1.2709E+00,1.2708E+00,1.2707E+00,1.2706E+00,1.2704E+00,
	1.2703E+00,1.2701E+00,1.2699E+00,1.2696E+00,1.2694E+00,1.2691E+00,1.2689E+00,1.2686E+00,
	1.2684E+00,1.2681E+00,1.2678E+00,1.2676E+00,1.2673E+00,1.2671E+00,1.2668E+00,1.2666E+00,
	1.2664E+00,1.2662E+00,1.2660E+00,1.2659E+00,1.2657E+00,1.2656E+00,1.2655E+00,1.2655E+00,
	1.2655E+00,1.2654E+00,1.2654E+00,1.2653E+00,1.2651E+00,1.2649E+00,1.2646E+00,1.2643E+00,
	1.2640E+00,1.2636E+00,1.2632E+00,1.2627E+00,1.2622E+00,1.2617E+00,1.2612E+00,1.2607E+00,
	1.2602E+00,1.2597E+00,1.2591E+00,1.2586E+00,1.2581E+00,1.2576E+00,1.2571E+00,1.2567E+00,
	1.2563E+00,1.2559E+00,1.2556E+00,1.2552E+00,1.2550E+00,1.2548E+00,1.2546E+00,1.2545E+00,
	1.2544E+00,1.2544E+00,1.2543E+00,1.2540E+00,1.2537E+00,1.2533E+00,1.2528E+00,1.2522E+00,
	1.2515E+00,1.2507E+00,1.2499E+00,1.2490E+00,1.2481E+00,1.2471E+00,1.2461E+00,1.2451E+00,
	1.2440E+00,1.2430E+00,1.2420E+00,1.2409E+00,1.2399E+00,1.2390E+00,1.2380E+00,1.2372E+00,
	1.2364E+00,1.2356E+00,1.2350E+00,1.2344E+00,1.2339E+00,1.2334E+00,1.2331E+00,1.2329E+00,
	1.2328E+00,1.2327E+00,1.2324E+00,1.2320E+00,1.2314E+00,1.2305E+00,1.2295E+00,1.2284E+00,
	1.2270E+00,1.2256E+00,1.2240E+00,1.2222E+00,1.2204E+00,1.2185E+00,1.2166E+00,1.2146E+00,
	1.2126E+00,1.2105E+00,1.2085E+00,1.2065E+00,1.2046E+00,1.2028E+00,1.2010E+00,1.1993E+00,
	1.1978E+00,1.1963E+00,1.1950E+00,1.1939E+00,1.1929E+00,1.1922E+00,1.1915E+00,1.1911E+00,
	1.1909E+00,1.1907E+00,1.1902E+00,1.1894E+00,1.1882E+00,1.1866E+00,1.1847E+00,1.1824E+00,
	1.1799E+00,1.1771E+00,1.1741E+00,1.1708E+00,1.1674E+00,1.1638E+00,1.1601E+00,1.1564E+00,
	1.1526E+00,1.1488E+00,1.1450E+00,1.1413E+00,1.1378E+00,1.1343E+00,1.1310E+00,1.1279E+00,
	1.1250E+00,1.1224E+00,1.1200E+00,1.1180E+00,1.1162E+00,1.1147E+00,1.1136E+00,1.1128E+00,
	1.1124E+00,1.1121E+00,1.1112E+00,1.1096E+00,1.1074E+00,1.1045E+00,1.1010E+00,1.0970E+00,
	1.0924E+00,1.0873E+00,1.0818E+00,1.0760E+00,1.0698E+00,1.0634E+00,1.0569E+00,1.0502E+00,
	1.0435E+00,1.0369E+00,1.0303E+00,1.0238E+00,1.0176E+00,1.0116E+00,1.0059E+00,1.0006E+00,
	9.9563E-01,9.9112E-01,9.8708E-01,9.8354E-01,9.8052E-01,9.7805E-01,9.7614E-01,9.7482E-01,
	9.7407E-01,9.7355E-01,9.7207E-01,9.6943E-01,9.6566E-01,9.6082E-01,9.5497E-01,9.4819E-01,
	9.4056E-01,9.3219E-01,9.2317E-01,9.1361E-01,9.0363E-01,8.9333E-01,8.8283E-01,8.7224E-01,
	8.6165E-01,8.5117E-01,8.4089E-01,8.3091E-01,8.2130E-01,8.1215E-01,8.0351E-01,7.9545E-01,
	7.8803E-01,7.8129E-01,7.7528E-01,7.7003E-01,7.6557E-01,7.6193E-01,7.5913E-01,7.5718E-01,
	7.5609E-01,7.5533E-01,7.5316E-01,7.4931E-01,7.4383E-01,7.3682E-01,7.2839E-01,7.1869E-01,
	7.0787E-01,6.9608E-01,6.8351E-01,6.7032E-01,6.5668E-01,6.4277E-01,6.2874E-01,6.1475E-01,
	6.0092E-01,5.8739E-01,5.7439E-01,5.6234E-01,5.5088E-01,5.4007E-01,5.2998E-01,5.2067E-01,
	5.1216E-01,5.0451E-01,4.9773E-01,4.9186E-01,4.8690E-01,4.8287E-01,4.7978E-01,4.7763E-01,
	4.7644E-01,4.7560E-01,4.7323E-01,4.6903E-01,4.6308E-01,4.5554E-01,4.4656E-01,4.3633E-01,
	4.2507E-01,4.1297E-01,4.0026E-01,3.8714E-01,3.7380E-01,3.6043E-01,3.4719E-01,3.3421E-01,
	3.2162E-01,3.0953E-01,2.9810E-01,2.8768E-01,2.7792E-01,2.6887E-01,2.6056E-01,2.5298E-01,
	2.4616E-01,2.4010E-01,2.3478E-01,2.3022E-01,2.2640E-01,2.2331E-01,2.2096E-01,2.1934E-01,
	2.1844E-01,2.1780E-01,2.1602E-01,2.1287E-01,2.0845E-01,2.0290E-01,1.9638E-01,1.8907E-01,
	1.8115E-01,1.7281E-01,1.6448E-01,1.5612E-01,1.4782E-01,1.3970E-01,1.3186E-01,1.2436E-01,
	1.1728E-01,1.1063E-01,1.0448E-01,9.8938E-02,9.3863E-02,8.9254E-02,8.5099E-02,8.1385E-02,
	7.8096E-02,7.5216E-02,7.2727E-02,7.0615E-02,6.8865E-02,6.7500E-02,6.6477E-02,6.5773E-02,
	6.5383E-02,6.5111E-02,6.4344E-02,6.3000E-02,6.1134E-02,5.8820E-02,5.6146E-02,5.3204E-02,
	5.0085E-02,4.6875E-02,4.3652E-02,4.0481E-02,3.7414E-02,3.4492E-02,3.1742E-02,2.9184E-02,
	2.6855E-02,2.4823E-02,2.2980E-02,2.1319E-02,1.9831E-02,1.8507E-02,1.7336E-02,1.6308E-02,
	1.5412E-02,1.4639E-02,1.3980E-02,1.3427E-02,1.2974E-02,1.2636E-02,1.2387E-02,1.2217E-02,
	1.2123E-02,1.2057E-02,1.1873E-02,1.1551E-02,1.1109E-02,1.0566E-02,9.9464E-03,9.2749E-03,
	8.5750E-03,7.8680E-03,7.1719E-03,6.5010E-03,5.8659E-03,5.2737E-03,4.7288E-03,4.2328E-03,
	3.7971E-03,3.4434E-03,3.1282E-03,2.8490E-03,2.6030E-03,2.3874E-03,2.1996E-03,2.0369E-03,
	1.8969E-03,1.7775E-03,1.6767E-03,1.5929E-03,1.5248E-03,1.4751E-03,1.4391E-03,1.4144E-03,
	1.4009E-03,1.3914E-03,1.3650E-03,1.3190E-03,1.2562E-03,1.1798E-03,1.0937E-03,1.0017E-03,
	9.0723E-04,8.1346E-04,7.2287E-04,6.4215E-04,5.6983E-04,5.0360E-04,4.4372E-04,3.9019E-04,
	3.4353E-04,3.0494E-04,2.7127E-04,2.4207E-04,2.1687E-04,1.9525E-04,1.7764E-04,1.6264E-04,
	1.4992E-04,1.3923E-04,1.3031E-04,1.2297E-04,1.1706E-04,1.1258E-04,1.0928E-04,1.0704E-04,
	1.0581E-04,1.0495E-04,1.0256E-04,9.8440E-05,9.2855E-05,8.6152E-05,7.8711E-05,7.0942E-05,
	6.3380E-05,5.6028E-05,4.9084E-05,5.3794E-05},
	/*5 202*/
	{
	1.9028E+00,1.9028E+00,1.9028E+00,1.9026E+00,1.9025E+00,1.9022E+00,1.9019E+00,1.9016E+00,
	1.9012E+00,1.9008E+00,1.9004E+00,1.8999E+00,1.8994E+00,1.8988E+00,1.8983E+00,1.8977E+00,
	1.8971E+00,1.8966E+00,1.8960E+00,1.8954E+00,1.8949E+00,1.8943E+00,1.8938E+00,1.8933E+00,
	1.8929E+00,1.8925E+00,1.8921E+00,1.8918E+00,1.8915E+00,1.8913E+00,1.8911E+00,1.8910E+00,
	1.8909E+00,1.8909E+00,1.8908E+00,1.8907E+00,1.8905E+00,1.8903E+00,1.8900E+00,1.8897E+00,
	1.8893E+00,1.8889E+00,1.8884E+00,1.8880E+00,1.8875E+00,1.8869E+00,1.8864E+00,1.8858E+00,
	1.8852E+00,1.8847E+00,1.8841E+00,1.8835E+00,1.8830E+00,1.8825E+00,1.8820E+00,1.8815E+00,
	1.8810E+00,1.8806E+00,1.8803E+00,1.8799E+00,1.8797E+00,1.8794E+00,1.8792E+00,1.8791E+00,
	1.8791E+00,1.8790E+00,1.8789E+00,1.8786E+00,1.8783E+00,1.8778E+00,1.8773E+00,1.8766E+00,
	1.8759E+00,1.8751E+00,1.8742E+00,1.8732E+00,1.8722E+00,1.8712E+00,1.8701E+00,1.8690E+00,
	1.8679E+00,1.8667E+00,1.8656E+00,1.8645E+00,1.8634E+00,1.8624E+00,1.8614E+00,1.8604E+00,
	1.8596E+00,1.8587E+00,1.8580E+00,1.8574E+00,1.8568E+00,1.8564E+00,1.8560E+00,1.8558E+00,
	1.8557E+00,1.8556E+00,1.8553E+00,1.8548E+00,1.8541E+00,1.8532E+00,1.8521E+00,1.8509E+00,
	1.8494E+00,1.8478E+00,1.8461E+00,1.8442E+00,1.8423E+00,1.8402E+00,1.8381E+00,1.8359E+00,
	1.8337E+00,1.8315E+00,1.8293E+00,1.8272E+00,1.8251E+00,1.8231E+00,1.8211E+00,1.8193E+00,
	1.8176E+00,1.8160E+00,1.8146E+00,1.8134E+00,1.8124E+00,1.8115E+00,1.8108E+00,1.8103E+00,
	1.8101E+00,1.8099E+00,1.8094E+00,1.8084E+00,1.8071E+00,1.8054E+00,1.8033E+00,1.8008E+00,
	1.7981E+00,1.7950E+00,1.7917E+00,1.7881E+00,1.7843E+00,1.7804E+00,1.7763E+00,1.7722E+00,
	1.7680E+00,1.7638E+00,1.7597E+00,1.7556E+00,1.7516E+00,1.7478E+00,1.7441E+00,1.7407E+00,
	1.7375E+00,1.7346E+00,1.7319E+00,1.7296E+00,1.7277E+00,1.7260E+00,1.7248E+00,1.7239E+00,
	1.7234E+00,1.7231E+00,1.7221E+00,1.7204E+00,1.7179E+00,1.7146E+00,1.7107E+00,1.7062E+00,
	1.7010E+00,1.6954E+00,1.6892E+00,1.6826E+00,1.6757E+00,1.6685E+00,1.6611E+00,1.6536E+00,
	1.6460E+00,1.6384E+00,1.6309E+00,1.6235E+00,1.6164E+00,1.6095E+00,1.6030E+00,1.5969E+00,
	1.5912E+00,1.5860E+00,1.5814E+00,1.5773E+00,1.5738E+00,1.5709E+00,1.5687E+00,1.5672E+00,
	1.5663E+00,1.5657E+00,1.5640E+00,1.5610E+00,1.5566E+00,1.5510E+00,1.5442E+00,1.5363E+00,
	1.5274E+00,1.5176E+00,1.5070E+00,1.4958E+00,1.4840E+00,1.4718E+00,1.4594E+00,1.4468E+00,
	1.4341E+00,1.4216E+00,1.4092E+00,1.3972E+00,1.3856E+00,1.3745E+00,1.3640E+00,1.3541E+00,
	1.3451E+00,1.3368E+00,1.3294E+00,1.3230E+00,1.3175E+00,1.3130E+00,1.3095E+00,1.3071E+00,
	1.3058E+00,1.3048E+00,1.3022E+00,1.2974E+00,1.2906E+00,1.2819E+00,1.2714E+00,1.2592E+00,
	1.2457E+00,1.2308E+00,1.2149E+00,1.1981E+00,1.1807E+00,1.1629E+00,1.1448E+00,1.1266E+00,
	1.1086E+00,1.0909E+00,1.0736E+00,1.0569E+00,1.0409E+00,1.0258E+00,1.0116E+00,9.9839E-01,
	9.8630E-01,9.7537E-01,9.6565E-01,9.5720E-01,9.5004E-01,9.4421E-01,9.3973E-01,9.3662E-01,
	9.3489E-01,9.3367E-01,9.3022E-01,9.2409E-01,9.1540E-01,9.0433E-01,8.9108E-01,8.7592E-01,
	8.5909E-01,8.4090E-01,8.2164E-01,8.0159E-01,7.8103E-01,7.6023E-01,7.3944E-01,7.1887E-01,
	6.9873E-01,6.7919E-01,6.6046E-01,6.4287E-01,6.2628E-01,6.1077E-01,5.9641E-01,5.8324E-01,
	5.7131E-01,5.6064E-01,5.5124E-01,5.4312E-01,5.3631E-01,5.3078E-01,5.2656E-01,5.2364E-01,
	5.2202E-01,5.2088E-01,5.1766E-01,5.1196E-01,5.0394E-01,4.9381E-01,4.8183E-01,4.6829E-01,
	4.5351E-01,4.3779E-01,4.2143E-01,4.0472E-01,3.8793E-01,3.7128E-01,3.5497E-01,3.3917E-01,
	3.2401E-01,3.0959E-01,2.9609E-01,2.8376E-01,2.7236E-01,2.6188E-01,2.5233E-01,2.4371E-01,
	2.3600E-01,2.2920E-01,2.2327E-01,2.1821E-01,2.1399E-01,2.1060E-01,2.0802E-01,2.0624E-01,
	2.0526E-01,2.0457E-01,2.0262E-01,1.9920E-01,1.9442E-01,1.8844E-01,1.8147E-01,1.7372E-01,
	1.6539E-01,1.5671E-01,1.4808E-01,1.3950E-01,1.3107E-01,1.2292E-01,1.1512E-01,1.0775E-01,
	1.0085E-01,9.4443E-02,8.8568E-02,8.3313E-02,7.8548E-02,7.4257E-02,7.0420E-02,6.7014E-02,
	6.4018E-02,6.1408E-02,5.9166E-02,5.7271E-02,5.5707E-02,5.4489E-02,5.3577E-02,5.2951E-02,
	5.2605E-02,5.2364E-02,5.1684E-02,5.0496E-02,4.8853E-02,4.6826E-02,4.4497E-02,4.1952E-02,
	3.9276E-02,3.6544E-02,3.3825E-02,3.1173E-02,2.8630E-02,2.6229E-02,2.3989E-02,2.1922E-02,
	2.0054E-02,1.8434E-02,1.6975E-02,1.5669E-02,1.4508E-02,1.3480E-02,1.2576E-02,1.1787E-02,
	1.1102E-02,1.0513E-02,1.0012E-02,9.5940E-03,9.2517E-03,8.9966E-03,8.8092E-03,8.6810E-03,
	8.6104E-03,8.5611E-03,8.4228E-03,8.1820E-03,7.8512E-03,7.4470E-03,6.9878E-03,6.4927E-03,
	5.9794E-03,5.4640E-03,4.9596E-03,4.4764E-03,4.0218E-03,3.6006E-03,3.2152E-03,2.8664E-03,
	2.5619E-03,2.3168E-03,2.0988E-03,1.9063E-03,1.7370E-03,1.5889E-03,1.4601E-03,1.3487E-03,
	1.2530E-03,1.1714E-03,1.1027E-03,1.0456E-03,9.9913E-04,9.6570E-04,9.4149E-04,9.2499E-04,
	9.1592E-04,9.0958E-04,8.9186E-04,8.6110E-04,8.1910E-04,7.6814E-04,7.1077E-04,6.4956E-04,
	5.8689E-04,5.2481E-04,4.6497E-04,4.1187E-04,3.6453E-04,3.2135E-04,2.8247E-04,2.4783E-04,
	2.1772E-04,1.9283E-04,1.7119E-04,1.5247E-04,1.3636E-04,1.2258E-04,1.1136E-04,1.0182E-04,
	9.3752E-05,8.6973E-05,8.1331E-05,7.6697E-05,7.2965E-05,7.0134E-05,6.8052E-05,6.6639E-05,
	6.5864E-05,6.5324E-05,6.3818E-05,6.1221E-05,5.7705E-05,5.3492E-05,4.8822E-05,4.3954E-05,
	3.9215E-05,3.4619E-05,3.0286E-05,3.5747E-05},
	/*6 202*/
	{
	2.5377E+00,2.5377E+00,2.5376E+00,2.5374E+00,2.5370E+00,2.5366E+00,2.5361E+00,2.5355E+00,
	2.5348E+00,2.5340E+00,2.5332E+00,2.5323E+00,2.5313E+00,2.5303E+00,2.5293E+00,2.5283E+00,
	2.5272E+00,2.5261E+00,2.5251E+00,2.5240E+00,2.5230E+00,2.5220E+00,2.5211E+00,2.5202E+00,
	2.5193E+00,2.5186E+00,2.5179E+00,2.5173E+00,2.5168E+00,2.5163E+00,2.5160E+00,2.5158E+00,
	2.5157E+00,2.5156E+00,2.5155E+00,2.5152E+00,2.5149E+00,2.5145E+00,2.5140E+00,2.5134E+00,
	2.5127E+00,2.5119E+00,2.5111E+00,2.5102E+00,2.5093E+00,2.5083E+00,2.5073E+00,2.5063E+00,
	2.5052E+00,2.5042E+00,2.5031E+00,2.5021E+00,2.5011E+00,2.5001E+00,2.4992E+00,2.4983E+00,
	2.4975E+00,2.4967E+00,2.4960E+00,2.4954E+00,2.4949E+00,2.4945E+00,2.4942E+00,2.4940E+00,
	2.4938E+00,2.4937E+00,2.4935E+00,2.4930E+00,2.4924E+00,2.4916E+00,2.4905E+00,2.4893E+00,
	2.4880E+00,2.4865E+00,2.4849E+00,2.4831E+00,2.4813E+00,2.4794E+00,2.4774E+00,2.4753E+00,
	2.4733E+00,2.4712E+00,2.4691E+00,2.4671E+00,2.4651E+00,2.4632E+00,2.4614E+00,2.4597E+00,
	2.4581E+00,2.4566E+00,2.4553E+00,2.4541E+00,2.4531E+00,2.4523E+00,2.4517E+00,2.4512E+00,
	2.4510E+00,2.4508E+00,2.4503E+00,2.4494E+00,2.4482E+00,2.4465E+00,2.4446E+00,2.4423E+00,
	2.4396E+00,2.4367E+00,2.4336E+00,2.4302E+00,2.4266E+00,2.4229E+00,2.4190E+00,2.4151E+00,
	2.4111E+00,2.4071E+00,2.4032E+00,2.3993E+00,2.3955E+00,2.3918E+00,2.3883E+00,2.3850E+00,
	2.3819E+00,2.3791E+00,2.3766E+00,2.3744E+00,2.3725E+00,2.3709E+00,2.3697E+00,2.3689E+00,
	2.3684E+00,2.3681E+00,2.3671E+00,2.3655E+00,2.3631E+00,2.3600E+00,2.3562E+00,2.3518E+00,
	2.3468E+00,2.3413E+00,2.3354E+00,2.3290E+00,2.3223E+00,2.3153E+00,2.3081E+00,2.3007E+00,
	2.2933E+00,2.2859E+00,2.2785E+00,2.2713E+00,2.2643E+00,2.2575E+00,2.2511E+00,2.2450E+00,
	2.2394E+00,2.2343E+00,2.2296E+00,2.2256E+00,2.2221E+00,2.2193E+00,2.2171E+00,2.2156E+00,
	2.2147E+00,2.2141E+00,2.2124E+00,2.2094E+00,2.2050E+00,2.1994E+00,2.1926E+00,2.1847E+00,
	2.1757E+00,2.1659E+00,2.1552E+00,2.1439E+00,2.1319E+00,2.1196E+00,2.1069E+00,2.0940E+00,
	2.0810E+00,2.0681E+00,2.0554E+00,2.0429E+00,2.0308E+00,2.0193E+00,2.0083E+00,1.9981E+00,
	1.9885E+00,1.9799E+00,1.9721E+00,1.9653E+00,1.9595E+00,1.9547E+00,1.9511E+00,1.9485E+00,
	1.9471E+00,1.9461E+00,1.9433E+00,1.9382E+00,1.9310E+00,1.9217E+00,1.9104E+00,1.8975E+00,
	1.8829E+00,1.8668E+00,1.8496E+00,1.8314E+00,1.8123E+00,1.7927E+00,1.7728E+00,1.7526E+00,
	1.7325E+00,1.7127E+00,1.6932E+00,1.6743E+00,1.6562E+00,1.6389E+00,1.6226E+00,1.6074E+00,
	1.5934E+00,1.5808E+00,1.5694E+00,1.5596E+00,1.5512E+00,1.5444E+00,1.5391E+00,1.5355E+00,
	1.5334E+00,1.5320E+00,1.5279E+00,1.5207E+00,1.5104E+00,1.4973E+00,1.4816E+00,1.4634E+00,
	1.4432E+00,1.4212E+00,1.3978E+00,1.3733E+00,1.3480E+00,1.3221E+00,1.2962E+00,1.2703E+00,
	1.2447E+00,1.2197E+00,1.1956E+00,1.1724E+00,1.1503E+00,1.1295E+00,1.1101E+00,1.0922E+00,
	1.0759E+00,1.0612E+00,1.0481E+00,1.0368E+00,1.0273E+00,1.0195E+00,1.0136E+00,1.0095E+00,
	1.0072E+00,1.0056E+00,1.0010E+00,9.9293E-01,9.8150E-01,9.6699E-01,9.4972E-01,9.3005E-01,
	9.0837E-01,8.8509E-01,8.6061E-01,8.3533E-01,8.0962E-01,7.8382E-01,7.5823E-01,7.3314E-01,
	7.0877E-01,6.8533E-01,6.6293E-01,6.4167E-01,6.2180E-01,6.0337E-01,5.8643E-01,5.7102E-01,
	5.5714E-01,5.4479E-01,5.3398E-01,5.2469E-01,5.1691E-01,5.1063E-01,5.0585E-01,5.0254E-01,
	5.0070E-01,4.9942E-01,4.9578E-01,4.8937E-01,4.8038E-01,4.6908E-01,4.5579E-01,4.4089E-01,
	4.2473E-01,4.0770E-01,3.9015E-01,3.7239E-01,3.5471E-01,3.3735E-01,3.2052E-01,3.0437E-01,
	2.8903E-01,2.7457E-01,2.6111E-01,2.4880E-01,2.3749E-01,2.2719E-01,2.1788E-01,2.0953E-01,
	2.0211E-01,1.9559E-01,1.8994E-01,1.8514E-01,1.8115E-01,1.7796E-01,1.7553E-01,1.7386E-01,
	1.7294E-01,1.7229E-01,1.7047E-01,1.6727E-01,1.6282E-01,1.5729E-01,1.5087E-01,1.4376E-01,
	1.3619E-01,1.2835E-01,1.2060E-01,1.1295E-01,1.0550E-01,9.8366E-02,9.1603E-02,8.5264E-02,
	7.9376E-02,7.3952E-02,6.9014E-02,6.4618E-02,6.0655E-02,5.7107E-02,5.3949E-02,5.1160E-02,
	4.8715E-02,4.6595E-02,4.4778E-02,4.3247E-02,4.1987E-02,4.1009E-02,4.0278E-02,3.9777E-02,
	3.9500E-02,3.9307E-02,3.8765E-02,3.7817E-02,3.6511E-02,3.4905E-02,3.3067E-02,3.1068E-02,
	2.8976E-02,2.6853E-02,2.4752E-02,2.2714E-02,2.0772E-02,1.8948E-02,1.7256E-02,1.5702E-02,
	1.4307E-02,1.3105E-02,1.2027E-02,1.1067E-02,1.0216E-02,9.4664E-03,8.8090E-03,8.2362E-03,
	7.7406E-03,7.3157E-03,6.9554E-03,6.6545E-03,6.4088E-03,6.2265E-03,6.0928E-03,6.0014E-03,
	5.9511E-03,5.9160E-03,5.8176E-03,5.6463E-03,5.4115E-03,5.1250E-03,4.8003E-03,4.4510E-03,
	4.0899E-03,3.7284E-03,3.3756E-03,3.0387E-03,2.7227E-03,2.4306E-03,2.1641E-03,1.9236E-03,
	1.7144E-03,1.5469E-03,1.3985E-03,1.2678E-03,1.1532E-03,1.0533E-03,9.6657E-04,8.9171E-04,
	8.2751E-04,7.7289E-04,7.2693E-04,6.8880E-04,6.5782E-04,6.3544E-04,6.1923E-04,6.0819E-04,
	6.0211E-04,5.9787E-04,5.8602E-04,5.6547E-04,5.3743E-04,5.0346E-04,4.6528E-04,4.2463E-04,
	3.8310E-04,3.4206E-04,3.0259E-04,2.6760E-04,2.3646E-04,2.0812E-04,1.8267E-04,1.6004E-04,
	1.4041E-04,1.2420E-04,1.1013E-04,9.7987E-05,8.7551E-05,7.8632E-05,7.1371E-05,6.5204E-05,
	5.9994E-05,5.5621E-05,5.1985E-05,4.9001E-05,4.6598E-05,4.4778E-05,4.3440E-05,4.2532E-05,
	4.2034E-05,4.1687E-05,4.0720E-05,3.9052E-05,3.6795E-05,3.4094E-05,3.1101E-05,2.7983E-05,
	2.4948E-05,2.2008E-05,1.9239E-05,2.3963E-05},
	/*7 202*/
	{
	3.2038E+00,3.2037E+00,3.2035E+00,3.2031E+00,3.2025E+00,3.2018E+00,3.2009E+00,3.1999E+00,
	3.1987E+00,3.1974E+00,3.1960E+00,3.1945E+00,3.1929E+00,3.1912E+00,3.1895E+00,3.1877E+00,
	3.1859E+00,3.1841E+00,3.1823E+00,3.1805E+00,3.1788E+00,3.1771E+00,3.1755E+00,3.1740E+00,
	3.1726E+00,3.1713E+00,3.1701E+00,3.1691E+00,3.1683E+00,3.1675E+00,3.1670E+00,3.1666E+00,
	3.1664E+00,3.1663E+00,3.1661E+00,3.1657E+00,3.1651E+00,3.1644E+00,3.1635E+00,3.1625E+00,
	3.1614E+00,3.1601E+00,3.1587E+00,3.1572E+00,3.1556E+00,3.1540E+00,3.1522E+00,3.1505E+00,
	3.1487E+00,3.1469E+00,3.1452E+00,3.1434E+00,3.1417E+00,3.1401E+00,3.1385E+00,3.1370E+00,
	3.1357E+00,3.1344E+00,3.1333E+00,3.1323E+00,3.1314E+00,3.1307E+00,3.1301E+00,3.1298E+00,
	3.1296E+00,3.1294E+00,3.1290E+00,3.1282E+00,3.1271E+00,3.1257E+00,3.1240E+00,3.1220E+00,
	3.1197E+00,3.1172E+00,3.1145E+00,3.1116E+00,3.1085E+00,3.1052E+00,3.1019E+00,3.0985E+00,
	3.0950E+00,3.0916E+00,3.0881E+00,3.0847E+00,3.0814E+00,3.0782E+00,3.0752E+00,3.0723E+00,
	3.0696E+00,3.0672E+00,3.0650E+00,3.0630E+00,3.0614E+00,3.0600E+00,3.0590E+00,3.0582E+00,
	3.0578E+00,3.0575E+00,3.0567E+00,3.0552E+00,3.0531E+00,3.0504E+00,3.0471E+00,3.0432E+00,
	3.0389E+00,3.0341E+00,3.0288E+00,3.0232E+00,3.0173E+00,3.0111E+00,3.0047E+00,2.9982E+00,
	2.9916E+00,2.9850E+00,2.9784E+00,2.9720E+00,2.9657E+00,2.9597E+00,2.9540E+00,2.9485E+00,
	2.9435E+00,2.9389E+00,2.9347E+00,2.9311E+00,2.9280E+00,2.9254E+00,2.9234E+00,2.9221E+00,
	2.9213E+00,2.9208E+00,2.9192E+00,2.9165E+00,2.9125E+00,2.9075E+00,2.9013E+00,2.8941E+00,
	2.8860E+00,2.8771E+00,2.8674E+00,2.8570E+00,2.8461E+00,2.8347E+00,2.8231E+00,2.8112E+00,
	2.7992E+00,2.7872E+00,2.7754E+00,2.7638E+00,2.7526E+00,2.7417E+00,2.7315E+00,2.7218E+00,
	2.7128E+00,2.7046E+00,2.6973E+00,2.6909E+00,2.6854E+00,2.6808E+00,2.6774E+00,2.6749E+00,
	2.6736E+00,2.6726E+00,2.6699E+00,2.6651E+00,2.6582E+00,2.6493E+00,2.6386E+00,2.6261E+00,
	2.6121E+00,2.5966E+00,2.5799E+00,2.5622E+00,2.5436E+00,2.5243E+00,2.5046E+00,2.4847E+00,
	2.4647E+00,2.4449E+00,2.4254E+00,2.4064E+00,2.3880E+00,2.3704E+00,2.3538E+00,2.3383E+00,
	2.3239E+00,2.3109E+00,2.2992E+00,2.2890E+00,2.2803E+00,2.2732E+00,2.2677E+00,2.2639E+00,
	2.2618E+00,2.2603E+00,2.2560E+00,2.2485E+00,2.2377E+00,2.2239E+00,2.2073E+00,2.1881E+00,
	2.1666E+00,2.1430E+00,2.1178E+00,2.0913E+00,2.0637E+00,2.0353E+00,2.0066E+00,1.9778E+00,
	1.9492E+00,1.9211E+00,1.8936E+00,1.8671E+00,1.8418E+00,1.8177E+00,1.7951E+00,1.7742E+00,
	1.7550E+00,1.7376E+00,1.7221E+00,1.7087E+00,1.6973E+00,1.6880E+00,1.6809E+00,1.6759E+00,
	1.6732E+00,1.6712E+00,1.6658E+00,1.6560E+00,1.6422E+00,1.6245E+00,1.6034E+00,1.5793E+00,
	1.5525E+00,1.5235E+00,1.4928E+00,1.4608E+00,1.4280E+00,1.3948E+00,1.3615E+00,1.3287E+00,
	1.2965E+00,1.2652E+00,1.2351E+00,1.2064E+00,1.1793E+00,1.1539E+00,1.1303E+00,1.1087E+00,
	1.0890E+00,1.0714E+00,1.0558E+00,1.0424E+00,1.0311E+00,1.0219E+00,1.0149E+00,1.0100E+00,
	1.0073E+00,1.0054E+00,1.0000E+00,9.9054E-01,9.7716E-01,9.6022E-01,9.4015E-01,9.1742E-01,
	8.9251E-01,8.6593E-01,8.3817E-01,8.0972E-01,7.8099E-01,7.5238E-01,7.2423E-01,6.9683E-01,
	6.7042E-01,6.4518E-01,6.2116E-01,5.9817E-01,5.7685E-01,5.5722E-01,5.3930E-01,5.2310E-01,
	5.0859E-01,4.9575E-01,4.8455E-01,4.7498E-01,4.6699E-01,4.6056E-01,4.5567E-01,4.5229E-01,
	4.5042E-01,4.4911E-01,4.4541E-01,4.3890E-01,4.2980E-01,4.1841E-01,4.0509E-01,3.9024E-01,
	3.7425E-01,3.5752E-01,3.4040E-01,3.2322E-01,3.0626E-01,2.8975E-01,2.7387E-01,2.5875E-01,
	2.4449E-01,2.3117E-01,2.1882E-01,2.0751E-01,1.9721E-01,1.8788E-01,1.7950E-01,1.7202E-01,
	1.6542E-01,1.5964E-01,1.5466E-01,1.5043E-01,1.4694E-01,1.4414E-01,1.4202E-01,1.4057E-01,
	1.3976E-01,1.3920E-01,1.3762E-01,1.3484E-01,1.3098E-01,1.2621E-01,1.2069E-01,1.1462E-01,
	1.0818E-01,1.0155E-01,9.5005E-02,8.8586E-02,8.2383E-02,7.6471E-02,7.0906E-02,6.5721E-02,
	6.0933E-02,5.6547E-02,5.2576E-02,4.9061E-02,4.5908E-02,4.3098E-02,4.0608E-02,3.8416E-02,
	3.6502E-02,3.4847E-02,3.3433E-02,3.2244E-02,3.1267E-02,3.0505E-02,2.9935E-02,2.9546E-02,
	2.9331E-02,2.9180E-02,2.8759E-02,2.8024E-02,2.7012E-02,2.5771E-02,2.4355E-02,2.2821E-02,
	2.1223E-02,1.9607E-02,1.8014E-02,1.6476E-02,1.5017E-02,1.3653E-02,1.2393E-02,1.1240E-02,
	1.0212E-02,9.3352E-03,8.5520E-03,7.8559E-03,7.2406E-03,6.6994E-03,6.2259E-03,5.8141E-03,
	5.4584E-03,5.1538E-03,4.8958E-03,4.6806E-03,4.5050E-03,4.3747E-03,4.2792E-03,4.2140E-03,
	4.1781E-03,4.1530E-03,4.0828E-03,3.9607E-03,3.7933E-03,3.5893E-03,3.3584E-03,3.1103E-03,
	2.8542E-03,2.5981E-03,2.3486E-03,2.1107E-03,1.8878E-03,1.6821E-03,1.4946E-03,1.3256E-03,
	1.1789E-03,1.0619E-03,9.5856E-04,8.6771E-04,7.8825E-04,7.1907E-04,6.5913E-04,6.0748E-04,
	5.6325E-04,5.2567E-04,4.9407E-04,4.6789E-04,4.4664E-04,4.3130E-04,4.2020E-04,4.1264E-04,
	4.0848E-04,4.0558E-04,3.9747E-04,3.8341E-04,3.6424E-04,3.4104E-04,3.1499E-04,2.8728E-04,
	2.5900E-04,2.3109E-04,2.0427E-04,1.8050E-04,1.5934E-04,1.4011E-04,1.2286E-04,1.0755E-04,
	9.4286E-05,8.3335E-05,7.3844E-05,6.5660E-05,5.8633E-05,5.2633E-05,4.7750E-05,4.3606E-05,
	4.0108E-05,3.7173E-05,3.4734E-05,3.2733E-05,3.1122E-05,2.9900E-05,2.9001E-05,2.8391E-05,
	2.8057E-05,2.7824E-05,2.7174E-05,2.6054E-05,2.4540E-05,2.2728E-05,2.0723E-05,1.8635E-05,
	1.6607E-05,1.4643E-05,1.2796E-05,1.6387E-05},
	/*8 202*/
	{
	3.8927E+00,3.8926E+00,3.8922E+00,3.8916E+00,3.8907E+00,3.8896E+00,3.8882E+00,3.8865E+00,
	3.8847E+00,3.8826E+00,3.8804E+00,3.8780E+00,3.8754E+00,3.8728E+00,3.8700E+00,3.8672E+00,
	3.8644E+00,3.8615E+00,3.8587E+00,3.8559E+00,3.8531E+00,3.8505E+00,3.8480E+00,3.8456E+00,
	3.8434E+00,3.8414E+00,3.8396E+00,3.8379E+00,3.8366E+00,3.8354E+00,3.8346E+00,3.8340E+00,
	3.8336E+00,3.8335E+00,3.8331E+00,3.8325E+00,3.8316E+00,3.8305E+00,3.8291E+00,3.8275E+00,
	3.8257E+00,3.8237E+00,3.8215E+00,3.8192E+00,3.8167E+00,3.8141E+00,3.8114E+00,3.8086E+00,
	3.8059E+00,3.8031E+00,3.8003E+00,3.7976E+00,3.7949E+00,3.7923E+00,3.7899E+00,3.7875E+00,
	3.7854E+00,3.7834E+00,3.7816E+00,3.7800E+00,3.7787E+00,3.7776E+00,3.7767E+00,3.7761E+00,
	3.7758E+00,3.7756E+00,3.7749E+00,3.7737E+00,3.7720E+00,3.7698E+00,3.7671E+00,3.7640E+00,
	3.7605E+00,3.7565E+00,3.7523E+00,3.7477E+00,3.7428E+00,3.7378E+00,3.7326E+00,3.7273E+00,
	3.7219E+00,3.7165E+00,3.7111E+00,3.7058E+00,3.7007E+00,3.6957E+00,3.6910E+00,3.6865E+00,
	3.6824E+00,3.6786E+00,3.6751E+00,3.6721E+00,3.6695E+00,3.6674E+00,3.6658E+00,3.6647E+00,
	3.6640E+00,3.6636E+00,3.6623E+00,3.6600E+00,3.6568E+00,3.6525E+00,3.6474E+00,3.6415E+00,
	3.6347E+00,3.6273E+00,3.6192E+00,3.6105E+00,3.6013E+00,3.5918E+00,3.5820E+00,3.5720E+00,
	3.5619E+00,3.5518E+00,3.5417E+00,3.5319E+00,3.5223E+00,3.5131E+00,3.5043E+00,3.4960E+00,
	3.4883E+00,3.4813E+00,3.4750E+00,3.4694E+00,3.4647E+00,3.4608E+00,3.4578E+00,3.4557E+00,
	3.4546E+00,3.4537E+00,3.4514E+00,3.4472E+00,3.4412E+00,3.4336E+00,3.4242E+00,3.4134E+00,
	3.4011E+00,3.3876E+00,3.3730E+00,3.3574E+00,3.3409E+00,3.3239E+00,3.3064E+00,3.2886E+00,
	3.2708E+00,3.2529E+00,3.2353E+00,3.2181E+00,3.2014E+00,3.1854E+00,3.1702E+00,3.1560E+00,
	3.1428E+00,3.1307E+00,3.1199E+00,3.1105E+00,3.1024E+00,3.0958E+00,3.0907E+00,3.0872E+00,
	3.0852E+00,3.0838E+00,3.0798E+00,3.0728E+00,3.0627E+00,3.0498E+00,3.0341E+00,3.0160E+00,
	2.9956E+00,2.9732E+00,2.9491E+00,2.9236E+00,2.8969E+00,2.8693E+00,2.8412E+00,2.8129E+00,
	2.7845E+00,2.7565E+00,2.7290E+00,2.7022E+00,2.6765E+00,2.6520E+00,2.6288E+00,2.6072E+00,
	2.5873E+00,2.5693E+00,2.5532E+00,2.5391E+00,2.5271E+00,2.5174E+00,2.5099E+00,2.5046E+00,
	2.5017E+00,2.4997E+00,2.4939E+00,2.4835E+00,2.4688E+00,2.4500E+00,2.4274E+00,2.4013E+00,
	2.3723E+00,2.3406E+00,2.3068E+00,2.2714E+00,2.2347E+00,2.1972E+00,2.1595E+00,2.1218E+00,
	2.0845E+00,2.0480E+00,2.0126E+00,1.9786E+00,1.9461E+00,1.9155E+00,1.8869E+00,1.8605E+00,
	1.8363E+00,1.8145E+00,1.7952E+00,1.7784E+00,1.7643E+00,1.7527E+00,1.7439E+00,1.7378E+00,
	1.7344E+00,1.7320E+00,1.7252E+00,1.7131E+00,1.6961E+00,1.6745E+00,1.6487E+00,1.6192E+00,
	1.5867E+00,1.5518E+00,1.5150E+00,1.4768E+00,1.4380E+00,1.3989E+00,1.3600E+00,1.3219E+00,
	1.2847E+00,1.2488E+00,1.2146E+00,1.1821E+00,1.1515E+00,1.1231E+00,1.0968E+00,1.0728E+00,
	1.0511E+00,1.0317E+00,1.0146E+00,9.9993E-01,9.8760E-01,9.7763E-01,9.7001E-01,9.6474E-01,
	9.6181E-01,9.5976E-01,9.5396E-01,9.4370E-01,9.2928E-01,9.1110E-01,8.8964E-01,8.6544E-01,
	8.3906E-01,8.1109E-01,7.8205E-01,7.5247E-01,7.2280E-01,6.9345E-01,6.6476E-01,6.3702E-01,
	6.1044E-01,5.8520E-01,5.6120E-01,5.3797E-01,5.1656E-01,4.9698E-01,4.7922E-01,4.6323E-01,
	4.4899E-01,4.3645E-01,4.2557E-01,4.1628E-01,4.0856E-01,4.0236E-01,3.9765E-01,3.9441E-01,
	3.9261E-01,3.9136E-01,3.8781E-01,3.8158E-01,3.7289E-01,3.6206E-01,3.4946E-01,3.3547E-01,
	3.2050E-01,3.0492E-01,2.8909E-01,2.7332E-01,2.5785E-01,2.4289E-01,2.2859E-01,2.1507E-01,
	2.0241E-01,1.9065E-01,1.7979E-01,1.6987E-01,1.6087E-01,1.5276E-01,1.4552E-01,1.3908E-01,
	1.3341E-01,1.2848E-01,1.2423E-01,1.2064E-01,1.1767E-01,1.1531E-01,1.1352E-01,1.1229E-01,
	1.1161E-01,1.1114E-01,1.0980E-01,1.0746E-01,1.0423E-01,1.0023E-01,9.5617E-02,9.0565E-02,
	8.5230E-02,7.9761E-02,7.4374E-02,6.9114E-02,6.4056E-02,5.9261E-02,5.4769E-02,5.0604E-02,
	4.6776E-02,4.3284E-02,4.0134E-02,3.7350E-02,3.4864E-02,3.2657E-02,3.0708E-02,2.8998E-02,
	2.7510E-02,2.6226E-02,2.5131E-02,2.4213E-02,2.3459E-02,2.2876E-02,2.2441E-02,2.2143E-02,
	2.1979E-02,2.1865E-02,2.1543E-02,2.0982E-02,2.0211E-02,1.9267E-02,1.8190E-02,1.7025E-02,
	1.5812E-02,1.4587E-02,1.3382E-02,1.2220E-02,1.1118E-02,1.0089E-02,9.1389E-03,8.2712E-03,
	7.4967E-03,6.8376E-03,6.2504E-03,5.7301E-03,5.2714E-03,4.8690E-03,4.5178E-03,4.2129E-03,
	3.9501E-03,3.7255E-03,3.5355E-03,3.3772E-03,3.2482E-03,3.1527E-03,3.0827E-03,3.0350E-03,
	3.0087E-03,2.9904E-03,2.9390E-03,2.8497E-03,2.7275E-03,2.5787E-03,2.4105E-03,2.2300E-03,
	2.0441E-03,1.8584E-03,1.6779E-03,1.5061E-03,1.3455E-03,1.1975E-03,1.0629E-03,9.4172E-04,
	8.3663E-04,7.5292E-04,6.7905E-04,6.1420E-04,5.5755E-04,5.0829E-04,4.6565E-04,4.2893E-04,
	3.9752E-04,3.7085E-04,3.4844E-04,3.2988E-04,3.1482E-04,3.0394E-04,2.9606E-04,2.9069E-04,
	2.8774E-04,2.8568E-04,2.7993E-04,2.6996E-04,2.5637E-04,2.3993E-04,2.2149E-04,2.0189E-04,
	1.8191E-04,1.6220E-04,1.4329E-04,1.2653E-04,1.1163E-04,9.8097E-05,8.5970E-05,7.5217E-05,
	6.5903E-05,5.8216E-05,5.1558E-05,4.5822E-05,4.0900E-05,3.6700E-05,3.3282E-05,3.0383E-05,
	2.7936E-05,2.5885E-05,2.4181E-05,2.2783E-05,2.1659E-05,2.0806E-05,2.0179E-05,1.9754E-05,
	1.9521E-05,1.9359E-05,1.8906E-05,1.8125E-05,1.7070E-05,1.5808E-05,1.4411E-05,1.2957E-05,
	1.1543E-05,1.0174E-05,8.8869E-06,1.1661E-05},
	/*9 202*/
	{
	4.6239E+00,4.6238E+00,4.6233E+00,4.6223E+00,4.6210E+00,4.6193E+00,4.6171E+00,4.6147E+00,
	4.6119E+00,4.6087E+00,4.6054E+00,4.6017E+00,4.5979E+00,4.5939E+00,4.5897E+00,4.5855E+00,
	4.5812E+00,4.5769E+00,4.5726E+00,4.5684E+00,4.5643E+00,4.5604E+00,4.5566E+00,4.5530E+00,
	4.5497E+00,4.5466E+00,4.5439E+00,4.5415E+00,4.5394E+00,4.5377E+00,4.5364E+00,4.5355E+00,
	4.5350E+00,4.5347E+00,4.5342E+00,4.5333E+00,4.5320E+00,4.5303E+00,4.5282E+00,4.5258E+00,
	4.5231E+00,4.5201E+00,4.5168E+00,4.5133E+00,4.5095E+00,4.5056E+00,4.5016E+00,4.4975E+00,
	4.4933E+00,4.4891E+00,4.4850E+00,4.4809E+00,4.4769E+00,4.4730E+00,4.4694E+00,4.4659E+00,
	4.4627E+00,4.4597E+00,4.4570E+00,4.4547E+00,4.4527E+00,4.4510E+00,4.4497E+00,4.4489E+00,
	4.4484E+00,4.4480E+00,4.4470E+00,4.4452E+00,4.4427E+00,4.4394E+00,4.4354E+00,4.4308E+00,
	4.4255E+00,4.4196E+00,4.4133E+00,4.4064E+00,4.3992E+00,4.3917E+00,4.3840E+00,4.3761E+00,
	4.3681E+00,4.3601E+00,4.3521E+00,4.3443E+00,4.3366E+00,4.3293E+00,4.3223E+00,4.3157E+00,
	4.3095E+00,4.3039E+00,4.2988E+00,4.2944E+00,4.2906E+00,4.2875E+00,4.2851E+00,4.2834E+00,
	4.2824E+00,4.2818E+00,4.2799E+00,4.2765E+00,4.2717E+00,4.2655E+00,4.2580E+00,4.2492E+00,
	4.2393E+00,4.2283E+00,4.2164E+00,4.2037E+00,4.1903E+00,4.1763E+00,4.1619E+00,4.1473E+00,
	4.1325E+00,4.1178E+00,4.1031E+00,4.0888E+00,4.0749E+00,4.0615E+00,4.0487E+00,4.0368E+00,
	4.0256E+00,4.0155E+00,4.0063E+00,3.9983E+00,3.9915E+00,3.9858E+00,3.9815E+00,3.9785E+00,
	3.9768E+00,3.9756E+00,3.9722E+00,3.9662E+00,3.9576E+00,3.9466E+00,3.9332E+00,3.9176E+00,
	3.9000E+00,3.8806E+00,3.8596E+00,3.8374E+00,3.8140E+00,3.7897E+00,3.7649E+00,3.7397E+00,
	3.7144E+00,3.6892E+00,3.6644E+00,3.6402E+00,3.6168E+00,3.5943E+00,3.5731E+00,3.5532E+00,
	3.5348E+00,3.5180E+00,3.5030E+00,3.4899E+00,3.4787E+00,3.4696E+00,3.4625E+00,3.4576E+00,
	3.4548E+00,3.4529E+00,3.4474E+00,3.4377E+00,3.4238E+00,3.4059E+00,3.3844E+00,3.3595E+00,
	3.3315E+00,3.3009E+00,3.2680E+00,3.2333E+00,3.1971E+00,3.1599E+00,3.1220E+00,3.0840E+00,
	3.0461E+00,3.0087E+00,2.9721E+00,2.9367E+00,2.9028E+00,2.8705E+00,2.8401E+00,2.8119E+00,
	2.7859E+00,2.7624E+00,2.7414E+00,2.7232E+00,2.7077E+00,2.6951E+00,2.6854E+00,2.6786E+00,
	2.6749E+00,2.6722E+00,2.6648E+00,2.6514E+00,2.6325E+00,2.6084E+00,2.5795E+00,2.5463E+00,
	2.5094E+00,2.4693E+00,2.4267E+00,2.3822E+00,2.3364E+00,2.2899E+00,2.2432E+00,2.1968E+00,
	2.1512E+00,2.1068E+00,2.0639E+00,2.0229E+00,1.9840E+00,1.9474E+00,1.9134E+00,1.8821E+00,
	1.8535E+00,1.8279E+00,1.8052E+00,1.7856E+00,1.7691E+00,1.7557E+00,1.7454E+00,1.7382E+00,
	1.7343E+00,1.7315E+00,1.7236E+00,1.7097E+00,1.6900E+00,1.6650E+00,1.6354E+00,1.6017E+00,
	1.5648E+00,1.5252E+00,1.4837E+00,1.4410E+00,1.3977E+00,1.3545E+00,1.3118E+00,1.2701E+00,
	1.2297E+00,1.1910E+00,1.1543E+00,1.1196E+00,1.0872E+00,1.0572E+00,1.0295E+00,1.0044E+00,
	9.8178E-01,9.6164E-01,9.4400E-01,9.2884E-01,9.1615E-01,9.0591E-01,8.9809E-01,8.9269E-01,
	8.8970E-01,8.8759E-01,8.8166E-01,8.7119E-01,8.5650E-01,8.3803E-01,8.1631E-01,7.9192E-01,
	7.6546E-01,7.3754E-01,7.0873E-01,6.7953E-01,6.5041E-01,6.2178E-01,5.9394E-01,5.6717E-01,
	5.4166E-01,5.1757E-01,4.9469E-01,4.7237E-01,4.5192E-01,4.3331E-01,4.1652E-01,4.0148E-01,
	3.8814E-01,3.7643E-01,3.6630E-01,3.5769E-01,3.5055E-01,3.4482E-01,3.4048E-01,3.3749E-01,
	3.3584E-01,3.3468E-01,3.3143E-01,3.2571E-01,3.1776E-01,3.0787E-01,2.9640E-01,2.8374E-01,
	2.7024E-01,2.5627E-01,2.4215E-01,2.2815E-01,2.1450E-01,2.0136E-01,1.8889E-01,1.7715E-01,
	1.6621E-01,1.5609E-01,1.4678E-01,1.3826E-01,1.3057E-01,1.2367E-01,1.1752E-01,1.1209E-01,
	1.0731E-01,1.0317E-01,9.9610E-02,9.6609E-02,9.4134E-02,9.2162E-02,9.0673E-02,8.9652E-02,
	8.9088E-02,8.8693E-02,8.7585E-02,8.5646E-02,8.2966E-02,7.9660E-02,7.5861E-02,7.1710E-02,
	6.7340E-02,6.2876E-02,5.8485E-02,5.4213E-02,5.0121E-02,4.6255E-02,4.2647E-02,3.9313E-02,
	3.6259E-02,3.3482E-02,3.0983E-02,2.8777E-02,2.6812E-02,2.5071E-02,2.3537E-02,2.2195E-02,
	2.1028E-02,2.0023E-02,1.9167E-02,1.8450E-02,1.7862E-02,1.7409E-02,1.7072E-02,1.6842E-02,
	1.6715E-02,1.6626E-02,1.6377E-02,1.5943E-02,1.5347E-02,1.4618E-02,1.3787E-02,1.2889E-02,
	1.1956E-02,1.1015E-02,1.0091E-02,9.2003E-03,8.3578E-03,7.5718E-03,6.8473E-03,6.1864E-03,
	5.5977E-03,5.0989E-03,4.6554E-03,4.2631E-03,3.9179E-03,3.6154E-03,3.3518E-03,3.1234E-03,
	2.9266E-03,2.7586E-03,2.6166E-03,2.4984E-03,2.4021E-03,2.3307E-03,2.2785E-03,2.2428E-03,
	2.2232E-03,2.2095E-03,2.1711E-03,2.1045E-03,2.0133E-03,1.9024E-03,1.7771E-03,1.6428E-03,
	1.5047E-03,1.3669E-03,1.2332E-03,1.1060E-03,9.8723E-04,8.7798E-04,7.7871E-04,6.8947E-04,
	6.1211E-04,5.5044E-04,4.9607E-04,4.4840E-04,4.0679E-04,3.7064E-04,3.3938E-04,3.1248E-04,
	2.8948E-04,2.6996E-04,2.5357E-04,2.4000E-04,2.2900E-04,2.2105E-04,2.1529E-04,2.1137E-04,
	2.0921E-04,2.0771E-04,2.0351E-04,1.9623E-04,1.8631E-04,1.7432E-04,1.6087E-04,1.4658E-04,
	1.3203E-04,1.1768E-04,1.0392E-04,9.1727E-05,8.0889E-05,7.1055E-05,6.2248E-05,5.4442E-05,
	4.7684E-05,4.2103E-05,3.7273E-05,3.3113E-05,2.9546E-05,2.6504E-05,2.4032E-05,2.1936E-05,
	2.0167E-05,1.8685E-05,1.7453E-05,1.6443E-05,1.5631E-05,1.5014E-05,1.4560E-05,1.4253E-05,
	1.4084E-05,1.3967E-05,1.3639E-05,1.3074E-05,1.2311E-05,1.1399E-05,1.0389E-05,9.3381E-06,
	8.3178E-06,7.3307E-06,6.4027E-06,8.5366E-06},
	/*10 202*/
	{
	5.3945E+00,5.3943E+00,5.3935E+00,5.3922E+00,5.3902E+00,5.3877E+00,5.3846E+00,5.3810E+00,
	5.3769E+00,5.3724E+00,5.3675E+00,5.3623E+00,5.3567E+00,5.3509E+00,5.3449E+00,5.3388E+00,
	5.3326E+00,5.3263E+00,5.3201E+00,5.3141E+00,5.3081E+00,5.3024E+00,5.2969E+00,5.2918E+00,
	5.2870E+00,5.2826E+00,5.2786E+00,5.2751E+00,5.2721E+00,5.2697E+00,5.2678E+00,5.2665E+00,
	5.2657E+00,5.2654E+00,5.2647E+00,5.2633E+00,5.2614E+00,5.2590E+00,5.2560E+00,5.2526E+00,
	5.2486E+00,5.2443E+00,5.2396E+00,5.2345E+00,5.2291E+00,5.2235E+00,5.2177E+00,5.2118E+00,
	5.2058E+00,5.1998E+00,5.1938E+00,5.1879E+00,5.1822E+00,5.1767E+00,5.1714E+00,5.1664E+00,
	5.1618E+00,5.1575E+00,5.1537E+00,5.1503E+00,5.1475E+00,5.1451E+00,5.1433E+00,5.1420E+00,
	5.1413E+00,5.1408E+00,5.1394E+00,5.1368E+00,5.1332E+00,5.1285E+00,5.1228E+00,5.1161E+00,
	5.1085E+00,5.1002E+00,5.0911E+00,5.0814E+00,5.0711E+00,5.0604E+00,5.0493E+00,5.0381E+00,
	5.0267E+00,5.0153E+00,5.0040E+00,4.9928E+00,4.9820E+00,4.9716E+00,4.9616E+00,4.9523E+00,
	4.9436E+00,4.9356E+00,4.9284E+00,4.9221E+00,4.9168E+00,4.9123E+00,4.9089E+00,4.9065E+00,
	4.9052E+00,4.9043E+00,4.9016E+00,4.8969E+00,4.8901E+00,4.8813E+00,4.8707E+00,4.8583E+00,
	4.8444E+00,4.8289E+00,4.8121E+00,4.7943E+00,4.7755E+00,4.7559E+00,4.7358E+00,4.7153E+00,
	4.6946E+00,4.6740E+00,4.6537E+00,4.6337E+00,4.6144E+00,4.5958E+00,4.5781E+00,4.5615E+00,
	4.5461E+00,4.5320E+00,4.5194E+00,4.5083E+00,4.4988E+00,4.4911E+00,4.4851E+00,4.4810E+00,
	4.4786E+00,4.4770E+00,4.4723E+00,4.4641E+00,4.4522E+00,4.4370E+00,4.4186E+00,4.3972E+00,
	4.3731E+00,4.3466E+00,4.3179E+00,4.2875E+00,4.2557E+00,4.2228E+00,4.1891E+00,4.1550E+00,
	4.1209E+00,4.0870E+00,4.0537E+00,4.0212E+00,3.9899E+00,3.9599E+00,3.9316E+00,3.9052E+00,
	3.8807E+00,3.8585E+00,3.8386E+00,3.8212E+00,3.8065E+00,3.7944E+00,3.7851E+00,3.7786E+00,
	3.7750E+00,3.7724E+00,3.7652E+00,3.7524E+00,3.7341E+00,3.7107E+00,3.6825E+00,3.6499E+00,
	3.6134E+00,3.5736E+00,3.5309E+00,3.4859E+00,3.4393E+00,3.3915E+00,3.3430E+00,3.2945E+00,
	3.2463E+00,3.1989E+00,3.1528E+00,3.1082E+00,3.0656E+00,3.0253E+00,2.9875E+00,2.9524E+00,
	2.9202E+00,2.8911E+00,2.8653E+00,2.8428E+00,2.8238E+00,2.8083E+00,2.7965E+00,2.7882E+00,
	2.7836E+00,2.7804E+00,2.7712E+00,2.7549E+00,2.7319E+00,2.7025E+00,2.6674E+00,2.6272E+00,
	2.5827E+00,2.5346E+00,2.4837E+00,2.4307E+00,2.3765E+00,2.3217E+00,2.2669E+00,2.2128E+00,
	2.1599E+00,2.1085E+00,2.0593E+00,2.0123E+00,1.9680E+00,1.9266E+00,1.8881E+00,1.8529E+00,
	1.8209E+00,1.7922E+00,1.7670E+00,1.7452E+00,1.7268E+00,1.7120E+00,1.7006E+00,1.6928E+00,
	1.6884E+00,1.6853E+00,1.6766E+00,1.6613E+00,1.6397E+00,1.6123E+00,1.5800E+00,1.5434E+00,
	1.5033E+00,1.4607E+00,1.4162E+00,1.3707E+00,1.3248E+00,1.2793E+00,1.2345E+00,1.1911E+00,
	1.1493E+00,1.1094E+00,1.0717E+00,1.0364E+00,1.0035E+00,9.7317E-01,9.4540E-01,9.2022E-01,
	8.9762E-01,8.7758E-01,8.6008E-01,8.4508E-01,8.3254E-01,8.2244E-01,8.1475E-01,8.0944E-01,
	8.0649E-01,8.0442E-01,7.9860E-01,7.8833E-01,7.7395E-01,7.5591E-01,7.3478E-01,7.1113E-01,
	6.8558E-01,6.5874E-01,6.3115E-01,6.0334E-01,5.7574E-01,5.4872E-01,5.2258E-01,4.9756E-01,
	4.7383E-01,4.5150E-01,4.3032E-01,4.0950E-01,3.9052E-01,3.7333E-01,3.5789E-01,3.4411E-01,
	3.3193E-01,3.2129E-01,3.1210E-01,3.0430E-01,2.9785E-01,2.9269E-01,2.8879E-01,2.8610E-01,
	2.8462E-01,2.8358E-01,2.8066E-01,2.7553E-01,2.6841E-01,2.5959E-01,2.4939E-01,2.3816E-01,
	2.2624E-01,2.1396E-01,2.0160E-01,1.8940E-01,1.7756E-01,1.6623E-01,1.5550E-01,1.4546E-01,
	1.3613E-01,1.2755E-01,1.1967E-01,1.1246E-01,1.0597E-01,1.0017E-01,9.5022E-02,9.0476E-02,
	8.6496E-02,8.3046E-02,8.0091E-02,7.7602E-02,7.5552E-02,7.3921E-02,7.2691E-02,7.1847E-02,
	7.1381E-02,7.1056E-02,7.0141E-02,6.8543E-02,6.6336E-02,6.3618E-02,6.0502E-02,5.7103E-02,
	5.3534E-02,4.9898E-02,4.6329E-02,4.2865E-02,3.9556E-02,3.6438E-02,3.3536E-02,3.0861E-02,
	2.8416E-02,2.6198E-02,2.4206E-02,2.2453E-02,2.0894E-02,1.9515E-02,1.8302E-02,1.7241E-02,
	1.6320E-02,1.5527E-02,1.4853E-02,1.4288E-02,1.3826E-02,1.3470E-02,1.3205E-02,1.3024E-02,
	1.2925E-02,1.2855E-02,1.2660E-02,1.2320E-02,1.1852E-02,1.1281E-02,1.0631E-02,9.9301E-03,
	9.2018E-03,8.4688E-03,7.7492E-03,7.0573E-03,6.4035E-03,5.7943E-03,5.2335E-03,4.7226E-03,
	4.2680E-03,3.8838E-03,3.5426E-03,3.2412E-03,2.9764E-03,2.7446E-03,2.5428E-03,2.3681E-03,
	2.2178E-03,2.0895E-03,1.9811E-03,1.8910E-03,1.8176E-03,1.7632E-03,1.7233E-03,1.6962E-03,
	1.6812E-03,1.6707E-03,1.6415E-03,1.5907E-03,1.5213E-03,1.4369E-03,1.3416E-03,1.2396E-03,
	1.1347E-03,1.0302E-03,9.2884E-04,8.3255E-04,7.4272E-04,6.6015E-04,5.8520E-04,5.1788E-04,
	4.5948E-04,4.1271E-04,3.7154E-04,3.3548E-04,3.0406E-04,2.7679E-04,2.5323E-04,2.3298E-04,
	2.1568E-04,2.0102E-04,1.8871E-04,1.7853E-04,1.7028E-04,1.6437E-04,1.6010E-04,1.5720E-04,
	1.5560E-04,1.5449E-04,1.5137E-04,1.4598E-04,1.3863E-04,1.2974E-04,1.1976E-04,1.0917E-04,
	9.8371E-05,8.7723E-05,7.7507E-05,6.8413E-05,6.0309E-05,5.2961E-05,4.6383E-05,4.0556E-05,
	3.5513E-05,3.1351E-05,2.7750E-05,2.4649E-05,2.1991E-05,1.9725E-05,1.7881E-05,1.6317E-05,
	1.4999E-05,1.3894E-05,1.2976E-05,1.2224E-05,1.1619E-05,1.1160E-05,1.0822E-05,1.0593E-05,
	1.0467E-05,1.0380E-05,1.0136E-05,9.7163E-06,9.1487E-06,8.4698E-06,7.7188E-06,6.9376E-06,
	6.1785E-06,5.4444E-06,4.7543E-06,6.4029E-06}};


/*He_cross_section returns cross section (cm^-2), 
 * given EgammaRyd, the photon energy in Ryd,
 * ipLevel, the index of the level, 0 is ground, 3 within 2 3P,
 * nelem is charge, equal to 1 for Helium 
 * this is a wrapper for cross_section */
double He_cross_section( double EgammaRyd , long ipLevel , long nelem )
{
	double cs;
	static double ThreshCS; 
	static long ipCurrentLevel=-1;

#define NUM_PREF_TCS 31

	/* These corrections ensure helium cross-sections agree with the highly accurate 
	 * values given by Hummer and Storey (1998).  The index is [ipLev - 1]	*/
	/* >>refer He	PCS	Hummer, D.G., & Storey, P.J. 1998, MNRAS 297, 1073	*/  
	double PreferredThres[NUM_PREF_TCS] = {
		/* n = 1 */
		7.394,
		/* n = 2 */
		5.480,9.253,15.98,15.98,15.98,13.48,
		/* n = 3 */
		8.025,14.49,28.52,18.48,18.13,26.99,
		/* n = 4 */
		10.77,20.38,41.59,36.70,35.75,-1.00,-1.00,41.75,
		/* n = 5, these are calculated using the Hummer and Storey 
		 * oscillator strength extrapolation procedure */
		13.54,26.58,55.71,-1.00,-1.00,-1.00,-1.00,-1.00,-1.00,58.07};

	globalZ = nelem;

	ipLev = ipLevel;

	EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLev];
	
	/* make sure this routine not called within collapsed high levels */
	ASSERT( ipLevel < iso.numLevels_max[ipHE_LIKE][nelem] - iso.nCollapsed_max[ipHE_LIKE][nelem] );

	/* this is a resolved level */
	cs = (1.e-18)*cross_section( EgammaRyd );

	/* Scale to preferred threshold values */
	if( nelem==ipHELIUM && ipLevel<NUM_PREF_TCS && !helike.lgSetBenjamin )
	{
		/* On the first pass, find the threshold cross-section calculated in the
		 * algorithms below...*/
		if( ipCurrentLevel!=ipLevel )
		{
			ThreshCS = cross_section( EthRyd );
			ipCurrentLevel = ipLevel;
		}

		/* ...and then use that value to rescale all cross-sections by the ratio of
		 * the preferred threshold cross-section to the calculated one. */
		if( PreferredThres[ipLevel] > 0. && ThreshCS > 0. )
			cs *= PreferredThres[ipLevel]/ThreshCS;
	}

	return cs;
}

/*cross_section calculates the photoionization cross_section for a given level and photon energy*/
static double cross_section(double EgammaRyd)
{
	/* These fit parameters (E0, sigma, y_a, P, y_w, yzero, and yone) all come from the following work: */
	/* >>refer	He	pcs	Verner, D. A., Verner, E. M., \& Ferland , G. J. 1996,
	 * >>refercon	Atomic Data and Nuclear Data Tables, Vol. 64, p.1 */
	double E0[29] = {
	1.36E+01,2.01E+01,1.76E+01,3.34E+01,4.62E+01,6.94E+01,8.71E+01,1.13E+02,1.59E+02,2.27E+02,
	2.04E+02,2.74E+02,2.75E+02,3.38E+02,4.39E+02,4.17E+02,4.47E+02,5.18E+02,6.30E+02,6.27E+02,
	8.66E+02,7.67E+02,9.70E+02,9.66E+02,1.06E+03,1.25E+03,1.35E+03,1.43E+03,1.56E+03};
	double sigma[29] = {
	9.49E+02,3.20E+02,5.46E+02,2.85E+02,2.34E+02,1.52E+02,1.33E+02,1.04E+02,6.70E+01,4.00E+01,
	6.14E+01,4.04E+01,4.75E+01,3.65E+01,2.45E+01,3.14E+01,3.11E+01,2.59E+01,1.94E+01,2.18E+01,
	1.23E+01,1.76E+01,1.19E+01,1.31E+01,1.20E+01,9.05E+00,8.38E+00,8.06E+00,7.17E+00};
	double y_a[29] = {
	1.47E+00,7.39E+00,1.72E+01,2.16E+01,2.18E+01,2.63E+01,2.54E+01,2.66E+01,3.35E+01,5.32E+01,
	2.78E+01,3.57E+01,2.85E+01,3.25E+01,4.41E+01,3.16E+01,3.04E+01,3.28E+01,3.92E+01,3.45E+01,
	5.89E+01,3.88E+01,5.35E+01,4.83E+01,5.77E+01,6.79E+01,7.43E+01,7.91E+01,9.10E+01};
	double P[29] = {
	3.19E+00,2.92E+00,3.16E+00,2.62E+00,2.58E+00,2.32E+00,2.34E+00,2.26E+00,2.00E+00,1.68E+00,
	2.16E+00,1.92E+00,2.14E+00,2.00E+00,1.77E+00,2.04E+00,2.09E+00,2.02E+00,1.86E+00,2.00E+00,
	1.62E+00,1.93E+00,1.70E+00,1.79E+00,1.72E+00,1.61E+00,1.59E+00,1.58E+00,1.54E+00};
	double y_w[29] = 
	{2.039,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	double yzero[29] =
	{0.4434,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	double yone[29] =
	{2.136,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
	
	/* These numbers are for the calculation of S cross-sections for n = 2 to n = 10. */
	/* indices are [n-2][s][a,b]	*/
	double heliumSparameters[9][2][2] = {
		{{0.0242024555,0.9702079},	{0.095921966,0.7337243}},

		{{0.0141220876,3.5176515},	{0.062991353,3.3995626}},

		{{0.0091014916,8.4545141},	{0.044113776,9.5657406}},
		
		{{0.0059539267,16.705745},	{0.032224774,20.933772}},
	
		{{0.0039485621,28.762086},	{0.024926753,38.464649}},
	
		{{0.0026053536,45.352333},	{0.020215452,63.043223}},
		
		{{0.0016632439,67.296132},	{0.016490176,96.296457}},
		
		{{0.0009867019,95.189826},	{0.013689704,138.82733}},
		
		{{0.0005155132,129.46774},	{0.011708145,191.73963}}};

	/* These are for the calculation of P cross-sections ...
	 * for n = 2 and n = 3 of triplets... */
	/* indices are [n-2][a,b]	*/
	double heTripP_params[2][2] = {
		{0.00489878046,3.0995085}, {0.01284809745,14.191429}};
	/* and for n=2 through n=9 of singlets. */
	/* indices are [n-2][a,b,c]	*/
	double heSingP_params[8][3] = {
		{-2.23352104,	-0.25979099,	-3.81078671},
		{-3.34344798,	-0.22891612,	-3.51168670},
		{-4.06915276,	-0.18554441,	-3.32255666},
		{-4.64178338,	-0.15390749,	-3.19447260},
		{-5.09796105,	-0.13059073,	-3.09491264},
		{-5.47775263,	-0.11232976,	-3.01326975},
		{-5.83522711,	-0.10848632,	-2.98479642},
		{-5.98242671,	-0.06820050,	-2.79942712}};

	double pcs,Egamma,lnE,a,b,c,y,F,x;
	long nelem = globalZ;

	Egamma = EgammaRyd * EVRYD;

	/* >>refer Helike	PCS	Verner, D.A., Ferland, G.J., Korista, K.T., & Yakovlev, D.G.
	 * >>refercon	1996a, ApJ 465,487	*/
	/* All ground state cross-sections calculated here.	*/
	if (ipLev == ipHe1s1S)
	{
		x = Egamma/E0[nelem-1] - yzero[nelem-1];
		y = sqrt(x*x + yone[nelem-1]*yone[nelem-1]);
		F = ((x-1)*(x-1)+y_w[nelem-1]*y_w[nelem-1])
			* pow(y,0.5*P[nelem-1]-5.5) * pow((1+sqrt(y/y_a[nelem-1])),-P[nelem-1]);
		pcs = sigma[nelem-1]*F;
		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
	/* Due to the highly non-hydrogenic nature of S orbitals, both singlet and triplet,
	 * a special formula needs to be applied to them.  The hydrogenic formula is 
	 * unsatisfactory since its errors get larger with increasing n.	*/
	
	/* Fix a better one for n = 2 as well?	*/
	else if ( ( L_(ipLev) == 0 ) && ( N_(ipLev) > 2 ) && ( nelem >= ipCARBON ) )
	{
		lnE = log(EgammaRyd * POW2(1.667*N_(ipLev)/nelem));
		pcs = POW2(5./(double)nelem) * exp(1.790166293 - 0.127439504 * POW2(lnE) - 1.660569112*lnE);
		pcs *= pow(N_(ipLev)/3.,1.365);
		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
	/* In addition to the above, the S orbitals of helium need to be handled separately.	*/
	else if ( ( nelem == ipHELIUM ) && ( L_(ipLev) == 0 ) )
	{
		if ( N_(ipLev)<=10 )
		{
			/* These are fits to TOPbase data	*/
			/* >>refer	He-like	PCS	Fernley, J. A., Taylor, K. T., & Seaton, M. J. 1987,
			 * >>refercon	J. Phys. B: At. Mol. Phys. 20, 6457 */
			/* >>refer	He-like	PCS	Cunto, W., Mendoza, C., Ochsenbein, F., & Zeippen, C.J. 1993,
			 * >>refercon	A{\&}A 275, L5 */
			a = heliumSparameters[N_(ipLev)-2][S_(ipLev)][0];
			b = heliumSparameters[N_(ipLev)-2][S_(ipLev)][1];
		}
		/* These are fits to the parameters of the above fits!	*/
		else if ( S_(ipLev) == 0 )
		{
			a = 4.94 / pow((double)N_(ipLev),3.9);
			b = 0.147 * pow((double)N_(ipLev), 2.947);
		}
		else
		{
			a = 0.364/ pow((double)N_(ipLev), 1.49);
			b = 0.138 * pow((double)N_(ipLev), 3.146);
		}
			
		/* Put E^-3 tail on helium at energies beyond first resonance.	*/
		if ( EgammaRyd > 2.75 + EthRyd)
		{
			/* The value of the above fit at the point of the first resonance is approximately 
			 * equal to 1/(2.75^2 * b).  An E^-3 tail has a value of about 1/(2.75^3).  
			 * (EthRyd is much smaller than 2.75 even at n=2, just as parameter a is dwarved by b.)
			 * So the scale factor is 2.75/b.	*/
			 pcs = 2.75 * POW3(1./EgammaRyd) / b;
		}
		else
		{
			pcs = 1./(a+b*POW2(EgammaRyd));
		}

		if( N_(ipLev)>=6 )
		{
			double tweakle_dee[2][20] = {
			{0.9540,0.9485,0.9463,0.9449,0.9426,0.9505,0.9533,0.9562,0.9590,0.9617,
				0.9644,0.9670,0.9694,0.9717,0.9740,0.9761,0.9782,0.9802,0.9821,0.9839},
			{1.0418,1.0400,1.0378,1.0322,1.0265,1.0383,1.0439,1.0502,1.0569,1.0638,
				1.0709,1.0779,1.0850,1.0919,1.0987,1.1054,1.1120,1.1185,1.1248,1.1309}};

			if( N_(ipLev) <=25 )
			{
				pcs *= tweakle_dee[S_(ipLev)][N_(ipLev)-6];
			}
			else
			{
				pcs *= tweakle_dee[S_(ipLev)][19];
			}
		}

		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
	else if ( ( nelem == ipHELIUM ) && ( L_(ipLev) == 1 ) && (S_(ipLev) == 1) && ( N_(ipLev) <= 10 ) )
	{
		/* Fit to TOPbase data for triplet P...only n=2 and 3 are available.	*/
		if( N_(ipLev) <= 3 )
		{
			a = heTripP_params[N_(ipLev)-2][0];
			b = heTripP_params[N_(ipLev)-2][1];
			pcs = 1./(a+b*POW3(EgammaRyd));
		}
		/* Interpolate on a set of values obtained from Rob Bauman's Peach implementation */
		else
		{
			if( EgammaRyd - EthRyd < 1.0 )
			{
				pcs = pwlinear( PeachE, TripP[N_(ipLev)-4], 460, EgammaRyd - EthRyd );
				/* must convert from atomic units	*/
				pcs /= 0.035710595746;
			}
			else
			{
				/* need a tail */
				pcs = TripP[N_(ipLev)-4][458]/0.035710595746*pow(EgammaRyd - EthRyd, -2.6)*pow(0.98807,2.6);
			}
		}

		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
	/* D levels: interpolate on a set of values obtained from Rob Bauman's Peach implementation */
	else if ( ( nelem == ipHELIUM ) && ( L_(ipLev) == 2 ) && ( N_(ipLev) <= 10 ) )
	{
		double EelecRyd = EgammaRyd - EthRyd;
		EelecRyd = MAX2( 0., EelecRyd );

		if( S_(ipLev) == 1 )
		{
			if( EelecRyd < 1.0 )
			{
				pcs = pwlinear( PeachE, TripD[N_(ipLev)-3], 460, EelecRyd );
				/* must convert from atomic units	*/
				pcs /= 0.035710595746;
			}
			else
			{
				/* need a tail */
				pcs = TripD[N_(ipLev)-3][458]/0.035710595746*pow(EelecRyd, -3.65)*pow(0.98807,-3.65);
			}
		}
		else
		{
			if( EelecRyd < 1.0 )
			{
				pcs = pwlinear( PeachE, SingD[N_(ipLev)-3], 460, EelecRyd );
				/* must convert from atomic units	*/
				pcs /= 0.035710595746;
			}
			else
			{
				/* need a tail */
				pcs = SingD[N_(ipLev)-3][458]/0.035710595746*pow(EelecRyd, -3.75)*pow(0.98807,-3.75);
			}
		}

		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
	/* Fit to TOPbase data for singlet P.	*/
	else if ( ( nelem == ipHELIUM ) && ( L_(ipLev) == 1 ) && (S_(ipLev) == 0) && ( N_(ipLev) <= 7 ) )
	{	
		a = heSingP_params[N_(ipLev)-2][0];
		b = heSingP_params[N_(ipLev)-2][1];
		c = heSingP_params[N_(ipLev)-2][2];
		pcs = exp( a ) * pow( EgammaRyd, b*log(EgammaRyd)+c );

		ASSERT( pcs > 0. && pcs < 1.E10 );
	}
#if	0
	else if ( ( nelem == ipHELIUM ) && ( L_(ipLev) <= 2 ) )
	{
		double EelecRyd = EgammaRyd - EthRyd + SMALLFLOAT;
		EelecRyd = MAX2( SMALLFLOAT, EelecRyd );

		pcs = (1.e18)*PeachCrossSection( nelem, L_(ipLev), S_(ipLev), EelecRyd );
		if( pcs >= 1.E10 )
		{
			ASSERT( pcs > 0. && pcs < 1.E10 );
		}
	}
#endif
	/* To everything else we apply a hydrogenic routine.	*/
	else 
	{
		double rel_photon_energy;
		
		/* >>chng 02 apr 24, more protection against calling with too small an energy */
		/* evaluating H-like photo cs at He energies, may be below threshold -
		 * prevent this from happening */
		rel_photon_energy = EgammaRyd / EthRyd;
		rel_photon_energy = MAX2( rel_photon_energy , 1. + FLT_EPSILON*2. );

		/* >>chng 02 apr 25, only one routine exposed, chooses which to evaluate */
		pcs = (1.e18)*H_photo_cs(rel_photon_energy , N_(ipLev), L_(ipLev), nelem);

		/* Rescale by ratio of real energy to hydrogenic energy.*/	
		pcs *= EthRyd * POW2( (double)N_(ipLev)/(double)nelem );

		/* This scale factor does a good job of making the SINGLET p threshold values smooth
		 * with increasing n */
		if( ( L_(ipLev) == 1 ) && (S_(ipLev) == 0) )
			pcs *= 1.06;

		/* This scale factor does a good job of making the TRIPLET p threshold values smooth
		 * with increasing n */
		if( ( L_(ipLev) == 1 ) && (S_(ipLev) == 1) )
			pcs *= 1.07;
			

		ASSERT( pcs > 0. && pcs < 1.E10 );

	}

	return pcs;
}
#if	0
static double PeachCrossSection( long int nelem, long int l, long int s, double electron_energy )
{
	double pcs = 0.;
	double C_LPrime, Peach_G_LPrime, Sum, neff, mu_Prime, Chi_LPrime, Zeta;
	/* electron_energy is in Rydberg units */

	Zeta = 1.;
	
	neff  = (double)nelem/sqrt(EthRyd);
	
	global_s = s;
	mu_Prime = Get_mu_prime(s, l+1);
	C_LPrime = (l+1.)/(2.*l+1.);
	Chi_LPrime = -0.05+electron_energy/3.;
	Peach_G_LPrime =  Get_Peach_g( neff, l, electron_energy, l+1 );
	Peach_G_LPrime *= sqrt(PI/2.) * POW2(1. + electron_energy*neff*neff) * sqrt(Zeta);
	Peach_G_LPrime /= cos(PI*(neff+mu_Prime+Chi_LPrime));

	Sum = C_LPrime * pow( Peach_G_LPrime * cos(PI*(neff+mu_Prime+Chi_LPrime)), 2. );

	if( l>0 )
	{
		mu_Prime = Get_mu_prime(s, l-1);
		C_LPrime = ((double)l)/(2.*l+1.);
		Chi_LPrime = -0.05+electron_energy/3.;
		Peach_G_LPrime =  Get_Peach_g( neff, l, electron_energy, l-1 );
		Peach_G_LPrime *= sqrt(PI/2.) * POW2(1. + electron_energy*neff*neff) * sqrt(Zeta);
		Peach_G_LPrime /= cos(PI*(neff+mu_Prime+Chi_LPrime));
		
		Sum += C_LPrime * pow( Peach_G_LPrime * cos(PI*(neff+mu_Prime+Chi_LPrime)), 2. );
	}

	pcs =  (8./3.) * FINE_STRUCTURE * POW3(neff) * pow(1. + electron_energy*neff*neff, -3.) *
		POW2(5.29177249e-9/nelem) * Sum / Zeta;

	return pcs;
}

static double Get_mu_prime(long s, long lprime)
{
	double mu_Prime = 0.;

	ASSERT( lprime <= 3 );

	if( s == 0 )
	{
		double singlet_mu[4] = {1.40005E-01,-1.20673E-02,2.08056E-03,4.21484E-04};
		mu_Prime = singlet_mu[lprime];
	}
	else
	{
		double triplet_mu[4] = {2.97063E-01,6.81567E-02,2.82381E-03,4.27703E-04};
		mu_Prime = triplet_mu[lprime];
	}

	return mu_Prime;
}

static double Get_Peach_g( double neff, long l, double electron_energy, long lprime )
{
	double g, lastg, R1, R2, step;
	global_nu = neff;
	global_l = l;
	global_k = globalZ * sqrt(electron_energy);
	global_lprime = lprime;
	
	/* Get first part via Simpson's Rule */
	g = (2.4975/3.)*( /* g_integrand( 0.005 ) + */ 4.* g_integrand( 2.5025 ) + g_integrand( 5.0 ) );

	R1 = 5.0;
	step = 1.0;
	R2 = R1 + step;
	g += qg32( R1, R2, g_integrand);
	do
	{
		lastg = g;
		R1 = R2;
		step *= 1.5;
		R2 = R1 + step;
		g += qg32( R1, R2, g_integrand);

	} while ( (fabs( (g-lastg)/g ) > 0.001) && (R2<=100) );

	g *= EthRyd;

	return g;
}

static double g_integrand( double radius )
{
	double K, P, kprime, Fkl, Hprime, Hkl, resultFp, resultHdoubleprime, eF, eH /*, bfa */;
	double nu, b, muprime, rho, zeta, Sum;
	double tau_l, PeachGkl /*, energy*/;
	double u[10] = { 0., 0., 0., 0., 0., 0., 0., 0., 0., 0.};
	long t, t0, nelem, l, lp, s;

	nu = global_nu;
	nelem = globalZ;
	l = global_l;
	lp = global_lprime;
	s = global_s;
	rho = radius * globalZ;
	kprime = global_k; 
	zeta = 1.;
	muprime = Get_mu_prime(s, l+1);
	
	/* I'm pretty sure Peach has this wrong...and the exponent
	 * should be -1/2 instead of 1/2. */
	K = 1./( nu * sqrt( zeta * gammafun(nu+l+1.) * gammafun(nu-l) ) );
	Sum = 1.;
	b = 1.;
	/* Following Peach, go out to ceil(nu+l-1), and then use an Euler transformation
	 * for the rest of the sequence */
	t0 = (int)(ceil(nu + l));
	for( t = 1; t < t0; t++ )
	{
		b *= (nu/2./(double)t)*((double)l*(l-1.)-(nu-t)*(nu-t+1.));
		Sum += b * pow( rho, -1.*t);
	}

	for( t = t0; t <= t0 + 9; t++ )
	{
		b *= (nu/2./(double)t)*((double)l*(l-1.)-(nu-t)*(nu-t+1.));
		u[t-t0] = b * pow( rho, -1.*t);
	}

	Sum += (-1. /  4.)*(u[1] -    u[0]);
	Sum +=  (1. /  8.)*(u[2] - 2.*u[1] +    u[0]);
	Sum += (-1./  16.)*(u[3] - 3.*u[2] + 3.*u[1] -    u[0]);
	Sum +=	(1./  32.)*(u[4] - 4.*u[3] + 6.*u[2] - 4.*u[1] +     u[0]);
	Sum += (-1./  64.)*(u[5] - 5.*u[4] +10.*u[3] -10.*u[2] +  5.*u[1] -     u[0]);
	Sum +=  (1./ 128.)*(u[6] - 6.*u[5] +15.*u[4] -20.*u[3] + 15.*u[2] -  6.*u[1] +   u[0]);
	Sum += (-1./ 256.)*(u[7] - 7.*u[6] +21.*u[5] -35.*u[4] + 35.*u[3] - 21.*u[2] + 7.*u[1] -    u[0]);
	Sum +=  (1./ 512.)*(u[8] - 8.*u[7] +28.*u[6] -56.*u[5] + 70.*u[4] - 56.*u[3] +28.*u[2] - 8.*u[1] +   u[0]);
	Sum += (-1./1024.)*(u[9] - 9.*u[8] +36.*u[7] -84.*u[6] +126.*u[5] -126.*u[4] +84.*u[3] -36.*u[2] +9.*u[1] - u[0]);

	P = sqrt(globalZ) * K * pow( 2.*rho/nu, nu) * sexp( rho/nu ) * Sum;

	/***************************/
	/*  Now find G(nu,l,e',l') */
	/***************************/

	if( lp==0 )
	{
		/* something big, doesn't matter */
		tau_l = 1.E10;
	}
	else
	{
		tau_l = 10./(double)lp/((double)lp+1.);
	}

	gsl_sf_coulomb_wave_FG_e(
			-1.*nelem/kprime,
			radius*kprime,
			(double)lp,
			0,
			&Fkl,
			&resultFp,
			&Hprime,
			&resultHdoubleprime,
			&eF,
			&eH );

	/* This GSL function uses the Abramowitz and Stegun (1955) notation,
	 * but that G is Peach's H' */
	Hkl = pow( 1.-exp(-tau_l*radius), 2.*lp + 1. ) * Hprime /* / POW2(kprime)*/;

	PeachGkl = (1./sqrt((double)nelem)) * (Fkl*cos(PI*muprime) + Hkl*sin(PI*muprime) );

/* 
	bfa = sqrt(8*rho); 
	energy = POW2(kprime/(double)nelem);
	
	Fkl = gsl_sf_bessel_Jn_e(2*lp+1,bfa)+ (energy/12)*
		( lp*(lp+1)*(2*lp+1)*gsl_sf_bessel_Jn_e(2*lp+1,bfa)-
		3*(lp+1)*(2*rho)*gsl_sf_bessel_Jn_e(2*lp+3,bfa)+
		(pow(2.*rho, 1.5))*gsl_sf_bessel_Jn_e(2*lp+4,bfa) );
	Fkl *= sqrt(PI*rho);

	Hkl = gsl_sf_bessel_Yn_e(2*lp+1,bfa)+ (energy/12)*
		( lp*(lp+1)*(2*lp+1)*gsl_sf_bessel_Yn_e(2*lp+1,bfa)-
		3*(lp+1)*(2*rho)*gsl_sf_bessel_Yn_e(2*lp+3,bfa)+
		(pow(2.*rho, 1.5))*gsl_sf_bessel_Yn_e(2*lp+4,bfa) );
	Hkl *= pow(1.-sexp(tau_l*rho), 2.*lp+1.) * sqrt(PI*rho);

	PeachGkl = (1./sqrt((double)nelem)) * (Fkl*cos(PI*muprime) - Hkl*sin(PI*muprime));
*/

	return P * PeachGkl * radius;
}

#endif

/*radrecomb calculates radiative recombination coefficients. */
static double radrecomb(double temp, long nelem, long ipLo)
{
	double alpha,RecomIntegral=0.,b,E1,E2,step,OldRecomIntegral,TotChangeLastFive;
	double change[5] = {0.,0.,0.,0.,0.};
	
	/* Factors outside integral in Milne relation.	*/
	 b = MILNE_CONST * (2.*L_(ipLo)+1.) * (2.*S_(ipLo)+1.) * pow(temp,-1.5) / 4;
	/* kT in Rydbergs.	*/
	kTRyd = temp / TE1RYD;
	globalZ = nelem;
	ipLev = ipLo;

	/* Begin integration.	*/
	/* First define characteristic step as the minimum of kTRyd and 3EthRyd.	*/
	E1 = EthRyd;
	step = MIN( 0.25*kTRyd, 0.5*E1 );
	E2 = E1 + step;
	/* Perform initial integration, from threshold to threshold + step.	*/
	RecomIntegral = qg32( E1, E2, RecomInt);
	/* Repeat the integration, adding each new result to the total, 
	 * except that the step size is doubled every time, since values away from 
	 * threshold tend to fall off more slowly.	*/
	do
	{
		OldRecomIntegral = RecomIntegral;
		E1 = E2;
		step *= 1.25;
		E2 = E1 + step;
		RecomIntegral += qg32( E1, E2, RecomInt);
		change[4] = change[3];
		change[3] = change[2];
		change[2] = change[1];
		change[1] = change[0];
		change[0] = (RecomIntegral - OldRecomIntegral)/RecomIntegral;
		TotChangeLastFive = change[0] + change[1] + change[2] + change[3] + change[4];
	/* Continue integration until the upper limit exceeds 100kTRyd, an arbitrary
	 * point at which the integrand component exp(electron energy/kT) is very small,
	 * making neglible cross-sections at photon energies beyond that point,
	 * OR when the last five steps resulted in less than a 1 percent change.	*/
	} while ( ((E2-EthRyd) < 100.*kTRyd) && ( TotChangeLastFive > 0.0001) );

	/* Calculate recombination coefficient.	*/
	alpha = b * RecomIntegral;

	if ( (ipLo == ipHe2p3P0) || (ipLo == ipHe2p3P1) || (ipLo == ipHe2p3P2) )
	{
		/* Split 2 trip P term.	*/
		alpha *= (2.*(ipLo-3L)+1.) / 9.;
	}

	/* chng 05 dec 12, no need to limit this to SMALLFLOAT.  Quantity is a double,
	 * let it be a double. */
	/* if ( alpha < SMALLFLOAT )
		alpha = SMALLFLOAT; */
	alpha = MAX2( alpha, SMALLDOUBLE );

	return alpha;
}

/*RecomInt, used in comput milne relation for he rec - the energy is photon Rydbergs.	*/
static double RecomInt(double ERyd)
{
	double x1, temp;

	/* Milne relation integrand	*/
	x1 = ERyd * ERyd * exp(-1.0 * ( ERyd - EthRyd ) / kTRyd);
	temp = He_cross_section( ERyd , ipLev, globalZ );
	x1 *= temp;
	
	return x1;
}

/* >>refer	He-like	RR	Seaton, M.J. 1959, MNRAS 119, 81S */
static double Recomb_Seaton59( long nelem, double temp, long n)
{
	double lambda = TE1RYD * nelem * nelem / temp;
	/* smallest x ever used here should be lowest Z, highest T, highest n...
	 * using helium, logt = 10., and n = 100, we get xmin = 1.5789E-9.	*/
	double x = lambda / n / n;
	double AlphaN;
	double SzeroOfX = 0.;
	double SoneOfX = 0.;
	double StwoOfX = 0.;
	double SnOfLambda = 0.;
	double lowerlimit, upperlimit, step;

	Xn_S59 = x;

	/* Equation 12	*/
	lowerlimit = x;
	step = 3. * x;
	upperlimit = lowerlimit + step;
	SzeroOfX = qg32( lowerlimit, upperlimit, ExponentialInt);

	do
	{
		lowerlimit = upperlimit;
		step *= 2;
		upperlimit = lowerlimit + step;
		SzeroOfX += qg32( lowerlimit, upperlimit, ExponentialInt);
	} while ( upperlimit < 20. );

	/* This must be placed inside integral...too big to be 
	 * handled separately.	
	SzeroOfX *= exp( x );	*/

	/* Equations 13 and 14 */
	lowerlimit = 0.;
	step = 0.5;
	upperlimit = lowerlimit + step;
	SoneOfX = qg32( lowerlimit, upperlimit, X1Int);
	StwoOfX = qg32( lowerlimit, upperlimit, X2Int);

	do
	{
		lowerlimit = upperlimit;
		step *= 2;
		upperlimit = lowerlimit + step;
		SoneOfX += qg32( lowerlimit, upperlimit, X1Int);
		SoneOfX += qg32( lowerlimit, upperlimit, X2Int);
	} while ( upperlimit < 200. );

	SoneOfX *= 0.1728 * pow( x, 1./3. );
	StwoOfX *= -0.0496 * pow( x, 2./3. );

	/* Equation 11	*/
	SnOfLambda = SzeroOfX + pow(1./lambda, 1./3.)*SoneOfX + pow(1./lambda, 2./3.)*StwoOfX;

	AlphaN = 5.197E-14 * nelem * pow(x, 1.5) * SnOfLambda;

	return AlphaN;

}

static double ExponentialInt( double v )
{
	double Integrand;

	Integrand = exp( -1. * v + Xn_S59) / v;

	return Integrand;
}

static double X1Int( double u )
{

	double Integrand;

	Integrand = pow(1./(u + 1.), 5./3.) * (u - 1.) * exp( -1. * Xn_S59 * u );

	return Integrand;
}

static double X2Int( double u )
{

	double Integrand;

	Integrand = pow(1./(u + 1.), 7./3.) * (u*u + 4./3.*u + 1.) * exp( -1. * Xn_S59 * u );

	return Integrand;
}

#undef DEBUG_LOC 


void HelikeRecombSetup( void )
{

	double RadRecombReturn;
	long int i, i1, i2, i3, i4, i5;
	long int ipLo, nelem, *NumHeLevRecomb=NULL;

#	define chLine_LENGTH 1000
	char chLine[chLine_LENGTH] , 
		/* this must be longer than chDataPath, set in path.h */
		chFilename[FILENAME_PATH_LENGTH_2];

	FILE *ioDATA;
	int lgEOL;

	/*double **TotalRecomb[LIMELM][N_HE_TE_RECOMB]*/
	if( (TotalRecomb = (double **)MALLOC(sizeof(double *)*(unsigned)LIMELM ) )==NULL )
		BadMalloc();

	for( nelem=ipHELIUM; nelem < LIMELM; ++nelem )
	{
		/*double ***TotalRecomb[LIMELM][N_HE_TE_RECOMB]*/
		if( (TotalRecomb[nelem] = (double*)MALLOC(sizeof(double)*(unsigned)N_HE_TE_RECOMB ))==NULL )
			BadMalloc();
	}

	if( !helike.lgNoRecombInterp )
	{
		/*double ***RRCoef[LIMELM][MaxLevels][N_HE_TE_RECOMB]*/
		if( (RRCoef = (double ***)MALLOC(sizeof(double **)*(unsigned)LIMELM ) )==NULL )
			BadMalloc();

		/* The number of recombination coefficients to be read from file for each element.	*/
		if( (NumHeLevRecomb = (long*)MALLOC(sizeof(long)*(unsigned)LIMELM ) )==NULL )
			BadMalloc();

		for( nelem=ipHELIUM; nelem < LIMELM; ++nelem )
		{
			long int MaxLevels;

			if( nelem == ipHELIUM )
				NumHeLevRecomb[nelem]= ( ( 1 + HE_RREC_MAXN ) * HE_RREC_MAXN + 1 );
			else
				NumHeLevRecomb[nelem]= ( ( 1 + HE_LIKE_RREC_MAXN ) * HE_LIKE_RREC_MAXN + 1 );

			if( nelem==ipHELIUM || dense.lgElmtOn[nelem] )
			{
				/* must always have at least NumHeLevRecomb[nelem] levels since that is number 
				* that will be read in from he rec data file, but possible to require more */
				MaxLevels = MAX2( NumHeLevRecomb[nelem] , iso.numLevels_max[ipHE_LIKE][nelem] );

				/* always define this */
				/* >>chng 02 jan 24, RRCoef will be iso.numLevels_max[ipHE_LIKE][nelem] levels, no iso.numLevels_max,
				* code will stop if more than this is requested */
				if( (RRCoef[nelem] = (double**)MALLOC(sizeof(double*)*(unsigned)(MaxLevels) ))==NULL )
					BadMalloc();
					
				for( ipLo=ipHe1s1S; ipLo < MaxLevels ;++ipLo )
				{
					if( (RRCoef[nelem][ipLo] = (double*)MALLOC(sizeof(double)*(unsigned)N_HE_TE_RECOMB ))==NULL )
						BadMalloc();
				}
			}
		}
	}

	/******************************************************************/
	/**  Establish radiative recombination rate coefficients - RRC	***/
	/******************************************************************/
	if( helike.lgCompileRecomb || !helike.lgNoRecombInterp )
	{
		for (i = 0; i < N_HE_TE_RECOMB; i++)
		{
			/* this is the vector of temperatures */
			TeRRCoef[i] = 0.25*(i);
		}
	}

	/* This flag says we are not compiling the data file	*/
	if ( !helike.lgCompileRecomb && !helike.lgNoRecombInterp )
	{
		/* check on path if file not here and path set */
		/* path was parsed in getset */
		if( lgDataPathSet == TRUE )
		{
			/*path set, so look only there */
			strcpy( chFilename , chDataPath );
			strcat( chFilename , "he_iso_recomb.dat" );
		}
		else
		{
			/* path not set, check local space only */
			strcpy( chFilename , "he_iso_recomb.dat" );
		}

		if( trace.lgTrace )
			fprintf( ioQQQ," HeCreate opening he_iso_recomb.dat:");

		/* Now try to read from file...*/
		if( ( ioDATA = fopen( chFilename , "r" ) ) == NULL )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n" );
			if( lgDataPathSet == TRUE )
				fprintf( ioQQQ, " even tried path\n" );

			if( lgDataPathSet == TRUE )
			{
				fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat\n");
				fprintf( ioQQQ, " path is ==%s==\n",chDataPath );
				fprintf( ioQQQ, " final path is ==%s==\n",chFilename );
				fprintf( ioQQQ, " Defaulting to on-the-fly computation, ");
				fprintf( ioQQQ, " but the code runs much faster if you compile he_iso_recomb.dat!\n");
			}

			/* Do on the fly computation of R.R. Coef's instead.	*/
			for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
			{
				if( dense.lgElmtOn[nelem] )
				{
					/* Zero out the recombination sum array.	*/
					for(i = 0; i < N_HE_TE_RECOMB; i++)
					{
						TotalRecomb[nelem][i] = 0.;
					}
					
					/* NumHeLevRecomb[nelem] corresponds to n = 40 for He and 10 for ions, at present	*/
					/* There is no need to fill in values for collapsed levels, because we do not need to
					* interpolate for a given temperature, just calculate it directly with a hydrogenic routine.	*/
					for( ipLo=ipHe1s1S; ipLo < iso.numLevels_max[ipHE_LIKE][nelem]-iso.nCollapsed_max[ipHE_LIKE][nelem]; ipLo++ )
					{
						EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo];

						/* loop over temperatures to produce array of recombination coefficients	*/
						for (i = 0; i < N_HE_TE_RECOMB; i++)
						{
							/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
							RadRecombReturn = radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo);
							TotalRecomb[nelem][i] += RadRecombReturn;
							RRCoef[nelem][ipLo][i] = log10(RadRecombReturn);
						}
					}
					for (i = 0; i < N_HE_TE_RECOMB; i++)
					{
						/* for( i1 = ( (nelem == ipHELIUM) ? (HE_RREC_MAXN + 1) : (HE_LIKE_RREC_MAXN + 1) ); i1<=SumUpToThisN; i1++ )*/
						for( i1 = iso.n_HighestResolved_max[ipHE_LIKE][nelem]+1; i1<=SumUpToThisN; i1++ )
						{
							TotalRecomb[nelem][i] += Recomb_Seaton59( nelem, pow(10.,TeRRCoef[i]), i1 );
						}
						TotalRecomb[nelem][i] = log10( TotalRecomb[nelem][i] );
					}
				}
			}
		}
		/* Data file is present and readable...begin read.	*/
		else 
		{
			/* check that magic number is ok */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			{
				fprintf( ioQQQ, " HeCreate could not read first line of he_iso_recomb.dat.\n");
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}
			i = 1;
			i1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i4 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			if( i1 !=RECOMBMAGIC || i2 !=NumHeLevRecomb[ipHELIUM] || i3 !=NumHeLevRecomb[ipLITHIUM] || i4 !=N_HE_TE_RECOMB )
			{
				fprintf( ioQQQ, 
					" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
				fprintf( ioQQQ, 
					" HeCreate: I expected to find the numbers  %i %li %li %i and got %li %li %li %li instead.\n" ,
					RECOMBMAGIC ,
					NumHeLevRecomb[ipHELIUM],
					NumHeLevRecomb[ipLITHIUM],
					N_HE_TE_RECOMB,
					i1 , i2 , i3, i4 );
				fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
				fprintf( ioQQQ, 
					" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}

			i5 = 1;
			/* now read in the data */
			for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
			{
				for( ipLo=ipHe1s1S; ipLo <= NumHeLevRecomb[nelem]; ipLo++ )
				{
					i5++;
					/* get next line image */
					if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
					{
						fprintf( ioQQQ, " HeCreate could not read line %li of he_iso_recomb.dat.\n", i5);
						puts( "[Stop in HeCreate]" );
						cdEXIT(EXIT_FAILURE);
					}
					/* each line starts with element and level number */
					i3 = 1;
					i1 = (long)FFmtRead(chLine,&i3,INPUT_LINE_LENGTH,&lgEOL);
					i2 = (long)FFmtRead(chLine,&i3,INPUT_LINE_LENGTH,&lgEOL);
					/* check that these number are correct */
					if( i1!=nelem || i2!=ipLo )
					{
						fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
						fprintf( ioQQQ, 
							" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
						puts( "[Stop in HeCreate]" );
						cdEXIT(EXIT_FAILURE);
					}

					/* loop over temperatures to produce array of recombination coefficients	*/
					for (i = 0; i < N_HE_TE_RECOMB; i++)
					{
						double ThisCoef = FFmtRead(chLine,&i3,chLine_LENGTH,&lgEOL);

						if( nelem == ipHELIUM || dense.lgElmtOn[nelem] )
						{
							/* The last line for each element is the total recombination for each temp.	*/
							if( ipLo == NumHeLevRecomb[nelem] )
							{
								TotalRecomb[nelem][i] = ThisCoef;
							}
							else
								RRCoef[nelem][ipLo][i] = ThisCoef;
						}

						if( lgEOL )
						{
							fprintf( ioQQQ, " HeCreate detected insanity in he_iso_recomb.dat.\n");
							fprintf( ioQQQ, 
								" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
							puts( "[Stop in HeCreate]" );
							cdEXIT(EXIT_FAILURE);
						}
					}
				}

				/* following loop only executed if we need more levels than are
				 * stored in the recom coef data set
				 * do not do collapsed levels since will use H-like recom there */
				if( nelem == ipHELIUM || dense.lgElmtOn[nelem] ) 
				{
					for( ipLo=NumHeLevRecomb[nelem]; ipLo < iso.numLevels_max[ipHE_LIKE][nelem]-iso.nCollapsed_max[ipHE_LIKE][nelem]; ipLo++ )
				    {
					     EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo];
						 
						 for (i = 0; i < N_HE_TE_RECOMB; i++)
					     {
						     /* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
						     RRCoef[nelem][ipLo][i] = log10(radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo));
					     }
				     }
			     }
			}

			/* check that ending magic number is ok */
			if( fgets( chLine , (int)sizeof(chLine) , ioDATA ) == NULL )
			{
				fprintf( ioQQQ, " HeCreate could not read last line of he_iso_recomb.dat.\n");
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}
			i = 1;
			i1 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i2 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i3 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			i4 = (long)FFmtRead(chLine,&i,INPUT_LINE_LENGTH,&lgEOL);
			
			if( i1 !=RECOMBMAGIC || i2 !=NumHeLevRecomb[ipHELIUM] || i3 !=NumHeLevRecomb[ipLITHIUM] || i4 !=N_HE_TE_RECOMB )
			{
				fprintf( ioQQQ, 
					" HeCreate: the version of he_iso_recomb.dat is not the current version.\n" );
				fprintf( ioQQQ, 
					" HeCreate: I expected to find the numbers  %i %li %li %i and got %li %li %li %li instead.\n" ,
					RECOMBMAGIC ,
					NumHeLevRecomb[ipHELIUM],
					NumHeLevRecomb[ipLITHIUM],
					N_HE_TE_RECOMB,
					i1 , i2 , i3, i4 );
				fprintf( ioQQQ, "Here is the line image:\n==%s==\n", chLine );
				fprintf( ioQQQ, 
					" HeCreate: please recompile the data file with the COMPILE HE-LIKE command.\n" );
				puts( "[Stop in HeCreate]" );
				cdEXIT(EXIT_FAILURE);
			}

			/* close the data file */
			fclose( ioDATA );
		}
	}
	/* We are compiling the he_iso_recomb.dat data file.	*/
	else if( helike.lgCompileRecomb )
	{
		/* option to create table of recombination coefficients,
		 * executed with the compile he-like command */
		FILE *ioRECOMB;

		ASSERT( helike.lgNoRecombInterp == FALSE );
		
		/*RECOMBMAGIC the magic number for the table of recombination coefficients */
		/*NumHeLevRecomb[nelem] the number of levels that will be done */
		/* create recombination coefficients  */
		if( ( ioRECOMB = fopen( "he_iso_recomb.dat" , "w" ) ) == NULL )
		{
			fprintf( ioQQQ, " HeCreate could not open he_iso_recomb.dat for writing.\n" );
			puts( "[Stop in HeCreate]" );
			cdEXIT(EXIT_FAILURE);
		}
		fprintf(ioRECOMB,"%i\t%li\t%li\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %li He levels, %li ion levels, and %i temperatures.\n",
			RECOMBMAGIC ,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB );

		for( nelem = ipHELIUM; nelem < LIMELM; nelem++ )
		{
			/* this must pass since compile he-like command reset numLevels to the macro */
			ASSERT( NumHeLevRecomb[nelem] <= iso.numLevels_max[ipHE_LIKE][nelem] );

			/* Zero out the recombination sum array.	*/
			for(i = 0; i < N_HE_TE_RECOMB; i++)
			{
				TotalRecomb[nelem][i] = 0.;
			}
			
			for( ipLo=ipHe1s1S; ipLo < NumHeLevRecomb[nelem] ; ipLo++ )
			{
				EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo];
				
				fprintf(ioRECOMB, "%li\t%li", nelem, ipLo );
				/* loop over temperatures to produce array of recombination coefficients	*/
				for (i = 0; i < N_HE_TE_RECOMB; i++)
				{
					/* Store log of recombination coefficients, in N_HE_TE_RECOMB half dec steps */
					RadRecombReturn = radrecomb( pow( 10.,TeRRCoef[i] ) ,nelem,ipLo);
					TotalRecomb[nelem][i] += RadRecombReturn;
					RRCoef[nelem][ipLo][i] = log10(RadRecombReturn);
					fprintf(ioRECOMB, "\t%f", RRCoef[nelem][ipLo][i] );
				}
				fprintf(ioRECOMB, "\n" );
			}
			
			/* Store one additional line in he_iso_recomb.dat that gives the total recombination,
			 * as computed by the sum so far, plus levels up to SumUpToThisN using Seaton 59, 
			 * for each element and each temperature.	*/
			fprintf(ioRECOMB, "%li\t%li", nelem, NumHeLevRecomb[nelem] );
			for (i = 0; i < N_HE_TE_RECOMB; i++)
			{
				for( i1 = ( (nelem == ipHELIUM) ? (HE_RREC_MAXN + 1) : (HE_LIKE_RREC_MAXN + 1) ); i1<=SumUpToThisN; i1++ )
				{
					TotalRecomb[nelem][i] += Recomb_Seaton59( nelem, pow(10.,TeRRCoef[i]), i1 );
				}
				fprintf(ioRECOMB, "\t%f", log10( TotalRecomb[nelem][i] ) );
			}
			fprintf(ioRECOMB, "\n" );
		}
		/* end the file with the same information */
		fprintf(ioRECOMB,"%i\t%li\t%li\t%i\tHe-like isoelectronic sequence recomb data, created by COMPILE HE-LIKE command, with %li He levels, %li ion levels, and %i temperatures.\n",
			RECOMBMAGIC ,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB,
			NumHeLevRecomb[ipHELIUM],
			NumHeLevRecomb[ipLITHIUM],
			N_HE_TE_RECOMB );

		fclose( ioRECOMB );

		fprintf( ioQQQ, "HeCreate: compilation complete, he_iso_recomb.dat created.\n" );
		puts( "[Stop in HeCreate]" );
		cdEXIT(EXIT_FAILURE);
	}
	/* End of recombination setup.	*/
	
	if( !helike.lgNoRecombInterp )
		free( NumHeLevRecomb );

	return;
}

/*===================================================================================*/
/* HeRecom do recomb coef for He, called by HeLike */
void HeRecom( 
			/* nelem on the c scale, He is 1 */
			long nelem )
{
	static double ditcrt[LIMELM]= {0.,5e4,1.20E+04,1.10E+04,4.40E+05,7.00E+05,1.00E+06,1.50E+06,
	  1.50E+06,3.80E+06,3.40E+04,4.00E+06,5.50E+06,6.30E+06,5.00E+06,9.00E+06,1.40E+07,
	  1.40E+07,2.00E+05,2.00E+05,2.00E+05,1.00E+05,7.00E+04,4.00E+04,6.00E+06,1.00E+11,
	  1.00E+20,1.00E+20,1.00E+20,1.00E+20};

	static double dicoef0[LIMELM]= {0.,1.9e-3,6.15E-03,1.62E-03,4.78E-02,3.22E-02,4.61E-02,6.23E-02,
	  2.52E-01,1.44E-01,1.80E-01,1.44E-01,3.40E+00,1.89E-01,4.02E-01,2.41E-01,1.10E+01,2.97E-01,
	  5.49E-01,3.55E-01,2.30E-01,1.40E-01,1.10E-01,4.10E-02,7.47E-01,5.19E-01,5.75E-01,5.75E-01,
	  5.75E-01,5.75E-01};

	static double dite0[LIMELM]= {0.,4.7e5,1.41E+05,8.19E+04,3.44E+06,4.06E+06,5.44E+06,7.01E+06,1.40E+07,
	  1.50E+07,1.30E+07,1.50E+07,1.70E+07,1.99E+07,2.41E+07,2.54E+07,2.80E+07,3.13E+07,3.65E+07,
	  3.78E+07,1.09E+06,9.62E+05,7.23E+05,4.23E+05,5.84E+07,6.01E+07,6.81E+07,6.81E+07,6.81E+07,
	  6.81E+07};

	static double dicoef1[LIMELM]= {0.,0.3,5.88E-02,3.43E-01,3.62E-01,3.15E-01,3.09E-01,3.04E-01,
	  3.04E-01,2.96E-01,0.00E+00,2.91E-01,0.00E+00,2.86E-01,2.98E-01,2.81E-01,0.00E+00,2.77E-01,
	  2.92E-01,2.75E-01,3.60E+00,4.90E+00,1.60E+00,4.20E+00,2.84E-01,2.79E-01,0.00E+00,2.86E-01,
	  2.86E-01,2.86E-01};

	static double dite1[LIMELM]= {0.,9.4e4,1.41E+05,1.59E+05,5.87E+05,8.31E+05,1.14E+06,1.47E+06,
	  1.80E+06,2.24E+06,0.00E+00,3.09E+06,0.00E+00,4.14E+06,4.67E+06,5.30E+06,0.00E+00,6.54E+06,
	  7.25E+06,7.68E+06,1.10E+07,8.34E+06,1.01E+07,1.07E+07,1.17E+07,9.97E+06,0.00E+00,9.08E+06,
	  9.08E+06,9.08E+06};

	long ipFirstCollapsed, LastN=0L, ThisN=1L, ipLevel; 
	double RecUnitVolTotal, topoff, TotMinusExplicitResolved,
		TotRRThisN=0., TotRRLastN=0., DielRec, t;
	double RecTriplets;
	static double RecExplictLevels[LIMELM], TotalRadRecomb[LIMELM], RecCollapsed[LIMELM];

#	ifdef DEBUG_FUN
	fputs( "<+>HeRecom()\n", debug_fp );
#	endif

	/* evaluate recombination escape probability for all levels */

	/* define radiative recombination rates for all levels */ 
	/* this will be the sum of all levels explicitly included in the model atom */
	RecExplictLevels[nelem] = 0.;
	RecTriplets = 0.;

	/* this is the number of resolved levels, so first collapsed level is [ipFirstCollapsed] */
	ipFirstCollapsed = iso.numLevels_max[ipHE_LIKE][nelem]-iso.nCollapsed_max[ipHE_LIKE][nelem]; 


	for( ipLevel=0; ipLevel<ipFirstCollapsed; ++ipLevel )
	{
		double BenDirectRadRecHe[31] = {
		1.54E-13,1.49E-14,5.55E-15,6.23E-15,1.87E-14,3.12E-14,1.26E-14,3.72E-15,1.62E-15,
		1.95E-14,1.33E-14,4.31E-15,5.01E-15,1.50E-15,7.00E-16,9.04E-15,8.29E-15,2.67E-15,
		4.15E-15,1.38E-15,2.43E-15,7.51E-16,3.66E-16,4.84E-15,4.99E-15,1.60E-15,3.65E-15,
		1.22E-15,1.34E-15,4.46E-16,1.35E-15};

		double BenDirectRadExpHe[31] = {
		0.486,0.381,0.451,0.639,0.639,0.639,0.695,0.344,0.444,0.632,0.868,0.872,0.700,0.328,0.445,
		0.632,0.867,0.871,1.046,1.046,0.708,0.327,0.454,0.636,0.870,0.873,1.048,1.048,1.183,1.183,0.718};

		double BenIndirectRadRecHe[31] = {
		0.      ,7.07E-15,2.25E-15,1.12E-15,3.36E-15,5.61E-15,3.11E-15,1.51E-15,6.09E-16,
		3.63E-15,8.19E-15,2.64E-15,1.30E-15,5.16E-16,2.44E-16,1.62E-15,5.19E-15,1.64E-15,
		6.94E-15,2.31E-15,6.50E-16,2.09E-16,1.16E-16,8.63E-16,3.21E-15,9.86E-16,5.83E-15,
		1.94E-15,8.39E-15,2.80E-15,3.82E-16};

		double BenIndirectRadExpHe[31] = {
		0.000,0.675,0.766,0.901,0.901,0.901,0.879,0.696,0.782,0.873,1.097,1.114,0.876,0.701,0.783,
		0.845,1.078,1.106,1.271,1.271,0.854,0.711,0.786,0.813,1.055,1.096,1.267,1.267,1.375,1.375,0.832};

		double BenIndirectDensExpHe[31] = {
		0.000,-0.001,0.000,0.009,0.009,0.009,0.008,-0.001,0.000,0.010,0.002,0.003,0.008,
		-0.001,0.000,0.010,0.002,0.003,-0.001,-0.001,0.008,-0.001,0.000,0.010,0.002,0.003,
		-0.001,-0.001,-0.007,-0.007,0.008};

		if( helike.lgSetBenjamin && (nelem == ipHELIUM) && ( ipLevel <= 30 ) 
			&& ( phycon.te < 1000000.1f ) && ( phycon.te > 99.9f ) )
		{
			iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] = 
				BenDirectRadRecHe[ipLevel] * pow( 10000./phycon.te, BenDirectRadExpHe[ipLevel] ) +
				BenIndirectRadRecHe[ipLevel] * pow( 10000./phycon.te, BenIndirectRadExpHe[ipLevel] ) * 
				pow( dense.eden/100., BenIndirectDensExpHe[ipLevel] );
		}
		else
		{
			/* this is radiative recombination rate coefficient */
			double RadRec;
			
			if( helike.lgNoRecombInterp==FALSE )
			{
				RadRec = He_RRCoef_Te( nelem , ipLevel );
			}
			else
			{
				EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLevel];
				RadRec = radrecomb( phycon.te, nelem, ipLevel);
			}

			iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] = RadRec;
	
			
			if( helike.lgRandErrGen )
			{
				iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] *=
					helike.ErrorFactor[nelem][ iso.numLevels_max[ipHE_LIKE][nelem] ][ipLevel][IPRAD];
			}
		}

		RecExplictLevels[nelem] += iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad];
		if( S_(ipLevel)==1 )
			RecTriplets += iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad];
		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] > 0. );
	}

	/* do not include ground in this sum	*/
	RecExplictLevels[nelem] -= iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad];	

	/**************************************************/
	/***  Add on recombination to collapsed levels  ***/
	/**************************************************/
	RecCollapsed[nelem] = 0.;
	for( ipLevel=ipFirstCollapsed; ipLevel<iso.numLevels_max[ipHE_LIKE][nelem]; ++ipLevel )
	{
		double RadRec = atmdat_H_rad_rec(nelem, N_(ipLevel), phycon.te);

		/* this is radiative recombination rate coefficient */
		iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] = RadRec;

		/* This kills recombination into the collapsed level so that the forced
		 * statistical weighting can be bypassed */
		if( helike.lgTopoff==FALSE )
		{
			iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] *= 1E-10;
		}

		if( helike.lgRandErrGen )
		{
			iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] *= 
				helike.ErrorFactor[nelem][ iso.numLevels_max[ipHE_LIKE][nelem] ][ipLevel][IPRAD];
		}

		RecCollapsed[nelem] += iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad];

		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] > 0. );
	}

	RecTriplets += 0.75 * RecCollapsed[nelem];

	for( ipLevel = ipHe1s1S; ipLevel<iso.numLevels_max[ipHE_LIKE][nelem]; ipLevel++ )
	{

		if( N_(ipLevel) == ThisN )
		{
			TotRRThisN += iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad];
		}
		else
		{
			ASSERT( (TotRRThisN<TotRRLastN) || (ThisN<=2L) || (phycon.te>3E7) || (nelem!=ipHELIUM) );
			LastN = ThisN;
			ThisN = N_(ipLevel);
			TotRRLastN = TotRRThisN;
			TotRRThisN = iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad];

			{
				/* Print the sum of recombination coefficients per n at current temp.	*/
				/*@-redef@*/
				enum {DEBUG_LOC=FALSE};
				/*@+redef@*/
				static long RUNONCE = FALSE;
				
				if( !RUNONCE && DEBUG_LOC )
				{
					static long FIRSTTIME = TRUE;

					if( FIRSTTIME )
					{
						fprintf( ioQQQ,"Sum of Radiative Recombination at current temp, T=%.2f\n", phycon.te);
						FIRSTTIME= FALSE;
					}
					
					fprintf( ioQQQ,"%li\t%.2e\n",LastN,TotRRLastN );
				}
				RUNONCE = TRUE;
			}
		}
	}

	if( helike.lgSetBenjamin && (nelem == ipHELIUM) )
        /* total topoff to Benjamin numbers */
		TotalRadRecomb[nelem] = 4.27E-13 * pow(10000./phycon.te, 0.678);
	else
	{
		if( !helike.lgNoRecombInterp )
		{
			/* We are interpolating, and total was calculated HelikeRecombSetup */
			TotalRadRecomb[nelem] = He_RRCoef_Te( nelem , ipFirstCollapsed );
		}
		else
		{
			/* We are not interpolating, must calculate total now, as sum of explicit
			 * resolved and collapsed levels, plus additional levels out to the macro
			 * SumUpToThisN */
			TotalRadRecomb[nelem] = RecCollapsed[nelem] + RecExplictLevels[nelem] +
				iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad];

			for( ipLevel= N_(ipFirstCollapsed) + iso.nCollapsed_max[ipHE_LIKE][nelem]; ipLevel<=SumUpToThisN; ipLevel++ )
			{
				TotalRadRecomb[nelem] += Recomb_Seaton59( nelem, phycon.te, ipLevel );
			}
		}

		if( helike.lgRandErrGen )
		{
			TotalRadRecomb[nelem] *= 
				helike.ErrorFactor[nelem][iso.numLevels_max[ipHE_LIKE][nelem]][iso.numLevels_max[ipHE_LIKE][nelem]][IPRAD];
		}
	}

	RecTriplets += 0.75*(TotalRadRecomb[nelem] - RecCollapsed[nelem] - RecExplictLevels[nelem] -
				iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad] );

	/* this is case B recombination, sum without the ground included */
	iso.RadRec_caseB[ipHE_LIKE][nelem] = TotalRadRecomb[nelem] - iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad];
	ASSERT( iso.RadRec_caseB[ipHE_LIKE][nelem] > 0.);

	iso.RadRec_caseB[ipHE_LIKE][nelem] = RecTriplets;

	/* this is used for radiative recombination cooling in CoolEvaluate, this does not include
	 * optical depth effects - may be reevaluated in level solver with opt dep included */
	ionbal.RR_rate_coef_used[nelem][nelem-1] = TotalRadRecomb[nelem];

	/* >>chng 02 jan 07, do not add three body, complete model atom will automatically include it
	 * problem is that most elements will not go to high n */
	/* add three-body recombination, convert to per unit vol 
	RecUnitVolTotal = dense.eden*(TotalRadRecomb[nelem] + ionbal.CotaRate[nelem-1]);*/
	/* convert to per unit vol */
	RecUnitVolTotal = dense.eden*TotalRadRecomb[nelem];

	/**********************************************************************/
	/***  Add topoff (excess) recombination	to top level.  This is only	***/
	/***  done if not using Benjamin command and atom is not full size. ***/
	/**********************************************************************/
	if( !helike.lgSetBenjamin && !iso.lgFullSize[ipHE_LIKE][nelem] )
	{
		/* at this point we have RecExplictLevels[nelem], the sum of radiative recombination 
		 * to all levels explicitly included in the model atom, and RecUnitVolTotal, the total 
		 * recombination rate.  The difference is the amount of "topoff" we will need to do */
		TotMinusExplicitResolved = RecUnitVolTotal/dense.eden - 
			iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad] - RecExplictLevels[nelem];
		
		topoff = TotMinusExplicitResolved - RecCollapsed[nelem];

		/* the atmdat_rad_rec fits are too small at high temperatures, so this atom is
		 * better than the topoff.  Only a problem for helium itself, at high temperatures,
		 * when physics disabled with the atom he-like benjamin command.
		 * complain if the negative topoff is not for this case */
		/* >>chng 02 jul 30, do not print if benjamin is set - this causes several known problems */
		if( topoff < 0. && (nelem!=ipHELIUM || phycon.te < 1e5 ) )
		{
			fprintf(ioQQQ," PROBLEM  negative topoff for %li, rel error was %.2e, temp was %.2f\n",  
				nelem, topoff/(RecUnitVolTotal/dense.eden), phycon.te );
		}
		
		if( helike.lgTopoff==FALSE )
			topoff *= 1E-20;
		
		/* >>chng 02 feb 24...now always have at least one collapsed level...so
		 * this bit greatly simplifies.	*/
		iso.RadRecomb[ipHE_LIKE][nelem][iso.numLevels_max[ipHE_LIKE][nelem]-1][ipRecRad] += 
			MAX2(0., topoff );

		/*for( ipLevel=QuantumNumbers2Index[nelem][16][0][1]; ipLevel<ipFirstCollapsed; ipLevel++ )
		{
			iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad] += 
				(2.*L_(ipLevel)+1)*(2.*S_(ipLevel)+1)*topoff/1024;
		}*/
	}

	/* If we have the variable set, go ahead and find effective recom, print, and exit. */
	if( nelem == ipHELIUM && helike.lgHugeCaseB )
	{
		long ipLo, ipHi, ipHigher;

		if( helike.lgRandErrGen )
		{
			fprintf( ioQQQ, "ipHi\tipLo\tWL\tEmiss\tSigmaEmiss\tRadEffec\tSigRadEff\tBrRat\tSigBrRat\n" );
		}
		else
		{
			fprintf( ioQQQ, "ipHi\tipLo\tWL\tEmiss\n" );
		}

		/* Find effective recombinations and errors thereof */
		for( ipHi=ipHe2s3S; ipHi < iso.numLevels_max[ipHE_LIKE][nelem]; ipHi++ )
		{
			helike.RadEffec[ipHi] = 0.;
			if( helike.lgRandErrGen )
			{
				helike.SigmaRadEffec[ipHi] = 0.;
			}

			for( ipHigher=ipHi; ipHigher < iso.numLevels_max[ipHE_LIKE][nelem]; ipHigher++ )
			{
				helike.RadEffec[ipHi] += helike.CascadeProb[ipHigher][ipHi] *
					iso.RadRecomb[ipHE_LIKE][nelem][ipHigher][ipRecRad];

				if( helike.lgRandErrGen )
				{
					ASSERT( helike.Error[nelem][iso.numLevels_max[ipHE_LIKE][nelem]][ipHigher][IPRAD] >= 0. );
					ASSERT( helike.CascadeProb[ipHigher][ipHi] >= 0. );
					ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][ipHigher][ipRecRad] >= 0. );
					ASSERT( helike.SigmaCascadeProb[ipHigher][ipHi] >= 0. );
					
					helike.SigmaRadEffec[ipHi] += pow( helike.Error[nelem][iso.numLevels_max[ipHE_LIKE][nelem]][ipHigher][IPRAD] *
						helike.CascadeProb[ipHigher][ipHi] * iso.RadRecomb[ipHE_LIKE][nelem][ipHigher][ipRecRad], 2.) + 
						pow( helike.SigmaCascadeProb[ipHigher][ipHi] * iso.RadRecomb[ipHE_LIKE][nelem][ipHigher][ipRecRad], 2.);
				}
			}

			ASSERT( helike.SigmaRadEffec[ipHi] >= 0. );
			helike.SigmaRadEffec[ipHi] = sqrt( helike.SigmaRadEffec[ipHi] );

			if( N_(ipHi) <= 7 )
			{
				fprintf( ioQQQ, "%li\t%li\t%li\t%li\t%e\t%e\n",
					ipHi, 
					N_(ipHi),
					L_(ipHi),
					S_(ipHi),
					helike.RadEffec[ipHi],
					helike.SigmaRadEffec[ipHi]);
			}

			for( ipLo = ipHe1s1S; ipLo < ipHi; ipLo++ )
			{
				if( (( L_(ipLo) == L_(ipHi) + 1 ) || ( L_(ipHi) == L_(ipLo) + 1 )) )
				{
					double EnergyInRydbergs = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipLo] - iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][ipHi];
					float wavelength = (float)(RYDLAM/EnergyInRydbergs);
					double emissivity = helike.RadEffec[ipHi] * helike.BranchRatio[nelem][ipHi][ipLo] * EN1RYD * EnergyInRydbergs;
					double sigma_emiss = 0., SigmaBranchRatio = 0.;

					if( ipLo==ipHe2p3P0 )
					{
						emissivity =  helike.RadEffec[ipHi] * EN1RYD * EnergyInRydbergs *
							(helike.BranchRatio[nelem][ipHi][ipLo] +
							helike.BranchRatio[nelem][ipHi][ipLo+1] +
							helike.BranchRatio[nelem][ipHi][ipLo+2]);
					}

					if( helike.lgRandErrGen )
					{
						SigmaBranchRatio = helike.BranchRatio[nelem][ipHi][ipLo] * sqrt( 
							pow( (double)helike.Error[nelem][ipHi][ipLo][IPRAD], 2. ) +
							pow( helike.SigmaAtot[nelem][ipHi]*helike.Lifetime[nelem][ipHi], 2.) );

						sigma_emiss =  EN1RYD * EnergyInRydbergs * sqrt( 
							pow( (double)helike.SigmaRadEffec[ipHi] * helike.BranchRatio[nelem][ipHi][ipLo], 2.) +
							pow( SigmaBranchRatio * helike.RadEffec[ipHi], 2.) );

					}

					if( ( emissivity > 2.E-29 ) && ( wavelength < 1.E6 ) &&
						(ipLo!=ipHe2p3P1) && (ipLo!=ipHe2p3P2) && (N_(ipHi)<=5) )
					{
						fprintf( ioQQQ, "%li\t%li\t%e\t%e\t%e\t%e\t%e\t%e\t%e\n", 
							ipHi, ipLo, wavelength, emissivity,
							helike.BranchRatio[nelem][ipHi][ipLo],
							helike.RadEffec[ipHi],
							helike.Lifetime[nelem][ipHi],
							SigmaBranchRatio,
							sigma_emiss);
					}
				}
			}
		}

		cdEXIT(EXIT_SUCCESS);
	}
		

	/***********************************************/
	/***  Add on Dielectronic recombination	- DR ***/
	/***********************************************/
	DielRec = 0.;
	if( helike.lgDielRecom > 0 )
	{
		if( nelem==ipIRON )
		{
			/* for case of iron, add on Dima's fits to diel part, above was zero,
			 * this requests He-like Fe */
			DielRec = atmdat_dielrec_fe(25,phycon.te);
		}
		else if( phycon.te > (ditcrt[nelem]*0.1) )
		{
			/* the suppression factor is on a charge scale, so rec to neutral
			 * is always 0, hence the nelem-1 */
			if( nelem == ipHELIUM )
			{
				/* this is the dr rate that was in the original He singlets coding, and has been
				 * part of cloudy since the beginning - based on Aldrovandi & Pequignot */
				/* >>refer He	DR	Aldrovandi, S.M.V., & Pequignot, D. 1973, A&A 25, 137	*/
				/* >>chng 04 oct 15, there was no need for the check for over 50000K.  
				 * The function is well-behaved, the rate is not quite neglible below 50000K, 
				 * it saves no computation time, and flow doesn't get here unless temp passes
				 * test just above. */
				t = (float)(1.9e-3/phycon.te32*sexp(4.7e5/phycon.te)*(1. + 0.3*sexp(9.4e4/phycon.te)));
				DielRec = t;
			}
			else
			{
				if( helike.lgDielRecom == 1 && nelem >= 12 )
				{
					long i;
					double EoverT;
					double ParamsE[28][3] ={
						{5.966E+01,	7.165E+01,	7.686E+01}, {1.066E+02,	1.289E+02,	1.398E+02},
						{1.672E+02,	2.032E+02,	2.223E+02},	{2.415E+02,	2.948E+02,	3.248E+02},
						{3.296E+02,	4.038E+02,	4.476E+02},	{4.314E+02,	5.303E+02,	5.909E+02},
						{5.471E+02,	6.744E+02,	7.548E+02},	{6.767E+02,	8.362E+02,	9.397E+02},
						{8.202E+02,	1.016E+03,	1.146E+03},	{9.749E+02,	1.200E+03,	1.353E+03},
						{1.149E+03,	1.428E+03,	1.622E+03},	{1.335E+03,	1.662E+03,	1.892E+03},
						{1.533E+03,	1.913E+03,	2.184E+03},	{1.749E+03,	2.195E+03,	2.520E+03},
						{1.974E+03,	2.470E+03,	2.833E+03},	{2.218E+03,	2.799E+03,	3.228E+03},
						{2.470E+03,	3.099E+03,	3.570E+03},	{2.741E+03,	3.469E+03,	4.014E+03},
						{3.023E+03,	3.801E+03,	4.396E+03},	{3.320E+03,	4.180E+03,	4.843E+03},
						{3.631E+03,	4.577E+03,	5.312E+03},	{3.957E+03,	4.992E+03,	5.803E+03},
						{4.297E+03,	5.426E+03,	6.317E+03},	{4.641E+03,	5.797E+03,	6.740E+03},
						{5.018E+03,	6.348E+03,	7.414E+03},	{5.399E+03,	6.845E+03,	8.000E+03},
						{5.796E+03,	7.345E+03,	8.601E+03},	{6.207E+03,	7.871E+03,	9.230E+03}	};
					double ParamsC[28][3] ={
						{6.774E-13,	2.089E-12,	1.116E-12}, {6.774E-13,	2.089E-12,	1.116E-12},
						{6.774E-13,	2.089E-12,	1.116E-12}, {6.774E-13,	2.089E-12,	1.116E-12},
						{6.774E-13,	2.089E-12,	1.116E-12}, {6.774E-13,	2.089E-12,	1.116E-12},
						{6.774E-13,	2.089E-12,	1.116E-12}, {6.774E-13,	2.089E-12,	1.116E-12},
						{6.774E-13,	2.089E-12,	1.116E-12}, {6.774E-13,	2.089E-12,	1.116E-12},
						{7.301E-13,	2.039E-12,	9.005E-13}, {7.827E-13,	1.988E-12,	6.850E-13},
						{8.197E-13,	1.867E-12,	5.788E-13}, {8.567E-13,	1.746E-12,	4.726E-13},
						{8.762E-13,	1.609E-12,	4.190E-13}, {8.956E-13,	1.471E-12,	3.653E-13},
						{8.942E-13,	1.337E-12,	3.374E-13}, {8.927E-13,	1.202E-12,	3.095E-13},
						{8.584E-13,	1.100E-12,	3.059E-13}, {8.241E-13,	9.970E-13,	3.022E-13},
						{7.899E-13,	8.945E-13,	2.986E-13}, {7.556E-13,	7.920E-13,	2.950E-13},
						{7.213E-13,	6.895E-13,	2.913E-13}, {6.870E-13,	5.870E-13,	2.877E-13},
						{6.548E-13,	5.686E-13,	2.348E-13}, {6.226E-13,	5.502E-13,	1.818E-13},
						{6.226E-13,	5.502E-13,	1.818E-13}, {6.226E-13,	5.502E-13,	1.818E-13}  };

					/* Use Gu data	*/
					/* >>refer He	Diel Recom 	Gu, M. F. private communication, 2003	*/  
					DielRec = 0.;
					for( i = 1; i <= 3; i++ )
					{
						ASSERT( nelem <= 29 );
						EoverT = ParamsE[nelem-2][i-1]*TE1RYD/phycon.te/EVRYD;
						DielRec += ParamsC[nelem-2][i-1]*pow(EoverT,1.5)*sexp(EoverT);
					}
				}
				else
				{
					/* Use old treatment	*/
					/* TODO	2	put Burgess reference here	*/
					/* TODO	2	I can't find this reference, should be killed anyway? --RLP */
					DielRec = ionbal.DielSupprs[0][nelem-1]/phycon.te32*dicoef0[nelem]*
						exp(-dite0[nelem]/phycon.te)*(1. + dicoef1[nelem]*sexp(dite1[nelem]/phycon.te));
				}
			}
		}
	}

	iso.RadRecomb[ipHE_LIKE][nelem][iso.numLevels_max[ipHE_LIKE][nelem]-1][ipRecRad] += 
			MAX2(0., DielRec );

	RecUnitVolTotal += DielRec*dense.eden;
	/* fprintf( ioQQQ,"nelem\t%li\DielRec\t%.3e\n",nelem,DielRec); */

	/* option for case b conditions, kill ground state recombination */
	if( opac.lgCaseB )
	{
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecEsc] = 1e-10;
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecNetEsc] = 1e-10;
	}
	else
	{
		double temp = RT_recom_effic(iso.ipIsoLevNIonCon[ipHE_LIKE][nelem][ipHe1s1S]);
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecEsc] = 
			MAX2( temp , opac.otsmin);
		iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecNetEsc] = 
			MIN2(1.,iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecEsc]+
			(1.-iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecEsc])*
			iso.ConOpacRatio[ipHE_LIKE][nelem][ipHe1s1S]);
	}
	/* total continuum effective escape prob for all levels  */
	/* RadRec_effec is total effective radiative recombination */
	iso.RadRec_effec[ipHE_LIKE][nelem] = iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecRad]*
		  iso.RadRecomb[ipHE_LIKE][nelem][ipHe1s1S][ipRecNetEsc];

	/**************************************************************/
	/***  Stuff escape probabilities, and effective rad recomb  ***/
	/**************************************************************/
	for( ipLevel=ipHe1s1S+1; ipLevel<iso.numLevels_max[ipHE_LIKE][nelem]; ++ipLevel )
	{
		/* this is escape probability */
		iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc] = 
			RT_recom_effic(iso.ipIsoLevNIonCon[ipHE_LIKE][nelem][ipLevel]);

		/* correct above value for possible otsmin value
		 * otsmin set to zero in zerologic, set to 1 with 
		 * the NO ON THE SPOT command */
		iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc] = 
			MAX2(iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc], opac.otsmin);

		/* net escape prob includes dest by background opacity */
		iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecNetEsc] = 
			MIN2(1.,iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc]+
		  (1.-iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc])*iso.ConOpacRatio[ipHE_LIKE][nelem][ipLevel]);

		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecEsc] >= 0. );
		ASSERT( iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecNetEsc] >= 0. );

		/* sum of all effective rad rec */
		iso.RadRec_effec[ipHE_LIKE][nelem] += iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecRad]*
		  iso.RadRecomb[ipHE_LIKE][nelem][ipLevel][ipRecNetEsc];
	}

#	ifdef DEBUG_FUN
	fputs( " <->HeRecom()\n", debug_fp );
#	endif

	return;
}

/* TempInterp - interpolate on an array
 *TODO	2	use a canned interpolation routine, no need for special one here */
static double TempInterp( double* TempArray , double* ValueArray, long NumElements )
{
	static long int ipTe=-1;
	double rate = 0.;
	double Xarray[4], Yarray[4];
	long i, index[4], numterms = 3;

#	ifdef DEBUG_FUN
	fputs( "<+>TempInterp()\n", debug_fp );
#	endif

	ASSERT( fabs( 1. - (double)phycon.alogte/log10((double)phycon.te) ) < 0.0001 );

	if( ipTe < 0 )
	{
		/* te totally unknown */
		if( ( phycon.alogte < TempArray[0] ) || 
			( phycon.alogte > TempArray[NumElements-1] ) )
		{
			fprintf(ioQQQ," TempInterp called with te out of bounds \n");
			puts( "[Stop in TempInterp]" );
			cdEXIT(EXIT_FAILURE);
		}
		/* now search for temperature */
		for( i=0; i<NumElements-1; ++i )
		{
			if( ( phycon.alogte > TempArray[i] ) && 
				( phycon.alogte <= TempArray[i+1] ) )
			{
				/* found the temperature - use it */
				ipTe = i;
				break;
			}
		}
		ASSERT( (ipTe >=0) && (ipTe < NumElements-1)  );
			
	}
	else if( phycon.alogte < TempArray[ipTe] )
	{
		/* temp is too low, must also lower ipTe */
		ASSERT( phycon.alogte > TempArray[0] );
		/* decrement ipTe until it is correct */
		while( ( phycon.alogte < TempArray[ipTe] ) && ipTe > 0)
			--ipTe;
	}
	else if( phycon.alogte > TempArray[ipTe+1] )
	{
		/* temp is too high */
		ASSERT( phycon.alogte <= TempArray[NumElements-1] );
		/* increment ipTe until it is correct */
		while( ( phycon.alogte > TempArray[ipTe+1] ) && ipTe < NumElements-1)
			++ipTe;
	}

	ASSERT( (ipTe >=0) && (ipTe < NumElements-1)  );

	/* ipTe should now be valid */
	ASSERT( ( phycon.alogte >= TempArray[ipTe] )
		&& ( phycon.alogte <= TempArray[ipTe+1] ) && ( ipTe < NumElements-1 ) );

#if	0
	rate = ((double)phycon.alogte - TempArray[ipTe]) / (TempArray[ipTe+1] - TempArray[ipTe]) *
		(ValueArray[ipTe+1] - ValueArray[ipTe]) + ValueArray[ipTe] ;
#endif
	
	/* Do a four-point interpolation */
	if( ipTe==0 || ipTe==1 )
	{
		index[0]=0;
		index[1]=1;
		index[2]=2;
		index[3]=3;
	}
	else if( ipTe==NumElements-2 )
	{
		index[0]=NumElements-4;
		index[1]=NumElements-3;
		index[2]=NumElements-2;
		index[3]=NumElements-1;
	}
	else
	{
		index[0]=ipTe-1;
		index[1]=ipTe;
		index[2]=ipTe+1;
		index[3]=ipTe+2;
	}

	for( i=0; i<=3; i++ )
	{
		Xarray[i] = TempArray[index[i]];
		Yarray[i] = ValueArray[index[i]];
	}

	interp(Xarray, Yarray, 4, &numterms, phycon.alogte, &rate);

#	ifdef DEBUG_FUN
	fputs( " <->TempInterp()\n", debug_fp );
#	endif

	return rate;
}

/*He_RRCoef_Te evaluated radiative recombination coef at some temperature */
static double He_RRCoef_Te( long nelem , long n )
{
	double rate = 0.;

#	ifdef DEBUG_FUN
	fputs( "<+>He_RRCoef_Te()\n", debug_fp );
#	endif

	ASSERT( helike.lgNoRecombInterp == FALSE );

	/* if n is equal to the number of levels, return the total recomb, else recomb for given level.	*/
	if( n == iso.numLevels_max[ipHE_LIKE][nelem] - iso.nCollapsed_max[ipHE_LIKE][nelem] )
	{
		rate = TempInterp( TeRRCoef, TotalRecomb[nelem], N_HE_TE_RECOMB );
	}
	else
	{
		rate = TempInterp( TeRRCoef, RRCoef[nelem][n], N_HE_TE_RECOMB );
	}

	/* that was the log, now make linear */
	rate = pow( 10. , rate );

#	ifdef DEBUG_FUN
	fputs( " <->He_RRCoef_Te()\n", debug_fp );
#	endif

	return rate;
}

/*HelikeCheckRecomb called by SanityCheck to confirm that recombination coef are ok
 * return value is relative error between new calculation of recom, and interp value */
double HelikeCheckRecomb(
	/* the chemical element, 1 for He */
	long nelem ,
	/* the level, 0 for ground */
	long level ,
	/* the temperature to be used */
	double temperature )
{
	double RecombRelError ,
		RecombInterp,
		RecombCalc,
		SaveTemp;

#	ifdef DEBUG_FUN
	fputs( "<+>HelikeCheckRecomb()\n", debug_fp );
#	endif

	/* this global variable is needed by radrecomb */
	EthRyd = iso.xIsoLevNIonRyd[ipHE_LIKE][nelem][level];

	/* first set temp todesired value */
	SaveTemp = phycon.te;
	phycon.te = temperature;
	tfidle(TRUE);

	/* actually calculate the recomb coef from the Milne relation,
	 * normally only done due to compile he-like command */
	RecombCalc = radrecomb( temperature , nelem , level );

	/* interpolate the recomb coef, this is the usual method */
	RecombInterp = He_RRCoef_Te( nelem , level );

	/* reset temp */
	phycon.te = SaveTemp;
	tfidle(TRUE);

	RecombRelError = ( RecombInterp - RecombCalc ) / MAX2( RecombInterp , RecombCalc );

#	ifdef DEBUG_FUN
	fputs( " <->HelikeCheckRecomb()\n", debug_fp );
#	endif

	return RecombRelError ;
}


/*lint +e662 creation of  out of bound pointer */

#undef PARALLEL
/*lint +e662 creation of  out of bound pointer */
/*lint +e661 creation of  out of bound pointer */









