/* This file is part of Cloudy and is copyright (C)1978-2006 by Gary J. Ferland
 * For conditions of distribution and use see copyright notice in license.txt */
/*kurucz79 obtain interpolated Kurucz stellar atmosphere */
#include "cddefines.h"
#define	NDATA	56
#include "rfield.h"
#include "stars.h"

void kurucz79(long int *nstar, 
  double par1, 
  double par2)
{
	long int i, 
	  mtemp;

	double
		alogg; 
	float 
	  fac1, 
	  fac2, 
	  sc30, 
	  sc35, 
	  sc40, 
	  sc45, 
	  sc50,
	  temp;
	static int lgFirst = TRUE;
	/* wavelengths in nanometers */
	static float wl[NDATA]={22.9f,23.4f,24.3f,25.2f,25.6f,26.0f,26.6f,27.5f,
	  28.3f,29.3f,30.1f,30.9f,32.5f,34.2f,35.1f,35.6f,36.5f,37.5f,38.4f,40.0f,
	  41.5f,42.5f,44.5f,47.2f,49.4f,50.6f,51.6f,54.0f,56.4f,58.7f,61.2f,64.0f,
	  67.1f,70.5f,73.6f,77.0f,81.0f,85.0f,89.0f,92.0f,94.0f,118.7f,143.3f,194.0f,
	  256.2f,331.2f,406.2f,471.2f,592.5f,775.0f,1035.f,1335.f,5000.f,6500.f,
	  1.e4f,9e6f};

	/* following atmospheres LTE from Kurucz 1979f, Ap.J. Sup 40f, 1. and
	 * Kurucz (1989) private communicationf, newer opacitiesf,
	 * log f_lam */
	static float t30g4[NDATA]={2.67f,3.75f,4.16f,4.53f,4.32f,4.81f,4.92f,
	  5.19f,5.42f,5.65f,5.53f,5.83f,6.30f,6.63f,6.77f,6.82f,6.97f,7.09f,7.24f,
	  7.44f,7.61f,7.70f,7.91f,8.13f,8.31f,9.55f,9.74f,9.74f,9.81f,9.83f,9.90f,
	  9.94f,9.98f,10.02f,10.06f,10.12f,10.14f,10.13f,10.19f,10.72f,11.24f,11.16f,
	  10.96f,10.59f,10.25f,9.87f,9.63f,9.40f,9.03f,8.59f,8.12f,7.69f,5.43f,4.97f,
	  4.22f,-7.60f};
	static float t35g45[NDATA]={4.96f,5.32f,5.55f,5.86f,6.00f,6.19f,6.53f,
	  6.78f,7.07f,7.23f,7.59f,7.84f,8.32f,8.60f,8.8f,9.00f,9.30f,9.30f,9.40f,9.60f,
	  9.70f,9.73f,9.83f,9.92f,10.00f,10.34f,10.92f,10.78f,10.97f,10.97f,10.92f,
	  10.99f,11.00f,10.88f,11.02f,10.99f,10.98f,10.95f,10.97f,11.30f,11.57f,
	  11.36f,11.19f,10.81f,10.43f,10.04f,9.76f,9.53f,9.16f,8.72f,8.24f,7.80f,
	  5.52f,5.07f,4.32f,-7.50f};
	static float t40g45[NDATA]={6.93f,7.39f,7.60f,7.90f,8.02f,8.13f,8.85f,
	  9.00f,9.23f,9.35f,9.53f,10.06f,10.35f,10.49f,10.55f,10.80f,11.10f,11.08f,
	  10.95f,11.33f,11.32f,11.28f,11.33f,11.27f,11.28f,11.28f,11.53f,11.38f,
	  11.51f,11.53f,11.45f,11.54f,11.52f,11.41f,11.52f,11.43f,11.45f,11.42f,
	  11.40f,11.43f,11.68f,11.51f,11.31f,10.93f,10.54f,10.15f,9.85f,9.62f,9.24f,
	  8.79f,8.31f,7.88f,5.60f,5.14f,4.39f,-7.43f};
	static float t45g45[NDATA]={8.32f,9.34f,9.40f,10.14f,10.10f,10.51f,11.46f,
	  11.40f,11.40f,11.52f,10.68f,11.55f,11.89f,11.95f,11.85f,11.84f,11.84f,
	  11.77f,11.48f,11.72f,11.74f,11.73f,11.80f,11.77f,11.80f,11.79f,11.83f,
	  11.80f,11.81f,11.80f,11.79f,11.77f,11.76f,11.73f,11.73f,11.68f,11.67f,
	  11.63f,11.61f,11.64f,11.79f,11.62f,11.37f,10.99f,10.60f,10.20f,9.90f,9.66f,
	  9.28f,8.84f,8.36f,7.92f,5.64f,5.18f,4.43f,-7.39f};
	static float t50g45[NDATA]={9.36f,11.12f,11.11f,11.65f,11.54f,11.68f,
	  12.12f,12.07f,12.09f,12.17f,11.65f,12.04f,12.22f,12.27f,12.16f,12.13f,
	  12.13f,12.08f,11.77f,12.00f,12.03f,12.03f,12.10f,12.07f,12.11f,12.09f,
	  12.10f,12.06f,12.06f,12.04f,12.03f,11.98f,11.98f,11.93f,11.93f,11.86f,
	  11.85f,11.81f,11.78f,11.78f,11.88f,11.70f,11.45f,11.06f,10.66f,10.26f,
	  9.95f,9.71f,9.33f,8.88f,8.40f,7.97f,5.68f,5.22f,4.47f,-7.35f};

#	ifdef DEBUG_FUN
	fputs( "<+>kurucz79()\n", debug_fp );
#	endif

	/* create arrays and zero unused elements */
	rfield.lgContMalloc[rfield.nspec] = TRUE;
	/* these do need to be init to zero */
	if( (rfield.tNuRyd[rfield.nspec] = (float*)CALLOC( (size_t)NCELL , sizeof(float) ) ) == NULL )
		BadMalloc();
	if( (rfield.tslop[rfield.nspec] = (float*)CALLOC( (size_t)NCELL , sizeof(float) ) ) == NULL )
		BadMalloc();
	if( (rfield.tFluxLog[rfield.nspec] = (float*)CALLOC( (size_t)NCELL , sizeof(float) ) ) == NULL )
		BadMalloc();

	if( lgFirst )
	{
		sc30 = t30g4[38];
		sc35 = t35g45[38];
		sc40 = t40g45[38];
		sc45 = t45g45[38];
		sc50 = t50g45[38];

		for( i=0; i < NDATA; i++ )
		{
			t30g4[i] -= sc30;
			t35g45[i] -= sc35;
			t40g45[i] -= sc40;
			t45g45[i] -= sc45;
			t50g45[i] -= sc50;
		}
		lgFirst = FALSE;
	}

	/* PAR1 and 2 are LOG(g) and temp */
	if( par1 > 10. )
	{
		temp = (float)par1;
		mtemp = (long)((par1 + 10.)/100.);
		alogg = par2;
	}
	else
	{
		temp = (float)par2;
		mtemp = (long)((par2 + 10.)/100.);
		alogg = par1;
	}

	if( alogg != 4. )
	{
		fprintf( ioQQQ, " only LOG(g)=4 in table at present.\n" );
		puts( "[Stop in kurucz79]" );
		cdEXIT(EXIT_FAILURE);
	}

	for( i=0; i < NDATA; i++ )
	{
		/* convert nanometers to Rydbergs */
		rfield.tNuRyd[rfield.nspec][NDATA-i-1] = 91.16f/wl[i];
	}

	if( mtemp == 300 )
	{
		for( i=0; i < NDATA; i++ )
		{
			rfield.tslop[rfield.nspec][NDATA-i-1] = t30g4[i];
		}
	}

	else if( mtemp == 350 )
	{
		for( i=0; i < NDATA; i++ )
		{
			rfield.tslop[rfield.nspec][NDATA-i-1] = t35g45[i];
		}
	}

	else if( mtemp == 400 )
	{
		for( i=0; i < NDATA; i++ )
		{
			rfield.tslop[rfield.nspec][NDATA-i-1] = t40g45[i];
		}
	}

	else if( mtemp == 450 )
	{
		for( i=0; i < NDATA; i++ )
		{
			rfield.tslop[rfield.nspec][NDATA-i-1] = t45g45[i];
		}
	}

	else if( mtemp == 500 )
	{
		for( i=0; i < NDATA; i++ )
		{
			rfield.tslop[rfield.nspec][NDATA-i-1] = t50g45[i];
		}
	}

	else if( mtemp < 300 || mtemp > 500 )
	{
		fprintf( ioQQQ, " This temp is not inside Kurucz 79 table.\n" );
		puts( "[Stop in kurucz79]" );
		cdEXIT(EXIT_FAILURE);
	}
	else
	{
		/* must interpolate on grid */
		if( mtemp > 300 && mtemp <= 350 )
		{
			fac1 = (temp - 30000.f)/5000.f;
			fac2 = 1.f - fac1;

			for( i=0; i < NDATA; i++ )
			{
				rfield.tslop[rfield.nspec][NDATA-i-1] = (fac2*
				  t30g4[i] + fac1*t35g45[i]);
			}
		}

		else if( mtemp > 350 && mtemp <= 400 )
		{
			fac1 = (temp - 35000.f)/5000.f;
			fac2 = 1.f - fac1;
			for( i=0; i < NDATA; i++ )
			{
				rfield.tslop[rfield.nspec][NDATA-i-1] = (fac2*
				  t35g45[i] + fac1*t40g45[i]);
			}
		}

		else if( mtemp > 400 && mtemp <= 450 )
		{
			fac1 = (temp - 40000.f)/5000.f;
			fac2 = 1.f - fac1;
			for( i=0; i < NDATA; i++ )
			{
				rfield.tslop[rfield.nspec][NDATA-i-1] = (fac2*
				  t40g45[i] + fac1*t45g45[i]);
			}
		}

		else if( mtemp > 450 && mtemp <= 500 )
		{
			fac1 = (temp - 45000.f)/5000.f;
			fac2 = 1.f - fac1;
			for( i=0; i < NDATA; i++ )
			{
				rfield.tslop[rfield.nspec][NDATA-i-1] = (fac2*
				  t45g45[i] + fac1*t50g45[i]);
			}
		}
	}

	/* now convert to f-nu (above is f-lam) */
	for( i=0; i < NDATA; i++ )
	{
		rfield.tslop[rfield.nspec][i] += 
			(float)(-2.*log10(rfield.tNuRyd[rfield.nspec][i]));
	}

	/* we now have the final product loaded in tnu and tslop, but need to
	 * lower the lower limit to the current low energy limit of the code,
	 * assume all stars are on Rayleigh-Jeans tail of planck function,
	 * so fnu propto nu^2 */

	/* this is the log of the low-energy f_nu */
	rfield.tslop[rfield.nspec][0] += 
		(float)(2.*log10( rfield.emm / rfield.tNuRyd[rfield.nspec][0] ) );

	/* now reset lowest energy to current bounds of code */
	rfield.tNuRyd[rfield.nspec][0] = rfield.emm ;

	*nstar = NDATA;

#	ifdef DEBUG_FUN
	fputs( " <->kurucz79()\n", debug_fp );
#	endif
	return;
}

