/* This file is part of Cloudy and is copyright (C)1978-2006 by Gary J. Ferland
 * For conditions of distribution and use see copyright notice in license.txt */
/*CO_step fills in matrix for heavy elements molecular routines */
#include "cddefines.h"
#include "mole.h"
/* Nick Abel between July and October of 2003 assisted Dr. Ferland in improving the heavy element 
 * molecular network in Cloudy. Before this routine would predict negative abundances if 
 * the fraction of carbon in the form of molecules came close to 100%. A reorganizing of 
 * the reaction network detected several bugs.  Treatment of "coupled reactions",
 * in which both densities in the reaction rate were being predicted by Cloudy, were also 
 * added.  Due to these improvements, Cloudy can now perform calculations
 * where 100% of the carbon is in the form of CO without predicting negative abundances
 *
 * Additional changes were made in November of 2003 so that our reaction 
 * network would include all reactions from the TH85 paper.  This involved 
 * adding silicon to the chemical network.  Also the reaction rates were
 * labeled to make identification with the reaction easier and the matrix 
 * elements of atomic C, O, and Si are now done in a loop, which makes 
 * the addition of future chemical species (like N or S) easy.
 * */
/*lint -e778 constant expression evaluatess to 0 in operation '-' */
/*=================================================================*/


/*CO_step fills in matrix for heavy elements molecular routines */
void CO_step(void )
{
	long int i, j ;

		/* zero out array used for formation rates */
	for ( i=0; i < NUM_COMOLE_CALC; i++ )
	{
		mole.b[i] = 0.;
		for ( j=0; j < NUM_COMOLE_CALC; j++ )
		{
			mole.c[j][i] = 0.;
		}
	}

/* Call all the previous routines that set up the matrix */	
	CO_gasgrain();
	CO_monox();
	CO_ele_recomb();
	CO_photo();
	CO_h_he();
	CO_rates();

/******************************************************************************
*																			  *
*																			  *
*						THE REACTION NETWORK								  *
*																			  *
*******************************************************************************

This part of the code sets of the matrix elements used to predict molecular abundances 
for all "heavy element" molecules.  This means molecules that contain either a carbon, 
oxygen, or a silicon atom.  H2 formation and destruction is not treated here, it is 
done separately in h2.c and hmole.c.

The network of reactions include every reaction in the appendix of the TH85 paper
(Tables 5-14, see references below) although many of the rates come from the later
paper by Hollenbach and McKee (1989--Tables 3,4, and 5 and henceforth referred to 
as HM89).  Other reactions not included in the TH85 paper but presented in other
papers are also part of our reaction network.  These include cosmic ray 
photodissociation reactions and many neutral-neutral reactions given in the HM89 paper.
	
Each term is put in with its reference so that a table of reactions can be put together. 
In order to put a reaction in, one should follow the following format. */


/* >>refer <tab> molecule <tab> reactant1_+_reactant2_=>_product1_product2 <tab> reference 

	
Here the underscore denotes spacing and the <tab> means places where the tab key should
be struck.  In the future more reactions will be inserted, so it is a good idea to 
continue putting reactions into Cloudy in this format so future tables can be made easily  
by running the perl script extract_atomic_data.pl.  In the future, if one wants to insert
a new molecule into the network, three files must be modified

1.  The reactions must be put into this program (co.c)
2.  The new molecule must be defined and the molecule variables updated in co.h
3.  pressuretotal.c must be adjusted to account for the new species

Additionally, if the new molecule has as a constituent an element that is not 
a part of the chemical network, the elements ionization must account for the amount
of that element in molecules.  This is done in the series of files labeled ion(element).c
Examples are ioncarbo.c and ionsilic.c.  The adjusting of the new elements file will
work by exact analogy with the previously mentioned examples.*/

/* Define all the products of two densities that comole tries to update here, these are used in coupled
	* reactions.  The purpose of doing this is to assure that the code always uses the same sequence in multiplying
	* two densities, if this is not done then the code can have problems with precision when 100% CO is reached */

/*****************************************************************************************
*																					     *
*  Certain reactions require special attention because one of						     *
*  the densities that we multiply by in the square matrix A is a density we are also     *
*  trying to solve for.  This non-linearity has to be dealt with in order to converge on *
*  an accurate solution.  We will use a scheme outlined by E. Muller on pg. 421-423 of   *
*  "Computational Methods for Astrophysical Fluid Flows".  For a reaction rate of the    *
*  form:																				 *
*																						 *
*                    Rate = (+/-)k*X(new)Y(new)                                          *
*																						 *
*  where X(new) and Y(new) are the densities that we are solving for we can approximate  *
*  this product by assuming that the difference between the old and new densities are    *
*  small.  Defining this difference as:													 *
*																						 *
*					deltax = X(new) - X(old)											 *													
*					deltay = Y(new) - Y(old)											 *
*																						 *
*  and then keeping terms to first order in delta gives the rate as:					 *
*																						 *
*                  Rate = (+/-)k*[X(new)Y(old) + X(old)Y(new) - X(old)Y(old)]		     *
*																						 *
*  Setting the rate = 0 for a time steady solution then gives:							 *
*																						 *
*				(+/-)k*X(old)Y(old) = (+/-)k*[X(new)Y(old) + X(old)Y(new)]				 *
*																						 *
*  For the matrix equation:																 *
*																						 *
*								Ax = B													 *	
*																						 *
*  The new densities get put into the solution column matrix x, the product of old		 * 
*  density times k gets put into the square matrix A, and the product of k times the	 *
*  old densities gets put intot the column vector B (here labeled as mole.b[i])			 *
*																						 *
*****************************************************************************************/  



/********************************************************************************

								COUPLED REACTIONS

	********************************************************************************/

	/* This is the C2 and C2+ chemistry */


	/* >>refer	mol	C		CH	=>		C2		H		UMIST	*/
	
	mole.c[ipCH][ipATC] -= rate_COmole.C_CH_C2_H_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_CH_C2_H_1;
	mole.b[ipATC]     -= rate_COmole.b_C_CH_C2_H;
	
	mole.c[ipCH][ipCH] -= rate_COmole.C_CH_C2_H_2;
	mole.c[ipATC][ipCH] -= rate_COmole.C_CH_C2_H_1;
	mole.b[ipCH]     -= rate_COmole.b_C_CH_C2_H;

	mole.c[ipCH][ipC2]  += rate_COmole.C_CH_C2_H_2;
	mole.c[ipATC][ipC2]  += rate_COmole.C_CH_C2_H_1;
	mole.b[ipC2]      += rate_COmole.b_C_CH_C2_H;

	/* >>refer	mol	C		CN	=>		C2		N		UMIST	*/
	
	mole.c[ipCN][ipCN] -= rate_COmole.C_CN_C2_N_2;
	mole.c[ipATC][ipCN] -= rate_COmole.C_CN_C2_N_1;
	mole.b[ipCN]     -= rate_COmole.b_C_CN_C2_N;
	
	mole.c[ipCN][ipATC] -= rate_COmole.C_CN_C2_N_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_CN_C2_N_1;
	mole.b[ipATC]     -= rate_COmole.b_C_CN_C2_N;

	mole.c[ipCN][ipC2]  += rate_COmole.C_CN_C2_N_2;
	mole.c[ipATC][ipC2]  += rate_COmole.C_CN_C2_N_1;
	mole.b[ipC2]      += rate_COmole.b_C_CN_C2_N;

	mole.c[ipCN][ipATN]  += rate_COmole.C_CN_C2_N_2;
	mole.c[ipATC][ipATN]  += rate_COmole.C_CN_C2_N_1;
	mole.b[ipATN]      += rate_COmole.b_C_CN_C2_N;

	/* >>refer	mol	C		CS	=>		S		C2		UMIST	*/
	
	mole.c[ipCS][ipCS] -= rate_COmole.C_CS_S_C2_2;
	mole.c[ipATC][ipCS] -= rate_COmole.C_CS_S_C2_1;
	mole.b[ipCS]     -= rate_COmole.b_C_CS_S_C2;
	
	mole.c[ipCS][ipATC] -= rate_COmole.C_CS_S_C2_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_CS_S_C2_1;
	mole.b[ipATC]     -= rate_COmole.b_C_CS_S_C2;

	mole.c[ipCS][ipATS]  += rate_COmole.C_CS_S_C2_2;
	mole.c[ipATC][ipATS]  += rate_COmole.C_CS_S_C2_1;
	mole.b[ipATS]      += rate_COmole.b_C_CS_S_C2;

	mole.c[ipCS][ipC2]  += rate_COmole.C_CS_S_C2_2;
	mole.c[ipATC][ipC2]  += rate_COmole.C_CS_S_C2_1;
	mole.b[ipC2]      += rate_COmole.b_C_CS_S_C2;

	/* >>refer	mol	N		C2	=>		CN		C		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.N_C2_CN_C_2;
	mole.c[ipATN][ipC2] -= rate_COmole.N_C2_CN_C_1;
	mole.b[ipC2]     -= rate_COmole.b_N_C2_CN_C;
	
	mole.c[ipC2][ipATN] -= rate_COmole.N_C2_CN_C_2;
	mole.c[ipATN][ipATN] -= rate_COmole.N_C2_CN_C_1;
	mole.b[ipATN]     -= rate_COmole.b_N_C2_CN_C;

	mole.c[ipC2][ipCN]  += rate_COmole.N_C2_CN_C_2;
	mole.c[ipATN][ipCN]  += rate_COmole.N_C2_CN_C_1;
	mole.b[ipCN]      += rate_COmole.b_N_C2_CN_C;

	mole.c[ipC2][ipATC]  += rate_COmole.N_C2_CN_C_2;
	mole.c[ipATN][ipATC]  += rate_COmole.N_C2_CN_C_1;
	mole.b[ipATC]      += rate_COmole.b_N_C2_CN_C;

	/* >>refer	mol	C2		S	=>		CS		C		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.C2_S_CS_C_1;
	mole.c[ipATS][ipC2] -= rate_COmole.C2_S_CS_C_2;
	mole.b[ipC2]     -= rate_COmole.b_C2_S_CS_C;
	
	mole.c[ipC2][ipATS] -= rate_COmole.C2_S_CS_C_1;
	mole.c[ipATS][ipATS] -= rate_COmole.C2_S_CS_C_2;
	mole.b[ipATS]     -= rate_COmole.b_C2_S_CS_C;

	mole.c[ipC2][ipCS]  += rate_COmole.C2_S_CS_C_1;
	mole.c[ipATS][ipCS]  += rate_COmole.C2_S_CS_C_2;
	mole.b[ipCS]      += rate_COmole.b_C2_S_CS_C;

	mole.c[ipC2][ipATC]  += rate_COmole.C2_S_CS_C_1;
	mole.c[ipATS][ipATC]  += rate_COmole.C2_S_CS_C_2;
	mole.b[ipATC]      += rate_COmole.b_C2_S_CS_C;


	/* >>refer	mol	NH+		C2	=>		HCN+	C		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.NHP_C2_HCNP_C_2;
	mole.c[ipNHP][ipC2] -= rate_COmole.NHP_C2_HCNP_C_1;
	mole.b[ipC2]     -= rate_COmole.b_NHP_C2_HCNP_C;
	
	mole.c[ipC2][ipNHP] -= rate_COmole.NHP_C2_HCNP_C_2;
	mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_C2_HCNP_C_1;
	mole.b[ipNHP]     -= rate_COmole.b_NHP_C2_HCNP_C;

	mole.c[ipC2][ipHCNP]  += rate_COmole.NHP_C2_HCNP_C_2;
	mole.c[ipNHP][ipHCNP]  += rate_COmole.NHP_C2_HCNP_C_1;
	mole.b[ipHCNP]      += rate_COmole.b_NHP_C2_HCNP_C;

	mole.c[ipC2][ipATC]  += rate_COmole.NHP_C2_HCNP_C_2;
	mole.c[ipNHP][ipATC]  += rate_COmole.NHP_C2_HCNP_C_1;
	mole.b[ipATC]      += rate_COmole.b_NHP_C2_HCNP_C;

	/* >>refer	mol	O+		C2	=>		CO+		C		UMIST	*/
	
	mole.c[ipOP][ipOP] -= rate_COmole.OP_C2_COP_C_1;
	mole.c[ipC2][ipOP] -= rate_COmole.OP_C2_COP_C_2;
	mole.b[ipOP]     -= rate_COmole.b_OP_C2_COP_C;
	
	mole.c[ipOP][ipC2] -= rate_COmole.OP_C2_COP_C_1;
	mole.c[ipC2][ipC2] -= rate_COmole.OP_C2_COP_C_2;
	mole.b[ipC2]     -= rate_COmole.b_OP_C2_COP_C;

	mole.c[ipOP][ipCOP]  += rate_COmole.OP_C2_COP_C_1;
	mole.c[ipC2][ipCOP]  += rate_COmole.OP_C2_COP_C_2;
	mole.b[ipCOP]      += rate_COmole.b_OP_C2_COP_C;

	mole.c[ipOP][ipATC]  += rate_COmole.OP_C2_COP_C_1;
	mole.c[ipC2][ipATC]  += rate_COmole.OP_C2_COP_C_2;
	mole.b[ipATC]      += rate_COmole.b_OP_C2_COP_C;


	/*>>chng 05 aug 1, charge transfer between C and S is an important process, one which we
	    did not include.  Now it is added */

	/* >>refer	mol	C+		S	=>		C		S+		UMIST	*/
	
	mole.c[ipATS][ipATS] -= rate_COmole.CP_S_C_SP_1;
	mole.c[ipCP][ipATS] -= rate_COmole.CP_S_C_SP_2;
	mole.b[ipATS]     -= rate_COmole.b_CP_S_C_SP;
	
	mole.c[ipATS][ipCP] -= rate_COmole.CP_S_C_SP_1;
	mole.c[ipCP][ipCP] -= rate_COmole.CP_S_C_SP_2;
	mole.b[ipCP]     -= rate_COmole.b_CP_S_C_SP;

	mole.c[ipATS][ipATC]  += rate_COmole.CP_S_C_SP_1;
	mole.c[ipCP][ipATC]  += rate_COmole.CP_S_C_SP_2;
	mole.b[ipATC]      += rate_COmole.b_CP_S_C_SP;

	mole.c[ipATS][ipSP]  += rate_COmole.CP_S_C_SP_1;
	mole.c[ipCP][ipSP]  += rate_COmole.CP_S_C_SP_2;
	mole.b[ipSP]      += rate_COmole.b_CP_S_C_SP;

	/* >>refer	mol	C2		S+	=>		CS+		C		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.C2_SP_CSP_C_1;
	mole.c[ipSP][ipC2] -= rate_COmole.C2_SP_CSP_C_2;
	mole.b[ipC2]     -= rate_COmole.b_C2_SP_CSP_C;
	
	mole.c[ipC2][ipSP] -= rate_COmole.C2_SP_CSP_C_1;
	mole.c[ipSP][ipSP] -= rate_COmole.C2_SP_CSP_C_2;
	mole.b[ipSP]     -= rate_COmole.b_C2_SP_CSP_C;

	mole.c[ipC2][ipCSP]  += rate_COmole.C2_SP_CSP_C_1;
	mole.c[ipSP][ipCSP]  += rate_COmole.C2_SP_CSP_C_2;
	mole.b[ipCSP]      += rate_COmole.b_C2_SP_CSP_C;

	mole.c[ipC2][ipATC]  += rate_COmole.C2_SP_CSP_C_1;
	mole.c[ipSP][ipATC]  += rate_COmole.C2_SP_CSP_C_2;
	mole.b[ipATC]      += rate_COmole.b_C2_SP_CSP_C;


	/* >>refer	mol	C		C2+	=>		C2		C+		UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.C_C2P_C2_CP_2;
	mole.c[ipATC][ipC2P] -= rate_COmole.C_C2P_C2_CP_1;
	mole.b[ipC2P]     -= rate_COmole.b_C_C2P_C2_CP;
	
	mole.c[ipC2P][ipATC] -= rate_COmole.C_C2P_C2_CP_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_C2P_C2_CP_1;
	mole.b[ipATC]     -= rate_COmole.b_C_C2P_C2_CP;

	mole.c[ipC2P][ipC2]  += rate_COmole.C_C2P_C2_CP_2;
	mole.c[ipATC][ipC2]  += rate_COmole.C_C2P_C2_CP_1;
	mole.b[ipC2]      += rate_COmole.b_C_C2P_C2_CP;

	mole.c[ipC2P][ipCP]  += rate_COmole.C_C2P_C2_CP_2;
	mole.c[ipATC][ipCP]  += rate_COmole.C_C2P_C2_CP_1;
	mole.b[ipCP]      += rate_COmole.b_C_C2P_C2_CP;

	/* >>refer	mol	CH		C2+	=>		C2		CH+		UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.CH_C2P_C2_CHP_2;
	mole.c[ipCH][ipC2P] -= rate_COmole.CH_C2P_C2_CHP_1;
	mole.b[ipC2P]     -= rate_COmole.b_CH_C2P_C2_CHP;
	
	mole.c[ipC2P][ipCH] -= rate_COmole.CH_C2P_C2_CHP_2;
	mole.c[ipCH][ipCH] -= rate_COmole.CH_C2P_C2_CHP_1;
	mole.b[ipCH]     -= rate_COmole.b_CH_C2P_C2_CHP;

	mole.c[ipC2P][ipC2]  += rate_COmole.CH_C2P_C2_CHP_2;
	mole.c[ipCH][ipC2]  += rate_COmole.CH_C2P_C2_CHP_1;
	mole.b[ipC2]      += rate_COmole.b_CH_C2P_C2_CHP;

	mole.c[ipC2P][ipCHP]  += rate_COmole.CH_C2P_C2_CHP_2;
	mole.c[ipCH][ipCHP]  += rate_COmole.CH_C2P_C2_CHP_1;
	mole.b[ipCHP]      += rate_COmole.b_CH_C2P_C2_CHP;

	/* >>refer	mol	N+		C2	=>		C2+		N		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.NP_C2_C2P_N_2;
	mole.c[ipNP][ipC2] -= rate_COmole.NP_C2_C2P_N_1;
	mole.b[ipC2]     -= rate_COmole.b_NP_C2_C2P_N;
	
	mole.c[ipC2][ipNP] -= rate_COmole.NP_C2_C2P_N_2;
	mole.c[ipNP][ipNP] -= rate_COmole.NP_C2_C2P_N_1;
	mole.b[ipNP]     -= rate_COmole.b_NP_C2_C2P_N;

	mole.c[ipC2][ipC2P]  += rate_COmole.NP_C2_C2P_N_2;
	mole.c[ipNP][ipC2P]  += rate_COmole.NP_C2_C2P_N_1;
	mole.b[ipC2P]      += rate_COmole.b_NP_C2_C2P_N;

	mole.c[ipC2][ipATN]  += rate_COmole.NP_C2_C2P_N_2;
	mole.c[ipNP][ipATN]  += rate_COmole.NP_C2_C2P_N_1;
	mole.b[ipATN]      += rate_COmole.b_NP_C2_C2P_N;

	/* >>refer	mol	CH2		C2+	=>		C2		CH2+	UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.CH2_C2P_C2_CH2P_2;
	mole.c[ipCH2][ipC2P] -= rate_COmole.CH2_C2P_C2_CH2P_1;
	mole.b[ipC2P]     -= rate_COmole.b_CH2_C2P_C2_CH2P;
	
	mole.c[ipC2P][ipCH2] -= rate_COmole.CH2_C2P_C2_CH2P_2;
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_C2P_C2_CH2P_1;
	mole.b[ipCH2]     -= rate_COmole.b_CH2_C2P_C2_CH2P;

	mole.c[ipC2P][ipC2]  += rate_COmole.CH2_C2P_C2_CH2P_2;
	mole.c[ipCH2][ipC2]  += rate_COmole.CH2_C2P_C2_CH2P_1;
	mole.b[ipC2]      += rate_COmole.b_CH2_C2P_C2_CH2P;

	mole.c[ipC2P][ipCH2P]  += rate_COmole.CH2_C2P_C2_CH2P_2;
	mole.c[ipCH2][ipCH2P]  += rate_COmole.CH2_C2P_C2_CH2P_1;
	mole.b[ipCH2P]      += rate_COmole.b_CH2_C2P_C2_CH2P;

	/* >>refer	mol	O+		C2	=>		C2+		O		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.OP_C2_C2P_O_2;
	mole.c[ipOP][ipC2] -= rate_COmole.OP_C2_C2P_O_1;
	mole.b[ipC2]     -= rate_COmole.b_OP_C2_C2P_O;
	
	mole.c[ipC2][ipOP] -= rate_COmole.OP_C2_C2P_O_2;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_C2_C2P_O_1;
	mole.b[ipOP]     -= rate_COmole.b_OP_C2_C2P_O;

	mole.c[ipC2][ipC2P]  += rate_COmole.OP_C2_C2P_O_2;
	mole.c[ipOP][ipC2P]  += rate_COmole.OP_C2_C2P_O_1;
	mole.b[ipC2P]      += rate_COmole.b_OP_C2_C2P_O;

	mole.c[ipC2][ipATO]  += rate_COmole.OP_C2_C2P_O_2;
	mole.c[ipOP][ipATO]  += rate_COmole.OP_C2_C2P_O_1;
	mole.b[ipATO]      += rate_COmole.b_OP_C2_C2P_O;

	/* >>refer	mol	NH2		C2+	=>		C2		NH2+	UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.NH2_C2P_C2_NH2P_2;
	mole.c[ipNH2][ipC2P] -= rate_COmole.NH2_C2P_C2_NH2P_1;
	mole.b[ipC2P]     -= rate_COmole.b_NH2_C2P_C2_NH2P;
	
	mole.c[ipC2P][ipNH2] -= rate_COmole.NH2_C2P_C2_NH2P_2;
	mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_C2P_C2_NH2P_1;
	mole.b[ipNH2]     -= rate_COmole.b_NH2_C2P_C2_NH2P;

	mole.c[ipC2P][ipC2]  += rate_COmole.NH2_C2P_C2_NH2P_2;
	mole.c[ipNH2][ipC2]  += rate_COmole.NH2_C2P_C2_NH2P_1;
	mole.b[ipC2]      += rate_COmole.b_NH2_C2P_C2_NH2P;

	mole.c[ipC2P][ipNH2P]  += rate_COmole.NH2_C2P_C2_NH2P_2;
	mole.c[ipNH2][ipNH2P]  += rate_COmole.NH2_C2P_C2_NH2P_1;
	mole.b[ipNH2P]      += rate_COmole.b_NH2_C2P_C2_NH2P;

	/* >>refer	mol	OH+		C2	=>		C2+		OH		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.OHP_C2_C2P_OH_2;
	mole.c[ipOHP][ipC2] -= rate_COmole.OHP_C2_C2P_OH_1;
	mole.b[ipC2]     -= rate_COmole.b_OHP_C2_C2P_OH;
	
	mole.c[ipC2][ipOHP] -= rate_COmole.OHP_C2_C2P_OH_2;
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_C2_C2P_OH_1;
	mole.b[ipOHP]     -= rate_COmole.b_OHP_C2_C2P_OH;

	mole.c[ipC2][ipC2P]  += rate_COmole.OHP_C2_C2P_OH_2;
	mole.c[ipOHP][ipC2P]  += rate_COmole.OHP_C2_C2P_OH_1;
	mole.b[ipC2P]      += rate_COmole.b_OHP_C2_C2P_OH;

	mole.c[ipC2][ipOH]  += rate_COmole.OHP_C2_C2P_OH_2;
	mole.c[ipOHP][ipOH]  += rate_COmole.OHP_C2_C2P_OH_1;
	mole.b[ipOH]      += rate_COmole.b_OHP_C2_C2P_OH;

	/* >>refer	mol	OH		C2+	=>		C2		OH+		UMIST	*/
	
	mole.c[ipOH][ipOH] -= rate_COmole.OH_C2P_C2_OHP_1;
	mole.c[ipC2P][ipOH] -= rate_COmole.OH_C2P_C2_OHP_2;
	mole.b[ipOH]     -= rate_COmole.b_OH_C2P_C2_OHP;
	
	mole.c[ipOH][ipC2P] -= rate_COmole.OH_C2P_C2_OHP_1;
	mole.c[ipC2P][ipC2P] -= rate_COmole.OH_C2P_C2_OHP_2;
	mole.b[ipC2P]     -= rate_COmole.b_OH_C2P_C2_OHP;

	mole.c[ipOH][ipC2]  += rate_COmole.OH_C2P_C2_OHP_1;
	mole.c[ipC2P][ipC2]  += rate_COmole.OH_C2P_C2_OHP_2;
	mole.b[ipC2]      += rate_COmole.b_OH_C2P_C2_OHP;

	mole.c[ipOH][ipOHP]  += rate_COmole.OH_C2P_C2_OHP_1;
	mole.c[ipC2P][ipOHP]  += rate_COmole.OH_C2P_C2_OHP_2;
	mole.b[ipOHP]      += rate_COmole.b_OH_C2P_C2_OHP;

	/* >>refer	mol	H2O+	C2	=>		C2+		H2O		UMIST	*/
	
	mole.c[ipC2][ipC2] -= rate_COmole.H2OP_C2_C2P_H2O_2;
	mole.c[ipH2OP][ipC2] -= rate_COmole.H2OP_C2_C2P_H2O_1;
	mole.b[ipC2]     -= rate_COmole.b_H2OP_C2_C2P_H2O;
	
	mole.c[ipC2][ipH2OP] -= rate_COmole.H2OP_C2_C2P_H2O_2;
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_C2_C2P_H2O_1;
	mole.b[ipH2OP]     -= rate_COmole.b_H2OP_C2_C2P_H2O;

	mole.c[ipC2][ipC2P]  += rate_COmole.H2OP_C2_C2P_H2O_2;
	mole.c[ipH2OP][ipC2P]  += rate_COmole.H2OP_C2_C2P_H2O_1;
	mole.b[ipC2P]      += rate_COmole.b_H2OP_C2_C2P_H2O;

	mole.c[ipC2][ipH2O]  += rate_COmole.H2OP_C2_C2P_H2O_2;
	mole.c[ipH2OP][ipH2O]  += rate_COmole.H2OP_C2_C2P_H2O_1;
	mole.b[ipH2O]      += rate_COmole.b_H2OP_C2_C2P_H2O;

	/* >>refer	mol	C2		CN+	=>		CN		C2+		UMIST	*/
	
	mole.c[ipCNP][ipCNP] -= rate_COmole.C2_CNP_CN_C2P_2;
	mole.c[ipC2][ipCNP] -= rate_COmole.C2_CNP_CN_C2P_1;
	mole.b[ipCNP]     -= rate_COmole.b_C2_CNP_CN_C2P;
	
	mole.c[ipCNP][ipC2] -= rate_COmole.C2_CNP_CN_C2P_2;
	mole.c[ipC2][ipC2] -= rate_COmole.C2_CNP_CN_C2P_1;
	mole.b[ipC2]     -= rate_COmole.b_C2_CNP_CN_C2P;

	mole.c[ipCNP][ipCN]  += rate_COmole.C2_CNP_CN_C2P_2;
	mole.c[ipC2][ipCN]  += rate_COmole.C2_CNP_CN_C2P_1;
	mole.b[ipCN]      += rate_COmole.b_C2_CNP_CN_C2P;

	mole.c[ipCNP][ipC2P]  += rate_COmole.C2_CNP_CN_C2P_2;
	mole.c[ipC2][ipC2P]  += rate_COmole.C2_CNP_CN_C2P_1;
	mole.b[ipC2P]      += rate_COmole.b_C2_CNP_CN_C2P;

	/* >>refer	mol	C2		N2+	=>		N2		C2+		UMIST	*/
	
	mole.c[ipN2P][ipN2P] -= rate_COmole.C2_N2P_N2_C2P_2;
	mole.c[ipC2][ipN2P] -= rate_COmole.C2_N2P_N2_C2P_1;
	mole.b[ipN2P]     -= rate_COmole.b_C2_N2P_N2_C2P;
	
	mole.c[ipN2P][ipC2] -= rate_COmole.C2_N2P_N2_C2P_2;
	mole.c[ipC2][ipC2] -= rate_COmole.C2_N2P_N2_C2P_1;
	mole.b[ipC2]     -= rate_COmole.b_C2_N2P_N2_C2P;

	mole.c[ipN2P][ipN2]  += rate_COmole.C2_N2P_N2_C2P_2;
	mole.c[ipC2][ipN2]  += rate_COmole.C2_N2P_N2_C2P_1;
	mole.b[ipN2]      += rate_COmole.b_C2_N2P_N2_C2P;

	mole.c[ipN2P][ipC2P]  += rate_COmole.C2_N2P_N2_C2P_2;
	mole.c[ipC2][ipC2P]  += rate_COmole.C2_N2P_N2_C2P_1;
	mole.b[ipC2P]      += rate_COmole.b_C2_N2P_N2_C2P;

	/* >>refer	mol	C2+		NO	=>		NO+		C2		UMIST	*/
	
	mole.c[ipNO][ipNO] -= rate_COmole.C2P_NO_NOP_C2_2;
	mole.c[ipC2P][ipNO] -= rate_COmole.C2P_NO_NOP_C2_1;
	mole.b[ipNO]     -= rate_COmole.b_C2P_NO_NOP_C2;
	
	mole.c[ipNO][ipC2P] -= rate_COmole.C2P_NO_NOP_C2_2;
	mole.c[ipC2P][ipC2P] -= rate_COmole.C2P_NO_NOP_C2_1;
	mole.b[ipC2P]     -= rate_COmole.b_C2P_NO_NOP_C2;

	mole.c[ipNO][ipNOP]  += rate_COmole.C2P_NO_NOP_C2_2;
	mole.c[ipC2P][ipNOP]  += rate_COmole.C2P_NO_NOP_C2_1;
	mole.b[ipNOP]      += rate_COmole.b_C2P_NO_NOP_C2;

	mole.c[ipNO][ipC2]  += rate_COmole.C2P_NO_NOP_C2_2;
	mole.c[ipC2P][ipC2]  += rate_COmole.C2P_NO_NOP_C2_1;
	mole.b[ipC2]      += rate_COmole.b_C2P_NO_NOP_C2;

	/* >>refer	mol	C2		O2+	=>		O2		C2+		UMIST	*/
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.C2_O2P_O2_C2P_2;
	mole.c[ipC2][ipO2P] -= rate_COmole.C2_O2P_O2_C2P_1;
	mole.b[ipO2P]     -= rate_COmole.b_C2_O2P_O2_C2P;
	
	mole.c[ipO2P][ipC2] -= rate_COmole.C2_O2P_O2_C2P_2;
	mole.c[ipC2][ipC2] -= rate_COmole.C2_O2P_O2_C2P_1;
	mole.b[ipC2]     -= rate_COmole.b_C2_O2P_O2_C2P;

	mole.c[ipO2P][ipO2]  += rate_COmole.C2_O2P_O2_C2P_2;
	mole.c[ipC2][ipO2]  += rate_COmole.C2_O2P_O2_C2P_1;
	mole.b[ipO2]      += rate_COmole.b_C2_O2P_O2_C2P;

	mole.c[ipO2P][ipC2P]  += rate_COmole.C2_O2P_O2_C2P_2;
	mole.c[ipC2][ipC2P]  += rate_COmole.C2_O2P_O2_C2P_1;
	mole.b[ipC2P]      += rate_COmole.b_C2_O2P_O2_C2P;

	/* >>refer	mol	C2+		S	=>		S+		C2		UMIST	*/
	
	mole.c[ipATS][ipATS] -= rate_COmole.C2P_S_SP_C2_2;
	mole.c[ipC2P][ipATS] -= rate_COmole.C2P_S_SP_C2_1;
	mole.b[ipATS]     -= rate_COmole.b_C2P_S_SP_C2;
	
	mole.c[ipATS][ipC2P] -= rate_COmole.C2P_S_SP_C2_2;
	mole.c[ipC2P][ipC2P] -= rate_COmole.C2P_S_SP_C2_1;
	mole.b[ipC2P]     -= rate_COmole.b_C2P_S_SP_C2;

	mole.c[ipATS][ipSP]  += rate_COmole.C2P_S_SP_C2_2;
	mole.c[ipC2P][ipSP]  += rate_COmole.C2P_S_SP_C2_1;
	mole.b[ipSP]      += rate_COmole.b_C2P_S_SP_C2;

	mole.c[ipATS][ipC2]  += rate_COmole.C2P_S_SP_C2_2;
	mole.c[ipC2P][ipC2]  += rate_COmole.C2P_S_SP_C2_1;
	mole.b[ipC2]      += rate_COmole.b_C2P_S_SP_C2;

	/* >>refer	mol	C		C	=>		C2		nu		UMIST	*/
	
	mole.c[ipATC][ipATC] -= rate_COmole.C_C_C2_nu_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_C_C2_nu_1;
	mole.b[ipATC]     -= rate_COmole.b_C_C_C2_nu;
	
	mole.c[ipATC][ipC2]  += 0.5*rate_COmole.C_C_C2_nu_2;
	mole.c[ipATC][ipC2]  += 0.5*rate_COmole.C_C_C2_nu_1;
	mole.b[ipC2]      += 0.5*rate_COmole.b_C_C_C2_nu;


	/* >>refer	mol	C+		CH	=>		C2+		H		UMIST	*/
	
	mole.c[ipCH][ipCH] -= rate_COmole.CP_CH_C2P_H_2;
	mole.c[ipCP][ipCH] -= rate_COmole.CP_CH_C2P_H_1;
	mole.b[ipCH]     -= rate_COmole.b_CP_CH_C2P_H;
	
	mole.c[ipCH][ipCP] -= rate_COmole.CP_CH_C2P_H_2;
	mole.c[ipCP][ipCP] -= rate_COmole.CP_CH_C2P_H_1;
	mole.b[ipCP]     -= rate_COmole.b_CP_CH_C2P_H;

	mole.c[ipCH][ipC2P]  += rate_COmole.CP_CH_C2P_H_2;
	mole.c[ipCP][ipC2P]  += rate_COmole.CP_CH_C2P_H_1;
	mole.b[ipC2P]      += rate_COmole.b_CP_CH_C2P_H;

	/* >>refer	mol	C		CH+	=>		C2+		H		UMIST	*/
	
	mole.c[ipCHP][ipCHP] -= rate_COmole.C_CHP_C2P_H_2;
	mole.c[ipATC][ipCHP] -= rate_COmole.C_CHP_C2P_H_1;
	mole.b[ipCHP]     -= rate_COmole.b_C_CHP_C2P_H;
	
	mole.c[ipCHP][ipATC] -= rate_COmole.C_CHP_C2P_H_2;
	mole.c[ipATC][ipATC] -= rate_COmole.C_CHP_C2P_H_1;
	mole.b[ipATC]     -= rate_COmole.b_C_CHP_C2P_H;

	mole.c[ipCHP][ipC2P]  += rate_COmole.C_CHP_C2P_H_2;
	mole.c[ipATC][ipC2P]  += rate_COmole.C_CHP_C2P_H_1;
	mole.b[ipC2P]      += rate_COmole.b_C_CHP_C2P_H;

	/* >>refer	mol	CH+		CH	=>		C2+		H2		UMIST	*/
	
	mole.c[ipCH][ipCH] -= rate_COmole.CHP_CH_C2P_H2_2;
	mole.c[ipCHP][ipCH] -= rate_COmole.CHP_CH_C2P_H2_1;
	mole.b[ipCH]     -= rate_COmole.b_CHP_CH_C2P_H2;
	
	mole.c[ipCH][ipCHP] -= rate_COmole.CHP_CH_C2P_H2_2;
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_CH_C2P_H2_1;
	mole.b[ipCHP]     -= rate_COmole.b_CHP_CH_C2P_H2;

	mole.c[ipCH][ipC2P]  += rate_COmole.CHP_CH_C2P_H2_2;
	mole.c[ipCHP][ipC2P]  += rate_COmole.CHP_CH_C2P_H2_1;
	mole.b[ipC2P]      += rate_COmole.b_CHP_CH_C2P_H2;

	/* >>refer	mol	N		C2+	=>		CN		C+		UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.N_C2P_CN_CP_2;
	mole.c[ipATN][ipC2P] -= rate_COmole.N_C2P_CN_CP_1;
	mole.b[ipC2P]     -= rate_COmole.b_N_C2P_CN_CP;
	
	mole.c[ipC2P][ipATN] -= rate_COmole.N_C2P_CN_CP_2;
	mole.c[ipATN][ipATN] -= rate_COmole.N_C2P_CN_CP_1;
	mole.b[ipATN]     -= rate_COmole.b_N_C2P_CN_CP;

	mole.c[ipC2P][ipCN]  += rate_COmole.N_C2P_CN_CP_2;
	mole.c[ipATN][ipCN]  += rate_COmole.N_C2P_CN_CP_1;
	mole.b[ipCN]      += rate_COmole.b_N_C2P_CN_CP;

	mole.c[ipC2P][ipCP]  += rate_COmole.N_C2P_CN_CP_2;
	mole.c[ipATN][ipCP]  += rate_COmole.N_C2P_CN_CP_1;
	mole.b[ipCP]      += rate_COmole.b_N_C2P_CN_CP;

	/* >>refer	mol	O		C2+	=>		CO+		C		UMIST	*/
	
	mole.c[ipC2P][ipC2P] -= rate_COmole.O_C2P_COP_C_2;
	mole.c[ipATO][ipC2P] -= rate_COmole.O_C2P_COP_C_1;
	mole.b[ipC2P]     -= rate_COmole.b_O_C2P_COP_C;
	
	mole.c[ipC2P][ipATO] -= rate_COmole.O_C2P_COP_C_2;
	mole.c[ipATC][ipATO] -= rate_COmole.O_C2P_COP_C_1;
	mole.b[ipATO]     -= rate_COmole.b_O_C2P_COP_C;

	mole.c[ipC2P][ipCOP]  += rate_COmole.O_C2P_COP_C_2;
	mole.c[ipATO][ipCOP]  += rate_COmole.O_C2P_COP_C_1;
	mole.b[ipCOP]      += rate_COmole.b_O_C2P_COP_C;

	mole.c[ipC2P][ipATC]  += rate_COmole.O_C2P_COP_C_2;
	mole.c[ipATO][ipATC]  += rate_COmole.O_C2P_COP_C_1;
	mole.b[ipATC]      += rate_COmole.b_O_C2P_COP_C;

	/* >>refer	mol	C2+		S	=>		CS+		C		UMIST	*/
	
	mole.c[ipATS][ipATS] -= rate_COmole.C2P_S_CSP_C_2;
	mole.c[ipC2P][ipATS] -= rate_COmole.C2P_S_CSP_C_1;
	mole.b[ipATS]     -= rate_COmole.b_C2P_S_CSP_C;
	
	mole.c[ipATS][ipC2P] -= rate_COmole.C2P_S_CSP_C_2;
	mole.c[ipC2P][ipC2P] -= rate_COmole.C2P_S_CSP_C_1;
	mole.b[ipC2P]     -= rate_COmole.b_C2P_S_CSP_C;

	mole.c[ipATS][ipCS]  += rate_COmole.C2P_S_CSP_C_2;
	mole.c[ipC2P][ipCS]  += rate_COmole.C2P_S_CSP_C_1;
	mole.b[ipCS]      += rate_COmole.b_C2P_S_CSP_C;

	mole.c[ipATS][ipATC]  += rate_COmole.C2P_S_CSP_C_2;
	mole.c[ipC2P][ipATC]  += rate_COmole.C2P_S_CSP_C_1;
	mole.b[ipATC]      += rate_COmole.b_C2P_S_CSP_C;


	/* >>refer	mol	C+		C	=>		C2+		nu		UMIST	*/
	
	mole.c[ipATC][ipATC] -= rate_COmole.CP_C_C2P_nu_2;
	mole.c[ipCP][ipATC] -= rate_COmole.CP_C_C2P_nu_1;
	mole.b[ipATC]     -= rate_COmole.b_CP_C_C2P_nu;
	
	mole.c[ipATC][ipCP] -= rate_COmole.CP_C_C2P_nu_2;
	mole.c[ipCP][ipCP] -= rate_COmole.CP_C_C2P_nu_1;
	mole.b[ipCP]     -= rate_COmole.b_CP_C_C2P_nu;

	mole.c[ipATC][ipC2P]  += rate_COmole.CP_C_C2P_nu_2;
	mole.c[ipCP][ipC2P]  += rate_COmole.CP_C_C2P_nu_1;
	mole.b[ipC2P]      += rate_COmole.b_CP_C_C2P_nu;

	/* end of C2 and C2+ chemistry */
	
	/* >>refer	mol	C	CH2	=>	CH	CH	UMIST	*/
			
	mole.c[ipCH2][ipCH2] -= rate_COmole.C_CH2_CH_CH_1;
	mole.c[ipATC][ipCH2] -= rate_COmole.C_CH2_CH_CH_2;
	mole.b[ipCH2]     -= rate_COmole.b_C_CH2_CH_CH;
			
	mole.c[ipCH2][ipATC] -= rate_COmole.C_CH2_CH_CH_1;
	mole.c[ipATC][ipATC] -= rate_COmole.C_CH2_CH_CH_2;
	mole.b[ipATC]     -= rate_COmole.b_C_CH2_CH_CH;

	mole.c[ipCH2][ipCH]  += 2*rate_COmole.C_CH2_CH_CH_1;
	mole.c[ipATC][ipCH]  += 2*rate_COmole.C_CH2_CH_CH_2;
	mole.b[ipCH]      += 2*rate_COmole.b_C_CH2_CH_CH;

	/* >>refer	mol	C	H2O+	=>	OH	CH+	UMIST	*/
		
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.C_H2OP_OH_CHP_1;
	mole.c[ipATC][ipH2OP] -= rate_COmole.C_H2OP_OH_CHP_2;
	mole.b[ipH2OP]     -= rate_COmole.b_C_H2OP_OH_CHP;
		
	mole.c[ipH2OP][ipATC] -= rate_COmole.C_H2OP_OH_CHP_1;
	mole.c[ipATC][ipATC] -= rate_COmole.C_H2OP_OH_CHP_2;
	mole.b[ipATC]     -= rate_COmole.b_C_H2OP_OH_CHP;
	
	mole.c[ipH2OP][ipOH]  += rate_COmole.C_H2OP_OH_CHP_1;
	mole.c[ipATC][ipOH]  += rate_COmole.C_H2OP_OH_CHP_2;
	mole.b[ipOH]      += rate_COmole.b_C_H2OP_OH_CHP;
	
	mole.c[ipH2OP][ipCHP] += rate_COmole.C_H2OP_OH_CHP_1;
	mole.c[ipATC][ipCHP] += rate_COmole.C_H2OP_OH_CHP_2;
	mole.b[ipCHP]     += rate_COmole.b_C_H2OP_OH_CHP;
	
	/* >>refer	mol	C	H3O+	=>	HCO+	H2	UMIST	*/
		
	mole.c[ipH3OP][ipH3OP] -= rate_COmole.C_H3OP_HCOP_H2_1;
	mole.c[ipATC][ipH3OP]  -= rate_COmole.C_H3OP_HCOP_H2_2;
	mole.b[ipH3OP]      -= rate_COmole.b_C_H3OP_HCOP_H2;
		
	mole.c[ipH3OP][ipATC] -= rate_COmole.C_H3OP_HCOP_H2_1;
	mole.c[ipATC][ipATC]  -= rate_COmole.C_H3OP_HCOP_H2_2;
	mole.b[ipATC]      -= rate_COmole.b_C_H3OP_HCOP_H2;
	
	mole.c[ipH3OP][ipHCOP] += rate_COmole.C_H3OP_HCOP_H2_1;
	mole.c[ipATC][ipHCOP]  += rate_COmole.C_H3OP_HCOP_H2_2;
	mole.b[ipHCOP]      += rate_COmole.b_C_H3OP_HCOP_H2;
	
	
	/* >>refer	mol	C	O2+	=>	O2	C+	UMIST	*/
		
	mole.c[ipO2P][ipO2P] -= rate_COmole.C_O2P_O2_CP_1;
	mole.c[ipATC][ipO2P] -= rate_COmole.C_O2P_O2_CP_2;
	mole.b[ipO2P]     -= rate_COmole.b_C_O2P_O2_CP;
		
	mole.c[ipO2P][ipATC] -= rate_COmole.C_O2P_O2_CP_1;
	mole.c[ipATC][ipATC] -= rate_COmole.C_O2P_O2_CP_2;
	mole.b[ipATC]     -= rate_COmole.b_C_O2P_O2_CP;
	
	mole.c[ipO2P][ipO2]  += rate_COmole.C_O2P_O2_CP_1;
	mole.c[ipATC][ipO2]  += rate_COmole.C_O2P_O2_CP_2;
	mole.b[ipO2]      += rate_COmole.b_C_O2P_O2_CP;
	
	mole.c[ipO2P][ipCP]  += rate_COmole.C_O2P_O2_CP_1;
	mole.c[ipATC][ipCP]  += rate_COmole.C_O2P_O2_CP_2;
	mole.b[ipCP]      += rate_COmole.b_C_O2P_O2_CP;
	
	/* >>refer	mol	C	O2+	=>	CO+	O	UMIST	*/
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.C_O2P_COP_O_1;
	mole.c[ipATC][ipO2P] -= rate_COmole.C_O2P_COP_O_2;
	mole.b[ipO2P]     -= rate_COmole.b_C_O2P_COP_O;
	
	mole.c[ipO2P][ipATC] -= rate_COmole.C_O2P_COP_O_1;
	mole.c[ipATC][ipATC] -= rate_COmole.C_O2P_COP_O_2;
	mole.b[ipATC]     -= rate_COmole.b_C_O2P_COP_O;
	
	mole.c[ipO2P][ipCOP] += rate_COmole.C_O2P_COP_O_1;
	mole.c[ipATC][ipCOP] += rate_COmole.C_O2P_COP_O_2;
	mole.b[ipCOP]     += rate_COmole.b_C_O2P_COP_O;
	
	mole.c[ipO2P][ipATO] += rate_COmole.C_O2P_COP_O_1;
	mole.c[ipATC][ipATO] += rate_COmole.C_O2P_COP_O_2;
	mole.b[ipATO]     += rate_COmole.b_C_O2P_COP_O;
	
	/* >>refer	mol	C	OH	=>	O	CH	UMIST	*/
	
	mole.c[ipOH][ipOH]   -= rate_COmole.C_OH_O_CH_1;
	mole.c[ipATC][ipOH]  -= rate_COmole.C_OH_O_CH_2;
	mole.b[ipOH]      -= rate_COmole.b_C_OH_O_CH;
	
	mole.c[ipOH][ipATC]   -= rate_COmole.C_OH_O_CH_1;
	mole.c[ipATC][ipATC]  -= rate_COmole.C_OH_O_CH_2;
	mole.b[ipATC]      -= rate_COmole.b_C_OH_O_CH;
	
	mole.c[ipOH][ipCH]   += rate_COmole.C_OH_O_CH_1;
	mole.c[ipATC][ipCH]  += rate_COmole.C_OH_O_CH_2;
	mole.b[ipCH]      += rate_COmole.b_C_OH_O_CH;
	
	mole.c[ipOH][ipATO]   += rate_COmole.C_OH_O_CH_1;
	mole.c[ipATC][ipATO]  += rate_COmole.C_OH_O_CH_2;
	mole.b[ipATO]      += rate_COmole.b_C_OH_O_CH;
	
	/* >>refer	mol	C	OH+	=>	O	CH+	UMIST	*/
		
	mole.c[ipOHP][ipOHP] -= rate_COmole.C_OHP_O_CHP_1;
	mole.c[ipATC][ipOHP] -= rate_COmole.C_OHP_O_CHP_2;
	mole.b[ipOHP]     -= rate_COmole.b_C_OHP_O_CHP;
		
	mole.c[ipOHP][ipATC] -= rate_COmole.C_OHP_O_CHP_1;
	mole.c[ipATC][ipATC] -= rate_COmole.C_OHP_O_CHP_2;
	mole.b[ipATC]     -= rate_COmole.b_C_OHP_O_CHP;
	
	mole.c[ipOHP][ipCHP] += rate_COmole.C_OHP_O_CHP_1;
	mole.c[ipATC][ipCHP] += rate_COmole.C_OHP_O_CHP_2;
	mole.b[ipCHP]     += rate_COmole.b_C_OHP_O_CHP;
	
	mole.c[ipOHP][ipATO] += rate_COmole.C_OHP_O_CHP_1;
	mole.c[ipATC][ipATO] += rate_COmole.C_OHP_O_CHP_2;
	mole.b[ipATO]     += rate_COmole.b_C_OHP_O_CHP;
	
	/* >>refer	mol	C+	CH	=>	CH+	C	UMIST	*/
		
	mole.c[ipCH][ipCP]  -= rate_COmole.CP_CH_CHP_C_1;
	mole.c[ipCP][ipCP]  -= rate_COmole.CP_CH_CHP_C_2;
	mole.b[ipCP]     -= rate_COmole.b_CP_CH_CHP_C;
		
	mole.c[ipCH][ipCH]  -= rate_COmole.CP_CH_CHP_C_1;
	mole.c[ipCP][ipCH]  -= rate_COmole.CP_CH_CHP_C_2;
	mole.b[ipCH]     -= rate_COmole.b_CP_CH_CHP_C;
	
	mole.c[ipCH][ipCHP] += rate_COmole.CP_CH_CHP_C_1;
	mole.c[ipCP][ipCHP] += rate_COmole.CP_CH_CHP_C_2;
	mole.b[ipCHP]    += rate_COmole.b_CP_CH_CHP_C;
	
	mole.c[ipCH][ipATC] += rate_COmole.CP_CH_CHP_C_1;
	mole.c[ipCP][ipATC] += rate_COmole.CP_CH_CHP_C_2;
	mole.b[ipATC]    += rate_COmole.b_CP_CH_CHP_C;

	/* >>refer	mol	C+	CH2	=>	CH2+	C	UMIST	*/
		
	mole.c[ipCH2][ipCP]  -= rate_COmole.CP_CH2_CH2P_C_1;
	mole.c[ipCP][ipCP]   -= rate_COmole.CP_CH2_CH2P_C_2;
	mole.b[ipCP]      -= rate_COmole.b_CP_CH2_CH2P_C;
		
	mole.c[ipCH2][ipCH2]  -= rate_COmole.CP_CH2_CH2P_C_1;
	mole.c[ipCP][ipCH2]   -= rate_COmole.CP_CH2_CH2P_C_2;
	mole.b[ipCH2]      -= rate_COmole.b_CP_CH2_CH2P_C;
	
	mole.c[ipCH2][ipCH2P] += rate_COmole.CP_CH2_CH2P_C_1;
	mole.c[ipCP][ipCH2P]  += rate_COmole.CP_CH2_CH2P_C_2;
	mole.b[ipCH2P]     += rate_COmole.b_CP_CH2_CH2P_C;
	
	mole.c[ipCH2][ipATC] += rate_COmole.CP_CH2_CH2P_C_1;
	mole.c[ipCP][ipATC]  += rate_COmole.CP_CH2_CH2P_C_2;
	mole.b[ipATC]     += rate_COmole.b_CP_CH2_CH2P_C;
	
	/* >>refer	mol	C+	H2O	=>	HCO+	H	UMIST	*/
		
	mole.c[ipH2O][ipCP]  -= rate_COmole.CP_H2O_HCOP_H_1;
	mole.c[ipCP][ipCP]   -= rate_COmole.CP_H2O_HCOP_H_2;
	mole.b[ipCP]      -= rate_COmole.b_CP_H2O_HCOP_H;
		
	mole.c[ipH2O][ipH2O]  -= rate_COmole.CP_H2O_HCOP_H_1;
	mole.c[ipCP][ipH2O]   -= rate_COmole.CP_H2O_HCOP_H_2;
	mole.b[ipH2O]      -= rate_COmole.b_CP_H2O_HCOP_H;
	
	mole.c[ipH2O][ipHCOP] += rate_COmole.CP_H2O_HCOP_H_1;
	mole.c[ipCP][ipHCOP]  += rate_COmole.CP_H2O_HCOP_H_2;
	mole.b[ipHCOP]     += rate_COmole.b_CP_H2O_HCOP_H;
	
	/* >>refer	mol	C+	O	=>	CO+	PHOTON	UMIST	*/
		
	mole.c[ipATO][ipATO] -= rate_COmole.CP_O_COP_nu_1;
	mole.c[ipCP][ipATO]  -= rate_COmole.CP_O_COP_nu_2;
	mole.b[ipATO]     -= rate_COmole.b_CP_O_COP_nu;
		
	mole.c[ipATO][ipCP] -= rate_COmole.CP_O_COP_nu_1;
	mole.c[ipCP][ipCP]  -= rate_COmole.CP_O_COP_nu_2;
	mole.b[ipCP]     -= rate_COmole.b_CP_O_COP_nu;
		
	mole.c[ipATO][ipCOP] += rate_COmole.CP_O_COP_nu_1;
	mole.c[ipCP][ipCOP]  += rate_COmole.CP_O_COP_nu_2;
	mole.b[ipCOP]     += rate_COmole.b_CP_O_COP_nu;
	
	/* >>refer	mol	C+	O2	=>	CO+	O	UMIST	*/
	
	mole.c[ipO2][ipCP]  -= rate_COmole.CP_O2_COP_O_1;
	mole.c[ipCP][ipCP]  -= rate_COmole.CP_O2_COP_O_2;
	mole.b[ipCP]     -= rate_COmole.b_CP_O2_COP_O;
	
	mole.c[ipO2][ipO2]  -= rate_COmole.CP_O2_COP_O_1;
	mole.c[ipCP][ipO2]  -= rate_COmole.CP_O2_COP_O_2;
	mole.b[ipO2]     -= rate_COmole.b_CP_O2_COP_O;
	
	mole.c[ipO2][ipCOP] += rate_COmole.CP_O2_COP_O_1;
	mole.c[ipCP][ipCOP] += rate_COmole.CP_O2_COP_O_2;
	mole.b[ipCOP]    += rate_COmole.b_CP_O2_COP_O;
	
	mole.c[ipO2][ipATO] += rate_COmole.CP_O2_COP_O_1;
	mole.c[ipCP][ipATO] += rate_COmole.CP_O2_COP_O_2;
	mole.b[ipATO]    += rate_COmole.b_CP_O2_COP_O;
	
	/* >>refer	mol	C+	OH	=>	CO+	H	UMIST	*/
		
	mole.c[ipOH][ipCP]  -= rate_COmole.CP_OH_COP_H_1;
	mole.c[ipCP][ipCP]  -= rate_COmole.CP_OH_COP_H_2;
	mole.b[ipCP]     -= rate_COmole.b_CP_OH_COP_H;
		
	mole.c[ipOH][ipOH]  -= rate_COmole.CP_OH_COP_H_1;
	mole.c[ipCP][ipOH]  -= rate_COmole.CP_OH_COP_H_2;
	mole.b[ipOH]     -= rate_COmole.b_CP_OH_COP_H;
	
	mole.c[ipOH][ipCOP] += rate_COmole.CP_OH_COP_H_1;
	mole.c[ipCP][ipCOP] += rate_COmole.CP_OH_COP_H_2;
	mole.b[ipCOP]    += rate_COmole.b_CP_OH_COP_H;
	
	/* >>refer	mol	O	CH	=>	OH	C	UMIST	*/
		
	mole.c[ipCH][ipCH]  -= rate_COmole.O_CH_OH_C_1;
	mole.c[ipATO][ipCH] -= rate_COmole.O_CH_OH_C_2;
	mole.b[ipCH]     -= rate_COmole.b_O_CH_OH_C;
		
	mole.c[ipCH][ipATO]  -= rate_COmole.O_CH_OH_C_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_CH_OH_C_2;
	mole.b[ipATO]     -= rate_COmole.b_O_CH_OH_C;
	
	mole.c[ipCH][ipOH]  += rate_COmole.O_CH_OH_C_1;
	mole.c[ipATO][ipOH] += rate_COmole.O_CH_OH_C_2;
	mole.b[ipOH]     += rate_COmole.b_O_CH_OH_C;
	
	mole.c[ipCH][ipATC]  += rate_COmole.O_CH_OH_C_1;
	mole.c[ipATO][ipATC] += rate_COmole.O_CH_OH_C_2;
	mole.b[ipATC]     += rate_COmole.b_O_CH_OH_C;
	
	/* >>refer	mol	O	CH	=>	HCO+	e	UMIST	*/
		
	mole.c[ipCH][ipCH]    -= rate_COmole.O_CH_HCOP_e_1;
	mole.c[ipATO][ipCH]   -= rate_COmole.O_CH_HCOP_e_2;
	mole.b[ipCH]       -= rate_COmole.b_O_CH_HCOP_e;
		
	mole.c[ipCH][ipATO]    -= rate_COmole.O_CH_HCOP_e_1;
	mole.c[ipATO][ipATO]   -= rate_COmole.O_CH_HCOP_e_2;
	mole.b[ipATO]       -= rate_COmole.b_O_CH_HCOP_e;
	
	mole.c[ipCH][ipHCOP]  += rate_COmole.O_CH_HCOP_e_1;
	mole.c[ipATO][ipHCOP] += rate_COmole.O_CH_HCOP_e_2;
	mole.b[ipHCOP]     += rate_COmole.b_O_CH_HCOP_e;
	
	/* >>refer	mol	O	CH+	=>	CO+	H	UMIST	*/
		
	mole.c[ipCHP][ipCHP] -= rate_COmole.O_CHP_COP_H_1;
	mole.c[ipATO][ipCHP] -= rate_COmole.O_CHP_COP_H_2;
	mole.b[ipCHP]     -= rate_COmole.b_O_CHP_COP_H;
		
	mole.c[ipCHP][ipATO] -= rate_COmole.O_CHP_COP_H_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_CHP_COP_H_2;
	mole.b[ipATO]     -= rate_COmole.b_O_CHP_COP_H;
	
	mole.c[ipCHP][ipCOP] += rate_COmole.O_CHP_COP_H_1;
	mole.c[ipATO][ipCOP] += rate_COmole.O_CHP_COP_H_2;
	mole.b[ipCOP]     += rate_COmole.b_O_CHP_COP_H;
	
	/* >>refer	mol	O	CH2	=>	OH	CH	UMIST	*/
	
	mole.c[ipCH2][ipCH2] -= rate_COmole.O_CH2_OH_CH_1;
	mole.c[ipATO][ipCH2] -= rate_COmole.O_CH2_OH_CH_2;
	mole.b[ipCH2]     -= rate_COmole.b_O_CH2_OH_CH;
	
	mole.c[ipCH2][ipATO] -= rate_COmole.O_CH2_OH_CH_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_CH2_OH_CH_2;
	mole.b[ipATO]     -= rate_COmole.b_O_CH2_OH_CH;
	
	mole.c[ipCH2][ipOH]  += rate_COmole.O_CH2_OH_CH_1;
	mole.c[ipATO][ipOH]  += rate_COmole.O_CH2_OH_CH_2;
	mole.b[ipOH]      += rate_COmole.b_O_CH2_OH_CH;
	
	mole.c[ipCH2][ipCH]  += rate_COmole.O_CH2_OH_CH_1;
	mole.c[ipATO][ipCH]  += rate_COmole.O_CH2_OH_CH_2;
	mole.b[ipCH]      += rate_COmole.b_O_CH2_OH_CH;
	
	
	/* >>refer	mol	O	CH2+	=>	HCO+	H	UMIST	*/
	
	mole.c[ipCH2P][ipCH2P] -= rate_COmole.O_CH2P_HCOP_H_1;
	mole.c[ipATO][ipCH2P]  -= rate_COmole.O_CH2P_HCOP_H_2;
	mole.b[ipCH2P]      -= rate_COmole.b_O_CH2P_HCOP_H;
	
	mole.c[ipCH2P][ipATO] -= rate_COmole.O_CH2P_HCOP_H_1;
	mole.c[ipATO][ipATO]  -= rate_COmole.O_CH2P_HCOP_H_2;
	mole.b[ipATO]      -= rate_COmole.b_O_CH2P_HCOP_H;
	
	mole.c[ipCH2P][ipHCOP] += rate_COmole.O_CH2P_HCOP_H_1;
	mole.c[ipATO][ipHCOP]  += rate_COmole.O_CH2P_HCOP_H_2;
	mole.b[ipHCOP]      += rate_COmole.b_O_CH2P_HCOP_H;
	
	/* >>refer	mol	O	CH3+	=>	HCO+	H2	UMIST	*/
		
	mole.c[ipCH3P][ipCH3P] -= rate_COmole.O_CH3P_HCOP_H2_1;
	mole.c[ipATO][ipCH3P]  -= rate_COmole.O_CH3P_HCOP_H2_2;
	mole.b[ipCH3P]      -= rate_COmole.b_O_CH3P_HCOP_H2;
		
	mole.c[ipCH3P][ipATO] -= rate_COmole.O_CH3P_HCOP_H2_1;
	mole.c[ipATO][ipATO]  -= rate_COmole.O_CH3P_HCOP_H2_2;
	mole.b[ipATO]      -= rate_COmole.b_O_CH3P_HCOP_H2;
	
	mole.c[ipCH3P][ipHCOP] += rate_COmole.O_CH3P_HCOP_H2_1;
	mole.c[ipATO][ipHCOP]  += rate_COmole.O_CH3P_HCOP_H2_2;
	mole.b[ipHCOP]      += rate_COmole.b_O_CH3P_HCOP_H2;
	/* >>refer	mol	O	H2O	=>	OH	OH	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= rate_COmole.O_H2O_OH_OH_1;
	mole.c[ipATO][ipH2O] -= rate_COmole.O_H2O_OH_OH_2;
	mole.b[ipH2O]     -= rate_COmole.b_O_H2O_OH_OH;
		
	mole.c[ipH2O][ipATO] -= rate_COmole.O_H2O_OH_OH_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_H2O_OH_OH_2;
	mole.b[ipATO]     -= rate_COmole.b_O_H2O_OH_OH;
	
	mole.c[ipH2O][ipOH]  += 2*rate_COmole.O_H2O_OH_OH_1;
	mole.c[ipATO][ipOH]  += 2*rate_COmole.O_H2O_OH_OH_2;
	mole.b[ipOH]      += 2*rate_COmole.b_O_H2O_OH_OH;
	
	/* >>refer	mol	O	H2O+	=>	O2+	H2	UMIST	*/
		
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.O_H2OP_O2P_H2_1;
	mole.c[ipATO][ipH2OP]  -= rate_COmole.O_H2OP_O2P_H2_2;
	mole.b[ipH2OP]      -= rate_COmole.b_O_H2OP_O2P_H2;
		
	mole.c[ipH2OP][ipATO] -= rate_COmole.O_H2OP_O2P_H2_1;
	mole.c[ipATO][ipATO]  -= rate_COmole.O_H2OP_O2P_H2_2;
	mole.b[ipATO]      -= rate_COmole.b_O_H2OP_O2P_H2;
	
	mole.c[ipH2OP][ipO2P]  += rate_COmole.O_H2OP_O2P_H2_1;
	mole.c[ipATO][ipO2P]   += rate_COmole.O_H2OP_O2P_H2_2;
	mole.b[ipO2P]       += rate_COmole.b_O_H2OP_O2P_H2;
	
	/* >>refer	mol	O	O	=>	O2	PHOTON	UMIST	*/
	
	mole.c[ipATO][ipO2] += 0.5*rate_COmole.O_O_O2_nu_1;
	mole.c[ipATO][ipO2] += 0.5*rate_COmole.O_O_O2_nu_2;
	mole.b[ipO2]     += 0.5*rate_COmole.b_O_O_O2_nu;
	
	mole.c[ipATO][ipATO] -= rate_COmole.O_O_O2_nu_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_O_O2_nu_2;
	mole.b[ipATO]     -= rate_COmole.b_O_O_O2_nu;
	
	/* >>refer	mol	O	OH	=>	O2	H	UMIST	*/
		
	mole.c[ipOH][ipOH]    -= rate_COmole.O_OH_O2_H_1;
	mole.c[ipATO][ipOH]   -= rate_COmole.O_OH_O2_H_2;
	mole.b[ipOH]       -= rate_COmole.b_O_OH_O2_H;
		
	mole.c[ipOH][ipATO]    -= rate_COmole.O_OH_O2_H_1;
	mole.c[ipATO][ipATO]   -= rate_COmole.O_OH_O2_H_2;
	mole.b[ipATO]       -= rate_COmole.b_O_OH_O2_H;
	
	mole.c[ipOH][ipO2]    += rate_COmole.O_OH_O2_H_1;
	mole.c[ipATO][ipO2]   += rate_COmole.O_OH_O2_H_2;
	mole.b[ipO2]       += rate_COmole.b_O_OH_O2_H;
	
	/* >>refer	mol	O	OH+	=>	O2+	H	UMIST	*/
		
	mole.c[ipOHP][ipOHP] -= rate_COmole.O_OHP_O2P_H_1;
	mole.c[ipATO][ipOHP] -= rate_COmole.O_OHP_O2P_H_2;
	mole.b[ipOHP]     -= rate_COmole.b_O_OHP_O2P_H;
		
	mole.c[ipOHP][ipATO] -= rate_COmole.O_OHP_O2P_H_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_OHP_O2P_H_2;
	mole.b[ipATO]     -= rate_COmole.b_O_OHP_O2P_H;
	
	mole.c[ipOHP][ipO2P] += rate_COmole.O_OHP_O2P_H_1;
	mole.c[ipATO][ipO2P] += rate_COmole.O_OHP_O2P_H_2;
	mole.b[ipO2P]     += rate_COmole.b_O_OHP_O2P_H;
	
	/* >>refer	mol	O	Si	=>	SiO	PHOTON	UMIST	*/
		
	mole.c[ipATSI][ipATO] -= rate_COmole.O_Si_SiO_nu_1;
	mole.c[ipATO][ipATO]   -= rate_COmole.O_Si_SiO_nu_2;
	mole.b[ipATO]      -= rate_COmole.b_O_Si_SiO_nu;
		
	mole.c[ipATSI][ipATSI] -= rate_COmole.O_Si_SiO_nu_1;
	mole.c[ipATO][ipATSI]   -= rate_COmole.O_Si_SiO_nu_2;
	mole.b[ipATSI]      -= rate_COmole.b_O_Si_SiO_nu;
		
	mole.c[ipATSI][ipSIO] += rate_COmole.O_Si_SiO_nu_1;
	mole.c[ipATO][ipSIO]   += rate_COmole.O_Si_SiO_nu_2;
	mole.b[ipSIO]      += rate_COmole.b_O_Si_SiO_nu;
	
	/* >>refer	mol	O	Si+	=>	SiO+	PHOTON	UMIST	*/
			
	mole.c[ipSIP][ipSIP] -= rate_COmole.O_SiP_SiOP_nu_1;
	mole.c[ipATO][ipSIP] -= rate_COmole.O_SiP_SiOP_nu_2;
	mole.b[ipSIP]     -= rate_COmole.b_O_SiP_SiOP_nu;
			
	mole.c[ipSIP][ipATO] -= rate_COmole.O_SiP_SiOP_nu_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_SiP_SiOP_nu_2;
	mole.b[ipATO]     -= rate_COmole.b_O_SiP_SiOP_nu;
			
	mole.c[ipSIP][ipSIOP] += rate_COmole.O_SiP_SiOP_nu_1;
	mole.c[ipATO][ipSIOP] += rate_COmole.O_SiP_SiOP_nu_2;
	mole.b[ipSIOP]     += rate_COmole.b_O_SiP_SiOP_nu;
	
	/* >>refer	mol	O	SiH	=>	SiO	H	UMIST	*/
		
	mole.c[ipSIH][ipSIH] -= rate_COmole.O_SiH_SiO_H_1;
	mole.c[ipATO][ipSIH] -= rate_COmole.O_SiH_SiO_H_2;
	mole.b[ipSIH]     -= rate_COmole.b_O_SiH_SiO_H;
		
	mole.c[ipSIH][ipATO] -= rate_COmole.O_SiH_SiO_H_1;
	mole.c[ipATO][ipATO] -= rate_COmole.O_SiH_SiO_H_2;
	mole.b[ipATO]     -= rate_COmole.b_O_SiH_SiO_H;
	
	mole.c[ipSIH][ipSIO] += rate_COmole.O_SiH_SiO_H_1;
	mole.c[ipATO][ipSIO] += rate_COmole.O_SiH_SiO_H_2;
	mole.b[ipSIO]     += rate_COmole.b_O_SiH_SiO_H;
	
	/* >>refer	mol	O	SiH2+	=>	SiOH+	H	UMIST	*/
			
	mole.c[ipSIH2P][ipSIH2P] -= rate_COmole.O_SiH2P_SiOHP_H_1;
	mole.c[ipATO][ipSIH2P]   -= rate_COmole.O_SiH2P_SiOHP_H_2;
	mole.b[ipSIH2P]       -= rate_COmole.b_O_SiH2P_SiOHP_H;
			
	mole.c[ipSIH2P][ipATO] -= rate_COmole.O_SiH2P_SiOHP_H_1;
	mole.c[ipATO][ipATO]   -= rate_COmole.O_SiH2P_SiOHP_H_2;
	mole.b[ipATO]       -= rate_COmole.b_O_SiH2P_SiOHP_H;
	
	mole.c[ipSIH2P][ipHOSIP] += rate_COmole.O_SiH2P_SiOHP_H_1;
	mole.c[ipATO][ipHOSIP]   += rate_COmole.O_SiH2P_SiOHP_H_2;
	mole.b[ipHOSIP]       += rate_COmole.b_O_SiH2P_SiOHP_H;
	
	/* >>refer	mol	O	SiO+	=>	O2	Si+	UMIST	*/
			
	mole.c[ipSIOP][ipSIOP] -= rate_COmole.O_SiOP_O2_SiP_1;
	mole.c[ipATO][ipSIOP]  -= rate_COmole.O_SiOP_O2_SiP_2;
	mole.b[ipSIOP]      -= rate_COmole.b_O_SiOP_O2_SiP;
			
	mole.c[ipSIOP][ipATO] -= rate_COmole.O_SiOP_O2_SiP_1;
	mole.c[ipATO][ipATO]  -= rate_COmole.O_SiOP_O2_SiP_2;
	mole.b[ipATO]      -= rate_COmole.b_O_SiOP_O2_SiP;
	
	mole.c[ipSIOP][ipO2]   += rate_COmole.O_SiOP_O2_SiP_1;
	mole.c[ipATO][ipO2]    += rate_COmole.O_SiOP_O2_SiP_2;
	mole.b[ipO2]        += rate_COmole.b_O_SiOP_O2_SiP;
	
	mole.c[ipSIOP][ipSIP]   += rate_COmole.O_SiOP_O2_SiP_1;
	mole.c[ipATO][ipSIP]    += rate_COmole.O_SiOP_O2_SiP_2;
	mole.b[ipSIP]        += rate_COmole.b_O_SiOP_O2_SiP;
	
	/* >>refer	mol	O+	CH	=>	O	CH+	UMIST	*/
		
	
	mole.c[ipCH][ipOP] -= rate_COmole.OP_CH_O_CHP_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_CH_O_CHP_2;
	mole.b[ipOP] -= rate_COmole.b_OP_CH_O_CHP;
	
	mole.c[ipCH][ipCH] -= rate_COmole.OP_CH_O_CHP_1;
	mole.c[ipOP][ipCH] -= rate_COmole.OP_CH_O_CHP_2;
	mole.b[ipCH] -= rate_COmole.b_OP_CH_O_CHP;
	
	mole.c[ipCH][ipCHP] += rate_COmole.OP_CH_O_CHP_1;
	mole.c[ipOP][ipCHP] += rate_COmole.OP_CH_O_CHP_2;
	mole.b[ipCHP] += rate_COmole.b_OP_CH_O_CHP;
	
	mole.c[ipCH][ipATO] += rate_COmole.OP_CH_O_CHP_1;
	mole.c[ipOP][ipATO] += rate_COmole.OP_CH_O_CHP_2;
	mole.b[ipATO] += rate_COmole.b_OP_CH_O_CHP;
	
	/* >>refer	mol	O+	CH	=>	CO+	H	UMIST	*/
		
	
	mole.c[ipCH][ipOP] -= rate_COmole.OP_CH_COP_H_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_CH_COP_H_2;
	mole.b[ipOP] -= rate_COmole.b_OP_CH_COP_H;
	
	mole.c[ipCH][ipCH] -= rate_COmole.OP_CH_COP_H_1;
	mole.c[ipOP][ipCH] -= rate_COmole.OP_CH_COP_H_2;
	mole.b[ipCH] -= rate_COmole.b_OP_CH_COP_H;
	
	mole.c[ipCH][ipCOP] += rate_COmole.OP_CH_COP_H_1;
	mole.c[ipOP][ipCOP] += rate_COmole.OP_CH_COP_H_2;
	mole.b[ipCOP] += rate_COmole.b_OP_CH_COP_H;
	
	/* >>refer	mol	O+	CH2	=>	O	CH2+	UMIST	*/
		
	
	mole.c[ipCH2][ipOP] -= rate_COmole.OP_CH2_O_CH2P_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_CH2_O_CH2P_2;
	mole.b[ipOP] -= rate_COmole.b_OP_CH2_O_CH2P;

	mole.c[ipCH2][ipCH2] -= rate_COmole.OP_CH2_O_CH2P_1;
	mole.c[ipOP][ipCH2] -= rate_COmole.OP_CH2_O_CH2P_2;
	mole.b[ipCH2] -= rate_COmole.b_OP_CH2_O_CH2P;
	
	mole.c[ipCH2][ipCH2P] += rate_COmole.OP_CH2_O_CH2P_1;
	mole.c[ipOP][ipCH2P] += rate_COmole.OP_CH2_O_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_OP_CH2_O_CH2P;
	
	mole.c[ipCH2][ipATO] += rate_COmole.OP_CH2_O_CH2P_1;
	mole.c[ipOP][ipATO] += rate_COmole.OP_CH2_O_CH2P_2;
	mole.b[ipATO] += rate_COmole.b_OP_CH2_O_CH2P;
	
	/* >>refer	mol	O+	H2O	=>	H2O+	O	UMIST	*/
		
	
	mole.c[ipH2O][ipOP] -= rate_COmole.OP_H2O_H2OP_O_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_H2O_H2OP_O_2;
	mole.b[ipOP] -= rate_COmole.b_OP_H2O_H2OP_O;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.OP_H2O_H2OP_O_1;
	mole.c[ipOP][ipH2O] -= rate_COmole.OP_H2O_H2OP_O_2;
	mole.b[ipH2O] -= rate_COmole.b_OP_H2O_H2OP_O;
	
	mole.c[ipH2O][ipH2OP] += rate_COmole.OP_H2O_H2OP_O_1;
	mole.c[ipOP][ipH2OP] += rate_COmole.OP_H2O_H2OP_O_2;
	mole.b[ipH2OP] += rate_COmole.b_OP_H2O_H2OP_O;
	
	mole.c[ipH2O][ipATO] += rate_COmole.OP_H2O_H2OP_O_1;
	mole.c[ipOP][ipATO] += rate_COmole.OP_H2O_H2OP_O_2;
	mole.b[ipATO] += rate_COmole.b_OP_H2O_H2OP_O;
	
	/* >>refer	mol	O+	O2	=>	O2+	O	UMIST	*/
		
	
	mole.c[ipO2][ipOP] -= rate_COmole.OP_O2_O2P_O_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_O2_O2P_O_2;
	mole.b[ipOP] -= rate_COmole.b_OP_O2_O2P_O;
	
	mole.c[ipO2][ipO2] -= rate_COmole.OP_O2_O2P_O_1;
	mole.c[ipOP][ipO2] -= rate_COmole.OP_O2_O2P_O_2;
	mole.b[ipO2] -= rate_COmole.b_OP_O2_O2P_O;
	
	mole.c[ipO2][ipO2P] += rate_COmole.OP_O2_O2P_O_1;
	mole.c[ipOP][ipO2P] += rate_COmole.OP_O2_O2P_O_2;
	mole.b[ipO2P] += rate_COmole.b_OP_O2_O2P_O;
	
	mole.c[ipO2][ipATO] += rate_COmole.OP_O2_O2P_O_1;
	mole.c[ipOP][ipATO] += rate_COmole.OP_O2_O2P_O_2;
	mole.b[ipATO] += rate_COmole.b_OP_O2_O2P_O;
	
	/* >>refer	mol	O+	OH	=>	O2+	H	UMIST	*/
		
	
	mole.c[ipOH][ipOP] -= rate_COmole.OP_OH_O2P_H_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_OH_O2P_H_2;
	mole.b[ipOP] -= rate_COmole.b_OP_OH_O2P_H;
	
	mole.c[ipOH][ipOH] -= rate_COmole.OP_OH_O2P_H_1;
	mole.c[ipOP][ipOH] -= rate_COmole.OP_OH_O2P_H_2;
	mole.b[ipOH] -= rate_COmole.b_OP_OH_O2P_H;
	
	
	mole.c[ipOH][ipO2P] += rate_COmole.OP_OH_O2P_H_1;
	mole.c[ipOP][ipO2P] += rate_COmole.OP_OH_O2P_H_2;
	mole.b[ipO2P] += rate_COmole.b_OP_OH_O2P_H;
	
	/* >>refer	mol	O+	OH	=>	OH+	O	UMIST	*/
		
	
	mole.c[ipOH][ipOP] -= rate_COmole.OP_OH_OHP_O_1;
	mole.c[ipOP][ipOP] -= rate_COmole.OP_OH_OHP_O_2;
	mole.b[ipOP] -= rate_COmole.b_OP_OH_OHP_O;
	
	mole.c[ipOH][ipOH] -= rate_COmole.OP_OH_OHP_O_1;
	mole.c[ipOP][ipOH] -= rate_COmole.OP_OH_OHP_O_2;
	mole.b[ipOH] -= rate_COmole.b_OP_OH_OHP_O;
	
	mole.c[ipOH][ipOHP] += rate_COmole.OP_OH_OHP_O_1;
	mole.c[ipOP][ipOHP] += rate_COmole.OP_OH_OHP_O_2;
	mole.b[ipOHP] += rate_COmole.b_OP_OH_OHP_O;
	
	mole.c[ipOH][ipATO] += rate_COmole.OP_OH_OHP_O_1;
	mole.c[ipOP][ipATO] += rate_COmole.OP_OH_OHP_O_2;
	mole.b[ipATO] += rate_COmole.b_OP_OH_OHP_O;
	
	/* >>refer	mol	Si	CH+	=>	Si+	CH	UMIST	*/
		
	mole.c[ipCHP][ipCHP] -= rate_COmole.Si_CHP_SiP_CH_1;
	mole.c[ipATSI][ipCHP] -= rate_COmole.Si_CHP_SiP_CH_2;
	mole.b[ipCHP] -= rate_COmole.b_Si_CHP_SiP_CH;
		
	mole.c[ipCHP][ipATSI] -= rate_COmole.Si_CHP_SiP_CH_1;
	mole.c[ipATSI][ipATSI] -= rate_COmole.Si_CHP_SiP_CH_2;
	mole.b[ipATSI] -= rate_COmole.b_Si_CHP_SiP_CH;
	
	mole.c[ipCHP][ipCH] += rate_COmole.Si_CHP_SiP_CH_1;
	mole.c[ipATSI][ipCH] += rate_COmole.Si_CHP_SiP_CH_2;
	mole.b[ipCH] += rate_COmole.b_Si_CHP_SiP_CH;
	
	mole.c[ipCHP][ipSIP] += rate_COmole.Si_CHP_SiP_CH_1;
	mole.c[ipATSI][ipSIP] += rate_COmole.Si_CHP_SiP_CH_2;
	mole.b[ipSIP] += rate_COmole.b_Si_CHP_SiP_CH;
	
	/* >>refer	mol	Si	H2O+	=>	Si+	H2O	UMIST	*/
			
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.Si_H2OP_SiP_H2O_1;
	mole.c[ipATSI][ipH2OP] -= rate_COmole.Si_H2OP_SiP_H2O_2;
	mole.b[ipH2OP] -= rate_COmole.b_Si_H2OP_SiP_H2O;
			
	mole.c[ipH2OP][ipATSI] -= rate_COmole.Si_H2OP_SiP_H2O_1;
	mole.c[ipATSI][ipATSI] -= rate_COmole.Si_H2OP_SiP_H2O_2;
	mole.b[ipATSI] -= rate_COmole.b_Si_H2OP_SiP_H2O;
	
	mole.c[ipH2OP][ipSIP] += rate_COmole.Si_H2OP_SiP_H2O_1;
	mole.c[ipATSI][ipSIP] += rate_COmole.Si_H2OP_SiP_H2O_2;
	mole.b[ipSIP] += rate_COmole.b_Si_H2OP_SiP_H2O;

	mole.c[ipH2OP][ipH2O] += rate_COmole.Si_H2OP_SiP_H2O_1;
	mole.c[ipATSI][ipH2O] += rate_COmole.Si_H2OP_SiP_H2O_2;
	mole.b[ipH2O] += rate_COmole.b_Si_H2OP_SiP_H2O;
	
	/* >>refer	mol	Si	OH	=>	SiO	H	UMIST	*/
		
	mole.c[ipOH][ipOH] -= rate_COmole.Si_OH_SiO_H_1;
	mole.c[ipATSI][ipOH] -= rate_COmole.Si_OH_SiO_H_2;
	mole.b[ipOH] -= rate_COmole.b_Si_OH_SiO_H;
		
	mole.c[ipOH][ipATSI] -= rate_COmole.Si_OH_SiO_H_1;
	mole.c[ipATSI][ipATSI] -= rate_COmole.Si_OH_SiO_H_2;
	mole.b[ipATSI] -= rate_COmole.b_Si_OH_SiO_H;
	
	mole.c[ipOH][ipSIO] += rate_COmole.Si_OH_SiO_H_1;
	mole.c[ipATSI][ipSIO] += rate_COmole.Si_OH_SiO_H_2;
	mole.b[ipSIO] += rate_COmole.b_Si_OH_SiO_H;
	
	/* >>refer	mol	Si	O2+	=>	O2	Si+	UMIST	*/
		
	mole.c[ipO2P][ipO2P] -= rate_COmole.Si_O2P_O2_SiP_1;
	mole.c[ipATSI][ipO2P] -= rate_COmole.Si_O2P_O2_SiP_2;
	mole.b[ipO2P] -= rate_COmole.b_Si_O2P_O2_SiP;
			
	mole.c[ipO2P][ipATSI] -= rate_COmole.Si_O2P_O2_SiP_1;
	mole.c[ipATSI][ipATSI] -= rate_COmole.Si_O2P_O2_SiP_2;
	mole.b[ipATSI] -= rate_COmole.b_Si_O2P_O2_SiP;
	
	mole.c[ipO2P][ipO2] += rate_COmole.Si_O2P_O2_SiP_1;
	mole.c[ipATSI][ipO2] += rate_COmole.Si_O2P_O2_SiP_2;
	mole.b[ipO2] += rate_COmole.b_Si_O2P_O2_SiP;
	
	mole.c[ipO2P][ipSIP] += rate_COmole.Si_O2P_O2_SiP_1;
	mole.c[ipATSI][ipSIP] += rate_COmole.Si_O2P_O2_SiP_2;
	mole.b[ipSIP] += rate_COmole.b_Si_O2P_O2_SiP;
	
	/* >>refer	mol	Si+	H2O	=>	SiOH+	H	UMIST	*/
		
	mole.c[ipH2O][ipSIP] -= rate_COmole.SiP_H2O_SiOHP_H_1;
	mole.c[ipSIP][ipSIP] -= rate_COmole.SiP_H2O_SiOHP_H_2;
	mole.b[ipSIP] -= rate_COmole.b_SiP_H2O_SiOHP_H;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.SiP_H2O_SiOHP_H_1;
	mole.c[ipSIP][ipH2O] -= rate_COmole.SiP_H2O_SiOHP_H_2;
	mole.b[ipH2O] -= rate_COmole.b_SiP_H2O_SiOHP_H;
	
	mole.c[ipH2O][ipHOSIP] += rate_COmole.SiP_H2O_SiOHP_H_1;
	mole.c[ipSIP][ipHOSIP] += rate_COmole.SiP_H2O_SiOHP_H_2;
	mole.b[ipHOSIP] += rate_COmole.b_SiP_H2O_SiOHP_H;
	
	/* >>refer	mol	Si+	OH	=>	SiO+	H	UMIST	*/
		
	mole.c[ipOH][ipSIP] -= rate_COmole.SiP_OH_SiOP_H_1;
	mole.c[ipSIP][ipSIP] -= rate_COmole.SiP_OH_SiOP_H_2;
	mole.b[ipSIP] -= rate_COmole.b_SiP_OH_SiOP_H;
	
	mole.c[ipOH][ipOH] -= rate_COmole.SiP_OH_SiOP_H_1;
	mole.c[ipSIP][ipOH] -= rate_COmole.SiP_OH_SiOP_H_2;
	mole.b[ipOH] -= rate_COmole.b_SiP_OH_SiOP_H;
	
	mole.c[ipOH][ipSIOP] += rate_COmole.SiP_OH_SiOP_H_1;
	mole.c[ipSIP][ipSIOP] += rate_COmole.SiP_OH_SiOP_H_2;
	mole.b[ipSIOP] += rate_COmole.b_SiP_OH_SiOP_H;
	
	/* >>refer	mol	Si+	O2	=>	SiO+	O	UMIST	*/
			
	mole.c[ipO2][ipSIP] -= rate_COmole.SiP_O2_SiOP_O_1; 
	mole.c[ipSIP][ipSIP] -= rate_COmole.SiP_O2_SiOP_O_2;
	mole.b[ipSIP] -= rate_COmole.b_SiP_O2_SiOP_O;
	
	mole.c[ipO2][ipO2] -= rate_COmole.SiP_O2_SiOP_O_1; 
	mole.c[ipSIP][ipO2] -= rate_COmole.SiP_O2_SiOP_O_2;
	mole.b[ipO2] -= rate_COmole.b_SiP_O2_SiOP_O;
	
	mole.c[ipO2][ipSIOP] += rate_COmole.SiP_O2_SiOP_O_1; 
	mole.c[ipSIP][ipSIOP] += rate_COmole.SiP_O2_SiOP_O_2;
	mole.b[ipSIOP] += rate_COmole.b_SiP_O2_SiOP_O;
		
	mole.c[ipO2][ipATO] += rate_COmole.SiP_O2_SiOP_O_1; 
	mole.c[ipSIP][ipATO] += rate_COmole.SiP_O2_SiOP_O_2;
	mole.b[ipATO] += rate_COmole.b_SiP_O2_SiOP_O;
	
	/* >>refer	mol	CH	CO+	=>	HCO+	C	UMIST	*/
			
	mole.c[ipCOP][ipCH] -= rate_COmole.CH_COP_HCOP_C_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_COP_HCOP_C_2;
	mole.b[ipCH] -= rate_COmole.b_CH_COP_HCOP_C;
	
	mole.c[ipCOP][ipCOP] -= rate_COmole.CH_COP_HCOP_C_1; 
	mole.c[ipCH][ipCOP] -= rate_COmole.CH_COP_HCOP_C_2;
	mole.b[ipCOP] -= rate_COmole.b_CH_COP_HCOP_C;
	
	mole.c[ipCOP][ipHCOP] += rate_COmole.CH_COP_HCOP_C_1; 
	mole.c[ipCH][ipHCOP] += rate_COmole.CH_COP_HCOP_C_2;
	mole.b[ipHCOP] += rate_COmole.b_CH_COP_HCOP_C;
	
	mole.c[ipCOP][ipATC] += rate_COmole.CH_COP_HCOP_C_1; 
	mole.c[ipCH][ipATC] += rate_COmole.CH_COP_HCOP_C_2;
	mole.b[ipATC] += rate_COmole.b_CH_COP_HCOP_C;
	
	/* >>refer	mol	CH	H2O+	=>	H2O	CH+	UMIST	*/
		
	mole.c[ipH2OP][ipCH] -= rate_COmole.CH_H2OP_H2O_CHP_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_H2OP_H2O_CHP_2;
	mole.b[ipCH] -= rate_COmole.b_CH_H2OP_H2O_CHP;
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.CH_H2OP_H2O_CHP_1; 
	mole.c[ipCH][ipH2OP] -= rate_COmole.CH_H2OP_H2O_CHP_2;
	mole.b[ipH2OP] -= rate_COmole.b_CH_H2OP_H2O_CHP;
	
	mole.c[ipH2OP][ipH2O] += rate_COmole.CH_H2OP_H2O_CHP_1; 
	mole.c[ipCH][ipH2O] += rate_COmole.CH_H2OP_H2O_CHP_2;
	mole.b[ipH2O] += rate_COmole.b_CH_H2OP_H2O_CHP;
	
	mole.c[ipH2OP][ipCHP] += rate_COmole.CH_H2OP_H2O_CHP_1; 
	mole.c[ipCH][ipCHP] += rate_COmole.CH_H2OP_H2O_CHP_2;
	mole.b[ipCHP] += rate_COmole.b_CH_H2OP_H2O_CHP;
	
	/* >>refer	mol	CH	H2O+	=>	OH	CH2+	UMIST	*/
		
	mole.c[ipH2OP][ipCH] -= rate_COmole.CH_H2OP_OH_CH2P_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_H2OP_OH_CH2P_2;
	mole.b[ipCH] -= rate_COmole.b_CH_H2OP_OH_CH2P;
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.CH_H2OP_OH_CH2P_1; 
	mole.c[ipCH][ipH2OP] -= rate_COmole.CH_H2OP_OH_CH2P_2;
	mole.b[ipH2OP] -= rate_COmole.b_CH_H2OP_OH_CH2P;
	
	mole.c[ipH2OP][ipOH] += rate_COmole.CH_H2OP_OH_CH2P_1; 
	mole.c[ipCH][ipOH] += rate_COmole.CH_H2OP_OH_CH2P_2;
	mole.b[ipOH] += rate_COmole.b_CH_H2OP_OH_CH2P;
	
	mole.c[ipH2OP][ipCH2P] += rate_COmole.CH_H2OP_OH_CH2P_1; 
	mole.c[ipCH][ipCH2P] += rate_COmole.CH_H2OP_OH_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH_H2OP_OH_CH2P;
	
	
	/* >>refer	mol	CH	H3O+	=>	H2O	CH2+	UMIST	*/
		
	mole.c[ipH3OP][ipCH] -= rate_COmole.CH_H3OP_H2O_CH2P_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_H3OP_H2O_CH2P_2;
	mole.b[ipCH] -= rate_COmole.b_CH_H3OP_H2O_CH2P;
	
	mole.c[ipH3OP][ipH3OP] -= rate_COmole.CH_H3OP_H2O_CH2P_1; 
	mole.c[ipCH][ipH3OP] -= rate_COmole.CH_H3OP_H2O_CH2P_2;
	mole.b[ipH3OP] -= rate_COmole.b_CH_H3OP_H2O_CH2P;
	
	mole.c[ipH3OP][ipH2O] += rate_COmole.CH_H3OP_H2O_CH2P_1; 
	mole.c[ipCH][ipH2O] += rate_COmole.CH_H3OP_H2O_CH2P_2;
	mole.b[ipH2O] += rate_COmole.b_CH_H3OP_H2O_CH2P;
	
	mole.c[ipH3OP][ipCH2P] += rate_COmole.CH_H3OP_H2O_CH2P_1; 
	mole.c[ipCH][ipCH2P] += rate_COmole.CH_H3OP_H2O_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH_H3OP_H2O_CH2P;
	

	/* >>refer	mol	CH	O2+	=>	O2	CH+	UMIST	*/
		
	mole.c[ipO2P][ipCH] -= rate_COmole.CH_O2P_O2_CHP_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_O2P_O2_CHP_2;
	mole.b[ipCH] -= rate_COmole.b_CH_O2P_O2_CHP;
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.CH_O2P_O2_CHP_1; 
	mole.c[ipCH][ipO2P] -= rate_COmole.CH_O2P_O2_CHP_2;
	mole.b[ipO2P] -= rate_COmole.b_CH_O2P_O2_CHP;
	
	mole.c[ipO2P][ipO2] += rate_COmole.CH_O2P_O2_CHP_1; 
	mole.c[ipCH][ipO2] += rate_COmole.CH_O2P_O2_CHP_2;
	mole.b[ipO2] += rate_COmole.b_CH_O2P_O2_CHP;
	
	mole.c[ipO2P][ipCHP] += rate_COmole.CH_O2P_O2_CHP_1; 
	mole.c[ipCH][ipCHP] += rate_COmole.CH_O2P_O2_CHP_2;
	mole.b[ipCHP] += rate_COmole.b_CH_O2P_O2_CHP;
	
	/* >>refer	mol	CH	O2+	=>	HCO+	O	UMIST	*/
		
	mole.c[ipO2P][ipCH] -= rate_COmole.CH_O2P_HCOP_O_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_O2P_HCOP_O_2;
	mole.b[ipCH] -= rate_COmole.b_CH_O2P_HCOP_O;
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.CH_O2P_HCOP_O_1; 
	mole.c[ipCH][ipO2P] -= rate_COmole.CH_O2P_HCOP_O_2;
	mole.b[ipO2P] -= rate_COmole.b_CH_O2P_HCOP_O;
	
	mole.c[ipO2P][ipHCOP] += rate_COmole.CH_O2P_HCOP_O_1; 
	mole.c[ipCH][ipHCOP] += rate_COmole.CH_O2P_HCOP_O_2;
	mole.b[ipHCOP] += rate_COmole.b_CH_O2P_HCOP_O;
	
	mole.c[ipO2P][ipATO] += rate_COmole.CH_O2P_HCOP_O_1; 
	mole.c[ipCH][ipATO] += rate_COmole.CH_O2P_HCOP_O_2;
	mole.b[ipATO] += rate_COmole.b_CH_O2P_HCOP_O;
	
	/* >>refer	mol	CH	OH+	=>	OH	CH+	UMIST	*/
		
	mole.c[ipOHP][ipCH] -= rate_COmole.CH_OHP_OH_CHP_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_OHP_OH_CHP_2;
	mole.b[ipCH] -= rate_COmole.b_CH_OHP_OH_CHP;
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.CH_OHP_OH_CHP_1; 
	mole.c[ipCH][ipOHP] -= rate_COmole.CH_OHP_OH_CHP_2;
	mole.b[ipOHP] -= rate_COmole.b_CH_OHP_OH_CHP;
	
	mole.c[ipOHP][ipOH] += rate_COmole.CH_OHP_OH_CHP_1; 
	mole.c[ipCH][ipOH] += rate_COmole.CH_OHP_OH_CHP_2;
	mole.b[ipOH] += rate_COmole.b_CH_OHP_OH_CHP;
	
	mole.c[ipOHP][ipCHP] += rate_COmole.CH_OHP_OH_CHP_1; 
	mole.c[ipCH][ipCHP] += rate_COmole.CH_OHP_OH_CHP_2;
	mole.b[ipCHP] += rate_COmole.b_CH_OHP_OH_CHP;
	
	/* >>refer	mol	CH	OH+	=>	O	CH2+	UMIST	*/
	
	mole.c[ipOHP][ipCH] -= rate_COmole.CH_OHP_O_CH2P_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_OHP_O_CH2P_2;
	mole.b[ipCH] -= rate_COmole.b_CH_OHP_O_CH2P;
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.CH_OHP_O_CH2P_1; 
	mole.c[ipCH][ipOHP] -= rate_COmole.CH_OHP_O_CH2P_2;
	mole.b[ipOHP] -= rate_COmole.b_CH_OHP_O_CH2P;
	
	mole.c[ipOHP][ipCH2P] += rate_COmole.CH_OHP_O_CH2P_1; 
	mole.c[ipCH][ipCH2P] += rate_COmole.CH_OHP_O_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH_OHP_O_CH2P;
	
	mole.c[ipOHP][ipATO] += rate_COmole.CH_OHP_O_CH2P_1; 
	mole.c[ipCH][ipATO] += rate_COmole.CH_OHP_O_CH2P_2;
	mole.b[ipATO] += rate_COmole.b_CH_OHP_O_CH2P;
	
	/* >>refer	mol	CH	SiO+	=>	HCO+	Si	UMIST	*/
	
	mole.c[ipSIOP][ipCH] -= rate_COmole.CH_SiOP_HCOP_Si_1; 
	mole.c[ipCH][ipCH] -= rate_COmole.CH_SiOP_HCOP_Si_2;
	mole.b[ipCH] -= rate_COmole.b_CH_SiOP_HCOP_Si;
	
	mole.c[ipSIOP][ipSIOP] -= rate_COmole.CH_SiOP_HCOP_Si_1; 
	mole.c[ipCH][ipSIOP] -= rate_COmole.CH_SiOP_HCOP_Si_2;
	mole.b[ipSIOP] -= rate_COmole.b_CH_SiOP_HCOP_Si;
	
	mole.c[ipSIOP][ipHCOP] += rate_COmole.CH_SiOP_HCOP_Si_1; 
	mole.c[ipCH][ipHCOP] += rate_COmole.CH_SiOP_HCOP_Si_2;
	mole.b[ipHCOP] += rate_COmole.b_CH_SiOP_HCOP_Si;
		
	mole.c[ipSIOP][ipATSI] += rate_COmole.CH_SiOP_HCOP_Si_1; 
	mole.c[ipCH][ipATSI] += rate_COmole.CH_SiOP_HCOP_Si_2;
	mole.b[ipATSI] += rate_COmole.b_CH_SiOP_HCOP_Si;
	
	/* >>refer	mol	CH+	H2O	=>	H3O+	C	UMIST	*/
	
	mole.c[ipH2O][ipCHP] -= rate_COmole.CHP_H2O_H3OP_C_1; 
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_H2O_H3OP_C_2;
	mole.b[ipCHP] -= rate_COmole.b_CHP_H2O_H3OP_C;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.CHP_H2O_H3OP_C_1; 
	mole.c[ipCHP][ipH2O] -= rate_COmole.CHP_H2O_H3OP_C_2;
	mole.b[ipH2O] -= rate_COmole.b_CHP_H2O_H3OP_C;
	
	mole.c[ipH2O][ipH3OP] += rate_COmole.CHP_H2O_H3OP_C_1; 
	mole.c[ipCHP][ipH3OP] += rate_COmole.CHP_H2O_H3OP_C_2;
	mole.b[ipH3OP] += rate_COmole.b_CHP_H2O_H3OP_C;
	
	mole.c[ipH2O][ipATC] += rate_COmole.CHP_H2O_H3OP_C_1; 
	mole.c[ipCHP][ipATC] += rate_COmole.CHP_H2O_H3OP_C_2;
	mole.b[ipATC] += rate_COmole.b_CHP_H2O_H3OP_C;
	
	/* >>refer	mol	CH+	H2O	=>	HCO+	H2	UMIST	*/
		
	mole.c[ipH2O][ipCHP] -= rate_COmole.CHP_H2O_HCOP_H2_1; 
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_H2O_HCOP_H2_2;
	mole.b[ipCHP] -= rate_COmole.b_CHP_H2O_HCOP_H2;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.CHP_H2O_HCOP_H2_1; 
	mole.c[ipCHP][ipH2O] -= rate_COmole.CHP_H2O_HCOP_H2_2;
	mole.b[ipH2O] -= rate_COmole.b_CHP_H2O_HCOP_H2;
	
	mole.c[ipH2O][ipHCOP] += rate_COmole.CHP_H2O_HCOP_H2_1; 
	mole.c[ipCHP][ipHCOP] += rate_COmole.CHP_H2O_HCOP_H2_2;
	mole.b[ipHCOP] += rate_COmole.b_CHP_H2O_HCOP_H2;
	
	/* >>refer	mol	CH+	O2	=>	HCO+	O	UMIST	*/
	
	mole.c[ipO2][ipCHP] -= rate_COmole.CHP_O2_HCOP_O_1; 
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_O2_HCOP_O_2;
	mole.b[ipCHP] -= rate_COmole.b_CHP_O2_HCOP_O;
	
	mole.c[ipO2][ipO2] -= rate_COmole.CHP_O2_HCOP_O_1; 
	mole.c[ipCHP][ipO2] -= rate_COmole.CHP_O2_HCOP_O_2;
	mole.b[ipO2] -= rate_COmole.b_CHP_O2_HCOP_O;
	
	mole.c[ipO2][ipHCOP] += rate_COmole.CHP_O2_HCOP_O_1; 
	mole.c[ipCHP][ipHCOP] += rate_COmole.CHP_O2_HCOP_O_2;
	mole.b[ipHCOP] += rate_COmole.b_CHP_O2_HCOP_O;
	
	mole.c[ipO2][ipATO] += rate_COmole.CHP_O2_HCOP_O_1; 
	mole.c[ipCHP][ipATO] += rate_COmole.CHP_O2_HCOP_O_2;
	mole.b[ipATO] += rate_COmole.b_CHP_O2_HCOP_O;


	/* >>refer	mol	CH+	OH	=>	CO+	H2	UMIST	*/
	
	mole.c[ipOH][ipCHP] -= rate_COmole.CHP_OH_COP_H2_1; 
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_OH_COP_H2_2;
	mole.b[ipCHP] -= rate_COmole.b_CHP_OH_COP_H2;
	
	mole.c[ipOH][ipOH] -= rate_COmole.CHP_OH_COP_H2_1; 
	mole.c[ipCHP][ipOH] -= rate_COmole.CHP_OH_COP_H2_2;
	mole.b[ipOH] -= rate_COmole.b_CHP_OH_COP_H2;
	
	mole.c[ipOH][ipCOP] += rate_COmole.CHP_OH_COP_H2_1; 
	mole.c[ipCHP][ipCOP] += rate_COmole.CHP_OH_COP_H2_2;
	mole.b[ipCOP] += rate_COmole.b_CHP_OH_COP_H2;

	
	/* >>refer	mol	CH+	O2	=>	CO+	OH	UMIST	*/
	
	mole.c[ipO2][ipCHP] -= rate_COmole.CHP_O2_COP_OH_1; 
	mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_O2_COP_OH_2;
	mole.b[ipCHP] -= rate_COmole.b_CHP_O2_COP_OH;
	
	mole.c[ipO2][ipO2] -= rate_COmole.CHP_O2_COP_OH_1; 
	mole.c[ipCHP][ipO2] -= rate_COmole.CHP_O2_COP_OH_2;
	mole.b[ipO2] -= rate_COmole.b_CHP_O2_COP_OH;
	
	mole.c[ipO2][ipCOP] += rate_COmole.CHP_O2_COP_OH_1; 
	mole.c[ipCHP][ipCOP] += rate_COmole.CHP_O2_COP_OH_2;
	mole.b[ipCOP] += rate_COmole.b_CHP_O2_COP_OH;
	
	mole.c[ipO2][ipOH] += rate_COmole.CHP_O2_COP_OH_1; 
	mole.c[ipCHP][ipOH] += rate_COmole.CHP_O2_COP_OH_2;
	mole.b[ipOH] += rate_COmole.b_CHP_O2_COP_OH;
	
	/* >>refer	mol	CH2	CO+	=>	HCO+	CH	UMIST	*/
	
	mole.c[ipCOP][ipCH2] -= rate_COmole.CH2_COP_HCOP_CH_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_COP_HCOP_CH_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_COP_HCOP_CH;
	
	mole.c[ipCOP][ipCOP] -= rate_COmole.CH2_COP_HCOP_CH_1; 
	mole.c[ipCH2][ipCOP] -= rate_COmole.CH2_COP_HCOP_CH_2;
	mole.b[ipCOP] -= rate_COmole.b_CH2_COP_HCOP_CH;
	
	mole.c[ipCOP][ipHCOP] += rate_COmole.CH2_COP_HCOP_CH_1; 
	mole.c[ipCH2][ipHCOP] += rate_COmole.CH2_COP_HCOP_CH_2;
	mole.b[ipHCOP] += rate_COmole.b_CH2_COP_HCOP_CH;
	
	mole.c[ipCOP][ipCH] += rate_COmole.CH2_COP_HCOP_CH_1; 
	mole.c[ipCH2][ipCH] += rate_COmole.CH2_COP_HCOP_CH_2;
	mole.b[ipCH] += rate_COmole.b_CH2_COP_HCOP_CH;
	

	/* >>refer	mol	CH2	H2O+	=>	H2O	CH2+	UMIST	*/
	
	mole.c[ipH2OP][ipCH2] -= rate_COmole.CH2_H2OP_H2O_CH2P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_H2OP_H2O_CH2P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_H2OP_H2O_CH2P;
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.CH2_H2OP_H2O_CH2P_1; 
	mole.c[ipCH2][ipH2OP] -= rate_COmole.CH2_H2OP_H2O_CH2P_2;
	mole.b[ipH2OP] -= rate_COmole.b_CH2_H2OP_H2O_CH2P;
	
	mole.c[ipH2OP][ipH2O] += rate_COmole.CH2_H2OP_H2O_CH2P_1; 
	mole.c[ipCH2][ipH2O] += rate_COmole.CH2_H2OP_H2O_CH2P_2;
	mole.b[ipH2O] += rate_COmole.b_CH2_H2OP_H2O_CH2P;
	
	mole.c[ipH2OP][ipCH2P] += rate_COmole.CH2_H2OP_H2O_CH2P_1; 
	mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_H2OP_H2O_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH2_H2OP_H2O_CH2P;
	
	/* >>refer	mol	CH2	H2O+	=>	OH	CH3+	UMIST	*/
	
	mole.c[ipH2OP][ipCH2] -= rate_COmole.CH2_H2OP_OH_CH3P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_H2OP_OH_CH3P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_H2OP_OH_CH3P;
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.CH2_H2OP_OH_CH3P_1; 
	mole.c[ipCH2][ipH2OP] -= rate_COmole.CH2_H2OP_OH_CH3P_2;
	mole.b[ipH2OP] -= rate_COmole.b_CH2_H2OP_OH_CH3P;
	
	mole.c[ipH2OP][ipOH] += rate_COmole.CH2_H2OP_OH_CH3P_1; 
	mole.c[ipCH2][ipOH] += rate_COmole.CH2_H2OP_OH_CH3P_2;
	mole.b[ipOH] += rate_COmole.b_CH2_H2OP_OH_CH3P;
	
	mole.c[ipH2OP][ipCH3P] += rate_COmole.CH2_H2OP_OH_CH3P_1; 
	mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_H2OP_OH_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_CH2_H2OP_OH_CH3P;
	
	/* >>refer	mol	CH2	H3O+	=>	H2O	CH3+	UMIST	*/
	
	mole.c[ipH3OP][ipCH2] -= rate_COmole.CH2_H3OP_H2O_CH3P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_H3OP_H2O_CH3P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_H3OP_H2O_CH3P;
	
	mole.c[ipH3OP][ipH3OP] -= rate_COmole.CH2_H3OP_H2O_CH3P_1; 
	mole.c[ipCH2][ipH3OP] -= rate_COmole.CH2_H3OP_H2O_CH3P_2;
	mole.b[ipH3OP] -= rate_COmole.b_CH2_H3OP_H2O_CH3P;
	
	mole.c[ipH3OP][ipH2O] += rate_COmole.CH2_H3OP_H2O_CH3P_1; 
	mole.c[ipCH2][ipH2O] += rate_COmole.CH2_H3OP_H2O_CH3P_2;
	mole.b[ipH2O] += rate_COmole.b_CH2_H3OP_H2O_CH3P;
	
	mole.c[ipH3OP][ipCH3P] += rate_COmole.CH2_H3OP_H2O_CH3P_1; 
	mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_H3OP_H2O_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_CH2_H3OP_H2O_CH3P;
	
	/* >>refer	mol	CH2	O2+	=>	O2	CH2+	UMIST	*/
	
	mole.c[ipO2P][ipCH2] -= rate_COmole.CH2_O2P_O2_CH2P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_O2P_O2_CH2P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_O2P_O2_CH2P;
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.CH2_O2P_O2_CH2P_1; 
	mole.c[ipCH2][ipO2P] -= rate_COmole.CH2_O2P_O2_CH2P_2;
	mole.b[ipO2P] -= rate_COmole.b_CH2_O2P_O2_CH2P;
	
	mole.c[ipO2P][ipO2] += rate_COmole.CH2_O2P_O2_CH2P_1; 
	mole.c[ipCH2][ipO2] += rate_COmole.CH2_O2P_O2_CH2P_2;
	mole.b[ipO2] += rate_COmole.b_CH2_O2P_O2_CH2P;
	
	mole.c[ipO2P][ipCH2P] += rate_COmole.CH2_O2P_O2_CH2P_1; 
	mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_O2P_O2_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH2_O2P_O2_CH2P;
	
	/* >>refer	mol	CH2	OH	=>	H2O	CH	UMIST	*/
	
	mole.c[ipOH][ipCH2] -= rate_COmole.CH2_OH_H2O_CH_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_OH_H2O_CH_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_OH_H2O_CH;
	
	mole.c[ipOH][ipOH] -= rate_COmole.CH2_OH_H2O_CH_1; 
	mole.c[ipCH2][ipOH] -= rate_COmole.CH2_OH_H2O_CH_2;
	mole.b[ipOH] -= rate_COmole.b_CH2_OH_H2O_CH;
	
	mole.c[ipOH][ipH2O] += rate_COmole.CH2_OH_H2O_CH_1; 
	mole.c[ipCH2][ipH2O] += rate_COmole.CH2_OH_H2O_CH_2;
	mole.b[ipH2O] += rate_COmole.b_CH2_OH_H2O_CH;
	
	mole.c[ipOH][ipCH] += rate_COmole.CH2_OH_H2O_CH_1; 
	mole.c[ipCH2][ipCH] += rate_COmole.CH2_OH_H2O_CH_2;
	mole.b[ipCH] += rate_COmole.b_CH2_OH_H2O_CH;
	
	
	/* >>refer	mol	CH2	OH+	=>	OH	CH2+	UMIST	*/
		
	mole.c[ipOHP][ipCH2] -= rate_COmole.CH2_OHP_OH_CH2P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_OHP_OH_CH2P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_OHP_OH_CH2P;
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.CH2_OHP_OH_CH2P_1; 
	mole.c[ipCH2][ipOHP] -= rate_COmole.CH2_OHP_OH_CH2P_2;
	mole.b[ipOHP] -= rate_COmole.b_CH2_OHP_OH_CH2P;
	
	mole.c[ipOHP][ipOH] += rate_COmole.CH2_OHP_OH_CH2P_1; 
	mole.c[ipCH2][ipOH] += rate_COmole.CH2_OHP_OH_CH2P_2;
	mole.b[ipOH] += rate_COmole.b_CH2_OHP_OH_CH2P;
	
	mole.c[ipOHP][ipCH2P] += rate_COmole.CH2_OHP_OH_CH2P_1; 
	mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_OHP_OH_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH2_OHP_OH_CH2P;
	
	/* >>refer	mol	CH2	OH+	=>	O	CH3+	UMIST	*/
	
	mole.c[ipOHP][ipCH2] -= rate_COmole.CH2_OHP_O_CH3P_1; 
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_OHP_O_CH3P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_OHP_O_CH3P;
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.CH2_OHP_O_CH3P_1; 
	mole.c[ipCH2][ipOHP] -= rate_COmole.CH2_OHP_O_CH3P_2;
	mole.b[ipOHP] -= rate_COmole.b_CH2_OHP_O_CH3P;
	
	mole.c[ipOHP][ipCH3P] += rate_COmole.CH2_OHP_O_CH3P_1; 
	mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_OHP_O_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_CH2_OHP_O_CH3P;
	
	mole.c[ipOHP][ipATO] += rate_COmole.CH2_OHP_O_CH3P_1; 
	mole.c[ipCH2][ipATO] += rate_COmole.CH2_OHP_O_CH3P_2;
	mole.b[ipATO] += rate_COmole.b_CH2_OHP_O_CH3P;
	
	/* >>refer	mol	CH2+	O2+	=>	HCO+	OH	UMIST	*/
		
	mole.c[ipO2P][ipCH2P] -= rate_COmole.CH2P_O2_HCOP_OH_1; 
	mole.c[ipCH2P][ipCH2P] -= rate_COmole.CH2P_O2_HCOP_OH_2;
	mole.b[ipCH2P] -= rate_COmole.b_CH2P_O2_HCOP_OH;
	
	mole.c[ipO2P][ipO2P] -= rate_COmole.CH2P_O2_HCOP_OH_1; 
	mole.c[ipCH2P][ipO2P] -= rate_COmole.CH2P_O2_HCOP_OH_2;
	mole.b[ipO2P] -= rate_COmole.b_CH2P_O2_HCOP_OH;
	
	mole.c[ipO2P][ipHCOP] += rate_COmole.CH2P_O2_HCOP_OH_1; 
	mole.c[ipCH2P][ipHCOP] += rate_COmole.CH2P_O2_HCOP_OH_2;
	mole.b[ipHCOP] += rate_COmole.b_CH2P_O2_HCOP_OH;
	
	mole.c[ipO2P][ipOH] += rate_COmole.CH2P_O2_HCOP_OH_1; 
	mole.c[ipCH2P][ipOH] += rate_COmole.CH2P_O2_HCOP_OH_2;
	mole.b[ipOH] += rate_COmole.b_CH2P_O2_HCOP_OH;
	

	/* >>refer	mol	H2O	CO+	=>	HCO+	OH	UMIST	*/
	
	mole.c[ipCOP][ipH2O] -= rate_COmole.H2O_COP_HCOP_OH_1; 
	mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_COP_HCOP_OH_2;
	mole.b[ipH2O] -= rate_COmole.b_H2O_COP_HCOP_OH;
	
	mole.c[ipCOP][ipCOP] -= rate_COmole.H2O_COP_HCOP_OH_1; 
	mole.c[ipH2O][ipCOP] -= rate_COmole.H2O_COP_HCOP_OH_2;
	mole.b[ipCOP] -= rate_COmole.b_H2O_COP_HCOP_OH;
	
	mole.c[ipCOP][ipHCOP] += rate_COmole.H2O_COP_HCOP_OH_1; 
	mole.c[ipH2O][ipHCOP] += rate_COmole.H2O_COP_HCOP_OH_2;
	mole.b[ipHCOP] += rate_COmole.b_H2O_COP_HCOP_OH;
	
	mole.c[ipCOP][ipOH] += rate_COmole.H2O_COP_HCOP_OH_1; 
	mole.c[ipH2O][ipOH] += rate_COmole.H2O_COP_HCOP_OH_2;
	mole.b[ipOH] += rate_COmole.b_H2O_COP_HCOP_OH;
	

	
	/* >>refer	mol	H2O+	H2O	=>	H3O+	OH	UMIST	*/
		
	mole.c[ipH2O][ipH2OP] -= rate_COmole.H2OP_H2O_H3OP_OH_1; 
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_H2O_H3OP_OH_2;
	mole.b[ipH2OP] -= rate_COmole.b_H2OP_H2O_H3OP_OH;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.H2OP_H2O_H3OP_OH_1; 
	mole.c[ipH2OP][ipH2O] -= rate_COmole.H2OP_H2O_H3OP_OH_2;
	mole.b[ipH2O] -= rate_COmole.b_H2OP_H2O_H3OP_OH;
	
	mole.c[ipH2O][ipH3OP] += rate_COmole.H2OP_H2O_H3OP_OH_1; 
	mole.c[ipH2OP][ipH3OP] += rate_COmole.H2OP_H2O_H3OP_OH_2;
	mole.b[ipH3OP] += rate_COmole.b_H2OP_H2O_H3OP_OH;
	
	mole.c[ipH2O][ipOH] += rate_COmole.H2OP_H2O_H3OP_OH_1; 
	mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_H2O_H3OP_OH_2;
	mole.b[ipOH] += rate_COmole.b_H2OP_H2O_H3OP_OH;
	
	/* >>refer	mol	H2O+	O2	=>	O2+	H2O	UMIST	*/
		
	
	mole.c[ipO2][ipH2OP] -= rate_COmole.H2OP_O2_O2P_H2O_1; 
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_O2_O2P_H2O_2;
	mole.b[ipH2OP] -= rate_COmole.b_H2OP_O2_O2P_H2O;
	
	mole.c[ipO2][ipO2] -= rate_COmole.H2OP_O2_O2P_H2O_1; 
	mole.c[ipH2OP][ipO2] -= rate_COmole.H2OP_O2_O2P_H2O_2;
	mole.b[ipO2] -= rate_COmole.b_H2OP_O2_O2P_H2O;
	
	mole.c[ipO2][ipO2P] += rate_COmole.H2OP_O2_O2P_H2O_1; 
	mole.c[ipH2OP][ipO2P] += rate_COmole.H2OP_O2_O2P_H2O_2;
	mole.b[ipO2P] += rate_COmole.b_H2OP_O2_O2P_H2O;
	
	mole.c[ipO2][ipH2O] += rate_COmole.H2OP_O2_O2P_H2O_1; 
	mole.c[ipH2OP][ipH2O] += rate_COmole.H2OP_O2_O2P_H2O_2;
	mole.b[ipH2O] += rate_COmole.b_H2OP_O2_O2P_H2O;
	
	/* >>refer	mol	H3O+	SiH	=>	SiH2+	H2O	UMIST	*/
		
	
	mole.c[ipSIH][ipH3OP] -= rate_COmole.H3OP_SiH_SiH2P_H2O_1; 
	mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_SiH_SiH2P_H2O_2;
	mole.b[ipH3OP] -= rate_COmole.b_H3OP_SiH_SiH2P_H2O;
	
	mole.c[ipSIH][ipSIH] -= rate_COmole.H3OP_SiH_SiH2P_H2O_1; 
	mole.c[ipH3OP][ipSIH] -= rate_COmole.H3OP_SiH_SiH2P_H2O_2;
	mole.b[ipSIH] -= rate_COmole.b_H3OP_SiH_SiH2P_H2O;
	
	mole.c[ipSIH][ipSIH2P] += rate_COmole.H3OP_SiH_SiH2P_H2O_1; 
	mole.c[ipH3OP][ipSIH2P] += rate_COmole.H3OP_SiH_SiH2P_H2O_2;
	mole.b[ipSIH2P] += rate_COmole.b_H3OP_SiH_SiH2P_H2O;
	
	mole.c[ipSIH][ipH2O] += rate_COmole.H3OP_SiH_SiH2P_H2O_1; 
	mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_SiH_SiH2P_H2O_2;
	mole.b[ipH2O] += rate_COmole.b_H3OP_SiH_SiH2P_H2O;
	
	/* >>refer	mol	H3O+	SiO	=>	SiOH+	H2O	UMIST	*/
		
	
	mole.c[ipSIO][ipH3OP] -= rate_COmole.H3OP_SiO_SiOHP_H2O_1; 
	mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_SiO_SiOHP_H2O_2;
	mole.b[ipH3OP] -= rate_COmole.b_H3OP_SiO_SiOHP_H2O;
	
	mole.c[ipSIO][ipSIO] -= rate_COmole.H3OP_SiO_SiOHP_H2O_1; 
	mole.c[ipH3OP][ipSIO] -= rate_COmole.H3OP_SiO_SiOHP_H2O_2;
	mole.b[ipSIO] -= rate_COmole.b_H3OP_SiO_SiOHP_H2O;
	
	mole.c[ipSIO][ipHOSIP] += rate_COmole.H3OP_SiO_SiOHP_H2O_1; 
	mole.c[ipH3OP][ipHOSIP] += rate_COmole.H3OP_SiO_SiOHP_H2O_2;
	mole.b[ipHOSIP] += rate_COmole.b_H3OP_SiO_SiOHP_H2O;
	
	mole.c[ipSIO][ipH2O] += rate_COmole.H3OP_SiO_SiOHP_H2O_1; 
	mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_SiO_SiOHP_H2O_2;
	mole.b[ipH2O] += rate_COmole.b_H3OP_SiO_SiOHP_H2O;
	
	/* >>refer	mol	OH	CO+	=>	HCO+	O	UMIST	*/
	
	mole.c[ipCOP][ipOH] -= rate_COmole.OH_COP_HCOP_O_1; 
	mole.c[ipOH][ipOH] -= rate_COmole.OH_COP_HCOP_O_2;
	mole.b[ipOH] -= rate_COmole.b_OH_COP_HCOP_O;
	
	mole.c[ipCOP][ipCOP] -= rate_COmole.OH_COP_HCOP_O_1; 
	mole.c[ipOH][ipCOP] -= rate_COmole.OH_COP_HCOP_O_2;
	mole.b[ipCOP] -= rate_COmole.b_OH_COP_HCOP_O;
	
	mole.c[ipCOP][ipHCOP] += rate_COmole.OH_COP_HCOP_O_1; 
	mole.c[ipOH][ipHCOP] += rate_COmole.OH_COP_HCOP_O_2;
	mole.b[ipHCOP] += rate_COmole.b_OH_COP_HCOP_O;
	
	mole.c[ipCOP][ipATO] += rate_COmole.OH_COP_HCOP_O_1; 
	mole.c[ipOH][ipATO] += rate_COmole.OH_COP_HCOP_O_2;
	mole.b[ipATO] += rate_COmole.b_OH_COP_HCOP_O;
	

	/* >>refer	mol	OH	H2O+	=>	H3O+	O	UMIST	*/
	
	mole.c[ipH2OP][ipOH] -= rate_COmole.OH_H2OP_H3OP_O_1; 
	mole.c[ipOH][ipOH] -= rate_COmole.OH_H2OP_H3OP_O_2;
	mole.b[ipOH] -= rate_COmole.b_OH_H2OP_H3OP_O;
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.OH_H2OP_H3OP_O_1; 
	mole.c[ipOH][ipH2OP] -= rate_COmole.OH_H2OP_H3OP_O_2;
	mole.b[ipH2OP] -= rate_COmole.b_OH_H2OP_H3OP_O;
	
	mole.c[ipH2OP][ipH3OP] += rate_COmole.OH_H2OP_H3OP_O_1; 
	mole.c[ipOH][ipH3OP] += rate_COmole.OH_H2OP_H3OP_O_2;
	mole.b[ipH3OP] += rate_COmole.b_OH_H2OP_H3OP_O;
	
	mole.c[ipH2OP][ipATO] += rate_COmole.OH_H2OP_H3OP_O_1; 
	mole.c[ipOH][ipATO] += rate_COmole.OH_H2OP_H3OP_O_2;
	mole.b[ipATO] += rate_COmole.b_OH_H2OP_H3OP_O;
	
	/* >>refer	mol	OH	OH	=>	H2O	H	UMIST	*/
	
	mole.c[ipOH][ipOH] -= rate_COmole.OH_OH_H2O_O_1; 
	mole.c[ipOH][ipOH] -= rate_COmole.OH_OH_H2O_O_2;
	mole.b[ipOH] -= rate_COmole.b_OH_OH_H2O_O;
	
	mole.c[ipOH][ipH2O] += 0.5*rate_COmole.OH_OH_H2O_O_1; 
	mole.c[ipOH][ipH2O] += 0.5*rate_COmole.OH_OH_H2O_O_2;
	mole.b[ipH2O] += 0.5*rate_COmole.b_OH_OH_H2O_O;

	/* >>refer	mol	OH+	H2O	=>	H3O+	O	UMIST	*/
	
	mole.c[ipH2O][ipOHP] -= rate_COmole.OHP_H2O_H3OP_O_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_H2O_H3OP_O_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_H2O_H3OP_O;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.OHP_H2O_H3OP_O_1; 
	mole.c[ipOHP][ipH2O] -= rate_COmole.OHP_H2O_H3OP_O_2;
	mole.b[ipH2O] -= rate_COmole.b_OHP_H2O_H3OP_O;
	
	mole.c[ipH2O][ipH3OP] += rate_COmole.OHP_H2O_H3OP_O_1; 
	mole.c[ipOHP][ipH3OP] += rate_COmole.OHP_H2O_H3OP_O_2;
	mole.b[ipH3OP] += rate_COmole.b_OHP_H2O_H3OP_O;
	
	mole.c[ipH2O][ipATO] += rate_COmole.OHP_H2O_H3OP_O_1; 
	mole.c[ipOHP][ipATO] += rate_COmole.OHP_H2O_H3OP_O_2;
	mole.b[ipATO] += rate_COmole.b_OHP_H2O_H3OP_O;
	
	/* >>refer	mol	OH+	H2O	=>	H2O+	OH	UMIST	*/
	
	mole.c[ipH2O][ipOHP] -= rate_COmole.OHP_H2O_H2OP_OH_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_H2O_H2OP_OH_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_H2O_H2OP_OH;
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.OHP_H2O_H2OP_OH_1; 
	mole.c[ipOHP][ipH2O] -= rate_COmole.OHP_H2O_H2OP_OH_2;
	mole.b[ipH2O] -= rate_COmole.b_OHP_H2O_H2OP_OH;
	
	mole.c[ipH2O][ipH2OP] += rate_COmole.OHP_H2O_H2OP_OH_1; 
	mole.c[ipOHP][ipH2OP] += rate_COmole.OHP_H2O_H2OP_OH_2;
	mole.b[ipH2OP] += rate_COmole.b_OHP_H2O_H2OP_OH;
	
	mole.c[ipH2O][ipOH] += rate_COmole.OHP_H2O_H2OP_OH_1; 
	mole.c[ipOHP][ipOH] += rate_COmole.OHP_H2O_H2OP_OH_2;
	mole.b[ipOH] += rate_COmole.b_OHP_H2O_H2OP_OH;
	
	/* >>refer	mol	OH+	O2	=>	O2+	OH	UMIST	*/
	
	mole.c[ipO2][ipOHP] -= rate_COmole.OHP_O2_O2P_OH_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_O2_O2P_OH_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_O2_O2P_OH;
	
	mole.c[ipO2][ipO2] -= rate_COmole.OHP_O2_O2P_OH_1; 
	mole.c[ipOHP][ipO2] -= rate_COmole.OHP_O2_O2P_OH_2;
	mole.b[ipO2] -= rate_COmole.b_OHP_O2_O2P_OH;
	
	mole.c[ipO2][ipO2P] += rate_COmole.OHP_O2_O2P_OH_1; 
	mole.c[ipOHP][ipO2P] += rate_COmole.OHP_O2_O2P_OH_2;
	mole.b[ipO2P] += rate_COmole.b_OHP_O2_O2P_OH;
	
	mole.c[ipO2][ipOH] += rate_COmole.OHP_O2_O2P_OH_1; 
	mole.c[ipOHP][ipOH] += rate_COmole.OHP_O2_O2P_OH_2;
	mole.b[ipOH] += rate_COmole.b_OHP_O2_O2P_OH;
	
	/* >>refer	mol	OH+	OH	=>	H2O+	O	UMIST	*/
	
	mole.c[ipOH][ipOHP] -= rate_COmole.OHP_OH_H2OP_O_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_OH_H2OP_O_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_OH_H2OP_O;
	
	mole.c[ipOH][ipOH] -= rate_COmole.OHP_OH_H2OP_O_1; 
	mole.c[ipOHP][ipOH] -= rate_COmole.OHP_OH_H2OP_O_2;
	mole.b[ipOH] -= rate_COmole.b_OHP_OH_H2OP_O;
	
	mole.c[ipOH][ipH2OP] += rate_COmole.OHP_OH_H2OP_O_1; 
	mole.c[ipOHP][ipH2OP] += rate_COmole.OHP_OH_H2OP_O_2;
	mole.b[ipH2OP] += rate_COmole.b_OHP_OH_H2OP_O;
	
	mole.c[ipOH][ipATO] += rate_COmole.OHP_OH_H2OP_O_1; 
	mole.c[ipOHP][ipATO] += rate_COmole.OHP_OH_H2OP_O_2;
	mole.b[ipATO] += rate_COmole.b_OHP_OH_H2OP_O;
	
	/* >>refer	mol	OH+	SiH	=>	SiH2+	O	UMIST	*/
	
	mole.c[ipSIH][ipOHP] -= rate_COmole.OHP_SiH_SiH2P_O_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_SiH_SiH2P_O_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_SiH_SiH2P_O;
	
	mole.c[ipSIH][ipSIH] -= rate_COmole.OHP_SiH_SiH2P_O_1; 
	mole.c[ipOHP][ipSIH] -= rate_COmole.OHP_SiH_SiH2P_O_2;
	mole.b[ipSIH] -= rate_COmole.b_OHP_SiH_SiH2P_O;
	
	mole.c[ipSIH][ipSIH2P] += rate_COmole.OHP_SiH_SiH2P_O_1; 
	mole.c[ipOHP][ipSIH2P] += rate_COmole.OHP_SiH_SiH2P_O_2;
	mole.b[ipSIH2P] += rate_COmole.b_OHP_SiH_SiH2P_O;
	
	mole.c[ipSIH][ipATO] += rate_COmole.OHP_SiH_SiH2P_O_1; 
	mole.c[ipOHP][ipATO] += rate_COmole.OHP_SiH_SiH2P_O_2;
	mole.b[ipATO] += rate_COmole.b_OHP_SiH_SiH2P_O;
	
	/* >>refer	mol	OH+	SiO	=>	SiOH+	O	UMIST	*/
	
	mole.c[ipSIO][ipOHP] -= rate_COmole.OHP_SiO_SiOHP_O_1; 
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_SiO_SiOHP_O_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_SiO_SiOHP_O;
	
	mole.c[ipSIO][ipSIO] -= rate_COmole.OHP_SiO_SiOHP_O_1; 
	mole.c[ipOHP][ipSIO] -= rate_COmole.OHP_SiO_SiOHP_O_2;
	mole.b[ipSIO] -= rate_COmole.b_OHP_SiO_SiOHP_O;
	
	mole.c[ipSIO][ipHOSIP] += rate_COmole.OHP_SiO_SiOHP_O_1; 
	mole.c[ipOHP][ipHOSIP] += rate_COmole.OHP_SiO_SiOHP_O_2;
	mole.b[ipHOSIP] += rate_COmole.b_OHP_SiO_SiOHP_O;
	
	mole.c[ipSIO][ipATO] += rate_COmole.OHP_SiO_SiOHP_O_1; 
	mole.c[ipOHP][ipATO] += rate_COmole.OHP_SiO_SiOHP_O_2;
	mole.b[ipATO] += rate_COmole.b_OHP_SiO_SiOHP_O;


	/* new coupled reactions involving CH3, CH4, CH4+, and CH5+ */

	/* >>refer mol   C CH5P  =>  CH4 CHP   UMIST    */
	
	mole.c[ipATC][ipCH5P] -= rate_COmole.C_CH5P_CH4_CHP_1;
	mole.c[ipCH5P][ipCH5P] -= rate_COmole.C_CH5P_CH4_CHP_2;
	mole.b[ipCH5P] -= rate_COmole.b_C_CH5P_CH4_CHP;
	
	mole.c[ipATC][ipATC] -= rate_COmole.C_CH5P_CH4_CHP_1;
	mole.c[ipCH5P][ipATC] -= rate_COmole.C_CH5P_CH4_CHP_2;
	mole.b[ipATC] -= rate_COmole.b_C_CH5P_CH4_CHP;
	
	mole.c[ipATC][ipCH4] += rate_COmole.C_CH5P_CH4_CHP_1;
	mole.c[ipCH5P][ipCH4] += rate_COmole.C_CH5P_CH4_CHP_2;
	mole.b[ipCH4] += rate_COmole.b_C_CH5P_CH4_CHP;
	
	mole.c[ipATC][ipCHP] += rate_COmole.C_CH5P_CH4_CHP_1;
	mole.c[ipCH5P][ipCHP] += rate_COmole.C_CH5P_CH4_CHP_2;
	mole.b[ipCHP] += rate_COmole.b_C_CH5P_CH4_CHP;
	
	/* >>refer mol   O CH4  =>  OH CH3   UMIST    */
	
	mole.c[ipATO][ipCH4] -= rate_COmole.O_CH4_OH_CH3_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.O_CH4_OH_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_O_CH4_OH_CH3;
		
	mole.c[ipATO][ipATO] -= rate_COmole.O_CH4_OH_CH3_1;
	mole.c[ipCH4][ipATO] -= rate_COmole.O_CH4_OH_CH3_2;
	mole.b[ipATO] -= rate_COmole.b_O_CH4_OH_CH3;
	
	mole.c[ipATO][ipOH] += rate_COmole.O_CH4_OH_CH3_1;
	mole.c[ipCH4][ipOH] += rate_COmole.O_CH4_OH_CH3_2;
	mole.b[ipOH] += rate_COmole.b_O_CH4_OH_CH3;
	
	mole.c[ipATO][ipCH3] += rate_COmole.O_CH4_OH_CH3_1;
	mole.c[ipCH4][ipCH3] += rate_COmole.O_CH4_OH_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_O_CH4_OH_CH3;
	
	
	/* >>refer mol   O CH4P  =>  OH CH3P   UMIST    */
	
	mole.c[ipATO][ipCH4P] -= rate_COmole.O_CH4P_OH_CH3P_1;
	mole.c[ipCH4P][ipCH4P] -= rate_COmole.O_CH4P_OH_CH3P_2;
	mole.b[ipCH4P] -= rate_COmole.b_O_CH4P_OH_CH3P;
	
	mole.c[ipATO][ipATO] -= rate_COmole.O_CH4P_OH_CH3P_1;
	mole.c[ipCH4P][ipATO] -= rate_COmole.O_CH4P_OH_CH3P_2;
	mole.b[ipATO] -= rate_COmole.b_O_CH4P_OH_CH3P;
	
	mole.c[ipATO][ipOH] += rate_COmole.O_CH4P_OH_CH3P_1;
	mole.c[ipCH4P][ipOH] += rate_COmole.O_CH4P_OH_CH3P_2;
	mole.b[ipOH] += rate_COmole.b_O_CH4P_OH_CH3P;
	
	mole.c[ipATO][ipCH3P] += rate_COmole.O_CH4P_OH_CH3P_1;
	mole.c[ipCH4P][ipCH3P] += rate_COmole.O_CH4P_OH_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_O_CH4P_OH_CH3P;
			
	/* >>refer mol   O CH5P  =>  H3OP CH2   UMIST    */
	
	mole.c[ipATO][ipCH5P] -= rate_COmole.O_CH5P_H3OP_CH2_1;
	mole.c[ipCH5P][ipCH5P] -= rate_COmole.O_CH5P_H3OP_CH2_2;
	mole.b[ipCH5P] -= rate_COmole.b_O_CH5P_H3OP_CH2;
		
	mole.c[ipATO][ipATO] -= rate_COmole.O_CH5P_H3OP_CH2_1;
	mole.c[ipCH5P][ipATO] -= rate_COmole.O_CH5P_H3OP_CH2_2;
	mole.b[ipATO] -= rate_COmole.b_O_CH5P_H3OP_CH2;
	
	mole.c[ipATO][ipH3OP] += rate_COmole.O_CH5P_H3OP_CH2_1;
	mole.c[ipCH5P][ipH3OP] += rate_COmole.O_CH5P_H3OP_CH2_2;
	mole.b[ipH3OP] += rate_COmole.b_O_CH5P_H3OP_CH2;
	
	mole.c[ipATO][ipCH2] += rate_COmole.O_CH5P_H3OP_CH2_1;
	mole.c[ipCH5P][ipCH2] += rate_COmole.O_CH5P_H3OP_CH2_2;
	mole.b[ipCH2] += rate_COmole.b_O_CH5P_H3OP_CH2;
	
	
	/* >>refer mol   OP CH4  =>  OH CH3P   UMIST    */
	
	mole.c[ipOP][ipOP] -= rate_COmole.OP_CH4_OH_CH3P_1;
	mole.c[ipCH4][ipOP] -= rate_COmole.OP_CH4_OH_CH3P_2;
	mole.b[ipOP] -= rate_COmole.b_OP_CH4_OH_CH3P;
	
	mole.c[ipOP][ipCH4] -= rate_COmole.OP_CH4_OH_CH3P_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.OP_CH4_OH_CH3P_2;
	mole.b[ipCH4] -= rate_COmole.b_OP_CH4_OH_CH3P;

	mole.c[ipOP][ipOH] += rate_COmole.OP_CH4_OH_CH3P_1;
	mole.c[ipCH4][ipOH] += rate_COmole.OP_CH4_OH_CH3P_2;
	mole.b[ipOH] += rate_COmole.b_OP_CH4_OH_CH3P;
	
	mole.c[ipOP][ipCH3P] += rate_COmole.OP_CH4_OH_CH3P_1;
	mole.c[ipCH4][ipCH3P] += rate_COmole.OP_CH4_OH_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_OP_CH4_OH_CH3P;
	
	
	/* >>refer mol   OP CH4  =>  CH4P O   UMIST    */
	
	mole.c[ipOP][ipOP] -= rate_COmole.OP_CH4_CH4P_O_1;
	mole.c[ipCH4][ipOP] -= rate_COmole.OP_CH4_CH4P_O_2;
	mole.b[ipOP] -= rate_COmole.b_OP_CH4_CH4P_O;
	
	mole.c[ipOP][ipCH4] -= rate_COmole.OP_CH4_CH4P_O_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.OP_CH4_CH4P_O_2;
	mole.b[ipCH4] -= rate_COmole.b_OP_CH4_CH4P_O;
	
	mole.c[ipOP][ipCH4P] += rate_COmole.OP_CH4_CH4P_O_1;
	mole.c[ipCH4][ipCH4P] += rate_COmole.OP_CH4_CH4P_O_2;
	mole.b[ipCH4P] += rate_COmole.b_OP_CH4_CH4P_O;
		
	mole.c[ipOP][ipATO] += rate_COmole.OP_CH4_CH4P_O_1;
	mole.c[ipCH4][ipATO] += rate_COmole.OP_CH4_CH4P_O_2;
	mole.b[ipATO] += rate_COmole.b_OP_CH4_CH4P_O;
	
	
	/* >>refer mol   CH4 CH  =>  CH3 CH2   UMIST    */
	
	mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_CH_CH3_CH2_1;
	mole.c[ipCH][ipCH4] -= rate_COmole.CH4_CH_CH3_CH2_2;
	mole.b[ipCH4] -= rate_COmole.b_CH4_CH_CH3_CH2;
	
	mole.c[ipCH4][ipCH] -= rate_COmole.CH4_CH_CH3_CH2_1;
	mole.c[ipCH][ipCH] -= rate_COmole.CH4_CH_CH3_CH2_2;
	mole.b[ipCH] -= rate_COmole.b_CH4_CH_CH3_CH2;
	
	mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CH_CH3_CH2_1;
	mole.c[ipCH][ipCH3] += rate_COmole.CH4_CH_CH3_CH2_2;
	mole.b[ipCH3] += rate_COmole.b_CH4_CH_CH3_CH2;
	
	mole.c[ipCH4][ipCH2] += rate_COmole.CH4_CH_CH3_CH2_1;
	mole.c[ipCH][ipCH2] += rate_COmole.CH4_CH_CH3_CH2_2;
	mole.b[ipCH2] += rate_COmole.b_CH4_CH_CH3_CH2;
	
	
	/* >>refer mol   CH5P CH  =>  CH4 CH2P   UMIST    */

	mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_CH_CH4_CH2P_1;
	mole.c[ipCH][ipCH5P] -= rate_COmole.CH5P_CH_CH4_CH2P_2;
	mole.b[ipCH5P] -= rate_COmole.b_CH5P_CH_CH4_CH2P;
	
	mole.c[ipCH5P][ipCH] -= rate_COmole.CH5P_CH_CH4_CH2P_1;
	mole.c[ipCH][ipCH] -= rate_COmole.CH5P_CH_CH4_CH2P_2;
	mole.b[ipCH] -= rate_COmole.b_CH5P_CH_CH4_CH2P;
	
	mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_CH_CH4_CH2P_1;
	mole.c[ipCH][ipCH4] += rate_COmole.CH5P_CH_CH4_CH2P_2;
	mole.b[ipCH4] += rate_COmole.b_CH5P_CH_CH4_CH2P;
	
	mole.c[ipCH5P][ipCH2P] += rate_COmole.CH5P_CH_CH4_CH2P_1;
	mole.c[ipCH][ipCH2P] += rate_COmole.CH5P_CH_CH4_CH2P_2;
	mole.b[ipCH2P] += rate_COmole.b_CH5P_CH_CH4_CH2P;
	
	
	/* >>refer mol   CH2 CH2  =>  CH3 CH   UMIST    */
	
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_CH3_CH_1;
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_CH3_CH_2;
	mole.b[ipCH2] -= rate_COmole.b_CH2_CH2_CH3_CH;
	
	mole.c[ipCH2][ipCH3] += 0.5*rate_COmole.CH2_CH2_CH3_CH_1;
	mole.c[ipCH2][ipCH3] += 0.5*rate_COmole.CH2_CH2_CH3_CH_2;
	mole.b[ipCH3] += 0.5*rate_COmole.b_CH2_CH2_CH3_CH;
	
	mole.c[ipCH2][ipCH] += 0.5*rate_COmole.CH2_CH2_CH3_CH_1;
	mole.c[ipCH2][ipCH] += 0.5*rate_COmole.CH2_CH2_CH3_CH_2;
	mole.b[ipCH] += 0.5*rate_COmole.b_CH2_CH2_CH3_CH;
	
	
	/* >>refer mol   CH4 CH2  =>  CH3 CH3   UMIST    */
	
	mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_CH2_CH3_CH3_1;
	mole.c[ipCH2][ipCH4] -= rate_COmole.CH4_CH2_CH3_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_CH4_CH2_CH3_CH3;
	
	mole.c[ipCH4][ipCH2] -= rate_COmole.CH4_CH2_CH3_CH3_1;
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH4_CH2_CH3_CH3_2;
	mole.b[ipCH2] -= rate_COmole.b_CH4_CH2_CH3_CH3;
	
	mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CH2_CH3_CH3_1;
	mole.c[ipCH2][ipCH3] += rate_COmole.CH4_CH2_CH3_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_CH4_CH2_CH3_CH3;
	
	mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CH2_CH3_CH3_1;
	mole.c[ipCH2][ipCH3] += rate_COmole.CH4_CH2_CH3_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_CH4_CH2_CH3_CH3;
	
	
	/* >>refer mol   OH CH2  =>  O CH3   UMIST    */
	
	mole.c[ipOH][ipOH] -= rate_COmole.OH_CH2_O_CH3_1;	
	mole.c[ipCH2][ipOH] -= rate_COmole.OH_CH2_O_CH3_2;
	mole.b[ipOH] -= rate_COmole.b_OH_CH2_O_CH3;
	
	mole.c[ipOH][ipCH2] -= rate_COmole.OH_CH2_O_CH3_1;
	mole.c[ipCH2][ipCH2] -= rate_COmole.OH_CH2_O_CH3_2;
	mole.b[ipCH2] -= rate_COmole.b_OH_CH2_O_CH3;
	
	mole.c[ipOH][ipCH3] += rate_COmole.OH_CH2_O_CH3_1;
	mole.c[ipCH2][ipCH3] += rate_COmole.OH_CH2_O_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_OH_CH2_O_CH3;
	
	mole.c[ipOH][ipATO] += rate_COmole.OH_CH2_O_CH3_1;
	mole.c[ipCH2][ipATO] += rate_COmole.OH_CH2_O_CH3_2;
	mole.b[ipATO] += rate_COmole.b_OH_CH2_O_CH3;
	
	
	/* >>refer mol   CH5P CH2  =>  CH4 CH3P   UMIST    */
	
	mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_CH2_CH4_CH3P_1;
	mole.c[ipCH2][ipCH5P] -= rate_COmole.CH5P_CH2_CH4_CH3P_2;
	mole.b[ipCH5P] -= rate_COmole.b_CH5P_CH2_CH4_CH3P;
	
	mole.c[ipCH5P][ipCH2] -= rate_COmole.CH5P_CH2_CH4_CH3P_1;
	mole.c[ipCH2][ipCH2] -= rate_COmole.CH5P_CH2_CH4_CH3P_2;
	mole.b[ipCH2] -= rate_COmole.b_CH5P_CH2_CH4_CH3P;
	
	mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_CH2_CH4_CH3P_1;
	mole.c[ipCH2][ipCH4] += rate_COmole.CH5P_CH2_CH4_CH3P_2;
	mole.b[ipCH4] += rate_COmole.b_CH5P_CH2_CH4_CH3P;
	
	mole.c[ipCH5P][ipCH3P] += rate_COmole.CH5P_CH2_CH4_CH3P_1;
	mole.c[ipCH2][ipCH3P] += rate_COmole.CH5P_CH2_CH4_CH3P_2;
	mole.b[ipCH3P] += rate_COmole.b_CH5P_CH2_CH4_CH3P;
	
	
	/* >>refer mol   OH CH3  =>  CH4 O   UMIST    */
	
	mole.c[ipOH][ipOH] -= rate_COmole.OH_CH3_CH4_O_1;
	mole.c[ipCH3][ipOH] -= rate_COmole.OH_CH3_CH4_O_2;
	mole.b[ipOH] -= rate_COmole.b_OH_CH3_CH4_O;
	
	mole.c[ipOH][ipCH3] -= rate_COmole.OH_CH3_CH4_O_1;
	mole.c[ipCH3][ipCH3] -= rate_COmole.OH_CH3_CH4_O_2;
	mole.b[ipCH3] -= rate_COmole.b_OH_CH3_CH4_O;
	
	mole.c[ipOH][ipCH4] += rate_COmole.OH_CH3_CH4_O_1;
	mole.c[ipCH3][ipCH4] += rate_COmole.OH_CH3_CH4_O_2;
	mole.b[ipCH4] += rate_COmole.b_OH_CH3_CH4_O;
		
	mole.c[ipOH][ipATO] += rate_COmole.OH_CH3_CH4_O_1;
	mole.c[ipCH3][ipATO] += rate_COmole.OH_CH3_CH4_O_2;
	mole.b[ipATO] += rate_COmole.b_OH_CH3_CH4_O;
	
	
	/* >>refer mol   OH CH3  =>  H2O CH2   UMIST    */
	
	mole.c[ipOH][ipOH] -= rate_COmole.OH_CH3_H2O_CH2_1;
	mole.c[ipCH3][ipOH] -= rate_COmole.OH_CH3_H2O_CH2_2;
	mole.b[ipOH] -= rate_COmole.b_OH_CH3_H2O_CH2;

	mole.c[ipOH][ipCH3] -= rate_COmole.OH_CH3_H2O_CH2_1;
	mole.c[ipCH3][ipCH3] -= rate_COmole.OH_CH3_H2O_CH2_2;
	mole.b[ipCH3] -= rate_COmole.b_OH_CH3_H2O_CH2;
	
	mole.c[ipOH][ipH2O] += rate_COmole.OH_CH3_H2O_CH2_1;
	mole.c[ipCH3][ipH2O] += rate_COmole.OH_CH3_H2O_CH2_2;
	mole.b[ipH2O] += rate_COmole.b_OH_CH3_H2O_CH2;
	
	mole.c[ipOH][ipCH2] += rate_COmole.OH_CH3_H2O_CH2_1;
	mole.c[ipCH3][ipCH2] += rate_COmole.OH_CH3_H2O_CH2_2;
	mole.b[ipCH2] += rate_COmole.b_OH_CH3_H2O_CH2;
	
	
	/* >>refer mol   H2O CH3  =>  OH CH4   UMIST    */
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_CH3_OH_CH4_1;
	mole.c[ipCH3][ipH2O] -= rate_COmole.H2O_CH3_OH_CH4_2;
	mole.b[ipH2O] -= rate_COmole.b_H2O_CH3_OH_CH4;
	
	mole.c[ipH2O][ipCH3] -= rate_COmole.H2O_CH3_OH_CH4_1;
	mole.c[ipCH3][ipCH3] -= rate_COmole.H2O_CH3_OH_CH4_2;
	mole.b[ipCH3] -= rate_COmole.b_H2O_CH3_OH_CH4;
	
	mole.c[ipH2O][ipOH] += rate_COmole.H2O_CH3_OH_CH4_1;
	mole.c[ipCH3][ipOH] += rate_COmole.H2O_CH3_OH_CH4_2;
	mole.b[ipOH] += rate_COmole.b_H2O_CH3_OH_CH4;
	
	mole.c[ipH2O][ipCH4] += rate_COmole.H2O_CH3_OH_CH4_1;
	mole.c[ipCH3][ipCH4] += rate_COmole.H2O_CH3_OH_CH4_2;
	mole.b[ipCH4] += rate_COmole.b_H2O_CH3_OH_CH4;


	/* >>refer mol   CH3 CH3  =>  CH4 CH2   UMIST    */
	
	mole.c[ipCH3][ipCH3] -= rate_COmole.CH3_CH3_CH4_CH2_1;
	mole.c[ipCH3][ipCH3] -= rate_COmole.CH3_CH3_CH4_CH2_2;
	mole.b[ipCH3] -= rate_COmole.b_CH3_CH3_CH4_CH2;
	
	mole.c[ipCH3][ipCH4] += 0.5*rate_COmole.CH3_CH3_CH4_CH2_1;
	mole.c[ipCH3][ipCH4] += 0.5*rate_COmole.CH3_CH3_CH4_CH2_2;
	mole.b[ipCH4] += 0.5*rate_COmole.b_CH3_CH3_CH4_CH2;
	
	mole.c[ipCH3][ipCH2] += 0.5*rate_COmole.CH3_CH3_CH4_CH2_1;
	mole.c[ipCH3][ipCH2] += 0.5*rate_COmole.CH3_CH3_CH4_CH2_2;
	mole.b[ipCH2] += 0.5*rate_COmole.b_CH3_CH3_CH4_CH2;
	
	
	/* >>refer mol   OH CH4  =>  H2O CH3   UMIST    */

	mole.c[ipOH][ipOH] -= rate_COmole.OH_CH4_H2O_CH3_1;
	mole.c[ipCH4][ipOH] -= rate_COmole.OH_CH4_H2O_CH3_2;
	mole.b[ipOH] -= rate_COmole.b_OH_CH4_H2O_CH3;
	
	mole.c[ipOH][ipCH4] -= rate_COmole.OH_CH4_H2O_CH3_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.OH_CH4_H2O_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_OH_CH4_H2O_CH3;
	
	mole.c[ipOH][ipH2O] += rate_COmole.OH_CH4_H2O_CH3_1;
	mole.c[ipCH4][ipH2O] += rate_COmole.OH_CH4_H2O_CH3_2;
	mole.b[ipH2O] += rate_COmole.b_OH_CH4_H2O_CH3;
	
	mole.c[ipOH][ipCH3] += rate_COmole.OH_CH4_H2O_CH3_1;
	mole.c[ipCH4][ipCH3] += rate_COmole.OH_CH4_H2O_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_OH_CH4_H2O_CH3;
	
	
	/* >>refer mol   OHP CH4  =>  CH5P O   UMIST    */
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_CH4_CH5P_O_1;
	mole.c[ipCH4][ipOHP] -= rate_COmole.OHP_CH4_CH5P_O_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_CH4_CH5P_O;
	
	mole.c[ipOHP][ipCH4] -= rate_COmole.OHP_CH4_CH5P_O_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.OHP_CH4_CH5P_O_2;
	mole.b[ipCH4] -= rate_COmole.b_OHP_CH4_CH5P_O;
	
	mole.c[ipOHP][ipCH5P] += rate_COmole.OHP_CH4_CH5P_O_1;
	mole.c[ipCH4][ipCH5P] += rate_COmole.OHP_CH4_CH5P_O_2;
	mole.b[ipCH5P] += rate_COmole.b_OHP_CH4_CH5P_O;
	
	mole.c[ipOHP][ipATO] += rate_COmole.OHP_CH4_CH5P_O_1;
	mole.c[ipCH4][ipATO] += rate_COmole.OHP_CH4_CH5P_O_2;
	mole.b[ipATO] += rate_COmole.b_OHP_CH4_CH5P_O;
	
	
	/* >>refer mol   OHP CH4  =>  H3OP CH2   UMIST    */
	
	mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_CH4_H3OP_CH2_1;
	mole.c[ipCH4][ipOHP] -= rate_COmole.OHP_CH4_H3OP_CH2_2;
	mole.b[ipOHP] -= rate_COmole.b_OHP_CH4_H3OP_CH2;
	
	mole.c[ipOHP][ipCH4] -= rate_COmole.OHP_CH4_H3OP_CH2_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.OHP_CH4_H3OP_CH2_2;
	mole.b[ipCH4] -= rate_COmole.b_OHP_CH4_H3OP_CH2;
	
	mole.c[ipOHP][ipH3OP] += rate_COmole.OHP_CH4_H3OP_CH2_1;
	mole.c[ipCH4][ipH3OP] += rate_COmole.OHP_CH4_H3OP_CH2_2;
	mole.b[ipH3OP] += rate_COmole.b_OHP_CH4_H3OP_CH2;
	
	mole.c[ipOHP][ipCH2] += rate_COmole.OHP_CH4_H3OP_CH2_1;
	mole.c[ipCH4][ipCH2] += rate_COmole.OHP_CH4_H3OP_CH2_2;
	mole.b[ipCH2] += rate_COmole.b_OHP_CH4_H3OP_CH2;
	
	
	/* >>refer mol   H2OP CH4  =>  H3OP CH3   UMIST    */
	
	mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_CH4_H3OP_CH3_1;
	mole.c[ipCH4][ipH2OP] -= rate_COmole.H2OP_CH4_H3OP_CH3_2;
	mole.b[ipH2OP] -= rate_COmole.b_H2OP_CH4_H3OP_CH3;
	
	mole.c[ipH2OP][ipCH4] -= rate_COmole.H2OP_CH4_H3OP_CH3_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.H2OP_CH4_H3OP_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_H2OP_CH4_H3OP_CH3;
	
	mole.c[ipH2OP][ipH3OP] += rate_COmole.H2OP_CH4_H3OP_CH3_1;
	mole.c[ipCH4][ipH3OP] += rate_COmole.H2OP_CH4_H3OP_CH3_2;
	mole.b[ipH3OP] += rate_COmole.b_H2OP_CH4_H3OP_CH3;
	
	mole.c[ipH2OP][ipCH3] += rate_COmole.H2OP_CH4_H3OP_CH3_1;
	mole.c[ipCH4][ipCH3] += rate_COmole.H2OP_CH4_H3OP_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_H2OP_CH4_H3OP_CH3;
	
	
	/* >>refer mol   COP CH4  =>  HCOP CH3   UMIST    */
	
	mole.c[ipCOP][ipCOP] -= rate_COmole.COP_CH4_HCOP_CH3_1;
	mole.c[ipCH4][ipCOP] -= rate_COmole.COP_CH4_HCOP_CH3_2;
	mole.b[ipCOP] -= rate_COmole.b_COP_CH4_HCOP_CH3;
	
	mole.c[ipCOP][ipCH4] -= rate_COmole.COP_CH4_HCOP_CH3_1;
	mole.c[ipCH4][ipCH4] -= rate_COmole.COP_CH4_HCOP_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_COP_CH4_HCOP_CH3;
	
	mole.c[ipCOP][ipHCOP] += rate_COmole.COP_CH4_HCOP_CH3_1;
	mole.c[ipCH4][ipHCOP] += rate_COmole.COP_CH4_HCOP_CH3_2;
	mole.b[ipHCOP] += rate_COmole.b_COP_CH4_HCOP_CH3;
	
	mole.c[ipCOP][ipCH3] += rate_COmole.COP_CH4_HCOP_CH3_1;
	mole.c[ipCH4][ipCH3] += rate_COmole.COP_CH4_HCOP_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_COP_CH4_HCOP_CH3;
	
	/* >>refer mol   CH4 CH4P  =>  CH5P CH3   UMIST    */
	
	mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_CH4P_CH5P_CH3_1;
	mole.c[ipCH4P][ipCH4] -= rate_COmole.CH4_CH4P_CH5P_CH3_2;
	mole.b[ipCH4] -= rate_COmole.b_CH4_CH4P_CH5P_CH3;
	
	mole.c[ipCH4][ipCH4P] -= rate_COmole.CH4_CH4P_CH5P_CH3_1;
	mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4_CH4P_CH5P_CH3_2;
	mole.b[ipCH4P] -= rate_COmole.b_CH4_CH4P_CH5P_CH3;
	
	mole.c[ipCH4][ipCH5P] += rate_COmole.CH4_CH4P_CH5P_CH3_1;
	mole.c[ipCH4P][ipCH5P] += rate_COmole.CH4_CH4P_CH5P_CH3_2;
	mole.b[ipCH5P] += rate_COmole.b_CH4_CH4P_CH5P_CH3;
	
	mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CH4P_CH5P_CH3_1;
	mole.c[ipCH4P][ipCH3] += rate_COmole.CH4_CH4P_CH5P_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_CH4_CH4P_CH5P_CH3;
	
	
	/* >>refer mol   H2O CH4P  =>  H3OP CH3   UMIST    */
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_CH4P_H3OP_CH3_1;
	mole.c[ipCH4P][ipH2O] -= rate_COmole.H2O_CH4P_H3OP_CH3_2;
	mole.b[ipH2O] -= rate_COmole.b_H2O_CH4P_H3OP_CH3;
	
	mole.c[ipH2O][ipCH4P] -= rate_COmole.H2O_CH4P_H3OP_CH3_1;
	mole.c[ipCH4P][ipCH4P] -= rate_COmole.H2O_CH4P_H3OP_CH3_2;
	mole.b[ipCH4P] -= rate_COmole.b_H2O_CH4P_H3OP_CH3;
	
	mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_CH4P_H3OP_CH3_1;
	mole.c[ipCH4P][ipH3OP] += rate_COmole.H2O_CH4P_H3OP_CH3_2;
	mole.b[ipH3OP] += rate_COmole.b_H2O_CH4P_H3OP_CH3;
	
	mole.c[ipH2O][ipCH3] += rate_COmole.H2O_CH4P_H3OP_CH3_1;
	mole.c[ipCH4P][ipCH3] += rate_COmole.H2O_CH4P_H3OP_CH3_2;
	mole.b[ipCH3] += rate_COmole.b_H2O_CH4P_H3OP_CH3;
	
	
	/* >>refer mol   O2 CH4P  =>  O2P CH4   UMIST    */
	
	mole.c[ipO2][ipO2] -= rate_COmole.O2_CH4P_O2P_CH4_1;
	mole.c[ipCH4P][ipO2] -= rate_COmole.O2_CH4P_O2P_CH4_2;
	mole.b[ipO2] -= rate_COmole.b_O2_CH4P_O2P_CH4;
	
	mole.c[ipO2][ipCH4P] -= rate_COmole.O2_CH4P_O2P_CH4_1;
	mole.c[ipCH4P][ipCH4P] -= rate_COmole.O2_CH4P_O2P_CH4_2;
	mole.b[ipCH4P] -= rate_COmole.b_O2_CH4P_O2P_CH4;
	
	mole.c[ipO2][ipO2P] += rate_COmole.O2_CH4P_O2P_CH4_1;
	mole.c[ipCH4P][ipO2P] += rate_COmole.O2_CH4P_O2P_CH4_2;
	mole.b[ipO2P] += rate_COmole.b_O2_CH4P_O2P_CH4;
	
	mole.c[ipO2][ipCH4] += rate_COmole.O2_CH4P_O2P_CH4_1;
	mole.c[ipCH4P][ipCH4] += rate_COmole.O2_CH4P_O2P_CH4_2;
	mole.b[ipCH4] += rate_COmole.b_O2_CH4P_O2P_CH4;
	
	/* >>refer mol   H2O CH5P  =>  H3OP CH4   UMIST    */
	
	mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_CH5P_H3OP_CH4_1;
	mole.c[ipCH5P][ipH2O] -= rate_COmole.H2O_CH5P_H3OP_CH4_2;
	mole.b[ipH2O] -= rate_COmole.b_H2O_CH5P_H3OP_CH4;
	
	mole.c[ipH2O][ipCH5P] -= rate_COmole.H2O_CH5P_H3OP_CH4_1;
	mole.c[ipCH5P][ipCH5P] -= rate_COmole.H2O_CH5P_H3OP_CH4_2;
	mole.b[ipCH5P] -= rate_COmole.b_H2O_CH5P_H3OP_CH4;

	mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_CH5P_H3OP_CH4_1;
	mole.c[ipCH5P][ipH3OP] += rate_COmole.H2O_CH5P_H3OP_CH4_2;
	mole.b[ipH3OP] += rate_COmole.b_H2O_CH5P_H3OP_CH4;
	
	mole.c[ipH2O][ipCH4] += rate_COmole.H2O_CH5P_H3OP_CH4_1;
	mole.c[ipCH5P][ipCH4] += rate_COmole.H2O_CH5P_H3OP_CH4_2;
	mole.b[ipCH4] += rate_COmole.b_H2O_CH5P_H3OP_CH4;
	
	
	/* >>refer mol   CH5P OH  =>  H2OP CH4   UMIST    */
	
	mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_OH_H2OP_CH4_1;
	mole.c[ipOH][ipCH5P] -= rate_COmole.CH5P_OH_H2OP_CH4_2;
	mole.b[ipCH5P] -= rate_COmole.b_CH5P_OH_H2OP_CH4;
	
	mole.c[ipCH5P][ipOH] -= rate_COmole.CH5P_OH_H2OP_CH4_1;
	mole.c[ipOH][ipOH] -= rate_COmole.CH5P_OH_H2OP_CH4_2;
	mole.b[ipOH] -= rate_COmole.b_CH5P_OH_H2OP_CH4;
	
	mole.c[ipCH5P][ipH2OP] += rate_COmole.CH5P_OH_H2OP_CH4_1;
	mole.c[ipOH][ipH2OP] += rate_COmole.CH5P_OH_H2OP_CH4_2;
	mole.b[ipH2OP] += rate_COmole.b_CH5P_OH_H2OP_CH4;
	
	mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_OH_H2OP_CH4_1;
	mole.c[ipOH][ipCH4] += rate_COmole.CH5P_OH_H2OP_CH4_2;
	mole.b[ipCH4] += rate_COmole.b_CH5P_OH_H2OP_CH4;


		/*  >>refer   mol  C      +  NH    =>   N     +     CH    UMIST  */

			mole.c[ipNH][ipNH] -= rate_COmole.C_NH_N_CH_1;
			mole.c[ipATC][ipNH] -= rate_COmole.C_NH_N_CH_2;
			mole.b[ipNH] -= rate_COmole.b_C_NH_N_CH;

			mole.c[ipNH][ipATC] -= rate_COmole.C_NH_N_CH_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NH_N_CH_2;
			mole.b[ipATC] -= rate_COmole.b_C_NH_N_CH;
			
			mole.c[ipNH][ipCH] += rate_COmole.C_NH_N_CH_1;
			mole.c[ipATC][ipCH] += rate_COmole.C_NH_N_CH_2;
			mole.b[ipCH] += rate_COmole.b_C_NH_N_CH;
			
			mole.c[ipNH][ipATN] += rate_COmole.C_NH_N_CH_1;
			mole.c[ipATC][ipATN] += rate_COmole.C_NH_N_CH_2;
			mole.b[ipATN] += rate_COmole.b_C_NH_N_CH;

		/*  >>refer   mol  C      +  NH    =>   CN    +     H	    UMIST  */

			mole.c[ipNH][ipNH] -= rate_COmole.C_NH_CN_H_1;
			mole.c[ipATC][ipNH] -= rate_COmole.C_NH_CN_H_2;
			mole.b[ipNH] -= rate_COmole.b_C_NH_CN_H;

			mole.c[ipNH][ipATC] -= rate_COmole.C_NH_CN_H_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NH_CN_H_2;
			mole.b[ipATC] -= rate_COmole.b_C_NH_CN_H;
			
			mole.c[ipNH][ipCN] += rate_COmole.C_NH_CN_H_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_NH_CN_H_2;
			mole.b[ipCN] += rate_COmole.b_C_NH_CN_H;

		/*  >>refer   mol  C      +  N2    =>   CN    +     N	    UMIST  */

			mole.c[ipN2][ipN2] -= rate_COmole.C_N2_CN_N_1;
			mole.c[ipATC][ipN2] -= rate_COmole.C_N2_CN_N_2;
			mole.b[ipN2] -= rate_COmole.b_C_N2_CN_N;

			mole.c[ipN2][ipATC] -= rate_COmole.C_N2_CN_N_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_N2_CN_N_2;
			mole.b[ipATC] -= rate_COmole.b_C_N2_CN_N;
			
			mole.c[ipN2][ipCN] += rate_COmole.C_N2_CN_N_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_N2_CN_N_2;
			mole.b[ipCN] += rate_COmole.b_C_N2_CN_N;
			
			mole.c[ipN2][ipATN] += rate_COmole.C_N2_CN_N_1;
			mole.c[ipATC][ipATN] += rate_COmole.C_N2_CN_N_2;
			mole.b[ipATN] += rate_COmole.b_C_N2_CN_N;

		/*  >>refer   mol  C      +  NO    =>   CN    +     O	    UMIST  */

			mole.c[ipNO][ipNO] -= rate_COmole.C_NO_CN_O_1;
			mole.c[ipATC][ipNO] -= rate_COmole.C_NO_CN_O_2;
			mole.b[ipNO] -= rate_COmole.b_C_NO_CN_O;
			
			mole.c[ipNO][ipATC] -= rate_COmole.C_NO_CN_O_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NO_CN_O_2;
			mole.b[ipATC] -= rate_COmole.b_C_NO_CN_O;
			
			mole.c[ipNO][ipCN] += rate_COmole.C_NO_CN_O_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_NO_CN_O_2;
			mole.b[ipCN] += rate_COmole.b_C_NO_CN_O;

			mole.c[ipNO][ipATO] += rate_COmole.C_NO_CN_O_1;
			mole.c[ipATC][ipATO] += rate_COmole.C_NO_CN_O_2;
			mole.b[ipATO] += rate_COmole.b_C_NO_CN_O;

			/*  >>refer   mol  C      +  HS    =>   S     +     CH    UMIST  */

			mole.c[ipHS][ipHS] -= rate_COmole.C_HS_S_CH_1;
			mole.c[ipATC][ipHS] -= rate_COmole.C_HS_S_CH_2;
			mole.b[ipHS] -= rate_COmole.b_C_HS_S_CH;
	
			mole.c[ipHS][ipATC] -= rate_COmole.C_HS_S_CH_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_HS_S_CH_2;
			mole.b[ipATC] -= rate_COmole.b_C_HS_S_CH;
			
			mole.c[ipHS][ipCH] += rate_COmole.C_HS_S_CH_1;
			mole.c[ipATC][ipCH] += rate_COmole.C_HS_S_CH_2;
			mole.b[ipCH] += rate_COmole.b_C_HS_S_CH;
		
			mole.c[ipHS][ipATS] += rate_COmole.C_HS_S_CH_1;
			mole.c[ipATC][ipATS] += rate_COmole.C_HS_S_CH_2;
			mole.b[ipATS] += rate_COmole.b_C_HS_S_CH;

		/*  >>refer   mol  C      +  HS    =>   CS    +     H	    UMIST  */

			mole.c[ipHS][ipHS] -= rate_COmole.C_HS_CS_H_1;
			mole.c[ipATC][ipHS] -= rate_COmole.C_HS_CS_H_2;
			mole.b[ipHS] -= rate_COmole.b_C_HS_CS_H;

			mole.c[ipHS][ipATC] -= rate_COmole.C_HS_CS_H_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_HS_CS_H_2;
			mole.b[ipATC] -= rate_COmole.b_C_HS_CS_H;
			
			mole.c[ipHS][ipCS] += rate_COmole.C_HS_CS_H_1;
			mole.c[ipATC][ipCS] += rate_COmole.C_HS_CS_H_2;
			mole.b[ipCS] += rate_COmole.b_C_HS_CS_H;

		/*  >>refer   mol  C      +  NS    =>   S     +     CN    UMIST  */

			mole.c[ipNS][ipATC] -= rate_COmole.C_NS_S_CN_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NS_S_CN_2;
			mole.b[ipATC] -= rate_COmole.b_C_NS_S_CN;
			
			mole.c[ipNS][ipNS] -= rate_COmole.C_NS_S_CN_1;
			mole.c[ipATC][ipNS] -= rate_COmole.C_NS_S_CN_2;
			mole.b[ipNS] -= rate_COmole.b_C_NS_S_CN;
			
			mole.c[ipNS][ipCN] += rate_COmole.C_NS_S_CN_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_NS_S_CN_2;
			mole.b[ipCN] += rate_COmole.b_C_NS_S_CN;

			mole.c[ipNS][ipATS] += rate_COmole.C_NS_S_CN_1;
			mole.c[ipATC][ipATS] += rate_COmole.C_NS_S_CN_2;
			mole.b[ipATS] += rate_COmole.b_C_NS_S_CN;

		/*  >>refer   mol  C      +  NS    =>   CS    +     N	    UMIST  */

			mole.c[ipNS][ipATC] -= rate_COmole.C_NS_CS_N_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NS_CS_N_2;
			mole.b[ipATC] -= rate_COmole.b_C_NS_CS_N;
			
			mole.c[ipNS][ipNS] -= rate_COmole.C_NS_CS_N_1;
			mole.c[ipATC][ipNS] -= rate_COmole.C_NS_CS_N_2;
			mole.b[ipNS] -= rate_COmole.b_C_NS_CS_N;
			
			mole.c[ipNS][ipCS] += rate_COmole.C_NS_CS_N_1;
			mole.c[ipATC][ipCS] += rate_COmole.C_NS_CS_N_2;
			mole.b[ipCS] += rate_COmole.b_C_NS_CS_N;

			mole.c[ipNS][ipATN] += rate_COmole.C_NS_CS_N_1;
			mole.c[ipATC][ipATN] += rate_COmole.C_NS_CS_N_2;
			mole.b[ipATN] += rate_COmole.b_C_NS_CS_N;

		/*  >>refer   mol  C      +  S2    =>   CS    +     S	    UMIST  */

			mole.c[ipS2][ipATC] -= rate_COmole.C_S2_CS_S_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_S2_CS_S_2;
			mole.b[ipATC] -= rate_COmole.b_C_S2_CS_S;
			
			mole.c[ipS2][ipS2] -= rate_COmole.C_S2_CS_S_1;
			mole.c[ipATC][ipS2] -= rate_COmole.C_S2_CS_S_2;
			mole.b[ipS2] -= rate_COmole.b_C_S2_CS_S;
			
			mole.c[ipS2][ipCS] += rate_COmole.C_S2_CS_S_1;
			mole.c[ipATC][ipCS] += rate_COmole.C_S2_CS_S_2;
			mole.b[ipCS] += rate_COmole.b_C_S2_CS_S;

			mole.c[ipS2][ipATS] += rate_COmole.C_S2_CS_S_1;
			mole.c[ipATC][ipATS] += rate_COmole.C_S2_CS_S_2;
			mole.b[ipATS] += rate_COmole.b_C_S2_CS_S;

		/*  >>refer   mol  CH     +  N     =>   NH    +     C	    UMIST  */

			mole.c[ipATN][ipCH] -= rate_COmole.CH_N_NH_C_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_N_NH_C_2;
			mole.b[ipCH] -= rate_COmole.b_CH_N_NH_C;
			
			mole.c[ipATN][ipATN] -= rate_COmole.CH_N_NH_C_1;
			mole.c[ipCH][ipATN] -= rate_COmole.CH_N_NH_C_2;
			mole.b[ipATN] -= rate_COmole.b_CH_N_NH_C;
			
			mole.c[ipATN][ipNH] += rate_COmole.CH_N_NH_C_1;
			mole.c[ipCH][ipNH] += rate_COmole.CH_N_NH_C_2;
			mole.b[ipNH] += rate_COmole.b_CH_N_NH_C;

			mole.c[ipATN][ipATC] += rate_COmole.CH_N_NH_C_1;
			mole.c[ipCH][ipATC] += rate_COmole.CH_N_NH_C_2;
			mole.b[ipATC] += rate_COmole.b_CH_N_NH_C;

		/*  >>refer   mol  CH     +  N     =>   CN    +     H	    UMIST  */

			mole.c[ipATN][ipCH] -= rate_COmole.CH_N_CN_H_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_N_CN_H_2;
			mole.b[ipCH] -= rate_COmole.b_CH_N_CN_H;

			mole.c[ipATN][ipATN] -= rate_COmole.CH_N_CN_H_1;
			mole.c[ipCH][ipATN] -= rate_COmole.CH_N_CN_H_2;
			mole.b[ipATN] -= rate_COmole.b_CH_N_CN_H;
			
			mole.c[ipATN][ipCN] += rate_COmole.CH_N_CN_H_1;
			mole.c[ipCH][ipCN] += rate_COmole.CH_N_CN_H_2;
			mole.b[ipCN] += rate_COmole.b_CH_N_CN_H;

		/*  >>refer   mol  CH     +  N2    =>   HCN   +     N	    UMIST  */

			mole.c[ipN2][ipCH] -= rate_COmole.CH_N2_HCN_N_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_N2_HCN_N_2;
			mole.b[ipCH] -= rate_COmole.b_CH_N2_HCN_N;
			
			mole.c[ipN2][ipN2] -= rate_COmole.CH_N2_HCN_N_1;
			mole.c[ipCH][ipN2] -= rate_COmole.CH_N2_HCN_N_2;
			mole.b[ipN2] -= rate_COmole.b_CH_N2_HCN_N;
			
			mole.c[ipN2][ipHCN] += rate_COmole.CH_N2_HCN_N_1;
			mole.c[ipCH][ipHCN] += rate_COmole.CH_N2_HCN_N_2;
			mole.b[ipHCN] += rate_COmole.b_CH_N2_HCN_N;
			
			mole.c[ipN2][ipATN] += rate_COmole.CH_N2_HCN_N_1;
			mole.c[ipCH][ipATN] += rate_COmole.CH_N2_HCN_N_2;
			mole.b[ipATN] += rate_COmole.b_CH_N2_HCN_N;

		/*  >>refer   mol  CH     +  NO    =>   HCN   +     O	    UMIST  */

			mole.c[ipNO][ipCH] -= rate_COmole.CH_NO_HCN_O_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NO_HCN_O_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NO_HCN_O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH_NO_HCN_O_1;
			mole.c[ipCH][ipNO] -= rate_COmole.CH_NO_HCN_O_2;
			mole.b[ipNO] -= rate_COmole.b_CH_NO_HCN_O;
			
			mole.c[ipNO][ipHCN] += rate_COmole.CH_NO_HCN_O_1;
			mole.c[ipCH][ipHCN] += rate_COmole.CH_NO_HCN_O_2;
			mole.b[ipHCN] += rate_COmole.b_CH_NO_HCN_O;
			
			mole.c[ipNO][ipATO] += rate_COmole.CH_NO_HCN_O_1;
			mole.c[ipCH][ipATO] += rate_COmole.CH_NO_HCN_O_2;
			mole.b[ipATO] += rate_COmole.b_CH_NO_HCN_O;

		/*  >>refer   mol  CH     +  NO    =>   CN    +     OH    UMIST  */

			mole.c[ipNO][ipCH] -= rate_COmole.CH_NO_CN_OH_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NO_CN_OH_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NO_CN_OH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH_NO_CN_OH_1;
			mole.c[ipCH][ipNO] -= rate_COmole.CH_NO_CN_OH_2;
			mole.b[ipNO] -= rate_COmole.b_CH_NO_CN_OH;
			
			mole.c[ipNO][ipCN] += rate_COmole.CH_NO_CN_OH_1;
			mole.c[ipCH][ipCN] += rate_COmole.CH_NO_CN_OH_2;
			mole.b[ipCN] += rate_COmole.b_CH_NO_CN_OH;
			
			mole.c[ipNO][ipOH] += rate_COmole.CH_NO_CN_OH_1;
			mole.c[ipCH][ipOH] += rate_COmole.CH_NO_CN_OH_2;
			mole.b[ipOH] += rate_COmole.b_CH_NO_CN_OH;

		/*  >>refer   mol  CH     +  NO    =>   OCN   +     H	    UMIST  */

			mole.c[ipNO][ipCH] -= rate_COmole.CH_NO_OCN_H_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NO_OCN_H_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NO_OCN_H;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH_NO_OCN_H_1;
			mole.c[ipCH][ipNO] -= rate_COmole.CH_NO_OCN_H_2;
			mole.b[ipNO] -= rate_COmole.b_CH_NO_OCN_H;
			
			mole.c[ipNO][ipOCN] += rate_COmole.CH_NO_OCN_H_1;
			mole.c[ipCH][ipOCN] += rate_COmole.CH_NO_OCN_H_2;
			mole.b[ipOCN] += rate_COmole.b_CH_NO_OCN_H;

		/*  >>refer   mol  CH     +  HNO   =>   NO    +     CH2   UMIST  */

			mole.c[ipHNO][ipCH] -= rate_COmole.CH_HNO_NO_CH2_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_HNO_NO_CH2_2;
			mole.b[ipCH] -= rate_COmole.b_CH_HNO_NO_CH2;
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.CH_HNO_NO_CH2_1;
			mole.c[ipCH][ipHNO] -= rate_COmole.CH_HNO_NO_CH2_2;
			mole.b[ipHNO] -= rate_COmole.b_CH_HNO_NO_CH2;
			
			mole.c[ipHNO][ipNO] += rate_COmole.CH_HNO_NO_CH2_1;
			mole.c[ipCH][ipNO] += rate_COmole.CH_HNO_NO_CH2_2;
			mole.b[ipNO] += rate_COmole.b_CH_HNO_NO_CH2;
			
			mole.c[ipHNO][ipCH2] += rate_COmole.CH_HNO_NO_CH2_1;
			mole.c[ipCH][ipCH2] += rate_COmole.CH_HNO_NO_CH2_2;
			mole.b[ipCH2] += rate_COmole.b_CH_HNO_NO_CH2;

		/*  >>refer   mol  CH     +  S     =>   HS    +     C	    UMIST  */

			mole.c[ipATS][ipCH] -= rate_COmole.CH_S_HS_C_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_S_HS_C_2;
			mole.b[ipCH] -= rate_COmole.b_CH_S_HS_C;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CH_S_HS_C_1;
			mole.c[ipCH][ipATS] -= rate_COmole.CH_S_HS_C_2;
			mole.b[ipATS] -= rate_COmole.b_CH_S_HS_C;
			
			mole.c[ipATS][ipHS] += rate_COmole.CH_S_HS_C_1;
			mole.c[ipCH][ipHS] += rate_COmole.CH_S_HS_C_2;
			mole.b[ipHS] += rate_COmole.b_CH_S_HS_C;

			mole.c[ipATS][ipATC] += rate_COmole.CH_S_HS_C_1;
			mole.c[ipCH][ipATC] += rate_COmole.CH_S_HS_C_2;
			mole.b[ipATC] += rate_COmole.b_CH_S_HS_C;

		/*  >>refer   mol  CH     +  S     =>   CS    +     H	    UMIST  */

			mole.c[ipATS][ipCH] -= rate_COmole.CH_S_CS_H_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_S_CS_H_2;
			mole.b[ipCH] -= rate_COmole.b_CH_S_CS_H;

			mole.c[ipATS][ipATS] -= rate_COmole.CH_S_CS_H_1;
			mole.c[ipCH][ipATS] -= rate_COmole.CH_S_CS_H_2;
			mole.b[ipATS] -= rate_COmole.b_CH_S_CS_H;
			
			mole.c[ipATS][ipCS] += rate_COmole.CH_S_CS_H_1;
			mole.c[ipCH][ipCS] += rate_COmole.CH_S_CS_H_2;
			mole.b[ipCS] += rate_COmole.b_CH_S_CS_H;

		/*  >>refer   mol  N      +  NH    =>   N2    +     H	    UMIST  */

			mole.c[ipNH][ipNH] -= rate_COmole.N_NH_N2_H_1;
			mole.c[ipATN][ipNH] -= rate_COmole.N_NH_N2_H_2;
			mole.b[ipNH] -= rate_COmole.b_N_NH_N2_H;

			mole.c[ipNH][ipATN] -= rate_COmole.N_NH_N2_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_NH_N2_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_NH_N2_H;
			
			mole.c[ipNH][ipN2] += rate_COmole.N_NH_N2_H_1;
			mole.c[ipATN][ipN2] += rate_COmole.N_NH_N2_H_2;
			mole.b[ipN2] += rate_COmole.b_N_NH_N2_H;

		/*  >>refer   mol  N      +  CH3   =>   HCN   +     H  +  H      UMIST  */

			mole.c[ipCH3][ipCH3] -= rate_COmole.N_CH3_HCN_H_H_1;
			mole.c[ipATN][ipCH3] -= rate_COmole.N_CH3_HCN_H_H_2;
			mole.b[ipCH3] -= rate_COmole.b_N_CH3_HCN_H_H;

			mole.c[ipCH3][ipATN] -= rate_COmole.N_CH3_HCN_H_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_CH3_HCN_H_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_CH3_HCN_H_H;
			
			mole.c[ipCH3][ipHCN] += rate_COmole.N_CH3_HCN_H_H_1;
			mole.c[ipATN][ipHCN] += rate_COmole.N_CH3_HCN_H_H_2;
			mole.b[ipHCN] += rate_COmole.b_N_CH3_HCN_H_H;

		/*  >>refer   mol  N      +  CH3   =>   HCN   +     H2    UMIST  */

			mole.c[ipCH3][ipCH3] -= rate_COmole.N_CH3_HCN_H2_1;
			mole.c[ipATN][ipCH3] -= rate_COmole.N_CH3_HCN_H2_2;
			mole.b[ipCH3] -= rate_COmole.b_N_CH3_HCN_H2;

			mole.c[ipCH3][ipATN] -= rate_COmole.N_CH3_HCN_H2_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_CH3_HCN_H2_2;
			mole.b[ipATN] -= rate_COmole.b_N_CH3_HCN_H2;
			
			mole.c[ipCH3][ipHCN] += rate_COmole.N_CH3_HCN_H2_1;
			mole.c[ipATN][ipHCN] += rate_COmole.N_CH3_HCN_H2_2;
			mole.b[ipHCN] += rate_COmole.b_N_CH3_HCN_H2;

		/*  >>refer   mol  N      +  OH    =>   O     +     NH    UMIST  */

			mole.c[ipOH][ipATN] -= rate_COmole.N_OH_O_NH_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_OH_O_NH_2;
			mole.b[ipATN] -= rate_COmole.b_N_OH_O_NH;
			
			mole.c[ipOH][ipOH] -= rate_COmole.N_OH_O_NH_1;
			mole.c[ipATN][ipOH] -= rate_COmole.N_OH_O_NH_2;
			mole.b[ipOH] -= rate_COmole.b_N_OH_O_NH;
			
			mole.c[ipOH][ipNH] += rate_COmole.N_OH_O_NH_1;
			mole.c[ipATN][ipNH] += rate_COmole.N_OH_O_NH_2;
			mole.b[ipNH] += rate_COmole.b_N_OH_O_NH;

			mole.c[ipOH][ipATO] += rate_COmole.N_OH_O_NH_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_OH_O_NH_2;
			mole.b[ipATO] += rate_COmole.b_N_OH_O_NH;

		/*  >>refer   mol  N      +  OH    =>   NO    +     H	    UMIST  */

			mole.c[ipOH][ipOH] -= rate_COmole.N_OH_NO_H_1;
			mole.c[ipATN][ipOH] -= rate_COmole.N_OH_NO_H_2;
			mole.b[ipOH] -= rate_COmole.b_N_OH_NO_H;

			mole.c[ipOH][ipATN] -= rate_COmole.N_OH_NO_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_OH_NO_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_OH_NO_H;
			
			mole.c[ipOH][ipNO] += rate_COmole.N_OH_NO_H_1;
			mole.c[ipATN][ipNO] += rate_COmole.N_OH_NO_H_2;
			mole.b[ipNO] += rate_COmole.b_N_OH_NO_H;

		/*  >>refer   mol  N      +  CN    =>   N2    +     C	    UMIST  */

			mole.c[ipCN][ipCN] -= rate_COmole.N_CN_N2_C_1;
			mole.c[ipATN][ipCN] -= rate_COmole.N_CN_N2_C_2;
			mole.b[ipCN] -= rate_COmole.b_N_CN_N2_C;
			
			mole.c[ipCN][ipATN] -= rate_COmole.N_CN_N2_C_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_CN_N2_C_2;
			mole.b[ipATN] -= rate_COmole.b_N_CN_N2_C;
			
			mole.c[ipCN][ipN2] += rate_COmole.N_CN_N2_C_1;
			mole.c[ipATN][ipN2] += rate_COmole.N_CN_N2_C_2;
			mole.b[ipN2] += rate_COmole.b_N_CN_N2_C;

			mole.c[ipCN][ipATC] += rate_COmole.N_CN_N2_C_1;
			mole.c[ipATN][ipATC] += rate_COmole.N_CN_N2_C_2;
			mole.b[ipATC] += rate_COmole.b_N_CN_N2_C;

		/*  >>refer   mol  N      +  SIH   =>   SIN   +     H	    UMIST  */

			mole.c[ipSIH][ipSIH] -= rate_COmole.N_SIH_SIN_H_1;
			mole.c[ipATN][ipSIH] -= rate_COmole.N_SIH_SIN_H_2;
			mole.b[ipSIH] -= rate_COmole.b_N_SIH_SIN_H;

			mole.c[ipSIH][ipATN] -= rate_COmole.N_SIH_SIN_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SIH_SIN_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_SIH_SIN_H;
			
			mole.c[ipSIH][ipSIN] += rate_COmole.N_SIH_SIN_H_1;
			mole.c[ipATN][ipSIN] += rate_COmole.N_SIH_SIN_H_2;
			mole.b[ipSIN] += rate_COmole.b_N_SIH_SIN_H;

		/*  >>refer   mol  N      +  NO    =>   N2    +     O	    UMIST  */

			mole.c[ipNO][ipNO] -= rate_COmole.N_NO_N2_O_1;
			mole.c[ipATN][ipNO] -= rate_COmole.N_NO_N2_O_2;
			mole.b[ipNO] -= rate_COmole.b_N_NO_N2_O;
			
			mole.c[ipNO][ipATN] -= rate_COmole.N_NO_N2_O_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_NO_N2_O_2;
			mole.b[ipATN] -= rate_COmole.b_N_NO_N2_O;
			
			mole.c[ipNO][ipN2] += rate_COmole.N_NO_N2_O_1;
			mole.c[ipATN][ipN2] += rate_COmole.N_NO_N2_O_2;
			mole.b[ipN2] += rate_COmole.b_N_NO_N2_O;

			mole.c[ipNO][ipATO] += rate_COmole.N_NO_N2_O_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_NO_N2_O_2;
			mole.b[ipATO] += rate_COmole.b_N_NO_N2_O;

		/*  >>refer   mol  N      +  HNO   =>   NO    +     NH    UMIST  */

			mole.c[ipHNO][ipHNO] -= rate_COmole.N_HNO_NO_NH_1;
			mole.c[ipATN][ipHNO] -= rate_COmole.N_HNO_NO_NH_2;
			mole.b[ipHNO] -= rate_COmole.b_N_HNO_NO_NH;
			
			mole.c[ipHNO][ipATN] -= rate_COmole.N_HNO_NO_NH_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_HNO_NO_NH_2;
			mole.b[ipATN] -= rate_COmole.b_N_HNO_NO_NH;

			mole.c[ipHNO][ipNO] += rate_COmole.N_HNO_NO_NH_1;
			mole.c[ipATN][ipNO] += rate_COmole.N_HNO_NO_NH_2;
			mole.b[ipNO] += rate_COmole.b_N_HNO_NO_NH;
			
			mole.c[ipHNO][ipNH] += rate_COmole.N_HNO_NO_NH_1;
			mole.c[ipATN][ipNH] += rate_COmole.N_HNO_NO_NH_2;
			mole.b[ipNH] += rate_COmole.b_N_HNO_NO_NH;

		/*  >>refer   mol  N      +  HNO   =>   N2O   +     H	    UMIST  */

			mole.c[ipHNO][ipHNO] -= rate_COmole.N_HNO_N2O_H_1;
			mole.c[ipATN][ipHNO] -= rate_COmole.N_HNO_N2O_H_2;
			mole.b[ipHNO] -= rate_COmole.b_N_HNO_N2O_H;
			
			mole.c[ipHNO][ipATN] -= rate_COmole.N_HNO_N2O_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_HNO_N2O_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_HNO_N2O_H;

			mole.c[ipHNO][ipN2O] += rate_COmole.N_HNO_N2O_H_1;
			mole.c[ipATN][ipN2O] += rate_COmole.N_HNO_N2O_H_2;
			mole.b[ipN2O] += rate_COmole.b_N_HNO_N2O_H;

		/*  >>refer   mol  N      +  O2    =>   NO    +     O	    UMIST  */

			mole.c[ipO2][ipO2] -= rate_COmole.N_O2_NO_O_1;
			mole.c[ipATN][ipO2] -= rate_COmole.N_O2_NO_O_2;
			mole.b[ipO2] -= rate_COmole.b_N_O2_NO_O;
			
			mole.c[ipO2][ipATN] -= rate_COmole.N_O2_NO_O_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_O2_NO_O_2;
			mole.b[ipATN] -= rate_COmole.b_N_O2_NO_O;
			
			mole.c[ipO2][ipNO] += rate_COmole.N_O2_NO_O_1;
			mole.c[ipATN][ipNO] += rate_COmole.N_O2_NO_O_2;
			mole.b[ipNO] += rate_COmole.b_N_O2_NO_O;

			mole.c[ipO2][ipATO] += rate_COmole.N_O2_NO_O_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_O2_NO_O_2;
			mole.b[ipATO] += rate_COmole.b_N_O2_NO_O;

		/*  >>refer   mol  N      +  HS    =>   NS    +     H     UMIST  */

			mole.c[ipHS][ipHS] -= rate_COmole.N_HS_NS_H_1;
			mole.c[ipATN][ipHS] -= rate_COmole.N_HS_NS_H_2;
			mole.b[ipHS] -= rate_COmole.b_N_HS_NS_H;

			mole.c[ipHS][ipATN] -= rate_COmole.N_HS_NS_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_HS_NS_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_HS_NS_H;
			
			mole.c[ipHS][ipNS] += rate_COmole.N_HS_NS_H_1;
			mole.c[ipATN][ipNS] += rate_COmole.N_HS_NS_H_2;
			mole.b[ipNS] += rate_COmole.b_N_HS_NS_H;

		/*  >>refer   mol  N      +  HS    =>   S     +     NH    UMIST  */

			mole.c[ipHS][ipATN] -= rate_COmole.N_HS_S_NH_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_HS_S_NH_2;
			mole.b[ipATN] -= rate_COmole.b_N_HS_S_NH;
			
			mole.c[ipHS][ipHS] -= rate_COmole.N_HS_S_NH_1;
			mole.c[ipATN][ipHS] -= rate_COmole.N_HS_S_NH_2;
			mole.b[ipHS] -= rate_COmole.b_N_HS_S_NH;
			
			mole.c[ipHS][ipNH] += rate_COmole.N_HS_S_NH_1;
			mole.c[ipATN][ipNH] += rate_COmole.N_HS_S_NH_2;
			mole.b[ipNH] += rate_COmole.b_N_HS_S_NH;

			mole.c[ipHS][ipATS] += rate_COmole.N_HS_S_NH_1;
			mole.c[ipATN][ipATS] += rate_COmole.N_HS_S_NH_2;
			mole.b[ipATS] += rate_COmole.b_N_HS_S_NH;

		/*  >>refer   mol  N      +  CS    =>   S     +     CN    UMIST  */

			mole.c[ipCS][ipATN] -= rate_COmole.N_CS_S_CN_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_CS_S_CN_2;
			mole.b[ipATN] -= rate_COmole.b_N_CS_S_CN;
			
			mole.c[ipCS][ipCS] -= rate_COmole.N_CS_S_CN_1;
			mole.c[ipATN][ipCS] -= rate_COmole.N_CS_S_CN_2;
			mole.b[ipCS] -= rate_COmole.b_N_CS_S_CN;
			
			mole.c[ipCS][ipCN] += rate_COmole.N_CS_S_CN_1;
			mole.c[ipATN][ipCN] += rate_COmole.N_CS_S_CN_2;
			mole.b[ipCN] += rate_COmole.b_N_CS_S_CN;

			mole.c[ipCS][ipATS] += rate_COmole.N_CS_S_CN_1;
			mole.c[ipATN][ipATS] += rate_COmole.N_CS_S_CN_2;
			mole.b[ipATS] += rate_COmole.b_N_CS_S_CN;

		/*  >>refer   mol  N      +  NS    =>   S     +     N2    UMIST  */

			mole.c[ipNS][ipNS] -= rate_COmole.N_NS_S_N2_1;
			mole.c[ipATN][ipNS] -= rate_COmole.N_NS_S_N2_2;
			mole.b[ipNS] -= rate_COmole.b_N_NS_S_N2;
			
			mole.c[ipNS][ipATN] -= rate_COmole.N_NS_S_N2_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_NS_S_N2_2;
			mole.b[ipATN] -= rate_COmole.b_N_NS_S_N2;
			
			mole.c[ipNS][ipN2] += rate_COmole.N_NS_S_N2_1;
			mole.c[ipATN][ipN2] += rate_COmole.N_NS_S_N2_2;
			mole.b[ipN2] += rate_COmole.b_N_NS_S_N2;

			mole.c[ipNS][ipATS] += rate_COmole.N_NS_S_N2_1;
			mole.c[ipATN][ipATS] += rate_COmole.N_NS_S_N2_2;
			mole.b[ipATS] += rate_COmole.b_N_NS_S_N2;

		/*  >>refer   mol  N      +  SO    =>   NS    +     O	    UMIST  */

			mole.c[ipSO][ipSO] -= rate_COmole.N_SO_NS_O_1;
			mole.c[ipATN][ipSO] -= rate_COmole.N_SO_NS_O_2;
			mole.b[ipSO] -= rate_COmole.b_N_SO_NS_O;

			mole.c[ipSO][ipATN] -= rate_COmole.N_SO_NS_O_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SO_NS_O_2;
			mole.b[ipATN] -= rate_COmole.b_N_SO_NS_O;

			mole.c[ipSO][ipNS] += rate_COmole.N_SO_NS_O_1;
			mole.c[ipATN][ipNS] += rate_COmole.N_SO_NS_O_2;
			mole.b[ipNS] += rate_COmole.b_N_SO_NS_O;

			mole.c[ipSO][ipATO] += rate_COmole.N_SO_NS_O_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_SO_NS_O_2;
			mole.b[ipATO] += rate_COmole.b_N_SO_NS_O;

		/*  >>refer   mol  N      +  SO    =>   S     +     NO    UMIST  */

			mole.c[ipSO][ipSO] -= rate_COmole.N_SO_S_NO_1;
			mole.c[ipATN][ipSO] -= rate_COmole.N_SO_S_NO_2;
			mole.b[ipSO] -= rate_COmole.b_N_SO_S_NO;
			
			mole.c[ipSO][ipATN] -= rate_COmole.N_SO_S_NO_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SO_S_NO_2;
			mole.b[ipATN] -= rate_COmole.b_N_SO_S_NO;
			
			mole.c[ipSO][ipNO] += rate_COmole.N_SO_S_NO_1;
			mole.c[ipATN][ipNO] += rate_COmole.N_SO_S_NO_2;
			mole.b[ipNO] += rate_COmole.b_N_SO_S_NO;

			mole.c[ipSO][ipATS] += rate_COmole.N_SO_S_NO_1;
			mole.c[ipATN][ipATS] += rate_COmole.N_SO_S_NO_2;
			mole.b[ipATS] += rate_COmole.b_N_SO_S_NO;

		/*  >>refer   mol  N      +  S2    =>   NS    +     S     UMIST  */

			mole.c[ipS2][ipS2] -= rate_COmole.N_S2_NS_S_1;
			mole.c[ipATN][ipS2] -= rate_COmole.N_S2_NS_S_2;
			mole.b[ipS2] -= rate_COmole.b_N_S2_NS_S;
			
			mole.c[ipS2][ipATN] -= rate_COmole.N_S2_NS_S_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_S2_NS_S_2;
			mole.b[ipATN] -= rate_COmole.b_N_S2_NS_S;
			
			mole.c[ipS2][ipNS] += rate_COmole.N_S2_NS_S_1;
			mole.c[ipATN][ipNS] += rate_COmole.N_S2_NS_S_2;
			mole.b[ipNS] += rate_COmole.b_N_S2_NS_S;

			mole.c[ipS2][ipATS] += rate_COmole.N_S2_NS_S_1;
			mole.c[ipATN][ipATS] += rate_COmole.N_S2_NS_S_2;
			mole.b[ipATS] += rate_COmole.b_N_S2_NS_S;

		/*  >>refer   mol  CH2    +  CN    =>   HCN   +     CH    UMIST  */

			mole.c[ipCN][ipCH2] -= rate_COmole.CH2_CN_HCN_CH_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CN_HCN_CH_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_CN_HCN_CH;
			
			mole.c[ipCN][ipCN] -= rate_COmole.CH2_CN_HCN_CH_1;
			mole.c[ipCH2][ipCN] -= rate_COmole.CH2_CN_HCN_CH_2;
			mole.b[ipCN] -= rate_COmole.b_CH2_CN_HCN_CH;
			
			mole.c[ipCN][ipHCN] += rate_COmole.CH2_CN_HCN_CH_1;
			mole.c[ipCH2][ipHCN] += rate_COmole.CH2_CN_HCN_CH_2;
			mole.b[ipHCN] += rate_COmole.b_CH2_CN_HCN_CH;
			
			mole.c[ipCN][ipCH] += rate_COmole.CH2_CN_HCN_CH_1;
			mole.c[ipCH2][ipCH] += rate_COmole.CH2_CN_HCN_CH_2;
			mole.b[ipCH] += rate_COmole.b_CH2_CN_HCN_CH;

		/*  >>refer   mol  CH2    +  NO    =>   HCN   +     OH    UMIST  */

			mole.c[ipNO][ipCH2] -= rate_COmole.CH2_NO_HCN_OH_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_NO_HCN_OH_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_NO_HCN_OH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH2_NO_HCN_OH_1;
			mole.c[ipCH2][ipNO] -= rate_COmole.CH2_NO_HCN_OH_2;
			mole.b[ipNO] -= rate_COmole.b_CH2_NO_HCN_OH;
			
			mole.c[ipNO][ipHCN] += rate_COmole.CH2_NO_HCN_OH_1;
			mole.c[ipCH2][ipHCN] += rate_COmole.CH2_NO_HCN_OH_2;
			mole.b[ipHCN] += rate_COmole.b_CH2_NO_HCN_OH;
			
			mole.c[ipNO][ipOH] += rate_COmole.CH2_NO_HCN_OH_1;
			mole.c[ipCH2][ipOH] += rate_COmole.CH2_NO_HCN_OH_2;
			mole.b[ipOH] += rate_COmole.b_CH2_NO_HCN_OH;

		/*  >>refer   mol  CH2    +  HNO   =>   NO    +     CH3   UMIST  */

			mole.c[ipHNO][ipHNO] -= rate_COmole.CH2_HNO_NO_CH3_1;
			mole.c[ipCH2][ipHNO] -= rate_COmole.CH2_HNO_NO_CH3_2;
			mole.b[ipHNO] -= rate_COmole.b_CH2_HNO_NO_CH3;
			
			mole.c[ipHNO][ipCH2] -= rate_COmole.CH2_HNO_NO_CH3_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_HNO_NO_CH3_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_HNO_NO_CH3;
			
			mole.c[ipHNO][ipNO] += rate_COmole.CH2_HNO_NO_CH3_1;
			mole.c[ipCH2][ipNO] += rate_COmole.CH2_HNO_NO_CH3_2;
			mole.b[ipNO] += rate_COmole.b_CH2_HNO_NO_CH3;
			
			mole.c[ipHNO][ipCH3] += rate_COmole.CH2_HNO_NO_CH3_1;
			mole.c[ipCH2][ipCH3] += rate_COmole.CH2_HNO_NO_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH2_HNO_NO_CH3;

		/*  >>refer   mol  NH     +  NH    =>   N2    +     H  +  H      UMIST  */

			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH_N2_H_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH_N2_H_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NH_N2_H_H;
			
			mole.c[ipNH][ipN2] += 0.5*rate_COmole.NH_NH_N2_H_H_1;
			mole.c[ipNH][ipN2] +=0.5* rate_COmole.NH_NH_N2_H_H_2;
			mole.b[ipN2] += 0.5*rate_COmole.b_NH_NH_N2_H_H;
			
		/*  >>refer   mol  NH     +  NH    =>   N2    +     H2    UMIST  */

			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH_N2_H2_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH_N2_H2_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NH_N2_H2;
			
			mole.c[ipNH][ipN2] += 0.5*rate_COmole.NH_NH_N2_H2_1;
			mole.c[ipNH][ipN2] += 0.5*rate_COmole.NH_NH_N2_H2_2;
			mole.b[ipN2] += 0.5*rate_COmole.b_NH_NH_N2_H2;
			
		/*  >>refer   mol  NH     +  O     =>   OH    +     N     UMIST  */

			mole.c[ipATO][ipNH] -= rate_COmole.NH_O_OH_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_O_OH_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_O_OH_N;
			
			mole.c[ipATO][ipATO] -= rate_COmole.NH_O_OH_N_1;
			mole.c[ipNH][ipATO] -= rate_COmole.NH_O_OH_N_2;
			mole.b[ipATO] -= rate_COmole.b_NH_O_OH_N;
			
			mole.c[ipATO][ipOH] += rate_COmole.NH_O_OH_N_1;
			mole.c[ipNH][ipOH] += rate_COmole.NH_O_OH_N_2;
			mole.b[ipOH] += rate_COmole.b_NH_O_OH_N;

			mole.c[ipATO][ipATN] += rate_COmole.NH_O_OH_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_O_OH_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_O_OH_N;

		/*  >>refer   mol  NH     +  O     =>   NO    +     H     UMIST  */

			mole.c[ipATO][ipNH] -= rate_COmole.NH_O_NO_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_O_NO_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_O_NO_H;

			mole.c[ipATO][ipATO] -= rate_COmole.NH_O_NO_H_1;
			mole.c[ipNH][ipATO] -= rate_COmole.NH_O_NO_H_2;
			mole.b[ipATO] -= rate_COmole.b_NH_O_NO_H;
			
			mole.c[ipATO][ipNO] += rate_COmole.NH_O_NO_H_1;
			mole.c[ipNH][ipNO] += rate_COmole.NH_O_NO_H_2;
			mole.b[ipNO] += rate_COmole.b_NH_O_NO_H;

		/*  >>refer   mol  NH     +  OH    =>   HNO   +     H     UMIST  */

			mole.c[ipOH][ipNH] -= rate_COmole.NH_OH_HNO_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OH_HNO_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OH_HNO_H;
			
			mole.c[ipOH][ipOH] -= rate_COmole.NH_OH_HNO_H_1;
			mole.c[ipNH][ipOH] -= rate_COmole.NH_OH_HNO_H_2;
			mole.b[ipOH] -= rate_COmole.b_NH_OH_HNO_H;
			
			mole.c[ipOH][ipHNO] += rate_COmole.NH_OH_HNO_H_1;
			mole.c[ipNH][ipHNO] += rate_COmole.NH_OH_HNO_H_2;
			mole.b[ipHNO] += rate_COmole.b_NH_OH_HNO_H;

		/*  >>refer   mol  NH     +  OH    =>   NH2   +     O     UMIST  */

			mole.c[ipOH][ipOH] -= rate_COmole.NH_OH_NH2_O_1;
			mole.c[ipNH][ipOH] -= rate_COmole.NH_OH_NH2_O_2;
			mole.b[ipOH] -= rate_COmole.b_NH_OH_NH2_O;
			
			mole.c[ipOH][ipNH] -= rate_COmole.NH_OH_NH2_O_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OH_NH2_O_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OH_NH2_O;
			
			mole.c[ipOH][ipNH2] += rate_COmole.NH_OH_NH2_O_1;
			mole.c[ipNH][ipNH2] += rate_COmole.NH_OH_NH2_O_2;
			mole.b[ipNH2] += rate_COmole.b_NH_OH_NH2_O;
			
			mole.c[ipOH][ipATO] += rate_COmole.NH_OH_NH2_O_1;
			mole.c[ipNH][ipATO] += rate_COmole.NH_OH_NH2_O_2;
			mole.b[ipATO] += rate_COmole.b_NH_OH_NH2_O;
			
		/*  >>refer   mol  NH     +  OH    =>   H2O   +     N     UMIST  */

			mole.c[ipOH][ipOH] -= rate_COmole.NH_OH_H2O_N_1;
			mole.c[ipNH][ipOH] -= rate_COmole.NH_OH_H2O_N_2;
			mole.b[ipOH] -= rate_COmole.b_NH_OH_H2O_N;
			
			mole.c[ipOH][ipNH] -= rate_COmole.NH_OH_H2O_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OH_H2O_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OH_H2O_N;
			
			mole.c[ipOH][ipH2O] += rate_COmole.NH_OH_H2O_N_1;
			mole.c[ipNH][ipH2O] += rate_COmole.NH_OH_H2O_N_2;
			mole.b[ipH2O] += rate_COmole.b_NH_OH_H2O_N;
			
			mole.c[ipOH][ipATN] += rate_COmole.NH_OH_H2O_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_OH_H2O_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_OH_H2O_N;

		/*  >>refer   mol  NH     +  H2O   =>   OH    +     NH2   UMIST  */

			mole.c[ipH2O][ipNH] -= rate_COmole.NH_H2O_OH_NH2_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_H2O_OH_NH2_2;
			mole.b[ipNH] -= rate_COmole.b_NH_H2O_OH_NH2;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NH_H2O_OH_NH2_1;
			mole.c[ipNH][ipH2O] -= rate_COmole.NH_H2O_OH_NH2_2;
			mole.b[ipH2O] -= rate_COmole.b_NH_H2O_OH_NH2;
			
			mole.c[ipH2O][ipOH] += rate_COmole.NH_H2O_OH_NH2_1;
			mole.c[ipNH][ipOH] += rate_COmole.NH_H2O_OH_NH2_2;
			mole.b[ipOH] += rate_COmole.b_NH_H2O_OH_NH2;
			
			mole.c[ipH2O][ipNH2] += rate_COmole.NH_H2O_OH_NH2_1;
			mole.c[ipNH][ipNH2] += rate_COmole.NH_H2O_OH_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH_H2O_OH_NH2;

		/*  >>refer   mol  NH     +  CN    =>   HCN   +     N     UMIST  */

			mole.c[ipCN][ipNH] -= rate_COmole.NH_CN_HCN_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_CN_HCN_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_CN_HCN_N;
			
			mole.c[ipCN][ipCN] -= rate_COmole.NH_CN_HCN_N_1;
			mole.c[ipNH][ipCN] -= rate_COmole.NH_CN_HCN_N_2;
			mole.b[ipCN] -= rate_COmole.b_NH_CN_HCN_N;
			
			mole.c[ipCN][ipHCN] += rate_COmole.NH_CN_HCN_N_1;
			mole.c[ipNH][ipHCN] += rate_COmole.NH_CN_HCN_N_2;
			mole.b[ipHCN] += rate_COmole.b_NH_CN_HCN_N;
			
			mole.c[ipCN][ipATN] += rate_COmole.NH_CN_HCN_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_CN_HCN_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_CN_HCN_N;
			
		/*  >>refer   mol  NH     +  NO    =>   N2O   +     H     UMIST  */

			mole.c[ipNO][ipNH] -= rate_COmole.NH_NO_N2O_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NO_N2O_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NO_N2O_H;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH_NO_N2O_H_1;
			mole.c[ipNH][ipNO] -= rate_COmole.NH_NO_N2O_H_2;
			mole.b[ipNO] -= rate_COmole.b_NH_NO_N2O_H;
			
			mole.c[ipNO][ipN2O] += rate_COmole.NH_NO_N2O_H_1;
			mole.c[ipNH][ipN2O] += rate_COmole.NH_NO_N2O_H_2;
			mole.b[ipN2O] += rate_COmole.b_NH_NO_N2O_H;

		/*  >>refer   mol  NH     +  NO    =>   N2    +     O  +  H      UMIST  */

			mole.c[ipNO][ipNH] -= rate_COmole.NH_NO_N2_O_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NO_N2_O_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NO_N2_O_H;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH_NO_N2_O_H_1;
			mole.c[ipNH][ipNO] -= rate_COmole.NH_NO_N2_O_H_2;
			mole.b[ipNO] -= rate_COmole.b_NH_NO_N2_O_H;
			
			mole.c[ipNO][ipN2] += rate_COmole.NH_NO_N2_O_H_1;
			mole.c[ipNH][ipN2] += rate_COmole.NH_NO_N2_O_H_2;
			mole.b[ipN2] += rate_COmole.b_NH_NO_N2_O_H;
			
			mole.c[ipNO][ipATO] += rate_COmole.NH_NO_N2_O_H_1;
			mole.c[ipNH][ipATO] += rate_COmole.NH_NO_N2_O_H_2;
			mole.b[ipATO] += rate_COmole.b_NH_NO_N2_O_H;

		/*  >>refer   mol  NH     +  NO    =>   N2    +     OH    UMIST  */

			mole.c[ipNO][ipNH] -= rate_COmole.NH_NO_N2_OH_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NO_N2_OH_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NO_N2_OH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH_NO_N2_OH_1;
			mole.c[ipNH][ipNO] -= rate_COmole.NH_NO_N2_OH_2;
			mole.b[ipNO] -= rate_COmole.b_NH_NO_N2_OH;
			
			mole.c[ipNO][ipN2] += rate_COmole.NH_NO_N2_OH_1;
			mole.c[ipNH][ipN2] += rate_COmole.NH_NO_N2_OH_2;
			mole.b[ipN2] += rate_COmole.b_NH_NO_N2_OH;
			
			mole.c[ipNO][ipOH] += rate_COmole.NH_NO_N2_OH_1;
			mole.c[ipNH][ipOH] += rate_COmole.NH_NO_N2_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH_NO_N2_OH;

		/*  >>refer   mol  NH     +  S     =>   HS    +     N     UMIST  */

			mole.c[ipATS][ipNH] -= rate_COmole.NH_S_HS_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_S_HS_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_S_HS_N;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NH_S_HS_N_1;
			mole.c[ipNH][ipATS] -= rate_COmole.NH_S_HS_N_2;
			mole.b[ipATS] -= rate_COmole.b_NH_S_HS_N;
			
			mole.c[ipATS][ipHS] += rate_COmole.NH_S_HS_N_1;
			mole.c[ipNH][ipHS] += rate_COmole.NH_S_HS_N_2;
			mole.b[ipHS] += rate_COmole.b_NH_S_HS_N;

			mole.c[ipATS][ipATN] += rate_COmole.NH_S_HS_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_S_HS_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_S_HS_N;

		/*  >>refer   mol  NH     +  S     =>   NS    +     H     UMIST  */

			mole.c[ipATS][ipNH] -= rate_COmole.NH_S_NS_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_S_NS_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_S_NS_H;

			mole.c[ipATS][ipATS] -= rate_COmole.NH_S_NS_H_1;
			mole.c[ipNH][ipATS] -= rate_COmole.NH_S_NS_H_2;
			mole.b[ipATS] -= rate_COmole.b_NH_S_NS_H;
			
			mole.c[ipATS][ipNS] += rate_COmole.NH_S_NS_H_1;
			mole.c[ipNH][ipNS] += rate_COmole.NH_S_NS_H_2;
			mole.b[ipNS] += rate_COmole.b_NH_S_NS_H;

		/*  >>refer   mol  NH     +  NO2   =>   HNO   +     NO    UMIST  */

			mole.c[ipNO2][ipNH] -= rate_COmole.NH_NO2_HNO_NO_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NO2_HNO_NO_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NO2_HNO_NO;
			
			mole.c[ipNO2][ipNO2] -= rate_COmole.NH_NO2_HNO_NO_1;
			mole.c[ipNH][ipNO2] -= rate_COmole.NH_NO2_HNO_NO_2;
			mole.b[ipNO2] -= rate_COmole.b_NH_NO2_HNO_NO;
			
			mole.c[ipNO2][ipHNO] += rate_COmole.NH_NO2_HNO_NO_1;
			mole.c[ipNH][ipHNO] += rate_COmole.NH_NO2_HNO_NO_2;
			mole.b[ipHNO] += rate_COmole.b_NH_NO2_HNO_NO;
			
			mole.c[ipNO2][ipNO] += rate_COmole.NH_NO2_HNO_NO_1;
			mole.c[ipNH][ipNO] += rate_COmole.NH_NO2_HNO_NO_2;
			mole.b[ipNO] += rate_COmole.b_NH_NO2_HNO_NO;

		/*  >>refer   mol  NH     +  NO2   =>   N2O   +     OH    UMIST  */

			mole.c[ipNO2][ipNH] -= rate_COmole.NH_NO2_N2O_OH_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NO2_N2O_OH_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NO2_N2O_OH;
			
			mole.c[ipNO2][ipNO2] -= rate_COmole.NH_NO2_N2O_OH_1;
			mole.c[ipNH][ipNO2] -= rate_COmole.NH_NO2_N2O_OH_2;
			mole.b[ipNO2] -= rate_COmole.b_NH_NO2_N2O_OH;
			
			mole.c[ipNO2][ipN2O] += rate_COmole.NH_NO2_N2O_OH_1;
			mole.c[ipNH][ipN2O] += rate_COmole.NH_NO2_N2O_OH_2;
			mole.b[ipN2O] += rate_COmole.b_NH_NO2_N2O_OH;
			
			mole.c[ipNO2][ipOH] += rate_COmole.NH_NO2_N2O_OH_1;
			mole.c[ipNH][ipOH] += rate_COmole.NH_NO2_N2O_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH_NO2_N2O_OH;

		/*  >>refer   mol  CH3    +  NH3   =>   CH4   +     NH2   UMIST  */

			mole.c[ipNH3][ipCH3] -= rate_COmole.CH3_NH3_CH4_NH2_1;
			mole.c[ipCH3][ipCH3] -= rate_COmole.CH3_NH3_CH4_NH2_2;
			mole.b[ipCH3] -= rate_COmole.b_CH3_NH3_CH4_NH2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CH3_NH3_CH4_NH2_1;
			mole.c[ipCH3][ipNH3] -= rate_COmole.CH3_NH3_CH4_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_CH3_NH3_CH4_NH2;
			
			mole.c[ipNH3][ipCH4] += rate_COmole.CH3_NH3_CH4_NH2_1;
			mole.c[ipCH3][ipCH4] += rate_COmole.CH3_NH3_CH4_NH2_2;
			mole.b[ipCH4] += rate_COmole.b_CH3_NH3_CH4_NH2;
			
			mole.c[ipNH3][ipNH2] += rate_COmole.CH3_NH3_CH4_NH2_1;
			mole.c[ipCH3][ipNH2] += rate_COmole.CH3_NH3_CH4_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_CH3_NH3_CH4_NH2;

		/*  >>refer   mol  CH3    +  CN    =>   HCN   +     CH2   UMIST  */

			mole.c[ipCN][ipCH3] -= rate_COmole.CH3_CN_HCN_CH2_1;
			mole.c[ipCH3][ipCH3] -= rate_COmole.CH3_CN_HCN_CH2_2;
			mole.b[ipCH3] -= rate_COmole.b_CH3_CN_HCN_CH2;
			
			mole.c[ipCN][ipCN] -= rate_COmole.CH3_CN_HCN_CH2_1;
			mole.c[ipCH3][ipCN] -= rate_COmole.CH3_CN_HCN_CH2_2;
			mole.b[ipCN] -= rate_COmole.b_CH3_CN_HCN_CH2;
			
			mole.c[ipCN][ipCH2] += rate_COmole.CH3_CN_HCN_CH2_1;
			mole.c[ipCH3][ipCH2] += rate_COmole.CH3_CN_HCN_CH2_2;
			mole.b[ipCH2] += rate_COmole.b_CH3_CN_HCN_CH2;
			
			mole.c[ipCN][ipHCN] += rate_COmole.CH3_CN_HCN_CH2_1;
			mole.c[ipCH3][ipHCN] += rate_COmole.CH3_CN_HCN_CH2_2;
			mole.b[ipHCN] += rate_COmole.b_CH3_CN_HCN_CH2;

		/*  >>refer   mol  CH3    +  HNO   =>   NO    +     CH4   UMIST  */

			mole.c[ipHNO][ipCH3] -= rate_COmole.CH3_HNO_NO_CH4_1;
			mole.c[ipCH3][ipCH3] -= rate_COmole.CH3_HNO_NO_CH4_2;
			mole.b[ipCH3] -= rate_COmole.b_CH3_HNO_NO_CH4;
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.CH3_HNO_NO_CH4_1;
			mole.c[ipCH3][ipHNO] -= rate_COmole.CH3_HNO_NO_CH4_2;
			mole.b[ipHNO] -= rate_COmole.b_CH3_HNO_NO_CH4;
			
			mole.c[ipHNO][ipNO] += rate_COmole.CH3_HNO_NO_CH4_1;
			mole.c[ipCH3][ipNO] += rate_COmole.CH3_HNO_NO_CH4_2;
			mole.b[ipNO] += rate_COmole.b_CH3_HNO_NO_CH4;
			
			mole.c[ipHNO][ipCH4] += rate_COmole.CH3_HNO_NO_CH4_1;
			mole.c[ipCH3][ipCH4] += rate_COmole.CH3_HNO_NO_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_CH3_HNO_NO_CH4;

		/*  >>refer   mol  O      +  NH2   =>   OH    +     NH    UMIST  */

			mole.c[ipNH2][ipNH2] -= rate_COmole.O_NH2_OH_NH_1;
			mole.c[ipATO][ipNH2] -= rate_COmole.O_NH2_OH_NH_2;
			mole.b[ipNH2] -= rate_COmole.b_O_NH2_OH_NH;

			mole.c[ipNH2][ipATO] -= rate_COmole.O_NH2_OH_NH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NH2_OH_NH_2;
			mole.b[ipATO] -= rate_COmole.b_O_NH2_OH_NH;
			
			mole.c[ipNH2][ipOH] += rate_COmole.O_NH2_OH_NH_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_NH2_OH_NH_2;
			mole.b[ipOH] += rate_COmole.b_O_NH2_OH_NH;
			
			mole.c[ipNH2][ipNH] += rate_COmole.O_NH2_OH_NH_1;
			mole.c[ipATO][ipNH] += rate_COmole.O_NH2_OH_NH_2;
			mole.b[ipNH] += rate_COmole.b_O_NH2_OH_NH;

		/*  >>refer   mol  O      +  NH2   =>   NO    +     H2    UMIST  */
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.O_NH2_NO_H2_1;
			mole.c[ipATO][ipNH2] -= rate_COmole.O_NH2_NO_H2_2;
			mole.b[ipNH2] -= rate_COmole.b_O_NH2_NO_H2;
			
			mole.c[ipNH2][ipATO] -= rate_COmole.O_NH2_NO_H2_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NH2_NO_H2_2;
			mole.b[ipATO] -= rate_COmole.b_O_NH2_NO_H2;
			
			mole.c[ipNH2][ipNO] += rate_COmole.O_NH2_NO_H2_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_NH2_NO_H2_2;
			mole.b[ipNO] += rate_COmole.b_O_NH2_NO_H2;
			
		/*  >>refer   mol  O      +  NH3   =>   OH    +     NH2   UMIST  */

			mole.c[ipNH3][ipNH3] -= rate_COmole.O_NH3_OH_NH2_1;
			mole.c[ipATO][ipNH3] -= rate_COmole.O_NH3_OH_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_O_NH3_OH_NH2;

			mole.c[ipNH3][ipATO] -= rate_COmole.O_NH3_OH_NH2_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NH3_OH_NH2_2;
			mole.b[ipATO] -= rate_COmole.b_O_NH3_OH_NH2;
			
			mole.c[ipNH3][ipOH] += rate_COmole.O_NH3_OH_NH2_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_NH3_OH_NH2_2;
			mole.b[ipOH] += rate_COmole.b_O_NH3_OH_NH2;
			
			mole.c[ipNH3][ipNH2] += rate_COmole.O_NH3_OH_NH2_1;
			mole.c[ipATO][ipNH2] += rate_COmole.O_NH3_OH_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_O_NH3_OH_NH2;

		/*  >>refer   mol  O      +  CN    =>   NO    +     C     UMIST  */
			
			mole.c[ipCN][ipCN] -= rate_COmole.O_CN_NO_C_1;
			mole.c[ipATO][ipCN] -= rate_COmole.O_CN_NO_C_2;
			mole.b[ipCN] -= rate_COmole.b_O_CN_NO_C;
						
			mole.c[ipCN][ipATO] -= rate_COmole.O_CN_NO_C_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_CN_NO_C_2;
			mole.b[ipATO] -= rate_COmole.b_O_CN_NO_C;
			
			mole.c[ipCN][ipNO] += rate_COmole.O_CN_NO_C_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_CN_NO_C_2;
			mole.b[ipNO] += rate_COmole.b_O_CN_NO_C;

			mole.c[ipCN][ipATC] += rate_COmole.O_CN_NO_C_1;
			mole.c[ipATO][ipATC] += rate_COmole.O_CN_NO_C_2;
			mole.b[ipATC] += rate_COmole.b_O_CN_NO_C;

		/*  >>refer   mol  O      +  HCN   =>   CN    +     OH    UMIST  */

			mole.c[ipHCN][ipHCN] -= rate_COmole.O_HCN_CN_OH_1;
			mole.c[ipATO][ipHCN] -= rate_COmole.O_HCN_CN_OH_2;
			mole.b[ipHCN] -= rate_COmole.b_O_HCN_CN_OH;
			
			mole.c[ipHCN][ipATO] -= rate_COmole.O_HCN_CN_OH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HCN_CN_OH_2;
			mole.b[ipATO] -= rate_COmole.b_O_HCN_CN_OH;

			mole.c[ipHCN][ipCN] += rate_COmole.O_HCN_CN_OH_1;
			mole.c[ipATO][ipCN] += rate_COmole.O_HCN_CN_OH_2;
			mole.b[ipCN] += rate_COmole.b_O_HCN_CN_OH;
			
			mole.c[ipHCN][ipOH] += rate_COmole.O_HCN_CN_OH_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_HCN_CN_OH_2;
			mole.b[ipOH] += rate_COmole.b_O_HCN_CN_OH;

		/*  >>refer   mol  O      +  HCN   =>   OCN   +     H     UMIST  */
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.O_HCN_OCN_H_1;
			mole.c[ipATO][ipHCN] -= rate_COmole.O_HCN_OCN_H_2;
			mole.b[ipHCN] -= rate_COmole.b_O_HCN_OCN_H;
				
			mole.c[ipHCN][ipATO] -= rate_COmole.O_HCN_OCN_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HCN_OCN_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HCN_OCN_H;
			
			mole.c[ipHCN][ipOCN] += rate_COmole.O_HCN_OCN_H_1;
			mole.c[ipATO][ipOCN] += rate_COmole.O_HCN_OCN_H_2;
			mole.b[ipOCN] += rate_COmole.b_O_HCN_OCN_H;

		/*  >>refer   mol  O      +  N2    =>   NO    +     N     UMIST  */

			mole.c[ipN2][ipN2] -= rate_COmole.O_N2_NO_N_1;
			mole.c[ipATO][ipN2] -= rate_COmole.O_N2_NO_N_2;
			mole.b[ipN2] -= rate_COmole.b_O_N2_NO_N;
			
			mole.c[ipN2][ipATO] -= rate_COmole.O_N2_NO_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_N2_NO_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_N2_NO_N;
			
			mole.c[ipN2][ipNO] += rate_COmole.O_N2_NO_N_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_N2_NO_N_2;
			mole.b[ipNO] += rate_COmole.b_O_N2_NO_N;

			mole.c[ipN2][ipATN] += rate_COmole.O_N2_NO_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_N2_NO_N_2;
			mole.b[ipATN] += rate_COmole.b_O_N2_NO_N;

		/*  >>refer   mol  O      +  NO    =>   O2    +     N     UMIST  */

			mole.c[ipNO][ipNO] -= rate_COmole.O_NO_O2_N_1;
			mole.c[ipATO][ipNO] -= rate_COmole.O_NO_O2_N_2;
			mole.b[ipNO] -= rate_COmole.b_O_NO_O2_N;
			
			mole.c[ipNO][ipATO] -= rate_COmole.O_NO_O2_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NO_O2_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_NO_O2_N;
			
			mole.c[ipNO][ipO2] += rate_COmole.O_NO_O2_N_1;
			mole.c[ipATO][ipO2] += rate_COmole.O_NO_O2_N_2;
			mole.b[ipO2] += rate_COmole.b_O_NO_O2_N;

			mole.c[ipNO][ipATN] += rate_COmole.O_NO_O2_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_NO_O2_N_2;
			mole.b[ipATN] += rate_COmole.b_O_NO_O2_N;

		/*  >>refer   mol  O      +  HNO   =>   NO    +     OH    UMIST  */
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.O_HNO_NO_OH_1;
			mole.c[ipATO][ipHNO] -= rate_COmole.O_HNO_NO_OH_2;
			mole.b[ipHNO] -= rate_COmole.b_O_HNO_NO_OH;
						
			mole.c[ipHNO][ipATO] -= rate_COmole.O_HNO_NO_OH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HNO_NO_OH_2;
			mole.b[ipATO] -= rate_COmole.b_O_HNO_NO_OH;

			mole.c[ipHNO][ipNO] += rate_COmole.O_HNO_NO_OH_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_HNO_NO_OH_2;
			mole.b[ipNO] += rate_COmole.b_O_HNO_NO_OH;
			
			mole.c[ipHNO][ipOH] += rate_COmole.O_HNO_NO_OH_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_HNO_NO_OH_2;
			mole.b[ipOH] += rate_COmole.b_O_HNO_NO_OH;

		/*  >>refer   mol  O      +  HNO   =>   O2    +     NH    UMIST  */
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.O_HNO_O2_NH_1;
			mole.c[ipATO][ipHNO] -= rate_COmole.O_HNO_O2_NH_2;
			mole.b[ipHNO] -= rate_COmole.b_O_HNO_O2_NH;
			
			mole.c[ipHNO][ipATO] -= rate_COmole.O_HNO_O2_NH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HNO_O2_NH_2;
			mole.b[ipATO] -= rate_COmole.b_O_HNO_O2_NH;
			
			mole.c[ipHNO][ipO2] += rate_COmole.O_HNO_O2_NH_1;
			mole.c[ipATO][ipO2] += rate_COmole.O_HNO_O2_NH_2;
			mole.b[ipO2] += rate_COmole.b_O_HNO_O2_NH;
			
			mole.c[ipHNO][ipNH] += rate_COmole.O_HNO_O2_NH_1;
			mole.c[ipATO][ipNH] += rate_COmole.O_HNO_O2_NH_2;
			mole.b[ipNH] += rate_COmole.b_O_HNO_O2_NH;

		/*  >>refer   mol  O      +  HNO   =>   NO2   +     H     UMIST  */
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.O_HNO_NO2_H_1;
			mole.c[ipATO][ipHNO] -= rate_COmole.O_HNO_NO2_H_2;
			mole.b[ipHNO] -= rate_COmole.b_O_HNO_NO2_H;
						
			mole.c[ipHNO][ipATO] -= rate_COmole.O_HNO_NO2_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HNO_NO2_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HNO_NO2_H;

			mole.c[ipHNO][ipNO2] += rate_COmole.O_HNO_NO2_H_1;
			mole.c[ipATO][ipNO2] += rate_COmole.O_HNO_NO2_H_2;
			mole.b[ipNO2] += rate_COmole.b_O_HNO_NO2_H;

		/*  >>refer   mol  O      +  HS    =>   S     +     OH    UMIST  */

			mole.c[ipHS][ipHS] -= rate_COmole.O_HS_S_OH_1;
			mole.c[ipATO][ipHS] -= rate_COmole.O_HS_S_OH_2;
			mole.b[ipHS] -= rate_COmole.b_O_HS_S_OH;
			
			mole.c[ipHS][ipATO] -= rate_COmole.O_HS_S_OH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HS_S_OH_2;
			mole.b[ipATO] -= rate_COmole.b_O_HS_S_OH;
			
			mole.c[ipHS][ipOH] += rate_COmole.O_HS_S_OH_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_HS_S_OH_2;
			mole.b[ipOH] += rate_COmole.b_O_HS_S_OH;

			mole.c[ipHS][ipATS] += rate_COmole.O_HS_S_OH_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_HS_S_OH_2;
			mole.b[ipATS] += rate_COmole.b_O_HS_S_OH;

		/*  >>refer   mol  O      +  HS    =>   SO    +     H     UMIST  */
			
			mole.c[ipHS][ipHS] -= rate_COmole.O_HS_SO_H_1;
			mole.c[ipATO][ipHS] -= rate_COmole.O_HS_SO_H_2;
			mole.b[ipHS] -= rate_COmole.b_O_HS_SO_H;
			
			mole.c[ipHS][ipATO] -= rate_COmole.O_HS_SO_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HS_SO_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HS_SO_H;
			
			mole.c[ipHS][ipSO] += rate_COmole.O_HS_SO_H_1;
			mole.c[ipATO][ipSO] += rate_COmole.O_HS_SO_H_2;
			mole.b[ipSO] += rate_COmole.b_O_HS_SO_H;

		/*  >>refer   mol  O      +  OCN   =>   O2    +     CN    UMIST  */
			
			mole.c[ipOCN][ipOCN] -= rate_COmole.O_OCN_O2_CN_1;
			mole.c[ipATO][ipOCN] -= rate_COmole.O_OCN_O2_CN_2;
			mole.b[ipOCN] -= rate_COmole.b_O_OCN_O2_CN;
			
			mole.c[ipOCN][ipATO] -= rate_COmole.O_OCN_O2_CN_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_OCN_O2_CN_2;
			mole.b[ipATO] -= rate_COmole.b_O_OCN_O2_CN;
			
			mole.c[ipOCN][ipO2] += rate_COmole.O_OCN_O2_CN_1;
			mole.c[ipATO][ipO2] += rate_COmole.O_OCN_O2_CN_2;
			mole.b[ipO2] += rate_COmole.b_O_OCN_O2_CN;
			
			mole.c[ipOCN][ipCN] += rate_COmole.O_OCN_O2_CN_1;
			mole.c[ipATO][ipCN] += rate_COmole.O_OCN_O2_CN_2;
			mole.b[ipCN] += rate_COmole.b_O_OCN_O2_CN;

		/*  >>refer   mol  O      +  SIN   =>   NO    +     SI    UMIST  */

			mole.c[ipSIN][ipSIN] -= rate_COmole.O_SIN_NO_SI_1;
			mole.c[ipATO][ipSIN] -= rate_COmole.O_SIN_NO_SI_2;
			mole.b[ipSIN] -= rate_COmole.b_O_SIN_NO_SI;
			
			mole.c[ipSIN][ipATO] -= rate_COmole.O_SIN_NO_SI_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_SIN_NO_SI_2;
			mole.b[ipATO] -= rate_COmole.b_O_SIN_NO_SI;
			
			mole.c[ipSIN][ipNO] += rate_COmole.O_SIN_NO_SI_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_SIN_NO_SI_2;
			mole.b[ipNO] += rate_COmole.b_O_SIN_NO_SI;

			mole.c[ipSIN][ipATSI] += rate_COmole.O_SIN_NO_SI_1;
			mole.c[ipATO][ipATSI] += rate_COmole.O_SIN_NO_SI_2;
			mole.b[ipATSI] += rate_COmole.b_O_SIN_NO_SI;
			
		/*  >>refer   mol  O      +  SIN   =>   SIO   +     N     UMIST  */

			mole.c[ipSIN][ipSIN] -= rate_COmole.O_SIN_SIO_N_1;
			mole.c[ipATO][ipSIN] -= rate_COmole.O_SIN_SIO_N_2;
			mole.b[ipSIN] -= rate_COmole.b_O_SIN_SIO_N;
			
			mole.c[ipSIN][ipATO] -= rate_COmole.O_SIN_SIO_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_SIN_SIO_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_SIN_SIO_N;
			
			mole.c[ipSIN][ipSIO] += rate_COmole.O_SIN_SIO_N_1;
			mole.c[ipATO][ipSIO] += rate_COmole.O_SIN_SIO_N_2;
			mole.b[ipSIO] += rate_COmole.b_O_SIN_SIO_N;

			mole.c[ipSIN][ipATN] += rate_COmole.O_SIN_SIO_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_SIN_SIO_N_2;
			mole.b[ipATN] += rate_COmole.b_O_SIN_SIO_N;

		/*  >>refer   mol  O      +  N2O   =>   NO    +     NO    UMIST  */

			mole.c[ipN2O][ipN2O] -= rate_COmole.O_N2O_NO_NO_1;
			mole.c[ipATO][ipN2O] -= rate_COmole.O_N2O_NO_NO_2;
			mole.b[ipN2O] -= rate_COmole.b_O_N2O_NO_NO;

			mole.c[ipN2O][ipATO] -= rate_COmole.O_N2O_NO_NO_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_N2O_NO_NO_2;
			mole.b[ipATO] -= rate_COmole.b_O_N2O_NO_NO;
			
			mole.c[ipN2O][ipNO] += rate_COmole.O_N2O_NO_NO_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_N2O_NO_NO_2;
			mole.b[ipNO] += rate_COmole.b_O_N2O_NO_NO;
			
			mole.c[ipN2O][ipNO] += rate_COmole.O_N2O_NO_NO_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_N2O_NO_NO_2;
			mole.b[ipNO] += rate_COmole.b_O_N2O_NO_NO;

		/*  >>refer   mol  O      +  N2O   =>   O2    +     N2    UMIST  */

			mole.c[ipN2O][ipN2O] -= rate_COmole.O_N2O_O2_N2_1;
			mole.c[ipATO][ipN2O] -= rate_COmole.O_N2O_O2_N2_2;
			mole.b[ipN2O] -= rate_COmole.b_O_N2O_O2_N2;

			mole.c[ipN2O][ipATO] -= rate_COmole.O_N2O_O2_N2_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_N2O_O2_N2_2;
			mole.b[ipATO] -= rate_COmole.b_O_N2O_O2_N2;
			
			mole.c[ipN2O][ipO2] += rate_COmole.O_N2O_O2_N2_1;
			mole.c[ipATO][ipO2] += rate_COmole.O_N2O_O2_N2_2;
			mole.b[ipO2] += rate_COmole.b_O_N2O_O2_N2;
			
			mole.c[ipN2O][ipN2] += rate_COmole.O_N2O_O2_N2_1;
			mole.c[ipATO][ipN2] += rate_COmole.O_N2O_O2_N2_2;
			mole.b[ipN2] += rate_COmole.b_O_N2O_O2_N2;

		/*  >>refer   mol  O      +  CS    =>   SO    +     C     UMIST  */

			mole.c[ipCS][ipCS] -= rate_COmole.O_CS_SO_C_1;
			mole.c[ipATO][ipCS] -= rate_COmole.O_CS_SO_C_2;
			mole.b[ipCS] -= rate_COmole.b_O_CS_SO_C;
			
			mole.c[ipCS][ipATO] -= rate_COmole.O_CS_SO_C_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_CS_SO_C_2;
			mole.b[ipATO] -= rate_COmole.b_O_CS_SO_C;
			
			mole.c[ipCS][ipSO] += rate_COmole.O_CS_SO_C_1;
			mole.c[ipATO][ipSO] += rate_COmole.O_CS_SO_C_2;
			mole.b[ipSO] += rate_COmole.b_O_CS_SO_C;

			mole.c[ipCS][ipATC] += rate_COmole.O_CS_SO_C_1;
			mole.c[ipATO][ipATC] += rate_COmole.O_CS_SO_C_2;
			mole.b[ipATC] += rate_COmole.b_O_CS_SO_C;

		/*  >>refer   mol  O      +  NS    =>   S     +     NO    UMIST  */

			mole.c[ipNS][ipNS] -= rate_COmole.O_NS_S_NO_1;
			mole.c[ipATO][ipNS] -= rate_COmole.O_NS_S_NO_2;
			mole.b[ipNS] -= rate_COmole.b_O_NS_S_NO;
			
			mole.c[ipNS][ipATO] -= rate_COmole.O_NS_S_NO_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NS_S_NO_2;
			mole.b[ipATO] -= rate_COmole.b_O_NS_S_NO;
			
			mole.c[ipNS][ipNO] += rate_COmole.O_NS_S_NO_1;
			mole.c[ipATO][ipNO] += rate_COmole.O_NS_S_NO_2;
			mole.b[ipNO] += rate_COmole.b_O_NS_S_NO;

			mole.c[ipNS][ipATS] += rate_COmole.O_NS_S_NO_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_NS_S_NO_2;
			mole.b[ipATS] += rate_COmole.b_O_NS_S_NO;

		/*  >>refer   mol  O      +  NS    =>   SO    +     N     UMIST  */

			mole.c[ipNS][ipNS] -= rate_COmole.O_NS_SO_N_1;
			mole.c[ipATO][ipNS] -= rate_COmole.O_NS_SO_N_2;
			mole.b[ipNS] -= rate_COmole.b_O_NS_SO_N;
			
			mole.c[ipNS][ipATO] -= rate_COmole.O_NS_SO_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NS_SO_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_NS_SO_N;
			
			mole.c[ipNS][ipSO] += rate_COmole.O_NS_SO_N_1;
			mole.c[ipATO][ipSO] += rate_COmole.O_NS_SO_N_2;
			mole.b[ipSO] += rate_COmole.b_O_NS_SO_N;

			mole.c[ipNS][ipATN] += rate_COmole.O_NS_SO_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_NS_SO_N_2;
			mole.b[ipATN] += rate_COmole.b_O_NS_SO_N;

		/*  >>refer   mol  O      +  SO    =>   S     +     O2    UMIST  */

			mole.c[ipSO][ipSO] -= rate_COmole.O_SO_S_O2_1;
			mole.c[ipATO][ipSO] -= rate_COmole.O_SO_S_O2_2;
			mole.b[ipSO] -= rate_COmole.b_O_SO_S_O2;
			
			mole.c[ipSO][ipATO] -= rate_COmole.O_SO_S_O2_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_SO_S_O2_2;
			mole.b[ipATO] -= rate_COmole.b_O_SO_S_O2;
			
			mole.c[ipSO][ipO2] += rate_COmole.O_SO_S_O2_1;
			mole.c[ipATO][ipO2] += rate_COmole.O_SO_S_O2_2;
			mole.b[ipO2] += rate_COmole.b_O_SO_S_O2;

			mole.c[ipSO][ipATS] += rate_COmole.O_SO_S_O2_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_SO_S_O2_2;
			mole.b[ipATS] += rate_COmole.b_O_SO_S_O2;

		/*  >>refer   mol  O      +  S2    =>   SO    +     S     UMIST  */

			mole.c[ipS2][ipS2] -= rate_COmole.O_S2_SO_S_1;
			mole.c[ipATO][ipS2] -= rate_COmole.O_S2_SO_S_2;
			mole.b[ipS2] -= rate_COmole.b_O_S2_SO_S;
			
			mole.c[ipS2][ipATO] -= rate_COmole.O_S2_SO_S_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_S2_SO_S_2;
			mole.b[ipATO] -= rate_COmole.b_O_S2_SO_S;
			
			mole.c[ipS2][ipSO] += rate_COmole.O_S2_SO_S_1;
			mole.c[ipATO][ipSO] += rate_COmole.O_S2_SO_S_2;
			mole.b[ipSO] += rate_COmole.b_O_S2_SO_S;

			mole.c[ipS2][ipATS] += rate_COmole.O_S2_SO_S_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_S2_SO_S_2;
			mole.b[ipATS] += rate_COmole.b_O_S2_SO_S;

		/*  >>refer   mol  NH2    +  OH    =>   NH3   +     O     UMIST  */
			
			mole.c[ipOH][ipNH2] -= rate_COmole.NH2_OH_NH3_O_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_OH_NH3_O_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_OH_NH3_O;

			mole.c[ipOH][ipOH] -= rate_COmole.NH2_OH_NH3_O_1;
			mole.c[ipNH2][ipOH] -= rate_COmole.NH2_OH_NH3_O_2;
			mole.b[ipOH] -= rate_COmole.b_NH2_OH_NH3_O;
			
			mole.c[ipOH][ipNH3] += rate_COmole.NH2_OH_NH3_O_1;
			mole.c[ipNH2][ipNH3] += rate_COmole.NH2_OH_NH3_O_2;
			mole.b[ipNH3] += rate_COmole.b_NH2_OH_NH3_O;
			
			mole.c[ipOH][ipATO] += rate_COmole.NH2_OH_NH3_O_1;
			mole.c[ipNH2][ipATO] += rate_COmole.NH2_OH_NH3_O_2;
			mole.b[ipATO] += rate_COmole.b_NH2_OH_NH3_O;

		/*  >>refer   mol  NH2    +  OH    =>   H2O   +     NH    UMIST  */
			
			mole.c[ipOH][ipNH2] -= rate_COmole.NH2_OH_H2O_NH_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_OH_H2O_NH_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_OH_H2O_NH;
			
			mole.c[ipOH][ipOH] -= rate_COmole.NH2_OH_H2O_NH_1;
			mole.c[ipNH2][ipOH] -= rate_COmole.NH2_OH_H2O_NH_2;
			mole.b[ipOH] -= rate_COmole.b_NH2_OH_H2O_NH;
			
			mole.c[ipOH][ipH2O] += rate_COmole.NH2_OH_H2O_NH_1;
			mole.c[ipNH2][ipH2O] += rate_COmole.NH2_OH_H2O_NH_2;
			mole.b[ipH2O] += rate_COmole.b_NH2_OH_H2O_NH;
			
			mole.c[ipOH][ipNH] += rate_COmole.NH2_OH_H2O_NH_1;
			mole.c[ipNH2][ipNH] += rate_COmole.NH2_OH_H2O_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2_OH_H2O_NH;

		/*  >>refer   mol  NH2    +  NO    =>   N2    +     OH +  H      UMIST  */

			mole.c[ipNO][ipNH2] -= rate_COmole.NH2_NO_N2_OH_H_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_NO_N2_OH_H_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_NO_N2_OH_H;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH2_NO_N2_OH_H_1;
			mole.c[ipNH2][ipNO] -= rate_COmole.NH2_NO_N2_OH_H_2;
			mole.b[ipNO] -= rate_COmole.b_NH2_NO_N2_OH_H;
			
			mole.c[ipNO][ipN2] += rate_COmole.NH2_NO_N2_OH_H_1;
			mole.c[ipNH2][ipN2] += rate_COmole.NH2_NO_N2_OH_H_2;
			mole.b[ipN2] += rate_COmole.b_NH2_NO_N2_OH_H;
			
			mole.c[ipNO][ipOH] += rate_COmole.NH2_NO_N2_OH_H_1;
			mole.c[ipNH2][ipOH] += rate_COmole.NH2_NO_N2_OH_H_2;
			mole.b[ipOH] += rate_COmole.b_NH2_NO_N2_OH_H;

		/*  >>refer   mol  NH2    +  NO    =>   N2    +     H2O   UMIST  */

			mole.c[ipNO][ipNH2] -= rate_COmole.NH2_NO_N2_H2O_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_NO_N2_H2O_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_NO_N2_H2O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH2_NO_N2_H2O_1;
			mole.c[ipNH2][ipNO] -= rate_COmole.NH2_NO_N2_H2O_2;
			mole.b[ipNO] -= rate_COmole.b_NH2_NO_N2_H2O;
			
			mole.c[ipNO][ipN2] += rate_COmole.NH2_NO_N2_H2O_1;
			mole.c[ipNH2][ipN2] += rate_COmole.NH2_NO_N2_H2O_2;
			mole.b[ipN2] += rate_COmole.b_NH2_NO_N2_H2O;
			
			mole.c[ipNO][ipH2O] += rate_COmole.NH2_NO_N2_H2O_1;
			mole.c[ipNH2][ipH2O] += rate_COmole.NH2_NO_N2_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_NH2_NO_N2_H2O;

		/*  >>refer   mol  CH4    +  CN    =>   HCN   +     CH3   UMIST  */

			mole.c[ipCN][ipCH4] -= rate_COmole.CH4_CN_HCN_CH3_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_CN_HCN_CH3_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_CN_HCN_CH3;
			
			mole.c[ipCN][ipCN] -= rate_COmole.CH4_CN_HCN_CH3_1;
			mole.c[ipCH4][ipCN] -= rate_COmole.CH4_CN_HCN_CH3_2;
			mole.b[ipCN] -= rate_COmole.b_CH4_CN_HCN_CH3;
			
			mole.c[ipCN][ipHCN] += rate_COmole.CH4_CN_HCN_CH3_1;
			mole.c[ipCH4][ipHCN] += rate_COmole.CH4_CN_HCN_CH3_2;
			mole.b[ipHCN] += rate_COmole.b_CH4_CN_HCN_CH3;
			
			mole.c[ipCN][ipCH3] += rate_COmole.CH4_CN_HCN_CH3_1;
			mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CN_HCN_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH4_CN_HCN_CH3;

		/*  >>refer   mol  OH     +  NH3   =>   H2O   +     NH2   UMIST  */

			mole.c[ipNH3][ipOH] -= rate_COmole.OH_NH3_H2O_NH2_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_NH3_H2O_NH2_2;
			mole.b[ipOH] -= rate_COmole.b_OH_NH3_H2O_NH2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.OH_NH3_H2O_NH2_1;
			mole.c[ipOH][ipNH3] -= rate_COmole.OH_NH3_H2O_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_OH_NH3_H2O_NH2;
			
			mole.c[ipNH3][ipH2O] += rate_COmole.OH_NH3_H2O_NH2_1;
			mole.c[ipOH][ipH2O] += rate_COmole.OH_NH3_H2O_NH2_2;
			mole.b[ipH2O] += rate_COmole.b_OH_NH3_H2O_NH2;
			
			mole.c[ipNH3][ipNH2] += rate_COmole.OH_NH3_H2O_NH2_1;
			mole.c[ipOH][ipNH2] += rate_COmole.OH_NH3_H2O_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_OH_NH3_H2O_NH2;

		/*  >>refer   mol  OH     +  CN    =>   HCN   +     O     UMIST  */

			mole.c[ipCN][ipOH] -= rate_COmole.OH_CN_HCN_O_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_CN_HCN_O_2;
			mole.b[ipOH] -= rate_COmole.b_OH_CN_HCN_O;
			
			mole.c[ipCN][ipCN] -= rate_COmole.OH_CN_HCN_O_1;
			mole.c[ipOH][ipCN] -= rate_COmole.OH_CN_HCN_O_2;
			mole.b[ipCN] -= rate_COmole.b_OH_CN_HCN_O;
			
			mole.c[ipCN][ipHCN] += rate_COmole.OH_CN_HCN_O_1;
			mole.c[ipOH][ipHCN] += rate_COmole.OH_CN_HCN_O_2;
			mole.b[ipHCN] += rate_COmole.b_OH_CN_HCN_O;
			
			mole.c[ipCN][ipATO] += rate_COmole.OH_CN_HCN_O_1;
			mole.c[ipOH][ipATO] += rate_COmole.OH_CN_HCN_O_2;
			mole.b[ipATO] += rate_COmole.b_OH_CN_HCN_O;
			
		/*  >>refer   mol  OH     +  CN    =>   OCN   +     H     UMIST  */

			mole.c[ipCN][ipOH] -= rate_COmole.OH_CN_OCN_H_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_CN_OCN_H_2;
			mole.b[ipOH] -= rate_COmole.b_OH_CN_OCN_H;
			
			mole.c[ipCN][ipCN] -= rate_COmole.OH_CN_OCN_H_1;
			mole.c[ipOH][ipCN] -= rate_COmole.OH_CN_OCN_H_2;
			mole.b[ipCN] -= rate_COmole.b_OH_CN_OCN_H;
			
			mole.c[ipCN][ipOCN] += rate_COmole.OH_CN_OCN_H_1;
			mole.c[ipOH][ipOCN] += rate_COmole.OH_CN_OCN_H_2;
			mole.b[ipOCN] += rate_COmole.b_OH_CN_OCN_H;

		/*  >>refer   mol  OH     +  HCN   =>   CN    +     H2O   UMIST  */

			mole.c[ipHCN][ipOH] -= rate_COmole.OH_HCN_CN_H2O_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_HCN_CN_H2O_2;
			mole.b[ipOH] -= rate_COmole.b_OH_HCN_CN_H2O;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.OH_HCN_CN_H2O_1;
			mole.c[ipOH][ipHCN] -= rate_COmole.OH_HCN_CN_H2O_2;
			mole.b[ipHCN] -= rate_COmole.b_OH_HCN_CN_H2O;
			
			mole.c[ipHCN][ipCN] += rate_COmole.OH_HCN_CN_H2O_1;
			mole.c[ipOH][ipCN] += rate_COmole.OH_HCN_CN_H2O_2;
			mole.b[ipCN] += rate_COmole.b_OH_HCN_CN_H2O;
			
			mole.c[ipHCN][ipH2O] += rate_COmole.OH_HCN_CN_H2O_1;
			mole.c[ipOH][ipH2O] += rate_COmole.OH_HCN_CN_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_OH_HCN_CN_H2O;

		/*  >>refer   mol  OH     +  NO    =>   NO2   +     H     UMIST  */

			mole.c[ipNO][ipOH] -= rate_COmole.OH_NO_NO2_H_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_NO_NO2_H_2;
			mole.b[ipOH] -= rate_COmole.b_OH_NO_NO2_H;
			
			mole.c[ipNO][ipNO] -= rate_COmole.OH_NO_NO2_H_1;
			mole.c[ipOH][ipNO] -= rate_COmole.OH_NO_NO2_H_2;
			mole.b[ipNO] -= rate_COmole.b_OH_NO_NO2_H;
			
			mole.c[ipNO][ipNO2] += rate_COmole.OH_NO_NO2_H_1;
			mole.c[ipOH][ipNO2] += rate_COmole.OH_NO_NO2_H_2;
			mole.b[ipNO2] += rate_COmole.b_OH_NO_NO2_H;
			
		/*  >>refer   mol  OH     +  S     =>   HS    +     O     UMIST  */

			mole.c[ipATS][ipOH] -= rate_COmole.OH_S_HS_O_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_S_HS_O_2;
			mole.b[ipOH] -= rate_COmole.b_OH_S_HS_O;
			
			mole.c[ipATS][ipATS] -= rate_COmole.OH_S_HS_O_1;
			mole.c[ipOH][ipATS] -= rate_COmole.OH_S_HS_O_2;
			mole.b[ipATS] -= rate_COmole.b_OH_S_HS_O;
			
			mole.c[ipATS][ipHS] += rate_COmole.OH_S_HS_O_1;
			mole.c[ipOH][ipHS] += rate_COmole.OH_S_HS_O_2;
			mole.b[ipHS] += rate_COmole.b_OH_S_HS_O;

			mole.c[ipATS][ipATO] += rate_COmole.OH_S_HS_O_1;
			mole.c[ipOH][ipATO] += rate_COmole.OH_S_HS_O_2;
			mole.b[ipATO] += rate_COmole.b_OH_S_HS_O;

		/*  >>refer   mol  OH     +  S     =>   SO    +     H     UMIST  */

			mole.c[ipATS][ipOH] -= rate_COmole.OH_S_SO_H_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_S_SO_H_2;
			mole.b[ipOH] -= rate_COmole.b_OH_S_SO_H;

			mole.c[ipATS][ipATS] -= rate_COmole.OH_S_SO_H_1;
			mole.c[ipOH][ipATS] -= rate_COmole.OH_S_SO_H_2;
			mole.b[ipATS] -= rate_COmole.b_OH_S_SO_H;
			
			mole.c[ipATS][ipSO] += rate_COmole.OH_S_SO_H_1;
			mole.c[ipOH][ipSO] += rate_COmole.OH_S_SO_H_2;
			mole.b[ipSO] += rate_COmole.b_OH_S_SO_H;

		/*  >>refer   mol  OH     +  N2O   =>   HNO   +     NO    UMIST  */

			mole.c[ipN2O][ipOH] -= rate_COmole.OH_N2O_HNO_NO_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_N2O_HNO_NO_2;
			mole.b[ipOH] -= rate_COmole.b_OH_N2O_HNO_NO;
			
			mole.c[ipN2O][ipN2O] -= rate_COmole.OH_N2O_HNO_NO_1;
			mole.c[ipOH][ipN2O] -= rate_COmole.OH_N2O_HNO_NO_2;
			mole.b[ipN2O] -= rate_COmole.b_OH_N2O_HNO_NO;
			
			mole.c[ipN2O][ipHNO] += rate_COmole.OH_N2O_HNO_NO_1;
			mole.c[ipOH][ipHNO] += rate_COmole.OH_N2O_HNO_NO_2;
			mole.b[ipHNO] += rate_COmole.b_OH_N2O_HNO_NO;
			
			mole.c[ipN2O][ipNO] += rate_COmole.OH_N2O_HNO_NO_1;
			mole.c[ipOH][ipNO] += rate_COmole.OH_N2O_HNO_NO_2;
			mole.b[ipNO] += rate_COmole.b_OH_N2O_HNO_NO;

		/*  >>refer   mol  OH     +  CS    =>   OCS   +     H     UMIST  */
			
			mole.c[ipCS][ipOH] -= rate_COmole.OH_CS_OCS_H_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_CS_OCS_H_2;
			mole.b[ipOH] -= rate_COmole.b_OH_CS_OCS_H;

			mole.c[ipCS][ipCS] -= rate_COmole.OH_CS_OCS_H_1;
			mole.c[ipOH][ipCS] -= rate_COmole.OH_CS_OCS_H_2;
			mole.b[ipCS] -= rate_COmole.b_OH_CS_OCS_H;
			
			mole.c[ipCS][ipOCS] += rate_COmole.OH_CS_OCS_H_1;
			mole.c[ipOH][ipOCS] += rate_COmole.OH_CS_OCS_H_2;
			mole.b[ipOCS] += rate_COmole.b_OH_CS_OCS_H;
			
		/*  >>refer   mol  NH3    +  CN    =>   HCN   +     NH2   UMIST  */

			mole.c[ipCN][ipNH3] -= rate_COmole.NH3_CN_HCN_NH2_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_CN_HCN_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_CN_HCN_NH2;
			
			mole.c[ipCN][ipCN] -= rate_COmole.NH3_CN_HCN_NH2_1;
			mole.c[ipNH3][ipCN] -= rate_COmole.NH3_CN_HCN_NH2_2;
			mole.b[ipCN] -= rate_COmole.b_NH3_CN_HCN_NH2;
			
			mole.c[ipCN][ipHCN] += rate_COmole.NH3_CN_HCN_NH2_1;
			mole.c[ipNH3][ipHCN] += rate_COmole.NH3_CN_HCN_NH2_2;
			mole.b[ipHCN] += rate_COmole.b_NH3_CN_HCN_NH2;
			
			mole.c[ipCN][ipNH2] += rate_COmole.NH3_CN_HCN_NH2_1;
			mole.c[ipNH3][ipNH2] += rate_COmole.NH3_CN_HCN_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH3_CN_HCN_NH2;

		/*  >>refer   mol  CN     +  NO    =>   OCN   +     N     UMIST  */

			mole.c[ipNO][ipCN] -= rate_COmole.CN_NO_OCN_N_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_NO_OCN_N_2;
			mole.b[ipCN] -= rate_COmole.b_CN_NO_OCN_N;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CN_NO_OCN_N_1;
			mole.c[ipCN][ipNO] -= rate_COmole.CN_NO_OCN_N_2;
			mole.b[ipNO] -= rate_COmole.b_CN_NO_OCN_N;
			
			mole.c[ipNO][ipOCN] += rate_COmole.CN_NO_OCN_N_1;
			mole.c[ipCN][ipOCN] += rate_COmole.CN_NO_OCN_N_2;
			mole.b[ipOCN] += rate_COmole.b_CN_NO_OCN_N;
			
			mole.c[ipNO][ipATN] += rate_COmole.CN_NO_OCN_N_1;
			mole.c[ipCN][ipATN] += rate_COmole.CN_NO_OCN_N_2;
			mole.b[ipATN] += rate_COmole.b_CN_NO_OCN_N;
			
		/*  >>refer   mol  CN     +  HNO   =>   NO    +     HCN   UMIST  */

			mole.c[ipHNO][ipCN] -= rate_COmole.CN_HNO_NO_HCN_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_HNO_NO_HCN_2;
			mole.b[ipCN] -= rate_COmole.b_CN_HNO_NO_HCN;
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.CN_HNO_NO_HCN_1;
			mole.c[ipCN][ipHNO] -= rate_COmole.CN_HNO_NO_HCN_2;
			mole.b[ipHNO] -= rate_COmole.b_CN_HNO_NO_HCN;
			
			mole.c[ipHNO][ipNO] += rate_COmole.CN_HNO_NO_HCN_1;
			mole.c[ipCN][ipNO] += rate_COmole.CN_HNO_NO_HCN_2;
			mole.b[ipNO] += rate_COmole.b_CN_HNO_NO_HCN;
			
			mole.c[ipHNO][ipHCN] += rate_COmole.CN_HNO_NO_HCN_1;
			mole.c[ipCN][ipHCN] += rate_COmole.CN_HNO_NO_HCN_2;
			mole.b[ipHCN] += rate_COmole.b_CN_HNO_NO_HCN;

		/*  >>refer   mol  CN     +  S     =>   NS    +     C     UMIST  */

			mole.c[ipATS][ipCN] -= rate_COmole.CN_S_NS_C_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_S_NS_C_2;
			mole.b[ipCN] -= rate_COmole.b_CN_S_NS_C;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CN_S_NS_C_1;
			mole.c[ipCN][ipATS] -= rate_COmole.CN_S_NS_C_2;
			mole.b[ipATS] -= rate_COmole.b_CN_S_NS_C;
			
			mole.c[ipATS][ipNS] += rate_COmole.CN_S_NS_C_1;
			mole.c[ipCN][ipNS] += rate_COmole.CN_S_NS_C_2;
			mole.b[ipNS] += rate_COmole.b_CN_S_NS_C;

			mole.c[ipATS][ipATC] += rate_COmole.CN_S_NS_C_1;
			mole.c[ipCN][ipATC] += rate_COmole.CN_S_NS_C_2;
			mole.b[ipATC] += rate_COmole.b_CN_S_NS_C;
			
		/*  >>refer   mol  CN     +  S     =>   CS    +     N     UMIST  */

			mole.c[ipATS][ipCN] -= rate_COmole.CN_S_CS_N_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_S_CS_N_2;
			mole.b[ipCN] -= rate_COmole.b_CN_S_CS_N;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CN_S_CS_N_1;
			mole.c[ipCN][ipATS] -= rate_COmole.CN_S_CS_N_2;
			mole.b[ipATS] -= rate_COmole.b_CN_S_CS_N;
			
			mole.c[ipATS][ipCS] += rate_COmole.CN_S_CS_N_1;
			mole.c[ipCN][ipCS] += rate_COmole.CN_S_CS_N_2;
			mole.b[ipCS] += rate_COmole.b_CN_S_CS_N;

			mole.c[ipATS][ipATN] += rate_COmole.CN_S_CS_N_1;
			mole.c[ipCN][ipATN] += rate_COmole.CN_S_CS_N_2;
			mole.b[ipATN] += rate_COmole.b_CN_S_CS_N;

		/*  >>refer   mol  N2     +  O2    =>   N2O   +     O     UMIST  */

			mole.c[ipO2][ipN2] -= rate_COmole.N2_O2_N2O_O_1;
			mole.c[ipN2][ipN2] -= rate_COmole.N2_O2_N2O_O_2;
			mole.b[ipN2] -= rate_COmole.b_N2_O2_N2O_O;
			
			mole.c[ipO2][ipO2] -= rate_COmole.N2_O2_N2O_O_1;
			mole.c[ipN2][ipO2] -= rate_COmole.N2_O2_N2O_O_2;
			mole.b[ipO2] -= rate_COmole.b_N2_O2_N2O_O;
			
			mole.c[ipO2][ipN2O] += rate_COmole.N2_O2_N2O_O_1;
			mole.c[ipN2][ipN2O] += rate_COmole.N2_O2_N2O_O_2;
			mole.b[ipN2O] += rate_COmole.b_N2_O2_N2O_O;
			
			mole.c[ipO2][ipATO] += rate_COmole.N2_O2_N2O_O_1;
			mole.c[ipN2][ipATO] += rate_COmole.N2_O2_N2O_O_2;
			mole.b[ipATO] += rate_COmole.b_N2_O2_N2O_O;
			
		/*  >>refer   mol  NO     +  NO    =>   O2    +     N2    UMIST  */

			mole.c[ipNO][ipNO] -= rate_COmole.NO_NO_O2_N2_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_NO_O2_N2_2;
			mole.b[ipNO] -= rate_COmole.b_NO_NO_O2_N2;
			
			mole.c[ipNO][ipO2] += 0.5*rate_COmole.NO_NO_O2_N2_1;
			mole.c[ipNO][ipO2] += 0.5*rate_COmole.NO_NO_O2_N2_2;
			mole.b[ipO2] += 0.5*rate_COmole.b_NO_NO_O2_N2;
			
			mole.c[ipNO][ipN2] += 0.5*rate_COmole.NO_NO_O2_N2_1;
			mole.c[ipNO][ipN2] += 0.5*rate_COmole.NO_NO_O2_N2_2;
			mole.b[ipN2] += 0.5*rate_COmole.b_NO_NO_O2_N2;

		/*  >>refer   mol  NO     +  NO    =>   N2O   +     O     UMIST  */

			mole.c[ipNO][ipNO] -= rate_COmole.NO_NO_N2O_O_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_NO_N2O_O_2;
			mole.b[ipNO] -= rate_COmole.b_NO_NO_N2O_O;
			
			mole.c[ipNO][ipN2O] += 0.5*rate_COmole.NO_NO_N2O_O_1;
			mole.c[ipNO][ipN2O] += 0.5*rate_COmole.NO_NO_N2O_O_2;
			mole.b[ipN2O] += 0.5*rate_COmole.b_NO_NO_N2O_O;
			
			mole.c[ipNO][ipATO] += 0.5*rate_COmole.NO_NO_N2O_O_1;
			mole.c[ipNO][ipATO] += 0.5*rate_COmole.NO_NO_N2O_O_2;
			mole.b[ipATO] += 0.5*rate_COmole.b_NO_NO_N2O_O;
			
		/*  >>refer   mol  NO     +  HNO   =>   N2O   +     OH    UMIST  */

			mole.c[ipHNO][ipNO] -= rate_COmole.NO_HNO_N2O_OH_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_HNO_N2O_OH_2;
			mole.b[ipNO] -= rate_COmole.b_NO_HNO_N2O_OH;
			
			mole.c[ipHNO][ipHNO] -= rate_COmole.NO_HNO_N2O_OH_1;
			mole.c[ipNO][ipHNO] -= rate_COmole.NO_HNO_N2O_OH_2;
			mole.b[ipHNO] -= rate_COmole.b_NO_HNO_N2O_OH;
			
			mole.c[ipHNO][ipN2O] += rate_COmole.NO_HNO_N2O_OH_1;
			mole.c[ipNO][ipN2O] += rate_COmole.NO_HNO_N2O_OH_2;
			mole.b[ipN2O] += rate_COmole.b_NO_HNO_N2O_OH;
			
			mole.c[ipHNO][ipOH] += rate_COmole.NO_HNO_N2O_OH_1;
			mole.c[ipNO][ipOH] += rate_COmole.NO_HNO_N2O_OH_2;
			mole.b[ipOH] += rate_COmole.b_NO_HNO_N2O_OH;

		/*  >>refer   mol  NO     +  S     =>   NS    +     O     UMIST  */

			mole.c[ipATS][ipNO] -= rate_COmole.NO_S_NS_O_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_S_NS_O_2;
			mole.b[ipNO] -= rate_COmole.b_NO_S_NS_O;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NO_S_NS_O_1;
			mole.c[ipNO][ipATS] -= rate_COmole.NO_S_NS_O_2;
			mole.b[ipATS] -= rate_COmole.b_NO_S_NS_O;
			
			mole.c[ipATS][ipNS] += rate_COmole.NO_S_NS_O_1;
			mole.c[ipNO][ipNS] += rate_COmole.NO_S_NS_O_2;
			mole.b[ipNS] += rate_COmole.b_NO_S_NS_O;
	
			mole.c[ipATS][ipATO] += rate_COmole.NO_S_NS_O_1;
			mole.c[ipNO][ipATO] += rate_COmole.NO_S_NS_O_2;
			mole.b[ipATO] += rate_COmole.b_NO_S_NS_O;
			
		/*  >>refer   mol  NO     +  S     =>   SO    +     N     UMIST  */

			mole.c[ipATS][ipNO] -= rate_COmole.NO_S_SO_N_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_S_SO_N_2;
			mole.b[ipNO] -= rate_COmole.b_NO_S_SO_N;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NO_S_SO_N_1;
			mole.c[ipNO][ipATS] -= rate_COmole.NO_S_SO_N_2;
			mole.b[ipATS] -= rate_COmole.b_NO_S_SO_N;
			
			mole.c[ipATS][ipSO] += rate_COmole.NO_S_SO_N_1;
			mole.c[ipNO][ipSO] += rate_COmole.NO_S_SO_N_2;
			mole.b[ipSO] += rate_COmole.b_NO_S_SO_N;

			mole.c[ipATS][ipATN] += rate_COmole.NO_S_SO_N_1;
			mole.c[ipNO][ipATN] += rate_COmole.NO_S_SO_N_2;
			mole.b[ipATN] += rate_COmole.b_NO_S_SO_N;
			
		/*  >>refer   mol  O2     +  S     =>   SO    +     O     UMIST  */

			mole.c[ipATS][ipO2] -= rate_COmole.O2_S_SO_O_1;
			mole.c[ipO2][ipO2] -= rate_COmole.O2_S_SO_O_2;
			mole.b[ipO2] -= rate_COmole.b_O2_S_SO_O;
			
			mole.c[ipATS][ipATS] -= rate_COmole.O2_S_SO_O_1;
			mole.c[ipO2][ipATS] -= rate_COmole.O2_S_SO_O_2;
			mole.b[ipATS] -= rate_COmole.b_O2_S_SO_O;
			
			mole.c[ipATS][ipSO] += rate_COmole.O2_S_SO_O_1;
			mole.c[ipO2][ipSO] += rate_COmole.O2_S_SO_O_2;
			mole.b[ipSO] += rate_COmole.b_O2_S_SO_O;

			mole.c[ipATS][ipATO] += rate_COmole.O2_S_SO_O_1;
			mole.c[ipO2][ipATO] += rate_COmole.O2_S_SO_O_2;
			mole.b[ipATO] += rate_COmole.b_O2_S_SO_O;
			
		/*  >>refer   mol  S      +  SO    =>   S2    +     O     UMIST  */
			
			mole.c[ipSO][ipSO] -= rate_COmole.S_SO_S2_O_1;
			mole.c[ipATS][ipSO] -= rate_COmole.S_SO_S2_O_2;
			mole.b[ipSO] -= rate_COmole.b_S_SO_S2_O;
						
			mole.c[ipSO][ipATS] -= rate_COmole.S_SO_S2_O_1;
			mole.c[ipATS][ipATS] -= rate_COmole.S_SO_S2_O_2;
			mole.b[ipATS] -= rate_COmole.b_S_SO_S2_O;
			
			mole.c[ipSO][ipS2] += rate_COmole.S_SO_S2_O_1;
			mole.c[ipATS][ipS2] += rate_COmole.S_SO_S2_O_2;
			mole.b[ipS2] += rate_COmole.b_S_SO_S2_O;

			mole.c[ipSO][ipATO] += rate_COmole.S_SO_S2_O_1;
			mole.c[ipATS][ipATO] += rate_COmole.S_SO_S2_O_2;
			mole.b[ipATO] += rate_COmole.b_S_SO_S2_O;
			


		/*  >>refer   mol  C      +  NH+   =>   N     +     CH+   UMIST  */
			
			mole.c[ipNHP][ipNHP] -= rate_COmole.C_NHP_N_CHP_1;
			mole.c[ipATC][ipNHP] -= rate_COmole.C_NHP_N_CHP_2;
			mole.b[ipNHP] -= rate_COmole.b_C_NHP_N_CHP;
						
			mole.c[ipNHP][ipATC] -= rate_COmole.C_NHP_N_CHP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NHP_N_CHP_2;
			mole.b[ipATC] -= rate_COmole.b_C_NHP_N_CHP;
			
			mole.c[ipNHP][ipCHP] += rate_COmole.C_NHP_N_CHP_1;
			mole.c[ipATC][ipCHP] += rate_COmole.C_NHP_N_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_C_NHP_N_CHP;

			mole.c[ipNHP][ipATN] += rate_COmole.C_NHP_N_CHP_1;
			mole.c[ipATC][ipATN] += rate_COmole.C_NHP_N_CHP_2;
			mole.b[ipATN] += rate_COmole.b_C_NHP_N_CHP;

		/*  >>refer   mol  C+     +  NH    =>   CN+   +     H     UMIST  */

			mole.c[ipNH][ipCP] -= rate_COmole.CP_NH_CNP_H_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NH_CNP_H_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NH_CNP_H;
			
			mole.c[ipNH][ipNH] -= rate_COmole.CP_NH_CNP_H_1;
			mole.c[ipCP][ipNH] -= rate_COmole.CP_NH_CNP_H_2;
			mole.b[ipNH] -= rate_COmole.b_CP_NH_CNP_H;
			
			mole.c[ipNH][ipCNP] += rate_COmole.CP_NH_CNP_H_1;
			mole.c[ipCP][ipCNP] += rate_COmole.CP_NH_CNP_H_2;
			mole.b[ipCNP] += rate_COmole.b_CP_NH_CNP_H;

			/*  >>refer   mol  C+     +  NH    =>   CH+   +     N     Zsargo & Federman 2003  */

			mole.c[ipNH][ipCP] -= rate_COmole.CP_NH_CHP_N_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NH_CHP_N_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NH_CHP_N;
			
			mole.c[ipNH][ipNH] -= rate_COmole.CP_NH_CHP_N_1;
			mole.c[ipCP][ipNH] -= rate_COmole.CP_NH_CHP_N_2;
			mole.b[ipNH] -= rate_COmole.b_CP_NH_CHP_N;
			
			mole.c[ipNH][ipCHP] += rate_COmole.CP_NH_CHP_N_1;
			mole.c[ipCP][ipCHP] += rate_COmole.CP_NH_CHP_N_2;
			mole.b[ipCHP] += rate_COmole.b_CP_NH_CHP_N;

			mole.c[ipNH][ipATN] += rate_COmole.CP_NH_CHP_N_1;
			mole.c[ipCP][ipATN] += rate_COmole.CP_NH_CHP_N_2;
			mole.b[ipATN] += rate_COmole.b_CP_NH_CHP_N;


		/*  >>refer   mol  C+     +  NH2   =>   HCN+  +     H     UMIST  */

			mole.c[ipNH2][ipCP] -= rate_COmole.CP_NH2_HCNP_H_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NH2_HCNP_H_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NH2_HCNP_H;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.CP_NH2_HCNP_H_1;
			mole.c[ipCP][ipNH2] -= rate_COmole.CP_NH2_HCNP_H_2;
			mole.b[ipNH2] -= rate_COmole.b_CP_NH2_HCNP_H;
			
			mole.c[ipNH2][ipHCNP] += rate_COmole.CP_NH2_HCNP_H_1;
			mole.c[ipCP][ipHCNP] += rate_COmole.CP_NH2_HCNP_H_2;
			mole.b[ipHCNP] += rate_COmole.b_CP_NH2_HCNP_H;

		/*  >>refer   mol  C      +  NH2+  =>   NH    +     CH+   UMIST  */

			mole.c[ipNH2P][ipNH2P] -= rate_COmole.C_NH2P_NH_CHP_1;
			mole.c[ipATC][ipNH2P] -= rate_COmole.C_NH2P_NH_CHP_2;
			mole.b[ipNH2P] -= rate_COmole.b_C_NH2P_NH_CHP;

			mole.c[ipNH2P][ipATC] -= rate_COmole.C_NH2P_NH_CHP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NH2P_NH_CHP_2;
			mole.b[ipATC] -= rate_COmole.b_C_NH2P_NH_CHP;
			
			mole.c[ipNH2P][ipNH] += rate_COmole.C_NH2P_NH_CHP_1;
			mole.c[ipATC][ipNH] += rate_COmole.C_NH2P_NH_CHP_2;
			mole.b[ipNH] += rate_COmole.b_C_NH2P_NH_CHP;
			
			mole.c[ipNH2P][ipCHP] += rate_COmole.C_NH2P_NH_CHP_1;
			mole.c[ipATC][ipCHP] += rate_COmole.C_NH2P_NH_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_C_NH2P_NH_CHP;

		/*  >>refer   mol  C+     +  NH3   =>   HCN+  +     H2    UMIST  */

			mole.c[ipNH3][ipCP] -= rate_COmole.CP_NH3_HCNP_H2_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NH3_HCNP_H2_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NH3_HCNP_H2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CP_NH3_HCNP_H2_1;
			mole.c[ipCP][ipNH3] -= rate_COmole.CP_NH3_HCNP_H2_2;
			mole.b[ipNH3] -= rate_COmole.b_CP_NH3_HCNP_H2;
			
			mole.c[ipNH3][ipHCNP] += rate_COmole.CP_NH3_HCNP_H2_1;
			mole.c[ipCP][ipHCNP] += rate_COmole.CP_NH3_HCNP_H2_2;
			mole.b[ipHCNP] += rate_COmole.b_CP_NH3_HCNP_H2;

		/*  >>refer   mol  C      +  NH3+  =>   NH    +     CH2+  UMIST  */

			mole.c[ipNH3P][ipNH3P] -= rate_COmole.C_NH3P_NH_CH2P_1;
			mole.c[ipATC][ipNH3P] -= rate_COmole.C_NH3P_NH_CH2P_2;
			mole.b[ipNH3P] -= rate_COmole.b_C_NH3P_NH_CH2P;

			mole.c[ipNH3P][ipATC] -= rate_COmole.C_NH3P_NH_CH2P_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_NH3P_NH_CH2P_2;
			mole.b[ipATC] -= rate_COmole.b_C_NH3P_NH_CH2P;
			
			mole.c[ipNH3P][ipNH] += rate_COmole.C_NH3P_NH_CH2P_1;
			mole.c[ipATC][ipNH] += rate_COmole.C_NH3P_NH_CH2P_2;
			mole.b[ipNH] += rate_COmole.b_C_NH3P_NH_CH2P;
			
			mole.c[ipNH3P][ipCH2P] += rate_COmole.C_NH3P_NH_CH2P_1;
			mole.c[ipATC][ipCH2P] += rate_COmole.C_NH3P_NH_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_C_NH3P_NH_CH2P;

		/*  >>refer   mol  C      +  HCN+  =>   CN    +     CH+   UMIST  */

			mole.c[ipHCNP][ipHCNP] -= rate_COmole.C_HCNP_CN_CHP_1;
			mole.c[ipATC][ipHCNP] -= rate_COmole.C_HCNP_CN_CHP_2;
			mole.b[ipHCNP] -= rate_COmole.b_C_HCNP_CN_CHP;

			mole.c[ipHCNP][ipATC] -= rate_COmole.C_HCNP_CN_CHP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_HCNP_CN_CHP_2;
			mole.b[ipATC] -= rate_COmole.b_C_HCNP_CN_CHP;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.C_HCNP_CN_CHP_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_HCNP_CN_CHP_2;
			mole.b[ipCN] += rate_COmole.b_C_HCNP_CN_CHP;
			
			mole.c[ipHCNP][ipCHP] += rate_COmole.C_HCNP_CN_CHP_1;
			mole.c[ipATC][ipCHP] += rate_COmole.C_HCNP_CN_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_C_HCNP_CN_CHP;

		/*  >>refer   mol  C      +  HNO+  =>   NO    +     CH+   UMIST  */

			mole.c[ipHNOP][ipHNOP] -= rate_COmole.C_HNOP_NO_CHP_1;
			mole.c[ipATC][ipHNOP] -= rate_COmole.C_HNOP_NO_CHP_2;
			mole.b[ipHNOP] -= rate_COmole.b_C_HNOP_NO_CHP;

			mole.c[ipHNOP][ipATC] -= rate_COmole.C_HNOP_NO_CHP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_HNOP_NO_CHP_2;
			mole.b[ipATC] -= rate_COmole.b_C_HNOP_NO_CHP;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.C_HNOP_NO_CHP_1;
			mole.c[ipATC][ipNO] += rate_COmole.C_HNOP_NO_CHP_2;
			mole.b[ipNO] += rate_COmole.b_C_HNOP_NO_CHP;
			
			mole.c[ipHNOP][ipCHP] += rate_COmole.C_HNOP_NO_CHP_1;
			mole.c[ipATC][ipCHP] += rate_COmole.C_HNOP_NO_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_C_HNOP_NO_CHP;

		/*  >>refer   mol  C+     +  HS    =>   CS+   +     H     UMIST  */

			mole.c[ipHS][ipCP] -= rate_COmole.CP_HS_CSP_H_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_HS_CSP_H_2;
			mole.b[ipCP] -= rate_COmole.b_CP_HS_CSP_H;
			
			mole.c[ipHS][ipHS] -= rate_COmole.CP_HS_CSP_H_1;
			mole.c[ipCP][ipHS] -= rate_COmole.CP_HS_CSP_H_2;
			mole.b[ipHS] -= rate_COmole.b_CP_HS_CSP_H;
			
			mole.c[ipHS][ipCSP] += rate_COmole.CP_HS_CSP_H_1;
			mole.c[ipCP][ipCSP] += rate_COmole.CP_HS_CSP_H_2;
			mole.b[ipCSP] += rate_COmole.b_CP_HS_CSP_H;

		/*  >>refer   mol  C      +  HS+   =>   CS+   +     H     UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.C_HSP_CSP_H_1;
			mole.c[ipATC][ipHSP] -= rate_COmole.C_HSP_CSP_H_2;
			mole.b[ipHSP] -= rate_COmole.b_C_HSP_CSP_H;

			mole.c[ipHSP][ipATC] -= rate_COmole.C_HSP_CSP_H_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_HSP_CSP_H_2;
			mole.b[ipATC] -= rate_COmole.b_C_HSP_CSP_H;
			
			mole.c[ipHSP][ipCSP] += rate_COmole.C_HSP_CSP_H_1;
			mole.c[ipATC][ipCSP] += rate_COmole.C_HSP_CSP_H_2;
			mole.b[ipCSP] += rate_COmole.b_C_HSP_CSP_H;

		/*  >>refer   mol  C+     +  OCN   =>   CO+   +     CN    UMIST  */

			mole.c[ipOCN][ipCP] -= rate_COmole.CP_OCN_COP_CN_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_OCN_COP_CN_2;
			mole.b[ipCP] -= rate_COmole.b_CP_OCN_COP_CN;
			
			mole.c[ipOCN][ipOCN] -= rate_COmole.CP_OCN_COP_CN_1;
			mole.c[ipCP][ipOCN] -= rate_COmole.CP_OCN_COP_CN_2;
			mole.b[ipOCN] -= rate_COmole.b_CP_OCN_COP_CN;
			
			mole.c[ipOCN][ipCOP] += rate_COmole.CP_OCN_COP_CN_1;
			mole.c[ipCP][ipCOP] += rate_COmole.CP_OCN_COP_CN_2;
			mole.b[ipCOP] += rate_COmole.b_CP_OCN_COP_CN;
			
			mole.c[ipOCN][ipCN] += rate_COmole.CP_OCN_COP_CN_1;
			mole.c[ipCP][ipCN] += rate_COmole.CP_OCN_COP_CN_2;
			mole.b[ipCN] += rate_COmole.b_CP_OCN_COP_CN;

		/*  >>refer   mol  C+     +  NS    =>   CS+   +     N     UMIST  */

			mole.c[ipNS][ipCP] -= rate_COmole.CP_NS_CSP_N_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NS_CSP_N_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NS_CSP_N;
			
			mole.c[ipNS][ipNS] -= rate_COmole.CP_NS_CSP_N_1;
			mole.c[ipCP][ipNS] -= rate_COmole.CP_NS_CSP_N_2;
			mole.b[ipNS] -= rate_COmole.b_CP_NS_CSP_N;
			
			mole.c[ipNS][ipCSP] += rate_COmole.CP_NS_CSP_N_1;
			mole.c[ipCP][ipCSP] += rate_COmole.CP_NS_CSP_N_2;
			mole.b[ipCSP] += rate_COmole.b_CP_NS_CSP_N;
			
			mole.c[ipNS][ipATN] += rate_COmole.CP_NS_CSP_N_1;
			mole.c[ipCP][ipATN] += rate_COmole.CP_NS_CSP_N_2;
			mole.b[ipATN] += rate_COmole.b_CP_NS_CSP_N;

		/*  >>refer   mol  C+     +  SO    =>   S     +     CO+   UMIST  */

			mole.c[ipSO][ipCP] -= rate_COmole.CP_SO_S_COP_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_SO_S_COP_2;
			mole.b[ipCP] -= rate_COmole.b_CP_SO_S_COP;
			
			mole.c[ipSO][ipSO] -= rate_COmole.CP_SO_S_COP_1;
			mole.c[ipCP][ipSO] -= rate_COmole.CP_SO_S_COP_2;
			mole.b[ipSO] -= rate_COmole.b_CP_SO_S_COP;
			
			mole.c[ipSO][ipCOP] += rate_COmole.CP_SO_S_COP_1;
			mole.c[ipCP][ipCOP] += rate_COmole.CP_SO_S_COP_2;
			mole.b[ipCOP] += rate_COmole.b_CP_SO_S_COP;
			
			mole.c[ipSO][ipATS] += rate_COmole.CP_SO_S_COP_1;
			mole.c[ipCP][ipATS] += rate_COmole.CP_SO_S_COP_2;
			mole.b[ipATS] += rate_COmole.b_CP_SO_S_COP;

		/*  >>refer   mol  C+     +  SO    =>   CS+   +     O     UMIST  */

			mole.c[ipSO][ipCP] -= rate_COmole.CP_SO_CSP_O_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_SO_CSP_O_2;
			mole.b[ipCP] -= rate_COmole.b_CP_SO_CSP_O;
			
			mole.c[ipSO][ipSO] -= rate_COmole.CP_SO_CSP_O_1;
			mole.c[ipCP][ipSO] -= rate_COmole.CP_SO_CSP_O_2;
			mole.b[ipSO] -= rate_COmole.b_CP_SO_CSP_O;
			
			mole.c[ipSO][ipCSP] += rate_COmole.CP_SO_CSP_O_1;
			mole.c[ipCP][ipCSP] += rate_COmole.CP_SO_CSP_O_2;
			mole.b[ipCSP] += rate_COmole.b_CP_SO_CSP_O;
			
			mole.c[ipSO][ipATO] += rate_COmole.CP_SO_CSP_O_1;
			mole.c[ipCP][ipATO] += rate_COmole.CP_SO_CSP_O_2;
			mole.b[ipATO] += rate_COmole.b_CP_SO_CSP_O;
			
		/*  >>refer   mol  CH+    +  N     =>   CN+   +     H     UMIST  */

			mole.c[ipATN][ipCHP] -= rate_COmole.CHP_N_CNP_H_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_N_CNP_H_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_N_CNP_H;

			mole.c[ipATN][ipATN] -= rate_COmole.CHP_N_CNP_H_1;
			mole.c[ipCHP][ipATN] -= rate_COmole.CHP_N_CNP_H_2;
			mole.b[ipATN] -= rate_COmole.b_CHP_N_CNP_H;
			
			mole.c[ipATN][ipCNP] += rate_COmole.CHP_N_CNP_H_1;
			mole.c[ipCHP][ipCNP] += rate_COmole.CHP_N_CNP_H_2;
			mole.b[ipCNP] += rate_COmole.b_CHP_N_CNP_H;

		/*  >>refer   mol  CH     +  N+    =>   CN+   +     H     UMIST  */

			mole.c[ipNP][ipCH] -= rate_COmole.CH_NP_CNP_H_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NP_CNP_H_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NP_CNP_H;
			
			mole.c[ipNP][ipNP] -= rate_COmole.CH_NP_CNP_H_1;
			mole.c[ipCH][ipNP] -= rate_COmole.CH_NP_CNP_H_2;
			mole.b[ipNP] -= rate_COmole.b_CH_NP_CNP_H;
			
			mole.c[ipNP][ipCNP] += rate_COmole.CH_NP_CNP_H_1;
			mole.c[ipCH][ipCNP] += rate_COmole.CH_NP_CNP_H_2;
			mole.b[ipCNP] += rate_COmole.b_CH_NP_CNP_H;
			
		/*  >>refer   mol  CH     +  NH+   =>   CH2+  +     N     UMIST  */

			mole.c[ipNHP][ipCH] -= rate_COmole.CH_NHP_CH2P_N_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NHP_CH2P_N_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NHP_CH2P_N;
			
			mole.c[ipNHP][ipNHP] -= rate_COmole.CH_NHP_CH2P_N_1;
			mole.c[ipCH][ipNHP] -= rate_COmole.CH_NHP_CH2P_N_2;
			mole.b[ipNHP] -= rate_COmole.b_CH_NHP_CH2P_N;
			
			mole.c[ipNHP][ipCH2P] += rate_COmole.CH_NHP_CH2P_N_1;
			mole.c[ipCH][ipCH2P] += rate_COmole.CH_NHP_CH2P_N_2;
			mole.b[ipCH2P] += rate_COmole.b_CH_NHP_CH2P_N;
			
			mole.c[ipNHP][ipATN] += rate_COmole.CH_NHP_CH2P_N_1;
			mole.c[ipCH][ipATN] += rate_COmole.CH_NHP_CH2P_N_2;
			mole.b[ipATN] += rate_COmole.b_CH_NHP_CH2P_N;
			
		/*  >>refer   mol  CH+    +  NH    =>   CN+   +     H2    UMIST  */

			mole.c[ipNH][ipCHP] -= rate_COmole.CHP_NH_CNP_H2_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_NH_CNP_H2_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_NH_CNP_H2;
			
			mole.c[ipNH][ipNH] -= rate_COmole.CHP_NH_CNP_H2_1;
			mole.c[ipCHP][ipNH] -= rate_COmole.CHP_NH_CNP_H2_2;
			mole.b[ipNH] -= rate_COmole.b_CHP_NH_CNP_H2;
			
			mole.c[ipNH][ipCNP] += rate_COmole.CHP_NH_CNP_H2_1;
			mole.c[ipCHP][ipCNP] += rate_COmole.CHP_NH_CNP_H2_2;
			mole.b[ipCNP] += rate_COmole.b_CHP_NH_CNP_H2;
			
		/*  >>refer   mol  CH+    +  NH2   =>   HCN+  +     H2    UMIST  */

			mole.c[ipNH2][ipCHP] -= rate_COmole.CHP_NH2_HCNP_H2_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_NH2_HCNP_H2_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_NH2_HCNP_H2;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.CHP_NH2_HCNP_H2_1;
			mole.c[ipCHP][ipNH2] -= rate_COmole.CHP_NH2_HCNP_H2_2;
			mole.b[ipNH2] -= rate_COmole.b_CHP_NH2_HCNP_H2;
			
			mole.c[ipNH2][ipHCNP] += rate_COmole.CHP_NH2_HCNP_H2_1;
			mole.c[ipCHP][ipHCNP] += rate_COmole.CHP_NH2_HCNP_H2_2;
			mole.b[ipHCNP] += rate_COmole.b_CHP_NH2_HCNP_H2;

		/*  >>refer   mol  CH     +  NH2+  =>   NH    +     CH2+  UMIST  */

			mole.c[ipNH2P][ipCH] -= rate_COmole.CH_NH2P_NH_CH2P_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NH2P_NH_CH2P_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NH2P_NH_CH2P;
			
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.CH_NH2P_NH_CH2P_1;
			mole.c[ipCH][ipNH2P] -= rate_COmole.CH_NH2P_NH_CH2P_2;
			mole.b[ipNH2P] -= rate_COmole.b_CH_NH2P_NH_CH2P;
			
			mole.c[ipNH2P][ipNH] += rate_COmole.CH_NH2P_NH_CH2P_1;
			mole.c[ipCH][ipNH] += rate_COmole.CH_NH2P_NH_CH2P_2;
			mole.b[ipNH] += rate_COmole.b_CH_NH2P_NH_CH2P;
			
			mole.c[ipNH2P][ipCH2P] += rate_COmole.CH_NH2P_NH_CH2P_1;
			mole.c[ipCH][ipCH2P] += rate_COmole.CH_NH2P_NH_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH_NH2P_NH_CH2P;

		/*  >>refer   mol  CH+    +  NH3   =>   NH4+  +     C     UMIST  */

			mole.c[ipNH3][ipCHP] -= rate_COmole.CHP_NH3_NH4P_C_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_NH3_NH4P_C_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_NH3_NH4P_C;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CHP_NH3_NH4P_C_1;
			mole.c[ipCHP][ipNH3] -= rate_COmole.CHP_NH3_NH4P_C_2;
			mole.b[ipNH3] -= rate_COmole.b_CHP_NH3_NH4P_C;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.CHP_NH3_NH4P_C_1;
			mole.c[ipCHP][ipNH4P] += rate_COmole.CHP_NH3_NH4P_C_2;
			mole.b[ipNH4P] += rate_COmole.b_CHP_NH3_NH4P_C;
			
			mole.c[ipNH3][ipATC] += rate_COmole.CHP_NH3_NH4P_C_1;
			mole.c[ipCHP][ipATC] += rate_COmole.CHP_NH3_NH4P_C_2;
			mole.b[ipATC] += rate_COmole.b_CHP_NH3_NH4P_C;

		/*  >>refer   mol  CH     +  NH3+  =>   NH4+  +     C     UMIST  */

			mole.c[ipNH3P][ipCH] -= rate_COmole.CH_NH3P_NH4P_C_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NH3P_NH4P_C_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NH3P_NH4P_C;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.CH_NH3P_NH4P_C_1;
			mole.c[ipCH][ipNH3P] -= rate_COmole.CH_NH3P_NH4P_C_2;
			mole.b[ipNH3P] -= rate_COmole.b_CH_NH3P_NH4P_C;
			
			mole.c[ipNH3P][ipNH4P] += rate_COmole.CH_NH3P_NH4P_C_1;
			mole.c[ipCH][ipNH4P] += rate_COmole.CH_NH3P_NH4P_C_2;
			mole.b[ipNH4P] += rate_COmole.b_CH_NH3P_NH4P_C;
			
			mole.c[ipNH3P][ipATC] += rate_COmole.CH_NH3P_NH4P_C_1;
			mole.c[ipCH][ipATC] += rate_COmole.CH_NH3P_NH4P_C_2;
			mole.b[ipATC] += rate_COmole.b_CH_NH3P_NH4P_C;

		/*  >>refer   mol  CH     +  HCN+  =>   CN    +     CH2+  UMIST  */

			mole.c[ipHCNP][ipCH] -= rate_COmole.CH_HCNP_CN_CH2P_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_HCNP_CN_CH2P_2;
			mole.b[ipCH] -= rate_COmole.b_CH_HCNP_CN_CH2P;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.CH_HCNP_CN_CH2P_1;
			mole.c[ipCH][ipHCNP] -= rate_COmole.CH_HCNP_CN_CH2P_2;
			mole.b[ipHCNP] -= rate_COmole.b_CH_HCNP_CN_CH2P;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.CH_HCNP_CN_CH2P_1;
			mole.c[ipCH][ipCN] += rate_COmole.CH_HCNP_CN_CH2P_2;
			mole.b[ipCN] += rate_COmole.b_CH_HCNP_CN_CH2P;
			
			mole.c[ipHCNP][ipCH2P] += rate_COmole.CH_HCNP_CN_CH2P_1;
			mole.c[ipCH][ipCH2P] += rate_COmole.CH_HCNP_CN_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH_HCNP_CN_CH2P;

		/*  >>refer   mol  CH     +  HNO+  =>   NO    +     CH2+  UMIST  */

			mole.c[ipHNOP][ipCH] -= rate_COmole.CH_HNOP_NO_CH2P_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_HNOP_NO_CH2P_2;
			mole.b[ipCH] -= rate_COmole.b_CH_HNOP_NO_CH2P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.CH_HNOP_NO_CH2P_1;
			mole.c[ipCH][ipHNOP] -= rate_COmole.CH_HNOP_NO_CH2P_2;
			mole.b[ipHNOP] -= rate_COmole.b_CH_HNOP_NO_CH2P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.CH_HNOP_NO_CH2P_1;
			mole.c[ipCH][ipNO] += rate_COmole.CH_HNOP_NO_CH2P_2;
			mole.b[ipNO] += rate_COmole.b_CH_HNOP_NO_CH2P;
			
			mole.c[ipHNOP][ipCH2P] += rate_COmole.CH_HNOP_NO_CH2P_1;
			mole.c[ipCH][ipCH2P] += rate_COmole.CH_HNOP_NO_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH_HNOP_NO_CH2P;

		/*  >>refer   mol  CH+    +  S     =>   HS+   +     C     UMIST  */
			
			mole.c[ipATS][ipCHP] -= rate_COmole.CHP_S_HSP_C_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_S_HSP_C_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_S_HSP_C;
						
			mole.c[ipATS][ipATS] -= rate_COmole.CHP_S_HSP_C_1;
			mole.c[ipCHP][ipATS] -= rate_COmole.CHP_S_HSP_C_2;
			mole.b[ipATS] -= rate_COmole.b_CHP_S_HSP_C;
			
			mole.c[ipATS][ipHSP] += rate_COmole.CHP_S_HSP_C_1;
			mole.c[ipCHP][ipHSP] += rate_COmole.CHP_S_HSP_C_2;
			mole.b[ipHSP] += rate_COmole.b_CHP_S_HSP_C;

			mole.c[ipATS][ipATC] += rate_COmole.CHP_S_HSP_C_1;
			mole.c[ipCHP][ipATC] += rate_COmole.CHP_S_HSP_C_2;
			mole.b[ipATC] += rate_COmole.b_CHP_S_HSP_C;

		/*  >>refer   mol  CH+    +  S     =>   CS+   +     H     UMIST  */
			
			mole.c[ipATS][ipCHP] -= rate_COmole.CHP_S_CSP_H_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_S_CSP_H_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_S_CSP_H;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CHP_S_CSP_H_1;
			mole.c[ipCHP][ipATS] -= rate_COmole.CHP_S_CSP_H_2;
			mole.b[ipATS] -= rate_COmole.b_CHP_S_CSP_H;
			
			mole.c[ipATS][ipCSP] += rate_COmole.CHP_S_CSP_H_1;
			mole.c[ipCHP][ipCSP] += rate_COmole.CHP_S_CSP_H_2;
			mole.b[ipCSP] += rate_COmole.b_CHP_S_CSP_H;

		/*  >>refer   mol  CH     +  S+    =>   CS+   +     H     UMIST  */

			mole.c[ipSP][ipCH] -= rate_COmole.CH_SP_CSP_H_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_SP_CSP_H_2;
			mole.b[ipCH] -= rate_COmole.b_CH_SP_CSP_H;
			
			mole.c[ipSP][ipSP] -= rate_COmole.CH_SP_CSP_H_1;
			mole.c[ipCH][ipSP] -= rate_COmole.CH_SP_CSP_H_2;
			mole.b[ipSP] -= rate_COmole.b_CH_SP_CSP_H;
			
			mole.c[ipSP][ipCSP] += rate_COmole.CH_SP_CSP_H_1;
			mole.c[ipCH][ipCSP] += rate_COmole.CH_SP_CSP_H_2;
			mole.b[ipCSP] += rate_COmole.b_CH_SP_CSP_H;

		/*  >>refer   mol  CH     +  HS+   =>   S     +     CH2+  UMIST  */

			mole.c[ipHSP][ipCH] -= rate_COmole.CH_HSP_S_CH2P_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_HSP_S_CH2P_2;
			mole.b[ipCH] -= rate_COmole.b_CH_HSP_S_CH2P;
			
			mole.c[ipHSP][ipHSP] -= rate_COmole.CH_HSP_S_CH2P_1;
			mole.c[ipCH][ipHSP] -= rate_COmole.CH_HSP_S_CH2P_2;
			mole.b[ipHSP] -= rate_COmole.b_CH_HSP_S_CH2P;
			
			mole.c[ipHSP][ipCH2P] += rate_COmole.CH_HSP_S_CH2P_1;
			mole.c[ipCH][ipCH2P] += rate_COmole.CH_HSP_S_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH_HSP_S_CH2P;
			
			mole.c[ipHSP][ipATS] += rate_COmole.CH_HSP_S_CH2P_1;
			mole.c[ipCH][ipATS] += rate_COmole.CH_HSP_S_CH2P_2;
			mole.b[ipATS] += rate_COmole.b_CH_HSP_S_CH2P;

		/*  >>refer   mol  N      +  CH2+  =>   HCN+  +     H     UMIST  */

			mole.c[ipCH2P][ipCH2P] -= rate_COmole.N_CH2P_HCNP_H_1;
			mole.c[ipATN][ipCH2P] -= rate_COmole.N_CH2P_HCNP_H_2;
			mole.b[ipCH2P] -= rate_COmole.b_N_CH2P_HCNP_H;

			mole.c[ipCH2P][ipATN] -= rate_COmole.N_CH2P_HCNP_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_CH2P_HCNP_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_CH2P_HCNP_H;
			
			mole.c[ipCH2P][ipHCNP] += rate_COmole.N_CH2P_HCNP_H_1;
			mole.c[ipATN][ipHCNP] += rate_COmole.N_CH2P_HCNP_H_2;
			mole.b[ipHCNP] += rate_COmole.b_N_CH2P_HCNP_H;

		/*  >>refer   mol  N+     +  NH    =>   N2+   +     H     UMIST  */

			mole.c[ipNH][ipNP] -= rate_COmole.NP_NH_N2P_H_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NH_N2P_H_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NH_N2P_H;
			
			mole.c[ipNH][ipNH] -= rate_COmole.NP_NH_N2P_H_1;
			mole.c[ipNP][ipNH] -= rate_COmole.NP_NH_N2P_H_2;
			mole.b[ipNH] -= rate_COmole.b_NP_NH_N2P_H;
			
			mole.c[ipNH][ipN2P] += rate_COmole.NP_NH_N2P_H_1;
			mole.c[ipNP][ipN2P] += rate_COmole.NP_NH_N2P_H_2;
			mole.b[ipN2P] += rate_COmole.b_NP_NH_N2P_H;
			
		/*  >>refer   mol  N      +  NH+   =>   N2+   +     H     UMIST  */

			mole.c[ipNHP][ipNHP] -= rate_COmole.N_NHP_N2P_H_1;
			mole.c[ipATN][ipNHP] -= rate_COmole.N_NHP_N2P_H_2;
			mole.b[ipNHP] -= rate_COmole.b_N_NHP_N2P_H;

			mole.c[ipNHP][ipATN] -= rate_COmole.N_NHP_N2P_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_NHP_N2P_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_NHP_N2P_H;
			
			mole.c[ipNHP][ipN2P] += rate_COmole.N_NHP_N2P_H_1;
			mole.c[ipATN][ipN2P] += rate_COmole.N_NHP_N2P_H_2;
			mole.b[ipN2P] += rate_COmole.b_N_NHP_N2P_H;

		/*  >>refer   mol  N+     +  CH4   =>   HCN+  +     H2 +  H      UMIST  */

			mole.c[ipCH4][ipNP] -= rate_COmole.NP_CH4_HCNP_H2_H_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_CH4_HCNP_H2_H_2;
			mole.b[ipNP] -= rate_COmole.b_NP_CH4_HCNP_H2_H;
			
			mole.c[ipCH4][ipCH4] -= rate_COmole.NP_CH4_HCNP_H2_H_1;
			mole.c[ipNP][ipCH4] -= rate_COmole.NP_CH4_HCNP_H2_H_2;
			mole.b[ipCH4] -= rate_COmole.b_NP_CH4_HCNP_H2_H;
			
			mole.c[ipCH4][ipHCNP] += rate_COmole.NP_CH4_HCNP_H2_H_1;
			mole.c[ipNP][ipHCNP] += rate_COmole.NP_CH4_HCNP_H2_H_2;
			mole.b[ipHCNP] += rate_COmole.b_NP_CH4_HCNP_H2_H;
			
		/*  >>refer   mol  N      +  OH+   =>   NO+   +     H     UMIST  */

			mole.c[ipOHP][ipOHP] -= rate_COmole.N_OHP_NOP_H_1;
			mole.c[ipATN][ipOHP] -= rate_COmole.N_OHP_NOP_H_2;
			mole.b[ipOHP] -= rate_COmole.b_N_OHP_NOP_H;

			mole.c[ipOHP][ipATN] -= rate_COmole.N_OHP_NOP_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_OHP_NOP_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_OHP_NOP_H;
			
			mole.c[ipOHP][ipNOP] += rate_COmole.N_OHP_NOP_H_1;
			mole.c[ipATN][ipNOP] += rate_COmole.N_OHP_NOP_H_2;
			mole.b[ipNOP] += rate_COmole.b_N_OHP_NOP_H;

		/*  >>refer   mol  N+     +  NH3   =>   NH2+  +     NH    UMIST  */

			mole.c[ipNH3][ipNP] -= rate_COmole.NP_NH3_NH2P_NH_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NH3_NH2P_NH_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NH3_NH2P_NH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NP_NH3_NH2P_NH_1;
			mole.c[ipNP][ipNH3] -= rate_COmole.NP_NH3_NH2P_NH_2;
			mole.b[ipNH3] -= rate_COmole.b_NP_NH3_NH2P_NH;
			
			mole.c[ipNH3][ipNH2P] += rate_COmole.NP_NH3_NH2P_NH_1;
			mole.c[ipNP][ipNH2P] += rate_COmole.NP_NH3_NH2P_NH_2;
			mole.b[ipNH2P] += rate_COmole.b_NP_NH3_NH2P_NH;
			
			mole.c[ipNH3][ipNH] += rate_COmole.NP_NH3_NH2P_NH_1;
			mole.c[ipNP][ipNH] += rate_COmole.NP_NH3_NH2P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NP_NH3_NH2P_NH;

		/*  >>refer   mol  N      +  H2O+  =>   HNO+  +     H     UMIST  */

			mole.c[ipH2OP][ipH2OP] -= rate_COmole.N_H2OP_HNOP_H_1;
			mole.c[ipATN][ipH2OP] -= rate_COmole.N_H2OP_HNOP_H_2;
			mole.b[ipH2OP] -= rate_COmole.b_N_H2OP_HNOP_H;

			mole.c[ipH2OP][ipATN] -= rate_COmole.N_H2OP_HNOP_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_H2OP_HNOP_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_H2OP_HNOP_H;
			
			mole.c[ipH2OP][ipHNOP] += rate_COmole.N_H2OP_HNOP_H_1;
			mole.c[ipATN][ipHNOP] += rate_COmole.N_H2OP_HNOP_H_2;
			mole.b[ipHNOP] += rate_COmole.b_N_H2OP_HNOP_H;

		/*  >>refer   mol  N+     +  NO    =>   N2+   +     O     UMIST  */

			mole.c[ipNO][ipNP] -= rate_COmole.NP_NO_N2P_O_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NO_N2P_O_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NO_N2P_O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NP_NO_N2P_O_1;
			mole.c[ipNP][ipNO] -= rate_COmole.NP_NO_N2P_O_2;
			mole.b[ipNO] -= rate_COmole.b_NP_NO_N2P_O;
			
			mole.c[ipNO][ipN2P] += rate_COmole.NP_NO_N2P_O_1;
			mole.c[ipNP][ipN2P] += rate_COmole.NP_NO_N2P_O_2;
			mole.b[ipN2P] += rate_COmole.b_NP_NO_N2P_O;
			
			mole.c[ipNO][ipATO] += rate_COmole.NP_NO_N2P_O_1;
			mole.c[ipNP][ipATO] += rate_COmole.NP_NO_N2P_O_2;
			mole.b[ipATO] += rate_COmole.b_NP_NO_N2P_O;
			
		/*  >>refer   mol  N+     +  O2    =>   NO    +     O+    UMIST  */

			mole.c[ipO2][ipNP] -= rate_COmole.NP_O2_NO_OP_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_O2_NO_OP_2;
			mole.b[ipNP] -= rate_COmole.b_NP_O2_NO_OP;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NP_O2_NO_OP_1;
			mole.c[ipNP][ipO2] -= rate_COmole.NP_O2_NO_OP_2;
			mole.b[ipO2] -= rate_COmole.b_NP_O2_NO_OP;
			
			mole.c[ipO2][ipNO] += rate_COmole.NP_O2_NO_OP_1;
			mole.c[ipNP][ipNO] += rate_COmole.NP_O2_NO_OP_2;
			mole.b[ipNO] += rate_COmole.b_NP_O2_NO_OP;
			
			mole.c[ipO2][ipOP] += rate_COmole.NP_O2_NO_OP_1;
			mole.c[ipNP][ipOP] += rate_COmole.NP_O2_NO_OP_2;
			mole.b[ipOP] += rate_COmole.b_NP_O2_NO_OP;

		/*  >>refer   mol  N+     +  O2    =>   NO+   +     O     UMIST  */

			mole.c[ipO2][ipNP] -= rate_COmole.NP_O2_NOP_O_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_O2_NOP_O_2;
			mole.b[ipNP] -= rate_COmole.b_NP_O2_NOP_O;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NP_O2_NOP_O_1;
			mole.c[ipNP][ipO2] -= rate_COmole.NP_O2_NOP_O_2;
			mole.b[ipO2] -= rate_COmole.b_NP_O2_NOP_O;
			
			mole.c[ipO2][ipNOP] += rate_COmole.NP_O2_NOP_O_1;
			mole.c[ipNP][ipNOP] += rate_COmole.NP_O2_NOP_O_2;
			mole.b[ipNOP] += rate_COmole.b_NP_O2_NOP_O;
			
			mole.c[ipO2][ipATO] += rate_COmole.NP_O2_NOP_O_1;
			mole.c[ipNP][ipATO] += rate_COmole.NP_O2_NOP_O_2;
			mole.b[ipATO] += rate_COmole.b_NP_O2_NOP_O;

		/*  >>refer   mol  N      +  O2+   =>   NO+   +     O     UMIST  */

			mole.c[ipO2P][ipO2P] -= rate_COmole.N_O2P_NOP_O_1;
			mole.c[ipATN][ipO2P] -= rate_COmole.N_O2P_NOP_O_2;
			mole.b[ipO2P] -= rate_COmole.b_N_O2P_NOP_O;
			
			mole.c[ipO2P][ipATN] -= rate_COmole.N_O2P_NOP_O_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_O2P_NOP_O_2;
			mole.b[ipATN] -= rate_COmole.b_N_O2P_NOP_O;
			
			mole.c[ipO2P][ipNOP] += rate_COmole.N_O2P_NOP_O_1;
			mole.c[ipATN][ipNOP] += rate_COmole.N_O2P_NOP_O_2;
			mole.b[ipNOP] += rate_COmole.b_N_O2P_NOP_O;

			mole.c[ipO2P][ipATO] += rate_COmole.N_O2P_NOP_O_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_O2P_NOP_O_2;
			mole.b[ipATO] += rate_COmole.b_N_O2P_NOP_O;

		/*  >>refer   mol  N      +  HS+   =>   NS+   +     H     UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.N_HSP_NSP_H_1;
			mole.c[ipATN][ipHSP] -= rate_COmole.N_HSP_NSP_H_2;
			mole.b[ipHSP] -= rate_COmole.b_N_HSP_NSP_H;

			mole.c[ipHSP][ipATN] -= rate_COmole.N_HSP_NSP_H_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_HSP_NSP_H_2;
			mole.b[ipATN] -= rate_COmole.b_N_HSP_NSP_H;
			
			mole.c[ipHSP][ipNSP] += rate_COmole.N_HSP_NSP_H_1;
			mole.c[ipATN][ipNSP] += rate_COmole.N_HSP_NSP_H_2;
			mole.b[ipNSP] += rate_COmole.b_N_HSP_NSP_H;

		/*  >>refer   mol  N      +  SIO+  =>   NO+   +     Si    UMIST  */

			mole.c[ipSIOP][ipSIOP] -= rate_COmole.N_SIOP_NOP_SI_1;
			mole.c[ipATN][ipSIOP] -= rate_COmole.N_SIOP_NOP_SI_2;
			mole.b[ipSIOP] -= rate_COmole.b_N_SIOP_NOP_SI;
			
			mole.c[ipSIOP][ipATN] -= rate_COmole.N_SIOP_NOP_SI_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SIOP_NOP_SI_2;
			mole.b[ipATN] -= rate_COmole.b_N_SIOP_NOP_SI;
			
			mole.c[ipSIOP][ipNOP] += rate_COmole.N_SIOP_NOP_SI_1;
			mole.c[ipATN][ipNOP] += rate_COmole.N_SIOP_NOP_SI_2;
			mole.b[ipNOP] += rate_COmole.b_N_SIOP_NOP_SI;

			mole.c[ipSIOP][ipATSI] += rate_COmole.N_SIOP_NOP_SI_1;
			mole.c[ipATN][ipATSI] += rate_COmole.N_SIOP_NOP_SI_2;
			mole.b[ipATSI] += rate_COmole.b_N_SIOP_NOP_SI;

		/*  >>refer   mol  N      +  SIO+  =>   NO    +     SI+   UMIST  */

			mole.c[ipSIOP][ipSIOP] -= rate_COmole.N_SIOP_NO_SIP_1;
			mole.c[ipATN][ipSIOP] -= rate_COmole.N_SIOP_NO_SIP_2;
			mole.b[ipSIOP] -= rate_COmole.b_N_SIOP_NO_SIP;

			mole.c[ipSIOP][ipATN] -= rate_COmole.N_SIOP_NO_SIP_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SIOP_NO_SIP_2;
			mole.b[ipATN] -= rate_COmole.b_N_SIOP_NO_SIP;
			
			mole.c[ipSIOP][ipNO] += rate_COmole.N_SIOP_NO_SIP_1;
			mole.c[ipATN][ipNO] += rate_COmole.N_SIOP_NO_SIP_2;
			mole.b[ipNO] += rate_COmole.b_N_SIOP_NO_SIP;
			
			mole.c[ipSIOP][ipSIP] += rate_COmole.N_SIOP_NO_SIP_1;
			mole.c[ipATN][ipSIP] += rate_COmole.N_SIOP_NO_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_N_SIOP_NO_SIP;

		/*  >>refer   mol  N      +  SO+   =>   NS+   +     O     UMIST  */

			mole.c[ipSOP][ipSOP] -= rate_COmole.N_SOP_NSP_O_1;
			mole.c[ipATN][ipSOP] -= rate_COmole.N_SOP_NSP_O_2;
			mole.b[ipSOP] -= rate_COmole.b_N_SOP_NSP_O;
			
			mole.c[ipSOP][ipATN] -= rate_COmole.N_SOP_NSP_O_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_SOP_NSP_O_2;
			mole.b[ipATN] -= rate_COmole.b_N_SOP_NSP_O;
			
			mole.c[ipSOP][ipNSP] += rate_COmole.N_SOP_NSP_O_1;
			mole.c[ipATN][ipNSP] += rate_COmole.N_SOP_NSP_O_2;
			mole.b[ipNSP] += rate_COmole.b_N_SOP_NSP_O;
			
			mole.c[ipSOP][ipATO] += rate_COmole.N_SOP_NSP_O_1;
			mole.c[ipATN][ipATO] += rate_COmole.N_SOP_NSP_O_2;
			mole.b[ipATO] += rate_COmole.b_N_SOP_NSP_O;

		/*  >>refer   mol  N+     +  OCS   =>   CS+   +     NO    UMIST  */

			mole.c[ipOCS][ipNP] -= rate_COmole.NP_OCS_CSP_NO_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_OCS_CSP_NO_2;
			mole.b[ipNP] -= rate_COmole.b_NP_OCS_CSP_NO;
			
			mole.c[ipOCS][ipOCS] -= rate_COmole.NP_OCS_CSP_NO_1;
			mole.c[ipNP][ipOCS] -= rate_COmole.NP_OCS_CSP_NO_2;
			mole.b[ipOCS] -= rate_COmole.b_NP_OCS_CSP_NO;
			
			mole.c[ipOCS][ipCSP] += rate_COmole.NP_OCS_CSP_NO_1;
			mole.c[ipNP][ipCSP] += rate_COmole.NP_OCS_CSP_NO_2;
			mole.b[ipCSP] += rate_COmole.b_NP_OCS_CSP_NO;
			
			mole.c[ipOCS][ipNO] += rate_COmole.NP_OCS_CSP_NO_1;
			mole.c[ipNP][ipNO] += rate_COmole.NP_OCS_CSP_NO_2;
			mole.b[ipNO] += rate_COmole.b_NP_OCS_CSP_NO;

		/*  >>refer   mol  CH2    +  NH+   =>   CH3+  +     N     UMIST  */

			mole.c[ipNHP][ipCH2] -= rate_COmole.CH2_NHP_CH3P_N_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_NHP_CH3P_N_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_NHP_CH3P_N;
			
			mole.c[ipNHP][ipNHP] -= rate_COmole.CH2_NHP_CH3P_N_1;
			mole.c[ipCH2][ipNHP] -= rate_COmole.CH2_NHP_CH3P_N_2;
			mole.b[ipNHP] -= rate_COmole.b_CH2_NHP_CH3P_N;
			
			mole.c[ipNHP][ipCH3P] += rate_COmole.CH2_NHP_CH3P_N_1;
			mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_NHP_CH3P_N_2;
			mole.b[ipCH3P] += rate_COmole.b_CH2_NHP_CH3P_N;
			
			mole.c[ipNHP][ipATN] += rate_COmole.CH2_NHP_CH3P_N_1;
			mole.c[ipCH2][ipATN] += rate_COmole.CH2_NHP_CH3P_N_2;
			mole.b[ipATN] += rate_COmole.b_CH2_NHP_CH3P_N;

		/*  >>refer   mol  CH2    +  NH2+  =>   CH3+  +     NH    UMIST  */

			mole.c[ipNH2P][ipCH2] -= rate_COmole.CH2_NH2P_CH3P_NH_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_NH2P_CH3P_NH_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_NH2P_CH3P_NH;
			
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.CH2_NH2P_CH3P_NH_1;
			mole.c[ipCH2][ipNH2P] -= rate_COmole.CH2_NH2P_CH3P_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_CH2_NH2P_CH3P_NH;
			
			mole.c[ipNH2P][ipCH3P] += rate_COmole.CH2_NH2P_CH3P_NH_1;
			mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_NH2P_CH3P_NH_2;
			mole.b[ipCH3P] += rate_COmole.b_CH2_NH2P_CH3P_NH;
			
			mole.c[ipNH2P][ipNH] += rate_COmole.CH2_NH2P_CH3P_NH_1;
			mole.c[ipCH2][ipNH] += rate_COmole.CH2_NH2P_CH3P_NH_2;
			mole.b[ipNH] += rate_COmole.b_CH2_NH2P_CH3P_NH;

		/*  >>refer   mol  CH2+   +  NH3   =>   NH4+  +     CH    UMIST  */

			mole.c[ipNH3][ipCH2P] -= rate_COmole.CH2P_NH3_NH4P_CH_1;
			mole.c[ipCH2P][ipCH2P] -= rate_COmole.CH2P_NH3_NH4P_CH_2;
			mole.b[ipCH2P] -= rate_COmole.b_CH2P_NH3_NH4P_CH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CH2P_NH3_NH4P_CH_1;
			mole.c[ipCH2P][ipNH3] -= rate_COmole.CH2P_NH3_NH4P_CH_2;
			mole.b[ipNH3] -= rate_COmole.b_CH2P_NH3_NH4P_CH;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.CH2P_NH3_NH4P_CH_1;
			mole.c[ipCH2P][ipNH4P] += rate_COmole.CH2P_NH3_NH4P_CH_2;
			mole.b[ipNH4P] += rate_COmole.b_CH2P_NH3_NH4P_CH;
			
			mole.c[ipNH3][ipCH] += rate_COmole.CH2P_NH3_NH4P_CH_1;
			mole.c[ipCH2P][ipCH] += rate_COmole.CH2P_NH3_NH4P_CH_2;
			mole.b[ipCH] += rate_COmole.b_CH2P_NH3_NH4P_CH;

		/*  >>refer   mol  CH2    +  NH3+  =>   NH2   +     CH3+  UMIST  */

			mole.c[ipNH3P][ipCH2] -= rate_COmole.CH2_NH3P_NH2_CH3P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_NH3P_NH2_CH3P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_NH3P_NH2_CH3P;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.CH2_NH3P_NH2_CH3P_1;
			mole.c[ipCH2][ipNH3P] -= rate_COmole.CH2_NH3P_NH2_CH3P_2;
			mole.b[ipNH3P] -= rate_COmole.b_CH2_NH3P_NH2_CH3P;
			
			mole.c[ipNH3P][ipNH2] += rate_COmole.CH2_NH3P_NH2_CH3P_1;
			mole.c[ipCH2][ipNH2] += rate_COmole.CH2_NH3P_NH2_CH3P_2;
			mole.b[ipNH2] += rate_COmole.b_CH2_NH3P_NH2_CH3P;
			
			mole.c[ipNH3P][ipCH3P] += rate_COmole.CH2_NH3P_NH2_CH3P_1;
			mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_NH3P_NH2_CH3P_2;
			mole.b[ipCH3P] += rate_COmole.b_CH2_NH3P_NH2_CH3P;

		/*  >>refer   mol  CH2    +  HCN+  =>   CN    +     CH3+  UMIST  */

			mole.c[ipHCNP][ipCH2] -= rate_COmole.CH2_HCNP_CN_CH3P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_HCNP_CN_CH3P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_HCNP_CN_CH3P;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.CH2_HCNP_CN_CH3P_1;
			mole.c[ipCH2][ipHCNP] -= rate_COmole.CH2_HCNP_CN_CH3P_2;
			mole.b[ipHCNP] -= rate_COmole.b_CH2_HCNP_CN_CH3P;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.CH2_HCNP_CN_CH3P_1;
			mole.c[ipCH2][ipCN] += rate_COmole.CH2_HCNP_CN_CH3P_2;
			mole.b[ipCN] += rate_COmole.b_CH2_HCNP_CN_CH3P;
			
			mole.c[ipHCNP][ipCH3P] += rate_COmole.CH2_HCNP_CN_CH3P_1;
			mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_HCNP_CN_CH3P_2;
			mole.b[ipCH3P] += rate_COmole.b_CH2_HCNP_CN_CH3P;

		/*  >>refer   mol  CH2    +  HNO+  =>   NO    +     CH3+  UMIST  */

			mole.c[ipHNOP][ipCH2] -= rate_COmole.CH2_HNOP_NO_CH3P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_HNOP_NO_CH3P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_HNOP_NO_CH3P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.CH2_HNOP_NO_CH3P_1;
			mole.c[ipCH2][ipHNOP] -= rate_COmole.CH2_HNOP_NO_CH3P_2;
			mole.b[ipHNOP] -= rate_COmole.b_CH2_HNOP_NO_CH3P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.CH2_HNOP_NO_CH3P_1;
			mole.c[ipCH2][ipNO] += rate_COmole.CH2_HNOP_NO_CH3P_2;
			mole.b[ipNO] += rate_COmole.b_CH2_HNOP_NO_CH3P;
			
			mole.c[ipHNOP][ipCH3P] += rate_COmole.CH2_HNOP_NO_CH3P_1;
			mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_HNOP_NO_CH3P_2;
			mole.b[ipCH3P] += rate_COmole.b_CH2_HNOP_NO_CH3P;

		/*  >>refer   mol  CH2+   +  S     =>   HCS+  +     H     UMIST  */

			mole.c[ipATS][ipCH2P] -= rate_COmole.CH2P_S_HCSP_H_1;
			mole.c[ipCH2P][ipCH2P] -= rate_COmole.CH2P_S_HCSP_H_2;
			mole.b[ipCH2P] -= rate_COmole.b_CH2P_S_HCSP_H;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CH2P_S_HCSP_H_1;
			mole.c[ipCH2P][ipATS] -= rate_COmole.CH2P_S_HCSP_H_2;
			mole.b[ipATS] -= rate_COmole.b_CH2P_S_HCSP_H;
			
			mole.c[ipATS][ipHCSP] += rate_COmole.CH2P_S_HCSP_H_1;
			mole.c[ipCH2P][ipHCSP] += rate_COmole.CH2P_S_HCSP_H_2;
			mole.b[ipHCSP] += rate_COmole.b_CH2P_S_HCSP_H;

		/*  >>refer   mol  CH2    +  S+    =>   HCS+  +     H     UMIST  */

			mole.c[ipSP][ipCH2] -= rate_COmole.CH2_SP_HCSP_H_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_SP_HCSP_H_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_SP_HCSP_H;
			
			mole.c[ipSP][ipSP] -= rate_COmole.CH2_SP_HCSP_H_1;
			mole.c[ipCH2][ipSP] -= rate_COmole.CH2_SP_HCSP_H_2;
			mole.b[ipSP] -= rate_COmole.b_CH2_SP_HCSP_H;
			
			mole.c[ipSP][ipHCSP] += rate_COmole.CH2_SP_HCSP_H_1;
			mole.c[ipCH2][ipHCSP] += rate_COmole.CH2_SP_HCSP_H_2;
			mole.b[ipHCSP] += rate_COmole.b_CH2_SP_HCSP_H;
			
		/*  >>refer   mol  NH+    +  NH    =>   NH2+  +     N     UMIST  */

			mole.c[ipNH][ipNHP] -= rate_COmole.NHP_NH_NH2P_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_NH_NH2P_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_NH_NH2P_N;
			
			mole.c[ipNH][ipNH] -= rate_COmole.NHP_NH_NH2P_N_1;
			mole.c[ipNHP][ipNH] -= rate_COmole.NHP_NH_NH2P_N_2;
			mole.b[ipNH] -= rate_COmole.b_NHP_NH_NH2P_N;
			
			mole.c[ipNH][ipNH2P] += rate_COmole.NHP_NH_NH2P_N_1;
			mole.c[ipNHP][ipNH2P] += rate_COmole.NHP_NH_NH2P_N_2;
			mole.b[ipNH2P] += rate_COmole.b_NHP_NH_NH2P_N;
			
			mole.c[ipNH][ipATN] += rate_COmole.NHP_NH_NH2P_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_NH_NH2P_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_NH_NH2P_N;
			
		/*  >>refer   mol  NH+    +  O     =>   OH+   +     N     UMIST  */

			mole.c[ipATO][ipNHP] -= rate_COmole.NHP_O_OHP_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_O_OHP_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_O_OHP_N;
			
			mole.c[ipATO][ipATO] -= rate_COmole.NHP_O_OHP_N_1;
			mole.c[ipNHP][ipATO] -= rate_COmole.NHP_O_OHP_N_2;
			mole.b[ipATO] -= rate_COmole.b_NHP_O_OHP_N;
			
			mole.c[ipATO][ipOHP] += rate_COmole.NHP_O_OHP_N_1;
			mole.c[ipNHP][ipOHP] += rate_COmole.NHP_O_OHP_N_2;
			mole.b[ipOHP] += rate_COmole.b_NHP_O_OHP_N;

			mole.c[ipATO][ipATN] += rate_COmole.NHP_O_OHP_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_O_OHP_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_O_OHP_N;

		/*  >>refer   mol  NH     +  O+    =>   NO+   +     H     UMIST  */

			mole.c[ipOP][ipNH] -= rate_COmole.NH_OP_NOP_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OP_NOP_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OP_NOP_H;
			
			mole.c[ipOP][ipOP] -= rate_COmole.NH_OP_NOP_H_1;
			mole.c[ipNH][ipOP] -= rate_COmole.NH_OP_NOP_H_2;
			mole.b[ipOP] -= rate_COmole.b_NH_OP_NOP_H;
			
			mole.c[ipOP][ipNOP] += rate_COmole.NH_OP_NOP_H_1;
			mole.c[ipNH][ipNOP] += rate_COmole.NH_OP_NOP_H_2;
			mole.b[ipNOP] += rate_COmole.b_NH_OP_NOP_H;

		/*  >>refer   mol  NH+    +  NH2   =>   NH3+  +     N     UMIST  */

			mole.c[ipNH2][ipNHP] -= rate_COmole.NHP_NH2_NH3P_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_NH2_NH3P_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_NH2_NH3P_N;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.NHP_NH2_NH3P_N_1;
			mole.c[ipNHP][ipNH2] -= rate_COmole.NHP_NH2_NH3P_N_2;
			mole.b[ipNH2] -= rate_COmole.b_NHP_NH2_NH3P_N;
			
			mole.c[ipNH2][ipNH3P] += rate_COmole.NHP_NH2_NH3P_N_1;
			mole.c[ipNHP][ipNH3P] += rate_COmole.NHP_NH2_NH3P_N_2;
			mole.b[ipNH3P] += rate_COmole.b_NHP_NH2_NH3P_N;
			
			mole.c[ipNH2][ipATN] += rate_COmole.NHP_NH2_NH3P_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_NH2_NH3P_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_NH2_NH3P_N;
			
		/*  >>refer   mol  NH     +  NH2+  =>   NH3+  +     N     UMIST  */

			mole.c[ipNH2P][ipNH] -= rate_COmole.NH_NH2P_NH3P_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH2P_NH3P_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NH2P_NH3P_N;
			
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH_NH2P_NH3P_N_1;
			mole.c[ipNH][ipNH2P] -= rate_COmole.NH_NH2P_NH3P_N_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH_NH2P_NH3P_N;
			
			mole.c[ipNH2P][ipNH3P] += rate_COmole.NH_NH2P_NH3P_N_1;
			mole.c[ipNH][ipNH3P] += rate_COmole.NH_NH2P_NH3P_N_2;
			mole.b[ipNH3P] += rate_COmole.b_NH_NH2P_NH3P_N;
			
			mole.c[ipNH2P][ipATN] += rate_COmole.NH_NH2P_NH3P_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_NH2P_NH3P_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_NH2P_NH3P_N;
			
		/*  >>refer   mol  NH+    +  OH    =>   H2O+  +     N     UMIST  */
			
			mole.c[ipOH][ipNHP] -= rate_COmole.NHP_OH_H2OP_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_OH_H2OP_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_OH_H2OP_N;

			mole.c[ipOH][ipOH] -= rate_COmole.NHP_OH_H2OP_N_1;
			mole.c[ipNHP][ipOH] -= rate_COmole.NHP_OH_H2OP_N_2;
			mole.b[ipOH] -= rate_COmole.b_NHP_OH_H2OP_N;
			
			mole.c[ipOH][ipH2OP] += rate_COmole.NHP_OH_H2OP_N_1;
			mole.c[ipNHP][ipH2OP] += rate_COmole.NHP_OH_H2OP_N_2;
			mole.b[ipH2OP] += rate_COmole.b_NHP_OH_H2OP_N;
			
			mole.c[ipOH][ipATN] += rate_COmole.NHP_OH_H2OP_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_OH_H2OP_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_OH_H2OP_N;
			
		/*  >>refer   mol  NH     +  OH+   =>   NH2+  +     O     UMIST  */

			mole.c[ipOHP][ipNH] -= rate_COmole.NH_OHP_NH2P_O_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OHP_NH2P_O_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OHP_NH2P_O;
			
			mole.c[ipOHP][ipOHP] -= rate_COmole.NH_OHP_NH2P_O_1;
			mole.c[ipNH][ipOHP] -= rate_COmole.NH_OHP_NH2P_O_2;
			mole.b[ipOHP] -= rate_COmole.b_NH_OHP_NH2P_O;
			
			mole.c[ipOHP][ipNH2P] += rate_COmole.NH_OHP_NH2P_O_1;
			mole.c[ipNH][ipNH2P] += rate_COmole.NH_OHP_NH2P_O_2;
			mole.b[ipNH2P] += rate_COmole.b_NH_OHP_NH2P_O;
			
			mole.c[ipOHP][ipATO] += rate_COmole.NH_OHP_NH2P_O_1;
			mole.c[ipNH][ipATO] += rate_COmole.NH_OHP_NH2P_O_2;
			mole.b[ipATO] += rate_COmole.b_NH_OHP_NH2P_O;
			
		/*  >>refer   mol  NH+    +  NH3   =>   NH4+  +     N     UMIST  */

			mole.c[ipNH3][ipNHP] -= rate_COmole.NHP_NH3_NH4P_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_NH3_NH4P_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_NH3_NH4P_N;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NHP_NH3_NH4P_N_1;
			mole.c[ipNHP][ipNH3] -= rate_COmole.NHP_NH3_NH4P_N_2;
			mole.b[ipNH3] -= rate_COmole.b_NHP_NH3_NH4P_N;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.NHP_NH3_NH4P_N_1;
			mole.c[ipNHP][ipNH4P] += rate_COmole.NHP_NH3_NH4P_N_2;
			mole.b[ipNH4P] += rate_COmole.b_NHP_NH3_NH4P_N;
			
			mole.c[ipNH3][ipATN] += rate_COmole.NHP_NH3_NH4P_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_NH3_NH4P_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_NH3_NH4P_N;
			
		/*  >>refer   mol  NH     +  NH3+  =>   NH4+  +     N     UMIST  */

			mole.c[ipNH3P][ipNH] -= rate_COmole.NH_NH3P_NH4P_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_NH3P_NH4P_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_NH3P_NH4P_N;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH_NH3P_NH4P_N_1;
			mole.c[ipNH][ipNH3P] -= rate_COmole.NH_NH3P_NH4P_N_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH_NH3P_NH4P_N;
			
			mole.c[ipNH3P][ipNH4P] += rate_COmole.NH_NH3P_NH4P_N_1;
			mole.c[ipNH][ipNH4P] += rate_COmole.NH_NH3P_NH4P_N_2;
			mole.b[ipNH4P] += rate_COmole.b_NH_NH3P_NH4P_N;
			
			mole.c[ipNH3P][ipATN] += rate_COmole.NH_NH3P_NH4P_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_NH3P_NH4P_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_NH3P_NH4P_N;

		/*  >>refer   mol  NH     +  CH5+  =>   CH4   +     NH2+  UMIST  */

			mole.c[ipCH5P][ipNH] -= rate_COmole.NH_CH5P_CH4_NH2P_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_CH5P_CH4_NH2P_2;
			mole.b[ipNH] -= rate_COmole.b_NH_CH5P_CH4_NH2P;
			
			mole.c[ipCH5P][ipCH5P] -= rate_COmole.NH_CH5P_CH4_NH2P_1;
			mole.c[ipNH][ipCH5P] -= rate_COmole.NH_CH5P_CH4_NH2P_2;
			mole.b[ipCH5P] -= rate_COmole.b_NH_CH5P_CH4_NH2P;
			
			mole.c[ipCH5P][ipCH4] += rate_COmole.NH_CH5P_CH4_NH2P_1;
			mole.c[ipNH][ipCH4] += rate_COmole.NH_CH5P_CH4_NH2P_2;
			mole.b[ipCH4] += rate_COmole.b_NH_CH5P_CH4_NH2P;
			
			mole.c[ipCH5P][ipNH2P] += rate_COmole.NH_CH5P_CH4_NH2P_1;
			mole.c[ipNH][ipNH2P] += rate_COmole.NH_CH5P_CH4_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH_CH5P_CH4_NH2P;

		/*  >>refer   mol  NH+    +  H2O   =>   NH3+  +     O     UMIST  */

			mole.c[ipH2O][ipNHP] -= rate_COmole.NHP_H2O_NH3P_O_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_H2O_NH3P_O_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_H2O_NH3P_O;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NHP_H2O_NH3P_O_1;
			mole.c[ipNHP][ipH2O] -= rate_COmole.NHP_H2O_NH3P_O_2;
			mole.b[ipH2O] -= rate_COmole.b_NHP_H2O_NH3P_O;
			
			mole.c[ipH2O][ipNH3P] += rate_COmole.NHP_H2O_NH3P_O_1;
			mole.c[ipNHP][ipNH3P] += rate_COmole.NHP_H2O_NH3P_O_2;
			mole.b[ipNH3P] += rate_COmole.b_NHP_H2O_NH3P_O;		
			
			mole.c[ipH2O][ipATO] += rate_COmole.NHP_H2O_NH3P_O_1;
			mole.c[ipNHP][ipATO] += rate_COmole.NHP_H2O_NH3P_O_2;
			mole.b[ipATO] += rate_COmole.b_NHP_H2O_NH3P_O;		

		/*  >>refer   mol  NH+    +  H2O   =>   H3O+  +     N     UMIST  */

			mole.c[ipH2O][ipNHP] -= rate_COmole.NHP_H2O_H3OP_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_H2O_H3OP_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_H2O_H3OP_N;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NHP_H2O_H3OP_N_1;
			mole.c[ipNHP][ipH2O] -= rate_COmole.NHP_H2O_H3OP_N_2;
			mole.b[ipH2O] -= rate_COmole.b_NHP_H2O_H3OP_N;
			
			mole.c[ipH2O][ipH3OP] += rate_COmole.NHP_H2O_H3OP_N_1;
			mole.c[ipNHP][ipH3OP] += rate_COmole.NHP_H2O_H3OP_N_2;
			mole.b[ipH3OP] += rate_COmole.b_NHP_H2O_H3OP_N;
			
			mole.c[ipH2O][ipATN] += rate_COmole.NHP_H2O_H3OP_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_H2O_H3OP_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_H2O_H3OP_N;

		/*  >>refer   mol  NH+    +  H2O   =>   HNO+  +     H2    UMIST  */

			mole.c[ipH2O][ipNHP] -= rate_COmole.NHP_H2O_HNOP_H2_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_H2O_HNOP_H2_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_H2O_HNOP_H2;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NHP_H2O_HNOP_H2_1;
			mole.c[ipNHP][ipH2O] -= rate_COmole.NHP_H2O_HNOP_H2_2;
			mole.b[ipH2O] -= rate_COmole.b_NHP_H2O_HNOP_H2;
			
			mole.c[ipH2O][ipHNOP] += rate_COmole.NHP_H2O_HNOP_H2_1;
			mole.c[ipNHP][ipHNOP] += rate_COmole.NHP_H2O_HNOP_H2_2;
			mole.b[ipHNOP] += rate_COmole.b_NHP_H2O_HNOP_H2;
			
		/*  >>refer   mol  NH     +  H2O+  =>   H3O+  +     N     UMIST  */

			mole.c[ipH2OP][ipNH] -= rate_COmole.NH_H2OP_H3OP_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_H2OP_H3OP_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_H2OP_H3OP_N;
			
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.NH_H2OP_H3OP_N_1;
			mole.c[ipNH][ipH2OP] -= rate_COmole.NH_H2OP_H3OP_N_2;
			mole.b[ipH2OP] -= rate_COmole.b_NH_H2OP_H3OP_N;
			
			mole.c[ipH2OP][ipH3OP] += rate_COmole.NH_H2OP_H3OP_N_1;
			mole.c[ipNH][ipH3OP] += rate_COmole.NH_H2OP_H3OP_N_2;
			mole.b[ipH3OP] += rate_COmole.b_NH_H2OP_H3OP_N;
			
			mole.c[ipH2OP][ipATN] += rate_COmole.NH_H2OP_H3OP_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_H2OP_H3OP_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_H2OP_H3OP_N;

		/*  >>refer   mol  NH+    +  H2O   =>   OH    +     NH2+  UMIST  */

			mole.c[ipH2O][ipNHP] -= rate_COmole.NHP_H2O_OH_NH2P_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_H2O_OH_NH2P_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_H2O_OH_NH2P;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NHP_H2O_OH_NH2P_1;
			mole.c[ipNHP][ipH2O] -= rate_COmole.NHP_H2O_OH_NH2P_2;
			mole.b[ipH2O] -= rate_COmole.b_NHP_H2O_OH_NH2P;
			
			mole.c[ipH2O][ipOH] += rate_COmole.NHP_H2O_OH_NH2P_1;
			mole.c[ipNHP][ipOH] += rate_COmole.NHP_H2O_OH_NH2P_2;
			mole.b[ipOH] += rate_COmole.b_NHP_H2O_OH_NH2P;
			
			mole.c[ipH2O][ipNH2P] += rate_COmole.NHP_H2O_OH_NH2P_1;
			mole.c[ipNHP][ipNH2P] += rate_COmole.NHP_H2O_OH_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NHP_H2O_OH_NH2P;

		/*  >>refer   mol  NH+    +  CN    =>   HCN+  +     N     UMIST  */

			mole.c[ipCN][ipCN] -= rate_COmole.NHP_CN_HCNP_N_1;
			mole.c[ipNHP][ipCN] -= rate_COmole.NHP_CN_HCNP_N_2;
			mole.b[ipCN] -= rate_COmole.b_NHP_CN_HCNP_N;
			
			mole.c[ipCN][ipNHP] -= rate_COmole.NHP_CN_HCNP_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_CN_HCNP_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_CN_HCNP_N;
			
			mole.c[ipCN][ipHCNP] += rate_COmole.NHP_CN_HCNP_N_1;
			mole.c[ipNHP][ipHCNP] += rate_COmole.NHP_CN_HCNP_N_2;
			mole.b[ipHCNP] += rate_COmole.b_NHP_CN_HCNP_N;
			
			mole.c[ipCN][ipATN] += rate_COmole.NHP_CN_HCNP_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_CN_HCNP_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_CN_HCNP_N;
			
		/*  >>refer   mol  NH     +  HCN+  =>   CN    +     NH2+  UMIST  */

			mole.c[ipHCNP][ipNH] -= rate_COmole.NH_HCNP_CN_NH2P_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_HCNP_CN_NH2P_2;
			mole.b[ipNH] -= rate_COmole.b_NH_HCNP_CN_NH2P;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.NH_HCNP_CN_NH2P_1;
			mole.c[ipNH][ipHCNP] -= rate_COmole.NH_HCNP_CN_NH2P_2;
			mole.b[ipHCNP] -= rate_COmole.b_NH_HCNP_CN_NH2P;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.NH_HCNP_CN_NH2P_1;
			mole.c[ipNH][ipCN] += rate_COmole.NH_HCNP_CN_NH2P_2;
			mole.b[ipCN] += rate_COmole.b_NH_HCNP_CN_NH2P;
			
			mole.c[ipHCNP][ipNH2P] += rate_COmole.NH_HCNP_CN_NH2P_1;
			mole.c[ipNH][ipNH2P] += rate_COmole.NH_HCNP_CN_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH_HCNP_CN_NH2P;

		/*  >>refer   mol  NH     +  CO+   =>   HCO+  +     N     UMIST  */

			mole.c[ipCOP][ipNH] -= rate_COmole.NH_COP_HCOP_N_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_COP_HCOP_N_2;
			mole.b[ipNH] -= rate_COmole.b_NH_COP_HCOP_N;
			
			mole.c[ipCOP][ipCOP] -= rate_COmole.NH_COP_HCOP_N_1;
			mole.c[ipNH][ipCOP] -= rate_COmole.NH_COP_HCOP_N_2;
			mole.b[ipCOP] -= rate_COmole.b_NH_COP_HCOP_N;
			
			mole.c[ipCOP][ipHCOP] += rate_COmole.NH_COP_HCOP_N_1;
			mole.c[ipNH][ipHCOP] += rate_COmole.NH_COP_HCOP_N_2;
			mole.b[ipHCOP] += rate_COmole.b_NH_COP_HCOP_N;
			
			mole.c[ipCOP][ipATN] += rate_COmole.NH_COP_HCOP_N_1;
			mole.c[ipNH][ipATN] += rate_COmole.NH_COP_HCOP_N_2;
			mole.b[ipATN] += rate_COmole.b_NH_COP_HCOP_N;
			
		/*  >>refer   mol  NH     +  SI+   =>   SIN+  +     H     UMIST  */

			mole.c[ipSIP][ipNH] -= rate_COmole.NH_SIP_SINP_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_SIP_SINP_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_SIP_SINP_H;
			
			mole.c[ipSIP][ipSIP] -= rate_COmole.NH_SIP_SINP_H_1;
			mole.c[ipNH][ipSIP] -= rate_COmole.NH_SIP_SINP_H_2;
			mole.b[ipSIP] -= rate_COmole.b_NH_SIP_SINP_H;
			
			mole.c[ipSIP][ipSINP] += rate_COmole.NH_SIP_SINP_H_1;
			mole.c[ipNH][ipSINP] += rate_COmole.NH_SIP_SINP_H_2;
			mole.b[ipSINP] += rate_COmole.b_NH_SIP_SINP_H;
			
		/*  >>refer   mol  NH     +  HNO+  =>   NO    +     NH2+  UMIST  */

			mole.c[ipHNOP][ipNH] -= rate_COmole.NH_HNOP_NO_NH2P_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_HNOP_NO_NH2P_2;
			mole.b[ipNH] -= rate_COmole.b_NH_HNOP_NO_NH2P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.NH_HNOP_NO_NH2P_1;
			mole.c[ipNH][ipHNOP] -= rate_COmole.NH_HNOP_NO_NH2P_2;
			mole.b[ipHNOP] -= rate_COmole.b_NH_HNOP_NO_NH2P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.NH_HNOP_NO_NH2P_1;
			mole.c[ipNH][ipNO] += rate_COmole.NH_HNOP_NO_NH2P_2;
			mole.b[ipNO] += rate_COmole.b_NH_HNOP_NO_NH2P;
			
			mole.c[ipHNOP][ipNH2P] += rate_COmole.NH_HNOP_NO_NH2P_1;
			mole.c[ipNH][ipNH2P] += rate_COmole.NH_HNOP_NO_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH_HNOP_NO_NH2P;

		/*  >>refer   mol  NH     +  O2+   =>   HNO+  +     O     UMIST  */
			
			mole.c[ipO2P][ipO2P] -= rate_COmole.NH_O2P_HNOP_O_1;
			mole.c[ipNH][ipO2P] -= rate_COmole.NH_O2P_HNOP_O_2;
			mole.b[ipO2P] -= rate_COmole.b_NH_O2P_HNOP_O;

			mole.c[ipO2P][ipNH] -= rate_COmole.NH_O2P_HNOP_O_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_O2P_HNOP_O_2;
			mole.b[ipNH] -= rate_COmole.b_NH_O2P_HNOP_O;
			
			mole.c[ipO2P][ipHNOP] += rate_COmole.NH_O2P_HNOP_O_1;
			mole.c[ipNH][ipHNOP] += rate_COmole.NH_O2P_HNOP_O_2;
			mole.b[ipHNOP] += rate_COmole.b_NH_O2P_HNOP_O;
			
			mole.c[ipO2P][ipATO] += rate_COmole.NH_O2P_HNOP_O_1;
			mole.c[ipNH][ipATO] += rate_COmole.NH_O2P_HNOP_O_2;
			mole.b[ipATO] += rate_COmole.b_NH_O2P_HNOP_O;
			
		/*  >>refer   mol  NH+    +  O2    =>   NO+   +     OH    UMIST  */

			mole.c[ipO2][ipNHP] -= rate_COmole.NHP_O2_NOP_OH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_O2_NOP_OH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_O2_NOP_OH;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NHP_O2_NOP_OH_1;
			mole.c[ipNHP][ipO2] -= rate_COmole.NHP_O2_NOP_OH_2;
			mole.b[ipO2] -= rate_COmole.b_NHP_O2_NOP_OH;
			
			mole.c[ipO2][ipNOP] += rate_COmole.NHP_O2_NOP_OH_1;
			mole.c[ipNHP][ipNOP] += rate_COmole.NHP_O2_NOP_OH_2;
			mole.b[ipNOP] += rate_COmole.b_NHP_O2_NOP_OH;
			
			mole.c[ipO2][ipOH] += rate_COmole.NHP_O2_NOP_OH_1;
			mole.c[ipNHP][ipOH] += rate_COmole.NHP_O2_NOP_OH_2;
			mole.b[ipOH] += rate_COmole.b_NHP_O2_NOP_OH;

		/*  >>refer   mol  NH     +  O2+   =>   NO2+  +     H     UMIST  */

			mole.c[ipO2P][ipNH] -= rate_COmole.NH_O2P_NO2P_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_O2P_NO2P_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_O2P_NO2P_H;
			
			mole.c[ipO2P][ipO2P] -= rate_COmole.NH_O2P_NO2P_H_1;
			mole.c[ipNH][ipO2P] -= rate_COmole.NH_O2P_NO2P_H_2;
			mole.b[ipO2P] -= rate_COmole.b_NH_O2P_NO2P_H;
			
			mole.c[ipO2P][ipNO2P] += rate_COmole.NH_O2P_NO2P_H_1;
			mole.c[ipNH][ipNO2P] += rate_COmole.NH_O2P_NO2P_H_2;
			mole.b[ipNO2P] += rate_COmole.b_NH_O2P_NO2P_H;
			
		/*  >>refer   mol  NH+    +  S     =>   HS+   +     N     UMIST  */

			mole.c[ipATS][ipNHP] -= rate_COmole.NHP_S_HSP_N_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_S_HSP_N_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_S_HSP_N;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NHP_S_HSP_N_1;
			mole.c[ipNHP][ipATS] -= rate_COmole.NHP_S_HSP_N_2;
			mole.b[ipATS] -= rate_COmole.b_NHP_S_HSP_N;
			
			mole.c[ipATS][ipHSP] += rate_COmole.NHP_S_HSP_N_1;
			mole.c[ipNHP][ipHSP] += rate_COmole.NHP_S_HSP_N_2;
			mole.b[ipHSP] += rate_COmole.b_NHP_S_HSP_N;

			mole.c[ipATS][ipATN] += rate_COmole.NHP_S_HSP_N_1;
			mole.c[ipNHP][ipATN] += rate_COmole.NHP_S_HSP_N_2;
			mole.b[ipATN] += rate_COmole.b_NHP_S_HSP_N;
			
		/*  >>refer   mol  NH+    +  S     =>   NS+   +     H     UMIST  */

			mole.c[ipATS][ipNHP] -= rate_COmole.NHP_S_NSP_H_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_S_NSP_H_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_S_NSP_H;

			mole.c[ipATS][ipATS] -= rate_COmole.NHP_S_NSP_H_1;
			mole.c[ipNHP][ipATS] -= rate_COmole.NHP_S_NSP_H_2;
			mole.b[ipATS] -= rate_COmole.b_NHP_S_NSP_H;
			
			mole.c[ipATS][ipNSP] += rate_COmole.NHP_S_NSP_H_1;
			mole.c[ipNHP][ipNSP] += rate_COmole.NHP_S_NSP_H_2;
			mole.b[ipNSP] += rate_COmole.b_NHP_S_NSP_H;

		/*  >>refer   mol  NH     +  S+    =>   NS+   +     H     UMIST  */

			mole.c[ipSP][ipNH] -= rate_COmole.NH_SP_NSP_H_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_SP_NSP_H_2;
			mole.b[ipNH] -= rate_COmole.b_NH_SP_NSP_H;
			
			mole.c[ipSP][ipSP] -= rate_COmole.NH_SP_NSP_H_1;
			mole.c[ipNH][ipSP] -= rate_COmole.NH_SP_NSP_H_2;
			mole.b[ipSP] -= rate_COmole.b_NH_SP_NSP_H;
			
			mole.c[ipSP][ipNSP] += rate_COmole.NH_SP_NSP_H_1;
			mole.c[ipNH][ipNSP] += rate_COmole.NH_SP_NSP_H_2;
			mole.b[ipNSP] += rate_COmole.b_NH_SP_NSP_H;

		/*  >>refer   mol  CH3+   +  NH3   =>   NH4+  +     CH2   UMIST  */

			mole.c[ipNH3][ipCH3P] -= rate_COmole.CH3P_NH3_NH4P_CH2_1;
			mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH3P_NH3_NH4P_CH2_2;
			mole.b[ipCH3P] -= rate_COmole.b_CH3P_NH3_NH4P_CH2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CH3P_NH3_NH4P_CH2_1;
			mole.c[ipCH3P][ipNH3] -= rate_COmole.CH3P_NH3_NH4P_CH2_2;
			mole.b[ipNH3] -= rate_COmole.b_CH3P_NH3_NH4P_CH2;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.CH3P_NH3_NH4P_CH2_1;
			mole.c[ipCH3P][ipNH4P] += rate_COmole.CH3P_NH3_NH4P_CH2_2;
			mole.b[ipNH4P] += rate_COmole.b_CH3P_NH3_NH4P_CH2;
			
			mole.c[ipNH3][ipCH2] += rate_COmole.CH3P_NH3_NH4P_CH2_1;
			mole.c[ipCH3P][ipCH2] += rate_COmole.CH3P_NH3_NH4P_CH2_2;
			mole.b[ipCH2] += rate_COmole.b_CH3P_NH3_NH4P_CH2;

		/*  >>refer   mol  CH3+   +  S     =>   HCS+  +     H2    UMIST  */

			mole.c[ipATS][ipCH3P] -= rate_COmole.CH3P_S_HCSP_H2_1;
			mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH3P_S_HCSP_H2_2;
			mole.b[ipCH3P] -= rate_COmole.b_CH3P_S_HCSP_H2;

			mole.c[ipATS][ipATS] -= rate_COmole.CH3P_S_HCSP_H2_1;
			mole.c[ipCH3P][ipATS] -= rate_COmole.CH3P_S_HCSP_H2_2;
			mole.b[ipATS] -= rate_COmole.b_CH3P_S_HCSP_H2;
			
			mole.c[ipATS][ipHCSP] += rate_COmole.CH3P_S_HCSP_H2_1;
			mole.c[ipCH3P][ipHCSP] += rate_COmole.CH3P_S_HCSP_H2_2;
			mole.b[ipHCSP] += rate_COmole.b_CH3P_S_HCSP_H2;

		/*  >>refer   mol  O      +  NH2+  =>   HNO+  +     H     UMIST  */

			mole.c[ipNH2P][ipNH2P] -= rate_COmole.O_NH2P_HNOP_H_1;
			mole.c[ipATO][ipNH2P] -= rate_COmole.O_NH2P_HNOP_H_2;
			mole.b[ipNH2P] -= rate_COmole.b_O_NH2P_HNOP_H;

			mole.c[ipNH2P][ipATO] -= rate_COmole.O_NH2P_HNOP_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NH2P_HNOP_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_NH2P_HNOP_H;
			
			mole.c[ipNH2P][ipHNOP] += rate_COmole.O_NH2P_HNOP_H_1;
			mole.c[ipATO][ipHNOP] += rate_COmole.O_NH2P_HNOP_H_2;
			mole.b[ipHNOP] += rate_COmole.b_O_NH2P_HNOP_H;

		/*  >>refer   mol  O      +  NH3+  =>   HNO+  +     H2    UMIST  */

			mole.c[ipNH3P][ipNH3P] -= rate_COmole.O_NH3P_HNOP_H2_1;
			mole.c[ipATO][ipNH3P] -= rate_COmole.O_NH3P_HNOP_H2_2;
			mole.b[ipNH3P] -= rate_COmole.b_O_NH3P_HNOP_H2;

			mole.c[ipNH3P][ipATO] -= rate_COmole.O_NH3P_HNOP_H2_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NH3P_HNOP_H2_2;
			mole.b[ipATO] -= rate_COmole.b_O_NH3P_HNOP_H2;
			
			mole.c[ipNH3P][ipHNOP] += rate_COmole.O_NH3P_HNOP_H2_1;
			mole.c[ipATO][ipHNOP] += rate_COmole.O_NH3P_HNOP_H2_2;
			mole.b[ipHNOP] += rate_COmole.b_O_NH3P_HNOP_H2;

		/*  >>refer   mol  O+     +  CN    =>   NO+   +     C     UMIST  */

			mole.c[ipCN][ipOP] -= rate_COmole.OP_CN_NOP_C_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_CN_NOP_C_2;
			mole.b[ipOP] -= rate_COmole.b_OP_CN_NOP_C;
			
			mole.c[ipCN][ipCN] -= rate_COmole.OP_CN_NOP_C_1;
			mole.c[ipOP][ipCN] -= rate_COmole.OP_CN_NOP_C_2;
			mole.b[ipCN] -= rate_COmole.b_OP_CN_NOP_C;
			
			mole.c[ipCN][ipNOP] += rate_COmole.OP_CN_NOP_C_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_CN_NOP_C_2;
			mole.b[ipNOP] += rate_COmole.b_OP_CN_NOP_C;
			
			mole.c[ipCN][ipATC] += rate_COmole.OP_CN_NOP_C_1;
			mole.c[ipOP][ipATC] += rate_COmole.OP_CN_NOP_C_2;
			mole.b[ipATC] += rate_COmole.b_OP_CN_NOP_C;

		/*  >>refer   mol  O+     +  HCN   =>   CO+   +     NH    UMIST  */

			mole.c[ipHCN][ipOP] -= rate_COmole.OP_HCN_COP_NH_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_HCN_COP_NH_2;
			mole.b[ipOP] -= rate_COmole.b_OP_HCN_COP_NH;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.OP_HCN_COP_NH_1;
			mole.c[ipOP][ipHCN] -= rate_COmole.OP_HCN_COP_NH_2;
			mole.b[ipHCN] -= rate_COmole.b_OP_HCN_COP_NH;
			
			mole.c[ipHCN][ipCOP] += rate_COmole.OP_HCN_COP_NH_1;
			mole.c[ipOP][ipCOP] += rate_COmole.OP_HCN_COP_NH_2;
			mole.b[ipCOP] += rate_COmole.b_OP_HCN_COP_NH;
			
			mole.c[ipHCN][ipNH] += rate_COmole.OP_HCN_COP_NH_1;
			mole.c[ipOP][ipNH] += rate_COmole.OP_HCN_COP_NH_2;
			mole.b[ipNH] += rate_COmole.b_OP_HCN_COP_NH;

		/*  >>refer   mol  O+     +  HCN   =>   NO+   +     CH    UMIST  */

			mole.c[ipHCN][ipOP] -= rate_COmole.OP_HCN_NOP_CH_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_HCN_NOP_CH_2;
			mole.b[ipOP] -= rate_COmole.b_OP_HCN_NOP_CH;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.OP_HCN_NOP_CH_1;
			mole.c[ipOP][ipHCN] -= rate_COmole.OP_HCN_NOP_CH_2;
			mole.b[ipHCN] -= rate_COmole.b_OP_HCN_NOP_CH;
			
			mole.c[ipHCN][ipNOP] += rate_COmole.OP_HCN_NOP_CH_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_HCN_NOP_CH_2;
			mole.b[ipNOP] += rate_COmole.b_OP_HCN_NOP_CH;
			
			mole.c[ipHCN][ipCH] += rate_COmole.OP_HCN_NOP_CH_1;
			mole.c[ipOP][ipCH] += rate_COmole.OP_HCN_NOP_CH_2;
			mole.b[ipCH] += rate_COmole.b_OP_HCN_NOP_CH;

		/*  >>refer   mol  O+     +  HCN   =>   HCO+  +     N     UMIST  */

			mole.c[ipHCN][ipOP] -= rate_COmole.OP_HCN_HCOP_N_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_HCN_HCOP_N_2;
			mole.b[ipOP] -= rate_COmole.b_OP_HCN_HCOP_N;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.OP_HCN_HCOP_N_1;
			mole.c[ipOP][ipHCN] -= rate_COmole.OP_HCN_HCOP_N_2;
			mole.b[ipHCN] -= rate_COmole.b_OP_HCN_HCOP_N;
			
			mole.c[ipHCN][ipHCOP] += rate_COmole.OP_HCN_HCOP_N_1;
			mole.c[ipOP][ipHCOP] += rate_COmole.OP_HCN_HCOP_N_2;
			mole.b[ipHCOP] += rate_COmole.b_OP_HCN_HCOP_N;
			
			mole.c[ipHCN][ipATN] += rate_COmole.OP_HCN_HCOP_N_1;
			mole.c[ipOP][ipATN] += rate_COmole.OP_HCN_HCOP_N_2;
			mole.b[ipATN] += rate_COmole.b_OP_HCN_HCOP_N;

		/*  >>refer   mol  O+     +  N2    =>   NO+   +     N     UMIST  */

			mole.c[ipN2][ipOP] -= rate_COmole.OP_N2_NOP_N_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_N2_NOP_N_2;
			mole.b[ipOP] -= rate_COmole.b_OP_N2_NOP_N;
			
			mole.c[ipN2][ipN2] -= rate_COmole.OP_N2_NOP_N_1;
			mole.c[ipOP][ipN2] -= rate_COmole.OP_N2_NOP_N_2;
			mole.b[ipN2] -= rate_COmole.b_OP_N2_NOP_N;
			
			mole.c[ipN2][ipNOP] += rate_COmole.OP_N2_NOP_N_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_N2_NOP_N_2;
			mole.b[ipNOP] += rate_COmole.b_OP_N2_NOP_N;
			
			mole.c[ipN2][ipATN] += rate_COmole.OP_N2_NOP_N_1;
			mole.c[ipOP][ipATN] += rate_COmole.OP_N2_NOP_N_2;
			mole.b[ipATN] += rate_COmole.b_OP_N2_NOP_N;
			
		/*  >>refer   mol  O      +  N2+   =>   NO+   +     N     UMIST  */

			mole.c[ipN2P][ipN2P] -= rate_COmole.O_N2P_NOP_N_1;
			mole.c[ipATO][ipN2P] -= rate_COmole.O_N2P_NOP_N_2;
			mole.b[ipN2P] -= rate_COmole.b_O_N2P_NOP_N;
			
			mole.c[ipN2P][ipATO] -= rate_COmole.O_N2P_NOP_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_N2P_NOP_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_N2P_NOP_N;
			
			mole.c[ipN2P][ipNOP] += rate_COmole.O_N2P_NOP_N_1;
			mole.c[ipATO][ipNOP] += rate_COmole.O_N2P_NOP_N_2;
			mole.b[ipNOP] += rate_COmole.b_O_N2P_NOP_N;

			mole.c[ipN2P][ipATN] += rate_COmole.O_N2P_NOP_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_N2P_NOP_N_2;
			mole.b[ipATN] += rate_COmole.b_O_N2P_NOP_N;

		/*  >>refer   mol  O      +  HNO+  =>   NO2+  +     H     UMIST  */

			mole.c[ipHNOP][ipHNOP] -= rate_COmole.O_HNOP_NO2P_H_1;
			mole.c[ipATO][ipHNOP] -= rate_COmole.O_HNOP_NO2P_H_2;
			mole.b[ipHNOP] -= rate_COmole.b_O_HNOP_NO2P_H;

			mole.c[ipHNOP][ipATO] -= rate_COmole.O_HNOP_NO2P_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HNOP_NO2P_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HNOP_NO2P_H;
			
			mole.c[ipHNOP][ipNO2P] += rate_COmole.O_HNOP_NO2P_H_1;
			mole.c[ipATO][ipNO2P] += rate_COmole.O_HNOP_NO2P_H_2;
			mole.b[ipNO2P] += rate_COmole.b_O_HNOP_NO2P_H;

		/*  >>refer   mol  O      +  HS+   =>   S+    +     OH    UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.O_HSP_SP_OH_1;
			mole.c[ipATO][ipHSP] -= rate_COmole.O_HSP_SP_OH_2;
			mole.b[ipHSP] -= rate_COmole.b_O_HSP_SP_OH;

			mole.c[ipHSP][ipATO] -= rate_COmole.O_HSP_SP_OH_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HSP_SP_OH_2;
			mole.b[ipATO] -= rate_COmole.b_O_HSP_SP_OH;
			
			mole.c[ipHSP][ipSP] += rate_COmole.O_HSP_SP_OH_1;
			mole.c[ipATO][ipSP] += rate_COmole.O_HSP_SP_OH_2;
			mole.b[ipSP] += rate_COmole.b_O_HSP_SP_OH;
			
			mole.c[ipHSP][ipOH] += rate_COmole.O_HSP_SP_OH_1;
			mole.c[ipATO][ipOH] += rate_COmole.O_HSP_SP_OH_2;
			mole.b[ipOH] += rate_COmole.b_O_HSP_SP_OH;

		/*  >>refer   mol  O      +  HS+   =>   SO+   +     H     UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.O_HSP_SOP_H_1;
			mole.c[ipATO][ipHSP] -= rate_COmole.O_HSP_SOP_H_2;
			mole.b[ipHSP] -= rate_COmole.b_O_HSP_SOP_H;

			mole.c[ipHSP][ipATO] -= rate_COmole.O_HSP_SOP_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HSP_SOP_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HSP_SOP_H;
			
			mole.c[ipHSP][ipSOP] += rate_COmole.O_HSP_SOP_H_1;
			mole.c[ipATO][ipSOP] += rate_COmole.O_HSP_SOP_H_2;
			mole.b[ipSOP] += rate_COmole.b_O_HSP_SOP_H;

		/*  >>refer   mol  O      +  SIN+  =>   SIO+  +     N     UMIST  */

			mole.c[ipSINP][ipSINP] -= rate_COmole.O_SINP_SIOP_N_1;
			mole.c[ipATO][ipSINP] -= rate_COmole.O_SINP_SIOP_N_2;
			mole.b[ipSINP] -= rate_COmole.b_O_SINP_SIOP_N;
			
			mole.c[ipSINP][ipATO] -= rate_COmole.O_SINP_SIOP_N_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_SINP_SIOP_N_2;
			mole.b[ipATO] -= rate_COmole.b_O_SINP_SIOP_N;
			
			mole.c[ipSINP][ipSIOP] += rate_COmole.O_SINP_SIOP_N_1;
			mole.c[ipATO][ipSIOP] += rate_COmole.O_SINP_SIOP_N_2;
			mole.b[ipSIOP] += rate_COmole.b_O_SINP_SIOP_N;

			mole.c[ipSINP][ipATN] += rate_COmole.O_SINP_SIOP_N_1;
			mole.c[ipATO][ipATN] += rate_COmole.O_SINP_SIOP_N_2;
			mole.b[ipATN] += rate_COmole.b_O_SINP_SIOP_N;

		/*  >>refer   mol  O+     +  N2O   =>   NO+   +     NO    UMIST  */

			mole.c[ipN2O][ipOP] -= rate_COmole.OP_N2O_NOP_NO_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_N2O_NOP_NO_2;
			mole.b[ipOP] -= rate_COmole.b_OP_N2O_NOP_NO;
			
			mole.c[ipN2O][ipN2O] -= rate_COmole.OP_N2O_NOP_NO_1;
			mole.c[ipOP][ipN2O] -= rate_COmole.OP_N2O_NOP_NO_2;
			mole.b[ipN2O] -= rate_COmole.b_OP_N2O_NOP_NO;
			
			mole.c[ipN2O][ipNOP] += rate_COmole.OP_N2O_NOP_NO_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_N2O_NOP_NO_2;
			mole.b[ipNOP] += rate_COmole.b_OP_N2O_NOP_NO;
			
			mole.c[ipN2O][ipNO] += rate_COmole.OP_N2O_NOP_NO_1;
			mole.c[ipOP][ipNO] += rate_COmole.OP_N2O_NOP_NO_2;
			mole.b[ipNO] += rate_COmole.b_OP_N2O_NOP_NO;

		/*  >>refer   mol  O      +  CS+   =>   S     +     CO+   UMIST  */

			mole.c[ipCSP][ipCSP] -= rate_COmole.O_CSP_S_COP_1;
			mole.c[ipATO][ipCSP] -= rate_COmole.O_CSP_S_COP_2;
			mole.b[ipCSP] -= rate_COmole.b_O_CSP_S_COP;
			
			mole.c[ipCSP][ipATO] -= rate_COmole.O_CSP_S_COP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_CSP_S_COP_2;
			mole.b[ipATO] -= rate_COmole.b_O_CSP_S_COP;
			
			mole.c[ipCSP][ipCOP] += rate_COmole.O_CSP_S_COP_1;
			mole.c[ipATO][ipCOP] += rate_COmole.O_CSP_S_COP_2;
			mole.b[ipCOP] += rate_COmole.b_O_CSP_S_COP;

			mole.c[ipCSP][ipATS] += rate_COmole.O_CSP_S_COP_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_CSP_S_COP_2;
			mole.b[ipATS] += rate_COmole.b_O_CSP_S_COP;

		/*  >>refer   mol  O      +  HCS+  =>   S     +     HCO+  UMIST  */

			mole.c[ipHCSP][ipHCSP] -= rate_COmole.O_HCSP_S_HCOP_1;
			mole.c[ipATO][ipHCSP] -= rate_COmole.O_HCSP_S_HCOP_2;
			mole.b[ipHCSP] -= rate_COmole.b_O_HCSP_S_HCOP;
			
			mole.c[ipHCSP][ipATO] -= rate_COmole.O_HCSP_S_HCOP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HCSP_S_HCOP_2;
			mole.b[ipATO] -= rate_COmole.b_O_HCSP_S_HCOP;
			
			mole.c[ipHCSP][ipHCOP] += rate_COmole.O_HCSP_S_HCOP_1;
			mole.c[ipATO][ipHCOP] += rate_COmole.O_HCSP_S_HCOP_2;
			mole.b[ipHCOP] += rate_COmole.b_O_HCSP_S_HCOP;

			mole.c[ipHCSP][ipATS] += rate_COmole.O_HCSP_S_HCOP_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_HCSP_S_HCOP_2;
			mole.b[ipATS] += rate_COmole.b_O_HCSP_S_HCOP;

		/*  >>refer   mol  O      +  HCS+  =>   OCS+  +     H     UMIST  */

			mole.c[ipHCSP][ipHCSP] -= rate_COmole.O_HCSP_OCSP_H_1;
			mole.c[ipATO][ipHCSP] -= rate_COmole.O_HCSP_OCSP_H_2;
			mole.b[ipHCSP] -= rate_COmole.b_O_HCSP_OCSP_H;

			mole.c[ipHCSP][ipATO] -= rate_COmole.O_HCSP_OCSP_H_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HCSP_OCSP_H_2;
			mole.b[ipATO] -= rate_COmole.b_O_HCSP_OCSP_H;
			
			mole.c[ipHCSP][ipOCSP] += rate_COmole.O_HCSP_OCSP_H_1;
			mole.c[ipATO][ipOCSP] += rate_COmole.O_HCSP_OCSP_H_2;
			mole.b[ipOCSP] += rate_COmole.b_O_HCSP_OCSP_H;

		/*  >>refer   mol  O+     +  NO2   =>   O2    +     NO+   UMIST  */

			mole.c[ipNO2][ipOP] -= rate_COmole.OP_NO2_O2_NOP_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_NO2_O2_NOP_2;
			mole.b[ipOP] -= rate_COmole.b_OP_NO2_O2_NOP;
			
			mole.c[ipNO2][ipNO2] -= rate_COmole.OP_NO2_O2_NOP_1;
			mole.c[ipOP][ipNO2] -= rate_COmole.OP_NO2_O2_NOP_2;
			mole.b[ipNO2] -= rate_COmole.b_OP_NO2_O2_NOP;
			
			mole.c[ipNO2][ipO2] += rate_COmole.OP_NO2_O2_NOP_1;
			mole.c[ipOP][ipO2] += rate_COmole.OP_NO2_O2_NOP_2;
			mole.b[ipO2] += rate_COmole.b_OP_NO2_O2_NOP;
			
			mole.c[ipNO2][ipNOP] += rate_COmole.OP_NO2_O2_NOP_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_NO2_O2_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_OP_NO2_O2_NOP;

		/*  >>refer   mol  O      +  NS+   =>   S     +     NO+   UMIST  */

			mole.c[ipNSP][ipATO] -= rate_COmole.O_NSP_S_NOP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_NSP_S_NOP_2;
			mole.b[ipATO] -= rate_COmole.b_O_NSP_S_NOP;
			
			mole.c[ipNSP][ipNSP] -= rate_COmole.O_NSP_S_NOP_1;
			mole.c[ipATO][ipNSP] -= rate_COmole.O_NSP_S_NOP_2;
			mole.b[ipNSP] -= rate_COmole.b_O_NSP_S_NOP;
			
			mole.c[ipNSP][ipNOP] += rate_COmole.O_NSP_S_NOP_1;
			mole.c[ipATO][ipNOP] += rate_COmole.O_NSP_S_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_O_NSP_S_NOP;

			mole.c[ipNSP][ipATS] += rate_COmole.O_NSP_S_NOP_1;
			mole.c[ipATO][ipATS] += rate_COmole.O_NSP_S_NOP_2;
			mole.b[ipATS] += rate_COmole.b_O_NSP_S_NOP;

		/*  >>refer   mol  NH2+   +  NH2   =>   NH3+  +     NH    UMIST  */

			mole.c[ipNH2][ipNH2P] -= rate_COmole.NH2P_NH2_NH3P_NH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_NH2_NH3P_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_NH2_NH3P_NH;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2P_NH2_NH3P_NH_1;
			mole.c[ipNH2P][ipNH2] -= rate_COmole.NH2P_NH2_NH3P_NH_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2P_NH2_NH3P_NH;
			
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2P_NH2_NH3P_NH_1;
			mole.c[ipNH2P][ipNH3P] += rate_COmole.NH2P_NH2_NH3P_NH_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2P_NH2_NH3P_NH;
			
			mole.c[ipNH2][ipNH] += rate_COmole.NH2P_NH2_NH3P_NH_1;
			mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_NH2_NH3P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2P_NH2_NH3P_NH;

		/*  >>refer   mol  NH2    +  OH+   =>   NH3+  +     O     UMIST  */

			mole.c[ipOHP][ipNH2] -= rate_COmole.NH2_OHP_NH3P_O_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_OHP_NH3P_O_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_OHP_NH3P_O;
			
			mole.c[ipOHP][ipOHP] -= rate_COmole.NH2_OHP_NH3P_O_1;
			mole.c[ipNH2][ipOHP] -= rate_COmole.NH2_OHP_NH3P_O_2;
			mole.b[ipOHP] -= rate_COmole.b_NH2_OHP_NH3P_O;
			
			mole.c[ipOHP][ipNH3P] += rate_COmole.NH2_OHP_NH3P_O_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_OHP_NH3P_O_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_OHP_NH3P_O;
			
			mole.c[ipOHP][ipATO] += rate_COmole.NH2_OHP_NH3P_O_1;
			mole.c[ipNH2][ipATO] += rate_COmole.NH2_OHP_NH3P_O_2;
			mole.b[ipATO] += rate_COmole.b_NH2_OHP_NH3P_O;

		/*  >>refer   mol  NH2+   +  OH    =>   H2O+  +     NH    UMIST  */

			mole.c[ipOH][ipNH2P] -= rate_COmole.NH2P_OH_H2OP_NH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_OH_H2OP_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_OH_H2OP_NH;
			
			mole.c[ipOH][ipOH] -= rate_COmole.NH2P_OH_H2OP_NH_1;
			mole.c[ipNH2P][ipOH] -= rate_COmole.NH2P_OH_H2OP_NH_2;
			mole.b[ipOH] -= rate_COmole.b_NH2P_OH_H2OP_NH;
						
			mole.c[ipOH][ipH2OP] += rate_COmole.NH2P_OH_H2OP_NH_1;
			mole.c[ipNH2P][ipH2OP] += rate_COmole.NH2P_OH_H2OP_NH_2;
			mole.b[ipH2OP] += rate_COmole.b_NH2P_OH_H2OP_NH;
			
			mole.c[ipOH][ipNH] += rate_COmole.NH2P_OH_H2OP_NH_1;
			mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_OH_H2OP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2P_OH_H2OP_NH;

		/*  >>refer   mol  NH2+   +  NH3   =>   NH4+  +     NH    UMIST  */

			mole.c[ipNH3][ipNH2P] -= rate_COmole.NH2P_NH3_NH4P_NH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_NH3_NH4P_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_NH3_NH4P_NH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH2P_NH3_NH4P_NH_1;
			mole.c[ipNH2P][ipNH3] -= rate_COmole.NH2P_NH3_NH4P_NH_2;
			mole.b[ipNH3] -= rate_COmole.b_NH2P_NH3_NH4P_NH;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH2P_NH3_NH4P_NH_1;
			mole.c[ipNH2P][ipNH4P] += rate_COmole.NH2P_NH3_NH4P_NH_2;
			mole.b[ipNH4P] += rate_COmole.b_NH2P_NH3_NH4P_NH;
			
			mole.c[ipNH3][ipNH] += rate_COmole.NH2P_NH3_NH4P_NH_1;
			mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_NH3_NH4P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2P_NH3_NH4P_NH;

		/*  >>refer   mol  NH2    +  NH3+  =>   NH4+  +     NH    UMIST  */

			mole.c[ipNH3P][ipNH2] -= rate_COmole.NH2_NH3P_NH4P_NH_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_NH3P_NH4P_NH_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_NH3P_NH4P_NH;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH2_NH3P_NH4P_NH_1;
			mole.c[ipNH2][ipNH3P] -= rate_COmole.NH2_NH3P_NH4P_NH_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH2_NH3P_NH4P_NH;
			
			mole.c[ipNH3P][ipNH4P] += rate_COmole.NH2_NH3P_NH4P_NH_1;
			mole.c[ipNH2][ipNH4P] += rate_COmole.NH2_NH3P_NH4P_NH_2;
			mole.b[ipNH4P] += rate_COmole.b_NH2_NH3P_NH4P_NH;
			
			mole.c[ipNH3P][ipNH] += rate_COmole.NH2_NH3P_NH4P_NH_1;
			mole.c[ipNH2][ipNH] += rate_COmole.NH2_NH3P_NH4P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2_NH3P_NH4P_NH;

		/*  >>refer   mol  NH2    +  CH5+  =>   NH3+  +     CH4   UMIST  */

			mole.c[ipCH5P][ipNH2] -= rate_COmole.NH2_CH5P_NH3P_CH4_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_CH5P_NH3P_CH4_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_CH5P_NH3P_CH4;
			
			mole.c[ipCH5P][ipCH5P] -= rate_COmole.NH2_CH5P_NH3P_CH4_1;
			mole.c[ipNH2][ipCH5P] -= rate_COmole.NH2_CH5P_NH3P_CH4_2;
			mole.b[ipCH5P] -= rate_COmole.b_NH2_CH5P_NH3P_CH4;
			
			mole.c[ipCH5P][ipNH3P] += rate_COmole.NH2_CH5P_NH3P_CH4_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_CH5P_NH3P_CH4_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_CH5P_NH3P_CH4;
			
			mole.c[ipCH5P][ipCH4] += rate_COmole.NH2_CH5P_NH3P_CH4_1;
			mole.c[ipNH2][ipCH4] += rate_COmole.NH2_CH5P_NH3P_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_NH2_CH5P_NH3P_CH4;

		/*  >>refer   mol  NH2+   +  H2O   =>   NH3+  +     OH    UMIST  */

			mole.c[ipH2O][ipNH2P] -= rate_COmole.NH2P_H2O_NH3P_OH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_H2O_NH3P_OH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_H2O_NH3P_OH;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NH2P_H2O_NH3P_OH_1;
			mole.c[ipNH2P][ipH2O] -= rate_COmole.NH2P_H2O_NH3P_OH_2;
			mole.b[ipH2O] -= rate_COmole.b_NH2P_H2O_NH3P_OH;
			
			mole.c[ipH2O][ipNH3P] += rate_COmole.NH2P_H2O_NH3P_OH_1;
			mole.c[ipNH2P][ipNH3P] += rate_COmole.NH2P_H2O_NH3P_OH_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2P_H2O_NH3P_OH;
			
			mole.c[ipH2O][ipOH] += rate_COmole.NH2P_H2O_NH3P_OH_1;
			mole.c[ipNH2P][ipOH] += rate_COmole.NH2P_H2O_NH3P_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH2P_H2O_NH3P_OH;

		/*  >>refer   mol  NH2+   +  H2O   =>   H3O+  +     NH    UMIST  */

			mole.c[ipH2O][ipNH2P] -= rate_COmole.NH2P_H2O_H3OP_NH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_H2O_H3OP_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_H2O_H3OP_NH;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NH2P_H2O_H3OP_NH_1;
			mole.c[ipNH2P][ipH2O] -= rate_COmole.NH2P_H2O_H3OP_NH_2;
			mole.b[ipH2O] -= rate_COmole.b_NH2P_H2O_H3OP_NH;
			
			mole.c[ipH2O][ipH3OP] += rate_COmole.NH2P_H2O_H3OP_NH_1;
			mole.c[ipNH2P][ipH3OP] += rate_COmole.NH2P_H2O_H3OP_NH_2;
			mole.b[ipH3OP] += rate_COmole.b_NH2P_H2O_H3OP_NH;
			
			mole.c[ipH2O][ipNH] += rate_COmole.NH2P_H2O_H3OP_NH_1;
			mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_H2O_H3OP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2P_H2O_H3OP_NH;

		/*  >>refer   mol  NH2    +  H2O+  =>   NH3+  +     OH    UMIST  */

			mole.c[ipH2OP][ipNH2] -= rate_COmole.NH2_H2OP_NH3P_OH_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_H2OP_NH3P_OH_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_H2OP_NH3P_OH;
			
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.NH2_H2OP_NH3P_OH_1;
			mole.c[ipNH2][ipH2OP] -= rate_COmole.NH2_H2OP_NH3P_OH_2;
			mole.b[ipH2OP] -= rate_COmole.b_NH2_H2OP_NH3P_OH;
			
			mole.c[ipH2OP][ipNH3P] += rate_COmole.NH2_H2OP_NH3P_OH_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_H2OP_NH3P_OH_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_H2OP_NH3P_OH;
			
			mole.c[ipH2OP][ipOH] += rate_COmole.NH2_H2OP_NH3P_OH_1;
			mole.c[ipNH2][ipOH] += rate_COmole.NH2_H2OP_NH3P_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH2_H2OP_NH3P_OH;

		/*  >>refer   mol  NH2+   +  H2O   =>   NH4+  +     O     UMIST  */

			mole.c[ipH2O][ipNH2P] -= rate_COmole.NH2P_H2O_NH4P_O_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_H2O_NH4P_O_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_H2O_NH4P_O;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NH2P_H2O_NH4P_O_1;
			mole.c[ipNH2P][ipH2O] -= rate_COmole.NH2P_H2O_NH4P_O_2;
			mole.b[ipH2O] -= rate_COmole.b_NH2P_H2O_NH4P_O;
			
			mole.c[ipH2O][ipNH4P] += rate_COmole.NH2P_H2O_NH4P_O_1;
			mole.c[ipNH2P][ipNH4P] += rate_COmole.NH2P_H2O_NH4P_O_2;
			mole.b[ipNH4P] += rate_COmole.b_NH2P_H2O_NH4P_O;
			
			mole.c[ipH2O][ipATO] += rate_COmole.NH2P_H2O_NH4P_O_1;
			mole.c[ipNH2P][ipATO] += rate_COmole.NH2P_H2O_NH4P_O_2;
			mole.b[ipATO] += rate_COmole.b_NH2P_H2O_NH4P_O;
			
		/*  >>refer   mol  NH2    +  H3O+  =>   H2O   +     NH3+  UMIST  */

			mole.c[ipH3OP][ipNH2] -= rate_COmole.NH2_H3OP_H2O_NH3P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_H3OP_H2O_NH3P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_H3OP_H2O_NH3P;
			
			mole.c[ipH3OP][ipH3OP] -= rate_COmole.NH2_H3OP_H2O_NH3P_1;
			mole.c[ipNH2][ipH3OP] -= rate_COmole.NH2_H3OP_H2O_NH3P_2;
			mole.b[ipH3OP] -= rate_COmole.b_NH2_H3OP_H2O_NH3P;
			
			mole.c[ipH3OP][ipH2O] += rate_COmole.NH2_H3OP_H2O_NH3P_1;
			mole.c[ipNH2][ipH2O] += rate_COmole.NH2_H3OP_H2O_NH3P_2;
			mole.b[ipH2O] += rate_COmole.b_NH2_H3OP_H2O_NH3P;
			
			mole.c[ipH3OP][ipNH3P] += rate_COmole.NH2_H3OP_H2O_NH3P_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_H3OP_H2O_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_H3OP_H2O_NH3P;

		/*  >>refer   mol  NH2    +  HCN+  =>   CN    +     NH3+  UMIST  */

			mole.c[ipHCNP][ipNH2] -= rate_COmole.NH2_HCNP_CN_NH3P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_HCNP_CN_NH3P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_HCNP_CN_NH3P;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.NH2_HCNP_CN_NH3P_1;
			mole.c[ipNH2][ipHCNP] -= rate_COmole.NH2_HCNP_CN_NH3P_2;
			mole.b[ipHCNP] -= rate_COmole.b_NH2_HCNP_CN_NH3P;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.NH2_HCNP_CN_NH3P_1;
			mole.c[ipNH2][ipCN] += rate_COmole.NH2_HCNP_CN_NH3P_2;
			mole.b[ipCN] += rate_COmole.b_NH2_HCNP_CN_NH3P;
			
			mole.c[ipHCNP][ipNH3P] += rate_COmole.NH2_HCNP_CN_NH3P_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_HCNP_CN_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_HCNP_CN_NH3P;

		/*  >>refer   mol  NH2    +  CO+   =>   HCO+  +     NH    UMIST  */

			mole.c[ipCOP][ipNH2] -= rate_COmole.NH2_COP_HCOP_NH_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_COP_HCOP_NH_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_COP_HCOP_NH;
			
			mole.c[ipCOP][ipCOP] -= rate_COmole.NH2_COP_HCOP_NH_1;
			mole.c[ipNH2][ipCOP] -= rate_COmole.NH2_COP_HCOP_NH_2;
			mole.b[ipCOP] -= rate_COmole.b_NH2_COP_HCOP_NH;
			
			mole.c[ipCOP][ipHCOP] += rate_COmole.NH2_COP_HCOP_NH_1;
			mole.c[ipNH2][ipHCOP] += rate_COmole.NH2_COP_HCOP_NH_2;
			mole.b[ipHCOP] += rate_COmole.b_NH2_COP_HCOP_NH;
			
			mole.c[ipCOP][ipNH] += rate_COmole.NH2_COP_HCOP_NH_1;
			mole.c[ipNH2][ipNH] += rate_COmole.NH2_COP_HCOP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2_COP_HCOP_NH;

		/*  >>refer   mol  NH2    +  HNO+  =>   NO    +     NH3+  UMIST  */

			mole.c[ipHNOP][ipNH2] -= rate_COmole.NH2_HNOP_NO_NH3P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_HNOP_NO_NH3P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_HNOP_NO_NH3P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.NH2_HNOP_NO_NH3P_1;
			mole.c[ipNH2][ipHNOP] -= rate_COmole.NH2_HNOP_NO_NH3P_2;
			mole.b[ipHNOP] -= rate_COmole.b_NH2_HNOP_NO_NH3P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.NH2_HNOP_NO_NH3P_1;
			mole.c[ipNH2][ipNO] += rate_COmole.NH2_HNOP_NO_NH3P_2;
			mole.b[ipNO] += rate_COmole.b_NH2_HNOP_NO_NH3P;
			
			mole.c[ipHNOP][ipNH3P] += rate_COmole.NH2_HNOP_NO_NH3P_1;
			mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_HNOP_NO_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2_HNOP_NO_NH3P;

		/*  >>refer   mol  NH2+   +  O2    =>   HNO+  +     OH    UMIST  */

			mole.c[ipO2][ipNH2P] -= rate_COmole.NH2P_O2_HNOP_OH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_O2_HNOP_OH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_O2_HNOP_OH;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NH2P_O2_HNOP_OH_1;
			mole.c[ipNH2P][ipO2] -= rate_COmole.NH2P_O2_HNOP_OH_2;
			mole.b[ipO2] -= rate_COmole.b_NH2P_O2_HNOP_OH;
			
			mole.c[ipO2][ipHNOP] += rate_COmole.NH2P_O2_HNOP_OH_1;
			mole.c[ipNH2P][ipHNOP] += rate_COmole.NH2P_O2_HNOP_OH_2;
			mole.b[ipHNOP] += rate_COmole.b_NH2P_O2_HNOP_OH;
			
			mole.c[ipO2][ipOH] += rate_COmole.NH2P_O2_HNOP_OH_1;
			mole.c[ipNH2P][ipOH] += rate_COmole.NH2P_O2_HNOP_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH2P_O2_HNOP_OH;

		/*  >>refer   mol  NH2+   +  S     =>   HS+   +     NH    UMIST  */

			mole.c[ipATS][ipNH2P] -= rate_COmole.NH2P_S_HSP_NH_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_S_HSP_NH_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_S_HSP_NH;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NH2P_S_HSP_NH_1;
			mole.c[ipNH2P][ipATS] -= rate_COmole.NH2P_S_HSP_NH_2;
			mole.b[ipATS] -= rate_COmole.b_NH2P_S_HSP_NH;
			
			mole.c[ipATS][ipHSP] += rate_COmole.NH2P_S_HSP_NH_1;
			mole.c[ipNH2P][ipHSP] += rate_COmole.NH2P_S_HSP_NH_2;
			mole.b[ipHSP] += rate_COmole.b_NH2P_S_HSP_NH;
			
			mole.c[ipATS][ipNH] += rate_COmole.NH2P_S_HSP_NH_1;
			mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_S_HSP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NH2P_S_HSP_NH;

		/*  >>refer   mol  CH4+   +  NH3   =>   NH4+  +     CH3   UMIST  */

			mole.c[ipNH3][ipCH4P] -= rate_COmole.CH4P_NH3_NH4P_CH3_1;
			mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4P_NH3_NH4P_CH3_2;
			mole.b[ipCH4P] -= rate_COmole.b_CH4P_NH3_NH4P_CH3;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CH4P_NH3_NH4P_CH3_1;
			mole.c[ipCH4P][ipNH3] -= rate_COmole.CH4P_NH3_NH4P_CH3_2;
			mole.b[ipNH3] -= rate_COmole.b_CH4P_NH3_NH4P_CH3;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.CH4P_NH3_NH4P_CH3_1;
			mole.c[ipCH4P][ipNH4P] += rate_COmole.CH4P_NH3_NH4P_CH3_2;
			mole.b[ipNH4P] += rate_COmole.b_CH4P_NH3_NH4P_CH3;
			
			mole.c[ipNH3][ipCH3] += rate_COmole.CH4P_NH3_NH4P_CH3_1;
			mole.c[ipCH4P][ipCH3] += rate_COmole.CH4P_NH3_NH4P_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH4P_NH3_NH4P_CH3;

		/*  >>refer   mol  CH4    +  NH3+  =>   NH4+  +     CH3   UMIST  */

			mole.c[ipNH3P][ipCH4] -= rate_COmole.CH4_NH3P_NH4P_CH3_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_NH3P_NH4P_CH3_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_NH3P_NH4P_CH3;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.CH4_NH3P_NH4P_CH3_1;
			mole.c[ipCH4][ipNH3P] -= rate_COmole.CH4_NH3P_NH4P_CH3_2;
			mole.b[ipNH3P] -= rate_COmole.b_CH4_NH3P_NH4P_CH3;
			
			mole.c[ipNH3P][ipNH4P] += rate_COmole.CH4_NH3P_NH4P_CH3_1;
			mole.c[ipCH4][ipNH4P] += rate_COmole.CH4_NH3P_NH4P_CH3_2;
			mole.b[ipNH4P] += rate_COmole.b_CH4_NH3P_NH4P_CH3;
			
			mole.c[ipNH3P][ipCH3] += rate_COmole.CH4_NH3P_NH4P_CH3_1;
			mole.c[ipCH4][ipCH3] += rate_COmole.CH4_NH3P_NH4P_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH4_NH3P_NH4P_CH3;

		/*  >>refer   mol  CH4    +  N2+   =>   N2    +     CH2+ + H2    UMIST  */

			mole.c[ipN2P][ipCH4] -= rate_COmole.CH4_N2P_N2_CH2P_H2_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_N2P_N2_CH2P_H2_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_N2P_N2_CH2P_H2;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.CH4_N2P_N2_CH2P_H2_1;
			mole.c[ipCH4][ipN2P] -= rate_COmole.CH4_N2P_N2_CH2P_H2_2;
			mole.b[ipN2P] -= rate_COmole.b_CH4_N2P_N2_CH2P_H2;
			
			mole.c[ipN2P][ipN2] += rate_COmole.CH4_N2P_N2_CH2P_H2_1;
			mole.c[ipCH4][ipN2] += rate_COmole.CH4_N2P_N2_CH2P_H2_2;
			mole.b[ipN2] += rate_COmole.b_CH4_N2P_N2_CH2P_H2;
			
			mole.c[ipN2P][ipCH2P] += rate_COmole.CH4_N2P_N2_CH2P_H2_1;
			mole.c[ipCH4][ipCH2P] += rate_COmole.CH4_N2P_N2_CH2P_H2_2;
			mole.b[ipCH2P] += rate_COmole.b_CH4_N2P_N2_CH2P_H2;

		/*  >>refer   mol  CH4    +  N2+   =>   N2    +     CH3+ + H     UMIST  */

			mole.c[ipN2P][ipCH4] -= rate_COmole.CH4_N2P_N2_CH3P_H_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_N2P_N2_CH3P_H_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_N2P_N2_CH3P_H;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.CH4_N2P_N2_CH3P_H_1;
			mole.c[ipCH4][ipN2P] -= rate_COmole.CH4_N2P_N2_CH3P_H_2;
			mole.b[ipN2P] -= rate_COmole.b_CH4_N2P_N2_CH3P_H;
			
			mole.c[ipN2P][ipN2] += rate_COmole.CH4_N2P_N2_CH3P_H_1;
			mole.c[ipCH4][ipN2] += rate_COmole.CH4_N2P_N2_CH3P_H_2;
			mole.b[ipN2] += rate_COmole.b_CH4_N2P_N2_CH3P_H;
			
			mole.c[ipN2P][ipCH3P] += rate_COmole.CH4_N2P_N2_CH3P_H_1;
			mole.c[ipCH4][ipCH3P] += rate_COmole.CH4_N2P_N2_CH3P_H_2;
			mole.b[ipCH3P] += rate_COmole.b_CH4_N2P_N2_CH3P_H;

		/*  >>refer   mol  CH4    +  HNO+  =>   NO    +     CH5+  UMIST  */

			mole.c[ipHNOP][ipCH4] -= rate_COmole.CH4_HNOP_NO_CH5P_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_HNOP_NO_CH5P_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_HNOP_NO_CH5P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.CH4_HNOP_NO_CH5P_1;
			mole.c[ipCH4][ipHNOP] -= rate_COmole.CH4_HNOP_NO_CH5P_2;
			mole.b[ipHNOP] -= rate_COmole.b_CH4_HNOP_NO_CH5P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.CH4_HNOP_NO_CH5P_1;
			mole.c[ipCH4][ipNO] += rate_COmole.CH4_HNOP_NO_CH5P_2;
			mole.b[ipNO] += rate_COmole.b_CH4_HNOP_NO_CH5P;
			
			mole.c[ipHNOP][ipCH5P] += rate_COmole.CH4_HNOP_NO_CH5P_1;
			mole.c[ipCH4][ipCH5P] += rate_COmole.CH4_HNOP_NO_CH5P_2;
			mole.b[ipCH5P] += rate_COmole.b_CH4_HNOP_NO_CH5P;

		/*  >>refer   mol  CH4    +  S+    =>   HCS+  +     H2   + H     UMIST  */

			mole.c[ipSP][ipCH4] -= rate_COmole.CH4_SP_HCSP_H2_H_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_SP_HCSP_H2_H_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_SP_HCSP_H2_H;
			
			mole.c[ipSP][ipSP] -= rate_COmole.CH4_SP_HCSP_H2_H_1;
			mole.c[ipCH4][ipSP] -= rate_COmole.CH4_SP_HCSP_H2_H_2;
			mole.b[ipSP] -= rate_COmole.b_CH4_SP_HCSP_H2_H;
			
			mole.c[ipSP][ipHCSP] += rate_COmole.CH4_SP_HCSP_H2_H_1;
			mole.c[ipCH4][ipHCSP] += rate_COmole.CH4_SP_HCSP_H2_H_2;
			mole.b[ipHCSP] += rate_COmole.b_CH4_SP_HCSP_H2_H;
			
		/*  >>refer   mol  CH4    +  CS+   =>   HCS+  +     CH3   UMIST  */

			mole.c[ipCSP][ipCH4] -= rate_COmole.CH4_CSP_HCSP_CH3_1;
			mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_CSP_HCSP_CH3_2;
			mole.b[ipCH4] -= rate_COmole.b_CH4_CSP_HCSP_CH3;
			
			mole.c[ipCSP][ipCSP] -= rate_COmole.CH4_CSP_HCSP_CH3_1;
			mole.c[ipCH4][ipCSP] -= rate_COmole.CH4_CSP_HCSP_CH3_2;
			mole.b[ipCSP] -= rate_COmole.b_CH4_CSP_HCSP_CH3;
			
			mole.c[ipCSP][ipHCSP] += rate_COmole.CH4_CSP_HCSP_CH3_1;
			mole.c[ipCH4][ipHCSP] += rate_COmole.CH4_CSP_HCSP_CH3_2;
			mole.b[ipHCSP] += rate_COmole.b_CH4_CSP_HCSP_CH3;
			
			mole.c[ipCSP][ipCH3] += rate_COmole.CH4_CSP_HCSP_CH3_1;
			mole.c[ipCH4][ipCH3] += rate_COmole.CH4_CSP_HCSP_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH4_CSP_HCSP_CH3;

		/*  >>refer   mol  OH+    +  NH3   =>   NH4+  +     O     UMIST  */

			mole.c[ipNH3][ipOHP] -= rate_COmole.OHP_NH3_NH4P_O_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_NH3_NH4P_O_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_NH3_NH4P_O;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.OHP_NH3_NH4P_O_1;
			mole.c[ipOHP][ipNH3] -= rate_COmole.OHP_NH3_NH4P_O_2;
			mole.b[ipNH3] -= rate_COmole.b_OHP_NH3_NH4P_O;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.OHP_NH3_NH4P_O_1;
			mole.c[ipOHP][ipNH4P] += rate_COmole.OHP_NH3_NH4P_O_2;
			mole.b[ipNH4P] += rate_COmole.b_OHP_NH3_NH4P_O;
			
			mole.c[ipNH3][ipATO] += rate_COmole.OHP_NH3_NH4P_O_1;
			mole.c[ipOHP][ipATO] += rate_COmole.OHP_NH3_NH4P_O_2;
			mole.b[ipATO] += rate_COmole.b_OHP_NH3_NH4P_O;
			
		/*  >>refer   mol  OH     +  NH3+  =>   NH4+  +     O     UMIST  */

			mole.c[ipNH3P][ipOH] -= rate_COmole.OH_NH3P_NH4P_O_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_NH3P_NH4P_O_2;
			mole.b[ipOH] -= rate_COmole.b_OH_NH3P_NH4P_O;
			
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.OH_NH3P_NH4P_O_1;
			mole.c[ipOH][ipNH3P] -= rate_COmole.OH_NH3P_NH4P_O_2;
			mole.b[ipNH3P] -= rate_COmole.b_OH_NH3P_NH4P_O;
			
			mole.c[ipNH3P][ipNH4P] += rate_COmole.OH_NH3P_NH4P_O_1;
			mole.c[ipOH][ipNH4P] += rate_COmole.OH_NH3P_NH4P_O_2;
			mole.b[ipNH4P] += rate_COmole.b_OH_NH3P_NH4P_O;
			
			mole.c[ipNH3P][ipATO] += rate_COmole.OH_NH3P_NH4P_O_1;
			mole.c[ipOH][ipATO] += rate_COmole.OH_NH3P_NH4P_O_2;
			mole.b[ipATO] += rate_COmole.b_OH_NH3P_NH4P_O;

		/*  >>refer   mol  OH+    +  CN    =>   HCN+  +     O     UMIST  */

			mole.c[ipCN][ipOHP] -= rate_COmole.OHP_CN_HCNP_O_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_CN_HCNP_O_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_CN_HCNP_O;
			
			mole.c[ipCN][ipCN] -= rate_COmole.OHP_CN_HCNP_O_1;
			mole.c[ipOHP][ipCN] -= rate_COmole.OHP_CN_HCNP_O_2;
			mole.b[ipCN] -= rate_COmole.b_OHP_CN_HCNP_O;
			
			mole.c[ipCN][ipHCNP] += rate_COmole.OHP_CN_HCNP_O_1;
			mole.c[ipOHP][ipHCNP] += rate_COmole.OHP_CN_HCNP_O_2;
			mole.b[ipHCNP] += rate_COmole.b_OHP_CN_HCNP_O;
			
			mole.c[ipCN][ipATO] += rate_COmole.OHP_CN_HCNP_O_1;
			mole.c[ipOHP][ipATO] += rate_COmole.OHP_CN_HCNP_O_2;
			mole.b[ipATO] += rate_COmole.b_OHP_CN_HCNP_O;

		/*  >>refer   mol  OH     +  HCN+  =>   CN    +     H2O+  UMIST  */

			mole.c[ipHCNP][ipOH] -= rate_COmole.OH_HCNP_CN_H2OP_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_HCNP_CN_H2OP_2;
			mole.b[ipOH] -= rate_COmole.b_OH_HCNP_CN_H2OP;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.OH_HCNP_CN_H2OP_1;
			mole.c[ipOH][ipHCNP] -= rate_COmole.OH_HCNP_CN_H2OP_2;
			mole.b[ipHCNP] -= rate_COmole.b_OH_HCNP_CN_H2OP;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.OH_HCNP_CN_H2OP_1;
			mole.c[ipOH][ipCN] += rate_COmole.OH_HCNP_CN_H2OP_2;
			mole.b[ipCN] += rate_COmole.b_OH_HCNP_CN_H2OP;
			
			mole.c[ipHCNP][ipH2OP] += rate_COmole.OH_HCNP_CN_H2OP_1;
			mole.c[ipOH][ipH2OP] += rate_COmole.OH_HCNP_CN_H2OP_2;
			mole.b[ipH2OP] += rate_COmole.b_OH_HCNP_CN_H2OP;

		/*  >>refer   mol  OH+    +  NO    =>   HNO+  +     O     UMIST  */

			mole.c[ipNO][ipOHP] -= rate_COmole.OHP_NO_HNOP_O_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_NO_HNOP_O_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_NO_HNOP_O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.OHP_NO_HNOP_O_1;
			mole.c[ipOHP][ipNO] -= rate_COmole.OHP_NO_HNOP_O_2;
			mole.b[ipNO] -= rate_COmole.b_OHP_NO_HNOP_O;
			
			mole.c[ipNO][ipHNOP] += rate_COmole.OHP_NO_HNOP_O_1;
			mole.c[ipOHP][ipHNOP] += rate_COmole.OHP_NO_HNOP_O_2;
			mole.b[ipHNOP] += rate_COmole.b_OHP_NO_HNOP_O;
			
			mole.c[ipNO][ipATO] += rate_COmole.OHP_NO_HNOP_O_1;
			mole.c[ipOHP][ipATO] += rate_COmole.OHP_NO_HNOP_O_2;
			mole.b[ipATO] += rate_COmole.b_OHP_NO_HNOP_O;
			
		/*  >>refer   mol  OH     +  HNO+  =>   NO    +     H2O+  UMIST  */

			mole.c[ipHNOP][ipOH] -= rate_COmole.OH_HNOP_NO_H2OP_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_HNOP_NO_H2OP_2;
			mole.b[ipOH] -= rate_COmole.b_OH_HNOP_NO_H2OP;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.OH_HNOP_NO_H2OP_1;
			mole.c[ipOH][ipHNOP] -= rate_COmole.OH_HNOP_NO_H2OP_2;
			mole.b[ipHNOP] -= rate_COmole.b_OH_HNOP_NO_H2OP;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.OH_HNOP_NO_H2OP_1;
			mole.c[ipOH][ipNO] += rate_COmole.OH_HNOP_NO_H2OP_2;
			mole.b[ipNO] += rate_COmole.b_OH_HNOP_NO_H2OP;
			
			mole.c[ipHNOP][ipH2OP] += rate_COmole.OH_HNOP_NO_H2OP_1;
			mole.c[ipOH][ipH2OP] += rate_COmole.OH_HNOP_NO_H2OP_2;
			mole.b[ipH2OP] += rate_COmole.b_OH_HNOP_NO_H2OP;

		/*  >>refer   mol  OH+    +  S     =>   HS+   +     O     UMIST  */

			mole.c[ipATS][ipOHP] -= rate_COmole.OHP_S_HSP_O_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_S_HSP_O_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_S_HSP_O;

			mole.c[ipATS][ipATS] -= rate_COmole.OHP_S_HSP_O_1;
			mole.c[ipOHP][ipATS] -= rate_COmole.OHP_S_HSP_O_2;
			mole.b[ipATS] -= rate_COmole.b_OHP_S_HSP_O;
			
			mole.c[ipATS][ipHSP] += rate_COmole.OHP_S_HSP_O_1;
			mole.c[ipOHP][ipHSP] += rate_COmole.OHP_S_HSP_O_2;
			mole.b[ipHSP] += rate_COmole.b_OHP_S_HSP_O;
			
			mole.c[ipATS][ipATO] += rate_COmole.OHP_S_HSP_O_1;
			mole.c[ipOHP][ipATO] += rate_COmole.OHP_S_HSP_O_2;
			mole.b[ipATO] += rate_COmole.b_OHP_S_HSP_O;

		/*  >>refer   mol  OH+    +  S     =>   SO+   +     H     UMIST  */

			mole.c[ipATS][ipOHP] -= rate_COmole.OHP_S_SOP_H_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_S_SOP_H_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_S_SOP_H;

			mole.c[ipATS][ipATS] -= rate_COmole.OHP_S_SOP_H_1;
			mole.c[ipOHP][ipATS] -= rate_COmole.OHP_S_SOP_H_2;
			mole.b[ipATS] -= rate_COmole.b_OHP_S_SOP_H;
			
			mole.c[ipATS][ipSOP] += rate_COmole.OHP_S_SOP_H_1;
			mole.c[ipOHP][ipSOP] += rate_COmole.OHP_S_SOP_H_2;
			mole.b[ipSOP] += rate_COmole.b_OHP_S_SOP_H;

		/*  >>refer   mol  OH     +  S+    =>   SO+   +     H     UMIST  */

			mole.c[ipSP][ipOH] -= rate_COmole.OH_SP_SOP_H_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_SP_SOP_H_2;
			mole.b[ipOH] -= rate_COmole.b_OH_SP_SOP_H;
			
			mole.c[ipSP][ipSP] -= rate_COmole.OH_SP_SOP_H_1;
			mole.c[ipOH][ipSP] -= rate_COmole.OH_SP_SOP_H_2;
			mole.b[ipSP] -= rate_COmole.b_OH_SP_SOP_H;
			
			mole.c[ipSP][ipSOP] += rate_COmole.OH_SP_SOP_H_1;
			mole.c[ipOH][ipSOP] += rate_COmole.OH_SP_SOP_H_2;
			mole.b[ipSOP] += rate_COmole.b_OH_SP_SOP_H;
			
		/*  >>refer   mol  NH3+   +  NH3   =>   NH4+  +     NH2   UMIST  */

			mole.c[ipNH3][ipNH3P] -= rate_COmole.NH3P_NH3_NH4P_NH2_1;
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH3P_NH3_NH4P_NH2_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH3P_NH3_NH4P_NH2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3P_NH3_NH4P_NH2_1;
			mole.c[ipNH3P][ipNH3] -= rate_COmole.NH3P_NH3_NH4P_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3P_NH3_NH4P_NH2;
			
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3P_NH3_NH4P_NH2_1;
			mole.c[ipNH3P][ipNH4P] += rate_COmole.NH3P_NH3_NH4P_NH2_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3P_NH3_NH4P_NH2;
			
			mole.c[ipNH3][ipNH2] += rate_COmole.NH3P_NH3_NH4P_NH2_1;
			mole.c[ipNH3P][ipNH2] += rate_COmole.NH3P_NH3_NH4P_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH3P_NH3_NH4P_NH2;

		/*  >>refer   mol  NH3    +  CH5+  =>   NH4+  +     CH4   UMIST  */

			mole.c[ipCH5P][ipNH3] -= rate_COmole.NH3_CH5P_NH4P_CH4_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_CH5P_NH4P_CH4_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_CH5P_NH4P_CH4;
			
			mole.c[ipCH5P][ipCH5P] -= rate_COmole.NH3_CH5P_NH4P_CH4_1;
			mole.c[ipNH3][ipCH5P] -= rate_COmole.NH3_CH5P_NH4P_CH4_2;
			mole.b[ipCH5P] -= rate_COmole.b_NH3_CH5P_NH4P_CH4;
			
			mole.c[ipCH5P][ipNH4P] += rate_COmole.NH3_CH5P_NH4P_CH4_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_CH5P_NH4P_CH4_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_CH5P_NH4P_CH4;
			
			mole.c[ipCH5P][ipCH4] += rate_COmole.NH3_CH5P_NH4P_CH4_1;
			mole.c[ipNH3][ipCH4] += rate_COmole.NH3_CH5P_NH4P_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_NH3_CH5P_NH4P_CH4;

		/*  >>refer   mol  NH3+   +  H2O   =>   NH4+  +     OH    UMIST  */

			mole.c[ipH2O][ipNH3P] -= rate_COmole.NH3P_H2O_NH4P_OH_1;
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH3P_H2O_NH4P_OH_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH3P_H2O_NH4P_OH;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NH3P_H2O_NH4P_OH_1;
			mole.c[ipNH3P][ipH2O] -= rate_COmole.NH3P_H2O_NH4P_OH_2;
			mole.b[ipH2O] -= rate_COmole.b_NH3P_H2O_NH4P_OH;
			
			mole.c[ipH2O][ipNH4P] += rate_COmole.NH3P_H2O_NH4P_OH_1;
			mole.c[ipNH3P][ipNH4P] += rate_COmole.NH3P_H2O_NH4P_OH_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3P_H2O_NH4P_OH;
			
			mole.c[ipH2O][ipOH] += rate_COmole.NH3P_H2O_NH4P_OH_1;
			mole.c[ipNH3P][ipOH] += rate_COmole.NH3P_H2O_NH4P_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH3P_H2O_NH4P_OH;

		/*  >>refer   mol  NH3    +  H2O+  =>   NH4+  +     OH    UMIST  */

			mole.c[ipH2OP][ipNH3] -= rate_COmole.NH3_H2OP_NH4P_OH_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_H2OP_NH4P_OH_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_H2OP_NH4P_OH;
			
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.NH3_H2OP_NH4P_OH_1;
			mole.c[ipNH3][ipH2OP] -= rate_COmole.NH3_H2OP_NH4P_OH_2;
			mole.b[ipH2OP] -= rate_COmole.b_NH3_H2OP_NH4P_OH;
			
			mole.c[ipH2OP][ipNH4P] += rate_COmole.NH3_H2OP_NH4P_OH_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_H2OP_NH4P_OH_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_H2OP_NH4P_OH;
			
			mole.c[ipH2OP][ipOH] += rate_COmole.NH3_H2OP_NH4P_OH_1;
			mole.c[ipNH3][ipOH] += rate_COmole.NH3_H2OP_NH4P_OH_2;
			mole.b[ipOH] += rate_COmole.b_NH3_H2OP_NH4P_OH;

		/*  >>refer   mol  NH3    +  H3O+  =>   NH4+  +     H2O   UMIST  */

			mole.c[ipH3OP][ipNH3] -= rate_COmole.NH3_H3OP_NH4P_H2O_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_H3OP_NH4P_H2O_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_H3OP_NH4P_H2O;
			
			mole.c[ipH3OP][ipH3OP] -= rate_COmole.NH3_H3OP_NH4P_H2O_1;
			mole.c[ipNH3][ipH3OP] -= rate_COmole.NH3_H3OP_NH4P_H2O_2;
			mole.b[ipH3OP] -= rate_COmole.b_NH3_H3OP_NH4P_H2O;
			
			mole.c[ipH3OP][ipNH4P] += rate_COmole.NH3_H3OP_NH4P_H2O_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_H3OP_NH4P_H2O_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_H3OP_NH4P_H2O;
			
			mole.c[ipH3OP][ipH2O] += rate_COmole.NH3_H3OP_NH4P_H2O_1;
			mole.c[ipNH3][ipH2O] += rate_COmole.NH3_H3OP_NH4P_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_NH3_H3OP_NH4P_H2O;

		/*  >>refer   mol  NH3    +  CO+   =>   HCO+  +     NH2   UMIST  */

			mole.c[ipCOP][ipNH3] -= rate_COmole.NH3_COP_HCOP_NH2_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_COP_HCOP_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_COP_HCOP_NH2;
			
			mole.c[ipCOP][ipCOP] -= rate_COmole.NH3_COP_HCOP_NH2_1;
			mole.c[ipNH3][ipCOP] -= rate_COmole.NH3_COP_HCOP_NH2_2;
			mole.b[ipCOP] -= rate_COmole.b_NH3_COP_HCOP_NH2;
			
			mole.c[ipCOP][ipHCOP] += rate_COmole.NH3_COP_HCOP_NH2_1;
			mole.c[ipNH3][ipHCOP] += rate_COmole.NH3_COP_HCOP_NH2_2;
			mole.b[ipHCOP] += rate_COmole.b_NH3_COP_HCOP_NH2;
			
			mole.c[ipCOP][ipNH2] += rate_COmole.NH3_COP_HCOP_NH2_1;
			mole.c[ipNH3][ipNH2] += rate_COmole.NH3_COP_HCOP_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH3_COP_HCOP_NH2;

		/*  >>refer   mol  NH3    +  HNO+  =>   NO    +     NH4+  UMIST  */

			mole.c[ipHNOP][ipNH3] -= rate_COmole.NH3_HNOP_NO_NH4P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HNOP_NO_NH4P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_HNOP_NO_NH4P;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.NH3_HNOP_NO_NH4P_1;
			mole.c[ipNH3][ipHNOP] -= rate_COmole.NH3_HNOP_NO_NH4P_2;
			mole.b[ipHNOP] -= rate_COmole.b_NH3_HNOP_NO_NH4P;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.NH3_HNOP_NO_NH4P_1;
			mole.c[ipNH3][ipNO] += rate_COmole.NH3_HNOP_NO_NH4P_2;
			mole.b[ipNO] += rate_COmole.b_NH3_HNOP_NO_NH4P;
			
			mole.c[ipHNOP][ipNH4P] += rate_COmole.NH3_HNOP_NO_NH4P_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_HNOP_NO_NH4P_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_HNOP_NO_NH4P;

		/*  >>refer   mol  NH3    +  HS+   =>   S     +     NH4+  UMIST  */

			mole.c[ipHSP][ipNH3] -= rate_COmole.NH3_HSP_S_NH4P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HSP_S_NH4P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_HSP_S_NH4P;
			
			mole.c[ipHSP][ipHSP] -= rate_COmole.NH3_HSP_S_NH4P_1;
			mole.c[ipNH3][ipHSP] -= rate_COmole.NH3_HSP_S_NH4P_2;
			mole.b[ipHSP] -= rate_COmole.b_NH3_HSP_S_NH4P;
			
			mole.c[ipHSP][ipNH4P] += rate_COmole.NH3_HSP_S_NH4P_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_HSP_S_NH4P_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_HSP_S_NH4P;
			
			mole.c[ipHSP][ipATS] += rate_COmole.NH3_HSP_S_NH4P_1;
			mole.c[ipNH3][ipATS] += rate_COmole.NH3_HSP_S_NH4P_2;
			mole.b[ipATS] += rate_COmole.b_NH3_HSP_S_NH4P;

		/*  >>refer   mol  NH3    +  HCS+  =>   CS    +     NH4+  UMIST  */

			mole.c[ipHCSP][ipNH3] -= rate_COmole.NH3_HCSP_CS_NH4P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HCSP_CS_NH4P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_HCSP_CS_NH4P;
			
			mole.c[ipHCSP][ipHCSP] -= rate_COmole.NH3_HCSP_CS_NH4P_1;
			mole.c[ipNH3][ipHCSP] -= rate_COmole.NH3_HCSP_CS_NH4P_2;
			mole.b[ipHCSP] -= rate_COmole.b_NH3_HCSP_CS_NH4P;
			
			mole.c[ipHCSP][ipCS] += rate_COmole.NH3_HCSP_CS_NH4P_1;
			mole.c[ipNH3][ipCS] += rate_COmole.NH3_HCSP_CS_NH4P_2;
			mole.b[ipCS] += rate_COmole.b_NH3_HCSP_CS_NH4P;
			
			mole.c[ipHCSP][ipNH4P] += rate_COmole.NH3_HCSP_CS_NH4P_1;
			mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_HCSP_CS_NH4P_2;
			mole.b[ipNH4P] += rate_COmole.b_NH3_HCSP_CS_NH4P;

		/*  >>refer   mol  CH5+   +  S     =>   HS+   +     CH4   UMIST  */

			mole.c[ipATS][ipCH5P] -= rate_COmole.CH5P_S_HSP_CH4_1;
			mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_S_HSP_CH4_2;
			mole.b[ipCH5P] -= rate_COmole.b_CH5P_S_HSP_CH4;

			mole.c[ipATS][ipATS] -= rate_COmole.CH5P_S_HSP_CH4_1;
			mole.c[ipCH5P][ipATS] -= rate_COmole.CH5P_S_HSP_CH4_2;
			mole.b[ipATS] -= rate_COmole.b_CH5P_S_HSP_CH4;
			
			mole.c[ipATS][ipHSP] += rate_COmole.CH5P_S_HSP_CH4_1;
			mole.c[ipCH5P][ipHSP] += rate_COmole.CH5P_S_HSP_CH4_2;
			mole.b[ipHSP] += rate_COmole.b_CH5P_S_HSP_CH4;
			
			mole.c[ipATS][ipCH4] += rate_COmole.CH5P_S_HSP_CH4_1;
			mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_S_HSP_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_CH5P_S_HSP_CH4;

		/*  >>refer   mol  H2O    +  CN+   =>   HCN+  +     OH    UMIST  */

			mole.c[ipCNP][ipH2O] -= rate_COmole.H2O_CNP_HCNP_OH_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_CNP_HCNP_OH_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_CNP_HCNP_OH;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.H2O_CNP_HCNP_OH_1;
			mole.c[ipH2O][ipCNP] -= rate_COmole.H2O_CNP_HCNP_OH_2;
			mole.b[ipCNP] -= rate_COmole.b_H2O_CNP_HCNP_OH;
			
			mole.c[ipCNP][ipHCNP] += rate_COmole.H2O_CNP_HCNP_OH_1;
			mole.c[ipH2O][ipHCNP] += rate_COmole.H2O_CNP_HCNP_OH_2;
			mole.b[ipHCNP] += rate_COmole.b_H2O_CNP_HCNP_OH;
			
			mole.c[ipCNP][ipOH] += rate_COmole.H2O_CNP_HCNP_OH_1;
			mole.c[ipH2O][ipOH] += rate_COmole.H2O_CNP_HCNP_OH_2;
			mole.b[ipOH] += rate_COmole.b_H2O_CNP_HCNP_OH;

		/*  >>refer   mol  H2O    +  CN+   =>   HCO+  +     NH    UMIST  */

			mole.c[ipCNP][ipH2O] -= rate_COmole.H2O_CNP_HCOP_NH_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_CNP_HCOP_NH_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_CNP_HCOP_NH;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.H2O_CNP_HCOP_NH_1;
			mole.c[ipH2O][ipCNP] -= rate_COmole.H2O_CNP_HCOP_NH_2;
			mole.b[ipCNP] -= rate_COmole.b_H2O_CNP_HCOP_NH;
			
			mole.c[ipCNP][ipHCOP] += rate_COmole.H2O_CNP_HCOP_NH_1;
			mole.c[ipH2O][ipHCOP] += rate_COmole.H2O_CNP_HCOP_NH_2;
			mole.b[ipHCOP] += rate_COmole.b_H2O_CNP_HCOP_NH;
			
			mole.c[ipCNP][ipNH] += rate_COmole.H2O_CNP_HCOP_NH_1;
			mole.c[ipH2O][ipNH] += rate_COmole.H2O_CNP_HCOP_NH_2;
			mole.b[ipNH] += rate_COmole.b_H2O_CNP_HCOP_NH;

		/*  >>refer   mol  H2O    +  HCN+  =>   CN    +     H3O+  UMIST  */

			mole.c[ipHCNP][ipH2O] -= rate_COmole.H2O_HCNP_CN_H3OP_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_HCNP_CN_H3OP_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_HCNP_CN_H3OP;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.H2O_HCNP_CN_H3OP_1;
			mole.c[ipH2O][ipHCNP] -= rate_COmole.H2O_HCNP_CN_H3OP_2;
			mole.b[ipHCNP] -= rate_COmole.b_H2O_HCNP_CN_H3OP;
			
			mole.c[ipHCNP][ipCN] += rate_COmole.H2O_HCNP_CN_H3OP_1;
			mole.c[ipH2O][ipCN] += rate_COmole.H2O_HCNP_CN_H3OP_2;
			mole.b[ipCN] += rate_COmole.b_H2O_HCNP_CN_H3OP;
			
			mole.c[ipHCNP][ipH3OP] += rate_COmole.H2O_HCNP_CN_H3OP_1;
			mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_HCNP_CN_H3OP_2;
			mole.b[ipH3OP] += rate_COmole.b_H2O_HCNP_CN_H3OP;

		/*  >>refer   mol  H2O    +  HNO+  =>   NO    +     H3O+  UMIST  */

			mole.c[ipHNOP][ipH2O] -= rate_COmole.H2O_HNOP_NO_H3OP_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_HNOP_NO_H3OP_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_HNOP_NO_H3OP;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.H2O_HNOP_NO_H3OP_1;
			mole.c[ipH2O][ipHNOP] -= rate_COmole.H2O_HNOP_NO_H3OP_2;
			mole.b[ipHNOP] -= rate_COmole.b_H2O_HNOP_NO_H3OP;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.H2O_HNOP_NO_H3OP_1;
			mole.c[ipH2O][ipNO] += rate_COmole.H2O_HNOP_NO_H3OP_2;
			mole.b[ipNO] += rate_COmole.b_H2O_HNOP_NO_H3OP;
			
			mole.c[ipHNOP][ipH3OP] += rate_COmole.H2O_HNOP_NO_H3OP_1;
			mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_HNOP_NO_H3OP_2;
			mole.b[ipH3OP] += rate_COmole.b_H2O_HNOP_NO_H3OP;

		/*  >>refer   mol  H2O+   +  S     =>   HS+   +     OH    UMIST  */

			mole.c[ipATS][ipH2OP] -= rate_COmole.H2OP_S_HSP_OH_1;
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_S_HSP_OH_2;
			mole.b[ipH2OP] -= rate_COmole.b_H2OP_S_HSP_OH;

			mole.c[ipATS][ipATS] -= rate_COmole.H2OP_S_HSP_OH_1;
			mole.c[ipH2OP][ipATS] -= rate_COmole.H2OP_S_HSP_OH_2;
			mole.b[ipATS] -= rate_COmole.b_H2OP_S_HSP_OH;
			
			mole.c[ipATS][ipHSP] += rate_COmole.H2OP_S_HSP_OH_1;
			mole.c[ipH2OP][ipHSP] += rate_COmole.H2OP_S_HSP_OH_2;
			mole.b[ipHSP] += rate_COmole.b_H2OP_S_HSP_OH;
			
			mole.c[ipATS][ipOH] += rate_COmole.H2OP_S_HSP_OH_1;
			mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_S_HSP_OH_2;
			mole.b[ipOH] += rate_COmole.b_H2OP_S_HSP_OH;

		/*  >>refer   mol  H2O    +  HS+   =>   S     +     H3O+  UMIST  */

			mole.c[ipHSP][ipH2O] -= rate_COmole.H2O_HSP_S_H3OP_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_HSP_S_H3OP_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_HSP_S_H3OP;
			
			mole.c[ipHSP][ipHSP] -= rate_COmole.H2O_HSP_S_H3OP_1;
			mole.c[ipH2O][ipHSP] -= rate_COmole.H2O_HSP_S_H3OP_2;
			mole.b[ipHSP] -= rate_COmole.b_H2O_HSP_S_H3OP;
			
			mole.c[ipHSP][ipH3OP] += rate_COmole.H2O_HSP_S_H3OP_1;
			mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_HSP_S_H3OP_2;
			mole.b[ipH3OP] += rate_COmole.b_H2O_HSP_S_H3OP;
			
			mole.c[ipHSP][ipATS] += rate_COmole.H2O_HSP_S_H3OP_1;
			mole.c[ipH2O][ipATS] += rate_COmole.H2O_HSP_S_H3OP_2;
			mole.b[ipATS] += rate_COmole.b_H2O_HSP_S_H3OP;
			
		/*  >>refer   mol  H3O+   +  CS    =>   HCS+  +     H2O   UMIST  */

			mole.c[ipCS][ipH3OP] -= rate_COmole.H3OP_CS_HCSP_H2O_1;
			mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_CS_HCSP_H2O_2;
			mole.b[ipH3OP] -= rate_COmole.b_H3OP_CS_HCSP_H2O;
			
			mole.c[ipCS][ipCS] -= rate_COmole.H3OP_CS_HCSP_H2O_1;
			mole.c[ipH3OP][ipCS] -= rate_COmole.H3OP_CS_HCSP_H2O_2;
			mole.b[ipCS] -= rate_COmole.b_H3OP_CS_HCSP_H2O;
			
			mole.c[ipCS][ipHCSP] += rate_COmole.H3OP_CS_HCSP_H2O_1;
			mole.c[ipH3OP][ipHCSP] += rate_COmole.H3OP_CS_HCSP_H2O_2;
			mole.b[ipHCSP] += rate_COmole.b_H3OP_CS_HCSP_H2O;
			
			mole.c[ipCS][ipH2O] += rate_COmole.H3OP_CS_HCSP_H2O_1;
			mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_CS_HCSP_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_H3OP_CS_HCSP_H2O;

		/*  >>refer   mol  CN+    +  NO    =>   OCN+  +     N     UMIST  */

			mole.c[ipNO][ipCNP] -= rate_COmole.CNP_NO_OCNP_N_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_NO_OCNP_N_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_NO_OCNP_N;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CNP_NO_OCNP_N_1;
			mole.c[ipCNP][ipNO] -= rate_COmole.CNP_NO_OCNP_N_2;
			mole.b[ipNO] -= rate_COmole.b_CNP_NO_OCNP_N;
			
			mole.c[ipNO][ipOCNP] += rate_COmole.CNP_NO_OCNP_N_1;
			mole.c[ipCNP][ipOCNP] += rate_COmole.CNP_NO_OCNP_N_2;
			mole.b[ipOCNP] += rate_COmole.b_CNP_NO_OCNP_N;
			
			mole.c[ipNO][ipATN] += rate_COmole.CNP_NO_OCNP_N_1;
			mole.c[ipCNP][ipATN] += rate_COmole.CNP_NO_OCNP_N_2;
			mole.b[ipATN] += rate_COmole.b_CNP_NO_OCNP_N;

		/*  >>refer   mol  CN     +  HNO+  =>   NO    +     HCN+  UMIST  */

			mole.c[ipHNOP][ipCN] -= rate_COmole.CN_HNOP_NO_HCNP_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_HNOP_NO_HCNP_2;
			mole.b[ipCN] -= rate_COmole.b_CN_HNOP_NO_HCNP;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.CN_HNOP_NO_HCNP_1;
			mole.c[ipCN][ipHNOP] -= rate_COmole.CN_HNOP_NO_HCNP_2;
			mole.b[ipHNOP] -= rate_COmole.b_CN_HNOP_NO_HCNP;
			
			mole.c[ipHNOP][ipNO] += rate_COmole.CN_HNOP_NO_HCNP_1;
			mole.c[ipCN][ipNO] += rate_COmole.CN_HNOP_NO_HCNP_2;
			mole.b[ipNO] += rate_COmole.b_CN_HNOP_NO_HCNP;
			
			mole.c[ipHNOP][ipHCNP] += rate_COmole.CN_HNOP_NO_HCNP_1;
			mole.c[ipCN][ipHCNP] += rate_COmole.CN_HNOP_NO_HCNP_2;
			mole.b[ipHCNP] += rate_COmole.b_CN_HNOP_NO_HCNP;

		/*  >>refer   mol  CN+    +  O2    =>   OCN+  +     O     UMIST  */

			mole.c[ipO2][ipCNP] -= rate_COmole.CNP_O2_OCNP_O_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_O2_OCNP_O_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_O2_OCNP_O;
			
			mole.c[ipO2][ipO2] -= rate_COmole.CNP_O2_OCNP_O_1;
			mole.c[ipCNP][ipO2] -= rate_COmole.CNP_O2_OCNP_O_2;
			mole.b[ipO2] -= rate_COmole.b_CNP_O2_OCNP_O;
			
			mole.c[ipO2][ipOCNP] += rate_COmole.CNP_O2_OCNP_O_1;
			mole.c[ipCNP][ipOCNP] += rate_COmole.CNP_O2_OCNP_O_2;
			mole.b[ipOCNP] += rate_COmole.b_CNP_O2_OCNP_O;
			
			mole.c[ipO2][ipATO] += rate_COmole.CNP_O2_OCNP_O_1;
			mole.c[ipCNP][ipATO] += rate_COmole.CNP_O2_OCNP_O_2;
			mole.b[ipATO] += rate_COmole.b_CNP_O2_OCNP_O;
			
		/*  >>refer   mol  HCN+   +  S     =>   HS+   +     CN    UMIST  */

			mole.c[ipATS][ipHCNP] -= rate_COmole.HCNP_S_HSP_CN_1;
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_S_HSP_CN_2;
			mole.b[ipHCNP] -= rate_COmole.b_HCNP_S_HSP_CN;

			mole.c[ipATS][ipATS] -= rate_COmole.HCNP_S_HSP_CN_1;
			mole.c[ipHCNP][ipATS] -= rate_COmole.HCNP_S_HSP_CN_2;
			mole.b[ipATS] -= rate_COmole.b_HCNP_S_HSP_CN;
			
			mole.c[ipATS][ipHSP] += rate_COmole.HCNP_S_HSP_CN_1;
			mole.c[ipHCNP][ipHSP] += rate_COmole.HCNP_S_HSP_CN_2;
			mole.b[ipHSP] += rate_COmole.b_HCNP_S_HSP_CN;
			
			mole.c[ipATS][ipCN] += rate_COmole.HCNP_S_HSP_CN_1;
			mole.c[ipHCNP][ipCN] += rate_COmole.HCNP_S_HSP_CN_2;
			mole.b[ipCN] += rate_COmole.b_HCNP_S_HSP_CN;

		/*  >>refer   mol  HNO+   +  S     =>   HS+   +     NO    UMIST  */
			
			mole.c[ipATS][ipHNOP] -= rate_COmole.HNOP_S_HSP_NO_1;
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.HNOP_S_HSP_NO_2;
			mole.b[ipHNOP] -= rate_COmole.b_HNOP_S_HSP_NO;
			
			mole.c[ipATS][ipATS] -= rate_COmole.HNOP_S_HSP_NO_1;
			mole.c[ipHNOP][ipATS] -= rate_COmole.HNOP_S_HSP_NO_2;
			mole.b[ipATS] -= rate_COmole.b_HNOP_S_HSP_NO;
			
			mole.c[ipATS][ipHSP] += rate_COmole.HNOP_S_HSP_NO_1;
			mole.c[ipHNOP][ipHSP] += rate_COmole.HNOP_S_HSP_NO_2;
			mole.b[ipHSP] += rate_COmole.b_HNOP_S_HSP_NO;
			
			mole.c[ipATS][ipNO] += rate_COmole.HNOP_S_HSP_NO_1;
			mole.c[ipHNOP][ipNO] += rate_COmole.HNOP_S_HSP_NO_2;
			mole.b[ipNO] += rate_COmole.b_HNOP_S_HSP_NO;

		/*  >>refer   mol  O2     +  S+    =>   SO+   +     O     UMIST  */

			mole.c[ipSP][ipO2] -= rate_COmole.O2_SP_SOP_O_1;
			mole.c[ipO2][ipO2] -= rate_COmole.O2_SP_SOP_O_2;
			mole.b[ipO2] -= rate_COmole.b_O2_SP_SOP_O;
			
			mole.c[ipSP][ipSP] -= rate_COmole.O2_SP_SOP_O_1;
			mole.c[ipO2][ipSP] -= rate_COmole.O2_SP_SOP_O_2;
			mole.b[ipSP] -= rate_COmole.b_O2_SP_SOP_O;
			
			mole.c[ipSP][ipSOP] += rate_COmole.O2_SP_SOP_O_1;
			mole.c[ipO2][ipSOP] += rate_COmole.O2_SP_SOP_O_2;
			mole.b[ipSOP] += rate_COmole.b_O2_SP_SOP_O;
			
			mole.c[ipSP][ipATO] += rate_COmole.O2_SP_SOP_O_1;
			mole.c[ipO2][ipATO] += rate_COmole.O2_SP_SOP_O_2;
			mole.b[ipATO] += rate_COmole.b_O2_SP_SOP_O;

		/*  >>refer   mol  O2+    +  S     =>   SO+   +     O     UMIST  */

			mole.c[ipATS][ipO2P] -= rate_COmole.O2P_S_SOP_O_1;
			mole.c[ipO2P][ipO2P] -= rate_COmole.O2P_S_SOP_O_2;
			mole.b[ipO2P] -= rate_COmole.b_O2P_S_SOP_O;
			
			mole.c[ipATS][ipATS] -= rate_COmole.O2P_S_SOP_O_1;
			mole.c[ipO2P][ipATS] -= rate_COmole.O2P_S_SOP_O_2;
			mole.b[ipATS] -= rate_COmole.b_O2P_S_SOP_O;
			
			mole.c[ipATS][ipSOP] += rate_COmole.O2P_S_SOP_O_1;
			mole.c[ipO2P][ipSOP] += rate_COmole.O2P_S_SOP_O_2;
			mole.b[ipSOP] += rate_COmole.b_O2P_S_SOP_O;

			mole.c[ipATS][ipATO] += rate_COmole.O2P_S_SOP_O_1;
			mole.c[ipO2P][ipATO] += rate_COmole.O2P_S_SOP_O_2;
			mole.b[ipATO] += rate_COmole.b_O2P_S_SOP_O;

		/*  >>refer   mol  O2     +  CS+   =>   OCS+  +     O     UMIST  */

			mole.c[ipCSP][ipO2] -= rate_COmole.O2_CSP_OCSP_O_1;
			mole.c[ipO2][ipO2] -= rate_COmole.O2_CSP_OCSP_O_2;
			mole.b[ipO2] -= rate_COmole.b_O2_CSP_OCSP_O;
			
			mole.c[ipCSP][ipCSP] -= rate_COmole.O2_CSP_OCSP_O_1;
			mole.c[ipO2][ipCSP] -= rate_COmole.O2_CSP_OCSP_O_2;
			mole.b[ipCSP] -= rate_COmole.b_O2_CSP_OCSP_O;
			
			mole.c[ipCSP][ipOCSP] += rate_COmole.O2_CSP_OCSP_O_1;
			mole.c[ipO2][ipOCSP] += rate_COmole.O2_CSP_OCSP_O_2;
			mole.b[ipOCSP] += rate_COmole.b_O2_CSP_OCSP_O;
			
			mole.c[ipCSP][ipATO] += rate_COmole.O2_CSP_OCSP_O_1;
			mole.c[ipO2][ipATO] += rate_COmole.O2_CSP_OCSP_O_2;
			mole.b[ipATO] += rate_COmole.b_O2_CSP_OCSP_O;
			
		/*  >>refer   mol  S      +  SIO+  =>   SO    +     SI+   UMIST  */
			
			mole.c[ipSIOP][ipSIOP] -= rate_COmole.S_SIOP_SO_SIP_1;
			mole.c[ipATS][ipSIOP] -= rate_COmole.S_SIOP_SO_SIP_2;
			mole.b[ipSIOP] -= rate_COmole.b_S_SIOP_SO_SIP;
			
			mole.c[ipSIOP][ipATS] -= rate_COmole.S_SIOP_SO_SIP_1;
			mole.c[ipATS][ipATS] -= rate_COmole.S_SIOP_SO_SIP_2;
			mole.b[ipATS] -= rate_COmole.b_S_SIOP_SO_SIP;
			
			mole.c[ipSIOP][ipSO] += rate_COmole.S_SIOP_SO_SIP_1;
			mole.c[ipATS][ipSO] += rate_COmole.S_SIOP_SO_SIP_2;
			mole.b[ipSO] += rate_COmole.b_S_SIOP_SO_SIP;
			
			mole.c[ipSIOP][ipSIP] += rate_COmole.S_SIOP_SO_SIP_1;
			mole.c[ipATS][ipSIP] += rate_COmole.S_SIOP_SO_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_S_SIOP_SO_SIP;


		/*  >>refer   mol  C+     +  NH3   =>   NH3+  +     C     UMIST  */

			mole.c[ipNH3][ipCP] -= rate_COmole.CP_NH3_NH3P_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NH3_NH3P_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NH3_NH3P_C;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CP_NH3_NH3P_C_1;
			mole.c[ipCP][ipNH3] -= rate_COmole.CP_NH3_NH3P_C_2;
			mole.b[ipNH3] -= rate_COmole.b_CP_NH3_NH3P_C;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.CP_NH3_NH3P_C_1;
			mole.c[ipCP][ipNH3P] += rate_COmole.CP_NH3_NH3P_C_2;
			mole.b[ipNH3P] += rate_COmole.b_CP_NH3_NH3P_C;
			
			mole.c[ipNH3][ipATC] += rate_COmole.CP_NH3_NH3P_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_NH3_NH3P_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_NH3_NH3P_C;

		/*  >>refer   mol  C      +  CN+   =>   CN    +     C+    UMIST  */
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.C_CNP_CN_CP_1;
			mole.c[ipATC][ipCNP] -= rate_COmole.C_CNP_CN_CP_2;
			mole.b[ipCNP] -= rate_COmole.b_C_CNP_CN_CP;
			
			mole.c[ipCNP][ipATC] -= rate_COmole.C_CNP_CN_CP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_CNP_CN_CP_2;
			mole.b[ipATC] -= rate_COmole.b_C_CNP_CN_CP;
			
			mole.c[ipCNP][ipCN] += rate_COmole.C_CNP_CN_CP_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_CNP_CN_CP_2;
			mole.b[ipCN] += rate_COmole.b_C_CNP_CN_CP;
			
			mole.c[ipCNP][ipCP] += rate_COmole.C_CNP_CN_CP_1;
			mole.c[ipATC][ipCP] += rate_COmole.C_CNP_CN_CP_2;
			mole.b[ipCP] += rate_COmole.b_C_CNP_CN_CP;

		/*  >>refer   mol  C      +  N2+   =>   N2    +     C+    UMIST  */

			mole.c[ipN2P][ipN2P] -= rate_COmole.C_N2P_N2_CP_1;
			mole.c[ipATC][ipN2P] -= rate_COmole.C_N2P_N2_CP_2;
			mole.b[ipN2P] -= rate_COmole.b_C_N2P_N2_CP;

			mole.c[ipN2P][ipATC] -= rate_COmole.C_N2P_N2_CP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_N2P_N2_CP_2;
			mole.b[ipATC] -= rate_COmole.b_C_N2P_N2_CP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.C_N2P_N2_CP_1;
			mole.c[ipATC][ipN2] += rate_COmole.C_N2P_N2_CP_2;
			mole.b[ipN2] += rate_COmole.b_C_N2P_N2_CP;
			
			mole.c[ipN2P][ipCP] += rate_COmole.C_N2P_N2_CP_1;
			mole.c[ipATC][ipCP] += rate_COmole.C_N2P_N2_CP_2;
			mole.b[ipCP] += rate_COmole.b_C_N2P_N2_CP;

		/*  >>refer   mol  C+     +  NO    =>   NO+   +     C     UMIST  */

			mole.c[ipNO][ipCP] -= rate_COmole.CP_NO_NOP_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NO_NOP_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NO_NOP_C;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CP_NO_NOP_C_1;
			mole.c[ipCP][ipNO] -= rate_COmole.CP_NO_NOP_C_2;
			mole.b[ipNO] -= rate_COmole.b_CP_NO_NOP_C;
			
			mole.c[ipNO][ipNOP] += rate_COmole.CP_NO_NOP_C_1;
			mole.c[ipCP][ipNOP] += rate_COmole.CP_NO_NOP_C_2;
			mole.b[ipNOP] += rate_COmole.b_CP_NO_NOP_C;
			
			mole.c[ipNO][ipATC] += rate_COmole.CP_NO_NOP_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_NO_NOP_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_NO_NOP_C;
			
		/*  >>refer   mol  C+     +  SIN   =>   SIN+  +     C     UMIST  */

			mole.c[ipSIN][ipCP] -= rate_COmole.CP_SIN_SINP_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_SIN_SINP_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_SIN_SINP_C;
			
			mole.c[ipSIN][ipSIN] -= rate_COmole.CP_SIN_SINP_C_1;
			mole.c[ipCP][ipSIN] -= rate_COmole.CP_SIN_SINP_C_2;
			mole.b[ipSIN] -= rate_COmole.b_CP_SIN_SINP_C;
			
			mole.c[ipSIN][ipSINP] += rate_COmole.CP_SIN_SINP_C_1;
			mole.c[ipCP][ipSINP] += rate_COmole.CP_SIN_SINP_C_2;
			mole.b[ipSINP] += rate_COmole.b_CP_SIN_SINP_C;
			
			mole.c[ipSIN][ipATC] += rate_COmole.CP_SIN_SINP_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_SIN_SINP_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_SIN_SINP_C;
			
		/*  >>refer   mol  C      +  CS+   =>   CS    +     C+    UMIST  */

			mole.c[ipCSP][ipCSP] -= rate_COmole.C_CSP_CS_CP_1;
			mole.c[ipATC][ipCSP] -= rate_COmole.C_CSP_CS_CP_2;
			mole.b[ipCSP] -= rate_COmole.b_C_CSP_CS_CP;

			mole.c[ipCSP][ipATC] -= rate_COmole.C_CSP_CS_CP_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_CSP_CS_CP_2;
			mole.b[ipATC] -= rate_COmole.b_C_CSP_CS_CP;
			
			mole.c[ipCSP][ipCS] += rate_COmole.C_CSP_CS_CP_1;
			mole.c[ipATC][ipCS] += rate_COmole.C_CSP_CS_CP_2;
			mole.b[ipCS] += rate_COmole.b_C_CSP_CS_CP;
			
			mole.c[ipCSP][ipCP] += rate_COmole.C_CSP_CS_CP_1;
			mole.c[ipATC][ipCP] += rate_COmole.C_CSP_CS_CP_2;
			mole.b[ipCP] += rate_COmole.b_C_CSP_CS_CP;

		/*  >>refer   mol  C+     +  NS    =>   NS+   +     C     UMIST  */

			mole.c[ipNS][ipCP] -= rate_COmole.CP_NS_NSP_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_NS_NSP_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_NS_NSP_C;
			
			mole.c[ipNS][ipNS] -= rate_COmole.CP_NS_NSP_C_1;
			mole.c[ipCP][ipNS] -= rate_COmole.CP_NS_NSP_C_2;
			mole.b[ipNS] -= rate_COmole.b_CP_NS_NSP_C;
			
			mole.c[ipNS][ipNSP] += rate_COmole.CP_NS_NSP_C_1;
			mole.c[ipCP][ipNSP] += rate_COmole.CP_NS_NSP_C_2;
			mole.b[ipNSP] += rate_COmole.b_CP_NS_NSP_C;
			
			mole.c[ipNS][ipATC] += rate_COmole.CP_NS_NSP_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_NS_NSP_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_NS_NSP_C;
			
		/*  >>refer   mol  C+     +  SO    =>   SO+   +     C     UMIST  */

			mole.c[ipSO][ipCP] -= rate_COmole.CP_SO_SOP_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_SO_SOP_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_SO_SOP_C;
			
			mole.c[ipSO][ipSO] -= rate_COmole.CP_SO_SOP_C_1;
			mole.c[ipCP][ipSO] -= rate_COmole.CP_SO_SOP_C_2;
			mole.b[ipSO] -= rate_COmole.b_CP_SO_SOP_C;
			
			mole.c[ipSO][ipSOP] += rate_COmole.CP_SO_SOP_C_1;
			mole.c[ipCP][ipSOP] += rate_COmole.CP_SO_SOP_C_2;
			mole.b[ipSOP] += rate_COmole.b_CP_SO_SOP_C;
			
			mole.c[ipSO][ipATC] += rate_COmole.CP_SO_SOP_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_SO_SOP_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_SO_SOP_C;
			
		/*  >>refer   mol  C+     +  OCS   =>   OCS+  +     C     UMIST  */

			mole.c[ipOCS][ipCP] -= rate_COmole.CP_OCS_OCSP_C_1;
			mole.c[ipCP][ipCP] -= rate_COmole.CP_OCS_OCSP_C_2;
			mole.b[ipCP] -= rate_COmole.b_CP_OCS_OCSP_C;
			
			mole.c[ipOCS][ipOCS] -= rate_COmole.CP_OCS_OCSP_C_1;
			mole.c[ipCP][ipOCS] -= rate_COmole.CP_OCS_OCSP_C_2;
			mole.b[ipOCS] -= rate_COmole.b_CP_OCS_OCSP_C;
			
			mole.c[ipOCS][ipOCSP] += rate_COmole.CP_OCS_OCSP_C_1;
			mole.c[ipCP][ipOCSP] += rate_COmole.CP_OCS_OCSP_C_2;
			mole.b[ipOCSP] += rate_COmole.b_CP_OCS_OCSP_C;
				
			mole.c[ipOCS][ipATC] += rate_COmole.CP_OCS_OCSP_C_1;
			mole.c[ipCP][ipATC] += rate_COmole.CP_OCS_OCSP_C_2;
			mole.b[ipATC] += rate_COmole.b_CP_OCS_OCSP_C;
			
		/*  >>refer   mol  CH     +  NH2+  =>   NH2   +     CH+   UMIST  */

			mole.c[ipNH2P][ipCH] -= rate_COmole.CH_NH2P_NH2_CHP_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NH2P_NH2_CHP_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NH2P_NH2_CHP;
			
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.CH_NH2P_NH2_CHP_1;
			mole.c[ipCH][ipNH2P] -= rate_COmole.CH_NH2P_NH2_CHP_2;
			mole.b[ipNH2P] -= rate_COmole.b_CH_NH2P_NH2_CHP;
			
			mole.c[ipNH2P][ipNH2] += rate_COmole.CH_NH2P_NH2_CHP_1;
			mole.c[ipCH][ipNH2] += rate_COmole.CH_NH2P_NH2_CHP_2;
			mole.b[ipNH2] += rate_COmole.b_CH_NH2P_NH2_CHP;
			
			mole.c[ipNH2P][ipCHP] += rate_COmole.CH_NH2P_NH2_CHP_1;
			mole.c[ipCH][ipCHP] += rate_COmole.CH_NH2P_NH2_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_CH_NH2P_NH2_CHP;

		/*  >>refer   mol  CH+    +  NH3   =>   NH3+  +     CH    UMIST  */

			mole.c[ipNH3][ipCHP] -= rate_COmole.CHP_NH3_NH3P_CH_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_NH3_NH3P_CH_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_NH3_NH3P_CH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CHP_NH3_NH3P_CH_1;
			mole.c[ipCHP][ipNH3] -= rate_COmole.CHP_NH3_NH3P_CH_2;
			mole.b[ipNH3] -= rate_COmole.b_CHP_NH3_NH3P_CH;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.CHP_NH3_NH3P_CH_1;
			mole.c[ipCHP][ipNH3P] += rate_COmole.CHP_NH3_NH3P_CH_2;
			mole.b[ipNH3P] += rate_COmole.b_CHP_NH3_NH3P_CH;
			
			mole.c[ipNH3][ipCH] += rate_COmole.CHP_NH3_NH3P_CH_1;
			mole.c[ipCHP][ipCH] += rate_COmole.CHP_NH3_NH3P_CH_2;
			mole.b[ipCH] += rate_COmole.b_CHP_NH3_NH3P_CH;

		/*  >>refer   mol  CH     +  CN+   =>   CN    +     CH+   UMIST  */

			mole.c[ipCNP][ipCH] -= rate_COmole.CH_CNP_CN_CHP_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_CNP_CN_CHP_2;
			mole.b[ipCH] -= rate_COmole.b_CH_CNP_CN_CHP;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.CH_CNP_CN_CHP_1;
			mole.c[ipCH][ipCNP] -= rate_COmole.CH_CNP_CN_CHP_2;
			mole.b[ipCNP] -= rate_COmole.b_CH_CNP_CN_CHP;
			
			mole.c[ipCNP][ipCN] += rate_COmole.CH_CNP_CN_CHP_1;
			mole.c[ipCH][ipCN] += rate_COmole.CH_CNP_CN_CHP_2;
			mole.b[ipCN] += rate_COmole.b_CH_CNP_CN_CHP;
			
			mole.c[ipCNP][ipCHP] += rate_COmole.CH_CNP_CN_CHP_1;
			mole.c[ipCH][ipCHP] += rate_COmole.CH_CNP_CN_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_CH_CNP_CN_CHP;

		/*  >>refer   mol  CH     +  N2+   =>   N2    +     CH+   UMIST  */

			mole.c[ipN2P][ipCH] -= rate_COmole.CH_N2P_N2_CHP_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_N2P_N2_CHP_2;
			mole.b[ipCH] -= rate_COmole.b_CH_N2P_N2_CHP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.CH_N2P_N2_CHP_1;
			mole.c[ipCH][ipN2P] -= rate_COmole.CH_N2P_N2_CHP_2;
			mole.b[ipN2P] -= rate_COmole.b_CH_N2P_N2_CHP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.CH_N2P_N2_CHP_1;
			mole.c[ipCH][ipN2] += rate_COmole.CH_N2P_N2_CHP_2;
			mole.b[ipN2] += rate_COmole.b_CH_N2P_N2_CHP;
			
			mole.c[ipN2P][ipCHP] += rate_COmole.CH_N2P_N2_CHP_1;
			mole.c[ipCH][ipCHP] += rate_COmole.CH_N2P_N2_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_CH_N2P_N2_CHP;

		/*  >>refer   mol  CH+    +  NO    =>   NO+   +     CH    UMIST  */

			mole.c[ipNO][ipCHP] -= rate_COmole.CHP_NO_NOP_CH_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_NO_NOP_CH_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_NO_NOP_CH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CHP_NO_NOP_CH_1;
			mole.c[ipCHP][ipNO] -= rate_COmole.CHP_NO_NOP_CH_2;
			mole.b[ipNO] -= rate_COmole.b_CHP_NO_NOP_CH;
			
			mole.c[ipNO][ipNOP] += rate_COmole.CHP_NO_NOP_CH_1;
			mole.c[ipCHP][ipNOP] += rate_COmole.CHP_NO_NOP_CH_2;
			mole.b[ipNOP] += rate_COmole.b_CHP_NO_NOP_CH;
			
			mole.c[ipNO][ipCH] += rate_COmole.CHP_NO_NOP_CH_1;
			mole.c[ipCHP][ipCH] += rate_COmole.CHP_NO_NOP_CH_2;
			mole.b[ipCH] += rate_COmole.b_CHP_NO_NOP_CH;

		/*  >>refer   mol  N+     +  NH    =>   NH+   +     N     UMIST  */

			mole.c[ipNH][ipNP] -= rate_COmole.NP_NH_NHP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NH_NHP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NH_NHP_N;
			
			mole.c[ipNH][ipNH] -= rate_COmole.NP_NH_NHP_N_1;
			mole.c[ipNP][ipNH] -= rate_COmole.NP_NH_NHP_N_2;
			mole.b[ipNH] -= rate_COmole.b_NP_NH_NHP_N;
			
			mole.c[ipNH][ipNHP] += rate_COmole.NP_NH_NHP_N_1;
			mole.c[ipNP][ipNHP] += rate_COmole.NP_NH_NHP_N_2;
			mole.b[ipNHP] += rate_COmole.b_NP_NH_NHP_N;
			
			mole.c[ipNH][ipATN] += rate_COmole.NP_NH_NHP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_NH_NHP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_NH_NHP_N;
			
		/*  >>refer   mol  N+     +  NH2   =>   NH2+  +     N     UMIST  */

			mole.c[ipNH2][ipNP] -= rate_COmole.NP_NH2_NH2P_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NH2_NH2P_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NH2_NH2P_N;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.NP_NH2_NH2P_N_1;
			mole.c[ipNP][ipNH2] -= rate_COmole.NP_NH2_NH2P_N_2;
			mole.b[ipNH2] -= rate_COmole.b_NP_NH2_NH2P_N;
			
			mole.c[ipNH2][ipNH2P] += rate_COmole.NP_NH2_NH2P_N_1;
			mole.c[ipNP][ipNH2P] += rate_COmole.NP_NH2_NH2P_N_2;
			mole.b[ipNH2P] += rate_COmole.b_NP_NH2_NH2P_N;
			
			mole.c[ipNH2][ipATN] += rate_COmole.NP_NH2_NH2P_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_NH2_NH2P_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_NH2_NH2P_N;

		/*  >>refer   mol  N+     +  NH3   =>   NH3+  +     N     UMIST  */

			mole.c[ipNH3][ipNP] -= rate_COmole.NP_NH3_NH3P_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NH3_NH3P_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NH3_NH3P_N;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NP_NH3_NH3P_N_1;
			mole.c[ipNP][ipNH3] -= rate_COmole.NP_NH3_NH3P_N_2;
			mole.b[ipNH3] -= rate_COmole.b_NP_NH3_NH3P_N;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.NP_NH3_NH3P_N_1;
			mole.c[ipNP][ipNH3P] += rate_COmole.NP_NH3_NH3P_N_2;
			mole.b[ipNH3P] += rate_COmole.b_NP_NH3_NH3P_N;
			
			mole.c[ipNH3][ipATN] += rate_COmole.NP_NH3_NH3P_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_NH3_NH3P_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_NH3_NH3P_N;

		/*  >>refer   mol  N+     +  CN    =>   CN+   +     N     UMIST  */

			mole.c[ipCN][ipNP] -= rate_COmole.NP_CN_CNP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_CN_CNP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_CN_CNP_N;
			
			mole.c[ipCN][ipCN] -= rate_COmole.NP_CN_CNP_N_1;
			mole.c[ipNP][ipCN] -= rate_COmole.NP_CN_CNP_N_2;
			mole.b[ipCN] -= rate_COmole.b_NP_CN_CNP_N;
			
			mole.c[ipCN][ipCNP] += rate_COmole.NP_CN_CNP_N_1;
			mole.c[ipNP][ipCNP] += rate_COmole.NP_CN_CNP_N_2;
			mole.b[ipCNP] += rate_COmole.b_NP_CN_CNP_N;
			
			mole.c[ipCN][ipATN] += rate_COmole.NP_CN_CNP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_CN_CNP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_CN_CNP_N;

		/*  >>refer   mol  N+     +  HCN   =>   HCN+  +     N     UMIST  */

			mole.c[ipHCN][ipNP] -= rate_COmole.NP_HCN_HCNP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_HCN_HCNP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_HCN_HCNP_N;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.NP_HCN_HCNP_N_1;
			mole.c[ipNP][ipHCN] -= rate_COmole.NP_HCN_HCNP_N_2;
			mole.b[ipHCN] -= rate_COmole.b_NP_HCN_HCNP_N;
			
			mole.c[ipHCN][ipHCNP] += rate_COmole.NP_HCN_HCNP_N_1;
			mole.c[ipNP][ipHCNP] += rate_COmole.NP_HCN_HCNP_N_2;
			mole.b[ipHCNP] += rate_COmole.b_NP_HCN_HCNP_N;
			
			mole.c[ipHCN][ipATN] += rate_COmole.NP_HCN_HCNP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_HCN_HCNP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_HCN_HCNP_N;

		/*  >>refer   mol  N      +  N2+   =>   N2    +     N+    UMIST  */

			mole.c[ipN2P][ipN2P] -= rate_COmole.N_N2P_N2_NP_1;
			mole.c[ipATN][ipN2P] -= rate_COmole.N_N2P_N2_NP_2;
			mole.b[ipN2P] -= rate_COmole.b_N_N2P_N2_NP;

			mole.c[ipN2P][ipATN] -= rate_COmole.N_N2P_N2_NP_1;
			mole.c[ipATN][ipATN] -= rate_COmole.N_N2P_N2_NP_2;
			mole.b[ipATN] -= rate_COmole.b_N_N2P_N2_NP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.N_N2P_N2_NP_1;
			mole.c[ipATN][ipN2] += rate_COmole.N_N2P_N2_NP_2;
			mole.b[ipN2] += rate_COmole.b_N_N2P_N2_NP;
			
			mole.c[ipN2P][ipNP] += rate_COmole.N_N2P_N2_NP_1;
			mole.c[ipATN][ipNP] += rate_COmole.N_N2P_N2_NP_2;
			mole.b[ipNP] += rate_COmole.b_N_N2P_N2_NP;

		/*  >>refer   mol  N+     +  NO    =>   NO+   +     N     UMIST  */

			mole.c[ipNO][ipNP] -= rate_COmole.NP_NO_NOP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_NO_NOP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_NO_NOP_N;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NP_NO_NOP_N_1;
			mole.c[ipNP][ipNO] -= rate_COmole.NP_NO_NOP_N_2;
			mole.b[ipNO] -= rate_COmole.b_NP_NO_NOP_N;
			
			mole.c[ipNO][ipNOP] += rate_COmole.NP_NO_NOP_N_1;
			mole.c[ipNP][ipNOP] += rate_COmole.NP_NO_NOP_N_2;
			mole.b[ipNOP] += rate_COmole.b_NP_NO_NOP_N;
			
			mole.c[ipNO][ipATN] += rate_COmole.NP_NO_NOP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_NO_NOP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_NO_NOP_N;
			
		/*  >>refer   mol  N+     +  OCS   =>   OCS+  +     N     UMIST  */

			mole.c[ipOCS][ipNP] -= rate_COmole.NP_OCS_OCSP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_OCS_OCSP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_OCS_OCSP_N;
			
			mole.c[ipOCS][ipOCS] -= rate_COmole.NP_OCS_OCSP_N_1;
			mole.c[ipNP][ipOCS] -= rate_COmole.NP_OCS_OCSP_N_2;
			mole.b[ipOCS] -= rate_COmole.b_NP_OCS_OCSP_N;
			
			mole.c[ipOCS][ipOCSP] += rate_COmole.NP_OCS_OCSP_N_1;
			mole.c[ipNP][ipOCSP] += rate_COmole.NP_OCS_OCSP_N_2;
			mole.b[ipOCSP] += rate_COmole.b_NP_OCS_OCSP_N;
			
			mole.c[ipOCS][ipATN] += rate_COmole.NP_OCS_OCSP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_OCS_OCSP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_OCS_OCSP_N;

		/*  >>refer   mol  CH2    +  NH2+  =>   NH2   +     CH2+  UMIST  */

			mole.c[ipNH2P][ipCH2] -= rate_COmole.CH2_NH2P_NH2_CH2P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_NH2P_NH2_CH2P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_NH2P_NH2_CH2P;
			
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.CH2_NH2P_NH2_CH2P_1;
			mole.c[ipCH2][ipNH2P] -= rate_COmole.CH2_NH2P_NH2_CH2P_2;
			mole.b[ipNH2P] -= rate_COmole.b_CH2_NH2P_NH2_CH2P;
			
			mole.c[ipNH2P][ipNH2] += rate_COmole.CH2_NH2P_NH2_CH2P_1;
			mole.c[ipCH2][ipNH2] += rate_COmole.CH2_NH2P_NH2_CH2P_2;
			mole.b[ipNH2] += rate_COmole.b_CH2_NH2P_NH2_CH2P;
			
			mole.c[ipNH2P][ipCH2P] += rate_COmole.CH2_NH2P_NH2_CH2P_1;
			mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_NH2P_NH2_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH2_NH2P_NH2_CH2P;

		/*  >>refer   mol  CH2    +  CN+   =>   CN    +     CH2+  UMIST  */

			mole.c[ipCNP][ipCH2] -= rate_COmole.CH2_CNP_CN_CH2P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CNP_CN_CH2P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_CNP_CN_CH2P;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.CH2_CNP_CN_CH2P_1;
			mole.c[ipCH2][ipCNP] -= rate_COmole.CH2_CNP_CN_CH2P_2;
			mole.b[ipCNP] -= rate_COmole.b_CH2_CNP_CN_CH2P;
			
			mole.c[ipCNP][ipCN] += rate_COmole.CH2_CNP_CN_CH2P_1;
			mole.c[ipCH2][ipCN] += rate_COmole.CH2_CNP_CN_CH2P_2;
			mole.b[ipCN] += rate_COmole.b_CH2_CNP_CN_CH2P;
			
			mole.c[ipCNP][ipCH2P] += rate_COmole.CH2_CNP_CN_CH2P_1;
			mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_CNP_CN_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH2_CNP_CN_CH2P;

		/*  >>refer   mol  CH2    +  N2+   =>   N2    +     CH2+  UMIST  */

			mole.c[ipN2P][ipCH2] -= rate_COmole.CH2_N2P_N2_CH2P_1;
			mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_N2P_N2_CH2P_2;
			mole.b[ipCH2] -= rate_COmole.b_CH2_N2P_N2_CH2P;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.CH2_N2P_N2_CH2P_1;
			mole.c[ipCH2][ipN2P] -= rate_COmole.CH2_N2P_N2_CH2P_2;
			mole.b[ipN2P] -= rate_COmole.b_CH2_N2P_N2_CH2P;
			
			mole.c[ipN2P][ipN2] += rate_COmole.CH2_N2P_N2_CH2P_1;
			mole.c[ipCH2][ipN2] += rate_COmole.CH2_N2P_N2_CH2P_2;
			mole.b[ipN2] += rate_COmole.b_CH2_N2P_N2_CH2P;
			
			mole.c[ipN2P][ipCH2P] += rate_COmole.CH2_N2P_N2_CH2P_1;
			mole.c[ipCH2][ipCH2P] += rate_COmole.CH2_N2P_N2_CH2P_2;
			mole.b[ipCH2P] += rate_COmole.b_CH2_N2P_N2_CH2P;

		/*  >>refer   mol  CH2+   +  NO    =>   NO+   +     CH2   UMIST  */

			mole.c[ipNO][ipCH2P] -= rate_COmole.CH2P_NO_NOP_CH2_1;
			mole.c[ipCH2P][ipCH2P] -= rate_COmole.CH2P_NO_NOP_CH2_2;
			mole.b[ipCH2P] -= rate_COmole.b_CH2P_NO_NOP_CH2;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH2P_NO_NOP_CH2_1;
			mole.c[ipCH2P][ipNO] -= rate_COmole.CH2P_NO_NOP_CH2_2;
			mole.b[ipNO] -= rate_COmole.b_CH2P_NO_NOP_CH2;
			
			mole.c[ipNO][ipNOP] += rate_COmole.CH2P_NO_NOP_CH2_1;
			mole.c[ipCH2P][ipNOP] += rate_COmole.CH2P_NO_NOP_CH2_2;
			mole.b[ipNOP] += rate_COmole.b_CH2P_NO_NOP_CH2;
			
			mole.c[ipNO][ipCH2] += rate_COmole.CH2P_NO_NOP_CH2_1;
			mole.c[ipCH2P][ipCH2] += rate_COmole.CH2P_NO_NOP_CH2_2;
			mole.b[ipCH2] += rate_COmole.b_CH2P_NO_NOP_CH2;

		/*  >>refer   mol  NH     +  O+    =>   O     +     NH+   UMIST  */

			mole.c[ipOP][ipOP] -= rate_COmole.NH_OP_O_NHP_1;
			mole.c[ipNH][ipOP] -= rate_COmole.NH_OP_O_NHP_2;
			mole.b[ipOP] -= rate_COmole.b_NH_OP_O_NHP;
			
			mole.c[ipOP][ipNH] -= rate_COmole.NH_OP_O_NHP_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OP_O_NHP_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OP_O_NHP;
			
			mole.c[ipOP][ipNHP] += rate_COmole.NH_OP_O_NHP_1;
			mole.c[ipNH][ipNHP] += rate_COmole.NH_OP_O_NHP_2;
			mole.b[ipNHP] += rate_COmole.b_NH_OP_O_NHP;
			
			mole.c[ipOP][ipATO] += rate_COmole.NH_OP_O_NHP_1;
			mole.c[ipNH][ipATO] += rate_COmole.NH_OP_O_NHP_2;
			mole.b[ipATO] += rate_COmole.b_NH_OP_O_NHP;
			
		/*  >>refer   mol  NH     +  OH+   =>   OH    +     NH+   UMIST  */

			mole.c[ipOHP][ipOHP] -= rate_COmole.NH_OHP_OH_NHP_1;
			mole.c[ipNH][ipOHP] -= rate_COmole.NH_OHP_OH_NHP_2;
			mole.b[ipOHP] -= rate_COmole.b_NH_OHP_OH_NHP;
			
			mole.c[ipOHP][ipNH] -= rate_COmole.NH_OHP_OH_NHP_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_OHP_OH_NHP_2;
			mole.b[ipNH] -= rate_COmole.b_NH_OHP_OH_NHP;
			
			mole.c[ipOHP][ipOH] += rate_COmole.NH_OHP_OH_NHP_1;
			mole.c[ipNH][ipOH] += rate_COmole.NH_OHP_OH_NHP_2;
			mole.b[ipOH] += rate_COmole.b_NH_OHP_OH_NHP;
			
			mole.c[ipOHP][ipNHP] += rate_COmole.NH_OHP_OH_NHP_1;
			mole.c[ipNH][ipNHP] += rate_COmole.NH_OHP_OH_NHP_2;
			mole.b[ipNHP] += rate_COmole.b_NH_OHP_OH_NHP;

		/*  >>refer   mol  NH+    +  NH3   =>   NH3+  +     NH    UMIST  */

			mole.c[ipNH3][ipNHP] -= rate_COmole.NHP_NH3_NH3P_NH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_NH3_NH3P_NH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_NH3_NH3P_NH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NHP_NH3_NH3P_NH_1;
			mole.c[ipNHP][ipNH3] -= rate_COmole.NHP_NH3_NH3P_NH_2;
			mole.b[ipNH3] -= rate_COmole.b_NHP_NH3_NH3P_NH;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.NHP_NH3_NH3P_NH_1;
			mole.c[ipNHP][ipNH3P] += rate_COmole.NHP_NH3_NH3P_NH_2;
			mole.b[ipNH3P] += rate_COmole.b_NHP_NH3_NH3P_NH;
			
			mole.c[ipNH3][ipNH] += rate_COmole.NHP_NH3_NH3P_NH_1;
			mole.c[ipNHP][ipNH] += rate_COmole.NHP_NH3_NH3P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NHP_NH3_NH3P_NH;

		/*  >>refer   mol  NH+    +  H2O   =>   H2O+  +     NH    UMIST  */

			mole.c[ipH2O][ipNHP] -= rate_COmole.NHP_H2O_H2OP_NH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_H2O_H2OP_NH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_H2O_H2OP_NH;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NHP_H2O_H2OP_NH_1;
			mole.c[ipNHP][ipH2O] -= rate_COmole.NHP_H2O_H2OP_NH_2;
			mole.b[ipH2O] -= rate_COmole.b_NHP_H2O_H2OP_NH;
			
			mole.c[ipH2O][ipH2OP] += rate_COmole.NHP_H2O_H2OP_NH_1;
			mole.c[ipNHP][ipH2OP] += rate_COmole.NHP_H2O_H2OP_NH_2;
			mole.b[ipH2OP] += rate_COmole.b_NHP_H2O_H2OP_NH;
			
			mole.c[ipH2O][ipNH] += rate_COmole.NHP_H2O_H2OP_NH_1;
			mole.c[ipNHP][ipNH] += rate_COmole.NHP_H2O_H2OP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NHP_H2O_H2OP_NH;

		/*  >>refer   mol  NH     +  CN+   =>   CN    +     NH+   UMIST  */

			mole.c[ipCNP][ipNH] -= rate_COmole.NH_CNP_CN_NHP_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_CNP_CN_NHP_2;
			mole.b[ipNH] -= rate_COmole.b_NH_CNP_CN_NHP;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.NH_CNP_CN_NHP_1;
			mole.c[ipNH][ipCNP] -= rate_COmole.NH_CNP_CN_NHP_2;
			mole.b[ipCNP] -= rate_COmole.b_NH_CNP_CN_NHP;
			
			mole.c[ipCNP][ipCN] += rate_COmole.NH_CNP_CN_NHP_1;
			mole.c[ipNH][ipCN] += rate_COmole.NH_CNP_CN_NHP_2;
			mole.b[ipCN] += rate_COmole.b_NH_CNP_CN_NHP;
			
			mole.c[ipCNP][ipNHP] += rate_COmole.NH_CNP_CN_NHP_1;
			mole.c[ipNH][ipNHP] += rate_COmole.NH_CNP_CN_NHP_2;
			mole.b[ipNHP] += rate_COmole.b_NH_CNP_CN_NHP;

		/*  >>refer   mol  NH     +  N2+   =>   N2    +     NH+   UMIST  */

			mole.c[ipN2P][ipNH] -= rate_COmole.NH_N2P_N2_NHP_1;
			mole.c[ipNH][ipNH] -= rate_COmole.NH_N2P_N2_NHP_2;
			mole.b[ipNH] -= rate_COmole.b_NH_N2P_N2_NHP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.NH_N2P_N2_NHP_1;
			mole.c[ipNH][ipN2P] -= rate_COmole.NH_N2P_N2_NHP_2;
			mole.b[ipN2P] -= rate_COmole.b_NH_N2P_N2_NHP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.NH_N2P_N2_NHP_1;
			mole.c[ipNH][ipN2] += rate_COmole.NH_N2P_N2_NHP_2;
			mole.b[ipN2] += rate_COmole.b_NH_N2P_N2_NHP;
			
			mole.c[ipN2P][ipNHP] += rate_COmole.NH_N2P_N2_NHP_1;
			mole.c[ipNH][ipNHP] += rate_COmole.NH_N2P_N2_NHP_2;
			mole.b[ipNHP] += rate_COmole.b_NH_N2P_N2_NHP;

		/*  >>refer   mol  NH+    +  NO    =>   NO+   +     NH    UMIST  */

			mole.c[ipNO][ipNHP] -= rate_COmole.NHP_NO_NOP_NH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_NO_NOP_NH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_NO_NOP_NH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NHP_NO_NOP_NH_1;
			mole.c[ipNHP][ipNO] -= rate_COmole.NHP_NO_NOP_NH_2;
			mole.b[ipNO] -= rate_COmole.b_NHP_NO_NOP_NH;
			
			mole.c[ipNO][ipNOP] += rate_COmole.NHP_NO_NOP_NH_1;
			mole.c[ipNHP][ipNOP] += rate_COmole.NHP_NO_NOP_NH_2;
			mole.b[ipNOP] += rate_COmole.b_NHP_NO_NOP_NH;
			
			mole.c[ipNO][ipNH] += rate_COmole.NHP_NO_NOP_NH_1;
			mole.c[ipNHP][ipNH] += rate_COmole.NHP_NO_NOP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NHP_NO_NOP_NH;

		/*  >>refer   mol  NH+    +  O2    =>   O2+   +     NH    UMIST  */

			mole.c[ipO2][ipNHP] -= rate_COmole.NHP_O2_O2P_NH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_O2_O2P_NH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_O2_O2P_NH;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NHP_O2_O2P_NH_1;
			mole.c[ipNHP][ipO2] -= rate_COmole.NHP_O2_O2P_NH_2;
			mole.b[ipO2] -= rate_COmole.b_NHP_O2_O2P_NH;
			
			mole.c[ipO2][ipO2P] += rate_COmole.NHP_O2_O2P_NH_1;
			mole.c[ipNHP][ipO2P] += rate_COmole.NHP_O2_O2P_NH_2;
			mole.b[ipO2P] += rate_COmole.b_NHP_O2_O2P_NH;
			
			mole.c[ipO2][ipNH] += rate_COmole.NHP_O2_O2P_NH_1;
			mole.c[ipNHP][ipNH] += rate_COmole.NHP_O2_O2P_NH_2;
			mole.b[ipNH] += rate_COmole.b_NHP_O2_O2P_NH;

		/*  >>refer   mol  NH+    +  S     =>   S+    +     NH    UMIST  */
			
			mole.c[ipATS][ipNHP] -= rate_COmole.NHP_S_SP_NH_1;
			mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_S_SP_NH_2;
			mole.b[ipNHP] -= rate_COmole.b_NHP_S_SP_NH;
			
			mole.c[ipATS][ipATS] -= rate_COmole.NHP_S_SP_NH_1;
			mole.c[ipNHP][ipATS] -= rate_COmole.NHP_S_SP_NH_2;
			mole.b[ipATS] -= rate_COmole.b_NHP_S_SP_NH;
			
			mole.c[ipATS][ipSP] += rate_COmole.NHP_S_SP_NH_1;
			mole.c[ipNHP][ipSP] += rate_COmole.NHP_S_SP_NH_2;
			mole.b[ipSP] += rate_COmole.b_NHP_S_SP_NH;
			
			mole.c[ipATS][ipNH] += rate_COmole.NHP_S_SP_NH_1;
			mole.c[ipNHP][ipNH] += rate_COmole.NHP_S_SP_NH_2;
			mole.b[ipNH] += rate_COmole.b_NHP_S_SP_NH;

		/*  >>refer   mol  CH3+   +  NO    =>   NO+   +     CH3   UMIST  */

			mole.c[ipNO][ipCH3P] -= rate_COmole.CH3P_NO_NOP_CH3_1;
			mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH3P_NO_NOP_CH3_2;
			mole.b[ipCH3P] -= rate_COmole.b_CH3P_NO_NOP_CH3;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CH3P_NO_NOP_CH3_1;
			mole.c[ipCH3P][ipNO] -= rate_COmole.CH3P_NO_NOP_CH3_2;
			mole.b[ipNO] -= rate_COmole.b_CH3P_NO_NOP_CH3;
			
			mole.c[ipNO][ipNOP] += rate_COmole.CH3P_NO_NOP_CH3_1;
			mole.c[ipCH3P][ipNOP] += rate_COmole.CH3P_NO_NOP_CH3_2;
			mole.b[ipNOP] += rate_COmole.b_CH3P_NO_NOP_CH3;
			
			mole.c[ipNO][ipCH3] += rate_COmole.CH3P_NO_NOP_CH3_1;
			mole.c[ipCH3P][ipCH3] += rate_COmole.CH3P_NO_NOP_CH3_2;
			mole.b[ipCH3] += rate_COmole.b_CH3P_NO_NOP_CH3;

		/*  >>refer   mol  O+     +  NH2   =>   NH2+  +     O     UMIST  */

			mole.c[ipNH2][ipOP] -= rate_COmole.OP_NH2_NH2P_O_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_NH2_NH2P_O_2;
			mole.b[ipOP] -= rate_COmole.b_OP_NH2_NH2P_O;
			
			mole.c[ipNH2][ipNH2] -= rate_COmole.OP_NH2_NH2P_O_1;
			mole.c[ipOP][ipNH2] -= rate_COmole.OP_NH2_NH2P_O_2;
			mole.b[ipNH2] -= rate_COmole.b_OP_NH2_NH2P_O;
			
			mole.c[ipNH2][ipNH2P] += rate_COmole.OP_NH2_NH2P_O_1;
			mole.c[ipOP][ipNH2P] += rate_COmole.OP_NH2_NH2P_O_2;
			mole.b[ipNH2P] += rate_COmole.b_OP_NH2_NH2P_O;
			
			mole.c[ipNH2][ipATO] += rate_COmole.OP_NH2_NH2P_O_1;
			mole.c[ipOP][ipATO] += rate_COmole.OP_NH2_NH2P_O_2;
			mole.b[ipATO] += rate_COmole.b_OP_NH2_NH2P_O;
			
		/*  >>refer   mol  O+     +  NH3   =>   NH3+  +     O     UMIST  */

			mole.c[ipNH3][ipOP] -= rate_COmole.OP_NH3_NH3P_O_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_NH3_NH3P_O_2;
			mole.b[ipOP] -= rate_COmole.b_OP_NH3_NH3P_O;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.OP_NH3_NH3P_O_1;
			mole.c[ipOP][ipNH3] -= rate_COmole.OP_NH3_NH3P_O_2;
			mole.b[ipNH3] -= rate_COmole.b_OP_NH3_NH3P_O;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.OP_NH3_NH3P_O_1;
			mole.c[ipOP][ipNH3P] += rate_COmole.OP_NH3_NH3P_O_2;
			mole.b[ipNH3P] += rate_COmole.b_OP_NH3_NH3P_O;
			
			mole.c[ipNH3][ipATO] += rate_COmole.OP_NH3_NH3P_O_1;
			mole.c[ipOP][ipATO] += rate_COmole.OP_NH3_NH3P_O_2;
			mole.b[ipATO] += rate_COmole.b_OP_NH3_NH3P_O;
			
		/*  >>refer   mol  O      +  CN+   =>   CN    +     O+    UMIST  */

			mole.c[ipCNP][ipCNP] -= rate_COmole.O_CNP_CN_OP_1;
			mole.c[ipATO][ipCNP] -= rate_COmole.O_CNP_CN_OP_2;
			mole.b[ipCNP] -= rate_COmole.b_O_CNP_CN_OP;

			mole.c[ipCNP][ipATO] -= rate_COmole.O_CNP_CN_OP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_CNP_CN_OP_2;
			mole.b[ipATO] -= rate_COmole.b_O_CNP_CN_OP;
			
			mole.c[ipCNP][ipCN] += rate_COmole.O_CNP_CN_OP_1;
			mole.c[ipATO][ipCN] += rate_COmole.O_CNP_CN_OP_2;
			mole.b[ipCN] += rate_COmole.b_O_CNP_CN_OP;
			
			mole.c[ipCNP][ipOP] += rate_COmole.O_CNP_CN_OP_1;
			mole.c[ipATO][ipOP] += rate_COmole.O_CNP_CN_OP_2;
			mole.b[ipOP] += rate_COmole.b_O_CNP_CN_OP;

		/*  >>refer   mol  O      +  HCN+  =>   HCN   +     O+    UMIST  */

			mole.c[ipHCNP][ipHCNP] -= rate_COmole.O_HCNP_HCN_OP_1;
			mole.c[ipATO][ipHCNP] -= rate_COmole.O_HCNP_HCN_OP_2;
			mole.b[ipHCNP] -= rate_COmole.b_O_HCNP_HCN_OP;

			mole.c[ipHCNP][ipATO] -= rate_COmole.O_HCNP_HCN_OP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_HCNP_HCN_OP_2;
			mole.b[ipATO] -= rate_COmole.b_O_HCNP_HCN_OP;
			
			mole.c[ipHCNP][ipHCN] += rate_COmole.O_HCNP_HCN_OP_1;
			mole.c[ipATO][ipHCN] += rate_COmole.O_HCNP_HCN_OP_2;
			mole.b[ipHCN] += rate_COmole.b_O_HCNP_HCN_OP;
			
			mole.c[ipHCNP][ipOP] += rate_COmole.O_HCNP_HCN_OP_1;
			mole.c[ipATO][ipOP] += rate_COmole.O_HCNP_HCN_OP_2;
			mole.b[ipOP] += rate_COmole.b_O_HCNP_HCN_OP;

		/*  >>refer   mol  O      +  N2+   =>   N2    +     O+    UMIST  */

			mole.c[ipN2P][ipN2P] -= rate_COmole.O_N2P_N2_OP_1;
			mole.c[ipATO][ipN2P] -= rate_COmole.O_N2P_N2_OP_2;
			mole.b[ipN2P] -= rate_COmole.b_O_N2P_N2_OP;

			mole.c[ipN2P][ipATO] -= rate_COmole.O_N2P_N2_OP_1;
			mole.c[ipATO][ipATO] -= rate_COmole.O_N2P_N2_OP_2;
			mole.b[ipATO] -= rate_COmole.b_O_N2P_N2_OP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.O_N2P_N2_OP_1;
			mole.c[ipATO][ipN2] += rate_COmole.O_N2P_N2_OP_2;
			mole.b[ipN2] += rate_COmole.b_O_N2P_N2_OP;
			
			mole.c[ipN2P][ipOP] += rate_COmole.O_N2P_N2_OP_1;
			mole.c[ipATO][ipOP] += rate_COmole.O_N2P_N2_OP_2;
			mole.b[ipOP] += rate_COmole.b_O_N2P_N2_OP;

		/*  >>refer   mol  O+     +  NO    =>   NO+   +     O     UMIST  */

			mole.c[ipNO][ipOP] -= rate_COmole.OP_NO_NOP_O_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_NO_NOP_O_2;
			mole.b[ipOP] -= rate_COmole.b_OP_NO_NOP_O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.OP_NO_NOP_O_1;
			mole.c[ipOP][ipNO] -= rate_COmole.OP_NO_NOP_O_2;
			mole.b[ipNO] -= rate_COmole.b_OP_NO_NOP_O;
			
			mole.c[ipNO][ipNOP] += rate_COmole.OP_NO_NOP_O_1;
			mole.c[ipOP][ipNOP] += rate_COmole.OP_NO_NOP_O_2;
			mole.b[ipNOP] += rate_COmole.b_OP_NO_NOP_O;
			
			mole.c[ipNO][ipATO] += rate_COmole.OP_NO_NOP_O_1;
			mole.c[ipOP][ipATO] += rate_COmole.OP_NO_NOP_O_2;
			mole.b[ipATO] += rate_COmole.b_OP_NO_NOP_O;
			
		/*  >>refer   mol  O+     +  OCS   =>   OCS+  +     O     UMIST  */

			mole.c[ipOCS][ipOP] -= rate_COmole.OP_OCS_OCSP_O_1;
			mole.c[ipOP][ipOP] -= rate_COmole.OP_OCS_OCSP_O_2;
			mole.b[ipOP] -= rate_COmole.b_OP_OCS_OCSP_O;
			
			mole.c[ipOCS][ipOCS] -= rate_COmole.OP_OCS_OCSP_O_1;
			mole.c[ipOP][ipOCS] -= rate_COmole.OP_OCS_OCSP_O_2;
			mole.b[ipOCS] -= rate_COmole.b_OP_OCS_OCSP_O;
			
			mole.c[ipOCS][ipOCSP] += rate_COmole.OP_OCS_OCSP_O_1;
			mole.c[ipOP][ipOCSP] += rate_COmole.OP_OCS_OCSP_O_2;
			mole.b[ipOCSP] += rate_COmole.b_OP_OCS_OCSP_O;
			
			mole.c[ipOCS][ipATO] += rate_COmole.OP_OCS_OCSP_O_1;
			mole.c[ipOP][ipATO] += rate_COmole.OP_OCS_OCSP_O_2;
			mole.b[ipATO] += rate_COmole.b_OP_OCS_OCSP_O;

		/*  >>refer   mol  NH2    +  OH+   =>   OH    +     NH2+  UMIST  */

			mole.c[ipOHP][ipNH2] -= rate_COmole.NH2_OHP_OH_NH2P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_OHP_OH_NH2P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_OHP_OH_NH2P;
			
			mole.c[ipOHP][ipOHP] -= rate_COmole.NH2_OHP_OH_NH2P_1;
			mole.c[ipNH2][ipOHP] -= rate_COmole.NH2_OHP_OH_NH2P_2;
			mole.b[ipOHP] -= rate_COmole.b_NH2_OHP_OH_NH2P;
			
			mole.c[ipOHP][ipOH] += rate_COmole.NH2_OHP_OH_NH2P_1;
			mole.c[ipNH2][ipOH] += rate_COmole.NH2_OHP_OH_NH2P_2;
			mole.b[ipOH] += rate_COmole.b_NH2_OHP_OH_NH2P;
			
			mole.c[ipOHP][ipNH2P] += rate_COmole.NH2_OHP_OH_NH2P_1;
			mole.c[ipNH2][ipNH2P] += rate_COmole.NH2_OHP_OH_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH2_OHP_OH_NH2P;

		/*  >>refer   mol  NH2+   +  NH3   =>   NH3+  +     NH2   UMIST  */

			mole.c[ipNH3][ipNH2P] -= rate_COmole.NH2P_NH3_NH3P_NH2_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_NH3_NH3P_NH2_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_NH3_NH3P_NH2;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH2P_NH3_NH3P_NH2_1;
			mole.c[ipNH2P][ipNH3] -= rate_COmole.NH2P_NH3_NH3P_NH2_2;
			mole.b[ipNH3] -= rate_COmole.b_NH2P_NH3_NH3P_NH2;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH2P_NH3_NH3P_NH2_1;
			mole.c[ipNH2P][ipNH3P] += rate_COmole.NH2P_NH3_NH3P_NH2_2;
			mole.b[ipNH3P] += rate_COmole.b_NH2P_NH3_NH3P_NH2;
			
			mole.c[ipNH3][ipNH2] += rate_COmole.NH2P_NH3_NH3P_NH2_1;
			mole.c[ipNH2P][ipNH2] += rate_COmole.NH2P_NH3_NH3P_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH2P_NH3_NH3P_NH2;

		/*  >>refer   mol  NH2    +  H2O+  =>   H2O   +     NH2+  UMIST  */

			mole.c[ipH2OP][ipNH2] -= rate_COmole.NH2_H2OP_H2O_NH2P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_H2OP_H2O_NH2P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_H2OP_H2O_NH2P;
			
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.NH2_H2OP_H2O_NH2P_1;
			mole.c[ipNH2][ipH2OP] -= rate_COmole.NH2_H2OP_H2O_NH2P_2;
			mole.b[ipH2OP] -= rate_COmole.b_NH2_H2OP_H2O_NH2P;
			
			mole.c[ipH2OP][ipH2O] += rate_COmole.NH2_H2OP_H2O_NH2P_1;
			mole.c[ipNH2][ipH2O] += rate_COmole.NH2_H2OP_H2O_NH2P_2;
			mole.b[ipH2O] += rate_COmole.b_NH2_H2OP_H2O_NH2P;
			
			mole.c[ipH2OP][ipNH2P] += rate_COmole.NH2_H2OP_H2O_NH2P_1;
			mole.c[ipNH2][ipNH2P] += rate_COmole.NH2_H2OP_H2O_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH2_H2OP_H2O_NH2P;

		/*  >>refer   mol  NH2    +  CN+   =>   CN    +     NH2+  UMIST  */

			mole.c[ipCNP][ipNH2] -= rate_COmole.NH2_CNP_CN_NH2P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_CNP_CN_NH2P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_CNP_CN_NH2P;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.NH2_CNP_CN_NH2P_1;
			mole.c[ipNH2][ipCNP] -= rate_COmole.NH2_CNP_CN_NH2P_2;
			mole.b[ipCNP] -= rate_COmole.b_NH2_CNP_CN_NH2P;
			
			mole.c[ipCNP][ipCN] += rate_COmole.NH2_CNP_CN_NH2P_1;
			mole.c[ipNH2][ipCN] += rate_COmole.NH2_CNP_CN_NH2P_2;
			mole.b[ipCN] += rate_COmole.b_NH2_CNP_CN_NH2P;
			
			mole.c[ipCNP][ipNH2P] += rate_COmole.NH2_CNP_CN_NH2P_1;
			mole.c[ipNH2][ipNH2P] += rate_COmole.NH2_CNP_CN_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH2_CNP_CN_NH2P;

		/*  >>refer   mol  NH2    +  N2+   =>   N2    +     NH2+  UMIST  */

			mole.c[ipN2P][ipNH2] -= rate_COmole.NH2_N2P_N2_NH2P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_N2P_N2_NH2P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_N2P_N2_NH2P;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.NH2_N2P_N2_NH2P_1;
			mole.c[ipNH2][ipN2P] -= rate_COmole.NH2_N2P_N2_NH2P_2;
			mole.b[ipN2P] -= rate_COmole.b_NH2_N2P_N2_NH2P;
			
			mole.c[ipN2P][ipN2] += rate_COmole.NH2_N2P_N2_NH2P_1;
			mole.c[ipNH2][ipN2] += rate_COmole.NH2_N2P_N2_NH2P_2;
			mole.b[ipN2] += rate_COmole.b_NH2_N2P_N2_NH2P;
			
			mole.c[ipN2P][ipNH2P] += rate_COmole.NH2_N2P_N2_NH2P_1;
			mole.c[ipNH2][ipNH2P] += rate_COmole.NH2_N2P_N2_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH2_N2P_N2_NH2P;

		/*  >>refer   mol  NH2+   +  NO    =>   NO+   +     NH2   UMIST  */

			mole.c[ipNO][ipNH2P] -= rate_COmole.NH2P_NO_NOP_NH2_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_NO_NOP_NH2_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_NO_NOP_NH2;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH2P_NO_NOP_NH2_1;
			mole.c[ipNH2P][ipNO] -= rate_COmole.NH2P_NO_NOP_NH2_2;
			mole.b[ipNO] -= rate_COmole.b_NH2P_NO_NOP_NH2;
			
			mole.c[ipNO][ipNOP] += rate_COmole.NH2P_NO_NOP_NH2_1;
			mole.c[ipNH2P][ipNOP] += rate_COmole.NH2P_NO_NOP_NH2_2;
			mole.b[ipNOP] += rate_COmole.b_NH2P_NO_NOP_NH2;
			
			mole.c[ipNO][ipNH2] += rate_COmole.NH2P_NO_NOP_NH2_1;
			mole.c[ipNH2P][ipNH2] += rate_COmole.NH2P_NO_NOP_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH2P_NO_NOP_NH2;

		/*  >>refer   mol  NH2    +  O2+   =>   O2    +     NH2+  UMIST  */

			mole.c[ipO2P][ipNH2] -= rate_COmole.NH2_O2P_O2_NH2P_1;
			mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_O2P_O2_NH2P_2;
			mole.b[ipNH2] -= rate_COmole.b_NH2_O2P_O2_NH2P;
			
			mole.c[ipO2P][ipO2P] -= rate_COmole.NH2_O2P_O2_NH2P_1;
			mole.c[ipNH2][ipO2P] -= rate_COmole.NH2_O2P_O2_NH2P_2;
			mole.b[ipO2P] -= rate_COmole.b_NH2_O2P_O2_NH2P;
			
			mole.c[ipO2P][ipO2] += rate_COmole.NH2_O2P_O2_NH2P_1;
			mole.c[ipNH2][ipO2] += rate_COmole.NH2_O2P_O2_NH2P_2;
			mole.b[ipO2] += rate_COmole.b_NH2_O2P_O2_NH2P;
			
			mole.c[ipO2P][ipNH2P] += rate_COmole.NH2_O2P_O2_NH2P_1;
			mole.c[ipNH2][ipNH2P] += rate_COmole.NH2_O2P_O2_NH2P_2;
			mole.b[ipNH2P] += rate_COmole.b_NH2_O2P_O2_NH2P;

		/*  >>refer   mol  NH2+   +  S     =>   S+    +     NH2   UMIST  */

			mole.c[ipATS][ipNH2P] -= rate_COmole.NH2P_S_SP_NH2_1;
			mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_S_SP_NH2_2;
			mole.b[ipNH2P] -= rate_COmole.b_NH2P_S_SP_NH2;

			mole.c[ipATS][ipATS] -= rate_COmole.NH2P_S_SP_NH2_1;
			mole.c[ipNH2P][ipATS] -= rate_COmole.NH2P_S_SP_NH2_2;
			mole.b[ipATS] -= rate_COmole.b_NH2P_S_SP_NH2;
			
			mole.c[ipATS][ipSP] += rate_COmole.NH2P_S_SP_NH2_1;
			mole.c[ipNH2P][ipSP] += rate_COmole.NH2P_S_SP_NH2_2;
			mole.b[ipSP] += rate_COmole.b_NH2P_S_SP_NH2;
			
			mole.c[ipATS][ipNH2] += rate_COmole.NH2P_S_SP_NH2_1;
			mole.c[ipNH2P][ipNH2] += rate_COmole.NH2P_S_SP_NH2_2;
			mole.b[ipNH2] += rate_COmole.b_NH2P_S_SP_NH2;

		/*  >>refer   mol  CH4+   +  NH3   =>   NH3+  +     CH4   UMIST  */

			mole.c[ipNH3][ipCH4P] -= rate_COmole.CH4P_NH3_NH3P_CH4_1;
			mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4P_NH3_NH3P_CH4_2;
			mole.b[ipCH4P] -= rate_COmole.b_CH4P_NH3_NH3P_CH4;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.CH4P_NH3_NH3P_CH4_1;
			mole.c[ipCH4P][ipNH3] -= rate_COmole.CH4P_NH3_NH3P_CH4_2;
			mole.b[ipNH3] -= rate_COmole.b_CH4P_NH3_NH3P_CH4;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.CH4P_NH3_NH3P_CH4_1;
			mole.c[ipCH4P][ipNH3P] += rate_COmole.CH4P_NH3_NH3P_CH4_2;
			mole.b[ipNH3P] += rate_COmole.b_CH4P_NH3_NH3P_CH4;
			
			mole.c[ipNH3][ipCH4] += rate_COmole.CH4P_NH3_NH3P_CH4_1;
			mole.c[ipCH4P][ipCH4] += rate_COmole.CH4P_NH3_NH3P_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_CH4P_NH3_NH3P_CH4;

		/*  >>refer   mol  CH4+   +  OCS   =>   OCS+  +     CH4   UMIST  */

			mole.c[ipOCS][ipCH4P] -= rate_COmole.CH4P_OCS_OCSP_CH4_1;
			mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4P_OCS_OCSP_CH4_2;
			mole.b[ipCH4P] -= rate_COmole.b_CH4P_OCS_OCSP_CH4;
			
			mole.c[ipOCS][ipOCS] -= rate_COmole.CH4P_OCS_OCSP_CH4_1;
			mole.c[ipCH4P][ipOCS] -= rate_COmole.CH4P_OCS_OCSP_CH4_2;
			mole.b[ipOCS] -= rate_COmole.b_CH4P_OCS_OCSP_CH4;
			
			mole.c[ipOCS][ipOCSP] += rate_COmole.CH4P_OCS_OCSP_CH4_1;
			mole.c[ipCH4P][ipOCSP] += rate_COmole.CH4P_OCS_OCSP_CH4_2;
			mole.b[ipOCSP] += rate_COmole.b_CH4P_OCS_OCSP_CH4;
			
			mole.c[ipOCS][ipCH4] += rate_COmole.CH4P_OCS_OCSP_CH4_1;
			mole.c[ipCH4P][ipCH4] += rate_COmole.CH4P_OCS_OCSP_CH4_2;
			mole.b[ipCH4] += rate_COmole.b_CH4P_OCS_OCSP_CH4;

		/*  >>refer   mol  OH+    +  NH3   =>   NH3+  +     OH    UMIST  */

			mole.c[ipNH3][ipOHP] -= rate_COmole.OHP_NH3_NH3P_OH_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_NH3_NH3P_OH_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_NH3_NH3P_OH;
			
			mole.c[ipNH3][ipNH3] -= rate_COmole.OHP_NH3_NH3P_OH_1;
			mole.c[ipOHP][ipNH3] -= rate_COmole.OHP_NH3_NH3P_OH_2;
			mole.b[ipNH3] -= rate_COmole.b_OHP_NH3_NH3P_OH;
			
			mole.c[ipNH3][ipNH3P] += rate_COmole.OHP_NH3_NH3P_OH_1;
			mole.c[ipOHP][ipNH3P] += rate_COmole.OHP_NH3_NH3P_OH_2;
			mole.b[ipNH3P] += rate_COmole.b_OHP_NH3_NH3P_OH;
			
			mole.c[ipNH3][ipOH] += rate_COmole.OHP_NH3_NH3P_OH_1;
			mole.c[ipOHP][ipOH] += rate_COmole.OHP_NH3_NH3P_OH_2;
			mole.b[ipOH] += rate_COmole.b_OHP_NH3_NH3P_OH;

		/*  >>refer   mol  OH     +  CN+   =>   CN    +     OH+   UMIST  */

			mole.c[ipCNP][ipOH] -= rate_COmole.OH_CNP_CN_OHP_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_CNP_CN_OHP_2;
			mole.b[ipOH] -= rate_COmole.b_OH_CNP_CN_OHP;
			
			mole.c[ipCNP][ipCNP] -= rate_COmole.OH_CNP_CN_OHP_1;
			mole.c[ipOH][ipCNP] -= rate_COmole.OH_CNP_CN_OHP_2;
			mole.b[ipCNP] -= rate_COmole.b_OH_CNP_CN_OHP;
			
			mole.c[ipCNP][ipCN] += rate_COmole.OH_CNP_CN_OHP_1;
			mole.c[ipOH][ipCN] += rate_COmole.OH_CNP_CN_OHP_2;
			mole.b[ipCN] += rate_COmole.b_OH_CNP_CN_OHP;
			
			mole.c[ipCNP][ipOHP] += rate_COmole.OH_CNP_CN_OHP_1;
			mole.c[ipOH][ipOHP] += rate_COmole.OH_CNP_CN_OHP_2;
			mole.b[ipOHP] += rate_COmole.b_OH_CNP_CN_OHP;

		/*  >>refer   mol  OH     +  N2+   =>   N2    +     OH+   UMIST  */

			mole.c[ipN2P][ipOH] -= rate_COmole.OH_N2P_N2_OHP_1;
			mole.c[ipOH][ipOH] -= rate_COmole.OH_N2P_N2_OHP_2;
			mole.b[ipOH] -= rate_COmole.b_OH_N2P_N2_OHP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.OH_N2P_N2_OHP_1;
			mole.c[ipOH][ipN2P] -= rate_COmole.OH_N2P_N2_OHP_2;
			mole.b[ipN2P] -= rate_COmole.b_OH_N2P_N2_OHP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.OH_N2P_N2_OHP_1;
			mole.c[ipOH][ipN2] += rate_COmole.OH_N2P_N2_OHP_2;
			mole.b[ipN2] += rate_COmole.b_OH_N2P_N2_OHP;
			
			mole.c[ipN2P][ipOHP] += rate_COmole.OH_N2P_N2_OHP_1;
			mole.c[ipOH][ipOHP] += rate_COmole.OH_N2P_N2_OHP_2;
			mole.b[ipOHP] += rate_COmole.b_OH_N2P_N2_OHP;

		/*  >>refer   mol  OH+    +  NO    =>   NO+   +     OH    UMIST  */

			mole.c[ipNO][ipOHP] -= rate_COmole.OHP_NO_NOP_OH_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_NO_NOP_OH_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_NO_NOP_OH;
			
			mole.c[ipNO][ipNO] -= rate_COmole.OHP_NO_NOP_OH_1;
			mole.c[ipOHP][ipNO] -= rate_COmole.OHP_NO_NOP_OH_2;
			mole.b[ipNO] -= rate_COmole.b_OHP_NO_NOP_OH;
			
			mole.c[ipNO][ipNOP] += rate_COmole.OHP_NO_NOP_OH_1;
			mole.c[ipOHP][ipNOP] += rate_COmole.OHP_NO_NOP_OH_2;
			mole.b[ipNOP] += rate_COmole.b_OHP_NO_NOP_OH;
			
			mole.c[ipNO][ipOH] += rate_COmole.OHP_NO_NOP_OH_1;
			mole.c[ipOHP][ipOH] += rate_COmole.OHP_NO_NOP_OH_2;
			mole.b[ipOH] += rate_COmole.b_OHP_NO_NOP_OH;

		/*  >>refer   mol  NH3    +  H2O+  =>   H2O   +     NH3+  UMIST  */

			mole.c[ipH2OP][ipNH3] -= rate_COmole.NH3_H2OP_H2O_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_H2OP_H2O_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_H2OP_H2O_NH3P;
			
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.NH3_H2OP_H2O_NH3P_1;
			mole.c[ipNH3][ipH2OP] -= rate_COmole.NH3_H2OP_H2O_NH3P_2;
			mole.b[ipH2OP] -= rate_COmole.b_NH3_H2OP_H2O_NH3P;
			
			mole.c[ipH2OP][ipH2O] += rate_COmole.NH3_H2OP_H2O_NH3P_1;
			mole.c[ipNH3][ipH2O] += rate_COmole.NH3_H2OP_H2O_NH3P_2;
			mole.b[ipH2O] += rate_COmole.b_NH3_H2OP_H2O_NH3P;
			
			mole.c[ipH2OP][ipNH3P] += rate_COmole.NH3_H2OP_H2O_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_H2OP_H2O_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_H2OP_H2O_NH3P;

		/*  >>refer   mol  NH3    +  HCN+  =>   HCN   +     NH3+  UMIST  */

			mole.c[ipHCNP][ipNH3] -= rate_COmole.NH3_HCNP_HCN_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HCNP_HCN_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_HCNP_HCN_NH3P;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.NH3_HCNP_HCN_NH3P_1;
			mole.c[ipNH3][ipHCNP] -= rate_COmole.NH3_HCNP_HCN_NH3P_2;
			mole.b[ipHCNP] -= rate_COmole.b_NH3_HCNP_HCN_NH3P;
			
			mole.c[ipHCNP][ipHCN] += rate_COmole.NH3_HCNP_HCN_NH3P_1;
			mole.c[ipNH3][ipHCN] += rate_COmole.NH3_HCNP_HCN_NH3P_2;
			mole.b[ipHCN] += rate_COmole.b_NH3_HCNP_HCN_NH3P;
			
			mole.c[ipHCNP][ipNH3P] += rate_COmole.NH3_HCNP_HCN_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_HCNP_HCN_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_HCNP_HCN_NH3P;

		/*  >>refer   mol  NH3    +  N2+   =>   N2    +     NH3+  UMIST  */

			mole.c[ipN2P][ipNH3] -= rate_COmole.NH3_N2P_N2_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_N2P_N2_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_N2P_N2_NH3P;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.NH3_N2P_N2_NH3P_1;
			mole.c[ipNH3][ipN2P] -= rate_COmole.NH3_N2P_N2_NH3P_2;
			mole.b[ipN2P] -= rate_COmole.b_NH3_N2P_N2_NH3P;
			
			mole.c[ipN2P][ipN2] += rate_COmole.NH3_N2P_N2_NH3P_1;
			mole.c[ipNH3][ipN2] += rate_COmole.NH3_N2P_N2_NH3P_2;
			mole.b[ipN2] += rate_COmole.b_NH3_N2P_N2_NH3P;
			
			mole.c[ipN2P][ipNH3P] += rate_COmole.NH3_N2P_N2_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_N2P_N2_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_N2P_N2_NH3P;

		/*  >>refer   mol  NH3+   +  SI    =>   SI+   +     NH3   UMIST  */

			mole.c[ipATSI][ipNH3P] -= rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH3P_SI_SIP_NH3;

			mole.c[ipATSI][ipATSI] -= rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipATSI] -= rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipATSI] -= rate_COmole.b_NH3P_SI_SIP_NH3;
			
			mole.c[ipATSI][ipSIP] += rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipSIP] += rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipSIP] += rate_COmole.b_NH3P_SI_SIP_NH3;
			
			mole.c[ipATSI][ipNH3] += rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNH3] += rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNH3] += rate_COmole.b_NH3P_SI_SIP_NH3;

		/*  >>refer   mol  NH3+   +  NO    =>   NO+   +     NH3   UMIST  */

			mole.c[ipNO][ipNH3P] -= rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNH3P] -= rate_COmole.b_NH3P_SI_SIP_NH3;
			
			mole.c[ipNO][ipNO] -= rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNO] -= rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNO] -= rate_COmole.b_NH3P_SI_SIP_NH3;
			
			mole.c[ipNO][ipNOP] += rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNOP] += rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNOP] += rate_COmole.b_NH3P_SI_SIP_NH3;
			
			mole.c[ipNO][ipNH3] += rate_COmole.NH3P_SI_SIP_NH3_1;
			mole.c[ipNH3P][ipNH3] += rate_COmole.NH3P_SI_SIP_NH3_2;
			mole.b[ipNH3] += rate_COmole.b_NH3P_SI_SIP_NH3;

		/*  >>refer   mol  NH3    +  O2+   =>   O2    +     NH3+  UMIST  */

			mole.c[ipO2P][ipNH3] -= rate_COmole.NH3_O2P_O2_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_O2P_O2_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_O2P_O2_NH3P;
			
			mole.c[ipO2P][ipO2P] -= rate_COmole.NH3_O2P_O2_NH3P_1;
			mole.c[ipNH3][ipO2P] -= rate_COmole.NH3_O2P_O2_NH3P_2;
			mole.b[ipO2P] -= rate_COmole.b_NH3_O2P_O2_NH3P;
			
			mole.c[ipO2P][ipO2] += rate_COmole.NH3_O2P_O2_NH3P_1;
			mole.c[ipNH3][ipO2] += rate_COmole.NH3_O2P_O2_NH3P_2;
			mole.b[ipO2] += rate_COmole.b_NH3_O2P_O2_NH3P;
			
			mole.c[ipO2P][ipNH3P] += rate_COmole.NH3_O2P_O2_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_O2P_O2_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_O2P_O2_NH3P;

		/*  >>refer   mol  NH3    +  S+    =>   S     +     NH3+  UMIST  */

			mole.c[ipSP][ipNH3] -= rate_COmole.NH3_SP_S_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_SP_S_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_SP_S_NH3P;
			
			mole.c[ipSP][ipSP] -= rate_COmole.NH3_SP_S_NH3P_1;
			mole.c[ipNH3][ipSP] -= rate_COmole.NH3_SP_S_NH3P_2;
			mole.b[ipSP] -= rate_COmole.b_NH3_SP_S_NH3P;
			
			mole.c[ipSP][ipNH3P] += rate_COmole.NH3_SP_S_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_SP_S_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_SP_S_NH3P;
			
			mole.c[ipSP][ipATS] += rate_COmole.NH3_SP_S_NH3P_1;
			mole.c[ipNH3][ipATS] += rate_COmole.NH3_SP_S_NH3P_2;
			mole.b[ipATS] += rate_COmole.b_NH3_SP_S_NH3P;
			
		/*  >>refer   mol  NH3    +  HS+   =>   HS    +     NH3+  UMIST  */

			mole.c[ipHSP][ipNH3] -= rate_COmole.NH3_HSP_HS_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HSP_HS_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_HSP_HS_NH3P;
			
			mole.c[ipHSP][ipHSP] -= rate_COmole.NH3_HSP_HS_NH3P_1;
			mole.c[ipNH3][ipHSP] -= rate_COmole.NH3_HSP_HS_NH3P_2;
			mole.b[ipHSP] -= rate_COmole.b_NH3_HSP_HS_NH3P;
			
			mole.c[ipHSP][ipHS] += rate_COmole.NH3_HSP_HS_NH3P_1;
			mole.c[ipNH3][ipHS] += rate_COmole.NH3_HSP_HS_NH3P_2;
			mole.b[ipHS] += rate_COmole.b_NH3_HSP_HS_NH3P;
			
			mole.c[ipHSP][ipNH3P] += rate_COmole.NH3_HSP_HS_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_HSP_HS_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_HSP_HS_NH3P;

		/*  >>refer   mol  NH3    +  SO+   =>   SO    +     NH3+  UMIST  */

			mole.c[ipSOP][ipNH3] -= rate_COmole.NH3_SOP_SO_NH3P_1;
			mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_SOP_SO_NH3P_2;
			mole.b[ipNH3] -= rate_COmole.b_NH3_SOP_SO_NH3P;
			
			mole.c[ipSOP][ipSOP] -= rate_COmole.NH3_SOP_SO_NH3P_1;
			mole.c[ipNH3][ipSOP] -= rate_COmole.NH3_SOP_SO_NH3P_2;
			mole.b[ipSOP] -= rate_COmole.b_NH3_SOP_SO_NH3P;
			
			mole.c[ipSOP][ipSO] += rate_COmole.NH3_SOP_SO_NH3P_1;
			mole.c[ipNH3][ipSO] += rate_COmole.NH3_SOP_SO_NH3P_2;
			mole.b[ipSO] += rate_COmole.b_NH3_SOP_SO_NH3P;
			
			mole.c[ipSOP][ipNH3P] += rate_COmole.NH3_SOP_SO_NH3P_1;
			mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_SOP_SO_NH3P_2;
			mole.b[ipNH3P] += rate_COmole.b_NH3_SOP_SO_NH3P;

		/*  >>refer   mol  H2O    +  HCN+  =>   HCN   +     H2O+  UMIST  */

			mole.c[ipHCNP][ipH2O] -= rate_COmole.H2O_HCNP_HCN_H2OP_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_HCNP_HCN_H2OP_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_HCNP_HCN_H2OP;
			
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.H2O_HCNP_HCN_H2OP_1;
			mole.c[ipH2O][ipHCNP] -= rate_COmole.H2O_HCNP_HCN_H2OP_2;
			mole.b[ipHCNP] -= rate_COmole.b_H2O_HCNP_HCN_H2OP;
			
			mole.c[ipHCNP][ipHCN] += rate_COmole.H2O_HCNP_HCN_H2OP_1;
			mole.c[ipH2O][ipHCN] += rate_COmole.H2O_HCNP_HCN_H2OP_2;
			mole.b[ipHCN] += rate_COmole.b_H2O_HCNP_HCN_H2OP;
			
			mole.c[ipHCNP][ipH2OP] += rate_COmole.H2O_HCNP_HCN_H2OP_1;
			mole.c[ipH2O][ipH2OP] += rate_COmole.H2O_HCNP_HCN_H2OP_2;
			mole.b[ipH2OP] += rate_COmole.b_H2O_HCNP_HCN_H2OP;

		/*  >>refer   mol  H2O    +  N2+   =>   N2    +     H2O+  UMIST  */

			mole.c[ipN2P][ipH2O] -= rate_COmole.H2O_N2P_N2_H2OP_1;
			mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_N2P_N2_H2OP_2;
			mole.b[ipH2O] -= rate_COmole.b_H2O_N2P_N2_H2OP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.H2O_N2P_N2_H2OP_1;
			mole.c[ipH2O][ipN2P] -= rate_COmole.H2O_N2P_N2_H2OP_2;
			mole.b[ipN2P] -= rate_COmole.b_H2O_N2P_N2_H2OP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.H2O_N2P_N2_H2OP_1;
			mole.c[ipH2O][ipN2] += rate_COmole.H2O_N2P_N2_H2OP_2;
			mole.b[ipN2] += rate_COmole.b_H2O_N2P_N2_H2OP;
			
			mole.c[ipN2P][ipH2OP] += rate_COmole.H2O_N2P_N2_H2OP_1;
			mole.c[ipH2O][ipH2OP] += rate_COmole.H2O_N2P_N2_H2OP_2;
			mole.b[ipH2OP] += rate_COmole.b_H2O_N2P_N2_H2OP;

		/*  >>refer   mol  H2O+   +  NO    =>   NO+   +     H2O   UMIST  */

			mole.c[ipNO][ipH2OP] -= rate_COmole.H2OP_NO_NOP_H2O_1;
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_NO_NOP_H2O_2;
			mole.b[ipH2OP] -= rate_COmole.b_H2OP_NO_NOP_H2O;
			
			mole.c[ipNO][ipNO] -= rate_COmole.H2OP_NO_NOP_H2O_1;
			mole.c[ipH2OP][ipNO] -= rate_COmole.H2OP_NO_NOP_H2O_2;
			mole.b[ipNO] -= rate_COmole.b_H2OP_NO_NOP_H2O;
			
			mole.c[ipNO][ipNOP] += rate_COmole.H2OP_NO_NOP_H2O_1;
			mole.c[ipH2OP][ipNOP] += rate_COmole.H2OP_NO_NOP_H2O_2;
			mole.b[ipNOP] += rate_COmole.b_H2OP_NO_NOP_H2O;
			
			mole.c[ipNO][ipH2O] += rate_COmole.H2OP_NO_NOP_H2O_1;
			mole.c[ipH2OP][ipH2O] += rate_COmole.H2OP_NO_NOP_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_H2OP_NO_NOP_H2O;

		/*  >>refer   mol  CN+    +  HCN   =>   HCN+  +     CN    UMIST  */

			mole.c[ipHCN][ipCNP] -= rate_COmole.CNP_HCN_HCNP_CN_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_HCN_HCNP_CN_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_HCN_HCNP_CN;
			
			mole.c[ipHCN][ipHCN] -= rate_COmole.CNP_HCN_HCNP_CN_1;
			mole.c[ipCNP][ipHCN] -= rate_COmole.CNP_HCN_HCNP_CN_2;
			mole.b[ipHCN] -= rate_COmole.b_CNP_HCN_HCNP_CN;
			
			mole.c[ipHCN][ipHCNP] += rate_COmole.CNP_HCN_HCNP_CN_1;
			mole.c[ipCNP][ipHCNP] += rate_COmole.CNP_HCN_HCNP_CN_2;
			mole.b[ipHCNP] += rate_COmole.b_CNP_HCN_HCNP_CN;
			
			mole.c[ipHCN][ipCN] += rate_COmole.CNP_HCN_HCNP_CN_1;
			mole.c[ipCNP][ipCN] += rate_COmole.CNP_HCN_HCNP_CN_2;
			mole.b[ipCN] += rate_COmole.b_CNP_HCN_HCNP_CN;

		/*  >>refer   mol  CN     +  N2+   =>   N2    +     CN+   UMIST  */

			mole.c[ipN2P][ipCN] -= rate_COmole.CN_N2P_N2_CNP_1;
			mole.c[ipCN][ipCN] -= rate_COmole.CN_N2P_N2_CNP_2;
			mole.b[ipCN] -= rate_COmole.b_CN_N2P_N2_CNP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.CN_N2P_N2_CNP_1;
			mole.c[ipCN][ipN2P] -= rate_COmole.CN_N2P_N2_CNP_2;
			mole.b[ipN2P] -= rate_COmole.b_CN_N2P_N2_CNP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.CN_N2P_N2_CNP_1;
			mole.c[ipCN][ipN2] += rate_COmole.CN_N2P_N2_CNP_2;
			mole.b[ipN2] += rate_COmole.b_CN_N2P_N2_CNP;
			
			mole.c[ipN2P][ipCNP] += rate_COmole.CN_N2P_N2_CNP_1;
			mole.c[ipCN][ipCNP] += rate_COmole.CN_N2P_N2_CNP_2;
			mole.b[ipCNP] += rate_COmole.b_CN_N2P_N2_CNP;

		/*  >>refer   mol  CN+    +  NO    =>   NO+   +     CN    UMIST  */

			mole.c[ipNO][ipCNP] -= rate_COmole.CNP_NO_NOP_CN_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_NO_NOP_CN_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_NO_NOP_CN;
			
			mole.c[ipNO][ipNO] -= rate_COmole.CNP_NO_NOP_CN_1;
			mole.c[ipCNP][ipNO] -= rate_COmole.CNP_NO_NOP_CN_2;
			mole.b[ipNO] -= rate_COmole.b_CNP_NO_NOP_CN;
			
			mole.c[ipNO][ipNOP] += rate_COmole.CNP_NO_NOP_CN_1;
			mole.c[ipCNP][ipNOP] += rate_COmole.CNP_NO_NOP_CN_2;
			mole.b[ipNOP] += rate_COmole.b_CNP_NO_NOP_CN;
			
			mole.c[ipNO][ipCN] += rate_COmole.CNP_NO_NOP_CN_1;
			mole.c[ipCNP][ipCN] += rate_COmole.CNP_NO_NOP_CN_2;
			mole.b[ipCN] += rate_COmole.b_CNP_NO_NOP_CN;

		/*  >>refer   mol  CN+    +  O2    =>   O2+   +     CN    UMIST  */

			mole.c[ipO2][ipCNP] -= rate_COmole.CNP_O2_O2P_CN_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_O2_O2P_CN_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_O2_O2P_CN;
			
			mole.c[ipO2][ipO2] -= rate_COmole.CNP_O2_O2P_CN_1;
			mole.c[ipCNP][ipO2] -= rate_COmole.CNP_O2_O2P_CN_2;
			mole.b[ipO2] -= rate_COmole.b_CNP_O2_O2P_CN;
			
			mole.c[ipO2][ipO2P] += rate_COmole.CNP_O2_O2P_CN_1;
			mole.c[ipCNP][ipO2P] += rate_COmole.CNP_O2_O2P_CN_2;
			mole.b[ipO2P] += rate_COmole.b_CNP_O2_O2P_CN;
			
			mole.c[ipO2][ipCN] += rate_COmole.CNP_O2_O2P_CN_1;
			mole.c[ipCNP][ipCN] += rate_COmole.CNP_O2_O2P_CN_2;
			mole.b[ipCN] += rate_COmole.b_CNP_O2_O2P_CN;

		/*  >>refer   mol  CN+    +  S     =>   S+    +     CN    UMIST  */

			mole.c[ipATS][ipCNP] -= rate_COmole.CNP_S_SP_CN_1;
			mole.c[ipCNP][ipCNP] -= rate_COmole.CNP_S_SP_CN_2;
			mole.b[ipCNP] -= rate_COmole.b_CNP_S_SP_CN;

			mole.c[ipATS][ipATS] -= rate_COmole.CNP_S_SP_CN_1;
			mole.c[ipCNP][ipATS] -= rate_COmole.CNP_S_SP_CN_2;
			mole.b[ipATS] -= rate_COmole.b_CNP_S_SP_CN;
			
			mole.c[ipATS][ipSP] += rate_COmole.CNP_S_SP_CN_1;
			mole.c[ipCNP][ipSP] += rate_COmole.CNP_S_SP_CN_2;
			mole.b[ipSP] += rate_COmole.b_CNP_S_SP_CN;
			
			mole.c[ipATS][ipCN] += rate_COmole.CNP_S_SP_CN_1;
			mole.c[ipCNP][ipCN] += rate_COmole.CNP_S_SP_CN_2;
			mole.b[ipCN] += rate_COmole.b_CNP_S_SP_CN;

		/*  >>refer   mol  HCN    +  N2+   =>   N2    +     HCN+  UMIST  */

			mole.c[ipN2P][ipHCN] -= rate_COmole.HCN_N2P_N2_HCNP_1;
			mole.c[ipHCN][ipHCN] -= rate_COmole.HCN_N2P_N2_HCNP_2;
			mole.b[ipHCN] -= rate_COmole.b_HCN_N2P_N2_HCNP;
			
			mole.c[ipN2P][ipN2P] -= rate_COmole.HCN_N2P_N2_HCNP_1;
			mole.c[ipHCN][ipN2P] -= rate_COmole.HCN_N2P_N2_HCNP_2;
			mole.b[ipN2P] -= rate_COmole.b_HCN_N2P_N2_HCNP;
			
			mole.c[ipN2P][ipN2] += rate_COmole.HCN_N2P_N2_HCNP_1;
			mole.c[ipHCN][ipN2] += rate_COmole.HCN_N2P_N2_HCNP_2;
			mole.b[ipN2] += rate_COmole.b_HCN_N2P_N2_HCNP;
			
			mole.c[ipN2P][ipHCNP] += rate_COmole.HCN_N2P_N2_HCNP_1;
			mole.c[ipHCN][ipHCNP] += rate_COmole.HCN_N2P_N2_HCNP_2;
			mole.b[ipHCNP] += rate_COmole.b_HCN_N2P_N2_HCNP;

		/*  >>refer   mol  HCN+   +  NO    =>   NO+   +     HCN   UMIST  */

			mole.c[ipNO][ipHCNP] -= rate_COmole.HCNP_NO_NOP_HCN_1;
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_NO_NOP_HCN_2;
			mole.b[ipHCNP] -= rate_COmole.b_HCNP_NO_NOP_HCN;
			
			mole.c[ipNO][ipNO] -= rate_COmole.HCNP_NO_NOP_HCN_1;
			mole.c[ipHCNP][ipNO] -= rate_COmole.HCNP_NO_NOP_HCN_2;
			mole.b[ipNO] -= rate_COmole.b_HCNP_NO_NOP_HCN;
			
			mole.c[ipNO][ipNOP] += rate_COmole.HCNP_NO_NOP_HCN_1;
			mole.c[ipHCNP][ipNOP] += rate_COmole.HCNP_NO_NOP_HCN_2;
			mole.b[ipNOP] += rate_COmole.b_HCNP_NO_NOP_HCN;
			
			mole.c[ipNO][ipHCN] += rate_COmole.HCNP_NO_NOP_HCN_1;
			mole.c[ipHCNP][ipHCN] += rate_COmole.HCNP_NO_NOP_HCN_2;
			mole.b[ipHCN] += rate_COmole.b_HCNP_NO_NOP_HCN;

		/*  >>refer   mol  HCN+   +  O2    =>   O2+   +     HCN   UMIST  */

			mole.c[ipO2][ipHCNP] -= rate_COmole.HCNP_O2_O2P_HCN_1;
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_O2_O2P_HCN_2;
			mole.b[ipHCNP] -= rate_COmole.b_HCNP_O2_O2P_HCN;
			
			mole.c[ipO2][ipO2] -= rate_COmole.HCNP_O2_O2P_HCN_1;
			mole.c[ipHCNP][ipO2] -= rate_COmole.HCNP_O2_O2P_HCN_2;
			mole.b[ipO2] -= rate_COmole.b_HCNP_O2_O2P_HCN;
			
			mole.c[ipO2][ipO2P] += rate_COmole.HCNP_O2_O2P_HCN_1;
			mole.c[ipHCNP][ipO2P] += rate_COmole.HCNP_O2_O2P_HCN_2;
			mole.b[ipO2P] += rate_COmole.b_HCNP_O2_O2P_HCN;
			
			mole.c[ipO2][ipHCN] += rate_COmole.HCNP_O2_O2P_HCN_1;
			mole.c[ipHCNP][ipHCN] += rate_COmole.HCNP_O2_O2P_HCN_2;
			mole.b[ipHCN] += rate_COmole.b_HCNP_O2_O2P_HCN;

		/*  >>refer   mol  HCN+   +  S     =>   S+    +     HCN   UMIST  */

			mole.c[ipATS][ipHCNP] -= rate_COmole.HCNP_S_SP_HCN_1;
			mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_S_SP_HCN_2;
			mole.b[ipHCNP] -= rate_COmole.b_HCNP_S_SP_HCN;

			mole.c[ipATS][ipATS] -= rate_COmole.HCNP_S_SP_HCN_1;
			mole.c[ipHCNP][ipATS] -= rate_COmole.HCNP_S_SP_HCN_2;
			mole.b[ipATS] -= rate_COmole.b_HCNP_S_SP_HCN;
			
			mole.c[ipATS][ipSP] += rate_COmole.HCNP_S_SP_HCN_1;
			mole.c[ipHCNP][ipSP] += rate_COmole.HCNP_S_SP_HCN_2;
			mole.b[ipSP] += rate_COmole.b_HCNP_S_SP_HCN;
			
			mole.c[ipATS][ipHCN] += rate_COmole.HCNP_S_SP_HCN_1;
			mole.c[ipHCNP][ipHCN] += rate_COmole.HCNP_S_SP_HCN_2;
			mole.b[ipHCN] += rate_COmole.b_HCNP_S_SP_HCN;

		/*  >>refer   mol  N2+    +  NO    =>   NO+   +     N2    UMIST  */

			mole.c[ipNO][ipN2P] -= rate_COmole.N2P_NO_NOP_N2_1;
			mole.c[ipN2P][ipN2P] -= rate_COmole.N2P_NO_NOP_N2_2;
			mole.b[ipN2P] -= rate_COmole.b_N2P_NO_NOP_N2;
			
			mole.c[ipNO][ipNO] -= rate_COmole.N2P_NO_NOP_N2_1;
			mole.c[ipN2P][ipNO] -= rate_COmole.N2P_NO_NOP_N2_2;
			mole.b[ipNO] -= rate_COmole.b_N2P_NO_NOP_N2;
			
			mole.c[ipNO][ipNOP] += rate_COmole.N2P_NO_NOP_N2_1;
			mole.c[ipN2P][ipNOP] += rate_COmole.N2P_NO_NOP_N2_2;
			mole.b[ipNOP] += rate_COmole.b_N2P_NO_NOP_N2;
			
			mole.c[ipNO][ipN2] += rate_COmole.N2P_NO_NOP_N2_1;
			mole.c[ipN2P][ipN2] += rate_COmole.N2P_NO_NOP_N2_2;
			mole.b[ipN2] += rate_COmole.b_N2P_NO_NOP_N2;

		/*  >>refer   mol  N2+    +  O2    =>   O2+   +     N2    UMIST  */

			mole.c[ipO2][ipN2P] -= rate_COmole.N2P_O2_O2P_N2_1;
			mole.c[ipN2P][ipN2P] -= rate_COmole.N2P_O2_O2P_N2_2;
			mole.b[ipN2P] -= rate_COmole.b_N2P_O2_O2P_N2;
			
			mole.c[ipO2][ipO2] -= rate_COmole.N2P_O2_O2P_N2_1;
			mole.c[ipN2P][ipO2] -= rate_COmole.N2P_O2_O2P_N2_2;
			mole.b[ipO2] -= rate_COmole.b_N2P_O2_O2P_N2;
			
			mole.c[ipO2][ipO2P] += rate_COmole.N2P_O2_O2P_N2_1;
			mole.c[ipN2P][ipO2P] += rate_COmole.N2P_O2_O2P_N2_2;
			mole.b[ipO2P] += rate_COmole.b_N2P_O2_O2P_N2;
			
			mole.c[ipO2][ipN2] += rate_COmole.N2P_O2_O2P_N2_1;
			mole.c[ipN2P][ipN2] += rate_COmole.N2P_O2_O2P_N2_2;
			mole.b[ipN2] += rate_COmole.b_N2P_O2_O2P_N2;

		/*  >>refer   mol  N2+    +  S     =>   S+    +     N2    UMIST  */

			mole.c[ipATS][ipN2P] -= rate_COmole.N2P_S_SP_N2_1;
			mole.c[ipN2P][ipN2P] -= rate_COmole.N2P_S_SP_N2_2;
			mole.b[ipN2P] -= rate_COmole.b_N2P_S_SP_N2;

			mole.c[ipATS][ipATS] -= rate_COmole.N2P_S_SP_N2_1;
			mole.c[ipN2P][ipATS] -= rate_COmole.N2P_S_SP_N2_2;
			mole.b[ipATS] -= rate_COmole.b_N2P_S_SP_N2;
			
			mole.c[ipATS][ipSP] += rate_COmole.N2P_S_SP_N2_1;
			mole.c[ipN2P][ipSP] += rate_COmole.N2P_S_SP_N2_2;
			mole.b[ipSP] += rate_COmole.b_N2P_S_SP_N2;
			
			mole.c[ipATS][ipN2] += rate_COmole.N2P_S_SP_N2_1;
			mole.c[ipN2P][ipN2] += rate_COmole.N2P_S_SP_N2_2;
			mole.b[ipN2] += rate_COmole.b_N2P_S_SP_N2;

		/*  >>refer   mol  SI     +  NO+   =>   NO    +     SI+   UMIST  */

			mole.c[ipNOP][ipNOP] -= rate_COmole.SI_NOP_NO_SIP_1;
			mole.c[ipATSI][ipNOP] -= rate_COmole.SI_NOP_NO_SIP_2;
			mole.b[ipNOP] -= rate_COmole.b_SI_NOP_NO_SIP;

			mole.c[ipNOP][ipATSI] -= rate_COmole.SI_NOP_NO_SIP_1;
			mole.c[ipATSI][ipATSI] -= rate_COmole.SI_NOP_NO_SIP_2;
			mole.b[ipATSI] -= rate_COmole.b_SI_NOP_NO_SIP;
			
			mole.c[ipNOP][ipNO] += rate_COmole.SI_NOP_NO_SIP_1;
			mole.c[ipATSI][ipNO] += rate_COmole.SI_NOP_NO_SIP_2;
			mole.b[ipNO] += rate_COmole.b_SI_NOP_NO_SIP;
			
			mole.c[ipNOP][ipSIP] += rate_COmole.SI_NOP_NO_SIP_1;
			mole.c[ipATSI][ipSIP] += rate_COmole.SI_NOP_NO_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_SI_NOP_NO_SIP;

		/*  >>refer   mol  SI     +  HS+   =>   HS    +     SI+   UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.SI_HSP_HS_SIP_1;
			mole.c[ipATSI][ipHSP] -= rate_COmole.SI_HSP_HS_SIP_2;
			mole.b[ipHSP] -= rate_COmole.b_SI_HSP_HS_SIP;

			mole.c[ipHSP][ipATSI] -= rate_COmole.SI_HSP_HS_SIP_1;
			mole.c[ipATSI][ipATSI] -= rate_COmole.SI_HSP_HS_SIP_2;
			mole.b[ipATSI] -= rate_COmole.b_SI_HSP_HS_SIP;
			
			mole.c[ipHSP][ipHS] += rate_COmole.SI_HSP_HS_SIP_1;
			mole.c[ipATSI][ipHS] += rate_COmole.SI_HSP_HS_SIP_2;
			mole.b[ipHS] += rate_COmole.b_SI_HSP_HS_SIP;
			
			mole.c[ipHSP][ipSIP] += rate_COmole.SI_HSP_HS_SIP_1;
			mole.c[ipATSI][ipSIP] += rate_COmole.SI_HSP_HS_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_SI_HSP_HS_SIP;
			
		/*  >>refer   mol  SI     +  CS+   =>   CS    +     SI+   UMIST  */

			mole.c[ipCSP][ipCSP] -= rate_COmole.SI_CSP_CS_SIP_1;
			mole.c[ipATSI][ipCSP] -= rate_COmole.SI_CSP_CS_SIP_2;
			mole.b[ipCSP] -= rate_COmole.b_SI_CSP_CS_SIP;

			mole.c[ipCSP][ipATSI] -= rate_COmole.SI_CSP_CS_SIP_1;
			mole.c[ipATSI][ipATSI] -= rate_COmole.SI_CSP_CS_SIP_2;
			mole.b[ipATSI] -= rate_COmole.b_SI_CSP_CS_SIP;
			
			mole.c[ipCSP][ipCS] += rate_COmole.SI_CSP_CS_SIP_1;
			mole.c[ipATSI][ipCS] += rate_COmole.SI_CSP_CS_SIP_2;
			mole.b[ipCS] += rate_COmole.b_SI_CSP_CS_SIP;
			
			mole.c[ipCSP][ipSIP] += rate_COmole.SI_CSP_CS_SIP_1;
			mole.c[ipATSI][ipSIP] += rate_COmole.SI_CSP_CS_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_SI_CSP_CS_SIP;

		/*  >>refer   mol  NO     +  HNO+  =>   HNO   +     NO+   UMIST  */

			mole.c[ipHNOP][ipNO] -= rate_COmole.NO_HNOP_HNO_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_HNOP_HNO_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_HNOP_HNO_NOP;
			
			mole.c[ipHNOP][ipHNOP] -= rate_COmole.NO_HNOP_HNO_NOP_1;
			mole.c[ipNO][ipHNOP] -= rate_COmole.NO_HNOP_HNO_NOP_2;
			mole.b[ipHNOP] -= rate_COmole.b_NO_HNOP_HNO_NOP;
			
			mole.c[ipHNOP][ipHNO] += rate_COmole.NO_HNOP_HNO_NOP_1;
			mole.c[ipNO][ipHNO] += rate_COmole.NO_HNOP_HNO_NOP_2;
			mole.b[ipHNO] += rate_COmole.b_NO_HNOP_HNO_NOP;
			
			mole.c[ipHNOP][ipNOP] += rate_COmole.NO_HNOP_HNO_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_HNOP_HNO_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_HNOP_HNO_NOP;

		/*  >>refer   mol  NO     +  O2+   =>   O2    +     NO+   UMIST  */

			mole.c[ipO2P][ipNO] -= rate_COmole.NO_O2P_O2_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_O2P_O2_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_O2P_O2_NOP;
			
			mole.c[ipO2P][ipO2P] -= rate_COmole.NO_O2P_O2_NOP_1;
			mole.c[ipNO][ipO2P] -= rate_COmole.NO_O2P_O2_NOP_2;
			mole.b[ipO2P] -= rate_COmole.b_NO_O2P_O2_NOP;
			
			mole.c[ipO2P][ipO2] += rate_COmole.NO_O2P_O2_NOP_1;
			mole.c[ipNO][ipO2] += rate_COmole.NO_O2P_O2_NOP_2;
			mole.b[ipO2] += rate_COmole.b_NO_O2P_O2_NOP;
			
			mole.c[ipO2P][ipNOP] += rate_COmole.NO_O2P_O2_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_O2P_O2_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_O2P_O2_NOP;

		/*  >>refer   mol  NO     +  S+    =>   S     +     NO+   UMIST  */

			mole.c[ipSP][ipNO] -= rate_COmole.NO_SP_S_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_SP_S_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_SP_S_NOP;
			
			mole.c[ipSP][ipSP] -= rate_COmole.NO_SP_S_NOP_1;
			mole.c[ipNO][ipSP] -= rate_COmole.NO_SP_S_NOP_2;
			mole.b[ipSP] -= rate_COmole.b_NO_SP_S_NOP;
			
			mole.c[ipSP][ipNOP] += rate_COmole.NO_SP_S_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_SP_S_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_SP_S_NOP;
			
			mole.c[ipSP][ipATS] += rate_COmole.NO_SP_S_NOP_1;
			mole.c[ipNO][ipATS] += rate_COmole.NO_SP_S_NOP_2;
			mole.b[ipATS] += rate_COmole.b_NO_SP_S_NOP;

		/*  >>refer   mol  NO     +  HS+   =>   HS    +     NO+   UMIST  */

			mole.c[ipHSP][ipNO] -= rate_COmole.NO_HSP_HS_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_HSP_HS_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_HSP_HS_NOP;
			
			mole.c[ipHSP][ipHSP] -= rate_COmole.NO_HSP_HS_NOP_1;
			mole.c[ipNO][ipHSP] -= rate_COmole.NO_HSP_HS_NOP_2;
			mole.b[ipHSP] -= rate_COmole.b_NO_HSP_HS_NOP;
			
			mole.c[ipHSP][ipHS] += rate_COmole.NO_HSP_HS_NOP_1;
			mole.c[ipNO][ipHS] += rate_COmole.NO_HSP_HS_NOP_2;
			mole.b[ipHS] += rate_COmole.b_NO_HSP_HS_NOP;
			
			mole.c[ipHSP][ipNOP] += rate_COmole.NO_HSP_HS_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_HSP_HS_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_HSP_HS_NOP;

		/*  >>refer   mol  NO     +  SIO+  =>   SIO   +     NO+   UMIST  */

			mole.c[ipSIOP][ipNO] -= rate_COmole.NO_SIOP_SIO_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_SIOP_SIO_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_SIOP_SIO_NOP;
			
			mole.c[ipSIOP][ipSIOP] -= rate_COmole.NO_SIOP_SIO_NOP_1;
			mole.c[ipNO][ipSIOP] -= rate_COmole.NO_SIOP_SIO_NOP_2;
			mole.b[ipSIOP] -= rate_COmole.b_NO_SIOP_SIO_NOP;
			
			mole.c[ipSIOP][ipSIO] += rate_COmole.NO_SIOP_SIO_NOP_1;
			mole.c[ipNO][ipSIO] += rate_COmole.NO_SIOP_SIO_NOP_2;
			mole.b[ipSIO] += rate_COmole.b_NO_SIOP_SIO_NOP;
			
			mole.c[ipSIOP][ipNOP] += rate_COmole.NO_SIOP_SIO_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_SIOP_SIO_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_SIOP_SIO_NOP;

		/*  >>refer   mol  NO     +  S2+   =>   S2    +     NO+   UMIST  */

			mole.c[ipS2P][ipNO] -= rate_COmole.NO_S2P_S2_NOP_1;
			mole.c[ipNO][ipNO] -= rate_COmole.NO_S2P_S2_NOP_2;
			mole.b[ipNO] -= rate_COmole.b_NO_S2P_S2_NOP;
			
			mole.c[ipS2P][ipS2P] -= rate_COmole.NO_S2P_S2_NOP_1;
			mole.c[ipNO][ipS2P] -= rate_COmole.NO_S2P_S2_NOP_2;
			mole.b[ipS2P] -= rate_COmole.b_NO_S2P_S2_NOP;
			
			mole.c[ipS2P][ipS2] += rate_COmole.NO_S2P_S2_NOP_1;
			mole.c[ipNO][ipS2] += rate_COmole.NO_S2P_S2_NOP_2;
			mole.b[ipS2] += rate_COmole.b_NO_S2P_S2_NOP;
			
			mole.c[ipS2P][ipNOP] += rate_COmole.NO_S2P_S2_NOP_1;
			mole.c[ipNO][ipNOP] += rate_COmole.NO_S2P_S2_NOP_2;
			mole.b[ipNOP] += rate_COmole.b_NO_S2P_S2_NOP;

		/*  >>refer   mol  O2+    +  NO2   =>   NO2+  +     O2    UMIST  */

			mole.c[ipNO2][ipO2P] -= rate_COmole.O2P_NO2_NO2P_O2_1;
			mole.c[ipO2P][ipO2P] -= rate_COmole.O2P_NO2_NO2P_O2_2;
			mole.b[ipO2P] -= rate_COmole.b_O2P_NO2_NO2P_O2;
			
			mole.c[ipNO2][ipNO2] -= rate_COmole.O2P_NO2_NO2P_O2_1;
			mole.c[ipO2P][ipNO2] -= rate_COmole.O2P_NO2_NO2P_O2_2;
			mole.b[ipNO2] -= rate_COmole.b_O2P_NO2_NO2P_O2;
			
			mole.c[ipNO2][ipNO2P] += rate_COmole.O2P_NO2_NO2P_O2_1;
			mole.c[ipO2P][ipNO2P] += rate_COmole.O2P_NO2_NO2P_O2_2;
			mole.b[ipNO2P] += rate_COmole.b_O2P_NO2_NO2P_O2;
			
			mole.c[ipNO2][ipO2] += rate_COmole.O2P_NO2_NO2P_O2_1;
			mole.c[ipO2P][ipO2] += rate_COmole.O2P_NO2_NO2P_O2_2;
			mole.b[ipO2] += rate_COmole.b_O2P_NO2_NO2P_O2;

		/*  >>refer   mol  S      +  HS+   =>   HS    +     S+    UMIST  */

			mole.c[ipHSP][ipHSP] -= rate_COmole.S_HSP_HS_SP_1;
			mole.c[ipATS][ipHSP] -= rate_COmole.S_HSP_HS_SP_2;
			mole.b[ipHSP] -= rate_COmole.b_S_HSP_HS_SP;

			mole.c[ipHSP][ipATS] -= rate_COmole.S_HSP_HS_SP_1;
			mole.c[ipATS][ipATS] -= rate_COmole.S_HSP_HS_SP_2;
			mole.b[ipATS] -= rate_COmole.b_S_HSP_HS_SP;
						
			mole.c[ipHSP][ipHS] += rate_COmole.S_HSP_HS_SP_1;
			mole.c[ipATS][ipHS] += rate_COmole.S_HSP_HS_SP_2;
			mole.b[ipHS] += rate_COmole.b_S_HSP_HS_SP;
			
			mole.c[ipHSP][ipSP] += rate_COmole.S_HSP_HS_SP_1;
			mole.c[ipATS][ipSP] += rate_COmole.S_HSP_HS_SP_2;
			mole.b[ipSP] += rate_COmole.b_S_HSP_HS_SP;
			

		/*  >>refer   mol  C      +  N     =>   CN    +     NU    UMIST  */

			mole.c[ipATN][ipATN]  -= rate_COmole.C_N_CN_NU_1;
			mole.c[ipATC][ipATN] -= rate_COmole.C_N_CN_NU_2;
			mole.b[ipATN] -= rate_COmole.b_C_N_CN_NU;

			mole.c[ipATN][ipATC]  -= rate_COmole.C_N_CN_NU_1;
			mole.c[ipATC][ipATC] -= rate_COmole.C_N_CN_NU_2;
			mole.b[ipATC] -= rate_COmole.b_C_N_CN_NU;

			mole.c[ipATN][ipCN]  += rate_COmole.C_N_CN_NU_1;
			mole.c[ipATC][ipCN] += rate_COmole.C_N_CN_NU_2;
			mole.b[ipCN] += rate_COmole.b_C_N_CN_NU;

		/*  >>refer   mol  C      +  S     =>   CS    +     NU    UMIST  */

			mole.c[ipATS][ipATC]  -= rate_COmole.C_S_CS_NU_1;
			mole.c[ipATC][ipATC]  -= rate_COmole.C_S_CS_NU_2;
			mole.b[ipATC]  -= rate_COmole.b_C_S_CS_NU;

			mole.c[ipATS][ipATS]  -= rate_COmole.C_S_CS_NU_1;
			mole.c[ipATC][ipATS] -= rate_COmole.C_S_CS_NU_2;
			mole.b[ipATS] -= rate_COmole.b_C_S_CS_NU;

			mole.c[ipATS][ipCS]  += rate_COmole.C_S_CS_NU_1;
			mole.c[ipATC][ipCS] += rate_COmole.C_S_CS_NU_2;
			mole.b[ipCS] += rate_COmole.b_C_S_CS_NU;

		/*  >>refer   mol  C+     +  S     =>   CS+   +     NU    UMIST  */

			mole.c[ipATS][ipCP]  -= rate_COmole.CP_S_CSP_NU_1;
			mole.c[ipCP][ipCP]  -= rate_COmole.CP_S_CSP_NU_2;
			mole.b[ipCP] -= rate_COmole.b_CP_S_CSP_NU;

			mole.c[ipATS][ipATS]  -= rate_COmole.CP_S_CSP_NU_1;
			mole.c[ipCP][ipATS]  -= rate_COmole.CP_S_CSP_NU_2;
			mole.b[ipATS] -= rate_COmole.b_CP_S_CSP_NU;

			mole.c[ipATN][ipCSP]  += rate_COmole.CP_S_CSP_NU_1;
			mole.c[ipCP][ipCSP]  += rate_COmole.CP_S_CSP_NU_2;
			mole.b[ipCSP] += rate_COmole.b_CP_S_CSP_NU;

		/*  >>refer   mol  N+     +  N     =>   N2+   +     NU    UMIST  */

			mole.c[ipATN][ipNP]  -= rate_COmole.NP_N_N2P_NU_1;
			mole.c[ipNP][ipNP]  -= rate_COmole.NP_N_N2P_NU_2;
			mole.b[ipNP] -= rate_COmole.b_NP_N_N2P_NU;

			mole.c[ipATN][ipATN]  -= rate_COmole.NP_N_N2P_NU_1;
			mole.c[ipNP][ipATN]  -= rate_COmole.NP_N_N2P_NU_2;
			mole.b[ipATN] -= rate_COmole.b_NP_N_N2P_NU;

			mole.c[ipATN][ipN2P]  += rate_COmole.NP_N_N2P_NU_1;
			mole.c[ipNP][ipN2P] += rate_COmole.NP_N_N2P_NU_2;
			mole.b[ipN2P] += rate_COmole.b_NP_N_N2P_NU;

		/*  >>refer   mol  CH     +  N+    =>   N     +     CH+   UMIST  */

			mole.c[ipNP][ipCH] -= rate_COmole.CH_NP_N_CHP_1;
			mole.c[ipCH][ipCH] -= rate_COmole.CH_NP_N_CHP_2;
			mole.b[ipCH] -= rate_COmole.b_CH_NP_N_CHP;
			
			mole.c[ipNP][ipNP] -= rate_COmole.CH_NP_N_CHP_1;
			mole.c[ipCH][ipNP] -= rate_COmole.CH_NP_N_CHP_2;
			mole.b[ipNP] -= rate_COmole.b_CH_NP_N_CHP;
			
			mole.c[ipNP][ipCHP] += rate_COmole.CH_NP_N_CHP_1;
			mole.c[ipCH][ipCHP] += rate_COmole.CH_NP_N_CHP_2;
			mole.b[ipCHP] += rate_COmole.b_CH_NP_N_CHP;
			
			mole.c[ipNP][ipATN] += rate_COmole.CH_NP_N_CHP_1;
			mole.c[ipCH][ipATN] += rate_COmole.CH_NP_N_CHP_2;
			mole.b[ipATN] += rate_COmole.b_CH_NP_N_CHP;

		/*  >>refer   mol  CH+    +  S     =>   S+    +     CH    UMIST  */
			
			mole.c[ipATS][ipCHP] -= rate_COmole.CHP_S_SP_CH_1;
			mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_S_SP_CH_2;
			mole.b[ipCHP] -= rate_COmole.b_CHP_S_SP_CH;
			
			mole.c[ipATS][ipATS] -= rate_COmole.CHP_S_SP_CH_1;
			mole.c[ipCHP][ipATS] -= rate_COmole.CHP_S_SP_CH_2;
			mole.b[ipATS] -= rate_COmole.b_CHP_S_SP_CH;
			
			mole.c[ipATS][ipSP] += rate_COmole.CHP_S_SP_CH_1;
			mole.c[ipCHP][ipSP] += rate_COmole.CHP_S_SP_CH_2;
			mole.b[ipSP] += rate_COmole.b_CHP_S_SP_CH;
			
			mole.c[ipATS][ipCH] += rate_COmole.CHP_S_SP_CH_1;
			mole.c[ipCHP][ipCH] += rate_COmole.CHP_S_SP_CH_2;
			mole.b[ipCH] += rate_COmole.b_CHP_S_SP_CH;

		/*  >>refer   mol  N+     +  CH2   =>   CH2+  +     N	    UMIST  */

			mole.c[ipCH2][ipNP] -= rate_COmole.NP_CH2_CH2P_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_CH2_CH2P_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_CH2_CH2P_N;

			mole.c[ipCH2][ipCH2] -= rate_COmole.NP_CH2_CH2P_N_1;
			mole.c[ipNP][ipCH2] -= rate_COmole.NP_CH2_CH2P_N_2;
			mole.b[ipCH2] -= rate_COmole.b_NP_CH2_CH2P_N;
						
			mole.c[ipCH2][ipCH2P] += rate_COmole.NP_CH2_CH2P_N_1;
			mole.c[ipNP][ipCH2P] += rate_COmole.NP_CH2_CH2P_N_2;
			mole.b[ipCH2P] += rate_COmole.b_NP_CH2_CH2P_N;
						
			mole.c[ipCH2][ipATN] += rate_COmole.NP_CH2_CH2P_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_CH2_CH2P_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_CH2_CH2P_N;
			
		/*  >>refer   mol  N+     +  CH4   =>   CH4+  +     N	    UMIST  */

			mole.c[ipCH4][ipNP] -= rate_COmole.NP_CH4_CH4P_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_CH4_CH4P_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_CH4_CH4P_N;
			
			mole.c[ipCH4][ipCH4] -= rate_COmole.NP_CH4_CH4P_N_1;
			mole.c[ipNP][ipCH4] -= rate_COmole.NP_CH4_CH4P_N_2;
			mole.b[ipCH4] -= rate_COmole.b_NP_CH4_CH4P_N;
			
			mole.c[ipCH4][ipCH4P] += rate_COmole.NP_CH4_CH4P_N_1;
			mole.c[ipNP][ipCH4P] += rate_COmole.NP_CH4_CH4P_N_2;
			mole.b[ipCH4P] += rate_COmole.b_NP_CH4_CH4P_N;
			
			mole.c[ipCH4][ipATN] += rate_COmole.NP_CH4_CH4P_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_CH4_CH4P_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_CH4_CH4P_N;
			
		/*  >>refer   mol  N+     +  OH    =>   OH+   +     N	    UMIST  */

			mole.c[ipOH][ipNP] -= rate_COmole.NP_OH_OHP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_OH_OHP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_OH_OHP_N;
			
			mole.c[ipOH][ipOH] -= rate_COmole.NP_OH_OHP_N_1;
			mole.c[ipNP][ipOH] -= rate_COmole.NP_OH_OHP_N_2;
			mole.b[ipOH] -= rate_COmole.b_NP_OH_OHP_N;
			
			mole.c[ipOH][ipOHP] += rate_COmole.NP_OH_OHP_N_1;
			mole.c[ipNP][ipOHP] += rate_COmole.NP_OH_OHP_N_2;
			mole.b[ipOHP] += rate_COmole.b_NP_OH_OHP_N;
			
			mole.c[ipOH][ipATN] += rate_COmole.NP_OH_OHP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_OH_OHP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_OH_OHP_N;
			
		/*  >>refer   mol  N+     +  H2O   =>   H2O+  +     N	    UMIST  */

			mole.c[ipH2O][ipNP] -= rate_COmole.NP_H2O_H2OP_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_H2O_H2OP_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_H2O_H2OP_N;
			
			mole.c[ipH2O][ipH2O] -= rate_COmole.NP_H2O_H2OP_N_1;
			mole.c[ipNP][ipH2O] -= rate_COmole.NP_H2O_H2OP_N_2;
			mole.b[ipH2O] -= rate_COmole.b_NP_H2O_H2OP_N;
			
			mole.c[ipH2O][ipH2OP] += rate_COmole.NP_H2O_H2OP_N_1;
			mole.c[ipNP][ipH2OP] += rate_COmole.NP_H2O_H2OP_N_2;
			mole.b[ipH2OP] += rate_COmole.b_NP_H2O_H2OP_N;
			
			mole.c[ipH2O][ipATN] += rate_COmole.NP_H2O_H2OP_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_H2O_H2OP_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_H2O_H2OP_N;
			
		/*  >>refer   mol  N+     +  O2    =>   O2+   +     N	    UMIST  */

			mole.c[ipO2][ipNP] -= rate_COmole.NP_O2_O2P_N_1;
			mole.c[ipNP][ipNP] -= rate_COmole.NP_O2_O2P_N_2;
			mole.b[ipNP] -= rate_COmole.b_NP_O2_O2P_N;
			
			mole.c[ipO2][ipO2] -= rate_COmole.NP_O2_O2P_N_1;
			mole.c[ipNP][ipO2] -= rate_COmole.NP_O2_O2P_N_2;
			mole.b[ipO2] -= rate_COmole.b_NP_O2_O2P_N;
			
			mole.c[ipO2][ipO2P] += rate_COmole.NP_O2_O2P_N_1;
			mole.c[ipNP][ipO2P] += rate_COmole.NP_O2_O2P_N_2;
			mole.b[ipO2P] += rate_COmole.b_NP_O2_O2P_N;
			
			mole.c[ipO2][ipATN] += rate_COmole.NP_O2_O2P_N_1;
			mole.c[ipNP][ipATN] += rate_COmole.NP_O2_O2P_N_2;
			mole.b[ipATN] += rate_COmole.b_NP_O2_O2P_N;

		/*  >>refer   mol  OH+    +  S     =>   S+    +     OH    UMIST  */

			mole.c[ipATS][ipOHP] -= rate_COmole.OHP_S_SP_OH_1;
			mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_S_SP_OH_2;
			mole.b[ipOHP] -= rate_COmole.b_OHP_S_SP_OH;

			mole.c[ipATS][ipATS] -= rate_COmole.OHP_S_SP_OH_1;
			mole.c[ipOHP][ipATS] -= rate_COmole.OHP_S_SP_OH_2;
			mole.b[ipATS] -= rate_COmole.b_OHP_S_SP_OH;
			
			mole.c[ipATS][ipSP] += rate_COmole.OHP_S_SP_OH_1;
			mole.c[ipOHP][ipSP] += rate_COmole.OHP_S_SP_OH_2;
			mole.b[ipSP] += rate_COmole.b_OHP_S_SP_OH;
			
			mole.c[ipATS][ipOH] += rate_COmole.OHP_S_SP_OH_1;
			mole.c[ipOHP][ipOH] += rate_COmole.OHP_S_SP_OH_2;
			mole.b[ipOH] += rate_COmole.b_OHP_S_SP_OH;

		/*  >>refer   mol  H2O+   +  S     =>   S+    +     H2O   UMIST  */

			mole.c[ipATS][ipH2OP] -= rate_COmole.H2OP_S_SP_H2O_1;
			mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_S_SP_H2O_2;
			mole.b[ipH2OP] -= rate_COmole.b_H2OP_S_SP_H2O;

			mole.c[ipATS][ipATS] -= rate_COmole.H2OP_S_SP_H2O_1;
			mole.c[ipH2OP][ipATS] -= rate_COmole.H2OP_S_SP_H2O_2;
			mole.b[ipATS] -= rate_COmole.b_H2OP_S_SP_H2O;
			
			mole.c[ipATS][ipSP] += rate_COmole.H2OP_S_SP_H2O_1;
			mole.c[ipH2OP][ipSP] += rate_COmole.H2OP_S_SP_H2O_2;
			mole.b[ipSP] += rate_COmole.b_H2OP_S_SP_H2O;
			
			mole.c[ipATS][ipH2O] += rate_COmole.H2OP_S_SP_H2O_1;
			mole.c[ipH2OP][ipH2O] += rate_COmole.H2OP_S_SP_H2O_2;
			mole.b[ipH2O] += rate_COmole.b_H2OP_S_SP_H2O;

		/*  >>refer   mol  SI     +  S+    =>   S     +     SI+   UMIST  */

			mole.c[ipSP][ipATSI] -= rate_COmole.SI_SP_S_SIP_1;
			mole.c[ipATSI][ipATSI] -= rate_COmole.SI_SP_S_SIP_2;
			mole.b[ipATSI] -= rate_COmole.b_SI_SP_S_SIP;
			
			mole.c[ipSP][ipSP] -= rate_COmole.SI_SP_S_SIP_1;
			mole.c[ipATSI][ipSP] -= rate_COmole.SI_SP_S_SIP_2;
			mole.b[ipSP] -= rate_COmole.b_SI_SP_S_SIP;
			
			mole.c[ipSP][ipSIP] += rate_COmole.SI_SP_S_SIP_1;
			mole.c[ipATSI][ipSIP] += rate_COmole.SI_SP_S_SIP_2;
			mole.b[ipSIP] += rate_COmole.b_SI_SP_S_SIP;

			mole.c[ipSP][ipATS] += rate_COmole.SI_SP_S_SIP_1;
			mole.c[ipATSI][ipATS] += rate_COmole.SI_SP_S_SIP_2;
			mole.b[ipATS] += rate_COmole.b_SI_SP_S_SIP;
			
		/*  >>refer   mol  O2+    +  S     =>   S+    +     O2    UMIST  */

			mole.c[ipATS][ipO2P] -= rate_COmole.O2P_S_SP_O2_1;
			mole.c[ipO2P][ipO2P] -= rate_COmole.O2P_S_SP_O2_2;
			mole.b[ipO2P] -= rate_COmole.b_O2P_S_SP_O2;

			mole.c[ipATS][ipATS] -= rate_COmole.O2P_S_SP_O2_1;
			mole.c[ipO2P][ipATS] -= rate_COmole.O2P_S_SP_O2_2;
			mole.b[ipATS] -= rate_COmole.b_O2P_S_SP_O2;
			
			mole.c[ipATS][ipSP] += rate_COmole.O2P_S_SP_O2_1;
			mole.c[ipO2P][ipSP] += rate_COmole.O2P_S_SP_O2_2;
			mole.b[ipSP] += rate_COmole.b_O2P_S_SP_O2;
			
			mole.c[ipATS][ipO2] += rate_COmole.O2P_S_SP_O2_1;
			mole.c[ipO2P][ipO2] += rate_COmole.O2P_S_SP_O2_2;
			mole.b[ipO2] += rate_COmole.b_O2P_S_SP_O2;

		/* >>chng Mar 23, 2005 -- Add Chlorine to network */

		/*  >>refer   mol  O    +  CCl     =>   ClO    +     C    UMIST  */

		mole.c[ipCCl][ipCCl] -= rate_COmole.O_CCl_ClO_C_2;
		mole.c[ipATO][ipCCl] -= rate_COmole.O_CCl_ClO_C_1;
		mole.b[ipCCl] -= rate_COmole.b_O_CCl_ClO_C;

		mole.c[ipCCl][ipATO] -= rate_COmole.O_CCl_ClO_C_2;
		mole.c[ipATO][ipATO] -= rate_COmole.O_CCl_ClO_C_1;
		mole.b[ipATO] -= rate_COmole.b_O_CCl_ClO_C;

		mole.c[ipCCl][ipClO] += rate_COmole.O_CCl_ClO_C_2;
		mole.c[ipATO][ipClO] += rate_COmole.O_CCl_ClO_C_1;
		mole.b[ipClO] += rate_COmole.b_O_CCl_ClO_C;

		mole.c[ipCCl][ipATC] += rate_COmole.O_CCl_ClO_C_2;
		mole.c[ipATO][ipATC] += rate_COmole.O_CCl_ClO_C_1;
		mole.b[ipATC] += rate_COmole.b_O_CCl_ClO_C;

		/*  >>refer   mol  O    +  ClO     =>   Cl    +     O2    UMIST  */

		mole.c[ipClO][ipClO] -= rate_COmole.O_ClO_Cl_O2_2;
		mole.c[ipATO][ipClO] -= rate_COmole.O_ClO_Cl_O2_1;
		mole.b[ipClO] -= rate_COmole.b_O_ClO_Cl_O2;

		mole.c[ipClO][ipATO] -= rate_COmole.O_ClO_Cl_O2_2;
		mole.c[ipATO][ipATO] -= rate_COmole.O_ClO_Cl_O2_1;
		mole.b[ipATO] -= rate_COmole.b_O_ClO_Cl_O2;
		
		mole.c[ipClO][ipATCl] += rate_COmole.O_ClO_Cl_O2_2;
		mole.c[ipATO][ipATCl] += rate_COmole.O_ClO_Cl_O2_1;
		mole.b[ipATCl] += rate_COmole.b_O_ClO_Cl_O2;

		mole.c[ipClO][ipO2] += rate_COmole.O_ClO_Cl_O2_2;
		mole.c[ipATO][ipO2] += rate_COmole.O_ClO_Cl_O2_1;
		mole.b[ipO2] += rate_COmole.b_O_ClO_Cl_O2;


		/*  >>refer   mol  C+    +  HCl     =>   CCl+    +     H    UMIST  */

		mole.c[ipHCl][ipHCl] -= rate_COmole.CP_HCl_CClP_H_2;
		mole.c[ipCP][ipHCl] -= rate_COmole.CP_HCl_CClP_H_1;
		mole.b[ipHCl] -= rate_COmole.b_CP_HCl_CClP_H;

		mole.c[ipHCl][ipCP] -= rate_COmole.CP_HCl_CClP_H_2;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_HCl_CClP_H_1;
		mole.b[ipCP] -= rate_COmole.b_CP_HCl_CClP_H;

		mole.c[ipHCl][ipCClP] += rate_COmole.CP_HCl_CClP_H_2;
		mole.c[ipCP][ipCClP] += rate_COmole.CP_HCl_CClP_H_1;
		mole.b[ipCClP] += rate_COmole.b_CP_HCl_CClP_H;

		/*  >>refer   mol  CH3+    +  HCl     =>   H2CCl+    +     H2    UMIST  */

		mole.c[ipHCl][ipHCl] -= rate_COmole.CH3P_HCl_H2CClP_H2_2;
		mole.c[ipCH3P][ipHCl] -= rate_COmole.CH3P_HCl_H2CClP_H2_1;
		mole.b[ipHCl] -= rate_COmole.b_CH3P_HCl_H2CClP_H2;

		mole.c[ipHCl][ipCH3P] += rate_COmole.CH3P_HCl_H2CClP_H2_2;
		mole.c[ipCH3P][ipCH3P] += rate_COmole.CH3P_HCl_H2CClP_H2_1;
		mole.b[ipCH3P] += rate_COmole.b_CH3P_HCl_H2CClP_H2;

		mole.c[ipHCl][ipH2CClP] += rate_COmole.CH3P_HCl_H2CClP_H2_2;
		mole.c[ipCH3P][ipH2CClP] += rate_COmole.CH3P_HCl_H2CClP_H2_1;
		mole.b[ipH2CClP] += rate_COmole.b_CH3P_HCl_H2CClP_H2;

		/*  >>refer   mol  H2O    +  H2Cl+     =>   HCl    +     H3O+    UMIST  */

		mole.c[ipH2ClP][ipH2ClP] -= rate_COmole.H2O_H2ClP_HCl_H3OP_2;
		mole.c[ipH2O][ipH2ClP] -= rate_COmole.H2O_H2ClP_HCl_H3OP_1;
		mole.b[ipH2ClP] -= rate_COmole.b_H2O_H2ClP_HCl_H3OP;

		mole.c[ipH2ClP][ipH2O] -= rate_COmole.H2O_H2ClP_HCl_H3OP_2;
		mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_H2ClP_HCl_H3OP_1;
		mole.b[ipH2O] -= rate_COmole.b_H2O_H2ClP_HCl_H3OP;

		mole.c[ipH2ClP][ipHCl] += rate_COmole.H2O_H2ClP_HCl_H3OP_2;
		mole.c[ipH2O][ipHCl] += rate_COmole.H2O_H2ClP_HCl_H3OP_1;
		mole.b[ipHCl] += rate_COmole.b_H2O_H2ClP_HCl_H3OP;

		mole.c[ipH2ClP][ipH3OP] += rate_COmole.H2O_H2ClP_HCl_H3OP_2;
		mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_H2ClP_HCl_H3OP_1;
		mole.b[ipH3OP] += rate_COmole.b_H2O_H2ClP_HCl_H3OP;

		/*  >>refer   mol  C+    +  CCl     =>   CCl+    +     C    UMIST  */

		mole.c[ipCCl][ipCCl] -= rate_COmole.CP_CCl_CClP_C_2;
		mole.c[ipCP][ipCCl] -= rate_COmole.CP_CCl_CClP_C_1;
		mole.b[ipCCl] -= rate_COmole.b_CP_CCl_CClP_C,

		mole.c[ipCCl][ipCP] -= rate_COmole.CP_CCl_CClP_C_2;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CCl_CClP_C_1;
		mole.b[ipCP] -= rate_COmole.b_CP_CCl_CClP_C;

		mole.c[ipCCl][ipCClP] += rate_COmole.CP_CCl_CClP_C_2;
		mole.c[ipCP][ipCClP] += rate_COmole.CP_CCl_CClP_C_1;
		mole.b[ipCClP] += rate_COmole.b_CP_CCl_CClP_C;

		mole.c[ipCCl][ipATC] += rate_COmole.CP_CCl_CClP_C_2;
		mole.c[ipCP][ipATC] += rate_COmole.CP_CCl_CClP_C_1;
		mole.b[ipATC] += rate_COmole.b_CP_CCl_CClP_C;

		/*  >>refer   mol  C+    +  ClO     =>   ClO+    +     C    UMIST  */

		mole.c[ipClO][ipClO] -= rate_COmole.CP_ClO_ClOP_C_2;
		mole.c[ipCP][ipClO] -= rate_COmole.CP_ClO_ClOP_C_1;
		mole.b[ipClO] -= rate_COmole.b_CP_ClO_ClOP_C;

		mole.c[ipClO][ipCP] -= rate_COmole.CP_ClO_ClOP_C_2;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_ClO_ClOP_C_1;
		mole.b[ipCP] -= rate_COmole.b_CP_ClO_ClOP_C;

		mole.c[ipClO][ipClOP] += rate_COmole.CP_ClO_ClOP_C_2;
		mole.c[ipCP][ipClOP] += rate_COmole.CP_ClO_ClOP_C_1;
		mole.b[ipClOP] += rate_COmole.b_CP_ClO_ClOP_C;

		mole.c[ipClO][ipATC] += rate_COmole.CP_ClO_ClOP_C_2;
		mole.c[ipCP][ipATC] += rate_COmole.CP_ClO_ClOP_C_1;
		mole.b[ipATC] += rate_COmole.b_CP_ClO_ClOP_C;

		/*  >>refer   mol  O2    +  Cl+     =>   Cl    +     O2+    UMIST  */

		mole.c[ipClP][ipClP] -= rate_COmole.O2_ClP_Cl_O2P_2;
		mole.c[ipO2][ipClP] -= rate_COmole.O2_ClP_Cl_O2P_1;
		mole.b[ipClP] -= rate_COmole.b_O2_ClP_Cl_O2P;

		mole.c[ipClP][ipO2] -= rate_COmole.O2_ClP_Cl_O2P_2;
		mole.c[ipO2][ipO2] -= rate_COmole.O2_ClP_Cl_O2P_1;
		mole.b[ipO2] -= rate_COmole.b_O2_ClP_Cl_O2P;

		mole.c[ipClP][ipATCl] += rate_COmole.O2_ClP_Cl_O2P_2;
		mole.c[ipO2][ipATCl] += rate_COmole.O2_ClP_Cl_O2P_1;
		mole.b[ipATCl] += rate_COmole.b_O2_ClP_Cl_O2P;

		mole.c[ipClP][ipO2P] += rate_COmole.O2_ClP_Cl_O2P_2;
		mole.c[ipO2][ipO2P] += rate_COmole.O2_ClP_Cl_O2P_1;
		mole.b[ipO2P] += rate_COmole.b_O2_ClP_Cl_O2P;


		/*  >>refer  mol  C   +   NH2   =   HNC   +   H   */

		mole.c[ipNH2][ipATC] -= rate_COmole.C_NH2_HNC_H_1; 
		mole.c[ipATC][ipATC] -= rate_COmole.C_NH2_HNC_H_2; 
		mole.b[ipATC] -= rate_COmole.b_C_NH2_HNC_H;

		mole.c[ipNH2][ipNH2] -= rate_COmole.C_NH2_HNC_H_1; 
		mole.c[ipATC][ipNH2] -= rate_COmole.C_NH2_HNC_H_2; 
		mole.b[ipNH2] -= rate_COmole.b_C_NH2_HNC_H;

		mole.c[ipNH2][ipHNC] += rate_COmole.C_NH2_HNC_H_1; 
		mole.c[ipATC][ipHNC] += rate_COmole.C_NH2_HNC_H_2; 
		mole.b[ipHNC] += rate_COmole.b_C_NH2_HNC_H;

		/*  >>refer  mol  N   +   CH2   =   HNC   +   H   */
		
		mole.c[ipCH2][ipATN] -= rate_COmole.N_CH2_HNC_H_1; 
		mole.c[ipATN][ipATN] -= rate_COmole.N_CH2_HNC_H_2; 
		mole.b[ipATN] -= rate_COmole.b_N_CH2_HNC_H;

		mole.c[ipCH2][ipCH2] -= rate_COmole.N_CH2_HNC_H_1; 
		mole.c[ipATN][ipCH2] -= rate_COmole.N_CH2_HNC_H_2; 
		mole.b[ipCH2] -= rate_COmole.b_N_CH2_HNC_H;

		mole.c[ipCH2][ipHNC] += rate_COmole.N_CH2_HNC_H_1; 
		mole.c[ipATN][ipHNC] += rate_COmole.N_CH2_HNC_H_2; 
		mole.b[ipHNC] += rate_COmole.b_N_CH2_HNC_H;

		/*  >>refer  mol  CHP   +   HNC   =   HCNHP   +   C   */

		mole.c[ipHNC][ipCHP] -= rate_COmole.CHP_HNC_HCNHP_C_1; 
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_HNC_HCNHP_C_2; 
		mole.b[ipCHP] -= rate_COmole.b_CHP_HNC_HCNHP_C;

		mole.c[ipHNC][ipHNC] -= rate_COmole.CHP_HNC_HCNHP_C_1; 
		mole.c[ipCHP][ipHNC] -= rate_COmole.CHP_HNC_HCNHP_C_2; 
		mole.b[ipHNC] -= rate_COmole.b_CHP_HNC_HCNHP_C;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.CHP_HNC_HCNHP_C_1; 
		mole.c[ipCHP][ipHCNHP] += rate_COmole.CHP_HNC_HCNHP_C_2; 
		mole.b[ipHCNHP] += rate_COmole.b_CHP_HNC_HCNHP_C;

		mole.c[ipHNC][ipATC] += rate_COmole.CHP_HNC_HCNHP_C_1; 
		mole.c[ipCHP][ipATC] += rate_COmole.CHP_HNC_HCNHP_C_2; 
		mole.b[ipATC] += rate_COmole.b_CHP_HNC_HCNHP_C;

		/*  >>refer  mol  CH   +   HCNHP   =   HNC   +   CH2P   */

		mole.c[ipHCNHP][ipCH] -= rate_COmole.CH_HCNHP_HNC_CH2P_1; 
		mole.c[ipCH][ipCH] -= rate_COmole.CH_HCNHP_HNC_CH2P_2; 
		mole.b[ipCH] -= rate_COmole.b_CH_HCNHP_HNC_CH2P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.CH_HCNHP_HNC_CH2P_1; 
		mole.c[ipCH][ipHCNHP] -= rate_COmole.CH_HCNHP_HNC_CH2P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_CH_HCNHP_HNC_CH2P;

		mole.c[ipHCNHP][ipHNC] += rate_COmole.CH_HCNHP_HNC_CH2P_1; 
		mole.c[ipCH][ipHNC] += rate_COmole.CH_HCNHP_HNC_CH2P_2; 
		mole.b[ipHNC] += rate_COmole.b_CH_HCNHP_HNC_CH2P;

		mole.c[ipHCNHP][ipCH2P] += rate_COmole.CH_HCNHP_HNC_CH2P_1; 
		mole.c[ipCH][ipCH2P] += rate_COmole.CH_HCNHP_HNC_CH2P_2; 
		mole.b[ipCH2P] += rate_COmole.b_CH_HCNHP_HNC_CH2P;

		/*  >>refer  mol  CH2   +   HCNHP   =   HNC   +   CH3P   */

		mole.c[ipHCNHP][ipCH2] -= rate_COmole.CH2_HCNHP_HNC_CH3P_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_HCNHP_HNC_CH3P_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_HCNHP_HNC_CH3P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.CH2_HCNHP_HNC_CH3P_1;
		mole.c[ipCH2][ipHCNHP] -= rate_COmole.CH2_HCNHP_HNC_CH3P_2;
		mole.b[ipHCNHP] -= rate_COmole.b_CH2_HCNHP_HNC_CH3P;

		mole.c[ipHCNHP][ipHNC] += rate_COmole.CH2_HCNHP_HNC_CH3P_1;
		mole.c[ipCH2][ipHNC] += rate_COmole.CH2_HCNHP_HNC_CH3P_2;
		mole.b[ipHNC] += rate_COmole.b_CH2_HCNHP_HNC_CH3P;

		mole.c[ipHCNHP][ipCH3P] += rate_COmole.CH2_HCNHP_HNC_CH3P_1;
		mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_HCNHP_HNC_CH3P_2;
		mole.b[ipCH3P] += rate_COmole.b_CH2_HCNHP_HNC_CH3P;

		/*  >>refer  mol  NHP   +   HNC   =   HCNHP   +   N   */
		
		mole.c[ipHNC][ipNHP] -= rate_COmole.NHP_HNC_HCNHP_N_1;
		mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_HNC_HCNHP_N_2; 
		mole.b[ipNHP] -= rate_COmole.b_NHP_HNC_HCNHP_N;

		mole.c[ipHNC][ipHNC] -= rate_COmole.NHP_HNC_HCNHP_N_1; 
		mole.c[ipNHP][ipHNC] -= rate_COmole.NHP_HNC_HCNHP_N_2; 
		mole.b[ipHNC] -= rate_COmole.b_NHP_HNC_HCNHP_N;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.NHP_HNC_HCNHP_N_1; 
		mole.c[ipNHP][ipHCNHP] += rate_COmole.NHP_HNC_HCNHP_N_2; 
		mole.b[ipHCNHP] += rate_COmole.b_NHP_HNC_HCNHP_N;

		mole.c[ipHNC][ipATN] += rate_COmole.NHP_HNC_HCNHP_N_1; 
		mole.c[ipNHP][ipATN] += rate_COmole.NHP_HNC_HCNHP_N_2; 
		mole.b[ipATN] += rate_COmole.b_NHP_HNC_HCNHP_N;

		/*  >>refer  mol  NH2P   +   HNC   =   HCNHP   +   NH   */
		
		mole.c[ipHNC][ipNH2P] -= rate_COmole.NH2P_HNC_HCNHP_NH_1; 
		mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_HNC_HCNHP_NH_2; 
		mole.b[ipNH2P] -= rate_COmole.b_NH2P_HNC_HCNHP_NH;

		mole.c[ipHNC][ipHNC] -= rate_COmole.NH2P_HNC_HCNHP_NH_1; 
		mole.c[ipNH2P][ipHNC] -= rate_COmole.NH2P_HNC_HCNHP_NH_2; 
		mole.b[ipHNC] -= rate_COmole.b_NH2P_HNC_HCNHP_NH;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.NH2P_HNC_HCNHP_NH_1; 
		mole.c[ipNH2P][ipHCNHP] += rate_COmole.NH2P_HNC_HCNHP_NH_2; 
		mole.b[ipHCNHP] += rate_COmole.b_NH2P_HNC_HCNHP_NH;

		mole.c[ipHNC][ipNH] += rate_COmole.NH2P_HNC_HCNHP_NH_1; 
		mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_HNC_HCNHP_NH_2; 
		mole.b[ipNH] += rate_COmole.b_NH2P_HNC_HCNHP_NH;

		/*  >>refer  mol  NH2   +   HCNHP   =   HNC   +   NH3P   */

		mole.c[ipHCNHP][ipNH2] -= rate_COmole.NH2_HCNHP_HNC_NH3P_1; 
		mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_HCNHP_HNC_NH3P_2; 
		mole.b[ipNH2] -= rate_COmole.b_NH2_HCNHP_HNC_NH3P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.NH2_HCNHP_HNC_NH3P_1; 
		mole.c[ipNH2][ipHCNHP] -= rate_COmole.NH2_HCNHP_HNC_NH3P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_NH2_HCNHP_HNC_NH3P;

		mole.c[ipHCNHP][ipHNC] += rate_COmole.NH2_HCNHP_HNC_NH3P_1; 
		mole.c[ipNH2][ipHNC] += rate_COmole.NH2_HCNHP_HNC_NH3P_2; 
		mole.b[ipHNC] += rate_COmole.b_NH2_HCNHP_HNC_NH3P;

		mole.c[ipHCNHP][ipNH3P] += rate_COmole.NH2_HCNHP_HNC_NH3P_1; 
		mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_HCNHP_HNC_NH3P_2; 
		mole.b[ipNH3P] += rate_COmole.b_NH2_HCNHP_HNC_NH3P;

		/*  >>refer  mol  OHP   +   HNC   =   HCNHP   +   O   */
		
		mole.c[ipHNC][ipOHP] -= rate_COmole.OHP_HNC_HCNHP_O_1; 
		mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_HNC_HCNHP_O_2; 
		mole.b[ipOHP] -= rate_COmole.b_OHP_HNC_HCNHP_O;

		mole.c[ipHNC][ipHNC] -= rate_COmole.OHP_HNC_HCNHP_O_1; 
		mole.c[ipOHP][ipHNC] -= rate_COmole.OHP_HNC_HCNHP_O_2; 
		mole.b[ipHNC] -= rate_COmole.b_OHP_HNC_HCNHP_O;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.OHP_HNC_HCNHP_O_1; 
		mole.c[ipOHP][ipHCNHP] += rate_COmole.OHP_HNC_HCNHP_O_2; 
		mole.b[ipHCNHP] += rate_COmole.b_OHP_HNC_HCNHP_O;

		mole.c[ipHNC][ipATO] += rate_COmole.OHP_HNC_HCNHP_O_1; 
		mole.c[ipOHP][ipATO] += rate_COmole.OHP_HNC_HCNHP_O_2; 
		mole.b[ipATO] += rate_COmole.b_OHP_HNC_HCNHP_O;

		/*  >>refer  mol  NH3   +   HCNHP   =   HNC   +   NH4P   */
	
		mole.c[ipHCNHP][ipNH3] -= rate_COmole.NH3_HCNHP_HNC_NH4P_1; 
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HCNHP_HNC_NH4P_2; 
		mole.b[ipNH3] -= rate_COmole.b_NH3_HCNHP_HNC_NH4P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.NH3_HCNHP_HNC_NH4P_1; 
		mole.c[ipNH3][ipHCNHP] -= rate_COmole.NH3_HCNHP_HNC_NH4P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_NH3_HCNHP_HNC_NH4P;

		mole.c[ipHCNHP][ipHNC] += rate_COmole.NH3_HCNHP_HNC_NH4P_1; 
		mole.c[ipNH3][ipHNC] += rate_COmole.NH3_HCNHP_HNC_NH4P_2; 
		mole.b[ipHNC] += rate_COmole.b_NH3_HCNHP_HNC_NH4P;

		mole.c[ipHCNHP][ipNH4P] += rate_COmole.NH3_HCNHP_HNC_NH4P_1; 
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_HCNHP_HNC_NH4P_2; 
		mole.b[ipNH4P] += rate_COmole.b_NH3_HCNHP_HNC_NH4P;

		/*  >>refer  mol  CH5P   +   HNC   =   HCNHP   +   CH4   */
	
		mole.c[ipHNC][ipCH5P] -= rate_COmole.CH5P_HNC_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_HNC_HCNHP_CH4_2; 
		mole.b[ipCH5P] -= rate_COmole.b_CH5P_HNC_HCNHP_CH4;

		mole.c[ipHNC][ipHNC] -= rate_COmole.CH5P_HNC_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipHNC] -= rate_COmole.CH5P_HNC_HCNHP_CH4_2; 
		mole.b[ipHNC] -= rate_COmole.b_CH5P_HNC_HCNHP_CH4;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.CH5P_HNC_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipHCNHP] += rate_COmole.CH5P_HNC_HCNHP_CH4_2; 
		mole.b[ipHCNHP] += rate_COmole.b_CH5P_HNC_HCNHP_CH4;

		mole.c[ipHNC][ipCH4] += rate_COmole.CH5P_HNC_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_HNC_HCNHP_CH4_2; 
		mole.b[ipCH4] += rate_COmole.b_CH5P_HNC_HCNHP_CH4;

		/*  >>refer  mol  H2OP   +   HNC   =   HCNHP   +   OH   */

		mole.c[ipHNC][ipH2OP] -= rate_COmole.H2OP_HNC_HCNHP_OH_1; 
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_HNC_HCNHP_OH_2; 
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_HNC_HCNHP_OH;

		mole.c[ipHNC][ipHNC] -= rate_COmole.H2OP_HNC_HCNHP_OH_1; 
		mole.c[ipH2OP][ipHNC] -= rate_COmole.H2OP_HNC_HCNHP_OH_2; 
		mole.b[ipHNC] -= rate_COmole.b_H2OP_HNC_HCNHP_OH;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.H2OP_HNC_HCNHP_OH_1; 
		mole.c[ipH2OP][ipHCNHP] += rate_COmole.H2OP_HNC_HCNHP_OH_2; 
		mole.b[ipHCNHP] += rate_COmole.b_H2OP_HNC_HCNHP_OH;

		mole.c[ipHNC][ipOH] += rate_COmole.H2OP_HNC_HCNHP_OH_1; 
		mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_HNC_HCNHP_OH_2; 
		mole.b[ipOH] += rate_COmole.b_H2OP_HNC_HCNHP_OH;

		/*  >>refer  mol  H3OP   +   HNC   =   HCNHP   +   H2O   */
	
		mole.c[ipHNC][ipH3OP] -= rate_COmole.H3OP_HNC_HCNHP_H2O_1;
		mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_HNC_HCNHP_H2O_2; 
		mole.b[ipH3OP] -= rate_COmole.b_H3OP_HNC_HCNHP_H2O;

		mole.c[ipHNC][ipHNC] -= rate_COmole.H3OP_HNC_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipHNC] -= rate_COmole.H3OP_HNC_HCNHP_H2O_2; 
		mole.b[ipHNC] -= rate_COmole.b_H3OP_HNC_HCNHP_H2O;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.H3OP_HNC_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipHCNHP] += rate_COmole.H3OP_HNC_HCNHP_H2O_2; 
		mole.b[ipHCNHP] += rate_COmole.b_H3OP_HNC_HCNHP_H2O;

		mole.c[ipHNC][ipH2O] += rate_COmole.H3OP_HNC_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_HNC_HCNHP_H2O_2; 
		mole.b[ipH2O] += rate_COmole.b_H3OP_HNC_HCNHP_H2O;

		/*  >>refer  mol  HCNP   +   HNC   =   HCNHP   +   CN   */
		
		mole.c[ipHNC][ipHCNP] -= rate_COmole.HCNP_HNC_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_HNC_HCNHP_CN_2; 
		mole.b[ipHCNP] -= rate_COmole.b_HCNP_HNC_HCNHP_CN;

		mole.c[ipHNC][ipHNC] -= rate_COmole.HCNP_HNC_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHNC] -= rate_COmole.HCNP_HNC_HCNHP_CN_2; 
		mole.b[ipHNC] -= rate_COmole.b_HCNP_HNC_HCNHP_CN;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.HCNP_HNC_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHCNHP] += rate_COmole.HCNP_HNC_HCNHP_CN_2; 
		mole.b[ipHCNHP] += rate_COmole.b_HCNP_HNC_HCNHP_CN;

		mole.c[ipHNC][ipCN] += rate_COmole.HCNP_HNC_HCNHP_CN_1; 
		mole.c[ipHCNP][ipCN] += rate_COmole.HCNP_HNC_HCNHP_CN_2; 
		mole.b[ipCN] += rate_COmole.b_HCNP_HNC_HCNHP_CN;

		/*  >>refer  mol  HNC   +   HNOP   =   NO   +   HCNHP   */

		mole.c[ipHNOP][ipHNC] -= rate_COmole.HNC_HNOP_NO_HCNHP_1; 
		mole.c[ipHNC][ipHNC] -= rate_COmole.HNC_HNOP_NO_HCNHP_2; 
		mole.b[ipHNC] -= rate_COmole.b_HNC_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipHNOP] -= rate_COmole.HNC_HNOP_NO_HCNHP_1; 
		mole.c[ipHNC][ipHNOP] -= rate_COmole.HNC_HNOP_NO_HCNHP_2; 
		mole.b[ipHNOP] -= rate_COmole.b_HNC_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipNO] += rate_COmole.HNC_HNOP_NO_HCNHP_1; 
		mole.c[ipHNC][ipNO] += rate_COmole.HNC_HNOP_NO_HCNHP_2; 
		mole.b[ipNO] += rate_COmole.b_HNC_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipHCNHP] += rate_COmole.HNC_HNOP_NO_HCNHP_1; 
		mole.c[ipHNC][ipHCNHP] += rate_COmole.HNC_HNOP_NO_HCNHP_2; 
		mole.b[ipHCNHP] += rate_COmole.b_HNC_HNOP_NO_HCNHP;

		/*  >>refer  mol  HNC   +   HSP   =   S   +   HCNHP   */
		
		mole.c[ipHSP][ipHNC] -= rate_COmole.HNC_HSP_S_HCNHP_1; 
		mole.c[ipHNC][ipHNC] -= rate_COmole.HNC_HSP_S_HCNHP_2; 
		mole.b[ipHNC] -= rate_COmole.b_HNC_HSP_S_HCNHP;

		mole.c[ipHSP][ipHSP] -= rate_COmole.HNC_HSP_S_HCNHP_1; 
		mole.c[ipHNC][ipHSP] -= rate_COmole.HNC_HSP_S_HCNHP_2; 
		mole.b[ipHSP] -= rate_COmole.b_HNC_HSP_S_HCNHP;

		mole.c[ipHSP][ipATS] += rate_COmole.HNC_HSP_S_HCNHP_1; 
		mole.c[ipHNC][ipATS] += rate_COmole.HNC_HSP_S_HCNHP_2; 
		mole.b[ipATS] += rate_COmole.b_HNC_HSP_S_HCNHP;

		mole.c[ipHSP][ipHCNHP] += rate_COmole.HNC_HSP_S_HCNHP_1; 
		mole.c[ipHNC][ipHCNHP] += rate_COmole.HNC_HSP_S_HCNHP_2; 
		mole.b[ipHCNHP] += rate_COmole.b_HNC_HSP_S_HCNHP;

		/*  >>refer  mol  CHP   +   HCN   =   HCNHP   +   C   */
		
		mole.c[ipHCN][ipCHP] -= rate_COmole.CHP_HCN_HCNHP_C_1; 
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_HCN_HCNHP_C_2; 
		mole.b[ipCHP] -= rate_COmole.b_CHP_HCN_HCNHP_C;
			
		mole.c[ipHCN][ipHCN] -= rate_COmole.CHP_HCN_HCNHP_C_1; 
		mole.c[ipCHP][ipHCN] -= rate_COmole.CHP_HCN_HCNHP_C_2; 
		mole.b[ipHCN] -= rate_COmole.b_CHP_HCN_HCNHP_C;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.CHP_HCN_HCNHP_C_1; 
		mole.c[ipCHP][ipHCNHP] += rate_COmole.CHP_HCN_HCNHP_C_2; 
		mole.b[ipHCNHP] += rate_COmole.b_CHP_HCN_HCNHP_C;
			
		mole.c[ipHCN][ipATC] += rate_COmole.CHP_HCN_HCNHP_C_1; 
		mole.c[ipCHP][ipATC] += rate_COmole.CHP_HCN_HCNHP_C_2; 
		mole.b[ipATC] += rate_COmole.b_CHP_HCN_HCNHP_C;

		/*  >>refer  mol  CH   +   HCNHP   =   HCN   +   CH2P   */
		
		mole.c[ipHCNHP][ipCH] -= rate_COmole.CH_HCNHP_HCN_CH2P_1; 
		mole.c[ipCH][ipCH] -= rate_COmole.CH_HCNHP_HCN_CH2P_2; 
		mole.b[ipCH] -= rate_COmole.b_CH_HCNHP_HCN_CH2P;
		
		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.CH_HCNHP_HCN_CH2P_1; 
		mole.c[ipCH][ipHCNHP] -= rate_COmole.CH_HCNHP_HCN_CH2P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_CH_HCNHP_HCN_CH2P;
		
		mole.c[ipHCNHP][ipHCN] += rate_COmole.CH_HCNHP_HCN_CH2P_1; 
		mole.c[ipCH][ipHCN] += rate_COmole.CH_HCNHP_HCN_CH2P_2; 
		mole.b[ipHCN] += rate_COmole.b_CH_HCNHP_HCN_CH2P;

		mole.c[ipHCNHP][ipCH2P] += rate_COmole.CH_HCNHP_HCN_CH2P_1; 
		mole.c[ipCH][ipCH2P] += rate_COmole.CH_HCNHP_HCN_CH2P_2; 
		mole.b[ipCH2P] += rate_COmole.b_CH_HCNHP_HCN_CH2P;

		/*  >>refer  mol  NP   +   CH4   =   HCNHP   +   H   +   H */

		mole.c[ipCH4][ipNP] -= rate_COmole.NP_CH4_HCNHP_H_H_1; 
		mole.c[ipNP][ipNP] -= rate_COmole.NP_CH4_HCNHP_H_H_2; 
		mole.b[ipNP] -= rate_COmole.b_NP_CH4_HCNHP_H_H;

		mole.c[ipCH4][ipCH4] -= rate_COmole.NP_CH4_HCNHP_H_H_1; 
		mole.c[ipNP][ipCH4] -= rate_COmole.NP_CH4_HCNHP_H_H_2; 
		mole.b[ipCH4] -= rate_COmole.b_NP_CH4_HCNHP_H_H;

		mole.c[ipCH4][ipHCNHP] += rate_COmole.NP_CH4_HCNHP_H_H_1; 
		mole.c[ipNP][ipHCNHP] += rate_COmole.NP_CH4_HCNHP_H_H_2; 
		mole.b[ipHCNHP] += rate_COmole.b_NP_CH4_HCNHP_H_H;

		/*  >>refer  mol  CH2   +   HCNHP   =   HCN   +   CH3P   */
		
		mole.c[ipHCNHP][ipCH2] -= rate_COmole.CH2_HCNHP_HCN_CH3P_1; 
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_HCNHP_HCN_CH3P_2; 
		mole.b[ipCH2] -= rate_COmole.b_CH2_HCNHP_HCN_CH3P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.CH2_HCNHP_HCN_CH3P_1; 
		mole.c[ipCH2][ipHCNHP] -= rate_COmole.CH2_HCNHP_HCN_CH3P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_CH2_HCNHP_HCN_CH3P;

		mole.c[ipHCNHP][ipHCN] += rate_COmole.CH2_HCNHP_HCN_CH3P_1; 
		mole.c[ipCH2][ipHCN] += rate_COmole.CH2_HCNHP_HCN_CH3P_2; 
		mole.b[ipHCN] += rate_COmole.b_CH2_HCNHP_HCN_CH3P;

		mole.c[ipHCNHP][ipCH3P] += rate_COmole.CH2_HCNHP_HCN_CH3P_1; 
		mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_HCNHP_HCN_CH3P_2; 
		mole.b[ipCH3P] += rate_COmole.b_CH2_HCNHP_HCN_CH3P;

		/*  >>refer  mol  NH   +   CH3P   =   HCNHP   +   H2   */
		
		mole.c[ipCH3P][ipNH] -= rate_COmole.NH_CH3P_HCNHP_H2_1; 
		mole.c[ipNH][ipNH] -= rate_COmole.NH_CH3P_HCNHP_H2_2; 
		mole.b[ipNH] -= rate_COmole.b_NH_CH3P_HCNHP_H2;

		mole.c[ipCH3P][ipCH3P] -= rate_COmole.NH_CH3P_HCNHP_H2_1; 
		mole.c[ipNH][ipCH3P] -= rate_COmole.NH_CH3P_HCNHP_H2_2; 
		mole.b[ipCH3P] -= rate_COmole.b_NH_CH3P_HCNHP_H2;

		mole.c[ipCH3P][ipHCNHP] += rate_COmole.NH_CH3P_HCNHP_H2_1; 
		mole.c[ipNH][ipHCNHP] += rate_COmole.NH_CH3P_HCNHP_H2_2; 
		mole.b[ipHCNHP] += rate_COmole.b_NH_CH3P_HCNHP_H2;

		/*  >>refer  mol  NHP   +   HCN   =   HCNHP   +   N   */

		mole.c[ipHCN][ipNHP] -= rate_COmole.NHP_HCN_HCNHP_N_1; 
		mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_HCN_HCNHP_N_2; 
		mole.b[ipNHP] -= rate_COmole.b_NHP_HCN_HCNHP_N;

		mole.c[ipHCN][ipHCN] -= rate_COmole.NHP_HCN_HCNHP_N_1;    
		mole.c[ipNHP][ipHCN] -= rate_COmole.NHP_HCN_HCNHP_N_2;    
		mole.b[ipHCN] -= rate_COmole.b_NHP_HCN_HCNHP_N;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.NHP_HCN_HCNHP_N_1;    
		mole.c[ipNHP][ipHCNHP] += rate_COmole.NHP_HCN_HCNHP_N_2;    
		mole.b[ipHCNHP] += rate_COmole.b_NHP_HCN_HCNHP_N;

		/*  >>refer  mol  NH2P   +   HCN   =   HCNHP   +   NH   */
		
		mole.c[ipHCN][ipNH2P] -= rate_COmole.NH2P_HCN_HCNHP_NH_1;  
		mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_HCN_HCNHP_NH_2;  
		mole.b[ipNH2P] -= rate_COmole.b_NH2P_HCN_HCNHP_NH;

		mole.c[ipHCN][ipHCN] -= rate_COmole.NH2P_HCN_HCNHP_NH_1;  
		mole.c[ipNH2P][ipHCN] -= rate_COmole.NH2P_HCN_HCNHP_NH_2;  
		mole.b[ipHCN] -= rate_COmole.b_NH2P_HCN_HCNHP_NH;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.NH2P_HCN_HCNHP_NH_1; 
		mole.c[ipNH2P][ipHCNHP] += rate_COmole.NH2P_HCN_HCNHP_NH_2;  
		mole.b[ipHCNHP] += rate_COmole.b_NH2P_HCN_HCNHP_NH;

		mole.c[ipHCN][ipNH] += rate_COmole.NH2P_HCN_HCNHP_NH_1;  
		mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_HCN_HCNHP_NH_2;  
		mole.b[ipNH] += rate_COmole.b_NH2P_HCN_HCNHP_NH;

		/*  >>refer  mol  NH2   +   HCNHP   =   HCN   +   NH3P   */
		
		mole.c[ipHCNHP][ipNH2] -= rate_COmole.NH2_HCNHP_HCN_NH3P_1; 
		mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_HCNHP_HCN_NH3P_2; 
		mole.b[ipNH2] -= rate_COmole.b_NH2_HCNHP_HCN_NH3P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.NH2_HCNHP_HCN_NH3P_1; 
		mole.c[ipNH2][ipHCNHP] -= rate_COmole.NH2_HCNHP_HCN_NH3P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_NH2_HCNHP_HCN_NH3P;

		mole.c[ipHCNHP][ipHCN] += rate_COmole.NH2_HCNHP_HCN_NH3P_1; 
   	    mole.c[ipNH2][ipHCN] += rate_COmole.NH2_HCNHP_HCN_NH3P_2; 
		mole.b[ipHCN] += rate_COmole.b_NH2_HCNHP_HCN_NH3P;

		mole.c[ipHCNHP][ipNH3P] += rate_COmole.NH2_HCNHP_HCN_NH3P_1; 
		mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_HCNHP_HCN_NH3P_2; 
		mole.b[ipNH3P] += rate_COmole.b_NH2_HCNHP_HCN_NH3P;

		/*  >>refer  mol  CH4   +   HCNP   =   HCNHP   +   CH3   */
	
		mole.c[ipHCNP][ipCH4] -= rate_COmole.CH4_HCNP_HCNHP_CH3_1; 
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_HCNP_HCNHP_CH3_2; 
		mole.b[ipCH4] -= rate_COmole.b_CH4_HCNP_HCNHP_CH3;

		mole.c[ipHCNP][ipHCNP] -= rate_COmole.CH4_HCNP_HCNHP_CH3_1; 
		mole.c[ipCH4][ipHCNP] -= rate_COmole.CH4_HCNP_HCNHP_CH3_2; 
		mole.b[ipHCNP] -= rate_COmole.b_CH4_HCNP_HCNHP_CH3;

		mole.c[ipHCNP][ipHCNHP] += rate_COmole.CH4_HCNP_HCNHP_CH3_1; 
		mole.c[ipCH4][ipHCNHP] += rate_COmole.CH4_HCNP_HCNHP_CH3_2; 
		mole.b[ipHCNHP] += rate_COmole.b_CH4_HCNP_HCNHP_CH3;

		mole.c[ipHCNP][ipCH3] += rate_COmole.CH4_HCNP_HCNHP_CH3_1; 
		mole.c[ipCH4][ipCH3] += rate_COmole.CH4_HCNP_HCNHP_CH3_2; 
		mole.b[ipCH3] += rate_COmole.b_CH4_HCNP_HCNHP_CH3;

		/*  >>refer  mol  OHP   +   HCN   =   HCNHP   +   O   */

		mole.c[ipHCN][ipOHP] -= rate_COmole.OHP_HCN_HCNHP_O_1;    
		mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_HCN_HCNHP_O_2;    
		mole.b[ipOHP] -= rate_COmole.b_OHP_HCN_HCNHP_O;

		mole.c[ipHCN][ipHCN] -= rate_COmole.OHP_HCN_HCNHP_O_1;    
		mole.c[ipOHP][ipHCN] -= rate_COmole.OHP_HCN_HCNHP_O_2;    
		mole.b[ipHCN] -= rate_COmole.b_OHP_HCN_HCNHP_O;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.OHP_HCN_HCNHP_O_1;    
		mole.c[ipOHP][ipHCNHP] += rate_COmole.OHP_HCN_HCNHP_O_2;    
		mole.b[ipHCNHP] += rate_COmole.b_OHP_HCN_HCNHP_O;

		mole.c[ipHCN][ipATO] += rate_COmole.OHP_HCN_HCNHP_O_1;    
		mole.c[ipOHP][ipATO] += rate_COmole.OHP_HCN_HCNHP_O_2;    
		mole.b[ipATO] += rate_COmole.b_OHP_HCN_HCNHP_O;

		/*  >>refer  mol  NH3   +   HCNP   =   HCNHP   +   NH2   */
		
		mole.c[ipHCNP][ipNH3] -= rate_COmole.NH3_HCNP_HCNHP_NH2_1; 
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HCNP_HCNHP_NH2_2; 
		mole.b[ipNH3] -= rate_COmole.b_NH3_HCNP_HCNHP_NH2;

		mole.c[ipHCNP][ipHCNP] -= rate_COmole.NH3_HCNP_HCNHP_NH2_1; 
		mole.c[ipNH3][ipHCNP] -= rate_COmole.NH3_HCNP_HCNHP_NH2_2; 
		mole.b[ipHCNP] -= rate_COmole.b_NH3_HCNP_HCNHP_NH2;

		mole.c[ipHCNP][ipHCNHP] += rate_COmole.NH3_HCNP_HCNHP_NH2_1; 
		mole.c[ipNH3][ipHCNHP] += rate_COmole.NH3_HCNP_HCNHP_NH2_2; 
		mole.b[ipHCNHP] += rate_COmole.b_NH3_HCNP_HCNHP_NH2;

		mole.c[ipHCNP][ipNH2] += rate_COmole.NH3_HCNP_HCNHP_NH2_1; 
		mole.c[ipNH3][ipNH2] += rate_COmole.NH3_HCNP_HCNHP_NH2_2; 
		mole.b[ipNH2] += rate_COmole.b_NH3_HCNP_HCNHP_NH2;

		/*  >>refer  mol  NH3   +   HCNHP   =   HCN   +   NH4P   */

		mole.c[ipHCNHP][ipNH3] -= rate_COmole.NH3_HCNHP_HCN_NH4P_1; 
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_HCNHP_HCN_NH4P_2; 
		mole.b[ipNH3] -= rate_COmole.b_NH3_HCNHP_HCN_NH4P;

		mole.c[ipHCNHP][ipHCNHP] -= rate_COmole.NH3_HCNHP_HCN_NH4P_1; 
		mole.c[ipNH3][ipHCNHP] -= rate_COmole.NH3_HCNHP_HCN_NH4P_2; 
		mole.b[ipHCNHP] -= rate_COmole.b_NH3_HCNHP_HCN_NH4P;

		mole.c[ipHCNHP][ipHCN] += rate_COmole.NH3_HCNHP_HCN_NH4P_1; 
		mole.c[ipNH3][ipHCN] += rate_COmole.NH3_HCNHP_HCN_NH4P_2; 
		mole.b[ipHCN] += rate_COmole.b_NH3_HCNHP_HCN_NH4P;

		mole.c[ipHCNHP][ipNH4P] += rate_COmole.NH3_HCNHP_HCN_NH4P_1; 
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_HCNHP_HCN_NH4P_2; 
		mole.b[ipNH4P] += rate_COmole.b_NH3_HCNHP_HCN_NH4P;

		/*  >>refer  mol  CH5P   +   HCN   =   HCNHP   +   CH4   */

		mole.c[ipHCN][ipCH5P] -= rate_COmole.CH5P_HCN_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_HCN_HCNHP_CH4_2; 
		mole.b[ipCH5P] -= rate_COmole.b_CH5P_HCN_HCNHP_CH4;

		mole.c[ipHCN][ipHCN] -= rate_COmole.CH5P_HCN_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipHCN] -= rate_COmole.CH5P_HCN_HCNHP_CH4_2; 
		mole.b[ipHCN] -= rate_COmole.b_CH5P_HCN_HCNHP_CH4;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.CH5P_HCN_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipHCNHP] += rate_COmole.CH5P_HCN_HCNHP_CH4_2; 
		mole.b[ipHCNHP] += rate_COmole.b_CH5P_HCN_HCNHP_CH4;

		mole.c[ipHCN][ipCH4] += rate_COmole.CH5P_HCN_HCNHP_CH4_1; 
		mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_HCN_HCNHP_CH4_2; 
		mole.b[ipCH4] += rate_COmole.b_CH5P_HCN_HCNHP_CH4;

		/*  >>refer  mol  H2OP   +   HCN   =   HCNHP   +   OH   */

		mole.c[ipHCN][ipH2OP] -= rate_COmole.H2OP_HCN_HCNHP_OH_1;  
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_HCN_HCNHP_OH_2;  
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_HCN_HCNHP_OH;

		mole.c[ipHCN][ipHCN] -= rate_COmole.H2OP_HCN_HCNHP_OH_1;  
		mole.c[ipH2OP][ipHCN] -= rate_COmole.H2OP_HCN_HCNHP_OH_2;  
		mole.b[ipHCN] -= rate_COmole.b_H2OP_HCN_HCNHP_OH;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.H2OP_HCN_HCNHP_OH_1;  
		mole.c[ipH2OP][ipHCNHP] += rate_COmole.H2OP_HCN_HCNHP_OH_2;  
		mole.b[ipHCNHP] += rate_COmole.b_H2OP_HCN_HCNHP_OH;

		mole.c[ipHCN][ipOH] += rate_COmole.H2OP_HCN_HCNHP_OH_1;  
		mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_HCN_HCNHP_OH_2;  
		mole.b[ipOH] += rate_COmole.b_H2OP_HCN_HCNHP_OH;

		/*  >>refer  mol  H3OP   +   HCN   =   HCNHP   +   H2O   */

		mole.c[ipHCN][ipH3OP] -= rate_COmole.H3OP_HCN_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_HCN_HCNHP_H2O_2; 
		mole.b[ipH3OP] -= rate_COmole.b_H3OP_HCN_HCNHP_H2O;

		mole.c[ipHCN][ipHCN] -= rate_COmole.H3OP_HCN_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipHCN] -= rate_COmole.H3OP_HCN_HCNHP_H2O_2; 
		mole.b[ipHCN] -= rate_COmole.b_H3OP_HCN_HCNHP_H2O;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.H3OP_HCN_HCNHP_H2O_1;
		mole.c[ipH3OP][ipHCNHP] += rate_COmole.H3OP_HCN_HCNHP_H2O_2; 
		mole.b[ipHCNHP] += rate_COmole.b_H3OP_HCN_HCNHP_H2O;

		mole.c[ipHCN][ipH2O] += rate_COmole.H3OP_HCN_HCNHP_H2O_1; 
		mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_HCN_HCNHP_H2O_2; 
		mole.b[ipH2O] += rate_COmole.b_H3OP_HCN_HCNHP_H2O;

		/*  >>refer  mol  HCNP   +   HCN   =   HCNHP   +   CN   */

		mole.c[ipHCN][ipHCNP] -= rate_COmole.HCNP_HCN_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHCNP] -= rate_COmole.HCNP_HCN_HCNHP_CN_2; 
		mole.b[ipHCNP] -= rate_COmole.b_HCNP_HCN_HCNHP_CN;

		mole.c[ipHCN][ipHCN] -= rate_COmole.HCNP_HCN_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHCN] -= rate_COmole.HCNP_HCN_HCNHP_CN_2; 
		mole.b[ipHCN] -= rate_COmole.b_HCNP_HCN_HCNHP_CN;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.HCNP_HCN_HCNHP_CN_1; 
		mole.c[ipHCNP][ipHCNHP] += rate_COmole.HCNP_HCN_HCNHP_CN_2; 
		mole.b[ipHCNHP] += rate_COmole.b_HCNP_HCN_HCNHP_CN;

		mole.c[ipHCN][ipCN] += rate_COmole.HCNP_HCN_HCNHP_CN_1; 
		mole.c[ipHCNP][ipCN] += rate_COmole.HCNP_HCN_HCNHP_CN_2; 
		mole.b[ipCN] += rate_COmole.b_HCNP_HCN_HCNHP_CN;

		/*  >>refer  mol  HCN   +   HNOP   =   NO   +   HCNHP   */

		mole.c[ipHNOP][ipHCN] -= rate_COmole.HCN_HNOP_NO_HCNHP_1;  
		mole.c[ipHCN][ipHCN] -= rate_COmole.HCN_HNOP_NO_HCNHP_2;  
		mole.b[ipHCN] -= rate_COmole.b_HCN_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipHNOP] -= rate_COmole.HCN_HNOP_NO_HCNHP_1;  
		mole.c[ipHCN][ipHNOP] -= rate_COmole.HCN_HNOP_NO_HCNHP_2;  
		mole.b[ipHNOP] -= rate_COmole.b_HCN_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipNO] += rate_COmole.HCN_HNOP_NO_HCNHP_1;  
		mole.c[ipHCN][ipNO] += rate_COmole.HCN_HNOP_NO_HCNHP_2;  
		mole.b[ipNO] += rate_COmole.b_HCN_HNOP_NO_HCNHP;

		mole.c[ipHNOP][ipHCNHP] += rate_COmole.HCN_HNOP_NO_HCNHP_1;  
		mole.c[ipHCN][ipHCNHP] += rate_COmole.HCN_HNOP_NO_HCNHP_2;  
		mole.b[ipHCNHP] += rate_COmole.b_HCN_HNOP_NO_HCNHP;

		/*  >>refer  mol  HCN   +   HSP   =   S   +   HCNHP   */

		mole.c[ipHSP][ipHCN] -= rate_COmole.HCN_HSP_S_HCNHP_1;    
		mole.c[ipHCN][ipHCN] -= rate_COmole.HCN_HSP_S_HCNHP_2;    
		mole.b[ipHCN] -= rate_COmole.b_HCN_HSP_S_HCNHP;

		mole.c[ipHSP][ipHSP] -= rate_COmole.HCN_HSP_S_HCNHP_1;    
		mole.c[ipHCN][ipHSP] -= rate_COmole.HCN_HSP_S_HCNHP_2;    
		mole.b[ipHSP] -= rate_COmole.b_HCN_HSP_S_HCNHP;

		mole.c[ipHSP][ipATS] += rate_COmole.HCN_HSP_S_HCNHP_1;    
		mole.c[ipHCN][ipATS] += rate_COmole.HCN_HSP_S_HCNHP_2;    
		mole.b[ipATS] += rate_COmole.b_HCN_HSP_S_HCNHP;

		mole.c[ipHSP][ipHCNHP] += rate_COmole.HCN_HSP_S_HCNHP_1;    
		mole.c[ipHCN][ipHCNHP] += rate_COmole.HCN_HSP_S_HCNHP_2;    
		mole.b[ipHCNHP] += rate_COmole.b_HCN_HSP_S_HCNHP;

		/* >>refer  mole   C   CH2   -->   C2H   H        */

		mole.c[ipCH2][ipCH2] -= rate_COmole.C_CH2_C2H_H_1;
		mole.c[ipATC][ipCH2] -= rate_COmole.C_CH2_C2H_H_2;
		mole.b[ipCH2] -= rate_COmole.b_C_CH2_C2H_H;

		mole.c[ipCH2][ipATC] -= rate_COmole.C_CH2_C2H_H_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_CH2_C2H_H_2;
		mole.b[ipATC] -= rate_COmole.b_C_CH2_C2H_H;

		mole.c[ipCH2][ipC2H] += rate_COmole.C_CH2_C2H_H_1;
		mole.c[ipATC][ipC2H] += rate_COmole.C_CH2_C2H_H_2;
		mole.b[ipC2H] += rate_COmole.b_C_CH2_C2H_H;

		/* >>refer  mole   O+   C2H   -->   CO+   CH        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.OP_C2H_COP_CH_1;
		mole.c[ipOP][ipC2H] -= rate_COmole.OP_C2H_COP_CH_2;
		mole.b[ipC2H] -= rate_COmole.b_OP_C2H_COP_CH;

		mole.c[ipC2H][ipOP] -= rate_COmole.OP_C2H_COP_CH_1;
		mole.c[ipOP][ipOP] -= rate_COmole.OP_C2H_COP_CH_2;
		mole.b[ipOP] -= rate_COmole.b_OP_C2H_COP_CH;

		mole.c[ipC2H][ipCOP] += rate_COmole.OP_C2H_COP_CH_1;
		mole.c[ipOP][ipCOP] += rate_COmole.OP_C2H_COP_CH_2;
		mole.b[ipCOP] += rate_COmole.b_OP_C2H_COP_CH;

		mole.c[ipC2H][ipCH] += rate_COmole.OP_C2H_COP_CH_1;
		mole.c[ipOP][ipCH] += rate_COmole.OP_C2H_COP_CH_2;
		mole.b[ipCH] += rate_COmole.b_OP_C2H_COP_CH;

		/* >>refer  mole   C2H   CO+   -->   HCO+   C2        */

		mole.c[ipCOP][ipCOP] -= rate_COmole.C2H_COP_HCOP_C2_1;
		mole.c[ipC2H][ipCOP] -= rate_COmole.C2H_COP_HCOP_C2_2;
		mole.b[ipCOP] -= rate_COmole.b_C2H_COP_HCOP_C2;

		mole.c[ipCOP][ipC2H] -= rate_COmole.C2H_COP_HCOP_C2_1;
		mole.c[ipC2H][ipC2H] -= rate_COmole.C2H_COP_HCOP_C2_2;
		mole.b[ipC2H] -= rate_COmole.b_C2H_COP_HCOP_C2;

		mole.c[ipCOP][ipHCOP] += rate_COmole.C2H_COP_HCOP_C2_1;
		mole.c[ipC2H][ipHCOP] += rate_COmole.C2H_COP_HCOP_C2_2;
		mole.b[ipHCOP] += rate_COmole.b_C2H_COP_HCOP_C2;

		mole.c[ipCOP][ipC2] += rate_COmole.C2H_COP_HCOP_C2_1;
		mole.c[ipC2H][ipC2] += rate_COmole.C2H_COP_HCOP_C2_2;
		mole.b[ipC2] += rate_COmole.b_C2H_COP_HCOP_C2;

		/* >>refer  mole   C   CH2+   -->   C2H+   H        */

		mole.c[ipCH2P][ipCH2P] -= rate_COmole.C_CH2P_C2HP_H_1;
		mole.c[ipATC][ipCH2P] -= rate_COmole.C_CH2P_C2HP_H_2;
		mole.b[ipCH2P] -= rate_COmole.b_C_CH2P_C2HP_H;

		mole.c[ipCH2P][ipATC] -= rate_COmole.C_CH2P_C2HP_H_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_CH2P_C2HP_H_2;
		mole.b[ipATC] -= rate_COmole.b_C_CH2P_C2HP_H;

		mole.c[ipCH2P][ipC2HP] += rate_COmole.C_CH2P_C2HP_H_1;
		mole.c[ipATC][ipC2HP] +=  rate_COmole.C_CH2P_C2HP_H_2;
		mole.b[ipC2HP] += rate_COmole.b_C_CH2P_C2HP_H;

		/* >>refer  mole   C   CH3+   -->   C2H+   H2        */

		mole.c[ipCH3P][ipCH3P] -= rate_COmole.C_CH3P_C2HP_H2_1;
		mole.c[ipATC][ipCH3P] -= rate_COmole.C_CH3P_C2HP_H2_2;
		mole.b[ipCH3P] -= rate_COmole.b_C_CH3P_C2HP_H2;

		mole.c[ipCH3P][ipATC] -= rate_COmole.C_CH3P_C2HP_H2_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_CH3P_C2HP_H2_2;
		mole.b[ipATC] -= rate_COmole.b_C_CH3P_C2HP_H2;

		mole.c[ipCH3P][ipC2HP] += rate_COmole.C_CH3P_C2HP_H2_1;
		mole.c[ipATC][ipC2HP] += rate_COmole.C_CH3P_C2HP_H2_2;
		mole.b[ipC2HP] += rate_COmole.b_C_CH3P_C2HP_H2;

		/* >>refer  mole   C2   HCN+   -->   CN   C2H+        */

		mole.c[ipHCNP][ipHCNP] -= rate_COmole.C2_HCNP_CN_C2HP_1;
		mole.c[ipC2][ipHCNP] -= rate_COmole.C2_HCNP_CN_C2HP_2;
		mole.b[ipHCNP] -= rate_COmole.b_C2_HCNP_CN_C2HP;

		mole.c[ipHCNP][ipC2] -= rate_COmole.C2_HCNP_CN_C2HP_1;
		mole.c[ipC2][ipC2] -= rate_COmole.C2_HCNP_CN_C2HP_2;
		mole.b[ipC2] -= rate_COmole.b_C2_HCNP_CN_C2HP;

		mole.c[ipHCNP][ipCN] += rate_COmole.C2_HCNP_CN_C2HP_1;
		mole.c[ipC2][ipCN] += rate_COmole.C2_HCNP_CN_C2HP_2;
		mole.b[ipCN] += rate_COmole.b_C2_HCNP_CN_C2HP;

		mole.c[ipHCNP][ipC2HP] += rate_COmole.C2_HCNP_CN_C2HP_1;
		mole.c[ipC2][ipC2HP] += rate_COmole.C2_HCNP_CN_C2HP_2;
		mole.b[ipC2HP] += rate_COmole.b_C2_HCNP_CN_C2HP;

		/* >>refer  mole   C2   HNO+   -->   NO   C2H+        */

		mole.c[ipHNOP][ipHNOP] -= rate_COmole.C2_HNOP_NO_C2HP_1;
		mole.c[ipC2][ipHNOP] -= rate_COmole.C2_HNOP_NO_C2HP_2;
		mole.b[ipHNOP] -= rate_COmole.b_C2_HNOP_NO_C2HP;

		mole.c[ipHNOP][ipC2] -= rate_COmole.C2_HNOP_NO_C2HP_1;
		mole.c[ipC2][ipC2] -= rate_COmole.C2_HNOP_NO_C2HP_2;
		mole.b[ipC2] -= rate_COmole.b_C2_HNOP_NO_C2HP;

		mole.c[ipHNOP][ipNO] += rate_COmole.C2_HNOP_NO_C2HP_1;
		mole.c[ipC2][ipNO] += rate_COmole.C2_HNOP_NO_C2HP_2;
		mole.b[ipNO] += rate_COmole.b_C2_HNOP_NO_C2HP;

		mole.c[ipHNOP][ipC2HP] += rate_COmole.C2_HNOP_NO_C2HP_1;
		mole.c[ipC2][ipC2HP] += rate_COmole.C2_HNOP_NO_C2HP_2;
		mole.b[ipC2HP] += rate_COmole.b_C2_HNOP_NO_C2HP;

		/* >>refer  mole   C2H   CN+   -->   CN   C2H+        */

		mole.c[ipCNP][ipCNP] -= rate_COmole.C2H_CNP_CN_C2HP_1;
		mole.c[ipC2H][ipCNP] -= rate_COmole.C2H_CNP_CN_C2HP_2;
		mole.b[ipCNP] -= rate_COmole.b_C2H_CNP_CN_C2HP;

		mole.c[ipCNP][ipC2H] -= rate_COmole.C2H_CNP_CN_C2HP_1;
		mole.c[ipC2H][ipC2H] -= rate_COmole.C2H_CNP_CN_C2HP_2;
		mole.b[ipC2H] -= rate_COmole.b_C2H_CNP_CN_C2HP;

		mole.c[ipCNP][ipCN] += rate_COmole.C2H_CNP_CN_C2HP_1;
		mole.c[ipC2H][ipCN] += rate_COmole.C2H_CNP_CN_C2HP_2;
		mole.b[ipCN] += rate_COmole.b_C2H_CNP_CN_C2HP;

		mole.c[ipCNP][ipC2HP] += rate_COmole.C2H_CNP_CN_C2HP_1;
		mole.c[ipC2H][ipC2HP] += rate_COmole.C2H_CNP_CN_C2HP_2;
		mole.b[ipC2HP] += rate_COmole.b_C2H_CNP_CN_C2HP;

		/* >>refer  mole   C2H   N2+   -->   N2   C2H+        */

		mole.c[ipN2P][ipN2P] -= rate_COmole.C2H_N2P_N2_C2HP_1;
		mole.c[ipC2H][ipN2P] -= rate_COmole.C2H_N2P_N2_C2HP_2;
		mole.b[ipN2P] -= rate_COmole.b_C2H_N2P_N2_C2HP;

		mole.c[ipN2P][ipC2H] -= rate_COmole.C2H_N2P_N2_C2HP_1;
		mole.c[ipC2H][ipC2H] -= rate_COmole.C2H_N2P_N2_C2HP_2;
		mole.b[ipC2H] -= rate_COmole.b_C2H_N2P_N2_C2HP;

		mole.c[ipN2P][ipN2] += rate_COmole.C2H_N2P_N2_C2HP_1;
		mole.c[ipC2H][ipN2] += rate_COmole.C2H_N2P_N2_C2HP_2;
		mole.b[ipN2] += rate_COmole.b_C2H_N2P_N2_C2HP;

		mole.c[ipN2P][ipC2HP] += rate_COmole.C2H_N2P_N2_C2HP_1;
		mole.c[ipC2H][ipC2HP] += rate_COmole.C2H_N2P_N2_C2HP_2;
		mole.b[ipC2HP] += rate_COmole.b_C2H_N2P_N2_C2HP;

		/* >>refer  mole   C2H+   HCN   -->   HCNH+   C2        */

		mole.c[ipHCN][ipHCN] -= rate_COmole.C2HP_HCN_HCNHP_C2_1;
		mole.c[ipC2HP][ipHCN] -= rate_COmole.C2HP_HCN_HCNHP_C2_2;
		mole.b[ipHCN] -= rate_COmole.b_C2HP_HCN_HCNHP_C2;

		mole.c[ipHCN][ipC2HP] -= rate_COmole.C2HP_HCN_HCNHP_C2_1;
		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C2HP_HCN_HCNHP_C2_2;
		mole.b[ipC2HP] -= rate_COmole.b_C2HP_HCN_HCNHP_C2;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.C2HP_HCN_HCNHP_C2_1;
		mole.c[ipC2HP][ipHCNHP] += rate_COmole.C2HP_HCN_HCNHP_C2_2;
		mole.b[ipHCNHP] += rate_COmole.b_C2HP_HCN_HCNHP_C2;

		mole.c[ipHCN][ipC2] += rate_COmole.C2HP_HCN_HCNHP_C2_1;
		mole.c[ipC2HP][ipC2] += rate_COmole.C2HP_HCN_HCNHP_C2_2;
		mole.b[ipC2] += rate_COmole.b_C2HP_HCN_HCNHP_C2;

		/* >>refer  mole   C2H+   HNC   -->   HCNH+   C2        */

		mole.c[ipHNC][ipHNC] -= rate_COmole.C2HP_HNC_HCNHP_C2_1;
		mole.c[ipC2HP][ipHNC] -= rate_COmole.C2HP_HNC_HCNHP_C2_2;
		mole.b[ipHNC] -= rate_COmole.b_C2HP_HNC_HCNHP_C2;

		mole.c[ipHNC][ipC2HP] -= rate_COmole.C2HP_HNC_HCNHP_C2_1;
		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C2HP_HNC_HCNHP_C2_2;
		mole.b[ipC2HP] -= rate_COmole.b_C2HP_HNC_HCNHP_C2;

		mole.c[ipHNC][ipHCNHP] += rate_COmole.C2HP_HNC_HCNHP_C2_1;
		mole.c[ipC2HP][ipHCNHP] += rate_COmole.C2HP_HNC_HCNHP_C2_2;
		mole.b[ipHCNHP] += rate_COmole.b_C2HP_HNC_HCNHP_C2;

		mole.c[ipHNC][ipC2] += rate_COmole.C2HP_HNC_HCNHP_C2_1;
		mole.c[ipC2HP][ipC2] += rate_COmole.C2HP_HNC_HCNHP_C2_2;
		mole.b[ipC2] += rate_COmole.b_C2HP_HNC_HCNHP_C2;

		/* >>refer  mole   C2H+   NO   -->   NO+   C2H        */

		mole.c[ipNO][ipNO] -= rate_COmole.C2HP_NO_NOP_C2H_1;
		mole.c[ipC2HP][ipNO] -= rate_COmole.C2HP_NO_NOP_C2H_2;
		mole.b[ipNO] -= rate_COmole.b_C2HP_NO_NOP_C2H;

		mole.c[ipNO][ipC2HP] -= rate_COmole.C2HP_NO_NOP_C2H_1;
		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C2HP_NO_NOP_C2H_2;
		mole.b[ipC2HP] -= rate_COmole.b_C2HP_NO_NOP_C2H;

		mole.c[ipNO][ipNOP] += rate_COmole.C2HP_NO_NOP_C2H_1;
		mole.c[ipC2HP][ipNOP] += rate_COmole.C2HP_NO_NOP_C2H_2;
		mole.b[ipNOP] += rate_COmole.b_C2HP_NO_NOP_C2H;

		mole.c[ipNO][ipC2H] += rate_COmole.C2HP_NO_NOP_C2H_1;
		mole.c[ipC2HP][ipC2H] += rate_COmole.C2HP_NO_NOP_C2H_2;
		mole.b[ipC2H] += rate_COmole.b_C2HP_NO_NOP_C2H;

		/* >>refer  mole   C2H+   S   -->   S+   C2H        */

		mole.c[ipATS][ipATS] -= rate_COmole.C2HP_S_SP_C2H_1;
		mole.c[ipC2HP][ipATS] -= rate_COmole.C2HP_S_SP_C2H_2;
		mole.b[ipATS] -= rate_COmole.b_C2HP_S_SP_C2H;

		mole.c[ipATS][ipC2HP] -= rate_COmole.C2HP_S_SP_C2H_1;
		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C2HP_S_SP_C2H_2;
		mole.b[ipC2HP] -= rate_COmole.b_C2HP_S_SP_C2H;

		mole.c[ipATS][ipSP] += rate_COmole.C2HP_S_SP_C2H_1;
		mole.c[ipC2HP][ipSP] += rate_COmole.C2HP_S_SP_C2H_2;
		mole.b[ipSP] += rate_COmole.b_C2HP_S_SP_C2H;

		mole.c[ipATS][ipC2H] += rate_COmole.C2HP_S_SP_C2H_1;
		mole.c[ipC2HP][ipC2H] += rate_COmole.C2HP_S_SP_C2H_2;
		mole.b[ipC2H] += rate_COmole.b_C2HP_S_SP_C2H;

		/* >>refer  mole   CH   C2H+   -->   C2   CH2+        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.CH_C2HP_C2_CH2P_1;
		mole.c[ipCH][ipC2HP] -= rate_COmole.CH_C2HP_C2_CH2P_2;
		mole.b[ipC2HP] -= rate_COmole.b_CH_C2HP_C2_CH2P;

		mole.c[ipC2HP][ipCH] -= rate_COmole.CH_C2HP_C2_CH2P_1;
		mole.c[ipCH][ipCH] -= rate_COmole.CH_C2HP_C2_CH2P_2;
		mole.b[ipCH] -= rate_COmole.b_CH_C2HP_C2_CH2P;

		mole.c[ipC2HP][ipC2] += rate_COmole.CH_C2HP_C2_CH2P_1;
		mole.c[ipCH][ipC2] += rate_COmole.CH_C2HP_C2_CH2P_2;
		mole.b[ipC2] += rate_COmole.b_CH_C2HP_C2_CH2P;

		mole.c[ipC2HP][ipCH2P] += rate_COmole.CH_C2HP_C2_CH2P_1;
		mole.c[ipCH][ipCH2P] += rate_COmole.CH_C2HP_C2_CH2P_2;
		mole.b[ipCH2P] += rate_COmole.b_CH_C2HP_C2_CH2P;

		/* >>refer  mole   CH2   C2H+   -->   C2   CH3+        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.CH2_C2HP_C2_CH3P_1;
		mole.c[ipCH2][ipC2HP] -= rate_COmole.CH2_C2HP_C2_CH3P_2;
		mole.b[ipC2HP] -= rate_COmole.b_CH2_C2HP_C2_CH3P;

		mole.c[ipC2HP][ipCH2] -= rate_COmole.CH2_C2HP_C2_CH3P_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_C2HP_C2_CH3P_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_C2HP_C2_CH3P;

		mole.c[ipC2HP][ipC2] += rate_COmole.CH2_C2HP_C2_CH3P_1;
		mole.c[ipCH2][ipC2] += rate_COmole.CH2_C2HP_C2_CH3P_2;
		mole.b[ipC2] += rate_COmole.b_CH2_C2HP_C2_CH3P;

		mole.c[ipC2HP][ipCH3P] += rate_COmole.CH2_C2HP_C2_CH3P_1;
		mole.c[ipCH2][ipCH3P] += rate_COmole.CH2_C2HP_C2_CH3P_2;
		mole.b[ipCH3P] += rate_COmole.b_CH2_C2HP_C2_CH3P;

		/* >>refer  mole   CH4   C2+   -->   C2H+   CH3        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.CH4_C2P_C2HP_CH3_1;
		mole.c[ipCH4][ipC2P] -= rate_COmole.CH4_C2P_C2HP_CH3_2;
		mole.b[ipC2P] -= rate_COmole.b_CH4_C2P_C2HP_CH3;

		mole.c[ipC2P][ipCH4] -= rate_COmole.CH4_C2P_C2HP_CH3_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_C2P_C2HP_CH3_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_C2P_C2HP_CH3;

		mole.c[ipC2P][ipC2HP] += rate_COmole.CH4_C2P_C2HP_CH3_1;
		mole.c[ipCH4][ipC2HP] += rate_COmole.CH4_C2P_C2HP_CH3_2;
		mole.b[ipC2HP] += rate_COmole.b_CH4_C2P_C2HP_CH3;

		mole.c[ipC2P][ipCH3] += rate_COmole.CH4_C2P_C2HP_CH3_1;
		mole.c[ipCH4][ipCH3] += rate_COmole.CH4_C2P_C2HP_CH3_2;
		mole.b[ipCH3] += rate_COmole.b_CH4_C2P_C2HP_CH3;

		/* >>refer  mole   CH5+   C2   -->   C2H+   CH4        */

		mole.c[ipC2][ipC2] -= rate_COmole.CH5P_C2_C2HP_CH4_1;
		mole.c[ipCH5P][ipC2] -= rate_COmole.CH5P_C2_C2HP_CH4_2;
		mole.b[ipC2] -= rate_COmole.b_CH5P_C2_C2HP_CH4;

		mole.c[ipC2][ipCH5P] -= rate_COmole.CH5P_C2_C2HP_CH4_1;
		mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_C2_C2HP_CH4_2;
		mole.b[ipCH5P] -= rate_COmole.b_CH5P_C2_C2HP_CH4;

		mole.c[ipC2][ipC2HP] += rate_COmole.CH5P_C2_C2HP_CH4_1;
		mole.c[ipCH5P][ipC2HP] += rate_COmole.CH5P_C2_C2HP_CH4_2;
		mole.b[ipC2HP] += rate_COmole.b_CH5P_C2_C2HP_CH4;

		mole.c[ipC2][ipCH4] += rate_COmole.CH5P_C2_C2HP_CH4_1;
		mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_C2_C2HP_CH4_2;
		mole.b[ipCH4] += rate_COmole.b_CH5P_C2_C2HP_CH4;

		/* >>refer  mole   CH+   CH2   -->   C2H+   H2        */

		mole.c[ipCH2][ipCH2] -= rate_COmole.CHP_CH2_C2HP_H2_1;
		mole.c[ipCHP][ipCH2] -= rate_COmole.CHP_CH2_C2HP_H2_2;
		mole.b[ipCH2] -= rate_COmole.b_CHP_CH2_C2HP_H2;

		mole.c[ipCH2][ipCHP] -= rate_COmole.CHP_CH2_C2HP_H2_1;
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_CH2_C2HP_H2_2;
		mole.b[ipCHP] -= rate_COmole.b_CHP_CH2_C2HP_H2;

		mole.c[ipCH2][ipC2HP] += rate_COmole.CHP_CH2_C2HP_H2_1;
		mole.c[ipCHP][ipC2HP] += rate_COmole.CHP_CH2_C2HP_H2_2;
		mole.b[ipC2HP] += rate_COmole.b_CHP_CH2_C2HP_H2;

		/* >>refer  mole   C+   CH2   -->   C2H+   H        */

		mole.c[ipCH2][ipCH2] -= rate_COmole.CP_CH2_C2HP_H_1;
		mole.c[ipCP][ipCH2] -= rate_COmole.CP_CH2_C2HP_H_2;
		mole.b[ipCH2] -= rate_COmole.b_CP_CH2_C2HP_H;

		mole.c[ipCH2][ipCP] -= rate_COmole.CP_CH2_C2HP_H_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CH2_C2HP_H_2;
		mole.b[ipCP] -= rate_COmole.b_CP_CH2_C2HP_H;

		mole.c[ipCH2][ipC2HP] += rate_COmole.CP_CH2_C2HP_H_1;
		mole.c[ipCP][ipC2HP] += rate_COmole.CP_CH2_C2HP_H_2;
		mole.b[ipC2HP] += rate_COmole.b_CP_CH2_C2HP_H;

		/* >>refer  mole   C+   CH3   -->   C2H+   H2        */

		mole.c[ipCH3][ipCH3] -= rate_COmole.CP_CH3_C2HP_H2_1;
		mole.c[ipCP][ipCH3] -= rate_COmole.CP_CH3_C2HP_H2_2;
		mole.b[ipCH3] -= rate_COmole.b_CP_CH3_C2HP_H2;

		mole.c[ipCH3][ipCP] -= rate_COmole.CP_CH3_C2HP_H2_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CH3_C2HP_H2_2;
		mole.b[ipCP] -= rate_COmole.b_CP_CH3_C2HP_H2;

		mole.c[ipCH3][ipC2HP] += rate_COmole.CP_CH3_C2HP_H2_1;
		mole.c[ipCP][ipC2HP] += rate_COmole.CP_CH3_C2HP_H2_2;
		mole.b[ipC2HP] += rate_COmole.b_CP_CH3_C2HP_H2;
		/* >>refer  mole   H2O   C2+   -->   C2H+   OH        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.H2O_C2P_C2HP_OH_1;
		mole.c[ipH2O][ipC2P] -= rate_COmole.H2O_C2P_C2HP_OH_2;
		mole.b[ipC2P] -= rate_COmole.b_H2O_C2P_C2HP_OH;

		mole.c[ipC2P][ipH2O] -= rate_COmole.H2O_C2P_C2HP_OH_1;
		mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_C2P_C2HP_OH_2;
		mole.b[ipH2O] -= rate_COmole.b_H2O_C2P_C2HP_OH;

		mole.c[ipC2P][ipC2HP] += rate_COmole.H2O_C2P_C2HP_OH_1;
		mole.c[ipH2O][ipC2HP] += rate_COmole.H2O_C2P_C2HP_OH_2;
		mole.b[ipC2HP] += rate_COmole.b_H2O_C2P_C2HP_OH;

		mole.c[ipC2P][ipOH] += rate_COmole.H2O_C2P_C2HP_OH_1;
		mole.c[ipH2O][ipOH] += rate_COmole.H2O_C2P_C2HP_OH_2;
		mole.b[ipOH] += rate_COmole.b_H2O_C2P_C2HP_OH;

		/* >>refer  mole   H2O+   C2   -->   C2H+   OH        */

		mole.c[ipC2][ipC2] -= rate_COmole.H2OP_C2_C2HP_OH_1;
		mole.c[ipH2OP][ipC2] -= rate_COmole.H2OP_C2_C2HP_OH_2;
		mole.b[ipC2] -= rate_COmole.b_H2OP_C2_C2HP_OH;

		mole.c[ipC2][ipH2OP] -= rate_COmole.H2OP_C2_C2HP_OH_1;
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_C2_C2HP_OH_2;
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_C2_C2HP_OH;

		mole.c[ipC2][ipC2HP] += rate_COmole.H2OP_C2_C2HP_OH_1;
		mole.c[ipH2OP][ipC2HP] += rate_COmole.H2OP_C2_C2HP_OH_2;
		mole.b[ipC2HP] += rate_COmole.b_H2OP_C2_C2HP_OH;

		mole.c[ipC2][ipOH] += rate_COmole.H2OP_C2_C2HP_OH_1;
		mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_C2_C2HP_OH_2;
		mole.b[ipOH] += rate_COmole.b_H2OP_C2_C2HP_OH;

		/* >>refer  mole   H2O+   C2H   -->   C2H+   H2O        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.H2OP_C2H_C2HP_H2O_1;
		mole.c[ipH2OP][ipC2H] -= rate_COmole.H2OP_C2H_C2HP_H2O_2;
		mole.b[ipC2H] -= rate_COmole.b_H2OP_C2H_C2HP_H2O;

		mole.c[ipC2H][ipH2OP] -= rate_COmole.H2OP_C2H_C2HP_H2O_1;
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_C2H_C2HP_H2O_2;
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_C2H_C2HP_H2O;

		mole.c[ipC2H][ipC2HP] += rate_COmole.H2OP_C2H_C2HP_H2O_1;
		mole.c[ipH2OP][ipC2HP] += rate_COmole.H2OP_C2H_C2HP_H2O_2;
		mole.b[ipC2HP] += rate_COmole.b_H2OP_C2H_C2HP_H2O;

		mole.c[ipC2H][ipH2O] += rate_COmole.H2OP_C2H_C2HP_H2O_1;
		mole.c[ipH2OP][ipH2O] += rate_COmole.H2OP_C2H_C2HP_H2O_2;
		mole.b[ipH2O] += rate_COmole.b_H2OP_C2H_C2HP_H2O;

		/* >>refer  mole   H3O+   C2   -->   C2H+   H2O        */

		mole.c[ipC2][ipC2] -= rate_COmole.H3OP_C2_C2HP_H2O_1;
		mole.c[ipH3OP][ipC2] -= rate_COmole.H3OP_C2_C2HP_H2O_2;
		mole.b[ipC2] -= rate_COmole.b_H3OP_C2_C2HP_H2O;

		mole.c[ipC2][ipH3OP] -= rate_COmole.H3OP_C2_C2HP_H2O_1;
		mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_C2_C2HP_H2O_2;
		mole.b[ipH3OP] -= rate_COmole.b_H3OP_C2_C2HP_H2O;

		mole.c[ipC2][ipC2HP] += rate_COmole.H3OP_C2_C2HP_H2O_1;
		mole.c[ipH3OP][ipC2HP] += rate_COmole.H3OP_C2_C2HP_H2O_2;
		mole.b[ipC2HP] += rate_COmole.b_H3OP_C2_C2HP_H2O;

		mole.c[ipC2][ipH2O] += rate_COmole.H3OP_C2_C2HP_H2O_1;
		mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_C2_C2HP_H2O_2;
		mole.b[ipH2O] += rate_COmole.b_H3OP_C2_C2HP_H2O;

		/* >>refer  mole   N   C2H+   -->   CN   CH+        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.N_C2HP_CN_CHP_1;
		mole.c[ipATN][ipC2HP] -= rate_COmole.N_C2HP_CN_CHP_2;
		mole.b[ipC2HP] -= rate_COmole.b_N_C2HP_CN_CHP;

		mole.c[ipC2HP][ipATN] -= rate_COmole.N_C2HP_CN_CHP_1;
		mole.c[ipATN][ipATN] -= rate_COmole.N_C2HP_CN_CHP_2;
		mole.b[ipATN] -= rate_COmole.b_N_C2HP_CN_CHP;

		mole.c[ipC2HP][ipCN] += rate_COmole.N_C2HP_CN_CHP_1;
		mole.c[ipATN][ipCN] += rate_COmole.N_C2HP_CN_CHP_2;
		mole.b[ipCN] += rate_COmole.b_N_C2HP_CN_CHP;

		mole.c[ipC2HP][ipCHP] += rate_COmole.N_C2HP_CN_CHP_1;
		mole.c[ipATN][ipCHP] += rate_COmole.N_C2HP_CN_CHP_2;
		mole.b[ipCHP] += rate_COmole.b_N_C2HP_CN_CHP;

		/* >>refer  mole   NH   C2+   -->   C2H+   N        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.NH_C2P_C2HP_N_1;
		mole.c[ipNH][ipC2P] -= rate_COmole.NH_C2P_C2HP_N_2;
		mole.b[ipC2P] -= rate_COmole.b_NH_C2P_C2HP_N;

		mole.c[ipC2P][ipNH] -= rate_COmole.NH_C2P_C2HP_N_1;
		mole.c[ipNH][ipNH] -= rate_COmole.NH_C2P_C2HP_N_2;
		mole.b[ipNH] -= rate_COmole.b_NH_C2P_C2HP_N;

		mole.c[ipC2P][ipC2HP] += rate_COmole.NH_C2P_C2HP_N_1;
		mole.c[ipNH][ipC2HP] += rate_COmole.NH_C2P_C2HP_N_2;
		mole.b[ipC2HP] += rate_COmole.b_NH_C2P_C2HP_N;

		mole.c[ipC2P][ipATN] += rate_COmole.NH_C2P_C2HP_N_1;
		mole.c[ipNH][ipATN] += rate_COmole.NH_C2P_C2HP_N_2;
		mole.b[ipATN] += rate_COmole.b_NH_C2P_C2HP_N;

		/* >>refer  mole   NH2   C2H+   -->   C2   NH3+        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.NH2_C2HP_C2_NH3P_1;
		mole.c[ipNH2][ipC2HP] -= rate_COmole.NH2_C2HP_C2_NH3P_2;
		mole.b[ipC2HP] -= rate_COmole.b_NH2_C2HP_C2_NH3P;

		mole.c[ipC2HP][ipNH2] -= rate_COmole.NH2_C2HP_C2_NH3P_1;
		mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_C2HP_C2_NH3P_2;
		mole.b[ipNH2] -= rate_COmole.b_NH2_C2HP_C2_NH3P;

		mole.c[ipC2HP][ipC2] += rate_COmole.NH2_C2HP_C2_NH3P_1;
		mole.c[ipNH2][ipC2] += rate_COmole.NH2_C2HP_C2_NH3P_2;
		mole.b[ipC2] += rate_COmole.b_NH2_C2HP_C2_NH3P;

		mole.c[ipC2HP][ipNH3P] += rate_COmole.NH2_C2HP_C2_NH3P_1;
		mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_C2HP_C2_NH3P_2;
		mole.b[ipNH3P] += rate_COmole.b_NH2_C2HP_C2_NH3P;

		/* >>refer  mole   NH2+   C2   -->   C2H+   NH        */

		mole.c[ipC2][ipC2] -= rate_COmole.NH2P_C2_C2HP_NH_1;
		mole.c[ipNH2P][ipC2] -= rate_COmole.NH2P_C2_C2HP_NH_2;
		mole.b[ipC2] -= rate_COmole.b_NH2P_C2_C2HP_NH;

		mole.c[ipC2][ipNH2P] -= rate_COmole.NH2P_C2_C2HP_NH_1;
		mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_C2_C2HP_NH_2;
		mole.b[ipNH2P] -= rate_COmole.b_NH2P_C2_C2HP_NH;

		mole.c[ipC2][ipC2HP] += rate_COmole.NH2P_C2_C2HP_NH_1;
		mole.c[ipNH2P][ipC2HP] += rate_COmole.NH2P_C2_C2HP_NH_2;
		mole.b[ipC2HP] += rate_COmole.b_NH2P_C2_C2HP_NH;

		mole.c[ipC2][ipNH] += rate_COmole.NH2P_C2_C2HP_NH_1;
		mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_C2_C2HP_NH_2;
		mole.b[ipNH] += rate_COmole.b_NH2P_C2_C2HP_NH;

		/* >>refer  mole   NH3   C2H+   -->   C2   NH4+        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.NH3_C2HP_C2_NH4P_1;
		mole.c[ipNH3][ipC2HP] -= rate_COmole.NH3_C2HP_C2_NH4P_2;
		mole.b[ipC2HP] -= rate_COmole.b_NH3_C2HP_C2_NH4P;

		mole.c[ipC2HP][ipNH3] -= rate_COmole.NH3_C2HP_C2_NH4P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C2HP_C2_NH4P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C2HP_C2_NH4P;

		mole.c[ipC2HP][ipC2] += rate_COmole.NH3_C2HP_C2_NH4P_1;
		mole.c[ipNH3][ipC2] += rate_COmole.NH3_C2HP_C2_NH4P_2;
		mole.b[ipC2] += rate_COmole.b_NH3_C2HP_C2_NH4P;

		mole.c[ipC2HP][ipNH4P] += rate_COmole.NH3_C2HP_C2_NH4P_1;
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_C2HP_C2_NH4P_2;
		mole.b[ipNH4P] += rate_COmole.b_NH3_C2HP_C2_NH4P;

		/* >>refer  mole   NH+   C2   -->   C2H+   N        */

		mole.c[ipC2][ipC2] -= rate_COmole.NHP_C2_C2HP_N_1;
		mole.c[ipNHP][ipC2] -= rate_COmole.NHP_C2_C2HP_N_2;
		mole.b[ipC2] -= rate_COmole.b_NHP_C2_C2HP_N;

		mole.c[ipC2][ipNHP] -= rate_COmole.NHP_C2_C2HP_N_1;
		mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_C2_C2HP_N_2;
		mole.b[ipNHP] -= rate_COmole.b_NHP_C2_C2HP_N;

		mole.c[ipC2][ipC2HP] += rate_COmole.NHP_C2_C2HP_N_1;
		mole.c[ipNHP][ipC2HP] += rate_COmole.NHP_C2_C2HP_N_2;
		mole.b[ipC2HP] += rate_COmole.b_NHP_C2_C2HP_N;

		mole.c[ipC2][ipATN] += rate_COmole.NHP_C2_C2HP_N_1;
		mole.c[ipNHP][ipATN] += rate_COmole.NHP_C2_C2HP_N_2;
		mole.b[ipATN] += rate_COmole.b_NHP_C2_C2HP_N;

		/* >>refer  mole   N+   C2H   -->   C2H+   N        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.NP_C2H_C2HP_N_1;
		mole.c[ipNP][ipC2H] -= rate_COmole.NP_C2H_C2HP_N_2;
		mole.b[ipC2H] -= rate_COmole.b_NP_C2H_C2HP_N;

		mole.c[ipC2H][ipNP] -= rate_COmole.NP_C2H_C2HP_N_1;
		mole.c[ipNP][ipNP] -= rate_COmole.NP_C2H_C2HP_N_2;
		mole.b[ipNP] -= rate_COmole.b_NP_C2H_C2HP_N;

		mole.c[ipC2H][ipC2HP] += rate_COmole.NP_C2H_C2HP_N_1;
		mole.c[ipNP][ipC2HP] += rate_COmole.NP_C2H_C2HP_N_2;
		mole.b[ipC2HP] += rate_COmole.b_NP_C2H_C2HP_N;

		mole.c[ipC2H][ipATN] += rate_COmole.NP_C2H_C2HP_N_1;
		mole.c[ipNP][ipATN] += rate_COmole.NP_C2H_C2HP_N_2;
		mole.b[ipATN] += rate_COmole.b_NP_C2H_C2HP_N;

		/* >>refer  mole   O   C2H+   -->   HCO+   C        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.O_C2HP_HCOP_C_1;
		mole.c[ipATO][ipC2HP] -= rate_COmole.O_C2HP_HCOP_C_2;
		mole.b[ipC2HP] -= rate_COmole.b_O_C2HP_HCOP_C;

		mole.c[ipC2HP][ipATO] -= rate_COmole.O_C2HP_HCOP_C_1;
		mole.c[ipATO][ipATO] -= rate_COmole.O_C2HP_HCOP_C_2;
		mole.b[ipATO] -= rate_COmole.b_O_C2HP_HCOP_C;

		mole.c[ipC2HP][ipHCOP] += rate_COmole.O_C2HP_HCOP_C_1;
		mole.c[ipATO][ipHCOP] += rate_COmole.O_C2HP_HCOP_C_2;
		mole.b[ipHCOP] += rate_COmole.b_O_C2HP_HCOP_C;

		mole.c[ipC2HP][ipATC] += rate_COmole.O_C2HP_HCOP_C_1;
		mole.c[ipATO][ipATC] += rate_COmole.O_C2HP_HCOP_C_2;
		mole.b[ipATC] += rate_COmole.b_O_C2HP_HCOP_C;

		/* >>refer  mole   OH+   C2   -->   C2H+   O        */

		mole.c[ipC2][ipC2] -= rate_COmole.OHP_C2_C2HP_O_1;
		mole.c[ipOHP][ipC2] -= rate_COmole.OHP_C2_C2HP_O_2;
		mole.b[ipC2] -= rate_COmole.b_OHP_C2_C2HP_O;

		mole.c[ipC2][ipOHP] -= rate_COmole.OHP_C2_C2HP_O_1;
		mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_C2_C2HP_O_2;
		mole.b[ipOHP] -= rate_COmole.b_OHP_C2_C2HP_O;

		mole.c[ipC2][ipC2HP] += rate_COmole.OHP_C2_C2HP_O_1;
		mole.c[ipOHP][ipC2HP] += rate_COmole.OHP_C2_C2HP_O_2;
		mole.b[ipC2HP] += rate_COmole.b_OHP_C2_C2HP_O;

		mole.c[ipC2][ipATO] += rate_COmole.OHP_C2_C2HP_O_1;
		mole.c[ipOHP][ipATO] += rate_COmole.OHP_C2_C2HP_O_2;
		mole.b[ipATO] += rate_COmole.b_OHP_C2_C2HP_O;

		/* >>refer  mole   OH+   C2H   -->   C2H+   OH        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.OHP_C2H_C2HP_OH_1;
		mole.c[ipOHP][ipC2H] -= rate_COmole.OHP_C2H_C2HP_OH_2;
		mole.b[ipC2H] -= rate_COmole.b_OHP_C2H_C2HP_OH;

		mole.c[ipC2H][ipOHP] -= rate_COmole.OHP_C2H_C2HP_OH_1;
		mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_C2H_C2HP_OH_2;
		mole.b[ipOHP] -= rate_COmole.b_OHP_C2H_C2HP_OH;

		mole.c[ipC2H][ipC2HP] += rate_COmole.OHP_C2H_C2HP_OH_1;
		mole.c[ipOHP][ipC2HP] += rate_COmole.OHP_C2H_C2HP_OH_2;
		mole.b[ipC2HP] += rate_COmole.b_OHP_C2H_C2HP_OH;

		mole.c[ipC2H][ipOH] += rate_COmole.OHP_C2H_C2HP_OH_1;
		mole.c[ipOHP][ipOH] += rate_COmole.OHP_C2H_C2HP_OH_2;
		mole.b[ipOH] += rate_COmole.b_OHP_C2H_C2HP_OH;

		/* >>refer  mole   O+   C2H   -->   C2H+   O        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.OP_C2H_C2HP_O_1;
		mole.c[ipOP][ipC2H] -= rate_COmole.OP_C2H_C2HP_O_2;
		mole.b[ipC2H] -= rate_COmole.b_OP_C2H_C2HP_O;

		mole.c[ipC2H][ipOP] -= rate_COmole.OP_C2H_C2HP_O_1;
		mole.c[ipOP][ipOP] -= rate_COmole.OP_C2H_C2HP_O_2;
		mole.b[ipOP] -= rate_COmole.b_OP_C2H_C2HP_O;

		mole.c[ipC2H][ipC2HP] += rate_COmole.OP_C2H_C2HP_O_1;
		mole.c[ipOP][ipC2HP] += rate_COmole.OP_C2H_C2HP_O_2;
		mole.b[ipC2HP] += rate_COmole.b_OP_C2H_C2HP_O;

		mole.c[ipC2H][ipATO] += rate_COmole.OP_C2H_C2HP_O_1;
		mole.c[ipOP][ipATO] += rate_COmole.OP_C2H_C2HP_O_2;
		mole.b[ipATO] += rate_COmole.b_OP_C2H_C2HP_O;

		/* >>refer  mole   C+   C2H   -->   C3+   H        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.CP_C2H_C3P_H_1;
		mole.c[ipCP][ipC2H] -= rate_COmole.CP_C2H_C3P_H_2;
		mole.b[ipC2H] -= rate_COmole.b_CP_C2H_C3P_H;

		mole.c[ipC2H][ipCP] -= rate_COmole.CP_C2H_C3P_H_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_C2H_C3P_H_2;
		mole.b[ipCP] -= rate_COmole.b_CP_C2H_C3P_H;

		mole.c[ipC2H][ipC3P] += rate_COmole.CP_C2H_C3P_H_1;
		mole.c[ipCP][ipC3P] += rate_COmole.CP_C2H_C3P_H_2;
		mole.b[ipC3P] += rate_COmole.b_CP_C2H_C3P_H;


		/* >>refer  mole   CH+   C2H   -->   C3+   H2        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.CHP_C2H_C3P_H2_1;
		mole.c[ipCHP][ipC2H] -= rate_COmole.CHP_C2H_C3P_H2_2;
		mole.b[ipC2H] -= rate_COmole.b_CHP_C2H_C3P_H2;

		mole.c[ipC2H][ipCHP] -= rate_COmole.CHP_C2H_C3P_H2_1;
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_C2H_C3P_H2_2;
		mole.b[ipCHP] -= rate_COmole.b_CHP_C2H_C3P_H2;

		mole.c[ipC2H][ipC3P] += rate_COmole.CHP_C2H_C3P_H2_1;
		mole.c[ipCHP][ipC3P] += rate_COmole.CHP_C2H_C3P_H2_2;
		mole.b[ipC3P] += rate_COmole.b_CHP_C2H_C3P_H2;

		/* >>refer  mole   C2+   C2   -->   C3+   C        */

		mole.c[ipC2][ipC2] -= rate_COmole.C2P_C2_C3P_C_1;
		mole.c[ipC2P][ipC2] -= rate_COmole.C2P_C2_C3P_C_2;
		mole.b[ipC2] -= rate_COmole.b_C2P_C2_C3P_C;

		mole.c[ipC2][ipC2P] -= rate_COmole.C2P_C2_C3P_C_1;
		mole.c[ipC2P][ipC2P] -= rate_COmole.C2P_C2_C3P_C_2;
		mole.b[ipC2P] -= rate_COmole.b_C2P_C2_C3P_C;

		mole.c[ipC2][ipC3P] += rate_COmole.C2P_C2_C3P_C_1;
		mole.c[ipC2P][ipC3P] += rate_COmole.C2P_C2_C3P_C_2;
		mole.b[ipC3P] += rate_COmole.b_C2P_C2_C3P_C;

		mole.c[ipC2][ipATC] += rate_COmole.C2P_C2_C3P_C_1;
		mole.c[ipC2P][ipATC] += rate_COmole.C2P_C2_C3P_C_2;
		mole.b[ipATC] += rate_COmole.b_C2P_C2_C3P_C;

		/* >>refer  mole   CH+   C2   -->   C3+   H        */

		mole.c[ipC2][ipC2] -= rate_COmole.CHP_C2_C3P_H_1;
		mole.c[ipCHP][ipC2] -= rate_COmole.CHP_C2_C3P_H_2;
		mole.b[ipC2] -= rate_COmole.b_CHP_C2_C3P_H;

		mole.c[ipC2][ipCHP] -= rate_COmole.CHP_C2_C3P_H_1;
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_C2_C3P_H_2;
		mole.b[ipCHP] -= rate_COmole.b_CHP_C2_C3P_H;

		mole.c[ipC2][ipC3P] += rate_COmole.CHP_C2_C3P_H_1;
		mole.c[ipCHP][ipC3P] += rate_COmole.CHP_C2_C3P_H_2;
		mole.b[ipC3P] += rate_COmole.b_CHP_C2_C3P_H;

		/* >>refer  mole   CH   C2+   -->   C3+   H        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.CH_C2P_C3P_H_1;
		mole.c[ipCH][ipC2P] -= rate_COmole.CH_C2P_C3P_H_2;
		mole.b[ipC2P] -= rate_COmole.b_CH_C2P_C3P_H;

		mole.c[ipC2P][ipCH] -= rate_COmole.CH_C2P_C3P_H_1;
		mole.c[ipCH][ipCH] -= rate_COmole.CH_C2P_C3P_H_2;
		mole.b[ipCH] -= rate_COmole.b_CH_C2P_C3P_H;

		mole.c[ipC2P][ipC3P] += rate_COmole.CH_C2P_C3P_H_1;
		mole.c[ipCH][ipC3P] += rate_COmole.CH_C2P_C3P_H_2;
		mole.b[ipC3P] += rate_COmole.b_CH_C2P_C3P_H;

		/* >>refer  mole   C   C2H+   -->   C3+   H        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C_C2HP_C3P_H_1;
		mole.c[ipATC][ipC2HP] -= rate_COmole.C_C2HP_C3P_H_2;
		mole.b[ipC2HP] -= rate_COmole.b_C_C2HP_C3P_H;

		mole.c[ipC2HP][ipATC] -= rate_COmole.C_C2HP_C3P_H_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_C2HP_C3P_H_2;
		mole.b[ipATC] -= rate_COmole.b_C_C2HP_C3P_H;

		mole.c[ipC2HP][ipC3P] += rate_COmole.C_C2HP_C3P_H_1;
		mole.c[ipATC][ipC3P] += rate_COmole.C_C2HP_C3P_H_2;
		mole.b[ipC3P] += rate_COmole.b_C_C2HP_C3P_H;

		/* >>refer  mole   C   C2H2   -->   C3H   H        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.C_C2H2_C3H_H_1;
		mole.c[ipATC][ipC2H2] -= rate_COmole.C_C2H2_C3H_H_2;
		mole.b[ipC2H2] -= rate_COmole.b_C_C2H2_C3H_H;

		mole.c[ipC2H2][ipATC] -= rate_COmole.C_C2H2_C3H_H_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_C2H2_C3H_H_2;
		mole.b[ipATC] -= rate_COmole.b_C_C2H2_C3H_H;

		mole.c[ipC2H2][ipC3H] += rate_COmole.C_C2H2_C3H_H_1;
		mole.c[ipATC][ipC3H] += rate_COmole.C_C2H2_C3H_H_2;
		mole.b[ipC3H] += rate_COmole.b_C_C2H2_C3H_H;

		/* >>refer  mole   C   C2H2+   -->   C3H+   H        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.C_C2H2P_C3HP_H_1;
		mole.c[ipATC][ipC2H2P] -= rate_COmole.C_C2H2P_C3HP_H_2;
		mole.b[ipC2H2P] -= rate_COmole.b_C_C2H2P_C3HP_H;

		mole.c[ipC2H2P][ipATC] -= rate_COmole.C_C2H2P_C3HP_H_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_C2H2P_C3HP_H_2;
		mole.b[ipATC] -= rate_COmole.b_C_C2H2P_C3HP_H;

		mole.c[ipC2H2P][ipC3HP] += rate_COmole.C_C2H2P_C3HP_H_1;
		mole.c[ipATC][ipC3HP] += rate_COmole.C_C2H2P_C3HP_H_2;
		mole.b[ipC3HP] += rate_COmole.b_C_C2H2P_C3HP_H;

		/* >>refer  mole   C2H   HCN+   -->   C2H2+   CN        */

		mole.c[ipHCNP][ipHCNP] -= rate_COmole.C2H_HCNP_C2H2P_CN_1;
		mole.c[ipC2H][ipHCNP] -= rate_COmole.C2H_HCNP_C2H2P_CN_2;
		mole.b[ipHCNP] -= rate_COmole.b_C2H_HCNP_C2H2P_CN;

		mole.c[ipHCNP][ipC2H] -= rate_COmole.C2H_HCNP_C2H2P_CN_1;
		mole.c[ipC2H][ipC2H] -= rate_COmole.C2H_HCNP_C2H2P_CN_2;
		mole.b[ipC2H] -= rate_COmole.b_C2H_HCNP_C2H2P_CN;

		mole.c[ipHCNP][ipC2H2P] += rate_COmole.C2H_HCNP_C2H2P_CN_1;
		mole.c[ipC2H][ipC2H2P] += rate_COmole.C2H_HCNP_C2H2P_CN_2;
		mole.b[ipC2H2P] += rate_COmole.b_C2H_HCNP_C2H2P_CN;

		mole.c[ipHCNP][ipCN] += rate_COmole.C2H_HCNP_C2H2P_CN_1;
		mole.c[ipC2H][ipCN] += rate_COmole.C2H_HCNP_C2H2P_CN_2;
		mole.b[ipCN] += rate_COmole.b_C2H_HCNP_C2H2P_CN;

		/* >>refer  mole   C2H2+   HCN   -->   HCNH+   C2H        */

		mole.c[ipHCN][ipHCN] -= rate_COmole.C2H2P_HCN_HCNHP_C2H_1;
		mole.c[ipC2H2P][ipHCN] -= rate_COmole.C2H2P_HCN_HCNHP_C2H_2;
		mole.b[ipHCN] -= rate_COmole.b_C2H2P_HCN_HCNHP_C2H;

		mole.c[ipHCN][ipC2H2P] -= rate_COmole.C2H2P_HCN_HCNHP_C2H_1;
		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.C2H2P_HCN_HCNHP_C2H_2;
		mole.b[ipC2H2P] -= rate_COmole.b_C2H2P_HCN_HCNHP_C2H;

		mole.c[ipHCN][ipHCNHP] += rate_COmole.C2H2P_HCN_HCNHP_C2H_1;
		mole.c[ipC2H2P][ipHCNHP] += rate_COmole.C2H2P_HCN_HCNHP_C2H_2;
		mole.b[ipHCNHP] += rate_COmole.b_C2H2P_HCN_HCNHP_C2H;

		mole.c[ipHCN][ipC2H] += rate_COmole.C2H2P_HCN_HCNHP_C2H_1;
		mole.c[ipC2H2P][ipC2H] += rate_COmole.C2H2P_HCN_HCNHP_C2H_2;
		mole.b[ipC2H] += rate_COmole.b_C2H2P_HCN_HCNHP_C2H;

		/* >>refer  mole   C2H2+   NO   -->   NO+   C2H2        */

		mole.c[ipNO][ipNO] -= rate_COmole.C2H2P_NO_NOP_C2H2_1;
		mole.c[ipC2H2P][ipNO] -= rate_COmole.C2H2P_NO_NOP_C2H2_2;
		mole.b[ipNO] -= rate_COmole.b_C2H2P_NO_NOP_C2H2;

		mole.c[ipNO][ipC2H2P] -= rate_COmole.C2H2P_NO_NOP_C2H2_1;
		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.C2H2P_NO_NOP_C2H2_2;
		mole.b[ipC2H2P] -= rate_COmole.b_C2H2P_NO_NOP_C2H2;

		mole.c[ipNO][ipNOP] += rate_COmole.C2H2P_NO_NOP_C2H2_1;
		mole.c[ipC2H2P][ipNOP] += rate_COmole.C2H2P_NO_NOP_C2H2_2;
		mole.b[ipNOP] += rate_COmole.b_C2H2P_NO_NOP_C2H2;

		mole.c[ipNO][ipC2H2] += rate_COmole.C2H2P_NO_NOP_C2H2_1;
		mole.c[ipC2H2P][ipC2H2] += rate_COmole.C2H2P_NO_NOP_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_C2H2P_NO_NOP_C2H2;

		/* >>refer  mole   C2H+   HCN   -->   C2H2+   CN        */

		mole.c[ipHCN][ipHCN] -= rate_COmole.C2HP_HCN_C2H2P_CN_1;
		mole.c[ipC2HP][ipHCN] -= rate_COmole.C2HP_HCN_C2H2P_CN_2;
		mole.b[ipHCN] -= rate_COmole.b_C2HP_HCN_C2H2P_CN;

		mole.c[ipHCN][ipC2HP] -= rate_COmole.C2HP_HCN_C2H2P_CN_1;
		mole.c[ipC2HP][ipC2HP] -= rate_COmole.C2HP_HCN_C2H2P_CN_2;
		mole.b[ipC2HP] -= rate_COmole.b_C2HP_HCN_C2H2P_CN;

		mole.c[ipHCN][ipC2H2P] += rate_COmole.C2HP_HCN_C2H2P_CN_1;
		mole.c[ipC2HP][ipC2H2P] += rate_COmole.C2HP_HCN_C2H2P_CN_2;
		mole.b[ipC2H2P] += rate_COmole.b_C2HP_HCN_C2H2P_CN;

		mole.c[ipHCN][ipCN] += rate_COmole.C2HP_HCN_C2H2P_CN_1;
		mole.c[ipC2HP][ipCN] += rate_COmole.C2HP_HCN_C2H2P_CN_2;
		mole.b[ipCN] += rate_COmole.b_C2HP_HCN_C2H2P_CN;

		/* >>refer  mole   CH   C2H+   -->   C3H+   H        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.CH_C2HP_C3HP_H_1;
		mole.c[ipCH][ipC2HP] -= rate_COmole.CH_C2HP_C3HP_H_2;
		mole.b[ipC2HP] -= rate_COmole.b_CH_C2HP_C3HP_H;

		mole.c[ipC2HP][ipCH] -= rate_COmole.CH_C2HP_C3HP_H_1;
		mole.c[ipCH][ipCH] -= rate_COmole.CH_C2HP_C3HP_H_2;
		mole.b[ipCH] -= rate_COmole.b_CH_C2HP_C3HP_H;

		mole.c[ipC2HP][ipC3HP] += rate_COmole.CH_C2HP_C3HP_H_1;
		mole.c[ipCH][ipC3HP] += rate_COmole.CH_C2HP_C3HP_H_2;
		mole.b[ipC3HP] += rate_COmole.b_CH_C2HP_C3HP_H;

		/* >>refer  mole   CH   CH3+   -->   C2H2+   H2        */

		mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH_CH3P_C2H2P_H2_1;
		mole.c[ipCH][ipCH3P] -= rate_COmole.CH_CH3P_C2H2P_H2_2;
		mole.b[ipCH3P] -= rate_COmole.b_CH_CH3P_C2H2P_H2;

		mole.c[ipCH3P][ipCH] -= rate_COmole.CH_CH3P_C2H2P_H2_1;
		mole.c[ipCH][ipCH] -= rate_COmole.CH_CH3P_C2H2P_H2_2;
		mole.b[ipCH] -= rate_COmole.b_CH_CH3P_C2H2P_H2;

		mole.c[ipCH3P][ipC2H2P] += rate_COmole.CH_CH3P_C2H2P_H2_1;
		mole.c[ipCH][ipC2H2P] += rate_COmole.CH_CH3P_C2H2P_H2_2;
		mole.b[ipC2H2P] += rate_COmole.b_CH_CH3P_C2H2P_H2;

		/* >>refer  mole   CH2   C2+   -->   C3H+   H        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.CH2_C2P_C3HP_H_1;
		mole.c[ipCH2][ipC2P] -= rate_COmole.CH2_C2P_C3HP_H_2;
		mole.b[ipC2P] -= rate_COmole.b_CH2_C2P_C3HP_H;

		mole.c[ipC2P][ipCH2] -= rate_COmole.CH2_C2P_C3HP_H_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_C2P_C3HP_H_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_C2P_C3HP_H;

		mole.c[ipC2P][ipC3HP] += rate_COmole.CH2_C2P_C3HP_H_1;
		mole.c[ipCH2][ipC3HP] += rate_COmole.CH2_C2P_C3HP_H_2;
		mole.b[ipC3HP] += rate_COmole.b_CH2_C2P_C3HP_H;

		/* >>refer  mole   CH2   CH2   -->   C2H2   H   H     */

		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H_H_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H_H_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_CH2_C2H2_H_H;

		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H_H_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H_H_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_CH2_C2H2_H_H;

		mole.c[ipCH2][ipC2H2] += rate_COmole.CH2_CH2_C2H2_H_H_1;
		mole.c[ipCH2][ipC2H2] += rate_COmole.CH2_CH2_C2H2_H_H_2;
		mole.b[ipC2H2] += rate_COmole.b_CH2_CH2_C2H2_H_H;

		/* >>refer  mole   CH2   CH2   -->   C2H2   H2        */

		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H2_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H2_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_CH2_C2H2_H2;

		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H2_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH2_C2H2_H2_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_CH2_C2H2_H2;

		mole.c[ipCH2][ipC2H2] += rate_COmole.CH2_CH2_C2H2_H2_1;
		mole.c[ipCH2][ipC2H2] += rate_COmole.CH2_CH2_C2H2_H2_2;
		mole.b[ipC2H2] += rate_COmole.b_CH2_CH2_C2H2_H2;

		/* >>refer  mole   CH3+   C2   -->   C3H+   H2        */

		mole.c[ipC2][ipC2] -= rate_COmole.CH3P_C2_C3HP_H2_1;
		mole.c[ipCH3P][ipC2] -= rate_COmole.CH3P_C2_C3HP_H2_2;
		mole.b[ipC2] -= rate_COmole.b_CH3P_C2_C3HP_H2;

		mole.c[ipC2][ipCH3P] -= rate_COmole.CH3P_C2_C3HP_H2_1;
		mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH3P_C2_C3HP_H2_2;
		mole.b[ipCH3P] -= rate_COmole.b_CH3P_C2_C3HP_H2;

		mole.c[ipC2][ipC3HP] += rate_COmole.CH3P_C2_C3HP_H2_1;
		mole.c[ipCH3P][ipC3HP] += rate_COmole.CH3P_C2_C3HP_H2_2;
		mole.b[ipC3HP] += rate_COmole.b_CH3P_C2_C3HP_H2;

		/* >>refer  mole   CH4   C2H+   -->   C2H2+   CH3        */

		mole.c[ipC2HP][ipC2HP] -= rate_COmole.CH4_C2HP_C2H2P_CH3_1;
		mole.c[ipCH4][ipC2HP] -= rate_COmole.CH4_C2HP_C2H2P_CH3_2;
		mole.b[ipC2HP] -= rate_COmole.b_CH4_C2HP_C2H2P_CH3;

		mole.c[ipC2HP][ipCH4] -= rate_COmole.CH4_C2HP_C2H2P_CH3_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_C2HP_C2H2P_CH3_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_C2HP_C2H2P_CH3;

		mole.c[ipC2HP][ipC2H2P] += rate_COmole.CH4_C2HP_C2H2P_CH3_1;
		mole.c[ipCH4][ipC2H2P] += rate_COmole.CH4_C2HP_C2H2P_CH3_2;
		mole.b[ipC2H2P] += rate_COmole.b_CH4_C2HP_C2H2P_CH3;

		mole.c[ipC2HP][ipCH3] += rate_COmole.CH4_C2HP_C2H2P_CH3_1;
		mole.c[ipCH4][ipCH3] += rate_COmole.CH4_C2HP_C2H2P_CH3_2;
		mole.b[ipCH3] += rate_COmole.b_CH4_C2HP_C2H2P_CH3;

		/* >>refer  mole   CH4   C2+   -->   C2H2+   CH2        */

		mole.c[ipC2P][ipC2P] -= rate_COmole.CH4_C2P_C2H2P_CH2_1;
		mole.c[ipCH4][ipC2P] -= rate_COmole.CH4_C2P_C2H2P_CH2_2;
		mole.b[ipC2P] -= rate_COmole.b_CH4_C2P_C2H2P_CH2;

		mole.c[ipC2P][ipCH4] -= rate_COmole.CH4_C2P_C2H2P_CH2_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_C2P_C2H2P_CH2_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_C2P_C2H2P_CH2;

		mole.c[ipC2P][ipC2H2P] += rate_COmole.CH4_C2P_C2H2P_CH2_1;
		mole.c[ipCH4][ipC2H2P] += rate_COmole.CH4_C2P_C2H2P_CH2_2;
		mole.b[ipC2H2P] += rate_COmole.b_CH4_C2P_C2H2P_CH2;

		mole.c[ipC2P][ipCH2] += rate_COmole.CH4_C2P_C2H2P_CH2_1;
		mole.c[ipCH4][ipCH2] += rate_COmole.CH4_C2P_C2H2P_CH2_2;
		mole.b[ipCH2] += rate_COmole.b_CH4_C2P_C2H2P_CH2;

		/* >>refer  mole   CH4   C2+   -->   C3H+   H2   H     */

		mole.c[ipC2P][ipC2P] -= rate_COmole.CH4_C2P_C3HP_H2_H_1;
		mole.c[ipCH4][ipC2P] -= rate_COmole.CH4_C2P_C3HP_H2_H_2;
		mole.b[ipC2P] -= rate_COmole.b_CH4_C2P_C3HP_H2_H;

		mole.c[ipC2P][ipCH4] -= rate_COmole.CH4_C2P_C3HP_H2_H_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_C2P_C3HP_H2_H_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_C2P_C3HP_H2_H;

		mole.c[ipC2P][ipC3HP] += rate_COmole.CH4_C2P_C3HP_H2_H_1;
		mole.c[ipCH4][ipC3HP] += rate_COmole.CH4_C2P_C3HP_H2_H_2;
		mole.b[ipC3HP] += rate_COmole.b_CH4_C2P_C3HP_H2_H;

		/* >>refer  mole   CH4+   C2H2   -->   C2H2+   CH4        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.CH4P_C2H2_C2H2P_CH4_1;
		mole.c[ipCH4P][ipC2H2] -= rate_COmole.CH4P_C2H2_C2H2P_CH4_2;
		mole.b[ipC2H2] -= rate_COmole.b_CH4P_C2H2_C2H2P_CH4;

		mole.c[ipC2H2][ipCH4P] -= rate_COmole.CH4P_C2H2_C2H2P_CH4_1;
		mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4P_C2H2_C2H2P_CH4_2;
		mole.b[ipCH4P] -= rate_COmole.b_CH4P_C2H2_C2H2P_CH4;

		mole.c[ipC2H2][ipC2H2P] += rate_COmole.CH4P_C2H2_C2H2P_CH4_1;
		mole.c[ipCH4P][ipC2H2P] += rate_COmole.CH4P_C2H2_C2H2P_CH4_2;
		mole.b[ipC2H2P] += rate_COmole.b_CH4P_C2H2_C2H2P_CH4;

		mole.c[ipC2H2][ipCH4] += rate_COmole.CH4P_C2H2_C2H2P_CH4_1;
		mole.c[ipCH4P][ipCH4] += rate_COmole.CH4P_C2H2_C2H2P_CH4_2;
		mole.b[ipCH4] += rate_COmole.b_CH4P_C2H2_C2H2P_CH4;

		/* >>refer  mole   CH5+   C2H   -->   C2H2+   CH4        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.CH5P_C2H_C2H2P_CH4_1;
		mole.c[ipCH5P][ipC2H] -= rate_COmole.CH5P_C2H_C2H2P_CH4_2;
		mole.b[ipC2H] -= rate_COmole.b_CH5P_C2H_C2H2P_CH4;

		mole.c[ipC2H][ipCH5P] -= rate_COmole.CH5P_C2H_C2H2P_CH4_1;
		mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_C2H_C2H2P_CH4_2;
		mole.b[ipCH5P] -= rate_COmole.b_CH5P_C2H_C2H2P_CH4;

		mole.c[ipC2H][ipC2H2P] += rate_COmole.CH5P_C2H_C2H2P_CH4_1;
		mole.c[ipCH5P][ipC2H2P] += rate_COmole.CH5P_C2H_C2H2P_CH4_2;
		mole.b[ipC2H2P] += rate_COmole.b_CH5P_C2H_C2H2P_CH4;

		mole.c[ipC2H][ipCH4] += rate_COmole.CH5P_C2H_C2H2P_CH4_1;
		mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_C2H_C2H2P_CH4_2;
		mole.b[ipCH4] += rate_COmole.b_CH5P_C2H_C2H2P_CH4;

		/* >>refer  mole   CH+   CH4   -->   C2H2+   H2   H     */

		mole.c[ipCH4][ipCH4] -= rate_COmole.CHP_CH4_C2H2P_H2_H_1;
		mole.c[ipCHP][ipCH4] -= rate_COmole.CHP_CH4_C2H2P_H2_H_2;
		mole.b[ipCH4] -= rate_COmole.b_CHP_CH4_C2H2P_H2_H;

		mole.c[ipCH4][ipCHP] -= rate_COmole.CHP_CH4_C2H2P_H2_H_1;
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_CH4_C2H2P_H2_H_2;
		mole.b[ipCHP] -= rate_COmole.b_CHP_CH4_C2H2P_H2_H;

		mole.c[ipCH4][ipC2H2P] += rate_COmole.CHP_CH4_C2H2P_H2_H_1;
		mole.c[ipCHP][ipC2H2P] += rate_COmole.CHP_CH4_C2H2P_H2_H_2;
		mole.b[ipC2H2P] += rate_COmole.b_CHP_CH4_C2H2P_H2_H;

		/* >>refer  mole   C+   C2H2   -->   C3H+   H        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.CP_C2H2_C3HP_H_1;
		mole.c[ipCP][ipC2H2] -= rate_COmole.CP_C2H2_C3HP_H_2;
		mole.b[ipC2H2] -= rate_COmole.b_CP_C2H2_C3HP_H;

		mole.c[ipC2H2][ipCP] -= rate_COmole.CP_C2H2_C3HP_H_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_C2H2_C3HP_H_2;
		mole.b[ipCP] -= rate_COmole.b_CP_C2H2_C3HP_H;

		mole.c[ipC2H2][ipC3HP] += rate_COmole.CP_C2H2_C3HP_H_1;
		mole.c[ipCP][ipC3HP] += rate_COmole.CP_C2H2_C3HP_H_2;
		mole.b[ipC3HP] += rate_COmole.b_CP_C2H2_C3HP_H;

		/* >>refer  mole   C+   CH3   -->   C2H2+   H        */

		mole.c[ipCH3][ipCH3] -= rate_COmole.CP_CH3_C2H2P_H_1;
		mole.c[ipCP][ipCH3] -= rate_COmole.CP_CH3_C2H2P_H_2;
		mole.b[ipCH3] -= rate_COmole.b_CP_CH3_C2H2P_H;

		mole.c[ipCH3][ipCP] -= rate_COmole.CP_CH3_C2H2P_H_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CH3_C2H2P_H_2;
		mole.b[ipCP] -= rate_COmole.b_CP_CH3_C2H2P_H;

		mole.c[ipCH3][ipC2H2P] += rate_COmole.CP_CH3_C2H2P_H_1;
		mole.c[ipCP][ipC2H2P] += rate_COmole.CP_CH3_C2H2P_H_2;
		mole.b[ipC2H2P] += rate_COmole.b_CP_CH3_C2H2P_H;

		/* >>refer  mole   C+   CH4   -->   C2H2+   H2        */

		mole.c[ipCH4][ipCH4] -= rate_COmole.CP_CH4_C2H2P_H2_1;
		mole.c[ipCP][ipCH4] -= rate_COmole.CP_CH4_C2H2P_H2_2;
		mole.b[ipCH4] -= rate_COmole.b_CP_CH4_C2H2P_H2;

		mole.c[ipCH4][ipCP] -= rate_COmole.CP_CH4_C2H2P_H2_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CH4_C2H2P_H2_2;
		mole.b[ipCP] -= rate_COmole.b_CP_CH4_C2H2P_H2;

		mole.c[ipCH4][ipC2H2P] += rate_COmole.CP_CH4_C2H2P_H2_1;
		mole.c[ipCP][ipC2H2P] += rate_COmole.CP_CH4_C2H2P_H2_2;
		mole.b[ipC2H2P] += rate_COmole.b_CP_CH4_C2H2P_H2;

		/* >>refer  mole   H2O   C2H2+   -->   C2H   H3O+        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.H2O_C2H2P_C2H_H3OP_1;
		mole.c[ipH2O][ipC2H2P] -= rate_COmole.H2O_C2H2P_C2H_H3OP_2;
		mole.b[ipC2H2P] -= rate_COmole.b_H2O_C2H2P_C2H_H3OP;

		mole.c[ipC2H2P][ipH2O] -= rate_COmole.H2O_C2H2P_C2H_H3OP_1;
		mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_C2H2P_C2H_H3OP_2;
		mole.b[ipH2O] -= rate_COmole.b_H2O_C2H2P_C2H_H3OP;

		mole.c[ipC2H2P][ipC2H] += rate_COmole.H2O_C2H2P_C2H_H3OP_1;
		mole.c[ipH2O][ipC2H] += rate_COmole.H2O_C2H2P_C2H_H3OP_2;
		mole.b[ipC2H] += rate_COmole.b_H2O_C2H2P_C2H_H3OP;

		mole.c[ipC2H2P][ipH3OP] += rate_COmole.H2O_C2H2P_C2H_H3OP_1;
		mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_C2H2P_C2H_H3OP_2;
		mole.b[ipH3OP] += rate_COmole.b_H2O_C2H2P_C2H_H3OP;

		/* >>refer  mole   H2O   C3H+   -->   HCO+   C2H2        */

		mole.c[ipC3HP][ipC3HP] -= rate_COmole.H2O_C3HP_HCOP_C2H2_1;
		mole.c[ipH2O][ipC3HP] -= rate_COmole.H2O_C3HP_HCOP_C2H2_2;
		mole.b[ipC3HP] -= rate_COmole.b_H2O_C3HP_HCOP_C2H2;

		mole.c[ipC3HP][ipH2O] -= rate_COmole.H2O_C3HP_HCOP_C2H2_1;
		mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_C3HP_HCOP_C2H2_2;
		mole.b[ipH2O] -= rate_COmole.b_H2O_C3HP_HCOP_C2H2;

		mole.c[ipC3HP][ipHCOP] += rate_COmole.H2O_C3HP_HCOP_C2H2_1;
		mole.c[ipH2O][ipHCOP] += rate_COmole.H2O_C3HP_HCOP_C2H2_2;
		mole.b[ipHCOP] += rate_COmole.b_H2O_C3HP_HCOP_C2H2;

		mole.c[ipC3HP][ipC2H2] += rate_COmole.H2O_C3HP_HCOP_C2H2_1;
		mole.c[ipH2O][ipC2H2] += rate_COmole.H2O_C3HP_HCOP_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_H2O_C3HP_HCOP_C2H2;

		/* >>refer  mole   H2O+   C2H   -->   C2H2+   OH        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.H2OP_C2H_C2H2P_OH_1;
		mole.c[ipH2OP][ipC2H] -= rate_COmole.H2OP_C2H_C2H2P_OH_2;
		mole.b[ipC2H] -= rate_COmole.b_H2OP_C2H_C2H2P_OH;

		mole.c[ipC2H][ipH2OP] -= rate_COmole.H2OP_C2H_C2H2P_OH_1;
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_C2H_C2H2P_OH_2;
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_C2H_C2H2P_OH;

		mole.c[ipC2H][ipC2H2P] += rate_COmole.H2OP_C2H_C2H2P_OH_1;
		mole.c[ipH2OP][ipC2H2P] += rate_COmole.H2OP_C2H_C2H2P_OH_2;
		mole.b[ipC2H2P] += rate_COmole.b_H2OP_C2H_C2H2P_OH;

		mole.c[ipC2H][ipOH] += rate_COmole.H2OP_C2H_C2H2P_OH_1;
		mole.c[ipH2OP][ipOH] += rate_COmole.H2OP_C2H_C2H2P_OH_2;
		mole.b[ipOH] += rate_COmole.b_H2OP_C2H_C2H2P_OH;

		/* >>refer  mole   H2O+   C2H2   -->   C2H2+   H2O        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.H2OP_C2H2_C2H2P_H2O_1;
		mole.c[ipH2OP][ipC2H2] -= rate_COmole.H2OP_C2H2_C2H2P_H2O_2;
		mole.b[ipC2H2] -= rate_COmole.b_H2OP_C2H2_C2H2P_H2O;

		mole.c[ipC2H2][ipH2OP] -= rate_COmole.H2OP_C2H2_C2H2P_H2O_1;
		mole.c[ipH2OP][ipH2OP] -= rate_COmole.H2OP_C2H2_C2H2P_H2O_2;
		mole.b[ipH2OP] -= rate_COmole.b_H2OP_C2H2_C2H2P_H2O;

		mole.c[ipC2H2][ipC2H2P] += rate_COmole.H2OP_C2H2_C2H2P_H2O_1;
		mole.c[ipH2OP][ipC2H2P] += rate_COmole.H2OP_C2H2_C2H2P_H2O_2;
		mole.b[ipC2H2P] += rate_COmole.b_H2OP_C2H2_C2H2P_H2O;

		mole.c[ipC2H2][ipH2O] += rate_COmole.H2OP_C2H2_C2H2P_H2O_1;
		mole.c[ipH2OP][ipH2O] += rate_COmole.H2OP_C2H2_C2H2P_H2O_2;
		mole.b[ipH2O] += rate_COmole.b_H2OP_C2H2_C2H2P_H2O;

		/* >>refer  mole   H3O+   C3   -->   C3H+   H2O        */

		mole.c[ipC3][ipC3] -= rate_COmole.H3OP_C3_C3HP_H2O_1;
		mole.c[ipH3OP][ipC3] -= rate_COmole.H3OP_C3_C3HP_H2O_2;
		mole.b[ipC3] -= rate_COmole.b_H3OP_C3_C3HP_H2O;

		mole.c[ipC3][ipH3OP] -= rate_COmole.H3OP_C3_C3HP_H2O_1;
		mole.c[ipH3OP][ipH3OP] -= rate_COmole.H3OP_C3_C3HP_H2O_2;
		mole.b[ipH3OP] -= rate_COmole.b_H3OP_C3_C3HP_H2O;

		mole.c[ipC3][ipC3HP] += rate_COmole.H3OP_C3_C3HP_H2O_1;
		mole.c[ipH3OP][ipC3HP] += rate_COmole.H3OP_C3_C3HP_H2O_2;
		mole.b[ipC3HP] += rate_COmole.b_H3OP_C3_C3HP_H2O;

		mole.c[ipC3][ipH2O] += rate_COmole.H3OP_C3_C3HP_H2O_1;
		mole.c[ipH3OP][ipH2O] += rate_COmole.H3OP_C3_C3HP_H2O_2;
		mole.b[ipH2O] += rate_COmole.b_H3OP_C3_C3HP_H2O;

		/* >>refer  mole   N   C2H2+   -->   HCN   CH+        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.N_C2H2P_HCN_CHP_1;
		mole.c[ipATN][ipC2H2P] -= rate_COmole.N_C2H2P_HCN_CHP_2;
		mole.b[ipC2H2P] -= rate_COmole.b_N_C2H2P_HCN_CHP;

		mole.c[ipC2H2P][ipATN] -= rate_COmole.N_C2H2P_HCN_CHP_1;
		mole.c[ipATN][ipATN] -= rate_COmole.N_C2H2P_HCN_CHP_2;
		mole.b[ipATN] -= rate_COmole.b_N_C2H2P_HCN_CHP;

		mole.c[ipC2H2P][ipHCN] += rate_COmole.N_C2H2P_HCN_CHP_1;
		mole.c[ipATN][ipHCN] += rate_COmole.N_C2H2P_HCN_CHP_2;
		mole.b[ipHCN] += rate_COmole.b_N_C2H2P_HCN_CHP;

		mole.c[ipC2H2P][ipCHP] += rate_COmole.N_C2H2P_HCN_CHP_1;
		mole.c[ipATN][ipCHP] += rate_COmole.N_C2H2P_HCN_CHP_2;
		mole.b[ipCHP] += rate_COmole.b_N_C2H2P_HCN_CHP;

		/* >>refer  mole   NH2   C2H2+   -->   C2H   NH3+        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.NH2_C2H2P_C2H_NH3P_1;
		mole.c[ipNH2][ipC2H2P] -= rate_COmole.NH2_C2H2P_C2H_NH3P_2;
		mole.b[ipC2H2P] -= rate_COmole.b_NH2_C2H2P_C2H_NH3P;

		mole.c[ipC2H2P][ipNH2] -= rate_COmole.NH2_C2H2P_C2H_NH3P_1;
		mole.c[ipNH2][ipNH2] -= rate_COmole.NH2_C2H2P_C2H_NH3P_2;
		mole.b[ipNH2] -= rate_COmole.b_NH2_C2H2P_C2H_NH3P;

		mole.c[ipC2H2P][ipC2H] += rate_COmole.NH2_C2H2P_C2H_NH3P_1;
		mole.c[ipNH2][ipC2H] += rate_COmole.NH2_C2H2P_C2H_NH3P_2;
		mole.b[ipC2H] += rate_COmole.b_NH2_C2H2P_C2H_NH3P;

		mole.c[ipC2H2P][ipNH3P] += rate_COmole.NH2_C2H2P_C2H_NH3P_1;
		mole.c[ipNH2][ipNH3P] += rate_COmole.NH2_C2H2P_C2H_NH3P_2;
		mole.b[ipNH3P] += rate_COmole.b_NH2_C2H2P_C2H_NH3P;

		/* >>refer  mole   NH2+   C2H   -->   C2H2+   NH        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.NH2P_C2H_C2H2P_NH_1;
		mole.c[ipNH2P][ipC2H] -= rate_COmole.NH2P_C2H_C2H2P_NH_2;
		mole.b[ipC2H] -= rate_COmole.b_NH2P_C2H_C2H2P_NH;

		mole.c[ipC2H][ipNH2P] -= rate_COmole.NH2P_C2H_C2H2P_NH_1;
		mole.c[ipNH2P][ipNH2P] -= rate_COmole.NH2P_C2H_C2H2P_NH_2;
		mole.b[ipNH2P] -= rate_COmole.b_NH2P_C2H_C2H2P_NH;

		mole.c[ipC2H][ipC2H2P] += rate_COmole.NH2P_C2H_C2H2P_NH_1;
		mole.c[ipNH2P][ipC2H2P] += rate_COmole.NH2P_C2H_C2H2P_NH_2;
		mole.b[ipC2H2P] += rate_COmole.b_NH2P_C2H_C2H2P_NH;

		mole.c[ipC2H][ipNH] += rate_COmole.NH2P_C2H_C2H2P_NH_1;
		mole.c[ipNH2P][ipNH] += rate_COmole.NH2P_C2H_C2H2P_NH_2;
		mole.b[ipNH] += rate_COmole.b_NH2P_C2H_C2H2P_NH;

		/* >>refer  mole   NH3   C2H2+   -->   C2H   NH4+        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.NH3_C2H2P_C2H_NH4P_1;
		mole.c[ipNH3][ipC2H2P] -= rate_COmole.NH3_C2H2P_C2H_NH4P_2;
		mole.b[ipC2H2P] -= rate_COmole.b_NH3_C2H2P_C2H_NH4P;

		mole.c[ipC2H2P][ipNH3] -= rate_COmole.NH3_C2H2P_C2H_NH4P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C2H2P_C2H_NH4P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C2H2P_C2H_NH4P;

		mole.c[ipC2H2P][ipC2H] += rate_COmole.NH3_C2H2P_C2H_NH4P_1;
		mole.c[ipNH3][ipC2H] += rate_COmole.NH3_C2H2P_C2H_NH4P_2;
		mole.b[ipC2H] += rate_COmole.b_NH3_C2H2P_C2H_NH4P;

		mole.c[ipC2H2P][ipNH4P] += rate_COmole.NH3_C2H2P_C2H_NH4P_1;
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_C2H2P_C2H_NH4P_2;
		mole.b[ipNH4P] += rate_COmole.b_NH3_C2H2P_C2H_NH4P;

		/* >>refer  mole   NH3   C2H2+   -->   C2H2   NH3+        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.NH3_C2H2P_C2H2_NH3P_1;
		mole.c[ipNH3][ipC2H2P] -= rate_COmole.NH3_C2H2P_C2H2_NH3P_2;
		mole.b[ipC2H2P] -= rate_COmole.b_NH3_C2H2P_C2H2_NH3P;

		mole.c[ipC2H2P][ipNH3] -= rate_COmole.NH3_C2H2P_C2H2_NH3P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C2H2P_C2H2_NH3P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C2H2P_C2H2_NH3P;

		mole.c[ipC2H2P][ipC2H2] += rate_COmole.NH3_C2H2P_C2H2_NH3P_1;
		mole.c[ipNH3][ipC2H2] += rate_COmole.NH3_C2H2P_C2H2_NH3P_2;
		mole.b[ipC2H2] += rate_COmole.b_NH3_C2H2P_C2H2_NH3P;

		mole.c[ipC2H2P][ipNH3P] += rate_COmole.NH3_C2H2P_C2H2_NH3P_1;
		mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_C2H2P_C2H2_NH3P_2;
		mole.b[ipNH3P] += rate_COmole.b_NH3_C2H2P_C2H2_NH3P;

		/* >>refer  mole   NH3   C3H+   -->   C3   NH4+        */

		mole.c[ipC3HP][ipC3HP] -= rate_COmole.NH3_C3HP_C3_NH4P_1;
		mole.c[ipNH3][ipC3HP] -= rate_COmole.NH3_C3HP_C3_NH4P_2;
		mole.b[ipC3HP] -= rate_COmole.b_NH3_C3HP_C3_NH4P;

		mole.c[ipC3HP][ipNH3] -= rate_COmole.NH3_C3HP_C3_NH4P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C3HP_C3_NH4P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C3HP_C3_NH4P;

		mole.c[ipC3HP][ipC3] += rate_COmole.NH3_C3HP_C3_NH4P_1;
		mole.c[ipNH3][ipC3] += rate_COmole.NH3_C3HP_C3_NH4P_2;
		mole.b[ipC3] += rate_COmole.b_NH3_C3HP_C3_NH4P;

		mole.c[ipC3HP][ipNH4P] += rate_COmole.NH3_C3HP_C3_NH4P_1;
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_C3HP_C3_NH4P_2;
		mole.b[ipNH4P] += rate_COmole.b_NH3_C3HP_C3_NH4P;

		/* >>refer  mole   NH3   C3H+   -->   C3H   NH3+        */

		mole.c[ipC3HP][ipC3HP] -= rate_COmole.NH3_C3HP_C3H_NH3P_1;
		mole.c[ipNH3][ipC3HP] -= rate_COmole.NH3_C3HP_C3H_NH3P_2;
		mole.b[ipC3HP] -= rate_COmole.b_NH3_C3HP_C3H_NH3P;

		mole.c[ipC3HP][ipNH3] -= rate_COmole.NH3_C3HP_C3H_NH3P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C3HP_C3H_NH3P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C3HP_C3H_NH3P;

		mole.c[ipC3HP][ipC3H] += rate_COmole.NH3_C3HP_C3H_NH3P_1;
		mole.c[ipNH3][ipC3H] += rate_COmole.NH3_C3HP_C3H_NH3P_2;
		mole.b[ipC3H] += rate_COmole.b_NH3_C3HP_C3H_NH3P;

		mole.c[ipC3HP][ipNH3P] += rate_COmole.NH3_C3HP_C3H_NH3P_1;
		mole.c[ipNH3][ipNH3P] += rate_COmole.NH3_C3HP_C3H_NH3P_2;
		mole.b[ipNH3P] += rate_COmole.b_NH3_C3HP_C3H_NH3P;

		/* >>refer  mole   NH3+   C2   -->   C2H2+   NH        */

		mole.c[ipC2][ipC2] -= rate_COmole.NH3P_C2_C2H2P_NH_1;
		mole.c[ipNH3P][ipC2] -= rate_COmole.NH3P_C2_C2H2P_NH_2;
		mole.b[ipC2] -= rate_COmole.b_NH3P_C2_C2H2P_NH;

		mole.c[ipC2][ipNH3P] -= rate_COmole.NH3P_C2_C2H2P_NH_1;
		mole.c[ipNH3P][ipNH3P] -= rate_COmole.NH3P_C2_C2H2P_NH_2;
		mole.b[ipNH3P] -= rate_COmole.b_NH3P_C2_C2H2P_NH;

		mole.c[ipC2][ipC2H2P] += rate_COmole.NH3P_C2_C2H2P_NH_1;
		mole.c[ipNH3P][ipC2H2P] += rate_COmole.NH3P_C2_C2H2P_NH_2;
		mole.b[ipC2H2P] += rate_COmole.b_NH3P_C2_C2H2P_NH;

		mole.c[ipC2][ipNH] += rate_COmole.NH3P_C2_C2H2P_NH_1;
		mole.c[ipNH3P][ipNH] += rate_COmole.NH3P_C2_C2H2P_NH_2;
		mole.b[ipNH] += rate_COmole.b_NH3P_C2_C2H2P_NH;

		/* >>refer  mole   NH+   C2H   -->   C2H2+   N        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.NHP_C2H_C2H2P_N_1;
		mole.c[ipNHP][ipC2H] -= rate_COmole.NHP_C2H_C2H2P_N_2;
		mole.b[ipC2H] -= rate_COmole.b_NHP_C2H_C2H2P_N;

		mole.c[ipC2H][ipNHP] -= rate_COmole.NHP_C2H_C2H2P_N_1;
		mole.c[ipNHP][ipNHP] -= rate_COmole.NHP_C2H_C2H2P_N_2;
		mole.b[ipNHP] -= rate_COmole.b_NHP_C2H_C2H2P_N;

		mole.c[ipC2H][ipC2H2P] += rate_COmole.NHP_C2H_C2H2P_N_1;
		mole.c[ipNHP][ipC2H2P] += rate_COmole.NHP_C2H_C2H2P_N_2;
		mole.b[ipC2H2P] += rate_COmole.b_NHP_C2H_C2H2P_N;

		mole.c[ipC2H][ipATN] += rate_COmole.NHP_C2H_C2H2P_N_1;
		mole.c[ipNHP][ipATN] += rate_COmole.NHP_C2H_C2H2P_N_2;
		mole.b[ipATN] += rate_COmole.b_NHP_C2H_C2H2P_N;

		/* >>refer  mole   NO   C3H+   -->   C3H   NO+        */

		mole.c[ipC3HP][ipC3HP] -= rate_COmole.NO_C3HP_C3H_NOP_1;
		mole.c[ipNO][ipC3HP] -= rate_COmole.NO_C3HP_C3H_NOP_2;
		mole.b[ipC3HP] -= rate_COmole.b_NO_C3HP_C3H_NOP;

		mole.c[ipC3HP][ipNO] -= rate_COmole.NO_C3HP_C3H_NOP_1;
		mole.c[ipNO][ipNO] -= rate_COmole.NO_C3HP_C3H_NOP_2;
		mole.b[ipNO] -= rate_COmole.b_NO_C3HP_C3H_NOP;

		mole.c[ipC3HP][ipC3H] += rate_COmole.NO_C3HP_C3H_NOP_1;
		mole.c[ipNO][ipC3H] += rate_COmole.NO_C3HP_C3H_NOP_2;
		mole.b[ipC3H] += rate_COmole.b_NO_C3HP_C3H_NOP;

		mole.c[ipC3HP][ipNOP] += rate_COmole.NO_C3HP_C3H_NOP_1;
		mole.c[ipNO][ipNOP] += rate_COmole.NO_C3HP_C3H_NOP_2;
		mole.b[ipNOP] += rate_COmole.b_NO_C3HP_C3H_NOP;

		/* >>refer  mole   O   C2H2   -->   C2H   OH        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.O_C2H2_C2H_OH_1;
		mole.c[ipATO][ipC2H2] -= rate_COmole.O_C2H2_C2H_OH_2;
		mole.b[ipC2H2] -= rate_COmole.b_O_C2H2_C2H_OH;

		mole.c[ipC2H2][ipATO] -= rate_COmole.O_C2H2_C2H_OH_1;
		mole.c[ipATO][ipATO] -= rate_COmole.O_C2H2_C2H_OH_2;
		mole.b[ipATO] -= rate_COmole.b_O_C2H2_C2H_OH;

		mole.c[ipC2H2][ipC2H] += rate_COmole.O_C2H2_C2H_OH_1;
		mole.c[ipATO][ipC2H] += rate_COmole.O_C2H2_C2H_OH_2;
		mole.b[ipC2H] += rate_COmole.b_O_C2H2_C2H_OH;

		mole.c[ipC2H2][ipOH] += rate_COmole.O_C2H2_C2H_OH_1;
		mole.c[ipATO][ipOH] += rate_COmole.O_C2H2_C2H_OH_2;
		mole.b[ipOH] += rate_COmole.b_O_C2H2_C2H_OH;

		/* >>refer  mole   O   C2H2+   -->   HCO+   CH        */

		mole.c[ipC2H2P][ipC2H2P] -= rate_COmole.O_C2H2P_HCOP_CH_1;
		mole.c[ipATO][ipC2H2P] -= rate_COmole.O_C2H2P_HCOP_CH_2;
		mole.b[ipC2H2P] -= rate_COmole.b_O_C2H2P_HCOP_CH;

		mole.c[ipC2H2P][ipATO] -= rate_COmole.O_C2H2P_HCOP_CH_1;
		mole.c[ipATO][ipATO] -= rate_COmole.O_C2H2P_HCOP_CH_2;
		mole.b[ipATO] -= rate_COmole.b_O_C2H2P_HCOP_CH;

		mole.c[ipC2H2P][ipHCOP] += rate_COmole.O_C2H2P_HCOP_CH_1;
		mole.c[ipATO][ipHCOP] += rate_COmole.O_C2H2P_HCOP_CH_2;
		mole.b[ipHCOP] += rate_COmole.b_O_C2H2P_HCOP_CH;

		mole.c[ipC2H2P][ipCH] += rate_COmole.O_C2H2P_HCOP_CH_1;
		mole.c[ipATO][ipCH] += rate_COmole.O_C2H2P_HCOP_CH_2;
		mole.b[ipCH] += rate_COmole.b_O_C2H2P_HCOP_CH;

		/* >>refer  mole   OH   C2H2   -->   C2H   H2O        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.OH_C2H2_C2H_H2O_1;
		mole.c[ipOH][ipC2H2] -= rate_COmole.OH_C2H2_C2H_H2O_2;
		mole.b[ipC2H2] -= rate_COmole.b_OH_C2H2_C2H_H2O;

		mole.c[ipC2H2][ipOH] -= rate_COmole.OH_C2H2_C2H_H2O_1;
		mole.c[ipOH][ipOH] -= rate_COmole.OH_C2H2_C2H_H2O_2;
		mole.b[ipOH] -= rate_COmole.b_OH_C2H2_C2H_H2O;

		mole.c[ipC2H2][ipC2H] += rate_COmole.OH_C2H2_C2H_H2O_1;
		mole.c[ipOH][ipC2H] += rate_COmole.OH_C2H2_C2H_H2O_2;
		mole.b[ipC2H] += rate_COmole.b_OH_C2H2_C2H_H2O;

		mole.c[ipC2H2][ipH2O] += rate_COmole.OH_C2H2_C2H_H2O_1;
		mole.c[ipOH][ipH2O] += rate_COmole.OH_C2H2_C2H_H2O_2;
		mole.b[ipH2O] += rate_COmole.b_OH_C2H2_C2H_H2O;

		/* >>refer  mole   OH+   C2H   -->   C2H2+   O        */

		mole.c[ipC2H][ipC2H] -= rate_COmole.OHP_C2H_C2H2P_O_1;
		mole.c[ipOHP][ipC2H] -= rate_COmole.OHP_C2H_C2H2P_O_2;
		mole.b[ipC2H] -= rate_COmole.b_OHP_C2H_C2H2P_O;

		mole.c[ipC2H][ipOHP] -= rate_COmole.OHP_C2H_C2H2P_O_1;
		mole.c[ipOHP][ipOHP] -= rate_COmole.OHP_C2H_C2H2P_O_2;
		mole.b[ipOHP] -= rate_COmole.b_OHP_C2H_C2H2P_O;

		mole.c[ipC2H][ipC2H2P] += rate_COmole.OHP_C2H_C2H2P_O_1;
		mole.c[ipOHP][ipC2H2P] += rate_COmole.OHP_C2H_C2H2P_O_2;
		mole.b[ipC2H2P] += rate_COmole.b_OHP_C2H_C2H2P_O;

		mole.c[ipC2H][ipATO] += rate_COmole.OHP_C2H_C2H2P_O_1;
		mole.c[ipOHP][ipATO] += rate_COmole.OHP_C2H_C2H2P_O_2;
		mole.b[ipATO] += rate_COmole.b_OHP_C2H_C2H2P_O;

		/* >>refer  mole   O+   C2H2   -->   C2H2+   O        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.OP_C2H2_C2H2P_O_1;
		mole.c[ipOP][ipC2H2] -= rate_COmole.OP_C2H2_C2H2P_O_2;
		mole.b[ipC2H2] -= rate_COmole.b_OP_C2H2_C2H2P_O;

		mole.c[ipC2H2][ipOP] -= rate_COmole.OP_C2H2_C2H2P_O_1;
		mole.c[ipOP][ipOP] -= rate_COmole.OP_C2H2_C2H2P_O_2;
		mole.b[ipOP] -= rate_COmole.b_OP_C2H2_C2H2P_O;

		mole.c[ipC2H2][ipC2H2P] += rate_COmole.OP_C2H2_C2H2P_O_1;
		mole.c[ipOP][ipC2H2P] += rate_COmole.OP_C2H2_C2H2P_O_2;
		mole.b[ipC2H2P] += rate_COmole.b_OP_C2H2_C2H2P_O;

		mole.c[ipC2H2][ipATO] += rate_COmole.OP_C2H2_C2H2P_O_1;
		mole.c[ipOP][ipATO] += rate_COmole.OP_C2H2_C2H2P_O_2;
		mole.b[ipATO] += rate_COmole.b_OP_C2H2_C2H2P_O;

		/* >>refer  mole   C   C2H3+   -->   C3H+   H2        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.C_C2H3P_C3HP_H2_1;
		mole.c[ipATC][ipC2H3P] -= rate_COmole.C_C2H3P_C3HP_H2_2;
		mole.b[ipC2H3P] -= rate_COmole.b_C_C2H3P_C3HP_H2;

		mole.c[ipC2H3P][ipATC] -= rate_COmole.C_C2H3P_C3HP_H2_1;
		mole.c[ipATC][ipATC] -= rate_COmole.C_C2H3P_C3HP_H2_2;
		mole.b[ipATC] -= rate_COmole.b_C_C2H3P_C3HP_H2;

		mole.c[ipC2H3P][ipC3HP] += rate_COmole.C_C2H3P_C3HP_H2_1;
		mole.c[ipATC][ipC3HP] += rate_COmole.C_C2H3P_C3HP_H2_2;
		mole.b[ipC3HP] += rate_COmole.b_C_C2H3P_C3HP_H2;

		/* >>refer  mole   C2H   C2H3+   -->   C2H2+   C2H2        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.C2H_C2H3P_C2H2P_C2H2_1;
		mole.c[ipC2H][ipC2H3P] -= rate_COmole.C2H_C2H3P_C2H2P_C2H2_2;
		mole.b[ipC2H3P] -= rate_COmole.b_C2H_C2H3P_C2H2P_C2H2;

		mole.c[ipC2H3P][ipC2H] -= rate_COmole.C2H_C2H3P_C2H2P_C2H2_1;
		mole.c[ipC2H][ipC2H] -= rate_COmole.C2H_C2H3P_C2H2P_C2H2_2;
		mole.b[ipC2H] -= rate_COmole.b_C2H_C2H3P_C2H2P_C2H2;

		mole.c[ipC2H3P][ipC2H2P] += rate_COmole.C2H_C2H3P_C2H2P_C2H2_1;
		mole.c[ipC2H][ipC2H2P] += rate_COmole.C2H_C2H3P_C2H2P_C2H2_2;
		mole.b[ipC2H2P] += rate_COmole.b_C2H_C2H3P_C2H2P_C2H2;

		mole.c[ipC2H3P][ipC2H2] += rate_COmole.C2H_C2H3P_C2H2P_C2H2_1;
		mole.c[ipC2H][ipC2H2] += rate_COmole.C2H_C2H3P_C2H2P_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_C2H_C2H3P_C2H2P_C2H2;

		/* >>refer  mole   CH2   CH3+   -->   C2H3+   H2        */

		mole.c[ipCH3P][ipCH3P] -= rate_COmole.CH2_CH3P_C2H3P_H2_1;
		mole.c[ipCH2][ipCH3P] -= rate_COmole.CH2_CH3P_C2H3P_H2_2;
		mole.b[ipCH3P] -= rate_COmole.b_CH2_CH3P_C2H3P_H2;

		mole.c[ipCH3P][ipCH2] -= rate_COmole.CH2_CH3P_C2H3P_H2_1;
		mole.c[ipCH2][ipCH2] -= rate_COmole.CH2_CH3P_C2H3P_H2_2;
		mole.b[ipCH2] -= rate_COmole.b_CH2_CH3P_C2H3P_H2;

		mole.c[ipCH3P][ipC2H3P] += rate_COmole.CH2_CH3P_C2H3P_H2_1;
		mole.c[ipCH2][ipC2H3P] += rate_COmole.CH2_CH3P_C2H3P_H2_2;
		mole.b[ipC2H3P] += rate_COmole.b_CH2_CH3P_C2H3P_H2;

		/* >>refer  mole   CH4   C3H+   -->   C2H3+   C2H2        */

		mole.c[ipC3HP][ipC3HP] -= rate_COmole.CH4_C3HP_C2H3P_C2H2_1;
		mole.c[ipCH4][ipC3HP] -= rate_COmole.CH4_C3HP_C2H3P_C2H2_2;
		mole.b[ipC3HP] -= rate_COmole.b_CH4_C3HP_C2H3P_C2H2;

		mole.c[ipC3HP][ipCH4] -= rate_COmole.CH4_C3HP_C2H3P_C2H2_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_C3HP_C2H3P_C2H2_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_C3HP_C2H3P_C2H2;

		mole.c[ipC3HP][ipC2H3P] += rate_COmole.CH4_C3HP_C2H3P_C2H2_1;
		mole.c[ipCH4][ipC2H3P] += rate_COmole.CH4_C3HP_C2H3P_C2H2_2;
		mole.b[ipC2H3P] += rate_COmole.b_CH4_C3HP_C2H3P_C2H2;

		mole.c[ipC3HP][ipC2H2] += rate_COmole.CH4_C3HP_C2H3P_C2H2_1;
		mole.c[ipCH4][ipC2H2] += rate_COmole.CH4_C3HP_C2H3P_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_CH4_C3HP_C2H3P_C2H2;

		/* >>refer  mole   CH4   HCN+   -->   C2H3+   NH2        */

		mole.c[ipHCNP][ipHCNP] -= rate_COmole.CH4_HCNP_C2H3P_NH2_1;
		mole.c[ipCH4][ipHCNP] -= rate_COmole.CH4_HCNP_C2H3P_NH2_2;
		mole.b[ipHCNP] -= rate_COmole.b_CH4_HCNP_C2H3P_NH2;

		mole.c[ipHCNP][ipCH4] -= rate_COmole.CH4_HCNP_C2H3P_NH2_1;
		mole.c[ipCH4][ipCH4] -= rate_COmole.CH4_HCNP_C2H3P_NH2_2;
		mole.b[ipCH4] -= rate_COmole.b_CH4_HCNP_C2H3P_NH2;

		mole.c[ipHCNP][ipC2H3P] += rate_COmole.CH4_HCNP_C2H3P_NH2_1;
		mole.c[ipCH4][ipC2H3P] += rate_COmole.CH4_HCNP_C2H3P_NH2_2;
		mole.b[ipC2H3P] += rate_COmole.b_CH4_HCNP_C2H3P_NH2;

		mole.c[ipHCNP][ipNH2] += rate_COmole.CH4_HCNP_C2H3P_NH2_1;
		mole.c[ipCH4][ipNH2] += rate_COmole.CH4_HCNP_C2H3P_NH2_2;
		mole.b[ipNH2] += rate_COmole.b_CH4_HCNP_C2H3P_NH2;

		/* >>refer  mole   CH4+   C2H2   -->   C2H3+   CH3        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.CH4P_C2H2_C2H3P_CH3_1;
		mole.c[ipCH4P][ipC2H2] -= rate_COmole.CH4P_C2H2_C2H3P_CH3_2;
		mole.b[ipC2H2] -= rate_COmole.b_CH4P_C2H2_C2H3P_CH3;

		mole.c[ipC2H2][ipCH4P] -= rate_COmole.CH4P_C2H2_C2H3P_CH3_1;
		mole.c[ipCH4P][ipCH4P] -= rate_COmole.CH4P_C2H2_C2H3P_CH3_2;
		mole.b[ipCH4P] -= rate_COmole.b_CH4P_C2H2_C2H3P_CH3;

		mole.c[ipC2H2][ipC2H3P] += rate_COmole.CH4P_C2H2_C2H3P_CH3_1;
		mole.c[ipCH4P][ipC2H3P] += rate_COmole.CH4P_C2H2_C2H3P_CH3_2;
		mole.b[ipC2H3P] += rate_COmole.b_CH4P_C2H2_C2H3P_CH3;

		mole.c[ipC2H2][ipCH3] += rate_COmole.CH4P_C2H2_C2H3P_CH3_1;
		mole.c[ipCH4P][ipCH3] += rate_COmole.CH4P_C2H2_C2H3P_CH3_2;
		mole.b[ipCH3] += rate_COmole.b_CH4P_C2H2_C2H3P_CH3;

		/* >>refer  mole   CH5+   C2H2   -->   C2H3+   CH4        */

		mole.c[ipC2H2][ipC2H2] -= rate_COmole.CH5P_C2H2_C2H3P_CH4_1;
		mole.c[ipCH5P][ipC2H2] -= rate_COmole.CH5P_C2H2_C2H3P_CH4_2;
		mole.b[ipC2H2] -= rate_COmole.b_CH5P_C2H2_C2H3P_CH4;

		mole.c[ipC2H2][ipCH5P] -= rate_COmole.CH5P_C2H2_C2H3P_CH4_1;
		mole.c[ipCH5P][ipCH5P] -= rate_COmole.CH5P_C2H2_C2H3P_CH4_2;
		mole.b[ipCH5P] -= rate_COmole.b_CH5P_C2H2_C2H3P_CH4;

		mole.c[ipC2H2][ipC2H3P] += rate_COmole.CH5P_C2H2_C2H3P_CH4_1;
		mole.c[ipCH5P][ipC2H3P] += rate_COmole.CH5P_C2H2_C2H3P_CH4_2;
		mole.b[ipC2H3P] += rate_COmole.b_CH5P_C2H2_C2H3P_CH4;

		mole.c[ipC2H2][ipCH4] += rate_COmole.CH5P_C2H2_C2H3P_CH4_1;
		mole.c[ipCH5P][ipCH4] += rate_COmole.CH5P_C2H2_C2H3P_CH4_2;
		mole.b[ipCH4] += rate_COmole.b_CH5P_C2H2_C2H3P_CH4;

		/* >>refer  mole   CH+   CH4   -->   C2H3+   H2        */

		mole.c[ipCH4][ipCH4] -= rate_COmole.CHP_CH4_C2H3P_H2_1;
		mole.c[ipCHP][ipCH4] -= rate_COmole.CHP_CH4_C2H3P_H2_2;
		mole.b[ipCH4] -= rate_COmole.b_CHP_CH4_C2H3P_H2;

		mole.c[ipCH4][ipCHP] -= rate_COmole.CHP_CH4_C2H3P_H2_1;
		mole.c[ipCHP][ipCHP] -= rate_COmole.CHP_CH4_C2H3P_H2_2;
		mole.b[ipCHP] -= rate_COmole.b_CHP_CH4_C2H3P_H2;

		mole.c[ipCH4][ipC2H3P] += rate_COmole.CHP_CH4_C2H3P_H2_1;
		mole.c[ipCHP][ipC2H3P] += rate_COmole.CHP_CH4_C2H3P_H2_2;
		mole.b[ipC2H3P] += rate_COmole.b_CHP_CH4_C2H3P_H2;

		/* >>refer  mole   C+   CH4   -->   C2H3+   H        */

		mole.c[ipCH4][ipCH4] -= rate_COmole.CP_CH4_C2H3P_H_1;
		mole.c[ipCP][ipCH4] -= rate_COmole.CP_CH4_C2H3P_H_2;
		mole.b[ipCH4] -= rate_COmole.b_CP_CH4_C2H3P_H;

		mole.c[ipCH4][ipCP] -= rate_COmole.CP_CH4_C2H3P_H_1;
		mole.c[ipCP][ipCP] -= rate_COmole.CP_CH4_C2H3P_H_2;
		mole.b[ipCP] -= rate_COmole.b_CP_CH4_C2H3P_H;

		mole.c[ipCH4][ipC2H3P] += rate_COmole.CP_CH4_C2H3P_H_1;
		mole.c[ipCP][ipC2H3P] += rate_COmole.CP_CH4_C2H3P_H_2;
		mole.b[ipC2H3P] += rate_COmole.b_CP_CH4_C2H3P_H;

		/* >>refer  mole   H2O   C2H3+   -->   C2H2   H3O+        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.H2O_C2H3P_C2H2_H3OP_1;
		mole.c[ipH2O][ipC2H3P] -= rate_COmole.H2O_C2H3P_C2H2_H3OP_2;
		mole.b[ipC2H3P] -= rate_COmole.b_H2O_C2H3P_C2H2_H3OP;

		mole.c[ipC2H3P][ipH2O] -= rate_COmole.H2O_C2H3P_C2H2_H3OP_1;
		mole.c[ipH2O][ipH2O] -= rate_COmole.H2O_C2H3P_C2H2_H3OP_2;
		mole.b[ipH2O] -= rate_COmole.b_H2O_C2H3P_C2H2_H3OP;

		mole.c[ipC2H3P][ipC2H2] += rate_COmole.H2O_C2H3P_C2H2_H3OP_1;
		mole.c[ipH2O][ipC2H2] += rate_COmole.H2O_C2H3P_C2H2_H3OP_2;
		mole.b[ipC2H2] += rate_COmole.b_H2O_C2H3P_C2H2_H3OP;

		mole.c[ipC2H3P][ipH3OP] += rate_COmole.H2O_C2H3P_C2H2_H3OP_1;
		mole.c[ipH2O][ipH3OP] += rate_COmole.H2O_C2H3P_C2H2_H3OP_2;
		mole.b[ipH3OP] += rate_COmole.b_H2O_C2H3P_C2H2_H3OP;

		/* >>refer  mole   HCN   C2H3+   -->   HCNH+   C2H2        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.HCN_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHCN][ipC2H3P] -= rate_COmole.HCN_C2H3P_HCNHP_C2H2_2;
		mole.b[ipC2H3P] -= rate_COmole.b_HCN_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipHCN] -= rate_COmole.HCN_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHCN][ipHCN] -= rate_COmole.HCN_C2H3P_HCNHP_C2H2_2;
		mole.b[ipHCN] -= rate_COmole.b_HCN_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipHCNHP] += rate_COmole.HCN_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHCN][ipHCNHP] += rate_COmole.HCN_C2H3P_HCNHP_C2H2_2;
		mole.b[ipHCNHP] += rate_COmole.b_HCN_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipC2H2] += rate_COmole.HCN_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHCN][ipC2H2] += rate_COmole.HCN_C2H3P_HCNHP_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_HCN_C2H3P_HCNHP_C2H2;

		/* >>refer  mole   HNC   C2H3+   -->   HCNH+   C2H2        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.HNC_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHNC][ipC2H3P] -= rate_COmole.HNC_C2H3P_HCNHP_C2H2_2;
		mole.b[ipC2H3P] -= rate_COmole.b_HNC_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipHNC] -= rate_COmole.HNC_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHNC][ipHNC] -= rate_COmole.HNC_C2H3P_HCNHP_C2H2_2;
		mole.b[ipHNC] -= rate_COmole.b_HNC_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipHCNHP] += rate_COmole.HNC_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHNC][ipHCNHP] += rate_COmole.HNC_C2H3P_HCNHP_C2H2_2;
		mole.b[ipHCNHP] += rate_COmole.b_HNC_C2H3P_HCNHP_C2H2;

		mole.c[ipC2H3P][ipC2H2] += rate_COmole.HNC_C2H3P_HCNHP_C2H2_1;
		mole.c[ipHNC][ipC2H2] += rate_COmole.HNC_C2H3P_HCNHP_C2H2_2;
		mole.b[ipC2H2] += rate_COmole.b_HNC_C2H3P_HCNHP_C2H2;

		/* >>refer  mole   NH3   C2H3+   -->   C2H2   NH4+        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.NH3_C2H3P_C2H2_NH4P_1;
		mole.c[ipNH3][ipC2H3P] -= rate_COmole.NH3_C2H3P_C2H2_NH4P_2;
		mole.b[ipC2H3P] -= rate_COmole.b_NH3_C2H3P_C2H2_NH4P;

		mole.c[ipC2H3P][ipNH3] -= rate_COmole.NH3_C2H3P_C2H2_NH4P_1;
		mole.c[ipNH3][ipNH3] -= rate_COmole.NH3_C2H3P_C2H2_NH4P_2;
		mole.b[ipNH3] -= rate_COmole.b_NH3_C2H3P_C2H2_NH4P;

		mole.c[ipC2H3P][ipC2H2] += rate_COmole.NH3_C2H3P_C2H2_NH4P_1;
		mole.c[ipNH3][ipC2H2] += rate_COmole.NH3_C2H3P_C2H2_NH4P_2;
		mole.b[ipC2H2] += rate_COmole.b_NH3_C2H3P_C2H2_NH4P;

		mole.c[ipC2H3P][ipNH4P] += rate_COmole.NH3_C2H3P_C2H2_NH4P_1;
		mole.c[ipNH3][ipNH4P] += rate_COmole.NH3_C2H3P_C2H2_NH4P_2;
		mole.b[ipNH4P] += rate_COmole.b_NH3_C2H3P_C2H2_NH4P;

		/* >>refer  mole   O   C2H3+   -->   HCO+   CH2        */

		mole.c[ipC2H3P][ipC2H3P] -= rate_COmole.O_C2H3P_HCOP_CH2_1;
		mole.c[ipATO][ipC2H3P] -= rate_COmole.O_C2H3P_HCOP_CH2_2;
		mole.b[ipC2H3P] -= rate_COmole.b_O_C2H3P_HCOP_CH2;

		mole.c[ipC2H3P][ipATO] -= rate_COmole.O_C2H3P_HCOP_CH2_1;
		mole.c[ipATO][ipATO] -= rate_COmole.O_C2H3P_HCOP_CH2_2;
		mole.b[ipATO] -= rate_COmole.b_O_C2H3P_HCOP_CH2;

		mole.c[ipC2H3P][ipHCOP] += rate_COmole.O_C2H3P_HCOP_CH2_1;
		mole.c[ipATO][ipHCOP] += rate_COmole.O_C2H3P_HCOP_CH2_2;
		mole.b[ipHCOP] += rate_COmole.b_O_C2H3P_HCOP_CH2;

		mole.c[ipC2H3P][ipCH2] += rate_COmole.O_C2H3P_HCOP_CH2_1;
		mole.c[ipATO][ipCH2] += rate_COmole.O_C2H3P_HCOP_CH2_2;
		mole.b[ipCH2] += rate_COmole.b_O_C2H3P_HCOP_CH2;

	/********************************************************************************************/
	
	/*							DONE WITH BALANCE EQUATIONS									    */
	
	/********************************************************************************************/
}


