/* This file is part of Cloudy and is copyright (C)1978-2006 by Gary J. Ferland
 * For conditions of distribution and use see copyright notice in license.txt */
/*CO_step fills in matrix for heavy elements molecular routines */
#include "cddefines.h"
#include "dense.h"
#include "hmi.h"
#include "phycon.h"
#include "mole.h"
/* Nick Abel between July and October of 2003 assisted Dr. Ferland in improving the heavy element 
 * molecular network in Cloudy. Before this routine would predict negative abundances if 
 * the fraction of carbon in the form of molecules came close to 100%. A reorganizing of 
 * the reaction network detected several bugs.  Treatment of "coupled reactions",
 * in which both densities in the reaction rate were being predicted by Cloudy, were also 
 * added.  Due to these improvements, Cloudy can now perform calculations
 * where 100% of the carbon is in the form of CO without predicting negative abundances
 *
 * Additional changes were made in November of 2003 so that our reaction 
 * network would include all reactions from the TH85 paper.  This involved 
 * adding silicon to the chemical network.  Also the reaction rates were
 * labeled to make identification with the reaction easier and the matrix 
 * elements of atomic C, O, and Si are now done in a loop, which makes 
 * the addition of future chemical species (like N or S) easy.
 * */
/*lint -e778 constant expression evaluatess to 0 in operation '-' */
/*=================================================================*/


/*CO_step fills in matrix for heavy elements molecular routines */
void CO_h_he(void )
{

double
	H_C2H3P_C2H2P_H2,
	H2P_C2H2_C2H3P_H,
	H3P_C2H2_C2H3P_H2,
	HeP_C2H2_C2HP_He_H,
	HeP_C3H_C3P_He_H,
	HeP_C2H2_C2P_He_H2,
	HeP_C2H2_CHP_CH_He,
	Hm_C2H_C2H2_e,
	HP_C2H2_C2H2P_H,
	HP_C2H2_C2HP_H2,
	HP_C3H_C3HP_H,
	HP_C3H_C3P_H2,
	H2P_C2H_C2H2P_H,
	H2P_C2H2_C2H2P_H2,
	H3P_C2H_C2H2P_H2,
	H3P_C3_C3HP_H2,
	H2_C2HP_C2H2P_H,
	H2_C3P_C3HP_H,
	HeP_C2H2_C2H2P_He,
	HeP_C3_C2_CP_He,
	HP_C3_C3P_H,
	HP_C2H_C2HP_H,
	H2P_C2_C2HP_H,
	H2P_C2H_C2HP_H2,
	H3P_C2_C2HP_H2,
	H2_C2P_C2HP_H,
	H2_S_HS_H,
	H_CH_C_H_H,
	H_OH_O_H_H,
	H_H2O_OH_H_H,
	H_CH_C_H2,
	H_CHP_CP_H2,
	H_CH2_CH_H2,
	H_CH3P_CH2P_H2,
	H_OH_O_H2,
	H_H2O_OH_H2,
	H_O2_OH_O,
	H_O2_O_O_H,
	H_C_CH_nu,
	H_CP_CHP_nu,
	H_OH_H2O_nu,
	H_O_OH_nu,
	Hminus_CH_CH2_e,
	Hminus_OH_H2O_e,
	Hminus_O_OH_e,
	Hminus_H3OP_H2O_H2,
	Hminus_H3OP_OH_H2_H,
	HP_CH_CHP_H,
	HP_CH2_CH2P_H,
	HP_H2O_H2OP_H,
	HP_O2_O2P_H,
	HP_OH_OHP_H,
	HP_SiO_SiOP_H,
	HP_CH2_CHP_H2,
	HP_SiH_SiP_H2,
	H2_C_CH_H,
	H2_CP_CHP_H,
	H2_CH_CH2_H,
	H2_CHP_CH2P_H,
	H2_OH_H2O_H,
	H2_OHP_H2OP_H,
	H2_H2OP_H3OP_H,
	H2_COP_HCOP_H,
	H2_O_OH_H,
	H2_OP_OHP_H,
	H2_SiOP_SiOHP_H,
	H2_CH_C_H2_H,
	H2_OH_O_H2_H,
	H2_CH2P_CH3P_H,
	H2_H2O_OH_H2_H,
	H2_O2_O_O_H2,
	H2_O2_OH_OH,
	H2_C_CH2_nu,
	H2_CP_CH2P_nu,
	H2_SiP_SiH2P_nu,
	H2P_C_CHP_H,
	H2P_CH_CH2P_H,
	H2P_CH2_CH3P_H,
	H2P_OH_H2OP_H,
	H2P_H2O_H3OP_H,
	H2P_O_OHP_H,
	H2P_CH_CHP_H2,
	H2P_CH2_CH2P_H2,
	H2P_H2O_H2OP_H2,
	H2P_O2_O2P_H2,
	H2P_OH_OHP_H2,
	H3P_C_CHP_H2,
	H3P_CH_CH2P_H2,
	H3P_CH2_CH3P_H2,
	H3P_OH_H2OP_H2,
	H3P_H2O_H3OP_H2,
	H3P_O_OHP_H2,
	H3P_SiH_SiH2P_H2,
	H3P_SiO_SiOHP_H2,
	HeP_CH_CHP_He,
	HeP_H2O_H2OP_He,
	HeP_O2_O2P_He,
	HeP_CH_CP_He_H,
	HeP_CH2_CHP_He_H,
	HeP_OH_OP_He_H,
	HeP_H2O_OHP_He_H,
	HeP_SiH_SiP_He_H,
	HeP_H2O_OH_He_HP,
	HeP_CH2_CP_He_H2,
	HeP_O2_OP_O_He,
	HeP_SiO_SiP_O_He,
	HeP_SiO_Si_OP_He,
	h2s_ch_ch2_h,
	h2s_o_oh_h,
	h2s_oh_h2o_h,
	h2s_o2_twooh,
	h2s_c_ch_h,
	h2s_cp_chp_h,
	h2s_op_ohp_h,

	/* These reactions were put in by NPA so that we can have EXACTLY the same 
	   reaction network as the others in Leiden */

	H2s_CH_C_H2_H, 
	H2s_OH_O_H2_H, 
	H2s_H2O_OH_H2_H, 
	H2s_O2_O_O_H2, 
	H2s_CH2_CH3_H,  
	H2s_CH_CH3_nu,
	H2s_CH3_CH4_H,

	H_CH3_CH2_H2,
	H_CH4P_CH3P_H2,
	H_CH5P_CH4P_H2,
	H2_CH2_CH3_H,
	H2_CH_CH3_nu,
	H2_CH3_CH4_H,
	H2_CH4P_CH5P_H,
	H2_CH3P_CH5P_nu,
	H2P_CH4_CH3P_H2,
	H2P_CH4_CH4P_H2,
	H2P_CH4_CH5P_H,
	H3P_CH3_CH4P_H2,
	H3P_CH4_CH5P_H2,
	HeP_CH3_CHP_He,
	HeP_CH4_CHP_He,
	HeP_CH4_CH2P_He,
	HeP_CH4_CH3_He,
	HeP_CH4_CH3P_He,
	HeP_CH4_CH4P_He,
	Hminus_CH2_CH3_e,
	Hminus_CH3_CH4_e,
	HP_CH3_CH3P_H,
	HP_CH4_CH3P_H2,
	HP_CH4_CH4P_H,
	H2_N_NH_H          , 
	H2_NH_NH2_H         ,
	H2_NH2_NH3_H        ,
	H2_CN_HCN_H         ,
	HP_HNO_NOP_H2       ,
	HP_HS_SP_H2         ,
	H_HSP_SP_H2         ,
	H2P_N_NHP_H         ,
	H2_NP_NHP_H         ,
	H2_NHP_N_H3P        ,
	H2P_NH_NH2P_H       ,
	H2_NHP_NH2P_H       ,
	H2_NH2P_NH3P_H      ,
	H2_NH3P_NH4P_H      ,
	H2P_CN_HCNP_H      ,
	H2_CNP_HCNP_H       ,
	H2P_NO_HNOP_H       ,
	H2_SP_HSP_H         ,
	H2_CSP_HCSP_H       ,
	H2_NO2P_NOP_H2O     ,
	H3P_NH_NH2P_H2      ,
	H3P_NH2_NH3P_H2     ,
	H3P_NH3_NH4P_H2     ,
	H3P_CN_HCNP_H2      ,
	H3P_NO_HNOP_H2      ,
	H3P_S_HSP_H2        ,
	H3P_CS_HCSP_H2      ,
	H3P_NO2_NOP_OH_H2   ,
	HeP_NH_NP_He_H      ,
	HeP_NH2_NP_He_H2    ,
	HeP_NH2_NHP_He_H    ,
	HeP_NH3_NHP_He_H2   ,
	HeP_NH3_NH2P_He_H   ,
	HeP_CN_N_CP_He      ,
	HeP_CN_NP_C_He      ,
	HeP_HCN_N_CHP_He    ,
	HeP_HCN_NP_CH_He    ,
	HeP_HCN_N_CP_He_H   ,
	HeP_HCN_CNP_He_H    ,
	HeP_N2_NP_N_He      ,
	HeP_NO_OP_N_He      ,
	HeP_NO_O_NP_He      ,
	HeP_HNO_NOP_He_H    ,
	HeP_HNO_NO_He_HP    ,
	HeP_HS_SP_He_H      ,
	HeP_OCN_CN_OP_He    ,
	HeP_OCN_CNP_O_He    ,
	HeP_SIN_SIP_N_He    ,
	HeP_N2O_N2_OP_He    ,
	HeP_N2O_N2P_O_He    ,
	HeP_N2O_NO_NP_He    ,
	HeP_N2O_NOP_N_He,
	HeP_CS_SP_C_He  ,
	HeP_CS_S_CP_He  ,
	HeP_NS_S_NP_He  ,
	HeP_NS_SP_N_He  ,
	HeP_SO_S_OP_He  ,
	HeP_SO_SP_O_He  ,
	HeP_OCS_S_COP_He,
	HeP_OCS_CSP_O_He,
	HeP_OCS_CS_OP_He,
	HeP_S2_SP_S_He  ,
	HP_NH_NHP_H ,
	HP_NH2_NH2P_H   ,
	HP_NH3_NH3P_H   ,
	H_CNP_CN_HP ,
	HP_HCN_HCNP_H   ,
	H_HCNP_HCN_HP   ,
	H_N2P_N2_HP ,
	HP_NO_NOP_H ,
	HP_HS_HSP_H ,
	HP_SIN_SINP_H   ,
	HP_CS_CSP_H ,
	HP_NS_NSP_H ,
	HP_SO_SOP_H ,
	HP_OCS_OCSP_H   ,
	HP_S2_S2P_H ,
	H2P_NH_NHP_H2   ,
	H2P_NH2_NH2P_H2 ,
	H2P_NH3_NH3P_H2 ,
	H2P_CN_CNP_H2   ,
	H2P_HCN_HCNP_H2 ,
	H2P_NO_NOP_H2   ,
	HeP_NH3_NH3P_He ,
	HeP_N2_N2P_He   ,
	Hminus_N_NH_e ,
	Hminus_NH_NH2_e   ,
	Hminus_NH2_NH3_e  ,
	Hminus_CN_HCN_e   ,
	Hminus_NH4P_NH3_H2,
	Hminus_NP_N_H ,
	H_C2_CH_C,
	HeP_C2_CP_C_He,
	HP_C2_C2P_H,
	H2P_C2_C2P_H2,
	HeP_C2_C2P_He,
	H2_ClP_HClP_H,
	H2_HClP_H2ClP_H,
	H3P_Cl_HClP_H2,
	H3P_HCl_H2ClP_H2,
	HeP_HCl_ClP_He_H,
	HeP_CCl_Cl_CP_He,
	HeP_ClO_ClP_O_He,
	HP_HCl_HClP_H,
	H3P_HNC_HCNHP_H2 ,
	HeP_HNC_NHP_C_He  ,
	HeP_HNC_N_CP_He_H,
	HeP_HNC_CNP_He_H  ,
	HP_HNC_HCN_HP    ,
	H_HNC_HCN_H     ,
	H2_HCNP_HCNHP_H  ,
	H3P_HCN_HCNHP_H2,
	HeP_C2H_CHP_C_He,
	HeP_C2H_CH_CP_He,
	HeP_C2H_C2P_He_H,
	Hm_C2_C2H_e,
	HP_C2H_C2P_H2,
	HeII, HII, HI,H2g, H2p, H3p,H2s, Hm;



	HeII = dense.xIonDense[ipHELIUM][1];
	HII = dense.xIonDense[ipHYDROGEN][1];
	HI = dense.xIonDense[ipHYDROGEN][0];
	H2g = hmi.Hmolec[ipMH2g];
	H2p = hmi.Hmolec[ipMH2p];
	H3p = hmi.Hmolec[ipMH3p];
	H2s = hmi.Hmolec[ipMH2s];
	Hm = hmi.Hmolec[ipMHm];

	H_CH_C_H_H			    = HMRATE(6.0e-9,0,40200)*HI;
	H_OH_O_H_H			    = HMRATE(6.0e-9,0,50900)*HI;
	H_H2O_OH_H_H		    = HMRATE((1e-9)*5.8,0,52900)*HI;
	H_CH_C_H2			    = HMRATE((1e-11)*2.7,0.38,0)*HI;
	H_CHP_CP_H2			    = HMRATE((1e-10)*7.5,0,0)*HI;
	H_CH2_CH_H2			    = HMRATE((1e-11)*6.64,0,0)*HI;
	H_CH3P_CH2P_H2		    = HMRATE(7.0e-10,0,10560)*HI;
	H_OH_O_H2			    = HMRATE(0.0000000000000699,2.8,1950)*HI;
	H_H2O_OH_H2			    = HMRATE((1e-11)*1.59,1.2,9610)*HI;
	H_O2_OH_O			    = HMRATE((1e-10)*2.61,0,8156)*HI;
	H_O2_O_O_H			    = HMRATE(6.0e-9,0,52300)*HI;
	H_C_CH_nu               = HMRATE((1e-17)*1.,0,0)*HI;
	H_CP_CHP_nu             = HMRATE((1e-17)*1.7,0,0)*HI;
	H_OH_H2O_nu             = HMRATE(5.26E-18,-5.22,90)*HI;
	H_O_OH_nu               = HMRATE(9.9E-19,-0.38,0)*HI;
	Hminus_CH_CH2_e         = HMRATE(1.0e-10,0,0)*Hm;
	Hminus_OH_H2O_e         = HMRATE(1.0e-10,0,0)*Hm;
	Hminus_O_OH_e           = HMRATE(1.0e-9,0,0)*Hm;
	Hminus_H3OP_H2O_H2      = HMRATE(0.00000023,-0.5,0)*Hm;
	Hminus_H3OP_OH_H2_H     = HMRATE(0.00000023,-0.5,0)*Hm;
	HP_CH_CHP_H             = HMRATE((1e-9)*1.9,0,0)*HII;
	HP_CH2_CH2P_H           = HMRATE((1e-9)*1.4,0,0)*HII;
	HP_H2O_H2OP_H           = HMRATE((1e-9)*6.9,0,0)*HII;
	HP_O2_O2P_H             = HMRATE(2.0e-9,0,0)*HII;
	HP_OH_OHP_H             = HMRATE((1e-9)*2.1,0,0)*HII;
	HP_SiO_SiOP_H           = HMRATE((1e-9)*3.3,0,0)*HII;
	HP_CH2_CHP_H2           = HMRATE((1e-9)*1.4,0,0)*HII;
	HP_SiH_SiP_H2           = HMRATE((1e-9)*1.7,0,0)*HII;
	H2_CHP_CH2P_H           = HMRATE(1.2e-9,0,0)*H2g;
	H2_CH2P_CH3P_H          = HMRATE((1e-9)*1.6,0,0)*H2g;
	H2_OHP_H2OP_H           = HMRATE((1e-9)*1.01,0,0)*H2g;
	H2_H2OP_H3OP_H          = HMRATE((1e-10)*6.4,0,0)*H2g;
	H2_COP_HCOP_H           = HMRATE((1e-9)*1.8,0,0)*H2g;
	H2_OP_OHP_H             = HMRATE((1e-9)*1.7,0,0)*H2g;
	H2_SiOP_SiOHP_H         = HMRATE((1e-10)*3.2,0,0)*H2g;
	H2_C_CH2_nu             = HMRATE((1e-17)*1.,0,0)*H2g;
	H2_CP_CH2P_nu           = HMRATE(0.0000000000000004,-0.2,0)*H2g;
	H2_SiP_SiH2P_nu         = HMRATE(0.000000000000000003,0,0)*H2g;
	H2_C_CH_H               = HMRATE((1e-10)*6.64,0,11700)*H2g;
	H2_CP_CHP_H             = HMRATE(1.0e-10,0,4640)*H2g;
	/* >>chng 05 aug 02, NA add following */
	H2_S_HS_H               = HMRATE(1.76e-13,2.88,6126)*H2g;
	H2_CH_CH2_H             = HMRATE((1e-10)*5.46,0,1943)*H2g;
	H2_OH_H2O_H             = HMRATE((1e-12)*2.05,1.52,1736)*H2g;
	H2_O_OH_H               = HMRATE((1e-13)*3.14,2.7,3150)*H2g;
	H2_CH_C_H2_H            = HMRATE(6.0e-9,0,40200)*H2g;
	H2_OH_O_H2_H            = HMRATE(6.0e-9,0,50900)*H2g;
	H2_H2O_OH_H2_H          = HMRATE((1e-9)*5.8,0,52900)*H2g;
	H2_O2_O_O_H2            = HMRATE(6.0e-9,0,52300)*H2g;
	H2_O2_OH_OH             = HMRATE((1e-10)*3.16,0,21890)*H2g;
	H2s_CH_C_H2_H           = HMRATE(6.0e-9,0,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2s_OH_O_H2_H           = HMRATE(6.0e-9,0,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2s_H2O_OH_H2_H         = HMRATE((1e-9)*5.8,0,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2s_O2_O_O_H2           = HMRATE(6.0e-9,0,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2P_C_CHP_H             = HMRATE((1e-9)*2.4,0,0)*H2p;
	H2P_CH_CH2P_H           = HMRATE((1e-10)*7.1,0,0)*H2p;
	H2P_CH2_CH3P_H          = HMRATE(1.0e-9,0,0)*H2p;
	H2P_OH_H2OP_H           = HMRATE((1e-10)*7.6,0,0)*H2p;
	H2P_H2O_H3OP_H          = HMRATE((1e-9)*3.4,0,0)*H2p;
	H2P_O_OHP_H             = HMRATE((1e-9)*1.5,0,0)*H2p;
	H2P_CH_CHP_H2           = HMRATE((1e-10)*7.1,0,0)*H2p;
	H2P_CH2_CH2P_H2         = HMRATE(1.0e-9,0,0)*H2p;
	H2P_H2O_H2OP_H2         = HMRATE((1e-9)*3.9,0,0)*H2p;
	H2P_O2_O2P_H2           = HMRATE((1e-10)*8.,0,0)*H2p;
	H2P_OH_OHP_H2           = HMRATE((1e-10)*7.6,0,0)*H2p;
	H3P_C_CHP_H2            = HMRATE(2.0e-9,0,0)*H3p;
	H3P_CH_CH2P_H2          = HMRATE(1.2e-9,0,0)*H3p;
	H3P_CH2_CH3P_H2         = HMRATE((1e-9)*1.7,0,0)*H3p;
	H3P_OH_H2OP_H2          = HMRATE(1.3e-9,0,0)*H3p;
	H3P_H2O_H3OP_H2         = HMRATE((1e-9)*5.9,0,0)*H3p;
	H3P_O_OHP_H2            = HMRATE((1e-10)*8.,0,0)*H3p;
	H3P_SiH_SiH2P_H2        = HMRATE(2.0e-9,0,0)*H3p;
	H3P_SiO_SiOHP_H2        = HMRATE(2.0e-9,0,0)*H3p;
	HeP_CH_CHP_He           = HMRATE(5.0e-10,0,0)*HeII;
	HeP_H2O_H2OP_He         = HMRATE((1e-11)*6.05,0,0)*HeII;
	HeP_O2_O2P_He           = HMRATE(3.3e-11,0,0)*HeII;
	HeP_CH_CP_He_H          = HMRATE(1.1e-9,0,0)*HeII;
	HeP_CH2_CHP_He_H        = HMRATE((1e-10)*7.5,0,0)*HeII;
	HeP_OH_OP_He_H          = HMRATE(1.1e-9,0,0)*HeII;
	HeP_H2O_OHP_He_H        = HMRATE((1e-10)*2.86,0,0)*HeII;
	HeP_SiH_SiP_He_H        = HMRATE((1e-9)*1.8,0,0)*HeII;
	HeP_H2O_OH_He_HP        = HMRATE((1e-10)*2.04,0,0)*HeII;
	HeP_CH2_CP_He_H2        = HMRATE((1e-10)*7.5,0,0)*HeII;
	HeP_O2_OP_O_He          = HMRATE(1.0e-9,0,0)*HeII;
	HeP_SiO_SiP_O_He        = HMRATE((1e-10)*8.6,0,0)*HeII;
	HeP_SiO_Si_OP_He        = HMRATE((1e-10)*8.6,0,0)*HeII;
	h2s_ch_ch2_h            = HMRATE((1e-10)*5.46,0, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	h2s_o_oh_h              = HMRATE((1e-13)*3.14,2.7, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	h2s_oh_h2o_h            = HMRATE((1e-12)*2.05,1.52, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	h2s_o2_twooh            = HMRATE((1e-10)*3.16,0, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	h2s_c_ch_h              = HMRATE((1e-10)*6.64,0, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	h2s_cp_chp_h            = HMRATE(1.0e-10,0, 0. )*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H_CH3_CH2_H2            = HMRATE(1.0e-10,0,7600)*HI;
	H_CH4P_CH3P_H2          = HMRATE(1.0e-11,0,0)*HI;
	H_CH5P_CH4P_H2          = HMRATE(2.0e-11,0,0)*HI;
	H2_CH2_CH3_H            = HMRATE((1e-11)*5.18,0.17,6400)*H2g;
	H2_CH_CH3_nu            = HMRATE(5.09E-18,-0.71,11.6)*H2g;
	H2_CH3_CH4_H            = HMRATE(0.0000000000000686,2.74,4740)*H2g;
	H2_CH4P_CH5P_H          = HMRATE(3.3e-11,0,0)*H2g;
	H2_CH3P_CH5P_nu         = HMRATE(0.000000000000013,-1,0)*H2g;
	H2s_CH2_CH3_H           = HMRATE((1e-11)*5.18,0.17,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2s_CH_CH3_nu           = HMRATE(5.09E-18,-0.71,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2s_CH3_CH4_H           = HMRATE(0.0000000000000686,2.74,0)*H2s*hmi.lgLeiden_Keep_ipMH2s;
	H2P_CH4_CH3P_H2         = HMRATE((1e-9)*2.3,0,0)*H2p;
	H2P_CH4_CH4P_H2         = HMRATE((1e-9)*1.4,0,0)*H2p;
	H2P_CH4_CH5P_H          = HMRATE((1e-10)*1.14,0,0)*H2p;
	H3P_CH3_CH4P_H2         = HMRATE((1e-9)*2.1,0,0)*H3p;
	H3P_CH4_CH5P_H2         = HMRATE((1e-9)*2.4,0,0)*H3p;
	HeP_CH3_CHP_He          = HMRATE((1e-9)*1.8,0,0)*HeII;
	HeP_CH4_CHP_He          = HMRATE((1e-10)*2.4,0,0)*HeII;
	HeP_CH4_CH2P_He         = HMRATE((1e-10)*9.5,0,0)*HeII;
	HeP_CH4_CH3_He          = HMRATE((1e-10)*4.8,0,0)*HeII;
	HeP_CH4_CH3P_He         = HMRATE((1e-11)*8.5,0,0)*HeII;
	HeP_CH4_CH4P_He         = HMRATE((1e-11)*5.1,0,0)*HeII;
	Hminus_CH2_CH3_e        = HMRATE(1.0e-9,0,0)*Hm;
	Hminus_CH3_CH4_e        = HMRATE(1.0e-9,0,0)*Hm;
	HP_CH3_CH3P_H           = HMRATE((1e-9)*3.4,0,0)*HII;
	HP_CH4_CH3P_H2          = HMRATE((1e-9)*2.3,0,0)*HII;
	HP_CH4_CH4P_H           = HMRATE((1e-9)*1.5,0,0)*HII;
	H_C2_CH_C = HMRATE((1e-10)*4.67,0.5,30450)*HI;
	HeP_C2_CP_C_He = HMRATE((1e-9)*1.6,0,0)*HeII;
	HP_C2_C2P_H = HMRATE((1e-9)*3.1,0,0)*HII;
	H2P_C2_C2P_H2 = HMRATE(1.1e-9,0,0)*H2p;
	HeP_C2_C2P_He = HMRATE(5.0e-10,0,0)*HeII;
	H2_N_NH_H           = HMRATE((1e-9)*1.69,0,18095)*H2g;
	H2_NH_NH2_H         = HMRATE((1e-11)*5.96,0,7782)*H2g;
	H2_NH2_NH3_H        = HMRATE((1e-15)*2.05,3.89,1400)*H2g;
	H2_CN_HCN_H         = HMRATE(0.000000000000404,2.87,820)*H2g;
	HP_HNO_NOP_H2       = HMRATE((1e-9)*4.,0,0)*HII;
	HP_HS_SP_H2         = HMRATE((1e-9)*1.6,0,0)*HII;
	H_HSP_SP_H2         = HMRATE(1.1e-10,0,0)*HI;
	H2P_N_NHP_H         = HMRATE((1e-9)*1.9,0,0)*H2p;
	H2_NP_NHP_H         = HMRATE(1.0e-9,0,85)*H2g;
	H2_NHP_N_H3P        = HMRATE((1e-10)*2.25,0,0)*H2g;
	H2P_NH_NH2P_H       = HMRATE((1e-10)*7.6,0,0)*H2p;
	H2_NHP_NH2P_H       = HMRATE((1e-9)*1.28,0,0)*H2g;
	H2_NH2P_NH3P_H      = HMRATE((1e-10)*2.7,0,0)*H2g;
	H2_NH3P_NH4P_H      = HMRATE(0.0000000000002,0,0)*H2g;
	H2P_CN_HCNP_H       = HMRATE(1.2e-9,0,0)*H2p;
	H2_CNP_HCNP_H       = HMRATE(1.0e-9,0,0)*H2g;
	H2P_NO_HNOP_H       = HMRATE(1.1e-9,0,0)*H2p;
	H2_SP_HSP_H         = HMRATE(1.1e-10,0,9860)*H2g;
	H2_CSP_HCSP_H       = HMRATE((1e-10)*4.5,0,0)*H2g;
	H2_NO2P_NOP_H2O     = HMRATE((1e-10)*1.5,0,0)*H2g;
	H3P_NH_NH2P_H2      = HMRATE(1.3e-9,0,0)*H3p;
	H3P_NH2_NH3P_H2     = HMRATE((1e-9)*1.8,0,0)*H3p;
	H3P_NH3_NH4P_H2     = HMRATE(2.7e-9,0,0)*H3p;
	H3P_CN_HCNP_H2      = HMRATE(2.0e-9,0,0)*H3p;
	H3P_NO_HNOP_H2      = HMRATE(1.1e-9,0,0)*H3p;
	H3P_S_HSP_H2        = HMRATE((1e-9)*2.6,0,0)*H3p;
	H3P_CS_HCSP_H2      = HMRATE((1e-9)*2.9,0,0)*H3p;
	H3P_NO2_NOP_OH_H2   = HMRATE(7.0e-10,0,0)*H3p;
	HeP_NH_NP_He_H      = HMRATE(1.1e-9,0,0)*HeII;
	HeP_NH2_NP_He_H2    = HMRATE((1e-10)*8.,0,0)*HeII;
	HeP_NH2_NHP_He_H    = HMRATE((1e-10)*8.,0,0)*HeII;
	HeP_NH3_NHP_He_H2   = HMRATE((1e-10)*1.76,0,0)*HeII;
	HeP_NH3_NH2P_He_H   = HMRATE((1e-9)*1.76,0,0)*HeII;
	HeP_CN_N_CP_He      = HMRATE((1e-10)*8.8,0,0)*HeII;
	HeP_CN_NP_C_He      = HMRATE((1e-10)*8.8,0,0)*HeII;
	HeP_HCN_N_CHP_He    = HMRATE((1e-10)*6.51,0,0)*HeII;
	HeP_HCN_NP_CH_He    = HMRATE((1e-10)*2.17,0,0)*HeII;
	HeP_HCN_N_CP_He_H   = HMRATE((1e-10)*7.75,0,0)*HeII;
	HeP_HCN_CNP_He_H    = HMRATE((1e-9)*1.46,0,0)*HeII;
	HeP_N2_NP_N_He      = HMRATE((1e-10)*9.6,0,0)*HeII;
	HeP_NO_OP_N_He      = HMRATE(2.0e-10,0,0)*HeII;
	HeP_NO_O_NP_He      = HMRATE((1e-9)*1.4,0,0)*HeII;
	HeP_HNO_NOP_He_H    = HMRATE(1.0e-9,0,0)*HeII;
	HeP_HNO_NO_He_HP    = HMRATE(1.0e-9,0,0)*HeII;
	HeP_HS_SP_He_H      = HMRATE((1e-9)*1.7,0,0)*HeII;
	HeP_OCN_CN_OP_He    = HMRATE(3.0e-9,0,0)*HeII;
	HeP_OCN_CNP_O_He    = HMRATE(3.0e-9,0,0)*HeII;
	HeP_SIN_SIP_N_He    = HMRATE(2.0e-9,0,0)*HeII;
	HeP_N2O_N2_OP_He    = HMRATE((1e-10)*2.76,0,0)*HeII;
	HeP_N2O_N2P_O_He    = HMRATE((1e-9)*1.24,0,0)*HeII;
	HeP_N2O_NO_NP_He    = HMRATE((1e-10)*3.,0,0)*HeII;
	HeP_N2O_NOP_N_He    = HMRATE((1e-10)*4.83,0,0)*HeII;
	HeP_CS_SP_C_He      = HMRATE(1.3e-9,0,0)*HeII;
	HeP_CS_S_CP_He      = HMRATE(1.3e-9,0,0)*HeII;
	HeP_NS_S_NP_He      = HMRATE(1.2e-9,0,0)*HeII;
	HeP_NS_SP_N_He      = HMRATE(1.2e-9,0,0)*HeII;
	HeP_SO_S_OP_He      = HMRATE((1e-10)*8.3,0,0)*HeII;
	HeP_SO_SP_O_He      = HMRATE((1e-10)*8.3,0,0)*HeII;
	HeP_OCS_S_COP_He    = HMRATE((1e-10)*7.6,0,0)*HeII;
	HeP_OCS_CSP_O_He    = HMRATE((1e-10)*7.6,0,0)*HeII;
	HeP_OCS_CS_OP_He    = HMRATE((1e-10)*7.6,0,0)*HeII;
	HeP_S2_SP_S_He      = HMRATE(2.0e-9,0,0)*HeII;
	HP_NH_NHP_H         = HMRATE((1e-9)*2.1,0,0)*HII;
	HP_NH2_NH2P_H       = HMRATE((1e-9)*2.9,0,0)*HII;
	HP_NH3_NH3P_H       = HMRATE(1.1e-9,0,0)*HII;
	H_CNP_CN_HP         = HMRATE((1e-10)*1.9,0,0)*HI;
	HP_HCN_HCNP_H       = HMRATE(0.0000000105,-0.13,0)*HII;
	H_HCNP_HCN_HP       = HMRATE((1e-11)*3.7,0,0)*HI;
	H_N2P_N2_HP         = HMRATE((1e-10)*1.2,0,0)*HI;
	HP_NO_NOP_H         = HMRATE((1e-9)*2.9,0,0)*HII;
	HP_HS_HSP_H         = HMRATE((1e-9)*1.6,0,0)*HII;
	HP_SIN_SINP_H       = HMRATE(3.0e-9,0,0)*HII;
	HP_CS_CSP_H         = HMRATE((1e-9)*4.9,0,0)*HII;
	HP_NS_NSP_H         = HMRATE((1e-9)*4.7,0,0)*HII;
	HP_SO_SOP_H         = HMRATE((1e-9)*3.2,0,0)*HII;
	HP_OCS_OCSP_H       = HMRATE((1e-9)*2.1,0,0)*HII;
	HP_S2_S2P_H         = HMRATE(3.0e-9,0,0)*HII;
	H2P_NH_NHP_H2       = HMRATE((1e-10)*7.6,0,0)*H2p;
	H2P_NH2_NH2P_H2     = HMRATE((1e-9)*2.1,0,0)*H2p;
	H2P_NH3_NH3P_H2     = HMRATE((1e-9)*5.7,0,0)*H2p;
	H2P_CN_CNP_H2       = HMRATE(1.2e-9,0,0)*H2p;
	H2P_HCN_HCNP_H2     = HMRATE(2.7e-9,0,0)*H2p;
	H2P_NO_NOP_H2       = HMRATE(1.1e-9,0,0)*H2p;
	HeP_NH3_NH3P_He     = HMRATE((1e-10)*2.64,0,0)*HeII;
	HeP_N2_N2P_He       = HMRATE((1e-10)*6.4,0,0)*HeII;
	Hminus_N_NH_e       = HMRATE(1.0e-9,0,0)*Hm;
	Hminus_NH_NH2_e     = HMRATE(1.0e-10,0,0)*Hm;
	Hminus_NH2_NH3_e    = HMRATE(1.0e-9,0,0)*Hm;
	Hminus_CN_HCN_e     = HMRATE(1.0e-10,0,0)*Hm;
	Hminus_NH4P_NH3_H2  = HMRATE(0.00000023,-0.5,0)*Hm;
	Hminus_NP_N_H       = HMRATE(0.00000023,-0.5,0)*Hm;
	H2_ClP_HClP_H = HMRATE(0.000000001,0,0)*H2g;
	H2_HClP_H2ClP_H = HMRATE(0.0000000013,0,0)*H2g;
	H3P_Cl_HClP_H2 = HMRATE(0.000000001,0,0)*H3p;
	H3P_HCl_H2ClP_H2 = HMRATE(0.0000000038,0,0)*H3p;
	HeP_HCl_ClP_He_H = HMRATE(0.0000000033,0,0)*HeII;
	HeP_CCl_Cl_CP_He = HMRATE(0.0000000033,0,0)*HeII;
	HeP_ClO_ClP_O_He = HMRATE(0.000000001,0,0)*HeII;
	HP_HCl_HClP_H = HMRATE(0.0000000033,1,0)*HII;
	/* >>chng 05 nov 17, TE added following reaction, the same rate as H2g is assumed*/
	h2s_op_ohp_h = HMRATE((1e-9)*1.7,0,0)*H2s;

	/* This turns on the diffuse cloud chemical rates from Tables 6 & 8 of
	>> refer Federman, S. R. & Zsargo, J. 2003, ApJ, 589, 319
	Without the "set rates federman" command, the default values are
	from UMIST */

	if(co.lgFederman)
	{
		H2_COP_HCOP_H = HMRATE((1e-9)*1.4,0,0)*H2g;
		H2_CH2P_CH3P_H = HMRATE((1e-9)*1.2,0,0)*H2g;
		H2_CP_CH2P_nu = HMRATE(1.0e-16,0,0)*H2g;
		H_CH_C_H2 = HMRATE((1e-11)*5.0,0.5,0)*HI;
	}
	/* >>chng 05 sept 13 - NPA:  Add HNC and HCNH+ to chemistry */

	/* non-coupled reactions */
	H3P_HNC_HCNHP_H2 = HMRATE(0.0000000081,0,0)*H3p;
	HeP_HNC_NHP_C_He = HMRATE(0.0000000005,0,0)*HeII;
	HeP_HNC_N_CP_He_H = HMRATE(0.0000000005,0,0)*HeII;
	HeP_HNC_CNP_He_H = HMRATE(0.0000000005,0,0)*HeII;
	HP_HNC_HCN_HP = HMRATE(0.000000001,0,0)*HII;
	H_HNC_HCN_H = HMRATE(0.000000000000001,0,0)*HI;
	H_HNC_HCN_H = HMRATE(0.000000000000136,4.48,0)*HI;
	H2_HCNP_HCNHP_H = HMRATE(0.0000000009,0,0)*H2g;
	H3P_HCN_HCNHP_H2 = HMRATE(0.0000000081,0,0)*H3p;
	
	HeP_C2H2_C2HP_He_H = HMRATE(0.000000001,0,0)*HeII;
	HeP_C3H_C3P_He_H = HMRATE(0.000000002,0,0)*HeII;
	HeP_C2H2_C2P_He_H2 = HMRATE(0.000000002,0,0)*HeII;
	HeP_C2H2_CHP_CH_He = HMRATE(0.0000000075,0,0)*HeII;
	Hm_C2H_C2H2_e = HMRATE(0.000000000294,0,0)*Hm;
	HP_C2H2_C2H2P_H = HMRATE(0.00000000086,0,0)*HII;
	HP_C2H2_C2HP_H2 = HMRATE(0.00000000077,0,0)*HII;
	HP_C3H_C3HP_H = HMRATE(0.00000000057,0,0)*HII;
	HP_C3H_C3P_H2 = HMRATE(0.0000000024,0,0)*HII;
	H2P_C2H_C2H2P_H = HMRATE(0.000000003,0,0)*H2p;
	H2P_C2H2_C2H2P_H2 = HMRATE(0.0000000014,0,0)*H2p;
	H3P_C2H_C2H2P_H2 = HMRATE(0.000000002,0,0)*H3p;
	H3P_C3_C3HP_H2 = HMRATE(0.0000000015,0,0)*H3p;
	H2_C2HP_C2H2P_H = HMRATE(0.000000001,0,0)*H2g;
	H2_C3P_C3HP_H = HMRATE(0.0000000022,0,0)*H2g;
	HeP_C2H2_C2H2P_He = HMRATE(0.000000001,0,0)*HeII;
	H_C2H3P_C2H2P_H2 = HMRATE(0.0000000000000003,-1,0)*HI;
	H3P_C2H2_C2H3P_H2 = HMRATE(0.000000001,0,0)*H3p;
	H2P_C2H2_C2H3P_H = HMRATE(0.0000000014,0,0)*H2p;
	HeP_C3_C2_CP_He = HMRATE(0.00000000094,0,0)*HeII;
	HP_C3_C3P_H = HMRATE(0.00000000485,0,0)*HII;
	HP_C2H_C2HP_H = HMRATE(0.0000000014,0,0)*HII;
	H2P_C2_C2HP_H = HMRATE(0.0000000015,0,0)*H2p;
	H2P_C2H_C2HP_H2 = HMRATE(0.0000000014,0,0)*H2p;
	H3P_C2_C2HP_H2 = HMRATE(0.000000002,0,0)*H3p;
	H2_C2P_C2HP_H = HMRATE(0.000000000254,0,0)*H2g;
	HeP_C2H_CHP_C_He = HMRATE(0.0000000015,0,0)*HeII;
	HeP_C2H_CH_CP_He = HMRATE(0.0000000015,0,0)*HeII;
    HeP_C2H_C2P_He_H = HMRATE(0.000000001,0,0)*HeII;
	Hm_C2_C2H_e = HMRATE(0.00000000235,0,0)*Hm;
	HP_C2H_C2P_H2 = HMRATE(0.0000000014,0,0)*HII;


	/* >>refer	mol	H	CH	=>	C	H	H	UMIST	*/
	
	mole.c[ipCH][ipCH] -= H_CH_C_H_H;
	mole.c[ipCH][ipATC] += H_CH_C_H_H;

	/* >>refer	mol	H	OH	=>	O	H	H	UMIST	*/

	mole.c[ipOH][ipOH] -= H_OH_O_H_H;
	mole.c[ipOH][ipATO] += H_OH_O_H_H;

	/* >>refer	mol	H	H2O	=>	OH	H	H	UMIST	*/

	mole.c[ipH2O][ipH2O] -= H_H2O_OH_H_H;
	mole.c[ipH2O][ipOH] += H_H2O_OH_H_H;
	
	/* >>refer	mol	H	CH	=>	C	H2	UMIST	*/
	
	mole.c[ipCH][ipCH] -= H_CH_C_H2;
	mole.c[ipCH][ipATC] += H_CH_C_H2;
	
	/* >>refer	mol	H	CH+	=>	C+	H2	UMIST	*/
	
	mole.c[ipCHP][ipCHP] -= H_CHP_CP_H2;
	mole.c[ipCHP][ipCP]  += H_CHP_CP_H2;
	
	/* >>refer	mol	H	CH2	=>	CH	H2	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= H_CH2_CH_H2;
	mole.c[ipCH2][ipCH] += H_CH2_CH_H2;
	
	/* >>refer	mol	H	CH3+	=>	CH2+	H2	UMIST	*/
	
	mole.c[ipCH3P][ipCH3P] -= H_CH3P_CH2P_H2;
	mole.c[ipCH3P][ipCH2P] += H_CH3P_CH2P_H2;
	
	/* >>refer	mol	H	OH	=>	O	H2	UMIST	*/
		
	mole.c[ipOH][ipOH] -= H_OH_O_H2;
	mole.c[ipOH][ipATO] += H_OH_O_H2;
	
	/* >>refer	mol	H	H2O	=>	OH	H2	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= H_H2O_OH_H2;
	mole.c[ipH2O][ipOH]  += H_H2O_OH_H2;
	
	/* >>refer	mol	H	O2	=>	OH	O	UMIST	*/
		
	mole.c[ipO2][ipO2] -= H_O2_OH_O;
	mole.c[ipO2][ipOH] += H_O2_OH_O;
	mole.c[ipO2][ipATO] += H_O2_OH_O;
	
	/* >>refer	mol	H	O2	=>	O	O	H	UMIST	*/
		
	mole.c[ipO2][ipO2] -= H_O2_O_O_H;
	mole.c[ipO2][ipATO] += 2*H_O2_O_O_H;
	
	/* >>refer	mol	H	C	=>	CH	PHOTON	UMIST	*/
	
	mole.c[ipATC][ipCH] += H_C_CH_nu;
	mole.c[ipATC][ipATC] -= H_C_CH_nu;
	
	/* >>refer	mol	H	C+	=>	CH+	PHOTON	UMIST	*/
		
	mole.c[ipCP][ipCP]  -= H_CP_CHP_nu;
	mole.c[ipCP][ipCHP] += H_CP_CHP_nu;
	
	/* >>refer	mol	H	OH	=>	H2O	PHOTON	UMIST	*/
		
	mole.c[ipOH][ipOH]  -= H_OH_H2O_nu;
	mole.c[ipOH][ipH2O] += H_OH_H2O_nu;
	
	/* >>refer	mol	H	O	=>	OH	PHOTON	UMIST	*/
		
	mole.c[ipATO][ipOH] += H_O_OH_nu;
	mole.c[ipATO][ipATO] -= H_O_OH_nu;
	
	/* >>refer	mol	H-	CH	=>	CH2	e	UMIST	*/
	
	mole.c[ipCH][ipCH]  -= Hminus_CH_CH2_e;
	mole.c[ipCH][ipCH2] += Hminus_CH_CH2_e;
	
	/* >>refer	mol	H-	C	=>	CH	e	UMIST	*/
	
	mole.c[ipATC][ipCH]  += Hminus_CH_CH2_e;
	mole.c[ipATC][ipATC]  -= Hminus_CH_CH2_e;
	
	/* >>refer	mol	H-	OH	=>	H2O	e	UMIST	*/
		
	mole.c[ipOH][ipOH]  -= Hminus_OH_H2O_e;
	mole.c[ipOH][ipH2O] += Hminus_OH_H2O_e;
	
	/* >>refer	mol	H-	O	=>	OH	e	UMIST	*/
		
	mole.c[ipATO][ipOH] += Hminus_O_OH_e;
	mole.c[ipATO][ipATO] -= Hminus_O_OH_e;
		
	
	/* >>refer	mol	H-	H3O+	=>	H2O	H2	UMIST	*/
		
	mole.c[ipH3OP][ipH3OP] -= Hminus_H3OP_H2O_H2;
	mole.c[ipH3OP][ipH2O] += Hminus_H3OP_H2O_H2;
	
	/* >>refer	mol	H-	H3O+	=>	OH	H2	H	UMIST	*/
		
	mole.c[ipH3OP][ipH3OP] -= Hminus_H3OP_OH_H2_H;
	mole.c[ipH3OP][ipOH] += Hminus_H3OP_OH_H2_H;
	
	/* >>refer	mol	H+	CH	=>	CH+	H	UMIST	*/
		
	mole.c[ipCH][ipCH] -= HP_CH_CHP_H;
	mole.c[ipCH][ipCHP] += HP_CH_CHP_H;
	
	/* >>refer	mol	H+	CH2	=>	CH2+	H	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= HP_CH2_CH2P_H;
	mole.c[ipCH2][ipCH2P] += HP_CH2_CH2P_H;
	
	/* >>refer	mol	H+	H2O	=>	H2O+	H	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= HP_H2O_H2OP_H;
	mole.c[ipH2O][ipH2OP] += HP_H2O_H2OP_H;
	
	/* >>refer	mol	H+	O2	=>	O2+	H	UMIST	*/
	
	mole.c[ipO2][ipO2] -= HP_O2_O2P_H;
	mole.c[ipO2][ipO2P] += HP_O2_O2P_H;
	
	/* >>refer	mol	H+	OH	=>	OH+	H	UMIST	*/
		
	mole.c[ipOH][ipOH] -= HP_OH_OHP_H;
	mole.c[ipOH][ipOHP] += HP_OH_OHP_H;
	
	/* >>refer	mol	H+	SiO	=>	SiO+	H	UMIST	*/
		
	mole.c[ipSIO][ipSIO] -= HP_SiO_SiOP_H;
	mole.c[ipSIO][ipSIOP] += HP_SiO_SiOP_H;
	
	/* >>refer	mol	H+	CH2	=>	CH+	H2	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= HP_CH2_CHP_H2;
	mole.c[ipCH2][ipCHP] += HP_CH2_CHP_H2;
	
	/* >>refer	mol	H+	SiH	=>	Si+	H2	UMIST	*/
		
	mole.c[ipSIH][ipSIH] -= HP_SiH_SiP_H2;
	mole.c[ipSIH][ipSIP] += HP_SiH_SiP_H2;
	
	/* >>refer	mol	H2	C	=>	CH	H	UMIST	*/
	
	mole.c[ipATC][ipCH] += H2_C_CH_H;
	mole.c[ipATC][ipATC] -= H2_C_CH_H;
	
	/* >>refer	mol	H2	C+	=>	CH+	H	UMIST	*/
		
	mole.c[ipCP][ipCP]  -= H2_CP_CHP_H;
	mole.c[ipCP][ipCHP] += H2_CP_CHP_H;
	
	/* >>refer	mol	H2	CH	=>	CH2	H	UMIST	*/
		
	mole.c[ipCH][ipCH] -= H2_CH_CH2_H;
	mole.c[ipCH][ipCH2] += H2_CH_CH2_H;
	
	/* >>refer	mol	H2	CH+	=>	CH2+	H	UMIST	*/
		
	mole.c[ipCHP][ipCHP] -= H2_CHP_CH2P_H;
	mole.c[ipCHP][ipCH2P] += H2_CHP_CH2P_H;
	
	/* >>refer	mol	H2	OH	=>	H2O	H	UMIST	*/
		
	mole.c[ipOH][ipOH] -= H2_OH_H2O_H;
	mole.c[ipOH][ipH2O] += H2_OH_H2O_H;
	
	/* >>refer	mol	H2	OH+	=>	H2O+	H	UMIST	*/
		
	mole.c[ipOHP][ipOHP] -= H2_OHP_H2OP_H;
	mole.c[ipOHP][ipH2OP] += H2_OHP_H2OP_H;
	
	/* >>refer	mol	H2	H2O+	=>	H3O+	H	UMIST	*/
		
	mole.c[ipH2OP][ipH2OP] -= H2_H2OP_H3OP_H;
	mole.c[ipH2OP][ipH3OP] += H2_H2OP_H3OP_H;
	
	/* >>refer	mol	H2	CO+	=>	HCO+	H	UMIST	*/
		
	mole.c[ipCOP][ipCOP] -= H2_COP_HCOP_H;
	mole.c[ipCOP][ipHCOP] += H2_COP_HCOP_H;
	
	/* >>refer	mol	H2	O	=>	OH	H	UMIST	*/
		
	mole.c[ipATO][ipOH] += H2_O_OH_H;
	mole.c[ipATO][ipATO] -= H2_O_OH_H;

	/* >>refer	mol	H2	CH2+	=>	CH3+	H	*/

	mole.c[ipCH2P][ipCH2P] -= H2_CH2P_CH3P_H;
	mole.c[ipCH2P][ipCH3P] += H2_CH2P_CH3P_H;
	
	/* >>refer	mol	H2	O+	=>	OH+	H	UMIST	*/
		
	mole.c[ipOP][ipOP]  -= H2_OP_OHP_H;
	mole.c[ipOP][ipOHP] += H2_OP_OHP_H;
	
	/* >>refer	mol	H2	SiO+	=>	SiOH+	H	UMIST	*/
	
	mole.c[ipSIOP][ipSIOP] -= H2_SiOP_SiOHP_H;
	mole.c[ipSIOP][ipHOSIP] += H2_SiOP_SiOHP_H;
	
	/* >>refer	mol	H2	CH	=>	C	H2	H	UMIST	*/
		
	mole.c[ipCH][ipCH] -= H2_CH_C_H2_H;
	mole.c[ipCH][ipATC] += H2_CH_C_H2_H;
	
	/* >>refer	mol	H2	OH	=>	O	H2	H	UMIST	*/
	
	mole.c[ipOH][ipOH] -= H2_OH_O_H2_H;
	mole.c[ipOH][ipATO] += H2_OH_O_H2_H;
	
	/* >>refer	mol	H2	H2O	=>	OH	H2	H	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= H2_H2O_OH_H2_H;
	mole.c[ipH2O][ipOH] += H2_H2O_OH_H2_H;
	
	/* >>refer	mol	H2	O2	=>	O	O	H2	UMIST	*/
		
	mole.c[ipO2][ipO2] -= H2_O2_O_O_H2;
	mole.c[ipO2][ipATO] += 2*H2_O2_O_O_H2;
	
	/* >>refer	mol	H2	O2	=>	OH	OH	UMIST	*/
	
	/**********************************************************************
	*                                                                     *
	*  A note here about the stoic. factors.  For a reaction of the type: *	
	*                                                                     *
	*   aA + bB => cC + dD                                                *
	*                                                                     *	
	*  The rates are related by:                                          *
	*                                                                     *
	*  (1/a)*D[A]/Dt = (1/b)*D[B]/Dt = - (1/c)*D[C]/Dt =  - (1/d)*D[D]/Dt *
	*                                                                     *
	*  (As a reference, look at most Physical Chemistry books)		      *
	*                                                                     *
	*                                                                     *
	***********************************************************************/

	mole.c[ipO2][ipO2] -= H2_O2_OH_OH;
	mole.c[ipO2][ipOH] += 2*H2_O2_OH_OH;
	
	/* >>refer	mol	H2	C	=>	CH2	PHOTON	UMIST	*/
		
	mole.c[ipATC][ipCH2] += H2_C_CH2_nu;
	mole.c[ipATC][ipATC] -= H2_C_CH2_nu;
	
	/* >>refer	mol	H2	C+	=>	CH2+	PHOTON	UMIST	*/
		
	mole.c[ipCP][ipCP]   -= H2_CP_CH2P_nu;
	mole.c[ipCP][ipCH2P] += H2_CP_CH2P_nu;
	
	/* >>refer	mol	H2	Si+	=>	SiH2+	PHOTON	UMIST	*/
		
	mole.c[ipSIP][ipSIP] -= H2_SiP_SiH2P_nu;
	mole.c[ipSIP][ipSIH2P] += H2_SiP_SiH2P_nu;
	
	/* >>refer	mol	H2+	C	=>	CH+	H	UMIST	*/
		
	mole.c[ipATC][ipCHP] += H2P_C_CHP_H;
	mole.c[ipATC][ipATC] -= H2P_C_CHP_H;
	
	/* >>refer	mol	H2+	CH	=>	CH2+	H	UMIST	*/
		
	mole.c[ipCH][ipCH] -= H2P_CH_CH2P_H;
	mole.c[ipCH][ipCH2P] += H2P_CH_CH2P_H;
	
	/* >>refer	mol	H2+	CH2	=>	CH3+	H	UMIST	*/
	
	mole.c[ipCH2][ipCH2] -= H2P_CH2_CH3P_H;
	mole.c[ipCH2][ipCH3P] += H2P_CH2_CH3P_H;
	
	/* >>refer	mol	H2+	OH	=>	H2O+	H	UMIST	*/
		
	mole.c[ipOH][ipOH] -= H2P_OH_H2OP_H;
	mole.c[ipOH][ipH2OP] += H2P_OH_H2OP_H;
	
	/* >>refer	mol	H2+	H2O	=>	H3O+	H	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= H2P_H2O_H3OP_H;
	mole.c[ipH2O][ipH3OP] += H2P_H2O_H3OP_H;
	
	/* >>refer	mol	H2+	O	=>	OH+	H	UMIST	*/
		
	mole.c[ipATO][ipOHP] += H2P_O_OHP_H;
	mole.c[ipATO][ipATO] -= H2P_O_OHP_H;
	
	/* >>refer	mol	H2+	CH	=>	CH+	H2	UMIST	*/
		
	mole.c[ipCH][ipCH] -= H2P_CH_CHP_H2;
	mole.c[ipCH][ipCHP] += H2P_CH_CHP_H2;
	
	/* >>refer	mol	H2+	CH2	=>	CH2+	H2	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= H2P_CH2_CH2P_H2;
	mole.c[ipCH2][ipCH2P] += H2P_CH2_CH2P_H2;
	
	/* >>refer	mol	H2 + S	=>	HS	H	UMIST	*/
	/* >>chng 05 aug 02, NA add following */
	mole.c[ipATS][ipATS] -= H2_S_HS_H;
	mole.c[ipATS][ipHS] += H2_S_HS_H;
	
	/* >>refer	mol	H2+	H2O	=>	H2O+	H2	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= H2P_H2O_H2OP_H2;
	mole.c[ipH2O][ipH2OP] += H2P_H2O_H2OP_H2;
	
	/* >>refer	mol	H2+	O2	=>	O2+	H2	UMIST	*/
		
	mole.c[ipO2][ipO2] -= H2P_O2_O2P_H2;
	mole.c[ipO2][ipO2P] += H2P_O2_O2P_H2;
	
	/* >>refer	mol	H2+	OH	=>	OH+	H2	UMIST	*/
		
	mole.c[ipOH][ipOH] -= H2P_OH_OHP_H2;
	mole.c[ipOH][ipOHP] += H2P_OH_OHP_H2;
	
	/* >>refer	mol	H3+	C	=>	CH+	H2	UMIST	*/
		
	mole.c[ipATC][ipCHP] += H3P_C_CHP_H2;
	mole.c[ipATC][ipATC] -= H3P_C_CHP_H2;
	
	/* >>refer	mol	H3+	CH	=>	CH2+	H2	UMIST	*/
		
	mole.c[ipCH][ipCH] -= H3P_CH_CH2P_H2;
	mole.c[ipCH][ipCH2P] += H3P_CH_CH2P_H2;
	
	/* >>refer	mol	H3+	CH2	=>	CH3P	H2	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= H3P_CH2_CH3P_H2;
	mole.c[ipCH2][ipCH3P] += H3P_CH2_CH3P_H2;
	
	/* >>refer	mol	H3+	OH	=>	H2O+	H2	UMIST	*/
		
	mole.c[ipOH][ipOH] -= H3P_OH_H2OP_H2;
	mole.c[ipOH][ipH2OP] += H3P_OH_H2OP_H2;
	
	/* >>refer	mol	H3+	H2O	=>	H3O+	H2	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= H3P_H2O_H3OP_H2;
	mole.c[ipH2O][ipH3OP] += H3P_H2O_H3OP_H2;
	
	/* >>refer	mol	H3+	O	=>	OH+	H2	UMIST	*/
	
	mole.c[ipATO][ipOHP] += H3P_O_OHP_H2;
	mole.c[ipATO][ipATO] -= H3P_O_OHP_H2;
	
	/* >>refer	mol	H3+	SiH	=>	SiH2+	H2	UMIST	*/
		
	mole.c[ipSIH][ipSIH] -= H3P_SiH_SiH2P_H2;
	mole.c[ipSIH][ipSIH2P] += H3P_SiH_SiH2P_H2;
	
	/* >>refer	mol	H3+	SiO	=>	SiOH+	H2	UMIST	*/
		
	mole.c[ipSIO][ipSIO] -= H3P_SiO_SiOHP_H2;
	mole.c[ipSIO][ipHOSIP] += H3P_SiO_SiOHP_H2;
	
	/* >>refer	mol	He+	CH	=>	CH+	He	UMIST	*/
		
	mole.c[ipCH][ipCH] -= HeP_CH_CHP_He;
	mole.c[ipCH][ipCHP] += HeP_CH_CHP_He;
	
	/* >>refer	mol	He+	H2O	=>	H2O+	He	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= HeP_H2O_H2OP_He;
	mole.c[ipH2O][ipH2OP] += HeP_H2O_H2OP_He;
	
	/* >>refer	mol	He+	O2	=>	O2+	He	UMIST	*/
		
	mole.c[ipO2][ipO2] -= HeP_O2_O2P_He;
	mole.c[ipO2][ipO2P] += HeP_O2_O2P_He;
	
	/* >>refer	mol	He+	CH	=>	C+	H	He	UMIST	*/
		
	mole.c[ipCH][ipCH] -= HeP_CH_CP_He_H;
	mole.c[ipCH][ipCP] += HeP_CH_CP_He_H;
	
	/* >>refer	mol	He+	CH2	=>	CH+	H	He	UMIST	*/
		
	mole.c[ipCH2][ipCH2] -= HeP_CH2_CHP_He_H;
	mole.c[ipCH2][ipCHP] += HeP_CH2_CHP_He_H;
	
	/* >>refer	mol	He+	OH	=>	O+	H	He	UMIST	*/
		
	mole.c[ipOH][ipOH] -= HeP_OH_OP_He_H;
	mole.c[ipOH][ipOP] += HeP_OH_OP_He_H;
	
	/* >>refer	mol	He+	H2O	=>	OH+	H	He	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= HeP_H2O_OHP_He_H;
	mole.c[ipH2O][ipOHP] += HeP_H2O_OHP_He_H;
	
	/* >>refer	mol	He+	SiH	=>	Si+	H	He	UMIST	*/
		
	mole.c[ipSIH][ipSIH] -= HeP_SiH_SiP_He_H;
	mole.c[ipSIH][ipSIP] += HeP_SiH_SiP_He_H;
	
	/* >>refer	mol	He+	H2O	=>	OH	H+	He	UMIST	*/
		
	mole.c[ipH2O][ipH2O] -= HeP_H2O_OH_He_HP;
	mole.c[ipH2O][ipOH] += HeP_H2O_OH_He_HP;
		
	/* >>refer	mol	He+	CH2	=>	C+	H2	He	UMIST	*/
	
	mole.c[ipCH2][ipCH2] -= HeP_CH2_CP_He_H2;
	mole.c[ipCH2][ipCP]  += HeP_CH2_CP_He_H2;
	
	/* >>refer	mol	He+	O2	=>	O+	O	He	UMIST	*/
		
	mole.c[ipO2][ipO2] -= HeP_O2_OP_O_He;
	mole.c[ipO2][ipOP] += HeP_O2_OP_O_He;
	mole.c[ipO2][ipATO] += HeP_O2_OP_O_He;
	
	/* >>refer	mol	He+	SiO	=>	Si+	O	He	UMIST	*/
		
	mole.c[ipSIO][ipSIO] -= HeP_SiO_SiP_O_He;
	mole.c[ipSIO][ipSIP] += HeP_SiO_SiP_O_He;
	mole.c[ipSIO][ipATO] += HeP_SiO_SiP_O_He;
	
	/* >>refer	mol	He+	SiO	=>	Si	O+	He	UMIST	*/
		
	mole.c[ipSIO][ipSIO] -= HeP_SiO_Si_OP_He;
	mole.c[ipSIO][ipOP]  += HeP_SiO_Si_OP_He;
	mole.c[ipSIO][ipATSI]  += HeP_SiO_Si_OP_He;
	

	
	/* new simpler reactions for CH3, CH4, CH4+, CH5+ */
	
	/* >>refer mol   H CH3  =>  CH2 H2   UMIST    */
	mole.c[ipCH3][ipCH3]  -=  H_CH3_CH2_H2;
	mole.c[ipCH3][ipCH2]  +=  H_CH3_CH2_H2;
	
	/* >>refer mol   H CH4P  =>  CH3P H2   UMIST    */
	mole.c[ipCH4P][ipCH4P]  -=  H_CH4P_CH3P_H2;
	mole.c[ipCH4P][ipCH3P]  +=  H_CH4P_CH3P_H2;
	
	/* >>refer mol   H CH5P  =>  CH4P H2   UMIST    */
	mole.c[ipCH5P][ipCH5P]  -=  H_CH5P_CH4P_H2;
	mole.c[ipCH5P][ipCH4P]  +=  H_CH5P_CH4P_H2;
	
	/* >>refer mol   H2 CH2  =>  CH3 H   UMIST    */
	mole.c[ipCH2][ipCH2]  -=  H2_CH2_CH3_H;
	mole.c[ipCH2][ipCH3]  +=  H2_CH2_CH3_H;
	
	/* >>refer mol   H2 CH  =>  CH3 nu   UMIST    */
	mole.c[ipCH][ipCH]  -=  H2_CH_CH3_nu;
	mole.c[ipCH][ipCH3]  +=  H2_CH_CH3_nu;
	
	/* >>refer mol   H2 CH3  =>  CH4 H   UMIST    */
	mole.c[ipCH3][ipCH3]  -=  H2_CH3_CH4_H;
	mole.c[ipCH3][ipCH4]  +=  H2_CH3_CH4_H;
	
	/* >>refer mol   H2 CH4P  =>  CH5P H   UMIST    */
	mole.c[ipCH4P][ipCH4P]  -=  H2_CH4P_CH5P_H;
	mole.c[ipCH4P][ipCH5P]  +=  H2_CH4P_CH5P_H;
	
	/* >>refer mol   H2 CH3P  =>  CH5P nu   UMIST    */
	mole.c[ipCH3P][ipCH3P]  -=  H2_CH3P_CH5P_nu;
	mole.c[ipCH3P][ipCH5P]  +=  H2_CH3P_CH5P_nu;
	
	/* >>refer mol   H2P CH4  =>  CH3P H2 H  UMIST    */
	mole.c[ipCH4][ipCH4]  -=  H2P_CH4_CH3P_H2;
	mole.c[ipCH4][ipCH3P]  +=  H2P_CH4_CH3P_H2;
	
	/* >>refer mol   H2P CH4  =>  CH4P H2   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  H2P_CH4_CH4P_H2;
	mole.c[ipCH4][ipCH4P]  +=  H2P_CH4_CH4P_H2;
	
	/* >>refer mol   H2P CH4  =>  CH5P H   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  H2P_CH4_CH5P_H;
	mole.c[ipCH4][ipCH5P]  +=  H2P_CH4_CH5P_H;
	
	/* >>refer mol   H3P CH3  =>  CH4P H2   UMIST    */
	mole.c[ipCH3][ipCH3]  -=  H3P_CH3_CH4P_H2;
	mole.c[ipCH3][ipCH4P]  +=  H3P_CH3_CH4P_H2;
	
	/* >>refer mol   H3P CH4  =>  CH5P H2   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  H3P_CH4_CH5P_H2;
	mole.c[ipCH4][ipCH5P]  +=  H3P_CH4_CH5P_H2;
	
	/* >>refer mol   HeP CH3  =>  CHP He H2  UMIST    */
	mole.c[ipCH3][ipCH3]  -=  HeP_CH3_CHP_He;
	mole.c[ipCH3][ipCHP]  +=  HeP_CH3_CHP_He;
	
	/* >>refer mol   HeP CH4  =>  CHP He H2 H UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HeP_CH4_CHP_He;
	mole.c[ipCH4][ipCHP]  +=  HeP_CH4_CHP_He;
	
	/* >>refer mol   HeP CH4  =>  CH2P He H2  UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HeP_CH4_CH2P_He;
	mole.c[ipCH4][ipCH2P]  +=  HeP_CH4_CH2P_He;
	
	/* >>refer mol   HeP CH4  =>  CH3 He HP  UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HeP_CH4_CH3_He;
	mole.c[ipCH4][ipCH3]  +=  HeP_CH4_CH3_He;
	
	/* >>refer mol   HeP CH4  =>  CH3P He H  UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HeP_CH4_CH3P_He;
	mole.c[ipCH4][ipCH3P]  +=  HeP_CH4_CH3P_He;
	
	/* >>refer mol   HeP CH4  =>  CH4P He   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HeP_CH4_CH4P_He;
	mole.c[ipCH4][ipCH4P]  +=  HeP_CH4_CH4P_He;
	
	/* >>refer mol   Hminus CH2  =>  CH3 e   UMIST    */
	mole.c[ipCH2][ipCH2]  -=  Hminus_CH2_CH3_e;
	mole.c[ipCH2][ipCH3]  +=  Hminus_CH2_CH3_e;
	
	/* >>refer mol   Hminus CH3  =>  CH4 e   UMIST    */
	mole.c[ipCH3][ipCH3]  -=  Hminus_CH3_CH4_e;
	mole.c[ipCH3][ipCH4]  +=  Hminus_CH3_CH4_e;
	
	/* >>refer mol   HP CH3  =>  CH3P H   UMIST    */
	mole.c[ipCH3][ipCH3]  -=  HP_CH3_CH3P_H;
	mole.c[ipCH3][ipCH3P]  +=  HP_CH3_CH3P_H;
	
	/* >>refer mol   HP CH4  =>  CH3P H2   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HP_CH4_CH3P_H2;
	mole.c[ipCH4][ipCH3P]  +=  HP_CH4_CH3P_H2;
	
	/* >>refer mol   HP CH4  =>  CH4P H   UMIST    */
	mole.c[ipCH4][ipCH4]  -=  HP_CH4_CH4P_H;
	mole.c[ipCH4][ipCH4P]  +=  HP_CH4_CH4P_H;
	
	/* >>refer	mol	He+		C2	=>		C+		C	He	UMIST	*/
	
	mole.c[ipC2][ipC2] -= HeP_C2_CP_C_He;
	mole.c[ipC2][ipCP]  += HeP_C2_CP_C_He;
	mole.c[ipC2][ipATC]  += HeP_C2_CP_C_He;
	/****************************Add all terms with H2star from TH85*********************/

	
	mole.c[ipCH][ipCH]  -= H2s_CH_C_H2_H; 
	mole.c[ipCH][ipATC]  += H2s_CH_C_H2_H; 
	
	mole.c[ipOH][ipOH]  -= H2s_OH_O_H2_H; 
	mole.c[ipOH][ipATO]  += H2s_OH_O_H2_H; 
	
	mole.c[ipH2O][ipH2O] -= H2s_H2O_OH_H2_H;
	mole.c[ipH2O][ipOH]  += H2s_H2O_OH_H2_H; 
	
	mole.c[ipO2][ipO2]   -= H2s_O2_O_O_H2;
	mole.c[ipO2][ipATO]   += 2*H2s_O2_O_O_H2;
	
	mole.c[ipCH2][ipCH2]  -= H2s_CH2_CH3_H;  
	mole.c[ipCH2][ipCH3]  += H2s_CH2_CH3_H; 
	
	mole.c[ipCH][ipCH]   -= H2s_CH_CH3_nu;
	mole.c[ipCH][ipCH3]  += H2s_CH_CH3_nu;

	mole.c[ipCH3][ipCH3]  -= H2s_CH3_CH4_H;
	mole.c[ipCH3][ipCH4]  += H2s_CH3_CH4_H;
	
	 
	/* >>refer	mol	H2* + CH => CH2 + H	 Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* Formation of CH2 */
	mole.c[ipCH][ipCH2] += h2s_ch_ch2_h;

	/* Destruction of CH */
	mole.c[ipCH][ipCH] -=h2s_ch_ch2_h;

	/* >>refer	mol	H2* + O => OH + H	 Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* Formation of OH */
	
	mole.c[ipATO][ipOH] += h2s_o_oh_h;
	mole.c[ipATO][ipATO] -= h2s_o_oh_h;

	/* >>refer	mol	H2* + OH => H2O + H	 Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* Formation of H2O */
	/* >>chng 03 sep 12, sec par had been 0.67, was supposed to
	 * be zero - Gargi Shaw */
	mole.c[ipOH][ipH2O] += h2s_o_oh_h;

	/* Destruction of OH */
	mole.c[ipOH][ipOH] -= h2s_oh_h2o_h;

	/* >>refer	mol	H2* + O2 => 2OH	 Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* >>chng 03 sep 12, sec par was supposed to be 0, was
	 * 0.67, caught by Gargi Shaw */
	/* Formation of OH */
	mole.c[ipO2][ipOH] += 2*h2s_o2_twooh;

	/* Destruction of O2 */
	mole.c[ipO2][ipO2] -= h2s_o2_twooh;

	/* >>refer	mol	H2* + C => CH + H	 Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* Formation of CH */
	mole.c[ipATC][ipCH] += h2s_c_ch_h;
	mole.c[ipATC][ipATC] -= h2s_c_ch_h;

	/* >>refer	mol	H2* + C+ => CH+ + H	Tielens & Hollenbach 1985, Ap. J. 291, 722 Table 9*/

	/* Formation of CH+ */
	mole.c[ipCP][ipCHP] += h2s_cp_chp_h;

	/* Destruction of C+ */
	mole.c[ipCP][ipCP] -= h2s_cp_chp_h;

	/* >>chng 05 nov 17, add this
	 *>>refer	mol	H2s	O+	=>	OH+	H	rate from UMIST	for H2g*/
		
	mole.c[ipOP][ipOP]  -= h2s_op_ohp_h;
	mole.c[ipOP][ipOHP] += h2s_op_ohp_h;

		/*  >>refer   mol  H2     +  N     =>   NH    +     H     UMIST  */

			mole.c[ipATN][ipNH]  += H2_N_NH_H;
			mole.c[ipATN][ipATN]  -= H2_N_NH_H;

		/*  >>refer   mol  H2     +  NH    =>   NH2   +     H	    UMIST  */
			
			mole.c[ipNH][ipNH]  -= H2_NH_NH2_H;
			mole.c[ipNH][ipNH2] += H2_NH_NH2_H;

		/*  >>refer   mol  H2     +  NH2   =>   NH3   +     H	    UMIST  */

			mole.c[ipNH2][ipNH2] -= H2_NH2_NH3_H;
			mole.c[ipNH2][ipNH3] += H2_NH2_NH3_H;

		/*  >>refer   mol  H2     +  CN    =>   HCN   +     H	    UMIST  */

			mole.c[ipCN][ipCN]  -= H2_CN_HCN_H;
			mole.c[ipCN][ipHCN] += H2_CN_HCN_H;



		/*  >>refer   mol  H+     +  HNO   =>   NO+   +     H2    UMIST  */

			mole.c[ipHNO][ipHNO] -= HP_HNO_NOP_H2;
			mole.c[ipHNO][ipNOP] += HP_HNO_NOP_H2;

		/*  >>refer   mol  H+     +  HS    =>   S+    +     H2    UMIST  */

			mole.c[ipHS][ipHS] -= HP_HS_SP_H2;
			mole.c[ipHS][ipSP] += HP_HS_SP_H2;

		/*  >>refer   mol  H      +  HS+   =>   S+    +     H2    UMIST  */

			mole.c[ipHSP][ipHSP] -= H_HSP_SP_H2;
			mole.c[ipHSP][ipSP] += H_HSP_SP_H2;

		/*  >>refer   mol  H2+    +  N     =>   NH+   +     H     UMIST  */

			mole.c[ipATN][ipNHP] += H2P_N_NHP_H;
			mole.c[ipATN][ipATN] -= H2P_N_NHP_H;

		/*  >>refer   mol  H2     +  N+    =>   NH+   +     H     UMIST  */

			mole.c[ipNP][ipNP] -= H2_NP_NHP_H;
			mole.c[ipNP][ipNHP] += H2_NP_NHP_H;

		/*  >>refer   mol  H2     +  NH+   =>   N     +     H3+   UMIST  */

			mole.c[ipNHP][ipNHP] -= H2_NHP_N_H3P;
			mole.c[ipNHP][ipATN] += H2_NHP_N_H3P;

		/*  >>refer   mol  H2+    +  NH    =>   NH2+  +     H     UMIST  */

			mole.c[ipNH][ipNH] -= H2P_NH_NH2P_H;
			mole.c[ipNH][ipNH2P] += H2P_NH_NH2P_H;

		/*  >>refer   mol  H2     +  NH+   =>   NH2+  +     H     UMIST  */

			mole.c[ipNHP][ipNHP] -= H2_NHP_NH2P_H;
			mole.c[ipNHP][ipNH2P] += H2_NHP_NH2P_H;

		/*  >>refer   mol  H2     +  NH2+  =>   NH3+  +     H     UMIST  */

			mole.c[ipNH2P][ipNH2P] -= H2_NH2P_NH3P_H;
			mole.c[ipNH2P][ipNH3P] += H2_NH2P_NH3P_H;

		/*  >>refer   mol  H2     +  NH3+  =>   NH4+  +     H     UMIST  */

			mole.c[ipNH3P][ipNH3P] -= H2_NH3P_NH4P_H;
			mole.c[ipNH3P][ipNH4P] += H2_NH3P_NH4P_H;

		/*  >>refer   mol  H2+    +  CN    =>   HCN+  +     H     UMIST  */

			mole.c[ipCN][ipCN] -= H2P_CN_HCNP_H;
			mole.c[ipCN][ipHCNP] += H2P_CN_HCNP_H;

		/*  >>refer   mol  H2     +  CN+   =>   HCN+  +     H     UMIST  */

			mole.c[ipCNP][ipCNP] -= H2_CNP_HCNP_H;
			mole.c[ipCNP][ipHCNP] += H2_CNP_HCNP_H;

		/*  >>refer   mol  H2+    +  NO    =>   HNO+  +     H     UMIST  */

			mole.c[ipNO][ipNO] -= H2P_NO_HNOP_H;
			mole.c[ipNO][ipHNOP] += H2P_NO_HNOP_H;

		/*  >>refer   mol  H2     +  S+    =>   HS+   +     H     UMIST  */

			mole.c[ipSP][ipSP] -= H2_SP_HSP_H;
			mole.c[ipSP][ipHSP] += H2_SP_HSP_H;

		/*  >>refer   mol  H2     +  CS+   =>   HCS+  +     H     UMIST  */

			mole.c[ipCSP][ipCSP] -= H2_CSP_HCSP_H;
			mole.c[ipCSP][ipHCSP] += H2_CSP_HCSP_H;

		/*  >>refer   mol  H2     +  NO2+  =>   NO+   +     H2O   UMIST  */

			mole.c[ipNO2P][ipNO2P] -= H2_NO2P_NOP_H2O;
			mole.c[ipNO2P][ipNOP] += H2_NO2P_NOP_H2O;
			mole.c[ipNO2P][ipH2O] += H2_NO2P_NOP_H2O;

		/*  >>refer   mol  H3+    +  NH    =>   NH2+  +     H2    UMIST  */

			mole.c[ipNH][ipNH] -= H3P_NH_NH2P_H2;
			mole.c[ipNH][ipNH2P] += H3P_NH_NH2P_H2;

		/*  >>refer   mol  H3+    +  NH2   =>   NH3+  +     H2    UMIST  */

			mole.c[ipNH2][ipNH2] -= H3P_NH2_NH3P_H2;
			mole.c[ipNH2][ipNH3P] += H3P_NH2_NH3P_H2;

		/*  >>refer   mol  H3+    +  NH3   =>   NH4+  +     H2    UMIST  */

			mole.c[ipNH3][ipNH3] -= H3P_NH3_NH4P_H2;
			mole.c[ipNH3][ipNH4P] += H3P_NH3_NH4P_H2;

		/*  >>refer   mol  H3+    +  CN    =>   HCN+  +     H2    UMIST  */

			mole.c[ipCN][ipCN] -= H3P_CN_HCNP_H2;
			mole.c[ipCN][ipHCNP] += H3P_CN_HCNP_H2;

		/*  >>refer   mol  H3+    +  NO    =>   HNO+  +     H2    UMIST  */

			mole.c[ipNO][ipNO] -= H3P_NO_HNOP_H2;
			mole.c[ipNO][ipHNOP] += H3P_NO_HNOP_H2;

		/*  >>refer   mol  H3+    +  S     =>   HS+   +     H2    UMIST  */

			mole.c[ipATS][ipHSP] += H3P_S_HSP_H2;
			mole.c[ipATS][ipATS] -= H3P_S_HSP_H2;

		/*  >>refer   mol  H3+    +  CS    =>   HCS+  +     H2    UMIST  */

			mole.c[ipCS][ipCS] -= H3P_CS_HCSP_H2;
			mole.c[ipCS][ipHCSP] += H3P_CS_HCSP_H2;

		/*  >>refer   mol  H3+    +  NO2   =>   NO+   +     OH +  H2     UMIST  */

			mole.c[ipNO2][ipNO2] -= H3P_NO2_NOP_OH_H2;
			mole.c[ipNO2][ipNOP] += H3P_NO2_NOP_OH_H2;

		/*  >>refer   mol  He+    +  NH    =>   N+    +     He +  H      UMIST  */

			mole.c[ipNH][ipNH] -= HeP_NH_NP_He_H;
			mole.c[ipNH][ipNP] += HeP_NH_NP_He_H;

		/*  >>refer   mol  He+    +  NH2   =>   N+    +     He +  H2     UMIST  */

			mole.c[ipNH2][ipNH2] -= HeP_NH2_NP_He_H2;
			mole.c[ipNH2][ipNP] += HeP_NH2_NP_He_H2;

		/*  >>refer   mol  He+    +  NH2   =>   NH+   +     He +  H      UMIST  */

			mole.c[ipNH2][ipNH2] -= HeP_NH2_NHP_He_H;
			mole.c[ipNH2][ipNHP] += HeP_NH2_NHP_He_H;

		/*  >>refer   mol  He+    +  NH3   =>   NH+   +     He +  H2     UMIST  */

			mole.c[ipNH3][ipNH3] -= HeP_NH3_NHP_He_H2;
			mole.c[ipNH3][ipNHP] += HeP_NH3_NHP_He_H2;

		/*  >>refer   mol  He+    +  NH3   =>   NH2+  +     He +  H      UMIST  */

			mole.c[ipNH3][ipNH3] -= HeP_NH3_NH2P_He_H;
			mole.c[ipNH3][ipNH2P] += HeP_NH3_NH2P_He_H;

		/*  >>refer   mol  He+    +  CN    =>   N     +     C+ +  He     UMIST  */

			mole.c[ipCN][ipCN] -= HeP_CN_N_CP_He;
			mole.c[ipCN][ipCP] += HeP_CN_N_CP_He;
			mole.c[ipCN][ipATN] += HeP_CN_N_CP_He;

		/*  >>refer   mol  He+    +  CN    =>   N+    +     C  +  He     UMIST  */

			mole.c[ipCN][ipCN] -= HeP_CN_NP_C_He;
			mole.c[ipCN][ipNP] += HeP_CN_NP_C_He;
			mole.c[ipCN][ipATC] += HeP_CN_NP_C_He;

		/*  >>refer   mol  He+    +  HCN   =>   N     +     CH+ +  He     UMIST  */

			mole.c[ipHCN][ipHCN] -= HeP_HCN_N_CHP_He;
			mole.c[ipHCN][ipCHP] += HeP_HCN_N_CHP_He;
			mole.c[ipHCN][ipATN] += HeP_HCN_N_CHP_He;

		/*  >>refer   mol  He+    +  HCN   =>   N+    +     CH +  He     UMIST  */

			mole.c[ipHCN][ipHCN] -= HeP_HCN_NP_CH_He;
			mole.c[ipHCN][ipNP] += HeP_HCN_NP_CH_He;
			mole.c[ipHCN][ipCH] += HeP_HCN_NP_CH_He;

		/*  >>refer   mol  He+    +  HCN   =>   N     +     C+ +  He +  H       UMIST  */

			mole.c[ipHCN][ipHCN] -= HeP_HCN_N_CP_He_H;
			mole.c[ipHCN][ipCP] += HeP_HCN_N_CP_He_H;
			mole.c[ipHCN][ipATN] += HeP_HCN_CNP_He_H;

		/*  >>refer   mol  He+    +  HCN   =>   CN+   +     He +  H      UMIST  */

			mole.c[ipHCN][ipHCN] -= HeP_HCN_CNP_He_H;
			mole.c[ipHCN][ipCNP]  += HeP_HCN_CNP_He_H;

		/*  >>refer   mol  He+    +  N2    =>   N+    +     N  +  He     UMIST  */

			mole.c[ipN2][ipN2] -= HeP_N2_NP_N_He;
			mole.c[ipN2][ipNP] += HeP_N2_NP_N_He;
			mole.c[ipN2][ipATN] += HeP_N2_NP_N_He;

		/*  >>refer   mol  He+    +  NO    =>   O+    +     N  +  He     UMIST  */

			mole.c[ipNO][ipNO] -= HeP_NO_OP_N_He;
			mole.c[ipNO][ipOP] += HeP_NO_OP_N_He;
			mole.c[ipNO][ipATN] += HeP_NO_OP_N_He;

		/*  >>refer   mol  He+    +  NO    =>   O     +     N+ +  He     UMIST  */

			mole.c[ipNO][ipNO] -= HeP_NO_O_NP_He;
			mole.c[ipNO][ipNP] += HeP_NO_O_NP_He;
			mole.c[ipNO][ipATO] += HeP_NO_O_NP_He;

		/*  >>refer   mol  He+    +  HNO   =>   NO+   +     He +  H      UMIST  */

			mole.c[ipHNO][ipHNO] -= HeP_HNO_NOP_He_H;
			mole.c[ipHNO][ipNOP] += HeP_HNO_NOP_He_H;

		/*  >>refer   mol  He+    +  HNO   =>   NO    +     He +  H+     UMIST  */

			mole.c[ipHNO][ipHNO] -= HeP_HNO_NO_He_HP;
			mole.c[ipHNO][ipNO] += HeP_HNO_NO_He_HP;

		/*  >>refer   mol  He+    +  HS    =>   S+    +     He +  H      UMIST  */

			mole.c[ipHS][ipHS] -= HeP_HS_SP_He_H;
			mole.c[ipHS][ipSP] += HeP_HS_SP_He_H;

		/*  >>refer   mol  He+    +  OCN   =>   CN    +     O+ +  He     UMIST  */

			mole.c[ipOCN][ipOCN] -= HeP_OCN_CN_OP_He;
			mole.c[ipOCN][ipCN] += HeP_OCN_CN_OP_He;
			mole.c[ipOCN][ipOP] += HeP_OCN_CN_OP_He;

		/*  >>refer   mol  He+    +  OCN   =>   CN+   +     O  +  He     UMIST  */

			mole.c[ipOCN][ipOCN] -= HeP_OCN_CNP_O_He;
			mole.c[ipOCN][ipCNP] += HeP_OCN_CNP_O_He;
			mole.c[ipOCN][ipATO] += HeP_OCN_CNP_O_He;

		/*  >>refer   mol  He+    +  SIN   =>   SI+   +     N  +  He     UMIST  */

			mole.c[ipSIN][ipSIN] -= HeP_SIN_SIP_N_He;
			mole.c[ipSIN][ipSIP] += HeP_SIN_SIP_N_He;
			mole.c[ipSIN][ipATN] += HeP_SIN_SIP_N_He;

		/*  >>refer   mol  He+    +  N2O   =>   N2    +     O+ +  He     UMIST  */

			mole.c[ipN2O][ipN2O] -= HeP_N2O_N2_OP_He;
			mole.c[ipN2O][ipN2] += HeP_N2O_N2_OP_He;
			mole.c[ipN2O][ipOP] += HeP_N2O_N2_OP_He;

		/*  >>refer   mol  He+    +  N2O   =>   N2+   +     O  +  He     UMIST  */

			mole.c[ipN2O][ipN2O] -= HeP_N2O_N2P_O_He;
			mole.c[ipN2O][ipN2P] += HeP_N2O_N2P_O_He;
			mole.c[ipN2O][ipATO] += HeP_N2O_N2P_O_He;

		/*  >>refer   mol  He+    +  N2O   =>   NO    +     N+ +  He     UMIST  */

			mole.c[ipN2O][ipN2O] -= HeP_N2O_NO_NP_He;
			mole.c[ipN2O][ipNO] += HeP_N2O_NO_NP_He;
			mole.c[ipN2O][ipNP] += HeP_N2O_NO_NP_He;

		/*  >>refer   mol  He+    +  N2O   =>   NO+   +     N  +  He     UMIST  */

			mole.c[ipN2O][ipN2O] -= HeP_N2O_NOP_N_He;
			mole.c[ipN2O][ipNOP] += HeP_N2O_NOP_N_He;
			mole.c[ipN2O][ipATN] += HeP_N2O_NOP_N_He;

		/*  >>refer   mol  He+    +  CS    =>   S+    +     C  +  He     UMIST  */

			mole.c[ipCS][ipCS] -= HeP_CS_SP_C_He;
			mole.c[ipCS][ipSP] += HeP_CS_SP_C_He;
			mole.c[ipCS][ipATC] += HeP_CS_SP_C_He;

		/*  >>refer   mol  He+    +  CS    =>   S     +     C+ +  He     UMIST  */

			mole.c[ipCS][ipCS] -= HeP_CS_S_CP_He;
			mole.c[ipCS][ipCP] += HeP_CS_S_CP_He;
			mole.c[ipCS][ipATS] += HeP_CS_S_CP_He;

		/*  >>refer   mol  He+    +  NS    =>   S     +     N+ +  He     UMIST  */

			mole.c[ipNS][ipNS] -= HeP_NS_S_NP_He;
			mole.c[ipNS][ipNP] += HeP_NS_S_NP_He;
			mole.c[ipNS][ipATS] += HeP_NS_S_NP_He;

		/*  >>refer   mol  He+    +  NS    =>   S+    +     N  +  He     UMIST  */

			mole.c[ipNS][ipNS] -= HeP_NS_SP_N_He;
			mole.c[ipNS][ipSP] += HeP_NS_SP_N_He;
			mole.c[ipNS][ipATN] += HeP_NS_SP_N_He;

		/*  >>refer   mol  He+    +  SO    =>   S     +     O+ +  He     UMIST  */

			mole.c[ipSO][ipSO] -= HeP_SO_S_OP_He;
			mole.c[ipSO][ipOP] += HeP_SO_S_OP_He;
			mole.c[ipSO][ipATS] += HeP_SO_S_OP_He;

		/*  >>refer   mol  He+    +  SO    =>   S+    +     O  +  He     UMIST  */

			mole.c[ipSO][ipSO] -= HeP_SO_SP_O_He;
			mole.c[ipSO][ipSP] += HeP_SO_SP_O_He;
			mole.c[ipSO][ipATO] += HeP_SO_SP_O_He;

		/*  >>refer   mol  He+    +  OCS   =>   S     +     CO+ +  He     UMIST  */

			mole.c[ipOCS][ipOCS] -= HeP_OCS_S_COP_He;
			mole.c[ipOCS][ipCOP] += HeP_OCS_S_COP_He;
			mole.c[ipOCS][ipATS] += HeP_OCS_S_COP_He;

		/*  >>refer   mol  He+    +  OCS   =>   CS+   +     O  +  He     UMIST  */

			mole.c[ipOCS][ipOCS] -= HeP_OCS_CSP_O_He;
			mole.c[ipOCS][ipCSP] += HeP_OCS_CSP_O_He;
			mole.c[ipOCS][ipATO] += HeP_OCS_CSP_O_He;

		/*  >>refer   mol  He+    +  OCS   =>   CS    +     O+ +  He     UMIST  */

			mole.c[ipOCS][ipOCS] -= HeP_OCS_CS_OP_He;
			mole.c[ipOCS][ipCS] += HeP_OCS_CS_OP_He;
			mole.c[ipOCS][ipOP] += HeP_OCS_CS_OP_He;

		/*  >>refer   mol  He+    +  S2    =>   S+    +     S  +  He     UMIST  */

			mole.c[ipS2][ipS2] -= HeP_S2_SP_S_He;
			mole.c[ipS2][ipSP] += HeP_S2_SP_S_He;
			mole.c[ipS2][ipATS] += HeP_S2_SP_S_He;

		/*  >>refer   mol  H+     +  NH    =>   NH+   +     H     UMIST  */

			mole.c[ipNH][ipNH] -= HP_NH_NHP_H;
			mole.c[ipNH][ipNHP] += HP_NH_NHP_H;

		/*  >>refer   mol  H+     +  NH2   =>   NH2+  +     H     UMIST  */

			mole.c[ipNH2][ipNH2] -= HP_NH2_NH2P_H;
			mole.c[ipNH2][ipNH2P] += HP_NH2_NH2P_H;

		/*  >>refer   mol  H+     +  NH3   =>   NH3+  +     H     UMIST  */

			mole.c[ipNH3][ipNH3] -= HP_NH3_NH3P_H;
			mole.c[ipNH3][ipNH3P] += HP_NH3_NH3P_H;

		/*  >>refer   mol  H      +  CN+   =>   CN    +     H+    UMIST  */

			mole.c[ipCNP][ipCNP] -= H_CNP_CN_HP;
			mole.c[ipCNP][ipCN] += H_CNP_CN_HP;

		/*  >>refer   mol  H+     +  HCN   =>   HCN+  +     H     UMIST  */

			mole.c[ipHCN][ipHCN] -= HP_HCN_HCNP_H;
			mole.c[ipHCN][ipHCNP] += HP_HCN_HCNP_H;

		/*  >>refer   mol  H      +  HCN+  =>   HCN   +     H+    UMIST  */

			mole.c[ipHCNP][ipHCNP] -= H_HCNP_HCN_HP;
			mole.c[ipHCNP][ipHCN] += H_HCNP_HCN_HP;

		/*  >>refer   mol  H      +  N2+   =>   N2    +     H+    UMIST  */

			mole.c[ipN2P][ipN2P] -= H_N2P_N2_HP;
			mole.c[ipN2P][ipN2] += H_N2P_N2_HP;

		/*  >>refer   mol  H+     +  NO    =>   NO+   +     H     UMIST  */

			mole.c[ipNO][ipNO] -= HP_NO_NOP_H;
			mole.c[ipNO][ipNOP] += HP_NO_NOP_H;

		/*  >>refer   mol  H+     +  HS    =>   HS+   +     H     UMIST  */

			mole.c[ipHS][ipHS] -= HP_HS_HSP_H;
			mole.c[ipHS][ipHSP] += HP_HS_HSP_H;

		/*  >>refer   mol  H+     +  SIN   =>   SIN+  +     H     UMIST  */

			mole.c[ipSIN][ipSIN] -= HP_SIN_SINP_H;
			mole.c[ipSIN][ipSINP] += HP_SIN_SINP_H;

		/*  >>refer   mol  H+     +  CS    =>   CS+   +     H     UMIST  */

			mole.c[ipCS][ipCS] -= HP_CS_CSP_H;
			mole.c[ipCS][ipCSP] += HP_CS_CSP_H;

		/*  >>refer   mol  H+     +  NS    =>   NS+   +     H     UMIST  */

			mole.c[ipNS][ipNS] -= HP_NS_NSP_H;
			mole.c[ipNS][ipNSP] += HP_NS_NSP_H;

		/*  >>refer   mol  H+     +  SO    =>   SO+   +     H     UMIST  */

			mole.c[ipSO][ipSO] -= HP_SO_SOP_H;
			mole.c[ipSO][ipSOP] += HP_SO_SOP_H;

		/*  >>refer   mol  H+     +  OCS   =>   OCS+  +     H     UMIST  */

			mole.c[ipOCS][ipOCS] -= HP_OCS_OCSP_H;
			mole.c[ipOCS][ipOCSP] += HP_OCS_OCSP_H;

		/*  >>refer   mol  H+     +  S2    =>   S2+   +     H     UMIST  */

			mole.c[ipS2][ipS2] -= HP_S2_S2P_H;
			mole.c[ipS2][ipS2P] += HP_S2_S2P_H;

		/*  >>refer   mol  H2+    +  NH    =>   NH+   +     H2    UMIST  */

			mole.c[ipNH][ipNH] -= H2P_NH_NHP_H2;
			mole.c[ipNH][ipNHP] += H2P_NH_NHP_H2;

		/*  >>refer   mol  H2+    +  NH2   =>   NH2+  +     H2    UMIST  */

			mole.c[ipNH2][ipNH2] -= H2P_NH2_NH2P_H2;
			mole.c[ipNH2][ipNH2P] += H2P_NH2_NH2P_H2;

		/*  >>refer   mol  H2+    +  NH3   =>   NH3+  +     H2    UMIST  */

			mole.c[ipNH3][ipNH3] -= H2P_NH3_NH3P_H2;
			mole.c[ipNH3][ipNH3P] += H2P_NH3_NH3P_H2;

		/*  >>refer   mol  H2+    +  CN    =>   CN+   +     H2    UMIST  */

			mole.c[ipCN][ipCN] -= H2P_CN_CNP_H2;
			mole.c[ipCN][ipCNP] += H2P_CN_CNP_H2;

		/*  >>refer   mol  H2+    +  HCN   =>   HCN+  +     H2    UMIST  */

			mole.c[ipHCN][ipHCN] -= H2P_HCN_HCNP_H2;
			mole.c[ipHCN][ipHCNP] += H2P_HCN_HCNP_H2;

		/*  >>refer   mol  H2+    +  NO    =>   NO+   +     H2    UMIST  */

			mole.c[ipNO][ipNO] -= H2P_NO_NOP_H2;
			mole.c[ipNO][ipNOP] += H2P_NO_NOP_H2;

		/*  >>refer   mol  He+    +  NH3   =>   NH3+  +     He    UMIST  */

			mole.c[ipNH3][ipNH3] -= HeP_NH3_NH3P_He;
			mole.c[ipNH3][ipNH3P] += HeP_NH3_NH3P_He;

		/*  >>refer   mol  He+    +  N2    =>   N2+   +     He    UMIST  */

			mole.c[ipN2][ipN2] -= HeP_N2_N2P_He;
			mole.c[ipN2][ipN2P] += HeP_N2_N2P_He;

		/*  >>refer   mol  H-     +  N     =>   NH    +     e     UMIST  */

			mole.c[ipATN][ipATN] -= Hminus_N_NH_e;
			mole.c[ipATN][ipNH] += Hminus_N_NH_e;
			
		/*  >>refer   mol  H-     +  NH    =>   NH2   +     e     UMIST  */

			mole.c[ipNH][ipNH] -= Hminus_NH_NH2_e;
			mole.c[ipNH][ipNH2] += Hminus_NH_NH2_e;

		/*  >>refer   mol  H-     +  NH2   =>   NH3   +     e     UMIST  */

			mole.c[ipNH2][ipNH2] -= Hminus_NH2_NH3_e;
			mole.c[ipNH2][ipNH3] += Hminus_NH2_NH3_e;

		/*  >>refer   mol  H-     +  CN    =>   HCN   +     e     UMIST  */

			mole.c[ipCN][ipCN] -= Hminus_CN_HCN_e;
			mole.c[ipCN][ipHCN] += Hminus_CN_HCN_e;

		/*  >>refer   mol  H-     +  NH4+  =>   NH3   +     H2    UMIST  */

			mole.c[ipNH4P][ipNH4P] -= Hminus_NH4P_NH3_H2;
			mole.c[ipNH4P][ipNH3] += Hminus_NH4P_NH3_H2;

		/*  >>refer   mol  H-     +  N+    =>   N     +     H	    UMIST  */

			mole.c[ipNP][ipNP] -= Hminus_NP_N_H;
			mole.c[ipNP][ipATN] += Hminus_NP_N_H;
		/*  >>refer   mol  H2    +  Cl+     =>   HCl+    +     H    UMIST  */

		mole.c[ipClP][ipClP] -= H2_ClP_HClP_H;
		mole.c[ipClP][ipHClP] += H2_ClP_HClP_H;

		/*  >>refer   mol  H2    +  HCl+     =>   H2Cl+    +     H    UMIST  */

		mole.c[ipHClP][ipHClP] -= H2_HClP_H2ClP_H;
		mole.c[ipHClP][ipH2ClP] += H2_HClP_H2ClP_H;

		/*  >>refer   mol  H3+    +  Cl     =>   HCl+    +     H2    UMIST  */

		mole.c[ipATCl][ipATCl] -= H3P_Cl_HClP_H2;
		mole.c[ipATCl][ipHClP] += H3P_Cl_HClP_H2;

		/*  >>refer   mol  H3+    +  HCl     =>   H2Cl+    +     H2    UMIST  */

		mole.c[ipHCl][ipHCl] -= H3P_HCl_H2ClP_H2;
		mole.c[ipHCl][ipH2ClP] += H3P_HCl_H2ClP_H2;

		/*  >>refer   mol  He+    +  HCl     =>   Cl+    +     He	+	H    UMIST  */

		mole.c[ipHCl][ipHCl] -= HeP_HCl_ClP_He_H;
		mole.c[ipHCl][ipClP] += HeP_HCl_ClP_He_H;

		/*  >>refer   mol  He+    +  CCl     =>   Cl    +     C+	+     He   UMIST  */

		mole.c[ipCCl][ipCCl] -= HeP_CCl_Cl_CP_He;
		mole.c[ipCCl][ipATCl] += HeP_CCl_Cl_CP_He;
		mole.c[ipCCl][ipCP] += HeP_CCl_Cl_CP_He;

		/*  >>refer   mol  He+    +  ClO     =>   Cl+    +     O	+     He   UMIST  */

		mole.c[ipClO][ipClO] -= HeP_ClO_ClP_O_He;
		mole.c[ipClO][ipClP] += HeP_ClO_ClP_O_He;
		mole.c[ipClO][ipATO] += HeP_ClO_ClP_O_He;
		/*  >>refer   mol  H+    +  HCl     =>   HCl+    +     H    UMIST  */

		mole.c[ipHCl][ipHCl] -= HP_HCl_HClP_H;
		mole.c[ipHCl][ipHClP] += HP_HCl_HClP_H;


		/* >>chng 05 sept 13 - NPA.  Add HNC and HCNH+ to chemistry */

		/*  >>refer  mol  H3P   +   HNC   =   HCNHP   +   H2   */
		
		mole.c[ipHNC][ipHNC] -= H3P_HNC_HCNHP_H2; 
		mole.c[ipHNC][ipHCNHP] += H3P_HNC_HCNHP_H2;

		/*  >>refer  mol  HeP   +   HNC   =   NHP   +   C   +   He */
		
		mole.c[ipHNC][ipHNC] -= HeP_HNC_NHP_C_He; 
		mole.c[ipHNC][ipNHP] += HeP_HNC_NHP_C_He; 
		mole.c[ipHNC][ipATC] += HeP_HNC_NHP_C_He; 

		/*  >>refer  mol  HeP   +   HNC   =   N   +   CP   +   He   +   H  */

		mole.c[ipHNC][ipHNC] -= HeP_HNC_N_CP_He_H; 
		mole.c[ipHNC][ipATN] += HeP_HNC_N_CP_He_H; 
		mole.c[ipHNC][ipCP] += HeP_HNC_N_CP_He_H; 

		/*  >>refer  mol  HeP   +   HNC   =   CNP   +   He   +   H */

		mole.c[ipHNC][ipHNC] -= HeP_HNC_CNP_He_H; 
		mole.c[ipHNC][ipCNP] += HeP_HNC_CNP_He_H; 

		/*  >>refer  mol  HP   +   HNC   =   HCN   +   HP   */

		mole.c[ipHNC][ipHNC] -= HP_HNC_HCN_HP; 
		mole.c[ipHNC][ipHCN] += HP_HNC_HCN_HP; 

		/*  >>refer  mol  H   +   HNC   =   HCN   +   H   */

		mole.c[ipHNC][ipHNC] -= H_HNC_HCN_H; 
		mole.c[ipHNC][ipHCN] += H_HNC_HCN_H; 

		/*  >>refer  mol  H2   +   HCNP   =   HCNHP   +   H   */

		mole.c[ipHCNP][ipHCNP] -= H2_HCNP_HCNHP_H; 
		mole.c[ipHCNP][ipHCNHP] += H2_HCNP_HCNHP_H; 

		/*  >>refer  mol  H3P   +   HCN   =   HCNHP   +   H2   */

		mole.c[ipHNC][ipHNC] -= H3P_HCN_HCNHP_H2; 
		mole.c[ipHNC][ipHCNHP] += H3P_HCN_HCNHP_H2; 

		/* >>refer	mol	H		C2	=>		CH		C		UMIST	*/

		mole.c[ipC2][ipC2] -= H_C2_CH_C;
		mole.c[ipC2][ipCH]  += H_C2_CH_C;
		mole.c[ipC2][ipATC]  += H_C2_CH_C;

		/* >>refer	mol	H+		C2	=>		C2+		H		UMIST	*/
		
		mole.c[ipC2][ipC2] -= HP_C2_C2P_H;
		mole.c[ipC2][ipC2P]  += HP_C2_C2P_H;

		/* >>refer	mol	H2+		C2	=>		C2+		H2		UMIST	*/
		
		mole.c[ipC2][ipC2] -= H2P_C2_C2P_H2;
		mole.c[ipC2][ipC2P]  += H2P_C2_C2P_H2;

		/* >>refer	mol	He+		C2	=>		C2+		He		UMIST	*/
		
		mole.c[ipC2][ipC2] -= HeP_C2_C2P_He;
		mole.c[ipC2][ipC2P]  += HeP_C2_C2P_He;

		/* >>refer  mole   He+   C2H   -->   C2+   He   H     */

		mole.c[ipC2H][ipC2H] -= HeP_C2H_C2P_He_H;
		mole.c[ipC2H][ipC2P] += HeP_C2H_C2P_He_H;

		/* >>refer  mole   He+   C2H   -->   CH+   C   He     */

		mole.c[ipC2H][ipC2H] -= HeP_C2H_CHP_C_He;
		mole.c[ipC2H][ipCHP] += HeP_C2H_CHP_C_He;
		mole.c[ipC2H][ipATC] += HeP_C2H_CHP_C_He;

		/* >>refer  mole   He+   C2H   -->   CH   C+   He     */

		mole.c[ipC2H][ipC2H] -= HeP_C2H_CH_CP_He;
		mole.c[ipC2H][ipCH] += HeP_C2H_CH_CP_He;
		mole.c[ipC2H][ipCP] += HeP_C2H_CH_CP_He;

		/* >>refer  mole   H-   C2   -->   C2H   e-        */

		mole.c[ipC2][ipC2] -= Hm_C2_C2H_e;
		mole.c[ipC2][ipC2H] += Hm_C2_C2H_e;
		
		/* >>refer  mole   H+   C2H   -->   C2+   H2        */

		mole.c[ipC2H][ipC2H] -= HP_C2H_C2P_H2;
		mole.c[ipC2H][ipC2P] += HP_C2H_C2P_H2;

		/* >>refer  mole   H+   C2H   -->   C2H+   H        */

		mole.c[ipC2H][ipC2H] -= HP_C2H_C2HP_H;
		mole.c[ipC2H][ipC2HP] += HP_C2H_C2HP_H;
	
		/* >>refer  mole   H2+   C2   -->   C2H+   H        */

		mole.c[ipC2][ipC2] -= H2P_C2_C2HP_H;
		mole.c[ipC2][ipC2HP] += H2P_C2_C2HP_H;
	
		/* >>refer  mole   H2+   C2H   -->   C2H+   H2        */

		mole.c[ipC2H][ipC2H] -= H2P_C2H_C2HP_H2;
		mole.c[ipC2H][ipC2HP] += H2P_C2H_C2HP_H2;

		/* >>refer  mole   H3+   C2   -->   C2H+   H2        */

		mole.c[ipC2][ipC2] -= H3P_C2_C2HP_H2;
		mole.c[ipC2][ipC2HP] += H3P_C2_C2HP_H2;

		/* >>refer  mole   H2   C2+   -->   C2H+   H        */

		mole.c[ipC2P][ipC2P] -= H2_C2P_C2HP_H;
		mole.c[ipC2P][ipC2HP] += H2_C2P_C2HP_H;

		/* >>refer  mole   H+   C3   -->   C3+   H        */

		mole.c[ipC3][ipC3] -= HP_C3_C3P_H;
		mole.c[ipC3][ipC3P] += HP_C3_C3P_H;

		/* >>refer  mole   He+   C3   -->   C2   C+   He     */

		mole.c[ipC3][ipC3] -= HeP_C3_C2_CP_He;
		mole.c[ipC3][ipC2] += HeP_C3_C2_CP_He;
		mole.c[ipC3][ipCP] += HeP_C3_C2_CP_He;

		/* >>refer  mole   He+   C2H2   -->   C2H+   He   H     */

		mole.c[ipC2H2][ipC2H2] -= HeP_C2H2_C2HP_He_H;
		mole.c[ipC2H2][ipC2HP] += HeP_C2H2_C2HP_He_H;

		/* >>refer  mole   He+   C3H   -->   C3+   He   H     */

		mole.c[ipC3H][ipC3H] -= HeP_C3H_C3P_He_H;
		mole.c[ipC3H][ipC3P] += HeP_C3H_C3P_He_H;

		/* >>refer  mole   He+   C2H2   -->   C2+   He   H2     */	

		mole.c[ipC2H2][ipC2H2] -= HeP_C2H2_C2P_He_H2;
		mole.c[ipC2H2][ipC2P] += HeP_C2H2_C2P_He_H2;

		/* >>refer  mole   He+   C2H2   -->   CH+   CH   He     */

		mole.c[ipC2H2][ipC2H2] -= HeP_C2H2_CHP_CH_He;
		mole.c[ipC2H2][ipCHP] += HeP_C2H2_CHP_CH_He;
		mole.c[ipC2H2][ipCH] += HeP_C2H2_CHP_CH_He;

		/* >>refer  mole   H-   C2H   -->   C2H2   e-        */

		mole.c[ipC2H][ipC2H] -= Hm_C2H_C2H2_e;
		mole.c[ipC2H][ipC2H2] += Hm_C2H_C2H2_e;

		/* >>refer  mole   H+   C2H2   -->   C2H2+   H        */

		mole.c[ipC2H2][ipC2H2] -= HP_C2H2_C2H2P_H;
		mole.c[ipC2H2][ipC2H2P] += HP_C2H2_C2H2P_H;

		/* >>refer  mole   H+   C2H2   -->   C2H+   H2        */

		mole.c[ipC2H2][ipC2H2] -= HP_C2H2_C2HP_H2;
		mole.c[ipC2H2][ipC2HP] += HP_C2H2_C2HP_H2;

		/* >>refer  mole   H+   C3H   -->   C3H+   H        */

		mole.c[ipC3H][ipC3H] -= HP_C3H_C3HP_H;
		mole.c[ipC3H][ipC3HP] += HP_C3H_C3HP_H;

		/* >>refer  mole   H+   C3H   -->   C3+   H2        */

		mole.c[ipC3H][ipC3H] -= HP_C3H_C3P_H2;
		mole.c[ipC3H][ipC3P] += HP_C3H_C3P_H2;

		/* >>refer  mole   H2+   C2H   -->   C2H2+   H        */

		mole.c[ipC2H][ipC2H] -= H2P_C2H_C2H2P_H;
		mole.c[ipC2H][ipC2H2P] += H2P_C2H_C2H2P_H;

		/* >>refer  mole   H2+   C2H2   -->   C2H2+   H2        */

		mole.c[ipC2H2][ipC2H2] -= H2P_C2H2_C2H2P_H2;
		mole.c[ipC2H2][ipC2H2P] += H2P_C2H2_C2H2P_H2;

		/* >>refer  mole   H3+   C2H   -->   C2H2+   H2        */

		mole.c[ipC2H][ipC2H] -= H3P_C2H_C2H2P_H2;
		mole.c[ipC2H][ipC2H2P] += H3P_C2H_C2H2P_H2;

		/* >>refer  mole   H3+   C3   -->   C3H+   H2        */

		mole.c[ipC3][ipC3] -= H3P_C3_C3HP_H2;
		mole.c[ipC3][ipC3HP] += H3P_C3_C3HP_H2;

		/* >>refer  mole   H2   C2H+   -->   C2H2+   H        */

		mole.c[ipC2HP][ipC2HP] -= H2_C2HP_C2H2P_H;
		mole.c[ipC2HP][ipC2H2P] += H2_C2HP_C2H2P_H;

		/* >>refer  mole   H2   C3+   -->   C3H+   H        */

		mole.c[ipC3P][ipC3P] -= H2_C3P_C3HP_H;
		mole.c[ipC3P][ipC3HP] += H2_C3P_C3HP_H;

		/* >>refer  mole   He+   C2H2   -->   C2H2+   He        */

		mole.c[ipC2H2][ipC2H2] -= HeP_C2H2_C2H2P_He;
		mole.c[ipC2H2][ipC2H2P] += HeP_C2H2_C2H2P_He;
											
		/* >>refer  mole   H   C2H3+   -->   C2H2+   H2        */										
												
		mole.c[ipC2H3P][ipC2H3P] -= H_C2H3P_C2H2P_H2;										
		mole.c[ipC2H3P][ipC2H2P] += H_C2H3P_C2H2P_H2;										

		/* >>refer  mole   H2+   C2H2   -->   C2H3+   H        */

		mole.c[ipC2H2][ipC2H2] -= H2P_C2H2_C2H3P_H;
		mole.c[ipC2H2][ipC2H3P] += H2P_C2H2_C2H3P_H;

		/* >>refer  mole   H3+   C2H2   -->   C2H3+   H2        */

		mole.c[ipC2H2][ipC2H2] -= H3P_C2H2_C2H3P_H2;
		mole.c[ipC2H2][ipC2H3P] += H3P_C2H2_C2H3P_H2;

}
